<?php
/**
 * Queue Management System Setup Script - Standalone Version
 * Run this file ONCE to initialize the queue management system
 * 
 * Usage: 
 * - Upload this file to your web root directory
 * - Run via web browser: http://yourdomain.com/setup_queue_system_fixed.php
 * - Or run via CLI: php setup_queue_system_fixed.php
 */

// Database configuration - Update these values
$db_config = [
    'hostname' => 'localhost',
    'username' => 'your_db_username',    // Update this
    'password' => 'your_db_password',    // Update this
    'database' => 'your_db_name',        // Update this
    'port' => 3306
];

// HTML header
echo "<!DOCTYPE html><html><head><title>Queue System Setup</title>";
echo "<style>body{font-family:Arial,sans-serif;margin:40px;} .success{color:green;} .warning{color:orange;} .error{color:red;} .info{background:#d4edda;border:1px solid #c3e6cb;padding:15px;margin:20px 0;border-radius:5px;}</style>";
echo "</head><body>";

echo "<h1>🏥 Queue Management System Setup</h1>";
echo "<p>Initializing queue management system for City Health HMS...</p>";

try {
    // Connect to database
    $mysqli = new mysqli(
        $db_config['hostname'],
        $db_config['username'],
        $db_config['password'],
        $db_config['database'],
        $db_config['port']
    );

    // Check connection
    if ($mysqli->connect_error) {
        throw new Exception("Database connection failed: " . $mysqli->connect_error);
    }
    
    echo "<p class='success'>✅ Database connection successful</p>";
    
    // Set charset
    $mysqli->set_charset("utf8mb4");
    
    // =============================================
    // 1. CREATE QUEUE TABLES
    // =============================================
    echo "<h2>📊 Creating Database Tables</h2>";
    
    // Check if patient_queue table exists
    $result = $mysqli->query("SHOW TABLES LIKE 'patient_queue'");
    if ($result->num_rows == 0) {
        echo "<p>Creating patient_queue table...</p>";
        
        $sql = "
        CREATE TABLE `patient_queue` (
            `id` int(11) NOT NULL AUTO_INCREMENT,
            `patient_id` int(11) NOT NULL,
            `queue_number` varchar(20) NOT NULL,
            `department` varchar(50) NOT NULL,
            `status` enum('waiting','in_progress','completed','skipped','transferred') DEFAULT 'waiting',
            `priority` enum('low','normal','high','urgent') DEFAULT 'normal',
            `queue_type` varchar(50) DEFAULT 'walk_in',
            `workflow_type` varchar(50) DEFAULT 'general',
            `estimated_wait_time` int(11) DEFAULT NULL COMMENT 'Estimated wait time in minutes',
            `actual_wait_time` int(11) DEFAULT NULL COMMENT 'Actual wait time in minutes',
            `called_at` datetime DEFAULT NULL,
            `completed_at` datetime DEFAULT NULL,
            `served_by` int(11) DEFAULT NULL COMMENT 'Staff ID who served the patient',
            `notes` text DEFAULT NULL,
            `next_queue` varchar(50) DEFAULT NULL COMMENT 'Next department if transferred',
            `created_at` datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
            `updated_at` datetime DEFAULT NULL ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (`id`),
            KEY `idx_patient_department` (`patient_id`,`department`),
            KEY `idx_department_status` (`department`,`status`),
            KEY `idx_queue_created` (`queue_number`,`created_at`)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
        ";
        
        if ($mysqli->query($sql)) {
            echo "<span class='success'>✅ patient_queue table created</span><br>";
        } else {
            throw new Exception("Error creating patient_queue table: " . $mysqli->error);
        }
    } else {
        echo "<span class='warning'>⚠️ patient_queue table already exists</span><br>";
    }
    
    // Check if queue_config table exists
    $result = $mysqli->query("SHOW TABLES LIKE 'queue_config'");
    if ($result->num_rows == 0) {
        echo "<p>Creating queue_config table...</p>";
        
        $sql = "
        CREATE TABLE `queue_config` (
            `id` int(11) NOT NULL AUTO_INCREMENT,
            `department` varchar(50) NOT NULL,
            `queue_prefix` varchar(10) NOT NULL DEFAULT 'Q',
            `counter_current` int(11) DEFAULT 1,
            `counter_reset` enum('daily','weekly','monthly','never') DEFAULT 'daily',
            `average_service_time` int(11) DEFAULT 15 COMMENT 'Average service time in minutes',
            `max_queue_size` int(11) DEFAULT 100,
            `auto_call_enabled` tinyint(1) DEFAULT 0,
            `sound_enabled` tinyint(1) DEFAULT 1,
            `display_enabled` tinyint(1) DEFAULT 1,
            `is_active` tinyint(1) DEFAULT 1,
            `created_at` datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
            `updated_at` datetime DEFAULT NULL ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (`id`),
            UNIQUE KEY `uk_department` (`department`)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
        ";
        
        if ($mysqli->query($sql)) {
            echo "<span class='success'>✅ queue_config table created</span><br>";
        } else {
            throw new Exception("Error creating queue_config table: " . $mysqli->error);
        }
    } else {
        echo "<span class='warning'>⚠️ queue_config table already exists</span><br>";
    }
    
    // Check if queue_display_settings table exists
    $result = $mysqli->query("SHOW TABLES LIKE 'queue_display_settings'");
    if ($result->num_rows == 0) {
        echo "<p>Creating queue_display_settings table...</p>";
        
        $sql = "
        CREATE TABLE `queue_display_settings` (
            `id` int(11) NOT NULL AUTO_INCREMENT,
            `department` varchar(50) NOT NULL,
            `display_name` varchar(100) NOT NULL,
            `theme_color` varchar(20) DEFAULT '#007bff',
            `icon_class` varchar(50) DEFAULT 'fa-users',
            `display_order` int(11) DEFAULT 0,
            `is_visible` tinyint(1) DEFAULT 1,
            PRIMARY KEY (`id`),
            KEY `idx_display_order` (`display_order`)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
        ";
        
        if ($mysqli->query($sql)) {
            echo "<span class='success'>✅ queue_display_settings table created</span><br>";
        } else {
            throw new Exception("Error creating queue_display_settings table: " . $mysqli->error);
        }
    } else {
        echo "<span class='warning'>⚠️ queue_display_settings table already exists</span><br>";
    }
    
    // Check if queue_flow_rules table exists
    $result = $mysqli->query("SHOW TABLES LIKE 'queue_flow_rules'");
    if ($result->num_rows == 0) {
        echo "<p>Creating queue_flow_rules table...</p>";
        
        $sql = "
        CREATE TABLE `queue_flow_rules` (
            `id` int(11) NOT NULL AUTO_INCREMENT,
            `rule_name` varchar(100) NOT NULL,
            `from_department` varchar(50) NOT NULL,
            `to_department` varchar(50) NOT NULL,
            `condition_type` enum('automatic','manual','conditional') DEFAULT 'manual',
            `condition_data` text DEFAULT NULL,
            `workflow_type` varchar(50) DEFAULT 'general',
            `priority_mapping` varchar(20) DEFAULT 'maintain' COMMENT 'maintain, increase, decrease, or specific priority',
            `is_active` tinyint(1) DEFAULT 1,
            `created_at` datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (`id`),
            KEY `idx_from_condition` (`from_department`,`condition_type`)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
        ";
        
        if ($mysqli->query($sql)) {
            echo "<span class='success'>✅ queue_flow_rules table created</span><br>";
        } else {
            throw new Exception("Error creating queue_flow_rules table: " . $mysqli->error);
        }
    } else {
        echo "<span class='warning'>⚠️ queue_flow_rules table already exists</span><br>";
    }
    
    // Check if queue_notifications table exists
    $result = $mysqli->query("SHOW TABLES LIKE 'queue_notifications'");
    if ($result->num_rows == 0) {
        echo "<p>Creating queue_notifications table...</p>";
        
        $sql = "
        CREATE TABLE `queue_notifications` (
            `id` int(11) NOT NULL AUTO_INCREMENT,
            `department` varchar(50) NOT NULL,
            `notification_type` varchar(50) NOT NULL,
            `patient_id` int(11) DEFAULT NULL,
            `queue_id` int(11) DEFAULT NULL,
            `message` text NOT NULL,
            `metadata` text DEFAULT NULL,
            `is_read` tinyint(1) DEFAULT 0,
            `expires_at` datetime DEFAULT NULL,
            `created_at` datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (`id`),
            KEY `idx_department_read` (`department`,`is_read`),
            KEY `idx_expires` (`expires_at`)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
        ";
        
        if ($mysqli->query($sql)) {
            echo "<span class='success'>✅ queue_notifications table created</span><br>";
        } else {
            throw new Exception("Error creating queue_notifications table: " . $mysqli->error);
        }
    } else {
        echo "<span class='warning'>⚠️ queue_notifications table already exists</span><br>";
    }
    
    // =============================================
    // 2. INSERT DEFAULT CONFIGURATION DATA
    // =============================================
    echo "<h2>⚙️ Inserting Default Configuration</h2>";
    
    // Default queue configuration
    $departments = [
        ['department' => 'registration', 'queue_prefix' => 'REG', 'avg_service_time' => 10],
        ['department' => 'triage', 'queue_prefix' => 'TRI', 'avg_service_time' => 15],
        ['department' => 'consultation', 'queue_prefix' => 'CON', 'avg_service_time' => 20],
        ['department' => 'pharmacy', 'queue_prefix' => 'PHM', 'avg_service_time' => 8],
        ['department' => 'laboratory', 'queue_prefix' => 'LAB', 'avg_service_time' => 12],
        ['department' => 'radiology', 'queue_prefix' => 'RAD', 'avg_service_time' => 25],
        ['department' => 'phlebotomy', 'queue_prefix' => 'PHL', 'avg_service_time' => 5],
        ['department' => 'emergency', 'queue_prefix' => 'EMR', 'avg_service_time' => 10]
    ];
    
    foreach ($departments as $dept) {
        // Check if config already exists
        $stmt = $mysqli->prepare("SELECT COUNT(*) as count FROM queue_config WHERE department = ?");
        $stmt->bind_param("s", $dept['department']);
        $stmt->execute();
        $result = $stmt->get_result();
        $row = $result->fetch_assoc();
        
        if ($row['count'] == 0) {
            $stmt = $mysqli->prepare("
                INSERT INTO queue_config 
                (department, queue_prefix, average_service_time, counter_current, counter_reset, max_queue_size, sound_enabled, display_enabled, is_active) 
                VALUES (?, ?, ?, 1, 'daily', 100, 1, 1, 1)
            ");
            $stmt->bind_param("ssi", $dept['department'], $dept['queue_prefix'], $dept['avg_service_time']);
            
            if ($stmt->execute()) {
                echo "<span class='success'>✅ {$dept['department']} queue config added</span><br>";
            } else {
                echo "<span class='error'>❌ Error adding {$dept['department']} config: " . $stmt->error . "</span><br>";
            }
        } else {
            echo "<span class='warning'>⚠️ {$dept['department']} queue config already exists</span><br>";
        }
    }
    
    // Default display settings
    $display_settings = [
        ['department' => 'registration', 'display_name' => 'Registration', 'theme_color' => '#28a745', 'icon_class' => 'fa-user-plus', 'display_order' => 1],
        ['department' => 'triage', 'display_name' => 'Triage', 'theme_color' => '#17a2b8', 'icon_class' => 'fa-stethoscope', 'display_order' => 2],
        ['department' => 'consultation', 'display_name' => 'Consultation', 'theme_color' => '#007bff', 'icon_class' => 'fa-user-md', 'display_order' => 3],
        ['department' => 'pharmacy', 'display_name' => 'Pharmacy', 'theme_color' => '#fd7e14', 'icon_class' => 'fa-pills', 'display_order' => 4],
        ['department' => 'laboratory', 'display_name' => 'Laboratory', 'theme_color' => '#6f42c1', 'icon_class' => 'fa-vial', 'display_order' => 5],
        ['department' => 'radiology', 'display_name' => 'Radiology', 'theme_color' => '#20c997', 'icon_class' => 'fa-x-ray', 'display_order' => 6],
        ['department' => 'phlebotomy', 'display_name' => 'Phlebotomy', 'theme_color' => '#e83e8c', 'icon_class' => 'fa-syringe', 'display_order' => 7],
        ['department' => 'emergency', 'display_name' => 'Emergency', 'theme_color' => '#dc3545', 'icon_class' => 'fa-ambulance', 'display_order' => 8]
    ];
    
    foreach ($display_settings as $setting) {
        $stmt = $mysqli->prepare("SELECT COUNT(*) as count FROM queue_display_settings WHERE department = ?");
        $stmt->bind_param("s", $setting['department']);
        $stmt->execute();
        $result = $stmt->get_result();
        $row = $result->fetch_assoc();
        
        if ($row['count'] == 0) {
            $stmt = $mysqli->prepare("
                INSERT INTO queue_display_settings 
                (department, display_name, theme_color, icon_class, display_order, is_visible) 
                VALUES (?, ?, ?, ?, ?, 1)
            ");
            $stmt->bind_param("ssssi", 
                $setting['department'], 
                $setting['display_name'], 
                $setting['theme_color'], 
                $setting['icon_class'], 
                $setting['display_order']
            );
            
            if ($stmt->execute()) {
                echo "<span class='success'>✅ {$setting['department']} display settings added</span><br>";
            } else {
                echo "<span class='error'>❌ Error adding {$setting['department']} display settings: " . $stmt->error . "</span><br>";
            }
        } else {
            echo "<span class='warning'>⚠️ {$setting['department']} display settings already exist</span><br>";
        }
    }
    
    // Default flow rules
    $flow_rules = [
        [
            'rule_name' => 'Registration to Triage',
            'from_department' => 'registration',
            'to_department' => 'triage',
            'condition_type' => 'automatic',
            'workflow_type' => 'general',
            'priority_mapping' => 'maintain'
        ],
        [
            'rule_name' => 'Triage to Consultation',
            'from_department' => 'triage',
            'to_department' => 'consultation',
            'condition_type' => 'manual',
            'workflow_type' => 'general',
            'priority_mapping' => 'maintain'
        ]
    ];
    
    foreach ($flow_rules as $rule) {
        $stmt = $mysqli->prepare("SELECT COUNT(*) as count FROM queue_flow_rules WHERE rule_name = ?");
        $stmt->bind_param("s", $rule['rule_name']);
        $stmt->execute();
        $result = $stmt->get_result();
        $row = $result->fetch_assoc();
        
        if ($row['count'] == 0) {
            $stmt = $mysqli->prepare("
                INSERT INTO queue_flow_rules 
                (rule_name, from_department, to_department, condition_type, workflow_type, priority_mapping, is_active) 
                VALUES (?, ?, ?, ?, ?, ?, 1)
            ");
            $stmt->bind_param("ssssss", 
                $rule['rule_name'], 
                $rule['from_department'], 
                $rule['to_department'], 
                $rule['condition_type'], 
                $rule['workflow_type'], 
                $rule['priority_mapping']
            );
            
            if ($stmt->execute()) {
                echo "<span class='success'>✅ Flow rule '{$rule['rule_name']}' added</span><br>";
            } else {
                echo "<span class='error'>❌ Error adding flow rule '{$rule['rule_name']}': " . $stmt->error . "</span><br>";
            }
        } else {
            echo "<span class='warning'>⚠️ Flow rule '{$rule['rule_name']}' already exists</span><br>";
        }
    }
    
    // =============================================
    // 3. VERIFY SETUP
    // =============================================
    echo "<h2>✅ Setup Verification</h2>";
    
    // Count records in each table
    $tables = ['patient_queue', 'queue_config', 'queue_display_settings', 'queue_flow_rules', 'queue_notifications'];
    foreach ($tables as $table) {
        $result = $mysqli->query("SELECT COUNT(*) as count FROM $table");
        $row = $result->fetch_assoc();
        echo "<p>📊 $table: {$row['count']} records</p>";
    }
    
    echo "<h2>🎉 Queue System Setup Complete!</h2>";
    echo "<div class='info'>";
    echo "<h3>Next Steps:</h3>";
    echo "<ol>";
    echo "<li><strong>Update Database Config:</strong> Make sure to update the database credentials in this script before running</li>";
    echo "<li><strong>Test Patient Registration:</strong> Register a new patient and verify they are added to triage queue</li>";
    echo "<li><strong>Check Department Modules:</strong> Visit triage, consultation, pharmacy pages to see queue integration</li>";
    echo "<li><strong>Configure Notifications:</strong> Verify sound notifications work in each department</li>";
    echo "<li><strong>Customize Settings:</strong> Visit admin/queue to configure department-specific settings</li>";
    echo "<li><strong>Train Staff:</strong> Show staff how to use the queue management features</li>";
    echo "</ol>";
    echo "<p><strong>⚠️ Important:</strong> Delete this setup file after successful installation for security.</p>";
    echo "</div>";
    
    echo "<hr><p><small>Queue Management System v1.0 - Installed successfully!</small></p>";
    
    // Close database connection
    $mysqli->close();
    
} catch (Exception $e) {
    echo "<div style='background: #f8d7da; border: 1px solid #f5c6cb; color: #721c24; padding: 15px; margin: 20px 0; border-radius: 5px;'>";
    echo "<h3>❌ Setup Failed</h3>";
    echo "<p><strong>Error:</strong> " . $e->getMessage() . "</p>";
    echo "<h4>Troubleshooting Tips:</h4>";
    echo "<ul>";
    echo "<li>Check database credentials in the script</li>";
    echo "<li>Ensure MySQL server is running</li>";
    echo "<li>Verify database user has CREATE and INSERT privileges</li>";
    echo "<li>Check if database name exists</li>";
    echo "</ul>";
    echo "</div>";
}

echo "</body></html>";
?>