/**
 * Queue Notification Service
 * Handles real-time notifications across all departments
 * Include this file in all queue management views
 */

class QueueNotificationService {
    constructor(department, baseUrl) {
        this.department = department;
        this.baseUrl = baseUrl;
        this.soundEnabled = true;
        this.notificationHistory = [];
        this.maxHistorySize = 50;
        
        this.init();
    }
    
    init() {
        console.log(`🔔 Initializing Queue Notifications for ${this.department.toUpperCase()}`);
        
        // Check for notifications every 3 seconds
        this.notificationInterval = setInterval(() => {
            this.checkForNotifications();
        }, 3000);
        
        // Initialize audio context
        this.initializeAudio();
        
        // Setup notification UI
        this.setupNotificationUI();
        
        console.log(`✅ Queue Notifications ready for ${this.department}`);
    }
    
    async checkForNotifications() {
        try {
            const response = await fetch(`${this.baseUrl}admin/${this.department}/check_incoming_notifications`);
            const data = await response.json();
            
            if (data.status === 'success' && data.has_notification) {
                this.handleNotification(data.notification);
            }
        } catch (error) {
            // Silently fail for notifications to avoid console spam
        }
    }
    
    handleNotification(notification) {
        // Add to history
        this.notificationHistory.unshift({
            ...notification,
            received_at: new Date()
        });
        
        // Trim history
        if (this.notificationHistory.length > this.maxHistorySize) {
            this.notificationHistory = this.notificationHistory.slice(0, this.maxHistorySize);
        }
        
        // Process notification based on type
        this.processNotification(notification);
        
        // Show visual notification
        this.showVisualNotification(notification);
        
        // Play sound if enabled
        if (this.soundEnabled && notification.sound) {
            this.playNotificationSound(notification.action);
        }
        
        // Trigger custom events for department-specific handling
        this.triggerNotificationEvent(notification);
    }
    
    processNotification(notification) {
        const { action, data, from_department } = notification;
        
        switch (action) {
            case 'new_patient_from_triage':
                this.handleNewPatientFromTriage(data);
                break;
            case 'new_patient_from_consultation':
                this.handleNewPatientFromConsultation(data);
                break;
            case 'lab_results_ready':
                this.handleLabResultsReady(data);
                break;
            case 'radiology_results_ready':
                this.handleRadiologyResultsReady(data);
                break;
            case 'urgent_patient_incoming':
                this.handleUrgentPatient(data);
                break;
            case 'patient_called_for_service':
                this.handlePatientCalled(data);
                break;
            default:
                console.log(`📨 Received notification: ${action}`, data);
        }
    }
    
    handleNewPatientFromTriage(data) {
        const message = `New ${data.priority || 'normal'} priority patient from Triage`;
        this.showNotificationBanner(message, 'info', 8000);
        
        // Auto-refresh queue after short delay
        setTimeout(() => {
            if (typeof loadQueueData === 'function') {
                loadQueueData();
            }
        }, 1000);
    }
    
    handleNewPatientFromConsultation(data) {
        const doctorInfo = data.doctor_name ? ` (Dr. ${data.doctor_name})` : '';
        const message = `New patient from Consultation${doctorInfo}`;
        this.showNotificationBanner(message, 'info', 8000);
        
        setTimeout(() => {
            if (typeof loadQueueData === 'function') {
                loadQueueData();
            }
        }, 1000);
    }
    
    handleLabResultsReady(data) {
        const priority = data.priority === 'high' ? 'URGENT' : '';
        const message = `${priority} Lab results ready for Patient ${data.patient_id}`;
        const alertType = data.priority === 'high' ? 'warning' : 'success';
        
        this.showNotificationBanner(message, alertType, 10000);
        
        setTimeout(() => {
            if (typeof loadQueueData === 'function') {
                loadQueueData();
            }
        }, 1000);
    }
    
    handleRadiologyResultsReady(data) {
        const message = `Radiology results ready: ${data.imaging_type} for Patient ${data.patient_id}`;
        this.showNotificationBanner(message, 'success', 8000);
        
        setTimeout(() => {
            if (typeof loadQueueData === 'function') {
                loadQueueData();
            }
        }, 1000);
    }
    
    handleUrgentPatient(data) {
        const message = `🚨 URGENT PATIENT INCOMING - Priority: ${data.priority?.toUpperCase()}`;
        this.showNotificationBanner(message, 'danger', 15000);
        
        // Flash the page briefly for urgent notifications
        this.flashPage();
        
        setTimeout(() => {
            if (typeof loadQueueData === 'function') {
                loadQueueData();
            }
        }, 500);
    }
    
    handlePatientCalled(data) {
        const message = `Patient ${data.queue_number} called for service`;
        this.showNotificationBanner(message, 'info', 5000);
    }
    
    showVisualNotification(notification) {
        // Show browser notification if permission granted
        if ('Notification' in window && Notification.permission === 'granted') {
            const title = `${this.department.toUpperCase()} - New Notification`;
            const body = this.getNotificationMessage(notification);
            
            const browserNotification = new Notification(title, {
                body: body,
                icon: `${this.baseUrl}assets/images/hospital-icon.png`,
                tag: `queue-${this.department}`,
                requireInteraction: notification.action.includes('urgent')
            });
            
            // Auto-close after 5 seconds
            setTimeout(() => {
                browserNotification.close();
            }, 5000);
        }
    }
    
    showNotificationBanner(message, type = 'info', duration = 5000) {
        // Remove existing banners
        $('.queue-notification-banner').remove();
        
        const alertClass = this.getAlertClass(type);
        const icon = this.getAlertIcon(type);
        
        const banner = $(`
            <div class="alert ${alertClass} alert-dismissible queue-notification-banner" 
                 style="position: fixed; top: 70px; right: 20px; z-index: 9999; min-width: 350px; 
                        animation: slideInRight 0.5s ease-out; box-shadow: 0 4px 8px rgba(0,0,0,0.2);">
                <button type="button" class="close" data-dismiss="alert">&times;</button>
                <h5><i class="fa ${icon}"></i> ${this.department.toUpperCase()} Alert</h5>
                <p>${message}</p>
                <small class="text-muted">Received: ${new Date().toLocaleTimeString()}</small>
            </div>
        `);
        
        $('body').append(banner);
        
        // Auto-remove after duration
        setTimeout(() => {
            banner.fadeOut(500, function() {
                $(this).remove();
            });
        }, duration);
    }
    
    getAlertClass(type) {
        switch (type) {
            case 'success': return 'alert-success';
            case 'warning': return 'alert-warning';
            case 'danger': return 'alert-danger';
            case 'info': 
            default: return 'alert-info';
        }
    }
    
    getAlertIcon(type) {
        switch (type) {
            case 'success': return 'fa-check-circle';
            case 'warning': return 'fa-exclamation-triangle';
            case 'danger': return 'fa-exclamation-circle';
            case 'info':
            default: return 'fa-info-circle';
        }
    }
    
    getNotificationMessage(notification) {
        const { action, data } = notification;
        
        switch (action) {
            case 'new_patient_from_triage':
                return `New patient from Triage (Priority: ${data.priority || 'normal'})`;
            case 'new_patient_from_consultation':
                return `New patient from Consultation`;
            case 'lab_results_ready':
                return `Lab results ready for patient`;
            case 'radiology_results_ready':
                return `Radiology results ready`;
            case 'urgent_patient_incoming':
                return `URGENT PATIENT INCOMING`;
            default:
                return `New notification: ${action}`;
        }
    }
    
    playNotificationSound(action) {
        try {
            const audioContext = new (window.AudioContext || window.webkitAudioContext)();
            const oscillator = audioContext.createOscillator();
            const gainNode = audioContext.createGain();
            
            oscillator.connect(gainNode);
            gainNode.connect(audioContext.destination);
            
            // Different sound patterns for different actions
            this.configureSoundForAction(oscillator, gainNode, audioContext, action);
            
        } catch (error) {
            console.log('Audio not supported:', error);
        }
    }
    
    configureSoundForAction(oscillator, gainNode, audioContext, action) {
        let frequencies = [];
        let duration = 0.8;
        
        switch (action) {
            case 'urgent_patient_incoming':
                // Urgent: Fast alternating high-low tones
                frequencies = [1000, 800, 1000, 800, 1000];
                duration = 1.5;
                break;
            case 'new_patient_from_triage':
            case 'new_patient_from_consultation':
                // New patient: Rising tone
                frequencies = [600, 800, 1000];
                break;
            case 'lab_results_ready':
            case 'radiology_results_ready':
                // Results ready: Two-tone chime
                frequencies = [800, 600];
                break;
            default:
                // Default: Simple tone
                frequencies = [700];
        }
        
        this.playToneSequence(oscillator, gainNode, audioContext, frequencies, duration);
    }
    
    playToneSequence(oscillator, gainNode, audioContext, frequencies, totalDuration) {
        const segmentDuration = totalDuration / frequencies.length;
        
        frequencies.forEach((freq, index) => {
            const startTime = audioContext.currentTime + (index * segmentDuration);
            oscillator.frequency.setValueAtTime(freq, startTime);
        });
        
        gainNode.gain.setValueAtTime(0.2, audioContext.currentTime);
        gainNode.gain.exponentialRampToValueAtTime(0.01, audioContext.currentTime + totalDuration);
        
        oscillator.start(audioContext.currentTime);
        oscillator.stop(audioContext.currentTime + totalDuration);
    }
    
    flashPage() {
        // Brief flash effect for urgent notifications
        const overlay = $('<div>')
            .css({
                position: 'fixed',
                top: 0,
                left: 0,
                width: '100%',
                height: '100%',
                backgroundColor: 'rgba(255, 0, 0, 0.1)',
                zIndex: 99999,
                pointerEvents: 'none'
            })
            .appendTo('body');
        
        setTimeout(() => {
            overlay.fadeOut(200, function() {
                $(this).remove();
            });
        }, 300);
    }
    
    triggerNotificationEvent(notification) {
        // Trigger custom event for department-specific handling
        const event = new CustomEvent('queueNotification', {
            detail: notification
        });
        document.dispatchEvent(event);
    }
    
    initializeAudio() {
        // Request audio permission and initialize context
        try {
            const audioContext = new (window.AudioContext || window.webkitAudioContext)();
            
            // Resume audio context (required for some browsers)
            if (audioContext.state === 'suspended') {
                audioContext.resume();
            }
            
            this.audioContext = audioContext;
        } catch (error) {
            console.log('Audio initialization failed:', error);
            this.soundEnabled = false;
        }
    }
    
    setupNotificationUI() {
        // Request browser notification permission
        if ('Notification' in window && Notification.permission === 'default') {
            Notification.requestPermission().then(permission => {
                console.log(`Browser notifications: ${permission}`);
            });
        }
        
        // Add notification controls to the page
        this.addNotificationControls();
    }
    
    addNotificationControls() {
        // Add notification history button if not exists
        if (!$('#notification-history-btn').length) {
            const historyBtn = $(`
                <button id="notification-history-btn" class="btn btn-sm btn-info" 
                        onclick="queueNotifications.showNotificationHistory()" 
                        style="position: fixed; bottom: 20px; right: 20px; z-index: 1000;">
                    <i class="fa fa-bell"></i> 
                    <span class="badge" id="notification-count">0</span>
                </button>
            `);
            $('body').append(historyBtn);
        }
    }
    
    showNotificationHistory() {
        let historyHtml = '<div class="notification-history">';
        
        if (this.notificationHistory.length === 0) {
            historyHtml += '<p class="text-muted">No recent notifications</p>';
        } else {
            historyHtml += '<div class="list-group">';
            
            this.notificationHistory.slice(0, 10).forEach(notification => {
                const timeStr = notification.received_at.toLocaleTimeString();
                const message = this.getNotificationMessage(notification);
                
                historyHtml += `
                    <div class="list-group-item">
                        <h6 class="list-group-item-heading">
                            ${notification.action.replace(/_/g, ' ').toUpperCase()}
                            <small class="pull-right text-muted">${timeStr}</small>
                        </h6>
                        <p class="list-group-item-text">${message}</p>
                        <small class="text-muted">From: ${notification.from_department || 'System'}</small>
                    </div>
                `;
            });
            
            historyHtml += '</div>';
        }
        
        historyHtml += '</div>';
        
        // Show in modal or popup
        this.showModal('Notification History', historyHtml);
        
        // Reset notification count
        $('#notification-count').text('0');
    }
    
    showModal(title, content) {
        const modalHtml = `
            <div class="modal fade" id="notificationModal" tabindex="-1" role="dialog">
                <div class="modal-dialog" role="document">
                    <div class="modal-content">
                        <div class="modal-header">
                            <button type="button" class="close" data-dismiss="modal">&times;</button>
                            <h4 class="modal-title">${title}</h4>
                        </div>
                        <div class="modal-body">
                            ${content}
                        </div>
                        <div class="modal-footer">
                            <button type="button" class="btn btn-default" data-dismiss="modal">Close</button>
                        </div>
                    </div>
                </div>
            </div>
        `;
        
        // Remove existing modal
        $('#notificationModal').remove();
        
        // Add new modal
        $('body').append(modalHtml);
        $('#notificationModal').modal('show');
    }
    
    updateNotificationCount() {
        $('#notification-count').text(this.notificationHistory.length);
    }
    
    enableSound() {
        this.soundEnabled = true;
        console.log('🔊 Queue notifications sound enabled');
    }
    
    disableSound() {
        this.soundEnabled = false;
        console.log('🔇 Queue notifications sound disabled');
    }
    
    destroy() {
        if (this.notificationInterval) {
            clearInterval(this.notificationInterval);
        }
        
        $('.queue-notification-banner').remove();
        $('#notification-history-btn').remove();
        
        console.log(`🔔 Queue notifications destroyed for ${this.department}`);
    }
}

// Global notification service instance
let queueNotifications = null;

// Initialize notifications when DOM is ready
$(document).ready(function() {
    // Get department and base URL from global variables or DOM
    const department = window.DEPARTMENT || $('body').data('department');
    const baseUrl = window.BASE_URL || $('body').data('base-url') || '';
    
    if (department) {
        queueNotifications = new QueueNotificationService(department, baseUrl);
        
        // Make available globally
        window.queueNotifications = queueNotifications;
    }
});

// Clean up on page unload
$(window).on('beforeunload', function() {
    if (queueNotifications) {
        queueNotifications.destroy();
    }
});

console.log('🔔 Queue Notification Service loaded');

// CSS styles for notifications
const notificationStyles = `
<style>
.queue-notification-banner {
    border-left: 4px solid;
    border-radius: 4px !important;
}

.queue-notification-banner.alert-info {
    border-left-color: #5bc0de;
}

.queue-notification-banner.alert-success {
    border-left-color: #5cb85c;
}

.queue-notification-banner.alert-warning {
    border-left-color: #f0ad4e;
}

.queue-notification-banner.alert-danger {
    border-left-color: #d9534f;
}

@keyframes slideInRight {
    from {
        transform: translateX(100%);
        opacity: 0;
    }
    to {
        transform: translateX(0);
        opacity: 1;
    }
}

@keyframes pulse {
    0% { transform: scale(1); }
    50% { transform: scale(1.05); }
    100% { transform: scale(1); }
}

.notification-pulse {
    animation: pulse 1s infinite;
}

#notification-history-btn {
    border-radius: 50px !important;
    padding: 10px 15px;
    box-shadow: 0 2px 5px rgba(0,0,0,0.2);
}

#notification-history-btn .badge {
    position: absolute;
    top: -5px;
    right: -5px;
    background-color: #d9534f;
}

.notification-history .list-group-item {
    border-left: 3px solid #337ab7;
    margin-bottom: 5px;
}

.notification-history .list-group-item-heading {
    color: #337ab7;
    font-weight: bold;
}
</style>
`;

// Inject notification styles
$('head').append(notificationStyles);