<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Triage Records - Hospital Management System</title>
    
    <!-- Bootstrap CSS -->
    <link href="https://cdnjs.cloudflare.com/ajax/libs/bootstrap/3.4.1/css/bootstrap.min.css" rel="stylesheet">
    <!-- Font Awesome -->
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/4.7.0/css/font-awesome.min.css" rel="stylesheet">
    <!-- DataTables CSS -->
    <link href="https://cdnjs.cloudflare.com/ajax/libs/datatables/1.10.21/css/dataTables.bootstrap.min.css" rel="stylesheet">
    
    <style>
        .content-wrapper {
            background-color: #ecf0f5;
            min-height: 100vh;
            padding: 20px;
        }
        
        .content-header h1 {
            margin: 0;
            font-size: 24px;
            color: #333;
        }
        
        .box {
            background: #fff;
            border-top: 3px solid #d2d6de;
            margin-bottom: 20px;
            box-shadow: 0 1px 1px rgba(0,0,0,0.1);
        }
        
        .box-header {
            border-bottom: 1px solid #f4f4f4;
            padding: 10px 15px;
            position: relative;
        }
        
        .box-title {
            font-size: 18px;
            margin: 0;
            line-height: 1.8;
        }
        
        .box-body {
            padding: 15px;
        }
        
        .priority-badge {
            padding: 4px 8px;
            border-radius: 3px;
            font-size: 11px;
            font-weight: bold;
            text-transform: uppercase;
        }
        
        .priority-emergency { background-color: #dd4b39; color: white; }
        .priority-urgent { background-color: #f39c12; color: white; }
        .priority-normal { background-color: #00a65a; color: white; }
        .priority-low { background-color: #3c8dbc; color: white; }
        
        .status-badge {
            padding: 4px 8px;
            border-radius: 3px;
            font-size: 11px;
            font-weight: bold;
        }
        
        .status-pending { background-color: #f39c12; color: white; }
        .status-in-progress { background-color: #00c0ef; color: white; }
        .status-completed { background-color: #00a65a; color: white; }
        .status-cancelled { background-color: #dd4b39; color: white; }
        
        .vital-summary {
            font-size: 11px;
            line-height: 1.3;
        }
        
        .table-responsive {
            border: 1px solid #f4f4f4;
        }
        
        .table > thead > tr > th {
            background-color: #f4f4f4;
            font-weight: 600;
            font-size: 13px;
        }
        
        .table > tbody > tr > td {
            font-size: 12px;
            vertical-align: middle;
        }
        
        .btn-xs {
            padding: 1px 5px;
            font-size: 10px;
        }
        
        .filter-section {
            background-color: #f9f9f9;
            padding: 15px;
            margin-bottom: 20px;
            border-radius: 3px;
        }
        
        .stats-row {
            margin-bottom: 20px;
        }
        
        .info-box {
            display: block;
            min-height: 70px;
            background: #fff;
            width: 100%;
            box-shadow: 0 1px 1px rgba(0,0,0,0.1);
            border-radius: 2px;
            margin-bottom: 15px;
        }
        
        .info-box-icon {
            border-top-left-radius: 2px;
            border-top-right-radius: 0;
            border-bottom-right-radius: 0;
            border-bottom-left-radius: 2px;
            display: block;
            float: left;
            height: 70px;
            width: 70px;
            text-align: center;
            font-size: 40px;
            line-height: 70px;
            background: rgba(0,0,0,0.2);
        }
        
        .info-box-content {
            padding: 5px 10px;
            margin-left: 70px;
        }
        
        .info-box-number {
            display: block;
            font-weight: bold;
            font-size: 18px;
        }
        
        .info-box-text {
            display: block;
            font-size: 13px;
            white-space: nowrap;
            overflow: hidden;
            text-overflow: ellipsis;
        }
        
        .bg-red { background-color: #dd4b39 !important; }
        .bg-yellow { background-color: #f39c12 !important; }
        .bg-green { background-color: #00a65a !important; }
        .bg-blue { background-color: #3c8dbc !important; }
        
        .dataTables_wrapper .dataTables_paginate .paginate_button {
            padding: 6px 12px;
            margin-left: -1px;
        }
    </style>
</head>
<body>
    <div class="content-wrapper">
        <!-- Content Header -->
        <section class="content-header">
            <h1>
                <i class="fa fa-heartbeat"></i> Triage Records
                <small>Patient Assessment & Priority Management</small>
            </h1>
        </section>

        <!-- Main content -->
        <section class="content">
            <!-- Statistics Row -->
            <div class="row stats-row">
                <div class="col-lg-3 col-xs-6">
                    <div class="info-box">
                        <span class="info-box-icon bg-red"><i class="fa fa-exclamation-triangle"></i></span>
                        <div class="info-box-content">
                            <span class="info-box-text">Emergency Cases</span>
                            <span class="info-box-number" id="emergency-count">0</span>
                        </div>
                    </div>
                </div>
                
                <div class="col-lg-3 col-xs-6">
                    <div class="info-box">
                        <span class="info-box-icon bg-yellow"><i class="fa fa-clock-o"></i></span>
                        <div class="info-box-content">
                            <span class="info-box-text">Urgent Cases</span>
                            <span class="info-box-number" id="urgent-count">0</span>
                        </div>
                    </div>
                </div>
                
                <div class="col-lg-3 col-xs-6">
                    <div class="info-box">
                        <span class="info-box-icon bg-green"><i class="fa fa-check-circle"></i></span>
                        <div class="info-box-content">
                            <span class="info-box-text">Completed Today</span>
                            <span class="info-box-number" id="completed-count">0</span>
                        </div>
                    </div>
                </div>
                
                <div class="col-lg-3 col-xs-6">
                    <div class="info-box">
                        <span class="info-box-icon bg-blue"><i class="fa fa-users"></i></span>
                        <div class="info-box-content">
                            <span class="info-box-text">Total Today</span>
                            <span class="info-box-number" id="total-count">0</span>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Filters -->
            <div class="box">
                <div class="box-header">
                    <h3 class="box-title"><i class="fa fa-filter"></i> Filters & Search</h3>
                </div>
                <div class="box-body">
                    <div class="filter-section">
                        <div class="row">
                            <div class="col-md-3">
                                <label>Date Range:</label>
                                <div class="input-group">
                                    <input type="date" class="form-control input-sm" id="date-from" value="">
                                    <span class="input-group-addon">to</span>
                                    <input type="date" class="form-control input-sm" id="date-to" value="">
                                </div>
                            </div>
                            <div class="col-md-2">
                                <label>Priority:</label>
                                <select class="form-control input-sm" id="priority-filter">
                                    <option value="">All Priorities</option>
                                    <option value="1">Emergency</option>
                                    <option value="2">Urgent</option>
                                    <option value="3">Normal</option>
                                    <option value="4">Low</option>
                                </select>
                            </div>
                            <div class="col-md-2">
                                <label>Status:</label>
                                <select class="form-control input-sm" id="status-filter">
                                    <option value="">All Status</option>
                                    <option value="pending">Pending</option>
                                    <option value="in_progress">In Progress</option>
                                    <option value="completed">Completed</option>
                                    <option value="cancelled">Cancelled</option>
                                </select>
                            </div>
                            <div class="col-md-2">
                                <label>Department:</label>
                                <select class="form-control input-sm" id="department-filter">
                                    <option value="">All Departments</option>
                                    <option value="emergency">Emergency</option>
                                    <option value="consultation">Consultation</option>
                                    <option value="pharmacy">Pharmacy</option>
                                    <option value="laboratory">Laboratory</option>
                                    <option value="radiology">Radiology</option>
                                    <option value="ipd">IPD</option>
                                </select>
                            </div>
                            <div class="col-md-3">
                                <label>&nbsp;</label>
                                <div>
                                    <button type="button" class="btn btn-primary btn-sm" onclick="applyFilters()">
                                        <i class="fa fa-search"></i> Apply Filters
                                    </button>
                                    <button type="button" class="btn btn-default btn-sm" onclick="clearFilters()">
                                        <i class="fa fa-refresh"></i> Clear
                                    </button>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Triage Records Table -->
            <div class="box">
                <div class="box-header">
                    <h3 class="box-title"><i class="fa fa-list"></i> Triage Records</h3>
                    <div class="box-tools pull-right">
                        <button type="button" class="btn btn-success btn-sm" onclick="exportToExcel()">
                            <i class="fa fa-download"></i> Export Excel
                        </button>
                        <button type="button" class="btn btn-info btn-sm" onclick="refreshTable()">
                            <i class="fa fa-refresh"></i> Refresh
                        </button>
                    </div>
                </div>
                <div class="box-body">
                    <div class="table-responsive">
                        <table class="table table-striped table-bordered table-hover" id="triageTable">
                            <thead>
                                <tr>
                                    <th width="5%">ID</th>
                                    <th width="12%">Date/Time</th>
                                    <th width="15%">Patient</th>
                                    <th width="8%">Priority</th>
                                    <th width="20%">Vital Signs</th>
                                    <th width="10%">Department</th>
                                    <th width="8%">Status</th>
                                    <th width="12%">Staff</th>
                                    <th width="10%">Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <!-- Data will be populated by DataTables from database -->
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </section>
    </div>

    <!-- View Record Modal -->
    <div class="modal fade" id="viewRecordModal" tabindex="-1" role="dialog">
        <div class="modal-dialog modal-lg" role="document">
            <div class="modal-content">
                <div class="modal-header">
                    <button type="button" class="close" data-dismiss="modal">&times;</button>
                    <h4 class="modal-title"><i class="fa fa-eye"></i> Triage Record Details</h4>
                </div>
                <div class="modal-body" id="recordDetails">
                    <!-- Record details will be loaded here -->
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-default" data-dismiss="modal">Close</button>
                    <button type="button" class="btn btn-primary" onclick="printCurrentRecord()">
                        <i class="fa fa-print"></i> Print
                    </button>
                </div>
            </div>
        </div>
    </div>

    <!-- Scripts -->
    <script src="https://cdnjs.cloudflare.com/ajax/libs/jquery/3.6.0/jquery.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/bootstrap/3.4.1/js/bootstrap.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/datatables/1.10.21/js/jquery.dataTables.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/datatables/1.10.21/js/dataTables.bootstrap.min.js"></script>

    <script>
        let triageTable;
        
        $(document).ready(function() {
            // Set today's date as default
            const today = new Date().toISOString().split('T')[0];
            $('#date-from').val(today);
            $('#date-to').val(today);
            
            // Initialize DataTable
            initializeTable();
            
            // Load statistics
            loadStatistics();
            
            // Auto-refresh every 30 seconds
            setInterval(function() {
                refreshTable();
                loadStatistics();
            }, 30000);
        });
        
        function initializeTable() {
            triageTable = $('#triageTable').DataTable({
                "processing": true,
                "serverSide": true,
                "pageLength": 25,
                "ordering": true,
                "searching": true,
                "order": [[1, "desc"]], // Order by date/time descending
                "ajax": {
                    "url": "<?php echo base_url('admin/triage/getTriageList'); ?>",
                    "type": "POST",
                    "data": function(d) {
                        // Add filter parameters
                        d.date_from = $('#date-from').val();
                        d.date_to = $('#date-to').val();
                        d.priority_filter = $('#priority-filter').val();
                        d.status_filter = $('#status-filter').val();
                        d.department_filter = $('#department-filter').val();
                        return d;
                    },
                    "error": function(xhr, error, code) {
                        console.error('DataTable Ajax Error:', xhr.responseText);
                        showMessage('Failed to load triage records. Please check console for details.', 'error');
                    }
                },
                "columns": [
                    { "data": "id", "title": "ID" },
                    { 
                        "data": "created_at", 
                        "title": "Date/Time",
                        "render": function(data, type, row) {
                            if (type === 'display') {
                                const date = new Date(data);
                                return `<strong>${date.toLocaleDateString()}</strong><br><small>${date.toLocaleTimeString()}</small>`;
                            }
                            return data;
                        }
                    },
                    { 
                        "data": "patient_info", 
                        "title": "Patient",
                        "render": function(data, type, row) {
                            if (type === 'display') {
                                let html = `<strong>${data.name || 'Unknown'}</strong><br>`;
                                html += `<small>ID: ${row.patient_id} | Age: ${data.age || 'N/A'} | ${data.gender || 'N/A'}</small><br>`;
                                if (data.phone) {
                                    html += `<small><i class="fa fa-phone"></i> ${data.phone}</small>`;
                                }
                                return html;
                            }
                            return data.name || 'Unknown';
                        }
                    },
                    { 
                        "data": "priority", 
                        "title": "Priority",
                        "render": function(data, type, row) {
                            if (type === 'display') {
                                const priorityClasses = {
                                    'urgent': 'priority-emergency',
                                    'high': 'priority-urgent',
                                    'normal': 'priority-normal',
                                    'low': 'priority-low'
                                };
                                const priorityIcons = {
                                    'urgent': 'fa-exclamation-triangle',
                                    'high': 'fa-clock-o',
                                    'normal': 'fa-check',
                                    'low': 'fa-minus'
                                };
                                return `<span class="priority-badge ${priorityClasses[data] || 'priority-normal'}">
                                    <i class="fa ${priorityIcons[data] || 'fa-check'}"></i> ${data.toUpperCase()}
                                </span>`;
                            }
                            return data;
                        }
                    },
                    { 
                        "data": "vitals", 
                        "title": "Vital Signs",
                        "render": function(data, type, row) {
                            if (type === 'display') {
                                try {
                                    const vitals = typeof data === 'string' ? JSON.parse(data) : data;
                                    let html = '<div class="vital-summary">';
                                    
                                    if (vitals.blood_pressure) {
                                        const bp = vitals.blood_pressure;
                                        const bpStatus = (bp.systolic > 140 || bp.diastolic > 90) ? 'text-danger' : 'text-success';
                                        html += `<strong>BP:</strong> ${bp.systolic}/${bp.diastolic} <span class="${bpStatus}">(${bpStatus === 'text-danger' ? 'High' : 'Normal'})</span><br>`;
                                    }
                                    
                                    if (vitals.temperature) {
                                        const tempStatus = vitals.temperature > 38 ? 'text-danger' : 'text-success';
                                        html += `<strong>Temp:</strong> ${vitals.temperature}°C <span class="${tempStatus}">(${tempStatus === 'text-danger' ? 'High' : 'Normal'})</span><br>`;
                                    }
                                    
                                    if (vitals.heart_rate || vitals.pulse_rate) {
                                        const hr = vitals.heart_rate || vitals.pulse_rate;
                                        const hrStatus = (hr > 100 || hr < 60) ? 'text-warning' : 'text-success';
                                        html += `<strong>HR:</strong> ${hr} bpm <span class="${hrStatus}">(${hrStatus === 'text-warning' ? 'Abnormal' : 'Normal'})</span><br>`;
                                    }
                                    
                                    if (vitals.oxygen_saturation) {
                                        const o2Status = vitals.oxygen_saturation < 95 ? 'text-danger' : 'text-success';
                                        html += `<strong>SpO2:</strong> ${vitals.oxygen_saturation}% <span class="${o2Status}">(${o2Status === 'text-danger' ? 'Low' : 'Normal'})</span><br>`;
                                    }
                                    
                                    if (vitals.pain_scale !== undefined) {
                                        const painLevel = vitals.pain_scale > 7 ? 'Severe' : vitals.pain_scale > 4 ? 'Moderate' : vitals.pain_scale > 0 ? 'Mild' : 'None';
                                        const painClass = vitals.pain_scale > 7 ? 'text-danger' : vitals.pain_scale > 4 ? 'text-warning' : 'text-success';
                                        html += `<strong>Pain:</strong> ${vitals.pain_scale}/10 <span class="${painClass}">(${painLevel})</span>`;
                                    }
                                    
                                    html += '</div>';
                                    return html;
                                } catch (e) {
                                    return '<small class="text-muted">Vitals not available</small>';
                                }
                            }
                            return data;
                        }
                    },
                    { 
                        "data": "department", 
                        "title": "Department",
                        "render": function(data, type, row) {
                            if (type === 'display') {
                                const deptClasses = {
                                    'emergency': 'label-danger',
                                    'consultation': 'label-primary',
                                    'pharmacy': 'label-success',
                                    'laboratory': 'label-warning',
                                    'radiology': 'label-info',
                                    'ipd': 'label-default'
                                };
                                return `<span class="label ${deptClasses[data] || 'label-default'}">${data || 'Not Assigned'}</span>`;
                            }
                            return data || 'Not Assigned';
                        }
                    },
                    { 
                        "data": "status", 
                        "title": "Status",
                        "render": function(data, type, row) {
                            if (type === 'display') {
                                const statusClasses = {
                                    'pending': 'status-pending',
                                    'in_progress': 'status-in-progress',
                                    'completed': 'status-completed',
                                    'cancelled': 'status-cancelled'
                                };
                                return `<span class="status-badge ${statusClasses[data] || 'status-pending'}">${data.replace('_', ' ').toUpperCase()}</span>`;
                            }
                            return data;
                        }
                    },
                    { 
                        "data": "staff_info", 
                        "title": "Staff",
                        "render": function(data, type, row) {
                            if (type === 'display') {
                                return `<strong>${data.name || 'Unknown'}</strong><br><small>${data.role || 'Staff'}</small>`;
                            }
                            return data.name || 'Unknown';
                        }
                    },
                    { 
                        "data": "id", 
                        "title": "Actions",
                        "orderable": false,
                        "searchable": false,
                        "render": function(data, type, row) {
                            if (type === 'display') {
                                return `<div class="btn-group">
                                    <button type="button" class="btn btn-primary btn-xs" onclick="viewRecord(${data})">
                                        <i class="fa fa-eye"></i>
                                    </button>
                                    <button type="button" class="btn btn-success btn-xs" onclick="printRecord(${data})">
                                        <i class="fa fa-print"></i>
                                    </button>
                                    <button type="button" class="btn btn-warning btn-xs" onclick="editRecord(${data})">
                                        <i class="fa fa-edit"></i>
                                    </button>
                                </div>`;
                            }
                            return data;
                        }
                    }
                ],
                "columnDefs": [
                    {
                        "targets": [8], // Actions column
                        "orderable": false,
                        "searchable": false
                    }
                ],
                "language": {
                    "processing": "Loading triage records...",
                    "emptyTable": "No triage records found",
                    "info": "Showing _START_ to _END_ of _TOTAL_ records",
                    "infoEmpty": "Showing 0 to 0 of 0 records",
                    "infoFiltered": "(filtered from _MAX_ total records)"
                }
            });
        }
        
        function loadStatistics() {
            // Fetch real statistics from backend
            $.ajax({
                url: "<?php echo base_url('admin/triage/getTriageStatistics'); ?>",
                type: "GET",
                data: {
                    date_from: $('#date-from').val(),
                    date_to: $('#date-to').val()
                },
                success: function(response) {
                    if (response.status === 'success') {
                        $('#emergency-count').text(response.data.emergency || 0);
                        $('#urgent-count').text(response.data.urgent || 0);
                        $('#completed-count').text(response.data.completed || 0);
                        $('#total-count').text(response.data.total || 0);
                    }
                },
                error: function() {
                    console.error('Failed to load statistics');
                }
            });
        }
        
        function applyFilters() {
            // Apply filters to DataTable
            const dateFrom = $('#date-from').val();
            const dateTo = $('#date-to').val();
            const priority = $('#priority-filter').val();
            const status = $('#status-filter').val();
            const department = $('#department-filter').val();
            
            // In real implementation, this would send AJAX request with filters
            triageTable.draw();
            loadStatistics();
        }
        
        function clearFilters() {
            const today = new Date().toISOString().split('T')[0];
            $('#date-from').val(today);
            $('#date-to').val(today);
            $('#priority-filter').val('');
            $('#status-filter').val('');
            $('#department-filter').val('');
            
            triageTable.search('').draw();
            loadStatistics();
        }
        
        function refreshTable() {
            if (triageTable) {
                triageTable.ajax.reload(null, false);
            }
        }
        
        function exportToExcel() {
            // Export current filtered data to Excel
            const filters = {
                date_from: $('#date-from').val(),
                date_to: $('#date-to').val(),
                priority_filter: $('#priority-filter').val(),
                status_filter: $('#status-filter').val(),
                department_filter: $('#department-filter').val()
            };
            
            const exportUrl = "<?php echo base_url('admin/triage/exportTriageRecords'); ?>?" + $.param(filters);
            window.open(exportUrl, '_blank');
        }
        
        function printRecord(id) {
            // Open print view for individual record
            const printUrl = "<?php echo base_url('admin/triage/printTriageRecord/'); ?>" + id;
            window.open(printUrl, '_blank');
        }
        
        function editRecord(id) {
            // Navigate to edit page
            window.location.href = "<?php echo base_url('admin/triage/editTriage/'); ?>" + id;
        }
        
        // Show message function
        function showMessage(message, type) {
            const alertClass = type === 'success' ? 'alert-success' : 
                             type === 'warning' ? 'alert-warning' : 
                             type === 'error' ? 'alert-danger' : 'alert-info';
            
            const alertHtml = `<div class="alert ${alertClass} alert-dismissible">
                <button type="button" class="close" data-dismiss="alert">&times;</button>
                ${message}
            </div>`;
            
            // Remove existing alerts
            $('.alert').remove();
            
            // Add new alert at top of content
            $('.content').prepend(alertHtml);
            
            // Auto-hide after 5 seconds
            setTimeout(function() {
                $('.alert').fadeOut();
            }, 5000);
        }
        
        function viewRecord(id) {
            // Load actual record details from database
            $.ajax({
                url: "<?php echo base_url('admin/triage/getTriageDetails'); ?>",
                type: "GET",
                data: { id: id },
                success: function(response) {
                    if (response.status === 'success') {
                        const record = response.data;
                        const vitals = typeof record.vitals === 'string' ? JSON.parse(record.vitals) : record.vitals;
                        
                        const recordContent = `
                            <div class="row">
                                <div class="col-md-6">
                                    <h4><i class="fa fa-user"></i> Patient Information</h4>
                                    <table class="table table-bordered table-condensed">
                                        <tr>
                                            <td width="40%"><strong>Patient Name:</strong></td>
                                            <td>${record.patient_name || 'N/A'}</td>
                                        </tr>
                                        <tr>
                                            <td><strong>Patient ID:</strong></td>
                                            <td>${record.patient_id}</td>
                                        </tr>
                                        <tr>
                                            <td><strong>Age/Gender:</strong></td>
                                            <td>${record.age || 'N/A'} years / ${record.gender || 'N/A'}</td>
                                        </tr>
                                        <tr>
                                            <td><strong>Contact:</strong></td>
                                            <td>${record.mobileno || 'N/A'}</td>
                                        </tr>
                                        <tr>
                                            <td><strong>Triage Date:</strong></td>
                                            <td>${new Date(record.created_at).toLocaleString()}</td>
                                        </tr>
                                        <tr>
                                            <td><strong>Triaged By:</strong></td>
                                            <td>${record.staff_name || 'N/A'}</td>
                                        </tr>
                                    </table>
                                </div>
                                
                                <div class="col-md-6">
                                    <h4><i class="fa fa-heartbeat"></i> Priority & Status</h4>
                                    <table class="table table-bordered table-condensed">
                                        <tr>
                                            <td width="40%"><strong>Priority:</strong></td>
                                            <td><span class="priority-badge priority-${record.priority}">${record.priority.toUpperCase()}</span></td>
                                        </tr>
                                        <tr>
                                            <td><strong>Status:</strong></td>
                                            <td><span class="status-badge status-${record.status}">${record.status.replace('_', ' ').toUpperCase()}</span></td>
                                        </tr>
                                        <tr>
                                            <td><strong>Department:</strong></td>
                                            <td><span class="label label-primary">${record.department || 'Not Assigned'}</span></td>
                                        </tr>
                                        <tr>
                                            <td><strong>Verification:</strong></td>
                                            <td><span class="label ${record.verification_status === 'verified' ? 'label-success' : 'label-warning'}">${record.verification_status || 'N/A'}</span></td>
                                        </tr>
                                    </table>
                                </div>
                            </div>
                            
                            <div class="row" style="margin-top: 20px;">
                                <div class="col-md-12">
                                    <h4><i class="fa fa-list"></i> Complete Vital Signs</h4>
                                    <div class="table-responsive">
                                        <table class="table table-striped table-bordered">
                                            <tbody>
                                                <tr>
                                                    <td width="25%"><strong>Temperature</strong></td>
                                                    <td width="25%">
                                                        ${vitals.temperature ? 
                                                            `<span class="label ${vitals.temperature > 38 ? 'label-danger' : 'label-success'}">${vitals.temperature}°C</span>` : 
                                                            '<span class="text-muted">Not recorded</span>'
                                                        }
                                                    </td>
                                                    <td width="25%"><strong>Blood Pressure</strong></td>
                                                    <td width="25%">
                                                        ${vitals.blood_pressure ? 
                                                            `<span class="label ${(vitals.blood_pressure.systolic > 140 || vitals.blood_pressure.diastolic > 90) ? 'label-danger' : 'label-success'}">${vitals.blood_pressure.systolic}/${vitals.blood_pressure.diastolic} mmHg</span>` : 
                                                            '<span class="text-muted">Not recorded</span>'
                                                        }
                                                    </td>
                                                </tr>
                                                <tr>
                                                    <td><strong>Heart Rate</strong></td>
                                                    <td>
                                                        ${(vitals.heart_rate || vitals.pulse_rate) ? 
                                                            `<span class="label ${((vitals.heart_rate || vitals.pulse_rate) > 100 || (vitals.heart_rate || vitals.pulse_rate) < 60) ? 'label-warning' : 'label-success'}">${vitals.heart_rate || vitals.pulse_rate} bpm</span>` : 
                                                            '<span class="text-muted">Not recorded</span>'
                                                        }
                                                    </td>
                                                    <td><strong>Respiratory Rate</strong></td>
                                                    <td>
                                                        ${vitals.respiratory_rate ? 
                                                            `<span class="label ${(vitals.respiratory_rate > 20 || vitals.respiratory_rate < 12) ? 'label-warning' : 'label-success'}">${vitals.respiratory_rate} /min</span>` : 
                                                            '<span class="text-muted">Not recorded</span>'
                                                        }
                                                    </td>
                                                </tr>
                                                <tr>
                                                    <td><strong>Oxygen Saturation</strong></td>
                                                    <td>
                                                        ${vitals.oxygen_saturation ? 
                                                            `<span class="label ${vitals.oxygen_saturation < 95 ? 'label-danger' : 'label-success'}">${vitals.oxygen_saturation}%</span>` : 
                                                            '<span class="text-muted">Not recorded</span>'
                                                        }
                                                    </td>
                                                    <td><strong>Pain Scale</strong></td>
                                                    <td>
                                                        ${vitals.pain_scale !== undefined ? 
                                                            `<span class="label ${vitals.pain_scale > 7 ? 'label-danger' : (vitals.pain_scale > 4 ? 'label-warning' : 'label-success')}">${vitals.pain_scale}/10</span>` : 
                                                            '<span class="text-muted">Not recorded</span>'
                                                        }
                                                    </td>
                                                </tr>
                                                <tr>
                                                    <td><strong>Weight</strong></td>
                                                    <td>
                                                        ${vitals.weight ? 
                                                            `<span class="label label-info">${vitals.weight} kg</span>` : 
                                                            '<span class="text-muted">Not recorded</span>'
                                                        }
                                                    </td>
                                                    <td><strong>Height</strong></td>
                                                    <td>
                                                        ${vitals.height ? 
                                                            `<span class="label label-info">${vitals.height} cm</span>` : 
                                                            '<span class="text-muted">Not recorded</span>'
                                                        }
                                                    </td>
                                                </tr>
                                            </tbody>
                                        </table>
                                    </div>
                                </div>
                            </div>
                            
                            ${record.notes ? `
                            <div class="row" style="margin-top: 20px;">
                                <div class="col-md-12">
                                    <h4><i class="fa fa-sticky-note"></i> Clinical Notes</h4>
                                    <div class="well">
                                        ${record.notes.replace(/\n/g, '<br>')}
                                    </div>
                                </div>
                            </div>
                            ` : ''}
                        `;
                        
                        $('#recordDetails').html(recordContent);
                        $('#viewRecordModal').modal('show');
                    } else {
                        showMessage('Failed to load record details: ' + (response.message || 'Unknown error'), 'error');
                    }
                },
                error: function() {
                    showMessage('Error loading record details', 'error');
                }
            });
        }
        
        function printRecord(id) {
            // Print individual record
            window.print();
        }
        
        function editRecord(id) {
            // Navigate to edit page or open edit modal
            alert('Edit functionality would redirect to triage form with record ID: ' + id);
        }
        
        function printCurrentRecord() {
            // Print the currently viewed record
            const printContent = document.getElementById('recordDetails').innerHTML;
            const printWindow = window.open('', '_blank');
            printWindow.document.write(`
                <html>
                <head>
                    <title>Triage Record</title>
                    <link href="https://cdnjs.cloudflare.com/ajax/libs/bootstrap/3.4.1/css/bootstrap.min.css" rel="stylesheet">
                    <style>
                        body { font-family: Arial, sans-serif; margin: 20px; }
                        .priority-badge, .status-badge, .label { 
                            padding: 4px 8px; border-radius: 3px; font-size: 11px; font-weight: bold; 
                        }
                        .priority-emergency { background-color: #dd4b39; color: white; }
                        .status-in-progress { background-color: #00c0ef; color: white; }
                        .label-danger { background-color: #dd4b39; color: white; }
                        .label-warning { background-color: #f39c12; color: white; }
                        .label-success { background-color: #00a65a; color: white; }
                    </style>
                </head>
                <body>
                    <div class="container">
                        <div class="text-center" style="margin-bottom: 30px;">
                            <h2>Hospital Management System</h2>
                            <h3>Triage Record Report</h3>
                            <p>Generated on: ${new Date().toLocaleString()}</p>
                        </div>
                        ${printContent}
                    </div>
                </body>
                </html>
            `);
            printWindow.document.close();
            printWindow.print();
        }
        
        // Additional helper functions
        function updateRecordStatus(id, newStatus) {
            // Update record status via AJAX
            $.ajax({
                url: 'path/to/update_triage_status.php',
                method: 'POST',
                data: {
                    id: id,
                    status: newStatus
                },
                success: function(response) {
                    refreshTable();
                    loadStatistics();
                },
                error: function() {
                    alert('Error updating record status');
                }
            });
        }
        
        function bulkExport() {
            const selectedFilters = {
                dateFrom: $('#date-from').val(),
                dateTo: $('#date-to').val(),
                priority: $('#priority-filter').val(),
                status: $('#status-filter').val(),
                department: $('#department-filter').val()
            };
            
            // Create export URL with filters
            const exportUrl = 'path/to/export_triage.php?' + $.param(selectedFilters);
            window.open(exportUrl, '_blank');
        }
        
        // Real-time updates using WebSocket (optional)
        function initializeRealTimeUpdates() {
            // if (typeof WebSocket !== 'undefined') {
            //     const ws = new WebSocket('ws://localhost:8080/triage-updates');
            //     ws.onmessage = function(event) {
            //         const data = JSON.parse(event.data);
            //         if (data.type === 'triage_update') {
            //             refreshTable();
            //             loadStatistics();
            //         }
            //     };
            // }
        }
        
        // Search functionality
        $('#triageTable_filter input').on('keyup', function() {
            const searchTerm = this.value;
            triageTable.search(searchTerm).draw();
        });
        
        // Keyboard shortcuts
        $(document).keydown(function(e) {
            // Ctrl+R for refresh
            if (e.ctrlKey && e.keyCode === 82) {
                e.preventDefault();
                refreshTable();
            }
            // Ctrl+E for export
            if (e.ctrlKey && e.keyCode === 69) {
                e.preventDefault();
                exportToExcel();
            }
        });
    </script>
</body>
</html>