<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Roles & Permissions Manager</title>
    <link href="https://cdnjs.cloudflare.com/ajax/libs/bootstrap/5.3.0/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css" rel="stylesheet">
    <style>
        body {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
        }
        
        .main-container {
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(10px);
            border-radius: 20px;
            box-shadow: 0 20px 40px rgba(0, 0, 0, 0.1);
            margin: 20px auto;
            padding: 30px;
        }
        
        .nav-tabs {
            border-bottom: 3px solid #e9ecef;
            margin-bottom: 30px;
        }
        
        .nav-tabs .nav-link {
            border: none;
            color: #6c757d;
            font-weight: 600;
            padding: 15px 25px;
            border-radius: 10px 10px 0 0;
            transition: all 0.3s ease;
        }
        
        .nav-tabs .nav-link.active {
            background: linear-gradient(45deg, #667eea, #764ba2);
            color: white;
            border: none;
        }
        
        .nav-tabs .nav-link:hover:not(.active) {
            background: rgba(102, 126, 234, 0.1);
            color: #667eea;
        }
        
        .form-section {
            background: white;
            border-radius: 15px;
            padding: 25px;
            margin-bottom: 20px;
            box-shadow: 0 5px 15px rgba(0, 0, 0, 0.08);
            border-left: 4px solid #667eea;
        }
        
        .form-label {
            font-weight: 600;
            color: #495057;
            margin-bottom: 8px;
        }
        
        .form-control, .form-select {
            border: 2px solid #e9ecef;
            border-radius: 10px;
            padding: 12px 15px;
            transition: all 0.3s ease;
        }
        
        .form-control:focus, .form-select:focus {
            border-color: #667eea;
            box-shadow: 0 0 0 0.2rem rgba(102, 126, 234, 0.25);
        }
        
        .btn-primary {
            background: linear-gradient(45deg, #667eea, #764ba2);
            border: none;
            padding: 12px 30px;
            border-radius: 25px;
            font-weight: 600;
            transition: all 0.3s ease;
        }
        
        .btn-primary:hover {
            transform: translateY(-2px);
            box-shadow: 0 10px 20px rgba(102, 126, 234, 0.3);
        }
        
        .btn-success {
            background: linear-gradient(45deg, #28a745, #20c997);
            border: none;
            padding: 10px 20px;
            border-radius: 20px;
            font-weight: 600;
        }
        
        .btn-danger {
            background: linear-gradient(45deg, #dc3545, #fd7e14);
            border: none;
            padding: 10px 20px;
            border-radius: 20px;
            font-weight: 600;
        }
        
        .permission-group {
            background: #f8f9fa;
            border-radius: 12px;
            padding: 20px;
            margin-bottom: 15px;
            border: 1px solid #dee2e6;
        }
        
        .permission-item {
            background: white;
            border-radius: 8px;
            padding: 15px;
            margin-bottom: 10px;
            border: 1px solid #e9ecef;
            transition: all 0.3s ease;
        }
        
        .permission-item:hover {
            box-shadow: 0 3px 10px rgba(0, 0, 0, 0.1);
            transform: translateY(-1px);
        }
        
        .form-check-input {
            margin-right: 8px;
            transform: scale(1.2);
        }
        
        .table {
            background: white;
            border-radius: 12px;
            overflow: hidden;
            box-shadow: 0 5px 15px rgba(0, 0, 0, 0.08);
        }
        
        .table th {
            background: linear-gradient(45deg, #667eea, #764ba2);
            color: white;
            border: none;
            padding: 15px;
            font-weight: 600;
        }
        
        .table td {
            padding: 12px 15px;
            border-color: #f1f3f4;
        }
        
        .badge {
            padding: 6px 12px;
            border-radius: 20px;
            font-weight: 500;
        }
        
        .alert {
            border: none;
            border-radius: 12px;
            padding: 15px 20px;
        }
        
        .alert-success {
            background: linear-gradient(45deg, rgba(40, 167, 69, 0.1), rgba(32, 201, 151, 0.1));
            color: #155724;
            border-left: 4px solid #28a745;
        }
        
        .alert-danger {
            background: linear-gradient(45deg, rgba(220, 53, 69, 0.1), rgba(253, 126, 20, 0.1));
            color: #721c24;
            border-left: 4px solid #dc3545;
        }
        
        .section-header {
            display: flex;
            align-items: center;
            margin-bottom: 20px;
            padding-bottom: 10px;
            border-bottom: 2px solid #e9ecef;
        }
        
        .section-header i {
            background: linear-gradient(45deg, #667eea, #764ba2);
            color: white;
            padding: 10px;
            border-radius: 50%;
            margin-right: 15px;
            font-size: 18px;
        }
        
        .section-header h4 {
            margin: 0;
            color: #495057;
            font-weight: 700;
        }
        
        .quick-actions {
            background: linear-gradient(135deg, rgba(102, 126, 234, 0.1), rgba(118, 75, 162, 0.1));
            border-radius: 15px;
            padding: 20px;
            margin-bottom: 25px;
        }
        
        .quick-actions h5 {
            color: #667eea;
            font-weight: 700;
            margin-bottom: 15px;
        }
        
        .role-card {
            background: white;
            border-radius: 12px;
            padding: 20px;
            margin-bottom: 15px;
            box-shadow: 0 3px 10px rgba(0, 0, 0, 0.1);
            border-left: 4px solid #667eea;
            transition: all 0.3s ease;
        }
        
        .role-card:hover {
            transform: translateY(-3px);
            box-shadow: 0 8px 25px rgba(0, 0, 0, 0.15);
        }
        
        @keyframes fadeIn {
            from { opacity: 0; transform: translateY(20px); }
            to { opacity: 1; transform: translateY(0); }
        }
        
        .fade-in {
            animation: fadeIn 0.5s ease-out;
        }
    </style>
</head>
<body>
    <div class="container-fluid">
        <div class="row justify-content-center">
            <div class="col-12">
                <div class="main-container">
                    <div class="text-center mb-4">
                        <h1 class="display-5 fw-bold text-primary">
                            <i class="fas fa-users-cog"></i> Roles & Permissions Manager
                        </h1>
                        <p class="lead text-muted">Comprehensive management system for roles and permissions</p>
                    </div>

                    <!-- Alert Messages -->
                    <div id="alertContainer"></div>

                    <!-- Navigation Tabs -->
                    <ul class="nav nav-tabs" id="mainTabs" role="tablist">
                        <li class="nav-item" role="presentation">
                            <button class="nav-link active" id="roles-tab" data-bs-toggle="tab" data-bs-target="#roles" type="button" role="tab">
                                <i class="fas fa-user-tie"></i> Manage Roles
                            </button>
                        </li>
                        <li class="nav-item" role="presentation">
                            <button class="nav-link" id="permissions-tab" data-bs-toggle="tab" data-bs-target="#permissions" type="button" role="tab">
                                <i class="fas fa-key"></i> Manage Permissions
                            </button>
                        </li>
                        <li class="nav-item" role="presentation">
                            <button class="nav-link" id="assign-tab" data-bs-toggle="tab" data-bs-target="#assign" type="button" role="tab">
                                <i class="fas fa-link"></i> Assign Permissions
                            </button>
                        </li>
                        <li class="nav-item" role="presentation">
                            <button class="nav-link" id="export-tab" data-bs-toggle="tab" data-bs-target="#export" type="button" role="tab">
                                <i class="fas fa-download"></i> Export/Import
                            </button>
                        </li>
                    </ul>

                    <!-- Tab Content -->
                    <div class="tab-content" id="mainTabContent">
                        
                        <!-- Roles Management Tab -->
                        <div class="tab-pane fade show active" id="roles" role="tabpanel">
                            <div class="row">
                                <div class="col-md-4">
                                    <div class="form-section">
                                        <div class="section-header">
                                            <i class="fas fa-plus"></i>
                                            <h4>Add New Role</h4>
                                        </div>
                                        
                                        <form id="addRoleForm">
                                            <div class="mb-3">
                                                <label for="roleName" class="form-label">Role Name</label>
                                                <input type="text" class="form-control" id="roleName" placeholder="e.g., Lab Manager" required>
                                            </div>
                                            
                                            <div class="mb-3">
                                                <label for="roleSlug" class="form-label">Role Slug</label>
                                                <input type="text" class="form-control" id="roleSlug" placeholder="e.g., lab-manager" readonly>
                                                <small class="form-text text-muted">Auto-generated from role name</small>
                                            </div>
                                            
                                            <div class="mb-3">
                                                <div class="form-check">
                                                    <input class="form-check-input" type="checkbox" id="isSystemRole">
                                                    <label class="form-check-label" for="isSystemRole">
                                                        System Role (Cannot be deleted)
                                                    </label>
                                                </div>
                                            </div>
                                            
                                            <div class="mb-3">
                                                <div class="form-check">
                                                    <input class="form-check-input" type="checkbox" id="isActive" checked>
                                                    <label class="form-check-label" for="isActive">
                                                        Active Role
                                                    </label>
                                                </div>
                                            </div>
                                            
                                            <button type="submit" class="btn btn-primary w-100">
                                                <i class="fas fa-plus"></i> Add Role
                                            </button>
                                        </form>
                                    </div>
                                </div>
                                
                                <div class="col-md-8">
                                    <div class="form-section">
                                        <div class="section-header">
                                            <i class="fas fa-list"></i>
                                            <h4>Existing Roles</h4>
                                        </div>
                                        
                                        <div class="table-responsive">
                                            <table class="table table-hover">
                                                <thead>
                                                    <tr>
                                                        <th>ID</th>
                                                        <th>Role Name</th>
                                                        <th>Type</th>
                                                        <th>Status</th>
                                                        <th>Actions</th>
                                                    </tr>
                                                </thead>
                                                <tbody id="rolesTableBody">
                                                    <!-- Roles will be populated here -->
                                                </tbody>
                                            </table>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <!-- Permissions Management Tab -->
                        <div class="tab-pane fade" id="permissions" role="tabpanel">
                            <div class="row">
                                <div class="col-md-5">
                                    <div class="form-section">
                                        <div class="section-header">
                                            <i class="fas fa-layer-group"></i>
                                            <h4>Add Permission Group</h4>
                                        </div>
                                        
                                        <form id="addPermissionGroupForm">
                                            <div class="mb-3">
                                                <label for="groupName" class="form-label">Group Name</label>
                                                <input type="text" class="form-control" id="groupName" placeholder="e.g., Laboratory Management" required>
                                            </div>
                                            
                                            <div class="mb-3">
                                                <label for="groupShortCode" class="form-label">Short Code</label>
                                                <input type="text" class="form-control" id="groupShortCode" placeholder="e.g., lab_mgmt" readonly>
                                            </div>
                                            
                                            <div class="mb-3">
                                                <label for="sortOrder" class="form-label">Sort Order</label>
                                                <input type="number" class="form-control" id="sortOrder" value="10" step="0.1">
                                            </div>
                                            
                                            <button type="submit" class="btn btn-primary w-100">
                                                <i class="fas fa-plus"></i> Add Group
                                            </button>
                                        </form>
                                    </div>
                                    
                                    <div class="form-section">
                                        <div class="section-header">
                                            <i class="fas fa-key"></i>
                                            <h4>Add Permission</h4>
                                        </div>
                                        
                                        <form id="addPermissionForm">
                                            <div class="mb-3">
                                                <label for="permissionGroup" class="form-label">Permission Group</label>
                                                <select class="form-select" id="permissionGroup" required>
                                                    <option value="">Select Group</option>
                                                </select>
                                            </div>
                                            
                                            <div class="mb-3">
                                                <label for="permissionName" class="form-label">Permission Name</label>
                                                <input type="text" class="form-control" id="permissionName" placeholder="e.g., Lab Test Results" required>
                                            </div>
                                            
                                            <div class="mb-3">
                                                <label for="permissionShortCode" class="form-label">Short Code</label>
                                                <input type="text" class="form-control" id="permissionShortCode" placeholder="e.g., lab_test_results" readonly>
                                            </div>
                                            
                                            <div class="mb-3">
                                                <label class="form-label">Available Actions</label>
                                                <div class="row">
                                                    <div class="col-6">
                                                        <div class="form-check">
                                                            <input class="form-check-input" type="checkbox" id="enableView" checked>
                                                            <label class="form-check-label" for="enableView">View</label>
                                                        </div>
                                                        <div class="form-check">
                                                            <input class="form-check-input" type="checkbox" id="enableAdd">
                                                            <label class="form-check-label" for="enableAdd">Add</label>
                                                        </div>
                                                    </div>
                                                    <div class="col-6">
                                                        <div class="form-check">
                                                            <input class="form-check-input" type="checkbox" id="enableEdit">
                                                            <label class="form-check-label" for="enableEdit">Edit</label>
                                                        </div>
                                                        <div class="form-check">
                                                            <input class="form-check-input" type="checkbox" id="enableDelete">
                                                            <label class="form-check-label" for="enableDelete">Delete</label>
                                                        </div>
                                                    </div>
                                                </div>
                                            </div>
                                            
                                            <button type="submit" class="btn btn-primary w-100">
                                                <i class="fas fa-plus"></i> Add Permission
                                            </button>
                                        </form>
                                    </div>
                                </div>
                                
                                <div class="col-md-7">
                                    <div class="form-section">
                                        <div class="section-header">
                                            <i class="fas fa-sitemap"></i>
                                            <h4>Permission Structure</h4>
                                        </div>
                                        
                                        <div id="permissionStructure">
                                            <!-- Permission groups and categories will be populated here -->
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <!-- Assign Permissions Tab -->
                        <div class="tab-pane fade" id="assign" role="tabpanel">
                            <div class="quick-actions">
                                <h5><i class="fas fa-lightning-bolt"></i> Quick Actions</h5>
                                <div class="row">
                                    <div class="col-md-4">
                                        <button class="btn btn-success btn-sm" onclick="copyPermissions()">
                                            <i class="fas fa-copy"></i> Copy Permissions
                                        </button>
                                    </div>
                                    <div class="col-md-4">
                                        <button class="btn btn-primary btn-sm" onclick="bulkAssign()">
                                            <i class="fas fa-users"></i> Bulk Assign
                                        </button>
                                    </div>
                                    <div class="col-md-4">
                                        <button class="btn btn-danger btn-sm" onclick="resetPermissions()">
                                            <i class="fas fa-undo"></i> Reset All
                                        </button>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="row">
                                <div class="col-md-3">
                                    <div class="form-section">
                                        <div class="section-header">
                                            <i class="fas fa-user-check"></i>
                                            <h4>Select Role</h4>
                                        </div>
                                        
                                        <div id="roleSelection">
                                            <!-- Role selection cards will be populated here -->
                                        </div>
                                    </div>
                                </div>
                                
                                <div class="col-md-9">
                                    <div class="form-section">
                                        <div class="section-header">
                                            <i class="fas fa-cogs"></i>
                                            <h4>Assign Permissions</h4>
                                        </div>
                                        
                                        <div id="permissionAssignment">
                                            <div class="text-center text-muted py-5">
                                                <i class="fas fa-arrow-left fa-3x mb-3"></i>
                                                <h5>Select a role to assign permissions</h5>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <!-- Export/Import Tab -->
                        <div class="tab-pane fade" id="export" role="tabpanel">
                            <div class="row">
                                <div class="col-md-6">
                                    <div class="form-section">
                                        <div class="section-header">
                                            <i class="fas fa-download"></i>
                                            <h4>Export Configuration</h4>
                                        </div>
                                        
                                        <p class="text-muted">Export your current roles and permissions configuration as SQL statements or JSON format.</p>
                                        
                                        <div class="mb-3">
                                            <label class="form-label">Export Format</label>
                                            <div class="form-check">
                                                <input class="form-check-input" type="radio" name="exportFormat" id="exportSQL" value="sql" checked>
                                                <label class="form-check-label" for="exportSQL">SQL Statements</label>
                                            </div>
                                            <div class="form-check">
                                                <input class="form-check-input" type="radio" name="exportFormat" id="exportJSON" value="json">
                                                <label class="form-check-label" for="exportJSON">JSON Format</label>
                                            </div>
                                        </div>
                                        
                                        <div class="mb-3">
                                            <div class="form-check">
                                                <input class="form-check-input" type="checkbox" id="includeRoles" checked>
                                                <label class="form-check-label" for="includeRoles">Include Roles</label>
                                            </div>
                                            <div class="form-check">
                                                <input class="form-check-input" type="checkbox" id="includePermissions" checked>
                                                <label class="form-check-label" for="includePermissions">Include Permissions</label>
                                            </div>
                                            <div class="form-check">
                                                <input class="form-check-input" type="checkbox" id="includeAssignments" checked>
                                                <label class="form-check-label" for="includeAssignments">Include Role Assignments</label>
                                            </div>
                                        </div>
                                        
                                        <button class="btn btn-primary w-100" onclick="exportConfiguration()">
                                            <i class="fas fa-download"></i> Export Configuration
                                        </button>
                                    </div>
                                </div>
                                
                                <div class="col-md-6">
                                    <div class="form-section">
                                        <div class="section-header">
                                            <i class="fas fa-upload"></i>
                                            <h4>Import Configuration</h4>
                                        </div>
                                        
                                        <p class="text-muted">Import roles and permissions from a previously exported configuration file.</p>
                                        
                                        <div class="mb-3">
                                            <label for="importFile" class="form-label">Configuration File</label>
                                            <input class="form-control" type="file" id="importFile" accept=".sql,.json">
                                        </div>
                                        
                                        <div class="mb-3">
                                            <div class="form-check">
                                                <input class="form-check-input" type="checkbox" id="backupBeforeImport" checked>
                                                <label class="form-check-label" for="backupBeforeImport">
                                                    Create backup before import
                                                </label>
                                            </div>
                                        </div>
                                        
                                        <button class="btn btn-success w-100" onclick="importConfiguration()">
                                            <i class="fas fa-upload"></i> Import Configuration
                                        </button>
                                    </div>
                                    
                                    <div class="form-section">
                                        <div class="section-header">
                                            <i class="fas fa-shield-alt"></i>
                                            <h4>Backup & Restore</h4>
                                        </div>
                                        
                                        <div class="d-grid gap-2">
                                            <button class="btn btn-secondary" onclick="createBackup()">
                                                <i class="fas fa-save"></i> Create Backup
                                            </button>
                                            <button class="btn btn-warning" onclick="restoreFromBackup()">
                                                <i class="fas fa-history"></i> Restore from Backup
                                            </button>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Modals -->
    
    <!-- Edit Role Modal -->
    <div class="modal fade" id="editRoleModal" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Edit Role</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <form id="editRoleForm">
                        <input type="hidden" id="editRoleId">
                        <div class="mb-3">
                            <label for="editRoleName" class="form-label">Role Name</label>
                            <input type="text" class="form-control" id="editRoleName" required>
                        </div>
                        <div class="mb-3">
                            <label for="editRoleSlug" class="form-label">Role Slug</label>
                            <input type="text" class="form-control" id="editRoleSlug" readonly>
                        </div>
                        <div class="mb-3">
                            <div class="form-check">
                                <input class="form-check-input" type="checkbox" id="editIsSystemRole">
                                <label class="form-check-label" for="editIsSystemRole">System Role</label>
                            </div>
                        </div>
                        <div class="mb-3">
                            <div class="form-check">
                                <input class="form-check-input" type="checkbox" id="editIsActive">
                                <label class="form-check-label" for="editIsActive">Active Role</label>
                            </div>
                        </div>
                    </form>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="button" class="btn btn-primary" onclick="updateRole()">Update Role</button>
                </div>
            </div>
        </div>
    </div>

    <!-- Copy Permissions Modal -->
    <div class="modal fade" id="copyPermissionsModal" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Copy Permissions</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <div class="mb-3">
                        <label for="sourceRole" class="form-label">Copy From (Source Role)</label>
                        <select class="form-select" id="sourceRole" required>
                            <option value="">Select source role</option>
                        </select>
                    </div>
                    <div class="mb-3">
                        <label for="targetRole" class="form-label">Copy To (Target Role)</label>
                        <select class="form-select" id="targetRole" required>
                            <option value="">Select target role</option>
                        </select>
                    </div>
                    <div class="mb-3">
                        <div class="form-check">
                            <input class="form-check-input" type="checkbox" id="overwriteExisting">
                            <label class="form-check-label" for="overwriteExisting">
                                Overwrite existing permissions
                            </label>
                        </div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="button" class="btn btn-success" onclick="executeCopyPermissions()">Copy Permissions</button>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdnjs.cloudflare.com/ajax/libs/bootstrap/5.3.0/js/bootstrap.bundle.min.js"></script>
    <script>
        // Sample data structure
        let roles = [
            {id: 1, name: 'Admin', slug: 'admin', is_system: 1, is_active: 1, is_superadmin: 0},
            {id: 2, name: 'Accountant', slug: 'accountant', is_system: 1, is_active: 1, is_superadmin: 0},
            {id: 3, name: 'Doctor', slug: 'doctor', is_system: 1, is_active: 1, is_superadmin: 0},
            {id: 4, name: 'Pharmacist', slug: 'pharmacist', is_system: 1, is_active: 1, is_superadmin: 0},
            {id: 5, name: 'Pathologist', slug: 'pathologist', is_system: 1, is_active: 1, is_superadmin: 0},
            {id: 6, name: 'Radiologist', slug: 'radiologist', is_system: 1, is_active: 1, is_superadmin: 0},
            {id: 7, name: 'Super Admin', slug: 'super-admin', is_system: 1, is_active: 1, is_superadmin: 1},
            {id: 8, name: 'Receptionist', slug: 'receptionist', is_system: 1, is_active: 1, is_superadmin: 0},
            {id: 9, name: 'Nurse', slug: 'nurse', is_system: 1, is_active: 1, is_superadmin: 0},
            {id: 10, name: 'Lab Tech', slug: 'lab-tech', is_system: 1, is_active: 1, is_superadmin: 0}
        ];

        let permissionGroups = [
            {id: 3, name: 'Income', short_code: 'income', is_active: 1, sort_order: 15.00},
            {id: 4, name: 'Expense', short_code: 'expense', is_active: 1, sort_order: 16.00},
            {id: 8, name: 'Content', short_code: 'content', is_active: 1, sort_order: 17.00},
            {id: 10, name: 'Inventory', short_code: 'inventory', is_active: 1, sort_order: 18.00},
            {id: 13, name: 'Communicate', short_code: 'communicate', is_active: 1, sort_order: 19.00},
            {id: 14, name: 'Reports', short_code: 'reports', is_active: 1, sort_order: 20.00},
            {id: 15, name: 'System Settings', short_code: 'system_settings', is_active: 1, sort_order: 21.00},
            {id: 16, name: 'Front CMS', short_code: 'front_cms', is_active: 1, sort_order: 22.00},
            {id: 17, name: 'Front Office', short_code: 'front_office', is_active: 1, sort_order: 23.00}
        ];

        let permissionCategories = [
            {id: 9, perm_group_id: 3, name: 'Income', short_code: 'income', enable_view: 1, enable_add: 1, enable_edit: 1, enable_delete: 1},
            {id: 10, perm_group_id: 3, name: 'Income Head', short_code: 'income_head', enable_view: 1, enable_add: 1, enable_edit: 1, enable_delete: 1},
            {id: 12, perm_group_id: 4, name: 'Expense', short_code: 'expense', enable_view: 1, enable_add: 1, enable_edit: 1, enable_delete: 1},
            {id: 13, perm_group_id: 4, name: 'Expense Head', short_code: 'expense_head', enable_view: 1, enable_add: 1, enable_edit: 1, enable_delete: 1},
            {id: 27, perm_group_id: 8, name: 'Content Type', short_code: 'content_type', enable_view: 1, enable_add: 1, enable_edit: 1, enable_delete: 1},
            {id: 31, perm_group_id: 10, name: 'Issue Item', short_code: 'issue_item', enable_view: 1, enable_add: 1, enable_edit: 0, enable_delete: 1},
            {id: 32, perm_group_id: 10, name: 'Item Stock', short_code: 'item_stock', enable_view: 1, enable_add: 1, enable_edit: 1, enable_delete: 1},
            {id: 33, perm_group_id: 10, name: 'Item', short_code: 'item', enable_view: 1, enable_add: 1, enable_edit: 1, enable_delete: 1},
            {id: 34, perm_group_id: 10, name: 'Store', short_code: 'store', enable_view: 1, enable_add: 1, enable_edit: 1, enable_delete: 1},
            {id: 35, perm_group_id: 10, name: 'Supplier', short_code: 'supplier', enable_view: 1, enable_add: 1, enable_edit: 1, enable_delete: 1}
        ];

        let rolePermissions = [
            {id: 1, role_id: 1, perm_cat_id: 9, can_view: 1, can_add: 1, can_edit: 1, can_delete: 1},
            {id: 2, role_id: 1, perm_cat_id: 10, can_view: 1, can_add: 1, can_edit: 1, can_delete: 1},
            {id: 3, role_id: 1, perm_cat_id: 12, can_view: 1, can_add: 1, can_edit: 1, can_delete: 1},
            {id: 4, role_id: 1, perm_cat_id: 13, can_view: 1, can_add: 1, can_edit: 1, can_delete: 1},
            {id: 5, role_id: 2, perm_cat_id: 9, can_view: 1, can_add: 1, can_edit: 1, can_delete: 0},
            {id: 6, role_id: 2, perm_cat_id: 12, can_view: 1, can_add: 1, can_edit: 1, can_delete: 0}
        ];

        // Initialize the application
        document.addEventListener('DOMContentLoaded', function() {
            initializeApp();
        });

        function initializeApp() {
            loadRoles();
            loadPermissionGroups();
            loadPermissionStructure();
            loadRoleSelection();
            setupEventListeners();
        }

        function setupEventListeners() {
            // Role name to slug conversion
            document.getElementById('roleName').addEventListener('input', function() {
                const slug = this.value.toLowerCase().replace(/[^a-z0-9]+/g, '-').replace(/(^-|-$)/g, '');
                document.getElementById('roleSlug').value = slug;
            });

            // Group name to short code conversion
            document.getElementById('groupName').addEventListener('input', function() {
                const shortCode = this.value.toLowerCase().replace(/[^a-z0-9]+/g, '_').replace(/(^_|_$)/g, '');
                document.getElementById('groupShortCode').value = shortCode;
            });

            // Permission name to short code conversion
            document.getElementById('permissionName').addEventListener('input', function() {
                const shortCode = this.value.toLowerCase().replace(/[^a-z0-9]+/g, '_').replace(/(^_|_$)/g, '');
                document.getElementById('permissionShortCode').value = shortCode;
            });

            // Form submissions
            document.getElementById('addRoleForm').addEventListener('submit', handleAddRole);
            document.getElementById('addPermissionGroupForm').addEventListener('submit', handleAddPermissionGroup);
            document.getElementById('addPermissionForm').addEventListener('submit', handleAddPermission);

            // Edit role name to slug conversion
            document.getElementById('editRoleName').addEventListener('input', function() {
                const slug = this.value.toLowerCase().replace(/[^a-z0-9]+/g, '-').replace(/(^-|-$)/g, '');
                document.getElementById('editRoleSlug').value = slug;
            });
        }

        function showAlert(message, type = 'success') {
            const alertContainer = document.getElementById('alertContainer');
            const alertHtml = `
                <div class="alert alert-${type} alert-dismissible fade show fade-in" role="alert">
                    <i class="fas fa-${type === 'success' ? 'check-circle' : 'exclamation-triangle'}"></i>
                    ${message}
                    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                </div>
            `;
            alertContainer.innerHTML = alertHtml;
            
            // Auto-dismiss after 5 seconds
            setTimeout(() => {
                const alert = alertContainer.querySelector('.alert');
                if (alert) {
                    alert.remove();
                }
            }, 5000);
        }

        function loadRoles() {
            const tbody = document.getElementById('rolesTableBody');
            tbody.innerHTML = '';

            roles.forEach(role => {
                const statusBadge = role.is_active ? 
                    '<span class="badge bg-success">Active</span>' : 
                    '<span class="badge bg-secondary">Inactive</span>';
                
                const typeBadge = role.is_superadmin ? 
                    '<span class="badge bg-danger">Super Admin</span>' : 
                    role.is_system ? 
                    '<span class="badge bg-primary">System</span>' : 
                    '<span class="badge bg-info">Custom</span>';

                const row = `
                    <tr>
                        <td>${role.id}</td>
                        <td><strong>${role.name}</strong></td>
                        <td>${typeBadge}</td>
                        <td>${statusBadge}</td>
                        <td>
                            <div class="btn-group btn-group-sm">
                                <button class="btn btn-outline-primary" onclick="editRole(${role.id})" title="Edit">
                                    <i class="fas fa-edit"></i>
                                </button>
                                <button class="btn btn-outline-info" onclick="viewPermissions(${role.id})" title="View Permissions">
                                    <i class="fas fa-key"></i>
                                </button>
                                ${!role.is_system ? `
                                <button class="btn btn-outline-danger" onclick="deleteRole(${role.id})" title="Delete">
                                    <i class="fas fa-trash"></i>
                                </button>` : ''}
                            </div>
                        </td>
                    </tr>
                `;
                tbody.innerHTML += row;
            });
        }

        function loadPermissionGroups() {
            const select = document.getElementById('permissionGroup');
            select.innerHTML = '<option value="">Select Group</option>';
            
            permissionGroups.forEach(group => {
                select.innerHTML += `<option value="${group.id}">${group.name}</option>`;
            });
        }

        function loadPermissionStructure() {
            const container = document.getElementById('permissionStructure');
            container.innerHTML = '';

            permissionGroups.forEach(group => {
                const groupCategories = permissionCategories.filter(cat => cat.perm_group_id === group.id);
                
                const groupHtml = `
                    <div class="permission-group">
                        <div class="d-flex justify-content-between align-items-center mb-3">
                            <h5 class="mb-0">
                                <i class="fas fa-folder text-primary"></i>
                                ${group.name}
                                <small class="text-muted">(${group.short_code})</small>
                            </h5>
                            <div class="btn-group btn-group-sm">
                                <button class="btn btn-outline-secondary" onclick="editPermissionGroup(${group.id})">
                                    <i class="fas fa-edit"></i>
                                </button>
                                <button class="btn btn-outline-danger" onclick="deletePermissionGroup(${group.id})">
                                    <i class="fas fa-trash"></i>
                                </button>
                            </div>
                        </div>
                        
                        <div class="row">
                            ${groupCategories.map(category => `
                                <div class="col-md-6 mb-2">
                                    <div class="permission-item">
                                        <div class="d-flex justify-content-between align-items-center">
                                            <div>
                                                <strong>${category.name}</strong>
                                                <br>
                                                <small class="text-muted">${category.short_code}</small>
                                            </div>
                                            <div>
                                                <div class="btn-group btn-group-sm">
                                                    ${category.enable_view ? '<span class="badge bg-success">V</span>' : ''}
                                                    ${category.enable_add ? '<span class="badge bg-primary">A</span>' : ''}
                                                    ${category.enable_edit ? '<span class="badge bg-warning">E</span>' : ''}
                                                    ${category.enable_delete ? '<span class="badge bg-danger">D</span>' : ''}
                                                </div>
                                                <div class="btn-group btn-group-sm ms-2">
                                                    <button class="btn btn-outline-secondary" onclick="editPermission(${category.id})">
                                                        <i class="fas fa-edit"></i>
                                                    </button>
                                                    <button class="btn btn-outline-danger" onclick="deletePermission(${category.id})">
                                                        <i class="fas fa-trash"></i>
                                                    </button>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            `).join('')}
                        </div>
                    </div>
                `;
                container.innerHTML += groupHtml;
            });
        }

        function loadRoleSelection() {
            const container = document.getElementById('roleSelection');
            container.innerHTML = '';

            roles.forEach(role => {
                const permissionCount = rolePermissions.filter(rp => rp.role_id === role.id).length;
                const statusClass = role.is_active ? 'border-success' : 'border-secondary';
                
                const roleCard = `
                    <div class="role-card ${statusClass}" onclick="selectRole(${role.id})" style="cursor: pointer;">
                        <div class="d-flex justify-content-between align-items-center">
                            <div>
                                <h6 class="mb-1">${role.name}</h6>
                                <small class="text-muted">${permissionCount} permissions</small>
                            </div>
                            <div>
                                ${role.is_superadmin ? '<i class="fas fa-crown text-warning"></i>' : ''}
                                ${role.is_system ? '<i class="fas fa-shield-alt text-primary"></i>' : ''}
                            </div>
                        </div>
                    </div>
                `;
                container.innerHTML += roleCard;
            });

            // Also populate dropdowns for copy permissions
            const sourceRole = document.getElementById('sourceRole');
            const targetRole = document.getElementById('targetRole');
            
            sourceRole.innerHTML = '<option value="">Select source role</option>';
            targetRole.innerHTML = '<option value="">Select target role</option>';
            
            roles.forEach(role => {
                sourceRole.innerHTML += `<option value="${role.id}">${role.name}</option>`;
                targetRole.innerHTML += `<option value="${role.id}">${role.name}</option>`;
            });
        }

        function selectRole(roleId) {
            const role = roles.find(r => r.id === roleId);
            const container = document.getElementById('permissionAssignment');
            
            // Highlight selected role
            document.querySelectorAll('.role-card').forEach(card => {
                card.classList.remove('border-primary');
            });
            event.currentTarget.classList.add('border-primary');
            
            // Load permissions for this role
            loadPermissionAssignment(roleId);
        }

        function loadPermissionAssignment(roleId) {
            const role = roles.find(r => r.id === roleId);
            const container = document.getElementById('permissionAssignment');
            
            let html = `
                <div class="d-flex justify-content-between align-items-center mb-4">
                    <h5>
                        <i class="fas fa-user-cog"></i>
                        Permissions for: <strong>${role.name}</strong>
                    </h5>
                    <button class="btn btn-success" onclick="saveRolePermissions(${roleId})">
                        <i class="fas fa-save"></i> Save Permissions
                    </button>
                </div>
            `;

            permissionGroups.forEach(group => {
                const groupCategories = permissionCategories.filter(cat => cat.perm_group_id === group.id);
                
                html += `
                    <div class="permission-group">
                        <h6 class="text-primary mb-3">
                            <i class="fas fa-folder"></i> ${group.name}
                        </h6>
                        
                        <div class="table-responsive">
                            <table class="table table-sm">
                                <thead>
                                    <tr>
                                        <th>Permission</th>
                                        <th width="80">View</th>
                                        <th width="80">Add</th>
                                        <th width="80">Edit</th>
                                        <th width="80">Delete</th>
                                    </tr>
                                </thead>
                                <tbody>
                `;
                
                groupCategories.forEach(category => {
                    const rolePermission = rolePermissions.find(rp => rp.role_id === roleId && rp.perm_cat_id === category.id);
                    
                    html += `
                        <tr>
                            <td>
                                <strong>${category.name}</strong>
                                <br><small class="text-muted">${category.short_code}</small>
                            </td>
                            <td>
                                ${category.enable_view ? `
                                <div class="form-check">
                                    <input class="form-check-input" type="checkbox" 
                                           id="perm_${category.id}_view" 
                                           data-role="${roleId}" 
                                           data-permission="${category.id}" 
                                           data-action="view"
                                           ${rolePermission && rolePermission.can_view ? 'checked' : ''}>
                                </div>` : '-'}
                            </td>
                            <td>
                                ${category.enable_add ? `
                                <div class="form-check">
                                    <input class="form-check-input" type="checkbox" 
                                           id="perm_${category.id}_add" 
                                           data-role="${roleId}" 
                                           data-permission="${category.id}" 
                                           data-action="add"
                                           ${rolePermission && rolePermission.can_add ? 'checked' : ''}>
                                </div>` : '-'}
                            </td>
                            <td>
                                ${category.enable_edit ? `
                                <div class="form-check">
                                    <input class="form-check-input" type="checkbox" 
                                           id="perm_${category.id}_edit" 
                                           data-role="${roleId}" 
                                           data-permission="${category.id}" 
                                           data-action="edit"
                                           ${rolePermission && rolePermission.can_edit ? 'checked' : ''}>
                                </div>` : '-'}
                            </td>
                            <td>
                                ${category.enable_delete ? `
                                <div class="form-check">
                                    <input class="form-check-input" type="checkbox" 
                                           id="perm_${category.id}_delete" 
                                           data-role="${roleId}" 
                                           data-permission="${category.id}" 
                                           data-action="delete"
                                           ${rolePermission && rolePermission.can_delete ? 'checked' : ''}>
                                </div>` : '-'}
                            </td>
                        </tr>
                    `;
                });
                
                html += `
                                </tbody>
                            </table>
                        </div>
                    </div>
                `;
            });
            
            container.innerHTML = html;
        }

        function handleAddRole(e) {
            e.preventDefault();
            
            const formData = {
                name: document.getElementById('roleName').value,
                slug: document.getElementById('roleSlug').value,
                is_system: document.getElementById('isSystemRole').checked ? 1 : 0,
                is_active: document.getElementById('isActive').checked ? 1 : 0,
                is_superadmin: 0
            };

            // Simulate API call
            const newId = Math.max(...roles.map(r => r.id)) + 1;
            roles.push({
                id: newId,
                ...formData
            });

            showAlert(`Role "${formData.name}" added successfully!`);
            loadRoles();
            loadRoleSelection();
            
            // Reset form
            document.getElementById('addRoleForm').reset();
        }

        function handleAddPermissionGroup(e) {
            e.preventDefault();
            
            const formData = {
                name: document.getElementById('groupName').value,
                short_code: document.getElementById('groupShortCode').value,
                sort_order: parseFloat(document.getElementById('sortOrder').value),
                is_active: 1,
                system: 0
            };

            // Simulate API call
            const newId = Math.max(...permissionGroups.map(g => g.id)) + 1;
            permissionGroups.push({
                id: newId,
                ...formData
            });

            showAlert(`Permission group "${formData.name}" added successfully!`);
            loadPermissionGroups();
            loadPermissionStructure();
            
            // Reset form
            document.getElementById('addPermissionGroupForm').reset();
        }

        function handleAddPermission(e) {
            e.preventDefault();
            
            const formData = {
                perm_group_id: parseInt(document.getElementById('permissionGroup').value),
                name: document.getElementById('permissionName').value,
                short_code: document.getElementById('permissionShortCode').value,
                enable_view: document.getElementById('enableView').checked ? 1 : 0,
                enable_add: document.getElementById('enableAdd').checked ? 1 : 0,
                enable_edit: document.getElementById('enableEdit').checked ? 1 : 0,
                enable_delete: document.getElementById('enableDelete').checked ? 1 : 0
            };

            // Simulate API call
            const newId = Math.max(...permissionCategories.map(p => p.id)) + 1;
            permissionCategories.push({
                id: newId,
                ...formData
            });

            showAlert(`Permission "${formData.name}" added successfully!`);
            loadPermissionStructure();
            
            // Reset form
            document.getElementById('addPermissionForm').reset();
        }

        function editRole(roleId) {
            const role = roles.find(r => r.id === roleId);
            
            document.getElementById('editRoleId').value = role.id;
            document.getElementById('editRoleName').value = role.name;
            document.getElementById('editRoleSlug').value = role.slug;
            document.getElementById('editIsSystemRole').checked = role.is_system === 1;
            document.getElementById('editIsActive').checked = role.is_active === 1;
            
            const modal = new bootstrap.Modal(document.getElementById('editRoleModal'));
            modal.show();
        }

        function updateRole() {
            const roleId = parseInt(document.getElementById('editRoleId').value);
            const roleIndex = roles.findIndex(r => r.id === roleId);
            
            if (roleIndex !== -1) {
                roles[roleIndex] = {
                    ...roles[roleIndex],
                    name: document.getElementById('editRoleName').value,
                    slug: document.getElementById('editRoleSlug').value,
                    is_system: document.getElementById('editIsSystemRole').checked ? 1 : 0,
                    is_active: document.getElementById('editIsActive').checked ? 1 : 0
                };
                
                showAlert(`Role updated successfully!`);
                loadRoles();
                loadRoleSelection();
                
                const modal = bootstrap.Modal.getInstance(document.getElementById('editRoleModal'));
                modal.hide();
            }
        }

        function deleteRole(roleId) {
            const role = roles.find(r => r.id === roleId);
            
            if (role.is_system) {
                showAlert('Cannot delete system roles!', 'danger');
                return;
            }
            
            if (confirm(`Are you sure you want to delete the role "${role.name}"?`)) {
                // Remove role and its permissions
                roles = roles.filter(r => r.id !== roleId);
                rolePermissions = rolePermissions.filter(rp => rp.role_id !== roleId);
                
                showAlert(`Role "${role.name}" deleted successfully!`);
                loadRoles();
                loadRoleSelection();
            }
        }

        function viewPermissions(roleId) {
            // Switch to assign permissions tab and select this role
            const assignTab = new bootstrap.Tab(document.getElementById('assign-tab'));
            assignTab.show();
            
            setTimeout(() => {
                selectRole(roleId);
            }, 100);
        }

        function saveRolePermissions(roleId) {
            const checkboxes = document.querySelectorAll(`input[data-role="${roleId}"]`);
            
            // Remove existing permissions for this role
            rolePermissions = rolePermissions.filter(rp => rp.role_id !== roleId);
            
            const permissionMap = {};
            
            checkboxes.forEach(checkbox => {
                const permissionId = parseInt(checkbox.dataset.permission);
                const action = checkbox.dataset.action;
                
                if (!permissionMap[permissionId]) {
                    permissionMap[permissionId] = {
                        role_id: roleId,
                        perm_cat_id: permissionId,
                        can_view: 0,
                        can_add: 0,
                        can_edit: 0,
                        can_delete: 0
                    };
                }
                
                if (checkbox.checked) {
                    permissionMap[permissionId][`can_${action}`] = 1;
                }
            });
            
            // Add new permissions
            Object.values(permissionMap).forEach(permission => {
                if (permission.can_view || permission.can_add || permission.can_edit || permission.can_delete) {
                    const newId = rolePermissions.length > 0 ? Math.max(...rolePermissions.map(rp => rp.id)) + 1 : 1;
                    rolePermissions.push({
                        id: newId,
                        ...permission
                    });
                }
            });
            
            showAlert('Permissions saved successfully!');
            loadRoleSelection(); // Update permission counts
        }

        function copyPermissions() {
            const modal = new bootstrap.Modal(document.getElementById('copyPermissionsModal'));
            modal.show();
        }

        function executeCopyPermissions() {
            const sourceRoleId = parseInt(document.getElementById('sourceRole').value);
            const targetRoleId = parseInt(document.getElementById('targetRole').value);
            const overwrite = document.getElementById('overwriteExisting').checked;
            
            if (!sourceRoleId || !targetRoleId) {
                showAlert('Please select both source and target roles!', 'danger');
                return;
            }
            
            if (sourceRoleId === targetRoleId) {
                showAlert('Source and target roles cannot be the same!', 'danger');
                return;
            }
            
            // Get source permissions
            const sourcePermissions = rolePermissions.filter(rp => rp.role_id === sourceRoleId);
            
            if (overwrite) {
                // Remove existing target permissions
                rolePermissions = rolePermissions.filter(rp => rp.role_id !== targetRoleId);
            }
            
            // Copy permissions
            sourcePermissions.forEach(perm => {
                const existingIndex = rolePermissions.findIndex(rp => 
                    rp.role_id === targetRoleId && rp.perm_cat_id === perm.perm_cat_id
                );
                
                if (existingIndex === -1 || overwrite) {
                    const newId = Math.max(...rolePermissions.map(rp => rp.id)) + 1;
                    rolePermissions.push({
                        id: newId,
                        role_id: targetRoleId,
                        perm_cat_id: perm.perm_cat_id,
                        can_view: perm.can_view,
                        can_add: perm.can_add,
                        can_edit: perm.can_edit,
                        can_delete: perm.can_delete
                    });
                }
            });
            
            const sourceRole = roles.find(r => r.id === sourceRoleId);
            const targetRole = roles.find(r => r.id === targetRoleId);
            
            showAlert(`Permissions copied from "${sourceRole.name}" to "${targetRole.name}" successfully!`);
            loadRoleSelection();
            
            const modal = bootstrap.Modal.getInstance(document.getElementById('copyPermissionsModal'));
            modal.hide();
        }

        function bulkAssign() {
            showAlert('Bulk assign feature coming soon!', 'info');
        }

        function resetPermissions() {
            if (confirm('Are you sure you want to reset all permissions? This action cannot be undone!')) {
                rolePermissions = [];
                showAlert('All permissions have been reset!', 'warning');
                loadRoleSelection();
            }
        }

        function exportConfiguration() {
            const format = document.querySelector('input[name="exportFormat"]:checked').value;
            const includeRoles = document.getElementById('includeRoles').checked;
            const includePermissions = document.getElementById('includePermissions').checked;
            const includeAssignments = document.getElementById('includeAssignments').checked;
            
            let exportData = {};
            
            if (includeRoles) {
                exportData.roles = roles;
            }
            
            if (includePermissions) {
                exportData.permission_groups = permissionGroups;
                exportData.permission_categories = permissionCategories;
            }
            
            if (includeAssignments) {
                exportData.role_permissions = rolePermissions;
            }
            
            if (format === 'sql') {
                exportAsSql(exportData);
            } else {
                exportAsJson(exportData);
            }
        }
        
        function exportAsSql(data) {
            let sql = "-- Roles and Permissions Export\n";
            sql += "-- Generated on: " + new Date().toISOString() + "\n\n";
            
            if (data.roles) {
                sql += "-- Roles\n";
                data.roles.forEach(role => {
                    sql += `INSERT INTO roles (id, name, slug, is_active, is_system, is_superadmin) VALUES (${role.id}, '${role.name}', '${role.slug}', ${role.is_active}, ${role.is_system}, ${role.is_superadmin});\n`;
                });
                sql += "\n";
            }
            
            if (data.permission_groups) {
                sql += "-- Permission Groups\n";
                data.permission_groups.forEach(group => {
                    sql += `INSERT INTO permission_group (id, name, short_code, is_active, system, sort_order) VALUES (${group.id}, '${group.name}', '${group.short_code}', ${group.is_active}, 0, ${group.sort_order});\n`;
                });
                sql += "\n";
            }
            
            if (data.permission_categories) {
                sql += "-- Permission Categories\n";
                data.permission_categories.forEach(category => {
                    sql += `INSERT INTO permission_category (id, perm_group_id, name, short_code, enable_view, enable_add, enable_edit, enable_delete) VALUES (${category.id}, ${category.perm_group_id}, '${category.name}', '${category.short_code}', ${category.enable_view}, ${category.enable_add}, ${category.enable_edit}, ${category.enable_delete});\n`;
                });
                sql += "\n";
            }
            
            if (data.role_permissions) {
                sql += "-- Role Permissions\n";
                data.role_permissions.forEach(rp => {
                    sql += `INSERT INTO roles_permissions (id, role_id, perm_cat_id, can_view, can_add, can_edit, can_delete) VALUES (${rp.id}, ${rp.role_id}, ${rp.perm_cat_id}, ${rp.can_view}, ${rp.can_add}, ${rp.can_edit}, ${rp.can_delete});\n`;
                });
            }
            
            downloadFile(sql, 'roles_permissions_export.sql', 'text/plain');
        }
        
        function exportAsJson(data) {
            const jsonData = JSON.stringify(data, null, 2);
            downloadFile(jsonData, 'roles_permissions_export.json', 'application/json');
        }
        
        function downloadFile(content, filename, contentType) {
            const blob = new Blob([content], { type: contentType });
            const url = URL.createObjectURL(blob);
            const link = document.createElement('a');
            link.href = url;
            link.download = filename;
            document.body.appendChild(link);
            link.click();
            document.body.removeChild(link);
            URL.revokeObjectURL(url);
            
            showAlert(`Configuration exported as ${filename}!`);
        }
        
        function importConfiguration() {
            const fileInput = document.getElementById('importFile');
            const file = fileInput.files[0];
            
            if (!file) {
                showAlert('Please select a file to import!', 'danger');
                return;
            }
            
            const reader = new FileReader();
            reader.onload = function(e) {
                try {
                    if (file.name.endsWith('.json')) {
                        const importData = JSON.parse(e.target.result);
                        processImportData(importData);
                    } else if (file.name.endsWith('.sql')) {
                        showAlert('SQL import feature coming soon!', 'info');
                    } else {
                        showAlert('Unsupported file format!', 'danger');
                    }
                } catch (error) {
                    showAlert('Error parsing file: ' + error.message, 'danger');
                }
            };
            reader.readAsText(file);
        }
        
        function processImportData(data) {
            const backup = document.getElementById('backupBeforeImport').checked;
            
            if (backup) {
                createBackup();
            }
            
            if (data.roles) {
                // Merge roles (avoiding duplicates)
                data.roles.forEach(importRole => {
                    const existingIndex = roles.findIndex(r => r.id === importRole.id);
                    if (existingIndex !== -1) {
                        roles[existingIndex] = importRole;
                    } else {
                        roles.push(importRole);
                    }
                });
            }
            
            if (data.permission_groups) {
                data.permission_groups.forEach(importGroup => {
                    const existingIndex = permissionGroups.findIndex(g => g.id === importGroup.id);
                    if (existingIndex !== -1) {
                        permissionGroups[existingIndex] = importGroup;
                    } else {
                        permissionGroups.push(importGroup);
                    }
                });
            }
            
            if (data.permission_categories) {
                data.permission_categories.forEach(importCategory => {
                    const existingIndex = permissionCategories.findIndex(c => c.id === importCategory.id);
                    if (existingIndex !== -1) {
                        permissionCategories[existingIndex] = importCategory;
                    } else {
                        permissionCategories.push(importCategory);
                    }
                });
            }
            
            if (data.role_permissions) {
                data.role_permissions.forEach(importRP => {
                    const existingIndex = rolePermissions.findIndex(rp => rp.id === importRP.id);
                    if (existingIndex !== -1) {
                        rolePermissions[existingIndex] = importRP;
                    } else {
                        rolePermissions.push(importRP);
                    }
                });
            }
            
            // Refresh all displays
            loadRoles();
            loadPermissionGroups();
            loadPermissionStructure();
            loadRoleSelection();
            
            showAlert('Configuration imported successfully!');
        }
        
        function createBackup() {
            const backupData = {
                timestamp: new Date().toISOString(),
                roles: roles,
                permission_groups: permissionGroups,
                permission_categories: permissionCategories,
                role_permissions: rolePermissions
            };
            
            const jsonData = JSON.stringify(backupData, null, 2);
            const filename = `backup_${new Date().toISOString().slice(0, 19).replace(/:/g, '-')}.json`;
            downloadFile(jsonData, filename, 'application/json');
        }
        
        function restoreFromBackup() {
            const fileInput = document.createElement('input');
            fileInput.type = 'file';
            fileInput.accept = '.json';
            fileInput.onchange = function(e) {
                const file = e.target.files[0];
                if (file) {
                    const reader = new FileReader();
                    reader.onload = function(e) {
                        try {
                            const backupData = JSON.parse(e.target.result);
                            
                            if (confirm('Are you sure you want to restore from this backup? This will overwrite current data!')) {
                                roles = backupData.roles || [];
                                permissionGroups = backupData.permission_groups || [];
                                permissionCategories = backupData.permission_categories || [];
                                rolePermissions = backupData.role_permissions || [];
                                
                                // Refresh all displays
                                loadRoles();
                                loadPermissionGroups();
                                loadPermissionStructure();
                                loadRoleSelection();
                                
                                showAlert('Backup restored successfully!');
                            }
                        } catch (error) {
                            showAlert('Error restoring backup: ' + error.message, 'danger');
                        }
                    };
                    reader.readAsText(file);
                }
            };
            fileInput.click();
        }
        
        function editPermissionGroup(groupId) {
            showAlert('Edit permission group feature coming soon!', 'info');
        }
        
        function deletePermissionGroup(groupId) {
            const group = permissionGroups.find(g => g.id === groupId);
            const hasCategories = permissionCategories.some(c => c.perm_group_id === groupId);
            
            if (hasCategories) {
                showAlert('Cannot delete group that contains permissions!', 'danger');
                return;
            }
            
            if (confirm(`Are you sure you want to delete the permission group "${group.name}"?`)) {
                permissionGroups = permissionGroups.filter(g => g.id !== groupId);
                showAlert(`Permission group "${group.name}" deleted successfully!`);
                loadPermissionGroups();
                loadPermissionStructure();
            }
        }
        
        function editPermission(categoryId) {
            showAlert('Edit permission feature coming soon!', 'info');
        }
        
        function deletePermission(categoryId) {
            const category = permissionCategories.find(c => c.id === categoryId);
            const hasAssignments = rolePermissions.some(rp => rp.perm_cat_id === categoryId);
            
            if (hasAssignments) {
                if (!confirm(`This permission is assigned to roles. Delete anyway? This will remove all assignments.`)) {
                    return;
                }
                // Remove role assignments
                rolePermissions = rolePermissions.filter(rp => rp.perm_cat_id !== categoryId);
            }
            
            if (confirm(`Are you sure you want to delete the permission "${category.name}"?`)) {
                permissionCategories = permissionCategories.filter(c => c.id !== categoryId);
                showAlert(`Permission "${category.name}" deleted successfully!`);
                loadPermissionStructure();
                loadRoleSelection();
            }
        }

        // Generate SQL for CodeIgniter Integration
        function generateCodeIgniterSQL() {
            let sql = "-- CodeIgniter HMS Roles & Permissions Integration\n";
            sql += "-- Copy and execute these SQL statements in your database\n\n";
            
            // Add new roles
            const customRoles = roles.filter(r => !r.is_system);
            if (customRoles.length > 0) {
                sql += "-- Insert New Custom Roles\n";
                customRoles.forEach(role => {
                    sql += `INSERT INTO roles (name, slug, is_active, is_system, is_superadmin) VALUES ('${role.name}', '${role.slug}', ${role.is_active}, 0, 0);\n`;
                });
                sql += "\n";
            }
            
            // Add new permission groups
            const customGroups = permissionGroups.filter(g => g.id > 20); // Assuming system groups have lower IDs
            if (customGroups.length > 0) {
                sql += "-- Insert New Permission Groups\n";
                customGroups.forEach(group => {
                    sql += `INSERT INTO permission_group (name, short_code, is_active, system, sort_order) VALUES ('${group.name}', '${group.short_code}', 1, 0, ${group.sort_order});\n`;
                });
                sql += "\n";
            }
            
            // Add new permission categories
            const customCategories = permissionCategories.filter(c => c.id > 50); // Assuming system categories have lower IDs
            if (customCategories.length > 0) {
                sql += "-- Insert New Permission Categories\n";
                customCategories.forEach(category => {
                    sql += `INSERT INTO permission_category (perm_group_id, name, short_code, enable_view, enable_add, enable_edit, enable_delete) VALUES ((SELECT id FROM permission_group WHERE short_code = '${permissionGroups.find(g => g.id === category.perm_group_id)?.short_code}'), '${category.name}', '${category.short_code}', ${category.enable_view}, ${category.enable_add}, ${category.enable_edit}, ${category.enable_delete});\n`;
                });
                sql += "\n";
            }
            
            // Add role permissions
            if (rolePermissions.length > 0) {
                sql += "-- Insert Role Permissions (Update as needed)\n";
                rolePermissions.forEach(rp => {
                    const role = roles.find(r => r.id === rp.role_id);
                    const category = permissionCategories.find(c => c.id === rp.perm_cat_id);
                    if (role && category) {
                        sql += `-- ${role.name} -> ${category.name}\n`;
                        sql += `INSERT INTO roles_permissions (role_id, perm_cat_id, can_view, can_add, can_edit, can_delete) VALUES ((SELECT id FROM roles WHERE name = '${role.name}'), (SELECT id FROM permission_category WHERE short_code = '${category.short_code}'), ${rp.can_view}, ${rp.can_add}, ${rp.can_edit}, ${rp.can_delete});\n`;
                    }
                });
            }
            
            downloadFile(sql, 'codeigniter_integration.sql', 'text/plain');
            showAlert('CodeIgniter integration SQL generated!');
        }

        // Add button to generate CodeIgniter SQL
        document.addEventListener('DOMContentLoaded', function() {
            // Add CodeIgniter integration button to export tab
            setTimeout(() => {
                const exportSection = document.querySelector('#export .col-md-6:first-child .form-section');
                if (exportSection) {
                    const ciButton = document.createElement('button');
                    ciButton.className = 'btn btn-info w-100 mt-3';
                    ciButton.innerHTML = '<i class="fas fa-code"></i> Generate CodeIgniter SQL';
                    ciButton.onclick = generateCodeIgniterSQL;
                    exportSection.appendChild(ciButton);
                }
            }, 100);
        });
    </script>
</body>
</html>