<!-- application/views/admin/radiology/reports.php -->
<!DOCTYPE html>
<html>
<head>
    <meta charset="utf-8">
    <title><?php echo $title; ?> - Radiology Reports</title>
    <style>
        .content-wrapper {
            background: #f8f9fa;
            min-height: 100vh;
            padding: 20px;
        }
        
        .page-header {
            background: white;
            border-bottom: 3px solid #17a2b8;
            padding: 25px 30px;
            margin-bottom: 30px;
            border-radius: 8px;
            box-shadow: 0 2px 8px rgba(0,0,0,0.1);
        }
        
        .page-title {
            color: #2c3e50;
            margin: 0;
            font-size: 32px;
            font-weight: 700;
        }
        
        .page-subtitle {
            color: #6c757d;
            margin: 8px 0 0 0;
            font-size: 16px;
        }
        
        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 20px;
            margin-bottom: 30px;
        }
        
        .stat-card {
            background: white;
            border-radius: 12px;
            padding: 25px;
            box-shadow: 0 4px 12px rgba(0,0,0,0.1);
            border-left: 4px solid;
            transition: transform 0.3s ease;
        }
        
        .stat-card:hover {
            transform: translateY(-2px);
        }
        
        .stat-card.blue { border-left-color: #007bff; }
        .stat-card.green { border-left-color: #28a745; }
        .stat-card.yellow { border-left-color: #ffc107; }
        .stat-card.red { border-left-color: #dc3545; }
        
        .stat-number {
            font-size: 36px;
            font-weight: 700;
            color: #2c3e50;
            margin: 0;
        }
        
        .stat-label {
            color: #6c757d;
            font-size: 14px;
            font-weight: 500;
            text-transform: uppercase;
            letter-spacing: 0.5px;
            margin-top: 5px;
        }
        
        .filter-panel {
            background: white;
            border-radius: 12px;
            padding: 20px;
            margin-bottom: 20px;
            box-shadow: 0 4px 12px rgba(0,0,0,0.1);
        }
        
        .filter-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 15px;
            align-items: end;
        }
        
        .form-group {
            display: flex;
            flex-direction: column;
        }
        
        .form-label {
            margin-bottom: 5px;
            font-weight: 600;
            color: #495057;
        }
        
        .form-control {
            padding: 10px 12px;
            border: 2px solid #e9ecef;
            border-radius: 6px;
            font-size: 14px;
        }
        
        .form-control:focus {
            border-color: #007bff;
            outline: none;
            box-shadow: 0 0 0 0.2rem rgba(0,123,255,0.25);
        }
        
        .btn {
            padding: 10px 20px;
            border: none;
            border-radius: 6px;
            cursor: pointer;
            font-weight: 600;
            text-decoration: none;
            display: inline-block;
            text-align: center;
            transition: all 0.3s ease;
        }
        
        .btn-primary { background: #007bff; color: white; }
        .btn-success { background: #28a745; color: white; }
        .btn-secondary { background: #6c757d; color: white; }
        
        .btn:hover { transform: translateY(-1px); }
        
        .content-panel {
            background: white;
            border-radius: 12px;
            box-shadow: 0 4px 12px rgba(0,0,0,0.1);
            overflow: hidden;
            margin-bottom: 20px;
        }
        
        .panel-header {
            background: #f8f9fa;
            border-bottom: 2px solid #dee2e6;
            padding: 20px 25px;
        }
        
        .panel-title {
            color: #495057;
            margin: 0;
            font-size: 18px;
            font-weight: 600;
        }
        
        .panel-body {
            padding: 25px;
        }
        
        .chart-container {
            position: relative;
            height: 300px;
            margin-bottom: 20px;
        }
        
        .table {
            width: 100%;
            border-collapse: collapse;
            margin-top: 15px;
        }
        
        .table th,
        .table td {
            padding: 12px 15px;
            text-align: left;
            border-bottom: 1px solid #e9ecef;
        }
        
        .table th {
            background: #f8f9fa;
            font-weight: 600;
            color: #495057;
        }
        
        .table tr:hover {
            background: #f8f9fa;
        }
        
        .text-right { text-align: right; }
        .text-center { text-align: center; }
        
        .badge {
            padding: 4px 10px;
            border-radius: 15px;
            font-size: 11px;
            font-weight: 600;
            text-transform: uppercase;
        }
        
        .badge-success { background: #28a745; color: white; }
        .badge-warning { background: #ffc107; color: white; }
        .badge-danger { background: #dc3545; color: white; }
        
        .empty-state {
            text-align: center;
            padding: 60px 20px;
            color: #6c757d;
        }
        
        .empty-state i {
            font-size: 64px;
            margin-bottom: 20px;
            opacity: 0.5;
        }
        
        .quick-ranges {
            display: flex;
            gap: 8px;
            margin-top: 10px;
            flex-wrap: wrap;
        }
        
        .quick-range-btn {
            background: #f8f9fa;
            border: 1px solid #dee2e6;
            padding: 6px 12px;
            border-radius: 4px;
            cursor: pointer;
            font-size: 12px;
            transition: all 0.3s ease;
        }
        
        .quick-range-btn:hover {
            background: #e9ecef;
        }
        
        @media (max-width: 768px) {
            .content-wrapper { padding: 10px; }
            .stats-grid { grid-template-columns: 1fr 1fr; }
            .filter-grid { grid-template-columns: 1fr; }
            .page-title { font-size: 24px; }
        }
    </style>
</head>
<body>
    <div class="content-wrapper">
        <!-- Page Header -->
        <div class="page-header">
            <h1 class="page-title">
                <i class="fa fa-bar-chart"></i> Radiology Reports & Analytics
            </h1>
            <p class="page-subtitle">Comprehensive reporting and performance analytics</p>
        </div>

        <!-- Date Range Filter -->
        <div class="filter-panel">
            <div class="filter-grid">
                <div class="form-group">
                    <label class="form-label">Start Date:</label>
                    <input type="date" id="startDate" class="form-control" value="<?php echo $start_date; ?>">
                </div>
                
                <div class="form-group">
                    <label class="form-label">End Date:</label>
                    <input type="date" id="endDate" class="form-control" value="<?php echo $end_date; ?>">
                </div>
                
                <div class="form-group">
                    <label class="form-label">Report Type:</label>
                    <select id="reportType" class="form-control">
                        <option value="summary">Summary Report</option>
                        <option value="detailed">Detailed Report</option>
                        <option value="financial">Financial Report</option>
                        <option value="productivity">Productivity Report</option>
                    </select>
                </div>
                
                <div class="form-group">
                    <button onclick="generateReports()" class="btn btn-primary">
                        <i class="fa fa-refresh"></i> Generate Reports
                    </button>
                </div>
                
                <div class="form-group">
                    <button onclick="exportReports()" class="btn btn-success">
                        <i class="fa fa-download"></i> Export PDF
                    </button>
                </div>
            </div>
            
            <!-- Quick Date Ranges -->
            <div class="quick-ranges">
                <span style="font-weight: 600; margin-right: 10px;">Quick Ranges:</span>
                <button class="quick-range-btn" onclick="setDateRange('today')">Today</button>
                <button class="quick-range-btn" onclick="setDateRange('week')">Last Week</button>
                <button class="quick-range-btn" onclick="setDateRange('month')">Last Month</button>
                <button class="quick-range-btn" onclick="setDateRange('quarter')">Last Quarter</button>
                <button class="quick-range-btn" onclick="setDateRange('year')">Last Year</button>
            </div>
        </div>

        <!-- Summary Statistics -->
        <div class="stats-grid">
            <?php
            $total_orders = isset($revenue_report) ? array_sum(array_column($revenue_report, 'order_count')) : 0;
            $total_revenue = isset($revenue_report) ? array_sum(array_column($revenue_report, 'total_revenue')) : 0;
            $avg_order_value = $total_orders > 0 ? $total_revenue / $total_orders : 0;
            $productivity_count = isset($productivity_report) ? count($productivity_report) : 0;
            ?>
            
            <div class="stat-card blue">
                <h3 class="stat-number"><?php echo number_format($total_orders); ?></h3>
                <p class="stat-label">Total Orders</p>
            </div>
            
            <div class="stat-card green">
                <h3 class="stat-number">KES <?php echo number_format($total_revenue, 0); ?></h3>
                <p class="stat-label">Total Revenue</p>
            </div>
            
            <div class="stat-card yellow">
                <h3 class="stat-number">KES <?php echo number_format($avg_order_value, 0); ?></h3>
                <p class="stat-label">Avg Order Value</p>
            </div>
            
            <div class="stat-card red">
                <h3 class="stat-number"><?php echo $productivity_count; ?></h3>
                <p class="stat-label">Active Staff</p>
            </div>
        </div>

        <!-- Charts Section -->
        <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 20px; margin-bottom: 30px;">
            <!-- Revenue Chart -->
            <div class="content-panel">
                <div class="panel-header">
                    <h3 class="panel-title">
                        <i class="fa fa-line-chart"></i> Daily Revenue Trend
                    </h3>
                </div>
                <div class="panel-body">
                    <div class="chart-container">
                        <canvas id="revenueChart"></canvas>
                    </div>
                </div>
            </div>
            
            <!-- Equipment Usage Chart -->
            <div class="content-panel">
                <div class="panel-header">
                    <h3 class="panel-title">
                        <i class="fa fa-pie-chart"></i> Equipment Usage
                    </h3>
                </div>
                <div class="panel-body">
                    <div class="chart-container">
                        <canvas id="equipmentChart"></canvas>
                    </div>
                </div>
            </div>
        </div>

        <!-- Revenue Report Table -->
        <div class="content-panel">
            <div class="panel-header">
                <h3 class="panel-title">
                    <i class="fa fa-table"></i> Daily Revenue Report
                </h3>
            </div>
            <div class="panel-body">
                <?php if (empty($revenue_report)): ?>
                    <div class="empty-state">
                        <i class="fa fa-chart-line"></i>
                        <h5>No Revenue Data</h5>
                        <p>No revenue data available for the selected period.</p>
                    </div>
                <?php else: ?>
                    <table class="table">
                        <thead>
                            <tr>
                                <th>Date</th>
                                <th class="text-center">Orders</th>
                                <th class="text-right">Revenue</th>
                                <th class="text-right">Avg Order Value</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($revenue_report as $row): ?>
                                <tr>
                                    <td style="font-weight: 600;">
                                        <?php echo date('M j, Y', strtotime($row['date'])); ?>
                                    </td>
                                    <td class="text-center">
                                        <?php echo number_format($row['order_count']); ?>
                                    </td>
                                    <td class="text-right" style="color: #28a745; font-weight: 600;">
                                        KES <?php echo number_format($row['total_revenue'], 2); ?>
                                    </td>
                                    <td class="text-right">
                                        KES <?php echo number_format($row['avg_order_value'], 2); ?>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                <?php endif; ?>
            </div>
        </div>

        <!-- Productivity Report -->
        <div class="content-panel">
            <div class="panel-header">
                <h3 class="panel-title">
                    <i class="fa fa-users"></i> Staff Productivity Report
                </h3>
            </div>
            <div class="panel-body">
                <?php if (empty($productivity_report)): ?>
                    <div class="empty-state">
                        <i class="fa fa-users"></i>
                        <h5>No Productivity Data</h5>
                        <p>No staff productivity data available for the selected period.</p>
                    </div>
                <?php else: ?>
                    <table class="table">
                        <thead>
                            <tr>
                                <th>Staff Member</th>
                                <th class="text-center">Orders Handled</th>
                                <th class="text-right">Revenue Generated</th>
                                <th class="text-center">Performance</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($productivity_report as $row): ?>
                                <?php 
                                $performance_score = $row['orders_handled'] * 10;
                                if ($performance_score >= 100) {
                                    $performance_class = 'badge-success';
                                    $performance_text = 'Excellent';
                                } elseif ($performance_score >= 50) {
                                    $performance_class = 'badge-warning';
                                    $performance_text = 'Good';
                                } else {
                                    $performance_class = 'badge-danger';
                                    $performance_text = 'Needs Improvement';
                                }
                                ?>
                                <tr>
                                    <td style="font-weight: 600;">
                                        <?php echo $row['staff_name'] ?: 'Unknown Staff'; ?>
                                    </td>
                                    <td class="text-center">
                                        <?php echo number_format($row['orders_handled']); ?>
                                    </td>
                                    <td class="text-right" style="color: #28a745; font-weight: 600;">
                                        KES <?php echo number_format($row['revenue_generated'], 2); ?>
                                    </td>
                                    <td class="text-center">
                                        <span class="badge <?php echo $performance_class; ?>">
                                            <?php echo $performance_text; ?>
                                        </span>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                <?php endif; ?>
            </div>
        </div>

        <!-- Equipment Usage Report -->
        <div class="content-panel">
            <div class="panel-header">
                <h3 class="panel-title">
                    <i class="fa fa-cogs"></i> Equipment Utilization Report
                </h3>
            </div>
            <div class="panel-body">
                <?php if (empty($equipment_usage)): ?>
                    <div class="empty-state">
                        <i class="fa fa-cogs"></i>
                        <h5>No Equipment Data</h5>
                        <p>No equipment usage data available.</p>
                    </div>
                <?php else: ?>
                    <table class="table">
                        <thead>
                            <tr>
                                <th>Equipment</th>
                                <th class="text-center">Usage Count</th>
                                <th class="text-right">Revenue</th>
                                <th class="text-center">Utilization Rate</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($equipment_usage as $equipment): ?>
                                <?php 
                                $utilization_color = $equipment['utilization_rate'] >= 70 ? '#28a745' : 
                                                   ($equipment['utilization_rate'] >= 40 ? '#ffc107' : '#dc3545');
                                ?>
                                <tr>
                                    <td style="font-weight: 600;">
                                        <?php echo $equipment['equipment_name']; ?>
                                    </td>
                                    <td class="text-center">
                                        <?php echo number_format($equipment['usage_count']); ?>
                                    </td>
                                    <td class="text-right" style="color: #28a745; font-weight: 600;">
                                        KES <?php echo number_format($equipment['total_revenue'], 2); ?>
                                    </td>
                                    <td class="text-center">
                                        <div style="display: flex; align-items: center; justify-content: center;">
                                            <div style="width: 60px; height: 8px; background: #e9ecef; border-radius: 4px; margin-right: 10px;">
                                                <div style="width: <?php echo $equipment['utilization_rate']; ?>%; height: 100%; background: <?php echo $utilization_color; ?>; border-radius: 4px;"></div>
                                            </div>
                                            <span style="font-weight: 600; color: <?php echo $utilization_color; ?>;">
                                                <?php echo $equipment['utilization_rate']; ?>%
                                            </span>
                                        </div>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                <?php endif; ?>
            </div>
        </div>
    </div>

    <!-- Include Chart.js -->
    <script src="https://cdnjs.cloudflare.com/ajax/libs/Chart.js/3.9.1/chart.min.js"></script>
    <script>
        // Initialize charts when document is ready
        document.addEventListener('DOMContentLoaded', function() {
            initializeCharts();
        });

        function initializeCharts() {
            // Revenue Chart
            const revenueData = <?php echo json_encode($revenue_report); ?>;
            if (revenueData && revenueData.length > 0) {
                const revenueLabels = revenueData.map(item => {
                    const date = new Date(item.date);
                    return date.toLocaleDateString('en-US', { month: 'short', day: 'numeric' });
                });
                const revenueValues = revenueData.map(item => parseFloat(item.total_revenue));
                
                const revenueCtx = document.getElementById('revenueChart').getContext('2d');
                new Chart(revenueCtx, {
                    type: 'line',
                    data: {
                        labels: revenueLabels,
                        datasets: [{
                            label: 'Daily Revenue (KES)',
                            data: revenueValues,
                            borderColor: '#007bff',
                            backgroundColor: 'rgba(0, 123, 255, 0.1)',
                            borderWidth: 3,
                            fill: true,
                            tension: 0.4
                        }]
                    },
                    options: {
                        responsive: true,
                        maintainAspectRatio: false,
                        plugins: {
                            legend: {
                                display: false
                            }
                        },
                        scales: {
                            y: {
                                beginAtZero: true,
                                ticks: {
                                    callback: function(value) {
                                        return 'KES ' + value.toLocaleString();
                                    }
                                }
                            }
                        }
                    }
                });
            }
            
            // Equipment Usage Chart
            const equipmentData = <?php echo json_encode($equipment_usage); ?>;
            if (equipmentData && equipmentData.length > 0) {
                const equipmentLabels = equipmentData.map(item => item.equipment_name);
                const equipmentValues = equipmentData.map(item => item.usage_count);
                const equipmentColors = ['#007bff', '#28a745', '#ffc107', '#dc3545', '#6f42c1', '#fd7e14'];
                
                const equipmentCtx = document.getElementById('equipmentChart').getContext('2d');
                new Chart(equipmentCtx, {
                    type: 'doughnut',
                    data: {
                        labels: equipmentLabels,
                        datasets: [{
                            data: equipmentValues,
                            backgroundColor: equipmentColors.slice(0, equipmentLabels.length),
                            borderWidth: 2,
                            borderColor: '#fff'
                        }]
                    },
                    options: {
                        responsive: true,
                        maintainAspectRatio: false,
                        plugins: {
                            legend: {
                                position: 'bottom',
                                labels: {
                                    padding: 20,
                                    usePointStyle: true
                                }
                            }
                        }
                    }
                });
            }
        }

        function generateReports() {
            const startDate = document.getElementById('startDate').value;
            const endDate = document.getElementById('endDate').value;
            
            if (!startDate || !endDate) {
                alert('Please select both start and end dates');
                return;
            }
            
            if (startDate > endDate) {
                alert('Start date cannot be after end date');
                return;
            }
            
            const url = '<?php echo base_url(); ?>admin/radiology/reports?start_date=' + startDate + '&end_date=' + endDate;
            window.location.href = url;
        }

        function exportReports() {
            const startDate = document.getElementById('startDate').value;
            const endDate = document.getElementById('endDate').value;
            const reportType = document.getElementById('reportType').value;
            
            // For now, print the page - you can implement PDF export later
            window.print();
        }

        function setDateRange(range) {
            const endDate = new Date();
            let startDate = new Date();
            
            switch(range) {
                case 'today':
                    startDate = new Date();
                    break;
                case 'week':
                    startDate.setDate(endDate.getDate() - 7);
                    break;
                case 'month':
                    startDate.setMonth(endDate.getMonth() - 1);
                    break;
                case 'quarter':
                    startDate.setMonth(endDate.getMonth() - 3);
                    break;
                case 'year':
                    startDate.setFullYear(endDate.getFullYear() - 1);
                    break;
            }
            
            document.getElementById('startDate').value = startDate.toISOString().split('T')[0];
            document.getElementById('endDate').value = endDate.toISOString().split('T')[0];
            
            generateReports();
        }

        // Keyboard shortcuts
        document.addEventListener('keydown', function(e) {
            if (e.ctrlKey) {
                switch(e.which) {
                    case 82: // Ctrl+R for Refresh
                        e.preventDefault();
                        generateReports();
                        break;
                    case 80: // Ctrl+P for Print
                        e.preventDefault();
                        window.print();
                        break;
                }
            }
        });
    </script>
</body>
</html>