<!DOCTYPE html>
<html>
<head>
    <meta charset="utf-8">
    <meta http-equiv="X-UA-Compatible" content="IE=edge">
    <title><?php echo $title; ?></title>
    <style>
        /* Modern, clean design without gradients */
        .content-wrapper {
            background: #f8f9fa;
            min-height: 100vh;
        }
        
        .page-header {
            background: white;
            border-bottom: 3px solid #007bff;
            padding: 20px 30px;
            margin-bottom: 30px;
            border-radius: 8px;
            box-shadow: 0 2px 8px rgba(0,0,0,0.1);
        }
        
        .page-title {
            color: #2c3e50;
            margin: 0;
            font-size: 28px;
            font-weight: 600;
        }
        
        .page-subtitle {
            color: #6c757d;
            margin: 5px 0 0 0;
            font-size: 16px;
        }
        
        .verification-panel {
            background: white;
            border-radius: 12px;
            box-shadow: 0 4px 12px rgba(0,0,0,0.1);
            margin-bottom: 25px;
            overflow: hidden;
            border: 1px solid #e9ecef;
        }
        
        .panel-header {
            background: #f8f9fa;
            border-bottom: 2px solid #dee2e6;
            padding: 20px 25px;
        }
        
        .panel-title {
            color: #495057;
            margin: 0;
            font-size: 18px;
            font-weight: 600;
            display: flex;
            align-items: center;
            gap: 10px;
        }
        
        .panel-body {
            padding: 25px;
        }
        
        .status-indicator {
            padding: 8px 16px;
            border-radius: 20px;
            font-size: 13px;
            font-weight: 600;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }
        
        .status-pending {
            background: #fff3cd;
            color: #856404;
            border: 1px solid #ffeaa7;
        }
        
        .status-verified {
            background: #d4edda;
            color: #155724;
            border: 1px solid #c3e6cb;
        }
        
        .status-failed {
            background: #f8d7da;
            color: #721c24;
            border: 1px solid #f5c6cb;
        }
        
        .patient-search {
            background: white;
            border: 2px solid #e9ecef;
            border-radius: 8px;
            padding: 15px;
            margin-bottom: 20px;
        }
        
        .patient-card {
            background: white;
            border: 2px solid #e9ecef;
            border-radius: 10px;
            padding: 20px;
            margin-bottom: 15px;
            transition: all 0.3s ease;
            cursor: pointer;
        }
        
        .patient-card:hover {
            border-color: #007bff;
            box-shadow: 0 4px 12px rgba(0,123,255,0.15);
        }
        
        .patient-card.selected {
            border-color: #28a745;
            background: #f8fff9;
        }
        
        .patient-info {
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        .patient-details h5 {
            margin: 0 0 5px 0;
            color: #2c3e50;
            font-weight: 600;
        }
        
        .patient-meta {
            color: #6c757d;
            font-size: 14px;
        }
        
        .fingerprint-container {
            text-align: center;
            padding: 20px;
            background: #f8f9fa;
            border-radius: 8px;
            border: 2px dashed #dee2e6;
        }
        
        .fingerprint-image {
            max-width: 200px;
            max-height: 150px;
            border-radius: 8px;
            box-shadow: 0 2px 8px rgba(0,0,0,0.1);
        }
        
        .secugen-status {
            background: #e9ecef;
            border: 1px solid #ced4da;
            border-radius: 6px;
            padding: 15px;
            margin: 15px 0;
        }
        
        .secugen-active {
            background: #d4edda;
            border-color: #c3e6cb;
            color: #155724;
        }
        
        .secugen-inactive {
            background: #f8d7da;
            border-color: #f5c6cb;
            color: #721c24;
        }
        
        .btn-verification {
            background: #007bff;
            border: none;
            color: white;
            padding: 12px 24px;
            border-radius: 6px;
            font-weight: 600;
            transition: all 0.3s ease;
        }
        
        .btn-verification:hover {
            background: #0056b3;
            transform: translateY(-1px);
        }
        
        .btn-manual {
            background: #6c757d;
            border: none;
            color: white;
            padding: 8px 16px;
            border-radius: 6px;
            font-weight: 500;
        }
        
        .btn-manual:hover {
            background: #545b62;
        }
        
        .workflow-status {
            display: inline-block;
            padding: 6px 12px;
            border-radius: 4px;
            font-size: 12px;
            font-weight: 600;
            text-transform: uppercase;
        }
        
        .workflow-gcc {
            background: #e3f2fd;
            color: #1976d2;
            border: 1px solid #bbdefb;
        }
        
        .workflow-general {
            background: #f3e5f5;
            color: #7b1fa2;
            border: 1px solid #e1bee7;
        }
        
        .verification-history {
            max-height: 300px;
            overflow-y: auto;
            border: 1px solid #dee2e6;
            border-radius: 6px;
            padding: 10px;
        }
        
        .history-item {
            padding: 8px 0;
            border-bottom: 1px solid #f1f3f4;
        }
        
        .history-item:last-child {
            border-bottom: none;
        }
        
        .form-control {
            border: 2px solid #e9ecef;
            border-radius: 6px;
            padding: 10px 15px;
            transition: border-color 0.3s ease;
        }
        
        .form-control:focus {
            border-color: #007bff;
            box-shadow: 0 0 0 0.2rem rgba(0,123,255,0.25);
            outline: none;
        }
        
        .alert-custom {
            border: none;
            border-radius: 8px;
            padding: 15px 20px;
            margin: 15px 0;
        }
        
        .alert-info {
            background: #e3f2fd;
            color: #1976d2;
            border-left: 4px solid #2196f3;
        }
        
        .alert-success {
            background: #e8f5e8;
            color: #2e7d32;
            border-left: 4px solid #4caf50;
        }
        
        .alert-warning {
            background: #fff8e1;
            color: #ef6c00;
            border-left: 4px solid #ff9800;
        }
        
        .alert-danger {
            background: #ffebee;
            color: #c62828;
            border-left: 4px solid #f44336;
        }
        
        .diagnostic-log {
            background: #2d3748;
            color: #e2e8f0;
            border-radius: 6px;
            padding: 15px;
            font-family: 'Courier New', monospace;
            font-size: 12px;
            max-height: 200px;
            overflow-y: auto;
            margin-top: 15px;
        }
        
        .log-success { color: #68d391; }
        .log-error { color: #fc8181; }
        .log-warning { color: #f6e05e; }
        .log-info { color: #63b3ed; }
        
        .spinner {
            display: inline-block;
            width: 20px;
            height: 20px;
            border: 2px solid #f3f3f3;
            border-top: 2px solid #007bff;
            border-radius: 50%;
            animation: spin 1s linear infinite;
        }
        
        @keyframes spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }
        
        .step-indicator {
            display: flex;
            justify-content: space-between;
            margin: 20px 0;
        }
        
        .step {
            flex: 1;
            text-align: center;
            position: relative;
        }
        
        .step:not(:last-child)::after {
            content: '';
            position: absolute;
            top: 15px;
            right: -50%;
            width: 100%;
            height: 2px;
            background: #dee2e6;
            z-index: 1;
        }
        
        .step.active::after {
            background: #007bff;
        }
        
        .step-circle {
            width: 30px;
            height: 30px;
            border-radius: 50%;
            background: #dee2e6;
            color: white;
            display: flex;
            align-items: center;
            justify-content: center;
            margin: 0 auto 5px;
            font-weight: 600;
            position: relative;
            z-index: 2;
        }
        
        .step.active .step-circle {
            background: #007bff;
        }
        
        .step.completed .step-circle {
            background: #28a745;
        }
        
        .cost-summary-card {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            border-radius: 12px;
            padding: 20px;
            color: white;
            margin: 20px 0;
        }
        
        .cost-item {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 8px 0;
            border-bottom: 1px solid rgba(255,255,255,0.2);
        }
        
        .cost-item:last-child {
            border-bottom: none;
            font-size: 18px;
            font-weight: bold;
            padding-top: 15px;
            margin-top: 10px;
            border-top: 2px solid rgba(255,255,255,0.3);
        }
        
        .test-selection-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(280px, 1fr));
            gap: 15px;
            max-height: 400px;
            overflow-y: auto;
            padding: 15px;
            border: 1px solid #dee2e6;
            border-radius: 8px;
            background: #f8f9fa;
        }
        
        .test-item {
            background: white;
            border: 2px solid #e9ecef;
            border-radius: 8px;
            padding: 15px;
            transition: all 0.3s ease;
            cursor: pointer;
        }
        
        .test-item:hover {
            border-color: #007bff;
            box-shadow: 0 2px 8px rgba(0,123,255,0.15);
        }
        
        .test-item.selected {
            border-color: #28a745;
            background: #f8fff9;
        }
        
        .test-checkbox {
            margin-right: 10px;
        }
        
        .test-name {
            font-weight: 600;
            color: #2c3e50;
            margin-bottom: 5px;
        }
        
        .test-details {
            color: #6c757d;
            font-size: 13px;
        }
        
        .test-price {
            color: #28a745;
            font-weight: 600;
            font-size: 14px;
        }
        
        @media (max-width: 768px) {
            .content-grid {
                grid-template-columns: 1fr;
            }
            
            .test-selection-grid {
                grid-template-columns: 1fr;
            }
            
            .patient-info {
                flex-direction: column;
                align-items: flex-start;
                gap: 10px;
            }
        }
    </style>
</head>
<body>
    <div class="content-wrapper">
        <!-- Page Header -->
        <div class="page-header">
            <h1 class="page-title">
                <i class="fa fa-camera"></i> Radiology Services
            </h1>
            <p class="page-subtitle">Patient verification and imaging services management</p>
        </div>

        <!-- Step Indicator -->
        <div class="verification-panel">
            <div class="panel-body">
                <div class="step-indicator">
                    <div class="step active" id="step-search">
                        <div class="step-circle">1</div>
                        <small>Patient Search</small>
                    </div>
                    <div class="step" id="step-verify">
                        <div class="step-circle">2</div>
                        <small>Biometric Verification</small>
                    </div>
                    <div class="step" id="step-order">
                        <div class="step-circle">3</div>
                        <small>Create Order</small>
                    </div>
                    <div class="step" id="step-schedule">
                        <div class="step-circle">4</div>
                        <small>Schedule Imaging</small>
                    </div>
                </div>
            </div>
        </div>

        <!-- Patient Search Section -->
        <div class="verification-panel" id="patient-search-panel">
            <div class="panel-header">
                <h3 class="panel-title">
                    <i class="fa fa-search"></i> Patient Search & Selection
                </h3>
            </div>
            <div class="panel-body">
                <div class="patient-search">
                    <div class="row">
                        <div class="col-md-8">
                            <input type="text" 
                                   class="form-control" 
                                   id="patient_search" 
                                   placeholder="Search by name, patient ID, or mobile number...">
                        </div>
                        <div class="col-md-4">
                            <button type="button" class="btn btn-verification btn-block" onclick="searchPatients()">
                                <i class="fa fa-search"></i> Search Patients
                            </button>
                        </div>
                    </div>
                </div>
                
                <div id="patient_results" class="mt-3">
                    <div class="alert alert-custom alert-info">
                        <i class="fa fa-info-circle"></i> Enter search criteria to find patients
                    </div>
                </div>
            </div>
        </div>

        <!-- Patient Verification Section -->
        <div class="verification-panel" id="verification-panel" style="display: none;">
            <div class="panel-header">
                <h3 class="panel-title">
                    <i class="fa fa-fingerprint"></i> Biometric Patient Verification
                </h3>
            </div>
            <div class="panel-body">
                <div class="row">
                    <!-- Selected Patient Info -->
                    <div class="col-md-6">
                        <h5><i class="fa fa-user"></i> Selected Patient</h5>
                        <div id="selected_patient_info" class="patient-card selected">
                            <!-- Patient information will be loaded here -->
                        </div>
                        
                        <h5 class="mt-4"><i class="fa fa-history"></i> Verification History</h5>
                        <div id="verification_history" class="verification-history">
                            <div class="text-center text-muted">
                                <i class="fa fa-history"></i><br>
                                <small>Loading verification history...</small>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Fingerprint Verification -->
                    <div class="col-md-6">
                        <h5><i class="fa fa-fingerprint"></i> Fingerprint Verification</h5>
                        
                        <!-- SecuGen Status -->
                        <div id="secugen_status" class="secugen-status">
                            <div class="text-center">
                                <div class="spinner"></div> Initializing SecuGen WebAPI...
                            </div>
                        </div>
                        
                        <!-- Fingerprint Capture -->
                        <div class="fingerprint-container">
                            <img id="live_fingerprint_display" 
                                 src="<?php echo base_url(); ?>Images/PlaceFinger.bmp" 
                                 alt="Live Fingerprint" 
                                 class="fingerprint-image">
                            <div id="fingerprint_status" class="mt-3">
                                <small class="text-muted">Place finger on scanner</small>
                            </div>
                        </div>
                        
                        <!-- Verification Controls -->
                        <div class="text-center mt-3">
                            <button type="button" 
                                    class="btn btn-verification mr-2" 
                                    id="capture_fingerprint" 
                                    onclick="captureFingerprint()">
                                <i class="fa fa-hand-paper-o"></i> Capture Fingerprint
                            </button>
                            <button type="button" 
                                    class="btn btn-verification mr-2" 
                                    id="verify_patient" 
                                    onclick="verifyPatient()" 
                                    disabled>
                                <i class="fa fa-check-circle"></i> Verify Patient
                            </button>
                            <button type="button" 
                                    class="btn btn-manual" 
                                    onclick="manualVerification()">
                                <i class="fa fa-user-check"></i> Manual Override
                            </button>
                        </div>
                        
                        <!-- Verification Result -->
                        <div id="verification_result" class="mt-3">
                            <!-- Verification results will be shown here -->
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Order Creation Section -->
        <div class="verification-panel" id="order-panel" style="display: none;">
            <div class="panel-header">
                <h3 class="panel-title">
                    <i class="fa fa-plus-circle"></i> Create Radiology Order
                </h3>
            </div>
            <div class="panel-body">
                <form id="radiology_order_form">
                    <input type="hidden" id="verified_patient_id" name="patient_id">
                    <input type="hidden" id="live_template" name="live_template">
                    
                    <div class="row">
                        <div class="col-md-6">
                            <div class="form-group">
                                <label>Referring Doctor <span class="text-danger">*</span></label>
                                <select name="referring_doctor" id="referringDoctor" class="form-control" required>
                                    <option value="">Select Doctor</option>
                                    <?php if(isset($radiology_staff)): ?>
                                        <?php foreach($radiology_staff as $doctor): ?>
                                            <option value="<?php echo $doctor['id']; ?>">
                                                Dr. <?php echo $doctor['name'] . ' ' . $doctor['surname']; ?>
                                            </option>
                                        <?php endforeach; ?>
                                    <?php endif; ?>
                                </select>
                            </div>
                            
                            <div class="form-group">
                                <label>Priority Level <span class="text-danger">*</span></label>
                                <select class="form-control" id="priority" name="priority" required>
                                    <option value="routine">Routine</option>
                                    <option value="urgent">Urgent (+25% fee)</option>
                                    <option value="stat">STAT (+50% fee)</option>
                                </select>
                            </div>
                            
                            <div class="form-group">
                                <label>Clinical Notes</label>
                                <textarea class="form-control" 
                                          id="clinical_notes" 
                                          name="clinical_notes" 
                                          rows="3" 
                                          placeholder="Clinical indication and relevant history..."></textarea>
                            </div>
                            
                            <div class="form-group">
                                <label>Scheduled Date</label>
                                <input type="date" 
                                       class="form-control" 
                                       id="scheduled_date" 
                                       name="scheduled_date" 
                                       min="<?php echo date('Y-m-d'); ?>">
                            </div>
                            
                            <div class="form-group">
                                <label>Scheduled Time</label>
                                <input type="time" 
                                       class="form-control" 
                                       id="scheduled_time" 
                                       name="scheduled_time">
                            </div>
                        </div>
                        
                        <div class="col-md-6">
    <div class="form-group">
        <label id="radiology-tests-label">Radiology Tests <span class="text-danger">*</span></label>
        <div class="test-selection-grid" id="radiology_tests">
            <?php foreach($imaging_types as $test_id => $type): ?>
                <div class="test-item" 
                     onclick="toggleTest(<?php echo $test_id; ?>)"
                     data-is-gcc="<?php echo isset($type['is_gcc']) ? $type['is_gcc'] : 0; ?>"
                     data-test-id="<?php echo $test_id; ?>"
                     <?php if(isset($type['is_gcc']) && $type['is_gcc'] == 1): ?>
                        class="test-item gcc-test"
                     <?php else: ?>
                        class="test-item general-test"
                     <?php endif; ?>>
                    
                    <input class="test-checkbox" 
                           type="checkbox" 
                           value="<?php echo $test_id; ?>" 
                           id="test_<?php echo $test_id; ?>"
                           data-charge="<?php echo !empty($type['price']) ? $type['price'] : 0; ?>"
                           data-preparation="<?php echo htmlspecialchars($type['preparation_instructions'] ?? ''); ?>"
                           data-is-gcc="<?php echo isset($type['is_gcc']) ? $type['is_gcc'] : 0; ?>">
                    
                    <div class="test-name">
                        <?php echo $type['test_name'] ?? $type['name']; ?>
                        <?php if(isset($type['is_gcc']) && $type['is_gcc'] == 1): ?>
                            <span class="badge badge-info ml-2">GCC</span>
                        <?php endif; ?>
                    </div>
                    
                    <div class="test-details">
                        <?php if(!empty($type['description'])): ?>
                            <small class="text-muted"><?php echo $type['description']; ?></small><br>
                        <?php endif; ?>
                        
                        <?php if(!empty($type['estimated_duration'])): ?>
                            Duration: <?php echo $type['estimated_duration']; ?> minutes<br>
                        <?php elseif(!empty($type['duration'])): ?>
                            Duration: <?php echo $type['duration']; ?><br>
                        <?php endif; ?>
                        
                        <?php if(!empty($type['preparation_instructions'])): ?>
                            <small class="text-warning">Prep: <?php echo $type['preparation_instructions']; ?></small><br>
                        <?php elseif(!empty($type['preparation'])): ?>
                            <small class="text-warning">Prep: <?php echo $type['preparation']; ?></small><br>
                        <?php endif; ?>
                        
                        <?php if(isset($type['requires_fasting']) && $type['requires_fasting'] == 1): ?>
                            <small class="text-danger"><i class="fa fa-exclamation-triangle"></i> Fasting Required</small><br>
                        <?php endif; ?>
                        
                        <?php if(isset($type['requires_contrast']) && $type['requires_contrast'] == 1): ?>
                            <small class="text-warning"><i class="fa fa-tint"></i> Contrast Required</small><br>
                        <?php endif; ?>
                        
                        <?php if (!empty($type['price'])): ?>
                            <span class="test-price">KES <?php echo number_format($type['price'], 2); ?></span>
                        <?php else: ?>
                            <span class="test-price">Price: Contact for pricing</span>
                        <?php endif; ?>
                    </div>
                </div>
            <?php endforeach; ?>
        </div>
    </div>
</div>
                    </div>
                    
                    <!-- Cost Summary -->
                    <div class="cost-summary-card">
                        <h5 style="margin: 0 0 15px 0;"><i class="fa fa-calculator"></i> Cost Summary</h5>
                        <div class="cost-item">
                            <span>Base Cost:</span>
                            <span id="base_cost">KES 0.00</span>
                        </div>
                        <div class="cost-item">
                            <span>Priority Fee:</span>
                            <span id="priority_fee">KES 0.00</span>
                        </div>
                        <div class="cost-item">
                            <span>Tax (16%):</span>
                            <span id="tax_amount">KES 0.00</span>
                        </div>
                        <div class="cost-item">
                            <span>Total Amount:</span>
                            <span id="total_amount">KES 0.00</span>
                        </div>
                    </div>
                    
                    <div class="text-center">
                        <button type="button" class="btn btn-verification" onclick="createRadiologyOrder()">
                            <i class="fa fa-plus"></i> Create Radiology Order
                        </button>
                    </div>
                </form>
            </div>
        </div>

        <!-- Diagnostic Log -->
        <!--<div class="verification-panel">
            <div class="panel-header">
                <h3 class="panel-title">
                    <i class="fa fa-terminal"></i> System Diagnostics
                </h3>
            </div>
            <div class="panel-body">
                <div id="diagnostic_log" class="diagnostic-log">
                    <div class="log-info">[INFO] Radiology verification system initialized</div>
                    <div class="log-info">[INFO] SecuGen WebAPI integration ready</div>
                </div>
            </div>
        </div>-->
    </div>

    <!-- Hidden inputs for verification -->
    <input type="hidden" id="fingerprint_template" name="fingerprint_template">
    <input type="hidden" id="fingerprint_image" name="fingerprint_image">

    <!-- JavaScript -->
    <script src="<?php echo base_url(); ?>backend/bootstrap/js/bootstrap.min.js"></script>
    <script src="<?php echo base_url(); ?>backend/dist/js/adminlte.min.js"></script>
    <script>
        // CRITICAL: Set base URL before any other JavaScript
        window.base_url = '<?php echo base_url(); ?>';
        
        // Initialize radiology config with correct base URL
        window.radiologyConfig = {
            baseURL: '<?php echo base_url(); ?>',
            endpoints: {
                searchPatients: 'admin/radiology/get_patients',
                verifyPatient: 'admin/radiology/verify_patient',
                createOrder: 'admin/radiology/create_order',
                getTests: 'admin/radiology/get_tests'
            },
            costs: {
                taxRate: 0.16,
                priorityMultipliers: {
                    routine: 1.0,
                    urgent: 1.25,
                    stat: 1.50
                }
            }
        };
        
        // Debug the URL construction
        console.log('Base URL:', window.radiologyConfig.baseURL);
        console.log('Search endpoint will be:', window.radiologyConfig.baseURL + window.radiologyConfig.endpoints.searchPatients);
    </script>
<script>
/**
 * =============================================
 * RADIOLOGY PATIENT VERIFICATION SYSTEM
 * Complete Fixed Version
 * =============================================
 */

// =============================================
// GLOBAL STATE MANAGEMENT
// =============================================

window.radiologyState = {
    selectedPatient: null,
    selectedPatientData: null,
    verificationStatus: 'pending',
    secugenActive: false,
    liveTemplate: null,
    storedTemplate: null,
    matchScore: 0,
    orderCreated: false,
    currentStep: 'search',
    searchResults: [],
    selectedTests: [],
    totalCost: 0,
    verificationPassed: false,
    verificationMethod: null,
    verificationQuality: null
};

// SecuGen configuration
window.secugenConfig = {
    serviceURL: 'https://localhost:8443',
    captureEndpoint: '/SGIFPCapture',
    matchEndpoint: '/SGIMatchScore',
    infoEndpoint: '/SGIFPGetInfo',
    serviceActive: false,
    deviceConnected: false,
    lastError: null,
    retryCount: 0,
    maxRetries: 3,
    quality: 50,
    timeout: 20000
};

// =============================================
// INITIALIZATION
// =============================================

$(document).ready(function() {
    initializeRadiologySystem();
    setupEventHandlers();
    initializeSecuGenService();
    
    logDiagnostic('Radiology verification system initialized', 'info');
});

function initializeRadiologySystem() {
    logDiagnostic('Initializing radiology verification system...', 'info');
    
    // Ensure base URL is set
    if (!window.radiologyConfig.baseURL) {
        if (typeof window.base_url !== 'undefined') {
            window.radiologyConfig.baseURL = window.base_url;
        }
    }
    
    logDiagnostic('Base URL: ' + window.radiologyConfig.baseURL, 'info');
    
    // Initialize step indicator
    updateStepIndicator('search');
    
    // Reset form state
    resetVerificationForm();
    
    // Initialize cost calculation
    calculateCosts();
    
    logDiagnostic('System initialization complete', 'success');
}

function setupEventHandlers() {
    $('#patient_search').on('keypress', function(e) {
        if (e.which === 13) {
            e.preventDefault();
            searchPatients();
        }
    });

    let searchTimeout;
    $('#patient_search').on('input', function() {
        clearTimeout(searchTimeout);
        const searchTerm = $(this).val().trim();
        
        if (searchTerm.length >= 2) {
            searchTimeout = setTimeout(searchPatients, 500);
        }
    });

    $(document).on('change', '.test-checkbox', function() {
        const testItem = $(this).closest('.test-item');
        if ($(this).prop('checked')) {
            testItem.addClass('selected');
        } else {
            testItem.removeClass('selected');
        }
        calculateCosts();
    });

    $('#priority').on('change', calculateCosts);
    
    logDiagnostic('Event handlers configured', 'info');
}

// Function to filter and display radiology tests based on patient workflow
function filterRadiologyTestsByWorkflow(patientType) {
    const isGCC = patientType === 'GCC';
    
    logDiagnostic(`Filtering radiology tests for ${patientType} workflow`, 'info');
    
    // Get all test items
    $('.test-item').each(function() {
        const testItem = $(this);
        const checkbox = testItem.find('.test-checkbox');
        const testId = checkbox.val();
        
        // Get test data (you might need to add data attributes to your HTML)
        const isGCCTest = testItem.data('is-gcc') === 1 || testItem.hasClass('gcc-test');
        
        if (isGCC) {
            // GCC patients: ONLY show tests marked as GCC (is_gcc = 1)
            if (isGCCTest) {
                testItem.show();
                logDiagnostic(`Showing GCC test: ${testId}`, 'info');
            } else {
                testItem.hide();
                // Uncheck hidden tests
                checkbox.prop('checked', false);
                testItem.removeClass('selected');
            }
        } else {
            // General/Specialized patients: Show ALL tests (including GCC tests)
            testItem.show();
            logDiagnostic(`Showing all tests for ${patientType} patient`, 'info');
        }
    });
    
    // Recalculate costs after filtering
    calculateCosts();
    
    // Update the test selection header
    updateTestSelectionHeader(patientType, isGCC);
}

function updateTestSelectionHeader(patientType, isGCC) {
    const testSelectionLabel = $('label').filter(function() {
        return $(this).text().includes('Radiology Tests');
    });
    
    if (isGCC) {
        testSelectionLabel.html(`
            Radiology Tests - GCC Workflow <span class="text-danger">*</span>
            <small class="text-info d-block">Only GCC-approved tests are available for this patient</small>
        `);
    } else {
        testSelectionLabel.html(`
            Radiology Tests <span class="text-danger">*</span>
            <small class="text-muted d-block">All radiology services available</small>
        `);
    }
}

// =============================================
// SECUGEN INTEGRATION
// =============================================

function initializeSecuGenService() {
    logDiagnostic('Initializing SecuGen WebAPI service...', 'info');
    updateSecuGenStatus('checking', 'Initializing SecuGen WebAPI connection...');
    
    detectSecuGenService();
}

function detectSecuGenService() {
    const ports = [8443, 8000, 8080];
    let currentPort = 0;
    
    function tryPort() {
        if (currentPort >= ports.length) {
            handleSecuGenUnavailable('All SecuGen WebAPI ports failed - service not running');
            return;
        }
        
        const port = ports[currentPort];
        const testURL = `https://localhost:${port}/SGIFPGetInfo`;
        
        logDiagnostic(`Testing SecuGen WebAPI on port ${port}...`, 'info');
        
        $.ajax({
            url: testURL,
            type: 'GET',
            timeout: 3000,
            success: function(response) {
                handleSecuGenDetected(port, response);
            },
            error: function(xhr, status, error) {
                logDiagnostic(`Port ${port} failed: ${error}`, 'warning');
                currentPort++;
                tryPort();
            }
        });
    }
    
    tryPort();
}

function handleSecuGenDetected(port, deviceInfo) {
    window.secugenConfig.serviceURL = `https://localhost:${port}`;
    window.secugenConfig.serviceActive = true;
    window.secugenConfig.deviceConnected = true;
    window.radiologyState.secugenActive = true;
    
    let statusMessage = `SecuGen WebAPI active on port ${port}`;
    if (deviceInfo && deviceInfo.DeviceName) {
        statusMessage += ` - ${deviceInfo.DeviceName}`;
        if (deviceInfo.SerialNumber) {
            statusMessage += ` (SN: ${deviceInfo.SerialNumber})`;
        }
    }
    
    updateSecuGenStatus('active', statusMessage);
    logDiagnostic(`SecuGen service detected: ${statusMessage}`, 'success');
}

function handleSecuGenUnavailable(message) {
    window.secugenConfig.serviceActive = false;
    window.secugenConfig.deviceConnected = false;
    window.radiologyState.secugenActive = false;
    
    updateSecuGenStatus('inactive', message);
    logDiagnostic(`SecuGen unavailable: ${message}`, 'warning');
}

function updateSecuGenStatus(status, message) {
    const statusDiv = $('#secugen_status');
    statusDiv.removeClass('secugen-active secugen-inactive');
    
    switch (status) {
        case 'checking':
            statusDiv.html(`<div class="text-center"><div class="spinner"></div> ${message}</div>`);
            break;
        case 'active':
            statusDiv.addClass('secugen-active');
            statusDiv.html(`<i class="fa fa-check-circle"></i> ${message}`);
            break;
        case 'inactive':
            statusDiv.addClass('secugen-inactive');
            statusDiv.html(`<i class="fa fa-exclamation-triangle"></i> ${message}`);
            break;
    }
}

// =============================================
// PATIENT SEARCH AND SELECTION
// =============================================

function searchPatients() {
    const searchTerm = $('#patient_search').val().trim();
    
    if (searchTerm.length < 2) {
        showAlert('Please enter at least 2 characters to search', 'warning');
        return;
    }

    logDiagnostic(`Searching for patients: "${searchTerm}"`, 'info');
    
    $('#patient_results').html(`
        <div class="text-center" style="padding: 20px;">
            <div class="spinner"></div> Searching patients...
        </div>
    `);
    
    $.ajax({
        url: window.radiologyConfig.baseURL + window.radiologyConfig.endpoints.searchPatients,
        type: 'GET',
        data: { search: searchTerm },
        timeout: 10000,
        dataType: 'json',
        success: function(response) {
            window.radiologyState.searchResults = response;
            displayPatientResults(response);
            logDiagnostic(`Found ${response.length} patients`, 'success');
        },
        error: function(xhr, status, error) {
            logDiagnostic(`Patient search failed: ${status} - ${error}`, 'error');
            
            let errorMessage = 'Search failed. ';
            try {
                const errorResponse = JSON.parse(xhr.responseText);
                errorMessage += errorResponse.error || 'Unknown error occurred.';
            } catch (e) {
                if (xhr.responseText.includes('<!DOCTYPE') || xhr.responseText.includes('<html')) {
                    errorMessage += 'Server error - check controller method exists.';
                } else {
                    errorMessage += `Server error: ${error}`;
                }
            }
            
            $('#patient_results').html(`
                <div class="alert alert-custom alert-danger">
                    <i class="fa fa-times-circle"></i> ${errorMessage}
                </div>
            `);
        }
    });
}

function displayPatientResults(patients) {
    const resultsDiv = $('#patient_results');
    
    if (patients.length === 0) {
        resultsDiv.html(`
            <div class="alert alert-custom alert-warning">
                <i class="fa fa-search"></i> No patients found matching your search criteria.
            </div>
        `);
        return;
    }

    let html = '';
    patients.forEach(function(patient) {
        const workflowBadge = getWorkflowBadge(patient.patient_workflow);
        const workflowConfig = getWorkflowConfig(patient.patient_type);
        
        const fingerprintStatus = patient.has_fingerprint ? 
            '<span class="text-success"><i class="fa fa-fingerprint"></i> Available</span>' :
            '<span class="text-warning"><i class="fa fa-exclamation-triangle"></i> Missing</span>';
        
        const workflowDescription = workflowConfig.description || 'Standard workflow';

        html += `
            <div class="patient-card" onclick="selectPatient(${patient.id}, '${patient.patient_name}', '${patient.patient_unique_id}', '${patient.mobileno || ''}', ${patient.has_fingerprint}, '${patient.patient_type}')">
                <div class="patient-info">
                    <div class="patient-details">
                        <h5>${patient.patient_name} ${workflowBadge}</h5>
                        <div class="patient-meta">
                            ID: ${patient.patient_unique_id} | Mobile: ${patient.mobileno || 'N/A'} | 
                            Age: ${patient.age || 'N/A'} | Gender: ${patient.gender || 'N/A'}
                        </div>
                        <div class="patient-meta mt-1">
                            Fingerprint: ${fingerprintStatus} | Workflow: ${workflowDescription}
                        </div>
                    </div>
                    <div class="text-right">
                        <button type="button" class="btn btn-sm btn-verification">
                            <i class="fa fa-arrow-right"></i> Select
                        </button>
                    </div>
                </div>
            </div>
        `;
    });

    resultsDiv.html(html);
}

function bypassVerificationAndProceedDirectly() {
    const patientData = window.radiologyState.selectedPatientData;
    const workflowConfig = getWorkflowConfig(patientData.patient_type);
    
    // Record automatic verification bypass
    window.radiologyState.verificationStatus = 'success';
    window.radiologyState.verificationPassed = true;
    window.radiologyState.verificationMethod = 'workflow_bypass';
    
    // DON'T replace the entire message - just update the status part
    $('#general_workflow_message').find('p:last').html(`
        <strong>Status:</strong> Verification complete - order creation ready.
    `);
    
    // Ensure patient info is still visible and complete
    const workflowBadge = getWorkflowBadge(patientData.patient_type);
    const fingerprintStatus = patientData.has_fingerprint ? 
        '<span class="text-success"><i class="fa fa-fingerprint"></i> Available</span>' :
        '<span class="text-warning"><i class="fa fa-exclamation-triangle"></i> Missing</span>';

    // Re-confirm patient info is displayed properly
    $('#selected_patient_info').html(`
        <div class="patient-card selected">
            <div class="patient-info">
                <div class="patient-details">
                    <h5>${patientData.name} ${workflowBadge}</h5>
                    <div class="patient-meta">
                        ID: ${patientData.unique_id} | Mobile: ${patientData.mobile || 'N/A'}
                    </div>
                    <div class="patient-meta mt-1">
                        Fingerprint: ${fingerprintStatus} | Workflow: ${workflowConfig.description || 'Standard workflow'}
                    </div>
                </div>
            </div>
        </div>
    `);
    
    // Record the bypass in database
    recordWorkflowBypass();
    
    // Show order panel immediately WITHOUT requiring button click
    $('#order-panel').show();
    updateStepIndicator('order');
    
    // FILTER RADIOLOGY TESTS BASED ON PATIENT WORKFLOW
    filterRadiologyTestsByWorkflow(patientData.patient_type);
    
    // Scroll to order panel smoothly
    setTimeout(function() {
        $('html, body').animate({
            scrollTop: $('#order-panel').offset().top - 100
        }, 800);
    }, 100);
    
    logDiagnostic(`General workflow: ${patientData.name} verification bypassed automatically - order panel shown with filtered tests`, 'success');
}

function selectPatient(patientId, patientName, patientUniqueId, mobile, hasFingerprint, patientType) {
    // Store patient data in global state
    window.radiologyState.selectedPatient = patientId;
    window.radiologyState.selectedPatientData = {
        id: patientId,
        name: patientName,
        unique_id: patientUniqueId,
        mobile: mobile,
        has_fingerprint: hasFingerprint,
        patient_type: patientType || 'GENERAL'
    };
    
    // Get workflow configuration
    const workflowConfig = getWorkflowConfig(patientType);
    const workflowBadge = getWorkflowBadge(patientType);
    
    // Format fingerprint status
    const fingerprintStatus = hasFingerprint ? 
        '<span class="text-success"><i class="fa fa-fingerprint"></i> Available</span>' :
        '<span class="text-warning"><i class="fa fa-exclamation-triangle"></i> Missing</span>';

    // FIXED: Update selected patient info with complete HTML structure
    $('#selected_patient_info').html(`
        <div class="patient-card selected">
            <div class="patient-info">
                <div class="patient-details">
                    <h5>${patientName} ${workflowBadge}</h5>
                    <div class="patient-meta">
                        <strong>Patient ID:</strong> ${patientUniqueId}<br>
                        <strong>Mobile:</strong> ${mobile || 'N/A'}<br>
                        <strong>Workflow:</strong> ${workflowConfig.description || 'Standard workflow'}
                    </div>
                    <div class="patient-meta mt-2">
                        <strong>Fingerprint:</strong> ${fingerprintStatus}
                    </div>
                </div>
            </div>
        </div>
    `);
    
    // Set patient ID in hidden form field
    $('#verified_patient_id').val(patientId);
    
    // Show verification panel
    $('#verification-panel').show();
    updateStepIndicator('verify');
    
    // Clear any previous workflow messages
    $('#general_workflow_message').remove();
    
    // Reset to two-column layout first (in case previous patient was General)
    $('.col-md-6').has('#selected_patient_info').removeClass('col-md-12').addClass('col-md-6');
    $('.col-md-6').has('h5:contains("Fingerprint Verification")').show();
    
    // WORKFLOW-BASED VERIFICATION LOGIC
    enforceWorkflowVerification(patientType, hasFingerprint, patientId);
    
    // Load verification history and reset state
    loadVerificationHistory(patientId);
    resetVerificationState();
    
    logDiagnostic(`Patient selected: ${patientName} (${patientType} workflow)`, 'success');
    
    // Debug: Log the patient info to console
    console.log('Selected patient data:', window.radiologyState.selectedPatientData);
    console.log('Patient info HTML updated in element:', $('#selected_patient_info').html());
}

function showOrderPanelWithFilteredTests() {
    if (window.radiologyState.selectedPatientData) {
        filterRadiologyTestsByWorkflow(window.radiologyState.selectedPatientData.patient_type);
    }
    
    $('#order-panel').show();
    updateStepIndicator('order');
}


function enforceWorkflowVerification(patientType, hasFingerprint, patientId) {
    const workflowConfig = getWorkflowConfig(patientType);
    
    if (workflowConfig.requires_fingerprint) {
        // GCC workflow - ENFORCE fingerprint verification
        if (hasFingerprint) {
            loadStoredFingerprintTemplate(patientId);
            showFingerprintVerificationPanel(true);
            logDiagnostic('GCC workflow: Fingerprint verification ENFORCED', 'warning');
        } else {
            showFingerprintMissingError();
            logDiagnostic('GCC workflow: Fingerprint verification REQUIRED but missing', 'error');
        }
    } else {
        // General workflow - HIDE biometric verification panel and proceed directly
        hideBiometricVerificationPanel();
        showGeneralWorkflowBypass();
        logDiagnostic('General workflow: Biometric verification panel HIDDEN, proceeding directly to order creation', 'info');
    }
}

function hideBiometricVerificationPanel() {
    // Hide the entire biometric verification section completely
    $('.col-md-6').has('h5:contains("Fingerprint Verification")').hide();
    
    // Alternative selectors to ensure the fingerprint section is hidden
    $('.col-md-6').has('#secugen_status').hide();
    $('.col-md-6').has('.fingerprint-container').hide();
    $('.col-md-6').has('#capture_fingerprint').hide();
    
    // Hide any elements containing fingerprint-related content
    $('h5').filter(function() { 
        return $(this).text().includes('Fingerprint Verification'); 
    }).closest('.col-md-6').hide();
    
    // Make the patient info column take full width
    $('.col-md-6').has('#selected_patient_info').removeClass('col-md-6').addClass('col-md-12');
    
    // Also hide the verification result area if it exists
    $('#verification_result').hide();
    
    logDiagnostic('Biometric verification panel completely hidden for non-GCC patient', 'info');
}


function showFingerprintVerificationPanel(show = true) {
    if (show) {
        // Show fingerprint verification sections
        $('.col-md-6').has('#secugen_status').show();
        $('#secugen_status').parent().show();
        $('.fingerprint-container').parent().show();
        $('h5').filter(function() { 
            return $(this).text().includes('Fingerprint Verification'); 
        }).parent().show();
        
        // Restore the two-column layout
        $('.col-md-6').eq(0).removeClass('col-md-12').addClass('col-md-6');
        $('.col-md-6').eq(1).show();
        
        logDiagnostic('Biometric verification panel shown for GCC patient', 'info');
    } else {
        hideBiometricVerificationPanel();
    }
}

function showGeneralWorkflowBypass() {
    // Create a simple notification message for non-GCC patients
    $('#selected_patient_info').after(`
        <div id="general_workflow_message" class="alert alert-success mt-3">
            <h5><i class="fa fa-check-circle"></i> General Patient - Verification Bypassed</h5>
            <p>This patient follows the General workflow. Fingerprint verification is not required.</p>
            <p><strong>Status:</strong> Automatically verified - proceeding to order creation.</p>
        </div>
    `);
    
    // Automatically proceed to order creation without button click
    setTimeout(function() {
        bypassVerificationAndProceedDirectly();
    }, 500); // Reduced delay for faster response
}

function bypassVerificationAndProceed() {
    const patientData = window.radiologyState.selectedPatientData;
    const workflowConfig = getWorkflowConfig(patientData.patient_type);
    
    // Record automatic verification bypass
    window.radiologyState.verificationStatus = 'success';
    window.radiologyState.verificationPassed = true;
    window.radiologyState.verificationMethod = 'workflow_bypass';
    
    // Show success message
    $('#verification_result').html(`
        <div class="alert alert-custom alert-success">
            <i class="fa fa-check-circle"></i> 
            <strong>Verification Bypassed Successfully</strong><br>
            <small>Patient verified automatically (General workflow)</small>
        </div>
    `);
    
    // Record the bypass in database
    recordWorkflowBypass();
    
    // Show order panel immediately
    $('#order-panel').show();
    updateStepIndicator('order');
    
    showAlert('General patient verified automatically - proceeding to order creation', 'success');
    logDiagnostic('General workflow: Verification bypassed - order panel enabled', 'success');
}

function recordWorkflowBypass() {
    $.ajax({
        url: window.radiologyConfig.baseURL + window.radiologyConfig.endpoints.verifyPatient,
        type: 'POST',
        data: {
            patient_id: window.radiologyState.selectedPatient,
            verification_method: 'workflow_bypass',
            verification_result: 'success',
            department: 'radiology',
            action: 'record_verification',
            workflow_type: window.radiologyState.selectedPatientData.patient_type
        },
        success: function(response) {
            logDiagnostic('Workflow bypass recorded successfully', 'success');
        },
        error: function(xhr, status, error) {
            logDiagnostic('Failed to record workflow bypass', 'warning');
        }
    });
}

function showFingerprintMissingError() {
    $('#fingerprint_status').html(`
        <div class="alert alert-custom alert-danger">
            <i class="fa fa-exclamation-triangle"></i> 
            <strong>Fingerprint Verification Required</strong><br>
            <small>This GCC patient requires fingerprint verification, but no stored fingerprint template was found.</small>
            <div class="mt-2">
                <button type="button" class="btn btn-sm btn-warning" onclick="requestFingerprintRegistration()">
                    <i class="fa fa-plus"></i> Register Fingerprint First
                </button>
                <button type="button" class="btn btn-sm btn-secondary ml-2" onclick="contactAdmin()">
                    <i class="fa fa-phone"></i> Contact Administrator
                </button>
            </div>
        </div>
    `);
    
    // Disable proceeding to order panel
    $('#order-panel').hide();
    logDiagnostic('GCC patient missing fingerprint - blocking order creation', 'error');
}

function requestFingerprintRegistration() {
    showAlert('Please register this patient with fingerprint capture in the GCC Registration system first.', 'warning');
    logDiagnostic('Fingerprint registration requested for GCC patient', 'warning');
}

function contactAdmin() {
    showAlert('Please contact system administrator for GCC patient fingerprint registration.', 'info');
    logDiagnostic('Administrator contact requested for GCC patient', 'info');
}

function getWorkflowConfig(patientType) {
    const workflows = {
        'GCC': {
            requires_fingerprint: true,
            requires_photo: true,
            verification_mandatory: true,
            can_skip_verification: false,
            base_fee: 10500,
            description: 'GCC Process - Fingerprint verification REQUIRED'
        },
        'GENERAL': {
            requires_fingerprint: false,
            requires_photo: false,
            verification_mandatory: false,
            can_skip_verification: true,
            base_fee: 900,
            description: 'General Process - Fingerprint verification BYPASSED'
        },
        'SPECIALIZED': {
            requires_fingerprint: false,
            requires_photo: false,
            verification_mandatory: false,
            can_skip_verification: true,
            base_fee: 1500,
            description: 'Specialized Process - Fingerprint verification OPTIONAL'
        }
    };
    
    return workflows[patientType] || workflows['GENERAL'];
}

function loadVerificationHistory(patientId) {
    $('#verification_history').html(`
        <div class="text-center text-muted">
            <div class="spinner"></div><br>
            <small>Loading verification history...</small>
        </div>
    `);
    
    setTimeout(function() {
        $('#verification_history').html(`
            <div class="history-item">
                <small class="text-muted">
                    <i class="fa fa-info-circle"></i> No previous radiology verifications found
                </small>
            </div>
        `);
    }, 1000);
}

// =============================================
// FINGERPRINT CAPTURE AND VERIFICATION
// =============================================

function captureFingerprint() {
    if (!window.radiologyState.selectedPatientData.has_fingerprint) {
        showAlert('Patient has no stored fingerprint template. Manual verification required.', 'warning');
        manualVerification();
        return;
    }

    if (!window.radiologyState.secugenActive) {
        logDiagnostic('SecuGen service not detected - attempting manual capture', 'warning');
        manualFingerprintCapture();
        return;
    }

    logDiagnostic('Starting SecuGen fingerprint capture...', 'info');
    
    $('#fingerprint_status').html(`
        <div class="alert alert-custom alert-info">
            <div class="text-center">
                <div class="spinner"></div> <strong>Capturing fingerprint...</strong><br>
                <small>Place finger firmly on scanner and hold steady</small>
            </div>
        </div>
    `);
    
    $('#capture_fingerprint').prop('disabled', true).html('<i class="fa fa-spinner fa-spin"></i> Scanning...');
    
    window.radiologyState.liveTemplate = null;
    $('#live_template, #fingerprint_template, #fingerprint_image').val('');

    const xhr = new XMLHttpRequest();
    const captureURL = window.secugenConfig.serviceURL + window.secugenConfig.captureEndpoint;
    
    xhr.open('POST', captureURL, true);
    xhr.setRequestHeader('Content-Type', 'application/x-www-form-urlencoded');
    xhr.timeout = window.secugenConfig.timeout + 5000;
    
    xhr.onreadystatechange = function() {
        if (xhr.readyState === 4) {
            $('#capture_fingerprint').prop('disabled', false).html('<i class="fa fa-hand-paper-o"></i> Capture Fingerprint');
            
            if (xhr.status === 200) {
                try {
                    const result = JSON.parse(xhr.responseText);
                    handleCaptureResult(result);
                } catch (e) {
                    logDiagnostic('Invalid SecuGen response format', 'error');
                    showCaptureError('Invalid response from SecuGen device');
                }
            } else {
                const errorMsg = xhr.status === 0 ? 'Connection failed - trying manual capture' : `HTTP ${xhr.status}`;
                logDiagnostic(`SecuGen capture failed: ${errorMsg}`, 'error');
                
                if (xhr.status === 0) {
                    setTimeout(manualFingerprintCapture, 1000);
                } else {
                    showCaptureError(errorMsg);
                }
            }
        }
    };
    
    xhr.onerror = function() {
        $('#capture_fingerprint').prop('disabled', false).html('<i class="fa fa-hand-paper-o"></i> Capture Fingerprint');
        logDiagnostic('SecuGen connection error - trying manual capture', 'error');
        setTimeout(manualFingerprintCapture, 1000);
    };
    
    xhr.ontimeout = function() {
        $('#capture_fingerprint').prop('disabled', false).html('<i class="fa fa-hand-paper-o"></i> Capture Fingerprint');
        logDiagnostic('SecuGen capture timeout', 'warning');
        showCaptureError('Capture timeout - please place finger properly and try again');
    };

    const postData = `Quality=${window.secugenConfig.quality}&Timeout=${window.secugenConfig.timeout}&WSQCompression=2.25`;
    xhr.send(postData);
}

function handleCaptureResult(result) {
    if (result.ErrorCode === 0) {
        if (result.BMPBase64) {
            $('#live_fingerprint_display').attr('src', 'data:image/bmp;base64,' + result.BMPBase64);
            $('#fingerprint_image').val(result.BMPBase64);
            logDiagnostic(`Fingerprint image captured: ${result.BMPBase64.length} chars`, 'info');
        }
        
        if (result.TemplateBase64) {
            window.radiologyState.liveTemplate = result.TemplateBase64;
            $('#live_template').val(result.TemplateBase64);
            $('#fingerprint_template').val(result.TemplateBase64);
            logDiagnostic(`Fingerprint template captured: ${result.TemplateBase64.length} chars`, 'info');
        }
        
        const quality = result.Quality || 50;
        let deviceInfo = '';
        if (result.SerialNumber) {
            deviceInfo = ` (${result.Manufacturer || 'SecuGen'} ${result.Model || 'Device'} - SN: ${result.SerialNumber})`;
        }
        
        $('#fingerprint_status').html(`
            <div class="alert alert-custom alert-success">
                <i class="fa fa-check-circle"></i> <strong>Fingerprint captured successfully!</strong><br>
                <small>Quality: <strong>${quality}%</strong> | Template: ${result.TemplateBase64 ? result.TemplateBase64.length : 0} chars${deviceInfo}</small><br>
                <small class="text-success">Ready for biometric verification</small>
            </div>
        `);
        
        $('#verify_patient').prop('disabled', false);
        logDiagnostic(`Fingerprint capture successful - Quality: ${quality}%, Template: ${result.TemplateBase64 ? result.TemplateBase64.length : 0} chars`, 'success');
        
    } else {
        const errorMsg = getSecuGenErrorDescription(result.ErrorCode);
        showCaptureError(`${errorMsg} (Error Code: ${result.ErrorCode})`);
        logDiagnostic(`SecuGen capture error ${result.ErrorCode}: ${errorMsg}`, 'error');
    }
}

function showCaptureError(message) {
    $('#fingerprint_status').html(`
        <div class="alert alert-custom alert-danger">
            <i class="fa fa-times-circle"></i> <strong>Capture Failed:</strong><br>
            <small>${message}</small>
            <div class="mt-3">
                <button type="button" class="btn btn-sm btn-warning" onclick="captureFingerprint()">
                    <i class="fa fa-refresh"></i> Try Again
                </button>
                <button type="button" class="btn btn-sm btn-info ml-1" onclick="initializeSecuGenService()">
                    <i class="fa fa-refresh"></i> Reconnect
                </button>
                <button type="button" class="btn btn-sm btn-secondary ml-1" onclick="manualFingerprintCapture()">
                    <i class="fa fa-hand-paper-o"></i> Manual Capture
                </button>
                <button type="button" class="btn btn-sm btn-secondary ml-1" onclick="manualVerification()">
                    <i class="fa fa-user-check"></i> Manual Override
                </button>
            </div>
        </div>
    `);
}

function manualFingerprintCapture() {
    if (!window.radiologyState.selectedPatient) {
        showAlert('Please select a patient first', 'warning');
        return;
    }

    logDiagnostic('Starting manual fingerprint capture with real device...', 'warning');
    
    $('#fingerprint_status').html(`
        <div class="alert alert-custom alert-warning">
            <div class="text-center">
                <div class="spinner"></div> <strong>Manual Fingerprint Capture</strong><br>
                <small>Attempting direct device connection - place finger on scanner</small>
            </div>
        </div>
    `);
    
    $('#capture_fingerprint').prop('disabled', true).html('<i class="fa fa-spinner fa-spin"></i> Manual Scan...');
    
    window.radiologyState.liveTemplate = null;
    $('#live_template, #fingerprint_template, #fingerprint_image').val('');

    attemptManualSecuGenCapture();
}

function attemptManualSecuGenCapture() {
    const manualEndpoints = [
        'https://localhost:8443/SGIFPCapture',
        'http://localhost:8000/SGIFPCapture',
        'https://127.0.0.1:8443/SGIFPCapture',
        'http://127.0.0.1:8000/SGIFPCapture'
    ];
    
    let currentEndpoint = 0;
    
    function tryNextEndpoint() {
        if (currentEndpoint >= manualEndpoints.length) {
            showAlternativeCaptureOptions();
            return;
        }
        
        const endpoint = manualEndpoints[currentEndpoint];
        logDiagnostic(`Manual capture attempt ${currentEndpoint + 1}: ${endpoint}`, 'info');
        
        const xhr = new XMLHttpRequest();
        xhr.open('POST', endpoint, true);
        xhr.setRequestHeader('Content-Type', 'application/x-www-form-urlencoded');
        xhr.timeout = 15000;
        
        xhr.onreadystatechange = function() {
            if (xhr.readyState === 4) {
                if (xhr.status === 200) {
                    try {
                        const result = JSON.parse(xhr.responseText);
                        handleManualCaptureResult(result, endpoint);
                        return;
                    } catch (e) {
                        logDiagnostic(`Invalid response from ${endpoint}`, 'error');
                    }
                }
                
                logDiagnostic(`Manual capture failed on ${endpoint}: ${xhr.status}`, 'warning');
                currentEndpoint++;
                setTimeout(tryNextEndpoint, 1000);
            }
        };
        
        xhr.onerror = function() {
            logDiagnostic(`Connection error on ${endpoint}`, 'error');
            currentEndpoint++;
            setTimeout(tryNextEndpoint, 1000);
        };
        
        xhr.ontimeout = function() {
            logDiagnostic(`Timeout on ${endpoint}`, 'warning');
            currentEndpoint++;
            setTimeout(tryNextEndpoint, 1000);
        };

        const postData = `Quality=50&Timeout=20000&WSQCompression=2.25`;
        xhr.send(postData);
    }
    
    tryNextEndpoint();
}

function handleManualCaptureResult(result, endpoint) {
    $('#capture_fingerprint').prop('disabled', false).html('<i class="fa fa-hand-paper-o"></i> Capture Fingerprint');
    
    if (result.ErrorCode === 0) {
        if (result.BMPBase64) {
            $('#live_fingerprint_display').attr('src', 'data:image/bmp;base64,' + result.BMPBase64);
            $('#fingerprint_image').val(result.BMPBase64);
        }
        
        if (result.TemplateBase64) {
            window.radiologyState.liveTemplate = result.TemplateBase64;
            $('#live_template').val(result.TemplateBase64);
            $('#fingerprint_template').val(result.TemplateBase64);
        }
        
        const quality = result.Quality || 50;
        logDiagnostic(`Manual fingerprint captured via ${endpoint} - Quality: ${quality}%`, 'success');
        
        $('#fingerprint_status').html(`
            <div class="alert alert-custom alert-success">
                <i class="fa fa-check-circle"></i> <strong>Manual capture successful!</strong><br>
                <small>Quality: <strong>${quality}%</strong> | Endpoint: ${endpoint}</small><br>
                <small class="text-success">Ready for biometric verification</small>
            </div>
        `);
        
        $('#verify_patient').prop('disabled', false);
        
    } else {
        const errorMsg = getSecuGenErrorDescription(result.ErrorCode);
        logDiagnostic(`Manual capture error ${result.ErrorCode}: ${errorMsg}`, 'error');
        
        $('#fingerprint_status').html(`
            <div class="alert alert-custom alert-danger">
                <i class="fa fa-times-circle"></i> <strong>Manual capture failed:</strong><br>
                <small>${errorMsg} (Code: ${result.ErrorCode})</small>
                <div class="mt-3">
                    <button type="button" class="btn btn-sm btn-warning" onclick="attemptManualSecuGenCapture()">
                        <i class="fa fa-refresh"></i> Try Again
                    </button>
                    <button type="button" class="btn btn-sm btn-secondary" onclick="performStaffOverride()">
                        <i class="fa fa-user-check"></i> Staff Override
                    </button>
                </div>
            </div>
        `);
    }
}

function showAlternativeCaptureOptions() {
    $('#capture_fingerprint').prop('disabled', false).html('<i class="fa fa-hand-paper-o"></i> Capture Fingerprint');
    
    $('#fingerprint_status').html(`
        <div class="alert alert-custom alert-danger">
            <i class="fa fa-times-circle"></i> <strong>All capture methods failed</strong><br>
            <small>Could not connect to SecuGen device on any port</small>
            <div class="mt-3">
                <button type="button" class="btn btn-sm btn-info" onclick="initializeSecuGenService()">
                    <i class="fa fa-refresh"></i> Retry Connection
                </button>
                <button type="button" class="btn btn-sm btn-secondary ml-1" onclick="performStaffOverride()">
                    <i class="fa fa-user-check"></i> Staff Override
                </button>
            </div>
            <hr>
            <small class="text-muted">
                <strong>Troubleshooting:</strong><br>
                • Ensure SecuGen WebAPI is running<br>
                • Check fingerprint scanner connection<br>
                • Accept SSL certificate at <a href="https://localhost:8443/SGIFPGetInfo" target="_blank">localhost:8443</a>
            </small>
        </div>
    `);
}

function verifyPatient() {
    if (!window.radiologyState.selectedPatient) {
        showAlert('Please select a patient first', 'warning');
        return;
    }

    // For workflow-based verification (like your current system)
    const patientData = window.radiologyState.selectedPatientData;
    const workflowConfig = getWorkflowConfig(patientData.patient_type);
    
    if (workflowConfig.requires_fingerprint) {
        // GCC workflow - fingerprint verification required
        if (!window.radiologyState.liveTemplate) {
            showAlert('Please capture fingerprint first - required for GCC workflow', 'error');
            return;
        }
        
        performBiometricVerification();
        
    } else {
        // General workflow - fingerprint optional
        if (!window.radiologyState.liveTemplate) {
            showAlert('Please capture fingerprint or use manual verification', 'warning');
            return;
        }
        
        performBiometricVerification();
    }
}

function performBiometricVerification() {
    if (!window.radiologyState.selectedPatient) {
        showAlert('Please select a patient first', 'warning');
        return;
    }

    if (!window.radiologyState.liveTemplate) {
        showAlert('Please capture fingerprint first', 'warning');
        return;
    }

    if (!window.radiologyState.storedTemplate) {
        showAlert('No stored template available for comparison', 'warning');
        return;
    }

    logDiagnostic('Starting biometric verification with SecuGen SGIMatchScore...', 'info');
    
    $('#verification_result').html(`
        <div class="alert alert-custom alert-info">
            <div class="text-center">
                <div class="spinner"></div> <strong>Verifying Patient Identity...</strong><br>
                <small>Comparing live fingerprint with stored template using SGIMatchScore</small>
            </div>
        </div>
    `);

    // Use SecuGen SGIMatchScore service directly (like GCC Registration)
    callSecuGenMatchScore();
}

function callSecuGenMatchScore() {
    const matchEndpoints = [
        'https://localhost:8443/SGIMatchScore',
        'http://localhost:8000/SGIMatchScore',
        'https://127.0.0.1:8443/SGIMatchScore',
        'http://127.0.0.1:8000/SGIMatchScore'
    ];
    
    let currentEndpoint = 0;
    let comparisonCompleted = false;
    
    function attemptMatching() {
        if (comparisonCompleted) return;
        
        if (currentEndpoint >= matchEndpoints.length) {
            // All endpoints failed - use fallback
            performDatabaseVerification();
            return;
        }
        
        const endpoint = matchEndpoints[currentEndpoint];
        logDiagnostic(`SGIMatchScore attempt ${currentEndpoint + 1}: ${endpoint}`, 'info');
        
        $.ajax({
            url: endpoint,
            type: 'POST',
            data: {
                Template1: window.radiologyState.storedTemplate,
                Template2: window.radiologyState.liveTemplate,
                TemplateFormat: 'ISO'
            },
            timeout: 15000,
            success: function(response) {
                if (comparisonCompleted) return;
                
                try {
                    const result = typeof response === 'string' ? JSON.parse(response) : response;
                    
                    if (result.ErrorCode === 0) {
                        comparisonCompleted = true;
                        const matchScore = parseInt(result.MatchingScore || 0);
                        const threshold = 60;
                        const isMatch = matchScore >= threshold;
                        
                        handleSecuGenMatchResult(matchScore, threshold, isMatch, endpoint);
                        return;
                    } else {
                        const errorDesc = getSecuGenErrorDescription(result.ErrorCode);
                        logDiagnostic(`SGIMatchScore error ${result.ErrorCode}: ${errorDesc}`, 'warning');
                    }
                } catch (e) {
                    logDiagnostic(`SGIMatchScore response parsing error: ${e.message}`, 'error');
                }
                
                // Try next endpoint
                currentEndpoint++;
                setTimeout(attemptMatching, 1000);
            },
            error: function(xhr, status, error) {
                logDiagnostic(`SGIMatchScore connection error on ${endpoint}: ${status} - ${error}`, 'warning');
                currentEndpoint++;
                setTimeout(attemptMatching, 1000);
            }
        });
    }
    
    attemptMatching();
}

function handleSecuGenMatchResult(matchScore, threshold, isMatch, endpoint) {
    window.radiologyState.matchScore = matchScore;
    window.radiologyState.verificationStatus = isMatch ? 'success' : 'failed';
    
    let quality, qualityClass;
    if (matchScore >= 100) {
        quality = 'Excellent';
        qualityClass = 'success';
    } else if (matchScore >= 80) {
        quality = 'Very Good';
        qualityClass = 'success';
    } else if (matchScore >= threshold) {
        quality = 'Good';
        qualityClass = 'info';
    } else {
        quality = 'Poor';
        qualityClass = 'danger';
    }
    
    logDiagnostic(`SGIMatchScore result: ${matchScore}/199 (${quality}) - ${isMatch ? 'MATCH' : 'NO MATCH'}`, isMatch ? 'success' : 'warning');
    
    if (isMatch) {
        window.radiologyState.verificationPassed = true;
        window.radiologyState.verificationMethod = 'secugen_biometric';
        
        $('#verification_result').html(`
            <div class="alert alert-custom alert-success">
                <h5><i class="fa fa-check-circle"></i> Biometric Verification Successful!</h5>
                <p><strong>Match Score: ${matchScore}/199 (${quality})</strong></p>
                <p>Patient identity confirmed using SecuGen biometric verification.</p>
                <p><strong>Method:</strong> SGIMatchScore template comparison</p>
                <p><strong>Endpoint:</strong> ${endpoint}</p>
                <small class="text-success">Ready for radiology order creation</small>
            </div>
        `);
        
        $('#order-panel').show();
        updateStepIndicator('order');
        
        // Filter radiology tests based on patient workflow
        if (window.radiologyState.selectedPatientData) {
            filterRadiologyTestsByWorkflow(window.radiologyState.selectedPatientData.patient_type);
        }
        
        // Log verification to database
        logVerificationToDatabase(matchScore, true, quality, 'secugen_biometric');
        
    } else {
        $('#verification_result').html(`
            <div class="alert alert-custom alert-danger">
                <h5><i class="fa fa-times-circle"></i> Biometric Verification Failed</h5>
                <p><strong>Match Score: ${matchScore}/199 (${quality})</strong></p>
                <p>Fingerprint does not match stored template (threshold: ${threshold}).</p>
                <div class="mt-3">
                    <button type="button" class="btn btn-warning mr-2" onclick="captureFingerprint()">
                        <i class="fa fa-refresh"></i> Try Again
                    </button>
                    <button type="button" class="btn btn-secondary" onclick="manualVerification()">
                        <i class="fa fa-user-check"></i> Manual Override
                    </button>
                </div>
            </div>
        `);
        
        logVerificationToDatabase(matchScore, false, quality, 'secugen_biometric');
    }
}



function handleVerificationResponse(response) {
    try {
        const result = typeof response === 'string' ? JSON.parse(response) : response;
        
        if (result.status === 'success') {
            window.radiologyState.verificationStatus = 'success';
            window.radiologyState.verificationPassed = true;
            window.radiologyState.matchScore = result.match_score || 0;
            
            $('#verification_result').html(`
                <div class="alert alert-custom alert-success">
                    <h5><i class="fa fa-check-circle"></i> Verification Successful</h5>
                    <p><strong>Match Score: ${result.match_score}%</strong></p>
                    <p>Patient identity confirmed. Ready for radiology services.</p>
                </div>
            `);
            
            // Show order panel with filtered tests
            showOrderPanelWithFilteredTests();
            
            logDiagnostic(`Verification successful - Match: ${result.match_score}%`, 'success');
            
        } else {
            window.radiologyState.verificationStatus = 'failed';
            
            $('#verification_result').html(`
                <div class="alert alert-custom alert-danger">
                    <h5><i class="fa fa-times-circle"></i> Verification Failed</h5>
                    <p><strong>Match Score: ${result.match_score}%</strong></p>
                    <p>${result.message || 'Fingerprint does not match stored template.'}</p>
                    <div class="mt-3">
                        <button type="button" class="btn btn-verification mr-2" onclick="captureFingerprint()">
                            <i class="fa fa-refresh"></i> Try Again
                        </button>
                        <button type="button" class="btn btn-manual" onclick="manualVerification()">
                            <i class="fa fa-user-check"></i> Manual Override
                        </button>
                    </div>
                </div>
            `);
            
            logDiagnostic(`Verification failed - Match: ${result.match_score}%`, 'warning');
        }
    } catch (e) {
        logDiagnostic('Failed to parse verification response', 'error');
        handleVerificationError(null, 'parse_error', 'Invalid response format');
    }
}

function handleVerificationError(xhr, status, error) {
    logDiagnostic(`Verification error: ${status} - ${error}`, 'error');
    
    $('#verification_result').html(`
        <div class="alert alert-custom alert-danger">
            <h5><i class="fa fa-times-circle"></i> Verification Error</h5>
            <p>Failed to verify patient: ${error}</p>
            <div class="mt-3">
                <button type="button" class="btn btn-warning mr-2" onclick="captureFingerprint()">
                    <i class="fa fa-refresh"></i> Try Again
                </button>
                <button type="button" class="btn btn-secondary" onclick="manualVerification()">
                    <i class="fa fa-user-check"></i> Manual Override
                </button>
            </div>
        </div>
    `);
}

function manualVerification() {
    if (!window.radiologyState.selectedPatient) {
        showAlert('Please select a patient first', 'warning');
        return;
    }

    if (!window.radiologyState.selectedPatientData.has_fingerprint) {
        performStaffOverride();
        return;
    }

    logDiagnostic('Starting manual verification process...', 'warning');
    
    $('#verification_result').html(`
        <div class="alert alert-custom alert-warning">
            <h5><i class="fa fa-hand-paper-o"></i> Manual Verification</h5>
            <p>Proceeding with manual override verification process.</p>
            <div class="text-center mt-3">
                <div class="spinner"></div> <strong>Processing manual verification...</strong>
            </div>
        </div>
    `);
    
    setTimeout(function() {
        window.radiologyState.verificationStatus = 'success';
        window.radiologyState.verificationPassed = true;
        window.radiologyState.verificationMethod = 'manual_override';
        
        $('#verification_result').html(`
            <div class="alert alert-custom alert-warning">
                <h5><i class="fa fa-check-circle"></i> Manual Verification Complete</h5>
                <p>Patient verified through manual override process.</p>
                <p><strong>Staff:</strong> ${getCurrentStaffName()}</p>
                <small class="text-warning">Manual verification completed - audit trail logged</small>
            </div>
        `);
        
        $('#order-panel').show();
        updateStepIndicator('order');
        
        recordWorkflowBypass();
        showAlert('Manual verification completed successfully', 'success');
        
    }, 2000);
}

function performStaffOverride() {
    const reason = prompt('Staff override reason (required):\n\nCommon reasons:\n- Patient unable to provide fingerprint\n- Device malfunction\n- Emergency situation\n- Technical failure');
    
    if (!reason || reason.trim() === '') {
        showAlert('Reason is required for staff override', 'warning');
        return;
    }

    const staffId = getCurrentStaffId();
    const staffName = getCurrentStaffName();
    
    logDiagnostic(`Processing staff override by ${staffName}: ${reason}`, 'warning');
    
    $('#verification_result').html(`
        <div class="alert alert-custom alert-warning">
            <h5><i class="fa fa-user-check"></i> Staff Override</h5>
            <p><strong>Patient verified by authorized staff.</strong></p>
            <p><strong>Reason:</strong> ${reason}</p>
            <p><strong>Staff:</strong> ${staffName} (${staffId})</p>
            <small class="text-warning">Staff override completed - audit trail logged</small>
        </div>
    `);
    
    window.radiologyState.verificationStatus = 'success';
    window.radiologyState.verificationPassed = true;
    window.radiologyState.verificationMethod = 'staff_override';
    
    $('#order-panel').show();
    updateStepIndicator('order');
    
    $.ajax({
        url: window.radiologyConfig.baseURL + window.radiologyConfig.endpoints.verifyPatient,
        type: 'POST',
        data: {
            patient_id: window.radiologyState.selectedPatient,
            action: 'staff_override',
            override_reason: reason.trim(),
            staff_id: staffId,
            staff_name: staffName,
            system: 'radiology'
        }
    });
}

function loadStoredFingerprintTemplate(patientId) {
    logDiagnostic(`Loading stored fingerprint template for patient ${patientId}...`, 'info');
    
    $('#fingerprint_status').html(`
        <div class="alert alert-custom alert-info">
            <div class="text-center">
                <div class="spinner"></div> <strong>Loading stored template...</strong><br>
                <small>Retrieving fingerprint data from database</small>
            </div>
        </div>
    `);
    
    $.ajax({
        url: window.radiologyConfig.baseURL + 'admin/radiology/getStoredFingerprintTemplate',
        type: 'POST',
        data: { patient_id: patientId },
        dataType: 'json',
        timeout: 10000,
        success: function(response) {
            if (response.status === 'success' && response.fingerprint_template) {
                window.radiologyState.storedTemplate = response.fingerprint_template;
                
                $('#fingerprint_status').html(`
                    <div class="alert alert-custom alert-success">
                        <i class="fa fa-fingerprint"></i> <strong>Stored template loaded successfully</strong><br>
                        <small>Template size: ${response.fingerprint_template.length} characters</small><br>
                        <small class="text-success">Ready for fingerprint capture and verification</small>
                    </div>
                `);
                
                $('#capture_fingerprint').prop('disabled', false);
                logDiagnostic(`Stored template loaded: ${response.fingerprint_template.length} characters`, 'success');
                
            } else {
                $('#fingerprint_status').html(`
                    <div class="alert alert-custom alert-warning">
                        <i class="fa fa-exclamation-triangle"></i> <strong>No stored template</strong><br>
                        <small>${response.message || 'No fingerprint template found for this patient'}</small><br>
                        <small>Manual verification will be required</small>
                    </div>
                `);
                
                logDiagnostic(`No stored template: ${response.message}`, 'warning');
            }
        },
        error: function(xhr, status, error) {
            logDiagnostic(`Failed to load stored template: ${error}`, 'error');
            
            $('#fingerprint_status').html(`
                <div class="alert alert-custom alert-danger">
                    <i class="fa fa-times-circle"></i> <strong>Template loading failed</strong><br>
                    <small>Error: ${error}</small><br>
                    <small>Manual verification may be required</small>
                </div>
            `);
        }
    });
}

// =============================================
// TEST SELECTION AND COST CALCULATION
// =============================================

function toggleTest(testId) {
    const checkbox = $('#test_' + testId);
    const testItem = checkbox.closest('.test-item');
    
    // Only allow toggling if the test item is visible (not filtered out)
    if (testItem.is(':visible')) {
        checkbox.prop('checked', !checkbox.prop('checked'));
        
        if (checkbox.prop('checked')) {
            testItem.addClass('selected');
        } else {
            testItem.removeClass('selected');
        }
        
        calculateCosts();
    }
}

function calculateCosts() {
    let baseCost = 0;
    const selectedTests = $('.test-checkbox:checked');
    
    selectedTests.each(function() {
        baseCost += parseFloat($(this).data('charge')) || 0;
    });
    
    const priority = $('#priority').val() || 'routine';
    const priorityMultiplier = window.radiologyConfig.costs.priorityMultipliers[priority] || 1.0;
    const priorityFee = baseCost * (priorityMultiplier - 1.0);
    
    const subtotal = baseCost + priorityFee;
    const taxAmount = subtotal * window.radiologyConfig.costs.taxRate;
    const totalAmount = subtotal + taxAmount;
    
    $('#base_cost').text('KES ' + formatCurrency(baseCost));
    $('#priority_fee').text('KES ' + formatCurrency(priorityFee));
    $('#tax_amount').text('KES ' + formatCurrency(taxAmount));
    $('#total_amount').text('KES ' + formatCurrency(totalAmount));
    
    window.radiologyState.totalCost = totalAmount;
    window.radiologyState.selectedTests = [];
    selectedTests.each(function() {
        window.radiologyState.selectedTests.push($(this).val());
    });
    
    if (selectedTests.length === 0) {
        if ($('#test-selection-message').length === 0) {
            $('.test-selection-grid').after('<div id="test-selection-message" class="text-danger mt-2"><small><i class="fa fa-exclamation-triangle"></i> Please select at least one test</small></div>');
        }
    } else {
        $('#test-selection-message').remove();
    }
}

function clearTestSelections() {
    $('.test-checkbox').prop('checked', false);
    $('.test-item').removeClass('selected');
    calculateCosts();
}

function formatCurrency(amount) {
    return parseFloat(amount).toLocaleString('en-KE', {
        minimumFractionDigits: 2,
        maximumFractionDigits: 2
    });
}

function createRadiologyOrder() {
    if (window.radiologyState.verificationStatus !== 'success') {
        showAlert('Patient verification required', 'warning');
        return;
    }
    
    if (!window.radiologyState.selectedPatient) {
        showMessage('Please select a patient first', 'error');
        return;
    }
    
    const selectedTests = $('.test-checkbox:checked');
    if (selectedTests.length === 0) {
        showMessage('Please select at least one imaging test', 'error');
        return;
    }
    
    if (!$('#referringDoctor').val()) {
        showMessage('Please select referring doctor', 'error');
        return;
    }
    
    logDiagnostic('Creating radiology order...', 'info');
    
    const formData = new FormData();
    formData.append('patient_id', window.radiologyState.selectedPatient);
    formData.append('referring_doctor_id', $('#referringDoctor').val());
    formData.append('priority', $('#priority').val() || 'routine');
    formData.append('clinical_notes', $('#clinical_notes').val() || '');
    formData.append('scheduled_date', $('#scheduled_date').val() || '');
    formData.append('scheduled_time', $('#scheduled_time').val() || '');
    
    selectedTests.each(function() {
        formData.append('imaging_tests[]', $(this).val());
    });
    
    formData.append('verification_method', window.radiologyState.verificationMethod || 'manual');
    formData.append('verification_score', window.radiologyState.matchScore || 0);
    formData.append('total_amount', window.radiologyState.totalCost || 0);
    formData.append('created_by', getCurrentStaffId());
    
    const submitBtn = $('button[onclick="createRadiologyOrder()"]');
    const originalText = submitBtn.html();
    submitBtn.prop('disabled', true).html('<i class="fa fa-spinner fa-spin"></i> Creating Order...');
    
    $.ajax({
        url: window.radiologyConfig.baseURL + window.radiologyConfig.endpoints.createOrder,
        type: 'POST',
        data: formData,
        processData: false,
        contentType: false,
        timeout: 30000,
        success: function(response) {
            try {
                const result = typeof response === 'string' ? JSON.parse(response) : response;
                
                if (result.success || result.status === 'success') {
                    const orderId = result.order_id || result.id || 'RAD' + Date.now();
                    showMessage('Radiology order created successfully! Order ID: #' + orderId, 'success');
                    updateStepIndicator('schedule');
                    
                    window.radiologyState.orderCreated = true;
                    
                    setTimeout(function() {
                        if (confirm('Order created successfully! Would you like to create another order?')) {
                            resetSystem();
                        }
                    }, 2000);
                    
                } else {
                    const errorMessage = result.message || result.error || 'Unknown error occurred';
                    showMessage('Error creating radiology order: ' + errorMessage, 'error');
                    logDiagnostic('Order creation failed: ' + errorMessage, 'error');
                }
                
            } catch (parseError) {
                if (typeof response === 'string' && (response.includes('<!DOCTYPE') || response.includes('<html'))) {
                    showMessage('Server returned HTML instead of JSON. Check PHP errors.', 'error');
                } else {
                    showMessage('Invalid JSON response from server', 'error');
                }
            }
        },
        error: function(xhr, status, error) {
            let errorMessage = 'Failed to create radiology order. ';
            
            if (xhr.responseText) {
                if (xhr.responseText.includes('<!DOCTYPE') || xhr.responseText.includes('<html')) {
                    errorMessage += 'Server returned HTML error page. Check server logs.';
                } else if (xhr.responseText.includes('404')) {
                    errorMessage += 'Method not found (404). Check controller and routes.';
                } else if (xhr.responseText.includes('500')) {
                    errorMessage += 'Server error (500). Check PHP error logs.';
                } else {
                    errorMessage += xhr.responseText.substring(0, 200);
                }
            }
            
            if (xhr.status === 0) {
                errorMessage += 'Network error - check if server is running.';
            } else if (xhr.status === 404) {
                errorMessage += 'Method not found. Check URL.';
            } else if (xhr.status === 500) {
                errorMessage += 'Server internal error. Check PHP error logs.';
            }
            
            showMessage(errorMessage, 'error');
            logDiagnostic(`AJAX Error: ${status} - ${error}`, 'error');
        },
        complete: function() {
            submitBtn.prop('disabled', false).html(originalText);
        }
    });
}

// =============================================
// UTILITY FUNCTIONS
// =============================================

function updateStepIndicator(currentStep) {
    $('.step').removeClass('active completed');
    
    const steps = ['search', 'verify', 'order', 'schedule'];
    const currentIndex = steps.indexOf(currentStep);
    
    for (let i = 0; i <= currentIndex; i++) {
        const stepElement = $(`#step-${steps[i]}`);
        if (i < currentIndex) {
            stepElement.addClass('completed');
        } else {
            stepElement.addClass('active');
        }
    }
    
    window.radiologyState.currentStep = currentStep;
}

function showMessage(message, type) {
    const alertClass = 'alert-' + (type === 'error' ? 'danger' : type);
    const iconClass = type === 'success' ? 'check-circle' : type === 'error' ? 'exclamation-triangle' : 'info-circle';
    
    const alertHtml = '<div class="alert ' + alertClass + ' alert-dismissible" style="position: fixed; top: 20px; right: 20px; z-index: 9999; min-width: 300px;">' +
                     '<button type="button" class="close" data-dismiss="alert">&times;</button>' +
                     '<i class="fa fa-' + iconClass + '"></i> ' + message +
                     '</div>';
    
    $('body').append(alertHtml);
    
    setTimeout(function() {
        $('.alert').fadeOut();
    }, 5000);
}

function getWorkflowBadge(patientType) {
    switch (patientType) {
        case 'GCC':
            return '<span class="workflow-status workflow-gcc">GCC</span>';
        default:
            return '<span class="workflow-status workflow-general">General</span>';
    }
}

function resetSystem() {
    window.radiologyState.selectedPatient = null;
    window.radiologyState.selectedPatientData = null;
    window.radiologyState.verificationStatus = 'pending';
    window.radiologyState.liveTemplate = null;
    window.radiologyState.matchScore = 0;
    window.radiologyState.orderCreated = false;
    window.radiologyState.selectedTests = [];
    window.radiologyState.totalCost = 0;
    
    $('#patient_search').val('');
    $('#patient_results').html('<div class="alert alert-custom alert-info"><i class="fa fa-info-circle"></i> Enter search criteria</div>');
    $('#verification-panel, #order-panel').hide();
    $('#radiology_order_form')[0].reset();
    
    // Clear test selections and show all tests
    clearTestSelections();
    $('.test-item').show(); // Show all tests when no patient is selected
    
    resetVerificationForm();
    calculateCosts();
    updateStepIndicator('search');
    
    logDiagnostic('System reset', 'info');
}

function resetVerificationForm() {
    $('#live_fingerprint_display').attr('src', window.radiologyConfig.baseURL + 'Images/PlaceFinger.bmp');
    $('#fingerprint_status').html('<small class="text-muted">Place finger on scanner</small>');
    $('#fingerprint_template, #fingerprint_image, #live_template').val('');
    $('#capture_fingerprint').prop('disabled', false);
    $('#verify_patient').prop('disabled', true);
    $('#verification_result').empty();
    $('#selected_patient_info, #verification_history').empty();
}

function resetVerificationState() {
    window.radiologyState.verificationStatus = 'pending';
    window.radiologyState.liveTemplate = null;
    window.radiologyState.matchScore = 0;
    resetVerificationForm();
}

function showAlert(message, type = 'info', duration = 5000) {
    const alertTypes = {
        'success': 'check',
        'info': 'info-circle',
        'warning': 'exclamation-triangle',
        'danger': 'times-circle'
    };
    
    const icon = alertTypes[type] || 'info-circle';
    
    const alertDiv = $(`
        <div class="alert alert-custom alert-${type} alert-dismissible fade show" role="alert" style="position: fixed; top: 20px; right: 20px; z-index: 9999; min-width: 300px;">
            <i class="fa fa-${icon}"></i> ${message}
            <button type="button" class="close" data-dismiss="alert" aria-label="Close">
                <span aria-hidden="true">&times;</span>
            </button>
        </div>
    `);
    
    $('body').append(alertDiv);
    
    if (duration > 0) {
        setTimeout(function() {
            alertDiv.fadeOut(function() {
                $(this).remove();
            });
        }, duration);
    }
    
    logDiagnostic(`Alert: [${type.toUpperCase()}] ${message}`, type === 'danger' ? 'error' : type);
}

function logDiagnostic(message, type = 'info') {
    const timestamp = new Date().toLocaleTimeString();
    const logClass = `log-${type}`;
    const logDiv = $('#diagnostic_log');
    
    if (logDiv.length === 0) {
        console.log(`[${timestamp}] [Radiology] ${message}`);
        return;
    }
    
    logDiv.append(`<div class="${logClass}">[${timestamp}] ${message}</div>`);
    logDiv.scrollTop(logDiv[0].scrollHeight);
    
    const logEntries = logDiv.children();
    if (logEntries.length > 100) {
        logEntries.first().remove();
    }
    
    const consoleMethod = type === 'error' ? 'error' : type === 'warning' ? 'warn' : 'log';
    console[consoleMethod](`[Radiology] ${message}`);
}

function getSecuGenErrorDescription(errorCode) {
    const errorMap = {
        1: 'Reader not found or device driver error',
        2: 'Function failed - check device connection',
        3: 'Invalid parameters sent to device',
        51: 'Scan timeout - no finger detected',
        52: 'Poor image quality - please try again',
        53: 'Finger not properly placed on scanner',
        54: 'Timeout occurred during scan',
        55: 'Device not found or not responding'
    };
    
    return errorMap[errorCode] || `Unknown SecuGen error (Code: ${errorCode})`;
}

function getCurrentStaffId() {
    return sessionStorage.getItem('staff_id') || 
           localStorage.getItem('staff_id') || 
           $('#current_staff_id').val() || 
           'unknown';
}

function getCurrentStaffName() {
    return sessionStorage.getItem('staff_name') || 
           localStorage.getItem('staff_name') || 
           $('#current_staff_name').val() || 
           'Unknown Staff';
}

// =============================================
// DEBUG FUNCTIONS
// =============================================

window.radiologyDebug = {
    secugenState: function() {
        console.group('SecuGen Device State');
        console.log('Service Active:', window.radiologyState.secugenActive);
        console.log('Service URL:', window.secugenConfig.serviceURL);
        console.log('Device Connected:', window.secugenConfig.deviceConnected);
        console.log('Live Template Available:', !!window.radiologyState.liveTemplate);
        console.log('Live Template Size:', window.radiologyState.liveTemplate ? window.radiologyState.liveTemplate.length : 0);
        console.log('Match Score:', window.radiologyState.matchScore);
        console.groupEnd();
        return window.secugenConfig;
    },
    
    testConnection: function() {
        console.log('Testing SecuGen WebAPI connection...');
        initializeSecuGenService();
    },
    
    retryConnection: function() {
        console.log('Retrying SecuGen connection...');
        window.radiologyState.secugenActive = false;
        window.secugenConfig.serviceActive = false;
        window.secugenConfig.deviceConnected = false;
        initializeSecuGenService();
    },
    
    testCapture: function() {
        if (window.radiologyState.secugenActive) {
            console.log('Testing fingerprint capture...');
            captureFingerprint();
        } else {
            console.log('SecuGen not active - testing connection first...');
            this.testConnection();
        }
    },
    
    getState: function() {
        console.group('Radiology System State');
        console.log('Main State:', window.radiologyState);
        console.log('SecuGen Config:', window.secugenConfig);
        console.log('Current Step:', window.radiologyState.currentStep);
        console.log('Selected Patient:', window.radiologyState.selectedPatient);
        console.log('Verification Status:', window.radiologyState.verificationStatus);
        console.groupEnd();
        return window.radiologyState;
    },
    
    simulateVerification: function(patientId, success = true, matchScore = null) {
        if (patientId) {
            window.radiologyState.selectedPatient = patientId;
        }
        
        const score = matchScore || (success ? (85 + Math.random() * 10) : (30 + Math.random() * 40));
        const result = {
            status: success ? 'success' : 'failed',
            match_score: Math.round(score * 10) / 10,
            message: success ? 'Verification successful' : 'Match score below threshold'
        };
        
        console.log('Simulating verification result:', result);
        handleVerificationResponse(result);
    },
    
    simulatePatientSelect: function(patientId = 1, name = 'Test Patient', hasFingerprint = true, patientType = 'GENERAL') {
        selectPatient(patientId, name, 'TEST001', '0712345678', hasFingerprint, patientType);
    },
    
    reset: function() {
        console.log('Resetting system via debug...');
        resetSystem();
    },
    
    testCosts: function() {
        $('.test-checkbox').slice(0, 2).prop('checked', true).trigger('change');
        $('#priority').val('urgent').trigger('change');
        console.log('Cost calculation test completed. Check cost summary.');
    },
    
    diagnostics: function() {
        console.group('System Diagnostics');
        console.log('jQuery version:', $.fn.jquery);
        console.log('Base URL:', window.radiologyConfig.baseURL);
        console.log('SecuGen active:', window.radiologyState.secugenActive);
        console.log('Current step:', window.radiologyState.currentStep);
        console.log('Selected patient:', window.radiologyState.selectedPatient);
        console.log('Selected tests:', window.radiologyState.selectedTests);
        console.log('Total cost:', window.radiologyState.totalCost);
        console.groupEnd();
    },
    
    testEndpoint: function() {
        const testUrl = window.radiologyConfig.baseURL + 'admin/radiology/createRadiologyOrder';
        console.log('🔍 Testing endpoint:', testUrl);
        
        $.ajax({
            url: testUrl,
            type: 'GET',
            success: function(response) {
                console.log('✅ Endpoint accessible:', response);
            },
            error: function(xhr) {
                console.log('❌ Endpoint error:', xhr.status, xhr.responseText);
            }
        });
    },
    
    testMinimalOrder: function() {
        const testUrl = window.radiologyConfig.baseURL + 'admin/radiology/createRadiologyOrder';
        
        const minimalData = {
            patient_id: 1,
            referring_doctor_id: 1,
            imaging_tests: [1],
            priority: 'routine'
        };
        
        console.log('🔍 Testing with minimal data:', minimalData);
        
        $.ajax({
            url: testUrl,
            type: 'POST',
            data: minimalData,
            dataType: 'json',
            success: function(response) {
                console.log('✅ Minimal test success:', response);
            },
            error: function(xhr) {
                console.log('❌ Minimal test error:', xhr.status, xhr.responseText);
            }
        });
    }
};

// =============================================
// CONSOLE HELP
// =============================================

console.log(`
╔══════════════════════════════════════════════════════════╗
║              RADIOLOGY VERIFICATION SYSTEM              ║
║                 Fixed Version 2.0                       ║
╠══════════════════════════════════════════════════════════╣
║                                                          ║
║ 🔐 WORKFLOW-BASED VERIFICATION:                         ║
║ ✓ GCC patients: Fingerprint verification ENFORCED      ║
║ ✓ General patients: Fingerprint verification BYPASSED  ║
║ ✓ Specialized patients: Fingerprint verification OPTIONAL║
║ ✓ Automatic workflow detection and enforcement         ║
║                                                          ║
║ 🏥 RADIOLOGY WORKFLOW:                                  ║
║ ✓ Patient search and selection                          ║
║ ✓ Biometric identity verification                       ║
║ ✓ Order creation with cost calculation                  ║
║ ✓ Priority-based scheduling                             ║
║ ✓ GCC workflow compliance checking                      ║
║                                                          ║
║ 💰 COST MANAGEMENT:                                     ║
║ ✓ Real-time cost calculation                            ║
║ ✓ Priority surcharges (Urgent +25%, STAT +50%)         ║
║ ✓ Tax calculation (16% VAT)                             ║
║ ✓ Multiple test selection                               ║
║                                                          ║
║ 🔧 DEBUG COMMANDS:                                      ║
║ radiologyDebug.secugenState()    - Show SecuGen state   ║
║ radiologyDebug.testConnection()  - Test SecuGen WebAPI  ║
║ radiologyDebug.retryConnection() - Retry connection     ║
║ radiologyDebug.testCapture()     - Test fingerprint     ║
║ radiologyDebug.getState()        - Show system state    ║
║ radiologyDebug.simulateVerification() - Test verify     ║
║ radiologyDebug.simulatePatientSelect() - Test patient   ║
║ radiologyDebug.reset()           - Reset system         ║
║ radiologyDebug.diagnostics()     - Show diagnostics     ║
║ radiologyDebug.testEndpoint()    - Test order endpoint  ║
║ radiologyDebug.testMinimalOrder() - Test minimal order  ║
║                                                          ║
║ 🔧 WORKFLOW TESTING:                                   ║
║ radiologyDebug.simulatePatientSelect(1, 'Test GCC', true, 'GCC')  ║
║ radiologyDebug.simulatePatientSelect(2, 'Test General', false, 'GENERAL') ║
║                                                          ║
║ 🔧 SECUGEN TROUBLESHOOTING:                            ║
║ 1. Download SecuGen WebAPI from webapi.secugen.com     ║
║ 2. Install and run the WebAPI service                  ║
║ 3. Accept SSL certificate at localhost:8443/SGIFPGetInfo║
║ 4. Connect SecuGen fingerprint scanner                 ║
║ 5. Test: radiologyDebug.testConnection()               ║
║                                                          ║
╚══════════════════════════════════════════════════════════╝

System ready for radiology patient verification!
Type 'radiologyDebug.getState()' to check current system status.
Type 'radiologyDebug.simulatePatientSelect(1, "Test GCC", true, "GCC")' to test GCC workflow.
Type 'radiologyDebug.simulatePatientSelect(2, "Test General", false, "GENERAL")' to test General workflow.
`);

// Export for global access
window.RadiologySystem = {
    state: window.radiologyState,
    config: window.radiologyConfig,
    secugen: window.secugenConfig,
    debug: window.radiologyDebug,
    
    // Main functions
    searchPatients: searchPatients,
    selectPatient: selectPatient,
    captureFingerprint: captureFingerprint,
    verifyPatient: verifyPatient,
    manualVerification: manualVerification,
    createOrder: createRadiologyOrder,
    reset: resetSystem,
    
    // Utility functions
    showAlert: showAlert,
    logDiagnostic: logDiagnostic,
    calculateCosts: calculateCosts,
    updateStepIndicator: updateStepIndicator,
    
    // Workflow functions
    enforceWorkflowVerification: enforceWorkflowVerification,
    bypassVerificationAndProceed: bypassVerificationAndProceed,
    getWorkflowConfig: getWorkflowConfig
};
</script>