<!-- 
Enhanced Queue Sound Notifications System
Include this view in any department page using: $this->load->view('admin/queue/sound_notifications');
-->

<!DOCTYPE html>
<html>
<head>
    <meta charset="utf-8">
    <title>Enhanced Queue Sound Notifications</title>
    <style>
        .notification-controls {
            position: fixed;
            top: 20px;
            right: 20px;
            background: white;
            border: 1px solid #ddd;
            border-radius: 8px;
            padding: 20px;
            box-shadow: 0 4px 20px rgba(0,0,0,0.15);
            z-index: 9999;
            min-width: 320px;
            max-width: 380px;
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
        }

        .notification-controls.minimized {
            height: 60px;
            overflow: hidden;
        }

        .notification-header {
            display: flex;
            align-items: center;
            justify-content: space-between;
            margin-bottom: 15px;
            border-bottom: 1px solid #eee;
            padding-bottom: 10px;
        }

        .notification-header h6 {
            margin: 0;
            color: #333;
            font-weight: 600;
        }

        .notification-toggle {
            display: flex;
            align-items: center;
            margin-bottom: 12px;
            padding: 8px;
            background: #f8f9fa;
            border-radius: 5px;
        }

        .notification-status {
            padding: 12px;
            border-radius: 5px;
            margin: 8px 0;
            font-size: 12px;
            font-weight: 500;
        }

        .status-active {
            background: #d4edda;
            color: #155724;
            border: 1px solid #c3e6cb;
        }

        .status-inactive {
            background: #f8d7da;
            color: #721c24;
            border: 1px solid #f5c6cb;
        }

        .volume-control {
            margin: 12px 0;
            padding: 8px;
            background: #f8f9fa;
            border-radius: 5px;
        }

        .volume-slider {
            width: 100%;
            margin: 5px 0;
        }

        .new-patient-alert {
            position: fixed;
            top: 100px;
            right: 20px;
            background: linear-gradient(135deg, #28a745, #20c997);
            color: white;
            padding: 20px;
            border-radius: 8px;
            box-shadow: 0 4px 20px rgba(0,0,0,0.25);
            z-index: 10000;
            min-width: 280px;
            max-width: 350px;
            animation: slideInBounce 0.5s ease-out;
            border-left: 5px solid #fff;
        }

        .new-patient-alert.urgent {
            background: linear-gradient(135deg, #dc3545, #e74c3c);
            animation: slideInShake 0.6s ease-out;
        }

        .new-patient-alert.priority-high {
            background: linear-gradient(135deg, #fd7e14, #f39c12);
        }

        @keyframes slideInBounce {
            0% {
                transform: translateX(100%) scale(0.9);
                opacity: 0;
            }
            60% {
                transform: translateX(-10px) scale(1.02);
                opacity: 1;
            }
            100% {
                transform: translateX(0) scale(1);
            }
        }

        @keyframes slideInShake {
            0% {
                transform: translateX(100%);
                opacity: 0;
            }
            30% {
                transform: translateX(-15px);
                opacity: 1;
            }
            50% {
                transform: translateX(10px);
            }
            70% {
                transform: translateX(-5px);
            }
            100% {
                transform: translateX(0);
            }
        }

        .alert-close {
            float: right;
            cursor: pointer;
            font-weight: bold;
            font-size: 18px;
            opacity: 0.8;
            transition: opacity 0.3s;
        }

        .alert-close:hover {
            opacity: 1;
        }

        .control-buttons {
            display: flex;
            gap: 8px;
            margin: 10px 0;
            flex-wrap: wrap;
        }

        .control-btn {
            flex: 1;
            background: #6c757d;
            color: white;
            border: none;
            padding: 6px 10px;
            border-radius: 4px;
            cursor: pointer;
            font-size: 11px;
            transition: background-color 0.3s;
        }

        .control-btn:hover {
            background: #5a6268;
        }

        .sound-test-btn {
            background: #007bff;
            flex: 0 0 auto;
        }

        .sound-test-btn:hover {
            background: #0056b3;
        }

        .minimize-btn {
            background: #17a2b8;
        }

        .minimize-btn:hover {
            background: #138496;
        }

        .clear-btn {
            background: #dc3545;
        }

        .clear-btn:hover {
            background: #c82333;
        }

        .debug-btn {
            background: #6f42c1;
        }

        .debug-btn:hover {
            background: #5a359a;
        }

        .notification-log {
            max-height: 180px;
            overflow-y: auto;
            border: 1px solid #ddd;
            padding: 10px;
            margin-top: 12px;
            font-size: 11px;
            background: #f8f9fa;
            border-radius: 5px;
        }

        .log-entry {
            margin: 4px 0;
            padding: 4px 8px;
            border-left: 3px solid #007bff;
            background: white;
            border-radius: 3px;
            font-family: 'Courier New', monospace;
        }

        .log-entry.success {
            border-left-color: #28a745;
        }

        .log-entry.error {
            border-left-color: #dc3545;
            background: #fff5f5;
        }

        .log-entry.warning {
            border-left-color: #ffc107;
            background: #fffdf5;
        }

        .department-info {
            background: #e3f2fd;
            padding: 8px;
            border-radius: 4px;
            margin-bottom: 10px;
            font-size: 11px;
            text-align: center;
            color: #1565c0;
            font-weight: 500;
        }

        .stats-display {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 8px;
            margin: 10px 0;
            font-size: 10px;
        }

        .stat-item {
            background: white;
            padding: 6px;
            border-radius: 3px;
            border: 1px solid #dee2e6;
            text-align: center;
        }

        .stat-number {
            font-weight: bold;
            font-size: 14px;
            color: #007bff;
        }

        .alert-header {
            font-weight: bold;
            margin-bottom: 8px;
            display: flex;
            align-items: center;
            justify-content: space-between;
        }

        .priority-badge {
            background: rgba(255,255,255,0.3);
            padding: 2px 8px;
            border-radius: 12px;
            font-size: 10px;
            font-weight: bold;
        }
    </style>
</head>
<body>

<!-- Enhanced Notification Controls Panel -->
<div class="notification-controls" id="notificationControls">
    <div class="notification-header">
        <h6><i class="fa fa-bell"></i> Queue Notifications</h6>
        <div class="stats-display" id="queueStats">
            <div class="stat-item">
                <div class="stat-number" id="waitingCount">0</div>
                <div>Waiting</div>
            </div>
            <div class="stat-item">
                <div class="stat-number" id="completedToday">0</div>
                <div>Completed</div>
            </div>
        </div>
        
        <div class="control-buttons">
            <button onclick="clearNotificationLog()" class="control-btn clear-btn">
                <i class="fa fa-trash"></i> Clear Log
            </button>
            <button onclick="debugQueueStatus()" class="control-btn debug-btn">
                <i class="fa fa-bug"></i> Debug
            </button>
            <button onclick="addTestPatient()" class="control-btn" style="background: #6f42c1;">
                <i class="fa fa-user-plus"></i> Test Patient
            </button>
        </div>
        
        <div class="notification-log" id="notificationLog">
            <div class="log-entry">System initializing...</div>
        </div>
    </div>
</div>

<!-- Enhanced Audio Elements with Multiple Formats -->
<audio id="newPatientSound" preload="auto">
    <source src="<?php echo base_url(); ?>assets/sounds/notifications/new_patient.wav" type="audio/wav">
    <source src="<?php echo base_url(); ?>assets/sounds/notifications/new_patient.mp3" type="audio/mpeg">
    <source src="<?php echo base_url(); ?>assets/sounds/notifications/new_patient.ogg" type="audio/ogg">
</audio>

<audio id="urgentPatientSound" preload="auto">
    <source src="<?php echo base_url(); ?>assets/sounds/notifications/urgent_patient.wav" type="audio/wav">
    <source src="<?php echo base_url(); ?>assets/sounds/notifications/urgent_patient.mp3" type="audio/mpeg">
    <source src="<?php echo base_url(); ?>assets/sounds/notifications/urgent_patient.ogg" type="audio/ogg">
</audio>

<audio id="transferredPatientSound" preload="auto">
    <source src="<?php echo base_url(); ?>assets/sounds/notifications/transferred_patient.wav" type="audio/wav">
    <source src="<?php echo base_url(); ?>assets/sounds/notifications/transferred_patient.mp3" type="audio/mpeg">
    <source src="<?php echo base_url(); ?>assets/sounds/notifications/transferred_patient.ogg" type="audio/ogg">
</audio>

<script>
// =============================================
// ENHANCED QUEUE NOTIFICATION SYSTEM
// =============================================

class EnhancedQueueNotifications {
    constructor() {
        this.enabled = true;
        this.volume = 0.7;
        this.lastCheck = Math.floor(Date.now() / 1000);
        this.checkInterval = 8000; // 8 seconds for better responsiveness
        this.intervalId = null;
        this.department = this.getCurrentDepartment();
        this.notificationCount = 0;
        this.isDebugMode = false;
        this.stats = { waiting: 0, completed: 0 };
        this.audioContext = null;
        this.baseUrl = '<?php echo base_url(); ?>';
        
        this.init();
    }
    
    init() {
        console.log('🔊 Enhanced Queue Notifications System v2.0 initialized');
        this.addLogEntry('Enhanced notification system started', 'success');
        
        this.setupEventListeners();
        this.startNotificationCheck();
        this.initializeAudio();
        this.updateDepartmentInfo();
        
        // Debug mode check
        if (window.location.href.includes('debug=true') || window.location.href.includes('test=true')) {
            this.isDebugMode = true;
            this.addLogEntry('Debug mode enabled', 'warning');
        }
        
        // Load initial stats
        this.loadQueueStats();
    }
    
    setupEventListeners() {
        const enableCheckbox = document.getElementById('enableNotifications');
        const volumeSlider = document.getElementById('notificationVolume');
        const volumeDisplay = document.getElementById('volumeDisplay');
        
        if (enableCheckbox) {
            enableCheckbox.addEventListener('change', (e) => {
                this.enabled = e.target.checked;
                this.updateStatus();
                
                if (this.enabled) {
                    this.startNotificationCheck();
                    this.addLogEntry('Notifications enabled', 'success');
                } else {
                    this.stopNotificationCheck();
                    this.addLogEntry('Notifications disabled', 'warning');
                }
            });
        }
        
        if (volumeSlider) {
            volumeSlider.addEventListener('input', (e) => {
                this.volume = e.target.value / 100;
                if (volumeDisplay) {
                    volumeDisplay.textContent = e.target.value + '%';
                }
                this.updateAudioVolume();
            });
        }
        
        // Auto-enable audio on first user interaction
        const enableAudioOnInteraction = () => {
            if (this.audioContext && this.audioContext.state === 'suspended') {
                this.audioContext.resume();
            }
            document.removeEventListener('click', enableAudioOnInteraction);
            document.removeEventListener('keydown', enableAudioOnInteraction);
        };
        
        document.addEventListener('click', enableAudioOnInteraction);
        document.addEventListener('keydown', enableAudioOnInteraction);
    }
    
    initializeAudio() {
        try {
            this.audioContext = new (window.AudioContext || window.webkitAudioContext)();
            this.addLogEntry('Audio system initialized', 'success');
            this.updateAudioVolume();
        } catch (error) {
            console.warn('Audio context not supported:', error);
            this.addLogEntry('Audio context not supported', 'warning');
        }
    }
    
    getCurrentDepartment() {
        const url = window.location.href;
        const departments = [
            'registration', 'triage', 'cardiology', 'neurology', 'orthopedics', 
            'dermatology', 'pediatrics', 'gynecology', 'psychiatry', 'psychology', 
            'ent', 'optometrics', 'urology', 'physiotherapy', 'physician', 
            'dental', 'surgery', 'nutrition', 'speech_therapy', 'queue', 
            'laboratory', 'radiology', 'pharmacy', 'phlebotomy', 'consultation'
        ];
        
        for (let dept of departments) {
            if (url.includes('/' + dept) || url.includes('/' + dept + '/')) {
                return dept;
            }
        }
        
        // Fallback: try to get from page title or body class
        const titleDept = document.title.toLowerCase();
        for (let dept of departments) {
            if (titleDept.includes(dept)) {
                return dept;
            }
        }
        
        return 'general';
    }
    
    updateDepartmentInfo() {
        const deptElement = document.getElementById('currentDepartment');
        if (deptElement) {
            const deptName = this.department.charAt(0).toUpperCase() + this.department.slice(1);
            deptElement.textContent = `${deptName} Department`;
        }
    }
    
    startNotificationCheck() {
        if (this.intervalId) {
            clearInterval(this.intervalId);
        }
        
        // Initial check
        this.checkForNotifications();
        
        this.intervalId = setInterval(() => {
            this.checkForNotifications();
        }, this.checkInterval);
        
        this.updateStatus();
    }
    
    stopNotificationCheck() {
        if (this.intervalId) {
            clearInterval(this.intervalId);
            this.intervalId = null;
        }
        this.updateStatus();
    }
    
    async checkForNotifications() {
        if (!this.enabled) return;
        
        try {
            const endpoint = this.getNotificationEndpoint();
            const params = new URLSearchParams({
                department: this.department === 'general' ? 'registration' : this.department,
                last_check: this.lastCheck
            });
            
            const url = `${this.baseUrl}${endpoint}?${params}`;
            
            if (this.isDebugMode) {
                console.log('🔍 Checking notifications:', url);
            }
            
            const response = await fetch(url);
            
            if (!response.ok) {
                throw new Error(`HTTP ${response.status} - ${response.statusText}`);
            }
            
            const data = await response.json();
            
            if (this.isDebugMode) {
                console.log('📨 Response:', data);
            }
            
            if (data.status === 'success') {
                if (data.notifications && data.notifications.length > 0) {
                    this.processNotifications(data.notifications);
                    this.addLogEntry(`${data.notifications.length} new notification(s)`, 'success');
                }
                
                // Update timestamp
                if (data.timestamp) {
                    this.lastCheck = data.timestamp;
                }
            } else if (data.status === 'error') {
                this.addLogEntry(`API Error: ${data.message}`, 'error');
            }
            
        } catch (error) {
            console.error('Notification check error:', error);
            this.addLogEntry(`Error: ${error.message}`, 'error');
            
            // Don't stop checking on error, just log it
            if (error.message.includes('404')) {
                this.addLogEntry('Endpoint not found - check Queue controller', 'error');
            }
        }
    }
    
    getNotificationEndpoint() {
        const url = window.location.href;
        
        if (url.includes('/admin/')) {
            return 'admin/queue/get_queue_notifications';
        } else {
            return 'queue/get_queue_notifications';
        }
    }
    
    processNotifications(notifications) {
        notifications.forEach(notification => {
            this.showNotification(notification);
            this.playNotificationSound(notification);
            this.logNotification(notification);
            this.notificationCount++;
        });
        
        this.updateStatus();
        this.loadQueueStats(); // Refresh stats after new notifications
    }
    
    showNotification(notification) {
        const alertDiv = document.createElement('div');
        alertDiv.className = 'new-patient-alert';
        
        // Determine alert styling based on priority and type
        const isUrgent = notification.priority === 'urgent' || notification.is_urgent;
        const isHigh = notification.priority === 'high';
        
        if (isUrgent) {
            alertDiv.classList.add('urgent');
        } else if (isHigh) {
            alertDiv.classList.add('priority-high');
        }
        
        // Determine icon based on notification type
        let icon = 'fa-user-plus';
        let title = 'New Patient';
        
        switch (notification.notification_type) {
            case 'transferred_patient':
                icon = 'fa-exchange';
                title = 'Patient Transferred';
                break;
            case 'urgent_patient':
                icon = 'fa-exclamation-triangle';
                title = 'Urgent Patient';
                break;
            case 'test_patient':
                icon = 'fa-flask';
                title = 'Test Patient';
                break;
        }
        
        alertDiv.innerHTML = `
            <div class="alert-header">
                <span><i class="fa ${icon}"></i> ${title}</span>
                <span class="priority-badge">${notification.priority?.toUpperCase() || 'NORMAL'}</span>
                <span class="alert-close" onclick="this.closest('.new-patient-alert').remove()">&times;</span>
            </div>
            <div style="margin-bottom: 8px;">
                <strong style="font-size: 16px;">${notification.patient_name}</strong>
                ${notification.patient_unique_id ? `<br><small>ID: ${notification.patient_unique_id}</small>` : ''}
            </div>
            <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 8px; font-size: 12px;">
                <div><strong>Queue:</strong> ${notification.queue_number}</div>
                <div><strong>Time:</strong> ${notification.time}</div>
                <div><strong>Dept:</strong> ${notification.department}</div>
                <div><strong>Type:</strong> ${notification.workflow_type || 'General'}</div>
            </div>
            ${notification.phone ? `<div style="margin-top: 8px; font-size: 11px;"><i class="fa fa-phone"></i> ${notification.phone}</div>` : ''}
            ${notification.from_department ? `<div style="margin-top: 4px; font-size: 10px; opacity: 0.8;">From: ${notification.from_department}</div>` : ''}
        `;
        
        document.body.appendChild(alertDiv);
        
        // Auto-remove based on priority
        const removeTime = isUrgent ? 15000 : (isHigh ? 12000 : 10000);
        setTimeout(() => {
            if (alertDiv.parentElement) {
                alertDiv.style.animation = 'slideOut 0.3s ease-in forwards';
                setTimeout(() => alertDiv.remove(), 300);
            }
        }, removeTime);
    }
    
    playNotificationSound(notification) {
        if (!this.enabled) return;
        
        try {
            let audio;
            
            // Choose sound based on priority and type
            if (notification.priority === 'urgent' || notification.is_urgent) {
                audio = document.getElementById('urgentPatientSound');
            } else if (notification.notification_type === 'transferred_patient') {
                audio = document.getElementById('transferredPatientSound');
            } else {
                audio = document.getElementById('newPatientSound');
            }
            
            if (audio && audio.readyState >= 2) { // HAVE_CURRENT_DATA
                audio.volume = this.volume;
                audio.currentTime = 0;
                
                const playPromise = audio.play();
                
                if (playPromise !== undefined) {
                    playPromise.catch(error => {
                        console.warn('Audio playback failed:', error);
                        this.generateBeepSound(notification);
                    });
                }
            } else {
                this.generateBeepSound(notification);
            }
            
        } catch (error) {
            console.error('Error playing sound:', error);
            this.generateBeepSound(notification);
        }
    }
    
    generateBeepSound(notification) {
        if (!this.audioContext) return;
        
        try {
            const isUrgent = notification?.priority === 'urgent' || notification?.is_urgent;
            
            // Create different beep patterns for different priorities
            if (isUrgent) {
                // Urgent: rapid triple beep
                this.createBeep(800, 0.2, 0);
                this.createBeep(800, 0.2, 0.3);
                this.createBeep(800, 0.2, 0.6);
            } else {
                // Normal: single beep
                this.createBeep(600, 0.5, 0);
            }
            
        } catch (error) {
            console.error('Error generating beep:', error);
        }
    }
    
    createBeep(frequency, duration, delay) {
        const oscillator = this.audioContext.createOscillator();
        const gainNode = this.audioContext.createGain();
        
        oscillator.connect(gainNode);
        gainNode.connect(this.audioContext.destination);
        
        oscillator.frequency.value = frequency;
        oscillator.type = 'sine';
        
        const startTime = this.audioContext.currentTime + delay;
        
        gainNode.gain.setValueAtTime(0, startTime);
        gainNode.gain.linearRampToValueAtTime(this.volume * 0.3, startTime + 0.01);
        gainNode.gain.exponentialRampToValueAtTime(0.01, startTime + duration);
        
        oscillator.start(startTime);
        oscillator.stop(startTime + duration);
    }
    
    updateAudioVolume() {
        const audioElements = document.querySelectorAll('audio');
        audioElements.forEach(audio => {
            audio.volume = this.volume;
        });
    }
    
    updateStatus() {
        const statusElement = document.getElementById('notificationStatus');
        
        if (!statusElement) return;
        
        if (this.enabled && this.intervalId) {
            statusElement.className = 'notification-status status-active';
            statusElement.innerHTML = `
                <i class="fa fa-check-circle"></i> 
                Active - ${this.notificationCount} received
                <br><small>Checking every ${this.checkInterval/1000}s | Last: ${new Date(this.lastCheck * 1000).toLocaleTimeString()}</small>
            `;
        } else {
            statusElement.className = 'notification-status status-inactive';
            statusElement.innerHTML = `
                <i class="fa fa-times-circle"></i> 
                Notifications Disabled
            `;
        }
    }
    
    async loadQueueStats() {
        try {
            const response = await fetch(`${this.baseUrl}admin/queue/debug_queue_status`);
            const data = await response.json();
            
    department_stats[this.department];
                
                if (deptStats) {
                    document.getElementById('waitingCount').textContent = deptStats.waiting_count || 0;
                    document.getElementById('completedToday').textContent = deptStats.completed_today || 0;
                }
            }
        } catch (error) {
            console.warn('Failed to load stats:', error);
        }
    }
    
    logNotification(notification) {
        const logMessage = `${notification.patient_name} (${notification.queue_number}) - ${notification.department} [${notification.priority}]`;
        this.addLogEntry(logMessage, 'success');
    }
    
    addLogEntry(message, type = 'info') {
        const logElement = document.getElementById('notificationLog');
        
        if (!logElement) return;
        
        const timestamp = new Date().toLocaleTimeString();
        
        const entry = document.createElement('div');
        entry.className = `log-entry ${type}`;
        entry.innerHTML = `<strong>${timestamp}</strong>: ${message}`;
        
        logElement.appendChild(entry);
        logElement.scrollTop = logElement.scrollHeight;
        
        // Keep only last 30 entries to save memory
        const entries = logElement.querySelectorAll('.log-entry');
        if (entries.length > 30) {
            entries[0].remove();
        }
    }
    
    // =============================================
    // DEBUG METHODS
    // =============================================
    
    async debugQueueStatus() {
        try {
            const response = await fetch(`${this.baseUrl}admin/queue/debug_queue_status`);
            const data = await response.json();
            
            console.log('🐛 Debug Queue Status:', data);
            this.addLogEntry(`Debug: ${data.total_waiting || 0} patients waiting across all departments`, 'info');
            
            return data;
        } catch (error) {
            console.error('Debug failed:', error);
            this.addLogEntry(`Debug failed: ${error.message}`, 'error');
        }
    }
    
    async addTestPatient(department = null) {
        try {
            const dept = department || this.department;
            const response = await fetch(`${this.baseUrl}admin/queue/add_test_patient_to_queue?department=${dept}`);
            const data = await response.json();
            
            console.log('🧪 Test Patient Added:', data);
            this.addLogEntry(`Test patient added to ${dept}: ${data.queue_number || 'Unknown'}`, 'success');
            
            return data;
        } catch (error) {
            console.error('Test patient failed:', error);
            this.addLogEntry(`Test patient failed: ${error.message}`, 'error');
        }
    }
}

// =============================================
// GLOBAL FUNCTIONS - Define immediately
// =============================================

// Define functions immediately in global scope
window.testNotificationSound = function() {
    console.log('🔊 Testing notification sound...');
    if (window.queueNotificationSystem) {
        const testNotification = {
            patient_id: 999,
            patient_name: 'Test Patient',
            queue_number: 'TEST-' + Date.now().toString().slice(-3),
            department: window.queueNotificationSystem.department,
            priority: 'normal',
            time: new Date().toLocaleTimeString(),
            workflow_type: 'test',
            phone: '0700123456',
            notification_type: 'test_patient',
            is_urgent: false
        };
        
        window.queueNotificationSystem.showNotification(testNotification);
        window.queueNotificationSystem.playNotificationSound(testNotification);
        window.queueNotificationSystem.addLogEntry('Sound test completed', 'success');
    } else {
        console.error('Queue notification system not initialized');
        alert('Notification system not ready. Please refresh the page.');
    }
};

window.clearNotificationLog = function() {
    console.log('🗑️ Clearing notification log...');
    const logElement = document.getElementById('notificationLog');
    if (logElement) {
        logElement.innerHTML = '<div class="log-entry success">Log cleared</div>';
    }
};

window.toggleNotificationPanel = function() {
    console.log('🔄 Toggling notification panel...');
    const panel = document.getElementById('notificationControls');
    const toggleBtn = document.getElementById('toggleBtn');
    
    if (!panel || !toggleBtn) {
        console.error('Panel or toggle button not found');
        return;
    }
    
    const isMinimized = panel.classList.contains('minimized');
    
    if (isMinimized) {
        panel.classList.remove('minimized');
        toggleBtn.innerHTML = '<i class="fa fa-minimize"></i>';
        console.log('Panel expanded');
    } else {
        panel.classList.add('minimized');
        toggleBtn.innerHTML = '<i class="fa fa-maximize"></i>';
        console.log('Panel minimized');
    }
};

window.debugQueueStatus = function() {
    console.log('🐛 Running queue debug...');
    if (window.queueNotificationSystem) {
        window.queueNotificationSystem.debugQueueStatus();
    } else {
        console.error('Queue notification system not initialized');
        alert('Notification system not ready. Please refresh the page.');
    }
};

window.addTestPatient = function(department = null) {
    console.log('🧪 Adding test patient...');
    if (window.queueNotificationSystem) {
        window.queueNotificationSystem.addTestPatient(department);
    } else {
        console.error('Queue notification system not initialized');
        alert('Notification system not ready. Please refresh the page.');
    }
};

// Add diagnostic function
window.diagnosticNotificationSystem = function() {
    console.log('🔍 Running notification system diagnostics...');
    
    const results = {
        timestamp: new Date().toISOString(),
        system_initialized: !!window.queueNotificationSystem,
        required_elements: {},
        global_functions: {},
        audio_support: {},
        current_state: {}
    };
    
    // Check required elements
    const requiredElements = [
        'notificationControls', 'enableNotifications', 'notificationVolume',
        'volumeDisplay', 'notificationStatus', 'notificationLog', 'currentDepartment'
    ];
    
    requiredElements.forEach(id => {
        const element = document.getElementById(id);
        results.required_elements[id] = {
            exists: !!element,
            visible: element ? element.offsetParent !== null : false
        };
    });
    
    // Check global functions
    const requiredFunctions = [
        'testNotificationSound', 'debugQueueStatus', 'addTestPatient',
        'clearNotificationLog', 'toggleNotificationPanel'
    ];
    
    requiredFunctions.forEach(func => {
        results.global_functions[func] = typeof window[func] === 'function';
    });
    
    // Check audio support
    results.audio_support = {
        audio_context: !!(window.AudioContext || window.webkitAudioContext),
        html5_audio: !!window.Audio
    };
    
    // Check current state if system is initialized
    if (window.queueNotificationSystem) {
        results.current_state = {
            department: window.queueNotificationSystem.department,
            enabled: window.queueNotificationSystem.enabled,
            volume: window.queueNotificationSystem.volume,
            notification_count: window.queueNotificationSystem.notificationCount,
            last_check: new Date(window.queueNotificationSystem.lastCheck * 1000).toISOString()
        };
    }
    
    console.table(results.required_elements);
    console.table(results.global_functions);
    console.table(results.audio_support);
    
    if (results.current_state) {
        console.table(results.current_state);
    }
    
    console.log('📊 Full diagnostic results:', results);
    
    // Show summary
    const issues = [];
    if (!results.system_initialized) issues.push('System not initialized');
    
    Object.entries(results.required_elements).forEach(([key, value]) => {
        if (!value.exists) issues.push(`Missing element: ${key}`);
    });
    
    Object.entries(results.global_functions).forEach(([key, value]) => {
        if (!value) issues.push(`Missing function: ${key}`);
    });
    
    if (issues.length === 0) {
        console.log('✅ All diagnostics passed - system should be working correctly');
        alert('✅ Notification system diagnostics passed!\n\nSystem appears to be working correctly.');
    } else {
        console.error('❌ Issues found:', issues);
        alert('❌ Issues found:\n\n' + issues.join('\n') + '\n\nCheck console for details.');
    }
    
    return results;
};

// Legacy function names for backwards compatibility
function testNotificationSound() { window.testNotificationSound(); }
function clearNotificationLog() { window.clearNotificationLog(); }
function toggleNotificationPanel() { window.toggleNotificationPanel(); }
function debugQueueStatus() { window.debugQueueStatus(); }
function addTestPatient(department) { window.addTestPatient(department); }
function diagnosticNotificationSystem() { window.diagnosticNotificationSystem(); }

// =============================================
// ENHANCED INITIALIZATION WITH ERROR HANDLING
// =============================================

// Initialize immediately when DOM is ready
(function() {
    console.log('🔄 Starting notification system initialization...');
    
    // Check if required elements exist
    function checkRequiredElements() {
        const required = [
            'notificationControls',
            'enableNotifications', 
            'notificationVolume',
            'volumeDisplay',
            'notificationStatus',
            'notificationLog'
        ];
        
        const missing = required.filter(id => !document.getElementById(id));
        
        if (missing.length > 0) {
            console.error('❌ Missing required elements:', missing);
            return false;
        }
        
        console.log('✅ All required elements found');
        return true;
    }
    
    // Initialize the notification system
    function initializeNotificationSystem() {
        try {
            if (!checkRequiredElements()) {
                console.error('❌ Cannot initialize - missing required elements');
                return false;
            }
            
            console.log('🚀 Initializing Enhanced Queue Notifications...');
            window.queueNotificationSystem = new EnhancedQueueNotifications();
            
            // Test basic functionality
            if (window.queueNotificationSystem && window.queueNotificationSystem.department) {
                console.log('✅ Queue notification system ready!');
                console.log(`📍 Department detected: ${window.queueNotificationSystem.department}`);
                console.log('🔧 Available commands:');
                console.log('  - testNotificationSound() - Test sound system');
                console.log('  - debugQueueStatus() - Check queue status');
                console.log('  - addTestPatient() - Add test patient');
                console.log('  - toggleNotificationPanel() - Toggle panel');
                console.log('  - clearNotificationLog() - Clear activity log');
                
                // Add a success message to the log
                window.queueNotificationSystem.addLogEntry('System initialized successfully', 'success');
                
                return true;
            } else {
                throw new Error('System initialization failed - object not created properly');
            }
            
        } catch (error) {
            console.error('❌ Failed to initialize notification system:', error);
            
            // Show error in the log if available
            const logElement = document.getElementById('notificationLog');
            if (logElement) {
                logElement.innerHTML = `<div class="log-entry error">Initialization failed: ${error.message}</div>`;
            }
            
            // Show browser alert as fallback
            setTimeout(() => {
                alert('Notification system failed to initialize. Please refresh the page or contact support.');
            }, 1000);
            
            return false;
        }
    }
    
    // Wait for DOM to be ready
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', function() {
            console.log('📄 DOM ready, checking for notification controls...');
            
            // Give a small delay to ensure all elements are rendered
            setTimeout(() => {
                const notificationControls = document.getElementById('notificationControls');
                
                if (notificationControls) {
                    console.log('🎛️ Notification controls found, initializing...');
                    initializeNotificationSystem();
                } else {
                    console.log('⚠️ Notification controls not found - skipping initialization');
                    console.log('💡 To enable notifications, include the sound_notifications view in your page');
                }
            }, 100);
        });
    } else {
        // DOM already ready
        console.log('📄 DOM already ready, checking for notification controls...');
        
        setTimeout(() => {
            const notificationControls = document.getElementById('notificationControls');
            
            if (notificationControls) {
                console.log('🎛️ Notification controls found, initializing...');
                initializeNotificationSystem();
            } else {
                console.log('⚠️ Notification controls not found - skipping initialization');
            }
        }, 100);
    }
    
})();

// Ensure functions are available globally immediately
console.log('🔧 Global notification functions loaded');

// Add CSS animation for slide out if not already present
if (!document.querySelector('style[data-notification-animations]')) {
    const style = document.createElement('style');
    style.setAttribute('data-notification-animations', 'true');
    style.textContent = `
        @keyframes slideOut {
            from {
                transform: translateX(0);
                opacity: 1;
            }
            to {
                transform: translateX(100%);
                opacity: 0;
            }
        }
        
        .notification-fade-out {
            animation: slideOut 0.3s ease-in forwards;
        }
    `;
    document.head.appendChild(style);
}
</script>

</body>
</html>