<!-- Department Queue Management -->
<div class="content-wrapper">
    <section class="content-header">
        <h1>
            <i class="fa fa-hospital-o"></i> <?php echo ucfirst($department); ?> Queue Management
            <small>Manage and monitor department queue</small>
        </h1>
        <ol class="breadcrumb">
            <li><a href="<?php echo base_url(); ?>admin/dashboard"><i class="fa fa-dashboard"></i> Home</a></li>
            <li><a href="<?php echo base_url('admin/queue'); ?>">Queue Management</a></li>
            <li class="active"><?php echo ucfirst($department); ?></li>
        </ol>
    </section>

    <section class="content">
        <!-- Current Status Row -->
        <div class="row">
            <div class="col-lg-3 col-xs-6">
                <div class="small-box bg-blue">
                    <div class="inner">
                        <h3 id="current-serving">-</h3>
                        <p>Currently Serving</p>
                    </div>
                    <div class="icon">
                        <i class="fa fa-user"></i>
                    </div>
                </div>
            </div>
            
            <div class="col-lg-3 col-xs-6">
                <div class="small-box bg-yellow">
                    <div class="inner">
                        <h3 id="waiting-count"><?php echo isset($queue_data['statistics']['waiting_count']) ? $queue_data['statistics']['waiting_count'] : 0; ?></h3>
                        <p>Waiting in Queue</p>
                    </div>
                    <div class="icon">
                        <i class="fa fa-clock-o"></i>
                    </div>
                </div>
            </div>
            
            <div class="col-lg-3 col-xs-6">
                <div class="small-box bg-green">
                    <div class="inner">
                        <h3 id="completed-today"><?php echo isset($queue_data['statistics']['completed_today']) ? $queue_data['statistics']['completed_today'] : 0; ?></h3>
                        <p>Completed Today</p>
                    </div>
                    <div class="icon">
                        <i class="fa fa-check"></i>
                    </div>
                </div>
            </div>
            
            <div class="col-lg-3 col-xs-6">
                <div class="small-box bg-red">
                    <div class="inner">
                        <h3 id="avg-wait"><?php echo isset($queue_data['statistics']['avg_wait_time']) ? $queue_data['statistics']['avg_wait_time'] : 0; ?></h3>
                        <p>Avg Wait (min)</p>
                    </div>
                    <div class="icon">
                        <i class="fa fa-clock-o"></i>
                    </div>
                </div>
            </div>
        </div>

        <!-- Queue Control Panel -->
        <div class="row">
            <div class="col-md-8">
                <div class="box box-primary">
                    <div class="box-header with-border">
                        <h3 class="box-title">
                            <i class="fa fa-list"></i> Queue List
                        </h3>
                        <div class="box-tools pull-right">
                            <button type="button" class="btn btn-sm btn-success" onclick="callNextPatient()">
                                <i class="fa fa-phone"></i> Call Next
                            </button>
                            <button type="button" class="btn btn-sm btn-default" onclick="refreshQueue()">
                                <i class="fa fa-refresh"></i> Refresh
                            </button>
                        </div>
                    </div>
                    
                    <div class="box-body">
                        <!-- Search -->
                        <div class="row mb-3">
                            <div class="col-md-6">
                                <div class="input-group">
                                    <input type="text" class="form-control" id="search-patient" 
                                           placeholder="Search by name, phone, or queue number...">
                                    <span class="input-group-btn">
                                        <button class="btn btn-default" type="button" onclick="searchPatient()">
                                            <i class="fa fa-search"></i>
                                        </button>
                                    </span>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <select class="form-control" id="priority-filter" onchange="filterByPriority()">
                                    <option value="">All Priorities</option>
                                    <option value="urgent">Urgent</option>
                                    <option value="high">High</option>
                                    <option value="normal">Normal</option>
                                    <option value="low">Low</option>
                                </select>
                            </div>
                        </div>

                        <!-- Queue Table -->
                        <div class="table-responsive">
                            <table class="table table-bordered table-hover" id="queue-table">
                                <thead>
                                    <tr>
                                        <th>Queue #</th>
                                        <th>Patient</th>
                                        <th>Contact</th>
                                        <th>Priority</th>
                                        <th>Wait Time</th>
                                        <th>Status</th>
                                        <th>Actions</th>
                                    </tr>
                                </thead>
                                <tbody id="queue-tbody">
                                    <!-- Queue items will be loaded here -->
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Current Patient Panel -->
            <div class="col-md-4">
                <div class="box box-info">
                    <div class="box-header with-border">
                        <h3 class="box-title">
                            <i class="fa fa-user"></i> Current Patient
                        </h3>
                    </div>
                    
                    <div class="box-body" id="current-patient-panel">
                        <div class="text-center text-muted" id="no-current-patient">
                            <i class="fa fa-user-times fa-3x"></i>
                            <p>No patient currently being served</p>
                            <button class="btn btn-primary" onclick="callNextPatient()">
                                <i class="fa fa-phone"></i> Call Next Patient
                            </button>
                        </div>
                        
                        <div id="current-patient-details" style="display: none;">
                            <div class="patient-info">
                                <h4 id="current-patient-name">-</h4>
                                <p class="text-muted" id="current-patient-contact">-</p>
                                <span class="label label-primary" id="current-queue-number">-</span>
                                <span class="label label-info" id="current-priority">-</span>
                            </div>
                            
                            <hr>
                            
                            <div class="service-actions">
                                <h5>Service Actions:</h5>
                                
                                <div class="form-group">
                                    <label>Next Department (optional):</label>
                                    <select class="form-control" id="next-department">
                                        <option value="">Complete Service</option>
                                        <option value="consultation">Consultation</option>
                                        <option value="laboratory">Laboratory</option>
                                        <option value="radiology">Radiology</option>
                                        <option value="pharmacy">Pharmacy</option>
                                        <option value="ipd">In-Patient</option>
                                        <option value="emergency">Emergency</option>
                                    </select>
                                </div>
                                
                                <div class="form-group">
                                    <label>Notes:</label>
                                    <textarea class="form-control" id="service-notes" rows="3" 
                                              placeholder="Service notes or comments..."></textarea>
                                </div>
                                
                                <div class="btn-group btn-group-justified">
                                    <div class="btn-group">
                                        <button class="btn btn-success" onclick="completeService()">
                                            <i class="fa fa-check"></i> Complete
                                        </button>
                                    </div>
                                    <div class="btn-group">
                                        <button class="btn btn-warning" onclick="skipPatient()">
                                            <i class="fa fa-skip-forward"></i> Skip
                                        </button>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Quick Add Panel -->
                <div class="box box-success">
                    <div class="box-header with-border">
                        <h3 class="box-title">
                            <i class="fa fa-plus"></i> Quick Add to Queue
                        </h3>
                    </div>
                    
                    <div class="box-body">
                        <div class="form-group">
                            <label>Patient ID:</label>
                            <input type="text" class="form-control" id="quick-patient-id" 
                                   placeholder="Enter patient ID...">
                        </div>
                        
                        <div class="form-group">
                            <label>Priority:</label>
                            <select class="form-control" id="quick-priority">
                                <option value="normal">Normal</option>
                                <option value="high">High</option>
                                <option value="urgent">Urgent</option>
                                <option value="low">Low</option>
                            </select>
                        </div>
                        
                        <button class="btn btn-success btn-block" onclick="quickAddToQueue()">
                            <i class="fa fa-plus"></i> Add to Queue
                        </button>
                    </div>
                </div>
            </div>
        </div>
    </section>
</div>

<!-- Patient Details Modal -->
<div class="modal fade" id="patientDetailsModal" tabindex="-1" role="dialog">
    <div class="modal-dialog modal-lg" role="document">
        <div class="modal-content">
            <div class="modal-header">
                <button type="button" class="close" data-dismiss="modal">&times;</button>
                <h4 class="modal-title">
                    <i class="fa fa-user"></i> Patient Details
                </h4>
            </div>
            
            <div class="modal-body" id="patient-details-content">
                <!-- Patient details will be loaded here -->
            </div>
            
            <div class="modal-footer">
                <button type="button" class="btn btn-default" data-dismiss="modal">Close</button>
            </div>
        </div>
    </div>
</div>

<script>
const DEPARTMENT = '<?php echo $department; ?>';
let currentServingPatient = null;
let queueUpdateInterval;

$(document).ready(function() {
    loadQueueData();
    
    // Auto-refresh every 15 seconds
    queueUpdateInterval = setInterval(refreshQueue, 15000);
    
    // Search on enter key
    $('#search-patient').on('keypress', function(e) {
        if (e.which === 13) {
            searchPatient();
        }
    });
});

function loadQueueData() {
    $.ajax({
        url: '<?php echo base_url("admin/queue/get_queue_status"); ?>',
        method: 'GET',
        data: { department: DEPARTMENT },
        dataType: 'json',
        success: function(response) {
            if (response.status === 'success') {
                updateQueueDisplay(response);
                updateCurrentPatient(response.current_serving);
                updateStatistics(response.statistics);
            } else {
                showAlert('error', response.message || 'Failed to load queue data');
            }
        },
        error: function() {
            showAlert('error', 'Failed to connect to server');
        }
    });
}

function updateQueueDisplay(data) {
    const tbody = $('#queue-tbody');
    tbody.empty();
    
    if (data.queue_list && data.queue_list.length > 0) {
        data.queue_list.forEach(function(patient) {
            const waitTime = calculateWaitTime(patient.created_at);
            const priorityClass = getPriorityClass(patient.priority);
            const statusClass = getStatusClass(patient.status);
            
            const row = `
                <tr data-queue-id="${patient.id}" class="${patient.status === 'in_progress' ? 'info' : ''}">
                    <td><strong>${patient.queue_number}</strong></td>
                    <td>
                        <a href="javascript:void(0)" onclick="showPatientDetails('${patient.patient_id}')">
                            ${patient.patient_name || 'N/A'}
                        </a>
                        <br><small class="text-muted">${patient.age || '-'} years, ${patient.gender || '-'}</small>
                    </td>
                    <td>${patient.mobileno || '-'}</td>
                    <td><span class="label ${priorityClass}">${patient.priority.toUpperCase()}</span></td>
                    <td>
                        <span class="wait-time" data-created="${patient.created_at}">${waitTime}</span>
                        ${patient.estimated_wait_time ? `<br><small class="text-muted">Est: ${patient.estimated_wait_time}m</small>` : ''}
                    </td>
                    <td><span class="label ${statusClass}">${patient.status.replace('_', ' ').toUpperCase()}</span></td>
                    <td>
                        <div class="btn-group">
                            ${patient.status === 'waiting' ? `
                                <button class="btn btn-xs btn-primary" onclick="callSpecificPatient(${patient.id})">
                                    <i class="fa fa-phone"></i> Call
                                </button>
                            ` : ''}
                            ${patient.status === 'in_progress' ? `
                                <button class="btn btn-xs btn-success" onclick="setCurrentPatient(${patient.id})">
                                    <i class="fa fa-check"></i> Complete
                                </button>
                            ` : ''}
                            <button class="btn btn-xs btn-warning" onclick="skipSpecificPatient(${patient.id})">
                                <i class="fa fa-skip-forward"></i> Skip
                            </button>
                            <button class="btn btn-xs btn-info" onclick="changePriority(${patient.id})">
                                <i class="fa fa-edit"></i>
                            </button>
                        </div>
                    </td>
                </tr>
            `;
            tbody.append(row);
        });
    } else {
        tbody.html('<tr><td colspan="7" class="text-center text-muted">No patients in queue</td></tr>');
    }
    
    // Update wait times every minute
    updateWaitTimes();
}

function updateCurrentPatient(servingPatient) {
    if (servingPatient) {
        currentServingPatient = servingPatient;
        $('#current-patient-name').text(servingPatient.patient_name || 'Unknown');
        $('#current-patient-contact').text(servingPatient.mobileno || 'No contact');
        $('#current-queue-number').text(servingPatient.queue_number);
        $('#current-priority').text(servingPatient.priority.toUpperCase());
        $('#current-serving').text(servingPatient.queue_number);
        
        $('#no-current-patient').hide();
        $('#current-patient-details').show();
    } else {
        currentServingPatient = null;
        $('#current-serving').text('-');
        $('#no-current-patient').show();
        $('#current-patient-details').hide();
    }
}

function updateStatistics(stats) {
    if (stats) {
        $('#waiting-count').text(stats.waiting_count || 0);
        $('#completed-today').text(stats.completed_today || 0);
        $('#avg-wait').text(stats.avg_wait_time || 0);
    }
}

function callNextPatient() {
    $.ajax({
        url: '<?php echo base_url("admin/queue/call_next"); ?>',
        method: 'POST',
        data: { department: DEPARTMENT },
        dataType: 'json',
        success: function(response) {
            if (response.status === 'success') {
                showAlert('success', `Called patient: ${response.patient.queue_number}`);
                loadQueueData();
                
                // Optional: Play announcement sound
                playAnnouncementSound();
            } else {
                showAlert('error', response.message || 'No patients in queue');
            }
        },
        error: function() {
            showAlert('error', 'Failed to call next patient');
        }
    });
}

function completeService() {
    if (!currentServingPatient) {
        showAlert('error', 'No patient currently being served');
        return;
    }
    
    const nextDepartment = $('#next-department').val();
    const notes = $('#service-notes').val();
    
    $.ajax({
        url: '<?php echo base_url("admin/queue/complete_service"); ?>',
        method: 'POST',
        data: {
            queue_id: currentServingPatient.id,
            next_department: nextDepartment,
            notes: notes
        },
        dataType: 'json',
        success: function(response) {
            if (response.status === 'success') {
                const message = nextDepartment 
                    ? `Service completed. Patient moved to ${nextDepartment}.`
                    : 'Service completed successfully.';
                showAlert('success', message);
                
                // Clear form
                $('#next-department').val('');
                $('#service-notes').val('');
                
                loadQueueData();
            } else {
                showAlert('error', response.message || 'Failed to complete service');
            }
        },
        error: function() {
            showAlert('error', 'Failed to complete service');
        }
    });
}

function skipPatient() {
    if (!currentServingPatient) {
        showAlert('error', 'No patient currently being served');
        return;
    }
    
    const reason = prompt('Reason for skipping this patient:');
    if (reason === null) return; // User cancelled
    
    skipSpecificPatient(currentServingPatient.id, reason);
}

function skipSpecificPatient(queueId, reason) {
    if (!reason) {
        reason = prompt('Reason for skipping this patient:');
        if (reason === null) return;
    }
    
    $.ajax({
        url: '<?php echo base_url("admin/queue/skip_patient"); ?>',
        method: 'POST',
        data: {
            queue_id: queueId,
            reason: reason
        },
        dataType: 'json',
        success: function(response) {
            if (response.status === 'success') {
                showAlert('success', 'Patient skipped successfully');
                loadQueueData();
            } else {
                showAlert('error', response.message || 'Failed to skip patient');
            }
        },
        error: function() {
            showAlert('error', 'Failed to skip patient');
        }
    });
}

function quickAddToQueue() {
    const patientId = $('#quick-patient-id').val().trim();
    const priority = $('#quick-priority').val();
    
    if (!patientId) {
        showAlert('error', 'Please enter a patient ID');
        return;
    }
    
    $.ajax({
        url: '<?php echo base_url("admin/queue/add_to_queue"); ?>',
        method: 'POST',
        data: {
            patient_id: patientId,
            department: DEPARTMENT,
            priority: priority
        },
        dataType: 'json',
        success: function(response) {
            if (response.status === 'success') {
                showAlert('success', `Patient added to queue: ${response.queue_number}`);
                $('#quick-patient-id').val('');
                loadQueueData();
            } else {
                showAlert('error', response.message || 'Failed to add patient to queue');
            }
        },
        error: function() {
            showAlert('error', 'Failed to add patient to queue');
        }
    });
}

function refreshQueue() {
    loadQueueData();
}

function searchPatient() {
    const searchTerm = $('#search-patient').val().trim();
    
    if (!searchTerm) {
        loadQueueData();
        return;
    }
    
    $.ajax({
        url: '<?php echo base_url("admin/queue/search_patient"); ?>',
        method: 'GET',
        data: {
            search: searchTerm,
            department: DEPARTMENT
        },
        dataType: 'json',
        success: function(response) {
            if (response.status === 'success') {
                const tbody = $('#queue-tbody');
                tbody.empty();
                
                if (response.patients.length > 0) {
                    response.patients.forEach(function(patient) {
                        const waitTime = calculateWaitTime(patient.created_at);
                        const priorityClass = getPriorityClass(patient.priority);
                        const statusClass = getStatusClass(patient.status);
                        
                        const row = `
                            <tr data-queue-id="${patient.id}">
                                <td><strong>${patient.queue_number}</strong></td>
                                <td>${patient.patient_name || 'N/A'}</td>
                                <td>${patient.mobileno || '-'}</td>
                                <td><span class="label ${priorityClass}">${patient.priority.toUpperCase()}</span></td>
                                <td>${waitTime}</td>
                                <td><span class="label ${statusClass}">${patient.status.replace('_', ' ').toUpperCase()}</span></td>
                                <td>
                                    <button class="btn btn-xs btn-primary" onclick="callSpecificPatient(${patient.id})">
                                        <i class="fa fa-phone"></i> Call
                                    </button>
                                </td>
                            </tr>
                        `;
                        tbody.append(row);
                    });
                } else {
                    tbody.html('<tr><td colspan="7" class="text-center text-muted">No patients found</td></tr>');
                }
            }
        },
        error: function() {
            showAlert('error', 'Search failed');
        }
    });
}

function filterByPriority() {
    const priority = $('#priority-filter').val();
    
    if (!priority) {
        loadQueueData();
        return;
    }
    
    $('#queue-tbody tr').each(function() {
        const rowPriority = $(this).find('.label').text().toLowerCase();
        if (rowPriority.includes(priority)) {
            $(this).show();
        } else {
            $(this).hide();
        }
    });
}

// Utility functions
function calculateWaitTime(createdAt) {
    const created = new Date(createdAt);
    const now = new Date();
    const diffMinutes = Math.floor((now - created) / 60000);
    
    if (diffMinutes < 60) {
        return diffMinutes + 'm';
    } else {
        const hours = Math.floor(diffMinutes / 60);
        const minutes = diffMinutes % 60;
        return hours + 'h ' + minutes + 'm';
    }
}

function updateWaitTimes() {
    $('.wait-time').each(function() {
        const createdAt = $(this).data('created');
        $(this).text(calculateWaitTime(createdAt));
    });
}

function getPriorityClass(priority) {
    switch (priority) {
        case 'urgent': return 'label-danger';
        case 'high': return 'label-warning';
        case 'normal': return 'label-info';
        case 'low': return 'label-default';
        default: return 'label-info';
    }
}

function getStatusClass(status) {
    switch (status) {
        case 'waiting': return 'label-warning';
        case 'in_progress': return 'label-primary';
        case 'completed': return 'label-success';
        case 'skipped': return 'label-danger';
        default: return 'label-default';
    }
}

function playAnnouncementSound() {
    // Optional: Play a notification sound
    try {
        const audio = new Audio('data:audio/wav;base64,UklGRnoGAABXQVZFZm10IBAAAAABAAEAQB8AAEAfAAABAAgAZGF0YQoGAACBhYqFbF1fdJivrJBhNjVgodDbq2EcBj+a2/LDciUFLIHO8tiJNwgZaLvt559NEAxQp+PwtmMcBjiR1/LMeSwFJHfH8N2QQAoUXrTp66hVFApGn+HyvmIhB');
        audio.play();
    } catch (e) {
        // Sound failed, ignore
    }
}

function showAlert(type, message) {
    const alertClass = type === 'success' ? 'alert-success' : 'alert-danger';
    const alertHtml = `<div class="alert ${alertClass} alert-dismissible">
        <button type="button" class="close" data-dismiss="alert">&times;</button>
        ${message}
    </div>`;
    
    $('.content').prepend(alertHtml);
    
    setTimeout(function() {
        $('.alert').fadeOut();
    }, 5000);
}

// Clean up interval on page unload
$(window).on('beforeunload', function() {
    if (queueUpdateInterval) {
        clearInterval(queueUpdateInterval);
    }
});
</script>

<style>
.patient-info h4 {
    margin-bottom: 5px;
}

.service-actions {
    margin-top: 15px;
}

.wait-time {
    font-weight: bold;
}

.table-hover tbody tr:hover {
    background-color: #f5f5f5;
}

#queue-table tbody tr.info {
    background-color: #d9edf7;
}

.btn-group .btn {
    margin: 1px;
}

.label {
    font-size: 11px;
    padding: 3px 6px;
}

.small-box h3 {
    font-size: 2.2em;
    font-weight: 700;
    margin: 0 0 10px 0;
}
</style>