<?php
/**
 * Loyalty Points Management Page
 * Path: application/views/admin/loyalty_points/index.php
 */
?>

<div class="content-wrapper">
    <!-- Content Header -->
    <section class="content-header">
        <h1>
            <i class="fa fa-star"></i> Loyalty Points Management
            <small>Manage patient loyalty rewards program</small>
        </h1>
        <ol class="breadcrumb">
            <li><a href="<?php echo base_url(); ?>admin/admin/dashboard"><i class="fa fa-dashboard"></i> Dashboard</a></li>
            <li class="active">Loyalty Points</li>
        </ol>
    </section>

    <!-- Main content -->
    <section class="content">
        <!-- Statistics Row -->
        <div class="row">
            <div class="col-lg-3 col-xs-6">
                <div class="small-box bg-yellow">
                    <div class="inner">
                        <h3 id="activeMembers">0</h3>
                        <p>Active Members</p>
                    </div>
                    <div class="icon">
                        <i class="fa fa-users"></i>
                    </div>
                </div>
            </div>
            <div class="col-lg-3 col-xs-6">
                <div class="small-box bg-green">
                    <div class="inner">
                        <h3 id="pointsEarnedToday">0</h3>
                        <p>Points Earned Today</p>
                    </div>
                    <div class="icon">
                        <i class="fa fa-plus"></i>
                    </div>
                </div>
            </div>
            <div class="col-lg-3 col-xs-6">
                <div class="small-box bg-red">
                    <div class="inner">
                        <h3 id="pointsRedeemedToday">0</h3>
                        <p>Points Redeemed Today</p>
                    </div>
                    <div class="icon">
                        <i class="fa fa-minus"></i>
                    </div>
                </div>
            </div>
            <div class="col-lg-3 col-xs-6">
                <div class="small-box bg-blue">
                    <div class="inner">
                        <h3 id="totalValue">KES 0</h3>
                        <p>Total Points Value</p>
                    </div>
                    <div class="icon">
                        <i class="fa fa-money"></i>
                    </div>
                </div>
            </div>
        </div>

        <!-- Main Content Row -->
        <div class="row">
            <div class="col-md-12">
                <!-- Patient Search Box -->
                <div class="box box-primary">
                    <div class="box-header with-border">
                        <h3 class="box-title"><i class="fa fa-search"></i> Patient Search</h3>
                    </div>
                    <div class="box-body">
                        <div class="row">
                            <div class="col-md-9">
                                <label>Search Patient</label>
                                <select id="loyaltyPatientSelect" class="form-control select2" style="width: 100%;">
                                    <option value="">Search by patient name, ID, or mobile number...</option>
                                </select>
                            </div>
                            <div class="col-md-3">
                                <label>&nbsp;</label>
                                <button type="button" class="btn btn-primary btn-block" onclick="loadLoyaltyPatientData()">
                                    <i class="fa fa-user"></i> Load Patient
                                </button>
                            </div>
                        </div>
                        
                        <!-- Selected Patient Loyalty Info -->
                        <div id="selectedLoyaltyPatientInfo" class="row" style="display: none; margin-top: 15px;">
                            <div class="col-md-12">
                                <div class="alert alert-success">
                                    <h4><i class="fa fa-star"></i> Patient Loyalty Information</h4>
                                    <div class="row">
                                        <div class="col-md-3">
                                            <strong>Name:</strong> <span id="loyaltyPatientName">-</span><br>
                                            <strong>ID:</strong> <span id="loyaltyPatientId">-</span>
                                        </div>
                                        <div class="col-md-3">
                                            <strong>Current Balance:</strong><br>
                                            <span class="label label-success" style="font-size: 16px; padding: 8px 12px;" id="currentPoints">0 Points</span>
                                        </div>
                                        <div class="col-md-3">
                                            <strong>Total Earned:</strong> <span id="totalEarned">0</span><br>
                                            <strong>Total Redeemed:</strong> <span id="totalRedeemed">0</span>
                                        </div>
                                        <div class="col-md-3">
                                            <strong>Total Spent:</strong> KES <span id="totalSpent">0</span><br>
                                            <button class="btn btn-sm btn-info" onclick="viewLoyaltyHistory()">
                                                <i class="fa fa-eye"></i> View History
                                            </button>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Award and Redeem Points Row -->
        <div class="row">
            <!-- Award Points -->
            <div class="col-md-6">
                <div class="box box-success">
                    <div class="box-header with-border">
                        <h3 class="box-title"><i class="fa fa-plus-circle"></i> Award Points</h3>
                    </div>
                    <div class="box-body">
                        <form id="awardPointsForm">
                            <div class="form-group">
                                <label>Points to Award <span class="text-red">*</span></label>
                                <input type="number" class="form-control" id="pointsToAward" min="1" max="1000" required>
                                <small class="help-block">Maximum 1000 points per transaction</small>
                            </div>
                            
                            <div class="form-group">
                                <label>Reason <span class="text-red">*</span></label>
                                <input type="text" class="form-control" id="awardReason" 
                                       placeholder="e.g., Exceptional service feedback" required>
                            </div>
                            
                            <div class="form-group">
                                <label>Reference Type</label>
                                <select class="form-control" id="awardReferenceType">
                                    <option value="">General</option>
                                    <option value="opd">OPD Visit</option>
                                    <option value="ipd">IPD Stay</option>
                                    <option value="pharmacy">Pharmacy Purchase</option>
                                    <option value="laboratory">Lab Tests</option>
                                    <option value="radiology">Radiology</option>
                                    <option value="feedback">Patient Feedback</option>
                                    <option value="referral">Patient Referral</option>
                                    <option value="birthday">Birthday Bonus</option>
                                    <option value="anniversary">Anniversary Bonus</option>
                                </select>
                            </div>
                            
                            <div class="form-group">
                                <label>Reference ID</label>
                                <input type="number" class="form-control" id="awardReferenceId" 
                                       placeholder="Optional: Related record ID">
                            </div>
                            
                            <div class="form-group">
                                <button type="submit" class="btn btn-success btn-lg btn-block">
                                    <i class="fa fa-plus"></i> Award Points
                                </button>
                            </div>
                        </form>
                    </div>
                </div>
            </div>

            <!-- Redeem Points -->
            <div class="col-md-6">
                <div class="box box-warning">
                    <div class="box-header with-border">
                        <h3 class="box-title"><i class="fa fa-gift"></i> Redeem Points</h3>
                    </div>
                    <div class="box-body">
                        <form id="redeemPointsForm">
                            <div class="form-group">
                                <label>Available Points</label>
                                <div class="form-control-static">
                                    <span class="label label-primary" style="font-size: 14px; padding: 6px 10px;" id="availablePointsDisplay">0 Points</span>
                                </div>
                            </div>
                            
                            <div class="form-group">
                                <label>Points to Redeem <span class="text-red">*</span></label>
                                <input type="number" class="form-control" id="pointsToRedeem" min="1" required>
                            </div>
                            
                            <div class="form-group">
                                <label>Service/Item <span class="text-red">*</span></label>
                                <select class="form-control" id="redeemService" required>
                                    <option value="">Select service...</option>
                                    <option value="consultation" data-value="100">Consultation Discount</option>
                                    <option value="laboratory" data-value="50">Lab Test Discount</option>
                                    <option value="pharmacy" data-value="75">Pharmacy Discount</option>
                                    <option value="radiology" data-value="150">Radiology Discount</option>
                                    <option value="ambulance" data-value="200">Ambulance Service</option>
                                    <option value="room_upgrade" data-value="300">Room Upgrade</option>
                                    <option value="custom">Custom Service</option>
                                </select>
                            </div>
                            
                            <div class="form-group">
                                <label>Redemption Value</label>
                                <div class="input-group">
                                    <span class="input-group-addon">KES</span>
                                    <input type="number" class="form-control" id="redemptionValue" step="0.01" readonly>
                                </div>
                                <small class="help-block">Default: 1 point = KES 50</small>
                            </div>
                            
                            <div class="form-group">
                                <label>Notes</label>
                                <textarea class="form-control" id="redeemNotes" rows="2" 
                                          placeholder="Optional notes about this redemption"></textarea>
                            </div>
                            
                            <div class="form-group">
                                <button type="submit" class="btn btn-warning btn-lg btn-block">
                                    <i class="fa fa-gift"></i> Redeem Points
                                </button>
                            </div>
                        </form>
                    </div>
                </div>
            </div>
        </div>

        <!-- Patient Points History -->
        <div class="row">
            <div class="col-md-12">
                <div class="box box-info">
                    <div class="box-header with-border">
                        <h3 class="box-title"><i class="fa fa-history"></i> Points History</h3>
                        <div class="box-tools pull-right">
                            <button type="button" class="btn btn-box-tool" onclick="refreshLoyaltyHistory()">
                                <i class="fa fa-refresh"></i>
                            </button>
                        </div>
                    </div>
                    <div class="box-body">
                        <div id="loyaltyHistoryList">
                            <div class="text-center text-muted" style="padding: 50px 20px;">
                                <i class="fa fa-star fa-3x"></i>
                                <p style="margin-top: 15px;">Select a patient to view loyalty history</p>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- All Activities Table -->
        <div class="row">
            <div class="col-md-12">
                <div class="box box-primary">
                    <div class="box-header with-border">
                        <h3 class="box-title"><i class="fa fa-list"></i> All Loyalty Activities</h3>
                        <div class="box-tools pull-right">
                            <button type="button" class="btn btn-success btn-sm" onclick="exportLoyaltyData()">
                                <i class="fa fa-download"></i> Export
                            </button>
                        </div>
                    </div>
                    <div class="box-body">
                        <!-- Filters -->
                        <div class="row" style="margin-bottom: 15px;">
                            <div class="col-md-2">
                                <select class="form-control input-sm" id="filterTransactionType">
                                    <option value="">All Types</option>
                                    <option value="earned">Earned</option>
                                    <option value="redeemed">Redeemed</option>
                                    <option value="adjusted">Adjusted</option>
                                    <option value="expired">Expired</option>
                                </select>
                            </div>
                            <div class="col-md-2">
                                <input type="date" class="form-control input-sm" id="filterActivityDateFrom" placeholder="From Date">
                            </div>
                            <div class="col-md-2">
                                <input type="date" class="form-control input-sm" id="filterActivityDateTo" placeholder="To Date">
                            </div>
                            <div class="col-md-3">
                                <input type="text" class="form-control input-sm" id="filterPatientName" placeholder="Search by patient name">
                            </div>
                            <div class="col-md-2">
                                <button class="btn btn-primary btn-sm" onclick="applyLoyaltyFilters()">
                                    <i class="fa fa-filter"></i> Filter
                                </button>
                            </div>
                            <div class="col-md-1">
                                <button class="btn btn-default btn-sm" onclick="clearLoyaltyFilters()">
                                    <i class="fa fa-times"></i>
                                </button>
                            </div>
                        </div>

                        <!-- Activities Table -->
                        <table id="loyaltyActivitiesTable" class="table table-bordered table-striped">
                            <thead>
                                <tr>
                                    <th>Date</th>
                                    <th>Patient</th>
                                    <th>Type</th>
                                    <th>Points</th>
                                    <th>Description</th>
                                    <th>Staff</th>
                                    <th>Reference</th>
                                </tr>
                            </thead>
                            <tbody>
                                <!-- Data loaded via AJAX -->
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>

        <!-- Loyalty Program Settings -->
        <div class="row">
            <div class="col-md-6">
                <div class="box box-solid">
                    <div class="box-header with-border">
                        <h3 class="box-title"><i class="fa fa-cog"></i> Program Settings</h3>
                    </div>
                    <div class="box-body">
                        <form id="loyaltySettingsForm">
                            <div class="form-group">
                                <label>Points per KES 1000 Spent</label>
                                <input type="number" class="form-control" name="points_per_1000_kes" 
                                       value="1" min="0" step="0.1">
                                <small class="help-block">How many points earned per KES 1000 spent</small>
                            </div>
                            
                            <div class="form-group">
                                <label>Point Value in KES</label>
                                <input type="number" class="form-control" name="point_value_kes" 
                                       value="50" min="1" step="0.01">
                                <small class="help-block">Value of 1 point when redeeming</small>
                            </div>
                            
                            <div class="form-group">
                                <label>Minimum Redemption Points</label>
                                <input type="number" class="form-control" name="minimum_redemption_points" 
                                       value="10" min="1">
                            </div>
                            
                            <div class="form-group">
                                <label>Max Points per Transaction</label>
                                <input type="number" class="form-control" name="max_points_per_transaction" 
                                       value="100" min="1">
                            </div>
                            
                            <div class="form-group">
                                <button type="submit" class="btn btn-primary">
                                    <i class="fa fa-save"></i> Update Settings
                                </button>
                            </div>
                        </form>
                    </div>
                </div>
            </div>
            
            <div class="col-md-6">
                <div class="box box-solid">
                    <div class="box-header with-border">
                        <h3 class="box-title"><i class="fa fa-trophy"></i> Top Loyalty Members</h3>
                    </div>
                    <div class="box-body">
                        <div id="topMembersList">
                            <!-- Top members list -->
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </section>
</div>

<!-- CSS Styles -->
<style>
.loyalty-history-item {
    border-left: 4px solid #f39c12;
    padding: 10px;
    margin-bottom: 10px;
    background: #fff;
    border-radius: 0 3px 3px 0;
    border: 1px solid #ddd;
}

.loyalty-history-item.earned {
    border-left-color: #00a65a;
    background: #f0fff0;
}

.loyalty-history-item.redeemed {
    border-left-color: #dd4b39;
    background: #fff5f5;
}

.loyalty-history-item.adjusted {
    border-left-color: #3c8dbc;
    background: #f0f8ff;
}

.loyalty-history-item.expired {
    border-left-color: #999;
    background: #f5f5f5;
}

.loyalty-history-item h5 {
    margin: 0 0 5px 0;
    font-size: 14px;
}

.loyalty-history-item .history-meta {
    font-size: 11px;
    color: #666;
    margin-bottom: 5px;
}

.points-display {
    font-size: 18px;
    font-weight: bold;
}

.points-earned {
    color: #00a65a;
}

.points-redeemed {
    color: #dd4b39;
}

.top-member-item {
    padding: 8px 0;
    border-bottom: 1px solid #eee;
}

.top-member-item:last-child {
    border-bottom: none;
}

.member-rank {
    width: 30px;
    height: 30px;
    border-radius: 50%;
    background: #f39c12;
    color: white;
    display: inline-flex;
    align-items: center;
    justify-content: center;
    font-weight: bold;
    margin-right: 10px;
}

.member-rank.rank-1 { background: #ffd700; color: #333; }
.member-rank.rank-2 { background: #c0c0c0; color: #333; }
.member-rank.rank-3 { background: #cd7f32; color: white; }
</style>

<!-- JavaScript -->
<script>
let currentLoyaltyPatientId = null;
let currentLoyaltyPatientData = null;

$(document).ready(function() {
    initializeLoyaltyComponents();
    loadLoyaltyDashboardStats();
    loadLoyaltySettings();
    loadTopMembers();
});

function initializeLoyaltyComponents() {
    // Initialize Select2
    $('#loyaltyPatientSelect').select2({
        placeholder: 'Search patient by name, ID, or mobile number',
        ajax: {
            url: '<?php echo base_url(); ?>admin/loyalty_points/search_patients',
            dataType: 'json',
            delay: 250,
            data: function (params) {
                return { q: params.term, page: params.page };
            },
            processResults: function (data, params) {
                params.page = params.page || 1;
                return {
                    results: data.items,
                    pagination: { more: (params.page * 30) < data.total_count }
                };
            },
            cache: true
        },
        minimumInputLength: 2
    });

    // Initialize DataTable
    $('#loyaltyActivitiesTable').DataTable({
        processing: true,
        serverSide: true,
        ajax: {
            url: '<?php echo base_url(); ?>admin/loyalty_points/get_all_activities',
            type: 'POST',
            data: function(d) {
                d.transaction_type = $('#filterTransactionType').val();
                d.date_from = $('#filterActivityDateFrom').val();
                d.date_to = $('#filterActivityDateTo').val();
                d.patient_name = $('#filterPatientName').val();
            }
        },
        columns: [
            { data: 'transaction_date', width: '100px' },
            { data: 'patient_name' },
            { data: 'transaction_type', width: '80px' },
            { data: 'points', width: '80px' },
            { data: 'description' },
            { data: 'staff_name', width: '100px' },
            { data: 'reference', width: '100px' }
        ],
        order: [[0, 'desc']],
        pageLength: 25
    });

    setupLoyaltyFormHandlers();
}

function setupLoyaltyFormHandlers() {
    $('#awardPointsForm').on('submit', function(e) {
        e.preventDefault();
        if (!currentLoyaltyPatientId) {
            toastr.warning('Please select a patient first');
            return;
        }
        awardPoints();
    });

    $('#redeemPointsForm').on('submit', function(e) {
        e.preventDefault();
        if (!currentLoyaltyPatientId) {
            toastr.warning('Please select a patient first');
            return;
        }
        redeemPoints();
    });

    $('#loyaltySettingsForm').on('submit', function(e) {
        e.preventDefault();
        updateLoyaltySettings();
    });

    // Auto-calculate redemption value
    $('#pointsToRedeem').on('input', function() {
        const points = parseInt($(this).val()) || 0;
        const pointValue = 50; // Default KES per point
        $('#redemptionValue').val(points * pointValue);
    });

    $('#redeemService').on('change', function() {
        const selectedOption = $(this).find('option:selected');
        const serviceValue = selectedOption.data('value');
        const points = parseInt($('#pointsToRedeem').val()) || 0;
        
        if (serviceValue && points > 0) {
            $('#redemptionValue').val(points * 50); // Base calculation
        }
    });
}

function loadLoyaltyPatientData() {
    const selectedPatient = $('#loyaltyPatientSelect').select2('data')[0];
    if (!selectedPatient) {
        toastr.warning('Please select a patient');
        return;
    }

    currentLoyaltyPatientId = selectedPatient.id;
    currentLoyaltyPatientData = selectedPatient;

    // Update patient info display
    $('#loyaltyPatientName').text(selectedPatient.patient_name);
    $('#loyaltyPatientId').text(selectedPatient.id);

    $('#selectedLoyaltyPatientInfo').show();
    
    loadPatientLoyaltyData();
}

function loadPatientLoyaltyData() {
    if (!currentLoyaltyPatientId) return;

    $.ajax({
        url: '<?php echo base_url(); ?>admin/loyalty_points/get_loyalty_points',
        method: 'GET',
        data: { patient_id: currentLoyaltyPatientId },
        dataType: 'json',
        success: function(response) {
            if (response.success) {
                const data = response.data;
                $('#currentPoints').text((data.current_balance || 0) + ' Points');
                $('#totalEarned').text(data.total_earned || 0);
                $('#totalRedeemed').text(data.total_redeemed || 0);
                $('#totalSpent').text(data.total_spent || 0);
                $('#availablePointsDisplay').text((data.current_balance || 0) + ' Points');
                
                loadPatientLoyaltyHistory();
            } else {
                toastr.error('Failed to load loyalty data');
            }
        },
        error: function() {
            toastr.error('Error loading loyalty data');
        }
    });
}

function loadPatientLoyaltyHistory() {
    if (!currentLoyaltyPatientId) return;

    $.ajax({
        url: '<?php echo base_url(); ?>admin/loyalty_points/get_loyalty_history',
        method: 'GET',
        data: { patient_id: currentLoyaltyPatientId, limit: 20 },
        dataType: 'json',
        success: function(response) {
            if (response.success) {
                displayLoyaltyHistory(response.data);
            }
        },
        error: function() {
            console.log('Error loading loyalty history');
        }
    });
}

function displayLoyaltyHistory(history) {
    const container = $('#loyaltyHistoryList');
    
    if (!history || history.length === 0) {
        container.html(`
            <div class="text-center text-muted" style="padding: 50px 20px;">
                <i class="fa fa-star fa-2x"></i>
                <p style="margin-top: 15px;">No loyalty history found</p>
            </div>
        `);
        return;
    }

    let html = '';
    history.forEach(function(transaction) {
        const isEarned = transaction.transaction_type === 'earned';
        const typeClass = transaction.transaction_type;
        const pointsClass = isEarned ? 'points-earned' : 'points-redeemed';
        const icon = isEarned ? 'plus' : 'minus';
        const sign = isEarned ? '+' : '-';

        html += `
            <div class="loyalty-history-item ${typeClass}">
                <h5>
                    <i class="fa fa-${icon}"></i> 
                    <span class="${pointsClass}">${sign}${Math.abs(transaction.points)} Points</span>
                </h5>
                <div class="history-meta">
                    ${transaction.description}
                </div>
                <div class="history-meta">
                    ${transaction.transaction_date} • 
                    ${transaction.staff_name ? 'by ' + transaction.staff_name : 'System'}
                    ${transaction.amount_spent ? ' • KES ' + transaction.amount_spent : ''}
                </div>
            </div>
        `;
    });

    container.html(html);
}

function awardPoints() {
    const points = parseInt($('#pointsToAward').val());
    const reason = $('#awardReason').val();
    const referenceType = $('#awardReferenceType').val();
    const referenceId = $('#awardReferenceId').val();

    if (!points || points <= 0) {
        toastr.warning('Please enter valid points amount');
        return;
    }

    if (!reason) {
        toastr.warning('Please enter a reason');
        return;
    }

    // Show loading
    const submitBtn = $('#awardPointsForm button[type="submit"]');
    const originalText = submitBtn.html();
    submitBtn.html('<i class="fa fa-spinner fa-spin"></i> Awarding...').prop('disabled', true);

    $.ajax({
        url: '<?php echo base_url(); ?>admin/loyalty_points/award_points',
        method: 'POST',
        data: {
            patient_id: currentLoyaltyPatientId,
            points: points,
            reason: reason,
            reference_type: referenceType,
            reference_id: referenceId
        },
        dataType: 'json',
        success: function(response) {
            if (response.success) {
                toastr.success('Points awarded successfully!');
                $('#awardPointsForm')[0].reset();
                loadPatientLoyaltyData();
                loadLoyaltyDashboardStats();
                $('#loyaltyActivitiesTable').DataTable().ajax.reload(null, false);
            } else {
                toastr.error('Award failed: ' + response.message);
            }
        },
        error: function() {
            toastr.error('Error awarding points');
        },
        complete: function() {
            submitBtn.html(originalText).prop('disabled', false);
        }
    });
}

function redeemPoints() {
    const points = parseInt($('#pointsToRedeem').val());
    const service = $('#redeemService').val();
    const value = parseFloat($('#redemptionValue').val());
    const notes = $('#redeemNotes').val();

    if (!points || points <= 0) {
        toastr.warning('Please enter valid points amount');
        return;
    }

    if (!service) {
        toastr.warning('Please select a service');
        return;
    }

    // Show loading
    const submitBtn = $('#redeemPointsForm button[type="submit"]');
    const originalText = submitBtn.html();
    submitBtn.html('<i class="fa fa-spinner fa-spin"></i> Redeeming...').prop('disabled', true);

    $.ajax({
        url: '<?php echo base_url(); ?>admin/loyalty_points/redeem_points',
        method: 'POST',
        data: {
            patient_id: currentLoyaltyPatientId,
            points: points,
            service: service,
            value: value,
            notes: notes
        },
        dataType: 'json',
        success: function(response) {
            if (response.success) {
                toastr.success('Points redeemed successfully!');
                $('#redeemPointsForm')[0].reset();
                loadPatientLoyaltyData();
                loadLoyaltyDashboardStats();
                $('#loyaltyActivitiesTable').DataTable().ajax.reload(null, false);
            } else {
                toastr.error('Redemption failed: ' + response.message);
            }
        },
        error: function() {
            toastr.error('Error redeeming points');
        },
        complete: function() {
            submitBtn.html(originalText).prop('disabled', false);
        }
    });
}

function loadLoyaltyDashboardStats() {
    $.ajax({
        url: '<?php echo base_url(); ?>admin/loyalty_points/get_dashboard_stats',
        method: 'GET',
        dataType: 'json',
        success: function(response) {
            if (response.success) {
                const stats = response.data;
                $('#activeMembers').text(stats.active_members || 0);
                $('#pointsEarnedToday').text(stats.points_earned_today || 0);
                $('#pointsRedeemedToday').text(stats.points_redeemed_today || 0);
                $('#totalValue').text('KES ' + (stats.total_points_value || 0));
            }
        },
        error: function() {
            console.log('Error loading dashboard stats');
        }
    });
}

function loadLoyaltySettings() {
    $.ajax({
        url: '<?php echo base_url(); ?>admin/loyalty_points/get_settings',
        method: 'GET',
        dataType: 'json',
        success: function(response) {
            if (response.success) {
                const settings = response.data;
                $('input[name="points_per_1000_kes"]').val(settings.points_per_1000_kes || 1);
                $('input[name="point_value_kes"]').val(settings.point_value_kes || 50);
                $('input[name="minimum_redemption_points"]').val(settings.minimum_redemption_points || 10);
                $('input[name="max_points_per_transaction"]').val(settings.max_points_per_transaction || 100);
            }
        },
        error: function() {
            console.log('Error loading settings');
        }
    });
}

function updateLoyaltySettings() {
    const settings = {
        points_per_1000_kes: $('input[name="points_per_1000_kes"]').val(),
        point_value_kes: $('input[name="point_value_kes"]').val(),
        minimum_redemption_points: $('input[name="minimum_redemption_points"]').val(),
        max_points_per_transaction: $('input[name="max_points_per_transaction"]').val()
    };

    $.ajax({
        url: '<?php echo base_url(); ?>admin/loyalty_points/update_settings',
        method: 'POST',
        data: { settings: settings },
        dataType: 'json',
        success: function(response) {
            if (response.success) {
                toastr.success('Settings updated successfully!');
            } else {
                toastr.error('Update failed: ' + response.message);
            }
        },
        error: function() {
            toastr.error('Error updating settings');
        }
    });
}

function loadTopMembers() {
    $.ajax({
        url: '<?php echo base_url(); ?>admin/loyalty_points/get_leaderboard',
        method: 'GET',
        data: { limit: 10 },
        dataType: 'json',
        success: function(response) {
            if (response.success) {
                displayTopMembers(response.data);
            }
        },
        error: function() {
            console.log('Error loading top members');
        }
    });
}

function displayTopMembers(members) {
    const container = $('#topMembersList');
    
    if (!members || members.length === 0) {
        container.html(`
            <div class="text-center text-muted" style="padding: 20px;">
                <i class="fa fa-trophy fa-2x"></i>
                <p style="margin-top: 10px;">No members found</p>
            </div>
        `);
        return;
    }

    let html = '';
    members.forEach(function(member, index) {
        const rank = index + 1;
        const rankClass = rank <= 3 ? `rank-${rank}` : '';

        html += `
            <div class="top-member-item">
                <div class="row">
                    <div class="col-xs-8">
                        <span class="member-rank ${rankClass}">${rank}</span>
                        <strong>${member.patient_name}</strong><br>
                        <small class="text-muted">ID: ${member.patient_id}</small>
                    </div>
                    <div class="col-xs-4 text-right">
                        <strong class="points-earned">${member.total_points} pts</strong><br>
                        <small class="text-muted">KES ${member.total_spent || 0}</small>
                    </div>
                </div>
            </div>
        `;
    });

    container.html(html);
}

function refreshLoyaltyHistory() {
    if (currentLoyaltyPatientId) {
        loadPatientLoyaltyHistory();
    } else {
        toastr.info('Please select a patient first');
    }
}

function viewLoyaltyHistory() {
    if (currentLoyaltyPatientId) {
        // Scroll to the history section
        $('html, body').animate({
            scrollTop: $("#loyaltyHistoryList").offset().top - 100
        }, 1000);
        
        toastr.info('Showing loyalty history below');
    }
}

function applyLoyaltyFilters() {
    $('#loyaltyActivitiesTable').DataTable().ajax.reload();
}

function clearLoyaltyFilters() {
    $('#filterTransactionType, #filterActivityDateFrom, #filterActivityDateTo, #filterPatientName').val('');
    $('#loyaltyActivitiesTable').DataTable().ajax.reload();
}

function exportLoyaltyData() {
    const params = new URLSearchParams({
        type: 'activities',
        transaction_type: $('#filterTransactionType').val(),
        date_from: $('#filterActivityDateFrom').val(),
        date_to: $('#filterActivityDateTo').val(),
        patient_name: $('#filterPatientName').val()
    });
    
    window.open('<?php echo base_url(); ?>admin/loyalty_points/export_data?' + params.toString(), '_blank');
}

// Auto-refresh dashboard stats every 5 minutes
setInterval(function() {
    loadLoyaltyDashboardStats();
}, 300000);

// Initialize tooltips
$(document).ready(function() {
    $('[data-toggle="tooltip"]').tooltip();
});
</script>