<?php
/**
 * Enhanced Laboratory Test Selection View with Fingerprint Verification
 * Save as: application/views/admin/laboratory/test_selection.php
 */
?>

<style>
/* Enhanced Styling for Laboratory Test Selection */
.test-category-card {
    border: 1px solid #ddd;
    border-radius: 5px;
    margin-bottom: 15px;
    transition: all 0.3s;
    background: #fff;
}

.test-category-card:hover {
    box-shadow: 0 2px 8px rgba(0,0,0,0.1);
    border-color: #3c8dbc;
}

.category-header {
    background: linear-gradient(135deg, #3c8dbc, #367fa9);
    color: white;
    padding: 12px 15px;
    border-radius: 5px 5px 0 0;
    cursor: pointer;
    transition: all 0.3s;
    position: relative;
}

.category-header:hover {
    background: linear-gradient(135deg, #367fa9, #2e6da4);
}

.category-tests {
    padding: 0;
    max-height: 350px;
    overflow-y: auto;
    border-radius: 0 0 5px 5px;
}

.test-item {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 12px 15px;
    border-bottom: 1px solid #f0f0f0;
    transition: all 0.2s;
    background: #fff;
}

.test-item:last-child {
    border-bottom: none;
}

.test-item:hover {
    background-color: #f8f9fa;
    padding-left: 20px;
}

.test-name {
    flex: 1;
    font-weight: 500;
    color: #333;
    font-size: 14px;
}

.test-price {
    color: #27ae60;
    font-weight: bold;
    margin-right: 15px;
    font-size: 14px;
    min-width: 100px;
    text-align: right;
}

.btn-add-test {
    background: linear-gradient(135deg, #28a745, #20a048);
    color: white;
    border: none;
    padding: 6px 12px;
    border-radius: 4px;
    font-size: 12px;
    font-weight: 500;
    transition: all 0.2s;
    min-width: 60px;
}

.btn-add-test:hover {
    background: linear-gradient(135deg, #218838, #1e7e34);
    color: white;
    transform: translateY(-1px);
}

.btn-add-test:disabled {
    background: #6c757d;
    cursor: not-allowed;
    transform: none;
}

.cart-item {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 12px 15px;
    border: 1px solid #e3e6f0;
    background: linear-gradient(135deg, #f8f9fa, #e9ecef);
    margin-bottom: 8px;
    border-radius: 5px;
    transition: all 0.2s;
}

.cart-item:hover {
    box-shadow: 0 2px 4px rgba(0,0,0,0.1);
}

.cart-item-name {
    flex: 1;
    font-weight: 500;
    color: #495057;
    font-size: 13px;
}

.cart-item-price {
    color: #28a745;
    font-weight: bold;
    margin-right: 10px;
    font-size: 13px;
    min-width: 90px;
    text-align: right;
}

.btn-remove-test {
    background: linear-gradient(135deg, #dc3545, #c82333);
    color: white;
    border: none;
    padding: 4px 8px;
    border-radius: 3px;
    font-size: 11px;
    transition: all 0.2s;
}

.btn-remove-test:hover {
    background: linear-gradient(135deg, #c82333, #bd2130);
    transform: scale(1.05);
    color: white;
}

.billing-summary {
    background: linear-gradient(135deg, #e8f4fd, #d1ecf1);
    border: 2px solid #bee5eb;
    border-radius: 8px;
    padding: 20px;
    margin-top: 20px;
    box-shadow: 0 2px 4px rgba(0,0,0,0.1);
}

.billing-summary h4 {
    color: #0c5460;
    margin-bottom: 15px;
    font-weight: 600;
}

.summary-row {
    display: flex;
    justify-content: space-between;
    margin-bottom: 10px;
    padding: 5px 0;
    font-size: 14px;
}

.summary-row span:first-child {
    color: #495057;
    font-weight: 500;
}

.summary-row span:last-child {
    color: #28a745;
    font-weight: 600;
}

.total-amount {
    font-size: 18px;
    font-weight: bold;
    color: #0c5460;
    border-top: 2px solid #3c8dbc;
    padding-top: 15px;
    margin-top: 15px;
    background: rgba(255,255,255,0.8);
    padding: 15px;
    border-radius: 5px;
}

.category-count {
    background: rgba(255,255,255,0.9);
    color: #3c8dbc;
    padding: 3px 10px;
    border-radius: 15px;
    font-size: 11px;
    font-weight: 600;
    margin-left: 10px;
    box-shadow: 0 1px 2px rgba(0,0,0,0.1);
}

.empty-cart {
    text-align: center;
    color: #6c757d;
    padding: 40px 20px;
    font-style: italic;
}

.empty-cart i {
    color: #dee2e6;
    margin-bottom: 15px;
}

.form-control:focus {
    border-color: #3c8dbc;
    box-shadow: inset 0 1px 1px rgba(0,0,0,.075), 0 0 0 3px rgba(60,141,188,.1);
}

.alert-custom {
    position: fixed;
    top: 80px;
    right: 20px;
    z-index: 9999;
    min-width: 350px;
    box-shadow: 0 4px 12px rgba(0,0,0,0.15);
    border: none;
    border-radius: 6px;
}

.patient-info-card {
    background: linear-gradient(135deg, #d4edda, #c3e6cb);
    border: 1px solid #c3e6cb;
    border-radius: 6px;
    padding: 15px;
    margin-bottom: 20px;
}

.search-highlight {
    background-color: #fff3cd;
    padding: 2px 4px;
    border-radius: 3px;
}

.priority-badge {
    display: inline-block;
    padding: 2px 6px;
    border-radius: 3px;
    font-size: 10px;
    font-weight: bold;
    text-transform: uppercase;
}

.priority-normal { background-color: #28a745; color: white; }
.priority-urgent { background-color: #ffc107; color: #212529; }
.priority-stat { background-color: #dc3545; color: white; }

.loading-overlay {
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: rgba(255,255,255,0.8);
    display: flex;
    align-items: center;
    justify-content: center;
    z-index: 1000;
    border-radius: 5px;
}

.spinner {
    border: 3px solid #f3f3f4;
    border-top: 3px solid #3c8dbc;
    border-radius: 50%;
    width: 30px;
    height: 30px;
    animation: spin 1s linear infinite;
}

@keyframes spin {
    0% { transform: rotate(0deg); }
    100% { transform: rotate(360deg); }
}

/* Fingerprint Verification Styles */
.verification-section {
    background: linear-gradient(135deg, #fff9e6, #fff3cd);
    border: 2px solid #ffc107;
    border-radius: 8px;
    padding: 20px;
    margin-bottom: 20px;
    box-shadow: 0 2px 4px rgba(0,0,0,0.1);
}

.verification-status {
    padding: 15px;
    border-radius: 6px;
    margin-bottom: 15px;
    text-align: center;
}

.status-info {
    background: #d1ecf1;
    border: 1px solid #bee5eb;
    color: #0c5460;
}

.status-success {
    background: #d4edda;
    border: 1px solid #c3e6cb;
    color: #155724;
}

.status-warning {
    background: #fff3cd;
    border: 1px solid #ffeaa7;
    color: #856404;
}

.status-error {
    background: #f8d7da;
    border: 1px solid #f5c6cb;
    color: #721c24;
}

.fingerprint-container {
    text-align: center;
    margin: 20px 0;
}

.fingerprint-image {
    max-width: 150px;
    max-height: 150px;
    border: 2px solid #ddd;
    border-radius: 8px;
    box-shadow: 0 2px 4px rgba(0,0,0,0.1);
}

.biometric-status {
    padding: 15px;
    background: #f8f9fa;
    border-radius: 6px;
    border: 1px solid #dee2e6;
}

.verification-actions {
    display: flex;
    gap: 10px;
    justify-content: center;
    flex-wrap: wrap;
}

.btn-verification {
    background: linear-gradient(135deg, #007bff, #0056b3);
    border: none;
    color: white;
    padding: 8px 16px;
    border-radius: 4px;
    transition: all 0.2s;
}

.btn-verification:hover {
    background: linear-gradient(135deg, #0056b3, #004085);
    color: white;
    transform: translateY(-1px);
}

.btn-manual {
    background: linear-gradient(135deg, #ffc107, #e0a800);
    border: none;
    color: #212529;
    padding: 8px 16px;
    border-radius: 4px;
    transition: all 0.2s;
}

.btn-manual:hover {
    background: linear-gradient(135deg, #e0a800, #d39e00);
    color: #212529;
    transform: translateY(-1px);
}

.verification-complete {
    border-color: #28a745 !important;
    background: linear-gradient(135deg, #d4edda, #c3e6cb) !important;
}

.verification-badge {
    display: inline-flex;
    align-items: center;
    gap: 5px;
    padding: 5px 10px;
    background: rgba(40, 167, 69, 0.1);
    border: 1px solid #28a745;
    border-radius: 15px;
    font-size: 12px;
    margin-top: 10px;
}

/* Responsive Design */
@media (max-width: 768px) {
    .test-item {
        flex-direction: column;
        align-items: flex-start;
        padding: 10px;
    }
    
    .test-price {
        margin: 5px 0;
        text-align: left;
    }
    
    .cart-item {
        flex-direction: column;
        align-items: flex-start;
    }
    
    .summary-row {
        font-size: 13px;
    }
    
    .verification-actions {
        flex-direction: column;
    }
}

/* Scroll Styling */
.category-tests::-webkit-scrollbar {
    width: 6px;
}

.category-tests::-webkit-scrollbar-track {
    background: #f1f1f1;
    border-radius: 3px;
}

.category-tests::-webkit-scrollbar-thumb {
    background: #3c8dbc;
    border-radius: 3px;
}

.category-tests::-webkit-scrollbar-thumb:hover {
    background: #367fa9;
}
</style>

<style>
/* Workflow-based verification styling */
.verification-mandatory {
    border-color: #dc3545 !important;
    background: linear-gradient(135deg, #fff5f5, #fee) !important;
}

.verification-optional {
    border-color: #28a745 !important;
    background: linear-gradient(135deg, #f8fff8, #efd) !important;
}

.workflow-exemption-badge {
    background: rgba(40, 167, 69, 0.1);
    border: 1px solid #28a745;
    border-radius: 15px;
    padding: 8px 12px;
    margin-top: 10px;
    text-align: center;
}

.pulse-highlight {
    animation: pulse-border 2s;
}

@keyframes pulse-border {
    0% { border-color: #3c8dbc; }
    50% { border-color: #dc3545; box-shadow: 0 0 10px rgba(220, 53, 69, 0.5); }
    100% { border-color: #3c8dbc; }
}

.workflow-badge {
    display: inline-block;
    padding: 2px 6px;
    border-radius: 3px;
    font-size: 10px;
    font-weight: bold;
    text-transform: uppercase;
    margin-left: 5px;
}

.workflow-gcc { 
    background-color: #dc3545; 
    color: white; 
}

.workflow-general { 
    background-color: #28a745; 
    color: white; 
}

.workflow-specialized { 
    background-color: #ffc107; 
    color: #212529; 
}
</style>


<div class="content-wrapper">
    <!-- Content Header -->
    <section class="content-header">
        <h1>
            <i class="fa fa-flask"></i> Laboratory Test Selection
            <small>Select and manage laboratory tests efficiently with biometric verification</small>
        </h1>
        <ol class="breadcrumb">
            <li><a href="<?php echo base_url(); ?>admin/admin/dashboard"><i class="fa fa-dashboard"></i> Home</a></li>
            <li><a href="<?php echo base_url(); ?>admin/laboratory">Laboratory</a></li>
            <li class="active">Test Selection</li>
        </ol>
    </section>

    <!-- Main content -->
    <section class="content">
        <div class="row">
            <!-- Patient Selection Panel -->
            <div class="col-md-12">
                <div class="box box-primary">
                    <div class="box-header with-border">
                        <h3 class="box-title"><i class="fa fa-user-circle"></i> Patient Selection</h3>
                        <div class="box-tools pull-right">
                            <button type="button" class="btn btn-box-tool" data-widget="collapse">
                                <i class="fa fa-minus"></i>
                            </button>
                        </div>
                    </div>
                    <div class="box-body">
                        <div class="row">
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="patient_search">
                                        <i class="fa fa-search"></i> Search Patient
                                    </label>
                                    <div class="input-group">
                                        <input type="text" class="form-control" id="patient_search" 
                                               placeholder="Search by name, phone, or patient ID">
                                        <span class="input-group-btn">
                                            <button class="btn btn-primary" type="button" id="search_patient_btn">
                                                <i class="fa fa-search"></i> Search
                                            </button>
                                        </span>
                                    </div>
                                    <small class="text-muted">Enter at least 2 characters to search</small>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="selected_patient">
                                        <i class="fa fa-user"></i> Selected Patient
                                    </label>
                                    <select class="form-control select2" id="selected_patient" name="patient_id">
                                        <option value="">-- Select a patient --</option>
                                        <?php if (!empty($patients)): ?>
                                            <?php foreach ($patients as $patient): ?>
                                                <option value="<?php echo $patient['id']; ?>" 
                                                        data-phone="<?php echo $patient['mobileno']; ?>"
                                                        data-age="<?php echo $patient['age']; ?>"
                                                        data-gender="<?php echo $patient['gender']; ?>"
                                                        data-unique-id="<?php echo $patient['patient_unique_id'] ?? ''; ?>"
                                                        data-has-fingerprint="<?php echo !empty($patient['fingerprint_template']) ? '1' : '0'; ?>">
                                                    <?php echo $patient['patient_name']; ?> - <?php echo $patient['mobileno']; ?>
                                                </option>
                                            <?php endforeach; ?>
                                        <?php endif; ?>
                                    </select>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Patient Info Display -->
                        <div id="patient_info" class="patient-info-card" style="display: none;">
                            <div class="row">
                                <div class="col-md-3">
                                    <strong><i class="fa fa-user"></i> Name:</strong><br>
                                    <span id="patient_name_display"></span>
                                </div>
                                <div class="col-md-2">
                                    <strong><i class="fa fa-phone"></i> Phone:</strong><br>
                                    <span id="patient_phone_display"></span>
                                </div>
                                <div class="col-md-2">
                                    <strong><i class="fa fa-birthday-cake"></i> Age:</strong><br>
                                    <span id="patient_age_display"></span>
                                </div>
                                <div class="col-md-2">
                                    <strong><i class="fa fa-venus-mars"></i> Gender:</strong><br>
                                    <span id="patient_gender_display"></span>
                                </div>
                                <div class="col-md-3">
                                    <strong><i class="fa fa-id-card"></i> Patient ID:</strong><br>
                                    <span id="patient_unique_id_display"></span>
                                </div>
                            </div>
                            <div class="row" style="margin-top: 10px;">
                                <div class="col-md-12">
                                    <strong><i class="fa fa-fingerprint"></i> Biometric Status:</strong>
                                    <span id="fingerprint_availability_display"></span>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Patient Verification Section -->
            <div id="patient_verification_section" class="col-md-12" style="display: none;">
                <div class="box box-warning">
                    <div class="box-header with-border">
                        <h3 class="box-title"><i class="fa fa-shield"></i> Patient Verification</h3>
                        <div class="box-tools pull-right">
                            <span id="secugen_connection_status" class="label label-info">Checking...</span>
                            <button type="button" class="btn btn-box-tool" data-widget="collapse">
                                <i class="fa fa-minus"></i>
                            </button>
                        </div>
                    </div>
                    <div class="box-body">
                        <div class="verification-section">
                            <div class="row">
                                <div class="col-md-6">
                                    <div id="verification_status" class="verification-status status-info">
                                        <i class="fa fa-info-circle"></i> 
                                        <strong>Patient verification required</strong>
                                        <br><small>Please verify patient identity before proceeding with lab tests</small>
                                    </div>
                                    <div class="verification-actions" style="margin-top: 15px;">
                                        <button type="button" class="btn btn-verification" id="verifyPatientBtn">
                                            <i class="fa fa-fingerprint"></i> Verify with Fingerprint
                                        </button>
                                        <button type="button" class="btn btn-manual" id="skipVerificationBtn">
                                            <i class="fa fa-hand-pointer-o"></i> Manual Fingerprint Verification
                                        </button>
                                        <button type="button" class="btn btn-danger" id="clearFingerprint" style="display: none;">
                                            <i class="fa fa-trash"></i> Clear Capture
                                        </button>
                                    </div>
                                    <div class="alert alert-info" style="margin-top: 15px;">
                                        <i class="fa fa-info-circle"></i>
                                        <strong>Note:</strong> Patient verification is mandatory for lab test orders. 
                                        Both verification methods use fingerprint capture and matching against stored templates.
                                    </div>
                                </div>
                                <div class="col-md-6">
                                    <div id="fingerprint_status" class="biometric-status">
                                        <i class="fa fa-fingerprint"></i> 
                                        <strong>Ready for fingerprint capture</strong>
                                        <br><small>Place finger on scanner when prompted</small>
                                    </div>
                                    <div class="fingerprint-container">
                                        <img id="patient_fingerprint_display" 
                                             src="<?php echo base_url(); ?>Images/PlaceFinger.bmp" 
                                             alt="Fingerprint Preview" 
                                             class="fingerprint-image">
                                        <div id="template_info" style="display: none; margin-top: 10px;">
                                            <small class="text-muted">Quality: <span id="fingerprint_quality"></span>%</small>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            <!-- Hidden form fields for fingerprint data -->
                            <input type="hidden" id="fingerprint_template" name="fingerprint_template">
                            <input type="hidden" id="fingerprint_image" name="fingerprint_image">
                            <input type="hidden" id="fingerprint_verified" name="fingerprint_verified" value="0">
                            <input type="hidden" id="verification_method" name="verification_method" value="">
                            <input type="hidden" id="verification_timestamp" name="verification_timestamp">
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <div class="row">
            <!-- Test Selection Panel -->
            <div class="col-md-8">
                <div class="nav-tabs-custom">
                    <ul class="nav nav-tabs pull-right">
                        <li class="active">
                            <a href="#tab_test_selection" data-toggle="tab">
                                <i class="fa fa-list-ul"></i> Select Tests
                            </a>
                        </li>
                        <li class="pull-left header">
                            <i class="fa fa-flask"></i> Available Laboratory Tests 
                            <span class="badge bg-blue" id="total_tests_count">0</span>
                        </li>
                    </ul>
                    <div class="tab-content">
                        <div class="tab-pane active" id="tab_test_selection">
                            <div class="box-body">
                                <!-- Search Bar -->
                                <div class="form-group">
                                    <div class="input-group">
                                        <input type="text" class="form-control" id="testSearch" 
                                               placeholder="Search for specific tests...">
                                        <span class="input-group-addon">
                                            <i class="fa fa-search"></i>
                                        </span>
                                    </div>
                                    <small class="text-muted">Type to filter tests across all categories</small>
                                </div>

                                <!-- Loading Indicator -->
                                <div id="tests_loading" class="text-center" style="padding: 40px;">
                                    <div class="spinner"></div>
                                    <p style="margin-top: 10px;">Loading laboratory tests...</p>
                                </div>

                                <!-- Test Categories -->
                                <div id="testCategories" style="display: none;">
                                    <!-- Categories will be populated here -->
                                </div>

                                <!-- No Results Message -->
                                <div id="no_results" style="display: none; text-align: center; padding: 40px; color: #6c757d;">
                                    <i class="fa fa-search fa-3x" style="margin-bottom: 15px; opacity: 0.3;"></i>
                                    <h4>No tests found</h4>
                                    <p>Try adjusting your search terms</p>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Cart and Billing Panel -->
            <div class="col-md-4">
                <div class="box box-success">
                    <div class="box-header with-border">
                        <h3 class="box-title">
                            <i class="fa fa-shopping-cart"></i> Selected Tests
                            <span class="badge bg-green" id="cartCount">0</span>
                        </h3>
                        <div class="box-tools pull-right">
                            <button type="button" class="btn btn-box-tool" data-widget="collapse">
                                <i class="fa fa-minus"></i>
                            </button>
                        </div>
                    </div>
                    <div class="box-body" style="position: relative;">
                        <!-- Loading Overlay for Cart -->
                        <div id="cart_loading" class="loading-overlay" style="display: none;">
                            <div class="spinner"></div>
                        </div>

                        <div id="selectedTests">
                            <div class="empty-cart">
                                <i class="fa fa-shopping-cart fa-3x"></i>
                                <p>No tests selected yet</p>
                                <small class="text-muted">Verify patient identity first, then add tests from the categories on the left</small>
                            </div>
                        </div>

                        <!-- Billing Summary -->
                        <div class="billing-summary" id="billingSummary" style="display: none;">
                            <h4><i class="fa fa-calculator"></i> Billing Summary</h4>
                            
                            <div class="summary-row">
                                <span>Subtotal:</span>
                                <span id="subtotal">KES 0</span>
                            </div>
                            
                            <div class="form-group" style="margin-top: 15px;">
                                <label for="discountPercent">
                                    <i class="fa fa-percent"></i> Discount (%)
                                </label>
                                <input type="number" class="form-control" id="discountPercent" 
                                       min="0" max="100" value="0" placeholder="0" step="0.1">
                            </div>
                            
                            <div class="summary-row">
                                <span>Discount Amount:</span>
                                <span id="discountAmount">KES 0</span>
                            </div>
                            
                            <div class="form-group">
                                <label for="taxPercent">
                                    <i class="fa fa-money"></i> Tax (%)
                                </label>
                                <input type="number" class="form-control" id="taxPercent" 
                                       min="0" max="100" value="16" placeholder="16" step="0.1">
                            </div>
                            
                            <div class="summary-row">
                                <span>Tax Amount:</span>
                                <span id="taxAmount">KES 0</span>
                            </div>
                            
                            <div class="summary-row total-amount">
                                <span><i class="fa fa-money"></i> Total Amount:</span>
                                <span id="totalAmount">KES 0</span>
                            </div>
                        </div>

                        <!-- Additional Options -->
                        <div id="labOptions" style="display: none; margin-top: 20px;">
                            <h4><i class="fa fa-cogs"></i> Test Options</h4>
                            
                            <div class="form-group">
                                <label for="test_priority">
                                    <i class="fa fa-exclamation-triangle"></i> Priority
                                </label>
                                <select class="form-control" id="test_priority">
                                    <option value="normal">Normal</option>
                                    <option value="urgent">Urgent (+50% fee)</option>
                                    <option value="stat">STAT (+100% fee)</option>
                                </select>
                            </div>
                            
                            <div class="form-group">
                                <label for="assigned_doctor">
                                    <i class="fa fa-user-md"></i> Requesting Doctor
                                </label>
                                <select class="form-control" id="assigned_doctor">
                                    <option value="">-- Loading doctors... --</option>
                                </select>
                            </div>
                            
                            <div class="form-group">
                                <label for="test_date">
                                    <i class="fa fa-calendar"></i> Test Date
                                </label>
                                <input type="date" class="form-control" id="test_date" 
                                       value="<?php echo date('Y-m-d'); ?>" min="<?php echo date('Y-m-d'); ?>">
                            </div>
                            
                            <div class="form-group">
                                <label for="payment_method">
                                    <i class="fa fa-credit-card"></i> Payment Method
                                </label>
                                <select class="form-control" id="payment_method">
                                    <option value="cash">Cash</option>
                                    <option value="card">Card</option>
                                    <option value="insurance">Insurance</option>
                                    <option value="credit">Credit</option>
                                </select>
                            </div>
                            
                            <div class="form-group">
                                <label for="lab_notes">
                                    <i class="fa fa-sticky-note"></i> Additional Notes
                                </label>
                                <textarea class="form-control" id="lab_notes" rows="3" 
                                          placeholder="Enter any special instructions or notes..."></textarea>
                            </div>
                        </div>

                        <!-- Action Buttons -->
                        <div style="margin-top: 20px;">
                            <button class="btn btn-warning btn-block" id="clearCart" 
                                    style="display: none;">
                                <i class="fa fa-trash"></i> Clear All Tests
                            </button>
                            <button class="btn btn-success btn-block" id="saveLabOrder" 
                                    style="display: none; margin-top: 10px;" disabled>
                                <i class="fa fa-save"></i> Save Laboratory Order
                            </button>
                        </div>
                    </div>
                </div>

                <!-- Quick Instructions -->
               <div class="box box-info">
                   <div class="box-header with-border">
                       <h3 class="box-title">
                           <i class="fa fa-lightbulb-o"></i> Quick Guide
                       </h3>
                   </div>
                   <div class="box-body">
                       <div class="list-group">
                           <div class="list-group-item">
                               <i class="fa fa-check text-green"></i> 
                               <strong>Step 1:</strong> Select a patient
                           </div>
                           <div class="list-group-item">
                               <i class="fa fa-check text-yellow"></i> 
                               <strong>Step 2:</strong> Verify patient identity
                           </div>
                           <div class="list-group-item">
                               <i class="fa fa-check text-green"></i> 
                               <strong>Step 3:</strong> Browse tests by category
                           </div>
                           <div class="list-group-item">
                               <i class="fa fa-check text-green"></i> 
                               <strong>Step 4:</strong> Add tests to cart
                           </div>
                           <div class="list-group-item">
                               <i class="fa fa-check text-green"></i> 
                               <strong>Step 5:</strong> Configure options & save
                           </div>
                       </div>
                       
                       <div style="margin-top: 15px; padding: 10px; background: #f8f9fa; border-radius: 4px;">
                           <small class="text-muted">
                               <i class="fa fa-info-circle"></i> 
                               <strong>Security:</strong> Biometric verification ensures patient safety and compliance
                           </small>
                       </div>
                   </div>
               </div>

               <!-- Statistics Box -->
               <div class="box box-primary">
                   <div class="box-header with-border">
                       <h3 class="box-title">
                           <i class="fa fa-bar-chart"></i> Session Stats
                       </h3>
                   </div>
                   <div class="box-body">
                       <div class="row">
                           <div class="col-xs-6">
                               <div class="description-block border-right">
                                   <span class="description-percentage text-green">
                                       <i class="fa fa-flask"></i>
                                   </span>
                                   <h5 class="description-header" id="session_tests">0</h5>
                                   <span class="description-text">Tests Added</span>
                               </div>
                           </div>
                           <div class="col-xs-6">
                               <div class="description-block">
                                   <span class="description-percentage text-blue">
                                       <i class="fa fa-money"></i>
                                   </span>
                                   <h5 class="description-header" id="session_amount">KES 0</h5>
                                   <span class="description-text">Total Value</span>
                               </div>
                           </div>
                       </div>
                   </div>
               </div>
           </div>
       </div>
   </section>
</div>

<!-- Confirmation Modal -->
<div class="modal fade" id="confirmModal" tabindex="-1" role="dialog">
   <div class="modal-dialog modal-lg" role="document">
       <div class="modal-content">
           <div class="modal-header bg-primary">
               <button type="button" class="close" data-dismiss="modal" aria-label="Close">
                   <span aria-hidden="true">&times;</span>
               </button>
               <h4 class="modal-title">
                   <i class="fa fa-check-circle"></i> Confirm Laboratory Order
               </h4>
           </div>
           <div class="modal-body">
               <div class="row">
                   <div class="col-md-6">
                       <h5><i class="fa fa-user"></i> Patient Information</h5>
                       <table class="table table-condensed">
                           <tr><td><strong>Name:</strong></td><td id="confirm_patient_name"></td></tr>
                           <tr><td><strong>Phone:</strong></td><td id="confirm_patient_phone"></td></tr>
                           <tr><td><strong>Age/Gender:</strong></td><td id="confirm_patient_details"></td></tr>
                           <tr><td><strong>Verification:</strong></td><td id="confirm_verification_status"></td></tr>
                       </table>
                   </div>
                   <div class="col-md-6">
                       <h5><i class="fa fa-cog"></i> Order Details</h5>
                       <table class="table table-condensed">
                           <tr><td><strong>Tests:</strong></td><td id="confirm_test_count"></td></tr>
                           <tr><td><strong>Priority:</strong></td><td id="confirm_priority"></td></tr>
                           <tr><td><strong>Total Amount:</strong></td><td id="confirm_total" class="text-green"></td></tr>
                       </table>
                   </div>
               </div>
               
               <h5><i class="fa fa-list"></i> Selected Tests</h5>
               <div id="confirm_tests_list" style="max-height: 200px; overflow-y: auto;"></div>
               
               <div class="alert alert-info">
                   <i class="fa fa-info-circle"></i> 
                   <strong>Note:</strong> This order will be saved and can be processed in the laboratory section.
               </div>
           </div>
           <div class="modal-footer">
               <button type="button" class="btn btn-default" data-dismiss="modal">
                   <i class="fa fa-times"></i> Cancel
               </button>
               <button type="button" class="btn btn-primary" id="finalConfirmBtn">
                   <i class="fa fa-check"></i> Confirm & Save Order
               </button>
           </div>
       </div>
   </div>
</div>

<script>
// Set base URL before any other JavaScript
window.base_url = '<?php echo base_url(); ?>';

// =============================================
// GLOBAL STATE MANAGEMENT
// =============================================

window.laboratoryState = {
   selectedPatient: null,
   selectedPatientData: null,
   verificationStatus: 'pending',
   secugenActive: false,
   liveTemplate: null,
   storedTemplate: null,
   matchScore: 0,
   selectedTests: [],
   isVerified: false,
   verificationMethod: null,
   verificationTimestamp: null
};

// SecuGen configuration matching existing implementations
window.secugenState = {
   serviceActive: false,
   currentProtocol: 'https',
   deviceConnected: false,
   lastError: null,
   retryCount: 0,
   maxRetries: 3,
   endpoints: {
       https: {
           info: 'https://localhost:8443/SGIFPGetInfo',
           capture: 'https://localhost:8443/SGIFPCapture',
           match: 'https://localhost:8443/SGIMatchScore'
       },
       http: {
           info: 'http://localhost:8000/SGIFPGetInfo',
           capture: 'http://localhost:8000/SGIFPCapture',
           match: 'http://localhost:8000/SGIMatchScore'
       }
   },
   patientVerification: {
       isVerified: false,
       verificationMethod: null,
       verificationTimestamp: null,
       patientId: null,
       verificationScore: 0,
       attempts: 0,
       maxAttempts: 3
   }
};

// Global variables from original working code
var selectedTests = [];
var allTestsData = {};
var isLoading = false;

$(document).ready(function() {
   // Initialize the application
   initializeTestSelection();
   
   // Initialize SecuGen service (delayed like other implementations)
   setTimeout(function() {
       initializeSecuGenService();
   }, 1500);
   
   // Patient selection handlers
   $('#selected_patient').on('change', function() {
       handlePatientSelection();
   });
   
   // Patient search functionality
   $('#search_patient_btn').on('click', function() {
       searchPatients();
   });
   
   $('#patient_search').on('keypress', function(e) {
       if (e.which === 13) {
           searchPatients();
       }
   });
   
   // Verification button handlers
   $('#verifyPatientBtn').on('click', function() {
       startPatientVerification();
   });
   
   $('#skipVerificationBtn').on('click', function() {
       showManualVerificationModal();
   });
   
   $('#clearFingerprint').on('click', function() {
       clearFingerprintCapture();
   });
   
   // Save lab order
   $('#saveLabOrder').on('click', function() {
       showConfirmationModal();
   });
   
   // Final confirmation
   $('#finalConfirmBtn').on('click', function() {
       saveLabOrder();
   });
   
   // Initialize select2 if available
   if ($.fn.select2) {
       $('#selected_patient').select2({
           placeholder: "Select a patient",
           allowClear: true
       });
   }
   
   console.log('Laboratory Test Selection with Biometric Verification loaded');
});

// =============================================
// CORE FUNCTIONS FROM ORIGINAL WORKING CODE
// =============================================

function initializeTestSelection() {
   loadTestCategories();
   loadDoctors(); // Load doctors from database
   bindTestSelectionEvents();
   updateSessionStats();
}

function handlePatientSelection() {
    var selectedOption = $('#selected_patient').find(':selected');
    var patientId = selectedOption.val();
    
    console.log('👤 Patient selection changed');
    console.log('📊 Patient ID:', patientId);
    
    if (patientId) {
        var hasFingerprint = selectedOption.data('has-fingerprint') == 1 || selectedOption.data('has-fingerprint') === true;
        var patientName = selectedOption.text().split(' - ')[0];
        var patientPhone = selectedOption.data('phone') || '';
        var patientAge = selectedOption.data('age') || 'N/A';
        var patientGender = selectedOption.data('gender') || 'N/A';
        var patientUniqueId = selectedOption.data('unique-id') || 'N/A';
        
        // 🔑 GET PATIENT WORKFLOW TYPE
        var patientWorkflow = selectedOption.data('workflow') || selectedOption.data('patient-type') || 'general';
        
        console.log('📊 Patient Name:', patientName);
        console.log('📊 Patient Workflow:', patientWorkflow);
        console.log('📊 Has Fingerprint (frontend):', hasFingerprint);
        
        // Update display fields
        $('#patient_name_display').text(patientName);
        $('#patient_phone_display').text(patientPhone);
        $('#patient_age_display').text(patientAge);
        $('#patient_gender_display').text(patientGender);
        $('#patient_unique_id_display').text(patientUniqueId);
        
        // Store patient data in global state
        window.laboratoryState.selectedPatient = patientId;
        window.laboratoryState.patientWorkflow = patientWorkflow.toLowerCase();
        window.laboratoryState.selectedPatientData = {
            id: patientId,
            name: patientName,
            phone: patientPhone,
            age: patientAge,
            gender: patientGender,
            unique_id: patientUniqueId,
            has_fingerprint: hasFingerprint,
            workflow: patientWorkflow.toLowerCase()
        };
        
        // 🔒 DETERMINE IF VERIFICATION IS REQUIRED BASED ON WORKFLOW
        determineVerificationRequirement(patientWorkflow.toLowerCase(), hasFingerprint);
        
        $('#patient_info').slideDown();
        
        console.log('✅ Patient selection completed');
        
    } else {
        console.log('🚫 No patient selected');
        $('#patient_info').slideUp();
        $('#patient_verification_section').slideUp();
        $('#labOptions').slideUp();
        resetPatientVerification();
        window.laboratoryState.selectedPatient = null;
        window.laboratoryState.selectedPatientData = null;
        window.laboratoryState.patientWorkflow = null;
        window.laboratoryState.requiresVerification = false;
    }
}

function searchPatients() {
   var searchTerm = $('#patient_search').val().trim();
   
   if (searchTerm.length < 2) {
       showAlert('Please enter at least 2 characters to search', 'warning');
       return;
   }
   
   $.ajax({
       url: window.base_url + 'admin/laboratory/searchPatients',
       type: 'POST',
       data: { search_term: searchTerm },
       dataType: 'json',
       beforeSend: function() {
           $('#search_patient_btn').prop('disabled', true).html('<i class="fa fa-spinner fa-spin"></i> Searching...');
       },
       success: function(response) {
           if (response.status === 'success') {
               updatePatientDropdown(response.data);
               showAlert('Found ' + response.count + ' patients', 'success');
           } else {
               showAlert(response.message, 'error');
           }
       },
       error: function() {
           showAlert('Error searching patients', 'error');
       },
       complete: function() {
           $('#search_patient_btn').prop('disabled', false).html('<i class="fa fa-search"></i> Search');
       }
   });
}

function updatePatientDropdown(patients) {
    var patientSelect = $('#selected_patient');
    patientSelect.empty().append('<option value="">-- Select a patient --</option>');
    
    $.each(patients, function(index, patient) {
        var hasFingerprint = patient.has_fingerprint == 1 || patient.has_fingerprint === true || patient.has_fingerprint === "1";
        
        // 🔑 GET WORKFLOW INFO
        var workflow = patient.patient_type || patient.workflow || patient.patient_workflow || 'general';
        
        console.log(`Patient ${patient.patient_name}: workflow = ${workflow}, has_fingerprint = ${patient.has_fingerprint}`);
        
        patientSelect.append(
            $('<option>', {
                value: patient.id,
                text: patient.patient_name + ' - ' + (patient.mobileno || 'No Phone'),
                'data-phone': patient.mobileno || '',
                'data-age': patient.age || '',
                'data-gender': patient.gender || '',
                'data-unique-id': patient.patient_unique_id || '',
                'data-has-fingerprint': hasFingerprint ? 1 : 0,
                'data-workflow': workflow,
                'data-patient-type': workflow
            })
        );
    });
    
    // Refresh select2 if initialized
    if (patientSelect.hasClass('select2-hidden-accessible')) {
        patientSelect.trigger('change');
    }
    
    console.log('Updated patient dropdown with', patients.length, 'patients');
}

window.LAB_DEBUG.testPatientFingerprintStatus = function(patientId = 133) {
    console.log('🧪 Testing patient fingerprint status...');
    
    $.ajax({
        url: window.base_url + 'admin/laboratory/testPatientFingerprintStatus?patient_id=' + patientId,
        type: 'GET',
        success: function(response) {
            console.log('✅ Patient fingerprint status test result:', response);
            
            if (response.status === 'success') {
                console.log('📊 Patient Data:', response.patient_data);
                console.log('🔍 Analysis:', response.analysis);
                console.log('🔗 SQL Query:', response.sql_query);
                
                if (response.analysis.has_fingerprint_calculation == 1) {
                    console.log('✅ Patient SHOULD show as having fingerprint');
                } else {
                    console.log('❌ Patient showing as NO fingerprint - check template data');
                }
            }
        },
        error: function(xhr, status, error) {
            console.error('❌ Test failed:', error);
        }
    });
};

function loadTestCategories() {
   $('#tests_loading').show();
   $('#testCategories').hide();
   
   console.log('Loading laboratory tests from database...');
   
   $.ajax({
       url: window.base_url + 'admin/laboratory/getLabTests',
       type: 'GET',
       dataType: 'json',
       timeout: 30000, // 30 second timeout
       success: function(response) {
           console.log('Lab tests response:', response);
           
           if (response && response.status === 'success' && response.data) {
               allTestsData = response.data;
               renderTestCategories(response.data);
               $('#total_tests_count').text(response.count || 0);
               console.log('Successfully loaded', response.count || 0, 'laboratory tests from database');
           } else {
               console.error('Invalid lab tests response:', response);
               showAlert('Error loading tests: ' + (response ? response.message : 'Invalid response'), 'error');
               renderTestCategories(getStaticTestData());
           }
       },
       error: function(xhr, status, error) {
           console.error('Lab tests loading error:', xhr, status, error);
           console.error('Response text:', xhr.responseText);
           
           showAlert('Failed to load tests from database: ' + error, 'error');
           renderTestCategories(getStaticTestData());
       },
       complete: function() {
           $('#tests_loading').hide();
           $('#testCategories').show();
       }
   });
}

function getStaticTestData() {
   // Fallback static data from original working code
   return {
       "Hematology": [
           { "name": "Complete Blood Count (CBC)", "charge": 1200, "method": "Automated", "sample_type": "Blood" },
           { "name": "Blood Culture", "charge": 6000, "method": "Culture", "sample_type": "Blood" },
           { "name": "ESR", "charge": 500, "method": "Westergren", "sample_type": "Blood" },
           { "name": "Hemoglobin", "charge": 500, "method": "Spectrophotometry", "sample_type": "Blood" }
       ],
       "Endocrinology": [
           { "name": "TSH", "charge": 2000, "method": "ELISA", "sample_type": "Serum" },
           { "name": "FT3", "charge": 2000, "method": "ELISA", "sample_type": "Serum" },
           { "name": "FT4", "charge": 2000, "method": "ELISA", "sample_type": "Serum" }
       ],
       "Diabetes": [
           { "name": "HBA1C", "charge": 3000, "method": "HPLC", "sample_type": "Blood" },
           { "name": "Fasting Blood Sugar", "charge": 500, "method": "Enzymatic", "sample_type": "Serum" }
       ]
   };
}

function loadDoctors() {
    console.log('Loading doctors from database...');
    
    $.ajax({
        // ✅ CORRECT: This endpoint exists in Laboratory controller
        url: window.base_url + 'admin/laboratory/getDoctors',
        type: 'GET',
        dataType: 'json',
        timeout: 15000,
        success: function(response) {
            console.log('Doctors response:', response);
            
            // ✅ FIXED: Handle correct response format (doctors, not data)
            if (response && response.status === 'success' && response.doctors && Array.isArray(response.doctors)) {
                populateDoctorsDropdown(response.doctors);
                console.log('Successfully loaded', response.doctors.length, 'doctors from database');
            } else {
                console.error('Invalid doctors response:', response);
                showAlert('Error loading doctors: ' + (response ? response.message : 'Invalid response'), 'warning');
                $('#assigned_doctor').html('<option value="">-- No doctors available --</option>');
            }
        },
        error: function(xhr, status, error) {
            console.error('Doctors loading error:', xhr, status, error);
            console.error('Response text:', xhr.responseText);
            
            // ✅ Better error handling
            let errorMessage = 'Failed to load doctors from database: ';
            
            if (xhr.status === 404) {
                errorMessage += 'Endpoint not found (check controller method exists)';
            } else if (xhr.status === 500) {
                errorMessage += 'Server error (check PHP error logs)';
            } else {
                errorMessage += error;
            }
            
            showAlert(errorMessage, 'error');
            $('#assigned_doctor').html('<option value="">-- Error loading doctors --</option>');
        }
    });
}

function populateDoctorsDropdown(doctors) {
    var doctorSelect = $('#assigned_doctor');
    doctorSelect.empty().append('<option value="">-- Select Doctor --</option>');
    
    $.each(doctors, function(index, doctor) {
        var doctorName = '';
        
        // ✅ Handle the actual response format from Laboratory controller
        if (doctor.full_name) {
            doctorName = doctor.full_name;
        } else if (doctor.name && doctor.surname) {
            doctorName = doctor.name + ' ' + doctor.surname;
        } else if (doctor.name) {
            doctorName = doctor.name;
        } else {
            doctorName = 'Dr. ' + (doctor.employee_id || doctor.id);
        }
        
        // ✅ Add "Dr." prefix if not present
        if (!doctorName.toLowerCase().includes('dr')) {
            doctorName = 'Dr. ' + doctorName;
        }
        
        doctorSelect.append(
            $('<option>', {
                value: doctor.id,
                text: doctorName + (doctor.employee_id ? ' (' + doctor.employee_id + ')' : ''),
                'data-employee-id': doctor.employee_id || '',
                'data-designation': doctor.designation || 'Doctor/Staff'
            })
        );
    });
    
    console.log('Populated doctors dropdown with', doctors.length, 'entries');
}

function renderTestCategories(testData) {
   var container = $('#testCategories');
   container.empty();

   if ($.isEmptyObject(testData)) {
       container.html('<div class="alert alert-warning"><i class="fa fa-exclamation-triangle"></i> No test categories available</div>');
       return;
   }

   $.each(testData, function(category, tests) {
       var categoryCard = createCategoryCard(category, tests);
       container.append(categoryCard);
   });
   
   // Initialize collapse functionality
   container.find('.category-header').on('click', function() {
       var target = $(this).data('target');
       $(target).collapse('toggle');
       
       var icon = $(this).find('.fa-chevron-down, .fa-chevron-right');
       setTimeout(function() {
           if ($(target).hasClass('in')) {
               icon.removeClass('fa-chevron-right').addClass('fa-chevron-down');
           } else {
               icon.removeClass('fa-chevron-down').addClass('fa-chevron-right');
           }
       }, 350);
   });
}

function createCategoryCard(category, tests) {
   var categoryId = category.replace(/\s+/g, '_').toLowerCase();
   
   var categoryCard = $('<div>', {
       'class': 'test-category-card',
       'data-category': category
   });
   
   var header = $('<div>', {
       'class': 'category-header',
       'data-toggle': 'collapse',
       'data-target': '#' + categoryId,
       html: '<i class="fa fa-chevron-down pull-right"></i>' +
             '<strong>' + category + '</strong>' +
             '<span class="category-count">' + tests.length + ' tests</span>'
   });
   
   var collapseDiv = $('<div>', {
       'id': categoryId,
       'class': 'collapse'
   });
   
   var testsDiv = $('<div>', {
       'class': 'category-tests'
   });
   
   $.each(tests, function(index, test) {
       var testItem = createTestItem(test, category);
       testsDiv.append(testItem);
   });
   
   collapseDiv.append(testsDiv);
   categoryCard.append(header).append(collapseDiv);
   
   return categoryCard;
}

function createTestItem(test, category) {
   var testItem = $('<div>', {
       'class': 'test-item',
       'data-test-name': test.name.toLowerCase(),
       'data-category': category
   });
   
   var testName = $('<div>', {
       'class': 'test-name',
       html: test.name + (test.method ? '<br><small class="text-muted">' + test.method + ' | ' + test.sample_type + '</small>' : '')
   });
   
   var testPrice = $('<div>', {
       'class': 'test-price',
       text: 'KES ' + test.charge.toLocaleString()
   });
   
   var isSelected = selectedTests.some(selected => 
       selected.name === test.name && selected.category === category
   );
   
   var addButton = $('<button>', {
       'class': 'btn-add-test',
       text: isSelected ? 'Added' : 'Add',
       disabled: isSelected,
       'data-test': JSON.stringify({...test, category: category})
   });
   
   testItem.append(testName).append(testPrice).append(addButton);
   return testItem;
}

function bindTestSelectionEvents() {
   // Use event delegation for dynamically created elements
   $(document).on('click', '.btn-add-test', function() {
       if ($(this).prop('disabled')) return;
       
       // Check if patient is verified before allowing test selection
       if (!window.laboratoryState.isVerified) {
           showAlert('Please verify patient identity before adding tests', 'warning');
           return;
       }
       
       var testData = JSON.parse($(this).attr('data-test'));
       addTestToCart(testData);
       $(this).text('Added').prop('disabled', true).addClass('disabled');
   });

   $(document).on('click', '.btn-remove-test', function() {
       var testIndex = parseInt($(this).attr('data-index'));
       removeTestFromCart(testIndex);
   });

   $('#testSearch').on('input', function() {
       var searchTerm = $(this).val().toLowerCase();
       filterTests(searchTerm);
   });

   $('#discountPercent, #taxPercent, #test_priority').on('input change', function() {
       updateBillingSummary();
   });

   $('#clearCart').on('click', function() {
       if (confirm('Are you sure you want to clear all selected tests?')) {
           clearCart();
       }
   });
}

function addTestToCart(test) {
   var existingIndex = selectedTests.findIndex(selected => 
       selected.name === test.name && selected.category === test.category
   );

   if (existingIndex === -1) {
       selectedTests.push({
           name: test.name,
           charge: test.charge,
           category: test.category,
           method: test.method || '',
           sample_type: test.sample_type || '',
           id: Date.now() + Math.random()
       });
       
       updateCartDisplay();
       updateBillingSummary();
       updateSessionStats();
       showAlert(test.name + ' added to cart', 'success');
   }
}

function removeTestFromCart(index) {
   var removedTest = selectedTests.splice(index, 1)[0];
   updateCartDisplay();
   updateBillingSummary();
   updateSessionStats();
   
   // Re-enable the add button
   $('.btn-add-test').each(function() {
       try {
           var testData = JSON.parse($(this).attr('data-test'));
           if (testData.name === removedTest.name && testData.category === removedTest.category) {
               $(this).text('Add').prop('disabled', false).removeClass('disabled');
           }
       } catch (e) {
           // Ignore parsing errors
       }
   });
   
   showAlert(removedTest.name + ' removed from cart', 'info');
}

function updateCartDisplay() {
   var container = $('#selectedTests');
   var cartCount = $('#cartCount');
   var clearBtn = $('#clearCart');
   var saveBtn = $('#saveLabOrder');
   var billingSummary = $('#billingSummary');

   cartCount.text(selectedTests.length);

   if (selectedTests.length === 0) {
       var emptyMessage = window.laboratoryState.isVerified ? 
           '<div class="empty-cart">' +
           '<i class="fa fa-shopping-cart fa-3x"></i>' +
           '<p>Patient verified - ready to add tests</p>' +
           '<small class="text-muted">Add tests from the categories on the left</small>' +
           '</div>' :
           '<div class="empty-cart">' +
           '<i class="fa fa-shopping-cart fa-3x"></i>' +
           '<p>No tests selected yet</p>' +
           '<small class="text-muted">Verify patient identity first, then add tests from the categories on the left</small>' +
           '</div>';
       
       container.html(emptyMessage);
       clearBtn.hide();
       saveBtn.hide();
       billingSummary.hide();
   } else {
       var cartHtml = '';
       $.each(selectedTests, function(index, test) {
           cartHtml += 
               '<div class="cart-item">' +
               '<div class="cart-item-name">' + 
               test.name + 
               '<br><small class="text-muted">' + test.category + '</small>' +
               '</div>' +
               '<div class="cart-item-price">KES ' + test.charge.toLocaleString() + '</div>' +
               '<button class="btn btn-xs btn-danger btn-remove-test" data-index="' + index + '" title="Remove test">' +
               '<i class="fa fa-times"></i></button>' +
               '</div>';
       });

       container.html(cartHtml);
       clearBtn.show();
       saveBtn.show();
       billingSummary.show();
   }
}

function updateBillingSummary() {
   if (selectedTests.length === 0) return;

   var subtotal = selectedTests.reduce(function(sum, test) {
       return sum + test.charge;
   }, 0);
   
   // Apply priority fees
   var priority = $('#test_priority').val();
   var priorityMultiplier = 1.0;
   
   if (priority === 'urgent') {
       priorityMultiplier = 1.5;
   } else if (priority === 'stat') {
       priorityMultiplier = 2.0;
   }
   
   var priorityAdjustedTotal = subtotal * priorityMultiplier;
   
   var discountPercent = parseFloat($('#discountPercent').val()) || 0;
   var taxPercent = parseFloat($('#taxPercent').val()) || 0;

   var discountAmount = (priorityAdjustedTotal * discountPercent) / 100;
   var taxableAmount = priorityAdjustedTotal - discountAmount;
   var taxAmount = (taxableAmount * taxPercent) / 100;
   var totalAmount = taxableAmount + taxAmount;

   $('#subtotal').text('KES ' + Math.round(priorityAdjustedTotal).toLocaleString());
   $('#discountAmount').text('KES ' + Math.round(discountAmount).toLocaleString());
   $('#taxAmount').text('KES ' + Math.round(taxAmount).toLocaleString());
   $('#totalAmount').text('KES ' + Math.round(totalAmount).toLocaleString());
}

function updateSessionStats() {
   $('#session_tests').text(selectedTests.length);
   
   var totalAmount = selectedTests.reduce(function(sum, test) {
       return sum + test.charge;
   }, 0);
   
   $('#session_amount').text('KES ' + totalAmount.toLocaleString());
}

function filterTests(searchTerm) {
   if (!searchTerm) {
       $('.test-category-card').show();
       $('.test-item').show();
       $('#no_results').hide();
       return;
   }

   var hasResults = false;
   
   $('.test-category-card').each(function() {
       var $category = $(this);
       var $tests = $category.find('.test-item');
       var hasVisibleTests = false;

       $tests.each(function() {
           var testName = $(this).data('test-name');
           if (testName && testName.includes(searchTerm)) {
               $(this).show();
               hasVisibleTests = true;
               hasResults = true;
               
               // Highlight search term
               var $testNameDiv = $(this).find('.test-name');
               var originalText = $testNameDiv.text();
               var highlightedText = originalText.replace(
                   new RegExp('(' + searchTerm + ')', 'gi'),
                   '<span class="search-highlight">$1</span>'
               );
               $testNameDiv.html(highlightedText);
           } else {
               $(this).hide();
           }
       });

       if (hasVisibleTests) {
           $category.show();
           $category.find('.collapse').addClass('in');
           $category.find('.fa-chevron-right').removeClass('fa-chevron-right').addClass('fa-chevron-down');
       } else {
           $category.hide();
       }
   });
   
   if (!hasResults) {
       $('#no_results').show();
   } else {
       $('#no_results').hide();
   }
}

function clearCart() {
   selectedTests = [];
   updateCartDisplay();
   updateBillingSummary();
   updateSessionStats();
   
   // Re-enable all add buttons
   $('.btn-add-test').text('Add').prop('disabled', false).removeClass('disabled');
   
   showAlert('Cart cleared successfully', 'info');
}

// =============================================
// FINGERPRINT VERIFICATION FUNCTIONS
// =============================================

function initializeSecuGenService() {
   console.log('Initializing SecuGen WebAPI service...');
   updateSecuGenConnectionStatus('checking', 'Initializing SecuGen WebAPI connection...');
   
   detectSecuGenService();
}

function detectSecuGenService() {
   const ports = [8443, 8000, 8080];
   let currentPort = 0;
   
   function tryPort() {
       if (currentPort >= ports.length) {
           handleSecuGenUnavailable('All SecuGen WebAPI ports failed - service not running');
           return;
       }
       
       const port = ports[currentPort];
       const testURL = `https://localhost:${port}/SGIFPGetInfo`;
       
       console.log(`Testing SecuGen WebAPI on port ${port}...`);
       
       $.ajax({
           url: testURL,
           type: 'GET',
           timeout: 3000,
           success: function(response) {
               handleSecuGenDetected(port, response);
           },
           error: function(xhr, status, error) {
               console.log(`Port ${port} failed:`, error);
               currentPort++;
               setTimeout(tryPort, 500);
           }
       });
   }
   
   tryPort();
}

function handleSecuGenDetected(port, response) {
   window.secugenState.serviceActive = true;
   window.secugenState.currentProtocol = 'https';
   window.secugenState.deviceConnected = true;
   
   updateSecuGenConnectionStatus('success', `SecuGen WebAPI Connected on port ${port}`);
   
   console.log(`✓ SecuGen service detected and ready on port ${port}`);
   
   // Parse device info if available
   if (response && typeof response === 'object') {
       console.log(`Device Info:`, response);
   }
   
   enableFingerprintCapture();
}

function handleSecuGenUnavailable(message) {
   window.secugenState.serviceActive = false;
   window.secugenState.deviceConnected = false;
   
   updateSecuGenConnectionStatus('error', message);
   
   console.log('SecuGen service not available - manual verification only');
   
   enableManualMode();
}

function updateSecuGenConnectionStatus(status, message) {
   const statusLabel = $('#secugen_connection_status');
   
   switch (status) {
       case 'checking':
           statusLabel.removeClass().addClass('label label-info').text('Checking...');
           break;
       case 'success':
           statusLabel.removeClass().addClass('label label-success').text('Connected');
           break;
       case 'error':
           statusLabel.removeClass().addClass('label label-danger').text('Not Available');
           break;
   }
   
   $('#fingerprint_status').html('<i class="fa fa-fingerprint"></i> <strong>Ready for fingerprint capture</strong><br><small>' + message + '</small>');
}

function enableFingerprintCapture() {
   $('#verifyPatientBtn').prop('disabled', false)
       .removeClass('btn-warning').addClass('btn-verification')
       .html('<i class="fa fa-fingerprint"></i> Verify with Fingerprint');
   
   console.log('Fingerprint capture enabled - SecuGen device ready');
}

function enableManualMode() {
   $('#verifyPatientBtn').prop('disabled', false)
       .removeClass('btn-verification').addClass('btn-warning')
       .html('<i class="fa fa-hand-pointer-o"></i> Manual Fingerprint Verification Only');
   
   $('#skipVerificationBtn').prop('disabled', false)
       .html('<i class="fa fa-hand-pointer-o"></i> Manual Fingerprint Verification');
   
   console.log('Manual fingerprint verification mode enabled - SecuGen device may be unavailable');
}

function resetPatientVerification() {
   window.secugenState.patientVerification = {
       isVerified: false,
       verificationMethod: null,
       verificationTimestamp: null,
       patientId: null,
       verificationScore: 0,
       attempts: 0,
       maxAttempts: 3
   };
   
   window.laboratoryState.isVerified = false;
   window.laboratoryState.verificationMethod = null;
   window.laboratoryState.liveTemplate = null;
   window.laboratoryState.storedTemplate = null;
   
   $('#labOptions').hide();
   $('#patient_verification_section').removeClass('verification-complete');
   $('.verification-badge').remove();
   $('#saveLabOrder').prop('disabled', true);
   
   updateVerificationStatus('info', 'Patient verification required', 
       'Please verify patient identity before proceeding with lab tests');
   
   // Reset fingerprint display
   $('#patient_fingerprint_display').attr('src', window.base_url + 'Images/PlaceFinger.bmp');
   $('#template_info').hide();
   $('#clearFingerprint').hide();
}

function startPatientVerification() {
   const patientId = $('#selected_patient').val();
   
   if (!patientId) {
       showAlert('Please select a patient first', 'warning');
       return;
   }
   
   // Reset verification state
   window.secugenState.patientVerification.attempts = 0;
   window.secugenState.patientVerification.patientId = patientId;
   
   updateVerificationStatus('info', 'Starting patient verification...', 'Initializing fingerprint scanner');
   
   // Check if SecuGen is available
   if (window.secugenState.serviceActive) {
       performFingerprintVerification();
   } else {
       updateVerificationStatus('warning', 'Fingerprint scanner not available', 'Manual verification required');
       showManualVerificationModal();
   }
}

// Replace your loadStoredFingerprintTemplate function with:
function loadStoredFingerprintTemplate(patientId) {
    console.log('Loading template for patient:', patientId);
    
    $.ajax({
        url: window.base_url + 'admin/laboratory/getStoredFingerprintTemplate',
        type: 'POST',
        data: { patient_id: patientId },
        dataType: 'json',
        success: function(response) {
            console.log('Response:', response);
            
            if (response.status === 'success') {
                window.laboratoryState.storedTemplate = response.fingerprint_template;
                updateVerificationStatus('success', 'Template loaded', 
                    'Ready for verification');
            } else {
                updateVerificationStatus('warning', 'No template', response.message);
            }
        },
        error: function(xhr, status, error) {
            console.error('Error:', error);
            console.error('Response:', xhr.responseText);
            updateVerificationStatus('error', 'Loading failed', error);
        }
    });
}

function performFingerprintVerification() {
   if (!window.laboratoryState.storedTemplate) {
       // No stored template, offer enrollment or manual verification
       showFingerprintEnrollmentOption();
       return;
   }
   
   updateVerificationStatus('info', 'Place finger on scanner', 'Capturing fingerprint for verification');
   
   $('#verifyPatientBtn').prop('disabled', true).html('<i class="fa fa-spinner fa-spin"></i> Scanning...');
   
   const captureUrl = window.secugenState.endpoints[window.secugenState.currentProtocol].capture;
   
   // Use XMLHttpRequest for better control (matching existing implementations)
   const xhr = new XMLHttpRequest();
   xhr.open('POST', captureUrl, true);
   xhr.setRequestHeader('Content-Type', 'application/x-www-form-urlencoded');
   xhr.timeout = 20000;
   
   xhr.onreadystatechange = function() {
       if (xhr.readyState === 4) {
           $('#verifyPatientBtn').prop('disabled', false).html('<i class="fa fa-fingerprint"></i> Verify with Fingerprint');
           
           if (xhr.status === 200) {
               try {
                   const result = JSON.parse(xhr.responseText);
                   handleCaptureResponse(result);
               } catch (e) {
                   handleVerificationError('Invalid response from fingerprint scanner: ' + e.message);
               }
           } else {
               handleVerificationError(`Fingerprint capture failed: HTTP ${xhr.status}`);
           }
       }
   };
   
   xhr.onerror = function() {
       $('#verifyPatientBtn').prop('disabled', false).html('<i class="fa fa-fingerprint"></i> Verify with Fingerprint');
       handleVerificationError('Network error during fingerprint capture');
   };
   
   xhr.ontimeout = function() {
       $('#verifyPatientBtn').prop('disabled', false).html('<i class="fa fa-fingerprint"></i> Verify with Fingerprint');
       handleVerificationError('Fingerprint capture timeout - no finger detected');
   };
   
   // Send capture request with quality parameters
   const postData = 'Quality=50&Timeout=15000&WSQCompression=2.25';
   xhr.send(postData);
   
   console.log('Fingerprint capture initiated for laboratory verification');
}

function handleCaptureResponse(result) {
   if (result.ErrorCode === 0 && result.TemplateBase64) {
       // Capture successful, now compare with stored template
       window.laboratoryState.liveTemplate = result.TemplateBase64;
       
       // Update fingerprint image if available
       if (result.BMPBase64) {
           $('#patient_fingerprint_display').attr('src', 'data:image/bmp;base64,' + result.BMPBase64);
           $('#clearFingerprint').show();
       }
       
       // Show template quality if available
       if (result.Quality) {
           $('#fingerprint_quality').text(result.Quality);
           $('#template_info').show();
       }
       
       updateVerificationStatus('info', 'Comparing fingerprints...', 'Verifying patient identity');
       
       // Compare captured fingerprint with stored template
       compareFingerprints(result.TemplateBase64, window.laboratoryState.storedTemplate);
       
       console.log(`Fingerprint captured successfully - Quality: ${result.Quality || 'N/A'}%`);
   } else {
       handleVerificationError(getSecuGenErrorDescription(result.ErrorCode));
   }
}

function compareFingerprints(capturedTemplate, storedTemplate) {
   const matchUrl = window.secugenState.endpoints[window.secugenState.currentProtocol].match;
   
   const xhr = new XMLHttpRequest();
   xhr.open('POST', matchUrl, true);
   xhr.setRequestHeader('Content-Type', 'application/x-www-form-urlencoded');
   xhr.timeout = 10000;
   
   xhr.onreadystatechange = function() {
       if (xhr.readyState === 4) {
           if (xhr.status === 200) {
               try {
                   const result = JSON.parse(xhr.responseText);
                   handleMatchResponse(result);
               } catch (e) {
                   handleVerificationError('Invalid match response: ' + e.message);
               }
           } else {
               handleVerificationError(`Match service error: HTTP ${xhr.status}`);
           }
       }
   };
   
   xhr.onerror = function() {
       handleVerificationError('Network error during fingerprint matching');
   };
   
   xhr.ontimeout = function() {
       handleVerificationError('Fingerprint matching timeout');
   };
   
   // Send match request
   const postData = `Template1=${encodeURIComponent(capturedTemplate)}&Template2=${encodeURIComponent(storedTemplate)}&TemplateFormat=ISO`;
   xhr.send(postData);
   
   console.log('Fingerprint matching initiated');
}

function handleMatchResponse(result) {
   if (result.ErrorCode === 0) {
       const matchScore = result.MatchingScore || 0;
       const minScore = 50; // Minimum match score for verification
       
       window.laboratoryState.matchScore = matchScore;
       
       if (matchScore >= minScore) {
           // Verification successful
           handleVerificationSuccess(matchScore);
       } else {
           // Verification failed
           handleVerificationFailure(matchScore, minScore);
       }
       
       console.log(`Fingerprint matching completed - Score: ${matchScore}%`);
   } else {
       handleVerificationError(`Match failed: ${getSecuGenErrorDescription(result.ErrorCode)}`);
   }
}

function handleVerificationSuccess(matchScore) {
    // Original verification success logic
    window.secugenState.patientVerification.isVerified = true;
    window.secugenState.patientVerification.verificationMethod = 'fingerprint';
    window.secugenState.patientVerification.verificationTimestamp = new Date().toISOString();
    window.secugenState.patientVerification.verificationScore = matchScore;
    
    window.laboratoryState.isVerified = true;
    window.laboratoryState.verificationMethod = 'fingerprint';
    window.laboratoryState.verificationTimestamp = new Date().toISOString();
    
    // Enhanced status message based on workflow
    const workflowMessage = window.laboratoryState.patientWorkflow === 'gcc' ? 
        'GCC patient verified successfully!' : 
        'Patient verified successfully!';
        
    updateVerificationStatus('success', workflowMessage, 
        `Fingerprint match: ${matchScore}% (${new Date().toLocaleTimeString()})`);
    
    enableLabOptions();
    
    console.log(`Patient verification successful - Match score: ${matchScore}%`);
    showAlert(`Patient identity verified (${matchScore}% match)`, 'success');
    
    // Store verification data
    $('#fingerprint_verified').val('1');
    $('#verification_method').val('fingerprint');
    $('#verification_timestamp').val(new Date().toISOString());
}


function handleVerificationFailure(matchScore, minScore) {
   window.secugenState.patientVerification.attempts++;
   
   if (window.secugenState.patientVerification.attempts >= window.secugenState.patientVerification.maxAttempts) {
       updateVerificationStatus('error', 'Verification failed', 
           `Maximum attempts reached. Match score: ${matchScore}% (minimum required: ${minScore}%)`);
       showManualVerificationModal();
   } else {
       updateVerificationStatus('warning', 'Verification failed - try again', 
           `Match score: ${matchScore}% (minimum ${minScore}%). Attempt ${window.secugenState.patientVerification.attempts}/${window.secugenState.patientVerification.maxAttempts}`);
       showAlert(`Fingerprint verification failed. ${window.secugenState.patientVerification.maxAttempts - window.secugenState.patientVerification.attempts} attempts remaining.`, 'warning');
   }
}

function handleVerificationError(errorMsg) {
   window.secugenState.patientVerification.attempts++;
   
   updateVerificationStatus('error', 'Verification error', errorMsg);
   
   if (window.secugenState.patientVerification.attempts >= window.secugenState.patientVerification.maxAttempts) {
       showManualVerificationModal();
   } else {
       showAlert(`Verification error: ${errorMsg}`, 'error');
   }
   
   console.log(`Verification error: ${errorMsg}`);
}

function showManualVerificationModal() {
   // Manual verification still uses fingerprint capture
   updateVerificationStatus('info', 'Manual fingerprint verification', 'Manual verification will capture fingerprint and match against stored template');
   
   // Start fingerprint capture process immediately
   performManualFingerprintVerification();
}

function performManualFingerprintVerification() {
   if (!window.laboratoryState.storedTemplate) {
       showAlert('No stored fingerprint template found for verification', 'error');
       showFingerprintEnrollmentOption();
       return;
   }
   
   updateVerificationStatus('info', 'Manual verification - place finger on scanner', 'Capturing fingerprint for manual verification process');
   
   const captureUrl = window.secugenState.endpoints[window.secugenState.currentProtocol].capture;
   
   // Use XMLHttpRequest for manual verification capture
   const xhr = new XMLHttpRequest();
   xhr.open('POST', captureUrl, true);
   xhr.setRequestHeader('Content-Type', 'application/x-www-form-urlencoded');
   xhr.timeout = 20000;
   
   xhr.onreadystatechange = function() {
       if (xhr.readyState === 4) {
           if (xhr.status === 200) {
               try {
                   const result = JSON.parse(xhr.responseText);
                   handleManualCaptureResponse(result);
               } catch (e) {
                   handleVerificationError('Invalid response from manual fingerprint capture: ' + e.message);
               }
           } else {
               handleVerificationError(`Manual fingerprint capture failed: HTTP ${xhr.status}`);
           }
       }
   };
   
   xhr.onerror = function() {
       handleVerificationError('Network error during manual fingerprint capture');
   };
   
   xhr.ontimeout = function() {
       handleVerificationError('Manual fingerprint capture timeout - no finger detected');
   };
   
   // Send capture request
   const postData = 'Quality=50&Timeout=15000&WSQCompression=2.25';
   xhr.send(postData);
   
   console.log('Manual fingerprint verification capture initiated');
}

function handleManualCaptureResponse(result) {
   if (result.ErrorCode === 0 && result.TemplateBase64) {
       // Capture successful for manual verification, now compare with stored template
       window.laboratoryState.liveTemplate = result.TemplateBase64;
       
       // Update fingerprint image if available
       if (result.BMPBase64) {
           $('#patient_fingerprint_display').attr('src', 'data:image/bmp;base64,' + result.BMPBase64);
           $('#clearFingerprint').show();
       }
       
       // Show template quality if available
       if (result.Quality) {
           $('#fingerprint_quality').text(result.Quality);
           $('#template_info').show();
       }
       
       updateVerificationStatus('info', 'Manual verification - comparing fingerprints...', 'Matching captured fingerprint with stored template');
       
       // Compare captured fingerprint with stored template for manual verification
       compareManualFingerprints(result.TemplateBase64, window.laboratoryState.storedTemplate);
       
       console.log(`Manual fingerprint captured successfully - Quality: ${result.Quality || 'N/A'}%`);
   } else {
       handleVerificationError(getSecuGenErrorDescription(result.ErrorCode));
   }
}

function compareManualFingerprints(capturedTemplate, storedTemplate) {
   const matchUrl = window.secugenState.endpoints[window.secugenState.currentProtocol].match;
   
   const xhr = new XMLHttpRequest();
   xhr.open('POST', matchUrl, true);
   xhr.setRequestHeader('Content-Type', 'application/x-www-form-urlencoded');
   xhr.timeout = 10000;
   
   xhr.onreadystatechange = function() {
       if (xhr.readyState === 4) {
           if (xhr.status === 200) {
               try {
                   const result = JSON.parse(xhr.responseText);
                   handleManualMatchResponse(result);
               } catch (e) {
                   handleVerificationError('Invalid manual match response: ' + e.message);
               }
           } else {
               handleVerificationError(`Manual match service error: HTTP ${xhr.status}`);
           }
       }
   };
   
   xhr.onerror = function() {
       handleVerificationError('Network error during manual fingerprint matching');
   };
   
   xhr.ontimeout = function() {
       handleVerificationError('Manual fingerprint matching timeout');
   };
   
   // Send match request
   const postData = `Template1=${encodeURIComponent(capturedTemplate)}&Template2=${encodeURIComponent(storedTemplate)}&TemplateFormat=ISO`;
   xhr.send(postData);
   
   console.log('Manual fingerprint matching initiated');
}

function handleManualMatchResponse(result) {
   if (result.ErrorCode === 0) {
       const matchScore = result.MatchingScore || 0;
       const minScore = 50; // Minimum match score for manual verification
       
       window.laboratoryState.matchScore = matchScore;
       
       if (matchScore >= minScore) {
           // Manual verification successful
           handleManualVerificationSuccess(matchScore);
       } else {
           // Manual verification failed
           handleManualVerificationFailure(matchScore, minScore);
       }
       
       console.log(`Manual fingerprint matching completed - Score: ${matchScore}%`);
   } else {
       handleVerificationError(`Manual match failed: ${getSecuGenErrorDescription(result.ErrorCode)}`);
   }
}

function handleManualVerificationSuccess(matchScore) {
   window.secugenState.patientVerification.isVerified = true;
   window.secugenState.patientVerification.verificationMethod = 'manual_fingerprint';
   window.secugenState.patientVerification.verificationTimestamp = new Date().toISOString();
   window.secugenState.patientVerification.verificationScore = matchScore;
   
   window.laboratoryState.isVerified = true;
   window.laboratoryState.verificationMethod = 'manual_fingerprint';
   window.laboratoryState.verificationTimestamp = new Date().toISOString();
   
   updateVerificationStatus('success', 'Manual verification successful!', 
       `Fingerprint match: ${matchScore}% - Manual verification completed (${new Date().toLocaleTimeString()})`);
   
   enableLabOptions();
   
   console.log(`Manual patient verification successful - Match score: ${matchScore}%`);
   showAlert(`Manual verification completed - Patient identity verified (${matchScore}% match)`, 'success');
   
   // Store verification data in hidden fields
   $('#fingerprint_verified').val('1');
   $('#verification_method').val('manual_fingerprint');
   $('#verification_timestamp').val(new Date().toISOString());
}

function handleManualVerificationFailure(matchScore, minScore) {
   window.secugenState.patientVerification.attempts++;
   
   if (window.secugenState.patientVerification.attempts >= window.secugenState.patientVerification.maxAttempts) {
       updateVerificationStatus('error', 'Manual verification failed', 
           `Maximum attempts reached. Match score: ${matchScore}% (minimum required: ${minScore}%)`);
       
       // After max attempts, show alternative verification
       showAlternativeVerificationOptions();
   } else {
       updateVerificationStatus('warning', 'Manual verification failed - try again', 
           `Match score: ${matchScore}% (minimum ${minScore}%). Attempt ${window.secugenState.patientVerification.attempts}/${window.secugenState.patientVerification.maxAttempts}`);
       showAlert(`Manual fingerprint verification failed. ${window.secugenState.patientVerification.maxAttempts - window.secugenState.patientVerification.attempts} attempts remaining.`, 'warning');
   }
}

function showFingerprintEnrollmentOption() {
   const patientName = $('#selected_patient option:selected').text().split(' - ')[0];
   
   if (confirm(`${patientName} does not have a stored fingerprint template. Would you like to enroll a fingerprint now, or proceed with alternative verification?`)) {
       // User wants to enroll fingerprint
       enrollPatientFingerprint();
   } else {
       // User wants alternative verification
       showAlternativeVerificationOptions();
   }
}

function enrollPatientFingerprint() {
   updateVerificationStatus('info', 'Enrolling fingerprint...', 'Place finger on scanner for enrollment');
   
   const captureUrl = window.secugenState.endpoints[window.secugenState.currentProtocol].capture;
   
   const xhr = new XMLHttpRequest();
   xhr.open('POST', captureUrl, true);
   xhr.setRequestHeader('Content-Type', 'application/x-www-form-urlencoded');
   xhr.timeout = 25000;
   
   xhr.onreadystatechange = function() {
       if (xhr.readyState === 4) {
           if (xhr.status === 200) {
               try {
                   const result = JSON.parse(xhr.responseText);
                   
                   if (result.ErrorCode === 0 && result.TemplateBase64) {
                       // Save the fingerprint template
                       saveFingerprintTemplate(result.TemplateBase64, result.BMPBase64);
                   } else {
                       handleVerificationError(`Enrollment failed: ${getSecuGenErrorDescription(result.ErrorCode)}`);
                   }
               } catch (e) {
                   handleVerificationError('Invalid enrollment response');
               }
           } else {
               handleVerificationError(`Enrollment capture failed: HTTP ${xhr.status}`);
           }
       }
   };
   
   xhr.onerror = function() {
       handleVerificationError('Network error during fingerprint enrollment');
   };
   
   xhr.ontimeout = function() {
       handleVerificationError('Enrollment timeout - no finger detected');
   };
   
   // Send capture request with higher quality for enrollment
   const postData = 'Quality=60&Timeout=20000&WSQCompression=2.25';
   xhr.send(postData);
}

function saveFingerprintTemplate(template, image) {
   const patientId = window.secugenState.patientVerification.patientId;
   
   $.ajax({
       url: window.base_url + 'admin/patient/saveFingerprintTemplate',
       type: 'POST',
       data: {
           patient_id: patientId,
           fingerprint_template: template,
           fingerprint_image: image,
           enrollment_date: new Date().toISOString()
       },
       dataType: 'json',
       success: function(response) {
           if (response.status === 'success') {
               updateVerificationStatus('success', 'Fingerprint enrolled successfully!', 
                   'Patient can now be verified using fingerprint');
               
               // Store the newly enrolled template and proceed with verification
               window.laboratoryState.storedTemplate = template;
               
               // Now proceed with verification using the newly enrolled template
               setTimeout(() => {
                   compareFingerprints(template, template); // Self-match for immediate verification
               }, 2000);
               
           } else {
               handleVerificationError('Failed to save fingerprint template: ' + response.message);
           }
       },
       error: function() {
           handleVerificationError('Error saving fingerprint template');
       }
   });
}

function showAlternativeVerificationOptions() {
   const patientData = $('#selected_patient option:selected');
   const patientName = patientData.text().split(' - ')[0];
   
   const method = prompt(`Alternative verification required for ${patientName}.\nPlease select verification method:\n\n1. National ID Card\n2. Passport\n3. Driving License\n4. Phone Verification\n5. Staff Recognition\n\nEnter 1-5:`, '1');
   
   if (method && method >= 1 && method <= 5) {
       const methods = ['id_card', 'passport', 'driving_license', 'phone_verification', 'staff_recognition'];
       const methodName = methods[method - 1];
       
       const details = prompt(`Please enter verification details for ${methodName.replace('_', ' ')}:`, '');
       
       if (details && details.trim()) {
           performAlternativeVerification(methodName, details.trim());
       } else {
           showAlert('Verification details required', 'warning');
       }
   }
}

function performAlternativeVerification(method, details) {
   // Save alternative verification
   $.ajax({
       url: window.base_url + 'admin/laboratory/saveAlternativeVerification',
       type: 'POST',
       data: {
           patient_id: window.secugenState.patientVerification.patientId,
           verification_method: method,
           verification_details: details,
           verified_by: '<?php echo $this->session->userdata('username'); ?>',
           verification_timestamp: new Date().toISOString(),
           fingerprint_attempts: window.secugenState.patientVerification.attempts,
           last_match_score: window.laboratoryState.matchScore
       },
       dataType: 'json',
       success: function(response) {
           if (response.status === 'success') {
               window.secugenState.patientVerification.isVerified = true;
               window.secugenState.patientVerification.verificationMethod = 'alternative_' + method;
               window.secugenState.patientVerification.verificationTimestamp = new Date().toISOString();
               
               window.laboratoryState.isVerified = true;
               window.laboratoryState.verificationMethod = 'alternative_' + method;
               window.laboratoryState.verificationTimestamp = new Date().toISOString();
               
               updateVerificationStatus('success', 'Alternative verification completed', 
                   `Method: ${method.replace('_', ' ').toUpperCase()} (${new Date().toLocaleTimeString()})`);
               
               enableLabOptions();
               
               showAlert('Patient verification completed using alternative method', 'success');
               
               // Store verification data in hidden fields
               $('#fingerprint_verified').val('1');
               $('#verification_method').val('alternative_' + method);
               $('#verification_timestamp').val(new Date().toISOString());
               
           } else {
               showAlert('Error saving alternative verification: ' + response.message, 'error');
           }
       },
       error: function() {
           showAlert('Error saving alternative verification', 'error');
       }
   });
}

function enableLabOptions() {
   $('#labOptions').slideDown();
   $('#patient_verification_section').addClass('verification-complete');
   $('#saveLabOrder').prop('disabled', false);
   
   // Add verification badge to patient info
   const verificationBadge = `
       <div class="verification-badge">
           <i class="fa fa-shield text-success"></i> 
           <span class="text-success">Verified</span>
           <small class="text-muted">${window.laboratoryState.verificationMethod}</small>
       </div>
   `;
   
   if ($('.verification-badge').length === 0) {
       $('#patient_info').append(verificationBadge);
   }
   
   // Update empty cart message
   if (selectedTests.length === 0) {
       $('#selectedTests .empty-cart').html(
           '<i class="fa fa-shopping-cart fa-3x"></i>' +
           '<p>Patient verified - ready to add tests</p>' +
           '<small class="text-muted">Add tests from the categories on the left</small>'
       );
   }
}

function clearFingerprintCapture() {
   window.laboratoryState.liveTemplate = null;
   $('#patient_fingerprint_display').attr('src', window.base_url + 'Images/PlaceFinger.bmp');
   $('#template_info').hide();
   $('#clearFingerprint').hide();
   
   updateVerificationStatus('info', 'Capture cleared', 'Ready for new fingerprint capture');
   
   console.log('Fingerprint capture data cleared');
}

function updateVerificationStatus(type, title, message) {
   const statusIcon = type === 'success' ? 'fa-check-circle' :
                     type === 'error' ? 'fa-times-circle' :
                     type === 'warning' ? 'fa-exclamation-triangle' : 'fa-info-circle';
   
   const statusClass = type === 'success' ? 'status-success' :
                      type === 'error' ? 'status-error' :
                      type === 'warning' ? 'status-warning' : 'status-info';
   
   $('#verification_status').html(`
       <div class="${statusClass}">
           <i class="fa ${statusIcon}"></i> 
           <strong>${title}</strong>
           <br><small>${message}</small>
       </div>
   `);
}

function getSecuGenErrorDescription(errorCode) {
   const errorMap = {
       1: 'Reader not found or device driver error',
       2: 'Function failed - check device connection', 
       3: 'Invalid parameters sent to device',
       51: 'Scan timeout - no finger detected',
       52: 'Poor image quality - please try again',
       53: 'Finger not properly placed on scanner',
       54: 'Timeout occurred during scan',
       55: 'Device not found or not responding'
   };
   
   return errorMap[errorCode] || `Unknown SecuGen error (Code: ${errorCode})`;
}

// =============================================
// ORDER MANAGEMENT FUNCTIONS
// =============================================

function showConfirmationModal() {
   // Validate required fields
   if (!$('#selected_patient').val()) {
       showAlert('Please select a patient first', 'warning');
       return;
   }
   
   if (!window.laboratoryState.isVerified) {
       showAlert('Please verify patient identity first', 'warning');
       return;
   }
   
   if (selectedTests.length === 0) {
       showAlert('Please select at least one test', 'warning');
       return;
   }
   
   // Populate confirmation modal
   var selectedPatient = $('#selected_patient option:selected');
   $('#confirm_patient_name').text(selectedPatient.text().split(' - ')[0]);
   $('#confirm_patient_phone').text(selectedPatient.data('phone'));
   $('#confirm_patient_details').text(selectedPatient.data('age') + ' years, ' + selectedPatient.data('gender'));
   
   // Show verification status
   var verificationStatus = window.laboratoryState.verificationMethod === 'fingerprint' ? 
       '<span class="text-success"><i class="fa fa-fingerprint"></i> Biometric Verified</span>' :
       window.laboratoryState.verificationMethod === 'manual_fingerprint' ?
       '<span class="text-info"><i class="fa fa-hand-pointer-o"></i> Manual Fingerprint Verified</span>' :
       window.laboratoryState.verificationMethod && window.laboratoryState.verificationMethod.startsWith('alternative_') ?
       '<span class="text-warning"><i class="fa fa-id-card"></i> Alternative Verification</span>' :
       '<span class="text-muted"><i class="fa fa-question-circle"></i> Unknown Method</span>';
   $('#confirm_verification_status').html(verificationStatus);
   
   $('#confirm_test_count').text(selectedTests.length + ' tests');
   
   var priority = $('#test_priority').val();
   var priorityText = priority.charAt(0).toUpperCase() + priority.slice(1);
   if (priority !== 'normal') {
       priorityText += ' <span class="priority-badge priority-' + priority + '">' + priority.toUpperCase() + '</span>';
   }
   $('#confirm_priority').html(priorityText);
   
   var billingInfo = calculateBillingInfo();
   $('#confirm_total').text('KES ' + Math.round(billingInfo.totalAmount).toLocaleString());
   
   // Show selected tests
   var testsListHtml = '<div class="table-responsive"><table class="table table-condensed table-striped">';
   testsListHtml += '<thead><tr><th>Test Name</th><th>Category</th><th class="text-right">Price</th></tr></thead><tbody>';
   
   $.each(selectedTests, function(index, test) {
       testsListHtml += '<tr>' +
           '<td>' + test.name + '</td>' +
           '<td><small class="text-muted">' + test.category + '</small></td>' +
           '<td class="text-right">KES ' + test.charge.toLocaleString() + '</td>' +
           '</tr>';
   });
   
   testsListHtml += '</tbody></table></div>';
   $('#confirm_tests_list').html(testsListHtml);
   
   // Show modal
   $('#confirmModal').modal('show');
}

function saveLabOrder() {
   // Validate again
   if (!$('#selected_patient').val() || selectedTests.length === 0 || !window.laboratoryState.isVerified) {
       showAlert('Missing required information or patient not verified', 'error');
       return;
   }
   
   // Calculate billing info
   var billingInfo = calculateBillingInfo();
   
   // Prepare data
   var orderData = {
       patient_id: $('#selected_patient').val(),
       selected_tests: JSON.stringify(selectedTests),
       billing_info: JSON.stringify(billingInfo),
       doctor_id: $('#assigned_doctor').val(),
       test_date: $('#test_date').val(),
       priority: $('#test_priority').val(),
       payment_method: $('#payment_method').val(),
       notes: $('#lab_notes').val(),
       verification_method: window.laboratoryState.verificationMethod,
       verification_timestamp: window.laboratoryState.verificationTimestamp,
       fingerprint_verified: window.laboratoryState.isVerified ? '1' : '0'
   };
   
   $.ajax({
       url: window.base_url + 'admin/laboratory/saveMultipleTests',
       type: 'POST',
       data: orderData,
       dataType: 'json',
       beforeSend: function() {
           $('#finalConfirmBtn').prop('disabled', true).html('<i class="fa fa-spinner fa-spin"></i> Saving Order...');
           $('#cart_loading').show();
       },
       success: function(response) {
           if (response.status === 'success') {
               $('#confirmModal').modal('hide');
               
               showAlert('Laboratory order saved successfully! Order includes ' + response.test_count + ' tests.', 'success');
               
               // Clear the cart after successful save
               setTimeout(function() {
                   clearCart();
                   resetForm();
               }, 2000);
               
               // Optionally redirect to lab management page
               if (confirm('Order saved successfully! Would you like to go to the Laboratory Management page?')) {
                   window.location.href = window.base_url + 'admin/laboratory';
               }
           } else {
               showAlert('Error: ' + response.message, 'error');
           }
       },
       error: function(xhr, status, error) {
           var errorMsg = 'Error saving laboratory order';
           try {
               var response = JSON.parse(xhr.responseText);
               errorMsg = response.message || errorMsg;
           } catch (e) {
               errorMsg += ': ' + error;
           }
           showAlert(errorMsg, 'error');
       },
       complete: function() {
           $('#finalConfirmBtn').prop('disabled', false).html('<i class="fa fa-check"></i> Confirm & Save Order');
           $('#cart_loading').hide();
       }
   });
}

function calculateBillingInfo() {
   var subtotal = selectedTests.reduce(function(sum, test) {
       return sum + test.charge;
   }, 0);
   
   var priority = $('#test_priority').val();
   var priorityMultiplier = priority === 'urgent' ? 1.5 : priority === 'stat' ? 2.0 : 1.0;
   var priorityAdjustedTotal = subtotal * priorityMultiplier;
   
   var discountPercent = parseFloat($('#discountPercent').val()) || 0;
   var taxPercent = parseFloat($('#taxPercent').val()) || 0;

   var discountAmount = (priorityAdjustedTotal * discountPercent) / 100;
   var taxableAmount = priorityAdjustedTotal - discountAmount;
   var taxAmount = (taxableAmount * taxPercent) / 100;
   var totalAmount = taxableAmount + taxAmount;

   return {
       subtotal: priorityAdjustedTotal,
       discountPercent: discountPercent,
       discountAmount: discountAmount,
       taxPercent: taxPercent,
       taxAmount: taxAmount,
       totalAmount: totalAmount
   };
}

function resetForm() {
   // Reset patient selection
   $('#selected_patient').val('').trigger('change');
   $('#patient_search').val('');
   $('#patient_info').hide();
   $('#patient_verification_section').hide();
   $('#labOptions').hide();
   
   // Reset verification state
   resetPatientVerification();
   
   // Reset form fields
   $('#discountPercent').val(0);
   $('#taxPercent').val(16);
   $('#test_priority').val('normal');
   $('#assigned_doctor').val('');
   $('#test_date').val('<?php echo date('Y-m-d'); ?>');
   $('#payment_method').val('cash');
   $('#lab_notes').val('');
   
   // Clear search
   $('#testSearch').val('');
   filterTests('');
}

// =============================================
// UTILITY FUNCTIONS
// =============================================

function showAlert(message, type) {
   var alertClass = type === 'success' ? 'alert-success' : 
                    type === 'warning' ? 'alert-warning' : 
                    type === 'error' ? 'alert-danger' : 'alert-info';
   
   var iconClass = type === 'success' ? 'fa-check-circle' :
                   type === 'warning' ? 'fa-exclamation-triangle' :
                   type === 'error' ? 'fa-times-circle' : 'fa-info-circle';
   
   var alert = $('<div>', {
       'class': 'alert ' + alertClass + ' alert-dismissible alert-custom',
       html: '<button type="button" class="close" data-dismiss="alert" aria-label="Close">' +
             '<span aria-hidden="true">&times;</span></button>' +
             '<i class="fa ' + iconClass + '"></i> ' + message
   });
   
   $('body').append(alert);
   
   // Auto-dismiss after 5 seconds
   setTimeout(function() {
       alert.fadeOut(400, function() {
           $(this).remove();
       });
   }, 5000);
}

// =============================================
// AUTO-SAVE & RECOVERY FUNCTIONS
// =============================================

function autoSave() {
   if (selectedTests.length > 0 && window.laboratoryState.selectedPatient) {
       localStorage.setItem('lab_cart_backup', JSON.stringify({
           patient_id: window.laboratoryState.selectedPatient,
           tests: selectedTests,
           verification_status: window.laboratoryState.isVerified,
           timestamp: new Date().getTime()
       }));
   }
}

function restoreFromAutoSave() {
   try {
       var backup = localStorage.getItem('lab_cart_backup');
       if (backup) {
           var data = JSON.parse(backup);
           var age = new Date().getTime() - data.timestamp;
           
           // Restore if less than 1 hour old
           if (age < 3600000 && data.tests && data.tests.length > 0) {
               if (confirm('Found ' + data.tests.length + ' unsaved tests from a previous session. Restore them?')) {
                   // Restore patient selection
                   if (data.patient_id) {
                       $('#selected_patient').val(data.patient_id).trigger('change');
                   }
                   
                   // Restore tests
                   selectedTests = data.tests;
                   updateCartDisplay();
                   updateBillingSummary();
                   updateSessionStats();
                   
                   showAlert('Restored ' + data.tests.length + ' tests from previous session', 'info');
               }
           }
       }
   } catch (e) {
       console.log('Error restoring auto-save:', e);
   }
}

// =============================================
// WORKFLOW-BASED VERIFICATION REQUIREMENT LOGIC
// =============================================

function determineVerificationRequirement(workflow, hasFingerprint) {
    console.log('🔍 Determining verification requirement...');
    console.log('📋 Workflow:', workflow);
    console.log('🔐 Has Fingerprint:', hasFingerprint);
    
    // 🔒 GCC WORKFLOW: MANDATORY VERIFICATION
    if (workflow === 'gcc' || workflow === 'GCC') {
        window.laboratoryState.requiresVerification = true;
        showVerificationSection(true, 'GCC workflow requires mandatory biometric verification');
        
        if (hasFingerprint) {
            loadStoredFingerprintTemplate(window.laboratoryState.selectedPatient);
        } else {
            updateVerificationStatus('warning', 'GCC patient without fingerprint template', 
                'Fingerprint enrollment required for GCC patients');
        }
        
        console.log('🔒 GCC WORKFLOW: Verification MANDATORY');
        
    // 🔓 GENERAL/SPECIALIZED WORKFLOW: OPTIONAL VERIFICATION
    } else {
        window.laboratoryState.requiresVerification = false;
        showVerificationSection(false, 'General workflow - verification optional');
        
        // Auto-enable lab options for non-GCC patients
        enableLabOptionsDirectly();
        
        console.log('🔓 GENERAL/SPECIALIZED WORKFLOW: Verification OPTIONAL');
    }
}

// =============================================
// DYNAMIC VERIFICATION SECTION DISPLAY
// =============================================

function showVerificationSection(isRequired, message) {
    if (isRequired) {
        // Show verification section for mandatory verification
        $('#patient_verification_section').slideDown();
        
        // Update UI to show mandatory verification
        $('.verification-section').removeClass('verification-optional').addClass('verification-mandatory');
        
        updateVerificationStatus('info', 'Verification Required', message);
        
        // Disable lab options until verified
        $('#labOptions').hide();
        $('#saveLabOrder').prop('disabled', true);
        
        // Update instruction text
        $('.empty-cart').html(
            '<i class="fa fa-shield fa-3x"></i>' +
            '<p>Patient verification required</p>' +
            '<small class="text-muted">GCC patients must be verified before adding tests</small>'
        );
        
    } else {
        // Hide verification section for optional verification
        $('#patient_verification_section').slideUp();
        
        // Show optional verification message
        updateVerificationStatus('success', 'Verification Optional', message);
        
        // Auto-enable lab options
        enableLabOptionsDirectly();
        
        // Update instruction text
        $('.empty-cart').html(
            '<i class="fa fa-shopping-cart fa-3x"></i>' +
            '<p>Ready to add tests</p>' +
            '<small class="text-muted">Verification not required for this patient type</small>'
        );
    }
}

// =============================================
// DIRECT LAB OPTIONS ENABLING (NON-GCC PATIENTS)
// =============================================

function enableLabOptionsDirectly() {
    console.log('🔓 Enabling lab options directly (no verification required)');
    
    // Mark as verified without biometric verification
    window.laboratoryState.isVerified = true;
    window.laboratoryState.verificationMethod = 'workflow_exempted';
    window.laboratoryState.verificationTimestamp = new Date().toISOString();
    
    // Enable lab options
    $('#labOptions').slideDown();
    $('#saveLabOrder').prop('disabled', false);
    
    // Add workflow exemption badge
    if ($('.workflow-exemption-badge').length === 0) {
        const exemptionBadge = `
            <div class="workflow-exemption-badge verification-badge">
                <i class="fa fa-check-circle text-success"></i> 
                <span class="text-success">Workflow Exempted</span>
                <small class="text-muted">General workflow - no verification required</small>
            </div>
        `;
        $('#patient_info').append(exemptionBadge);
    }
    
    console.log('✅ Lab options enabled for non-GCC patient');
}



// =============================================
// DEBUG FUNCTIONS
// =============================================

window.LAB_DEBUG = {
   secugenState: function() {
       console.log('SecuGen State:', window.secugenState);
       console.log('Laboratory State:', window.laboratoryState);
       return {
           secugen: window.secugenState,
           laboratory: window.laboratoryState
       };
   },
   
   testConnection: function() {
       console.log('Testing SecuGen connection...');
       initializeSecuGenService();
   },
   
   testLabTestsEndpoint: function() {
       console.log('Testing lab tests endpoint...');
       console.log('URL:', window.base_url + 'admin/laboratory/getLabTests');
       
       $.ajax({
           url: window.base_url + 'admin/laboratory/getLabTests',
           type: 'GET',
           success: function(response) {
               console.log('Lab tests endpoint response:', response);
           },
           error: function(xhr, status, error) {
               console.error('Lab tests endpoint error:', xhr, status, error);
               console.error('Response text:', xhr.responseText);
           }
       });
   },
   
   testDoctorsEndpoint: function() {
       console.log('Testing doctors endpoint...');
       console.log('URL:', window.base_url + 'admin/laboratory/getDoctors');
       
       $.ajax({
           url: window.base_url + 'admin/laboratory/getDoctors',
           type: 'GET',
           success: function(response) {
               console.log('Doctors endpoint response:', response);
           },
           error: function(xhr, status, error) {
               console.error('Doctors endpoint error:', xhr, status, error);
               console.error('Response text:', xhr.responseText);
           }
       });
   },
   
   reloadTests: function() {
       console.log('Reloading laboratory tests...');
       loadTestCategories();
   },
   
   reloadDoctors: function() {
       console.log('Reloading doctors...');
       loadDoctors();
   },
   
   validateOrder: function() {
       console.log('Order Validation:');
       console.log('- Patient Selected:', !!window.laboratoryState.selectedPatient);
       console.log('- Patient Verified:', window.laboratoryState.isVerified);
       console.log('- Tests Selected:', selectedTests.length);
       console.log('- Total Amount:', selectedTests.length > 0 ? calculateBillingInfo().totalAmount : 0);
       return {
           valid: !!(window.laboratoryState.selectedPatient && window.laboratoryState.isVerified && selectedTests.length > 0),
           details: {
               patient: !!window.laboratoryState.selectedPatient,
               verified: window.laboratoryState.isVerified,
               tests: selectedTests.length,
               amount: selectedTests.length > 0 ? calculateBillingInfo().totalAmount : 0
           }
       };
   },
   
   simulateVerification: function() {
       console.log('Simulating successful verification...');
       window.laboratoryState.isVerified = true;
       window.laboratoryState.verificationMethod = 'debug';
       window.laboratoryState.verificationTimestamp = new Date().toISOString();
       enableLabOptions();
       showAlert('Debug verification successful', 'success');
   },
   
   checkEndpoints: function() {
       console.log('Base URL:', window.base_url);
       console.log('Expected endpoints:');
       console.log('- Lab Tests:', window.base_url + 'admin/laboratory/getLabTests');
       console.log('- Doctors:', window.base_url + 'admin/laboratory/getDoctors');
       console.log('- Patient Search:', window.base_url + 'admin/laboratory/searchPatients');
       console.log('- Patient Fingerprint:', window.base_url + 'admin/laboratory/getPatientFingerprint');
       console.log('- Save Order:', window.base_url + 'admin/laboratory/saveMultipleTests');
       
       console.log('Testing all endpoints...');
       this.testLabTestsEndpoint();
       this.testDoctorsEndpoint();
   },
   
   resetSystem: function() {
       console.log('Resetting laboratory system...');
       clearCart();
       resetForm();
       window.laboratoryState = {
           selectedPatient: null,
           selectedPatientData: null,
           verificationStatus: 'pending',
           secugenActive: false,
           liveTemplate: null,
           storedTemplate: null,
           matchScore: 0,
           selectedTests: [],
           isVerified: false,
           verificationMethod: null,
           verificationTimestamp: null
       };
       showAlert('System reset complete', 'info');
   }
};

// =============================================
// INITIALIZATION & EVENT HANDLERS
// =============================================

// Initialize auto-save
$(document).ready(function() {
   // Check for auto-save data after a delay
   setTimeout(restoreFromAutoSave, 1000);
   
   // Auto-save every 30 seconds
   setInterval(autoSave, 30000);
   
   // 🔒 CHECK VERIFICATION REQUIREMENT
    if (window.laboratoryState.requiresVerification && !window.laboratoryState.isVerified) {
        const workflowType = window.laboratoryState.patientWorkflow?.toUpperCase() || 'GCC';
        showAlert(`${workflowType} patients must be verified before adding tests`, 'warning');
        
        // Highlight verification section
        $('#patient_verification_section').addClass('pulse-highlight');
        setTimeout(() => {
            $('#patient_verification_section').removeClass('pulse-highlight');
        }, 2000);
        
        return;
    }
   
   // Keyboard shortcuts
   $(document).on('keydown', function(e) {
       // Ctrl/Cmd + S to save
       if ((e.ctrlKey || e.metaKey) && e.which === 83) {
           e.preventDefault();
           if (selectedTests.length > 0 && $('#selected_patient').val() && window.laboratoryState.isVerified) {
               showConfirmationModal();
           }
       }
       
       // Escape to clear search
       if (e.which === 27) {
           $('#testSearch').val('').trigger('input');
       }
   });
   
   // Enhanced error handling
   window.onerror = function(msg, url, lineNo, columnNo, error) {
       console.error('JavaScript Error:', {
           message: msg,
           source: url,
           line: lineNo,
           column: columnNo,
           error: error
       });
       
       showAlert('An unexpected error occurred. Please refresh the page if issues persist.', 'error');
       return false;
   };
});


// Console help - using array join to avoid line break issues
console.log([
'=== Enhanced Laboratory Test Selection with Biometric Verification ===',
'',
'🔬 LABORATORY FEATURES:',
'✓ SecuGen WebAPI fingerprint verification integration',
'✓ Patient identity verification before test ordering (fingerprint-based)', 
'✓ Dynamic test selection loaded from database',
'✓ Dynamic doctor list loaded from database',
'✓ Real-time billing calculation with priorities',
'✓ Automated order management and tracking',
'✓ Session recovery and auto-save functionality',
'',
'🔐 VERIFICATION METHODS:',
'1. Standard Fingerprint Verification - Automatic capture and matching',
'2. Manual Fingerprint Verification - Manual initiation but still uses fingerprint capture and matching',  
'3. Alternative Verification - Only available after fingerprint verification fails (ID, passport, etc.)',
'',
'📋 DEBUG COMMANDS:',
'- LAB_DEBUG.secugenState() - Show SecuGen device state',
'- LAB_DEBUG.testConnection() - Test SecuGen WebAPI connection', 
'- LAB_DEBUG.checkEndpoints() - Test all database endpoints',
'- LAB_DEBUG.reloadTests() - Reload laboratory tests',
'- LAB_DEBUG.validateOrder() - Check order validation status',
'- LAB_DEBUG.simulateVerification() - Simulate successful verification',
'- LAB_DEBUG.resetSystem() - Reset entire system',
'',
'💡 WORKFLOW:',
'1. Select Patient → 2. Verify Identity (Fingerprint) → 3. Select Tests → 4. Configure Options → 5. Save Order',
'',
'IMPORTANT: All verification methods use fingerprint capture and matching.',
'Manual verification is simply a manual initiation of the fingerprint process.',
'Only after fingerprint verification fails are alternative methods offered.',
'',
'All data (tests, doctors, patients) is loaded dynamically from the database.'
].join('\n'));
</script>