<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Laboratory Test Selection</title>
    <link href="https://cdnjs.cloudflare.com/ajax/libs/bootstrap/3.4.1/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/4.7.0/css/font-awesome.min.css" rel="stylesheet">
    <style>
        .content-wrapper {
            background-color: #f4f4f4;
            min-height: 100vh;
            padding: 20px;
        }
        
        .box {
            background: #fff;
            border-top: 3px solid #d2d6de;
            margin-bottom: 20px;
            width: 100%;
            box-shadow: 0 1px 1px rgba(0,0,0,0.1);
            border-radius: 3px;
        }
        
        .box-header {
            border-bottom: 1px solid #f4f4f4;
            color: #444;
            display: block;
            padding: 10px 15px;
            position: relative;
        }
        
        .box-title {
            font-size: 18px;
            margin: 0;
            line-height: 1.8;
        }
        
        .box-body {
            border-top-left-radius: 0;
            border-top-right-radius: 0;
            border-bottom-right-radius: 3px;
            border-bottom-left-radius: 3px;
            padding: 10px;
        }
        
        .nav-tabs-custom {
            margin-bottom: 20px;
            background: #fff;
            box-shadow: 0 1px 1px rgba(0,0,0,0.1);
            border-radius: 3px;
        }
        
        .nav-tabs-custom > .nav-tabs {
            margin: 0;
            border-bottom-color: #f4f4f4;
            border-top-right-radius: 3px;
            border-top-left-radius: 3px;
        }
        
        .nav-tabs-custom > .nav-tabs > li.active {
            border-top-color: #3c8dbc;
        }
        
        .test-category-card {
            border: 1px solid #ddd;
            border-radius: 5px;
            margin-bottom: 15px;
            transition: all 0.3s;
        }
        
        .test-category-card:hover {
            box-shadow: 0 2px 5px rgba(0,0,0,0.1);
        }
        
        .category-header {
            background: linear-gradient(135deg, #3c8dbc, #367fa9);
            color: white;
            padding: 10px 15px;
            border-radius: 5px 5px 0 0;
            cursor: pointer;
        }
        
        .category-tests {
            padding: 10px;
            max-height: 300px;
            overflow-y: auto;
        }
        
        .test-item {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 8px;
            border-bottom: 1px solid #eee;
            transition: background-color 0.2s;
        }
        
        .test-item:hover {
            background-color: #f8f9fa;
        }
        
        .test-name {
            flex: 1;
            font-weight: 500;
        }
        
        .test-price {
            color: #27ae60;
            font-weight: bold;
            margin-right: 10px;
        }
        
        .btn-add-test {
            background: #28a745;
            color: white;
            border: none;
            padding: 5px 10px;
            border-radius: 3px;
            font-size: 12px;
            transition: background-color 0.2s;
        }
        
        .btn-add-test:hover {
            background: #218838;
            color: white;
        }
        
        .cart-item {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 10px;
            border-bottom: 1px solid #eee;
            background: #f8f9fa;
            margin-bottom: 5px;
            border-radius: 3px;
        }
        
        .cart-item-name {
            flex: 1;
            font-weight: 500;
        }
        
        .cart-item-price {
            color: #27ae60;
            font-weight: bold;
            margin-right: 10px;
        }
        
        .btn-remove-test {
            background: #dc3545;
            color: white;
            border: none;
            padding: 3px 8px;
            border-radius: 3px;
            font-size: 11px;
        }
        
        .btn-remove-test:hover {
            background: #c82333;
            color: white;
        }
        
        .billing-summary {
            background: #e8f4fd;
            border: 1px solid #bee5eb;
            border-radius: 5px;
            padding: 15px;
            margin-top: 20px;
        }
        
        .summary-row {
            display: flex;
            justify-content: space-between;
            margin-bottom: 8px;
        }
        
        .total-amount {
            font-size: 18px;
            font-weight: bold;
            color: #2c3e50;
            border-top: 2px solid #3c8dbc;
            padding-top: 10px;
            margin-top: 10px;
        }
        
        .search-box {
            margin-bottom: 20px;
        }
        
        .category-count {
            background: #fff;
            color: #3c8dbc;
            padding: 2px 8px;
            border-radius: 10px;
            font-size: 12px;
            margin-left: 10px;
        }
        
        .empty-cart {
            text-align: center;
            color: #6c757d;
            padding: 30px;
            font-style: italic;
        }
        
        .btn-primary {
            background-color: #3c8dbc;
            border-color: #367fa9;
        }
        
        .btn-primary:hover {
            background-color: #367fa9;
            border-color: #2e6da4;
        }
        
        .form-control:focus {
            border-color: #3c8dbc;
            box-shadow: inset 0 1px 1px rgba(0,0,0,.075), 0 0 0 3px rgba(60,141,188,.1);
        }
        
        .alert-info {
            background-color: #d9edf7;
            border-color: #bce8f1;
            color: #31708f;
        }
    </style>
</head>
<body>
    <div class="content-wrapper">
        <div class="container-fluid">
            <!-- Header -->
            <div class="row">
                <div class="col-md-12">
                    <div class="box">
                        <div class="box-header with-border">
                            <h3 class="box-title">
                                <i class="fa fa-flask"></i> Laboratory Test Selection
                            </h3>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Main Content -->
            <div class="row">
                <!-- Test Selection Panel -->
                <div class="col-md-8">
                    <div class="nav-tabs-custom">
                        <ul class="nav nav-tabs pull-right">
                            <li class="active">
                                <a href="#tab_test_selection" data-toggle="tab">
                                    <i class="fa fa-list"></i> Select Tests
                                </a>
                            </li>
                            <li class="pull-left header">
                                <i class="fa fa-flask"></i> Available Laboratory Tests
                            </li>
                        </ul>
                        <div class="tab-content">
                            <div class="tab-pane active" id="tab_test_selection">
                                <div class="box-body">
                                    <!-- Search Bar -->
                                    <div class="search-box">
                                        <div class="input-group">
                                            <input type="text" class="form-control" id="testSearch" 
                                                   placeholder="Search for tests...">
                                            <span class="input-group-addon">
                                                <i class="fa fa-search"></i>
                                            </span>
                                        </div>
                                    </div>

                                    <!-- Test Categories -->
                                    <div id="testCategories">
                                        <!-- Categories will be populated here -->
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Cart and Billing Panel -->
                <div class="col-md-4">
                    <div class="box">
                        <div class="box-header with-border">
                            <h3 class="box-title">
                                <i class="fa fa-shopping-cart"></i> Selected Tests
                                <span class="badge" id="cartCount">0</span>
                            </h3>
                        </div>
                        <div class="box-body">
                            <div id="selectedTests">
                                <div class="empty-cart">
                                    <i class="fa fa-shopping-cart fa-3x"></i>
                                    <p>No tests selected yet</p>
                                </div>
                            </div>

                            <!-- Billing Summary -->
                            <div class="billing-summary" id="billingSummary" style="display: none;">
                                <h4><i class="fa fa-calculator"></i> Billing Summary</h4>
                                
                                <div class="summary-row">
                                    <span>Subtotal:</span>
                                    <span id="subtotal">KES 0</span>
                                </div>
                                
                                <div class="form-group">
                                    <label for="discountPercent">Discount (%):</label>
                                    <input type="number" class="form-control" id="discountPercent" 
                                           min="0" max="100" value="0" placeholder="0">
                                </div>
                                
                                <div class="summary-row">
                                    <span>Discount Amount:</span>
                                    <span id="discountAmount">KES 0</span>
                                </div>
                                
                                <div class="form-group">
                                    <label for="taxPercent">Tax (%):</label>
                                    <input type="number" class="form-control" id="taxPercent" 
                                           min="0" max="100" value="16" placeholder="16">
                                </div>
                                
                                <div class="summary-row">
                                    <span>Tax Amount:</span>
                                    <span id="taxAmount">KES 0</span>
                                </div>
                                
                                <div class="summary-row total-amount">
                                    <span>Total Amount:</span>
                                    <span id="totalAmount">KES 0</span>
                                </div>
                            </div>

                            <!-- Action Buttons -->
                            <div style="margin-top: 20px;">
                                <button class="btn btn-danger btn-block" id="clearCart" 
                                        style="display: none;">
                                    <i class="fa fa-trash"></i> Clear All Tests
                                </button>
                                <button class="btn btn-primary btn-block" id="proceedToLab" 
                                        style="display: none; margin-top: 10px;">
                                    <i class="fa fa-arrow-right"></i> Proceed to Laboratory
                                </button>
                            </div>
                        </div>
                    </div>

                    <!-- Patient Info Card -->
                    <div class="box">
                        <div class="box-header with-border">
                            <h3 class="box-title">
                                <i class="fa fa-user"></i> Quick Instructions
                            </h3>
                        </div>
                        <div class="box-body">
                            <div class="alert alert-info">
                                <p><strong>How to use:</strong></p>
                                <ul>
                                    <li>Browse tests by category</li>
                                    <li>Use search to find specific tests</li>
                                    <li>Click "Add" to select tests</li>
                                    <li>Review billing and proceed</li>
                                </ul>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdnjs.cloudflare.com/ajax/libs/jquery/3.6.0/jquery.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/bootstrap/3.4.1/js/bootstrap.min.js"></script>
    <script>
        // Laboratory Tests Data
        const labTestsData = {
            "General Tests": [
                { "name": "C ANCA", "charge": 5500 },
                { "name": "P ANCA", "charge": 5500 },
                { "name": "AAFB test", "charge": 1000 },
                { "name": "ACR- Albumin Creatinine Ratio", "charge": 3200 },
                { "name": "ALPHA FETOPROTEIN", "charge": 3500 },
                { "name": "AMYLASE", "charge": 1500 },
                { "name": "Anti CCP", "charge": 5000 },
                { "name": "APTT", "charge": 1500 },
                { "name": "ASCITIC TAP", "charge": 3000 }
            ],
            "Hematology": [
                { "name": "Blood Culture", "charge": 6000 },
                { "name": "Blood ethanol levels", "charge": 5000 },
                { "name": "Blood Grouping", "charge": 400 },
                { "name": "Complete Blood Count (CBC)", "charge": 1200 },
                { "name": "ESR", "charge": 500 },
                { "name": "Full Blood Count", "charge": 1200 },
                { "name": "Hemoglobin", "charge": 500 },
                { "name": "Platelet Count", "charge": 800 },
                { "name": "White Blood Cell Count", "charge": 600 }
            ],
            "Endocrinology": [
                { "name": "ANTI THYROID PEROXIDASE ANTIBODIES( TPO )", "charge": 3000 },
                { "name": "FT3", "charge": 2000 },
                { "name": "FT4", "charge": 2000 },
                { "name": "TSH", "charge": 2000 },
                { "name": "T3", "charge": 1800 },
                { "name": "T4", "charge": 1800 },
                { "name": "Thyroid Function Tests", "charge": 4500 },
                { "name": "PTH", "charge": 4000 },
                { "name": "Cortisol", "charge": 3500 },
                { "name": "Growth Hormone", "charge": 5000 }
            ],
            "Diabetes": [
                { "name": "Fluid glucose", "charge": 100 },
                { "name": "HBA1C", "charge": 3000 },
                { "name": "Fasting Blood Sugar", "charge": 500 },
                { "name": "Random Blood Sugar", "charge": 500 },
                { "name": "Oral Glucose Tolerance Test", "charge": 1500 }
            ],
            "Lipid Profile": [
                { "name": "HDL Cholesterol", "charge": 500 },
                { "name": "LIPID PROFILE", "charge": 3000 },
                { "name": "LDL Cholesterol", "charge": 800 },
                { "name": "Total Cholesterol", "charge": 600 },
                { "name": "Triglycerides", "charge": 700 },
                { "name": "VLDL Cholesterol", "charge": 600 }
            ],
            "Hepatology": [
                { "name": "ALT/GPT (Alanine Aminotransferase.)", "charge": 500 },
                { "name": "AST/GOT", "charge": 500 },
                { "name": "HEPATITIS C VIRUS ANTIBODY", "charge": 1000 },
                { "name": "HEPATITIS B SURFACE ANTIGEN", "charge": 1000 },
                { "name": "Bilirubin Total", "charge": 600 },
                { "name": "Bilirubin Direct", "charge": 600 },
                { "name": "Alkaline Phosphatase", "charge": 700 },
                { "name": "GGT", "charge": 800 },
                { "name": "Total Protein", "charge": 500 },
                { "name": "Albumin", "charge": 500 }
            ],
            "Urology": [
                { "name": "Urinalysis", "charge": 400 },
                { "name": "URINALYSIS (corporate)", "charge": 500 },
                { "name": "URINE CREAT", "charge": 800 },
                { "name": "Urine Culture", "charge": 2000 },
                { "name": "Urine Microscopy", "charge": 600 },
                { "name": "24hr Urine Protein", "charge": 1500 },
                { "name": "Urine Glucose", "charge": 300 },
                { "name": "Urine Ketones", "charge": 400 }
            ],
            "Microbiology": [
                { "name": "CULTURE & SENSITIVITY", "charge": 2050 },
                { "name": "Fluid Culture & Sensitivity", "charge": 2500 },
                { "name": "Microscopy, culture & Sensitivity MCS", "charge": 1450 },
                { "name": "Stool Culture", "charge": 2000 },
                { "name": "Throat Swab", "charge": 1500 },
                { "name": "Wound Swab", "charge": 1800 }
            ],
            "Vitamins & Minerals": [
                { "name": "Folate (Folic Acid)", "charge": 2700 },
                { "name": "Vitamin B 12", "charge": 6000 },
                { "name": "Vitamin B12", "charge": 3450 },
                { "name": "Vitamin D", "charge": 4000 },
                { "name": "Iron Studies", "charge": 2500 },
                { "name": "Calcium", "charge": 600 },
                { "name": "Magnesium", "charge": 800 },
                { "name": "Phosphorus", "charge": 700 }
            ],
            "Cardiology": [
                { "name": "TROPONIN", "charge": 4000 },
                { "name": "CK-MB", "charge": 2500 },
                { "name": "LDH", "charge": 1200 },
                { "name": "CPK", "charge": 1500 },
                { "name": "BNP", "charge": 5000 }
            ],
            "Health Packages": [
                { "name": "15000 comprehensive female wellness", "charge": 15000 },
                { "name": "18000 Female Comprehensive Wellness", "charge": 18000 },
                { "name": "BAHRAIN MEDICAL PACKAGE", "charge": 4700 },
                { "name": "Basic Health Screen", "charge": 3500 },
                { "name": "Executive Health Package", "charge": 8500 },
                { "name": "Cardiac Risk Assessment", "charge": 6500 },
                { "name": "Diabetes Screen", "charge": 2500 },
                { "name": "Liver Function Package", "charge": 3200 }
            ]
        };

        // Shopping cart
        let selectedTests = [];

        // Initialize the application
        $(document).ready(function() {
            renderTestCategories();
            bindEvents();
        });

        function renderTestCategories() {
            const container = $('#testCategories');
            container.empty();

            Object.keys(labTestsData).forEach(category => {
                const tests = labTestsData[category];
                const categoryCard = createCategoryCard(category, tests);
                container.append(categoryCard);
            });
        }

        function createCategoryCard(category, tests) {
            const categoryId = category.replace(/\s+/g, '_').toLowerCase();
            
            return `
                <div class="test-category-card" data-category="${category}">
                    <div class="category-header" data-toggle="collapse" data-target="#${categoryId}">
                        <i class="fa fa-chevron-down pull-right"></i>
                        <strong>${category}</strong>
                        <span class="category-count">${tests.length} tests</span>
                    </div>
                    <div id="${categoryId}" class="collapse">
                        <div class="category-tests">
                            ${tests.map(test => createTestItem(test, category)).join('')}
                        </div>
                    </div>
                </div>
            `;
        }

        function createTestItem(test, category) {
            const testId = `${category}_${test.name}`.replace(/[^a-zA-Z0-9]/g, '_');
            const isSelected = selectedTests.some(selected => 
                selected.name === test.name && selected.category === category
            );
            
            return `
                <div class="test-item" data-test-name="${test.name}" data-category="${category}">
                    <div class="test-name">${test.name}</div>
                    <div class="test-price">KES ${test.charge.toLocaleString()}</div>
                    <button class="btn-add-test" data-test='${JSON.stringify({...test, category})}' 
                            ${isSelected ? 'disabled' : ''}>
                        ${isSelected ? 'Added' : 'Add'}
                    </button>
                </div>
            `;
        }

        function bindEvents() {
            // Add test to cart
            $(document).on('click', '.btn-add-test', function() {
                const testData = JSON.parse($(this).attr('data-test'));
                addTestToCart(testData);
                $(this).text('Added').prop('disabled', true);
            });

            // Remove test from cart
            $(document).on('click', '.btn-remove-test', function() {
                const testIndex = parseInt($(this).attr('data-index'));
                removeTestFromCart(testIndex);
            });

            // Search functionality
            $('#testSearch').on('input', function() {
                const searchTerm = $(this).val().toLowerCase();
                filterTests(searchTerm);
            });

            // Billing calculations
            $('#discountPercent, #taxPercent').on('input', function() {
                updateBillingSummary();
            });

            // Clear cart
            $('#clearCart').on('click', function() {
                if (confirm('Are you sure you want to clear all selected tests?')) {
                    clearCart();
                }
            });

            // Proceed to laboratory
            $('#proceedToLab').on('click', function() {
                proceedToLaboratory();
            });

            // Collapse/expand categories
            $(document).on('click', '.category-header', function() {
                const icon = $(this).find('.fa');
                setTimeout(() => {
                    if ($(this).attr('aria-expanded') === 'true') {
                        icon.removeClass('fa-chevron-right').addClass('fa-chevron-down');
                    } else {
                        icon.removeClass('fa-chevron-down').addClass('fa-chevron-right');
                    }
                }, 100);
            });
        }

        function addTestToCart(test) {
            // Check if test already exists
            const existingIndex = selectedTests.findIndex(selected => 
                selected.name === test.name && selected.category === test.category
            );

            if (existingIndex === -1) {
                selectedTests.push({
                    name: test.name,
                    charge: test.charge,
                    category: test.category,
                    id: Date.now() + Math.random()
                });
                
                updateCartDisplay();
                updateBillingSummary();
                
                // Show success notification
                showNotification(`${test.name} added to cart`, 'success');
            }
        }

        function removeTestFromCart(index) {
            const removedTest = selectedTests.splice(index, 1)[0];
            updateCartDisplay();
            updateBillingSummary();
            
            // Re-enable the add button
            $(`.btn-add-test[data-test*='"name":"${removedTest.name}"']`).text('Add').prop('disabled', false);
            
            showNotification(`${removedTest.name} removed from cart`, 'info');
        }

        function updateCartDisplay() {
            const container = $('#selectedTests');
            const cartCount = $('#cartCount');
            const clearBtn = $('#clearCart');
            const proceedBtn = $('#proceedToLab');
            const billingSummary = $('#billingSummary');

            cartCount.text(selectedTests.length);

            if (selectedTests.length === 0) {
                container.html(`
                    <div class="empty-cart">
                        <i class="fa fa-shopping-cart fa-3x"></i>
                        <p>No tests selected yet</p>
                    </div>
                `);
                clearBtn.hide();
                proceedBtn.hide();
                billingSummary.hide();
            } else {
                const cartHtml = selectedTests.map((test, index) => `
                    <div class="cart-item">
                        <div class="cart-item-name">${test.name}</div>
                        <div class="cart-item-price">KES ${test.charge.toLocaleString()}</div>
                        <button class="btn-remove-test" data-index="${index}">
                            <i class="fa fa-times"></i>
                        </button>
                    </div>
                `).join('');

                container.html(cartHtml);
                clearBtn.show();
                proceedBtn.show();
                billingSummary.show();
            }
        }

        function updateBillingSummary() {
            if (selectedTests.length === 0) return;

            const subtotal = selectedTests.reduce((sum, test) => sum + test.charge, 0);
            const discountPercent = parseFloat($('#discountPercent').val()) || 0;
            const taxPercent = parseFloat($('#taxPercent').val()) || 0;

            const discountAmount = (subtotal * discountPercent) / 100;
            const taxableAmount = subtotal - discountAmount;
            const taxAmount = (taxableAmount * taxPercent) / 100;
            const totalAmount = taxableAmount + taxAmount;

            $('#subtotal').text(`KES ${subtotal.toLocaleString()}`);
            $('#discountAmount').text(`KES ${discountAmount.toLocaleString()}`);
            $('#taxAmount').text(`KES ${taxAmount.toLocaleString()}`);
            $('#totalAmount').text(`KES ${totalAmount.toLocaleString()}`);
        }

        function filterTests(searchTerm) {
            if (!searchTerm) {
                $('.test-category-card').show();
                $('.test-item').show();
                return;
            }

            $('.test-category-card').each(function() {
                const $category = $(this);
                const $tests = $category.find('.test-item');
                let hasVisibleTests = false;

                $tests.each(function() {
                    const testName = $(this).find('.test-name').text().toLowerCase();
                    if (testName.includes(searchTerm)) {
                        $(this).show();
                        hasVisibleTests = true;
                    } else {
                        $(this).hide();
                    }
                });

                if (hasVisibleTests) {
                    $category.show();
                    $category.find('.collapse').addClass('in');
                } else {
                    $category.hide();
                }
            });
        }

        function clearCart() {
            selectedTests = [];
            updateCartDisplay();
            updateBillingSummary();
            
            // Re-enable all add buttons
            $('.btn-add-test').text('Add').prop('disabled', false);
            
            showNotification('Cart cleared', 'info');
        }

        function proceedToLaboratory() {
            if (selectedTests.length === 0) {
                showNotification('Please select at least one test', 'warning');
                return;
            }

            // Calculate final billing
            const subtotal = selectedTests.reduce((sum, test) => sum + test.charge, 0);
            const discountPercent = parseFloat($('#discountPercent').val()) || 0;
            const taxPercent = parseFloat($('#taxPercent').val()) || 0;
            const discountAmount = (subtotal * discountPercent) / 100;
            const taxableAmount = subtotal - discountAmount;
            const taxAmount = (taxableAmount * taxPercent) / 100;
            const totalAmount = taxableAmount + taxAmount;

            const labOrder = {
                tests: selectedTests,
                billing: {
                    subtotal: subtotal,
                    discountPercent: discountPercent,
                    discountAmount: discountAmount,
                    taxPercent: taxPercent,
                    taxAmount: taxAmount,
                    totalAmount: totalAmount
                },
                timestamp: new Date().toISOString()
            };

            // Store the order data (you can integrate this with your backend)
            localStorage.setItem('labOrder', JSON.stringify(labOrder));
            
            // Show confirmation
            const confirmModal = `
                <div class="modal fade" id="confirmModal" tabindex="-1">
                    <div class="modal-dialog">
                        <div class="modal-content">
                            <div class="modal-header">
                                <button type="button" class="close" data-dismiss="modal">&times;</button>
                                <h4 class="modal-title">Confirm Laboratory Order</h4>
                            </div>
                            <div class="modal-body">
                                <p><strong>Selected Tests:</strong> ${selectedTests.length}</p>
                                <p><strong>Total Amount:</strong> KES ${totalAmount.toLocaleString()}</p>
                                <p>Do you want to proceed with this laboratory order?</p>
                            </div>
                            <div class="modal-footer">
                                <button type="button" class="btn btn-default" data-dismiss="modal">Cancel</button>
                                <button type="button" class="btn btn-primary" onclick="confirmLabOrder()">Confirm Order</button>
                            </div>
                        </div>
                    </div>
                </div>
            `;
            
            $('body').append(confirmModal);
            $('#confirmModal').modal('show');
        }

        function confirmLabOrder() {
            $('#confirmModal').modal('hide');
            
            // Here you would typically send the data to your backend
            // For now, we'll just show a success message
            showNotification('Laboratory order created successfully!', 'success');
            
            // Clear the cart after successful order
            setTimeout(() => {
                clearCart();
            }, 2000);
        }

        function showNotification(message, type = 'info') {
            const alertClass = type === 'success' ? 'alert-success' : 
                             type === 'warning' ? 'alert-warning' : 
                             type === 'error' ? 'alert-danger' : 'alert-info';
            
            const notification = `
                <div class="alert ${alertClass} alert-dismissible" style="position: fixed; top: 20px; right: 20px; z-index: 9999; min-width: 300px;">
                    <button type="button" class="close" data-dismiss="alert">&times;</button>
                    ${message}
                </div>
            `;
            
            $('body').append(notification);
            
            // Auto-dismiss after 3 seconds
            setTimeout(() => {
                $('.alert').last().fadeOut();
            }, 3000);
        }

        // Integration functions for your existing system
        window.labTestSelection = {
            getSelectedTests: function() {
                return selectedTests;
            },
            
            getBillingInfo: function() {
                const subtotal = selectedTests.reduce((sum, test) => sum + test.charge, 0);
                const discountPercent = parseFloat($('#discountPercent').val()) || 0;
                const taxPercent = parseFloat($('#taxPercent').val()) || 0;
                const discountAmount = (subtotal * discountPercent) / 100;
                const taxableAmount = subtotal - discountAmount;
                const taxAmount = (taxableAmount * taxPercent) / 100;
                const totalAmount = taxableAmount + taxAmount;
                
                return {
                    subtotal,
                    discountPercent,
                    discountAmount,
                    taxPercent,
                    taxAmount,
                    totalAmount
                };
            },
            
            clearSelection: function() {
                clearCart();
            },
            
            addTest: function(testName, charge, category = 'General Tests') {
                addTestToCart({ name: testName, charge: charge, category: category });
            }
        };
    </script>