<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Complete HR Staff Management System</title>
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/4.7.0/css/all.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/twitter-bootstrap/3.4.1/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/datatables/1.10.21/css/dataTables.bootstrap.min.css" rel="stylesheet">
    <style>
        .content-wrapper {
            padding: 20px;
            background-color: #f4f4f4;
        }
        
        .box {
            background: #fff;
            border-radius: 3px;
            box-shadow: 0 1px 3px rgba(0,0,0,0.12), 0 1px 2px rgba(0,0,0,0.24);
            margin-bottom: 20px;
        }
        
        .box-header {
            border-bottom: 1px solid #f4f4f4;
            padding: 10px 15px;
            background: #fafafa;
        }
        
        .box-body {
            padding: 15px;
        }
        
        .nav-tabs-custom {
            margin-bottom: 20px;
        }
        
        .nav-tabs-custom .nav-tabs {
            border-bottom-color: #3c8dbc;
        }
        
        .nav-tabs-custom .nav-tabs li.active a {
            background-color: #3c8dbc;
            color: #fff;
        }
        
        .form-group {
            margin-bottom: 15px;
        }
        
        .req {
            color: #dd4b39;
        }
        
        .info-box {
            display: block;
            min-height: 90px;
            background: #fff;
            width: 100%;
            box-shadow: 0 1px 1px rgba(0,0,0,0.1);
            border-radius: 2px;
            margin-bottom: 15px;
        }
        
        .info-box-icon {
            border-top-left-radius: 2px;
            border-top-right-radius: 0;
            border-bottom-right-radius: 0;
            border-bottom-left-radius: 2px;
            display: block;
            float: left;
            height: 90px;
            width: 90px;
            text-align: center;
            font-size: 45px;
            line-height: 90px;
            background: rgba(0,0,0,0.2);
        }
        
        .info-box-content {
            padding: 5px 10px;
            margin-left: 90px;
        }
        
        .bg-blue { background-color: #3c8dbc !important; }
        .bg-green { background-color: #00a65a !important; }
        .bg-yellow { background-color: #f39c12 !important; }
        .bg-red { background-color: #dd4b39 !important; }
        
        .modal-lg {
            width: 900px;
        }
        
        .table-responsive {
            overflow-x: auto;
        }
        
        .btn-group-justified {
            display: table;
            width: 100%;
            table-layout: fixed;
            border-collapse: separate;
        }
        
        .progress {
            background-color: #f4f4f4;
            border-radius: 4px;
            height: 20px;
            margin-bottom: 5px;
        }
        
        .progress-bar {
            background-color: #3c8dbc;
            height: 100%;
            border-radius: 4px;
            transition: width 0.6s ease;
        }
        
        .staff-photo {
            width: 100px;
            height: 100px;
            object-fit: cover;
            border-radius: 50%;
            border: 3px solid #ddd;
        }
        
        .document-preview {
            max-width: 100%;
            max-height: 300px;
        }
        
        .status-badge {
            display: inline-block;
            padding: 0.25em 0.6em;
            font-size: 0.75em;
            font-weight: 700;
            line-height: 1;
            text-align: center;
            white-space: nowrap;
            vertical-align: baseline;
            border-radius: 0.25rem;
        }
        
        .status-active {
            background-color: #28a745;
            color: #fff;
        }
        
        .status-inactive {
            background-color: #dc3545;
            color: #fff;
        }
        
        .p9-controls {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 20px;
            border-radius: 8px;
            margin-bottom: 20px;
        }
        
        .dashboard-stats {
            background: #fff;
            padding: 20px;
            border-radius: 8px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
            margin-bottom: 20px;
        }
        
        .stat-card {
            text-align: center;
            padding: 15px;
        }
        
        .stat-number {
            font-size: 2em;
            font-weight: bold;
            margin-bottom: 5px;
        }
        
        .file-upload-area {
            border: 2px dashed #ccc;
            border-radius: 10px;
            padding: 20px;
            text-align: center;
            margin: 10px 0;
            transition: border-color 0.3s;
        }
        
        .file-upload-area:hover {
            border-color: #3c8dbc;
        }
        
        .file-upload-area.dragover {
            border-color: #28a745;
            background-color: #f8f9fa;
        }
    </style>
</head>
<body>
    <div class="content-wrapper">
        <!-- Header Section -->
        <section class="content-header">
            <h1>
                <i class="fa fa-users"></i> Staff Management System
                <small>Complete HR Management Dashboard</small>
            </h1>
            <ol class="breadcrumb">
                <li><a href="#"><i class="fa fa-dashboard"></i> Dashboard</a></li>
                <li><a href="#"><i class="fa fa-users"></i> HR</a></li>
                <li class="active">Staff Management</li>
            </ol>
        </section>

        <!-- Dashboard Stats -->
        <section class="content">
            <div class="row">
                <div class="col-md-3 col-sm-6">
                    <div class="info-box bg-blue">
                        <span class="info-box-icon"><i class="fa fa-users"></i></span>
                        <div class="info-box-content">
                            <span class="info-box-text">Total Staff</span>
                            <span class="info-box-number" id="totalStaff">Loading...</span>
                            <div class="progress">
                                <div class="progress-bar" style="width: 85%"></div>
                            </div>
                            <span class="progress-description">Active Staff</span>
                        </div>
                    </div>
                </div>

                <div class="col-md-3 col-sm-6">
                    <div class="info-box bg-green">
                        <span class="info-box-icon"><i class="fa fa-user-plus"></i></span>
                        <div class="info-box-content">
                            <span class="info-box-text">New This Month</span>
                            <span class="info-box-number" id="newStaff">Loading...</span>
                            <div class="progress">
                                <div class="progress-bar" style="width: 60%"></div>
                            </div>
                            <span class="progress-description">Recent hires</span>
                        </div>
                    </div>
                </div>

                <div class="col-md-3 col-sm-6">
                    <div class="info-box bg-yellow">
                        <span class="info-box-icon"><i class="fa fa-file-pdf-o"></i></span>
                        <div class="info-box-content">
                            <span class="info-box-text">P9 Generated</span>
                            <span class="info-box-number" id="p9Generated">Loading...</span>
                            <div class="progress">
                                <div class="progress-bar" style="width: 91%"></div>
                            </div>
                            <span class="progress-description">Completed</span>
                        </div>
                    </div>
                </div>

                <div class="col-md-3 col-sm-6">
                    <div class="info-box bg-red">
                        <span class="info-box-icon"><i class="fa fa-exclamation-triangle"></i></span>
                        <div class="info-box-content">
                            <span class="info-box-text">Pending Actions</span>
                            <span class="info-box-number" id="pendingActions">Loading...</span>
                            <div class="progress">
                                <div class="progress-bar" style="width: 15%"></div>
                            </div>
                            <span class="progress-description">Requires attention</span>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Main Staff Management Box -->
            <div class="box box-primary">
                <div class="box-header with-border">
                    <h3 class="box-title"><i class="fa fa-list"></i> Staff Directory</h3>
                    <div class="box-tools pull-right">
                        <button type="button" class="btn btn-primary btn-sm" onclick="addStaff()">
                            <i class="fa fa-plus"></i> Add Staff
                        </button>
                        <button type="button" class="btn btn-info btn-sm" onclick="exportStaff()">
                            <i class="fa fa-download"></i> Export
                        </button>
                        <button type="button" class="btn btn-success btn-sm" onclick="importStaff()">
                            <i class="fa fa-upload"></i> Import
                        </button>
                        <button type="button" class="btn btn-warning btn-sm" onclick="generateP9Bulk()">
                            <i class="fa fa-file-pdf-o"></i> Generate P9s
                        </button>
                    </div>
                </div>
                
                <!-- Filters Section -->
                <div class="box-body">
                    <div class="row">
                        <div class="col-md-3">
                            <div class="form-group">
                                <label>Department</label>
                                <select class="form-control" id="filter_department">
                                    <option value="">All Departments</option>
                                </select>
                            </div>
                        </div>
                        <div class="col-md-3">
                            <div class="form-group">
                                <label>Role</label>
                                <select class="form-control" id="filter_role">
                                    <option value="">All Roles</option>
                                </select>
                            </div>
                        </div>
                        <div class="col-md-3">
                            <div class="form-group">
                                <label>Employment Type</label>
                                <select class="form-control" id="filter_employment_type">
                                    <option value="">All Types</option>
                                    <option value="permanent">Permanent</option>
                                    <option value="contract">Contract</option>
                                    <option value="temporary">Temporary</option>
                                    <option value="intern">Intern</option>
                                </select>
                            </div>
                        </div>
                        <div class="col-md-3">
                            <div class="form-group">
                                <label>Status</label>
                                <select class="form-control" id="filter_status">
                                    <option value="">All Status</option>
                                    <option value="yes">Active</option>
                                    <option value="no">Inactive</option>
                                </select>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Staff Table -->
                    <div class="table-responsive">
                        <table class="table table-striped table-bordered table-hover" id="staff-table">
                            <thead>
                                <tr>
                                    <th>#</th>
                                    <th>Photo</th>
                                    <th>Employee ID</th>
                                    <th>Name</th>
                                    <th>Department</th>
                                    <th>Role</th>
                                    <th>Employment Type</th>
                                    <th>Joining Date</th>
                                    <th>Status</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <!-- Data will be loaded via AJAX -->
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>

            <!-- P9 Management Section -->
            <div class="box box-info">
                <div class="box-header with-border">
                    <h3 class="box-title"><i class="fa fa-file-text"></i> P9 Tax Certificate Management</h3>
                </div>
                <div class="box-body">
                    <div class="p9-controls">
                        <div class="row">
                            <div class="col-md-3">
                                <div class="form-group">
                                    <label style="color: white;">Tax Year</label>
                                    <select class="form-control" id="p9_year">
                                        <option value="2024" selected>2024</option>
                                        <option value="2023">2023</option>
                                        <option value="2022">2022</option>
                                    </select>
                                </div>
                            </div>
                            <div class="col-md-3">
                                <div class="form-group">
                                    <label style="color: white;">Department</label>
                                    <select class="form-control" id="p9_department">
                                        <option value="">All Departments</option>
                                    </select>
                                </div>
                            </div>
                            <div class="col-md-3">
                                <div class="form-group">
                                    <label style="color: white;">Action</label>
                                    <select class="form-control" id="p9_action">
                                        <option value="">Select Action</option>
                                        <option value="generate">Generate P9s</option>
                                        <option value="email">Email P9s</option>
                                        <option value="download">Download All</option>
                                    </select>
                                </div>
                            </div>
                            <div class="col-md-3">
                                <div class="form-group">
                                    <label style="color: white;">&nbsp;</label>
                                    <div>
                                        <button class="btn btn-warning btn-block" onclick="executeP9Action()">
                                            <i class="fa fa-cogs"></i> Execute Action
                                        </button>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </section>
    </div>

    <!-- Add/Edit Staff Modal -->
    <div class="modal fade" id="staffModal" tabindex="-1" role="dialog">
        <div class="modal-dialog modal-lg" role="document">
            <div class="modal-content">
                <div class="modal-header">
                    <button type="button" class="close" data-dismiss="modal">&times;</button>
                    <h4 class="modal-title" id="staffModalTitle">Add New Staff</h4>
                </div>
                <form id="staffForm">
                    <div class="modal-body">
                        <div class="nav-tabs-custom">
                            <ul class="nav nav-tabs">
                                <li class="active"><a href="#personal-info" data-toggle="tab">Personal Info</a></li>
                                <li><a href="#employment-info" data-toggle="tab">Employment Info</a></li>
                                <li><a href="#contact-info" data-toggle="tab">Contact Info</a></li>
                                <li><a href="#emergency-contact" data-toggle="tab">Emergency Contact</a></li>
                            </ul>
                            <div class="tab-content">
                                <!-- Personal Information Tab -->
                                <div class="active tab-pane" id="personal-info">
                                    <div class="row">
                                        <div class="col-md-4">
                                            <div class="form-group">
                                                <label>Photo</label>
                                                <div class="file-upload-area" id="photoUpload">
                                                    <i class="fa fa-camera fa-3x"></i>
                                                    <p>Click to upload photo or drag & drop</p>
                                                    <input type="file" id="photoFile" accept="image/*" style="display: none;">
                                                </div>
                                                <div id="photoPreview" style="display: none;">
                                                    <img id="previewImage" class="staff-photo" src="" alt="Preview">
                                                </div>
                                            </div>
                                        </div>
                                        <div class="col-md-8">
                                            <div class="row">
                                                <div class="col-md-6">
                                                    <div class="form-group">
                                                        <label>Employee ID <span class="req">*</span></label>
                                                        <input type="text" class="form-control" name="employee_id" id="employee_id" required>
                                                    </div>
                                                </div>
                                                <div class="col-md-6">
                                                    <div class="form-group">
                                                        <label>Staff Number</label>
                                                        <input type="text" class="form-control" name="staff_number" id="staff_number">
                                                    </div>
                                                </div>
                                            </div>
                                            <div class="row">
                                                <div class="col-md-6">
                                                    <div class="form-group">
                                                        <label>First Name <span class="req">*</span></label>
                                                        <input type="text" class="form-control" name="name" id="first_name" required>
                                                    </div>
                                                </div>
                                                <div class="col-md-6">
                                                    <div class="form-group">
                                                        <label>Last Name <span class="req">*</span></label>
                                                        <input type="text" class="form-control" name="surname" id="last_name" required>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                    </div>

                                    <div class="row">
                                        <div class="col-md-6">
                                            <div class="form-group">
                                                <label>Father's Name</label>
                                                <input type="text" class="form-control" name="father_name" id="father_name">
                                            </div>
                                        </div>
                                        <div class="col-md-6">
                                            <div class="form-group">
                                                <label>Mother's Name</label>
                                                <input type="text" class="form-control" name="mother_name" id="mother_name">
                                            </div>
                                        </div>
                                    </div>

                                    <div class="row">
                                        <div class="col-md-4">
                                            <div class="form-group">
                                                <label>Gender</label>
                                                <select class="form-control" name="gender" id="gender">
                                                    <option value="">Select Gender</option>
                                                    <option value="male">Male</option>
                                                    <option value="female">Female</option>
                                                    <option value="other">Other</option>
                                                </select>
                                            </div>
                                        </div>
                                        <div class="col-md-4">
                                            <div class="form-group">
                                                <label>Date of Birth</label>
                                                <input type="date" class="form-control" name="dob" id="date_of_birth">
                                            </div>
                                        </div>
                                        <div class="col-md-4">
                                            <div class="form-group">
                                                <label>Marital Status</label>
                                                <select class="form-control" name="marital_status" id="marital_status">
                                                    <option value="">Select Status</option>
                                                    <option value="single">Single</option>
                                                    <option value="married">Married</option>
                                                    <option value="divorced">Divorced</option>
                                                    <option value="widowed">Widowed</option>
                                                </select>
                                            </div>
                                        </div>
                                    </div>
                                </div>

                                <!-- Employment Information Tab -->
                                <div class="tab-pane" id="employment-info">
                                    <div class="row">
                                        <div class="col-md-6">
                                            <div class="form-group">
                                                <label>Department <span class="req">*</span></label>
                                                <select class="form-control" name="department" id="department_id" required>
                                                    <option value="">Select Department</option>
                                                </select>
                                            </div>
                                        </div>
                                        <div class="col-md-6">
                                            <div class="form-group">
                                                <label>Role <span class="req">*</span></label>
                                                <select class="form-control" name="role" id="role_id" required>
                                                    <option value="">Select Role</option>
                                                </select>
                                            </div>
                                        </div>
                                    </div>

                                    <div class="row">
                                        <div class="col-md-6">
                                            <div class="form-group">
                                                <label>Employment Type</label>
                                                <select class="form-control" name="employment_type" id="employment_type">
                                                    <option value="">Select Type</option>
                                                    <option value="permanent">Permanent</option>
                                                    <option value="contract">Contract</option>
                                                    <option value="temporary">Temporary</option>
                                                    <option value="intern">Intern</option>
                                                </select>
                                            </div>
                                        </div>
                                        <div class="col-md-6">
                                            <div class="form-group">
                                                <label>Joining Date</label>
                                                <input type="date" class="form-control" name="date_of_joining" id="joining_date">
                                            </div>
                                        </div>
                                    </div>

                                    <div class="row">
                                        <div class="col-md-6">
                                            <div class="form-group">
                                                <label>Basic Salary (KSh)</label>
                                                <input type="number" class="form-control" name="basic_salary" id="basic_salary" min="0" step="0.01">
                                            </div>
                                        </div>
                                        <div class="col-md-6">
                                            <div class="form-group">
                                                <label>EPF Number</label>
                                                <input type="text" class="form-control" name="epf" id="kra_pin">
                                            </div>
                                        </div>
                                    </div>

                                    <div class="row">
                                        <div class="col-md-4">
                                            <div class="form-group">
                                                <label>Blood Group</label>
                                                <select class="form-control" name="blood_group" id="blood_group">
                                                    <option value="">Select Blood Group</option>
                                                    <option value="A+">A+</option>
                                                    <option value="A-">A-</option>
                                                    <option value="B+">B+</option>
                                                    <option value="B-">B-</option>
                                                    <option value="AB+">AB+</option>
                                                    <option value="AB-">AB-</option>
                                                    <option value="O+">O+</option>
                                                    <option value="O-">O-</option>
                                                </select>
                                            </div>
                                        </div>
                                        <div class="col-md-4">
                                            <div class="form-group">
                                                <label>Qualification</label>
                                                <input type="text" class="form-control" name="qualification" id="qualification">
                                            </div>
                                        </div>
                                        <div class="col-md-4">
                                            <div class="form-group">
                                                <label>Work Experience</label>
                                                <input type="text" class="form-control" name="work_exp" id="work_exp">
                                            </div>
                                        </div>
                                    </div>
                                </div>

                                <!-- Contact Information Tab -->
                                <div class="tab-pane" id="contact-info">
                                    <div class="row">
                                        <div class="col-md-6">
                                            <div class="form-group">
                                                <label>Email <span class="req">*</span></label>
                                                <input type="email" class="form-control" name="email" id="email" required>
                                            </div>
                                        </div>
                                        <div class="col-md-6">
                                            <div class="form-group">
                                                <label>Phone <span class="req">*</span></label>
                                                <input type="text" class="form-control" name="phone" id="phone" required>
                                            </div>
                                        </div>
                                    </div>
                                    <div class="row">
                                        <div class="col-md-12">
                                            <div class="form-group">
                                                <label>Current Address</label>
                                                <textarea class="form-control" name="address" id="current_address" rows="3"></textarea>
                                            </div>
                                        </div>
                                    </div>
                                    <div class="row">
                                        <div class="col-md-12">
                                            <div class="form-group">
                                                <label>Permanent Address</label>
                                                <textarea class="form-control" name="permanent_address" id="permanent_address" rows="3"></textarea>
                                            </div>
                                        </div>
                                    </div>
                                </div>

                                <!-- Emergency Contact Tab -->
                                <div class="tab-pane" id="emergency-contact">
                                    <div class="row">
                                        <div class="col-md-6">
                                            <div class="form-group">
                                                <label>Emergency Contact Name</label>
                                                <input type="text" class="form-control" name="emergency_contact_name" id="emergency_contact_name">
                                            </div>
                                        </div>
                                        <div class="col-md-6">
                                            <div class="form-group">
                                                <label>Emergency Contact Phone</label>
                                                <input type="text" class="form-control" name="emergency_contact_phone" id="emergency_contact_phone">
                                            </div>
                                        </div>
                                    </div>
                                    <div class="row">
                                        <div class="col-md-12">
                                            <div class="form-group">
                                                <label>Additional Notes</label>
                                                <textarea class="form-control" name="note" id="notes" rows="4" placeholder="Additional notes about the staff member"></textarea>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <input type="hidden" name="staff_id" id="staff_id">
                        <button type="button" class="btn btn-default" data-dismiss="modal">Close</button>
                        <button type="submit" class="btn btn-primary" id="saveStaffBtn">
                            <i class="fa fa-save"></i> Save Staff
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- View Staff Details Modal -->
    <div class="modal fade" id="viewStaffModal" tabindex="-1" role="dialog">
        <div class="modal-dialog modal-lg" role="document">
            <div class="modal-content">
                <div class="modal-header">
                    <button type="button" class="close" data-dismiss="modal">&times;</button>
                    <h4 class="modal-title">Staff Details</h4>
                </div>
                <div class="modal-body" id="staffDetailsContent">
                    <!-- Staff details will be loaded here -->
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-default" data-dismiss="modal">Close</button>
                    <button type="button" class="btn btn-info" onclick="printStaffDetails()">
                        <i class="fa fa-print"></i> Print
                    </button>
                </div>
            </div>
        </div>
    </div>

    <!-- Documents Modal -->
    <div class="modal fade" id="documentsModal" tabindex="-1" role="dialog">
        <div class="modal-dialog modal-lg" role="document">
            <div class="modal-content">
                <div class="modal-header">
                    <button type="button" class="close" data-dismiss="modal">&times;</button>
                    <h4 class="modal-title">Staff Documents</h4>
                </div>
                <div class="modal-body">
                    <div class="row">
                        <div class="col-md-6">
                            <button type="button" class="btn btn-primary btn-sm" onclick="uploadDocument()">
                                <i class="fa fa-plus"></i> Upload Document
                            </button>
                        </div>
                        <div class="col-md-6 text-right">
                            <div class="btn-group btn-group-sm">
                                <button type="button" class="btn btn-info dropdown-toggle" data-toggle="dropdown">
                                    <i class="fa fa-cog"></i> Actions <span class="caret"></span>
                                </button>
                                <ul class="dropdown-menu">
                                    <li><a href="#" onclick="bulkDownloadDocuments()"><i class="fa fa-download"></i> Download All</a></li>
                                    <li><a href="#" onclick="organizeDocuments()"><i class="fa fa-folder"></i> Organize</a></li>
                                    <li class="divider"></li>
                                    <li><a href="#" onclick="auditDocuments()"><i class="fa fa-history"></i> Audit Trail</a></li>
                                </ul>
                            </div>
                        </div>
                    </div>
                    <br>
                    <div class="table-responsive">
                        <table class="table table-striped" id="documents-table">
                            <thead>
                                <tr>
                                    <th><input type="checkbox" id="selectAllDocs"></th>
                                    <th>Document Type</th>
                                    <th>Document Title</th>
                                    <th>Upload Date</th>
                                    <th>Expiry Date</th>
                                    <th>Status</th>
                                    <th>Action</th>
                                </tr>
                            </thead>
                            <tbody id="documentsTableBody">
                                <!-- Documents will be loaded via AJAX -->
                            </tbody>
                        </table>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-default" data-dismiss="modal">Close</button>
                </div>
            </div>
        </div>
    </div>

    <!-- Upload Document Modal -->
    <div class="modal fade" id="uploadDocumentModal" tabindex="-1" role="dialog">
        <div class="modal-dialog" role="document">
            <div class="modal-content">
                <div class="modal-header">
                    <button type="button" class="close" data-dismiss="modal">&times;</button>
                    <h4 class="modal-title">Upload Document</h4>
                </div>
                <form id="uploadDocumentForm" enctype="multipart/form-data">
                    <div class="modal-body">
                        <input type="hidden" name="staff_id" id="upload_staff_id">
                        <div class="form-group">
                            <label>Document Type <span class="req">*</span></label>
                            <select class="form-control" name="document_type" id="document_type" required>
                                <option value="">Select Document Type</option>
                                <option value="contract">Employment Contract</option>
                                <option value="id_copy">ID Copy</option>
                                <option value="certificate">Certificate</option>
                                <option value="license">Professional License</option>
                                <option value="resume">Resume/CV</option>
                                <option value="medical">Medical Certificate</option>
                                <option value="passport">Passport Copy</option>
                                <option value="other">Other</option>
                            </select>
                        </div>
                        <div class="form-group">
                            <label>Document Title <span class="req">*</span></label>
                            <input type="text" class="form-control" name="document_title" id="document_title" required>
                        </div>
                        <div class="form-group">
                            <label>Choose File <span class="req">*</span></label>
                            <div class="file-upload-area" id="documentUpload">
                                <i class="fa fa-file-o fa-3x"></i>
                                <p>Click to select file or drag & drop</p>
                                <input type="file" name="document_file" id="document_file" required accept=".pdf,.doc,.docx,.jpg,.jpeg,.png" style="display: none;">
                            </div>
                            <small class="text-muted">Supported formats: PDF, Word, Images (Max: 5MB)</small>
                        </div>
                        <div class="form-group">
                            <label>Expiry Date</label>
                            <input type="date" class="form-control" name="expiry_date" id="expiry_date">
                            <small class="text-muted">Leave blank if document doesn't expire</small>
                        </div>
                        <div class="form-group">
                            <label>Notes</label>
                            <textarea class="form-control" name="document_notes" id="document_notes" rows="3"></textarea>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-default" data-dismiss="modal">Cancel</button>
                        <button type="submit" class="btn btn-primary">
                            <i class="fa fa-upload"></i> Upload Document
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- P9 Generation Modal -->
    <div class="modal fade" id="p9Modal" tabindex="-1" role="dialog">
        <div class="modal-dialog modal-lg" role="document">
            <div class="modal-content">
                <div class="modal-header">
                    <button type="button" class="close" data-dismiss="modal">&times;</button>
                    <h4 class="modal-title">P9 Tax Certificate Generation</h4>
                </div>
                <div class="modal-body">
                    <div class="row">
                        <div class="col-md-6">
                            <h5>Generation Settings</h5>
                            <div class="form-group">
                                <label>Tax Year</label>
                                <select class="form-control" id="p9GenerationYear">
                                    <option value="2024" selected>2024</option>
                                    <option value="2023">2023</option>
                                    <option value="2022">2022</option>
                                </select>
                            </div>
                            <div class="form-group">
                                <label>Generation Type</label>
                                <select class="form-control" id="p9GenerationType">
                                    <option value="individual">Individual Staff</option>
                                    <option value="department">Entire Department</option>
                                    <option value="all">All Active Staff</option>
                                </select>
                            </div>
                            <div class="form-group">
                                <label>Delivery Method</label>
                                <div class="checkbox">
                                    <label><input type="checkbox" id="p9Email"> Email to Staff</label>
                                </div>
                                <div class="checkbox">
                                    <label><input type="checkbox" id="p9Download" checked> Download PDF</label>
                                </div>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <h5>Preview Summary</h5>
                            <div class="dashboard-stats">
                                <div class="row">
                                    <div class="col-md-6 stat-card">
                                        <div class="stat-number text-blue" id="p9StaffCount">0</div>
                                        <div>Staff to Process</div>
                                    </div>
                                    <div class="col-md-6 stat-card">
                                        <div class="stat-number text-green" id="p9EstimatedTime">0</div>
                                        <div>Est. Time (min)</div>
                                    </div>
                                </div>
                            </div>
                            <div class="alert alert-info">
                                <i class="fa fa-info-circle"></i>
                                <strong>Note:</strong> P9 forms will be generated based on payroll data for the selected year. Ensure all payroll records are up to date.
                            </div>
                        </div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-default" data-dismiss="modal">Cancel</button>
                    <button type="button" class="btn btn-primary" onclick="startP9Generation()">
                        <i class="fa fa-cogs"></i> Generate P9 Forms
                    </button>
                </div>
            </div>
        </div>
    </div>

    <!-- Progress Modal -->
    <div class="modal fade" id="progressModal" tabindex="-1" role="dialog" data-backdrop="static">
        <div class="modal-dialog" role="document">
            <div class="modal-content">
                <div class="modal-header">
                    <h4 class="modal-title">Processing...</h4>
                </div>
                <div class="modal-body">
                    <div class="progress">
                        <div class="progress-bar progress-bar-striped active" id="progressBar" style="width: 0%"></div>
                    </div>
                    <p id="progressText">Initializing...</p>
                </div>
            </div>
        </div>
    </div>

    <!-- Scripts -->
    <script src="https://cdnjs.cloudflare.com/ajax/libs/jquery/3.6.0/jquery.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/twitter-bootstrap/3.4.1/js/bootstrap.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/datatables/1.10.21/js/jquery.dataTables.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/datatables/1.10.21/js/dataTables.bootstrap.min.js"></script>

    <script>
        // Global variables
        var staffTable;
        var currentStaffId = null;
        var baseUrl = '<?php echo base_url(); ?>'; // Adjust this based on your CodeIgniter setup

        $(document).ready(function() {
            // Load initial data
            loadDashboardStats();
            loadDepartments();
            loadRoles();
            
            // Initialize DataTable
            initializeStaffTable();
            
            // Initialize file upload areas
            initializeFileUploads();
            
            // Form handlers
            $('#staffForm').on('submit', handleStaffFormSubmit);
            $('#uploadDocumentForm').on('submit', handleDocumentUpload);
            
            // Filter change events
            $('#filter_department, #filter_role, #filter_employment_type, #filter_status').change(function() {
                if (staffTable) {
                    staffTable.draw();
                }
            });

            console.log('HR Staff Management System Initialized');
        });

        // Load dashboard statistics from database
        function loadDashboardStats() {
            $.ajax({
                url: baseUrl + 'hr/getDashboardStats',
                type: 'GET',
                dataType: 'json',
                success: function(response) {
                    if (response.status === 'success') {
                        var stats = response.data;
                        $('#totalStaff').text(stats.total_staff || 0);
                        $('#newStaff').text(stats.recent_hires || 0);
                        $('#p9Generated').text(stats.p9_generated || 0);
                        $('#pendingActions').text(stats.pending_actions || 0);
                    }
                },
                error: function() {
                    $('#totalStaff').text('0');
                    $('#newStaff').text('0');
                    $('#p9Generated').text('0');
                    $('#pendingActions').text('0');
                }
            });
        }

        // Load departments from database
        function loadDepartments() {
            $.ajax({
                url: baseUrl + 'hr/getDepartments',
                type: 'GET',
                dataType: 'json',
                success: function(response) {
                    if (response.status === 'success') {
                        var departments = response.data;
                        var filterSelect = $('#filter_department');
                        var formSelect = $('#department_id');
                        var p9Select = $('#p9_department');
                        
                        filterSelect.empty().append('<option value="">All Departments</option>');
                        formSelect.empty().append('<option value="">Select Department</option>');
                        p9Select.empty().append('<option value="">All Departments</option>');
                        
                        $.each(departments, function(index, dept) {
                            var option = '<option value="' + dept.id + '">' + dept.department_name + '</option>';
                            filterSelect.append(option);
                            formSelect.append(option);
                            p9Select.append(option);
                        });
                    }
                },
                error: function() {
                    console.error('Failed to load departments');
                }
            });
        }

        // Load roles from database
        function loadRoles() {
            $.ajax({
                url: baseUrl + 'hr/getRoles',
                type: 'GET',
                dataType: 'json',
                success: function(response) {
                    if (response.status === 'success') {
                        var roles = response.data;
                        var filterSelect = $('#filter_role');
                        var formSelect = $('#role_id');
                        
                        filterSelect.empty().append('<option value="">All Roles</option>');
                        formSelect.empty().append('<option value="">Select Role</option>');
                        
                        $.each(roles, function(index, role) {
                            var option = '<option value="' + role.id + '">' + role.name + '</option>';
                            filterSelect.append(option);
                            formSelect.append(option);
                        });
                    }
                },
                error: function() {
                    console.error('Failed to load roles');
                }
            });
        }

        function initializeStaffTable() {
            staffTable = $('#staff-table').DataTable({
                processing: true,
                serverSide: true,
                responsive: true,
                ajax: {
                    url: baseUrl + 'hr/getStaffList',
                    type: 'POST',
                    data: function(d) {
                        d.filter_department = $('#filter_department').val();
                        d.filter_role = $('#filter_role').val();
                        d.filter_employment_type = $('#filter_employment_type').val();
                        d.filter_status = $('#filter_status').val();
                    }
                },
                order: [[2, 'asc']],
                language: {
                    processing: "Loading staff data..."
                },
                columnDefs: [
                    { targets: [0, 1, 9], orderable: false }
                ]
            });
        }

        function initializeFileUploads() {
            // Photo upload
            $('#photoUpload').on('click', function() {
                $('#photoFile').click();
            });

            $('#photoFile').on('change', function(e) {
                handlePhotoUpload(e.target.files[0]);
            });

            // Document upload
            $('#documentUpload').on('click', function() {
                $('#document_file').click();
            });

            // Drag and drop functionality
            $('.file-upload-area').on('dragover', function(e) {
                e.preventDefault();
                $(this).addClass('dragover');
            });

            $('.file-upload-area').on('dragleave', function(e) {
                e.preventDefault();
                $(this).removeClass('dragover');
            });

            $('.file-upload-area').on('drop', function(e) {
                e.preventDefault();
                $(this).removeClass('dragover');
                var files = e.originalEvent.dataTransfer.files;
                if (files.length > 0) {
                    if ($(this).attr('id') === 'photoUpload') {
                        handlePhotoUpload(files[0]);
                    } else {
                        $('#document_file')[0].files = files;
                    }
                }
            });
        }

        function handlePhotoUpload(file) {
            if (file && file.type.startsWith('image/')) {
                var reader = new FileReader();
                reader.onload = function(e) {
                    $('#previewImage').attr('src', e.target.result);
                    $('#photoPreview').show();
                    $('#photoUpload').hide();
                };
                reader.readAsDataURL(file);
            }
        }

        // Staff Management Functions
        function addStaff() {
            resetStaffForm();
            currentStaffId = null;
            $('#staffModalTitle').text('Add New Staff');
            $('#staffModal').modal('show');
        }

        function editStaff(staffId) {
            currentStaffId = staffId;
            $('#staffModalTitle').text('Edit Staff');
            loadStaffData(staffId);
            $('#staffModal').modal('show');
        }

        function viewStaff(staffId) {
            loadStaffDetails(staffId);
            $('#viewStaffModal').modal('show');
        }

        function viewDocuments(staffId) {
            currentStaffId = staffId;
            $('#upload_staff_id').val(staffId);
            loadStaffDocuments(staffId);
            $('#documentsModal').modal('show');
        }

        function generateP9(staffId) {
            currentStaffId = staffId;
            $('#p9Modal').modal('show');
            updateP9Preview();
        }

        function deleteStaff(staffId) {
            if (confirm('Are you sure you want to delete this staff member?')) {
                $.ajax({
                    url: baseUrl + 'hr/deleteStaff/' + staffId,
                    type: 'POST',
                    dataType: 'json',
                    success: function(response) {
                        if (response.status === 'success') {
                            showNotification('success', response.message);
                            staffTable.draw();
                        } else {
                            showNotification('error', response.message);
                        }
                    },
                    error: function() {
                        showNotification('error', 'Failed to delete staff member');
                    }
                });
            }
        }

        function loadStaffData(staffId) {
            $.ajax({
                url: baseUrl + 'hr/getStaffData/' + staffId,
                type: 'GET',
                dataType: 'json',
                success: function(response) {
                    if (response.status === 'success') {
                        var staff = response.data;
                        
                        // Populate form fields
                        $('#staff_id').val(staff.id);
                        $('#employee_id').val(staff.employee_id);
                        $('#first_name').val(staff.name);
                        $('#last_name').val(staff.surname);
                        $('#father_name').val(staff.father_name);
                        $('#mother_name').val(staff.mother_name);
                        $('#email').val(staff.email);
                        $('#phone').val(staff.phone);
                        $('#emergency_contact_name').val(staff.emergency_contact_name);
                        $('#emergency_contact_phone').val(staff.emergency_contact_phone);
                        $('#gender').val(staff.gender);
                        $('#date_of_birth').val(staff.dob);
                        $('#marital_status').val(staff.marital_status);
                        $('#blood_group').val(staff.blood_group);
                        $('#current_address').val(staff.address);
                        $('#permanent_address').val(staff.permanent_address);
                        $('#qualification').val(staff.qualification);
                        $('#work_exp').val(staff.work_exp);
                        $('#notes').val(staff.note);
                        $('#department_id').val(staff.department);
                        $('#role_id').val(staff.role);
                        $('#employment_type').val(staff.employment_type);
                        $('#joining_date').val(staff.date_of_joining);
                        $('#basic_salary').val(staff.basic_salary);
                        
                        // Show photo if exists
                        if (staff.photo) {
                            $('#previewImage').attr('src', baseUrl + staff.photo);
                            $('#photoPreview').show();
                            $('#photoUpload').hide();
                        }
                    } else {
                        showNotification('error', response.message);
                    }
                },
                error: function() {
                    showNotification('error', 'Failed to load staff data');
                }
            });
        }

        function loadStaffDetails(staffId) {
            $.ajax({
                url: baseUrl + 'hr/viewStaff/' + staffId,
                type: 'GET',
                dataType: 'json',
                success: function(response) {
                    if (response.status === 'success') {
                        var staff = response.data;
                        var photoUrl = staff.photo ? baseUrl + staff.photo : 'https://via.placeholder.com/150x150';
                        
                        var html = `
                            <div class="row">
                                <div class="col-md-4 text-center">
                                    <img src="${photoUrl}" class="staff-photo" alt="Staff Photo">
                                    <h4>${staff.name} ${staff.surname}</h4>
                                    <p class="text-muted">${staff.role_name || 'N/A'}</p>
                                </div>
                                <div class="col-md-8">
                                    <table class="table table-bordered">
                                        <tr><td><strong>Employee ID:</strong></td><td>${staff.employee_id || 'N/A'}</td></tr>
                                        <tr><td><strong>Department:</strong></td><td>${staff.department_name || 'N/A'}</td></tr>
                                        <tr><td><strong>Email:</strong></td><td>${staff.email || 'N/A'}</td></tr>
                                        <tr><td><strong>Phone:</strong></td><td>${staff.phone || 'N/A'}</td></tr>
                                        <tr><td><strong>Joining Date:</strong></td><td>${staff.date_of_joining || 'N/A'}</td></tr>
                                        <tr><td><strong>Employment Type:</strong></td><td>${staff.employment_type || 'N/A'}</td></tr>
                                        <tr><td><strong>Basic Salary:</strong></td><td>KSh ${staff.basic_salary || '0'}</td></tr>
                                    </table>
                                </div>
                            </div>
                        `;
                        $('#staffDetailsContent').html(html);
                    } else {
                        $('#staffDetailsContent').html('<p>Failed to load staff details</p>');
                    }
                },
                error: function() {
                    $('#staffDetailsContent').html('<p>Error loading staff details</p>');
                }
            });
        }

        function loadStaffDocuments(staffId) {
            $.ajax({
                url: baseUrl + 'hr/getStaffDocuments/' + staffId,
                type: 'GET',
                dataType: 'json',
                success: function(response) {
                    if (response.status === 'success') {
                        var documents = response.data;
                        var tbody = $('#documentsTableBody');
                        tbody.empty();
                        
                        if (documents.length > 0) {
                            $.each(documents, function(index, doc) {
                                var expiryStatus = 'Valid';
                                var statusClass = 'label-success';
                                
                                if (doc.expiry_date) {
                                    var expiryDate = new Date(doc.expiry_date);
                                    var today = new Date();
                                    if (expiryDate < today) {
                                        expiryStatus = 'Expired';
                                        statusClass = 'label-danger';
                                    }
                                }
                                
                                var row = `
                                    <tr>
                                        <td><input type="checkbox" class="doc-checkbox" value="${doc.id}"></td>
                                        <td><span class="label label-primary">${doc.document_type}</span></td>
                                        <td>${doc.document_title}</td>
                                        <td>${doc.upload_date || 'N/A'}</td>
                                        <td>${doc.expiry_date || 'N/A'}</td>
                                        <td><span class="label ${statusClass}">${expiryStatus}</span></td>
                                        <td>
                                            <div class="btn-group btn-group-xs">
                                                <button class="btn btn-info" onclick="viewDocument(${doc.id})" title="View"><i class="fa fa-eye"></i></button>
                                                <button class="btn btn-primary" onclick="downloadDocument(${doc.id})" title="Download"><i class="fa fa-download"></i></button>
                                                <button class="btn btn-danger" onclick="deleteDocument(${doc.id})" title="Delete"><i class="fa fa-trash"></i></button>
                                            </div>
                                        </td>
                                    </tr>
                                `;
                                tbody.append(row);
                            });
                        } else {
                            tbody.append('<tr><td colspan="7" class="text-center">No documents found</td></tr>');
                        }
                    }
                },
                error: function() {
                    $('#documentsTableBody').html('<tr><td colspan="7" class="text-center">Error loading documents</td></tr>');
                }
            });
        }

        function handleStaffFormSubmit(e) {
            e.preventDefault();
            
            var formData = new FormData($('#staffForm')[0]);
            var url = currentStaffId ? baseUrl + 'hr/editStaff/' + currentStaffId : baseUrl + 'hr/addStaff';
            
            $.ajax({
                url: url,
                type: 'POST',
                data: formData,
                processData: false,
                contentType: false,
                dataType: 'json',
                success: function(response) {
                    if (response.status === 'success') {
                        $('#staffModal').modal('hide');
                        showNotification('success', response.message);
                        staffTable.draw();
                        loadDashboardStats(); // Refresh stats
                    } else {
                        showNotification('error', response.message);
                    }
                },
                error: function() {
                    showNotification('error', 'Failed to save staff data');
                }
            });
        }

        function handleDocumentUpload(e) {
            e.preventDefault();
            
            var formData = new FormData($('#uploadDocumentForm')[0]);
            
            $.ajax({
                url: baseUrl + 'hr/uploadDocument',
                type: 'POST',
                data: formData,
                processData: false,
                contentType: false,
                dataType: 'json',
                success: function(response) {
                    if (response.status === 'success') {
                        $('#uploadDocumentModal').modal('hide');
                        showNotification('success', response.message);
                        loadStaffDocuments(currentStaffId);
                    } else {
                        showNotification('error', response.message);
                    }
                },
                error: function() {
                    showNotification('error', 'Failed to upload document');
                }
            });
        }

        function resetStaffForm() {
            $('#staffForm')[0].reset();
            $('#staff_id').val('');
            $('#photoPreview').hide();
            $('#photoUpload').show();
            currentStaffId = null;
            $('.nav-tabs a:first').tab('show');
        }

        // P9 Management Functions
        function generateP9Bulk() {
            $('#p9Modal').modal('show');
            updateP9Preview();
        }

        function updateP9Preview() {
            var type = $('#p9GenerationType').val();
            var staffCount = type === 'all' ? 156 : type === 'department' ? 25 : 1;
            var estimatedTime = Math.ceil(staffCount * 0.5);
            
            $('#p9StaffCount').text(staffCount);
            $('#p9EstimatedTime').text(estimatedTime);
        }

        function startP9Generation() {
            $('#p9Modal').modal('hide');
            showProgress('Generating P9 forms...', 0);
            
            var progress = 0;
            var interval = setInterval(function() {
                progress += Math.random() * 20;
                if (progress >= 100) {
                    progress = 100;
                    clearInterval(interval);
                    hideProgress();
                    showNotification('success', 'P9 forms generated successfully!');
                }
                updateProgress(progress, 'Processing staff ' + Math.ceil(progress/10) + ' of 10...');
            }, 500);
        }

        function executeP9Action() {
            var action = $('#p9_action').val();
            if (!action) {
                showNotification('warning', 'Please select an action');
                return;
            }
            
            showProgress('Executing ' + action + '...', 0);
            
            setTimeout(function() {
                hideProgress();
                showNotification('success', 'Action completed successfully!');
            }, 3000);
        }

        // Document Management Functions
        function uploadDocument() {
            if (!currentStaffId) {
                showNotification('error', 'Please select a staff member first');
                return;
            }
            $('#uploadDocumentModal').modal('show');
        }

        function viewDocument(docId) {
            window.open(baseUrl + 'hr/viewDocument/' + docId, '_blank');
        }

        function downloadDocument(docId) {
            window.location = baseUrl + 'hr/downloadDocument/' + docId;
        }

        function deleteDocument(docId) {
            if (confirm('Are you sure you want to delete this document?')) {
                $.ajax({
                    url: baseUrl + 'hr/deleteDocument/' + docId,
                    type: 'POST',
                    dataType: 'json',
                    success: function(response) {
                        if (response.status === 'success') {
                            showNotification('success', response.message);
                            loadStaffDocuments(currentStaffId);
                        } else {
                            showNotification('error', response.message);
                        }
                    },
                    error: function() {
                        showNotification('error', 'Failed to delete document');
                    }
                });
            }
        }

        // Utility Functions
        function exportStaff() {
            window.location = baseUrl + 'hr/exportStaff';
            showNotification('info', 'Export started. File will be downloaded shortly.');
        }

        function importStaff() {
            var input = document.createElement('input');
            input.type = 'file';
            input.accept = '.xlsx,.xls,.csv';
            input.onchange = function(e) {
                var file = e.target.files[0];
                if (file) {
                    var formData = new FormData();
                    formData.append('import_file', file);
                    
                    showProgress('Importing staff data...', 50);
                    
                    $.ajax({
                        url: baseUrl + 'hr/importStaff',
                        type: 'POST',
                        data: formData,
                        processData: false,
                        contentType: false,
                        dataType: 'json',
                        success: function(response) {
                            hideProgress();
                            if (response.status === 'success') {
                                showNotification('success', response.message);
                                staffTable.draw();
                                loadDashboardStats();
                            } else {
                                showNotification('error', response.message);
                            }
                        },
                        error: function() {
                            hideProgress();
                            showNotification('error', 'Failed to import staff data');
                        }
                    });
                }
            };
            input.click();
        }

        function showProgress(text, percent) {
            $('#progressText').text(text);
            $('#progressBar').css('width', percent + '%');
            $('#progressModal').modal('show');
        }

        function updateProgress(percent, text) {
            $('#progressBar').css('width', percent + '%');
            $('#progressText').text(text);
        }

        function hideProgress() {
            $('#progressModal').modal('hide');
        }

        function showNotification(type, message) {
            var alertClass = 'alert-info';
            if (type === 'success') alertClass = 'alert-success';
            else if (type === 'error') alertClass = 'alert-danger';
            else if (type === 'warning') alertClass = 'alert-warning';
            
            var notification = $('<div class="alert ' + alertClass + ' alert-dismissible" style="position: fixed; top: 20px; right: 20px; z-index: 9999; min-width: 300px;">' +
                '<button type="button" class="close" data-dismiss="alert">&times;</button>' +
                '<strong>' + type.charAt(0).toUpperCase() + type.slice(1) + ':</strong> ' + message +
                '</div>');
            
            $('body').append(notification);
            
            setTimeout(function() {
                notification.fadeOut(function() {
                    $(this).remove();
                });
            }, 5000);
        }

        function printStaffDetails() {
            window.print();
        }

        // Change handlers for P9 form
        $('#p9GenerationType').change(updateP9Preview);

        // Bulk actions for documents
        function bulkDownloadDocuments() {
            var selectedDocs = [];
            $('.doc-checkbox:checked').each(function() {
                selectedDocs.push($(this).val());
            });
            
            if (selectedDocs.length === 0) {
                showNotification('warning', 'Please select documents to download');
                return;
            }
            
            $.ajax({
                url: baseUrl + 'hr/bulkDownloadDocuments',
                type: 'POST',
                data: { document_ids: selectedDocs },
                dataType: 'json',
                success: function(response) {
                    if (response.status === 'success') {
                        window.location = response.download_url;
                        showNotification('success', 'Documents download started');
                    } else {
                        showNotification('error', response.message);
                    }
                },
                error: function() {
                    showNotification('error', 'Failed to download documents');
                }
            });
        }

        function organizeDocuments() {
            showNotification('info', 'Document organization feature coming soon');
        }

        function auditDocuments() {
            showNotification('info', 'Document audit trail feature coming soon');
        }

        // Select all documents checkbox
        $('#selectAllDocs').change(function() {
            $('.doc-checkbox').prop('checked', $(this).prop('checked'));
        });

        // Additional helper functions for dashboard
        function refreshDashboard() {
            loadDashboardStats();
            if (staffTable) {
                staffTable.draw();
            }
        }

        // Auto-refresh dashboard every 5 minutes
        setInterval(function() {
            refreshDashboard();
        }, 300000);

        // Additional AJAX endpoints you might need to add to your controller:
        
        // Get dashboard statistics
        function getDashboardStats() {
            return $.ajax({
                url: baseUrl + 'hr/getDashboardStats',
                type: 'GET',
                dataType: 'json'
            });
        }

        // Get departments
        function getDepartments() {
            return $.ajax({
                url: baseUrl + 'hr/getDepartments',
                type: 'GET',
                dataType: 'json'
            });
        }

        // Get roles
        function getRoles() {
            return $.ajax({
                url: baseUrl + 'hr/getRoles',
                type: 'GET',
                dataType: 'json'
            });
        }

        // Clock in/out functionality (if needed)
        function clockIn(staffId) {
            $.ajax({
                url: baseUrl + 'hr/clockIn',
                type: 'POST',
                data: { staff_id: staffId },
                dataType: 'json',
                success: function(response) {
                    if (response.status === 'success') {
                        showNotification('success', response.message);
                    } else {
                        showNotification('error', response.message);
                    }
                },
                error: function() {
                    showNotification('error', 'Failed to clock in');
                }
            });
        }

        function clockOut(staffId) {
            $.ajax({
                url: baseUrl + 'hr/clockOut',
                type: 'POST',
                data: { staff_id: staffId },
                dataType: 'json',
                success: function(response) {
                    if (response.status === 'success') {
                        showNotification('success', response.message);
                    } else {
                        showNotification('error', response.message);
                    }
                },
                error: function() {
                    showNotification('error', 'Failed to clock out');
                }
            });
        }

        // Debug function to check if everything is working
        function debugSystem() {
            console.log('Base URL:', baseUrl);
            console.log('Current Staff ID:', currentStaffId);
            console.log('Staff Table:', staffTable);
            
            // Test AJAX connection
            $.ajax({
                url: baseUrl + 'hr/test',
                type: 'GET',
                success: function(response) {
                    console.log('Controller connection test successful');
                },
                error: function() {
                    console.error('Controller connection test failed');
                }
            });
        }

        // Call debug on load (remove in production)
        // debugSystem();
    </script>
</body>
</html>