<?php
defined('BASEPATH') OR exit('No direct script access allowed');

class Wellness_model extends CI_Model {

    public function __construct() {
        parent::__construct();
    }

    /**
     * Create a new wellness visit
     */
    public function createWellnessVisit($data) {
        $visit_data = array(
            'visit_reference' => $this->generateVisitReference(),
            'location' => $data['location'],
            'visit_date' => date('Y-m-d'),
            'visit_time' => date('H:i:s'),
            'patient_name' => $data['patient_name'],
            'age' => $data['age'],
            'gender' => $data['gender'],
            'phone_number' => $data['phone_number'] ?? null,
            'national_id' => $data['national_id'] ?? null,
            'county' => $data['county'] ?? null,
            'sub_county' => $data['sub_county'] ?? null,
            'village' => $data['village'] ?? null,
            'emergency_contact' => $data['emergency_contact'] ?? null,
            'insurance_type' => $data['insurance_type'] ?? 'cash',
            'insurance_number' => $data['insurance_number'] ?? null,
            'chief_complaint' => $data['chief_complaint'] ?? null,
            'served_by' => $data['served_by'] ?? null,
            'notes' => $data['notes'] ?? null
        );

        $this->db->insert('wellness_visits', $visit_data);
        return $this->db->insert_id();
    }

    /**
     * Add service to wellness visit
     */
    public function addWellnessService($visit_id, $service_data) {
        // Get service details from catalog if service_code provided
        if (isset($service_data['service_code'])) {
            $service_info = $this->getServiceFromCatalog($service_data['service_code']);
            if ($service_info) {
                $service_data['service_name'] = $service_info->service_name;
                $service_data['unit_price'] = $service_data['unit_price'] ?? $service_info->standard_price;
            }
        }

        $quantity = $service_data['quantity'] ?? 1;
        $unit_price = $service_data['unit_price'];
        $total_price = $quantity * $unit_price;
        
        $discount_percentage = $service_data['discount_percentage'] ?? 0;
        $discount_amount = ($total_price * $discount_percentage) / 100;
        
        $tax_percentage = $service_data['tax_percentage'] ?? 0;
        $tax_amount = (($total_price - $discount_amount) * $tax_percentage) / 100;
        
        $final_amount = $total_price - $discount_amount + $tax_amount;

        $service_record = array(
            'visit_id' => $visit_id,
            'service_type' => $service_data['service_type'],
            'service_name' => $service_data['service_name'],
            'service_code' => $service_data['service_code'] ?? null,
            'quantity' => $quantity,
            'unit_price' => $unit_price,
            'total_price' => $total_price,
            'discount_percentage' => $discount_percentage,
            'discount_amount' => $discount_amount,
            'tax_percentage' => $tax_percentage,
            'tax_amount' => $tax_amount,
            'final_amount' => $final_amount,
            'provider_staff_id' => $service_data['provider_staff_id'] ?? null,
            'notes' => $service_data['notes'] ?? null,
            'status' => $service_data['status'] ?? 'pending'
        );

        $this->db->insert('wellness_services', $service_record);
        $service_id = $this->db->insert_id();

        // Update visit total
        $this->updateVisitTotal($visit_id);

        return $service_id;
    }

    /**
     * Update visit totals
     */
    private function updateVisitTotal($visit_id) {
        $this->db->select('SUM(final_amount) as total');
        $this->db->where('visit_id', $visit_id);
        $query = $this->db->get('wellness_services');
        $result = $query->row();
        
        $total_amount = $result->total ?? 0;
        
        $this->db->where('id', $visit_id);
        $this->db->update('wellness_visits', array(
            'total_amount' => $total_amount,
            'balance' => $total_amount - $this->getVisitPaidAmount($visit_id)
        ));
    }

    /**
     * Process payment for wellness visit
     */
    public function processPayment($visit_id, $payment_data) {
        $amount_paid = $payment_data['amount'];
        $payment_method = $payment_data['payment_method'] ?? 'cash';
        $receipt_number = $payment_data['receipt_number'] ?? $this->generateReceiptNumber();

        // Get current visit data
        $visit = $this->getWellnessVisit($visit_id);
        $new_paid_amount = $visit->amount_paid + $amount_paid;
        $balance = $visit->total_amount - $new_paid_amount;
        
        // Determine payment status
        if ($balance <= 0) {
            $payment_status = 'paid';
        } elseif ($new_paid_amount > 0) {
            $payment_status = 'partial';
        } else {
            $payment_status = 'pending';
        }

        // Update visit payment info
        $this->db->where('id', $visit_id);
        $this->db->update('wellness_visits', array(
            'amount_paid' => $new_paid_amount,
            'balance' => max(0, $balance),
            'payment_status' => $payment_status,
            'payment_method' => $payment_method,
            'receipt_number' => $receipt_number
        ));

        // Record in financial transactions
        $this->recordFinancialTransaction($visit_id, $payment_data);

        return $receipt_number;
    }

    /**
     * Get wellness visit details
     */
    public function getWellnessVisit($visit_id) {
        $this->db->select('wv.*, wl.location_name, wl.county as location_county, 
                          CONCAT(s.name, " ", s.surname) as served_by_name');
        $this->db->from('wellness_visits wv');
        $this->db->join('wellness_locations wl', 'wl.location_name = wv.location', 'left');
        $this->db->join('staff s', 's.id = wv.served_by', 'left');
        $this->db->where('wv.id', $visit_id);
        
        $query = $this->db->get();
        return $query->row();
    }

    /**
     * Get wellness visit services
     */
    public function getWellnessServices($visit_id) {
        $this->db->select('ws.*, CONCAT(s.name, " ", s.surname) as provider_name');
        $this->db->from('wellness_services ws');
        $this->db->join('staff s', 's.id = ws.provider_staff_id', 'left');
        $this->db->where('ws.visit_id', $visit_id);
        $this->db->order_by('ws.created_at', 'ASC');
        
        $query = $this->db->get();
        return $query->result();
    }

    /**
     * Generate wellness reports
     */
    public function getWellnessReport($filters = array()) {
        $this->db->select('wv.location, DATE(wv.visit_date) as visit_date,
                          COUNT(DISTINCT wv.id) as total_visits,
                          COUNT(CASE WHEN ws.service_type = "consultation" THEN 1 END) as consultations,
                          COUNT(CASE WHEN ws.service_type = "dental" THEN 1 END) as dental_services,
                          COUNT(CASE WHEN ws.service_type = "laboratory" THEN 1 END) as lab_services,
                          COUNT(CASE WHEN ws.service_type = "pharmacy" THEN 1 END) as pharmacy_services,
                          COUNT(CASE WHEN ws.service_type = "radiology" THEN 1 END) as radiology_services,
                          COUNT(CASE WHEN ws.service_type = "optical" THEN 1 END) as optical_services,
                          SUM(wv.total_amount) as total_revenue,
                          SUM(wv.amount_paid) as total_payments,
                          SUM(wv.balance) as total_outstanding');
        
        $this->db->from('wellness_visits wv');
        $this->db->join('wellness_services ws', 'ws.visit_id = wv.id', 'left');
        
        // Apply filters
        if (!empty($filters['location'])) {
            $this->db->where('wv.location', $filters['location']);
        }
        
        if (!empty($filters['date_from'])) {
            $this->db->where('wv.visit_date >=', $filters['date_from']);
        }
        
        if (!empty($filters['date_to'])) {
            $this->db->where('wv.visit_date <=', $filters['date_to']);
        }
        
        $this->db->group_by('wv.location, DATE(wv.visit_date)');
        $this->db->order_by('wv.visit_date', 'DESC');
        
        $query = $this->db->get();
        return $query->result();
    }

    /**
     * Get service catalog
     */
    public function getServiceCatalog($service_type = null) {
        $this->db->select('*');
        $this->db->from('wellness_service_catalog');
        $this->db->where('is_active', 1);
        
        if ($service_type) {
            $this->db->where('service_type', $service_type);
        }
        
        $this->db->order_by('service_type, service_name');
        
        $query = $this->db->get();
        return $query->result();
    }

    /**
     * Get locations
     */
    public function getWellnessLocations() {
        $this->db->select('*');
        $this->db->from('wellness_locations');
        $this->db->where('is_active', 1);
        $this->db->order_by('location_name');
        
        $query = $this->db->get();
        return $query->result();
    }

    // Helper methods
    private function generateVisitReference() {
        return 'WV-' . date('Ymd') . '-' . str_pad(mt_rand(1, 9999), 4, '0', STR_PAD_LEFT);
    }

    private function generateReceiptNumber() {
        return 'WR-' . date('Ymd-His') . '-' . mt_rand(100, 999);
    }

    private function getServiceFromCatalog($service_code) {
        $this->db->where('service_code', $service_code);
        $this->db->where('is_active', 1);
        $query = $this->db->get('wellness_service_catalog');
        return $query->row();
    }

    private function getVisitPaidAmount($visit_id) {
        $this->db->select('amount_paid');
        $this->db->where('id', $visit_id);
        $query = $this->db->get('wellness_visits');
        $result = $query->row();
        return $result ? $result->amount_paid : 0;
    }

    private function recordFinancialTransaction($visit_id, $payment_data) {
        $visit = $this->getWellnessVisit($visit_id);
        
        $transaction_data = array(
            'transaction_date' => date('Y-m-d H:i:s'),
            'transaction_type' => 'revenue',
            'category' => 'wellness_services',
            'subcategory' => 'outreach',
            'amount' => $payment_data['amount'],
            'description' => 'Wellness visit payment - ' . $visit->location,
            'reference_table' => 'wellness_visits',
            'reference_id' => $visit_id,
            'patient_id' => $visit->visit_reference,
            'department' => 'Wellness',
            'payment_method' => $payment_data['payment_method'],
            'receipt_number' => $payment_data['receipt_number'],
            'created_by' => $this->session->userdata('admin')['id'] ?? null
        );

        $this->db->insert('financial_transactions', $transaction_data);
    }
}