<?php

if (!defined('BASEPATH'))
    exit('No direct script access allowed');

class Triage_model extends CI_Model {

    public function __construct() {
        parent::__construct();
    }

    /**
     * Save a new triage record
     */
    public function saveTriageRecord($data) {
        $this->db->insert('triage', $data);
        return $this->db->insert_id();
    }

    /**
     * Update an existing triage record
     */
    public function updateTriageRecord($triageId, $data) {
        $this->db->where('id', $triageId);
        return $this->db->update('triage', $data);
    }

    /**
     * Get triage details by ID
     */
    public function getTriageDetails($triageId) {
        $this->db->select('t.*, p.patient_name, p.mobileno as patient_phone, p.age, p.gender, 
                          CONCAT(s.name, " ", s.surname) as staff_name');
        $this->db->from('triage t');
        $this->db->join('patients p', 'p.id = t.patient_id', 'left');
        $this->db->join('staff s', 's.id = t.triage_by', 'left');
        $this->db->where('t.id', $triageId);
        
        $query = $this->db->get();
        return $query->row_array();
    }

    /**
     * Get triage records list for DataTable
     */
    public function getTriageList($start = 0, $length = 10, $search = '', $orderColumn = 'id', $orderDir = 'desc') {
        $this->db->select('t.id, t.patient_id, t.vitals, t.notes, t.created_at,
                          p.patient_name, p.id as patient_id,
                          CONCAT(s.name, " ", s.surname) as staff_name');
        
        // Add verification_status if column exists
        if ($this->db->field_exists('verification_status', 'triage')) {
            $this->db->select('t.verification_status');
        }
        
        // Add triage_by if column exists
        if ($this->db->field_exists('triage_by', 'triage')) {
            $this->db->select('t.triage_by');
        }
        
        $this->db->from('triage t');
        $this->db->join('patients p', 'p.id = t.patient_id', 'left');
        $this->db->join('staff s', 's.id = t.triage_by', 'left');
        
        if (!empty($search)) {
            $this->db->group_start();
            $this->db->like('p.patient_name', $search);
            $this->db->or_like('t.vitals', $search);
            $this->db->or_like('t.notes', $search);
            $this->db->or_like('CONCAT(s.name, " ", s.surname)', $search);
            $this->db->group_end();
        }
        
        $this->db->order_by($orderColumn, $orderDir);
        $this->db->limit($length, $start);
        
        $query = $this->db->get();
        return $query->result_array();
    }

    /**
     * Get total triage records count
     */
    public function getTriageCount($search = '') {
        $this->db->from('triage t');
        $this->db->join('patients p', 'p.id = t.patient_id', 'left');
        $this->db->join('staff s', 's.id = t.triage_by', 'left');
        
        if (!empty($search)) {
            $this->db->group_start();
            $this->db->like('p.patient_name', $search);
            $this->db->or_like('t.vitals', $search);
            $this->db->or_like('t.notes', $search);
            $this->db->or_like('CONCAT(s.name, " ", s.surname)', $search);
            $this->db->group_end();
        }
        
        return $this->db->count_all_results();
    }

    /**
     * Get patient's triage history
     */
    public function getPatientTriageHistory($patientId, $limit = 10) {
        $this->db->select('t.id, t.created_at as date, t.vitals, t.notes,
                          CONCAT(s.name, " ", s.surname) as staff_name');
        
        // Add verification_status if column exists
        if ($this->db->field_exists('verification_status', 'triage')) {
            $this->db->select('t.verification_status');
        }
        
        $this->db->from('triage t');
        $this->db->join('staff s', 's.id = t.triage_by', 'left');
        $this->db->where('t.patient_id', $patientId);
        $this->db->order_by('t.created_at', 'DESC');
        $this->db->limit($limit);
        
        $query = $this->db->get();
        return $query->result_array();
    }

    /**
     * Delete a triage record
     */
    public function deleteTriageRecord($triageId) {
        $this->db->where('id', $triageId);
        return $this->db->delete('triage');
    }

    /**
     * Get triage records by date range
     */
    public function getTriageByDateRange($startDate, $endDate, $staffId = null) {
        $this->db->select('t.*, p.patient_name, CONCAT(s.name, " ", s.surname) as staff_name');
        $this->db->from('triage t');
        $this->db->join('patients p', 'p.id = t.patient_id', 'left');
        $this->db->join('staff s', 's.id = t.triage_by', 'left');
        $this->db->where('DATE(t.created_at) >=', $startDate);
        $this->db->where('DATE(t.created_at) <=', $endDate);
        
        if ($staffId) {
            $this->db->where('t.triage_by', $staffId);
        }
        
        $this->db->order_by('t.created_at', 'DESC');
        
        $query = $this->db->get();
        return $query->result_array();
    }

    /**
     * Get triage statistics
     */
    public function getTriageStats($dateFrom = null, $dateTo = null) {
        if (!$dateFrom) $dateFrom = date('Y-m-01'); // First day of current month
        if (!$dateTo) $dateTo = date('Y-m-d'); // Today
        
        // Total triage records
        $this->db->where('DATE(created_at) >=', $dateFrom);
        $this->db->where('DATE(created_at) <=', $dateTo);
        $totalTriage = $this->db->count_all_results('triage');
        
        // Triage by staff member
        $this->db->select('s.name, s.surname, COUNT(t.id) as triage_count');
        $this->db->from('triage t');
        $this->db->join('staff s', 's.id = t.triage_by', 'left');
        $this->db->where('DATE(t.created_at) >=', $dateFrom);
        $this->db->where('DATE(t.created_at) <=', $dateTo);
        $this->db->group_by('t.triage_by');
        $this->db->order_by('triage_count', 'DESC');
        $triageByStaff = $this->db->get()->result_array();
        
        // Daily triage records for chart
        $this->db->select('DATE(created_at) as date, COUNT(id) as count');
        $this->db->from('triage');
        $this->db->where('DATE(created_at) >=', $dateFrom);
        $this->db->where('DATE(created_at) <=', $dateTo);
        $this->db->group_by('DATE(created_at)');
        $this->db->order_by('date', 'ASC');
        $dailyTriage = $this->db->get()->result_array();
        
        // Verification status statistics (if column exists)
        $verificationStats = [];
        if ($this->db->field_exists('verification_status', 'triage')) {
            $this->db->select('verification_status, COUNT(id) as count');
            $this->db->from('triage');
            $this->db->where('DATE(created_at) >=', $dateFrom);
            $this->db->where('DATE(created_at) <=', $dateTo);
            $this->db->group_by('verification_status');
            $verificationStats = $this->db->get()->result_array();
        }
        
        return [
            'total_triage' => $totalTriage,
            'triage_by_staff' => $triageByStaff,
            'daily_triage' => $dailyTriage,
            'verification_stats' => $verificationStats
        ];
    }

    /**
     * Search triage records
     */
    public function searchTriageRecords($searchTerm, $limit = 50) {
        $this->db->select('t.id, t.created_at, t.vitals, t.notes,
                          p.patient_name, p.id as patient_id,
                          CONCAT(s.name, " ", s.surname) as staff_name');
        
        // Add verification_status if column exists
        if ($this->db->field_exists('verification_status', 'triage')) {
            $this->db->select('t.verification_status');
        }
        
        $this->db->from('triage t');
        $this->db->join('patients p', 'p.id = t.patient_id', 'left');
        $this->db->join('staff s', 's.id = t.triage_by', 'left');
        
        $this->db->group_start();
        $this->db->like('p.patient_name', $searchTerm);
        $this->db->or_like('t.vitals', $searchTerm);
        $this->db->or_like('t.notes', $searchTerm);
        $this->db->or_like('p.id', $searchTerm);
        $this->db->group_end();
        
        $this->db->order_by('t.created_at', 'DESC');
        $this->db->limit($limit);
        
        $query = $this->db->get();
        return $query->result_array();
    }

    /**
     * Get triage records by priority (if implemented)
     */
    public function getTriageByPriority($priority, $limit = 50) {
        $this->db->select('t.*, p.patient_name, CONCAT(s.name, " ", s.surname) as staff_name');
        $this->db->from('triage t');
        $this->db->join('patients p', 'p.id = t.patient_id', 'left');
        $this->db->join('staff s', 's.id = t.triage_by', 'left');
        
        // Assuming priority is stored in vitals JSON or notes
        $this->db->like('t.vitals', '"triage_priority":"' . $priority . '"');
        $this->db->or_like('t.notes', 'Priority: ' . $priority);
        
        $this->db->order_by('t.created_at', 'DESC');
        $this->db->limit($limit);
        
        $query = $this->db->get();
        return $query->result_array();
    }

    /**
     * Get urgent triage cases
     */
    public function getUrgentTriageCases($limit = 20) {
        $this->db->select('t.*, p.patient_name, p.mobileno, CONCAT(s.name, " ", s.surname) as staff_name');
        $this->db->from('triage t');
        $this->db->join('patients p', 'p.id = t.patient_id', 'left');
        $this->db->join('staff s', 's.id = t.triage_by', 'left');
        
        // Look for urgent indicators in vitals or notes
        $this->db->group_start();
        $this->db->like('t.vitals', '"triage_priority":"1"'); // Priority 1 = Immediate
        $this->db->or_like('t.vitals', '"triage_priority":"2"'); // Priority 2 = Urgent
        $this->db->or_like('t.notes', 'urgent', 'both');
        $this->db->or_like('t.notes', 'critical', 'both');
        $this->db->group_end();
        
        $this->db->where('DATE(t.created_at)', date('Y-m-d')); // Today only
        $this->db->order_by('t.created_at', 'DESC');
        $this->db->limit($limit);
        
        $query = $this->db->get();
        return $query->result_array();
    }

    /**
     * Get patients waiting for follow-up
     */
    public function getPatientsAwaitingFollowup($days = 7) {
        $this->db->select('t.*, p.patient_name, p.mobileno, CONCAT(s.name, " ", s.surname) as staff_name');
        $this->db->from('triage t');
        $this->db->join('patients p', 'p.id = t.patient_id', 'left');
        $this->db->join('staff s', 's.id = t.triage_by', 'left');
        
        // Look for follow-up indicators in notes
        $this->db->like('t.notes', 'follow', 'both');
        $this->db->where('t.created_at >=', date('Y-m-d H:i:s', strtotime("-{$days} days")));
        $this->db->order_by('t.created_at', 'DESC');
        
        $query = $this->db->get();
        return $query->result_array();
    }

    /**
     * Parse vitals JSON data
     */
    public function parseVitalsData($vitalsJson) {
        if (empty($vitalsJson)) {
            return [];
        }
        
        $vitals = json_decode($vitalsJson, true);
        if (json_last_error() !== JSON_ERROR_NONE) {
            return [];
        }
        
        return $vitals;
    }

    /**
     * Format vitals for display
     */
    public function formatVitalsForDisplay($vitalsJson) {
        $vitals = $this->parseVitalsData($vitalsJson);
        $formatted = [];
        
        if (!empty($vitals)) {
            // Blood Pressure
            if (!empty($vitals['blood_pressure']['systolic']) && !empty($vitals['blood_pressure']['diastolic'])) {
                $formatted['Blood Pressure'] = $vitals['blood_pressure']['systolic'] . '/' . $vitals['blood_pressure']['diastolic'] . ' mmHg';
            }
            
            // Heart Rate
            if (!empty($vitals['heart_rate'])) {
                $formatted['Heart Rate'] = $vitals['heart_rate'] . ' bpm';
            }
            
            // Temperature
            if (!empty($vitals['temperature'])) {
                $formatted['Temperature'] = $vitals['temperature'] . ' °C';
            }
            
            // Respiratory Rate
            if (!empty($vitals['respiratory_rate'])) {
                $formatted['Respiratory Rate'] = $vitals['respiratory_rate'] . ' rpm';
            }
            
            // Oxygen Saturation
            if (!empty($vitals['oxygen_saturation'])) {
                $formatted['SpO2'] = $vitals['oxygen_saturation'] . '%';
            }
            
            // Weight and Height
            if (!empty($vitals['weight'])) {
                $formatted['Weight'] = $vitals['weight'] . ' kg';
            }
            
            if (!empty($vitals['height'])) {
                $formatted['Height'] = $vitals['height'] . ' cm';
            }
            
            // Pain Scale
            if (!empty($vitals['pain_scale'])) {
                $formatted['Pain Scale'] = $vitals['pain_scale'] . '/10';
            }
            
            // Triage Priority
            if (!empty($vitals['triage_priority'])) {
                $priorities = [
                    '1' => 'Immediate (Red)',
                    '2' => 'Urgent (Orange)', 
                    '3' => 'Less Urgent (Yellow)',
                    '4' => 'Non-Urgent (Green)',
                    '5' => 'Fast Track (Blue)'
                ];
                $formatted['Triage Priority'] = $priorities[$vitals['triage_priority']] ?? $vitals['triage_priority'];
            }
        }
        
        return $formatted;
    }
}