<?php

if (!defined('BASEPATH'))
    exit('No direct script access allowed');

class Specialized_services_model extends CI_Model {

    public function __construct() {
        parent::__construct();
    }

    // =============================================
    // CATEGORIES MANAGEMENT
    // =============================================

    /**
     * Get all service categories
     */
    public function getServiceCategories($is_active = null) {
        $this->db->select('*');
        $this->db->from('specialized_service_categories');
        
        if ($is_active !== null) {
            $this->db->where('is_active', $is_active);
        }
        
        $this->db->order_by('category_type, category_name');
        $query = $this->db->get();
        
        return $query->result_array();
    }

    /**
     * Get categories by type
     */
    public function getCategoriesByType($category_type) {
        $this->db->select('*');
        $this->db->from('specialized_service_categories');
        $this->db->where('category_type', $category_type);
        $this->db->where('is_active', 1);
        $this->db->order_by('category_name');
        
        $query = $this->db->get();
        return $query->result_array();
    }

    /**
     * Add new category
     */
    public function addServiceCategory($data) {
        return $this->db->insert('specialized_service_categories', $data);
    }

    /**
     * Update category
     */
    public function updateServiceCategory($id, $data) {
        $this->db->where('id', $id);
        return $this->db->update('specialized_service_categories', $data);
    }

    /**
     * Delete category
     */
    public function deleteServiceCategory($id) {
        $this->db->where('id', $id);
        return $this->db->delete('specialized_service_categories');
    }

    /**
     * Get category by ID
     */
    public function getServiceCategory($id) {
        $this->db->where('id', $id);
        $query = $this->db->get('specialized_service_categories');
        return $query->row();
    }

    // =============================================
    // SERVICES MANAGEMENT
    // =============================================

    /**
     * Get all services with category information
     */
    public function getServices($filters = array()) {
        $this->db->select('
            specialized_services.*,
            specialized_service_categories.category_name,
            specialized_service_categories.category_type,
            specialized_service_categories.icon_class,
            specialized_service_categories.color_code
        ');
        
        $this->db->from('specialized_services');
        $this->db->join('specialized_service_categories', 'specialized_service_categories.id = specialized_services.category_id', 'left');
        
        // Apply filters
        if (!empty($filters['category_type'])) {
            $this->db->where('specialized_service_categories.category_type', $filters['category_type']);
        }
        
        if (!empty($filters['category_id'])) {
            $this->db->where('specialized_services.category_id', $filters['category_id']);
        }
        
        if (!empty($filters['service_type'])) {
            $this->db->where('specialized_services.service_type', $filters['service_type']);
        }
        
        if (!empty($filters['requires_specialist'])) {
            $this->db->where('specialized_services.requires_specialist', $filters['requires_specialist']);
        }
        
        if (isset($filters['is_active'])) {
            $this->db->where('specialized_services.is_active', $filters['is_active']);
        } else {
            $this->db->where('specialized_services.is_active', 1);
        }
        
        if (!empty($filters['search'])) {
            $this->db->group_start();
            $this->db->like('specialized_services.service_name', $filters['search']);
            $this->db->or_like('specialized_services.service_code', $filters['search']);
            $this->db->or_like('specialized_service_categories.category_name', $filters['search']);
            $this->db->group_end();
        }
        
        $this->db->order_by('specialized_service_categories.category_type, specialized_services.service_name');
        
        if (!empty($filters['limit'])) {
            $this->db->limit($filters['limit'], $filters['offset'] ?? 0);
        }
        
        $query = $this->db->get();
        return $query->result_array();
    }

    /**
     * Get service by ID with category information
     */
    public function getService($id) {
        $this->db->select('
            specialized_services.*,
            specialized_service_categories.category_name,
            specialized_service_categories.category_type,
            specialized_service_categories.icon_class,
            specialized_service_categories.color_code
        ');
        
        $this->db->from('specialized_services');
        $this->db->join('specialized_service_categories', 'specialized_service_categories.id = specialized_services.category_id', 'left');
        $this->db->where('specialized_services.id', $id);
        
        $query = $this->db->get();
        return $query->row();
    }

    /**
     * Add new service
     */
    public function addService($data) {
        return $this->db->insert('specialized_services', $data);
    }

    /**
     * Update service
     */
    public function updateService($id, $data) {
        $this->db->where('id', $id);
        return $this->db->update('specialized_services', $data);
    }

    /**
     * Delete service
     */
    public function deleteService($id) {
        $this->db->where('id', $id);
        return $this->db->delete('specialized_services');
    }

    /**
     * Check if service name exists in category
     */
    public function checkServiceExists($service_name, $category_id, $exclude_id = null) {
        $this->db->where('service_name', $service_name);
        $this->db->where('category_id', $category_id);
        
        if ($exclude_id) {
            $this->db->where('id !=', $exclude_id);
        }
        
        $query = $this->db->get('specialized_services');
        return $query->num_rows() > 0;
    }

    /**
     * Get services by category type (dental, optical, etc.)
     */
    public function getServicesByType($category_type, $is_active = 1) {
        $this->db->select('
            specialized_services.*,
            specialized_service_categories.category_name,
            specialized_service_categories.category_type
        ');
        
        $this->db->from('specialized_services');
        $this->db->join('specialized_service_categories', 'specialized_service_categories.id = specialized_services.category_id');
        $this->db->where('specialized_service_categories.category_type', $category_type);
        
        if ($is_active !== null) {
            $this->db->where('specialized_services.is_active', $is_active);
        }
        
        $this->db->order_by('specialized_services.service_name');
        
        $query = $this->db->get();
        return $query->result_array();
    }

    /**
     * Get dental services (based on your price list)
     */
    public function getDentalServices($is_active = 1) {
        return $this->getServicesByType('dental', $is_active);
    }

    /**
     * Get optical services
     */
    public function getOpticalServices($is_active = 1) {
        return $this->getServicesByType('optical', $is_active);
    }

    /**
     * Toggle service active status
     */
    public function toggleServiceStatus($id) {
        $this->db->select('is_active');
        $this->db->where('id', $id);
        $query = $this->db->get('specialized_services');
        $service = $query->row();
        
        if ($service) {
            $new_status = $service->is_active == 1 ? 0 : 1;
            $this->db->where('id', $id);
            return $this->db->update('specialized_services', array(
                'is_active' => $new_status,
                'updated_at' => date('Y-m-d H:i:s')
            ));
        }
        
        return false;
    }

    // =============================================
    // STATISTICS AND REPORTING
    // =============================================

    /**
     * Get setup statistics
     */
    public function getSetupStats() {
        $stats = array();
        
        // Total categories
        $stats['total_categories'] = $this->db->count_all('specialized_service_categories');
        
        // Total services
        $stats['total_services'] = $this->db->count_all('specialized_services');
        
        // Services by category type
        $this->db->select('
            specialized_service_categories.category_type,
            COUNT(specialized_services.id) as service_count
        ');
        $this->db->from('specialized_service_categories');
        $this->db->join('specialized_services', 'specialized_services.category_id = specialized_service_categories.id', 'left');
        $this->db->group_by('specialized_service_categories.category_type');
        $query = $this->db->get();
        
        $services_by_type = $query->result_array();
        foreach ($services_by_type as $type) {
            $stats[$type['category_type'] . '_services'] = $type['service_count'];
        }
        
        // Services by category
        $this->db->select('
            specialized_service_categories.category_name,
            specialized_service_categories.category_type,
            COUNT(specialized_services.id) as service_count
        ');
        $this->db->from('specialized_service_categories');
        $this->db->join('specialized_services', 'specialized_services.category_id = specialized_service_categories.id', 'left');
        $this->db->group_by('specialized_service_categories.id');
        $this->db->order_by('specialized_service_categories.category_type, specialized_service_categories.category_name');
        $query = $this->db->get();
        
        $stats['services_by_category'] = $query->result_array();
        
        // Average prices
        $this->db->select('AVG(cash_price) as avg_cash_price, AVG(insurance_price) as avg_insurance_price');
        $this->db->where('is_active', 1);
        $query = $this->db->get('specialized_services');
        $prices = $query->row();
        
        $stats['avg_cash_price'] = $prices->avg_cash_price ?? 0;
        $stats['avg_insurance_price'] = $prices->avg_insurance_price ?? 0;
        
        // Services requiring specialist
        $this->db->where('requires_specialist', 1);
        $this->db->where('is_active', 1);
        $stats['specialist_services'] = $this->db->count_all_results('specialized_services');
        
        // High-value services (above average price)
        if ($stats['avg_cash_price'] > 0) {
            $this->db->select('service_name, cash_price');
            $this->db->where('cash_price >', $stats['avg_cash_price']);
            $this->db->where('is_active', 1);
            $this->db->order_by('cash_price', 'DESC');
            $this->db->limit(5);
            $query = $this->db->get('specialized_services');
            $stats['high_value_services'] = $query->result_array();
        } else {
            $stats['high_value_services'] = array();
        }
        
        return $stats;
    }

    /**
     * Get services count by filters
     */
    public function getServicesCount($filters = array()) {
        $this->db->from('specialized_services');
        $this->db->join('specialized_service_categories', 'specialized_service_categories.id = specialized_services.category_id', 'left');
        
        // Apply same filters as getServices method
        if (!empty($filters['category_type'])) {
            $this->db->where('specialized_service_categories.category_type', $filters['category_type']);
        }
        
        if (!empty($filters['category_id'])) {
            $this->db->where('specialized_services.category_id', $filters['category_id']);
        }
        
        if (!empty($filters['service_type'])) {
            $this->db->where('specialized_services.service_type', $filters['service_type']);
        }
        
        if (!empty($filters['requires_specialist'])) {
            $this->db->where('specialized_services.requires_specialist', $filters['requires_specialist']);
        }
        
        if (isset($filters['is_active'])) {
            $this->db->where('specialized_services.is_active', $filters['is_active']);
        } else {
            $this->db->where('specialized_services.is_active', 1);
        }
        
        if (!empty($filters['search'])) {
            $this->db->group_start();
            $this->db->like('specialized_services.service_name', $filters['search']);
            $this->db->or_like('specialized_services.service_code', $filters['search']);
            $this->db->or_like('specialized_service_categories.category_name', $filters['search']);
            $this->db->group_end();
        }
        
        return $this->db->count_all_results();
    }

    // =============================================
    // UTILITY METHODS
    // =============================================

    /**
     * Get service pricing for a specific service
     */
    public function getServicePricing($service_id, $customer_type = 'cash') {
        $this->db->select('cash_price, insurance_price');
        $this->db->where('id', $service_id);
        $this->db->where('is_active', 1);
        $query = $this->db->get('specialized_services');
        
        $service = $query->row();
        if ($service) {
            if ($customer_type === 'insurance' && $service->insurance_price) {
                return $service->insurance_price;
            }
            return $service->cash_price;
        }
        
        return 0;
    }

    /**
     * Get services for dropdown/select options
     */
    public function getServicesForSelect($category_type = null) {
        $this->db->select('
            specialized_services.id,
            specialized_services.service_name,
            specialized_services.service_code,
            specialized_services.cash_price,
            specialized_service_categories.category_name,
            specialized_service_categories.category_type
        ');
        
        $this->db->from('specialized_services');
        $this->db->join('specialized_service_categories', 'specialized_service_categories.id = specialized_services.category_id');
        $this->db->where('specialized_services.is_active', 1);
        $this->db->where('specialized_service_categories.is_active', 1);
        
        if ($category_type) {
            $this->db->where('specialized_service_categories.category_type', $category_type);
        }
        
        $this->db->order_by('specialized_service_categories.category_type, specialized_services.service_name');
        
        $query = $this->db->get();
        return $query->result_array();
    }

    /**
     * Search services by name or code
     */
    public function searchServices($search_term, $category_type = null, $limit = 10) {
        $this->db->select('
            specialized_services.*,
            specialized_service_categories.category_name,
            specialized_service_categories.category_type
        ');
        
        $this->db->from('specialized_services');
        $this->db->join('specialized_service_categories', 'specialized_service_categories.id = specialized_services.category_id');
        $this->db->where('specialized_services.is_active', 1);
        
        if ($category_type) {
            $this->db->where('specialized_service_categories.category_type', $category_type);
        }
        
        $this->db->group_start();
        $this->db->like('specialized_services.service_name', $search_term);
        $this->db->or_like('specialized_services.service_code', $search_term);
        $this->db->group_end();
        
        $this->db->order_by('specialized_services.service_name');
        $this->db->limit($limit);
        
        $query = $this->db->get();
        return $query->result_array();
    }

    /**
     * Get popular services (this would need booking/usage data)
     */
    public function getPopularServices($limit = 5) {
        // This is a placeholder - you'd implement this based on your booking/usage tracking
        $this->db->select('
            specialized_services.*,
            specialized_service_categories.category_name,
            specialized_service_categories.category_type
        ');
        
        $this->db->from('specialized_services');
        $this->db->join('specialized_service_categories', 'specialized_service_categories.id = specialized_services.category_id');
        $this->db->where('specialized_services.is_active', 1);
        $this->db->order_by('specialized_services.created_at', 'DESC'); // Newest first as placeholder
        $this->db->limit($limit);
        
        $query = $this->db->get();
        return $query->result_array();
    }

    /**
     * Validate service data
     */
    public function validateServiceData($data, $exclude_id = null) {
        $errors = array();
        
        // Required fields
        if (empty($data['service_name'])) {
            $errors['service_name'] = 'Service name is required';
        }
        
        if (empty($data['category_id'])) {
            $errors['category_id'] = 'Category is required';
        }
        
        if (!isset($data['cash_price']) || $data['cash_price'] <= 0) {
            $errors['cash_price'] = 'Valid cash price is required';
        }
        
        // Check for duplicate service name in category
        if (!empty($data['service_name']) && !empty($data['category_id'])) {
            if ($this->checkServiceExists($data['service_name'], $data['category_id'], $exclude_id)) {
                $errors['service_name'] = 'A service with this name already exists in this category';
            }
        }
        
        // Validate service code if provided
        if (!empty($data['service_code'])) {
            $this->db->where('service_code', $data['service_code']);
            if ($exclude_id) {
                $this->db->where('id !=', $exclude_id);
            }
            $existing = $this->db->get('specialized_services');
            if ($existing->num_rows() > 0) {
                $errors['service_code'] = 'Service code already exists';
            }
        }
        
        return $errors;
    }

    /**
     * Bulk operations
     */
    public function bulkUpdateServices($service_ids, $update_data) {
        if (empty($service_ids) || !is_array($service_ids)) {
            return false;
        }
        
        $update_data['updated_at'] = date('Y-m-d H:i:s');
        
        $this->db->where_in('id', $service_ids);
        return $this->db->update('specialized_services', $update_data);
    }

    /**
     * Get services that require equipment
     */
    public function getServicesRequiringEquipment() {
        $this->db->select('
            specialized_services.*,
            specialized_service_categories.category_name,
            specialized_service_categories.category_type
        ');
        
        $this->db->from('specialized_services');
        $this->db->join('specialized_service_categories', 'specialized_service_categories.id = specialized_services.category_id');
        $this->db->where('specialized_services.requires_equipment', 1);
        $this->db->where('specialized_services.is_active', 1);
        $this->db->order_by('specialized_service_categories.category_type, specialized_services.service_name');
        
        $query = $this->db->get();
        return $query->result_array();
    }

    /**
     * Get services by risk level
     */
    public function getServicesByRiskLevel($risk_level) {
        $this->db->select('
            specialized_services.*,
            specialized_service_categories.category_name,
            specialized_service_categories.category_type
        ');
        
        $this->db->from('specialized_services');
        $this->db->join('specialized_service_categories', 'specialized_service_categories.id = specialized_services.category_id');
        $this->db->where('specialized_services.complications_risk_level', $risk_level);
        $this->db->where('specialized_services.is_active', 1);
        $this->db->order_by('specialized_services.service_name');
        
        $query = $this->db->get();
        return $query->result_array();
    }
}

?>