<?php

if (!defined('BASEPATH'))
    exit('No direct script access allowed');

/**
 * Procurement Model
 * File: application/models/Procurement_model.php
 * Handles all procurement-related database operations
 */
class Procurement_model extends CI_Model {

    public function __construct() {
        parent::__construct();
        $this->load->database();
    }

    // =============================================
    // REQUISITIONS METHODS
    // =============================================

    /**
     * Save purchase requisition
     */
    public function saveRequisition($data) {
        $this->db->insert('purchase_requisitions', $data);
        return $this->db->insert_id();
    }

    /**
     * Update purchase requisition
     */
    public function updateRequisition($requisition_id, $data) {
        $this->db->where('id', $requisition_id);
        return $this->db->update('purchase_requisitions', $data);
    }

    /**
     * Save requisition item
     */
    public function saveRequisitionItem($data) {
        $this->db->insert('purchase_requisition_items', $data);
        return $this->db->insert_id();
    }

    /**
     * Get requisitions list for DataTable
     */
    public function getRequisitionsList($start = 0, $length = 10, $search = '', $orderColumn = 0, $orderDir = 'desc') {
        $columns = [
            'pr.id',
            'pr.requisition_no',
            'pr.request_date',
            'pr.required_date',
            'pr.priority',
            'pr.status',
            'pr.total_estimated_cost',
            'd.department_name',
            'CONCAT(s.name, " ", s.surname)'
        ];

        // Check if tables exist
        if (!$this->db->table_exists('purchase_requisitions')) {
            return [
                'data' => [],
                'total' => 0,
                'filtered' => 0
            ];
        }

        $this->db->select('pr.*, d.department_name, CONCAT(COALESCE(s.name, ""), " ", COALESCE(s.surname, "")) as requested_by_name');
        $this->db->from('purchase_requisitions pr');
        
        // Left join with departments
        if ($this->db->table_exists('departments')) {
            $this->db->join('departments d', 'd.id = pr.requesting_department_id', 'left');
        } else {
            $this->db->select('pr.requesting_department_id as department_name', FALSE);
        }
        
        // Left join with staff
        if ($this->db->table_exists('staff')) {
            $this->db->join('staff s', 's.id = pr.requested_by', 'left');
        } else {
            $this->db->select('"Unknown User" as requested_by_name', FALSE);
        }

        // Search
        if (!empty($search)) {
            $this->db->group_start();
            $this->db->like('pr.requisition_no', $search);
            if ($this->db->table_exists('departments')) {
                $this->db->or_like('d.department_name', $search);
            }
            $this->db->or_like('pr.purpose', $search);
            $this->db->or_like('pr.status', $search);
            $this->db->group_end();
        }

        // Count total records
        $total_query = clone $this->db;
        $total = $total_query->count_all_results('', FALSE);

        // Count filtered records
        $filtered_query = clone $this->db;
        $filtered = $filtered_query->count_all_results('', FALSE);

        // Order and limit
        if (isset($columns[$orderColumn])) {
            $this->db->order_by($columns[$orderColumn], $orderDir);
        } else {
            $this->db->order_by('pr.id', 'desc');
        }

        $this->db->limit($length, $start);
        $query = $this->db->get();

        return [
            'data' => $query->result_array(),
            'total' => $total,
            'filtered' => $filtered
        ];
    }

    /**
     * Get requisition details
     */
    public function getRequisitionDetails($requisition_id) {
        if (!$this->db->table_exists('purchase_requisitions')) {
            return null;
        }

        $this->db->select('pr.*');
        $this->db->from('purchase_requisitions pr');
        
        if ($this->db->table_exists('departments')) {
            $this->db->select('d.department_name');
            $this->db->join('departments d', 'd.id = pr.requesting_department_id', 'left');
        }
        
        if ($this->db->table_exists('staff')) {
            $this->db->select('CONCAT(COALESCE(s.name, ""), " ", COALESCE(s.surname, "")) as requested_by_name');
            $this->db->join('staff s', 's.id = pr.requested_by', 'left');
        }
        
        $this->db->where('pr.id', $requisition_id);
        
        $query = $this->db->get();
        return $query->row_array();
    }

    /**
     * Get requisition items
     */
    public function getRequisitionItems($requisition_id) {
        if (!$this->db->table_exists('purchase_requisition_items')) {
            return [];
        }
    
        $this->db->select('pri.*');
        $this->db->from('purchase_requisition_items pri');
        
        if ($this->db->table_exists('items')) {
            $this->db->select('i.item_name, i.item_code, i.unit_of_measure');
            $this->db->join('items i', 'i.id = pri.item_id', 'left');
        }
        
        $this->db->where('pri.requisition_id', $requisition_id);
        $this->db->order_by('pri.id', 'asc');
        
        $query = $this->db->get();
        return $query->result_array();
    }

    /**
     * Get approved requisitions
     */
    public function getApprovedRequisitions() {
        if (!$this->db->table_exists('purchase_requisitions')) {
            return [];
        }

        $this->db->select('pr.*');
        $this->db->from('purchase_requisitions pr');
        
        if ($this->db->table_exists('departments')) {
            $this->db->select('d.department_name');
            $this->db->join('departments d', 'd.id = pr.requesting_department_id', 'left');
        }
        
        $this->db->where('pr.status', 'approved');
        $this->db->order_by('pr.request_date', 'desc');
        
        $query = $this->db->get();
        return $query->result_array();
    }

    /**
     * Count pending requisitions
     */
    public function countPendingRequisitions() {
        if (!$this->db->table_exists('purchase_requisitions')) {
            return 0;
        }

        $this->db->from('purchase_requisitions');
        $this->db->where_in('status', ['pending_approval', 'draft']);
        return $this->db->count_all_results();
    }

    /**
     * Get recent requisitions
     */
    public function getRecentRequisitions($limit = 5) {
        if (!$this->db->table_exists('purchase_requisitions')) {
            return [];
        }

        $this->db->select('pr.*');
        $this->db->from('purchase_requisitions pr');
        
        if ($this->db->table_exists('departments')) {
            $this->db->select('d.department_name');
            $this->db->join('departments d', 'd.id = pr.requesting_department_id', 'left');
        }
        
        $this->db->order_by('pr.created_at', 'desc');
        $this->db->limit($limit);
        
        $query = $this->db->get();
        return $query->result_array();
    }

    // =============================================
    // PURCHASE ORDERS METHODS
    // =============================================

    /**
     * Save purchase order
     */
    public function savePurchaseOrder($data) {
        $this->db->insert('purchase_orders', $data);
        return $this->db->insert_id();
    }

    /**
     * Update purchase order
     */
    public function updatePurchaseOrder($po_id, $data) {
        $this->db->where('id', $po_id);
        return $this->db->update('purchase_orders', $data);
    }

    /**
     * Save purchase order item
     */
    public function savePurchaseOrderItem($data) {
        $this->db->insert('purchase_order_items', $data);
        return $this->db->insert_id();
    }

    /**
     * Update PO item received quantity
     */
    public function updatePOItemReceived($po_item_id, $quantity_received) {
        $this->db->set('quantity_received', 'quantity_received + ' . (int)$quantity_received, FALSE);
        $this->db->set('quantity_remaining', 'quantity_ordered - quantity_received', FALSE);
        $this->db->where('id', $po_item_id);
        return $this->db->update('purchase_order_items');
    }

    /**
     * Get LPO details
     */
    public function getLPODetails($lpo_id) {
        if (!$this->db->table_exists('purchase_orders')) {
            return null;
        }

        $this->db->select('po.*');
        $this->db->from('purchase_orders po');
        
        if ($this->db->table_exists('suppliers')) {
            $this->db->select('s.supplier_name, s.contact_person, s.email, s.phone, s.address');
            $this->db->join('suppliers s', 's.id = po.supplier_id', 'left');
        }
        
        $this->db->where('po.id', $lpo_id);
        
        $query = $this->db->get();
        return $query->row_array();
    }

    /**
     * Get LPO items
     */
    public function getLPOItems($lpo_id) {
        if (!$this->db->table_exists('purchase_order_items')) {
            return [];
        }

        $this->db->select('poi.*');
        $this->db->from('purchase_order_items poi');
        
        if ($this->db->table_exists('items')) {
            $this->db->select('i.item_name, i.item_code, i.unit_of_measure, i.specifications');
            $this->db->join('items i', 'i.id = poi.item_id', 'left');
        }
        
        $this->db->where('poi.purchase_order_id', $lpo_id);
        
        $query = $this->db->get();
        return $query->result_array();
    }

    /**
     * Get pending orders
     */
    public function getPendingOrders() {
        if (!$this->db->table_exists('purchase_orders')) {
            return [];
        }

        $this->db->select('po.*');
        $this->db->from('purchase_orders po');
        
        if ($this->db->table_exists('suppliers')) {
            $this->db->select('s.supplier_name');
            $this->db->join('suppliers s', 's.id = po.supplier_id', 'left');
        }
        
        $this->db->where_in('po.status', ['sent', 'confirmed']);
        $this->db->order_by('po.order_date', 'desc');
        
        $query = $this->db->get();
        return $query->result_array();
    }

    /**
     * Count pending orders
     */
    public function countPendingOrders() {
        if (!$this->db->table_exists('purchase_orders')) {
            return 0;
        }

        $this->db->from('purchase_orders');
        $this->db->where_in('status', ['draft', 'sent']);
        return $this->db->count_all_results();
    }

    /**
     * Count pending deliveries
     */
    public function countPendingDeliveries() {
        if (!$this->db->table_exists('purchase_orders')) {
            return 0;
        }

        $this->db->from('purchase_orders');
        $this->db->where_in('status', ['confirmed', 'partially_received']);
        return $this->db->count_all_results();
    }

    /**
     * Get recent orders
     */
    public function getRecentOrders($limit = 5) {
        if (!$this->db->table_exists('purchase_orders')) {
            return [];
        }

        $this->db->select('po.*');
        $this->db->from('purchase_orders po');
        
        if ($this->db->table_exists('suppliers')) {
            $this->db->select('s.supplier_name');
            $this->db->join('suppliers s', 's.id = po.supplier_id', 'left');
        }
        
        $this->db->order_by('po.created_at', 'desc');
        $this->db->limit($limit);
        
        $query = $this->db->get();
        return $query->result_array();
    }

    /**
     * Get monthly spend
     */
    public function getMonthlySpend() {
        if (!$this->db->table_exists('purchase_orders')) {
            return 0;
        }

        $this->db->select('SUM(total_amount) as monthly_spend');
        $this->db->from('purchase_orders');
        $this->db->where('MONTH(order_date)', date('m'));
        $this->db->where('YEAR(order_date)', date('Y'));
        $this->db->where_in('status', ['confirmed', 'partially_received', 'fully_received']);
        
        $query = $this->db->get();
        $result = $query->row();
        return $result ? $result->monthly_spend : 0;
    }

    /**
     * Get budget utilization
     */
    public function getBudgetUtilization() {
        if (!$this->db->table_exists('procurement_budgets')) {
            return [
                'total_budget' => 0,
                'total_spent' => 0,
                'utilization_percentage' => 0
            ];
        }

        $this->db->select('
            SUM(allocated_amount) as total_budget,
            SUM(spent_amount) as total_spent,
            (SUM(spent_amount) / SUM(allocated_amount) * 100) as utilization_percentage
        ');
        $this->db->from('procurement_budgets');
        $this->db->where('budget_year', date('Y'));
        $this->db->where('is_active', 'yes');
        
        $query = $this->db->get();
        $result = $query->row_array();
        
        return $result ?: [
            'total_budget' => 0,
            'total_spent' => 0,
            'utilization_percentage' => 0
        ];
    }

    // =============================================
    // GOODS RECEIVED NOTES METHODS
    // =============================================

    /**
     * Save GRN
     */
    public function saveGRN($data) {
        $this->db->insert('goods_received_notes', $data);
        return $this->db->insert_id();
    }

    /**
     * Save GRN item
     */
    public function saveGRNItem($data) {
        $this->db->insert('grn_items', $data);
        return $this->db->insert_id();
    }

    /**
     * Get GRN details
     */
    public function getGRNDetails($grn_id) {
        if (!$this->db->table_exists('goods_received_notes')) {
            return null;
        }

        $this->db->select('grn.*');
        $this->db->from('goods_received_notes grn');
        
        if ($this->db->table_exists('purchase_orders')) {
            $this->db->select('po.lpo_number');
            $this->db->join('purchase_orders po', 'po.id = grn.purchase_order_id', 'left');
            
            if ($this->db->table_exists('suppliers')) {
                $this->db->select('s.supplier_name');
                $this->db->join('suppliers s', 's.id = po.supplier_id', 'left');
            }
        }
        
        $this->db->where('grn.id', $grn_id);
        
        $query = $this->db->get();
        return $query->row_array();
    }

    /**
     * Get GRN items
     */
    public function getGRNItems($grn_id) {
        if (!$this->db->table_exists('grn_items')) {
            return [];
        }

        $this->db->select('gi.*');
        $this->db->from('grn_items gi');
        
        if ($this->db->table_exists('items')) {
            $this->db->select('i.item_name, i.item_code, i.unit_of_measure');
            $this->db->join('items i', 'i.id = gi.item_id', 'left');
        }
        
        $this->db->where('gi.grn_id', $grn_id);
        
        $query = $this->db->get();
        return $query->result_array();
    }

    // =============================================
    // UTILITY METHODS
    // =============================================

    /**
     * Check if table exists and create if not
     */
    public function ensureTableExists($table_name) {
        return $this->db->table_exists($table_name);
    }

    /**
     * Get staff list (fallback if staff model not available)
     */
    public function getStaffList() {
        if (!$this->db->table_exists('staff')) {
            return [];
        }

        $this->db->select('id, name, surname, employee_id');
        $this->db->from('staff');
        $this->db->where('is_active', 'yes');
        $this->db->order_by('name', 'asc');
        
        $query = $this->db->get();
        return $query->result_array();
    }

    /**
     * Get departments list (fallback if department model not available)
     */
    public function getDepartmentsList() {
        if (!$this->db->table_exists('departments')) {
            // Return sample departments if table doesn't exist
            return [
                ['id' => 1, 'department_name' => 'Pharmacy', 'department_code' => 'PHARM'],
                ['id' => 2, 'department_name' => 'Laboratory', 'department_code' => 'LAB'],
                ['id' => 3, 'department_name' => 'Radiology', 'department_code' => 'RAD'],
                ['id' => 4, 'department_name' => 'Surgery', 'department_code' => 'SURG'],
                ['id' => 5, 'department_name' => 'Emergency', 'department_code' => 'EMER']
            ];
        }

        $this->db->select('id, department_name, department_code');
        $this->db->from('departments');
        $this->db->where('is_active', 'yes');
        $this->db->order_by('department_name', 'asc');
        
        $query = $this->db->get();
        return $query->result_array();
    }

    /**
     * Create basic tables if they don't exist
     */
    public function createBasicTables() {
        // Create departments table if it doesn't exist
        if (!$this->db->table_exists('departments')) {
            $sql = "CREATE TABLE `departments` (
                `id` int(11) NOT NULL AUTO_INCREMENT,
                `department_name` varchar(100) NOT NULL,
                `department_code` varchar(20) UNIQUE NOT NULL,
                `head_of_department` int(11) DEFAULT NULL,
                `budget_allocated` decimal(15,2) DEFAULT 0.00,
                `is_active` enum('yes','no') DEFAULT 'yes',
                `created_at` timestamp DEFAULT CURRENT_TIMESTAMP,
                PRIMARY KEY (`id`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8";
            
            $this->db->query($sql);
            
            // Insert sample departments
            $departments = [
                ['department_name' => 'Pharmacy', 'department_code' => 'PHARM'],
                ['department_name' => 'Laboratory', 'department_code' => 'LAB'],
                ['department_name' => 'Radiology', 'department_code' => 'RAD'],
                ['department_name' => 'Surgery', 'department_code' => 'SURG'],
                ['department_name' => 'Emergency', 'department_code' => 'EMER'],
                ['department_name' => 'Administration', 'department_code' => 'ADMIN'],
                ['department_name' => 'Maintenance', 'department_code' => 'MAINT']
            ];
            
            $this->db->insert_batch('departments', $departments);
        }
    }
}