<?php

if (!defined('BASEPATH'))
    exit('No direct script access allowed');

/**
 * Phlebotomy Model
 * Handles all database operations for phlebotomy module
 * 
 * @author HMS Development Team
 * @version 1.0
 */
class Phlebotomy_model extends CI_Model {

    public function __construct() {
        parent::__construct();
    }

    // =============================================
    // PATIENT AND TRIAGE MANAGEMENT
    // =============================================

    /**
     * Get patients from triage queue who need phlebotomy
     */
    public function getTriageQueuePatients() {
        $this->db->select('
            p.id,
            p.patient_name,
            p.patient_unique_id,
            p.age,
            p.gender,
            p.mobileno,
            p.workflow_type,
            t.id as triage_id,
            t.priority as triage_priority,
            t.department_assigned,
            t.created_at as triage_date,
            wsh.to_status as workflow_status
        ');
        $this->db->from('patients p');
        $this->db->join('triage t', 'p.id = t.patient_id', 'inner');
        $this->db->join('workflow_step_history wsh', 'p.id = wsh.patient_id AND wsh.step_name = "triage"', 'left');
        
        // Only get patients who have completed triage but not phlebotomy
        $this->db->where('t.status', 'completed');
        $this->db->where('p.is_active', 'yes');
        
        // Exclude patients who have already completed phlebotomy
        $this->db->where('p.id NOT IN (
            SELECT DISTINCT patient_id 
            FROM phlebotomy_process_completion 
            WHERE status = "completed"
        )', null, false);
        
        $this->db->order_by('t.priority DESC, t.created_at ASC');
        $this->db->limit(50); // Limit for performance
        
        $query = $this->db->get();
        return $query->result_array();
    }

    /**
     * Get detailed triage information for a patient
     */
    public function getPatientTriageDetails($patient_id) {
        $this->db->select('
            t.*,
            p.patient_name,
            p.patient_unique_id,
            s.name as staff_name,
            s.surname as staff_surname
        ');
        $this->db->from('triage t');
        $this->db->join('patients p', 't.patient_id = p.id', 'inner');
        $this->db->join('staff s', 't.staff_id = s.id', 'left');
        $this->db->where('t.patient_id', $patient_id);
        $this->db->order_by('t.id DESC');
        $this->db->limit(1);
        
        $query = $this->db->get();
        return $query->row_array();
    }

    /**
     * Check if patient has pending phlebotomy requirements
     */
    public function hasPhlebotomyRequirements($patient_id) {
        // Check if patient has lab tests ordered
        $this->db->select('id');
        $this->db->from('pathology');
        $this->db->where('patient_id', $patient_id);
        $this->db->where('collection_date IS NULL');
        
        $query = $this->db->get();
        return $query->num_rows() > 0;
    }

    // =============================================
    // SAMPLE COLLECTION MANAGEMENT
    // =============================================

    /**
     * Save sample collection data
     */
    public function saveSampleCollection($sample_data) {
        $this->db->insert('phlebotomy_samples', $sample_data);
        
        if ($this->db->affected_rows() > 0) {
            return $this->db->insert_id();
        }
        return false;
    }

    /**
     * Update sample status
     */
    public function updateSampleStatus($patient_id, $sample_type, $status) {
        $this->db->where('patient_id', $patient_id);
        $this->db->where('sample_type', $sample_type);
        $this->db->update('phlebotomy_samples', [
            'status' => $status,
            'updated_at' => date('Y-m-d H:i:s')
        ]);
        
        return $this->db->affected_rows() > 0;
    }

    /**
     * Get sample collection data for a patient
     */
    public function getPatientSamples($patient_id) {
        $this->db->select('
            ps.*,
            s.name as collected_by_name,
            s.surname as collected_by_surname
        ');
        $this->db->from('phlebotomy_samples ps');
        $this->db->join('staff s', 'ps.collected_by = s.id', 'left');
        $this->db->where('ps.patient_id', $patient_id);
        $this->db->order_by('ps.collection_date DESC');
        
        $query = $this->db->get();
        return $query->result_array();
    }

    /**
     * Get patient sample history
     */
    public function getPatientSampleHistory($patient_id) {
        $this->db->select('
            ps.*,
            pb.barcode,
            s.name as collected_by_name,
            s.surname as collected_by_surname
        ');
        $this->db->from('phlebotomy_samples ps');
        $this->db->join('phlebotomy_barcodes pb', 'ps.id = pb.sample_id', 'left');
        $this->db->join('staff s', 'ps.collected_by = s.id', 'left');
        $this->db->where('ps.patient_id', $patient_id);
        $this->db->order_by('ps.collection_date DESC');
        
        $query = $this->db->get();
        return $query->result_array();
    }

    /**
     * Get samples by status
     */
    public function getSamplesByStatus($status, $limit = 100) {
        $this->db->select('
            ps.*,
            p.patient_name,
            p.patient_unique_id,
            s.name as collected_by_name
        ');
        $this->db->from('phlebotomy_samples ps');
        $this->db->join('patients p', 'ps.patient_id = p.id', 'inner');
        $this->db->join('staff s', 'ps.collected_by = s.id', 'left');
        $this->db->where('ps.status', $status);
        $this->db->order_by('ps.collection_date DESC');
        $this->db->limit($limit);
        
        $query = $this->db->get();
        return $query->result_array();
    }

    // =============================================
    // BARCODE MANAGEMENT
    // =============================================

    /**
     * Save barcode data
     */
    public function saveBarcode($barcode_data) {
        $this->db->insert('phlebotomy_barcodes', $barcode_data);
        
        if ($this->db->affected_rows() > 0) {
            return $this->db->insert_id();
        }
        return false;
    }

    /**
     * Get next barcode sequence number for a date and sample type
     */
    public function getNextBarcodeSequence($date, $sample_type) {
        $prefix = strtoupper(substr($sample_type, 0, 3));
        
        $this->db->select_max('CAST(SUBSTRING(barcode, -3) AS UNSIGNED)', 'max_sequence');
        $this->db->from('phlebotomy_barcodes');
        $this->db->like('barcode', $prefix . '-' . $date, 'after');
        
        $query = $this->db->get();
        $result = $query->row_array();
        
        return ($result['max_sequence'] ?? 0) + 1;
    }

    /**
     * Get barcode information
     */
    public function getBarcodeInfo($barcode) {
        $this->db->select('
            pb.*,
            ps.sample_type,
            ps.collection_date,
            ps.notes as sample_notes,
            p.patient_name,
            p.patient_unique_id,
            p.age,
            p.gender
        ');
        $this->db->from('phlebotomy_barcodes pb');
        $this->db->join('phlebotomy_samples ps', 'pb.sample_id = ps.id', 'left');
        $this->db->join('patients p', 'pb.patient_id = p.id', 'inner');
        $this->db->where('pb.barcode', $barcode);
        
        $query = $this->db->get();
        return $query->row_array();
    }

    /**
     * Update barcode status
     */
    public function updateBarcodeStatus($barcode, $status, $additional_data = []) {
        $update_data = array_merge(['status' => $status], $additional_data);
        
        $this->db->where('barcode', $barcode);
        $this->db->update('phlebotomy_barcodes', $update_data);
        
        return $this->db->affected_rows() > 0;
    }

    /**
     * Mark barcode as printed
     */
    public function markBarcodeAsPrinted($barcode, $printed_by) {
        $this->db->where('barcode', $barcode);
        $this->db->update('phlebotomy_barcodes', [
            'printed_at' => date('Y-m-d H:i:s'),
            'printed_by' => $printed_by
        ]);
        
        return $this->db->affected_rows() > 0;
    }

    // =============================================
    // VERIFICATION MANAGEMENT
    // =============================================

    /**
     * Log verification attempt
     */
    public function logVerificationAttempt($log_data) {
        $this->db->insert('phlebotomy_verification_logs', $log_data);
        
        if ($this->db->affected_rows() > 0) {
            return $this->db->insert_id();
        }
        return false;
    }

    /**
     * Update verification log
     */
    public function updateVerificationLog($log_id, $update_data) {
        $this->db->where('id', $log_id);
        $this->db->update('phlebotomy_verification_logs', $update_data);
        
        return $this->db->affected_rows() > 0;
    }

    /**
     * Get verification logs for a patient
     */
    public function getPatientVerificationLogs($patient_id, $limit = 10) {
        $this->db->select('
            pvl.*,
            s.name as attempted_by_name,
            s.surname as attempted_by_surname
        ');
        $this->db->from('phlebotomy_verification_logs pvl');
        $this->db->join('staff s', 'pvl.attempted_by = s.id', 'left');
        $this->db->where('pvl.patient_id', $patient_id);
        $this->db->order_by('pvl.attempt_time DESC');
        $this->db->limit($limit);
        
        $query = $this->db->get();
        return $query->result_array();
    }

    // =============================================
    // PROCESS COMPLETION MANAGEMENT
    // =============================================

    /**
     * Mark phlebotomy process as completed
     */
    public function markProcessCompleted($completion_data) {
        $this->db->insert('phlebotomy_process_completion', $completion_data);
        
        if ($this->db->affected_rows() > 0) {
            return $this->db->insert_id();
        }
        return false;
    }

    /**
     * Get process completion status for a patient
     */
    public function getProcessCompletionStatus($patient_id) {
        $this->db->select('
            ppc.*,
            s.name as completed_by_name,
            s.surname as completed_by_surname
        ');
        $this->db->from('phlebotomy_process_completion ppc');
        $this->db->join('staff s', 'ppc.completed_by = s.id', 'left');
        $this->db->where('ppc.patient_id', $patient_id);
        $this->db->order_by('ppc.completion_date DESC');
        $this->db->limit(1);
        
        $query = $this->db->get();
        return $query->row_array();
    }

    /**
     * Check if patient has completed phlebotomy
     */
    public function hasCompletedPhlebotomy($patient_id) {
        $this->db->select('id');
        $this->db->from('phlebotomy_process_completion');
        $this->db->where('patient_id', $patient_id);
        $this->db->where('status', 'completed');
        
        $query = $this->db->get();
        return $query->num_rows() > 0;
    }

    // =============================================
    // WORKFLOW MANAGEMENT
    // =============================================

    /**
     * Log workflow step
     */
    public function logWorkflowStep($log_data) {
        $this->db->insert('workflow_step_history', $log_data);
        
        if ($this->db->affected_rows() > 0) {
            return $this->db->insert_id();
        }
        return false;
    }

    /**
     * Get workflow history for a patient
     */
    public function getPatientWorkflowHistory($patient_id) {
        $this->db->select('
            wsh.*,
            s.name as completed_by_name,
            s.surname as completed_by_surname
        ');
        $this->db->from('workflow_step_history wsh');
        $this->db->join('staff s', 'wsh.completed_by = s.id', 'left');
        $this->db->where('wsh.patient_id', $patient_id);
        $this->db->where('wsh.workflow_type', 'phlebotomy');
        $this->db->order_by('wsh.completion_time DESC');
        
        $query = $this->db->get();
        return $query->result_array();
    }

    // =============================================
    // REPORTING AND STATISTICS
    // =============================================

    /**
     * Get phlebotomy summary statistics
     */
    public function getPhlebotomySummary($date_from, $date_to) {
        $summary = [];
        
        // Total samples collected
        $this->db->select('COUNT(*) as total_samples');
        $this->db->from('phlebotomy_samples');
        $this->db->where('collection_date >=', $date_from);
        $this->db->where('collection_date <=', $date_to . ' 23:59:59');
        $query = $this->db->get();
        $summary['total_samples'] = $query->row()->total_samples;
        
        // Samples by type
        $this->db->select('sample_type, COUNT(*) as count');
        $this->db->from('phlebotomy_samples');
        $this->db->where('collection_date >=', $date_from);
        $this->db->where('collection_date <=', $date_to . ' 23:59:59');
        $this->db->group_by('sample_type');
        $query = $this->db->get();
        $summary['samples_by_type'] = $query->result_array();
        
        // Completed processes
        $this->db->select('COUNT(*) as completed_processes');
        $this->db->from('phlebotomy_process_completion');
        $this->db->where('completion_date >=', $date_from);
        $this->db->where('completion_date <=', $date_to . ' 23:59:59');
        $this->db->where('status', 'completed');
        $query = $this->db->get();
        $summary['completed_processes'] = $query->row()->completed_processes;
        
        // Average processing time (placeholder - would need more complex calculation)
        $summary['avg_processing_time'] = '25 minutes'; // This would be calculated from actual data
        
        return $summary;
    }

    /**
     * Get daily collection statistics
     */
    public function getDailyCollections($date_from, $date_to) {
        $this->db->select('
            DATE(collection_date) as collection_date,
            COUNT(*) as total_collections,
            COUNT(CASE WHEN sample_type = "blood" THEN 1 END) as blood_samples,
            COUNT(CASE WHEN sample_type = "stool" THEN 1 END) as stool_samples
        ');
        $this->db->from('phlebotomy_samples');
        $this->db->where('collection_date >=', $date_from);
        $this->db->where('collection_date <=', $date_to . ' 23:59:59');
        $this->db->group_by('DATE(collection_date)');
        $this->db->order_by('collection_date ASC');
        
        $query = $this->db->get();
        return $query->result_array();
    }

    /**
     * Get sample type statistics
     */
    public function getSampleTypeStats($date_from, $date_to) {
        $this->db->select('
            sample_type,
            COUNT(*) as count,
            COUNT(CASE WHEN status = "completed" THEN 1 END) as completed,
            COUNT(CASE WHEN status = "pending" THEN 1 END) as pending
        ');
        $this->db->from('phlebotomy_samples');
        $this->db->where('collection_date >=', $date_from);
        $this->db->where('collection_date <=', $date_to . ' 23:59:59');
        $this->db->group_by('sample_type');
        $this->db->order_by('count DESC');
        
        $query = $this->db->get();
        return $query->result_array();
    }

    /**
     * Get technician performance statistics
     */
    public function getTechnicianStats($date_from, $date_to) {
        $this->db->select('
            s.name,
            s.surname,
            COUNT(ps.id) as samples_collected,
            COUNT(DISTINCT ps.patient_id) as patients_served,
            COUNT(ppc.id) as processes_completed
        ');
        $this->db->from('phlebotomy_samples ps');
        $this->db->join('staff s', 'ps.collected_by = s.id', 'inner');
        $this->db->join('phlebotomy_process_completion ppc', 'ps.patient_id = ppc.patient_id AND ppc.completed_by = s.id', 'left');
        $this->db->where('ps.collection_date >=', $date_from);
        $this->db->where('ps.collection_date <=', $date_to . ' 23:59:59');
        $this->db->group_by('ps.collected_by, s.name, s.surname');
        $this->db->order_by('samples_collected DESC');
        
        $query = $this->db->get();
        return $query->result_array();
    }

    /**
     * Get pending samples that need attention
     */
    public function getPendingSamples() {
        $this->db->select('
            ps.*,
            p.patient_name,
            p.patient_unique_id,
            s.name as collected_by_name,
            TIMESTAMPDIFF(HOUR, ps.collection_date, NOW()) as hours_pending
        ');
        $this->db->from('phlebotomy_samples ps');
        $this->db->join('patients p', 'ps.patient_id = p.id', 'inner');
        $this->db->join('staff s', 'ps.collected_by = s.id', 'left');
        $this->db->where_in('ps.status', ['pending', 'collected']);
        $this->db->order_by('ps.collection_date ASC');
        $this->db->limit(50);
        
        $query = $this->db->get();
        return $query->result_array();
    }

    // =============================================
    // UTILITY METHODS
    // =============================================

    /**
     * Search patients for phlebotomy
     */
    public function searchPatients($search_term, $limit = 20) {
        $this->db->select('
            p.id,
            p.patient_name,
            p.patient_unique_id,
            p.age,
            p.gender,
            p.mobileno
        ');
        $this->db->from('patients p');
        $this->db->where('p.is_active', 'yes');
        
        // Search by name, ID, or phone
        $this->db->group_start();
        $this->db->like('p.patient_name', $search_term);
        $this->db->or_like('p.patient_unique_id', $search_term);
        $this->db->or_like('p.mobileno', $search_term);
        $this->db->group_end();
        
        $this->db->order_by('p.patient_name ASC');
        $this->db->limit($limit);
        
        $query = $this->db->get();
        return $query->result_array();
    }

    /**
     * Get sample statistics for dashboard
     */
    public function getDashboardStats() {
        $stats = [];
        
        // Today's collections
        $this->db->select('COUNT(*) as today_collections');
        $this->db->from('phlebotomy_samples');
        $this->db->where('DATE(collection_date)', date('Y-m-d'));
        $query = $this->db->get();
        $stats['today_collections'] = $query->row()->today_collections;
        
        // Pending samples
        $this->db->select('COUNT(*) as pending_samples');
        $this->db->from('phlebotomy_samples');
        $this->db->where('status', 'pending');
        $query = $this->db->get();
        $stats['pending_samples'] = $query->row()->pending_samples;
        
        // This week's collections
        $this->db->select('COUNT(*) as week_collections');
        $this->db->from('phlebotomy_samples');
        $this->db->where('collection_date >=', date('Y-m-d', strtotime('-7 days')));
        $query = $this->db->get();
        $stats['week_collections'] = $query->row()->week_collections;
        
        // Active barcodes
        $this->db->select('COUNT(*) as active_barcodes');
        $this->db->from('phlebotomy_barcodes');
        $this->db->where('status', 'active');
        $query = $this->db->get();
        $stats['active_barcodes'] = $query->row()->active_barcodes;
        
        return $stats;
    }

    /**
     * Validate barcode format
     */
    public function validateBarcodeFormat($barcode) {
        // Expected format: XXX-YYYYMMDD-NNN (e.g., BLO-20250720-001)
        $pattern = '/^[A-Z]{3}-\d{8}-\d{3}$/';
        return preg_match($pattern, $barcode);
    }

    /**
     * Check if barcode exists
     */
    public function barcodeExists($barcode) {
        $this->db->select('id');
        $this->db->from('phlebotomy_barcodes');
        $this->db->where('barcode', $barcode);
        
        $query = $this->db->get();
        return $query->num_rows() > 0;
    }

    /**
     * Get patient's current workflow step
     */
    public function getPatientWorkflowStep($patient_id) {
        $this->db->select('step_name, to_status');
        $this->db->from('workflow_step_history');
        $this->db->where('patient_id', $patient_id);
        $this->db->order_by('completion_time DESC');
        $this->db->limit(1);
        
        $query = $this->db->get();
        return $query->row_array();
    }

    /**
     * Update sample with additional information
     */
    public function updateSample($sample_id, $update_data) {
        $update_data['updated_at'] = date('Y-m-d H:i:s');
        
        $this->db->where('id', $sample_id);
        $this->db->update('phlebotomy_samples', $update_data);
        
        return $this->db->affected_rows() > 0;
    }

    /**
     * Delete sample (soft delete by changing status)
     */
    public function deleteSample($sample_id, $reason = null) {
        $update_data = [
            'status' => 'cancelled',
            'notes' => ($reason ? 'Cancelled: ' . $reason : 'Cancelled'),
            'updated_at' => date('Y-m-d H:i:s')
        ];
        
        $this->db->where('id', $sample_id);
        $this->db->update('phlebotomy_samples', $update_data);
        
        return $this->db->affected_rows() > 0;
    }
}

/* End of file Phlebotomy_model.php */
/* Location: ./application/models/Phlebotomy_model.php */