<?php

defined('BASEPATH') OR exit('No direct script access allowed');

class Patient_billing_model extends CI_Model {

    public function __construct() {
        parent::__construct();
        $this->load->database();
    }

    /**
     * Get all patients with their billing summary
     */
    public function getAllPatientsWithBilling() {
        try {
            $this->db->select('
                p.id,
                p.patient_name,
                p.patient_unique_id,
                p.age,
                p.gender,
                p.mobileno,
                p.email,
                p.address,
                p.admission_date,
                p.is_active,
                bg.name as blood_group_name,
                MAX(COALESCE(opd.appointment_date, ipd.date, lab.date, pharmacy.date, radio.date)) as last_visit,
                
                -- Consultation totals
                COALESCE(SUM(CASE WHEN opd.id IS NOT NULL THEN opd.amount ELSE 0 END), 0) as consultation_total,
                
                -- Laboratory totals  
                COALESCE(SUM(CASE WHEN lab.id IS NOT NULL THEN lab.amount ELSE 0 END), 0) as lab_total,
                
                -- Pharmacy totals
                COALESCE(SUM(CASE WHEN pharmacy.id IS NOT NULL THEN pharmacy.net_amount ELSE 0 END), 0) as pharmacy_total,
                
                -- Radiology totals
                COALESCE(SUM(CASE WHEN radio.id IS NOT NULL THEN radio.amount ELSE 0 END), 0) as radiology_total
            ');
            
            $this->db->from('patients p');
            $this->db->join('blood_group bg', 'bg.id = p.blood_group_id', 'left');
            
            // Join with OPD (consultations)
            $this->db->join('opd_details opd', 'opd.patient_id = p.id', 'left');
            
            // Join with Laboratory
            $this->db->join('pathology_billing lab', 'lab.patient_id = p.id', 'left');
            
            // Join with Pharmacy
            $this->db->join('pharmacy_bill_basic pharmacy', 'pharmacy.patient_id = p.id', 'left');
            
            // Join with Radiology
            $this->db->join('radiology_billing radio', 'radio.patient_id = p.id', 'left');
            
            $this->db->where('p.is_active', 'yes');
            $this->db->group_by('p.id');
            $this->db->order_by('p.patient_name', 'ASC');
            
            $query = $this->db->get();
            
            if ($query->num_rows() > 0) {
                $patients = $query->result_array();
                
                // Calculate total billing for each patient
                foreach ($patients as &$patient) {
                    $patient['total_billing'] = $patient['consultation_total'] + 
                                              $patient['lab_total'] + 
                                              $patient['pharmacy_total'] + 
                                              $patient['radiology_total'];
                }
                
                return $patients;
            }
            
            return [];
        } catch (Exception $e) {
            log_message('error', 'Error in getAllPatientsWithBilling: ' . $e->getMessage());
            return false;
        }
    }

    /**
     * Get dashboard statistics
     */
    public function getDashboardStatistics() {
        try {
            $stats = [];
            
            // Total patients
            $this->db->where('is_active', 'yes');
            $stats['total_patients'] = $this->db->count_all_results('patients');
            
            // Total revenue (all time)
            $total_revenue = 0;
            
            // OPD revenue
            $this->db->select_sum('amount');
            $opd_revenue = $this->db->get('opd_details')->row()->amount ?: 0;
            $total_revenue += $opd_revenue;
            
            // Laboratory revenue
            $this->db->select_sum('amount');
            $lab_revenue = $this->db->get('pathology_billing')->row()->amount ?: 0;
            $total_revenue += $lab_revenue;
            
            // Pharmacy revenue
            $this->db->select_sum('net_amount');
            $pharmacy_revenue = $this->db->get('pharmacy_bill_basic')->row()->net_amount ?: 0;
            $total_revenue += $pharmacy_revenue;
            
            // Radiology revenue
            $this->db->select_sum('amount');
            $radio_revenue = $this->db->get('radiology_billing')->row()->amount ?: 0;
            $total_revenue += $radio_revenue;
            
            $stats['total_revenue'] = $total_revenue;
            
            // Today's consultations
            $today = date('Y-m-d');
            $this->db->where('DATE(appointment_date)', $today);
            $stats['today_consultations'] = $this->db->count_all_results('opd_details');
            
            // Pending lab results
            $this->db->where('pathology_report', '');
            $this->db->or_where('pathology_report IS NULL');
            $stats['pending_labs'] = $this->db->count_all_results('pathology_billing');
            
            return $stats;
        } catch (Exception $e) {
            log_message('error', 'Error in getDashboardStatistics: ' . $e->getMessage());
            return [
                'total_patients' => 0,
                'total_revenue' => 0,
                'today_consultations' => 0,
                'pending_labs' => 0
            ];
        }
    }

    /**
     * Get detailed billing information for a specific patient
     */
    public function getPatientBillingDetails($patient_id) {
        try {
            $billing_details = [
                'consultations' => [],
                'laboratory' => [],
                'pharmacy' => [],
                'radiology' => [],
                'totals' => [
                    'consultations' => 0,
                    'laboratory' => 0,
                    'pharmacy' => 0,
                    'radiology' => 0,
                    'grand_total' => 0
                ]
            ];

            // Get consultation details
            $this->db->select('
                opd.id,
                opd.appointment_date as date,
                opd.amount,
                opd.symptoms,
                opd.refference,
                opd.cons_doctor,
                CONCAT(staff.name, " ", staff.surname) as doctor_name,
                "Consultation" as type,
                CASE WHEN payment.id IS NOT NULL THEN "Paid" ELSE "Pending" END as status
            ');
            $this->db->from('opd_details opd');
            $this->db->join('staff', 'staff.id = opd.cons_doctor', 'left');
            $this->db->join('transactions payment', 'payment.opd_id = opd.id', 'left');
            $this->db->where('opd.patient_id', $patient_id);
            $this->db->order_by('opd.appointment_date', 'DESC');
            
            $consultations = $this->db->get()->result_array();
            $billing_details['consultations'] = $consultations;
            $billing_details['totals']['consultations'] = array_sum(array_column($consultations, 'amount'));

            // Get laboratory details
            $this->db->select('
                pb.id,
                pb.date,
                pb.amount,
                p.test_name,
                pb.pathology_report as result,
                CASE WHEN pb.pathology_report != "" THEN "Completed" ELSE "Pending" END as status,
                CASE WHEN payment.id IS NOT NULL THEN "Paid" ELSE "Pending" END as payment_status
            ');
            $this->db->from('pathology_billing pb');
            $this->db->join('pathology p', 'p.id = pb.pathology_id', 'left');
            $this->db->join('transactions payment', 'payment.pathology_billing_id = pb.id', 'left');
            $this->db->where('pb.patient_id', $patient_id);
            $this->db->order_by('pb.date', 'DESC');
            
            $laboratory = $this->db->get()->result_array();
            $billing_details['laboratory'] = $laboratory;
            $billing_details['totals']['laboratory'] = array_sum(array_column($laboratory, 'amount'));

            // Get pharmacy details
            $this->db->select('
                pbb.id,
                pbb.date,
                pbb.net_amount as amount,
                GROUP_CONCAT(CONCAT(med.medicine_name, " (", pbd.quantity, ")") SEPARATOR ", ") as medications,
                pbb.discount,
                pbb.tax,
                CASE WHEN payment.id IS NOT NULL THEN "Paid" ELSE "Pending" END as status
            ');
            $this->db->from('pharmacy_bill_basic pbb');
            $this->db->join('pharmacy_bill_detail pbd', 'pbd.pharmacy_bill_basic_id = pbb.id', 'left');
            $this->db->join('medicine_dosage md', 'md.id = pbd.medicine_dosage_id', 'left');
            $this->db->join('medicine med', 'med.id = md.medicine_id', 'left');
            $this->db->join('transactions payment', 'payment.pharmacy_bill_basic_id = pbb.id', 'left');
            $this->db->where('pbb.patient_id', $patient_id);
            $this->db->group_by('pbb.id');
            $this->db->order_by('pbb.date', 'DESC');
            
            $pharmacy = $this->db->get()->result_array();
            $billing_details['pharmacy'] = $pharmacy;
            $billing_details['totals']['pharmacy'] = array_sum(array_column($pharmacy, 'amount'));

            // Get radiology details
            $this->db->select('
                rb.id,
                rb.date,
                rb.amount,
                r.test_name as scan_name,
                rb.radiology_report as result,
                CASE WHEN rb.radiology_report != "" THEN "Completed" ELSE "Pending" END as status,
                CASE WHEN payment.id IS NOT NULL THEN "Paid" ELSE "Pending" END as payment_status
            ');
            $this->db->from('radiology_billing rb');
            $this->db->join('radiology r', 'r.id = rb.radiology_id', 'left');
            $this->db->join('transactions payment', 'payment.radiology_billing_id = rb.id', 'left');
            $this->db->where('rb.patient_id', $patient_id);
            $this->db->order_by('rb.date', 'DESC');
            
            $radiology = $this->db->get()->result_array();
            $billing_details['radiology'] = $radiology;
            $billing_details['totals']['radiology'] = array_sum(array_column($radiology, 'amount'));

            // Calculate grand total
            $billing_details['totals']['grand_total'] = 
                $billing_details['totals']['consultations'] +
                $billing_details['totals']['laboratory'] +
                $billing_details['totals']['pharmacy'] +
                $billing_details['totals']['radiology'];

            return $billing_details;
        } catch (Exception $e) {
            log_message('error', 'Error in getPatientBillingDetails: ' . $e->getMessage());
            return false;
        }
    }

    /**
     * Search patients based on criteria
     */
    public function searchPatients($search_term = '', $status_filter = '', $sort_by = 'name') {
        try {
            $this->db->select('
                p.id,
                p.patient_name,
                p.patient_unique_id,
                p.age,
                p.gender,
                p.mobileno,
                p.email,
                p.address,
                p.admission_date,
                p.is_active,
                bg.name as blood_group_name,
                MAX(COALESCE(opd.appointment_date, ipd.date, lab.date, pharmacy.date, radio.date)) as last_visit,
                
                COALESCE(SUM(CASE WHEN opd.id IS NOT NULL THEN opd.amount ELSE 0 END), 0) as consultation_total,
                COALESCE(SUM(CASE WHEN lab.id IS NOT NULL THEN lab.amount ELSE 0 END), 0) as lab_total,
                COALESCE(SUM(CASE WHEN pharmacy.id IS NOT NULL THEN pharmacy.net_amount ELSE 0 END), 0) as pharmacy_total,
                COALESCE(SUM(CASE WHEN radio.id IS NOT NULL THEN radio.amount ELSE 0 END), 0) as radiology_total
            ');
            
            $this->db->from('patients p');
            $this->db->join('blood_group bg', 'bg.id = p.blood_group_id', 'left');
            $this->db->join('opd_details opd', 'opd.patient_id = p.id', 'left');
            $this->db->join('pathology_billing lab', 'lab.patient_id = p.id', 'left');
            $this->db->join('pharmacy_bill_basic pharmacy', 'pharmacy.patient_id = p.id', 'left');
            $this->db->join('radiology_billing radio', 'radio.patient_id = p.id', 'left');
            
            // Apply search filter
            if (!empty($search_term)) {
                $this->db->group_start();
                $this->db->like('p.patient_name', $search_term);
                $this->db->or_like('p.patient_unique_id', $search_term);
                $this->db->or_like('p.mobileno', $search_term);
                $this->db->or_like('p.email', $search_term);
                $this->db->group_end();
            }
            
            // Apply status filter
            if (!empty($status_filter)) {
                if ($status_filter === 'active') {
                    $this->db->where('p.is_active', 'yes');
                } elseif ($status_filter === 'inactive') {
                    $this->db->where('p.is_active', 'no');
                }
            } else {
                $this->db->where('p.is_active', 'yes');
            }
            
            $this->db->group_by('p.id');
            
            // Apply sorting
            switch ($sort_by) {
                case 'name':
                    $this->db->order_by('p.patient_name', 'ASC');
                    break;
                case 'last_visit':
                    $this->db->order_by('last_visit', 'DESC');
                    break;
                case 'total_billing':
                    $this->db->order_by('(consultation_total + lab_total + pharmacy_total + radiology_total)', 'DESC');
                    break;
                default:
                    $this->db->order_by('p.patient_name', 'ASC');
            }
            
            $query = $this->db->get();
            
            if ($query->num_rows() > 0) {
                $patients = $query->result_array();
                
                // Calculate total billing for each patient
                foreach ($patients as &$patient) {
                    $patient['total_billing'] = $patient['consultation_total'] + 
                                              $patient['lab_total'] + 
                                              $patient['pharmacy_total'] + 
                                              $patient['radiology_total'];
                }
                
                return $patients;
            }
            
            return [];
        } catch (Exception $e) {
            log_message('error', 'Error in searchPatients: ' . $e->getMessage());
            return false;
        }
    }

    /**
     * Get billing report for specified date range
     */
    public function getBillingReport($date_from, $date_to, $patient_id = null) {
        try {
            $report = [
                'summary' => [
                    'total_consultations' => 0,
                    'total_laboratory' => 0,
                    'total_pharmacy' => 0,
                    'total_radiology' => 0,
                    'grand_total' => 0
                ],
                'details' => []
            ];

            // Build date condition
            $date_condition = '';
            if (!empty($date_from) && !empty($date_to)) {
                $date_condition = " AND DATE(date_field) BETWEEN '$date_from' AND '$date_to'";
            }

            // Patient condition
            $patient_condition = '';
            if (!empty($patient_id)) {
                $patient_condition = " AND patient_id = $patient_id";
            }

            // Get consultation totals
            $consultation_query = "
                SELECT 
                    COUNT(*) as count,
                    COALESCE(SUM(amount), 0) as total
                FROM opd_details 
                WHERE 1=1 $patient_condition
                " . str_replace('date_field', 'appointment_date', $date_condition);
            
            $consultation_result = $this->db->query($consultation_query)->row_array();
            $report['summary']['total_consultations'] = $consultation_result['total'];

            // Get laboratory totals
            $lab_query = "
                SELECT 
                    COUNT(*) as count,
                    COALESCE(SUM(amount), 0) as total
                FROM pathology_billing 
                WHERE 1=1 $patient_condition
                " . str_replace('date_field', 'date', $date_condition);
            
            $lab_result = $this->db->query($lab_query)->row_array();
            $report['summary']['total_laboratory'] = $lab_result['total'];

            // Get pharmacy totals
            $pharmacy_query = "
                SELECT 
                    COUNT(*) as count,
                    COALESCE(SUM(net_amount), 0) as total
                FROM pharmacy_bill_basic 
                WHERE 1=1 $patient_condition
                " . str_replace('date_field', 'date', $date_condition);
            
            $pharmacy_result = $this->db->query($pharmacy_query)->row_array();
            $report['summary']['total_pharmacy'] = $pharmacy_result['total'];

            // Get radiology totals
            $radiology_query = "
                SELECT 
                    COUNT(*) as count,
                    COALESCE(SUM(amount), 0) as total
                FROM radiology_billing 
                WHERE 1=1 $patient_condition
                " . str_replace('date_field', 'date', $date_condition);
            
            $radiology_result = $this->db->query($radiology_query)->row_array();
            $report['summary']['total_radiology'] = $radiology_result['total'];

            // Calculate grand total
            $report['summary']['grand_total'] = 
                $report['summary']['total_consultations'] +
                $report['summary']['total_laboratory'] +
                $report['summary']['total_pharmacy'] +
                $report['summary']['total_radiology'];

            return $report;
        } catch (Exception $e) {
            log_message('error', 'Error in getBillingReport: ' . $e->getMessage());
            return false;
        }
    }

    /**
     * Get billing trends for charts
     */
    public function getBillingTrends($period = 'monthly') {
        try {
            $trends = [];
            
            // Determine date format based on period
            $date_format = ($period === 'daily') ? '%Y-%m-%d' : 
                          (($period === 'weekly') ? '%Y-%u' : '%Y-%m');
            
            // Get consultation trends
            $this->db->select("DATE_FORMAT(appointment_date, '$date_format') as period, SUM(amount) as total");
            $this->db->from('opd_details');
            $this->db->where('appointment_date >=', date('Y-m-d', strtotime('-12 months')));
            $this->db->group_by('period');
            $this->db->order_by('period');
            $consultation_trends = $this->db->get()->result_array();

            // Get laboratory trends
            $this->db->select("DATE_FORMAT(date, '$date_format') as period, SUM(amount) as total");
            $this->db->from('pathology_billing');
            $this->db->where('date >=', date('Y-m-d', strtotime('-12 months')));
            $this->db->group_by('period');
            $this->db->order_by('period');
            $lab_trends = $this->db->get()->result_array();

            // Combine trends data
            $trends = [
                'consultations' => $consultation_trends,
                'laboratory' => $lab_trends
            ];

            return $trends;
        } catch (Exception $e) {
            log_message('error', 'Error in getBillingTrends: ' . $e->getMessage());
            return [];
        }
    }

    /**
     * Get outstanding payments
     */
    public function getOutstandingPayments() {
        try {
            $outstanding = [];

            // Outstanding consultation payments
            $this->db->select('
                opd.id,
                opd.patient_id,
                p.patient_name,
                opd.appointment_date as date,
                opd.amount,
                "Consultation" as type
            ');
            $this->db->from('opd_details opd');
            $this->db->join('patients p', 'p.id = opd.patient_id');
            $this->db->join('transactions t', 't.opd_id = opd.id', 'left');
            $this->db->where('t.id IS NULL');
            $this->db->where('opd.amount >', 0);
            $consultation_outstanding = $this->db->get()->result_array();

            // Outstanding laboratory payments
            $this->db->select('
                pb.id,
                pb.patient_id,
                p.patient_name,
                pb.date,
                pb.amount,
                "Laboratory" as type
            ');
            $this->db->from('pathology_billing pb');
            $this->db->join('patients p', 'p.id = pb.patient_id');
            $this->db->join('transactions t', 't.pathology_billing_id = pb.id', 'left');
            $this->db->where('t.id IS NULL');
            $this->db->where('pb.amount >', 0);
            $lab_outstanding = $this->db->get()->result_array();

            // Outstanding pharmacy payments
            $this->db->select('
                pbb.id,
                pbb.patient_id,
                p.patient_name,
                pbb.date,
                pbb.net_amount as amount,
                "Pharmacy" as type
            ');
            $this->db->from('pharmacy_bill_basic pbb');
            $this->db->join('patients p', 'p.id = pbb.patient_id');
            $this->db->join('transactions t', 't.pharmacy_bill_basic_id = pbb.id', 'left');
            $this->db->where('t.id IS NULL');
            $this->db->where('pbb.net_amount >', 0);
            $pharmacy_outstanding = $this->db->get()->result_array();

            // Outstanding radiology payments
            $this->db->select('
                rb.id,
                rb.patient_id,
                p.patient_name,
                rb.date,
                rb.amount,
                "Radiology" as type
            ');
            $this->db->from('radiology_billing rb');
            $this->db->join('patients p', 'p.id = rb.patient_id');
            $this->db->join('transactions t', 't.radiology_billing_id = rb.id', 'left');
            $this->db->where('t.id IS NULL');
            $this->db->where('rb.amount >', 0);
            $radiology_outstanding = $this->db->get()->result_array();

            // Combine all outstanding payments
            $outstanding = array_merge(
                $consultation_outstanding,
                $lab_outstanding,
                $pharmacy_outstanding,
                $radiology_outstanding
            );

            // Sort by date (most recent first)
            usort($outstanding, function($a, $b) {
                return strtotime($b['date']) - strtotime($a['date']);
            });

            return $outstanding;
        } catch (Exception $e) {
            log_message('error', 'Error in getOutstandingPayments: ' . $e->getMessage());
            return [];
        }
    }

    /**
     * Update payment status
     */
    public function updatePaymentStatus($transaction_id, $status, $module) {
        try {
            $this->db->trans_start();

            // Update payment status in transactions table
            $update_data = [
                'payment_mode' => $status,
                'payment_date' => date('Y-m-d H:i:s'),
                'updated_at' => date('Y-m-d H:i:s')
            ];

            $this->db->where('id', $transaction_id);
            $this->db->update('transactions', $update_data);

            // Update status in respective module table
            switch ($module) {
                case 'consultation':
                    $this->db->where('opd_id', $transaction_id);
                    $this->db->update('opd_details', ['payment_status' => $status]);
                    break;
                case 'laboratory':
                    $this->db->where('pathology_billing_id', $transaction_id);
                    $this->db->update('pathology_billing', ['payment_status' => $status]);
                    break;
                case 'pharmacy':
                    $this->db->where('pharmacy_bill_basic_id', $transaction_id);
                    $this->db->update('pharmacy_bill_basic', ['payment_status' => $status]);
                    break;
                case 'radiology':
                    $this->db->where('radiology_billing_id', $transaction_id);
                    $this->db->update('radiology_billing', ['payment_status' => $status]);
                    break;
            }

            $this->db->trans_complete();

            if ($this->db->trans_status() === FALSE) {
                return false;
            }

            return true;
        } catch (Exception $e) {
            log_message('error', 'Error in updatePaymentStatus: ' . $e->getMessage());
            return false;
        }
    }

    /**
     * Generate billing summary report
     */
    public function generateBillingSummary($date_from, $date_to, $department = '') {
        try {
            $summary = [
                'period' => [
                    'from' => $date_from,
                    'to' => $date_to
                ],
                'departments' => [],
                'totals' => [
                    'gross_amount' => 0,
                    'discount_amount' => 0,
                    'tax_amount' => 0,
                    'net_amount' => 0,
                    'paid_amount' => 0,
                    'outstanding_amount' => 0
                ]
            ];

            $where_date = '';
            if (!empty($date_from) && !empty($date_to)) {
                $where_date = " AND DATE(date_field) BETWEEN '$date_from' AND '$date_to'";
            }

            // Consultation summary
            if (empty($department) || $department === 'consultation') {
                $consultation_query = "
                    SELECT 
                        'Consultation' as department,
                        COUNT(*) as transaction_count,
                        SUM(opd.amount) as gross_amount,
                        SUM(COALESCE(opd.discount, 0)) as discount_amount,
                        SUM(COALESCE(opd.tax, 0)) as tax_amount,
                        SUM(opd.amount) as net_amount,
                        SUM(CASE WHEN t.id IS NOT NULL THEN opd.amount ELSE 0 END) as paid_amount,
                        SUM(CASE WHEN t.id IS NULL THEN opd.amount ELSE 0 END) as outstanding_amount
                    FROM opd_details opd
                    LEFT JOIN transactions t ON t.opd_id = opd.id
                    WHERE 1=1 " . str_replace('date_field', 'opd.appointment_date', $where_date);
                
                $consultation_summary = $this->db->query($consultation_query)->row_array();
                $summary['departments']['consultation'] = $consultation_summary;
            }

            // Laboratory summary
            if (empty($department) || $department === 'laboratory') {
                $lab_query = "
                    SELECT 
                        'Laboratory' as department,
                        COUNT(*) as transaction_count,
                        SUM(pb.amount) as gross_amount,
                        SUM(COALESCE(pb.discount, 0)) as discount_amount,
                        SUM(COALESCE(pb.tax, 0)) as tax_amount,
                        SUM(pb.amount) as net_amount,
                        SUM(CASE WHEN t.id IS NOT NULL THEN pb.amount ELSE 0 END) as paid_amount,
                        SUM(CASE WHEN t.id IS NULL THEN pb.amount ELSE 0 END) as outstanding_amount
                    FROM pathology_billing pb
                    LEFT JOIN transactions t ON t.pathology_billing_id = pb.id
                    WHERE 1=1 " . str_replace('date_field', 'pb.date', $where_date);
                
                $lab_summary = $this->db->query($lab_query)->row_array();
                $summary['departments']['laboratory'] = $lab_summary;
            }

            // Pharmacy summary
            if (empty($department) || $department === 'pharmacy') {
                $pharmacy_query = "
                    SELECT 
                        'Pharmacy' as department,
                        COUNT(*) as transaction_count,
                        SUM(pbb.total) as gross_amount,
                        SUM(COALESCE(pbb.discount, 0)) as discount_amount,
                        SUM(COALESCE(pbb.tax, 0)) as tax_amount,
                        SUM(pbb.net_amount) as net_amount,
                        SUM(CASE WHEN t.id IS NOT NULL THEN pbb.net_amount ELSE 0 END) as paid_amount,
                        SUM(CASE WHEN t.id IS NULL THEN pbb.net_amount ELSE 0 END) as outstanding_amount
                    FROM pharmacy_bill_basic pbb
                    LEFT JOIN transactions t ON t.pharmacy_bill_basic_id = pbb.id
                    WHERE 1=1 " . str_replace('date_field', 'pbb.date', $where_date);
                
                $pharmacy_summary = $this->db->query($pharmacy_query)->row_array();
                $summary['departments']['pharmacy'] = $pharmacy_summary;
            }

            // Radiology summary
            if (empty($department) || $department === 'radiology') {
                $radiology_query = "
                    SELECT 
                        'Radiology' as department,
                        COUNT(*) as transaction_count,
                        SUM(rb.amount) as gross_amount,
                        SUM(COALESCE(rb.discount, 0)) as discount_amount,
                        SUM(COALESCE(rb.tax, 0)) as tax_amount,
                        SUM(rb.amount) as net_amount,
                        SUM(CASE WHEN t.id IS NOT NULL THEN rb.amount ELSE 0 END) as paid_amount,
                        SUM(CASE WHEN t.id IS NULL THEN rb.amount ELSE 0 END) as outstanding_amount
                    FROM radiology_billing rb
                    LEFT JOIN transactions t ON t.radiology_billing_id = rb.id
                    WHERE 1=1 " . str_replace('date_field', 'rb.date', $where_date);
                
                $radiology_summary = $this->db->query($radiology_query)->row_array();
                $summary['departments']['radiology'] = $radiology_summary;
            }

            // Calculate totals across all departments
            foreach ($summary['departments'] as $dept) {
                $summary['totals']['gross_amount'] += $dept['gross_amount'] ?: 0;
                $summary['totals']['discount_amount'] += $dept['discount_amount'] ?: 0;
                $summary['totals']['tax_amount'] += $dept['tax_amount'] ?: 0;
                $summary['totals']['net_amount'] += $dept['net_amount'] ?: 0;
                $summary['totals']['paid_amount'] += $dept['paid_amount'] ?: 0;
                $summary['totals']['outstanding_amount'] += $dept['outstanding_amount'] ?: 0;
            }

            return $summary;
        } catch (Exception $e) {
            log_message('error', 'Error in generateBillingSummary: ' . $e->getMessage());
            return false;
        }
    }

    /**
     * Get patient billing history with pagination
     */
    public function getPatientBillingHistory($patient_id, $limit = 10, $offset = 0) {
        try {
            $history = [];

            // Union query to get all billing records
            $query = "
                (SELECT 
                    'consultation' as type,
                    opd.id,
                    opd.appointment_date as date,
                    CONCAT('Dr. ', s.name, ' ', s.surname) as description,
                    opd.amount,
                    CASE WHEN t.id IS NOT NULL THEN 'Paid' ELSE 'Pending' END as status
                FROM opd_details opd
                LEFT JOIN staff s ON s.id = opd.cons_doctor
                LEFT JOIN transactions t ON t.opd_id = opd.id
                WHERE opd.patient_id = $patient_id)
                
                UNION ALL
                
                (SELECT 
                    'laboratory' as type,
                    pb.id,
                    pb.date,
                    p.test_name as description,
                    pb.amount,
                    CASE WHEN t.id IS NOT NULL THEN 'Paid' ELSE 'Pending' END as status
                FROM pathology_billing pb
                LEFT JOIN pathology p ON p.id = pb.pathology_id
                LEFT JOIN transactions t ON t.pathology_billing_id = pb.id
                WHERE pb.patient_id = $patient_id)
                
                UNION ALL
                
                (SELECT 
                    'pharmacy' as type,
                    pbb.id,
                    pbb.date,
                    'Pharmacy Bill' as description,
                    pbb.net_amount as amount,
                    CASE WHEN t.id IS NOT NULL THEN 'Paid' ELSE 'Pending' END as status
                FROM pharmacy_bill_basic pbb
                LEFT JOIN transactions t ON t.pharmacy_bill_basic_id = pbb.id
                WHERE pbb.patient_id = $patient_id)
                
                UNION ALL
                
                (SELECT 
                    'radiology' as type,
                    rb.id,
                    rb.date,
                    r.test_name as description,
                    rb.amount,
                    CASE WHEN t.id IS NOT NULL THEN 'Paid' ELSE 'Pending' END as status
                FROM radiology_billing rb
                LEFT JOIN radiology r ON r.id = rb.radiology_id
                LEFT JOIN transactions t ON t.radiology_billing_id = rb.id
                WHERE rb.patient_id = $patient_id)
                
                ORDER BY date DESC
                LIMIT $limit OFFSET $offset
            ";

            $history = $this->db->query($query)->result_array();

            return $history;
        } catch (Exception $e) {
            log_message('error', 'Error in getPatientBillingHistory: ' . $e->getMessage());
            return [];
        }
    }
}