<?php
if (!defined('BASEPATH'))
    exit('No direct script access allowed');

class Journal_model extends CI_Model {

    public function __construct() {
        parent::__construct();
    }

    /**
     * Add journal header
     */
    public function addJournalHeader($data) {
        $data['created_at'] = date('Y-m-d H:i:s');
        $data['status'] = $data['status'] ?? 'posted';
        
        if ($this->db->insert('journal_headers', $data)) {
            return $this->db->insert_id();
        }
        return false;
    }

    /**
     * Add journal line
     */
    public function addJournalLine($data) {
        $data['created_at'] = date('Y-m-d H:i:s');
        return $this->db->insert('journal_lines', $data);
    }

    /**
     * Get journal entries with lines
     */
    public function getJournalEntries($limit = 50, $offset = 0, $filters = []) {
        $this->db->select('
            jh.*,
            s.name as created_by_name,
            s.surname as created_by_surname
        ');
        $this->db->from('journal_headers jh');
        $this->db->join('staff s', 's.id = jh.created_by', 'left');
        
        // Apply filters
        if (!empty($filters['date_from'])) {
            $this->db->where('jh.date >=', $filters['date_from']);
        }
        
        if (!empty($filters['date_to'])) {
            $this->db->where('jh.date <=', $filters['date_to']);
        }
        
        if (!empty($filters['reference_no'])) {
            $this->db->like('jh.reference_no', $filters['reference_no']);
        }
        
        if (!empty($filters['source_module'])) {
            $this->db->where('jh.source_module', $filters['source_module']);
        }
        
        if (!empty($filters['status'])) {
            $this->db->where('jh.status', $filters['status']);
        }
        
        $this->db->order_by('jh.date', 'DESC');
        $this->db->order_by('jh.id', 'DESC');
        $this->db->limit($limit, $offset);
        
        $headers = $this->db->get()->result_array();
        
        // Get lines for each header
        foreach ($headers as &$header) {
            $header['lines'] = $this->getJournalLines($header['id']);
        }
        
        return $headers;
    }

    /**
     * Get journal lines for a header
     */
    public function getJournalLines($journal_id) {
        $this->db->select('
            jl.*,
            coa.account_code,
            coa.account_name,
            coa.account_type
        ');
        $this->db->from('journal_lines jl');
        $this->db->join('chart_of_accounts coa', 'coa.id = jl.account_id');
        $this->db->where('jl.journal_id', $journal_id);
        $this->db->order_by('jl.id');
        
        return $this->db->get()->result_array();
    }

    /**
     * Get journal entry by ID
     */
    public function getJournalEntry($id) {
        $this->db->select('
            jh.*,
            s.name as created_by_name,
            s.surname as created_by_surname
        ');
        $this->db->from('journal_headers jh');
        $this->db->join('staff s', 's.id = jh.created_by', 'left');
        $this->db->where('jh.id', $id);
        
        $header = $this->db->get()->row_array();
        
        if ($header) {
            $header['lines'] = $this->getJournalLines($id);
        }
        
        return $header;
    }

    /**
     * Update journal header
     */
    public function updateJournalHeader($id, $data) {
        $data['updated_at'] = date('Y-m-d H:i:s');
        $this->db->where('id', $id);
        return $this->db->update('journal_headers', $data);
    }

    /**
     * Delete journal entry (with lines)
     */
    public function deleteJournalEntry($id) {
        // Check if entry can be deleted (not posted or from billing)
        $header = $this->db->get_where('journal_headers', ['id' => $id])->row();
        
        if (!$header) {
            return false;
        }
        
        if ($header->status == 'posted' && $header->source_module) {
            return false; // Cannot delete posted entries from billing
        }
        
        $this->db->trans_start();
        
        // Delete lines first
        $this->db->where('journal_id', $id);
        $this->db->delete('journal_lines');
        
        // Delete header
        $this->db->where('id', $id);
        $this->db->delete('journal_headers');
        
        $this->db->trans_complete();
        
        return $this->db->trans_status();
    }

    /**
     * Reverse journal entry
     */
    public function reverseJournalEntry($id, $reversal_reason = '') {
        $original_entry = $this->getJournalEntry($id);
        
        if (!$original_entry || $original_entry['status'] == 'reversed') {
            return false;
        }
        
        $this->db->trans_start();
        
        // Create reversal header
        $reversal_data = [
            'reference_no' => 'REV-' . $original_entry['reference_no'],
            'date' => date('Y-m-d'),
            'description' => 'Reversal: ' . $original_entry['description'] . ($reversal_reason ? ' - ' . $reversal_reason : ''),
            'total_amount' => $original_entry['total_amount'],
            'source_module' => $original_entry['source_module'],
            'source_id' => $original_entry['source_id'],
            'patient_id' => $original_entry['patient_id'],
            'supplier_id' => $original_entry['supplier_id'],
            'created_by' => $this->session->userdata('staff_id') ?? $original_entry['created_by'],
            'reversed_entry_id' => $id,
            'status' => 'posted'
        ];
        
        $reversal_id = $this->addJournalHeader($reversal_data);
        
        if ($reversal_id) {
            // Create reversal lines (flip debits and credits)
            foreach ($original_entry['lines'] as $line) {
                $reversal_line = [
                    'journal_id' => $reversal_id,
                    'account_id' => $line['account_id'],
                    'debit_amount' => $line['credit_amount'], // Flip
                    'credit_amount' => $line['debit_amount'], // Flip
                    'description' => 'Reversal: ' . $line['description']
                ];
                
                $this->addJournalLine($reversal_line);
            }
            
            // Mark original as reversed
            $this->updateJournalHeader($id, [
                'status' => 'reversed',
                'reversed_by' => $this->session->userdata('staff_id'),
                'reversed_at' => date('Y-m-d H:i:s'),
                'reversal_reason' => $reversal_reason
            ]);
        }
        
        $this->db->trans_complete();
        
        return $this->db->trans_status() ? $reversal_id : false;
    }

    /**
     * Get journal entries for account
     */
    public function getAccountJournalEntries($account_id, $date_from = null, $date_to = null, $limit = 100) {
        $this->db->select('
            jh.date,
            jh.reference_no,
            jh.description as header_description,
            jl.description as line_description,
            jl.debit_amount,
            jl.credit_amount,
            jh.source_module,
            jh.status
        ');
        $this->db->from('journal_lines jl');
        $this->db->join('journal_headers jh', 'jh.id = jl.journal_id');
        $this->db->where('jl.account_id', $account_id);
        
        if ($date_from) {
            $this->db->where('jh.date >=', $date_from);
        }
        
        if ($date_to) {
            $this->db->where('jh.date <=', $date_to);
        }
        
        $this->db->order_by('jh.date', 'DESC');
        $this->db->order_by('jh.id', 'DESC');
        $this->db->limit($limit);
        
        return $this->db->get()->result_array();
    }

    /**
     * Validate journal entry balance
     */
    public function validateBalance($journal_lines) {
        $total_debits = 0;
        $total_credits = 0;
        
        foreach ($journal_lines as $line) {
            $total_debits += floatval($line['debit_amount'] ?? 0);
            $total_credits += floatval($line['credit_amount'] ?? 0);
        }
        
        return abs($total_debits - $total_credits) < 0.01; // Allow for rounding differences
    }

    /**
     * Create complete journal entry (header + lines)
     */
    public function createJournalEntry($header_data, $lines_data) {
        // Validate that lines balance
        if (!$this->validateBalance($lines_data)) {
            return false;
        }
        
        $this->db->trans_start();
        
        // Calculate total amount
        $total_amount = 0;
        foreach ($lines_data as $line) {
            $total_amount += floatval($line['debit_amount'] ?? 0);
        }
        $header_data['total_amount'] = $total_amount;
        
        // Create header
        $journal_id = $this->addJournalHeader($header_data);
        
        if ($journal_id) {
            // Create lines
            foreach ($lines_data as $line) {
                if (floatval($line['debit_amount'] ?? 0) > 0 || floatval($line['credit_amount'] ?? 0) > 0) {
                    $line['journal_id'] = $journal_id;
                    $this->addJournalLine($line);
                }
            }
        }
        
        $this->db->trans_complete();
        
        return $this->db->trans_status() ? $journal_id : false;
    }

    /**
     * Get journal entry summary statistics
     */
    public function getJournalStats($date_from = null, $date_to = null) {
        $this->db->select('
            COUNT(DISTINCT jh.id) as total_entries,
            COUNT(jl.id) as total_lines,
            SUM(jh.total_amount) as total_amount,
            SUM(CASE WHEN jh.source_module IS NOT NULL THEN 1 ELSE 0 END) as auto_entries,
            SUM(CASE WHEN jh.source_module IS NULL THEN 1 ELSE 0 END) as manual_entries
        ');
        $this->db->from('journal_headers jh');
        $this->db->join('journal_lines jl', 'jl.journal_id = jh.id');
        
        if ($date_from) {
            $this->db->where('jh.date >=', $date_from);
        }
        
        if ($date_to) {
            $this->db->where('jh.date <=', $date_to);
        }
        
        return $this->db->get()->row_array();
    }

    /**
     * Get entries by source module
     */
    public function getEntriesByModule($source_module, $date_from = null, $date_to = null) {
        $this->db->select('*');
        $this->db->from('journal_headers');
        $this->db->where('source_module', $source_module);
        
        if ($date_from) {
            $this->db->where('date >=', $date_from);
        }
        
        if ($date_to) {
            $this->db->where('date <=', $date_to);
        }
        
        $this->db->order_by('date', 'DESC');
        
        return $this->db->get()->result_array();
    }

    /**
     * Search journal entries
     */
    public function searchJournalEntries($search_term, $limit = 50) {
        $this->db->select('
            jh.*,
            s.name as created_by_name
        ');
        $this->db->from('journal_headers jh');
        $this->db->join('staff s', 's.id = jh.created_by', 'left');
        $this->db->group_start();
        $this->db->like('jh.reference_no', $search_term);
        $this->db->or_like('jh.description', $search_term);
        $this->db->group_end();
        $this->db->order_by('jh.date', 'DESC');
        $this->db->limit($limit);
        
        return $this->db->get()->result_array();
    }

    /**
     * Get recurring journal entries
     */
    public function getRecurringEntries() {
        $this->db->select('*');
        $this->db->from('recurring_journal_entries');
        $this->db->where('is_active', 1);
        $this->db->where('next_run_date <=', date('Y-m-d'));
        
        return $this->db->get()->result_array();
    }

    /**
     * Process recurring entries
     */
    public function processRecurringEntries() {
        $recurring_entries = $this->getRecurringEntries();
        $processed_count = 0;
        
        foreach ($recurring_entries as $recurring) {
            // Get the template lines
            $this->db->where('recurring_entry_id', $recurring['id']);
            $template_lines = $this->db->get('recurring_journal_lines')->result_array();
            
            if (!empty($template_lines)) {
                // Create header data
                $header_data = [
                    'reference_no' => $recurring['reference_prefix'] . '-' . date('Ymd'),
                    'date' => date('Y-m-d'),
                    'description' => $recurring['description'] . ' - ' . date('M Y'),
                    'source_module' => 'recurring',
                    'created_by' => 1 // System user
                ];
                
                // Prepare lines data
                $lines_data = [];
                foreach ($template_lines as $template_line) {
                    $lines_data[] = [
                        'account_id' => $template_line['account_id'],
                        'debit_amount' => $template_line['debit_amount'],
                        'credit_amount' => $template_line['credit_amount'],
                        'description' => $template_line['description']
                    ];
                }
                
                // Create the journal entry
                if ($this->createJournalEntry($header_data, $lines_data)) {
                    $processed_count++;
                    
                    // Update next run date
                    $next_date = $this->calculateNextRunDate($recurring['frequency'], $recurring['next_run_date']);
                    $this->db->where('id', $recurring['id']);
                    $this->db->update('recurring_journal_entries', [
                        'last_run_date' => date('Y-m-d'),
                        'next_run_date' => $next_date,
                        'run_count' => $recurring['run_count'] + 1
                    ]);
                }
            }
        }
        
        return $processed_count;
    }

    /**
     * Calculate next run date for recurring entries
     */
    private function calculateNextRunDate($frequency, $current_date) {
        $date = new DateTime($current_date);
        
        switch ($frequency) {
            case 'daily':
                $date->add(new DateInterval('P1D'));
                break;
            case 'weekly':
                $date->add(new DateInterval('P1W'));
                break;
            case 'monthly':
                $date->add(new DateInterval('P1M'));
                break;
            case 'quarterly':
                $date->add(new DateInterval('P3M'));
                break;
            case 'yearly':
                $date->add(new DateInterval('P1Y'));
                break;
            default:
                $date->add(new DateInterval('P1M'));
        }
        
        return $date->format('Y-m-d');
    }

    /**
     * Get journal entries for export
     */
    public function getEntriesForExport($date_from, $date_to, $format = 'csv') {
        $this->db->select('
            jh.date,
            jh.reference_no,
            jh.description as header_description,
            jl.description as line_description,
            coa.account_code,
            coa.account_name,
            jl.debit_amount,
            jl.credit_amount,
            jh.source_module,
            jh.status
        ');
        $this->db->from('journal_lines jl');
        $this->db->join('journal_headers jh', 'jh.id = jl.journal_id');
        $this->db->join('chart_of_accounts coa', 'coa.id = jl.account_id');
        $this->db->where('jh.date >=', $date_from);
        $this->db->where('jh.date <=', $date_to);
        $this->db->order_by('jh.date');
        $this->db->order_by('jh.id');
        $this->db->order_by('jl.id');
        
        return $this->db->get()->result_array();
    }

    /**
     * Get month-end closing entries
     */
    public function getClosingEntries($year, $month) {
        $date_from = sprintf('%04d-%02d-01', $year, $month);
        $date_to = date('Y-m-t', strtotime($date_from));
        
        $this->db->select('*');
        $this->db->from('journal_headers');
        $this->db->where('source_module', 'month_end_closing');
        $this->db->where('date >=', $date_from);
        $this->db->where('date <=', $date_to);
        
        return $this->db->get()->result_array();
    }

    /**
     * Create month-end closing entries
     */
    public function createMonthEndClosing($year, $month) {
        $date_from = sprintf('%04d-%02d-01', $year, $month);
        $date_to = date('Y-m-t', strtotime($date_from));
        $closing_date = $date_to;
        
        // Check if closing already exists
        if ($this->getClosingEntries($year, $month)) {
            return false; // Already closed
        }
        
        $this->db->trans_start();
        
        // Get revenue and expense account balances
        $this->db->select('
            coa.id as account_id,
            coa.account_code,
            coa.account_name,
            coa.account_type,
            SUM(jl.credit_amount - jl.debit_amount) as balance
        ');
        $this->db->from('chart_of_accounts coa');
        $this->db->join('journal_lines jl', 'jl.account_id = coa.id');
        $this->db->join('journal_headers jh', 'jh.id = jl.journal_id');
        $this->db->where_in('coa.account_type', ['revenue', 'expense']);
        $this->db->where('jh.date >=', $date_from);
        $this->db->where('jh.date <=', $date_to);
        $this->db->group_by('coa.id, coa.account_code, coa.account_name, coa.account_type');
        $this->db->having('SUM(jl.credit_amount - jl.debit_amount) != 0');
        
        $accounts = $this->db->get()->result_array();
        
        if (!empty($accounts)) {
            // Create closing entry header
            $header_data = [
                'reference_no' => 'CLOSE-' . $year . '-' . sprintf('%02d', $month),
                'date' => $closing_date,
                'description' => 'Month-end closing for ' . date('F Y', strtotime($date_from)),
                'source_module' => 'month_end_closing',
                'created_by' => $this->session->userdata('staff_id') ?? 1
            ];
            
            $lines_data = [];
            $net_income = 0;
            
            // Close revenue and expense accounts
            foreach ($accounts as $account) {
                if ($account['account_type'] == 'revenue' && $account['balance'] > 0) {
                    // Debit revenue accounts to close them
                    $lines_data[] = [
                        'account_id' => $account['account_id'],
                        'debit_amount' => $account['balance'],
                        'credit_amount' => 0,
                        'description' => 'Closing ' . $account['account_name']
                    ];
                    $net_income += $account['balance'];
                    
                } elseif ($account['account_type'] == 'expense' && $account['balance'] > 0) {
                    // Credit expense accounts to close them
                    $lines_data[] = [
                        'account_id' => $account['account_id'],
                        'debit_amount' => 0,
                        'credit_amount' => $account['balance'],
                        'description' => 'Closing ' . $account['account_name']
                    ];
                    $net_income -= $account['balance'];
                }
            }
            
            // Transfer net income to retained earnings
            $retained_earnings_account = $this->getRetainedEarningsAccount();
            if ($retained_earnings_account && $net_income != 0) {
                if ($net_income > 0) {
                    // Profit - credit retained earnings
                    $lines_data[] = [
                        'account_id' => $retained_earnings_account,
                        'debit_amount' => 0,
                        'credit_amount' => $net_income,
                        'description' => 'Net income transfer'
                    ];
                } else {
                    // Loss - debit retained earnings
                    $lines_data[] = [
                        'account_id' => $retained_earnings_account,
                        'debit_amount' => abs($net_income),
                        'credit_amount' => 0,
                        'description' => 'Net loss transfer'
                    ];
                }
            }
            
            // Create the closing entry
            $journal_id = $this->createJournalEntry($header_data, $lines_data);
        }
        
        $this->db->trans_complete();
        
        return $this->db->trans_status();
    }

    /**
     * Get retained earnings account
     */
    private function getRetainedEarningsAccount() {
        $this->db->select('id');
        $this->db->where('account_code', '3100'); // Retained Earnings
        $this->db->where('is_active', 1);
        $query = $this->db->get('chart_of_accounts');
        
        if ($query->num_rows() > 0) {
            return $query->row()->id;
        }
        
        return null;
    }

    /**
     * Get journal entry audit trail
     */
    public function getAuditTrail($journal_id) {
        $this->db->select('
            jat.*,
            s.name as user_name,
            s.surname as user_surname
        ');
        $this->db->from('journal_audit_trail jat');
        $this->db->join('staff s', 's.id = jat.user_id', 'left');
        $this->db->where('jat.journal_id', $journal_id);
        $this->db->order_by('jat.created_at');
        
        return $this->db->get()->result_array();
    }

    /**
     * Log journal entry changes
     */
    public function logJournalChange($journal_id, $action, $old_data = null, $new_data = null) {
        $audit_data = [
            'journal_id' => $journal_id,
            'action' => $action,
            'old_data' => $old_data ? json_encode($old_data) : null,
            'new_data' => $new_data ? json_encode($new_data) : null,
            'user_id' => $this->session->userdata('staff_id') ?? 1,
            'ip_address' => $this->input->ip_address(),
            'user_agent' => $this->input->user_agent(),
            'created_at' => date('Y-m-d H:i:s')
        ];
        
        return $this->db->insert('journal_audit_trail', $audit_data);
    }

    /**
     * Get accounts with highest activity
     */
    public function getMostActiveAccounts($limit = 10, $date_from = null, $date_to = null) {
        $this->db->select('
            coa.account_code,
            coa.account_name,
            COUNT(jl.id) as transaction_count,
            SUM(jl.debit_amount + jl.credit_amount) as total_activity
        ');
        $this->db->from('journal_lines jl');
        $this->db->join('journal_headers jh', 'jh.id = jl.journal_id');
        $this->db->join('chart_of_accounts coa', 'coa.id = jl.account_id');
        
        if ($date_from) {
            $this->db->where('jh.date >=', $date_from);
        }
        
        if ($date_to) {
            $this->db->where('jh.date <=', $date_to);
        }
        
        $this->db->group_by('coa.id, coa.account_code, coa.account_name');
        $this->db->order_by('transaction_count', 'DESC');
        $this->db->limit($limit);
        
        return $this->db->get()->result_array();
    }

    /**
     * Get journal entries requiring approval
     */
    public function getEntriesRequiringApproval() {
        $this->db->select('
            jh.*,
            s.name as created_by_name
        ');
        $this->db->from('journal_headers jh');
        $this->db->join('staff s', 's.id = jh.created_by', 'left');
        $this->db->where('jh.status', 'pending');
        $this->db->where('jh.total_amount >', $this->getApprovalThreshold());
        $this->db->order_by('jh.created_at');
        
        return $this->db->get()->result_array();
    }

    /**
     * Get approval threshold from settings
     */
    private function getApprovalThreshold() {
        // This would typically come from a settings table
        return 10000; // Default threshold
    }

    /**
     * Approve journal entry
     */
    public function approveJournalEntry($journal_id, $approver_id) {
        $update_data = [
            'status' => 'approved',
            'approved_by' => $approver_id,
            'approved_at' => date('Y-m-d H:i:s')
        ];
        
        $this->db->where('id', $journal_id);
        $result = $this->db->update('journal_headers', $update_data);
        
        if ($result) {
            $this->logJournalChange($journal_id, 'approved', null, $update_data);
        }
        
        return $result;
    }

    /**
     * Reject journal entry
     */
    public function rejectJournalEntry($journal_id, $approver_id, $rejection_reason) {
        $update_data = [
            'status' => 'rejected',
            'approved_by' => $approver_id,
            'approved_at' => date('Y-m-d H:i:s'),
            'rejection_reason' => $rejection_reason
        ];
        
        $this->db->where('id', $journal_id);
        $result = $this->db->update('journal_headers', $update_data);
        
        if ($result) {
            $this->logJournalChange($journal_id, 'rejected', null, $update_data);
        }
        
        return $result;
    }
}
?>