<?php
defined('BASEPATH') OR exit('No direct script access allowed');

/**
 * Import Model
 * Handles database operations for import functionality
 * 
 * @author Your Name
 * @version 1.0
 */
class Import_model extends CI_Model {

    public function __construct() {
        parent::__construct();
        $this->load->database();
    }

    /**
     * Log import activity
     */
public function log_import_activity($data) {
    // Create import_logs table if it doesn't exist
    $this->create_import_logs_table();
    
    $log_data = array(
        'import_type' => $data['import_type'],
        'filename' => $data['filename'],
        'records_processed' => isset($data['records_processed']) ? $data['records_processed'] : 0,
        'records_successful' => isset($data['records_successful']) ? $data['records_successful'] : 0,
        'records_failed' => isset($data['records_failed']) ? $data['records_failed'] : 0,
        'status' => $data['status'],
        'error_message' => isset($data['error_message']) ? $data['error_message'] : null,
        'batch_id' => isset($data['batch_id']) ? $data['batch_id'] : null,
        'user_id' => isset($data['user_id']) ? $data['user_id'] : null,
        'created_at' => date('Y-m-d H:i:s')
    );
    
    return $this->db->insert('import_logs', $log_data);
}

    /**
     * Get recent imports
     */
public function get_recent_imports($limit = 10) {
    // Make sure table exists
    $this->create_import_logs_table();
    
    $this->db->select('il.*, s.name as user_name');
    $this->db->from('import_logs il');
    $this->db->join('staff s', 'il.user_id = s.id', 'left');
    $this->db->order_by('il.created_at', 'DESC');
    $this->db->limit($limit);
    
    $query = $this->db->get();
    return $query ? $query->result() : array();
}


    /**
     * Get import history with pagination
     */
    public function get_import_history($limit = 50, $offset = 0) {
        $this->db->select('il.*, s.name as user_name');
        $this->db->from('import_logs il');
        $this->db->join('staff s', 'il.user_id = s.id', 'left');
        $this->db->order_by('il.created_at', 'DESC');
        $this->db->limit($limit, $offset);
        
        return $this->db->get()->result();
    }

    /**
     * Get import by ID
     */
public function get_import_by_id($import_id) {
    // Make sure table exists
    $this->create_import_logs_table();
    
    $this->db->select('il.*, s.name as user_name');
    $this->db->from('import_logs il');
    $this->db->join('staff s', 'il.user_id = s.id', 'left');
    $this->db->where('il.id', $import_id);
    
    $query = $this->db->get();
    return $query ? $query->row() : false;
}

    /**
     * Get import statistics
     */
public function get_import_statistics() {
    // Make sure table exists
    $this->create_import_logs_table();
    
    $stats = array();
    
    // Total imports
    $this->db->from('import_logs');
    $stats['total_imports'] = $this->db->count_all_results();
    
    // Successful imports
    $this->db->from('import_logs');
    $this->db->where('status', 'completed');
    $stats['successful_imports'] = $this->db->count_all_results();
    
    // Failed imports
    $this->db->from('import_logs');
    $this->db->where('status', 'failed');
    $stats['failed_imports'] = $this->db->count_all_results();
    
    // Total records processed
    $this->db->select_sum('records_processed');
    $this->db->from('import_logs');
    $this->db->where('status', 'completed');
    $query = $this->db->get();
    $result = $query->row();
    $stats['total_records_processed'] = $result ? $result->records_processed : 0;
    
    // Calculate success rate
    $stats['success_rate'] = $stats['total_imports'] > 0 ? 
        round(($stats['successful_imports'] / $stats['total_imports']) * 100, 2) : 0;
    
    return $stats;
}

/**
 * Clean up old import logs (optional maintenance method)
 */
public function cleanup_old_logs($days = 90) {
    $this->create_import_logs_table();
    
    $cutoff_date = date('Y-m-d H:i:s', strtotime("-{$days} days"));
    $this->db->where('created_at <', $cutoff_date);
    $this->db->where('status !=', 'processing'); // Don't delete processing logs
    
    return $this->db->delete('import_logs');
}

    /**
     * Check if consumable exists
     */
    public function consumable_exists($name, $category_id) {
        $this->db->where('consumable_name', $name);
        $this->db->where('category_id', $category_id);
        
        return $this->db->get('consumables')->num_rows() > 0;
    }

    /**
     * Get consumable by name and category
     */
    public function get_consumable($name, $category_id) {
        $this->db->where('consumable_name', $name);
        $this->db->where('category_id', $category_id);
        
        return $this->db->get('consumables')->row();
    }

    /**
     * Check if pharmacy stock item exists
     */
    public function pharmacy_item_exists($name, $group_name = 'Pharmacy') {
        $this->db->where('item_name', $name);
        $this->db->where('group_name', $group_name);
        
        return $this->db->get('pharmacy_stock_items')->num_rows() > 0;
    }

    /**
     * Get pharmacy stock item
     */
    public function get_pharmacy_item($name, $group_name = 'Pharmacy') {
        $this->db->where('item_name', $name);
        $this->db->where('group_name', $group_name);
        
        return $this->db->get('pharmacy_stock_items')->row();
    }

    /**
     * Backup data before import
     */
    public function backup_table_data($table_name, $backup_id) {
        // Create backup tables if they don't exist
        $this->create_backup_tables();
        
        $backup_data = array(
            'backup_id' => $backup_id,
            'backup_table' => $table_name,
            'records_backed_up' => 0,
            'created_at' => date('Y-m-d H:i:s'),
            'created_by' => $this->session->userdata('admin')['name'] ?? 'System'
        );
        
        // Count records to backup
        $count = $this->db->count_all_results($table_name);
        $backup_data['records_backed_up'] = $count;
        
        // Create backup table with timestamp
        $backup_table_name = $table_name . '_backup_' . date('Y_m_d_H_i_s');
        
        // Create backup table structure
        $this->db->query("CREATE TABLE `{$backup_table_name}` LIKE `{$table_name}`");
        
        // Copy data to backup table
        $this->db->query("INSERT INTO `{$backup_table_name}` SELECT * FROM `{$table_name}`");
        
        // Log backup
        if ($table_name == 'pharmacy_stock_items') {
            $this->db->insert('pharmacy_stock_backups', $backup_data);
        } else {
            // For consumables and other tables, we can create a generic backup log
            $this->log_backup_activity($backup_data);
        }
        
        return $backup_table_name;
    }

    /**
     * Log backup activity
     */
    private function log_backup_activity($data) {
        // Create generic backup logs table if needed
        $this->create_backup_logs_table();
        
        return $this->db->insert('backup_logs', $data);
    }

    /**
     * Get item categories
     */
    public function get_item_categories() {
        $this->db->where('is_active', 'yes');
        $this->db->order_by('category_name');
        
        return $this->db->get('item_categories')->result();
    }

    /**
     * Get consumable categories
     */
    public function get_consumable_categories() {
        // Assuming consumables use item_categories table
        return $this->get_item_categories();
    }

    /**
     * Validate import data structure
     */
    public function validate_import_data($data, $required_fields) {
        $errors = array();
        
        foreach ($data as $index => $row) {
            $row_errors = array();
            
            foreach ($required_fields as $field) {
                if (!isset($row[$field]) || empty(trim($row[$field]))) {
                    $row_errors[] = "Missing required field: $field";
                }
            }
            
            if (!empty($row_errors)) {
                $errors["Row " . ($index + 1)] = $row_errors;
            }
        }
        
        return $errors;
    }

    /**
     * Create import logs table if it doesn't exist
     */
public function create_import_logs_table() {
    $sql = "CREATE TABLE IF NOT EXISTS `import_logs` (
        `id` bigint NOT NULL AUTO_INCREMENT,
        `import_type` varchar(50) NOT NULL,
        `filename` varchar(255) NOT NULL,
        `records_processed` int NOT NULL DEFAULT '0',
        `records_successful` int NOT NULL DEFAULT '0',
        `records_failed` int NOT NULL DEFAULT '0',
        `status` enum('pending','processing','completed','failed') NOT NULL DEFAULT 'pending',
        `error_message` text,
        `batch_id` varchar(100) DEFAULT NULL,
        `user_id` bigint DEFAULT NULL,
        `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
        `updated_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
        PRIMARY KEY (`id`),
        KEY `idx_import_type` (`import_type`),
        KEY `idx_status` (`status`),
        KEY `idx_user_id` (`user_id`),
        KEY `idx_batch_id` (`batch_id`),
        KEY `idx_created_at` (`created_at`)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
    
    return $this->db->query($sql);
}


    /**
     * Create backup tables if they don't exist
     */
    private function create_backup_tables() {
        // Create pharmacy stock backups table if it doesn't exist
        if (!$this->db->table_exists('pharmacy_stock_backups')) {
            $this->db->query("
                CREATE TABLE `pharmacy_stock_backups` (
                    `id` bigint NOT NULL AUTO_INCREMENT,
                    `backup_id` varchar(100) NOT NULL,
                    `backup_table` varchar(100) NOT NULL,
                    `records_backed_up` int NOT NULL DEFAULT 0,
                    `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
                    `created_by` varchar(255) DEFAULT NULL,
                    PRIMARY KEY (`id`),
                    KEY `idx_backup_id` (`backup_id`),
                    KEY `idx_created_at` (`created_at`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
            ");
        }
    }

    /**
     * Create backup logs table if it doesn't exist
     */
    private function create_backup_logs_table() {
        if (!$this->db->table_exists('backup_logs')) {
            $sql = "
                CREATE TABLE `backup_logs` (
                    `id` int NOT NULL AUTO_INCREMENT,
                    `backup_id` varchar(100) NOT NULL,
                    `backup_table` varchar(100) NOT NULL,
                    `records_backed_up` int NOT NULL DEFAULT 0,
                    `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
                    `created_by` varchar(255) DEFAULT NULL,
                    PRIMARY KEY (`id`),
                    KEY `idx_backup_id` (`backup_id`),
                    KEY `idx_created_at` (`created_at`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
            ";
            
            $this->db->query($sql);
        }
    }

    /**
     * Clean old import logs (older than specified days)
     */
    public function clean_old_logs($days = 90) {
        $cutoff_date = date('Y-m-d', strtotime("-$days days"));
        
        $this->db->where('DATE(created_at) <', $cutoff_date);
        return $this->db->delete('import_logs');
    }

    /**
     * Get duplicate items in import data
     */
    public function find_duplicates($table, $field, $values) {
        $this->db->select($field);
        $this->db->where_in($field, $values);
        $results = $this->db->get($table)->result_array();
        
        return array_column($results, $field);
    }

    /**
     * Bulk insert with duplicate handling
     */
    public function bulk_insert_with_duplicates($table, $data, $update_on_duplicate = false) {
        if (empty($data)) {
            return array('inserted' => 0, 'updated' => 0, 'errors' => array());
        }
        
        $inserted = 0;
        $updated = 0;
        $errors = array();
        
        foreach ($data as $index => $row) {
            try {
                if ($update_on_duplicate) {
                    // Try to update first, then insert if no rows affected
                    $this->db->replace($table, $row);
                    $updated++;
                } else {
                    $this->db->insert($table, $row);
                    $inserted++;
                }
            } catch (Exception $e) {
                $errors[] = "Row $index: " . $e->getMessage();
            }
        }
        
        return array(
            'inserted' => $inserted,
            'updated' => $updated,
            'errors' => $errors
        );
    }

    /**
     * Update stock movements after import
     */
    public function update_stock_movements($item_id, $movement_data) {
        // This would integrate with your existing stock movement tracking
        // Implement based on your stock movement requirements
        
        $movement_record = array(
            'item_id' => $item_id,
            'movement_type' => 'import_adjustment',
            'quantity_before' => $movement_data['old_stock'],
            'quantity_after' => $movement_data['new_stock'],
            'reference_type' => 'import',
            'reference_id' => $movement_data['import_id'],
            'created_at' => date('Y-m-d H:i:s'),
            'created_by' => $this->session->userdata('admin')['id'] ?? null
        );
        
        // Insert into appropriate stock movement table based on item type
        if (isset($movement_data['table_type']) && $movement_data['table_type'] == 'pharmacy') {
            return $this->db->insert('pharmacy_stock_movements', $movement_record);
        } else {
            // For consumables or other stock movements
            return $this->db->insert('stock_movements', $movement_record);
        }
    }

    /**
     * Get filtered imports with pagination
     */
public function get_filtered_imports($filters = array(), $limit = 25, $offset = 0) {
    // Make sure table exists
    $this->create_import_logs_table();
    
    $this->db->select('il.*, s.name as user_name');
    $this->db->from('import_logs il');
    $this->db->join('staff s', 'il.user_id = s.id', 'left');
    
    // Apply filters
    if (!empty($filters['type'])) {
        $this->db->where('il.import_type', $filters['type']);
    }
    
    if (!empty($filters['status'])) {
        $this->db->where('il.status', $filters['status']);
    }
    
    if (!empty($filters['date_from'])) {
        $this->db->where('DATE(il.created_at) >=', $filters['date_from']);
    }
    
    if (!empty($filters['date_to'])) {
        $this->db->where('DATE(il.created_at) <=', $filters['date_to']);
    }
    
    $this->db->order_by('il.created_at', 'DESC');
    $this->db->limit($limit, $offset);
    
    $query = $this->db->get();
    return $query ? $query->result() : array();
}
    /**
     * Get count of filtered imports
     */
public function get_filtered_imports_count($filters = array()) {
    // Make sure table exists
    $this->create_import_logs_table();
    
    $this->db->from('import_logs il');
    
    // Apply same filters as get_filtered_imports
    if (!empty($filters['type'])) {
        $this->db->where('il.import_type', $filters['type']);
    }
    
    if (!empty($filters['status'])) {
        $this->db->where('il.status', $filters['status']);
    }
    
    if (!empty($filters['date_from'])) {
        $this->db->where('DATE(il.created_at) >=', $filters['date_from']);
    }
    
    if (!empty($filters['date_to'])) {
        $this->db->where('DATE(il.created_at) <=', $filters['date_to']);
    }
    
    return $this->db->count_all_results();
}

}