// =============================================
    // ON-DEMAND PAYROLL PROCESSING
    // =============================================

    public function processOnDemandPayroll($params) {
        $this->db->trans_start();
        
        try {
            $month = $params['month'];
            $year = $params['year'];
            $department_id = $params['department_id'] ?? null;
            $process_date = $params['process_date'];
            $payroll_type = $params['payroll_type'] ?? 'full_month';
            $overwrite = $params['overwrite'] ?? false;
            $processed_by = $params['processed_by'] ?? null;

            // If overwrite is true, delete existing payroll
            if ($overwrite) {
                $this->deleteExistingPayroll($month, $year, $department_id);
            }

            // Get active staff
            $this->db->select('s.*, d.name as department_name');
            $this->db->from('staff s');
            $this->db->join('departments d', 'd.id = s.department_id', 'left');
            $this->db->where('s.is_active', 'yes');
            
            if ($department_id) {
                $this->db->where('s.department_id', $department_id);
            }
            
            $staff_list = $this->db->get()->result();
            
            $processing_results = [
                'total_staff' => count($staff_list),
                'processed_successfully' => 0,
                'processing_errors' => [],
                'total_gross_salary' => 0,
                'total_net_salary' => 0,
                'total_deductions' => 0,
                'processing_date' => $process_date,
                'processed_by' => $processed_by
            ];

            foreach ($staff_list as $staff) {
                try {
                    $payroll_data = $this->calculateStaffPayrollOnDemand($staff->id, $month, $year, $process_date, $payroll_type);
                    
                    if ($payroll_data) {
                        $payroll_data['processed_by'] = $processed_by;
                        $payroll_data['processing_date'] = $process_date;
                        
                        $this->db->insert('staff_payroll', $payroll_data);
                        $processing_results['processed_successfully']++;
                        $processing_results['total_gross_salary'] += $payroll_data['gross_salary'];
                        $processing_results['total_net_salary'] += $payroll_data['net_salary'];
                        $processing_results['total_deductions'] += $payroll_data['total_deductions'];
                    }
                } catch (Exception $e) {
                    $processing_results['processing_errors'][] = [
                        'staff_id' => $staff->id,
                        'staff_name' => $staff->name . ' ' . $staff->surname,
                        'error' => $e->getMessage()
                    ];
                }
            }

            // Log the processing activity
            $this->logPayrollProcessing([
                'month' => $month,
                'year' => $year,
                'department_id' => $department_id,
                'process_date' => $process_date,
                'payroll_type' => $payroll_type,
                'results' => $processing_results,
                'processed_by' => $processed_by
            ]);

            $this->db->trans_complete();

            if ($this->db->trans_status() === FALSE) {
                return [
                    'status' => 'error',
                    'message' => 'Database transaction failed'
                ];
            }

            return [
                'status' => 'success',
                'message' => 'Payroll processed successfully on ' . date('jS F Y', strtotime($process_date)),
                'results' => $processing_results
            ];

        } catch (Exception $e) {
            $this->db->trans_rollback();
            return [
                'status' => 'error',
                'message' => 'Processing failed: ' . $e->getMessage()
            ];
        }
    }

    public function calculateStaffPayrollOnDemand($staff_id, $month, $year, $process_date, $payroll_type = 'full_month') {
        // Get staff salary structure
        $salary_components = $this->getStaffSalaryComponents($staff_id);
        
        if (empty($salary_components)) {
            throw new Exception('No salary structure found for staff ID: ' . $staff_id);
        }

        // Calculate date range based on payroll type
        $date_range = $this->calculateDateRange($month, $year, $process_date, $payroll_type);
        
        // Get attendance data for the calculated period
        $attendance_data = $this->getAttendanceDataForPeriod($staff_id, $date_range);
        
        // Get leave data for the period
        $leave_data = $this->getLeaveDataForPeriod($staff_id, $date_range);
        
        // Calculate working days based on payroll type
        $working_days = $this->calculateWorkingDays($date_range, $payroll_type);
        
        // Initialize payroll data
        $payroll_data = [
            'staff_id' => $staff_id,
            'month' => $month,
            'year' => $year,
            'working_days' => $working_days['total'],
            'worked_days' => $attendance_data['worked_days'],
            'leave_days' => $leave_data['paid_leave_days'],
            'absent_days' => max(0, $working_days['total'] - $attendance_data['worked_days'] - $leave_data['paid_leave_days']),
            'overtime_hours' => $attendance_data['overtime_hours'],
            'total_earnings' => 0,
            'total_deductions' => 0,
            'gross_salary' => 0,
            'net_salary' => 0,
            'payroll_type' => $payroll_type,
            'date_from' => $date_range['start_date'],
            'date_to' => $date_range['end_date'],
            'created_at' => date('Y-m-d H:i:s')
        ];

        $earnings = [];
        $deductions = [];

        // Calculate each salary component
        foreach ($salary_components as $component) {
            $amount = $this->calculateComponentAmountOnDemand($component, $payroll_data, $attendance_data, $working_days);
            
            if ($component->type == 'earning') {
                $earnings[$component->short_code] = $amount;
                $payroll_data['total_earnings'] += $amount;
            } else {
                $deductions[$component->short_code] = $amount;
                $payroll_data['total_deductions'] += $amount;
            }
        }

        $payroll_data['gross_salary'] = $payroll_data['total_earnings'];
        $payroll_data['net_salary'] = $payroll_data['gross_salary'] - $payroll_data['total_deductions'];
        $payroll_data['earnings_json'] = json_encode($earnings);
        $payroll_data['deductions_json'] = json_encode($deductions);

        return $payroll_data;
    }

    private function calculateDateRange($month, $year, $process_date, $payroll_type) {
        $start_date = $year . '-' . str_pad($month, 2, '0', STR_PAD_LEFT) . '-01';
        
        switch ($payroll_type) {
            case 'up_to_date':
                // From start of month to processing date
                $end_date = min($process_date, date('Y-m-t', strtotime($start_date)));
                break;
            case 'partial':
                // From start of month to processing date
                $end_date = $process_date;
                break;
            case 'full_month':
            default:
                // Full month regardless of processing date
                $end_date = date('Y-m-t', strtotime($start_date));
                break;
        }
        
        return [
            'start_date' => $start_date,
            'end_date' => $end_date,
            'process_date' => $process_date
        ];
    }

    private function calculateWorkingDays($date_range, $payroll_type) {
        $start = new DateTime($date_range['start_date']);
        $end = new DateTime($date_range['end_date']);
        $end->modify('+1 day');
        
        $interval = new DateInterval('P1D');
        $period = new DatePeriod($start, $interval, $end);
        
        $working_days = 0;
        $total_days = 0;
        
        foreach ($period as $date) {
            $total_days++;
            // Skip weekends (Saturday = 6, Sunday = 0)
            if ($date->format('w') != 0 && $date->format('w') != 6) {
                // Check if it's not a holiday
                if (!$this->isHoliday($date->format('Y-m-d'))) {
                    $working_days++;
                }
            }
        }
        
        return [
            'total' => $working_days,
            'calendar_days' => $total_days,
            'payroll_type' => $payroll_type
        ];
    }

    private function getAttendanceDataForPeriod($staff_id, $date_range) {
        $this->db->select('
            COUNT(*) as total_attendance_records,
            SUM(CASE WHEN clock_in_time IS NOT NULL THEN 1 ELSE 0 END) as worked_days,
            SUM(COALESCE(total_hours, 0)) as total_hours,
            SUM(COALESCE(overtime_hours, 0)) as overtime_hours
        ');
        $this->db->from('staff_attendance');
        $this->db->where('staff_id', $staff_id);
        $this->db->where('date >=', $date_range['start_date']);
        $this->db->where('date <=', $date_range['end_date']);
        
        $result = $this->db->get()->row();
        
        return [
            'worked_days' => $result ? $result->worked_days : 0,
            'total_hours' => $result ? $result->total_hours : 0,
            'overtime_hours' => $result ? $result->overtime_hours : 0
        ];
    }

    private function getLeaveDataForPeriod($staff_id, $date_range) {
        $this->db->select('
            COALESCE(SUM(
                CASE 
                    WHEN start_date <= "' . $date_range['start_date'] . '" AND end_date >= "' . $date_range['end_date'] . '" 
                    THEN DATEDIFF("' . $date_range['end_date'] . '", "' . $date_range['start_date'] . '") + 1
                    WHEN start_date <= "' . $date_range['start_date'] . '" AND end_date < "' . $date_range['end_date'] . '" AND end_date >= "' . $date_range['start_date'] . '"
                    THEN DATEDIFF(end_date, "' . $date_range['start_date'] . '") + 1
                    WHEN start_date > "' . $date_range['start_date'] . '" AND end_date >= "' . $date_range['end_date'] . '"
                    THEN DATEDIFF("' . $date_range['end_date'] . '", start_date) + 1
                    WHEN start_date >= "' . $date_range['start_date'] . '" AND end_date <= "' . $date_range['end_date'] . '"
                    THEN total_days
                    ELSE 0
                END
            ), 0) as paid_leave_days
        ');
        $this->db->from('staff_leave sl');
        $this->db->join('leave_types lt', 'lt.id = sl.leave_type_id', 'left');
        $this->db->where('sl.staff_id', $staff_id);
        $this->db->where('sl.status', 'approved');
        $this->db->where('lt.is_paid', 'yes');
        $this->db->group_start();
        $this->db->where('sl.start_date <=', $date_range['end_date']);
        $this->db->where('sl.end_date >=', $date_range['start_date']);
        $this->db->group_end();
        
        $result = $this->db->get()->row();
        
        return [
            'paid_leave_days' => $result ? $result->paid_leave_days : 0
        ];
    }

    private function calculateComponentAmountOnDemand($component, $payroll_data, $attendance_data, $working_days) {
        $amount = 0;

        switch ($component->calculation_type) {
            case 'fixed':
                if ($component->short_code == 'OVERTIME') {
                    // Calculate overtime amount
                    $hourly_rate = $component->amount;
                    $amount = $attendance_data['overtime_hours'] * $hourly_rate;
                } else {
                    // Pro-rate based on payroll type and worked days
                    if ($payroll_data['payroll_type'] === 'up_to_date' || $payroll_data['payroll_type'] === 'partial') {
                        // Pro-rate based on actual period
                        $days_in_period = $working_days['total'];
                        $days_in_full_month = $this->getWorkingDaysInMonth($payroll_data['month'], $payroll_data['year']);
                        $proration_factor = $days_in_period / $days_in_full_month;
                        $amount = $component->amount * $proration_factor;
                    } else {
                        // Full month amount, but pro-rate based on attendance
                        $effective_days = $payroll_data['worked_days'] + $payroll_data['leave_days'];
                        $amount = ($component->amount / $payroll_data['working_days']) * $effective_days;
                    }
                }
                break;

            case 'percentage':
                $base_amount = $this->getBaseSalaryForCalculation($payroll_data['staff_id']);
                if ($payroll_data['payroll_type'] !== 'full_month') {
                    // Pro-rate the base amount
                    $days_in_full_month = $this->getWorkingDaysInMonth($payroll_data['month'], $payroll_data['year']);
                    $base_amount = ($base_amount / $days_in_full_month) * $working_days['total'];
                }
                $amount = ($base_amount * $component->percentage) / 100;
                break;

            case 'formula':
                $amount = $this->evaluateFormula($component->formula, $payroll_data, $attendance_data);
                break;
        }

        return round($amount, 2);
    }

    public function checkExistingPayroll($month, $year, $department_id = null) {
        $this->db->select('sp.*, s.name, s.surname, s.employee_id');
        $this->db->from('staff_payroll sp');
        $this->db->join('staff s', 's.id = sp.staff_id', 'left');
        $this->db->where('sp.month', $month);
        $this->db->where('sp.year', $year);
        
        if ($department_id) {
            $this->db->where('s.department_id', $department_id);
        }
        
        return $this->db->get()->result();
    }

    public function deleteExistingPayroll($month, $year, $department_id = null) {
        if ($department_id) {
            $this->db->select('sp.id');
            $this->db->from('staff_payroll sp');
            $this->db->join('staff s', 's.id = sp.staff_id', 'left');
            $this->db->where('sp.month', $month);
            $this->db->where('sp.year', $year);
            $this->db->where('s.department_id', $department_id);
            $payroll_ids = $this->db->get()->result();
            
            foreach ($payroll_ids as $payroll) {
                $this->db->where('id', $payroll->id);
                $this->db->delete('staff_payroll');
            }
        } else {
            $this->db->where('month', $month);
            $this->db->where('year', $year);
            $this->db->delete('staff_payroll');
        }
        
        return true;
    }

    public function generatePayrollPreview($params) {
        $month = $params['month'];
        $year = $params['year'];
        $department_id = $params['department_id'] ?? null;
        $process_date = $params['process_date'];
        
        // Get staff list
        $this->db->select('s.*, d.name as department_name');
        $this->db->from('staff s');
        $this->db->join('departments d', 'd.id = s.department_id', 'left');
        $this->db->where('s.is_active', 'yes');
        
        if ($department_id) {
            $this->db->where('s.department_id', $department_id);
        }
        
        $this->db->limit(5); // Limit for preview
        $staff_list = $this->db->get()->result();
        
        $preview = [];
        $total_estimated_cost = 0;
        
        foreach ($staff_list as $staff) {
            try {
                $estimated_payroll = $this->calculateStaffPayrollOnDemand($staff->id, $month, $year, $process_date, 'up_to_date');
                
                $preview[] = [
                    'staff_name' => $staff->name . ' ' . $staff->surname,
                    'employee_id' => $staff->employee_id,
                    'department' => $staff->department_name,
                    'estimated_gross' => $estimated_payroll['gross_salary'],
                    'estimated_net' => $estimated_payroll['net_salary'],
                    'working_days' => $estimated_payroll['working_days'],
                    'worked_days' => $estimated_payroll['worked_days']
                ];
                
                $total_estimated_cost += $estimated_payroll['net_salary'];
                
            } catch (Exception $e) {
                $preview[] = [
                    'staff_name' => $staff->name . ' ' . $staff->surname,
                    'employee_id' => $staff->employee_id,
                    'department' => $staff->department_name,
                    'error' => $e->getMessage()
                ];
            }
        }
        
        return [
            'preview_data' => $preview,
            'total_staff' => count($staff_list),
            'estimated_total_cost' => $total_estimated_cost,
            'process_date' => $process_date
        ];
    }

    public function getPayrollStatus($month, $year) {
        $this->db->select('
            COUNT(*) as total_records,
            SUM(CASE WHEN status = "draft" THEN 1 ELSE 0 END) as draft_count,
            SUM(CASE WHEN status = "approved" THEN 1 ELSE 0 END) as approved_count,
            SUM(CASE WHEN status = "paid" THEN 1 ELSE 0 END) as paid_count,
            MAX(created_at) as last_processed_date
        ');
        $this->db->where('month', $month);
        $this->db->where('year', $year);
        
        return $this->db->get('staff_payroll')->row();
    }

    public function canProcessPayroll($month, $year) {
        // Check if it's reasonable to process payroll
        $current_date = new DateTime();
        $payroll_month = new DateTime($year . '-' . str_pad($month, 2, '0', STR_PAD_LEFT) . '-01');
        
        // Allow processing from the beginning of the month onwards
        return $payroll_month <= $current_date;
    }

    public function getLastProcessedDate($month, $year) {
        $this->db->select('MAX(created_at) as last_processed');
        $this->db->where('month', $month);
        $this->db->where('year', $year);
        $result = $this->db->get('staff_payroll')->row();
        
        return $result ? $result->last_processed : null;
    }

    public function logPayrollProcessing($data) {
        $log_data = [
            'month' => $data['month'],
            'year' => $data['year'],
            'department_id' => $data['department_id'],
            'process_date' => $data['process_date'],
            'payroll_type' => $data['payroll_type'],
            'total_staff_processed' => $data['results']['processed_successfully'],
            'total_errors' => count($data['results']['processing_errors']),
            'total_gross_amount' => $data['results']['total_gross_salary'],
            'total_net_amount' => $data['results']['total_net_salary'],
            'processing_details' => json_encode($data['results']),
            'processed_by' => $data['processed_by'],
            'created_at' => date('Y-m-d H:i:s')
        ];
        
        return $this->db->insert('payroll_processing_log', $log_data);
    }

    public function getPayrollProcessingHistory($limit = 50) {
        $this->db->select('ppl.*, s.name as processed_by_name, d.name as department_name');
        $this->db->from('payroll_processing_log ppl');
        $this->db->join('staff s', 's.id = ppl.processed_by', 'left');
        $this->db->join('departments d', 'd.id = ppl.department_id', 'left');
        $this->db->order_by('ppl.created_at', 'DESC');
        $this->db->limit($limit);
        
        return $this->db->get()->result();
    }

    // =============================================
    // INSTANT PAYSLIP GENERATION
    // =============================================

    public function generateInstantPayslips($month, $year, $department_id = null, $delivery_method = 'both') {
        $results = [
            'total_payslips' => 0,
            'emails_sent' => 0,
            'pdfs_generated' => 0,
            'errors' => [],
            'success_list' => []
        ];

        // Get payroll records
        $payroll_records = $this->getPayrollList($month, $year, $department_id);

        if (empty($payroll_records)) {
            throw new Exception('No payroll records found for the specified period');
        }

        $results['total_payslips'] = count($payroll_records);

        // Create temporary directory for PDFs if needed
        $temp_dir = null;
        if ($delivery_method === 'download' || $delivery_method === 'both') {
            $temp_dir = APPPATH . 'temp/payslips_' . date('YmdHis') . '/';
            if (!is_dir($temp_dir)) {
                mkdir($temp_dir, 0755, true);
            }
        }

        foreach ($payroll_records as $payroll) {
            try {
                $payslip_data = $this->generatePayslipData($payroll->id);
                
                if (!$payslip_data) {
                    $results['errors'][] = 'Failed to generate payslip data for ' . $payroll->name;
                    continue;
                }

                // Email delivery
                if (($delivery_method === 'email' || $delivery_method === 'both') && $payroll->email) {
                    $email_sent = $this->emailPayslip($payroll->id, $payroll->email);
                    if ($email_sent) {
                        $results['emails_sent']++;
                        $results['success_list'][] = [
                            'name' => $payroll->name . ' ' . $payroll->surname,
                            'action' => 'Email sent to ' . $payroll->email
                        ];
                    } else {
                        $results['errors'][] = 'Failed to email payslip to ' . $payroll->name;
                    }
                }

                // PDF generation
                if ($delivery_method === 'download' || $delivery_method === 'both') {
                    $pdf_filename = 'payslip_' . $payroll->employee_id . '_' . 
                                   date('Y_m', mktime(0, 0, 0, $month, 1, $year)) . '.pdf';
                    
                    // Generate PDF content (you'd implement this based on your PDF library)
                    $pdf_content = $this->generatePayslipPDFContent($payslip_data);
                    
                    if ($temp_dir) {
                        file_put_contents($temp_dir . $pdf_filename, $pdf_content);
                        $results['pdfs_generated']++;
                    }
                }

            } catch (Exception $e) {
                $results['errors'][] = 'Error processing ' . $payroll->name . ': ' . $e->getMessage();
            }
        }

        // Create ZIP file if PDFs were generated
        if ($temp_dir && $results['pdfs_generated'] > 0) {
            $this->load->library('zip');
            $this->zip->read_dir($temp_dir, FALSE);
            
            $zip_filename = 'payslips_' . date('Y_m', mktime(0, 0, 0, $month, 1, $year)) . '.zip';
            $this->zip->download($zip_filename);
            
            // Clean up temporary files
            $this->deleteDirectory($temp_dir);
        }

        return $results;
    }

    private function deleteDirectory($dir) {
        if (is_dir($dir)) {
            $files = array_diff(scandir($dir), array('.', '..'));
            foreach ($files as $file) {
                (is_dir("$dir/$file")) ? $this->deleteDirectory("$dir/$file") : unlink("$dir/$file");
            }
            rmdir($dir);
        }
    }

    // =============================================
    // DATABASE INITIALIZATION FOR ON-DEMAND PROCESSING
    // =============================================

    public function createOnDemandTables() {
        // Payroll Processing Log Table
        $this->db->query("
            CREATE TABLE IF NOT EXISTS `payroll_processing_log` (
                `id` int(11) NOT NULL AUTO_INCREMENT,
                `month` int(2) NOT NULL,
                `year` int(4) NOT NULL,
                `department_id` int(11) DEFAULT NULL,
                `process_date` date NOT NULL,
                `payroll_type` varchar(50) DEFAULT 'full_month',
                `total_staff_processed` int(11) DEFAULT 0,
                `total_errors` int(11) DEFAULT 0,
                `total_gross_amount` decimal(15,2) DEFAULT 0.00,
                `total_net_amount` decimal(15,2) DEFAULT 0.00,
                `processing_details` longtext,
                `processed_by` int(11) DEFAULT NULL,
                `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
                PRIMARY KEY (`id`),
                KEY `month_year` (`month`,`year`),
                KEY `processed_by` (`processed_by`),
                KEY `process_date` (`process_date`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8;
        ");

        // Add additional columns to staff_payroll table for on-demand processing
        $this->db->query("
            ALTER TABLE `staff_payroll` 
            ADD COLUMN IF NOT EXISTS `payroll_type` varchar(50) DEFAULT 'full_month',
            ADD COLUMN IF NOT EXISTS `date_from` date DEFAULT NULL,
            ADD COLUMN IF NOT EXISTS `date_to` date DEFAULT NULL,
            ADD COLUMN IF NOT EXISTS `processing_date` date DEFAULT NULL,
            ADD COLUMN IF NOT EXISTS `processed_by` int(11) DEFAULT NULL;
        ");

        return true;
    }<?php
defined('BASEPATH') OR exit('No direct script access allowed');

/**
 * HR Payroll Model
 * Handles payroll calculations and management
 */
class Hr_payroll_model extends CI_Model {

    public function __construct() {
        parent::__construct();
        $this->load->database();
    }

    // =============================================
    // PAYROLL GENERATION
    // =============================================

    public function generateMonthlyPayroll($month, $year, $department_id = null) {
        $this->db->trans_start();

        // Get active staff
        $this->db->select('s.*, d.name as department_name');
        $this->db->from('staff s');
        $this->db->join('departments d', 'd.id = s.department_id', 'left');
        $this->db->where('s.is_active', 'yes');
        
        if ($department_id) {
            $this->db->where('s.department_id', $department_id);
        }
        
        $staff_list = $this->db->get()->result();
        
        $processed_count = 0;
        
        foreach ($staff_list as $staff) {
            // Check if payroll already exists for this period
            $existing_payroll = $this->getStaffPayroll($staff->id, $month, $year);
            
            if (!$existing_payroll) {
                $payroll_data = $this->calculateStaffPayroll($staff->id, $month, $year);
                
                if ($payroll_data) {
                    $this->db->insert('staff_payroll', $payroll_data);
                    $processed_count++;
                }
            }
        }
        
        $this->db->trans_complete();
        
        if ($this->db->trans_status() === FALSE) {
            return false;
        }
        
        return ['records_processed' => $processed_count];
    }

    public function calculateStaffPayroll($staff_id, $month, $year) {
        // Get staff salary structure
        $salary_components = $this->getStaffSalaryComponents($staff_id);
        
        if (empty($salary_components)) {
            return false;
        }

        // Get attendance data for the month
        $attendance_data = $this->getMonthlyAttendanceData($staff_id, $month, $year);
        
        // Get leave data for the month
        $leave_data = $this->getMonthlyLeaveData($staff_id, $month, $year);
        
        // Calculate working days
        $working_days = $this->getWorkingDaysInMonth($month, $year);
        $worked_days = $attendance_data['worked_days'];
        $leave_days = $leave_data['paid_leave_days'];
        
        // Initialize payroll data
        $payroll_data = [
            'staff_id' => $staff_id,
            'month' => $month,
            'year' => $year,
            'working_days' => $working_days,
            'worked_days' => $worked_days,
            'leave_days' => $leave_days,
            'absent_days' => $working_days - $worked_days - $leave_days,
            'overtime_hours' => $attendance_data['overtime_hours'],
            'total_earnings' => 0,
            'total_deductions' => 0,
            'gross_salary' => 0,
            'net_salary' => 0,
            'created_at' => date('Y-m-d H:i:s')
        ];

        $earnings = [];
        $deductions = [];

        foreach ($salary_components as $component) {
            $amount = $this->calculateComponentAmount($component, $payroll_data, $attendance_data);
            
            if ($component->type == 'earning') {
                $earnings[$component->short_code] = $amount;
                $payroll_data['total_earnings'] += $amount;
            } else {
                $deductions[$component->short_code] = $amount;
                $payroll_data['total_deductions'] += $amount;
            }
        }

        $payroll_data['gross_salary'] = $payroll_data['total_earnings'];
        $payroll_data['net_salary'] = $payroll_data['gross_salary'] - $payroll_data['total_deductions'];
        $payroll_data['earnings_json'] = json_encode($earnings);
        $payroll_data['deductions_json'] = json_encode($deductions);

        return $payroll_data;
    }

    private function calculateComponentAmount($component, $payroll_data, $attendance_data) {
        $amount = 0;

        switch ($component->calculation_type) {
            case 'fixed':
                if ($component->short_code == 'OVERTIME') {
                    // Calculate overtime amount
                    $hourly_rate = $component->amount;
                    $amount = $attendance_data['overtime_hours'] * $hourly_rate;
                } else {
                    // Pro-rate based on worked days + paid leave days
                    $effective_days = $payroll_data['worked_days'] + $payroll_data['leave_days'];
                    $amount = ($component->amount / $payroll_data['working_days']) * $effective_days;
                }
                break;

            case 'percentage':
                $base_amount = $this->getBaseSalaryForCalculation($payroll_data['staff_id']);
                $amount = ($base_amount * $component->percentage) / 100;
                break;

            case 'formula':
                // Handle custom formulas (to be implemented)
                $amount = $this->evaluateFormula($component->formula, $payroll_data, $attendance_data);
                break;
        }

        return round($amount, 2);
    }

    private function getBaseSalaryForCalculation($staff_id) {
        $this->db->select('sss.amount');
        $this->db->from('staff_salary_structure sss');
        $this->db->join('salary_components sc', 'sc.id = sss.component_id', 'left');
        $this->db->where('sss.staff_id', $staff_id);
        $this->db->where('sc.short_code', 'BASIC');
        $this->db->where('sss.is_active', 'yes');
        
        $result = $this->db->get()->row();
        return $result ? $result->amount : 0;
    }

    // =============================================
    // PAYROLL DATA RETRIEVAL
    // =============================================

    public function getStaffPayroll($staff_id, $month, $year) {
        $this->db->where('staff_id', $staff_id);
        $this->db->where('month', $month);
        $this->db->where('year', $year);
        return $this->db->get('staff_payroll')->row();
    }

    public function getPayrollList($month = null, $year = null, $department_id = null) {
        $this->db->select('sp.*, s.name, s.surname, s.employee_id, d.name as department_name');
        $this->db->from('staff_payroll sp');
        $this->db->join('staff s', 's.id = sp.staff_id', 'left');
        $this->db->join('departments d', 'd.id = s.department_id', 'left');
        
        if ($month) {
            $this->db->where('sp.month', $month);
        }
        
        if ($year) {
            $this->db->where('sp.year', $year);
        }
        
        if ($department_id) {
            $this->db->where('s.department_id', $department_id);
        }
        
        $this->db->order_by('s.name', 'ASC');
        return $this->db->get()->result();
    }

    public function getCurrentMonthPayroll() {
        $current_month = date('n');
        $current_year = date('Y');
        
        $this->db->select('SUM(net_salary) as total_payroll');
        $this->db->where('month', $current_month);
        $this->db->where('year', $current_year);
        
        $result = $this->db->get('staff_payroll')->row();
        return $result ? $result->total_payroll : 0;
    }

    public function getPayrollById($payroll_id) {
        $this->db->select('sp.*, s.name, s.surname, s.employee_id, d.name as department_name');
        $this->db->from('staff_payroll sp');
        $this->db->join('staff s', 's.id = sp.staff_id', 'left');
        $this->db->join('departments d', 'd.id = s.department_id', 'left');
        $this->db->where('sp.id', $payroll_id);
        return $this->db->get()->row();
    }

    // =============================================
    // PAYSLIP GENERATION
    // =============================================

    public function generatePayslipData($payroll_id) {
        $payroll = $this->getPayrollById($payroll_id);
        
        if (!$payroll) {
            return false;
        }

        // Decode earnings and deductions
        $earnings = json_decode($payroll->earnings_json, true) ?: [];
        $deductions = json_decode($payroll->deductions_json, true) ?: [];

        // Get salary components for display names
        $salary_components = $this->getSalaryComponents();
        $component_names = [];
        foreach ($salary_components as $component) {
            $component_names[$component->short_code] = $component->name;
        }

        // Calculate additional payslip details
        $payslip_details = $this->calculatePayslipDetails($payroll, $earnings, $deductions);

        return [
            'payroll' => $payroll,
            'earnings' => $earnings,
            'deductions' => $deductions,
            'component_names' => $component_names,
            'details' => $payslip_details
        ];
    }

    private function calculatePayslipDetails($payroll, $earnings, $deductions) {
        // Calculate gross pay before tax deductions
        $gross_pay = array_sum($earnings);
        
        // Calculate taxable pay (gross minus non-taxable deductions)
        $non_taxable_deductions = ($deductions['NSSF'] ?? 0) + ($deductions['NHIF'] ?? 0);
        $taxable_pay = $gross_pay - $non_taxable_deductions;
        
        // Tax relief (Kenya standard relief)
        $tax_relief = 1408; // Monthly tax relief as of 2024
        
        // PAYE calculation details
        $paye_before_relief = $this->calculatePAYE($taxable_pay + $tax_relief);
        $paye_after_relief = max(0, $paye_before_relief - $tax_relief);
        
        return [
            'gross_pay' => $gross_pay,
            'taxable_pay' => $taxable_pay,
            'tax_relief' => $tax_relief,
            'paye_before_relief' => $paye_before_relief,
            'paye_after_relief' => $paye_after_relief,
            'total_statutory_deductions' => ($deductions['PAYE'] ?? 0) + ($deductions['NSSF'] ?? 0) + ($deductions['NHIF'] ?? 0) + ($deductions['AHL'] ?? 0),
            'total_other_deductions' => $payroll->total_deductions - (($deductions['PAYE'] ?? 0) + ($deductions['NSSF'] ?? 0) + ($deductions['NHIF'] ?? 0) + ($deductions['AHL'] ?? 0))
        ];
    }

    public function emailPayslip($payroll_id, $email_address = null) {
        $payslip_data = $this->generatePayslipData($payroll_id);
        
        if (!$payslip_data) {
            return false;
        }

        $staff_email = $email_address ?: $payslip_data['payroll']->email;
        
        if (!$staff_email) {
            return false;
        }

        // Load email library and send payslip
        $this->load->library('email');
        
        $payroll = $payslip_data['payroll'];
        $subject = 'Payslip for ' . date('F Y', mktime(0, 0, 0, $payroll->month, 1, $payroll->year));
        
        // Generate payslip HTML content
        $message = $this->generatePayslipEmailContent($payslip_data);
        
        $this->email->to($staff_email);
        $this->email->subject($subject);
        $this->email->message($message);
        
        return $this->email->send();
    }

    private function generatePayslipEmailContent($payslip_data) {
        $payroll = $payslip_data['payroll'];
        
        $message = "Dear " . $payroll->name . " " . $payroll->surname . ",\n\n";
        $message .= "Your payslip for " . date('F Y', mktime(0, 0, 0, $payroll->month, 1, $payroll->year)) . " is ready.\n\n";
        $message .= "PAYSLIP SUMMARY:\n";
        $message .= "Employee ID: " . $payroll->employee_id . "\n";
        $message .= "Department: " . $payroll->department_name . "\n";
        $message .= "Gross Salary: KSh " . number_format($payroll->gross_salary, 2) . "\n";
        $message .= "Total Deductions: KSh " . number_format($payroll->total_deductions, 2) . "\n";
        $message .= "NET PAY: KSh " . number_format($payroll->net_salary, 2) . "\n\n";
        $message .= "Please login to the system to view and download your detailed payslip.\n\n";
        $message .= "Best regards,\n";
        $message .= "HR Department\n";
        $message .= $this->config->item('hospital_name') ?: 'Hospital Management System';
        
        return $message;
    }

    // =============================================
    // STATUTORY CALCULATIONS
    // =============================================

    public function calculatePAYE($taxable_income) {
        // Kenya PAYE calculation as of 2024
        $tax = 0;
        
        if ($taxable_income <= 24000) {
            $tax = $taxable_income * 0.10;
        } elseif ($taxable_income <= 32333) {
            $tax = 2400 + (($taxable_income - 24000) * 0.25);
        } elseif ($taxable_income <= 500000) {
            $tax = 4483.25 + (($taxable_income - 32333) * 0.30);
        } elseif ($taxable_income <= 800000) {
            $tax = 144783.35 + (($taxable_income - 500000) * 0.325);
        } else {
            $tax = 242283.35 + (($taxable_income - 800000) * 0.35);
        }
        
        return round($tax, 2);
    }

    public function calculateNSSF($gross_salary) {
        // NSSF calculation - 6% of pensionable salary (max KSh 2,160)
        $nssf_rate = 0.06;
        $max_nssf = 2160;
        
        $nssf_amount = $gross_salary * $nssf_rate;
        return round(min($nssf_amount, $max_nssf), 2);
    }

    public function calculateNHIF($gross_salary) {
        // NHIF rates based on salary bands
        if ($gross_salary <= 5999) return 150;
        elseif ($gross_salary <= 7999) return 300;
        elseif ($gross_salary <= 11999) return 400;
        elseif ($gross_salary <= 14999) return 500;
        elseif ($gross_salary <= 19999) return 600;
        elseif ($gross_salary <= 24999) return 750;
        elseif ($gross_salary <= 29999) return 850;
        elseif ($gross_salary <= 34999) return 900;
        elseif ($gross_salary <= 39999) return 950;
        elseif ($gross_salary <= 44999) return 1000;
        elseif ($gross_salary <= 49999) return 1100;
        elseif ($gross_salary <= 59999) return 1200;
        elseif ($gross_salary <= 69999) return 1300;
        elseif ($gross_salary <= 79999) return 1400;
        elseif ($gross_salary <= 89999) return 1500;
        elseif ($gross_salary <= 99999) return 1600;
        else return 1700;
    }

    public function calculateAffordableHousingLevy($gross_salary) {
        // Housing Levy - 1.5% of gross salary
        return round($gross_salary * 0.015, 2);
    }

    // =============================================
    // ATTENDANCE AND LEAVE DATA
    // =============================================

    private function getMonthlyAttendanceData($staff_id, $month, $year) {
        $start_date = $year . '-' . str_pad($month, 2, '0', STR_PAD_LEFT) . '-01';
        $end_date = date('Y-m-t', strtotime($start_date));
        
        $this->db->select('
            COUNT(*) as total_attendance_records,
            SUM(CASE WHEN clock_in_time IS NOT NULL THEN 1 ELSE 0 END) as worked_days,
            SUM(COALESCE(total_hours, 0)) as total_hours,
            SUM(COALESCE(overtime_hours, 0)) as overtime_hours
        ');
        $this->db->from('staff_attendance');
        $this->db->where('staff_id', $staff_id);
        $this->db->where('date >=', $start_date);
        $this->db->where('date <=', $end_date);
        
        $result = $this->db->get()->row();
        
        return [
            'worked_days' => $result ? $result->worked_days : 0,
            'total_hours' => $result ? $result->total_hours : 0,
            'overtime_hours' => $result ? $result->overtime_hours : 0
        ];
    }

    private function getMonthlyLeaveData($staff_id, $month, $year) {
        $start_date = $year . '-' . str_pad($month, 2, '0', STR_PAD_LEFT) . '-01';
        $end_date = date('Y-m-t', strtotime($start_date));
        
        $this->db->select('
            COALESCE(SUM(
                CASE 
                    WHEN start_date <= "' . $start_date . '" AND end_date >= "' . $end_date . '" 
                    THEN DATEDIFF("' . $end_date . '", "' . $start_date . '") + 1
                    WHEN start_date <= "' . $start_date . '" AND end_date < "' . $end_date . '" AND end_date >= "' . $start_date . '"
                    THEN DATEDIFF(end_date, "' . $start_date . '") + 1
                    WHEN start_date > "' . $start_date . '" AND end_date >= "' . $end_date . '"
                    THEN DATEDIFF("' . $end_date . '", start_date) + 1
                    WHEN start_date >= "' . $start_date . '" AND end_date <= "' . $end_date . '"
                    THEN total_days
                    ELSE 0
                END
            ), 0) as paid_leave_days
        ');
        $this->db->from('staff_leave sl');
        $this->db->join('leave_types lt', 'lt.id = sl.leave_type_id', 'left');
        $this->db->where('sl.staff_id', $staff_id);
        $this->db->where('sl.status', 'approved');
        $this->db->where('lt.is_paid', 'yes');
        $this->db->group_start();
        $this->db->where('sl.start_date <=', $end_date);
        $this->db->where('sl.end_date >=', $start_date);
        $this->db->group_end();
        
        $result = $this->db->get()->row();
        
        return [
            'paid_leave_days' => $result ? $result->paid_leave_days : 0
        ];
    }

    private function getWorkingDaysInMonth($month, $year) {
        $start_date = $year . '-' . str_pad($month, 2, '0', STR_PAD_LEFT) . '-01';
        $end_date = date('Y-m-t', strtotime($start_date));
        
        $start = new DateTime($start_date);
        $end = new DateTime($end_date);
        $end->modify('+1 day');
        
        $interval = new DateInterval('P1D');
        $period = new DatePeriod($start, $interval, $end);
        
        $working_days = 0;
        foreach ($period as $date) {
            // Skip weekends (Saturday = 6, Sunday = 0)
            if ($date->format('w') != 0 && $date->format('w') != 6) {
                // Check if it's not a holiday
                if (!$this->isHoliday($date->format('Y-m-d'))) {
                    $working_days++;
                }
            }
        }
        
        return $working_days;
    }

    private function isHoliday($date) {
        $this->db->where('holiday_date', $date);
        $this->db->where('is_active', 'yes');
        $holiday = $this->db->get('holidays')->row();
        
        return $holiday ? true : false;
    }

    // =============================================
    // SALARY COMPONENTS
    // =============================================

    private function getStaffSalaryComponents($staff_id) {
        $this->db->select('sss.*, sc.name, sc.short_code, sc.type, sc.calculation_type, sc.is_statutory, sc.is_taxable, sss.percentage');
        $this->db->from('staff_salary_structure sss');
        $this->db->join('salary_components sc', 'sc.id = sss.component_id', 'left');
        $this->db->where('sss.staff_id', $staff_id);
        $this->db->where('sss.is_active', 'yes');
        $this->db->order_by('sc.display_order', 'ASC');
        return $this->db->get()->result();
    }

    private function getSalaryComponents() {
        $this->db->where('is_active', 'yes');
        $this->db->order_by('display_order', 'ASC');
        return $this->db->get('salary_components')->result();
    }

    // =============================================
    // PAYROLL REPORTS
    // =============================================

    public function getPayrollSummary($month, $year, $department_id = null) {
        $this->db->select('
            COUNT(sp.id) as total_staff,
            SUM(sp.gross_salary) as total_gross,
            SUM(sp.total_deductions) as total_deductions,
            SUM(sp.net_salary) as total_net,
            AVG(sp.net_salary) as avg_net_salary
        ');
        $this->db->from('staff_payroll sp');
        $this->db->join('staff s', 's.id = sp.staff_id', 'left');
        $this->db->where('sp.month', $month);
        $this->db->where('sp.year', $year);
        
        if ($department_id) {
            $this->db->where('s.department_id', $department_id);
        }
        
        return $this->db->get()->row();
    }

    public function getDepartmentPayrollSummary($month, $year) {
        $this->db->select('
            d.name as department,
            COUNT(sp.id) as staff_count,
            SUM(sp.gross_salary) as total_gross,
            SUM(sp.net_salary) as total_net,
            AVG(sp.net_salary) as avg_net_salary
        ');
        $this->db->from('staff_payroll sp');
        $this->db->join('staff s', 's.id = sp.staff_id', 'left');
        $this->db->join('departments d', 'd.id = s.department_id', 'left');
        $this->db->where('sp.month', $month);
        $this->db->where('sp.year', $year);
        $this->db->group_by('s.department_id');
        $this->db->order_by('total_net', 'DESC');
        
        return $this->db->get()->result();
    }

    public function getStatutoryRemittanceReport($month, $year) {
        $this->db->select('
            SUM(JSON_EXTRACT(deductions_json, "$.PAYE")) as total_paye,
            SUM(JSON_EXTRACT(deductions_json, "$.NSSF")) as total_nssf,
            SUM(JSON_EXTRACT(deductions_json, "$.NHIF")) as total_nhif,
            SUM(JSON_EXTRACT(deductions_json, "$.AHL")) as total_ahl
        ');
        $this->db->from('staff_payroll');
        $this->db->where('month', $month);
        $this->db->where('year', $year);
        
        return $this->db->get()->row();
    }

    // =============================================
    // PAYROLL ADJUSTMENTS
    // =============================================

    public function addPayrollAdjustment($data) {
        return $this->db->insert('payroll_adjustments', $data);
    }

    public function getPayrollAdjustments($staff_id = null, $month = null, $year = null) {
        $this->db->select('pa.*, s.name, s.surname, s.employee_id, sc.name as component_name');
        $this->db->from('payroll_adjustments pa');
        $this->db->join('staff s', 's.id = pa.staff_id', 'left');
        $this->db->join('salary_components sc', 'sc.id = pa.component_id', 'left');
        
        if ($staff_id) {
            $this->db->where('pa.staff_id', $staff_id);
        }
        
        if ($month && $year) {
            $this->db->where('pa.month', $month);
            $this->db->where('pa.year', $year);
        }
        
        $this->db->order_by('pa.created_at', 'DESC');
        return $this->db->get()->result();
    }

    // =============================================
    // UTILITY FUNCTIONS
    // =============================================

    private function evaluateFormula($formula, $payroll_data, $attendance_data) {
        // Simple formula evaluation - can be extended
        $variables = [
            'basic_salary' => $this->getBaseSalaryForCalculation($payroll_data['staff_id']),
            'worked_days' => $payroll_data['worked_days'],
            'working_days' => $payroll_data['working_days'],
            'overtime_hours' => $attendance_data['overtime_hours'],
            'gross_salary' => $payroll_data['total_earnings']
        ];
        
        // Replace variables in formula
        foreach ($variables as $var => $value) {
            $formula = str_replace('{' . $var . '}', $value, $formula);
        }
        
        // Basic evaluation (use with caution in production)
        try {
            return eval("return $formula;");
        } catch (Exception $e) {
            return 0;
        }
    }

    // =============================================
    // DATABASE INITIALIZATION
    // =============================================

    public function createPayrollTables() {
        // Staff Payroll Table
        $this->db->query("
            CREATE TABLE IF NOT EXISTS `staff_payroll` (
                `id` int(11) NOT NULL AUTO_INCREMENT,
                `staff_id` int(11) NOT NULL,
                `month` int(2) NOT NULL,
                `year` int(4) NOT NULL,
                `working_days` int(3) NOT NULL DEFAULT 0,
                `worked_days` int(3) NOT NULL DEFAULT 0,
                `leave_days` int(3) NOT NULL DEFAULT 0,
                `absent_days` int(3) NOT NULL DEFAULT 0,
                `overtime_hours` decimal(5,2) NOT NULL DEFAULT 0.00,
                `total_earnings` decimal(10,2) NOT NULL DEFAULT 0.00,
                `total_deductions` decimal(10,2) NOT NULL DEFAULT 0.00,
                `gross_salary` decimal(10,2) NOT NULL DEFAULT 0.00,
                `net_salary` decimal(10,2) NOT NULL DEFAULT 0.00,
                `earnings_json` text,
                `deductions_json` text,
                `status` enum('draft','approved','paid') DEFAULT 'draft',
                `approved_by` int(11) DEFAULT NULL,
                `approved_at` datetime DEFAULT NULL,
                `paid_at` datetime DEFAULT NULL,
                `notes` text,
                `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
                `updated_at` timestamp NULL DEFAULT NULL ON UPDATE CURRENT_TIMESTAMP,
                PRIMARY KEY (`id`),
                UNIQUE KEY `staff_month_year` (`staff_id`,`month`,`year`),
                KEY `staff_id` (`staff_id`),
                KEY `month_year` (`month`,`year`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8;
        ");

        // Payroll Adjustments Table
        $this->db->query("
            CREATE TABLE IF NOT EXISTS `payroll_adjustments` (
                `id` int(11) NOT NULL AUTO_INCREMENT,
                `staff_id` int(11) NOT NULL,
                `component_id` int(11) NOT NULL,
                `month` int(2) NOT NULL,
                `year` int(4) NOT NULL,
                `adjustment_type` enum('addition','deduction') NOT NULL,
                `amount` decimal(10,2) NOT NULL,
                `reason` text NOT NULL,
                `is_recurring` enum('yes','no') DEFAULT 'no',
                `created_by` int(11) NOT NULL,
                `approved_by` int(11) DEFAULT NULL,
                `status` enum('pending','approved','applied') DEFAULT 'pending',
                `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
                `updated_at` timestamp NULL DEFAULT NULL ON UPDATE CURRENT_TIMESTAMP,
                PRIMARY KEY (`id`),
                KEY `staff_id` (`staff_id`),
                KEY `component_id` (`component_id`),
                KEY `month_year` (`month`,`year`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8;
        ");

        // Payroll History Table
        $this->db->query("
            CREATE TABLE IF NOT EXISTS `payroll_history` (
                `id` int(11) NOT NULL AUTO_INCREMENT,
                `payroll_id` int(11) NOT NULL,
                `action` varchar(50) NOT NULL,
                `old_values` text,
                `new_values` text,
                `changed_by` int(11) NOT NULL,
                `change_reason` text,
                `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
                PRIMARY KEY (`id`),
                KEY `payroll_id` (`payroll_id`),
                KEY `changed_by` (`changed_by`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8;
        ");

        // Tax Brackets Table
        $this->db->query("
            CREATE TABLE IF NOT EXISTS `tax_brackets` (
                `id` int(11) NOT NULL AUTO_INCREMENT,
                `bracket_name` varchar(100) NOT NULL,
                `min_amount` decimal(10,2) NOT NULL,
                `max_amount` decimal(10,2) DEFAULT NULL,
                `tax_rate` decimal(5,2) NOT NULL,
                `fixed_amount` decimal(10,2) DEFAULT 0.00,
                `effective_from` date NOT NULL,
                `effective_to` date DEFAULT NULL,
                `is_active` enum('yes','no') DEFAULT 'yes',
                `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
                `updated_at` timestamp NULL DEFAULT NULL ON UPDATE CURRENT_TIMESTAMP,
                PRIMARY KEY (`id`),
                KEY `effective_from` (`effective_from`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8;
        ");

        return true;
    }

    public function insertDefaultTaxBrackets() {
        $tax_brackets = [
            ['bracket_name' => '10% Band', 'min_amount' => 0, 'max_amount' => 24000, 'tax_rate' => 10.00, 'fixed_amount' => 0, 'effective_from' => date('Y-01-01')],
            ['bracket_name' => '25% Band', 'min_amount' => 24001, 'max_amount' => 32333, 'tax_rate' => 25.00, 'fixed_amount' => 2400, 'effective_from' => date('Y-01-01')],
            ['bracket_name' => '30% Band', 'min_amount' => 32334, 'max_amount' => 500000, 'tax_rate' => 30.00, 'fixed_amount' => 4483.25, 'effective_from' => date('Y-01-01')],
            ['bracket_name' => '32.5% Band', 'min_amount' => 500001, 'max_amount' => 800000, 'tax_rate' => 32.50, 'fixed_amount' => 144783.35, 'effective_from' => date('Y-01-01')],
            ['bracket_name' => '35% Band', 'min_amount' => 800001, 'max_amount' => null, 'tax_rate' => 35.00, 'fixed_amount' => 242283.35, 'effective_from' => date('Y-01-01')]
        ];

        foreach ($tax_brackets as $bracket) {
            // Check if bracket already exists
            $this->db->where('bracket_name', $bracket['bracket_name']);
            $this->db->where('effective_from', $bracket['effective_from']);
            $existing = $this->db->get('tax_brackets')->row();
            
            if (!$existing) {
                $this->db->insert('tax_brackets', $bracket);
            }
        }

        return true;
    }
}