<?php
defined('BASEPATH') OR exit('No direct script access allowed');

/**
 * HR Leave Model
 * Handles leave management operations
 */
class Hr_leave_model extends CI_Model {

    public function __construct() {
        parent::__construct();
        $this->load->database();
    }

    // =============================================
    // LEAVE TYPES MANAGEMENT
    // =============================================

    public function getLeaveTypes() {
        $this->db->where('is_active', 'yes');
        $this->db->order_by('name', 'ASC');
        return $this->db->get('leave_types')->result();
    }

    public function addLeaveType($data) {
        return $this->db->insert('leave_types', $data);
    }

    public function updateLeaveType($type_id, $data) {
        $this->db->where('id', $type_id);
        return $this->db->update('leave_types', $data);
    }

    public function getLeaveTypeById($type_id) {
        $this->db->where('id', $type_id);
        return $this->db->get('leave_types')->row();
    }

    // =============================================
    // LEAVE APPLICATIONS
    // =============================================

    public function applyLeave($data) {
        $this->db->trans_start();
        
        // Insert leave application
        $this->db->insert('staff_leave', $data);
        $leave_id = $this->db->insert_id();
        
        // Log the application
        $this->db->insert('leave_approval_log', [
            'leave_id' => $leave_id,
            'action' => 'applied',
            'comments' => 'Leave application submitted',
            'action_by' => $data['staff_id'],
            'action_date' => date('Y-m-d H:i:s')
        ]);
        
        $this->db->trans_complete();
        
        if ($this->db->trans_status() === FALSE) {
            return false;
        }
        
        return $leave_id;
    }

    public function updateLeave($leave_id, $data) {
        $this->db->where('id', $leave_id);
        return $this->db->update('staff_leave', $data);
    }

    public function getLeaveById($leave_id) {
        $this->db->select('sl.*, lt.name as leave_type_name, s.name as staff_name, s.surname as staff_surname, s.employee_id');
        $this->db->from('staff_leave sl');
        $this->db->join('leave_types lt', 'lt.id = sl.leave_type_id', 'left');
        $this->db->join('staff s', 's.id = sl.staff_id', 'left');
        $this->db->where('sl.id', $leave_id);
        return $this->db->get()->row();
    }

    public function getLeaveApplications($staff_id = null, $status = null, $start_date = null, $end_date = null) {
        $this->db->select('sl.*, lt.name as leave_type_name, s.name as staff_name, s.surname as staff_surname, s.employee_id, d.name as department_name');
        $this->db->from('staff_leave sl');
        $this->db->join('leave_types lt', 'lt.id = sl.leave_type_id', 'left');
        $this->db->join('staff s', 's.id = sl.staff_id', 'left');
        $this->db->join('departments d', 'd.id = s.department_id', 'left');
        
        if ($staff_id) {
            $this->db->where('sl.staff_id', $staff_id);
        }
        
        if ($status) {
            $this->db->where('sl.status', $status);
        }
        
        if ($start_date && $end_date) {
            $this->db->where('sl.start_date >=', $start_date);
            $this->db->where('sl.end_date <=', $end_date);
        }
        
        $this->db->order_by('sl.applied_date', 'DESC');
        return $this->db->get()->result();
    }

    public function getPendingLeaveRequests() {
        $this->db->where('status', 'pending');
        return $this->db->count_all_results('staff_leave');
    }

    public function getRecentLeaveRequests($limit = 5) {
        $this->db->select('sl.*, lt.name as leave_type_name, s.name as staff_name, s.surname as staff_surname, s.employee_id');
        $this->db->from('staff_leave sl');
        $this->db->join('leave_types lt', 'lt.id = sl.leave_type_id', 'left');
        $this->db->join('staff s', 's.id = sl.staff_id', 'left');
        $this->db->order_by('sl.applied_date', 'DESC');
        $this->db->limit($limit);
        return $this->db->get()->result();
    }

    public function getStaffOnLeaveToday() {
        $today = date('Y-m-d');
        
        $this->db->select('COUNT(*) as count');
        $this->db->from('staff_leave');
        $this->db->where('start_date <=', $today);
        $this->db->where('end_date >=', $today);
        $this->db->where('status', 'approved');
        
        $result = $this->db->get()->row();
        return $result ? $result->count : 0;
    }

    // =============================================
    // LEAVE APPROVAL WORKFLOW
    // =============================================

    public function approveLeave($leave_id, $approved_by, $comments = '') {
        $this->db->trans_start();
        
        // Update leave status
        $this->db->where('id', $leave_id);
        $this->db->update('staff_leave', [
            'status' => 'approved',
            'approved_by' => $approved_by,
            'approved_date' => date('Y-m-d'),
            'updated_at' => date('Y-m-d H:i:s')
        ]);
        
        // Log the approval
        $this->db->insert('leave_approval_log', [
            'leave_id' => $leave_id,
            'action' => 'approved',
            'comments' => $comments,
            'action_by' => $approved_by,
            'action_date' => date('Y-m-d H:i:s')
        ]);
        
        // Get leave details for balance deduction
        $leave = $this->getLeaveById($leave_id);
        
        // Deduct from leave balance
        if ($leave) {
            $this->deductLeaveBalance($leave->staff_id, $leave->leave_type_id, $leave->total_days);
        }
        
        $this->db->trans_complete();
        return $this->db->trans_status();
    }

    public function rejectLeave($leave_id, $rejected_by, $comments = '') {
        $this->db->trans_start();
        
        // Update leave status
        $this->db->where('id', $leave_id);
        $this->db->update('staff_leave', [
            'status' => 'rejected',
            'rejected_by' => $rejected_by,
            'rejected_date' => date('Y-m-d'),
            'updated_at' => date('Y-m-d H:i:s')
        ]);
        
        // Log the rejection
        $this->db->insert('leave_approval_log', [
            'leave_id' => $leave_id,
            'action' => 'rejected',
            'comments' => $comments,
            'action_by' => $rejected_by,
            'action_date' => date('Y-m-d H:i:s')
        ]);
        
        $this->db->trans_complete();
        return $this->db->trans_status();
    }

    public function cancelLeave($leave_id, $cancelled_by, $comments = '') {
        $this->db->trans_start();
        
        // Get leave details before cancellation
        $leave = $this->getLeaveById($leave_id);
        
        // Update leave status
        $this->db->where('id', $leave_id);
        $this->db->update('staff_leave', [
            'status' => 'cancelled',
            'updated_at' => date('Y-m-d H:i:s')
        ]);
        
        // Log the cancellation
        $this->db->insert('leave_approval_log', [
            'leave_id' => $leave_id,
            'action' => 'cancelled',
            'comments' => $comments,
            'action_by' => $cancelled_by,
            'action_date' => date('Y-m-d H:i:s')
        ]);
        
        // Restore leave balance if it was already approved
        if ($leave && $leave->status == 'approved') {
            $this->restoreLeaveBalance($leave->staff_id, $leave->leave_type_id, $leave->total_days);
        }
        
        $this->db->trans_complete();
        return $this->db->trans_status();
    }

    public function getLeaveApprovalLog($leave_id) {
        $this->db->select('lal.*, s.name as action_by_name, s.surname as action_by_surname');
        $this->db->from('leave_approval_log lal');
        $this->db->join('staff s', 's.id = lal.action_by', 'left');
        $this->db->where('lal.leave_id', $leave_id);
        $this->db->order_by('lal.action_date', 'ASC');
        return $this->db->get()->result();
    }

    // =============================================
    // LEAVE BALANCE MANAGEMENT
    // =============================================

    public function getLeaveBalance($staff_id, $leave_type_id = null) {
        $this->db->select('slb.*, lt.name as leave_type_name');
        $this->db->from('staff_leave_balance slb');
        $this->db->join('leave_types lt', 'lt.id = slb.leave_type_id', 'left');
        $this->db->where('slb.staff_id', $staff_id);
        $this->db->where('slb.year', date('Y'));
        
        if ($leave_type_id) {
            $this->db->where('slb.leave_type_id', $leave_type_id);
            $result = $this->db->get()->row();
            return $result ? $result->remaining_days : 0;
        }
        
        return $this->db->get()->result();
    }

    public function initializeStaffLeaveBalance($staff_id, $year = null) {
        if (!$year) {
            $year = date('Y');
        }

        $this->db->trans_start();
        
        // Get all active leave types
        $leave_types = $this->getLeaveTypes();
        
        foreach ($leave_types as $type) {
            // Check if balance already exists
            $this->db->where('staff_id', $staff_id);
            $this->db->where('leave_type_id', $type->id);
            $this->db->where('year', $year);
            $existing = $this->db->get('staff_leave_balance')->row();
            
            if (!$existing) {
                $balance_data = [
                    'staff_id' => $staff_id,
                    'leave_type_id' => $type->id,
                    'year' => $year,
                    'allocated_days' => $type->default_days,
                    'used_days' => 0,
                    'remaining_days' => $type->default_days,
                    'carried_forward' => 0,
                    'created_at' => date('Y-m-d H:i:s')
                ];
                
                $this->db->insert('staff_leave_balance', $balance_data);
            }
        }
        
        $this->db->trans_complete();
        return $this->db->trans_status();
    }

    public function deductLeaveBalance($staff_id, $leave_type_id, $days) {
        $this->db->where('staff_id', $staff_id);
        $this->db->where('leave_type_id', $leave_type_id);
        $this->db->where('year', date('Y'));
        $this->db->set('used_days', 'used_days + ' . $days, FALSE);
        $this->db->set('remaining_days', 'remaining_days - ' . $days, FALSE);
        $this->db->set('updated_at', date('Y-m-d H:i:s'));
        return $this->db->update('staff_leave_balance');
    }

    public function restoreLeaveBalance($staff_id, $leave_type_id, $days) {
        $this->db->where('staff_id', $staff_id);
        $this->db->where('leave_type_id', $leave_type_id);
        $this->db->where('year', date('Y'));
        $this->db->set('used_days', 'used_days - ' . $days, FALSE);
        $this->db->set('remaining_days', 'remaining_days + ' . $days, FALSE);
        $this->db->set('updated_at', date('Y-m-d H:i:s'));
        return $this->db->update('staff_leave_balance');
    }

    public function adjustLeaveBalance($staff_id, $leave_type_id, $adjustment_days, $reason) {
        $this->db->trans_start();
        
        // Update balance
        $this->db->where('staff_id', $staff_id);
        $this->db->where('leave_type_id', $leave_type_id);
        $this->db->where('year', date('Y'));
        $this->db->set('allocated_days', 'allocated_days + ' . $adjustment_days, FALSE);
        $this->db->set('remaining_days', 'remaining_days + ' . $adjustment_days, FALSE);
        $this->db->set('updated_at', date('Y-m-d H:i:s'));
        $this->db->update('staff_leave_balance');
        
        // Log the adjustment
        $this->db->insert('leave_balance_adjustments', [
            'staff_id' => $staff_id,
            'leave_type_id' => $leave_type_id,
            'adjustment_days' => $adjustment_days,
            'reason' => $reason,
            'adjusted_by' => $this->session->userdata('admin_id'),
            'adjustment_date' => date('Y-m-d'),
            'created_at' => date('Y-m-d H:i:s')
        ]);
        
        $this->db->trans_complete();
        return $this->db->trans_status();
    }

    // =============================================
    // LEAVE CARRYOVER
    // =============================================

    public function processYearEndCarryover($year) {
        $this->db->trans_start();
        
        // Get all staff leave balances for the current year
        $this->db->select('slb.*, lt.max_carryover');
        $this->db->from('staff_leave_balance slb');
        $this->db->join('leave_types lt', 'lt.id = slb.leave_type_id', 'left');
        $this->db->where('slb.year', $year);
        $this->db->where('lt.allow_carryover', 'yes');
        $balances = $this->db->get()->result();
        
        foreach ($balances as $balance) {
            $carryover_days = min($balance->remaining_days, $balance->max_carryover);
            
            if ($carryover_days > 0) {
                // Check if next year balance exists
                $next_year = $year + 1;
                $this->db->where('staff_id', $balance->staff_id);
                $this->db->where('leave_type_id', $balance->leave_type_id);
                $this->db->where('year', $next_year);
                $next_year_balance = $this->db->get('staff_leave_balance')->row();
                
                if ($next_year_balance) {
                    // Update existing balance
                    $this->db->where('id', $next_year_balance->id);
                    $this->db->set('carried_forward', $carryover_days);
                    $this->db->set('allocated_days', 'allocated_days + ' . $carryover_days, FALSE);
                    $this->db->set('remaining_days', 'remaining_days + ' . $carryover_days, FALSE);
                    $this->db->update('staff_leave_balance');
                } else {
                    // Create new balance for next year
                    $leave_type = $this->getLeaveTypeById($balance->leave_type_id);
                    $new_balance = [
                        'staff_id' => $balance->staff_id,
                        'leave_type_id' => $balance->leave_type_id,
                        'year' => $next_year,
                        'allocated_days' => $leave_type->default_days + $carryover_days,
                        'used_days' => 0,
                        'remaining_days' => $leave_type->default_days + $carryover_days,
                        'carried_forward' => $carryover_days,
                        'created_at' => date('Y-m-d H:i:s')
                    ];
                    $this->db->insert('staff_leave_balance', $new_balance);
                }
            }
        }
        
        $this->db->trans_complete();
        return $this->db->trans_status();
    }

    // =============================================
    // LEAVE REPORTS
    // =============================================

    public function getLeaveReport($start_date, $end_date, $department_id = null, $leave_type_id = null) {
        $this->db->select('sl.*, lt.name as leave_type_name, s.name as staff_name, s.surname as staff_surname, s.employee_id, d.name as department_name');
        $this->db->from('staff_leave sl');
        $this->db->join('leave_types lt', 'lt.id = sl.leave_type_id', 'left');
        $this->db->join('staff s', 's.id = sl.staff_id', 'left');
        $this->db->join('departments d', 'd.id = s.department_id', 'left');
        $this->db->where('sl.start_date >=', $start_date);
        $this->db->where('sl.start_date <=', $end_date);
        $this->db->where('sl.status', 'approved');
        
        if ($department_id) {
            $this->db->where('s.department_id', $department_id);
        }
        
        if ($leave_type_id) {
            $this->db->where('sl.leave_type_id', $leave_type_id);
        }
        
        $this->db->order_by('sl.start_date', 'DESC');
        return $this->db->get()->result();
    }

    public function getLeaveStatistics($year = null) {
        if (!$year) {
            $year = date('Y');
        }

        $this->db->select('
            lt.name as leave_type,
            COUNT(sl.id) as total_applications,
            SUM(CASE WHEN sl.status = "approved" THEN 1 ELSE 0 END) as approved_applications,
            SUM(CASE WHEN sl.status = "approved" THEN sl.total_days ELSE 0 END) as total_days_taken
        ');
        $this->db->from('staff_leave sl');
        $this->db->join('leave_types lt', 'lt.id = sl.leave_type_id', 'left');
        $this->db->where('YEAR(sl.start_date)', $year);
        $this->db->group_by('sl.leave_type_id');
        $this->db->order_by('total_days_taken', 'DESC');
        
        return $this->db->get()->result();
    }

    public function getDepartmentLeaveStatistics($year = null) {
        if (!$year) {
            $year = date('Y');
        }

        $this->db->select('
            d.name as department,
            COUNT(sl.id) as total_applications,
            SUM(CASE WHEN sl.status = "approved" THEN sl.total_days ELSE 0 END) as total_days_taken,
            AVG(CASE WHEN sl.status = "approved" THEN sl.total_days ELSE NULL END) as avg_days_per_application
        ');
        $this->db->from('staff_leave sl');
        $this->db->join('staff s', 's.id = sl.staff_id', 'left');
        $this->db->join('departments d', 'd.id = s.department_id', 'left');
        $this->db->where('YEAR(sl.start_date)', $year);
        $this->db->group_by('s.department_id');
        $this->db->order_by('total_days_taken', 'DESC');
        
        return $this->db->get()->result();
    }

    // =============================================
    // UTILITY FUNCTIONS
    // =============================================

    public function calculateLeaveDays($start_date, $end_date, $exclude_weekends = true, $exclude_holidays = true) {
        $start = new DateTime($start_date);
        $end = new DateTime($end_date);
        $end->modify('+1 day'); // Include end date
        
        $interval = new DateInterval('P1D');
        $period = new DatePeriod($start, $interval, $end);
        
        $leave_days = 0;
        foreach ($period as $date) {
            $include_day = true;
            
            // Exclude weekends if required
            if ($exclude_weekends && ($date->format('w') == 0 || $date->format('w') == 6)) {
                $include_day = false;
            }
            
            // Exclude holidays if required
            if ($exclude_holidays && $this->isHoliday($date->format('Y-m-d'))) {
                $include_day = false;
            }
            
            if ($include_day) {
                $leave_days++;
            }
        }
        
        return $leave_days;
    }

    private function isHoliday($date) {
        $this->db->where('holiday_date', $date);
        $this->db->where('is_active', 'yes');
        $holiday = $this->db->get('holidays')->row();
        
        return $holiday ? true : false;
    }

    public function checkLeaveConflict($staff_id, $start_date, $end_date, $exclude_leave_id = null) {
        $this->db->where('staff_id', $staff_id);
        $this->db->where('status', 'approved');
        $this->db->group_start();
        $this->db->where('start_date <=', $end_date);
        $this->db->where('end_date >=', $start_date);
        $this->db->group_end();
        
        if ($exclude_leave_id) {
            $this->db->where('id !=', $exclude_leave_id);
        }
        
        return $this->db->get('staff_leave')->result();
    }

    // =============================================
    // DATABASE INITIALIZATION
    // =============================================

    public function createLeaveTables() {
        // Leave Types Table
        $this->db->query("
            CREATE TABLE IF NOT EXISTS `leave_types` (
                `id` int(11) NOT NULL AUTO_INCREMENT,
                `name` varchar(100) NOT NULL,
                `short_code` varchar(20) NOT NULL,
                `default_days` int(11) NOT NULL DEFAULT 0,
                `max_days_per_application` int(11) DEFAULT NULL,
                `require_approval` enum('yes','no') DEFAULT 'yes',
                `allow_carryover` enum('yes','no') DEFAULT 'no',
                `max_carryover` int(11) DEFAULT 0,
                `gender_specific` enum('male','female','both') DEFAULT 'both',
                `description` text,
                `is_active` enum('yes','no') DEFAULT 'yes',
                `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
                `updated_at` timestamp NULL DEFAULT NULL ON UPDATE CURRENT_TIMESTAMP,
                PRIMARY KEY (`id`),
                UNIQUE KEY `short_code` (`short_code`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8;
        ");

        // Staff Leave Table
        $this->db->query("
            CREATE TABLE IF NOT EXISTS `staff_leave` (
                `id` int(11) NOT NULL AUTO_INCREMENT,
                `staff_id` int(11) NOT NULL,
                `leave_type_id` int(11) NOT NULL,
                `start_date` date NOT NULL,
                `end_date` date NOT NULL,
                `total_days` int(11) NOT NULL,
                `reason` text NOT NULL,
                `emergency_contact` varchar(255) DEFAULT NULL,
                `handover_notes` text,
                `status` enum('pending','approved','rejected','cancelled') DEFAULT 'pending',
                `applied_date` date NOT NULL,
                `approved_by` int(11) DEFAULT NULL,
                `approved_date` date DEFAULT NULL,
                `rejected_by` int(11) DEFAULT NULL,
                `rejected_date` date DEFAULT NULL,
                `rejection_reason` text,
                `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
                `updated_at` timestamp NULL DEFAULT NULL ON UPDATE CURRENT_TIMESTAMP,
                PRIMARY KEY (`id`),
                KEY `staff_id` (`staff_id`),
                KEY `leave_type_id` (`leave_type_id`),
                KEY `status` (`status`),
                KEY `start_date` (`start_date`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8;
        ");

        // Staff Leave Balance Table
        $this->db->query("
            CREATE TABLE IF NOT EXISTS `staff_leave_balance` (
                `id` int(11) NOT NULL AUTO_INCREMENT,
                `staff_id` int(11) NOT NULL,
                `leave_type_id` int(11) NOT NULL,
                `year` int(4) NOT NULL,
                `allocated_days` int(11) NOT NULL DEFAULT 0,
                `used_days` int(11) NOT NULL DEFAULT 0,
                `remaining_days` int(11) NOT NULL DEFAULT 0,
                `carried_forward` int(11) NOT NULL DEFAULT 0,
                `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
                `updated_at` timestamp NULL DEFAULT NULL ON UPDATE CURRENT_TIMESTAMP,
                PRIMARY KEY (`id`),
                UNIQUE KEY `staff_leave_year` (`staff_id`,`leave_type_id`,`year`),
                KEY `staff_id` (`staff_id`),
                KEY `leave_type_id` (`leave_type_id`),
                KEY `year` (`year`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8;
        ");

        // Leave Approval Log Table
        $this->db->query("
            CREATE TABLE IF NOT EXISTS `leave_approval_log` (
                `id` int(11) NOT NULL AUTO_INCREMENT,
                `leave_id` int(11) NOT NULL,
                `action` enum('applied','approved','rejected','cancelled') NOT NULL,
                `comments` text,
                `action_by` int(11) NOT NULL,
                `action_date` datetime NOT NULL,
                `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
                PRIMARY KEY (`id`),
                KEY `leave_id` (`leave_id`),
                KEY `action_by` (`action_by`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8;
        ");

        // Leave Balance Adjustments Table
        $this->db->query("
            CREATE TABLE IF NOT EXISTS `leave_balance_adjustments` (
                `id` int(11) NOT NULL AUTO_INCREMENT,
                `staff_id` int(11) NOT NULL,
                `leave_type_id` int(11) NOT NULL,
                `adjustment_days` int(11) NOT NULL,
                `reason` text NOT NULL,
                `adjusted_by` int(11) NOT NULL,
                `adjustment_date` date NOT NULL,
                `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
                PRIMARY KEY (`id`),
                KEY `staff_id` (`staff_id`),
                KEY `leave_type_id` (`leave_type_id`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8;
        ");

        return true;
    }

    public function insertDefaultLeaveTypes() {
        $leave_types = [
            [
                'name' => 'Annual Leave',
                'short_code' => 'AL',
                'default_days' => 21,
                'max_days_per_application' => 21,
                'allow_carryover' => 'yes',
                'max_carryover' => 5,
                'description' => 'Annual vacation leave'
            ],
            [
                'name' => 'Sick Leave',
                'short_code' => 'SL',
                'default_days' => 14,
                'max_days_per_application' => 14,
                'allow_carryover' => 'no',
                'description' => 'Medical leave for illness'
            ],
            [
                'name' => 'Maternity Leave',
                'short_code' => 'ML',
                'default_days' => 90,
                'max_days_per_application' => 90,
                'gender_specific' => 'female',
                'allow_carryover' => 'no',
                'description' => 'Maternity leave for female staff'
            ],
            [
                'name' => 'Paternity Leave',
                'short_code' => 'PL',
                'default_days' => 14,
                'max_days_per_application' => 14,
                'gender_specific' => 'male',
                'allow_carryover' => 'no',
                'description' => 'Paternity leave for male staff'
            ],
            [
                'name' => 'Compassionate Leave',
                'short_code' => 'CL',
                'default_days' => 7,
                'max_days_per_application' => 7,
                'allow_carryover' => 'no',
                'description' => 'Leave for bereavement or family emergencies'
            ],
            [
                'name' => 'Study Leave',
                'short_code' => 'STL',
                'default_days' => 5,
                'max_days_per_application' => 30,
                'allow_carryover' => 'yes',
                'max_carryover' => 10,
                'description' => 'Leave for educational purposes'
            ]
        ];

        foreach ($leave_types as $type) {
            // Check if leave type already exists
            $this->db->where('short_code', $type['short_code']);
            $existing = $this->db->get('leave_types')->row();
            
            if (!$existing) {
                $this->db->insert('leave_types', $type);
            }
        }

        return true;
    }
}