<?php
defined('BASEPATH') OR exit('No direct script access allowed');

/**
 * HR Attendance Model
 * Handles attendance and time tracking operations
 */
class Hr_attendance_model extends CI_Model {

    public function __construct() {
        parent::__construct();
        $this->load->database();
    }

    // =============================================
    // ATTENDANCE TRACKING
    // =============================================

    public function getTodayAttendance($staff_id) {
        $this->db->where('staff_id', $staff_id);
        $this->db->where('date', date('Y-m-d'));
        return $this->db->get('staff_attendance')->row();
    }

    public function clockIn($data) {
        return $this->db->insert('staff_attendance', $data);
    }

    public function clockOut($attendance_id, $data) {
        $this->db->where('id', $attendance_id);
        return $this->db->update('staff_attendance', $data);
    }

    public function getAttendanceByDate($date, $department_id = null) {
        $this->db->select('sa.*, s.name, s.surname, s.employee_id, d.name as department_name');
        $this->db->from('staff_attendance sa');
        $this->db->join('staff s', 's.id = sa.staff_id', 'left');
        $this->db->join('departments d', 'd.id = s.department_id', 'left');
        $this->db->where('sa.date', $date);
        
        if ($department_id) {
            $this->db->where('s.department_id', $department_id);
        }
        
        $this->db->order_by('s.name', 'ASC');
        return $this->db->get()->result();
    }

    public function getAttendanceReport($start_date, $end_date, $staff_id = null, $department_id = null) {
        $this->db->select('sa.*, s.name, s.surname, s.employee_id, d.name as department_name');
        $this->db->from('staff_attendance sa');
        $this->db->join('staff s', 's.id = sa.staff_id', 'left');
        $this->db->join('departments d', 'd.id = s.department_id', 'left');
        $this->db->where('sa.date >=', $start_date);
        $this->db->where('sa.date <=', $end_date);
        
        if ($staff_id) {
            $this->db->where('sa.staff_id', $staff_id);
        }
        
        if ($department_id) {
            $this->db->where('s.department_id', $department_id);
        }
        
        $this->db->order_by('sa.date', 'DESC');
        $this->db->order_by('s.name', 'ASC');
        return $this->db->get()->result();
    }

    public function getCurrentMonthAttendanceRate() {
        $start_date = date('Y-m-01');
        $end_date = date('Y-m-t');
        
        // Get total working days (excluding weekends)
        $working_days = $this->getWorkingDaysBetween($start_date, $end_date);
        
        // Get total possible attendance (active staff * working days)
        $this->db->where('is_active', 'yes');
        $active_staff_count = $this->db->count_all_results('staff');
        $total_possible = $active_staff_count * $working_days;
        
        // Get actual attendance count
        $this->db->where('date >=', $start_date);
        $this->db->where('date <=', $end_date);
        $this->db->where('clock_in_time IS NOT NULL');
        $actual_attendance = $this->db->count_all_results('staff_attendance');
        
        if ($total_possible > 0) {
            return round(($actual_attendance / $total_possible) * 100, 2);
        }
        
        return 0;
    }

    public function getStaffOnLeaveToday() {
        $today = date('Y-m-d');
        
        $this->db->select('COUNT(DISTINCT sl.staff_id) as count');
        $this->db->from('staff_leave sl');
        $this->db->where('sl.start_date <=', $today);
        $this->db->where('sl.end_date >=', $today);
        $this->db->where('sl.status', 'approved');
        
        $result = $this->db->get()->row();
        return $result ? $result->count : 0;
    }

    // =============================================
    // SHIFT MANAGEMENT
    // =============================================

    public function getShifts() {
        $this->db->where('is_active', 'yes');
        $this->db->order_by('start_time', 'ASC');
        return $this->db->get('work_shifts')->result();
    }

    public function addShift($data) {
        return $this->db->insert('work_shifts', $data);
    }

    public function updateShift($shift_id, $data) {
        $this->db->where('id', $shift_id);
        return $this->db->update('work_shifts', $data);
    }

    public function getStaffShiftSchedule($staff_id, $start_date, $end_date) {
        $this->db->select('sss.*, ws.name as shift_name, ws.start_time, ws.end_time');
        $this->db->from('staff_shift_schedule sss');
        $this->db->join('work_shifts ws', 'ws.id = sss.shift_id', 'left');
        $this->db->where('sss.staff_id', $staff_id);
        $this->db->where('sss.schedule_date >=', $start_date);
        $this->db->where('sss.schedule_date <=', $end_date);
        $this->db->order_by('sss.schedule_date', 'ASC');
        return $this->db->get()->result();
    }

    public function assignShiftToStaff($staff_id, $shift_id, $dates) {
        $this->db->trans_start();
        
        foreach ($dates as $date) {
            // Check if schedule already exists
            $this->db->where('staff_id', $staff_id);
            $this->db->where('schedule_date', $date);
            $existing = $this->db->get('staff_shift_schedule')->row();
            
            if ($existing) {
                // Update existing schedule
                $this->db->where('id', $existing->id);
                $this->db->update('staff_shift_schedule', [
                    'shift_id' => $shift_id,
                    'updated_at' => date('Y-m-d H:i:s')
                ]);
            } else {
                // Insert new schedule
                $this->db->insert('staff_shift_schedule', [
                    'staff_id' => $staff_id,
                    'shift_id' => $shift_id,
                    'schedule_date' => $date,
                    'created_at' => date('Y-m-d H:i:s')
                ]);
            }
        }
        
        $this->db->trans_complete();
        return $this->db->trans_status();
    }

    // =============================================
    // OVERTIME TRACKING
    // =============================================

    public function getOvertimeRecords($staff_id = null, $start_date = null, $end_date = null) {
        $this->db->select('ot.*, s.name, s.surname, s.employee_id, d.name as department_name');
        $this->db->from('overtime_records ot');
        $this->db->join('staff s', 's.id = ot.staff_id', 'left');
        $this->db->join('departments d', 'd.id = s.department_id', 'left');
        
        if ($staff_id) {
            $this->db->where('ot.staff_id', $staff_id);
        }
        
        if ($start_date && $end_date) {
            $this->db->where('ot.overtime_date >=', $start_date);
            $this->db->where('ot.overtime_date <=', $end_date);
        }
        
        $this->db->order_by('ot.overtime_date', 'DESC');
        return $this->db->get()->result();
    }

    public function addOvertimeRecord($data) {
        return $this->db->insert('overtime_records', $data);
    }

    public function updateOvertimeRecord($record_id, $data) {
        $this->db->where('id', $record_id);
        return $this->db->update('overtime_records', $data);
    }

    public function approveOvertime($record_id, $approved_by) {
        $data = [
            'status' => 'approved',
            'approved_by' => $approved_by,
            'approved_at' => date('Y-m-d H:i:s'),
            'updated_at' => date('Y-m-d H:i:s')
        ];
        
        $this->db->where('id', $record_id);
        return $this->db->update('overtime_records', $data);
    }

    // =============================================
    // ATTENDANCE SUMMARY & STATISTICS
    // =============================================

    public function getMonthlyAttendanceSummary($staff_id, $month, $year) {
        $start_date = $year . '-' . str_pad($month, 2, '0', STR_PAD_LEFT) . '-01';
        $end_date = date('Y-m-t', strtotime($start_date));
        
        $this->db->select('
            COUNT(*) as total_days,
            SUM(CASE WHEN clock_in_time IS NOT NULL THEN 1 ELSE 0 END) as present_days,
            SUM(CASE WHEN clock_in_time IS NULL THEN 1 ELSE 0 END) as absent_days,
            SUM(total_hours) as total_hours_worked,
            AVG(total_hours) as avg_hours_per_day
        ');
        $this->db->from('staff_attendance');
        $this->db->where('staff_id', $staff_id);
        $this->db->where('date >=', $start_date);
        $this->db->where('date <=', $end_date);
        
        return $this->db->get()->row();
    }

    public function getLateArrivals($staff_id = null, $start_date = null, $end_date = null) {
        $this->db->select('sa.*, s.name, s.surname, s.employee_id, ws.start_time as scheduled_start');
        $this->db->from('staff_attendance sa');
        $this->db->join('staff s', 's.id = sa.staff_id', 'left');
        $this->db->join('staff_shift_schedule sss', 'sss.staff_id = sa.staff_id AND sss.schedule_date = sa.date', 'left');
        $this->db->join('work_shifts ws', 'ws.id = sss.shift_id', 'left');
        $this->db->where('TIME(sa.clock_in_time) > TIME(ws.start_time)');
        
        if ($staff_id) {
            $this->db->where('sa.staff_id', $staff_id);
        }
        
        if ($start_date && $end_date) {
            $this->db->where('sa.date >=', $start_date);
            $this->db->where('sa.date <=', $end_date);
        }
        
        $this->db->order_by('sa.date', 'DESC');
        return $this->db->get()->result();
    }

    public function getEarlyDepartures($staff_id = null, $start_date = null, $end_date = null) {
        $this->db->select('sa.*, s.name, s.surname, s.employee_id, ws.end_time as scheduled_end');
        $this->db->from('staff_attendance sa');
        $this->db->join('staff s', 's.id = sa.staff_id', 'left');
        $this->db->join('staff_shift_schedule sss', 'sss.staff_id = sa.staff_id AND sss.schedule_date = sa.date', 'left');
        $this->db->join('work_shifts ws', 'ws.id = sss.shift_id', 'left');
        $this->db->where('TIME(sa.clock_out_time) < TIME(ws.end_time)');
        $this->db->where('sa.clock_out_time IS NOT NULL');
        
        if ($staff_id) {
            $this->db->where('sa.staff_id', $staff_id);
        }
        
        if ($start_date && $end_date) {
            $this->db->where('sa.date >=', $start_date);
            $this->db->where('sa.date <=', $end_date);
        }
        
        $this->db->order_by('sa.date', 'DESC');
        return $this->db->get()->result();
    }

    // =============================================
    // UTILITY FUNCTIONS
    // =============================================

    private function getWorkingDaysBetween($start_date, $end_date) {
        $start = new DateTime($start_date);
        $end = new DateTime($end_date);
        $end->modify('+1 day'); // Include end date
        
        $interval = new DateInterval('P1D');
        $period = new DatePeriod($start, $interval, $end);
        
        $working_days = 0;
        foreach ($period as $date) {
            // Skip weekends (Saturday = 6, Sunday = 0)
            if ($date->format('w') != 0 && $date->format('w') != 6) {
                $working_days++;
            }
        }
        
        return $working_days;
    }

    public function isHoliday($date) {
        $this->db->where('holiday_date', $date);
        $this->db->where('is_active', 'yes');
        $holiday = $this->db->get('holidays')->row();
        
        return $holiday ? true : false;
    }

    public function getHolidays($year = null) {
        if (!$year) {
            $year = date('Y');
        }
        
        $this->db->where('YEAR(holiday_date)', $year);
        $this->db->where('is_active', 'yes');
        $this->db->order_by('holiday_date', 'ASC');
        return $this->db->get('holidays')->result();
    }

    // =============================================
    // DATABASE INITIALIZATION
    // =============================================

    public function createAttendanceTables() {
        // Staff Attendance Table
        $this->db->query("
            CREATE TABLE IF NOT EXISTS `staff_attendance` (
                `id` int(11) NOT NULL AUTO_INCREMENT,
                `staff_id` int(11) NOT NULL,
                `date` date NOT NULL,
                `clock_in_time` datetime DEFAULT NULL,
                `clock_out_time` datetime DEFAULT NULL,
                `total_hours` decimal(4,2) DEFAULT NULL,
                `break_hours` decimal(4,2) DEFAULT 0.00,
                `overtime_hours` decimal(4,2) DEFAULT 0.00,
                `clock_in_location` varchar(255) DEFAULT NULL,
                `clock_out_location` varchar(255) DEFAULT NULL,
                `clock_in_notes` text,
                `clock_out_notes` text,
                `status` enum('present','absent','late','half_day','on_leave') DEFAULT 'present',
                `approved_by` int(11) DEFAULT NULL,
                `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
                `updated_at` timestamp NULL DEFAULT NULL ON UPDATE CURRENT_TIMESTAMP,
                PRIMARY KEY (`id`),
                UNIQUE KEY `staff_date` (`staff_id`,`date`),
                KEY `staff_id` (`staff_id`),
                KEY `date` (`date`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8;
        ");

        // Work Shifts Table
        $this->db->query("
            CREATE TABLE IF NOT EXISTS `work_shifts` (
                `id` int(11) NOT NULL AUTO_INCREMENT,
                `name` varchar(100) NOT NULL,
                `short_code` varchar(20) NOT NULL,
                `start_time` time NOT NULL,
                `end_time` time NOT NULL,
                `break_duration` int(11) DEFAULT 60 COMMENT 'Break duration in minutes',
                `total_hours` decimal(4,2) NOT NULL,
                `is_night_shift` enum('yes','no') DEFAULT 'no',
                `overtime_rate` decimal(5,2) DEFAULT 1.5,
                `is_active` enum('yes','no') DEFAULT 'yes',
                `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
                `updated_at` timestamp NULL DEFAULT NULL ON UPDATE CURRENT_TIMESTAMP,
                PRIMARY KEY (`id`),
                UNIQUE KEY `short_code` (`short_code`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8;
        ");

        // Staff Shift Schedule Table
        $this->db->query("
            CREATE TABLE IF NOT EXISTS `staff_shift_schedule` (
                `id` int(11) NOT NULL AUTO_INCREMENT,
                `staff_id` int(11) NOT NULL,
                `shift_id` int(11) NOT NULL,
                `schedule_date` date NOT NULL,
                `is_active` enum('yes','no') DEFAULT 'yes',
                `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
                `updated_at` timestamp NULL DEFAULT NULL ON UPDATE CURRENT_TIMESTAMP,
                PRIMARY KEY (`id`),
                UNIQUE KEY `staff_schedule_date` (`staff_id`,`schedule_date`),
                KEY `staff_id` (`staff_id`),
                KEY `shift_id` (`shift_id`),
                KEY `schedule_date` (`schedule_date`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8;
        ");

        // Overtime Records Table
        $this->db->query("
            CREATE TABLE IF NOT EXISTS `overtime_records` (
                `id` int(11) NOT NULL AUTO_INCREMENT,
                `staff_id` int(11) NOT NULL,
                `overtime_date` date NOT NULL,
                `start_time` time NOT NULL,
                `end_time` time NOT NULL,
                `total_hours` decimal(4,2) NOT NULL,
                `overtime_rate` decimal(5,2) DEFAULT 1.5,
                `reason` text,
                `status` enum('pending','approved','rejected') DEFAULT 'pending',
                `approved_by` int(11) DEFAULT NULL,
                `approved_at` datetime DEFAULT NULL,
                `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
                `updated_at` timestamp NULL DEFAULT NULL ON UPDATE CURRENT_TIMESTAMP,
                PRIMARY KEY (`id`),
                KEY `staff_id` (`staff_id`),
                KEY `overtime_date` (`overtime_date`),
                KEY `status` (`status`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8;
        ");

        // Holidays Table
        $this->db->query("
            CREATE TABLE IF NOT EXISTS `holidays` (
                `id` int(11) NOT NULL AUTO_INCREMENT,
                `name` varchar(255) NOT NULL,
                `holiday_date` date NOT NULL,
                `description` text,
                `is_recurring` enum('yes','no') DEFAULT 'no',
                `is_active` enum('yes','no') DEFAULT 'yes',
                `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
                `updated_at` timestamp NULL DEFAULT NULL ON UPDATE CURRENT_TIMESTAMP,
                PRIMARY KEY (`id`),
                KEY `holiday_date` (`holiday_date`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8;
        ");

        return true;
    }

    public function insertDefaultShifts() {
        $shifts = [
            ['name' => 'Morning Shift', 'short_code' => 'MORNING', 'start_time' => '08:00:00', 'end_time' => '17:00:00', 'total_hours' => 8.00],
            ['name' => 'Evening Shift', 'short_code' => 'EVENING', 'start_time' => '14:00:00', 'end_time' => '22:00:00', 'total_hours' => 8.00],
            ['name' => 'Night Shift', 'short_code' => 'NIGHT', 'start_time' => '22:00:00', 'end_time' => '06:00:00', 'total_hours' => 8.00, 'is_night_shift' => 'yes'],
            ['name' => '12 Hour Day', 'short_code' => '12HOUR_DAY', 'start_time' => '06:00:00', 'end_time' => '18:00:00', 'total_hours' => 12.00],
            ['name' => '12 Hour Night', 'short_code' => '12HOUR_NIGHT', 'start_time' => '18:00:00', 'end_time' => '06:00:00', 'total_hours' => 12.00, 'is_night_shift' => 'yes']
        ];

        foreach ($shifts as $shift) {
            // Check if shift already exists
            $this->db->where('short_code', $shift['short_code']);
            $existing = $this->db->get('work_shifts')->row();
            
            if (!$existing) {
                $this->db->insert('work_shifts', $shift);
            }
        }

        return true;
    }

    public function insertDefaultHolidays($year = null) {
        if (!$year) {
            $year = date('Y');
        }

        $holidays = [
            ['name' => 'New Year\'s Day', 'holiday_date' => $year . '-01-01', 'is_recurring' => 'yes'],
            ['name' => 'Labour Day', 'holiday_date' => $year . '-05-01', 'is_recurring' => 'yes'],
            ['name' => 'Madaraka Day', 'holiday_date' => $year . '-06-01', 'is_recurring' => 'yes'],
            ['name' => 'Mashujaa Day', 'holiday_date' => $year . '-10-20', 'is_recurring' => 'yes'],
            ['name' => 'Jamhuri Day', 'holiday_date' => $year . '-12-12', 'is_recurring' => 'yes'],
            ['name' => 'Christmas Day', 'holiday_date' => $year . '-12-25', 'is_recurring' => 'yes'],
            ['name' => 'Boxing Day', 'holiday_date' => $year . '-12-26', 'is_recurring' => 'yes']
        ];

        foreach ($holidays as $holiday) {
            // Check if holiday already exists
            $this->db->where('holiday_date', $holiday['holiday_date']);
            $existing = $this->db->get('holidays')->row();
            
            if (!$existing) {
                $this->db->insert('holidays', $holiday);
            }
        }

        return true;
    }
}