<?php

if (!defined('BASEPATH'))
    exit('No direct script access allowed');

class Financial_model extends CI_Model {

    public function __construct() {
        parent::__construct();
        $this->load->database();
    }
  /**
     * Fixed getTrialBalance method with correct SQL syntax
     */
    public function getTrialBalance($as_of_date = null) {
        if ($as_of_date === null) {
            $as_of_date = date('Y-m-d');
        }

        try {
            // FIXED: Removed misplaced parentheses in WHERE clause
            $sql = "
                SELECT 
                    coa.account_code,
                    coa.account_name,
                    coa.account_type,
                    COALESCE(SUM(jl.debit_amount), 0) as total_debits,
                    COALESCE(SUM(jl.credit_amount), 0) as total_credits,
                    CASE 
                        WHEN coa.account_type IN ('asset', 'expense') THEN 
                            GREATEST((coa.opening_balance + COALESCE(SUM(jl.debit_amount - jl.credit_amount), 0)), 0)
                        ELSE 0 
                    END as debit_balance,
                    CASE 
                        WHEN coa.account_type IN ('liability', 'equity', 'revenue') THEN 
                            GREATEST((coa.opening_balance + COALESCE(SUM(jl.credit_amount - jl.debit_amount), 0)), 0)
                        WHEN coa.account_type IN ('asset', 'expense') AND 
                             (coa.opening_balance + COALESCE(SUM(jl.debit_amount - jl.credit_amount), 0)) < 0 THEN 
                            ABS((coa.opening_balance + COALESCE(SUM(jl.debit_amount - jl.credit_amount), 0)))
                        ELSE 0 
                    END as credit_balance
                FROM chart_of_accounts coa
                LEFT JOIN journal_lines jl ON jl.account_id = coa.id
                LEFT JOIN journal_headers jh ON jh.id = jl.journal_id
                WHERE coa.is_active = 1
                    AND (jh.date IS NULL OR jh.date <= ?)
                    AND (jh.status IS NULL OR jh.status = 'posted')
                GROUP BY coa.id, coa.account_code, coa.account_name, coa.account_type, coa.opening_balance
                HAVING debit_balance > 0 OR credit_balance > 0
                ORDER BY coa.account_code ASC
            ";

            $query = $this->db->query($sql, array($as_of_date));
            $accounts = $query->result_array();

            // Initialize account types
            $account_types = [
                'asset' => ['name' => 'Assets', 'debit' => 0, 'credit' => 0],
                'liability' => ['name' => 'Liabilities', 'debit' => 0, 'credit' => 0],
                'equity' => ['name' => 'Equity', 'debit' => 0, 'credit' => 0],
                'revenue' => ['name' => 'Revenue', 'debit' => 0, 'credit' => 0],
                'expense' => ['name' => 'Expenses', 'debit' => 0, 'credit' => 0]
            ];

            // Calculate totals
            $total_debit = 0;
            $total_credit = 0;

            foreach ($accounts as $account) {
                $type = $account['account_type'];
                if (isset($account_types[$type])) {
                    $account_types[$type]['debit'] += floatval($account['debit_balance']);
                    $account_types[$type]['credit'] += floatval($account['credit_balance']);
                }
                
                $total_debit += floatval($account['debit_balance']);
                $total_credit += floatval($account['credit_balance']);
            }

            // Check if balanced
            $difference = abs($total_debit - $total_credit);
            $is_balanced = $difference < 0.01; // Allow for small rounding differences

            return [
                'accounts' => $accounts,
                'account_types' => $account_types,
                'total_debit' => $total_debit,
                'total_credit' => $total_credit,
                'difference' => $difference,
                'is_balanced' => $is_balanced
            ];

        } catch (Exception $e) {
            log_message('error', 'Trial Balance SQL Error: ' . $e->getMessage());
            
            // Return empty structure on error
            return [
                'accounts' => [],
                'account_types' => [
                    'asset' => ['name' => 'Assets', 'debit' => 0, 'credit' => 0],
                    'liability' => ['name' => 'Liabilities', 'debit' => 0, 'credit' => 0],
                    'equity' => ['name' => 'Equity', 'debit' => 0, 'credit' => 0],
                    'revenue' => ['name' => 'Revenue', 'debit' => 0, 'credit' => 0],
                    'expense' => ['name' => 'Expenses', 'debit' => 0, 'credit' => 0]
                ],
                'total_debit' => 0,
                'total_credit' => 0,
                'difference' => 0,
                'is_balanced' => true
            ];
        }
    }
    
    /**
 * Create financial entries for pharmacy bills
 * This method should be added to your Financial_model class
 */
public function create_pharmacy_financial_entries($bill_id, $bill_data) {
    $CI = &get_instance();
    
    try {
        log_message('info', "Creating pharmacy financial entries for bill ID: {$bill_id}");
        
        // Get required account IDs with fallback for different account codes
        $accounts = $this->getPharmacyAccountIdsFixed();
        
        if (!$accounts) {
            log_message('error', 'Required pharmacy accounts not found');
            return false;
        }
        
        // Extract bill data
        $net_amount = floatval($bill_data['net_amount']);
        $payment_status = $bill_data['payment_status'];
        $patient_id = $bill_data['patient_id'];
        
        // Generate reference number
        $reference_no = 'PHARM-' . date('Ymd') . '-' . $bill_id;
        
        // Create journal header
        $journal_data = [
            'reference_no' => $reference_no,
            'date' => date('Y-m-d'),
            'description' => "Pharmacy Bill - Patient ID: {$patient_id}, Bill ID: {$bill_id}",
            'total_amount' => $net_amount,
            'source_module' => 'pharmacy',
            'source_id' => $bill_id,
            'patient_id' => $patient_id,
            'status' => 'posted',
            'created_by' => $this->getStaffIdSafe(),
            'journal_type' => 'revenue',
            'currency_code' => 'KES',
            'created_at' => date('Y-m-d H:i:s'),
            'updated_at' => date('Y-m-d H:i:s')
        ];
        
        $CI->db->insert('journal_headers', $journal_data);
        $journal_id = $CI->db->insert_id();
        
        if (!$journal_id) {
            log_message('error', 'Failed to create journal header for pharmacy bill');
            return false;
        }
        
        // Create journal lines
        $journal_lines = [];
        
        if ($payment_status === 'paid') {
            // DEBIT: Cash (Asset increases)
            $journal_lines[] = [
                'journal_id' => $journal_id,
                'account_id' => $accounts['cash_account_id'],
                'description' => 'Cash received from pharmacy sales',
                'debit_amount' => $net_amount,
                'credit_amount' => 0.00,
                'created_at' => date('Y-m-d H:i:s')
            ];
        } else {
            // DEBIT: Accounts Receivable (Asset increases) 
            $journal_lines[] = [
                'journal_id' => $journal_id,
                'account_id' => $accounts['receivable_account_id'],
                'description' => 'Outstanding payment for pharmacy sales',
                'debit_amount' => $net_amount,
                'credit_amount' => 0.00,
                'created_at' => date('Y-m-d H:i:s')
            ];
        }
        
        // CREDIT: Pharmacy Revenue (Revenue increases)
        $journal_lines[] = [
            'journal_id' => $journal_id,
            'account_id' => $accounts['pharmacy_revenue_id'],
            'description' => 'Pharmacy medicine sales revenue',
            'debit_amount' => 0.00,
            'credit_amount' => $net_amount,
            'created_at' => date('Y-m-d H:i:s')
        ];
        
        // Insert journal lines
        foreach ($journal_lines as $line) {
            $CI->db->insert('journal_lines', $line);
        }
        
        // Create general ledger entries
        foreach ($journal_lines as $line) {
            $gl_entry = [
                'transaction_date' => date('Y-m-d'),
                'account_id' => $line['account_id'],
                'reference_type' => 'pharmacy_bill',
                'reference_id' => $bill_id,
                'description' => $line['description'],
                'debit_amount' => $line['debit_amount'],
                'credit_amount' => $line['credit_amount'],
                'posted_by' => $this->getStaffIdSafe(),
                'created_at' => date('Y-m-d H:i:s')
            ];
            
            $CI->db->insert('general_ledger', $gl_entry);
        }
        
        // Create financial transaction record
        $financial_transaction = [
            'transaction_date' => date('Y-m-d'),
            'transaction_type' => 'revenue',
            'category' => 'pharmacy',
            'subcategory' => 'medicine_sales',
            'amount' => $net_amount,
            'description' => "Pharmacy medicine sales - Bill #{$bill_id}",
            'reference_type' => 'pharmacy_bill',
            'reference_id' => $bill_id,
            'patient_id' => $patient_id,
            'department' => 'pharmacy',
            'payment_method' => $payment_status === 'paid' ? 'cash' : 'credit',
            'receipt_number' => 'MDB' . str_pad($bill_id, 6, '0', STR_PAD_LEFT),
            'processed_by' => $this->getStaffIdSafe(),
            'created_at' => date('Y-m-d H:i:s')
        ];
        
        $CI->db->insert('financial_transactions', $financial_transaction);
        
        log_message('info', "Successfully created pharmacy financial entries for bill: {$bill_id}");
        return true;
        
    } catch (Exception $e) {
        log_message('error', "Error creating pharmacy financial entries: " . $e->getMessage());
        return false;
    }
}

/**
 * Get pharmacy account IDs with proper handling for different account setups
 */
private function getPharmacyAccountIdsFixed() {
    $CI = &get_instance();
    
    try {
        $accounts = [];
        
        // Cash account (1001)
        $CI->db->select('id');
        $CI->db->where('account_code', '1001');
        $CI->db->where('is_active', 1);
        $cash_account = $CI->db->get('chart_of_accounts')->row();
        
        if ($cash_account) {
            $accounts['cash_account_id'] = $cash_account->id;
        } else {
            log_message('error', "Cash account (1001) not found");
            return false;
        }
        
        // Pharmacy Revenue account (4005)
        $CI->db->select('id');
        $CI->db->where('account_code', '4005');
        $CI->db->where('is_active', 1);
        $revenue_account = $CI->db->get('chart_of_accounts')->row();
        
        if ($revenue_account) {
            $accounts['pharmacy_revenue_id'] = $revenue_account->id;
        } else {
            log_message('error', "Pharmacy revenue account (4005) not found");
            return false;
        }
        
        // Accounts Receivable - try 1002 first, then 2001
        $CI->db->select('id, account_type');
        $CI->db->where('account_code', '1200');
        $CI->db->where('is_active', 1);
        $receivable_account = $CI->db->get('chart_of_accounts')->row();
        
        if ($receivable_account && $receivable_account->account_type === 'asset') {
            $accounts['receivable_account_id'] = $receivable_account->id;
        } else {
            // Fallback to 2001 if it exists and warn about the issue
            $CI->db->select('id, account_type');
            $CI->db->where('account_code', '1200');
            $CI->db->where('is_active', 1);
            $fallback_account = $CI->db->get('chart_of_accounts')->row();
            
            if ($fallback_account) {
                $accounts['receivable_account_id'] = $fallback_account->id;
                log_message('warning', "Using account 1200 as receivable account, but it's type: " . $fallback_account->account_type);
            } else {
                log_message('error', "No receivable account found (tried 1002 and 1200)");
                return false;
            }
        }
        
        return $accounts;
        
    } catch (Exception $e) {
        log_message('error', 'Error getting pharmacy account IDs: ' . $e->getMessage());
        return false;
    }
}

private function getStaffIdSafe() {
    $CI = &get_instance();
    
    // Try session first
    $session_data = $CI->session->userdata('admin');
    if (isset($session_data['id'])) {
        return $session_data['id'];
    }
    
    // Try alternative session keys
    $user_data = $CI->session->userdata('user');
    if (isset($user_data['id'])) {
        return $user_data['id'];
    }
    
    // Default fallback
    return 1;
}
/**
 * Get pharmacy account IDs from chart of accounts
 */
private function getPharmacyAccountIds() {
    try {
        $accounts = [];
        
        // Map of account codes to variable names
        $account_mapping = [
            '1001' => 'cash_account_id',
            '4005' => 'pharmacy_revenue_id', 
            '1200' => 'receivable_account_id',
            '1201' => 'cogs_account_id'
        ];
        
        foreach ($account_mapping as $code => $key) {
            $this->db->select('id');
            $this->db->where('account_code', $code);
            $this->db->where('is_active', 1);
            $account = $this->db->get('chart_of_accounts')->row();
            
            if ($account) {
                $accounts[$key] = $account->id;
            } else {
                log_message('error', "Pharmacy account with code {$code} not found");
                return false;
            }
        }
        
        return $accounts;
        
    } catch (Exception $e) {
        log_message('error', 'Error getting pharmacy account IDs: ' . $e->getMessage());
        return false;
    }
}

/**
 * Create discount entry in financial records
 */
private function createDiscountEntry($journal_id, $bill_id, $discount_amount, $accounts) {
    try {
        // DEBIT: Discount Expense (if you have a discount account)
        // This reduces the revenue effectively
        
        // For now, we'll adjust the revenue entry
        // In a more complex system, you might have a separate discount account
        
        log_message('info', "Discount of {$discount_amount} applied to bill {$bill_id}");
        
        // You can add specific discount handling here if needed
        return true;
        
    } catch (Exception $e) {
        log_message('error', 'Error creating discount entry: ' . $e->getMessage());
        return false;
    }
}

/**
 * Create tax entry in financial records
 */
private function createTaxEntry($journal_id, $bill_id, $tax_amount, $accounts) {
    try {
        // You might want to create a separate entry for tax liability
        // CREDIT: Tax Payable (Liability account)
        
        log_message('info', "Tax of {$tax_amount} applied to bill {$bill_id}");
        
        // You can add specific tax handling here if needed
        return true;
        
    } catch (Exception $e) {
        log_message('error', 'Error creating tax entry: ' . $e->getMessage());
        return false;
    }
}

/**
 * Handle payment receipt for credit sales
 * Call this when a patient pays for a previously unpaid pharmacy bill
 */
public function record_pharmacy_payment($bill_id, $payment_amount, $payment_method = 'cash') {
    try {
        log_message('info', "Recording pharmacy payment for bill ID: {$bill_id}, Amount: {$payment_amount}");
        
        $accounts = $this->getPharmacyAccountIds();
        
        if (!$accounts) {
            return false;
        }
        
        // Generate reference number
        $reference_no = 'PHARM-PMT-' . date('Ymd') . '-' . $bill_id;
        
        // Create journal header for payment
        $journal_data = [
            'reference_no' => $reference_no,
            'date' => date('Y-m-d'),
            'description' => "Payment received for Pharmacy Bill ID: {$bill_id}",
            'total_amount' => $payment_amount,
            'source_module' => 'pharmacy_payment',
            'source_id' => $bill_id,
            'status' => 'posted',
            'created_by' => $this->getStaffId(),
            'journal_type' => 'payment'
        ];
        
        $this->db->insert('journal_headers', $journal_data);
        $journal_id = $this->db->insert_id();
        
        // Journal lines for payment
        $journal_lines = [
            [
                'journal_id' => $journal_id,
                'account_id' => $accounts['cash_account_id'],
                'description' => 'Cash received for pharmacy bill payment',
                'debit_amount' => $payment_amount,
                'credit_amount' => 0.00
            ],
            [
                'journal_id' => $journal_id,
                'account_id' => $accounts['receivable_account_id'],
                'description' => 'Payment against accounts receivable',
                'debit_amount' => 0.00,
                'credit_amount' => $payment_amount
            ]
        ];
        
        // Insert journal lines
        foreach ($journal_lines as $line) {
            $this->db->insert('journal_lines', $line);
        }
        
        // Create general ledger entries
        foreach ($journal_lines as $line) {
            $gl_entry = [
                'transaction_date' => date('Y-m-d'),
                'account_id' => $line['account_id'],
                'reference_type' => 'pharmacy_payment',
                'reference_id' => $bill_id,
                'description' => $line['description'],
                'debit_amount' => $line['debit_amount'],
                'credit_amount' => $line['credit_amount'],
                'posted_by' => $this->getStaffId(),
                'created_at' => date('Y-m-d H:i:s')
            ];
            
            $this->db->insert('general_ledger', $gl_entry);
        }
        
        log_message('info', "Successfully recorded pharmacy payment for bill: {$bill_id}");
        return true;
        
    } catch (Exception $e) {
        log_message('error', "Error recording pharmacy payment: " . $e->getMessage());
        return false;
    }
}

/**
 * Ensure required pharmacy accounts exist
 */
public function ensure_pharmacy_accounts() {
    try {
        $required_accounts = [
            [
                'account_code' => '1001',
                'account_name' => 'Cash',
                'account_type' => 'asset',
                'description' => 'Cash received from pharmacy sales',
                'opening_balance' => 0.00
            ],
            [
                'account_code' => '4005',
                'account_name' => 'Pharmacy Revenue',
                'account_type' => 'revenue',
                'description' => 'Revenue from pharmacy medicine sales',
                'opening_balance' => 0.00
            ],
            [
                'account_code' => '1200',
                'account_name' => 'Accounts Receivable',
                'account_type' => 'asset',
                'description' => 'Outstanding payments from patients',
                'opening_balance' => 0.00
            ],
            [
                'account_code' => '1201',
                'account_name' => 'Cost of Goods Sold - Pharmacy',
                'account_type' => 'expense',
                'description' => 'Cost of medicines sold',
                'opening_balance' => 0.00
            ]
        ];
        
        foreach ($required_accounts as $account) {
            $this->db->where('account_code', $account['account_code']);
            $existing = $this->db->get('chart_of_accounts')->row();
            
            if (!$existing) {
                $account['is_active'] = 1;
                $account['created_at'] = date('Y-m-d H:i:s');
                
                $this->db->insert('chart_of_accounts', $account);
                log_message('info', "Created pharmacy account: {$account['account_code']} - {$account['account_name']}");
            }
        }
        
        return true;
        
    } catch (Exception $e) {
        log_message('error', 'Error ensuring pharmacy accounts: ' . $e->getMessage());
        return false;
    }
}

    /**
     * Alternative method using your existing trial_balance_view
     * This is the simplest approach since you already have the view
     */
    public function getTrialBalanceUsingView($as_of_date = null) {
        if ($as_of_date === null) {
            $as_of_date = date('Y-m-d');
        }

        try {
            // First, get all accounts that should be included as of the specified date
            $sql = "
                SELECT 
                    account_code,
                    account_name,
                    account_type,
                    debit_balance,
                    credit_balance
                FROM trial_balance_view
                ORDER BY account_code ASC
            ";

            $query = $this->db->query($sql);
            $accounts = $query->result_array();

            // Now filter out transactions after the as_of_date
            if ($as_of_date < date('Y-m-d')) {
                // Recalculate balances excluding transactions after as_of_date
                foreach ($accounts as &$account) {
                    $account_id_query = $this->db->select('id')
                                                ->from('chart_of_accounts')
                                                ->where('account_code', $account['account_code'])
                                                ->get();
                    
                    if ($account_id_query->num_rows() > 0) {
                        $account_id = $account_id_query->row()->id;
                        
                        // Get transactions after the as_of_date
                        $future_transactions_sql = "
                            SELECT 
                                COALESCE(SUM(jl.debit_amount), 0) as future_debits,
                                COALESCE(SUM(jl.credit_amount), 0) as future_credits
                            FROM journal_lines jl
                            JOIN journal_headers jh ON jh.id = jl.journal_id
                            WHERE jl.account_id = ?
                                AND jh.date > ?
                                AND jh.status = 'posted'
                        ";
                        
                        $future_query = $this->db->query($future_transactions_sql, array($account_id, $as_of_date));
                        $future = $future_query->row_array();
                        
                        // Adjust balances
                        if ($account['account_type'] == 'asset' || $account['account_type'] == 'expense') {
                            $adjustment = $future['future_debits'] - $future['future_credits'];
                            $account['debit_balance'] = max(0, $account['debit_balance'] - $adjustment);
                            if ($account['debit_balance'] == 0 && $adjustment > 0) {
                                $account['credit_balance'] = $adjustment;
                            }
                        } else {
                            $adjustment = $future['future_credits'] - $future['future_debits'];
                            $account['credit_balance'] = max(0, $account['credit_balance'] - $adjustment);
                            if ($account['credit_balance'] == 0 && $adjustment > 0) {
                                $account['debit_balance'] = $adjustment;
                            }
                        }
                    }
                }
            }

            // Filter out zero balances
            $accounts = array_filter($accounts, function($account) {
                return $account['debit_balance'] > 0 || $account['credit_balance'] > 0;
            });

            // Group by account type and calculate totals
            $account_types = [
                'asset' => ['name' => 'Assets', 'debit' => 0, 'credit' => 0],
                'liability' => ['name' => 'Liabilities', 'debit' => 0, 'credit' => 0],
                'equity' => ['name' => 'Equity', 'debit' => 0, 'credit' => 0],
                'revenue' => ['name' => 'Revenue', 'debit' => 0, 'credit' => 0],
                'expense' => ['name' => 'Expenses', 'debit' => 0, 'credit' => 0]
            ];

            $total_debit = 0;
            $total_credit = 0;

            foreach ($accounts as $account) {
                $type = $account['account_type'];
                if (isset($account_types[$type])) {
                    $account_types[$type]['debit'] += floatval($account['debit_balance']);
                    $account_types[$type]['credit'] += floatval($account['credit_balance']);
                }
                
                $total_debit += floatval($account['debit_balance']);
                $total_credit += floatval($account['credit_balance']);
            }

            // Check if balanced
            $difference = abs($total_debit - $total_credit);
            $is_balanced = $difference < 0.01;

            return [
                'accounts' => $accounts,
                'account_types' => $account_types,
                'total_debit' => $total_debit,
                'total_credit' => $total_credit,
                'difference' => $difference,
                'is_balanced' => $is_balanced
            ];

        } catch (Exception $e) {
            log_message('error', 'Trial Balance View Error: ' . $e->getMessage());
            
            // Return empty structure on error
            return [
                'accounts' => [],
                'account_types' => [
                    'asset' => ['name' => 'Assets', 'debit' => 0, 'credit' => 0],
                    'liability' => ['name' => 'Liabilities', 'debit' => 0, 'credit' => 0],
                    'equity' => ['name' => 'Equity', 'debit' => 0, 'credit' => 0],
                    'revenue' => ['name' => 'Revenue', 'debit' => 0, 'credit' => 0],
                    'expense' => ['name' => 'Expenses', 'debit' => 0, 'credit' => 0]
                ],
                'total_debit' => 0,
                'total_credit' => 0,
                'difference' => 0,
                'is_balanced' => true
            ];
        }
    }

    /**
     * Check if financial system is properly set up
     */
   /* public function isSystemSetup() {
        // Check if required tables exist
        $required_tables = ['chart_of_accounts', 'journal_headers', 'journal_lines'];
        
        foreach ($required_tables as $table) {
            if (!$this->db->table_exists($table)) {
                return false;
            }
        }
        
        // Check if there are some accounts in chart of accounts
        $count = $this->db->count_all('chart_of_accounts');
        return $count > 0;
    }*/

    
public function create_gcc_financial_entries($patient_id) {
        log_message('info', "FINANCIAL DEBUG - Starting GCC financial entries for patient: {$patient_id}");
        
        // Start transaction
        $this->db->trans_begin();

        try {
            // Fetch patient details
            $this->db->select('patient_name, patient_unique_id, patient_type, patient_workflow');
            $patient = $this->db->get_where('patients', ['id' => $patient_id])->row_array();
            
            if (!$patient) {
                throw new Exception("Patient not found with ID: {$patient_id}");
            }

            log_message('info', "FINANCIAL DEBUG - Patient data: " . json_encode($patient));

            $patient_name = $patient['patient_name'] ?? 'Unknown Patient';
            $patient_unique_id = $patient['patient_unique_id'] ?? 'PAT-' . $patient_id;
            $patient_type = strtolower($patient['patient_type'] ?? '');
            $workflow_type = strtolower($patient['patient_workflow'] ?? '');

            log_message('info', "FINANCIAL DEBUG - Patient type: '{$patient_type}', Workflow: '{$workflow_type}'");

            // Check if this is a GCC patient (multiple ways to verify)
            $is_gcc = ($workflow_type === 'gcc' || $patient_type === 'gcc' || $patient_type === 'GCC');
            
            if (!$is_gcc) {
                log_message('info', "FINANCIAL DEBUG - Not a GCC patient, skipping financial entries");
                return true; // Not an error, just not applicable
            }

            // Ensure accounts exist
            $this->ensure_gcc_accounts();

            // Get account IDs with detailed logging
            $cash_id = $this->get_account_id('1001');
            $consultation_id = $this->get_account_id('4001');
            $lab_id = $this->get_account_id('4006');
            $radiology_id = $this->get_account_id('4004');

            log_message('info', "FINANCIAL DEBUG - Account IDs - Cash: {$cash_id}, Consultation: {$consultation_id}, Lab: {$lab_id}, Radiology: {$radiology_id}");

            if (!$cash_id || !$consultation_id) {
                throw new Exception("Required chart of accounts not found. Cash: {$cash_id}, Consultation: {$consultation_id}");
            }

            // Use fallback accounts if lab/radiology not found
            if (!$lab_id) {
                $lab_id = $consultation_id;
                log_message('info', "FINANCIAL DEBUG - Using consultation account for lab revenue");
            }
            if (!$radiology_id) {
                $radiology_id = $consultation_id;
                log_message('info', "FINANCIAL DEBUG - Using consultation account for radiology revenue");
            }

            // Define amounts
            $total_amount = 10500.00;
            $consultation_amount = 1500.00;
            $lab_amount = 8000.00;
            $radiology_amount = 1000.00;
            $reference_no = 'GCC-' . date('Ymd') . '-' . $patient_id;
            $user_id = $this->session->userdata('admin_id') ?: 1;

            log_message('info', "FINANCIAL DEBUG - Amounts - Total: {$total_amount}, Consultation: {$consultation_amount}, Lab: {$lab_amount}, Radiology: {$radiology_amount}");

            // Insert journal header
            $journal_data = [
                'reference_no' => $reference_no,
                'date' => date('Y-m-d'),
                'description' => "GCC Registration - Patient: {$patient_name} (ID: {$patient_unique_id})",
                'total_amount' => $total_amount,
                'source_module' => 'gcc_registration',
                'source_id' => $patient_id,
                'patient_id' => $patient_id,
                'status' => 'posted',
                'created_by' => $user_id,
                'created_at' => date('Y-m-d H:i:s'),
            ];

            log_message('info', "FINANCIAL DEBUG - Inserting journal header: " . json_encode($journal_data));
            $this->db->insert('journal_headers', $journal_data);
            
            if ($this->db->affected_rows() === 0) {
                throw new Exception('Failed to insert journal header: ' . $this->db->error()['message']);
            }
            
            $journal_id = $this->db->insert_id();
            log_message('info', "FINANCIAL DEBUG - Journal header created with ID: {$journal_id}");

            // Prepare journal lines
            $journal_lines = [
                [
                    'journal_id' => $journal_id,
                    'account_id' => $cash_id,
                    'debit_amount' => $total_amount,
                    'credit_amount' => 0.00,
                    'description' => 'Cash received for GCC services',
                    'created_at' => date('Y-m-d H:i:s')
                ],
                [
                    'journal_id' => $journal_id,
                    'account_id' => $consultation_id,
                    'debit_amount' => 0.00,
                    'credit_amount' => $consultation_amount,
                    'description' => 'GCC Consultation Revenue',
                    'created_at' => date('Y-m-d H:i:s')
                ],
                [
                    'journal_id' => $journal_id,
                    'account_id' => $lab_id,
                    'debit_amount' => 0.00,
                    'credit_amount' => $lab_amount,
                    'description' => 'GCC Laboratory Revenue',
                    'created_at' => date('Y-m-d H:i:s')
                ],
                [
                    'journal_id' => $journal_id,
                    'account_id' => $radiology_id,
                    'debit_amount' => 0.00,
                    'credit_amount' => $radiology_amount,
                    'description' => 'GCC Radiology Revenue',
                    'created_at' => date('Y-m-d H:i:s')
                ]
            ];

            // Insert journal lines
            foreach ($journal_lines as $line) {
                log_message('info', "FINANCIAL DEBUG - Inserting journal line: " . json_encode($line));
                $this->db->insert('journal_lines', $line);
                
                if ($this->db->affected_rows() === 0) {
                    throw new Exception('Failed to insert journal line: ' . $this->db->error()['message']);
                }
            }

            // Insert general ledger entries
            foreach ($journal_lines as $line) {
                $gl_data = [
                    'transaction_date' => date('Y-m-d'),
                    'account_id' => $line['account_id'],
                    'reference_type' => 'gcc_registration',
                    'reference_id' => $patient_id,
                    'description' => $line['description'],
                    'debit_amount' => $line['debit_amount'],
                    'credit_amount' => $line['credit_amount'],
                    'posted_by' => $user_id,
                    'created_at' => date('Y-m-d H:i:s'),
                ];
                
                log_message('info', "FINANCIAL DEBUG - Inserting general ledger: " . json_encode($gl_data));
                $this->db->insert('general_ledger', $gl_data);
                
                if ($this->db->affected_rows() === 0) {
                    throw new Exception('Failed to insert general ledger entry: ' . $this->db->error()['message']);
                }
            }

            // Insert financial transactions
            $financial_txns = [
                ['subcategory' => 'consultation', 'amount' => $consultation_amount, 'department' => 'OPD'],
                ['subcategory' => 'laboratory', 'amount' => $lab_amount, 'department' => 'Laboratory'],
                ['subcategory' => 'radiology', 'amount' => $radiology_amount, 'department' => 'Radiology'],
            ];

            foreach ($financial_txns as $txn) {
                $ft_data = [
                    'transaction_date' => date('Y-m-d H:i:s'),
                    'transaction_type' => 'revenue',
                    'category' => 'medical_services',
                    'subcategory' => $txn['subcategory'],
                    'amount' => $txn['amount'],
                    'description' => 'GCC ' . ucfirst($txn['subcategory']) . ' Services',
                    'reference_table' => 'patients',
                    'reference_id' => $patient_id,
                    'patient_id' => $patient_unique_id,
                    'department' => $txn['department'],
                    'payment_method' => 'cash',
                    'receipt_number' => $reference_no,
                    'created_by' => $user_id,
                    'created_at' => date('Y-m-d H:i:s'),
                ];
                
                log_message('info', "FINANCIAL DEBUG - Inserting financial transaction: " . json_encode($ft_data));
                $this->db->insert('financial_transactions', $ft_data);
                
                if ($this->db->affected_rows() === 0) {
                    throw new Exception('Failed to insert financial transaction: ' . $this->db->error()['message']);
                }
            }

            // Commit transaction
            $this->db->trans_commit();
            log_message('info', "FINANCIAL DEBUG - Successfully created all GCC financial entries for patient: {$patient_id}");
            return true;

        } catch (Exception $e) {
            $this->db->trans_rollback();
            log_message('error', 'FINANCIAL DEBUG - GCC Financial Entry Error: ' . $e->getMessage());
            log_message('error', 'FINANCIAL DEBUG - Database error: ' . $this->db->error()['message']);

            // Log to error table if it exists
            if ($this->db->table_exists('error_log')) {
                $this->db->insert('error_log', [
                    'error_type' => 'GCC_FINANCIAL_ENTRY',
                    'error_message' => $e->getMessage(),
                    'patient_id' => $patient_id,
                    'created_at' => date('Y-m-d H:i:s')
                ]);
            }
            
            return false;
        }
    }
    
    /**
     * Create financial entries for general workflow patients
     */
    public function create_general_financial_entries($patient_id, $amount = 900.00) {
        log_message('info', "FINANCIAL DEBUG - Creating general financial entries for patient: {$patient_id}, Amount: {$amount}");
        
        $this->db->trans_begin();

        try {
            // Ensure basic accounts exist
            $this->ensure_general_accounts();

            // Get patient details
            $patient = $this->db->get_where('patients', ['id' => $patient_id])->row_array();
            if (!$patient) {
                throw new Exception("Patient not found");
            }

            $patient_name = $patient['patient_name'] ?? 'Unknown Patient';
            $patient_unique_id = $patient['patient_unique_id'] ?? 'PAT-' . $patient_id;

            // Get account IDs
            $cash_id = $this->get_account_id('1000'); // Use existing cash account
            $general_revenue_id = $this->get_account_id('4000'); // General consultation revenue

            if (!$cash_id) {
                // Fallback to any cash account
                $cash_id = $this->get_account_id('1001');
            }

            if (!$general_revenue_id) {
                // Create or use consultation revenue account
                $general_revenue_id = $this->get_account_id('4001');
            }

            if (!$cash_id || !$general_revenue_id) {
                throw new Exception("Required accounts not found for general workflow");
            }

            $reference_no = 'GEN-' . date('Ymd') . '-' . $patient_id;
            $user_id = $this->session->userdata('admin_id') ?: 1;

            // Insert simple journal entry for general workflow
            $journal_data = [
                'reference_no' => $reference_no,
                'date' => date('Y-m-d'),
                'description' => "General Registration - Patient: {$patient_name}",
                'total_amount' => $amount,
                'source_module' => 'general_registration',
                'source_id' => $patient_id,
                'patient_id' => $patient_id,
                'status' => 'posted',
                'created_by' => $user_id,
                'created_at' => date('Y-m-d H:i:s'),
            ];

            $this->db->insert('journal_headers', $journal_data);
            $journal_id = $this->db->insert_id();

            // Journal lines
            $journal_lines = [
                [
                    'journal_id' => $journal_id,
                    'account_id' => $cash_id,
                    'debit_amount' => $amount,
                    'credit_amount' => 0.00,
                    'description' => 'Cash received for general services',
                    'created_at' => date('Y-m-d H:i:s')
                ],
                [
                    'journal_id' => $journal_id,
                    'account_id' => $general_revenue_id,
                    'debit_amount' => 0.00,
                    'credit_amount' => $amount,
                    'description' => 'General consultation revenue',
                    'created_at' => date('Y-m-d H:i:s')
                ]
            ];

            foreach ($journal_lines as $line) {
                $this->db->insert('journal_lines', $line);
                
                // Also insert to general ledger
                $gl_data = [
                    'transaction_date' => date('Y-m-d'),
                    'account_id' => $line['account_id'],
                    'reference_type' => 'general_registration',
                    'reference_id' => $patient_id,
                    'description' => $line['description'],
                    'debit_amount' => $line['debit_amount'],
                    'credit_amount' => $line['credit_amount'],
                    'posted_by' => $user_id,
                    'created_at' => date('Y-m-d H:i:s'),
                ];
                
                $this->db->insert('general_ledger', $gl_data);
            }

            // Insert financial transaction
            $ft_data = [
                'transaction_date' => date('Y-m-d H:i:s'),
                'transaction_type' => 'revenue',
                'category' => 'medical_services',
                'subcategory' => 'general_consultation',
                'amount' => $amount,
                'description' => 'General medical consultation',
                'reference_table' => 'patients',
                'reference_id' => $patient_id,
                'patient_id' => $patient_unique_id,
                'department' => 'OPD',
                'payment_method' => 'cash',
                'receipt_number' => $reference_no,
                'created_by' => $user_id,
                'created_at' => date('Y-m-d H:i:s'),
            ];
            
            $this->db->insert('financial_transactions', $ft_data);

            $this->db->trans_commit();
            log_message('info', "FINANCIAL DEBUG - General financial entries created successfully");
            return true;

        } catch (Exception $e) {
            $this->db->trans_rollback();
            log_message('error', 'General Financial Entry Error: ' . $e->getMessage());
            return false;
        }
    }

    /**
     * Ensure general accounts exist
     */
    private function ensure_general_accounts() {
        $general_accounts = [
            ['4000', 'General Consultation Revenue', 'revenue', 'Revenue from general consultations']
        ];
        
        foreach ($general_accounts as $account) {
            list($code, $name, $type, $description) = $account;
            
            $existing = $this->get_account_id($code);
            
            if (!$existing) {
                $account_data = [
                    'account_code' => $code,
                    'account_name' => $name,
                    'account_type' => $type,
                    'parent_account_id' => null,
                    'description' => $description,
                    'is_active' => 1,
                    'opening_balance' => 0.00,
                    'balance_type' => 'credit',
                    'created_at' => date('Y-m-d H:i:s'),
                    'updated_at' => date('Y-m-d H:i:s')
                ];
                
                $this->db->insert('chart_of_accounts', $account_data);
                log_message('info', "Created general account: {$code} - {$name}");
            }
        }
    }


/**
 * Ensure required chart of accounts exist
 */
public function ensure_gcc_accounts() {
        log_message('info', "FINANCIAL DEBUG - Ensuring GCC accounts exist");
        
        $required_accounts = [
            ['1001', 'Cash Account', 'asset', 'Cash received from patients'],
            ['4001', 'Consultation Revenue', 'revenue', 'Revenue from medical consultations'],
            ['4006', 'Laboratory Revenue', 'revenue', 'Revenue from laboratory services'],
            ['4004', 'Radiology Revenue', 'revenue', 'Revenue from radiology services']
        ];
        
        foreach ($required_accounts as $account) {
            list($code, $name, $type, $description) = $account;
            
            // Check if account exists
            $existing = $this->get_account_id($code);
            
            if (!$existing) {
                // Create the account
                $account_data = [
                    'account_code' => $code,
                    'account_name' => $name,
                    'account_type' => $type,
                    'parent_account_id' => null,
                    'description' => $description,
                    'is_active' => 1,
                    'opening_balance' => 0.00,
                    'balance_type' => ($type === 'asset') ? 'debit' : 'credit',
                    'created_at' => date('Y-m-d H:i:s'),
                    'updated_at' => date('Y-m-d H:i:s')
                ];
                
                $this->db->insert('chart_of_accounts', $account_data);
                
                if ($this->db->affected_rows() > 0) {
                    log_message('info', "FINANCIAL DEBUG - Created account: {$code} - {$name}");
                } else {
                    log_message('error', "FINANCIAL DEBUG - Failed to create account: {$code} - " . $this->db->error()['message']);
                }
            }
        }
    }

    /**
     * Get recent transactions
     */
    public function getRecentTransactions($limit = 10) {
        // Try to get from journal headers first if table exists
        if ($this->db->table_exists('journal_headers')) {
            try {
                $this->db->select('
                    source_module,
                    reference_no,
                    description,
                    total_amount,
                    date,
                    status
                ');
                
                $this->db->from('journal_headers');
                $this->db->where('status', 'posted');
                $this->db->order_by('date', 'DESC');
                $this->db->limit($limit);
                $query = $this->db->get();
                
                if ($query && $query->num_rows() > 0) {
                    $journal_entries = $query->result_array();
                    if (!empty($journal_entries)) {
                        return $journal_entries;
                    }
                }
            } catch (Exception $e) {
                log_message('error', 'Error getting journal entries: ' . $e->getMessage());
            }
        }

        // Fallback to available billing tables
        $transactions = array();
        
        // Check for transactions table (you have this one)
        if ($this->db->table_exists('transactions')) {
            try {
                $this->db->select('
                    "Transaction" as source_module,
                    CONCAT("TXN-", id) as reference_no,
                    "Transaction" as description,
                    amount as total_amount,
                    created_at as date,
                    "approved" as status
                ');
                $this->db->from('transactions');
                $this->db->where('amount > 0');
                $this->db->order_by('created_at', 'DESC');
                $this->db->limit($limit);
                $query = $this->db->get();
                
                if ($query && $query->num_rows() > 0) {
                    $txn_result = $query->result_array();
                    if ($txn_result) {
                        $transactions = array_merge($transactions, $txn_result);
                    }
                }
            } catch (Exception $e) {
                log_message('error', 'Error getting transactions: ' . $e->getMessage());
            }
        }
        
        // Check for OPD billing
        if ($this->db->table_exists('opd_billing')) {
            try {
                $this->db->select('
                    "OPD" as source_module,
                    CONCAT("OPD-", id) as reference_no,
                    "OPD Consultation" as description,
                    amount as total_amount,
                    date,
                    "approved" as status
                ');
                $this->db->from('opd_billing');
                $this->db->where('amount > 0');
                $this->db->order_by('date', 'DESC');
                $this->db->limit($limit);
                $query = $this->db->get();
                
                if ($query && $query->num_rows() > 0) {
                    $opd_result = $query->result_array();
                    if ($opd_result) {
                        $transactions = array_merge($transactions, $opd_result);
                    }
                }
            } catch (Exception $e) {
                log_message('error', 'Error getting OPD billing: ' . $e->getMessage());
            }
        }
        
        // Check for IPD billing
        if ($this->db->table_exists('ipd_billing')) {
            try {
                $this->db->select('
                    "IPD" as source_module,
                    CONCAT("IPD-", id) as reference_no,
                    "IPD Treatment" as description,
                    amount as total_amount,
                    date,
                    "approved" as status
                ');
                $this->db->from('ipd_billing');
                $this->db->where('amount > 0');
                $this->db->order_by('date', 'DESC');
                $this->db->limit($limit);
                $query = $this->db->get();
                
                if ($query && $query->num_rows() > 0) {
                    $ipd_result = $query->result_array();
                    if ($ipd_result) {
                        $transactions = array_merge($transactions, $ipd_result);
                    }
                }
            } catch (Exception $e) {
                log_message('error', 'Error getting IPD billing: ' . $e->getMessage());
            }
        }
        
        // Check for pathology billing
        if ($this->db->table_exists('pathology_billing')) {
            try {
                $this->db->select('
                    "Pathology" as source_module,
                    CONCAT("PATH-", id) as reference_no,
                    "Pathology Test" as description,
                    amount as total_amount,
                    date,
                    "approved" as status
                ');
                $this->db->from('pathology_billing');
                $this->db->where('amount > 0');
                $this->db->order_by('date', 'DESC');
                $this->db->limit($limit);
                $query = $this->db->get();
                
                if ($query && $query->num_rows() > 0) {
                    $path_result = $query->result_array();
                    if ($path_result) {
                        $transactions = array_merge($transactions, $path_result);
                    }
                }
            } catch (Exception $e) {
                log_message('error', 'Error getting pathology billing: ' . $e->getMessage());
            }
        }

        // Check for radiology billing
        if ($this->db->table_exists('radiology_billing')) {
            try {
                $this->db->select('
                    "Radiology" as source_module,
                    CONCAT("RAD-", id) as reference_no,
                    "Radiology Test" as description,
                    amount as total_amount,
                    date,
                    "approved" as status
                ');
                $this->db->from('radiology_billing');
                $this->db->where('amount > 0');
                $this->db->order_by('date', 'DESC');
                $this->db->limit($limit);
                $query = $this->db->get();
                
                if ($query && $query->num_rows() > 0) {
                    $rad_result = $query->result_array();
                    if ($rad_result) {
                        $transactions = array_merge($transactions, $rad_result);
                    }
                }
            } catch (Exception $e) {
                log_message('error', 'Error getting radiology billing: ' . $e->getMessage());
            }
        }

        // Check for pharmacy billing
        if ($this->db->table_exists('pharmacy_billing')) {
            try {
                $this->db->select('
                    "Pharmacy" as source_module,
                    CONCAT("PHARM-", id) as reference_no,
                    "Pharmacy Sale" as description,
                    amount as total_amount,
                    date,
                    "approved" as status
                ');
                $this->db->from('pharmacy_billing');
                $this->db->where('amount > 0');
                $this->db->order_by('date', 'DESC');
                $this->db->limit($limit);
                $query = $this->db->get();
                
                if ($query && $query->num_rows() > 0) {
                    $pharm_result = $query->result_array();
                    if ($pharm_result) {
                        $transactions = array_merge($transactions, $pharm_result);
                    }
                }
            } catch (Exception $e) {
                log_message('error', 'Error getting pharmacy billing: ' . $e->getMessage());
            }
        }
        
        // Sort by date and limit
        if (!empty($transactions)) {
            usort($transactions, function($a, $b) {
                return strtotime($b['date']) - strtotime($a['date']);
            });
            
            return array_slice($transactions, 0, $limit);
        }
        
        return array();
    }

    /**
     * Get pending approvals
     */
    public function getPendingApprovals() {
        // Check if journal_headers table exists and has pending entries
        if ($this->db->table_exists('journal_headers')) {
            try {
                $this->db->select('
                    id,
                    reference_no,
                    date,
                    description,
                    total_amount,
                    source_module,
                    created_at,
                    created_by,
                    "normal" as priority
                ');
                $this->db->from('journal_headers');
                $this->db->where_in('status', array('draft', 'pending'));
                $this->db->order_by('date', 'DESC');
                $this->db->limit(10);
                $query = $this->db->get();
                
                if ($query && $query->num_rows() > 0) {
                    return $query->result_array();
                }
            } catch (Exception $e) {
                log_message('error', 'Error getting pending approvals: ' . $e->getMessage());
            }
        }
        
        // For now, return empty array since there's no approval system in the current schema
        return array();
    }

    /**
     * Get monthly revenue data for charts
     */
    public function getMonthlyRevenue($months = 12) {
        $revenue_data = array();
        
        for ($i = $months - 1; $i >= 0; $i--) {
            $month = date('Y-m', strtotime("-$i months"));
            $total_revenue = 0;
            
            // Check transactions table
            if ($this->db->table_exists('transactions')) {
                try {
                    $this->db->select_sum('amount');
                    $this->db->from('transactions');
                    $this->db->where('DATE_FORMAT(created_at, "%Y-%m") =', $month);
                    $this->db->where('amount > 0');
                    $query = $this->db->get();
                    
                    if ($query && $query->num_rows() > 0) {
                        $txn_result = $query->row();
                        $txn_revenue = $txn_result ? ($txn_result->amount ?: 0) : 0;
                        $total_revenue += $txn_revenue;
                    }
                } catch (Exception $e) {
                    log_message('error', 'Error getting transactions revenue for month ' . $month . ': ' . $e->getMessage());
                }
            }
            
            // Check OPD billing table
            if ($this->db->table_exists('opd_billing')) {
                try {
                    $this->db->select_sum('amount');
                    $this->db->from('opd_billing');
                    $this->db->where('DATE_FORMAT(date, "%Y-%m") =', $month);
                    $this->db->where('amount > 0');
                    $query = $this->db->get();
                    
                    if ($query && $query->num_rows() > 0) {
                        $opd_result = $query->row();
                        $opd_revenue = $opd_result ? ($opd_result->amount ?: 0) : 0;
                        $total_revenue += $opd_revenue;
                    }
                } catch (Exception $e) {
                    log_message('error', 'Error getting OPD revenue for month ' . $month . ': ' . $e->getMessage());
                }
            }
            
            // Check IPD billing table
            if ($this->db->table_exists('ipd_billing')) {
                try {
                    $this->db->select_sum('amount');
                    $this->db->from('ipd_billing');
                    $this->db->where('DATE_FORMAT(date, "%Y-%m") =', $month);
                    $this->db->where('amount > 0');
                    $query = $this->db->get();
                    
                    if ($query && $query->num_rows() > 0) {
                        $ipd_result = $query->row();
                        $ipd_revenue = $ipd_result ? ($ipd_result->amount ?: 0) : 0;
                        $total_revenue += $ipd_revenue;
                    }
                } catch (Exception $e) {
                    log_message('error', 'Error getting IPD revenue for month ' . $month . ': ' . $e->getMessage());
                }
            }

            // Check pathology billing table
            if ($this->db->table_exists('pathology_billing')) {
                try {
                    $this->db->select_sum('amount');
                    $this->db->from('pathology_billing');
                    $this->db->where('DATE_FORMAT(date, "%Y-%m") =', $month);
                    $this->db->where('amount > 0');
                    $query = $this->db->get();
                    
                    if ($query && $query->num_rows() > 0) {
                        $path_result = $query->row();
                        $path_revenue = $path_result ? ($path_result->amount ?: 0) : 0;
                        $total_revenue += $path_revenue;
                    }
                } catch (Exception $e) {
                    log_message('error', 'Error getting pathology revenue for month ' . $month . ': ' . $e->getMessage());
                }
            }

            // Check radiology billing table
            if ($this->db->table_exists('radiology_billing')) {
                try {
                    $this->db->select_sum('amount');
                    $this->db->from('radiology_billing');
                    $this->db->where('DATE_FORMAT(date, "%Y-%m") =', $month);
                    $this->db->where('amount > 0');
                    $query = $this->db->get();
                    
                    if ($query && $query->num_rows() > 0) {
                        $rad_result = $query->row();
                        $rad_revenue = $rad_result ? ($rad_result->amount ?: 0) : 0;
                        $total_revenue += $rad_revenue;
                    }
                } catch (Exception $e) {
                    log_message('error', 'Error getting radiology revenue for month ' . $month . ': ' . $e->getMessage());
                }
            }

            // Check pharmacy billing table
            if ($this->db->table_exists('pharmacy_billing')) {
                try {
                    $this->db->select_sum('amount');
                    $this->db->from('pharmacy_billing');
                    $this->db->where('DATE_FORMAT(date, "%Y-%m") =', $month);
                    $this->db->where('amount > 0');
                    $query = $this->db->get();
                    
                    if ($query && $query->num_rows() > 0) {
                        $pharm_result = $query->row();
                        $pharm_revenue = $pharm_result ? ($pharm_result->amount ?: 0) : 0;
                        $total_revenue += $pharm_revenue;
                    }
                } catch (Exception $e) {
                    log_message('error', 'Error getting pharmacy revenue for month ' . $month . ': ' . $e->getMessage());
                }
            }
            
            $revenue_data[] = $total_revenue;
        }
        
        return $revenue_data;
    }

    /**
     * Get all accounts - FIXED for your database schema
     */
    public function getAllAccounts() {
    // Check if chart_of_accounts table exists
    if ($this->db->table_exists('chart_of_accounts')) {
        try {
            // SELECT ALL fields that your view needs, including is_active
            $this->db->select('id, account_code, account_name, account_type, parent_account_id, description, is_active, opening_balance, balance_type, normal_balance, created_at, updated_at');
            $this->db->from('chart_of_accounts');
            // REMOVED: Don't filter by is_active here - let the view show all accounts
            // $this->db->where('is_active', 1); 
            $this->db->order_by('account_code', 'ASC');
            
            $query = $this->db->get();
            
            if ($query) {
                return $query->result_array();
            }
            return array();
        } catch (Exception $e) {
            log_message('error', 'Error getting all accounts: ' . $e->getMessage());
            return array();
        }
    }

}

    /**
     * Get ledger entries for specific account - FIXED for your schema
     */
    public function getLedgerEntries($account_id, $date_from, $date_to) {
        // Check if we have proper journal entry tables
        if ($this->db->table_exists('journal_headers') && $this->db->table_exists('journal_lines')) {
            $this->db->select('
                jh.date,
                jh.reference_no,
                coa.account_code,
                coa.account_name,
                jh.description as header_description,
                jl.description as line_description,
                jl.debit_amount,
                jl.credit_amount,
                jh.source_module
            ');
            $this->db->from('journal_lines jl');
            $this->db->join('journal_headers jh', 'jh.id = jl.journal_id');
            $this->db->join('chart_of_accounts coa', 'coa.id = jl.account_id');
            $this->db->where('jl.account_id', $account_id);
            $this->db->where('jh.date >=', $date_from);
            $this->db->where('jh.date <=', $date_to);
            $this->db->where('jh.status', 'posted');
            $this->db->order_by('jh.date', 'DESC');
            
            return $this->db->get()->result_array();
        }
        
        // Fallback to billing data simulation
        if ($account_id == 4) { // OPD Revenue
            try {
                $this->db->select('
                    date,
                    CONCAT("OPD-", id) as reference_no,
                    "4001" as account_code,
                    "OPD Revenue" as account_name,
                    "OPD Consultation" as header_description,
                    "" as line_description,
                    0 as debit_amount,
                    amount as credit_amount,
                    "opd" as source_module
                ');
                $this->db->from('opd_billing');
                $this->db->where('date >=', $date_from);
                $this->db->where('date <=', $date_to);
                $this->db->where('amount > 0');
                $this->db->order_by('date', 'DESC');
                
                return $this->db->get()->result_array();
            } catch (Exception $e) {
                return array();
            }
        }
        
        if ($account_id == 5) { // IPD Revenue
            try {
                $this->db->select('
                    date,
                    CONCAT("IPD-", id) as reference_no,
                    "4002" as account_code,
                    "IPD Revenue" as account_name,
                    "IPD Treatment" as header_description,
                    "" as line_description,
                    0 as debit_amount,
                    amount as credit_amount,
                    "ipd" as source_module
                ');
                $this->db->from('ipd_billing');
                $this->db->where('date >=', $date_from);
                $this->db->where('date <=', $date_to);
                $this->db->where('amount > 0');
                $this->db->order_by('date', 'DESC');
                
                return $this->db->get()->result_array();
            } catch (Exception $e) {
                return array();
            }
        }
        
        // For other accounts, return empty for now
        return array();
    }

    /**
     * Get all ledger entries (combined) - FIXED for your schema
     */
    public function getAllLedgerEntries($date_from, $date_to) {
        // Check if we have proper journal entry tables - USING YOUR ACTUAL TABLE NAMES
        if ($this->db->table_exists('journal_headers') && $this->db->table_exists('journal_lines')) {
            try {
                $this->db->select('
                    jh.date,
                    jh.reference_no,
                    coa.account_code,
                    coa.account_name,
                    jh.description as header_description,
                    jl.description as line_description,
                    jl.debit_amount,
                    jl.credit_amount,
                    jh.source_module
                ');
                $this->db->from('journal_lines jl');
                $this->db->join('journal_headers jh', 'jh.id = jl.journal_id');
                $this->db->join('chart_of_accounts coa', 'coa.id = jl.account_id');
                $this->db->where('jh.date >=', $date_from);
                $this->db->where('jh.date <=', $date_to);
                $this->db->where('jh.status', 'posted');
                $this->db->order_by('jh.date', 'DESC');
                
                return $this->db->get()->result_array();
            } catch (Exception $e) {
                log_message('error', 'Error getting journal entries: ' . $e->getMessage());
            }
        }
        
        $entries = array();
        
        // Get entries from transactions table
        if ($this->db->table_exists('transactions')) {
            try {
                $this->db->select('
                    created_at as date,
                    CONCAT("TXN-", id) as reference_no,
                    "TXN" as account_code,
                    "Transaction" as account_name,
                    "Transaction" as header_description,
                    "" as line_description,
                    0 as debit_amount,
                    amount as credit_amount,
                    "transaction" as source_module
                ');
                $this->db->from('transactions');
                $this->db->where('created_at >=', $date_from);
                $this->db->where('created_at <=', $date_to);
                $this->db->where('amount > 0');
                $txn_entries = $this->db->get()->result_array();
                $entries = array_merge($entries, $txn_entries);
            } catch (Exception $e) {
                log_message('error', 'Error getting transaction entries: ' . $e->getMessage());
            }
        }
        
        // Try OPD and IPD if they exist
        if ($this->db->table_exists('opd_billing')) {
            try {
                $this->db->select('
                    date,
                    CONCAT("OPD-", id) as reference_no,
                    "4001" as account_code,
                    "OPD Revenue" as account_name,
                    "OPD Consultation" as header_description,
                    "" as line_description,
                    0 as debit_amount,
                    amount as credit_amount,
                    "opd" as source_module
                ');
                $this->db->from('opd_billing');
                $this->db->where('date >=', $date_from);
                $this->db->where('date <=', $date_to);
                $this->db->where('amount > 0');
                $opd_entries = $this->db->get()->result_array();
                $entries = array_merge($entries, $opd_entries);
            } catch (Exception $e) {
                // Table might not exist
            }
        }
        
        if ($this->db->table_exists('ipd_billing')) {
            try {
                $this->db->select('
                    date,
                    CONCAT("IPD-", id) as reference_no,
                    "4002" as account_code,
                    "IPD Revenue" as account_name,
                    "IPD Treatment" as header_description,
                    "" as line_description,
                    0 as debit_amount,
                    amount as credit_amount,
                    "ipd" as source_module
                ');
                $this->db->from('ipd_billing');
                $this->db->where('date >=', $date_from);
                $this->db->where('date <=', $date_to);
                $this->db->where('amount > 0');
                $ipd_entries = $this->db->get()->result_array();
                $entries = array_merge($entries, $ipd_entries);
            } catch (Exception $e) {
                // Table might not exist
            }
        }
        
        // Sort by date descending
        if (!empty($entries)) {
            usort($entries, function($a, $b) {
                return strtotime($b['date']) - strtotime($a['date']);
            });
        }
        
        return $entries;
    }

    /**
     * Get account summary
     */
    public function getAccountSummary($account_id, $date_from, $date_to) {
        $opening_balance = 0; // Would calculate from previous periods
        
        $entries = $this->getLedgerEntries($account_id, $date_from, $date_to);
        
        $total_debits = 0;
        $total_credits = 0;
        
        foreach ($entries as $entry) {
            $total_debits += $entry['debit_amount'];
            $total_credits += $entry['credit_amount'];
        }
        
        $closing_balance = $opening_balance + $total_debits - $total_credits;
        
        return array(
            'opening_balance' => $opening_balance,
            'total_debits' => $total_debits,
            'total_credits' => $total_credits,
            'closing_balance' => $closing_balance
        );
    }

    /**
     * Create basic financial tables if they don't exist - UPDATED FOR YOUR SCHEMA
     */
    public function createFinancialTables() {
        try {
            // Chart of Accounts - matches your schema
            if (!$this->db->table_exists('chart_of_accounts')) {
                $this->db->query("
                    CREATE TABLE `chart_of_accounts` (
                        `id` int NOT NULL AUTO_INCREMENT,
                        `account_code` varchar(20) NOT NULL,
                        `account_name` varchar(255) NOT NULL,
                        `account_type` enum('asset','liability','equity','revenue','expense') NOT NULL,
                        `parent_account_id` int DEFAULT NULL,
                        `description` text,
                        `is_active` tinyint(1) DEFAULT 1,
                        `opening_balance` decimal(15,2) DEFAULT 0.00,
                        `balance_type` enum('debit','credit') DEFAULT 'debit',
                        `created_at` timestamp DEFAULT CURRENT_TIMESTAMP,
                        `updated_at` timestamp DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                        PRIMARY KEY (`id`),
                        UNIQUE KEY `account_code` (`account_code`),
                        KEY `parent_account_id` (`parent_account_id`)
                    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb3
                ");
            }

            // Journal Headers - matches your schema
            if (!$this->db->table_exists('journal_headers')) {
                $this->db->query("
                    CREATE TABLE `journal_headers` (
                        `id` int NOT NULL AUTO_INCREMENT,
                        `reference_no` varchar(50) NOT NULL,
                        `date` date NOT NULL,
                        `description` text NOT NULL,
                        `total_amount` decimal(15,2) NOT NULL DEFAULT 0.00,
                        `source_module` varchar(50) DEFAULT NULL COMMENT 'opd, ipd, pathology, radiology, pharmacy, payment, manual',
                        `source_id` int DEFAULT NULL COMMENT 'ID from source module',
                        `patient_id` int DEFAULT NULL,
                        `supplier_id` int DEFAULT NULL,
                        `status` enum('draft','pending','approved','posted','reversed') DEFAULT 'posted',
                        `approved_by` int DEFAULT NULL,
                        `approved_at` timestamp NULL DEFAULT NULL,
                        `reversed_by` int DEFAULT NULL,
                        `reversed_at` timestamp NULL DEFAULT NULL,
                        `reversed_entry_id` int DEFAULT NULL,
                        `reversal_reason` text,
                        `rejection_reason` text,
                        `created_by` int NOT NULL,
                        `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
                        `updated_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                        PRIMARY KEY (`id`),
                        UNIQUE KEY `reference_no` (`reference_no`)
                    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb3
                ");
            }

            // Journal Lines - matches your schema
            if (!$this->db->table_exists('journal_lines')) {
                $this->db->query("
                    CREATE TABLE `journal_lines` (
                        `id` int NOT NULL AUTO_INCREMENT,
                        `journal_id` int NOT NULL,
                        `account_id` int NOT NULL,
                        `debit_amount` decimal(15,2) DEFAULT 0.00,
                        `credit_amount` decimal(15,2) DEFAULT 0.00,
                        `description` text,
                        `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
                        PRIMARY KEY (`id`),
                        KEY `journal_id` (`journal_id`),
                        KEY `account_id` (`account_id`)
                    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb3
                ");
            }

            // Insert default chart of accounts
            $this->insertDefaultAccounts();
            
            return true;
        } catch (Exception $e) {
            log_message('error', 'Error creating financial tables: ' . $e->getMessage());
            throw $e;
        }
    }

    private function insertDefaultAccounts() {
        $accounts = array(
            // Assets
            array('account_code' => '1001', 'account_name' => 'Cash', 'account_type' => 'asset', 'balance_type' => 'debit'),
            array('account_code' => '1002', 'account_name' => 'Bank Account', 'account_type' => 'asset', 'balance_type' => 'debit'),
            array('account_code' => '1200', 'account_name' => 'Accounts Receivable', 'account_type' => 'asset', 'balance_type' => 'debit'),
            array('account_code' => '1300', 'account_name' => 'Medical Equipment', 'account_type' => 'asset', 'balance_type' => 'debit'),
            array('account_code' => '1400', 'account_name' => 'Furniture & Fixtures', 'account_type' => 'asset', 'balance_type' => 'debit'),
           
           // Liabilities
           array('account_code' => '2001', 'account_name' => 'Accounts Payable', 'account_type' => 'liability', 'balance_type' => 'credit'),
           array('account_code' => '2100', 'account_name' => 'Equipment Loan', 'account_type' => 'liability', 'balance_type' => 'credit'),
           array('account_code' => '2200', 'account_name' => 'Bank Loan', 'account_type' => 'liability', 'balance_type' => 'credit'),
           
           // Equity
           array('account_code' => '3001', 'account_name' => 'Owner Equity', 'account_type' => 'equity', 'balance_type' => 'credit'),
           array('account_code' => '3100', 'account_name' => 'Retained Earnings', 'account_type' => 'equity', 'balance_type' => 'credit'),
           
           // Revenue
           array('account_code' => '4001', 'account_name' => 'OPD Revenue', 'account_type' => 'revenue', 'balance_type' => 'credit'),
           array('account_code' => '4002', 'account_name' => 'IPD Revenue', 'account_type' => 'revenue', 'balance_type' => 'credit'),
           array('account_code' => '4003', 'account_name' => 'Pathology Revenue', 'account_type' => 'revenue', 'balance_type' => 'credit'),
           array('account_code' => '4004', 'account_name' => 'Radiology Revenue', 'account_type' => 'revenue', 'balance_type' => 'credit'),
           array('account_code' => '4005', 'account_name' => 'Pharmacy Revenue', 'account_type' => 'revenue', 'balance_type' => 'credit'),
           array('account_code' => '4006', 'account_name' => 'Laboratory Revenue', 'account_type' => 'revenue', 'balance_type' => 'credit'),
           
           // Expenses
           array('account_code' => '5001', 'account_name' => 'Staff Salaries', 'account_type' => 'expense', 'balance_type' => 'debit'),
           array('account_code' => '5002', 'account_name' => 'Doctor Fees', 'account_type' => 'expense', 'balance_type' => 'debit'),
           array('account_code' => '5003', 'account_name' => 'Medical Supplies', 'account_type' => 'expense', 'balance_type' => 'debit'),
           array('account_code' => '5004', 'account_name' => 'Utilities', 'account_type' => 'expense', 'balance_type' => 'debit'),
           array('account_code' => '5005', 'account_name' => 'Equipment Maintenance', 'account_type' => 'expense', 'balance_type' => 'debit'),
           array('account_code' => '5006', 'account_name' => 'Office Expenses', 'account_type' => 'expense', 'balance_type' => 'debit')
       );

       foreach ($accounts as $account) {
           try {
               $this->db->where('account_code', $account['account_code']);
               $existing = $this->db->get('chart_of_accounts')->row();
               
               if (!$existing) {
                   $this->db->insert('chart_of_accounts', $account);
               }
           } catch (Exception $e) {
               log_message('error', 'Error inserting account ' . $account['account_code'] . ': ' . $e->getMessage());
           }
       }
   }

   /**
    * Auto-create journal entries from billing transactions - FIXED VERSION FOR YOUR SCHEMA
    */
   public function createJournalFromBilling($type, $billing_id, $amount, $date) {
       if (!$this->db->table_exists('journal_headers') || !$this->db->table_exists('journal_lines')) {
           return false;
       }

       $reference_no = strtoupper($type) . '-JE-' . $billing_id;
       
       // Check if journal entry already exists
       $this->db->where('reference_no', $reference_no);
       $existing = $this->db->get('journal_headers')->row();
       
       if ($existing) {
           return $existing->id;
       }

       try {
           // Create journal entry header - USING YOUR TABLE STRUCTURE
           $journal_data = array(
               'reference_no' => $reference_no,
               'date' => $date,
               'description' => ucfirst($type) . ' billing transaction',
               'total_amount' => $amount,
               'source_module' => $type,
               'source_id' => $billing_id,
               'status' => 'posted',
               'created_by' => 1 // Default user, you can get from session
           );
           
           $this->db->insert('journal_headers', $journal_data);
           $journal_entry_id = $this->db->insert_id();

           if (!$journal_entry_id) {
               return false;
           }

           // Get account IDs
           $cash_account = $this->getAccountByCode('1001'); // Cash
           $revenue_account = $this->getAccountByCode($type == 'opd' ? '4001' : '4002'); // OPD or IPD Revenue
           
           if (!$cash_account || !$revenue_account) {
               return false;
           }

           // Create journal entry lines - Debit Cash
           $this->db->insert('journal_lines', array(
               'journal_id' => $journal_entry_id,
               'account_id' => $cash_account['id'],
               'description' => 'Cash received for ' . ucfirst($type),
               'debit_amount' => $amount,
               'credit_amount' => 0
           ));
           
           // Create journal entry lines - Credit Revenue
           $this->db->insert('journal_lines', array(
               'journal_id' => $journal_entry_id,
               'account_id' => $revenue_account['id'],
               'description' => ucfirst($type) . ' revenue',
               'debit_amount' => 0,
               'credit_amount' => $amount
           ));

           return $journal_entry_id;
       } catch (Exception $e) {
           log_message('error', 'Error creating journal entry: ' . $e->getMessage());
           return false;
       }
   }

   private function getAccountByCode($code) {
       if (!$this->db->table_exists('chart_of_accounts')) {
           return false;
       }
       
       $this->db->where('account_code', $code);
       $this->db->where('is_active', 1);
       return $this->db->get('chart_of_accounts')->row_array();
   }

   /**
    * Get trial balance - FIXED FOR YOUR SCHEMA
    */
  /* public function getTrialBalance($date = null) {
       if (!$date) $date = date('Y-m-d');
       
       if (!$this->db->table_exists('chart_of_accounts') || 
           !$this->db->table_exists('journal_headers') || 
           !$this->db->table_exists('journal_lines')) {
           return array();
       }
       
       try {
           $this->db->select('
               coa.account_code,
               coa.account_name,
               coa.account_type,
               COALESCE(SUM(jl.debit_amount), 0) as total_debits,
               COALESCE(SUM(jl.credit_amount), 0) as total_credits
           ');
           $this->db->from('chart_of_accounts coa');
           $this->db->join('journal_lines jl', 'jl.account_id = coa.id', 'left');
           $this->db->join('journal_headers jh', 'jh.id = jl.journal_id', 'left');
           $this->db->where('coa.is_active', 1); // Use 1 instead of 'yes'
           $this->db->where('(jh.date IS NULL OR jh.date <=)', $date);
           $this->db->where('(jh.status IS NULL OR jh.status =)', 'posted');
           $this->db->group_by('coa.id');
           $this->db->order_by('coa.account_code');
           
           return $this->db->get()->result_array();
       } catch (Exception $e) {
           log_message('error', 'Error getting trial balance: ' . $e->getMessage());
           return array();
       }
   }*/

   /**
    * Get income statement - COMPLETELY FIXED FOR YOUR DATABASE SCHEMA
    */
   public function getIncomeStatement($date_from, $date_to) {
       // Initialize return array
       $income_statement = array(
           'revenue' => array(),
           'expenses' => array()
       );

       // Check if required tables exist - using your actual table names
       if (!$this->db->table_exists('chart_of_accounts') || 
           !$this->db->table_exists('journal_headers') || 
           !$this->db->table_exists('journal_lines')) {
           
           // If journal tables don't exist, try fallback data
           return $this->getIncomeStatementFallback($date_from, $date_to);
       }

       try {
           // Revenue Query - Fixed for your schema
           $revenue_query = "
               SELECT 
                   coa.id,
                   coa.account_code,
                   coa.account_name,
                   coa.account_type,
                   COALESCE(SUM(jl.credit_amount - jl.debit_amount), 0) as amount
               FROM chart_of_accounts coa
               LEFT JOIN journal_lines jl ON jl.account_id = coa.id
               LEFT JOIN journal_headers jh ON jh.id = jl.journal_id AND jh.status = 'posted'
               WHERE coa.account_type = 'revenue'
               AND coa.is_active = 1
               AND (jh.date IS NULL OR (jh.date >= ? AND jh.date <= ?))
               GROUP BY coa.id, coa.account_code, coa.account_name, coa.account_type
               HAVING amount != 0
               ORDER BY coa.account_code
           ";

           $revenue_result = $this->db->query($revenue_query, array($date_from, $date_to));
           if ($revenue_result && $revenue_result !== false) {
               $income_statement['revenue'] = $revenue_result->result_array();
           }

           // Expense Query - Fixed for your schema
           $expense_query = "
               SELECT 
                   coa.id,
                   coa.account_code,
                   coa.account_name,
                   coa.account_type,
                   COALESCE(SUM(jl.debit_amount - jl.credit_amount), 0) as amount
               FROM chart_of_accounts coa
               LEFT JOIN journal_lines jl ON jl.account_id = coa.id
               LEFT JOIN journal_headers jh ON jh.id = jl.journal_id AND jh.status = 'posted'
               WHERE coa.account_type = 'expense'
               AND coa.is_active = 1
               AND (jh.date IS NULL OR (jh.date >= ? AND jh.date <= ?))
               GROUP BY coa.id, coa.account_code, coa.account_name, coa.account_type
               HAVING amount != 0
               ORDER BY coa.account_code
           ";

           $expense_result = $this->db->query($expense_query, array($date_from, $date_to));
           if ($expense_result && $expense_result !== false) {
               $income_statement['expenses'] = $expense_result->result_array();
           }

           // Debug logging
           log_message('debug', 'Revenue query returned: ' . count($income_statement['revenue']) . ' accounts');
           log_message('debug', 'Expense query returned: ' . count($income_statement['expenses']) . ' accounts');

           return $income_statement;

       } catch (Exception $e) {
           log_message('error', 'Error getting income statement: ' . $e->getMessage());
           return $this->getIncomeStatementFallback($date_from, $date_to);
       }
   }

   /**
    * Fallback income statement using billing tables
    */
   private function getIncomeStatementFallback($date_from, $date_to) {
       $income_statement = array(
           'revenue' => array(),
           'expenses' => array()
       );

       // Get revenue from OPD billing
       if ($this->db->table_exists('opd_billing')) {
           try {
               $this->db->select('SUM(amount) as total_amount');
               $this->db->from('opd_billing');
               $this->db->where('date >=', $date_from);
               $this->db->where('date <=', $date_to);
               $this->db->where('amount > 0');
               $result = $this->db->get()->row();
               
               if ($result && $result->total_amount > 0) {
                   $income_statement['revenue'][] = array(
                       'id' => null,
                       'account_code' => '4001',
                       'account_name' => 'OPD Revenue',
                       'account_type' => 'revenue',
                       'amount' => $result->total_amount
                   );
               }
           } catch (Exception $e) {
               log_message('error', 'Error getting OPD revenue: ' . $e->getMessage());
           }
       }

       // Get revenue from IPD billing
       if ($this->db->table_exists('ipd_billing')) {
           try {
               $this->db->select('SUM(amount) as total_amount');
               $this->db->from('ipd_billing');
               $this->db->where('date >=', $date_from);
               $this->db->where('date <=', $date_to);
               $this->db->where('amount > 0');
               $result = $this->db->get()->row();
               
               if ($result && $result->total_amount > 0) {
                   $income_statement['revenue'][] = array(
                       'id' => null,
                       'account_code' => '4002',
                       'account_name' => 'IPD Revenue',
                       'account_type' => 'revenue',
                       'amount' => $result->total_amount
                   );
               }
           } catch (Exception $e) {
               log_message('error', 'Error getting IPD revenue: ' . $e->getMessage());
           }
       }

       // Get revenue from pathology billing
       if ($this->db->table_exists('pathology_billing')) {
           try {
               $this->db->select('SUM(amount) as total_amount');
               $this->db->from('pathology_billing');
               $this->db->where('date >=', $date_from);
               $this->db->where('date <=', $date_to);
               $this->db->where('amount > 0');
               $result = $this->db->get()->row();
               
               if ($result && $result->total_amount > 0) {
                   $income_statement['revenue'][] = array(
                       'id' => null,
                       'account_code' => '4003',
                       'account_name' => 'Pathology Revenue',
                       'account_type' => 'revenue',
                       'amount' => $result->total_amount
                   );
               }
           } catch (Exception $e) {
               log_message('error', 'Error getting Pathology revenue: ' . $e->getMessage());
           }
       }

       // Get revenue from radiology billing
       if ($this->db->table_exists('radiology_billing')) {
           try {
               $this->db->select('SUM(amount) as total_amount');
               $this->db->from('radiology_billing');
               $this->db->where('date >=', $date_from);
               $this->db->where('date <=', $date_to);
               $this->db->where('amount > 0');
               $result = $this->db->get()->row();
               
               if ($result && $result->total_amount > 0) {
                   $income_statement['revenue'][] = array(
                       'id' => null,
                       'account_code' => '4004',
                       'account_name' => 'Radiology Revenue',
                       'account_type' => 'revenue',
                       'amount' => $result->total_amount
                   );
               }
           } catch (Exception $e) {
               log_message('error', 'Error getting Radiology revenue: ' . $e->getMessage());
           }
       }

       // Get revenue from pharmacy billing
       if ($this->db->table_exists('pharmacy_billing')) {
           try {
               $this->db->select('SUM(amount) as total_amount');
               $this->db->from('pharmacy_billing');
               $this->db->where('date >=', $date_from);
               $this->db->where('date <=', $date_to);
               $this->db->where('amount > 0');
               $result = $this->db->get()->row();
               
               if ($result && $result->total_amount > 0) {
                   $income_statement['revenue'][] = array(
                       'id' => null,
                       'account_code' => '4005',
                       'account_name' => 'Pharmacy Revenue',
                       'account_type' => 'revenue',
                       'amount' => $result->total_amount
                   );
               }
           } catch (Exception $e) {
               log_message('error', 'Error getting Pharmacy revenue: ' . $e->getMessage());
           }
       }

       return $income_statement;
   }

   /**
    * Debug journal entries with your schema
    */
   public function debugJournalEntries($date_from = null, $date_to = null) {
       if (!$date_from) $date_from = '2020-01-01';
       if (!$date_to) $date_to = '2030-12-31';
       
       try {
           // Build the query based on your schema
           $query = "
               SELECT 
                   jh.id,
                   jh.reference_no,
                   jh.date,
                   jh.description,
                   jh.total_amount,
                   jh.status,
                   jl.account_id,
                   coa.account_code,
                   coa.account_name,
                   coa.account_type,
                   jl.debit_amount,
                   jl.credit_amount
               FROM journal_headers jh
               LEFT JOIN journal_lines jl ON jl.journal_id = jh.id
               LEFT JOIN chart_of_accounts coa ON coa.id = jl.account_id
               WHERE jh.date >= ? AND jh.date <= ?
               ORDER BY jh.date DESC, jh.id DESC
               LIMIT 50
           ";
           
           $result = $this->db->query($query, array($date_from, $date_to));
           
           if ($result === false) {
               // Log the error and return empty array
               $error = $this->db->error();
               log_message('error', 'debugJournalEntries query failed: ' . $error['message']);
               log_message('error', 'Query was: ' . $query);
               return array();
           }
           
           return $result->result_array();
           
       } catch (Exception $e) {
           log_message('error', 'Exception in debugJournalEntries: ' . $e->getMessage());
           return array();
       }
   }

   /**
    * Get chart of accounts for debugging
    */
   public function debugChartOfAccounts() {
       try {
           $this->db->select('*');
           $this->db->from('chart_of_accounts');
           $this->db->where('is_active', 1);
           $this->db->order_by('account_code');
           
           return $this->db->get()->result_array();
       } catch (Exception $e) {
           log_message('error', 'DEBUG: Error getting chart of accounts: ' . $e->getMessage());
           return array();
       }
   }

   /**
    * Get balance sheet data - Fixed version
    */
   public function getBalanceSheet($date) {
       $asset_accounts = $this->getAccountsByType('asset');
       $liability_accounts = $this->getAccountsByType('liability');
       $equity_accounts = $this->getAccountsByType('equity');
       
       $asset_data = array();
       $liability_data = array();
       $equity_data = array();
       $total_assets = 0;
       $total_liabilities = 0;
       $total_equity = 0;

       // Calculate assets
       foreach ($asset_accounts as $account) {
           $opening_balance = floatval($account['opening_balance'] ?? 0);
           
           // Get journal movements (will be 0 if no journal entries exist)
           $journal_balance = 0;
           if ($this->db->table_exists('journal_lines') && $this->db->table_exists('journal_headers')) {
               $journal_balance = $this->getAccountBalance($account['id'], null, $date, 'net');
           }
           
           // Calculate total balance
           $balance_type = $account['balance_type'] ?? 'debit';
           if ($balance_type == 'debit') {
               // Normal asset accounts (debit increases balance)
               $total_balance = $opening_balance + $journal_balance;
           } else {
               // Contra-asset accounts like Accumulated Depreciation (credit decreases asset)
               $total_balance = $opening_balance - $journal_balance;
           }
           
           $asset_data[] = array(
               'account_code' => $account['account_code'],
               'account_name' => $account['account_name'],
               'amount' => $total_balance,
               'opening_balance' => $opening_balance,
               'journal_balance' => $journal_balance
           );
           $total_assets += $total_balance;
       }

       // Calculate liabilities
       foreach ($liability_accounts as $account) {
           $opening_balance = floatval($account['opening_balance'] ?? 0);
           
           $journal_balance = 0;
           if ($this->db->table_exists('journal_lines') && $this->db->table_exists('journal_headers')) {
               $journal_balance = $this->getAccountBalance($account['id'], null, $date, 'net');
           }
           
           // For liabilities, credit balance is normal
           $balance_type = $account['balance_type'] ?? 'credit';
           if ($balance_type == 'credit') {
               // Normal liability accounts (credit increases balance)
               $total_balance = $opening_balance + abs($journal_balance);
           } else {
               // Unusual case
               $total_balance = $opening_balance + $journal_balance;
           }
           
           $liability_data[] = array(
               'account_code' => $account['account_code'],
               'account_name' => $account['account_name'],
               'amount' => $total_balance,
               'opening_balance' => $opening_balance,
               'journal_balance' => $journal_balance
           );
           $total_liabilities += $total_balance;
       }

       // Calculate equity
       foreach ($equity_accounts as $account) {
           $opening_balance = floatval($account['opening_balance'] ?? 0);
           
           $journal_balance = 0;
           if ($this->db->table_exists('journal_lines') && $this->db->table_exists('journal_headers')) {
               $journal_balance = $this->getAccountBalance($account['id'], null, $date, 'net');
           }
           
           // For equity, credit balance is normal
           $balance_type = $account['balance_type'] ?? 'credit';
           if ($balance_type == 'credit') {
               // Normal equity accounts (credit increases balance)
               $total_balance = $opening_balance + abs($journal_balance);
           } else {
               // Unusual case
               $total_balance = $opening_balance + $journal_balance;
           }
           
           $equity_data[] = array(
               'account_code' => $account['account_code'],
               'account_name' => $account['account_name'],
               'amount' => $total_balance,
               'opening_balance' => $opening_balance,
               'journal_balance' => $journal_balance
           );
           $total_equity += $total_balance;
       }

       return array(
           'asset_accounts' => $asset_data,
           'liability_accounts' => $liability_data,
           'equity_accounts' => $equity_data,
           'total_assets' => $total_assets,
           'total_liabilities' => $total_liabilities,
           'total_equity' => $total_equity,
           'date' => $date
       );
   }

   /**
    * Get account balance for specific date range
    */
   public function getAccountBalance($account_id, $date_from = null, $date_to = null, $balance_type = 'net') {
       // Check if required tables exist
       if (!$this->db->table_exists('journal_lines') || !$this->db->table_exists('journal_headers')) {
           return 0;
       }
       
       // Validate account_id
       if (!$account_id || !is_numeric($account_id)) {
           return 0;
       }
       
       // Check if there are any journal entries for this account
       $this->db->where('account_id', $account_id);
       $count = $this->db->count_all_results('journal_lines');
       
       if ($count == 0) {
           return 0; // No journal entries yet
       }
       
       try {
           $this->db->select('COALESCE(SUM(jl.debit_amount), 0) as total_debits, COALESCE(SUM(jl.credit_amount), 0) as total_credits');
           $this->db->from('journal_lines jl');
           $this->db->join('journal_headers jh', 'jh.id = jl.journal_id');
           $this->db->where('jl.account_id', $account_id);
           $this->db->where('jh.status', 'posted');
           
           if ($date_from) {
               $this->db->where('jh.date >=', $date_from);
           }
           if ($date_to) {
               $this->db->where('jh.date <=', $date_to);
           }
           
           $query = $this->db->get();
           
           if ($query === false) {
               $db_error = $this->db->error();
               log_message('error', 'Database error in getAccountBalance: ' . $db_error['message']);
               return 0;
           }
           
           $result = $query->row();
           
           if (!$result) {
               return 0;
           }
           
           $debits = $result->total_debits ?: 0;
           $credits = $result->total_credits ?: 0;
           
           switch ($balance_type) {
               case 'debit':
                   return $debits;
               case 'credit':
                   return $credits;
               case 'net':
               default:
                   return $debits - $credits;
           }
           
       } catch (Exception $e) {
           log_message('error', 'Exception in getAccountBalance: ' . $e->getMessage());
           return 0;
       }
   }

   /**
    * Get accounts by type
    */
   public function getAccountsByType($type) {
       if ($this->db->table_exists('chart_of_accounts')) {
           $this->db->where('account_type', $type);
           $this->db->where('is_active', 1);
           $this->db->order_by('account_code');
           return $this->db->get('chart_of_accounts')->result_array();
       }
       return array();
   }

   /**
    * Get revenue by source/module
    */
   public function getRevenueBySource($date_from = null, $date_to = null) {
       if (!$date_from) $date_from = date('Y-01-01');
       if (!$date_to) $date_to = date('Y-m-d');
       
       $revenue_sources = array();
       
       // Get revenue from transactions table
       if ($this->db->table_exists('transactions')) {
           try {
               $this->db->select_sum('amount');
               $this->db->from('transactions');
               $this->db->where('created_at >=', $date_from);
               $this->db->where('created_at <=', $date_to);
               $this->db->where('amount > 0');
               $result = $this->db->get()->row();
               $revenue_sources['transactions'] = $result ? ($result->amount ?: 0) : 0;
           } catch (Exception $e) {
               $revenue_sources['transactions'] = 0;
           }
       } else {
           $revenue_sources['transactions'] = 0;
       }
       
       // Try OPD Revenue if table exists
       if ($this->db->table_exists('opd_billing')) {
           try {
               $this->db->select_sum('amount');
               $this->db->from('opd_billing');
               $this->db->where('date >=', $date_from);
               $this->db->where('date <=', $date_to);
               $this->db->where('amount > 0');
               $result = $this->db->get()->row();
               $revenue_sources['opd'] = $result ? ($result->amount ?: 0) : 0;
           } catch (Exception $e) {
               $revenue_sources['opd'] = 0;
           }
       } else {
           $revenue_sources['opd'] = 0;
       }
       
       // Try IPD Revenue if table exists
       if ($this->db->table_exists('ipd_billing')) {
           try {
               $this->db->select_sum('amount');
               $this->db->from('ipd_billing');
               $this->db->where('date >=', $date_from);
               $this->db->where('date <=', $date_to);
               $this->db->where('amount > 0');
               $result = $this->db->get()->row();
               $revenue_sources['ipd'] = $result ? ($result->amount ?: 0) : 0;
           } catch (Exception $e) {
               $revenue_sources['ipd'] = 0;
           }
       } else {
           $revenue_sources['ipd'] = 0;
       }
       
       // Get other sources (pathology, radiology, pharmacy) if tables exist
       $other_modules = array('pathology', 'radiology', 'pharmacy');
       foreach ($other_modules as $module) {
           $table_name = $module . '_billing';
           if ($this->db->table_exists($table_name)) {
               try {
                   $this->db->select_sum('amount');
                   $this->db->from($table_name);
                   $this->db->where('date >=', $date_from);
                   $this->db->where('date <=', $date_to);
                   $this->db->where('amount > 0');
                   $result = $this->db->get()->row();
                   $revenue_sources[$module] = $result ? ($result->amount ?: 0) : 0;
               } catch (Exception $e) {
                   $revenue_sources[$module] = 0;
               }
           } else {
               $revenue_sources[$module] = 0;
           }
       }
       
       return $revenue_sources;
   }

   /**
    * Get cash flow data
    */
   public function getCashFlow($date_from, $date_to) {
       $cash_flow = array(
           'opening_balance' => 0,
           'cash_inflows' => array(),
           'cash_outflows' => array(),
           'closing_balance' => 0
       );
       
       // Get cash inflows (revenue)
       $revenue_sources = $this->getRevenueBySource($date_from, $date_to);
       $cash_flow['cash_inflows'] = array(
           array('description' => 'Transactions', 'amount' => $revenue_sources['transactions']),
           array('description' => 'OPD Revenue', 'amount' => $revenue_sources['opd']),
           array('description' => 'IPD Revenue', 'amount' => $revenue_sources['ipd']),
           array('description' => 'Pathology Revenue', 'amount' => $revenue_sources['pathology']),
           array('description' => 'Radiology Revenue', 'amount' => $revenue_sources['radiology']),
           array('description' => 'Pharmacy Revenue', 'amount' => $revenue_sources['pharmacy'])
       );
       
       // Get cash outflows (expenses) - placeholder for now
       $cash_flow['cash_outflows'] = array(
           array('description' => 'Staff Salaries', 'amount' => 0),
           array('description' => 'Medical Supplies', 'amount' => 0),
           array('description' => 'Utilities', 'amount' => 0),
           array('description' => 'Other Expenses', 'amount' => 0)
       );
       
       $total_inflows = array_sum(array_column($cash_flow['cash_inflows'], 'amount'));
       $total_outflows = array_sum(array_column($cash_flow['cash_outflows'], 'amount'));
       
       $cash_flow['closing_balance'] = $cash_flow['opening_balance'] + $total_inflows - $total_outflows;
       
       return $cash_flow;
   }

   /**
    * Get accounts receivable aging
    */
   public function getAccountsReceivableAging() {
       $aging_buckets = array(
           'current' => 0,      // 0-30 days
           'days_31_60' => 0,   // 31-60 days
           'days_61_90' => 0,   // 61-90 days
           'over_90' => 0       // Over 90 days
       );
       
       // Try OPD and IPD tables if they exist
       if ($this->db->table_exists('opd_billing')) {
           try {
               // Current (0-30 days)
               $this->db->select_sum('amount');
               $this->db->from('opd_billing');
               $this->db->where('status !=', 'paid');
               $this->db->where('date >=', date('Y-m-d', strtotime('-30 days')));
               $result = $this->db->get()->row();
               $aging_buckets['current'] += $result ? ($result->amount ?: 0) : 0;
               
               // 31-60 days
               $this->db->select_sum('amount');
               $this->db->from('opd_billing');
               $this->db->where('status !=', 'paid');
               $this->db->where('date <', date('Y-m-d', strtotime('-30 days')));
               $this->db->where('date >=', date('Y-m-d', strtotime('-60 days')));
               $result = $this->db->get()->row();
               $aging_buckets['days_31_60'] += $result ? ($result->amount ?: 0) : 0;
               
               // 61-90 days
               $this->db->select_sum('amount');
               $this->db->from('opd_billing');
               $this->db->where('status !=', 'paid');
               $this->db->where('date <', date('Y-m-d', strtotime('-60 days')));
               $this->db->where('date >=', date('Y-m-d', strtotime('-90 days')));
               $result = $this->db->get()->row();
               $aging_buckets['days_61_90'] += $result ? ($result->amount ?: 0) : 0;
               
               // Over 90 days
               $this->db->select_sum('amount');
               $this->db->from('opd_billing');
               $this->db->where('status !=', 'paid');
               $this->db->where('date <', date('Y-m-d', strtotime('-90 days')));
               $result = $this->db->get()->row();
               $aging_buckets['over_90'] += $result ? ($result->amount ?: 0) : 0;
               
           } catch (Exception $e) {
               log_message('error', 'Error getting OPD AR aging: ' . $e->getMessage());
           }
       }
       
       if ($this->db->table_exists('ipd_billing')) {
           try {
               // Current (0-30 days)
               $this->db->select_sum('amount');
               $this->db->from('ipd_billing');
               $this->db->where('status !=', 'paid');
               $this->db->where('date >=', date('Y-m-d', strtotime('-30 days')));
               $result = $this->db->get()->row();
               $aging_buckets['current'] += $result ? ($result->amount ?: 0) : 0;
               
               // 31-60 days
               $this->db->select_sum('amount');
               $this->db->from('ipd_billing');
               $this->db->where('status !=', 'paid');
               $this->db->where('date <', date('Y-m-d', strtotime('-30 days')));
               $this->db->where('date >=', date('Y-m-d', strtotime('-60 days')));
               $result = $this->db->get()->row();
               $aging_buckets['days_31_60'] += $result ? ($result->amount ?: 0) : 0;
               
               // 61-90 days
               $this->db->select_sum('amount');
               $this->db->from('ipd_billing');
               $this->db->where('status !=', 'paid');
               $this->db->where('date <', date('Y-m-d', strtotime('-60 days')));
               $this->db->where('date >=', date('Y-m-d', strtotime('-90 days')));
               $result = $this->db->get()->row();
               $aging_buckets['days_61_90'] += $result ? ($result->amount ?: 0) : 0;
               
               // Over 90 days
               $this->db->select_sum('amount');
               $this->db->from('ipd_billing');
               $this->db->where('status !=', 'paid');
               $this->db->where('date <', date('Y-m-d', strtotime('-90 days')));
               $result = $this->db->get()->row();
               $aging_buckets['over_90'] += $result ? ($result->amount ?: 0) : 0;
               
           } catch (Exception $e) {
               log_message('error', 'Error getting IPD AR aging: ' . $e->getMessage());
           }
       }
       
       return $aging_buckets;
   }

   /**
    * Get budget vs actual comparison
    */
   public function getBudgetVsActual($year = null) {
       if (!$year) $year = date('Y');
       
       $comparison = array();
       
       // Get actual revenue for the year
       $actual_revenue = $this->getRevenueBySource($year . '-01-01', $year . '-12-31');
       
       // Get budget data if budget table exists
       if ($this->db->table_exists('budgets')) {
           try {
               $this->db->select('coa.account_name, b.total_amount as budget_amount');
               $this->db->from('budgets b');
               $this->db->join('chart_of_accounts coa', 'coa.id = b.account_id');
               $this->db->where('b.budget_year', $year);
               $this->db->where('coa.account_type', 'revenue');
               $budget_data = $this->db->get()->result_array();
               
               foreach ($budget_data as $budget) {
                   $comparison[] = array(
                       'account' => $budget['account_name'],
                       'budget' => $budget['budget_amount'],
                       'actual' => 0, // Match with actual data
                       'variance' => 0
                   );
               }
           } catch (Exception $e) {
               log_message('error', 'Error getting budget data: ' . $e->getMessage());
           }
       }
       
       // If no budget data, create comparison with actual data only
       if (empty($comparison)) {
           $comparison = array(
               array(
                   'account' => 'Transactions',
                   'budget' => 0,
                   'actual' => $actual_revenue['transactions'],
                   'variance' => $actual_revenue['transactions']
               ),
               array(
                   'account' => 'OPD Revenue',
                   'budget' => 0,
                   'actual' => $actual_revenue['opd'],
                   'variance' => $actual_revenue['opd']
               ),
               array(
                   'account' => 'IPD Revenue',
                   'budget' => 0,
                   'actual' => $actual_revenue['ipd'],
                   'variance' => $actual_revenue['ipd']
               )
           );
       }
       
       return $comparison;
   }

   /**
    * Check if financial system is properly set up
    */
   public function isSystemSetup() {
       $required_tables = array('chart_of_accounts', 'journal_headers', 'journal_lines');
       
       foreach ($required_tables as $table) {
           if (!$this->db->table_exists($table)) {
               return false;
           }
       }
       
       // Check if we have some accounts
       try {
           $this->db->from('chart_of_accounts');
           $this->db->where('is_active', 1);
           $count = $this->db->count_all_results();
           return $count > 0;
       } catch (Exception $e) {
           return false;
       }
   }

   /**
    * Get dashboard summary metrics
    */
   public function getDashboardMetrics() {
       $current_year = date('Y');
       $current_month = date('Y-m');
       
       // Get revenue data
       $revenue_sources = $this->getRevenueBySource($current_year . '-01-01', date('Y-m-d'));
       $monthly_revenue = $this->getRevenueBySource($current_month . '-01', date('Y-m-d'));
       
       // Calculate totals
       $total_revenue = array_sum($revenue_sources);
       $monthly_revenue_total = array_sum($monthly_revenue);
       
       // Get accounts receivable
       $ar_aging = $this->getAccountsReceivableAging();
       $total_ar = array_sum($ar_aging);
       
       return array(
           'total_revenue' => $total_revenue,
           'monthly_revenue' => $monthly_revenue_total,
           'accounts_receivable' => $total_ar,
           'overdue_receivables' => $ar_aging['over_90'],
           'opd_revenue' => $revenue_sources['opd'],
           'ipd_revenue' => $revenue_sources['ipd'],
           'pathology_revenue' => $revenue_sources['pathology'],
           'radiology_revenue' => $revenue_sources['radiology'],
           'pharmacy_revenue' => $revenue_sources['pharmacy'],
           'collection_rate' => $this->getCollectionRate(),
           'cash_balance' => $this->getCashBalance()
       );
   }

   private function getCollectionRate() {
       try {
           // Calculate collection rate as percentage of bills paid
           $this->db->select('
               COUNT(*) as total_bills, 
               SUM(CASE WHEN status = "paid" THEN 1 ELSE 0 END) as paid_bills
           ');
           $this->db->from('opd_billing');
           $this->db->where('YEAR(date)', date('Y'));
           $this->db->where('amount > 0');
           $result = $this->db->get()->row();
           
           if ($result && $result->total_bills > 0) {
               return ($result->paid_bills / $result->total_bills) * 100;
           }
       } catch (Exception $e) {
           log_message('error', 'Error calculating collection rate: ' . $e->getMessage());
       }
       
       return 0;
   }

   private function getCashBalance() {
       // Get cash balance from chart of accounts if available
       if ($this->db->table_exists('chart_of_accounts') && 
           $this->db->table_exists('journal_headers') && 
           $this->db->table_exists('journal_lines')) {
           
           try {
               $cash_account = $this->getAccountByCode('1001');
               if ($cash_account) {
                   $this->db->select('
                       COALESCE(SUM(jl.debit_amount - jl.credit_amount), 0) as balance
                   ');
                   $this->db->from('journal_lines jl');
                   $this->db->join('journal_headers jh', 'jh.id = jl.journal_id');
                   $this->db->where('jl.account_id', $cash_account['id']);
                   $this->db->where('jh.status', 'posted');
                   $result = $this->db->get()->row();
                   
                   return $result ? $result->balance : 0;
               }
           } catch (Exception $e) {
               log_message('error', 'Error getting cash balance: ' . $e->getMessage());
           }
       }
       
       // Fallback: estimate from revenue
       try {
           $this->db->select_sum('amount');
           $this->db->from('opd_billing');
           $this->db->where('status', 'paid');
           $this->db->where('date >=', date('Y-m-01'));
           $opd_cash = $this->db->get()->row()->amount ?: 0;
           
           $this->db->select_sum('amount');
           $this->db->from('ipd_billing');
           $this->db->where('status', 'paid');
           $this->db->where('date >=', date('Y-m-01'));
           $ipd_cash = $this->db->get()->row()->amount ?: 0;
           
           return $opd_cash + $ipd_cash;
       } catch (Exception $e) {
           return 50000; // Default value
       }
   }

   /**
    * Sync billing data with financial system
    */
   public function syncBillingData($limit = 100) {
       $synced_count = 0;
       
       // Sync OPD billing
       try {
           $this->db->select('*');
           $this->db->from('opd_billing');
           
           // Check if journal_entry_id column exists
           if ($this->db->field_exists('journal_entry_id', 'opd_billing')) {
               $this->db->where('(journal_entry_id IS NULL OR journal_entry_id = 0)');
           }
           
           $this->db->where('amount > 0');
           $this->db->limit($limit);
           $opd_bills = $this->db->get()->result_array();
           
           foreach ($opd_bills as $bill) {
               $journal_id = $this->createJournalFromBilling('opd', $bill['id'], $bill['amount'], $bill['date']);
               if ($journal_id) {
                   // Update billing record if column exists
                   if ($this->db->field_exists('journal_entry_id', 'opd_billing')) {
                       $this->db->where('id', $bill['id']);
                       $this->db->update('opd_billing', array('journal_entry_id' => $journal_id));
                   }
                   $synced_count++;
               }
           }
       } catch (Exception $e) {
           log_message('error', 'Error syncing OPD billing: ' . $e->getMessage());
       }
       
       // Sync IPD billing
       try {
           $this->db->select('*');
           $this->db->from('ipd_billing');
           
           // Check if journal_entry_id column exists
           if ($this->db->field_exists('journal_entry_id', 'ipd_billing')) {
               $this->db->where('(journal_entry_id IS NULL OR journal_entry_id = 0)');
           }
           
           $this->db->where('amount > 0');
           $this->db->limit($limit);
           $ipd_bills = $this->db->get()->result_array();
           
           foreach ($ipd_bills as $bill) {
               $journal_id = $this->createJournalFromBilling('ipd', $bill['id'], $bill['amount'], $bill['date']);
               if ($journal_id) {
                   // Update billing record if column exists
                   if ($this->db->field_exists('journal_entry_id', 'ipd_billing')) {
                       $this->db->where('id', $bill['id']);
                       $this->db->update('ipd_billing', array('journal_entry_id' => $journal_id));
                   }
                   $synced_count++;
               }
           }
       } catch (Exception $e) {
           log_message('error', 'Error syncing IPD billing: ' . $e->getMessage());
       }
       
       return $synced_count;
   }
   
   /**
    * Test the fixed journal entry creation
    */
   public function testJournalCreation() {
       $test_result = $this->createJournalFromBilling('opd', 123, 1500.00, '2025-06-28');
       
       if ($test_result) {
           echo "✅ Journal entry created successfully! ID: " . $test_result;
           
           // Verify the entries were created
           $this->db->where('id', $test_result);
           $journal_entry = $this->db->get('journal_headers')->row();
           
           if ($journal_entry) {
               echo "<br>Journal Entry Details:";
               echo "<br>Reference: " . $journal_entry->reference_no;
               echo "<br>Amount: " . $journal_entry->total_amount;
               echo "<br>Date: " . $journal_entry->date;
               
               // Check journal lines
               $this->db->where('journal_id', $test_result);
               $lines = $this->db->get('journal_lines')->result();
               
               echo "<br><br>Journal Lines:";
               foreach ($lines as $line) {
                   echo "<br>Account ID: " . $line->account_id . " | Debit: " . $line->debit_amount . " | Credit: " . $line->credit_amount;
               }
           }
       } else {
           echo "❌ Failed to create journal entry";
           echo "<br>Last error: " . $this->db->error()['message'];
       }
   }
   
   /**
    * Debug profit and loss with detailed information
    */
   public function debugProfitLoss() {
       // Enable error logging
       ini_set('display_errors', 1);
       error_reporting(E_ALL);
       
       $date_from = $this->input->get('date_from') ?: date('Y-01-01');
       $date_to = $this->input->get('date_to') ?: date('Y-m-d');
       
       echo "<h2>Debug Information for Profit & Loss</h2>";
       echo "<p><strong>Date Range:</strong> {$date_from} to {$date_to}</p>";
       
       // Check table existence
       echo "<h3>1. Table Existence Check</h3>";
       $tables = array('chart_of_accounts', 'journal_headers', 'journal_lines');
       foreach ($tables as $table) {
           $exists = $this->db->table_exists($table) ? 'EXISTS' : 'DOES NOT EXIST';
           echo "<p>{$table}: <strong>{$exists}</strong></p>";
       }
       
       // Chart of Accounts Analysis
       echo "<h3>2. Chart of Accounts Analysis</h3>";
       try {
           $accounts = $this->debugChartOfAccounts();
           echo "<p>Found " . count($accounts) . " active accounts</p>";
           
           if (!empty($accounts)) {
               echo "<table border='1' style='border-collapse: collapse;'>";
               echo "<tr><th>Code</th><th>Name</th><th>Type</th><th>Active</th></tr>";
               foreach (array_slice($accounts, 0, 10) as $account) {
                   echo "<tr>";
                   echo "<td>{$account['account_code']}</td>";
                   echo "<td>{$account['account_name']}</td>";
                   echo "<td>{$account['account_type']}</td>";
                   echo "<td>{$account['is_active']}</td>";
                   echo "</tr>";
               }
               echo "</table>";
           }
       } catch (Exception $e) {
           echo "<p style='color: red;'>Error: " . $e->getMessage() . "</p>";
       }
       
       // Journal Entries Analysis
       echo "<h3>3. Journal Entries Analysis</h3>";
       try {
           $journal_entries = $this->debugJournalEntries($date_from, $date_to);
           echo "<p>Found " . count($journal_entries) . " journal entries in date range</p>";
           
           if (!empty($journal_entries)) {
               echo "<table border='1' style='border-collapse: collapse;'>";
               echo "<tr><th>Ref No</th><th>Date</th><th>Account</th><th>Debit</th><th>Credit</th></tr>";
               foreach (array_slice($journal_entries, 0, 10) as $entry) {
                   echo "<tr>";
                   echo "<td>{$entry['reference_no']}</td>";
                   echo "<td>{$entry['date']}</td>";
                   echo "<td>{$entry['account_code']} - {$entry['account_name']}</td>";
                   echo "<td>" . number_format($entry['debit_amount'], 2) . "</td>";
                   echo "<td>" . number_format($entry['credit_amount'], 2) . "</td>";
                   echo "</tr>";
               }
               echo "</table>";
           }
       } catch (Exception $e) {
           echo "<p style='color: red;'>Error: " . $e->getMessage() . "</p>";
       }
       
       // Test Income Statement
       echo "<h3>4. Income Statement Test</h3>";
       try {
           $income_statement = $this->getIncomeStatement($date_from, $date_to);
           echo "<p>Revenue accounts: " . count($income_statement['revenue']) . "</p>";
           echo "<p>Expense accounts: " . count($income_statement['expenses']) . "</p>";
           
           if (!empty($income_statement['revenue'])) {
               echo "<h4>Revenue:</h4>";
               foreach ($income_statement['revenue'] as $rev) {
                   echo "<p>{$rev['account_code']} - {$rev['account_name']}: " . number_format($rev['amount'], 2) . "</p>";
               }
           }
           
           if (!empty($income_statement['expenses'])) {
               echo "<h4>Expenses:</h4>";
               foreach ($income_statement['expenses'] as $exp) {
                   echo "<p>{$exp['account_code']} - {$exp['account_name']}: " . number_format($exp['amount'], 2) . "</p>";
               }
           }
       } catch (Exception $e) {
           echo "<p style='color: red;'>Error: " . $e->getMessage() . "</p>";
       }
       
       echo "<hr>";
       echo "<p><a href='" . base_url() . "admin/financial/profitloss?date_from={$date_from}&date_to={$date_to}'>Back to Profit & Loss</a></p>";
   }

   /**
    * Create a test journal entry for debugging
    */
   public function createTestJournalEntry() {
       if (!$this->db->table_exists('journal_headers') || !$this->db->table_exists('journal_lines')) {
           return 'Required journal tables do not exist';
       }

       try {
           // Get cash and revenue accounts
           $cash_account = $this->getAccountByCode('1001');
           $revenue_account = $this->getAccountByCode('4001');
           
           if (!$cash_account || !$revenue_account) {
               return 'Missing required accounts (Cash: 1001, OPD Revenue: 4001)';
           }

           // Create journal header
           $journal_data = array(
               'reference_no' => 'TEST-JE-' . date('YmdHis'),
               'description' => 'Test journal entry for system verification',
               'total_amount' => 1000.00,
               'date' => date('Y-m-d'),
               'status' => 'posted',
               'source_module' => 'test',
               'created_by' => 1
           );
           
           $this->db->insert('journal_headers', $journal_data);
           $journal_id = $this->db->insert_id();

           if (!$journal_id) {
               return 'Failed to create journal header';
           }

           // Create debit line (Cash)
           $this->db->insert('journal_lines', array(
               'journal_id' => $journal_id,
               'account_id' => $cash_account['id'],
               'description' => 'Test cash receipt',
               'debit_amount' => 1000.00,
               'credit_amount' => 0
           ));
           
           // Create credit line (Revenue)
           $this->db->insert('journal_lines', array(
               'journal_id' => $journal_id,
               'account_id' => $revenue_account['id'],
               'description' => 'Test revenue',
               'debit_amount' => 0,
               'credit_amount' => 1000.00
           ));

           return 'Test journal entry created successfully with ID: ' . $journal_id;
       } catch (Exception $e) {
           return 'Error creating test journal: ' . $e->getMessage();
       }
   }

   /**
    * Debug information about your actual database schema
    */
   public function debugDatabaseSchema() {
       $debug_info = array();
       
       // Check table existence
       $required_tables = array('chart_of_accounts', 'journal_headers', 'journal_lines');
       foreach ($required_tables as $table) {
           $debug_info['tables'][$table] = $this->db->table_exists($table);
       }
       
       // Check chart of accounts data
       if ($this->db->table_exists('chart_of_accounts')) {
           try {
               $this->db->select('COUNT(*) as total, account_type, is_active');
               $this->db->from('chart_of_accounts');
               $this->db->group_by('account_type, is_active');
               $debug_info['accounts'] = $this->db->get()->result_array();
           } catch (Exception $e) {
               $debug_info['accounts_error'] = $e->getMessage();
           }
       }
       
       // Check journal data
       if ($this->db->table_exists('journal_headers')) {
           try {
               $this->db->select('COUNT(*) as total, status');
               $this->db->from('journal_headers');
               $this->db->group_by('status');
               $debug_info['journals'] = $this->db->get()->result_array();
           } catch (Exception $e) {
               $debug_info['journals_error'] = $e->getMessage();
           }
       }
       
       return $debug_info;
   }
   
   /**
 * Get Profit & Loss data specifically for the P&L view - SAFE NEW METHOD
 * This method returns data in the exact format expected by the P&L view
 * without affecting other financial pages
 */
public function getProfitLossData($date_from, $date_to) {
    // Get the standard income statement data
    $standard_data = $this->getIncomeStatement($date_from, $date_to);
    
    // Transform it to match what the P&L view expects
    $profit_loss_data = array(
        // Keep original keys for backward compatibility
        'revenue' => $standard_data['revenue'],
        'expenses' => $standard_data['expenses'],
        
        // Add the keys that the P&L view specifically looks for
        'revenue_accounts' => $standard_data['revenue'],
        'expense_accounts' => $standard_data['expenses'],
        
        // Calculate totals
        'total_revenue' => 0,
        'total_expenses' => 0,
        'net_income' => 0
    );
    
    // Calculate total revenue
    if (!empty($profit_loss_data['revenue_accounts'])) {
        foreach ($profit_loss_data['revenue_accounts'] as $revenue) {
            $profit_loss_data['total_revenue'] += $revenue['amount'];
        }
    }
    
    // Calculate total expenses  
    if (!empty($profit_loss_data['expense_accounts'])) {
        foreach ($profit_loss_data['expense_accounts'] as $expense) {
            $profit_loss_data['total_expenses'] += $expense['amount'];
        }
    }
    
    // Calculate net income
    $profit_loss_data['net_income'] = $profit_loss_data['total_revenue'] - $profit_loss_data['total_expenses'];
    
    return $profit_loss_data;
}

/**
 * Enhanced fallback that works with your billing tables
 * This ensures you see data even without journal entries
 */
public function getProfitLossDataFallback($date_from, $date_to) {
    $revenue_accounts = array();
    $expense_accounts = array();
    
    // Updated billing tables with correct field mappings based on your schema
    $billing_tables = array(
        'opd_billing' => array(
            'code' => '4001', 
            'name' => 'OPD Revenue',
            'amount_field' => 'net_amount',
            'date_field' => 'date'
        ),
        'ipd_billing' => array(
            'code' => '4002', 
            'name' => 'IPD Revenue',
            'amount_field' => 'net_amount',
            'date_field' => 'date'
        ),
        'pathology_billing' => array(
            'code' => '4003', 
            'name' => 'Pathology Revenue',
            'amount_field' => 'net_amount',
            'date_field' => 'date'
        ),
        'radiology_billing' => array(
            'code' => '4004', 
            'name' => 'Radiology Revenue',
            'amount_field' => 'net_amount',
            'date_field' => 'date'
        ),
        'pharmacy_bill_basic' => array(  // Correct table name
            'code' => '4005', 
            'name' => 'Pharmacy Revenue',
            'amount_field' => 'net_amount',
            'date_field' => 'date'
        ),
        // Note: laboratory_billing doesn't exist in your schema, so removing it
        'appointment_payment' => array(
            'code' => '4007', 
            'name' => 'Appointment Revenue',
            'amount_field' => 'paid_amount',
            'date_field' => 'date'
        ),
        'transactions' => array(
            'code' => '4008', 
            'name' => 'Other Payments',
            'amount_field' => 'amount',
            'date_field' => 'payment_date'
        )
    );
    
    foreach ($billing_tables as $table => $info) {
        if ($this->db->table_exists($table)) {
            try {
                $amount_field = $info['amount_field'];
                $date_field = $info['date_field'];
                
                $this->db->select("SUM($amount_field) as total_amount, COUNT(*) as record_count");
                $this->db->from($table);
                
                // Handle different date field formats
                if ($table == 'transactions') {
                    // transactions table uses payment_date (datetime)
                    $this->db->where("DATE($date_field) >=", $date_from);
                    $this->db->where("DATE($date_field) <=", $date_to);
                    // Only include payment transactions, not refunds
                    $this->db->where('type', 'payment');
                } else {
                    // Other tables use date (datetime or date)
                    $this->db->where("DATE($date_field) >=", $date_from);
                    $this->db->where("DATE($date_field) <=", $date_to);
                }
                
                $this->db->where("$amount_field > 0");
                $this->db->where("$date_field IS NOT NULL");
                
                $result = $this->db->get();
                
                if ($result !== FALSE) {
                    $row = $result->row();
                    
                    if ($row && $row->total_amount > 0) {
                        $revenue_accounts[] = array(
                            'id' => null,
                            'account_code' => $info['code'],
                            'account_name' => $info['name'],
                            'account_type' => 'revenue',
                            'amount' => floatval($row->total_amount),
                            'record_count' => $row->record_count,
                            'source_table' => $table
                        );
                    }
                } else {
                    log_message('error', 'Query failed for table: ' . $table . ' - ' . $this->db->error()['message']);
                }
                
            } catch (Exception $e) {
                log_message('error', 'Error getting revenue from ' . $table . ': ' . $e->getMessage());
            }
        }
    }
    
    // Get expense data from supplier bills and other expense sources
    $expense_tables = array(
        'supplier_bill_basic' => array(
            'code' => '5001',
            'name' => 'Supplier Expenses',
            'amount_field' => 'net_amount',
            'date_field' => 'date'
        )
    );
    
    foreach ($expense_tables as $table => $info) {
        if ($this->db->table_exists($table)) {
            try {
                $amount_field = $info['amount_field'];
                $date_field = $info['date_field'];
                
                $this->db->select("SUM($amount_field) as total_amount, COUNT(*) as record_count");
                $this->db->from($table);
                $this->db->where("DATE($date_field) >=", $date_from);
                $this->db->where("DATE($date_field) <=", $date_to);
                $this->db->where("$amount_field > 0");
                $this->db->where("$date_field IS NOT NULL");
                
                $result = $this->db->get();
                
                if ($result !== FALSE) {
                    $row = $result->row();
                    
                    if ($row && $row->total_amount > 0) {
                        $expense_accounts[] = array(
                            'id' => null,
                            'account_code' => $info['code'],
                            'account_name' => $info['name'],
                            'account_type' => 'expense',
                            'amount' => floatval($row->total_amount),
                            'record_count' => $row->record_count,
                            'source_table' => $table
                        );
                    }
                } else {
                    log_message('error', 'Query failed for expense table: ' . $table . ' - ' . $this->db->error()['message']);
                }
                
            } catch (Exception $e) {
                log_message('error', 'Error getting expenses from ' . $table . ': ' . $e->getMessage());
            }
        }
    }
    
    // Calculate totals
    $total_revenue = 0;
    foreach ($revenue_accounts as $revenue) {
        $total_revenue += $revenue['amount'];
    }
    
    $total_expenses = 0;
    foreach ($expense_accounts as $expense) {
        $total_expenses += $expense['amount'];
    }
    
    return array(
        'revenue' => $revenue_accounts,
        'expenses' => $expense_accounts,
        'revenue_accounts' => $revenue_accounts,
        'expense_accounts' => $expense_accounts,
        'total_revenue' => $total_revenue,
        'total_expenses' => $total_expenses,
        'net_income' => $total_revenue - $total_expenses,
        'data_source' => 'billing_tables' // Indicator of data source
    );
}

/**
 * Check what billing data you actually have
 */
public function checkAvailableData($date_from, $date_to) {
    $available_data = array();
    
    // Updated billing tables based on your actual schema
    // Note: laboratory_billing doesn't exist in your schema, removing it
    $billing_tables = array(
        'opd_billing' => array(
            'amount_field' => 'net_amount',  // Using net_amount instead of amount
            'date_field' => 'date'
        ),
        'ipd_billing' => array(
            'amount_field' => 'net_amount',  // Using net_amount instead of amount
            'date_field' => 'date'
        ),
        'pathology_billing' => array(
            'amount_field' => 'net_amount',  // Using net_amount instead of amount
            'date_field' => 'date'
        ),
        'radiology_billing' => array(
            'amount_field' => 'net_amount',  // Using net_amount instead of amount
            'date_field' => 'date'
        ),
        'pharmacy_bill_basic' => array(  // Correct table name is pharmacy_bill_basic, not pharmacy_billing
            'amount_field' => 'net_amount',  // Using net_amount instead of amount
            'date_field' => 'date'
        ),
        // Adding transactions table for additional billing data
        'transactions' => array(
            'amount_field' => 'amount',
            'date_field' => 'payment_date'
        )
    );
    
    foreach ($billing_tables as $table => $config) {
        if ($this->db->table_exists($table)) {
            try {
                $amount_field = $config['amount_field'];
                $date_field = $config['date_field'];
                
                // Get overall statistics
                $this->db->select("COUNT(*) as total_records, SUM($amount_field) as total_amount, MIN($date_field) as earliest_date, MAX($date_field) as latest_date");
                $this->db->from($table);
                $this->db->where("$amount_field > 0");
                $this->db->where("$date_field IS NOT NULL");
                $result = $this->db->get()->row();
                
                $available_data[$table] = array(
                    'exists' => true,
                    'total_records' => $result->total_records ? $result->total_records : 0,
                    'total_amount' => $result->total_amount ? $result->total_amount : 0,
                    'earliest_date' => $result->earliest_date,
                    'latest_date' => $result->latest_date,
                    'in_date_range' => 0,
                    'range_amount' => 0
                );
                
                // Check records in the specific date range
                $this->db->select("COUNT(*) as range_records, SUM($amount_field) as range_amount");
                $this->db->from($table);
                
                // Handle different date field formats
                if ($table == 'transactions') {
                    // transactions table uses payment_date (datetime)
                    $this->db->where("DATE($date_field) >=", $date_from);
                    $this->db->where("DATE($date_field) <=", $date_to);
                } else {
                    // Other tables use date (datetime)
                    $this->db->where("DATE($date_field) >=", $date_from);
                    $this->db->where("DATE($date_field) <=", $date_to);
                }
                
                $this->db->where("$amount_field > 0");
                $this->db->where("$date_field IS NOT NULL");
                $range_result = $this->db->get()->row();
                
                $available_data[$table]['in_date_range'] = $range_result->range_records ? $range_result->range_records : 0;
                $available_data[$table]['range_amount'] = $range_result->range_amount ? $range_result->range_amount : 0;
                
            } catch (Exception $e) {
                $available_data[$table] = array('exists' => true, 'error' => $e->getMessage());
            }
        } else {
            $available_data[$table] = array('exists' => false);
        }
    }
    
    // Additional check for appointment payments
    if ($this->db->table_exists('appointment_payment')) {
        try {
            $this->db->select('COUNT(*) as total_records, SUM(paid_amount) as total_amount, MIN(date) as earliest_date, MAX(date) as latest_date');
            $this->db->from('appointment_payment');
            $this->db->where('paid_amount > 0');
            $result = $this->db->get()->row();
            
            $available_data['appointment_payment'] = array(
                'exists' => true,
                'total_records' => $result->total_records ? $result->total_records : 0,
                'total_amount' => $result->total_amount ? $result->total_amount : 0,
                'earliest_date' => $result->earliest_date,
                'latest_date' => $result->latest_date,
                'in_date_range' => 0,
                'range_amount' => 0
            );
            
            // Check records in the specific date range
            $this->db->select('COUNT(*) as range_records, SUM(paid_amount) as range_amount');
            $this->db->from('appointment_payment');
            $this->db->where('DATE(date) >=', $date_from);
            $this->db->where('DATE(date) <=', $date_to);
            $this->db->where('paid_amount > 0');
            $range_result = $this->db->get()->row();
            
            $available_data['appointment_payment']['in_date_range'] = $range_result->range_records ? $range_result->range_records : 0;
            $available_data['appointment_payment']['range_amount'] = $range_result->range_amount ? $range_result->range_amount : 0;
            
        } catch (Exception $e) {
            $available_data['appointment_payment'] = array('exists' => true, 'error' => $e->getMessage());
        }
    }
    
    return $available_data;
}

/**
 * Get account ID by account code
 */
private function get_account_id($account_code) {
        log_message('info', "FINANCIAL DEBUG - Looking for account code: {$account_code}");
        
        $this->db->select('id, account_name');
        $this->db->where('account_code', $account_code);
        $this->db->where('is_active', 1);
        $query = $this->db->get('chart_of_accounts');
        
        if ($query->num_rows() > 0) {
            $row = $query->row();
            log_message('info', "FINANCIAL DEBUG - Found account: {$account_code} = {$row->account_name} (ID: {$row->id})");
            return $row->id;
        }
        
        log_message('error', "FINANCIAL DEBUG - Account NOT found: {$account_code}");
        return null;
    }
}

