<?php

if (!defined('BASEPATH'))
   exit('No direct script access allowed');

class Consultation_model extends CI_Model {

   public function __construct() {
       parent::__construct();
   }

   // =============================================
   // CONSULTATION MANAGEMENT METHODS
   // =============================================

   /**
    * Save a new consultation with fingerprint verification status
    */
   public function saveConsultation($data) {
       // Ensure verification_status is set
       if (!isset($data['verification_status'])) {
           $data['verification_status'] = 'unverified';
       }
       
       $this->db->insert('consultations', $data);
       return $this->db->insert_id();
   }

   /**
    * Update an existing consultation
    */
   public function updateConsultation($consultationId, $data) {
       $this->db->where('id', $consultationId);
       return $this->db->update('consultations', $data);
   }

   /**
    * Get consultation details by ID with verification status and diagnosis information
    */
   public function getConsultationDetails($consultationId) {
       $this->db->select('c.*, p.patient_name, p.mobileno as patient_phone, p.age, p.gender, 
                         p.fingerprint_template, p.fingerprint_image,
                         CONCAT(s.name, " ", s.surname) as doctor_name,
                         CASE c.verification_status 
                           WHEN "verified" THEN "Verified" 
                           WHEN "skipped" THEN "Skipped" 
                           ELSE "Unverified" 
                         END as verification_display');
       $this->db->from('consultations c');
       $this->db->join('patients p', 'p.id = c.patient_id', 'left');
       $this->db->join('staff s', 's.id = c.doctor_id', 'left');
       $this->db->where('c.id', $consultationId);
       
       $query = $this->db->get();
       $consultation = $query->row_array();
       
       if ($consultation) {
           // Get associated ICD-10 diagnoses
           $consultation['diagnoses'] = $this->getConsultationDiagnoses($consultationId);
           $consultation['has_icd_diagnosis'] = count($consultation['diagnoses']) > 0;
       }
       
       return $consultation;
   }

   /**
    * Get consultations list for DataTable with fingerprint verification info
    */
   public function getConsultationsList($start = 0, $length = 10, $search = '', $orderColumn = 'id', $orderDir = 'desc') {
       $this->db->select('c.id, c.consultation_date, c.symptoms, c.diagnosis, c.created_at, c.verification_status,
                         p.patient_name, p.id as patient_id, p.fingerprint_template,
                         CONCAT(s.name, " ", s.surname) as doctor_name,
                         CASE c.verification_status 
                           WHEN "verified" THEN "Verified" 
                           WHEN "skipped" THEN "Skipped" 
                           ELSE "Unverified" 
                         END as verification_display');
       $this->db->from('consultations c');
       $this->db->join('patients p', 'p.id = c.patient_id', 'left');
       $this->db->join('staff s', 's.id = c.doctor_id', 'left');
       
       if (!empty($search)) {
           $this->db->group_start();
           $this->db->like('p.patient_name', $search);
           $this->db->or_like('c.symptoms', $search);
           $this->db->or_like('c.diagnosis', $search);
           $this->db->or_like('CONCAT(s.name, " ", s.surname)', $search);
           $this->db->or_like('c.verification_status', $search);
           $this->db->group_end();
       }
       
       $this->db->order_by($orderColumn, $orderDir);
       $this->db->limit($length, $start);
       
       $query = $this->db->get();
       $results = $query->result_array();
       
       // Add fingerprint availability info and ICD diagnosis info
       foreach ($results as &$result) {
           $result['patient_has_fingerprint'] = !empty($result['fingerprint_template']);
           $result['diagnoses'] = $this->getConsultationDiagnoses($result['id']);
           $result['has_icd_diagnosis'] = count($result['diagnoses']) > 0;
       }
       
       return $results;
   }

   /**
    * Get total consultations count with search support
    */
   public function getConsultationsCount($search = '') {
       $this->db->from('consultations c');
       $this->db->join('patients p', 'p.id = c.patient_id', 'left');
       $this->db->join('staff s', 's.id = c.doctor_id', 'left');
       
       if (!empty($search)) {
           $this->db->group_start();
           $this->db->like('p.patient_name', $search);
           $this->db->or_like('c.symptoms', $search);
           $this->db->or_like('c.diagnosis', $search);
           $this->db->or_like('CONCAT(s.name, " ", s.surname)', $search);
           $this->db->or_like('c.verification_status', $search);
           $this->db->group_end();
       }
       
       return $this->db->count_all_results();
   }

   /**
    * Get patient's consultation history with verification status
    */
   public function getPatientConsultations($patientId, $limit = 10) {
       $this->db->select('c.id, c.consultation_date as date, c.symptoms, c.diagnosis, c.notes, c.verification_status,
                         CONCAT(s.name, " ", s.surname) as doctor_name,
                         CASE c.verification_status 
                           WHEN "verified" THEN "Verified" 
                           WHEN "skipped" THEN "Skipped" 
                           ELSE "Unverified" 
                         END as verification_display');
       $this->db->from('consultations c');
       $this->db->join('staff s', 's.id = c.doctor_id', 'left');
       $this->db->where('c.patient_id', $patientId);
       $this->db->order_by('c.consultation_date', 'DESC');
       $this->db->limit($limit);
       
       $query = $this->db->get();
       $consultations = $query->result_array();
       
       // Add diagnosis information to each consultation
       foreach ($consultations as &$consultation) {
           $consultation['diagnoses'] = $this->getConsultationDiagnoses($consultation['id']);
           $consultation['has_icd_diagnosis'] = count($consultation['diagnoses']) > 0;
       }
       
       return $consultations;
   }

   /**
    * Delete a consultation
    */
   public function deleteConsultation($consultationId) {
       // Start transaction
       $this->db->trans_start();
       
       // Delete associated diagnosis records first
       $this->db->where('consultation_id', $consultationId);
       $this->db->delete('consultation_diagnosis');
       
       // Delete the consultation
       $this->db->where('id', $consultationId);
       $this->db->delete('consultations');
       
       // Complete transaction
       $this->db->trans_complete();
       
       return $this->db->trans_status();
   }

   // =============================================
   // ICD-10 DIAGNOSIS METHODS (NEW)
   // =============================================

   /**
    * Get all active ICD-10 diagnosis codes
    */
   public function getIcdDiagnosisCodes($icdVersion = 'ICD10', $limit = 1000) {
       $this->db->select('id, icd_version, code, title, description, category, subcategory');
       $this->db->from('diagnosis_codes');
       $this->db->where('is_active', 1);
       $this->db->where('icd_version', $icdVersion);
       $this->db->order_by('category ASC, code ASC');
       $this->db->limit($limit);
       
       $query = $this->db->get();
       return $query->result_array();
   }

   /**
    * Search ICD-10 diagnosis codes
    */
   public function searchIcdDiagnosisCodes($searchTerm, $icdVersion = 'ICD10', $category = null, $limit = 50) {
       $this->db->select('id, code, title, category, subcategory, description');
       $this->db->from('diagnosis_codes');
       $this->db->where('is_active', 1);
       $this->db->where('icd_version', $icdVersion);
       
       // Search in multiple fields
       if (!empty($searchTerm)) {
           $this->db->group_start();
           $this->db->like('code', $searchTerm);
           $this->db->or_like('title', $searchTerm);
           $this->db->or_like('description', $searchTerm);
           $this->db->or_like('category', $searchTerm);
           $this->db->group_end();
       }
       
       // Filter by category if specified
       if ($category) {
           $this->db->where('category', $category);
       }
       
       $this->db->order_by('code ASC');
       $this->db->limit($limit);
       
       $query = $this->db->get();
       return $query->result_array();
   }

   /**
    * Get diagnosis categories
    */
   public function getDiagnosisCategories($icdVersion = 'ICD10') {
       $this->db->select('category, COUNT(*) as count');
       $this->db->from('diagnosis_codes');
       $this->db->where('is_active', 1);
       $this->db->where('icd_version', $icdVersion);
       $this->db->where('category IS NOT NULL');
       $this->db->where('category !=', '');
       $this->db->group_by('category');
       $this->db->order_by('category ASC');
       
       $query = $this->db->get();
       return $query->result_array();
   }

   /**
    * Get diagnosis by ID
    */
   public function getDiagnosisById($diagnosisId) {
       $this->db->select('*');
       $this->db->from('diagnosis_codes');
       $this->db->where('id', $diagnosisId);
       $this->db->where('is_active', 1);
       
       $query = $this->db->get();
       return $query->row_array();
   }

   /**
    * Save consultation diagnosis association
    */
   public function saveConsultationDiagnosis($consultationId, $diagnosisCodeId, $isPrimary = 1, $notes = '') {
       // Remove existing primary diagnosis if setting a new primary
       if ($isPrimary) {
           $this->db->where('consultation_id', $consultationId);
           $this->db->set('is_primary', 0);
           $this->db->update('consultation_diagnosis');
       }
       
       $data = [
           'consultation_id' => $consultationId,
           'diagnosis_code_id' => $diagnosisCodeId,
           'is_primary' => $isPrimary ? 1 : 0,
           'notes' => $notes,
           'created_at' => date('Y-m-d H:i:s')
       ];
       
       $this->db->insert('consultation_diagnosis', $data);
       return $this->db->insert_id();
   }

   /**
    * Get consultation diagnoses with full diagnosis details
    */
   public function getConsultationDiagnoses($consultationId) {
       $this->db->select('cd.*, dc.code, dc.title, dc.category, dc.subcategory, dc.description, dc.icd_version');
       $this->db->from('consultation_diagnosis cd');
       $this->db->join('diagnosis_codes dc', 'cd.diagnosis_code_id = dc.id', 'left');
       $this->db->where('cd.consultation_id', $consultationId);
       $this->db->order_by('cd.is_primary DESC, cd.created_at ASC');
       
       $query = $this->db->get();
       return $query->result_array();
   }

   /**
    * Remove consultation diagnosis association
    */
   public function removeConsultationDiagnosis($consultationId, $diagnosisCodeId = null) {
       $this->db->where('consultation_id', $consultationId);
       
       if ($diagnosisCodeId) {
           $this->db->where('diagnosis_code_id', $diagnosisCodeId);
       }
       
       return $this->db->delete('consultation_diagnosis');
   }

   /**
    * Get most frequently used diagnoses
    */
   public function getMostUsedDiagnoses($limit = 20, $dateFrom = null, $dateTo = null) {
       $this->db->select('dc.code, dc.title, dc.category, COUNT(cd.id) as usage_count,
                         MAX(cd.created_at) as last_used');
       $this->db->from('consultation_diagnosis cd');
       $this->db->join('diagnosis_codes dc', 'cd.diagnosis_code_id = dc.id');
       
       if ($dateFrom) {
           $this->db->where('cd.created_at >=', $dateFrom);
       }
       
       if ($dateTo) {
           $this->db->where('cd.created_at <=', $dateTo);
       }
       
       $this->db->group_by('dc.id');
       $this->db->order_by('usage_count DESC');
       $this->db->limit($limit);
       
       $query = $this->db->get();
       return $query->result_array();
   }

   /**
    * Get diagnosis usage statistics by category
    */
   public function getDiagnosisUsageByCategory($dateFrom = null, $dateTo = null) {
       $this->db->select('dc.category, COUNT(cd.id) as usage_count,
                         COUNT(DISTINCT cd.consultation_id) as consultation_count,
                         COUNT(DISTINCT dc.id) as unique_diagnoses');
       $this->db->from('consultation_diagnosis cd');
       $this->db->join('diagnosis_codes dc', 'cd.diagnosis_code_id = dc.id');
       $this->db->where('dc.category IS NOT NULL');
       
       if ($dateFrom) {
           $this->db->where('cd.created_at >=', $dateFrom);
       }
       
       if ($dateTo) {
           $this->db->where('cd.created_at <=', $dateTo);
       }
       
       $this->db->group_by('dc.category');
       $this->db->order_by('usage_count DESC');
       
       $query = $this->db->get();
       return $query->result_array();
   }

   /**
    * Add new diagnosis code
    */
   public function addDiagnosisCode($data) {
       $data['created_at'] = date('Y-m-d H:i:s');
       $data['updated_at'] = date('Y-m-d H:i:s');
       
       $this->db->insert('diagnosis_codes', $data);
       return $this->db->insert_id();
   }

   /**
    * Update diagnosis code
    */
   public function updateDiagnosisCode($diagnosisId, $data) {
       $data['updated_at'] = date('Y-m-d H:i:s');
       
       $this->db->where('id', $diagnosisId);
       return $this->db->update('diagnosis_codes', $data);
   }

   /**
    * Bulk insert diagnosis codes (for importing ICD datasets)
    */
   public function bulkInsertDiagnosisCodes($diagnosisArray) {
       if (empty($diagnosisArray)) {
           return false;
       }
       
       // Add timestamps to all records
       foreach ($diagnosisArray as &$diagnosis) {
           $diagnosis['created_at'] = date('Y-m-d H:i:s');
           $diagnosis['updated_at'] = date('Y-m-d H:i:s');
       }
       
       return $this->db->insert_batch('diagnosis_codes', $diagnosisArray);
   }

   // =============================================
   // FINGERPRINT VERIFICATION METHODS
   // =============================================

   /**
    * Get patient's fingerprint data for verification
    */
   public function getPatientFingerprintData($patientId) {
       $this->db->select('id, patient_name, fingerprint_template, fingerprint_image');
       $this->db->from('patients');
       $this->db->where('id', $patientId);
       $this->db->where('is_active', 'yes');
       
       $query = $this->db->get();
       $result = $query->row_array();
       
       if ($result) {
           $result['has_fingerprint'] = !empty($result['fingerprint_template']);
           return $result;
       }
       
       return null;
   }

   /**
    * Log fingerprint verification attempt
    */
   public function logFingerprintVerification($patientId, $matchScore, $isMatch, $staffId = null, $ipAddress = null) {
       $logData = [
           'patient_id' => $patientId,
           'match_score' => $matchScore,
           'is_match' => $isMatch ? 1 : 0,
           'verification_date' => date('Y-m-d H:i:s'),
           'staff_id' => $staffId,
           'ip_address' => $ipAddress
       ];
       
       return $this->db->insert('fingerprint_verification_log', $logData);
   }

   /**
    * Get fingerprint verification history for a patient
    */
   public function getFingerprintVerificationHistory($patientId, $limit = 20) {
       $this->db->select('fvl.*, s.name as staff_name, s.surname as staff_surname');
       $this->db->from('fingerprint_verification_log fvl');
       $this->db->join('staff s', 's.id = fvl.staff_id', 'left');
       $this->db->where('fvl.patient_id', $patientId);
       $this->db->order_by('fvl.verification_date', 'DESC');
       $this->db->limit($limit);
       
       $query = $this->db->get();
       return $query->result_array();
   }

   /**
    * Get verification statistics
    */
   public function getVerificationStats($dateFrom = null, $dateTo = null) {
       if (!$dateFrom) $dateFrom = date('Y-m-01'); // First day of current month
       if (!$dateTo) $dateTo = date('Y-m-d'); // Today
       
       // Total verifications
       $this->db->where('verification_date >=', $dateFrom);
       $this->db->where('verification_date <=', $dateTo);
       $totalVerifications = $this->db->count_all_results('fingerprint_verification_log');
       
       // Successful verifications
       $this->db->where('verification_date >=', $dateFrom);
       $this->db->where('verification_date <=', $dateTo);
       $this->db->where('is_match', 1);
       $successfulVerifications = $this->db->count_all_results('fingerprint_verification_log');
       
       // Failed verifications
       $this->db->where('verification_date >=', $dateFrom);
       $this->db->where('verification_date <=', $dateTo);
       $this->db->where('is_match', 0);
       $failedVerifications = $this->db->count_all_results('fingerprint_verification_log');
       
       // Daily verification counts
       $this->db->select('DATE(verification_date) as date, 
                         COUNT(*) as total_count,
                         SUM(is_match) as successful_count');
       $this->db->from('fingerprint_verification_log');
       $this->db->where('verification_date >=', $dateFrom);
       $this->db->where('verification_date <=', $dateTo);
       $this->db->group_by('DATE(verification_date)');
       $this->db->order_by('date', 'ASC');
       $dailyStats = $this->db->get()->result_array();
       
       // Average match scores
       $this->db->select('AVG(match_score) as avg_match_score');
       $this->db->from('fingerprint_verification_log');
       $this->db->where('verification_date >=', $dateFrom);
       $this->db->where('verification_date <=', $dateTo);
       $this->db->where('is_match', 1);
       $avgSuccessScore = $this->db->get()->row()->avg_match_score ?? 0;
       
       $this->db->select('AVG(match_score) as avg_match_score');
       $this->db->from('fingerprint_verification_log');
       $this->db->where('verification_date >=', $dateFrom);
       $this->db->where('verification_date <=', $dateTo);
       $this->db->where('is_match', 0);
       $avgFailScore = $this->db->get()->row()->avg_match_score ?? 0;
       
       return [
           'total_verifications' => $totalVerifications,
           'successful_verifications' => $successfulVerifications,
           'failed_verifications' => $failedVerifications,
           'success_rate' => $totalVerifications > 0 ? round(($successfulVerifications / $totalVerifications) * 100, 2) : 0,
           'avg_success_score' => round($avgSuccessScore, 2),
           'avg_fail_score' => round($avgFailScore, 2),
           'daily_stats' => $dailyStats
       ];
   }

   // =============================================
   // ENHANCED CONSULTATION STATISTICS WITH ICD-10 DATA
   // =============================================

   /**
    * Get comprehensive consultation and diagnosis statistics
    */
   public function getComprehensiveConsultationStats($dateFrom = null, $dateTo = null) {
       if (!$dateFrom) $dateFrom = date('Y-m-01');
       if (!$dateTo) $dateTo = date('Y-m-d');
       
       $stats = [];
       
       // Basic consultation stats
       $stats['consultation_stats'] = $this->getConsultationStatsWithVerification($dateFrom, $dateTo);
       
       // Diagnosis-specific statistics
       $stats['diagnosis_stats'] = [
           'most_used_diagnoses' => $this->getMostUsedDiagnoses(20, $dateFrom, $dateTo),
           'category_usage' => $this->getDiagnosisUsageByCategory($dateFrom, $dateTo),
           'total_icd_codes_available' => $this->getTotalAvailableIcdCodes(),
           'unique_diagnoses_used' => $this->getUniqueDiagnosesUsedInPeriod($dateFrom, $dateTo)
       ];
       
       // Verification and diagnosis correlation
       $stats['verification_diagnosis_correlation'] = $this->getVerificationDiagnosisCorrelation($dateFrom, $dateTo);
       
       return $stats;
   }

   /**
    * Get comprehensive consultation statistics with verification and diagnosis data
    */
   public function getConsultationStatsWithVerification($dateFrom = null, $dateTo = null) {
       if (!$dateFrom) $dateFrom = date('Y-m-01');
       if (!$dateTo) $dateTo = date('Y-m-d');
       
       // Total consultations
       $this->db->where('consultation_date >=', $dateFrom);
       $this->db->where('consultation_date <=', $dateTo);
       $totalConsultations = $this->db->count_all_results('consultations');
       
       // Consultations by verification status
       $this->db->select('verification_status, COUNT(*) as count');
       $this->db->from('consultations');
       $this->db->where('consultation_date >=', $dateFrom);
       $this->db->where('consultation_date <=', $dateTo);
       $this->db->group_by('verification_status');
       $verificationBreakdown = $this->db->get()->result_array();
       
       // Consultations with ICD-10 diagnoses
       $this->db->distinct();
       $this->db->select('c.id');
       $this->db->from('consultations c');
       $this->db->join('consultation_diagnosis cd', 'c.id = cd.consultation_id');
       $this->db->where('c.consultation_date >=', $dateFrom);
       $this->db->where('c.consultation_date <=', $dateTo);
       $consultationsWithIcd = $this->db->count_all_results();
       
       // Consultations by doctor with verification rates
       $this->db->select('s.name, s.surname, s.id as doctor_id,
                         COUNT(c.id) as total_consultations,
                         SUM(CASE WHEN c.verification_status = "verified" THEN 1 ELSE 0 END) as verified_consultations,
                         SUM(CASE WHEN c.verification_status = "skipped" THEN 1 ELSE 0 END) as skipped_consultations,
                         SUM(CASE WHEN c.verification_status = "unverified" THEN 1 ELSE 0 END) as unverified_consultations');
       $this->db->from('consultations c');
       $this->db->join('staff s', 's.id = c.doctor_id', 'left');
       $this->db->where('c.consultation_date >=', $dateFrom);
       $this->db->where('c.consultation_date <=', $dateTo);
       $this->db->group_by('c.doctor_id');
       $this->db->order_by('total_consultations', 'DESC');
       $consultationsByDoctor = $this->db->get()->result_array();
       
       // Daily consultations with verification breakdown
       $this->db->select('DATE(consultation_date) as date,
                         COUNT(*) as total,
                         SUM(CASE WHEN verification_status = "verified" THEN 1 ELSE 0 END) as verified,
                         SUM(CASE WHEN verification_status = "skipped" THEN 1 ELSE 0 END) as skipped,
                         SUM(CASE WHEN verification_status = "unverified" THEN 1 ELSE 0 END) as unverified');
       $this->db->from('consultations');
       $this->db->where('consultation_date >=', $dateFrom);
       $this->db->where('consultation_date <=', $dateTo);
       $this->db->group_by('DATE(consultation_date)');
       $this->db->order_by('date', 'ASC');
       $dailyConsultations = $this->db->get()->result_array();
       
       // Patients with/without fingerprints consultation pattern
       $this->db->select('CASE WHEN p.fingerprint_template IS NOT NULL AND p.fingerprint_template != "" 
                               THEN "With Fingerprint" ELSE "Without Fingerprint" END as fingerprint_status,
                         COUNT(c.id) as consultation_count,
                         COUNT(DISTINCT c.patient_id) as unique_patients');
       $this->db->from('consultations c');
       $this->db->join('patients p', 'p.id = c.patient_id', 'left');
       $this->db->where('c.consultation_date >=', $dateFrom);
       $this->db->where('c.consultation_date <=', $dateTo);
       $this->db->group_by('fingerprint_status');
       $fingerprintPatternStats = $this->db->get()->result_array();
       
       // Get diagnosis usage statistics for the period
       $diagnosisStats = $this->getDiagnosisUsageByCategory($dateFrom, $dateTo);
       
       return [
           'total_consultations' => $totalConsultations,
           'consultations_with_icd' => $consultationsWithIcd,
           'verification_breakdown' => $verificationBreakdown,
           'consultations_by_doctor' => $consultationsByDoctor,
           'daily_consultations' => $dailyConsultations,
           'fingerprint_pattern_stats' => $fingerprintPatternStats,
           'diagnosis_usage_by_category' => $diagnosisStats
       ];
   }

   /**
    * Get total available ICD codes in system
    */
   public function getTotalAvailableIcdCodes($icdVersion = 'ICD10') {
       $this->db->where('is_active', 1);
       $this->db->where('icd_version', $icdVersion);
       return $this->db->count_all_results('diagnosis_codes');
   }

   /**
    * Get unique diagnoses used in a period
    */
   public function getUniqueDiagnosesUsedInPeriod($dateFrom, $dateTo) {
       $this->db->distinct();
       $this->db->select('cd.diagnosis_code_id');
       $this->db->from('consultation_diagnosis cd');
       $this->db->join('consultations c', 'c.id = cd.consultation_id');
       $this->db->where('c.consultation_date >=', $dateFrom);
       $this->db->where('c.consultation_date <=', $dateTo);
       
       return $this->db->count_all_results();
   }

   /**
    * Get correlation between verification status and diagnosis usage
    */
   public function getVerificationDiagnosisCorrelation($dateFrom, $dateTo) {
       $this->db->select('c.verification_status,
                         COUNT(c.id) as total_consultations,
                         COUNT(cd.id) as consultations_with_icd_diagnosis,
                         ROUND((COUNT(cd.id) / COUNT(c.id)) * 100, 2) as icd_usage_percentage');
       $this->db->from('consultations c');
       $this->db->join('consultation_diagnosis cd', 'c.id = cd.consultation_id', 'left');
       $this->db->where('c.consultation_date >=', $dateFrom);
       $this->db->where('c.consultation_date <=', $dateTo);
       $this->db->group_by('c.verification_status');
       $this->db->order_by('total_consultations', 'DESC');
       
       $query = $this->db->get();
       return $query->result_array();
   }

   /**
    * Get doctor-wise diagnosis pattern analysis
    */
   public function getDoctorDiagnosisPatterns($dateFrom = null, $dateTo = null, $limit = 10) {
       if (!$dateFrom) $dateFrom = date('Y-m-01');
       if (!$dateTo) $dateTo = date('Y-m-d');
       
       $this->db->select('s.name, s.surname, s.id as doctor_id,
                         COUNT(DISTINCT c.id) as total_consultations,
                         COUNT(DISTINCT cd.diagnosis_code_id) as unique_diagnoses_used,
                         COUNT(cd.id) as total_icd_diagnoses');
       $this->db->from('consultations c');
       $this->db->join('staff s', 's.id = c.doctor_id', 'left');
       $this->db->join('consultation_diagnosis cd', 'c.id = cd.consultation_id', 'left');
       $this->db->where('c.consultation_date >=', $dateFrom);
       $this->db->where('c.consultation_date <=', $dateTo);
       $this->db->group_by('s.id');
       $this->db->order_by('total_consultations', 'DESC');
       $this->db->limit($limit);
       
       $query = $this->db->get();
       return $query->result_array();
   }

   // =============================================
   // UTILITY AND MAINTENANCE METHODS
   // =============================================

   /**
    * Search consultations with advanced filters including verification status and diagnosis
    */
   public function searchConsultationsAdvanced($filters, $limit = 50) {
       $this->db->select('c.id, c.consultation_date, c.symptoms, c.diagnosis, c.verification_status,
                         p.patient_name, p.id as patient_id,
                         CONCAT(s.name, " ", s.surname) as doctor_name,
                         CASE WHEN p.fingerprint_template IS NOT NULL AND p.fingerprint_template != "" 
                              THEN 1 ELSE 0 END as patient_has_fingerprint');
       $this->db->from('consultations c');
       $this->db->join('patients p', 'p.id = c.patient_id', 'left');
       $this->db->join('staff s', 's.id = c.doctor_id', 'left');
       
       // Join diagnosis tables for ICD-10 search
       if (!empty($filters['diagnosis_code']) || !empty($filters['diagnosis_category'])) {
           $this->db->join('consultation_diagnosis cd', 'c.id = cd.consultation_id', 'left');
           $this->db->join('diagnosis_codes dc', 'cd.diagnosis_code_id = dc.id', 'left');
       }
       
       // Apply filters
       if (!empty($filters['search_term'])) {
           $this->db->group_start();
           $this->db->like('p.patient_name', $filters['search_term']);
           $this->db->or_like('c.symptoms', $filters['search_term']);
           $this->db->or_like('c.diagnosis', $filters['search_term']);
           $this->db->or_like('p.id', $filters['search_term']);
           
           // Search in ICD diagnosis codes and titles if tables are joined
           if (!empty($filters['diagnosis_code']) || !empty($filters['diagnosis_category'])) {
               $this->db->or_like('dc.code', $filters['search_term']);
               $this->db->or_like('dc.title', $filters['search_term']);
           }
           $this->db->group_end();
       }
       
       if (!empty($filters['verification_status'])) {
           $this->db->where('c.verification_status', $filters['verification_status']);
       }
       
       if (!empty($filters['doctor_id'])) {
           $this->db->where('c.doctor_id', $filters['doctor_id']);
       }
       
       if (!empty($filters['date_from'])) {
           $this->db->where('c.consultation_date >=', $filters['date_from']);
       }
       
       if (!empty($filters['date_to'])) {
           $this->db->where('c.consultation_date <=', $filters['date_to']);
       }
       
       if (!empty($filters['diagnosis_code'])) {
           $this->db->where('dc.code', $filters['diagnosis_code']);
       }
       
       if (!empty($filters['diagnosis_category'])) {
           $this->db->where('dc.category', $filters['diagnosis_category']);
       }
       
       if (isset($filters['has_fingerprint'])) {
           if ($filters['has_fingerprint']) {
               $this->db->where('p.fingerprint_template IS NOT NULL');
               $this->db->where('p.fingerprint_template !=', '');
           } else {
               $this->db->group_start();
               $this->db->where('p.fingerprint_template IS NULL');
               $this->db->or_where('p.fingerprint_template', '');
               $this->db->group_end();
           }
       }
       
       if (isset($filters['has_icd_diagnosis'])) {
           if ($filters['has_icd_diagnosis']) {
               $this->db->where('cd.id IS NOT NULL');
           } else {
               $this->db->where('cd.id IS NULL');
           }
       }
       
       $this->db->group_by('c.id'); // Prevent duplicates from joins
       $this->db->order_by('c.consultation_date', 'DESC');
       $this->db->limit($limit);
       
       $query = $this->db->get();
       $results = $query->result_array();
       
       // Add diagnosis information to each result
       foreach ($results as &$result) {
           $result['diagnoses'] = $this->getConsultationDiagnoses($result['id']);
           $result['has_icd_diagnosis'] = count($result['diagnoses']) > 0;
       }
       
       return $results;
   }

   /**
    * Get fingerprint enrollment statistics
    */
   public function getFingerprintEnrollmentStats() {
       // Total patients
       $this->db->where('is_active', 'yes');
       $totalPatients = $this->db->count_all_results('patients');
       
       // Patients with fingerprints
       $this->db->where('is_active', 'yes');
       $this->db->where('fingerprint_template IS NOT NULL');
       $this->db->where('fingerprint_template !=', '');
       $enrolledPatients = $this->db->count_all_results('patients');
       
       // Recent enrollments (last 30 days)
       $this->db->where('is_active', 'yes');
       $this->db->where('fingerprint_template IS NOT NULL');
       $this->db->where('fingerprint_template !=', '');
       $this->db->where('created_at >=', date('Y-m-d H:i:s', strtotime('-30 days')));
       $recentEnrollments = $this->db->count_all_results('patients');
       
       // Enrollment by age group
       $this->db->select('CASE 
                           WHEN age < 18 THEN "Under 18"
                           WHEN age BETWEEN 18 AND 35 THEN "18-35"
                           WHEN age BETWEEN 36 AND 55 THEN "36-55"
                           WHEN age > 55 THEN "Over 55"
                           ELSE "Unknown"
                         END as age_group,
                         COUNT(*) as total_patients,
                         SUM(CASE WHEN fingerprint_template IS NOT NULL AND fingerprint_template != "" 
                             THEN 1 ELSE 0 END) as enrolled_patients');
       $this->db->from('patients');
       $this->db->where('is_active', 'yes');
       $this->db->group_by('age_group');
       $enrollmentByAge = $this->db->get()->result_array();
       
       // Enrollment by gender
       $this->db->select('gender,
                         COUNT(*) as total_patients,
                         SUM(CASE WHEN fingerprint_template IS NOT NULL AND fingerprint_template != "" 
                             THEN 1 ELSE 0 END) as enrolled_patients');
       $this->db->from('patients');
       $this->db->where('is_active', 'yes');
       $this->db->group_by('gender');
       $enrollmentByGender = $this->db->get()->result_array();
       
       return [
           'total_patients' => $totalPatients,
           'enrolled_patients' => $enrolledPatients,
           'enrollment_rate' => $totalPatients > 0 ? round(($enrolledPatients / $totalPatients) * 100, 2) : 0,
           'recent_enrollments' => $recentEnrollments,
           'enrollment_by_age' => $enrollmentByAge,
           'enrollment_by_gender' => $enrollmentByGender
       ];
   }

   /**
    * Clean up old verification logs (for maintenance)
    */
   public function cleanupOldVerificationLogs($daysToKeep = 90) {
       $cutoffDate = date('Y-m-d H:i:s', strtotime("-{$daysToKeep} days"));
       
       $this->db->where('verification_date <', $cutoffDate);
       $deletedCount = $this->db->count_all_results('fingerprint_verification_log');
       
       $this->db->where('verification_date <', $cutoffDate);
       $this->db->delete('fingerprint_verification_log');
       
       return $deletedCount;
   }

   /**
    * Clean up orphaned consultation diagnoses (for maintenance)
    */
   public function cleanupOrphanedDiagnoses() {
       // Find consultation_diagnosis records with no matching consultation
       $this->db->select('cd.id');
       $this->db->from('consultation_diagnosis cd');
       $this->db->join('consultations c', 'cd.consultation_id = c.id', 'left');
       $this->db->where('c.id IS NULL');
       $orphanedQuery = $this->db->get();
       $orphanedCount = $orphanedQuery->num_rows();
       
       if ($orphanedCount > 0) {
           $orphanedIds = array_column($orphanedQuery->result_array(), 'id');
           $this->db->where_in('id', $orphanedIds);
           $this->db->delete('consultation_diagnosis');
       }
       
       return $orphanedCount;
   }

   /**
    * Get system health check for fingerprint verification and diagnosis system
    */
   public function getSystemHealthCheck() {
       $health = [];
       
       // Check if required tables exist
       $health['tables_exist'] = [
           'consultations' => $this->db->table_exists('consultations'),
           'fingerprint_verification_log' => $this->db->table_exists('fingerprint_verification_log'),
           'patients' => $this->db->table_exists('patients'),
           'diagnosis_codes' => $this->db->table_exists('diagnosis_codes'),
           'consultation_diagnosis' => $this->db->table_exists('consultation_diagnosis')
       ];
       
       // Check recent activity (last 24 hours)
       $yesterday = date('Y-m-d H:i:s', strtotime('-24 hours'));
       
       $this->db->where('created_at >=', $yesterday);
       $health['recent_consultations'] = $this->db->count_all_results('consultations');
       
       if ($health['tables_exist']['fingerprint_verification_log']) {
           $this->db->where('verification_date >=', $yesterday);
           $health['recent_verifications'] = $this->db->count_all_results('fingerprint_verification_log');
       } else {
           $health['recent_verifications'] = 0;
       }
       
       if ($health['tables_exist']['consultation_diagnosis']) {
           $this->db->where('created_at >=', $yesterday);
           $health['recent_diagnosis_assignments'] = $this->db->count_all_results('consultation_diagnosis');
       } else {
           $health['recent_diagnosis_assignments'] = 0;
       }
       
       // Check data integrity
       $this->db->select('COUNT(*) as count');
       $this->db->from('consultations c');
       $this->db->join('patients p', 'p.id = c.patient_id', 'left');
       $this->db->where('p.id IS NULL');
       $health['orphaned_consultations'] = $this->db->get()->row()->count;
       
       // Check orphaned diagnosis assignments
       if ($health['tables_exist']['consultation_diagnosis']) {
           $this->db->select('COUNT(*) as count');
           $this->db->from('consultation_diagnosis cd');
           $this->db->join('consultations c', 'cd.consultation_id = c.id', 'left');
           $this->db->where('c.id IS NULL');
           $health['orphaned_diagnoses'] = $this->db->get()->row()->count;
       } else {
           $health['orphaned_diagnoses'] = 0;
       }
       
       // Check fingerprint data quality
       $this->db->where('fingerprint_template IS NOT NULL');
       $this->db->where('fingerprint_template !=', '');
       $this->db->where('LENGTH(fingerprint_template) < 100'); // Templates should be longer
       $health['invalid_fingerprint_templates'] = $this->db->count_all_results('patients');
       
       // Check diagnosis codes availability
       if ($health['tables_exist']['diagnosis_codes']) {
           $this->db->where('is_active', 1);
           $health['active_diagnosis_codes'] = $this->db->count_all_results('diagnosis_codes');
           
           $this->db->where('is_active', 1);
           $this->db->where('icd_version', 'ICD10');
           $health['icd10_codes_available'] = $this->db->count_all_results('diagnosis_codes');
       } else {
           $health['active_diagnosis_codes'] = 0;
           $health['icd10_codes_available'] = 0;
       }
       
       // Overall system health score
       $totalChecks = 10;
       $passedChecks = 0;
       
       $passedChecks += array_sum($health['tables_exist']); // 5 table checks
       $passedChecks += ($health['recent_consultations'] > 0) ? 1 : 0;
       $passedChecks += ($health['orphaned_consultations'] == 0) ? 1 : 0;
       $passedChecks += ($health['orphaned_diagnoses'] == 0) ? 1 : 0;
       $passedChecks += ($health['invalid_fingerprint_templates'] == 0) ? 1 : 0;
       $passedChecks += ($health['icd10_codes_available'] > 0) ? 1 : 0;
       
       $health['system_health_score'] = round(($passedChecks / $totalChecks) * 100, 1);
       $health['system_status'] = $health['system_health_score'] >= 80 ? 'Good' : 
                                 ($health['system_health_score'] >= 60 ? 'Fair' : 'Poor');
       
       return $health;
   }

   /**
    * Export consultation data with diagnosis information for reporting
    */
   public function exportConsultationsWithDiagnoses($dateFrom, $dateTo, $format = 'array') {
       $this->db->select('c.id as consultation_id, c.consultation_date, c.symptoms, c.diagnosis as custom_diagnosis,
                         c.notes, c.verification_status, c.created_at,
                         p.id as patient_id, p.patient_name, p.age, p.gender, p.mobileno,
                         CONCAT(s.name, " ", s.surname) as doctor_name,
                         dc.code as icd_code, dc.title as icd_title, dc.category as icd_category,
                         cd.is_primary as is_primary_diagnosis, cd.notes as diagnosis_notes');
       $this->db->from('consultations c');
       $this->db->join('patients p', 'c.patient_id = p.id', 'left');
       $this->db->join('staff s', 'c.doctor_id = s.id', 'left');
       $this->db->join('consultation_diagnosis cd', 'c.id = cd.consultation_id', 'left');
       $this->db->join('diagnosis_codes dc', 'cd.diagnosis_code_id = dc.id', 'left');
       $this->db->where('c.consultation_date >=', $dateFrom);
       $this->db->where('c.consultation_date <=', $dateTo);
       $this->db->order_by('c.consultation_date DESC, c.id ASC, cd.is_primary DESC');
       
       $query = $this->db->get();
       $results = $query->result_array();
       
       if ($format === 'csv') {
           return $this->convertToCSV($results);
       }
       
       return $results;
   }

   /**
    * Convert array data to CSV format
    */
   private function convertToCSV($data) {
       if (empty($data)) {
           return '';
       }
       
       $csv = '';
       
       // Headers
       $headers = array_keys($data[0]);
       $csv .= implode(',', array_map(function($header) {
           return '"' . str_replace('"', '""', $header) . '"';
       }, $headers)) . "\n";
       
       // Data rows
       foreach ($data as $row) {
           $csvRow = [];
           foreach ($row as $value) {
               $csvRow[] = '"' . str_replace('"', '""', $value ?? '') . '"';
           }
           $csv .= implode(',', $csvRow) . "\n";
       }
       
       return $csv;
   }

   /**
    * Get consultation trend analysis with diagnosis patterns
    */
   public function getConsultationTrendAnalysis($months = 12) {
       $trends = [];
       
       for ($i = $months - 1; $i >= 0; $i--) {
           $month = date('Y-m', strtotime("-{$i} months"));
           $monthStart = $month . '-01';
           $monthEnd = date('Y-m-t', strtotime($monthStart));
           
           // Basic consultation stats for the month
           $this->db->where('consultation_date >=', $monthStart);
           $this->db->where('consultation_date <=', $monthEnd);
           $monthlyConsultations = $this->db->count_all_results('consultations');
           
           // Consultations with ICD diagnoses
           $this->db->distinct();
           $this->db->select('c.id');
           $this->db->from('consultations c');
           $this->db->join('consultation_diagnosis cd', 'c.id = cd.consultation_id');
           $this->db->where('c.consultation_date >=', $monthStart);
           $this->db->where('c.consultation_date <=', $monthEnd);
           $monthlyIcdConsultations = $this->db->count_all_results();
           
           // Verified consultations
           $this->db->where('consultation_date >=', $monthStart);
           $this->db->where('consultation_date <=', $monthEnd);
           $this->db->where('verification_status', 'verified');
           $monthlyVerifiedConsultations = $this->db->count_all_results('consultations');
           
           $trends[] = [
               'month' => $month,
               'month_name' => date('M Y', strtotime($monthStart)),
               'total_consultations' => $monthlyConsultations,
               'consultations_with_icd' => $monthlyIcdConsultations,
               'verified_consultations' => $monthlyVerifiedConsultations,
               'icd_usage_rate' => $monthlyConsultations > 0 ? round(($monthlyIcdConsultations / $monthlyConsultations) * 100, 1) : 0,
               'verification_rate' => $monthlyConsultations > 0 ? round(($monthlyVerifiedConsultations / $monthlyConsultations) * 100, 1) : 0
           ];
       }
       
       return $trends;
   }

   /**
    * Get diagnosis statistics for dashboard widgets
    */
   public function getDiagnosisDashboardStats() {
       $stats = [];
       
       // Total ICD codes available
       $this->db->where('is_active', 1);
       $this->db->where('icd_version', 'ICD10');
       $stats['total_icd_codes'] = $this->db->count_all_results('diagnosis_codes');
       
       // Most used diagnoses this month
       $monthStart = date('Y-m-01');
       $monthEnd = date('Y-m-d');
       $stats['top_diagnoses_this_month'] = $this->getMostUsedDiagnoses(5, $monthStart, $monthEnd);
       
       // ICD usage rate this month
       $this->db->where('consultation_date >=', $monthStart);
       $this->db->where('consultation_date <=', $monthEnd);
       $totalConsultationsThisMonth = $this->db->count_all_results('consultations');
       
       $this->db->distinct();
       $this->db->select('c.id');
       $this->db->from('consultations c');
       $this->db->join('consultation_diagnosis cd', 'c.id = cd.consultation_id');
       $this->db->where('c.consultation_date >=', $monthStart);
       $this->db->where('c.consultation_date <=', $monthEnd);
       $icdConsultationsThisMonth = $this->db->count_all_results();
       
       $stats['icd_usage_rate'] = $totalConsultationsThisMonth > 0 ? 
           round(($icdConsultationsThisMonth / $totalConsultationsThisMonth) * 100, 1) : 0;
       
       // Categories usage this month
       $stats['category_usage_this_month'] = $this->getDiagnosisUsageByCategory($monthStart, $monthEnd);
       
       return $stats;
   }

   /**
    * Get patient diagnosis history
    */
   public function getPatientDiagnosisHistory($patientId, $limit = 20) {
       $this->db->select('c.consultation_date, c.symptoms, c.notes,
                         dc.code, dc.title, dc.category, cd.is_primary,
                         CONCAT(s.name, " ", s.surname) as doctor_name');
       $this->db->from('consultations c');
       $this->db->join('consultation_diagnosis cd', 'c.id = cd.consultation_id');
       $this->db->join('diagnosis_codes dc', 'cd.diagnosis_code_id = dc.id');
       $this->db->join('staff s', 'c.doctor_id = s.id', 'left');
       $this->db->where('c.patient_id', $patientId);
       $this->db->order_by('c.consultation_date DESC, cd.is_primary DESC');
       $this->db->limit($limit);
       
       $query = $this->db->get();
       return $query->result_array();
   }

   /**
    * Get similar diagnoses based on category and symptoms
    */
   public function getSimilarDiagnoses($diagnosisCodeId, $limit = 5) {
       // Get the category of the current diagnosis
       $this->db->select('category');
       $this->db->from('diagnosis_codes');
       $this->db->where('id', $diagnosisCodeId);
       $currentDiagnosis = $this->db->get()->row();
       
       if (!$currentDiagnosis) {
           return [];
       }
       
       // Get other diagnoses in the same category
       $this->db->select('id, code, title, description');
       $this->db->from('diagnosis_codes');
       $this->db->where('category', $currentDiagnosis->category);
       $this->db->where('id !=', $diagnosisCodeId);
       $this->db->where('is_active', 1);
       $this->db->order_by('title ASC');
       $this->db->limit($limit);
       
       $query = $this->db->get();
       return $query->result_array();
   }

   /**
    * Get diagnosis code details with usage statistics
    */
   public function getDiagnosisCodeWithStats($diagnosisCodeId) {
       // Get basic diagnosis info
       $diagnosis = $this->getDiagnosisById($diagnosisCodeId);
       
       if (!$diagnosis) {
           return null;
       }
       
       // Add usage statistics
       $this->db->where('diagnosis_code_id', $diagnosisCodeId);
       $diagnosis['total_usage'] = $this->db->count_all_results('consultation_diagnosis');
       
       // Usage in last 30 days
       $this->db->where('diagnosis_code_id', $diagnosisCodeId);
       $this->db->where('created_at >=', date('Y-m-d H:i:s', strtotime('-30 days')));
       $diagnosis['recent_usage'] = $this->db->count_all_results('consultation_diagnosis');
       
       // Last used date
       $this->db->select('MAX(created_at) as last_used');
       $this->db->from('consultation_diagnosis');
       $this->db->where('diagnosis_code_id', $diagnosisCodeId);
       $lastUsed = $this->db->get()->row();
       $diagnosis['last_used'] = $lastUsed ? $lastUsed->last_used : null;
       
       // Most common doctors using this diagnosis
       $this->db->select('CONCAT(s.name, " ", s.surname) as doctor_name, COUNT(*) as usage_count');
       $this->db->from('consultation_diagnosis cd');
       $this->db->join('consultations c', 'cd.consultation_id = c.id');
       $this->db->join('staff s', 'c.doctor_id = s.id');
       $this->db->where('cd.diagnosis_code_id', $diagnosisCodeId);
       $this->db->group_by('c.doctor_id');
       $this->db->order_by('usage_count DESC');
       $this->db->limit(5);
       $diagnosis['top_users'] = $this->db->get()->result_array();
       
       return $diagnosis;
   }

   /**
    * Validate diagnosis code format
    */
   public function validateDiagnosisCode($code, $icdVersion = 'ICD10') {
       $errors = [];
       
       // Check if code already exists
       $this->db->where('code', $code);
       $this->db->where('icd_version', $icdVersion);
       if ($this->db->count_all_results('diagnosis_codes') > 0) {
           $errors[] = 'Diagnosis code already exists';
       }
       
       // Basic format validation for ICD-10
       if ($icdVersion === 'ICD10') {
           if (!preg_match('/^[A-Z][0-9]{2}(\.[0-9X]{1,4})?$/', $code)) {
               $errors[] = 'Invalid ICD-10 format. Expected format: A00 or A00.1 or A00.12';
           }
       }
       
       // Basic format validation for ICD-11
       if ($icdVersion === 'ICD11') {
           if (!preg_match('/^[0-9A-Z]{2,8}(\.[A-Z0-9]{1,4})?$/', $code)) {
               $errors[] = 'Invalid ICD-11 format';
           }
       }
       
       return [
           'valid' => empty($errors),
           'errors' => $errors
       ];
   }

   /**
    * Get diagnosis codes that need review (unused for long time)
    */
   public function getDiagnosisCodesForReview($unusedDays = 365) {
       $cutoffDate = date('Y-m-d H:i:s', strtotime("-{$unusedDays} days"));
       
       $this->db->select('dc.*, 
                         COALESCE(MAX(cd.created_at), dc.created_at) as last_usage,
                         COUNT(cd.id) as total_usage');
       $this->db->from('diagnosis_codes dc');
       $this->db->join('consultation_diagnosis cd', 'dc.id = cd.diagnosis_code_id', 'left');
       $this->db->where('dc.is_active', 1);
       $this->db->group_by('dc.id');
       $this->db->having('last_usage <', $cutoffDate);
       $this->db->order_by('last_usage ASC');
       
       $query = $this->db->get();
       return $query->result_array();
   }

   /**
    * Import ICD codes from CSV data
    */
   public function importIcdCodesFromCsv($csvData, $icdVersion = 'ICD10') {
       $imported = 0;
       $errors = [];
       $duplicates = 0;
       
       foreach ($csvData as $rowIndex => $row) {
           try {
               // Expected CSV columns: code, title, description, category, subcategory
               if (count($row) < 3) {
                   $errors[] = "Row {$rowIndex}: Insufficient data (need at least code, title, description)";
                   continue;
               }
               
               $code = trim($row[0]);
               $title = trim($row[1]);
               $description = isset($row[2]) ? trim($row[2]) : '';
               $category = isset($row[3]) ? trim($row[3]) : '';
               $subcategory = isset($row[4]) ? trim($row[4]) : '';
               
               // Validate code format
               $validation = $this->validateDiagnosisCode($code, $icdVersion);
               if (!$validation['valid']) {
                   $errors[] = "Row {$rowIndex}: " . implode(', ', $validation['errors']);
                   continue;
               }
               
               // Check for duplicates
               $this->db->where('code', $code);
               $this->db->where('icd_version', $icdVersion);
               if ($this->db->count_all_results('diagnosis_codes') > 0) {
                   $duplicates++;
                   continue;
               }
               
               // Insert the code
               $data = [
                   'icd_version' => $icdVersion,
                   'code' => $code,
                   'title' => $title,
                   'description' => $description,
                   'category' => $category,
                   'subcategory' => $subcategory,
                   'is_active' => 1,
                   'created_at' => date('Y-m-d H:i:s'),
                   'updated_at' => date('Y-m-d H:i:s')
               ];
               
               if ($this->addDiagnosisCode($data)) {
                   $imported++;
               }
               
           } catch (Exception $e) {
               $errors[] = "Row {$rowIndex}: " . $e->getMessage();
           }
       }
       
       return [
           'imported' => $imported,
           'errors' => $errors,
           'duplicates' => $duplicates,
           'total_processed' => count($csvData)
       ];
   }

   /**
    * Get consultation quality metrics including diagnosis completeness
    */
   public function getConsultationQualityMetrics($dateFrom = null, $dateTo = null) {
       if (!$dateFrom) $dateFrom = date('Y-m-01');
       if (!$dateTo) $dateTo = date('Y-m-d');
       
       // Total consultations in period
       $this->db->where('consultation_date >=', $dateFrom);
       $this->db->where('consultation_date <=', $dateTo);
       $totalConsultations = $this->db->count_all_results('consultations');
       
       // Consultations with ICD diagnoses
       $this->db->distinct();
       $this->db->select('c.id');
       $this->db->from('consultations c');
       $this->db->join('consultation_diagnosis cd', 'c.id = cd.consultation_id');
       $this->db->where('c.consultation_date >=', $dateFrom);
       $this->db->where('c.consultation_date <=', $dateTo);
       $consultationsWithIcd = $this->db->count_all_results();
       
       // Consultations with only custom diagnosis (legacy)
       $this->db->select('c.id');
       $this->db->from('consultations c');
       $this->db->join('consultation_diagnosis cd', 'c.id = cd.consultation_id', 'left');
       $this->db->where('c.consultation_date >=', $dateFrom);
       $this->db->where('c.consultation_date <=', $dateTo);
       $this->db->where('c.diagnosis IS NOT NULL');
       $this->db->where('c.diagnosis !=', '');
       $this->db->where('cd.id IS NULL');
       $consultationsWithCustomOnly = $this->db->count_all_results();
       
       // Consultations with no diagnosis at all
       $this->db->select('c.id');
       $this->db->from('consultations c');
       $this->db->join('consultation_diagnosis cd', 'c.id = cd.consultation_id', 'left');
       $this->db->where('c.consultation_date >=', $dateFrom);
       $this->db->where('c.consultation_date <=', $dateTo);
       $this->db->group_start();
       $this->db->where('c.diagnosis IS NULL');
       $this->db->or_where('c.diagnosis', '');
       $this->db->group_end();
       $this->db->where('cd.id IS NULL');
       $consultationsWithNoDiagnosis = $this->db->count_all_results();
       
       // Calculate quality scores
       $icdUsageRate = $totalConsultations > 0 ? 
           round(($consultationsWithIcd / $totalConsultations) * 100, 1) : 0;
       
       $diagnosisCompleteness = $totalConsultations > 0 ? 
           round((($totalConsultations - $consultationsWithNoDiagnosis) / $totalConsultations) * 100, 1) : 0;
       
       return [
           'total_consultations' => $totalConsultations,
           'consultations_with_icd' => $consultationsWithIcd,
           'consultations_with_custom_only' => $consultationsWithCustomOnly,
           'consultations_with_no_diagnosis' => $consultationsWithNoDiagnosis,
           'icd_usage_rate' => $icdUsageRate,
           'diagnosis_completeness' => $diagnosisCompleteness,
           'quality_score' => round(($icdUsageRate + $diagnosisCompleteness) / 2, 1)
       ];
   }

   /**
    * Archive old diagnosis codes (soft delete)
    */
   public function archiveUnusedDiagnosisCodes($unusedDays = 730, $dryRun = true) {
       $cutoffDate = date('Y-m-d H:i:s', strtotime("-{$unusedDays} days"));
       $archived = 0;
       
       // Find unused diagnosis codes
       $this->db->select('dc.id');
       $this->db->from('diagnosis_codes dc');
       $this->db->join('consultation_diagnosis cd', 'dc.id = cd.diagnosis_code_id', 'left');
       $this->db->where('dc.is_active', 1);
       $this->db->group_by('dc.id');
       $this->db->having('MAX(cd.created_at) <', $cutoffDate);
       $this->db->or_having('MAX(cd.created_at) IS NULL');
       
       $unusedCodes = $this->db->get()->result_array();
       
       if (!$dryRun && !empty($unusedCodes)) {
           $codeIds = array_column($unusedCodes, 'id');
           
           $this->db->where_in('id', $codeIds);
           $this->db->set('is_active', 0);
           $this->db->set('updated_at', date('Y-m-d H:i:s'));
           $this->db->update('diagnosis_codes');
           
           $archived = $this->db->affected_rows();
       }
       
       return [
           'found_unused' => count($unusedCodes),
           'archived' => $archived,
           'dry_run' => $dryRun,
           'cutoff_date' => $cutoffDate
       ];
   }

   // =============================================
   // PATIENT DATA METHODS WITH FINGERPRINT INFO
   // =============================================

   /**
    * Get all active patients with fingerprint availability
    */
   public function getAllPatientsWithFingerprintInfo($limit = 100) {
       $this->db->select('id, patient_name, mobileno, gender, age, 
                         fingerprint_template, fingerprint_image,
                         CASE WHEN fingerprint_template IS NOT NULL AND fingerprint_template != "" 
                              THEN 1 ELSE 0 END as has_fingerprint');
       $this->db->from('patients');
       $this->db->where('is_active', 'yes');
       $this->db->order_by('patient_name', 'ASC');
       $this->db->limit($limit);
       
       $query = $this->db->get();
       return $query->result_array();
   }

   /**
    * Search patients with fingerprint status
    */
   public function searchPatientsWithFingerprintInfo($searchTerm, $limit = 30) {
       $this->db->select('id, patient_name, mobileno, gender, age, 
                         fingerprint_template, fingerprint_image,
                         CASE WHEN fingerprint_template IS NOT NULL AND fingerprint_template != "" 
                              THEN 1 ELSE 0 END as has_fingerprint');
       $this->db->from('patients');
       
       $this->db->group_start();
       $this->db->like('patient_name', $searchTerm, 'both');
       $this->db->or_like('id', $searchTerm, 'both');
       $this->db->or_like('mobileno', $searchTerm, 'both');
       
       if (strlen($searchTerm) == 1) {
           $this->db->or_like('patient_name', $searchTerm, 'after');
       }
       $this->db->group_end();
       
       $this->db->where('is_active', 'yes');
       $this->db->order_by('patient_name', 'ASC');
       $this->db->limit($limit);
       
       $query = $this->db->get();
       return $query->result_array();
   }

   /**
    * Get patient details including fingerprint and verification history
    */
   public function getPatientDetailsWithVerificationInfo($patientId) {
       $this->db->select('p.*, bg.name as blood_group_name,
                         CASE WHEN p.fingerprint_template IS NOT NULL AND p.fingerprint_template != "" 
                              THEN 1 ELSE 0 END as has_fingerprint');
       $this->db->from('patients p');
       $this->db->join('blood_group bg', 'bg.id = p.blood_group', 'left');
       $this->db->where('p.id', $patientId);
       $this->db->where('p.is_active', 'yes');
       
       $query = $this->db->get();
       $patient = $query->row_array();
       
       if ($patient) {
           // Get recent verification attempts
           $patient['recent_verifications'] = $this->getFingerprintVerificationHistory($patientId, 5);
           
           // Get consultation count
           $this->db->where('patient_id', $patientId);
           $patient['total_consultations'] = $this->db->count_all_results('consultations');
           
           // Get verified consultation count
           $this->db->where('patient_id', $patientId);
           $this->db->where('verification_status', 'verified');
           $patient['verified_consultations'] = $this->db->count_all_results('consultations');
       }
       
       return $patient;
   }

   // =============================================
   // CONSULTATION REPORTING WITH VERIFICATION DATA
   // =============================================

   /**
    * Get consultations by date range with verification breakdown
    */
   public function getConsultationsByDateRangeWithVerification($startDate, $endDate, $doctorId = null) {
       $this->db->select('c.*, p.patient_name, p.fingerprint_template,
                         CONCAT(s.name, " ", s.surname) as doctor_name,
                         CASE WHEN p.fingerprint_template IS NOT NULL AND p.fingerprint_template != "" 
                              THEN 1 ELSE 0 END as patient_has_fingerprint');
       $this->db->from('consultations c');
       $this->db->join('patients p', 'p.id = c.patient_id', 'left');
       $this->db->join('staff s', 's.id = c.doctor_id', 'left');
       $this->db->where('c.consultation_date >=', $startDate);
       $this->db->where('c.consultation_date <=', $endDate);
       
       if ($doctorId) {
           $this->db->where('c.doctor_id', $doctorId);
       }
       
       $this->db->order_by('c.consultation_date', 'DESC');
       
       $query = $this->db->get();
       $consultations = $query->result_array();
       
       // Add diagnosis information to each consultation
       foreach ($consultations as &$consultation) {
           $consultation['diagnoses'] = $this->getConsultationDiagnoses($consultation['id']);
           $consultation['has_icd_diagnosis'] = count($consultation['diagnoses']) > 0;
       }
       
       return $consultations;
   }

   // =============================================
   // ADDITIONAL UTILITY METHODS
   // =============================================

   /**
    * Get consultation statistics summary for dashboard
    */
   public function getDashboardSummary($dateFrom = null, $dateTo = null) {
       if (!$dateFrom) $dateFrom = date('Y-m-01');
       if (!$dateTo) $dateTo = date('Y-m-d');
       
       $summary = [];
       
       // Basic consultation counts
       $this->db->where('consultation_date >=', $dateFrom);
       $this->db->where('consultation_date <=', $dateTo);
       $summary['total_consultations'] = $this->db->count_all_results('consultations');
       
       // Today's consultations
       $this->db->where('DATE(consultation_date)', date('Y-m-d'));
       $summary['today_consultations'] = $this->db->count_all_results('consultations');
       
       // Verified consultations
       $this->db->where('consultation_date >=', $dateFrom);
       $this->db->where('consultation_date <=', $dateTo);
       $this->db->where('verification_status', 'verified');
       $summary['verified_consultations'] = $this->db->count_all_results('consultations');
       
       // Consultations with ICD diagnoses
       $this->db->distinct();
       $this->db->select('c.id');
       $this->db->from('consultations c');
       $this->db->join('consultation_diagnosis cd', 'c.id = cd.consultation_id');
       $this->db->where('c.consultation_date >=', $dateFrom);
       $this->db->where('c.consultation_date <=', $dateTo);
       $summary['consultations_with_icd'] = $this->db->count_all_results();
       
       // Calculate rates
       $summary['verification_rate'] = $summary['total_consultations'] > 0 ? 
           round(($summary['verified_consultations'] / $summary['total_consultations']) * 100, 1) : 0;
       
       $summary['icd_usage_rate'] = $summary['total_consultations'] > 0 ? 
           round(($summary['consultations_with_icd'] / $summary['total_consultations']) * 100, 1) : 0;
       
       return $summary;
   }

   /**
    * Get recent consultation activity
    */
   public function getRecentConsultationActivity($limit = 10) {
       $this->db->select('c.id, c.consultation_date, c.symptoms, c.verification_status, c.created_at,
                         p.patient_name, p.id as patient_id,
                         CONCAT(s.name, " ", s.surname) as doctor_name');
       $this->db->from('consultations c');
       $this->db->join('patients p', 'p.id = c.patient_id', 'left');
       $this->db->join('staff s', 's.id = c.doctor_id', 'left');
       $this->db->order_by('c.created_at', 'DESC');
       $this->db->limit($limit);
       
       $query = $this->db->get();
       $results = $query->result_array();
       
       // Add diagnosis info
       foreach ($results as &$result) {
           $result['diagnoses'] = $this->getConsultationDiagnoses($result['id']);
           $result['has_icd_diagnosis'] = count($result['diagnoses']) > 0;
       }
       
       return $results;
   }

   /**
    * Get top performing doctors by consultation count
    */
   public function getTopDoctorsByConsultations($dateFrom = null, $dateTo = null, $limit = 5) {
       if (!$dateFrom) $dateFrom = date('Y-m-01');
       if (!$dateTo) $dateTo = date('Y-m-d');
       
       $this->db->select('s.name, s.surname, s.id as doctor_id,
                         COUNT(c.id) as consultation_count,
                         SUM(CASE WHEN c.verification_status = "verified" THEN 1 ELSE 0 END) as verified_count');
       $this->db->from('consultations c');
       $this->db->join('staff s', 's.id = c.doctor_id', 'left');
       $this->db->where('c.consultation_date >=', $dateFrom);
       $this->db->where('c.consultation_date <=', $dateTo);
       $this->db->group_by('c.doctor_id');
       $this->db->order_by('consultation_count', 'DESC');
       $this->db->limit($limit);
       
       $query = $this->db->get();
       return $query->result_array();
   }

   /**
    * Backup consultation data
    */
   public function backupConsultationData($dateFrom, $dateTo) {
       $backupData = [];
       
       // Export consultations
       $backupData['consultations'] = $this->exportConsultationsWithDiagnoses($dateFrom, $dateTo);
       
       // Export diagnosis codes used in this period
       $this->db->select('dc.*');
       $this->db->distinct();
       $this->db->from('diagnosis_codes dc');
       $this->db->join('consultation_diagnosis cd', 'dc.id = cd.diagnosis_code_id');
       $this->db->join('consultations c', 'c.id = cd.consultation_id');
       $this->db->where('c.consultation_date >=', $dateFrom);
       $this->db->where('c.consultation_date <=', $dateTo);
       $backupData['diagnosis_codes'] = $this->db->get()->result_array();
       
       // Export verification logs
       if ($this->db->table_exists('fingerprint_verification_log')) {
           $this->db->select('fvl.*');
           $this->db->from('fingerprint_verification_log fvl');
           $this->db->join('consultations c', 'c.patient_id = fvl.patient_id');
           $this->db->where('c.consultation_date >=', $dateFrom);
           $this->db->where('c.consultation_date <=', $dateTo);
           $this->db->group_by('fvl.id');
           $backupData['verification_logs'] = $this->db->get()->result_array();
       }
       
       $backupData['backup_info'] = [
           'created_at' => date('Y-m-d H:i:s'),
           'date_range' => ['from' => $dateFrom, 'to' => $dateTo],
           'total_consultations' => count($backupData['consultations']),
           'total_diagnosis_codes' => count($backupData['diagnosis_codes']),
           'total_verification_logs' => isset($backupData['verification_logs']) ? count($backupData['verification_logs']) : 0
       ];
       
       return $backupData;
   }

   /**
    * Get consultation completion rate by doctor
    */
   public function getConsultationCompletionRateByDoctor($dateFrom = null, $dateTo = null) {
       if (!$dateFrom) $dateFrom = date('Y-m-01');
       if (!$dateTo) $dateTo = date('Y-m-d');
       
       $this->db->select('s.name, s.surname, s.id as doctor_id,
                         COUNT(c.id) as total_consultations,
                         SUM(CASE WHEN c.diagnosis IS NOT NULL AND c.diagnosis != "" THEN 1 ELSE 0 END) as consultations_with_diagnosis,
                         SUM(CASE WHEN cd.id IS NOT NULL THEN 1 ELSE 0 END) as consultations_with_icd,
                         ROUND((SUM(CASE WHEN c.diagnosis IS NOT NULL AND c.diagnosis != "" THEN 1 ELSE 0 END) / COUNT(c.id)) * 100, 1) as diagnosis_completion_rate,
                         ROUND((SUM(CASE WHEN cd.id IS NOT NULL THEN 1 ELSE 0 END) / COUNT(c.id)) * 100, 1) as icd_usage_rate');
       $this->db->from('consultations c');
       $this->db->join('staff s', 's.id = c.doctor_id', 'left');
       $this->db->join('consultation_diagnosis cd', 'c.id = cd.consultation_id', 'left');
       $this->db->where('c.consultation_date >=', $dateFrom);
       $this->db->where('c.consultation_date <=', $dateTo);
       $this->db->group_by('c.doctor_id');
       $this->db->order_by('total_consultations', 'DESC');
       
       $query = $this->db->get();
       return $query->result_array();
   }

   /**
    * Get monthly consultation trends
    */
   public function getMonthlyConsultationTrends($months = 6) {
       $trends = [];
       
       for ($i = $months - 1; $i >= 0; $i--) {
           $month = date('Y-m', strtotime("-{$i} months"));
           $monthStart = $month . '-01';
           $monthEnd = date('Y-m-t', strtotime($monthStart));
           
           // Get consultation counts
           $this->db->where('consultation_date >=', $monthStart);
           $this->db->where('consultation_date <=', $monthEnd);
           $totalConsultations = $this->db->count_all_results('consultations');
           
           // Get verification counts
           $this->db->where('consultation_date >=', $monthStart);
           $this->db->where('consultation_date <=', $monthEnd);
           $this->db->where('verification_status', 'verified');
           $verifiedConsultations = $this->db->count_all_results('consultations');
           
           // Get ICD usage
           $this->db->distinct();
           $this->db->select('c.id');
           $this->db->from('consultations c');
           $this->db->join('consultation_diagnosis cd', 'c.id = cd.consultation_id');
           $this->db->where('c.consultation_date >=', $monthStart);
           $this->db->where('c.consultation_date <=', $monthEnd);
           $icdConsultations = $this->db->count_all_results();
           
           $trends[] = [
               'month' => $month,
               'month_name' => date('M Y', strtotime($monthStart)),
               'total_consultations' => $totalConsultations,
               'verified_consultations' => $verifiedConsultations,
               'icd_consultations' => $icdConsultations,
               'verification_rate' => $totalConsultations > 0 ? round(($verifiedConsultations / $totalConsultations) * 100, 1) : 0,
               'icd_usage_rate' => $totalConsultations > 0 ? round(($icdConsultations / $totalConsultations) * 100, 1) : 0
           ];
       }
       
       return $trends;
   }

   /**
    * Generate consultation report
    */
   public function generateConsultationReport($dateFrom, $dateTo, $doctorId = null, $reportType = 'summary') {
       $report = [
           'report_info' => [
               'generated_at' => date('Y-m-d H:i:s'),
               'date_range' => ['from' => $dateFrom, 'to' => $dateTo],
               'doctor_filter' => $doctorId,
               'report_type' => $reportType
           ]
       ];
       
       // Basic statistics
       $report['summary'] = $this->getConsultationStatsWithVerification($dateFrom, $dateTo);
       
       // Detailed consultations if requested
       if ($reportType === 'detailed') {
           $report['consultations'] = $this->getConsultationsByDateRangeWithVerification($dateFrom, $dateTo, $doctorId);
       }
       
       // Diagnosis statistics
       $report['diagnosis_stats'] = [
           'most_used' => $this->getMostUsedDiagnoses(20, $dateFrom, $dateTo),
           'category_usage' => $this->getDiagnosisUsageByCategory($dateFrom, $dateTo),
           'doctor_patterns' => $this->getDoctorDiagnosisPatterns($dateFrom, $dateTo)
       ];
       
       // Quality metrics
       $report['quality_metrics'] = $this->getConsultationQualityMetrics($dateFrom, $dateTo);
       
       // Verification statistics
       $report['verification_stats'] = $this->getVerificationStats($dateFrom, $dateTo);
       
       return $report;
   }

   /**
    * Get system performance metrics
    */
   public function getSystemPerformanceMetrics() {
       $metrics = [];
       
       // Database performance indicators
       $this->db->select('COUNT(*) as total_consultations');
       $this->db->from('consultations');
       $metrics['total_consultations'] = $this->db->get()->row()->total_consultations;
       
       $this->db->select('COUNT(*) as total_diagnosis_codes');
       $this->db->from('diagnosis_codes');
       $this->db->where('is_active', 1);
       $metrics['total_diagnosis_codes'] = $this->db->get()->row()->total_diagnosis_codes;
       
       $this->db->select('COUNT(*) as total_patients');
       $this->db->from('patients');
       $this->db->where('is_active', 'yes');
       $metrics['total_patients'] = $this->db->get()->row()->total_patients;
       
       // Recent activity (last 24 hours)
       $yesterday = date('Y-m-d H:i:s', strtotime('-24 hours'));
       
       $this->db->where('created_at >=', $yesterday);
       $metrics['recent_consultations'] = $this->db->count_all_results('consultations');
       
       if ($this->db->table_exists('fingerprint_verification_log')) {
           $this->db->where('verification_date >=', $yesterday);
           $metrics['recent_verifications'] = $this->db->count_all_results('fingerprint_verification_log');
       } else {
           $metrics['recent_verifications'] = 0;
       }
       
       // System health indicators
       $metrics['system_health'] = $this->getSystemHealthCheck();
       
       return $metrics;
   }

   /**
    * Get consultation statistics by time periods
    */
   public function getConsultationStatsByPeriod($period = 'daily', $limit = 30) {
       $stats = [];
       
       switch ($period) {
           case 'hourly':
               $this->db->select('HOUR(created_at) as period, DATE(created_at) as date, COUNT(*) as count');
               $this->db->from('consultations');
               $this->db->where('DATE(created_at)', date('Y-m-d'));
               $this->db->group_by('HOUR(created_at)');
               $this->db->order_by('HOUR(created_at)', 'ASC');
               break;
               
           case 'daily':
               $this->db->select('DATE(created_at) as period, COUNT(*) as count');
               $this->db->from('consultations');
               $this->db->where('created_at >=', date('Y-m-d H:i:s', strtotime("-{$limit} days")));
               $this->db->group_by('DATE(created_at)');
               $this->db->order_by('DATE(created_at)', 'ASC');
               break;
               
           case 'weekly':
               $this->db->select('YEARWEEK(created_at) as period, COUNT(*) as count');
               $this->db->from('consultations');
               $this->db->where('created_at >=', date('Y-m-d H:i:s', strtotime("-{$limit} weeks")));
               $this->db->group_by('YEARWEEK(created_at)');
               $this->db->order_by('YEARWEEK(created_at)', 'ASC');
               break;
               
           case 'monthly':
               $this->db->select('DATE_FORMAT(created_at, "%Y-%m") as period, COUNT(*) as count');
               $this->db->from('consultations');
               $this->db->where('created_at >=', date('Y-m-d H:i:s', strtotime("-{$limit} months")));
               $this->db->group_by('DATE_FORMAT(created_at, "%Y-%m")');
               $this->db->order_by('DATE_FORMAT(created_at, "%Y-%m")', 'ASC');
               break;
       }
       
       $this->db->limit($limit);
       $query = $this->db->get();
       
       return $query->result_array();
   }

   /**
    * Get diagnosis usage trends
    */
   public function getDiagnosisUsageTrends($months = 6, $topN = 10) {
       $trends = [];
       
       // Get top N diagnoses overall
       $topDiagnoses = $this->getMostUsedDiagnoses($topN);
       
       for ($i = $months - 1; $i >= 0; $i--) {
           $month = date('Y-m', strtotime("-{$i} months"));
           $monthStart = $month . '-01';
           $monthEnd = date('Y-m-t', strtotime($monthStart));
           
           $monthData = [
               'month' => $month,
               'month_name' => date('M Y', strtotime($monthStart)),
               'diagnoses' => []
           ];
           
           foreach ($topDiagnoses as $diagnosis) {
               $this->db->select('COUNT(cd.id) as usage_count');
               $this->db->from('consultation_diagnosis cd');
               $this->db->join('diagnosis_codes dc', 'cd.diagnosis_code_id = dc.id');
               $this->db->join('consultations c', 'cd.consultation_id = c.id');
               $this->db->where('dc.code', $diagnosis['code']);
               $this->db->where('c.consultation_date >=', $monthStart);
               $this->db->where('c.consultation_date <=', $monthEnd);
               
               $usage = $this->db->get()->row();
               
               $monthData['diagnoses'][] = [
                   'code' => $diagnosis['code'],
                   'title' => $diagnosis['title'],
                   'usage_count' => $usage ? $usage->usage_count : 0
               ];
           }
           
           $trends[] = $monthData;
       }
       
       return $trends;
   }
}

?>