<?php

if (!defined('BASEPATH')) {
    exit('No direct script access allowed');
}

class Community_card_model extends MY_Model
{ 
    /**
     * Verify a community card by card number
     * @param string $card_number
     * @return array|false
     */
    public function verify_card($card_number)
    {
        $this->db->where('card_number', $card_number);
        $card = $this->db->get('community_card_registry')->row_array();
        
        if ($card) {
            // Check if card is expired
            $today = date('Y-m-d');
            if ($card['expiry_date'] < $today && $card['status'] === 'active') {
                // Auto-update expired cards
                $this->db->where('id', $card['id']);
                $this->db->update('community_card_registry', ['status' => 'expired']);
                $card['status'] = 'expired';
            }
            
            return $card;
        }
        
        return false;
    }
    
    /**
     * Renew a community card
     * @param string $card_number
     * @return bool
     */
    public function renew_card($card_number)
    {
        $card = $this->verify_card($card_number);
        
        if (!$card) {
            return false;
        }
        
        $today = date('Y-m-d');
        $new_expiry = date('Y-m-d', strtotime('+1 year'));
        
        $renewal_data = [
            'expiry_date' => $new_expiry,
            'status' => 'active',
            'renewal_count' => $card['renewal_count'] + 1,
            'last_renewed' => $today,
            'updated_at' => date('Y-m-d H:i:s')
        ];
        
        $this->db->where('card_number', $card_number);
        return $this->db->update('community_card_registry', $renewal_data);
    }
    
    /**
     * Get card coverage details
     * @param string $coverage_level
     * @return array
     */
    public function get_coverage_details($coverage_level)
    {
        $coverage_rates = [
            'basic' => [
                'percentage' => 60,
                'services' => [
                    'consultation' => 60,
                    'laboratory' => 60,
                    'medication' => 60,
                    'preventive' => 70
                ],
                'annual_limit' => 50000
            ],
            'plus' => [
                'percentage' => 70,
                'services' => [
                    'consultation' => 70,
                    'laboratory' => 70,
                    'medication' => 70,
                    'preventive' => 80,
                    'radiology' => 60
                ],
                'annual_limit' => 100000
            ],
            'premium' => [
                'percentage' => 80,
                'services' => [
                    'consultation' => 80,
                    'laboratory' => 80,
                    'medication' => 80,
                    'preventive' => 90,
                    'radiology' => 80,
                    'emergency' => 70
                ],
                'annual_limit' => 200000
            ]
        ];
        
        return isset($coverage_rates[$coverage_level]) ? $coverage_rates[$coverage_level] : $coverage_rates['basic'];
    }
    
    /**
     * Log community card usage
     * @param array $usage_data
     * @return bool
     */
    public function log_usage($usage_data)
    {
        $required_fields = ['card_number', 'patient_id', 'service_type', 'original_amount', 'coverage_percentage', 'covered_amount', 'copay_amount'];
        
        foreach ($required_fields as $field) {
            if (!isset($usage_data[$field])) {
                return false;
            }
        }
        
        $usage_data['service_date'] = date('Y-m-d H:i:s');
        $usage_data['created_at'] = date('Y-m-d H:i:s');
        
        return $this->db->insert('community_card_usage_log', $usage_data);
    }
    
    /**
     * Get card usage history
     * @param string $card_number
     * @param int $limit
     * @return array
     */
    public function get_usage_history($card_number, $limit = 10)
    {
        $this->db->select('ccul.*, p.patient_name');
        $this->db->from('community_card_usage_log ccul');
        $this->db->join('patients p', 'p.id = ccul.patient_id', 'left');
        $this->db->where('ccul.card_number', $card_number);
        $this->db->order_by('ccul.service_date', 'DESC');
        $this->db->limit($limit);
        
        return $this->db->get()->result_array();
    }
    
    /**
     * Calculate card utilization for current year
     * @param string $card_number
     * @return array
     */
    public function get_card_utilization($card_number)
    {
        $year_start = date('Y-01-01');
        $year_end = date('Y-12-31');
        
        $this->db->select('SUM(covered_amount) as total_covered, COUNT(*) as service_count');
        $this->db->where('card_number', $card_number);
        $this->db->where('service_date >=', $year_start);
        $this->db->where('service_date <=', $year_end);
        
        $result = $this->db->get('community_card_usage_log')->row_array();
        
        return [
            'total_covered' => $result['total_covered'] ?: 0,
            'service_count' => $result['service_count'] ?: 0,
            'year' => date('Y')
        ];
    }
    
    /**
     * Get all active community cards
     * @return array
     */
    public function get_active_cards()
    {
        $this->db->where('status', 'active');
        $this->db->where('expiry_date >', date('Y-m-d'));
        $this->db->order_by('expiry_date', 'ASC');
        
        return $this->db->get('community_card_registry')->result_array();
    }
    
    /**
     * Get expiring cards (within 30 days)
     * @return array
     */
    public function get_expiring_cards()
    {
        $today = date('Y-m-d');
        $thirty_days_from_now = date('Y-m-d', strtotime('+30 days'));
        
        $this->db->where('status', 'active');
        $this->db->where('expiry_date >', $today);
        $this->db->where('expiry_date <=', $thirty_days_from_now);
        $this->db->order_by('expiry_date', 'ASC');
        
        return $this->db->get('community_card_registry')->result_array();
    }
    
    /**
     * Create new community card
     * @param array $card_data
     * @return int|false
     */
    public function create_card($card_data)
    {
        // Generate card number if not provided
        if (!isset($card_data['card_number'])) {
            $card_data['card_number'] = $this->generate_card_number();
        }
        
        // Set default dates if not provided
        if (!isset($card_data['issue_date'])) {
            $card_data['issue_date'] = date('Y-m-d');
        }
        
        if (!isset($card_data['expiry_date'])) {
            $card_data['expiry_date'] = date('Y-m-d', strtotime('+1 year'));
        }
        
        $card_data['created_at'] = date('Y-m-d H:i:s');
        
        if ($this->db->insert('community_card_registry', $card_data)) {
            return $this->db->insert_id();
        }
        
        return false;
    }
    
    /**
     * Generate unique card number
     * @return string
     */
    private function generate_card_number()
    {
        $year = date('Y');
        $branch = '0001'; // Default branch code
        
        // Get next sequential number
        $this->db->select_max('card_number');
        $this->db->like('card_number', "CHC-{$year}-{$branch}-", 'after');
        $result = $this->db->get('community_card_registry')->row_array();
        
        $next_number = 1;
        if ($result['card_number']) {
            $parts = explode('-', $result['card_number']);
            if (count($parts) === 4) {
                $next_number = intval($parts[3]) + 1;
            }
        }
        
        return sprintf('CHC-%s-%s-%04d', $year, $branch, $next_number);
    }
    
    /**
     * Update card status
     * @param string $card_number
     * @param string $status
     * @param string $notes
     * @return bool
     */
    public function update_card_status($card_number, $status, $notes = null)
    {
        $update_data = [
            'status' => $status,
            'updated_at' => date('Y-m-d H:i:s')
        ];
        
        if ($notes) {
            $update_data['notes'] = $notes;
        }
        
        $this->db->where('card_number', $card_number);
        return $this->db->update('community_card_registry', $update_data);
    }
}