<?php
if (!defined('BASEPATH')) {
    exit('No direct script access allowed');
}

class Billing_model extends CI_Model
{
    public function __construct()
    {
        parent::__construct();
        $this->load->database();
    }

    /**
     * Create a new invoice
     */
    public function createInvoice($invoice_data)
    {
        $this->db->insert('patient_invoices', $invoice_data);
        return $this->db->insert_id();
    }

    /**
     * Create invoice line item
     */
    public function createInvoiceLineItem($line_item_data)
    {
        $this->db->insert('patient_invoice_items', $line_item_data);
        return $this->db->insert_id();
    }

    /**
     * Get invoice details with line items
     */
    public function getInvoiceDetails($invoice_id)
    {
        // Get main invoice data
        $this->db->select('pi.*, p.patient_name, p.mobileno, p.email, p.patient_unique_id, p.address');
        $this->db->from('patient_invoices pi');
        $this->db->join('patients p', 'p.id = pi.patient_id', 'left');
        $this->db->where('pi.id', $invoice_id);
        $invoice = $this->db->get()->row_array();

        if (!empty($invoice)) {
            // Get line items
            $this->db->select('pii.*, pc.name as charge_name');
            $this->db->from('patient_invoice_items pii');
            $this->db->join('pathology_categories pc', 'pc.id = pii.module_id', 'left'); // This will need adjustment based on module
            $this->db->where('pii.invoice_id', $invoice_id);
            $invoice['line_items'] = $this->db->get()->result_array();

            // Get payments for this invoice
            $this->db->select('pip.*, s.name as received_by_name');
            $this->db->from('patient_invoice_payments pip');
            $this->db->join('staff s', 's.id = pip.received_by', 'left');
            $this->db->where('pip.invoice_id', $invoice_id);
            $this->db->order_by('pip.payment_date', 'desc');
            $invoice['payments'] = $this->db->get()->result_array();

            // Calculate payment totals
            $total_payments = 0;
            foreach ($invoice['payments'] as $payment) {
                $total_payments += $payment['amount'];
            }
            $invoice['total_paid'] = $total_payments;
            $invoice['balance_due'] = $invoice['total_amount'] - $total_payments;
        }

        return $invoice;
    }

    /**
     * Get all invoices for a patient by patient ID
     */
    public function getInvoicesByPatient($patient_id)
    {
        $this->db->select('pi.*, p.patient_name');
        $this->db->from('patient_invoices pi');
        $this->db->join('patients p', 'p.id = pi.patient_id', 'left');
        $this->db->where('pi.patient_id', $patient_id);
        $this->db->order_by('pi.invoice_date', 'desc');
        
        $invoices = $this->db->get()->result_array();
        
        // Get payment totals for each invoice
        foreach ($invoices as &$invoice) {
            $payment_total = $this->getTotalPaymentsForInvoice($invoice['id']);
            $invoice['total_paid'] = $payment_total;
            $invoice['balance_due'] = $invoice['total_amount'] - $payment_total;
            
            // Update status based on payments
            if ($payment_total >= $invoice['total_amount']) {
                $invoice['status'] = 'paid';
            } elseif ($payment_total > 0) {
                $invoice['status'] = 'partial';
            } else {
                $invoice['status'] = 'pending';
            }
        }
        
        return $invoices;
    }

    /**
     * Add payment for an invoice
     */
    public function addPayment($payment_data)
    {
        $this->db->insert('patient_invoice_payments', $payment_data);
        return $this->db->insert_id();
    }

    /**
     * Get total payments for an invoice
     */
    public function getTotalPaymentsForInvoice($invoice_id)
    {
        $this->db->select_sum('amount');
        $this->db->where('invoice_id', $invoice_id);
        $result = $this->db->get('patient_invoice_payments')->row();
        return $result ? (float)$result->amount : 0;
    }

    /**
     * Update invoice status
     */
    public function updateInvoiceStatus($invoice_id, $status)
    {
        $this->db->where('id', $invoice_id);
        return $this->db->update('patient_invoices', array('status' => $status, 'updated_at' => date('Y-m-d H:i:s')));
    }

    /**
     * Get invoice by ID
     */
    public function getInvoice($invoice_id)
    {
        $this->db->where('id', $invoice_id);
        return $this->db->get('patient_invoices')->row_array();
    }

    /**
     * Generate next invoice number for the month
     */
    public function getLastInvoiceNumber($year, $month)
    {
        $this->db->select('invoice_number');
        $this->db->like('invoice_number', 'INV' . $year . $month, 'after');
        $this->db->order_by('id', 'desc');
        $this->db->limit(1);
        $result = $this->db->get('patient_invoices')->row();
        
        if ($result) {
            // Extract number from invoice number (e.g., INV2025070001 -> 1)
            $number_part = substr($result->invoice_number, -4);
            return (int)$number_part;
        }
        
        return 0;
    }

    /**
     * Get billing statistics - With error handling
     */
    public function getTotalInvoicesCount($date = null)
    {
        try {
            if (!$this->db->table_exists('patient_invoices')) {
                return 0;
            }
            
            if ($date) {
                $this->db->where('DATE(invoice_date)', $date);
            }
            
            return $this->db->count_all_results('patient_invoices');
            
        } catch (Exception $e) {
            log_message('error', 'getTotalInvoicesCount error: ' . $e->getMessage());
            return 0;
        }
    }

    public function getTotalPaymentsSum($date = null)
    {
        try {
            if (!$this->db->table_exists('patient_invoice_payments')) {
                return 0;
            }
            
            $this->db->select_sum('amount');
            if ($date) {
                $this->db->where('DATE(payment_date)', $date);
            }
            
            $result = $this->db->get('patient_invoice_payments')->row();
            return $result ? (float)$result->amount : 0;
            
        } catch (Exception $e) {
            log_message('error', 'getTotalPaymentsSum error: ' . $e->getMessage());
            return 0;
        }
    }

    public function getPendingInvoicesCount()
    {
        try {
            if (!$this->db->table_exists('patient_invoices')) {
                return 0;
            }
            
            $this->db->where_in('status', array('pending', 'partial'));
            return $this->db->count_all_results('patient_invoices');
            
        } catch (Exception $e) {
            log_message('error', 'getPendingInvoicesCount error: ' . $e->getMessage());
            return 0;
        }
    }

    public function getOverdueInvoicesCount()
    {
        try {
            if (!$this->db->table_exists('patient_invoices')) {
                return 0;
            }
            
            $this->db->where('due_date <', date('Y-m-d'));
            $this->db->where_in('status', array('pending', 'partial'));
            return $this->db->count_all_results('patient_invoices');
            
        } catch (Exception $e) {
            log_message('error', 'getOverdueInvoicesCount error: ' . $e->getMessage());
            return 0;
        }
    }

    /**
     * Get recent invoices - With error handling
     */
    public function getRecentInvoices($limit = 10)
    {
        try {
            if (!$this->db->table_exists('patient_invoices')) {
                return array();
            }
            
            $this->db->select('pi.*, p.patient_name');
            $this->db->from('patient_invoices pi');
            $this->db->join('patients p', 'p.id = pi.patient_id', 'left');
            $this->db->order_by('pi.created_at', 'desc');
            $this->db->limit($limit);
            
            $result = $this->db->get();
            return $result ? $result->result_array() : array();
            
        } catch (Exception $e) {
            log_message('error', 'getRecentInvoices error: ' . $e->getMessage());
            return array();
        }
    }

    /**
     * Get pending payments (overdue invoices) - With error handling
     */
    public function getPendingPayments($limit = 10)
    {
        try {
            if (!$this->db->table_exists('patient_invoices') || !$this->db->table_exists('patient_invoice_payments')) {
                return array();
            }
            
            $this->db->select('pi.id, pi.invoice_number, pi.total_amount, pi.due_date, p.patient_name, pi.patient_id, 
                              (pi.total_amount - COALESCE(SUM(pip.amount), 0)) as amount_due');
            $this->db->from('patient_invoices pi');
            $this->db->join('patients p', 'p.id = pi.patient_id', 'left');
            $this->db->join('patient_invoice_payments pip', 'pip.invoice_id = pi.id', 'left');
            $this->db->where('pi.status !=', 'paid');
            $this->db->group_by('pi.id');
            $this->db->having('amount_due >', 0);
            $this->db->order_by('pi.due_date', 'asc');
            $this->db->limit($limit);
            
            $result = $this->db->get();
            return $result ? $result->result_array() : array();
            
        } catch (Exception $e) {
            log_message('error', 'getPendingPayments error: ' . $e->getMessage());
            return array();
        }
    }

    /**
     * Get saved invoices for download center
     */
    public function getSavedInvoices($limit, $offset)
    {
        $this->db->select('pi.*, p.patient_name');
        $this->db->from('patient_invoices pi');
        $this->db->join('patients p', 'p.id = pi.patient_id', 'left');
        $this->db->order_by('pi.created_at', 'desc');
        $this->db->limit($limit, $offset);
        return $this->db->get()->result_array();
    }

    public function getTotalSavedInvoices()
    {
        return $this->db->count_all('patient_invoices');
    }

    /**
     * Generate billing summary report
     */
    public function generateBillingSummaryReport($date_from, $date_to)
    {
        // Total invoices generated
        $this->db->where('invoice_date >=', $date_from);
        $this->db->where('invoice_date <=', $date_to);
        $total_invoices = $this->db->count_all_results('patient_invoices');

        // Total invoice amount
        $this->db->select_sum('total_amount');
        $this->db->where('invoice_date >=', $date_from);
        $this->db->where('invoice_date <=', $date_to);
        $total_billed = $this->db->get('patient_invoices')->row()->total_amount ?: 0;

        // Invoice status breakdown
        $this->db->select('status, COUNT(*) as count, SUM(total_amount) as amount');
        $this->db->where('invoice_date >=', $date_from);
        $this->db->where('invoice_date <=', $date_to);
        $this->db->group_by('status');
        $status_breakdown = $this->db->get('patient_invoices')->result_array();

        return array(
            'total_invoices' => $total_invoices,
            'total_billed' => $total_billed,
            'status_breakdown' => $status_breakdown,
            'date_range' => array('from' => $date_from, 'to' => $date_to)
        );
    }

    /**
     * Generate payment summary report
     */
    public function generatePaymentSummaryReport($date_from, $date_to)
    {
        // Total payments received
        $this->db->select_sum('amount');
        $this->db->where('payment_date >=', $date_from);
        $this->db->where('payment_date <=', $date_to);
        $total_collected = $this->db->get('patient_invoice_payments')->row()->amount ?: 0;

        // Payment mode breakdown
        $this->db->select('payment_mode, COUNT(*) as count, SUM(amount) as amount');
        $this->db->where('payment_date >=', $date_from);
        $this->db->where('payment_date <=', $date_to);
        $this->db->group_by('payment_mode');
        $mode_breakdown = $this->db->get('patient_invoice_payments')->result_array();

        // Daily collection trend
        $this->db->select('DATE(payment_date) as payment_date, SUM(amount) as daily_total');
        $this->db->where('payment_date >=', $date_from);
        $this->db->where('payment_date <=', $date_to);
        $this->db->group_by('DATE(payment_date)');
        $this->db->order_by('payment_date', 'asc');
        $daily_trend = $this->db->get('patient_invoice_payments')->result_array();

        return array(
            'total_collected' => $total_collected,
            'mode_breakdown' => $mode_breakdown,
            'daily_trend' => $daily_trend,
            'date_range' => array('from' => $date_from, 'to' => $date_to)
        );
    }

    /**
     * Generate outstanding summary report
     */
    public function generateOutstandingSummaryReport()
    {
        // Total outstanding amount
        $this->db->select('SUM(pi.total_amount - COALESCE(payment_totals.total_paid, 0)) as total_outstanding');
        $this->db->from('patient_invoices pi');
        $this->db->join('(SELECT invoice_id, SUM(amount) as total_paid FROM patient_invoice_payments GROUP BY invoice_id) payment_totals', 
                       'payment_totals.invoice_id = pi.id', 'left');
        $this->db->where('pi.status !=', 'paid');
        $total_outstanding = $this->db->get()->row()->total_outstanding ?: 0;

        // Aging analysis
        $aging_ranges = array(
            '0-30' => array('min' => 0, 'max' => 30),
            '31-60' => array('min' => 31, 'max' => 60),
            '61-90' => array('min' => 61, 'max' => 90),
            '90+' => array('min' => 91, 'max' => 9999)
        );

        $aging_analysis = array();
        foreach ($aging_ranges as $range => $days) {
            $this->db->select('COUNT(*) as count, SUM(pi.total_amount - COALESCE(payment_totals.total_paid, 0)) as amount');
            $this->db->from('patient_invoices pi');
            $this->db->join('(SELECT invoice_id, SUM(amount) as total_paid FROM patient_invoice_payments GROUP BY invoice_id) payment_totals', 
                           'payment_totals.invoice_id = pi.id', 'left');
            $this->db->where('pi.status !=', 'paid');
            $this->db->where('DATEDIFF(CURDATE(), pi.due_date) >=', $days['min']);
            if ($days['max'] < 9999) {
                $this->db->where('DATEDIFF(CURDATE(), pi.due_date) <=', $days['max']);
            }
            $result = $this->db->get()->row_array();
            $aging_analysis[$range] = array(
                'count' => $result['count'] ?: 0,
                'amount' => $result['amount'] ?: 0
            );
        }

        return array(
            'total_outstanding' => $total_outstanding,
            'aging_analysis' => $aging_analysis
        );
    }

    /**
     * Get payment history for a patient
     */
    public function getPaymentHistoryByPatient($patient_id)
    {
        $this->db->select('pip.*, pi.invoice_number, s.name as received_by_name');
        $this->db->from('patient_invoice_payments pip');
        $this->db->join('patient_invoices pi', 'pi.id = pip.invoice_id', 'left');
        $this->db->join('staff s', 's.id = pip.received_by', 'left');
        $this->db->where('pi.patient_id', $patient_id);
        $this->db->order_by('pip.payment_date', 'desc');
        return $this->db->get()->result_array();
    }

    /**
     * Search invoices by various criteria - Fixed query
     */
    public function searchInvoices($search_params)
    {
        $this->db->select('pi.*, p.patient_name, p.patient_unique_id');
        $this->db->from('patient_invoices pi');
        $this->db->join('patients p', 'p.id = pi.patient_id', 'left');

        if (!empty($search_params['invoice_number'])) {
            $this->db->like('pi.invoice_number', $search_params['invoice_number']);
        }

        if (!empty($search_params['patient_name'])) {
            $this->db->like('p.patient_name', $search_params['patient_name']);
        }

        if (!empty($search_params['patient_id'])) {
            $this->db->where('pi.patient_id', $search_params['patient_id']);
        }

        if (!empty($search_params['status'])) {
            $this->db->where('pi.status', $search_params['status']);
        }

        if (!empty($search_params['date_from'])) {
            $this->db->where('pi.invoice_date >=', $search_params['date_from']);
        }

        if (!empty($search_params['date_to'])) {
            $this->db->where('pi.invoice_date <=', $search_params['date_to']);
        }

        $this->db->order_by('pi.invoice_date', 'desc');
        return $this->db->get()->result_array();
    }

    /**
     * Get module-wise revenue breakdown
     */
    public function getModuleRevenueBreakdown($date_from, $date_to)
    {
        $this->db->select('pii.module_type, COUNT(*) as item_count, SUM(pii.total_price) as total_revenue');
        $this->db->from('patient_invoice_items pii');
        $this->db->join('patient_invoices pi', 'pi.id = pii.invoice_id', 'left');
        $this->db->where('pi.invoice_date >=', $date_from);
        $this->db->where('pi.invoice_date <=', $date_to);
        $this->db->group_by('pii.module_type');
        $this->db->order_by('total_revenue', 'desc');
        return $this->db->get()->result_array();
    }

    /**
     * Get top paying patients
     */
    public function getTopPayingPatients($limit = 10, $date_from = null, $date_to = null)
    {
        $this->db->select('p.patient_name, p.patient_unique_id, SUM(pip.amount) as total_paid, COUNT(DISTINCT pi.id) as invoice_count');
        $this->db->from('patient_invoice_payments pip');
        $this->db->join('patient_invoices pi', 'pi.id = pip.invoice_id', 'left');
        $this->db->join('patients p', 'p.id = pi.patient_id', 'left');

        if ($date_from) {
            $this->db->where('pip.payment_date >=', $date_from);
        }
        if ($date_to) {
            $this->db->where('pip.payment_date <=', $date_to);
        }

        $this->db->group_by('pi.patient_id');
        $this->db->order_by('total_paid', 'desc');
        $this->db->limit($limit);
        return $this->db->get()->result_array();
    }

    /**
     * Create database tables if they don't exist
     */
    public function createBillingTables()
    {
        // Patient Invoices Table
        if (!$this->db->table_exists('patient_invoices')) {
            $this->dbforge->add_field(array(
                'id' => array('type' => 'INT', 'auto_increment' => TRUE),
                'invoice_number' => array('type' => 'VARCHAR', 'constraint' => '50', 'unique' => TRUE),
                'case_id' => array('type' => 'VARCHAR', 'constraint' => '50'),
                'patient_id' => array('type' => 'INT'),
                'invoice_type' => array('type' => 'ENUM', 'constraint' => array('final', 'interim', 'partial')),
                'invoice_date' => array('type' => 'DATE'),
                'due_date' => array('type' => 'DATE'),
                'subtotal' => array('type' => 'DECIMAL', 'constraint' => '10,2'),
                'tax_amount' => array('type' => 'DECIMAL', 'constraint' => '10,2', 'default' => 0),
                'discount_amount' => array('type' => 'DECIMAL', 'constraint' => '10,2', 'default' => 0),
                'total_amount' => array('type' => 'DECIMAL', 'constraint' => '10,2'),
                'status' => array('type' => 'ENUM', 'constraint' => array('pending', 'partial', 'paid', 'cancelled'), 'default' => 'pending'),
                'notes' => array('type' => 'TEXT', 'null' => TRUE),
                'created_by' => array('type' => 'INT'),
                'created_at' => array('type' => 'TIMESTAMP', 'default' => 'CURRENT_TIMESTAMP'),
                'updated_at' => array('type' => 'TIMESTAMP', 'default' => 'CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP')
            ));
            $this->dbforge->add_key('id', TRUE);
            $this->dbforge->add_key('case_id');
            $this->dbforge->add_key('patient_id');
            $this->dbforge->add_key('invoice_date');
            $this->dbforge->create_table('patient_invoices');
        }

        // Patient Invoice Items Table
        if (!$this->db->table_exists('patient_invoice_items')) {
            $this->dbforge->add_field(array(
                'id' => array('type' => 'INT', 'auto_increment' => TRUE),
                'invoice_id' => array('type' => 'INT'),
                'module_type' => array('type' => 'VARCHAR', 'constraint' => '50'),
                'module_id' => array('type' => 'INT'),
                'description' => array('type' => 'TEXT'),
                'quantity' => array('type' => 'INT', 'default' => 1),
                'unit_price' => array('type' => 'DECIMAL', 'constraint' => '10,2'),
                'total_price' => array('type' => 'DECIMAL', 'constraint' => '10,2'),
                'created_at' => array('type' => 'TIMESTAMP', 'default' => 'CURRENT_TIMESTAMP')
            ));
            $this->dbforge->add_key('id', TRUE);
            $this->dbforge->add_key('invoice_id');
            $this->dbforge->create_table('patient_invoice_items');
        }

        // Patient Invoice Payments Table
        if (!$this->db->table_exists('patient_invoice_payments')) {
            $this->dbforge->add_field(array(
                'id' => array('type' => 'INT', 'auto_increment' => TRUE),
                'invoice_id' => array('type' => 'INT'),
                'amount' => array('type' => 'DECIMAL', 'constraint' => '10,2'),
                'payment_mode' => array('type' => 'VARCHAR', 'constraint' => '50'),
                'payment_date' => array('type' => 'DATE'),
                'cheque_no' => array('type' => 'VARCHAR', 'constraint' => '50', 'null' => TRUE),
                'cheque_date' => array('type' => 'DATE', 'null' => TRUE),
                'attachment' => array('type' => 'VARCHAR', 'constraint' => '255', 'null' => TRUE),
                'attachment_name' => array('type' => 'VARCHAR', 'constraint' => '255', 'null' => TRUE),
                'note' => array('type' => 'TEXT', 'null' => TRUE),
                'received_by' => array('type' => 'INT'),
                'created_at' => array('type' => 'TIMESTAMP', 'default' => 'CURRENT_TIMESTAMP')
            ));
            $this->dbforge->add_key('id', TRUE);
            $this->dbforge->add_key('invoice_id');
            $this->dbforge->add_key('payment_date');
            $this->dbforge->create_table('patient_invoice_payments');
        }
    }
}
?>