<?php

if (!defined('BASEPATH')) {
    exit('No direct script access allowed');
}

class Agent_model extends MY_Model
{ 
    /**
     * Get all active agents
     * @return array
     */
    public function get_active_agents()
    {
        $this->db->where('status', 'active');
        $this->db->order_by('agent_name', 'ASC');
        return $this->db->get('agents')->result_array();
    }
    
    /**
     * Get agent by ID with their active contracts
     * @param int $agent_id
     * @return array|false
     */
    public function get_agent_with_contracts($agent_id)
    {
        // Get agent details
        $this->db->where('id', $agent_id);
        $agent = $this->db->get('agents')->row_array();
        
        if (!$agent) {
            return false;
        }
        
        // Get active contracts for this agent
        $this->db->select('acc.*, ac.country_name, ac.country_code, ac.currency_code, ac.flag_image');
        $this->db->from('agent_country_contracts acc');
        $this->db->join('agent_countries ac', 'ac.id = acc.country_id');
        $this->db->where('acc.agent_id', $agent_id);
        $this->db->where('acc.status', 'active');
        $this->db->where('acc.contract_end_date >', date('Y-m-d'));
        $this->db->order_by('ac.country_name', 'ASC');
        
        $contracts = $this->db->get()->result_array();
        
        $agent['contracts'] = $contracts;
        return $agent;
    }
    
    /**
     * Get available countries for an agent
     * @param int $agent_id
     * @return array
     */
    public function get_agent_countries($agent_id)
    {
        $this->db->select('acc.id as contract_id, acc.total_cost_per_patient, acc.agent_commission_per_patient, 
                          acc.patient_limit, acc.patients_registered, acc.contract_name,
                          ac.id as country_id, ac.country_name, ac.country_code, ac.currency_code, ac.flag_image');
        $this->db->from('agent_country_contracts acc');
        $this->db->join('agent_countries ac', 'ac.id = acc.country_id');
        $this->db->where('acc.agent_id', $agent_id);
        $this->db->where('acc.status', 'active');
        $this->db->where('acc.contract_end_date >', date('Y-m-d'));
        $this->db->order_by('ac.country_name', 'ASC');
        
        return $this->db->get()->result_array();
    }
    
    /**
     * Check if agent can register more patients for a specific contract
     * @param int $contract_id
     * @return array
     */
    public function check_contract_availability($contract_id)
    {
        $this->db->where('id', $contract_id);
        $contract = $this->db->get('agent_country_contracts')->row_array();
        
        if (!$contract) {
            return ['status' => 'error', 'message' => 'Contract not found'];
        }
        
        if ($contract['status'] !== 'active') {
            return ['status' => 'error', 'message' => 'Contract is not active'];
        }
        
        if ($contract['contract_end_date'] < date('Y-m-d')) {
            return ['status' => 'error', 'message' => 'Contract has expired'];
        }
        
        $remaining = $contract['patient_limit'] - $contract['patients_registered'];
        
        if ($remaining <= 0) {
            return ['status' => 'error', 'message' => 'Patient limit reached for this contract'];
        }
        
        return [
            'status' => 'success',
            'remaining_slots' => $remaining,
            'total_limit' => $contract['patient_limit'],
            'registered' => $contract['patients_registered'],
            'contract' => $contract
        ];
    }
    
    /**
     * Register a patient under an agent contract
     * @param array $registration_data
     * @return array
     */
    public function register_patient($registration_data)
    {
        // Check contract availability first
        $availability = $this->check_contract_availability($registration_data['contract_id']);
        
        if ($availability['status'] !== 'success') {
            return $availability;
        }
        
        $this->db->trans_start();
        
        // Generate registration number
        $registration_data['registration_number'] = $this->generate_registration_number($registration_data['agent_id'], $registration_data['country_id']);
        
        // Insert registration record
        $registration_data['registration_date'] = date('Y-m-d H:i:s');
        $this->db->insert('agent_patient_registrations', $registration_data);
        $registration_id = $this->db->insert_id();
        
        // Update contract patient count
        $this->db->set('patients_registered', 'patients_registered + 1', FALSE);
        $this->db->where('id', $registration_data['contract_id']);
        $this->db->update('agent_country_contracts');
        
        // Check if contract is now full
        $updated_contract = $this->db->get_where('agent_country_contracts', ['id' => $registration_data['contract_id']])->row_array();
        if ($updated_contract['patients_registered'] >= $updated_contract['patient_limit']) {
            $this->db->where('id', $registration_data['contract_id']);
            $this->db->update('agent_country_contracts', ['status' => 'full']);
        }
        
        $this->db->trans_complete();
        
        if ($this->db->trans_status() === FALSE) {
            return ['status' => 'error', 'message' => 'Failed to register patient'];
        }
        
        return [
            'status' => 'success', 
            'registration_id' => $registration_id,
            'registration_number' => $registration_data['registration_number'],
            'message' => 'Patient registered successfully'
        ];
    }
    
    /**
     * Generate unique registration number
     * @param int $agent_id
     * @param int $country_id
     * @return string
     */
    private function generate_registration_number($agent_id, $country_id)
    {
        // Get agent code and country code
        $agent = $this->db->get_where('agents', ['id' => $agent_id])->row_array();
        $country = $this->db->get_where('agent_countries', ['id' => $country_id])->row_array();
        
        $year = date('Y');
        $prefix = $agent['agent_code'] . '-' . $country['country_code'] . '-' . $year . '-';
        
        // Get next sequential number
        $this->db->select_max('registration_number');
        $this->db->like('registration_number', $prefix, 'after');
        $result = $this->db->get('agent_patient_registrations')->row_array();
        
        $next_number = 1;
        if ($result['registration_number']) {
            $parts = explode('-', $result['registration_number']);
            if (count($parts) >= 4) {
                $next_number = intval(end($parts)) + 1;
            }
        }
        
        return $prefix . sprintf('%04d', $next_number);
    }
    
    /**
     * Get agent dashboard statistics
     * @param int $agent_id
     * @return array
     */
    public function get_agent_stats($agent_id)
    {
        // Total contracts
        $this->db->where('agent_id', $agent_id);
        $total_contracts = $this->db->count_all_results('agent_country_contracts');
        
        // Active contracts
        $this->db->where('agent_id', $agent_id);
        $this->db->where('status', 'active');
        $this->db->where('contract_end_date >', date('Y-m-d'));
        $active_contracts = $this->db->count_all_results('agent_country_contracts');
        
        // Total patients registered
        $this->db->select_sum('patients_registered');
        $this->db->where('agent_id', $agent_id);
        $result = $this->db->get('agent_country_contracts')->row_array();
        $total_patients = $result['patients_registered'] ?: 0;
        
        // Total commission earned (this year)
        $this->db->select_sum('agent_commission');
        $this->db->where('agent_id', $agent_id);
        $this->db->where('YEAR(registration_date)', date('Y'));
        $result = $this->db->get('agent_patient_registrations')->row_array();
        $total_commission = $result['agent_commission'] ?: 0;
        
        // Contracts near limit (>80% full)
        $this->db->where('agent_id', $agent_id);
        $this->db->where('status', 'active');
        $this->db->where('(patients_registered / patient_limit) >', 0.8);
        $contracts_near_limit = $this->db->count_all_results('agent_country_contracts');
        
        // Contracts expiring soon (within 30 days)
        $thirty_days_from_now = date('Y-m-d', strtotime('+30 days'));
        $this->db->where('agent_id', $agent_id);
        $this->db->where('status', 'active');
        $this->db->where('contract_end_date <=', $thirty_days_from_now);
        $this->db->where('contract_end_date >', date('Y-m-d'));
        $contracts_expiring = $this->db->count_all_results('agent_country_contracts');
        
        return [
            'total_contracts' => $total_contracts,
            'active_contracts' => $active_contracts,
            'total_patients' => $total_patients,
            'total_commission' => $total_commission,
            'contracts_near_limit' => $contracts_near_limit,
            'contracts_expiring' => $contracts_expiring
        ];
    }
    
    /**
     * Get contract details with remaining capacity
     * @param int $contract_id
     * @return array|false
     */
    public function get_contract_details($contract_id)
    {
        $this->db->select('acc.*, ac.country_name, ac.country_code, ac.currency_code, a.agent_name, a.agent_code');
        $this->db->from('agent_country_contracts acc');
        $this->db->join('agent_countries ac', 'ac.id = acc.country_id');
        $this->db->join('agents a', 'a.id = acc.agent_id');
        $this->db->where('acc.id', $contract_id);
        
        $contract = $this->db->get()->row_array();
        
        if ($contract) {
            $contract['remaining_slots'] = $contract['patient_limit'] - $contract['patients_registered'];
            $contract['utilization_percentage'] = $contract['patient_limit'] > 0 ? 
                round(($contract['patients_registered'] / $contract['patient_limit']) * 100, 2) : 0;
            
            // Check if renewal is needed
            $days_to_expiry = (strtotime($contract['contract_end_date']) - time()) / (60 * 60 * 24);
            $contract['days_to_expiry'] = floor($days_to_expiry);
            $contract['needs_renewal'] = $days_to_expiry <= $contract['renewal_notice_days'];
        }
        
        return $contract;
    }
    
    /**
     * Get all countries
     * @return array
     */
    public function get_all_countries()
    {
        $this->db->where('is_active', 'yes');
        $this->db->order_by('country_name', 'ASC');
        return $this->db->get('agent_countries')->result_array();
    }
    
    /**
     * Create new agent
     * @param array $agent_data
     * @return int|false
     */
    public function create_agent($agent_data)
    {
        // Generate agent code if not provided
        if (!isset($agent_data['agent_code'])) {
            $agent_data['agent_code'] = $this->generate_agent_code();
        }
        
        $agent_data['created_at'] = date('Y-m-d H:i:s');
        
        if ($this->db->insert('agents', $agent_data)) {
            return $this->db->insert_id();
        }
        
        return false;
    }
    
    /**
     * Generate unique agent code
     * @return string
     */
    private function generate_agent_code()
    {
        $this->db->select_max('agent_code');
        $this->db->like('agent_code', 'AG', 'after');
        $result = $this->db->get('agents')->row_array();
        
        $next_number = 1;
        if ($result['agent_code']) {
            $number = intval(substr($result['agent_code'], 2));
            $next_number = $number + 1;
        }
        
        return 'AG' . sprintf('%03d', $next_number);
    }
    
    /**
     * Create new contract
     * @param array $contract_data
     * @return int|false
     */
    public function create_contract($contract_data)
    {
        $contract_data['created_at'] = date('Y-m-d H:i:s');
        
        if ($this->db->insert('agent_country_contracts', $contract_data)) {
            return $this->db->insert_id();
        }
        
        return false;
    }
    
    /**
     * Update contract status
     * @param int $contract_id
     * @param string $status
     * @return bool
     */
    public function update_contract_status($contract_id, $status)
    {
        $this->db->where('id', $contract_id);
        return $this->db->update('agent_country_contracts', [
            'status' => $status,
            'updated_at' => date('Y-m-d H:i:s')
        ]);
    }
    
    /**
     * Get expiring contracts (for notifications)
     * @param int $days_ahead
     * @return array
     */
    public function get_expiring_contracts($days_ahead = 30)
    {
        $end_date = date('Y-m-d', strtotime("+{$days_ahead} days"));
        
        $this->db->select('acc.*, ac.country_name, a.agent_name, a.email');
        $this->db->from('agent_country_contracts acc');
        $this->db->join('agent_countries ac', 'ac.id = acc.country_id');
        $this->db->join('agents a', 'a.id = acc.agent_id');
        $this->db->where('acc.status', 'active');
        $this->db->where('acc.contract_end_date <=', $end_date);
        $this->db->where('acc.contract_end_date >', date('Y-m-d'));
        $this->db->order_by('acc.contract_end_date', 'ASC');
        
        return $this->db->get()->result_array();
    }
    
    /**
     * Get patient registration history for an agent
     * @param int $agent_id
     * @param int $limit
     * @return array
     */
    public function get_patient_history($agent_id, $limit = 50)
    {
        $this->db->select('apr.*, p.patient_name, ac.country_name, acc.contract_name');
        $this->db->from('agent_patient_registrations apr');
        $this->db->join('patients p', 'p.id = apr.patient_id');
        $this->db->join('agent_countries ac', 'ac.id = apr.country_id');
        $this->db->join('agent_country_contracts acc', 'acc.id = apr.contract_id');
        $this->db->where('apr.agent_id', $agent_id);
        $this->db->order_by('apr.registration_date', 'DESC');
        $this->db->limit($limit);
        
        return $this->db->get()->result_array();
    }
}