<?php

if (!defined('BASEPATH'))
    exit('No direct script access allowed');

/**
 * Integration Helper for Lab Reports and Loyalty Points
 * 
 * This helper file provides functions to integrate the lab reports and loyalty points
 * systems with your existing payment and transaction systems.
 */

if (!function_exists('award_loyalty_points_on_payment')) {
    /**
     * Award loyalty points when a patient makes a payment
     * 
     * @param int $patient_id Patient ID
     * @param float $amount Payment amount
     * @param string $reference_type Type of payment (opd, ipd, pharmacy, etc.)
     * @param int $reference_id Reference ID (bill ID, etc.)
     * @param int $staff_id Staff processing the payment
     * @return bool|int Returns inserted ID on success, false on failure
     */
    function award_loyalty_points_on_payment($patient_id, $amount, $reference_type = 'payment', $reference_id = null, $staff_id = null) {
        $CI =& get_instance();
        
        // Load loyalty points model if not already loaded
        if (!isset($CI->loyalty_points_model)) {
            $CI->load->model('loyalty_points_model');
        }
        
        try {
            return $CI->loyalty_points_model->autoAwardPoints(
                $patient_id, 
                $amount, 
                $reference_type, 
                $reference_id, 
                $staff_id
            );
        } catch (Exception $e) {
            log_message('error', 'Failed to award loyalty points: ' . $e->getMessage());
            return false;
        }
    }
}

if (!function_exists('get_patient_loyalty_balance')) {
    /**
     * Get patient's current loyalty points balance
     * 
     * @param int $patient_id Patient ID
     * @return int Current loyalty points balance
     */
    function get_patient_loyalty_balance($patient_id) {
        $CI =& get_instance();
        
        if (!isset($CI->loyalty_points_model)) {
            $CI->load->model('loyalty_points_model');
        }
        
        try {
            return $CI->loyalty_points_model->getPatientCurrentBalance($patient_id);
        } catch (Exception $e) {
            log_message('error', 'Failed to get loyalty balance: ' . $e->getMessage());
            return 0;
        }
    }
}

if (!function_exists('can_use_loyalty_points')) {
    /**
     * Check if patient can use loyalty points for payment
     * 
     * @param int $patient_id Patient ID
     * @param int $points_needed Points needed for payment
     * @return bool True if patient can use points
     */
    function can_use_loyalty_points($patient_id, $points_needed) {
        $CI =& get_instance();
        
        if (!isset($CI->loyalty_points_model)) {
            $CI->load->model('loyalty_points_model');
        }
        
        try {
            $current_balance = $CI->loyalty_points_model->getPatientCurrentBalance($patient_id);
            $min_redemption = $CI->loyalty_points_model->getSettingValue('minimum_redemption_points', 10);
            
            return ($current_balance >= $points_needed && $points_needed >= $min_redemption);
        } catch (Exception $e) {
            log_message('error', 'Failed to check loyalty points eligibility: ' . $e->getMessage());
            return false;
        }
    }
}

if (!function_exists('convert_points_to_amount')) {
    /**
     * Convert loyalty points to KES amount
     * 
     * @param int $points Number of points
     * @return float Amount in KES
     */
    function convert_points_to_amount($points) {
        $CI =& get_instance();
        
        if (!isset($CI->loyalty_points_model)) {
            $CI->load->model('loyalty_points_model');
        }
        
        try {
            $point_value = $CI->loyalty_points_model->getSettingValue('point_value_kes', 50);
            return $points * $point_value;
        } catch (Exception $e) {
            log_message('error', 'Failed to convert points to amount: ' . $e->getMessage());
            return 0;
        }
    }
}

if (!function_exists('get_patient_lab_reports_count')) {
    /**
     * Get count of lab reports for a patient
     * 
     * @param int $patient_id Patient ID
     * @return int Number of lab reports
     */
    function get_patient_lab_reports_count($patient_id) {
        $CI =& get_instance();
        
        if (!isset($CI->lab_reports_model)) {
            $CI->load->model('lab_reports_model');
        }
        
        try {
            return $CI->lab_reports_model->getPatientReportsCount($patient_id);
        } catch (Exception $e) {
            log_message('error', 'Failed to get lab reports count: ' . $e->getMessage());
            return 0;
        }
    }
}

if (!function_exists('get_patient_critical_reports')) {
    /**
     * Check if patient has any critical lab reports
     * 
     * @param int $patient_id Patient ID
     * @return array Critical reports
     */
    function get_patient_critical_reports($patient_id) {
        $CI =& get_instance();
        
        if (!isset($CI->lab_reports_model)) {
            $CI->load->model('lab_reports_model');
        }
        
        try {
            $CI->db->select('*');
            $CI->db->from('patient_lab_reports');
            $CI->db->where('patient_id', $patient_id);
            $CI->db->where('is_critical', 1);
            $CI->db->where('status !=', 'verified');
            $CI->db->order_by('created_at', 'DESC');
            
            $query = $CI->db->get();
            return $query->result_array();
        } catch (Exception $e) {
            log_message('error', 'Failed to get critical reports: ' . $e->getMessage());
            return [];
        }
    }
}

if (!function_exists('format_loyalty_points_display')) {
    /**
     * Format loyalty points for display
     * 
     * @param int $points Points amount
     * @param bool $show_value Whether to show KES value
     * @return string Formatted display string
     */
    function format_loyalty_points_display($points, $show_value = true) {
        $display = number_format($points) . ' point' . ($points != 1 ? 's' : '');
        
        if ($show_value && $points > 0) {
            $value = convert_points_to_amount($points);
            $display .= ' (KES ' . number_format($value, 2) . ')';
        }
        
        return $display;
    }
}

if (!function_exists('get_loyalty_program_status')) {
    /**
     * Check if loyalty program is enabled
     * 
     * @return bool True if enabled
     */
    function get_loyalty_program_status() {
        $CI =& get_instance();
        
        if (!isset($CI->loyalty_points_model)) {
            $CI->load->model('loyalty_points_model');
        }
        
        try {
            return (bool) $CI->loyalty_points_model->getSettingValue('enable_program', 1);
        } catch (Exception $e) {
            log_message('error', 'Failed to get loyalty program status: ' . $e->getMessage());
            return false;
        }
    }
}

if (!function_exists('attach_lab_report_to_consultation')) {
    /**
     * Link lab report to consultation/OPD record
     * 
     * @param int $report_id Lab report ID
     * @param int $consultation_id Consultation ID
     * @param int $opd_id OPD ID
     * @return bool Success status
     */
    function attach_lab_report_to_consultation($report_id, $consultation_id = null, $opd_id = null) {
        $CI =& get_instance();
        
        if (!isset($CI->lab_reports_model)) {
            $CI->load->model('lab_reports_model');
        }
        
        try {
            $update_data = [];
            if ($consultation_id) {
                $update_data['consultation_id'] = $consultation_id;
            }
            if ($opd_id) {
                $update_data['opd_id'] = $opd_id;
            }
            
            if (!empty($update_data)) {
                return $CI->lab_reports_model->updateReport($report_id, $update_data);
            }
            
            return false;
        } catch (Exception $e) {
            log_message('error', 'Failed to attach lab report to consultation: ' . $e->getMessage());
            return false;
        }
    }
}

// Add these functions to your existing payment processing hooks

/**
 * Hook for OPD payment processing
 * Add this to your OPD payment controller after successful payment
 */
if (!function_exists('process_opd_payment_loyalty')) {
    function process_opd_payment_loyalty($patient_id, $amount, $opd_id, $staff_id) {
        return award_loyalty_points_on_payment($patient_id, $amount, 'opd', $opd_id, $staff_id);
    }
}

/**
 * Hook for IPD payment processing
 * Add this to your IPD payment controller after successful payment
 */
if (!function_exists('process_ipd_payment_loyalty')) {
    function process_ipd_payment_loyalty($patient_id, $amount, $ipd_id, $staff_id) {
        return award_loyalty_points_on_payment($patient_id, $amount, 'ipd', $ipd_id, $staff_id);
    }
}

/**
 * Hook for pharmacy payment processing
 * Add this to your pharmacy payment controller after successful payment
 */
if (!function_exists('process_pharmacy_payment_loyalty')) {
    function process_pharmacy_payment_loyalty($patient_id, $amount, $pharmacy_id, $staff_id) {
        return award_loyalty_points_on_payment($patient_id, $amount, 'pharmacy', $pharmacy_id, $staff_id);
    }
}

/**
 * Hook for laboratory payment processing
 * Add this to your laboratory payment controller after successful payment
 */
if (!function_exists('process_laboratory_payment_loyalty')) {
    function process_laboratory_payment_loyalty($patient_id, $amount, $lab_id, $staff_id) {
        return award_loyalty_points_on_payment($patient_id, $amount, 'laboratory', $lab_id, $staff_id);
    }
}

/**
 * Hook for radiology payment processing
 * Add this to your radiology payment controller after successful payment
 */
if (!function_calls('process_radiology_payment_loyalty')) {
    function process_radiology_payment_loyalty($patient_id, $amount, $radiology_id, $staff_id) {
        return award_loyalty_points_on_payment($patient_id, $amount, 'radiology', $radiology_id, $staff_id);
    }
}

/**
 * Example usage in your payment controllers:
 * 
 * // In your OPD payment controller
 * if ($payment_successful) {
 *     // Award loyalty points
 *     process_opd_payment_loyalty($patient_id, $amount, $opd_id, $staff_id);
 * }
 * 
 * // In your consultation view, show patient's loyalty points
 * $loyalty_balance = get_patient_loyalty_balance($patient_id);
 * $lab_reports_count = get_patient_lab_reports_count($patient_id);
 * $critical_reports = get_patient_critical_reports($patient_id);
 * 
 * // In your billing system, allow points redemption
 * if (can_use_loyalty_points($patient_id, $points_to_use)) {
 *     $discount_amount = convert_points_to_amount($points_to_use);
 *     // Apply discount and redeem points
 * }
 */