<?php
if (!defined('BASEPATH'))
    exit('No direct script access allowed');

/**
 * Billing Integration Helper
 * 
 * This helper class handles automatic posting of billing transactions
 * from various modules (OPD, IPD, Pathology, Radiology, Pharmacy, etc.)
 * to the Financial module's General Ledger.
 * 
 * @version 2.0
 * @author Hospital Management System
 * 
 * Usage in billing modules:
 * $this->load->helper('billing_integration');
 * post_billing_to_gl($billing_data);
 */

if (!function_exists('post_billing_to_gl')) {
    /**
     * Post billing transaction to General Ledger
     * 
     * @param array $billing_data Array containing billing information
     * @return bool|int Success status or journal ID
     */
    function post_billing_to_gl($billing_data) {
        $CI =& get_instance();
        
        // Check if integration is enabled
        if (!is_financial_integration_enabled()) {
            log_message('info', 'Billing Integration: Financial integration is disabled');
            return false;
        }
        
        // Load required models
        $CI->load->model('financial_model');
        $CI->load->model('journal_model');
        
        try {
            // Validate billing data
            $validation = validate_billing_data($billing_data);
            if (!$validation['valid']) {
                log_message('error', 'Billing Integration: Validation failed - ' . implode(', ', $validation['errors']));
                return false;
            }
            
            // Check if financial period is open
            if (!get_financial_period_status($billing_data['bill_date'])) {
                log_message('error', 'Billing Integration: Financial period is closed for date ' . $billing_data['bill_date']);
                return false;
            }
            
            // Get account mapping for this module
            $account_mapping = get_billing_account_mapping($billing_data['module'], 'billing');
            if (!$account_mapping) {
                log_message('error', "Billing Integration: No account mapping found for module '{$billing_data['module']}'");
                return false;
            }
            
            // Prepare journal entry data
            $journal_data = [
                'reference_no' => generate_billing_reference($billing_data['module'], $billing_data['bill_id']),
                'date' => $billing_data['bill_date'],
                'description' => generate_billing_description($billing_data),
                'source_module' => $billing_data['module'],
                'source_id' => $billing_data['bill_id'],
                'patient_id' => $billing_data['patient_id'],
                'created_by' => $billing_data['created_by'] ?? $CI->session->userdata('staff_id') ?? 1,
                'currency' => $billing_data['currency'] ?? 'INR',
                'exchange_rate' => $billing_data['exchange_rate'] ?? 1.00
            ];
            
            // Prepare journal lines
            $journal_lines = [];
            
            // Calculate amounts
            $net_amount = floatval($billing_data['amount']);
            $tax_amount = isset($billing_data['tax_amount']) ? floatval($billing_data['tax_amount']) : 0;
            $discount_amount = isset($billing_data['discount_amount']) ? floatval($billing_data['discount_amount']) : 0;
            $gross_amount = $net_amount + $tax_amount - $discount_amount;
            
            // Debit: Accounts Receivable (or Cash if paid immediately)
            if (isset($billing_data['payment_mode']) && $billing_data['payment_mode'] !== 'credit') {
                // Cash payment - debit cash account
                $cash_account = get_cash_account_by_payment_mode($billing_data['payment_mode']);
                $journal_lines[] = [
                    'account_id' => $cash_account,
                    'debit_amount' => $gross_amount,
                    'credit_amount' => 0,
                    'description' => 'Cash payment - ' . $billing_data['payment_mode'],
                    'patient_id' => $billing_data['patient_id']
                ];
            } else {
                // Credit billing - debit accounts receivable
                $ar_account = $account_mapping['debit_account_id'] ?? get_accounts_receivable_account();
                $journal_lines[] = [
                    'account_id' => $ar_account,
                    'debit_amount' => $gross_amount,
                    'credit_amount' => 0,
                    'description' => 'Billing - ' . ($billing_data['patient_name'] ?? 'Patient'),
                    'patient_id' => $billing_data['patient_id']
                ];
            }
            
            // Credit: Revenue Account
            $journal_lines[] = [
                'account_id' => $account_mapping['credit_account_id'],
                'debit_amount' => 0,
                'credit_amount' => $net_amount,
                'description' => ucfirst($billing_data['module']) . ' Revenue',
                'patient_id' => $billing_data['patient_id']
            ];
            
            // Credit: Tax Account (if applicable)
            if ($tax_amount > 0 && isset($account_mapping['tax_account_id']) && $account_mapping['tax_account_id']) {
                $journal_lines[] = [
                    'account_id' => $account_mapping['tax_account_id'],
                    'debit_amount' => 0,
                    'credit_amount' => $tax_amount,
                    'description' => 'Tax on ' . ucfirst($billing_data['module']) . ' billing',
                    'patient_id' => $billing_data['patient_id']
                ];
            }
            
            // Handle discounts if present
            if ($discount_amount > 0) {
                $discount_account = get_discount_account($billing_data['module']);
                if ($discount_account) {
                    $journal_lines[] = [
                        'account_id' => $discount_account,
                        'debit_amount' => $discount_amount,
                        'credit_amount' => 0,
                        'description' => 'Discount given - ' . ($billing_data['discount_reason'] ?? 'General discount'),
                        'patient_id' => $billing_data['patient_id']
                    ];
                }
            }
            
            // Create the journal entry
            $journal_id = $CI->journal_model->createJournalEntry($journal_data, $journal_lines);
            
            if ($journal_id) {
                // Log successful integration
                log_message('info', "Billing Integration: Successfully posted {$billing_data['module']} billing #{$billing_data['bill_id']} to GL as Journal #{$journal_id}");
                
                // Update billing record with journal reference
                update_billing_journal_reference($billing_data['module'], $billing_data['bill_id'], $journal_id);
                
                // Post to analytics if enabled
                post_billing_analytics($billing_data, $journal_id);
                
                return $journal_id;
            } else {
                log_message('error', "Billing Integration: Failed to create journal entry for {$billing_data['module']} billing #{$billing_data['bill_id']}");
                return false;
            }
            
        } catch (Exception $e) {
            log_message('error', "Billing Integration Error: " . $e->getMessage());
            return false;
        }
    }
}

if (!function_exists('post_payment_to_gl')) {
    /**
     * Post payment transaction to General Ledger
     * 
     * @param array $payment_data Array containing payment information
     * @return bool|int Success status or journal ID
     */
    function post_payment_to_gl($payment_data) {
        $CI =& get_instance();
        
        // Check if integration is enabled
        if (!is_financial_integration_enabled()) {
            return false;
        }
        
        // Load required models
        $CI->load->model('financial_model');
        $CI->load->model('journal_model');
        
        try {
            // Validate required fields
            $required_fields = ['payment_id', 'amount', 'payment_mode', 'patient_id', 'payment_date'];
            foreach ($required_fields as $field) {
                if (!isset($payment_data[$field]) || empty($payment_data[$field])) {
                    log_message('error', "Payment Integration: Missing required field '$field'");
                    return false;
                }
            }
            
            // Check if financial period is open
            if (!get_financial_period_status($payment_data['payment_date'])) {
                log_message('error', 'Payment Integration: Financial period is closed for date ' . $payment_data['payment_date']);
                return false;
            }
            
            // Prepare journal entry data
            $journal_data = [
                'reference_no' => 'PAY-' . str_pad($payment_data['payment_id'], 8, '0', STR_PAD_LEFT),
                'date' => $payment_data['payment_date'],
                'description' => 'Payment received - ' . ucfirst($payment_data['payment_mode']),
                'source_module' => 'payment',
                'source_id' => $payment_data['payment_id'],
                'patient_id' => $payment_data['patient_id'],
                'created_by' => $payment_data['created_by'] ?? $CI->session->userdata('staff_id') ?? 1
            ];
            
            // Prepare journal lines
            $journal_lines = [];
            $payment_amount = floatval($payment_data['amount']);
            
            // Debit: Cash/Bank Account
            $cash_account = get_cash_account_by_payment_mode($payment_data['payment_mode']);
            $journal_lines[] = [
                'account_id' => $cash_account,
                'debit_amount' => $payment_amount,
                'credit_amount' => 0,
                'description' => 'Payment received - ' . $payment_data['payment_mode'],
                'patient_id' => $payment_data['patient_id']
            ];
            
            // Credit: Accounts Receivable
            $ar_account = get_accounts_receivable_account();
            $journal_lines[] = [
                'account_id' => $ar_account,
                'debit_amount' => 0,
                'credit_amount' => $payment_amount,
                'description' => 'Payment from ' . ($payment_data['patient_name'] ?? 'Patient'),
                'patient_id' => $payment_data['patient_id']
            ];
            
            // Handle payment charges/fees if applicable
            if (isset($payment_data['processing_fee']) && $payment_data['processing_fee'] > 0) {
                $fee_amount = floatval($payment_data['processing_fee']);
                
                // Debit: Payment Processing Fee Expense
                $fee_expense_account = get_account_id_by_code('5200'); // Payment Processing Fees
                $journal_lines[] = [
                    'account_id' => $fee_expense_account,
                    'debit_amount' => $fee_amount,
                    'credit_amount' => 0,
                    'description' => 'Payment processing fee',
                    'patient_id' => $payment_data['patient_id']
                ];
                
                // Adjust cash account for net amount
                $journal_lines[0]['debit_amount'] = $payment_amount - $fee_amount;
            }
            
            // Create the journal entry
            $journal_id = $CI->journal_model->createJournalEntry($journal_data, $journal_lines);
            
            if ($journal_id) {
                log_message('info', "Payment Integration: Successfully posted payment #{$payment_data['payment_id']} to GL as Journal #{$journal_id}");
                
                // Update payment record with journal reference
                update_payment_journal_reference($payment_data['payment_id'], $journal_id);
                
                return $journal_id;
            } else {
                log_message('error', "Payment Integration: Failed to create journal entry for payment #{$payment_data['payment_id']}");
                return false;
            }
            
        } catch (Exception $e) {
            log_message('error', "Payment Integration Error: " . $e->getMessage());
            return false;
        }
    }
}

if (!function_exists('post_refund_to_gl')) {
    /**
     * Post refund transaction to General Ledger
     * 
     * @param array $refund_data Array containing refund information
     * @return bool|int Success status or journal ID
     */
    function post_refund_to_gl($refund_data) {
        $CI =& get_instance();
        
        // Check if integration is enabled
        if (!is_financial_integration_enabled()) {
            return false;
        }
        
        // Load required models
        $CI->load->model('financial_model');
        $CI->load->model('journal_model');
        
        try {
            // Validate required fields
            $required_fields = ['refund_id', 'amount', 'refund_mode', 'patient_id', 'refund_date', 'module'];
            foreach ($required_fields as $field) {
                if (!isset($refund_data[$field]) || empty($refund_data[$field])) {
                    log_message('error', "Refund Integration: Missing required field '$field'");
                    return false;
                }
            }
            
            // Check if financial period is open
            if (!get_financial_period_status($refund_data['refund_date'])) {
                log_message('error', 'Refund Integration: Financial period is closed for date ' . $refund_data['refund_date']);
                return false;
            }
            
            // Prepare journal entry data
            $journal_data = [
                'reference_no' => 'REF-' . str_pad($refund_data['refund_id'], 8, '0', STR_PAD_LEFT),
                'date' => $refund_data['refund_date'],
                'description' => 'Refund issued - ' . ucfirst($refund_data['refund_mode']),
                'source_module' => 'refund',
                'source_id' => $refund_data['refund_id'],
                'patient_id' => $refund_data['patient_id'],
                'created_by' => $refund_data['created_by'] ?? $CI->session->userdata('staff_id') ?? 1
            ];
            
            // Prepare journal lines
            $journal_lines = [];
            $refund_amount = floatval($refund_data['amount']);
            
            // Get revenue account for the module
            $account_mapping = get_billing_account_mapping($refund_data['module'], 'billing');
            
            if ($account_mapping) {
                // Debit: Revenue Account (reverse the original revenue)
                $journal_lines[] = [
                    'account_id' => $account_mapping['credit_account_id'],
                    'debit_amount' => $refund_amount,
                    'credit_amount' => 0,
                    'description' => 'Refund - ' . ucfirst($refund_data['module']) . ' Revenue reversal',
                    'patient_id' => $refund_data['patient_id']
                ];
            } else {
                // Use general refund expense account
                $refund_expense_account = get_account_id_by_code('5300'); // Refunds Expense
                $journal_lines[] = [
                    'account_id' => $refund_expense_account,
                    'debit_amount' => $refund_amount,
                    'credit_amount' => 0,
                    'description' => 'Refund expense',
                    'patient_id' => $refund_data['patient_id']
                ];
            }
            
            // Credit: Cash/Bank Account
            $cash_account = get_cash_account_by_payment_mode($refund_data['refund_mode']);
            $journal_lines[] = [
                'account_id' => $cash_account,
                'debit_amount' => 0,
                'credit_amount' => $refund_amount,
                'description' => 'Refund paid - ' . $refund_data['refund_mode'],
                'patient_id' => $refund_data['patient_id']
            ];
            
            // Create the journal entry
            $journal_id = $CI->journal_model->createJournalEntry($journal_data, $journal_lines);
            
            if ($journal_id) {
                log_message('info', "Refund Integration: Successfully posted refund #{$refund_data['refund_id']} to GL as Journal #{$journal_id}");
                
                // Update refund record with journal reference
                update_refund_journal_reference($refund_data['refund_id'], $journal_id);
                
                return $journal_id;
            } else {
                log_message('error', "Refund Integration: Failed to create journal entry for refund #{$refund_data['refund_id']}");
                return false;
            }
            
        } catch (Exception $e) {
            log_message('error', "Refund Integration Error: " . $e->getMessage());
            return false;
        }
    }
}

if (!function_exists('post_inventory_transaction_to_gl')) {
    /**
     * Post inventory transaction to General Ledger
     * 
     * @param array $inventory_data Inventory transaction data
     * @return bool|int Success status or journal ID
     */
    function post_inventory_transaction_to_gl($inventory_data) {
        $CI =& get_instance();
        
        // Check if integration is enabled
        if (!is_financial_integration_enabled()) {
            return false;
        }
        
        // Load required models
        $CI->load->model('financial_model');
        $CI->load->model('journal_model');
        
        try {
            // Validate required fields
            $required_fields = ['transaction_type', 'item_id', 'quantity', 'unit_cost', 'total_cost', 'date'];
            foreach ($required_fields as $field) {
                if (!isset($inventory_data[$field])) {
                    log_message('error', "Inventory Integration: Missing required field '$field'");
                    return false;
                }
            }
            
            // Check if financial period is open
            if (!get_financial_period_status($inventory_data['date'])) {
                log_message('error', 'Inventory Integration: Financial period is closed for date ' . $inventory_data['date']);
                return false;
            }
            
            $journal_lines = [];
            $total_cost = floatval($inventory_data['total_cost']);
            
            // Prepare journal entry data
            $journal_data = [
                'reference_no' => 'INV-' . str_pad($inventory_data['transaction_id'], 8, '0', STR_PAD_LEFT),
                'date' => $inventory_data['date'],
                'description' => 'Inventory ' . $inventory_data['transaction_type'] . ' - ' . ($inventory_data['item_name'] ?? 'Item'),
                'source_module' => 'inventory',
                'source_id' => $inventory_data['transaction_id'],
                'created_by' => $inventory_data['created_by'] ?? 1
            ];
            
            if ($inventory_data['transaction_type'] == 'purchase') {
                // Debit: Inventory Account
                $inventory_account = get_inventory_account_by_category($inventory_data['category'] ?? 'general');
                $journal_lines[] = [
                    'account_id' => $inventory_account,
                    'debit_amount' => $total_cost,
                    'credit_amount' => 0,
                    'description' => 'Inventory purchase - ' . ($inventory_data['item_name'] ?? 'Item')
                ];
                
                // Credit: Accounts Payable or Cash
                if (isset($inventory_data['supplier_id']) && $inventory_data['supplier_id']) {
                    $payable_account = get_account_id_by_code('2000'); // Accounts Payable
                    $journal_lines[] = [
                        'account_id' => $payable_account,
                        'debit_amount' => 0,
                        'credit_amount' => $total_cost,
                        'description' => 'Purchase on credit - ' . ($inventory_data['supplier_name'] ?? 'Supplier')
                    ];
                } else {
                    $cash_account = get_account_id_by_code('1000'); // Cash
                    $journal_lines[] = [
                        'account_id' => $cash_account,
                        'debit_amount' => 0,
                        'credit_amount' => $total_cost,
                        'description' => 'Cash purchase'
                    ];
                }
                
            } elseif ($inventory_data['transaction_type'] == 'usage' || $inventory_data['transaction_type'] == 'consumption') {
                // Debit: Medical Supplies Expense
                $expense_account = get_expense_account_by_category($inventory_data['category'] ?? 'medical_supplies');
                $journal_lines[] = [
                    'account_id' => $expense_account,
                    'debit_amount' => $total_cost,
                    'credit_amount' => 0,
                    'description' => 'Inventory consumed - ' . ($inventory_data['department'] ?? 'Department')
                ];
                
                // Credit: Inventory Account
                $inventory_account = get_inventory_account_by_category($inventory_data['category'] ?? 'general');
                $journal_lines[] = [
                    'account_id' => $inventory_account,
                    'debit_amount' => 0,
                    'credit_amount' => $total_cost,
                    'description' => 'Inventory usage - ' . ($inventory_data['item_name'] ?? 'Item')
                ];
                
            } elseif ($inventory_data['transaction_type'] == 'adjustment') {
                $inventory_account = get_inventory_account_by_category($inventory_data['category'] ?? 'general');
                $adjustment_account = get_account_id_by_code('5400'); // Inventory Adjustment
                
                if ($inventory_data['adjustment_type'] == 'increase') {
                    // Debit: Inventory, Credit: Inventory Adjustment (gain)
                    $journal_lines[] = [
                        'account_id' => $inventory_account,
                        'debit_amount' => $total_cost,
                        'credit_amount' => 0,
                        'description' => 'Inventory adjustment - increase'
                    ];
                    $journal_lines[] = [
                        'account_id' => $adjustment_account,
                        'debit_amount' => 0,
                        'credit_amount' => $total_cost,
                        'description' => 'Inventory gain'
                    ];
                } else {
                    // Debit: Inventory Adjustment (loss), Credit: Inventory
                    $journal_lines[] = [
                        'account_id' => $adjustment_account,
                        'debit_amount' => $total_cost,
                        'credit_amount' => 0,
                        'description' => 'Inventory loss'
                    ];
                    $journal_lines[] = [
                        'account_id' => $inventory_account,
                        'debit_amount' => 0,
                        'credit_amount' => $total_cost,
                        'description' => 'Inventory adjustment - decrease'
                    ];
                }
            }
            
            if (!empty($journal_lines)) {
                $journal_id = $CI->journal_model->createJournalEntry($journal_data, $journal_lines);
                
                if ($journal_id) {
                    log_message('info', "Inventory Integration: Successfully posted inventory transaction #{$inventory_data['transaction_id']} to GL as Journal #{$journal_id}");
                    
                    // Update inventory record with journal reference
                    update_inventory_journal_reference($inventory_data['transaction_id'], $journal_id);
                    
                    return $journal_id;
                } else {
                    log_message('error', "Inventory Integration: Failed to create journal entry for transaction #{$inventory_data['transaction_id']}");
                    return false;
                }
            }
            
        } catch (Exception $e) {
            log_message('error', "Inventory Integration Error: " . $e->getMessage());
        }
        
        return false;
    }
}

if (!function_exists('post_expense_to_gl')) {
    /**
     * Post expense transaction to General Ledger
     * 
     * @param array $expense_data Expense data
     * @return bool|int Success status or journal ID
     */
    function post_expense_to_gl($expense_data) {
        $CI =& get_instance();
        
        // Check if integration is enabled
        if (!is_financial_integration_enabled()) {
            return false;
        }
        
        // Load required models
        $CI->load->model('financial_model');
        $CI->load->model('journal_model');
        
        try {
            // Validate required fields
            $required_fields = ['expense_id', 'amount', 'expense_category', 'expense_date'];
            foreach ($required_fields as $field) {
                if (!isset($expense_data[$field]) || empty($expense_data[$field])) {
                    log_message('error', "Expense Integration: Missing required field '$field'");
                    return false;
                }
            }
            
            // Check if financial period is open
            if (!get_financial_period_status($expense_data['expense_date'])) {
                log_message('error', 'Expense Integration: Financial period is closed for date ' . $expense_data['expense_date']);
                return false;
            }
            
            // Prepare journal entry data
            $journal_data = [
                'reference_no' => 'EXP-' . str_pad($expense_data['expense_id'], 8, '0', STR_PAD_LEFT),
                'date' => $expense_data['expense_date'],
                'description' => 'Expense - ' . ($expense_data['description'] ?? $expense_data['expense_category']),
                'source_module' => 'expense',
                'source_id' => $expense_data['expense_id'],
                'created_by' => $expense_data['created_by'] ?? 1
            ];
            
            // Prepare journal lines
            $journal_lines = [];
            $expense_amount = floatval($expense_data['amount']);
            
            // Debit: Expense Account
            $expense_account = get_expense_account_by_category($expense_data['expense_category']);
            $journal_lines[] = [
                'account_id' => $expense_account,
                'debit_amount' => $expense_amount,
                'credit_amount' => 0,
                'description' => $expense_data['description'] ?? 'Expense'
            ];
            
            // Credit: Cash/Bank or Accounts Payable
            if (isset($expense_data['payment_mode']) && $expense_data['payment_mode'] !== 'credit') {
                // Cash payment
                $cash_account = get_cash_account_by_payment_mode($expense_data['payment_mode']);
                $journal_lines[] = [
                    'account_id' => $cash_account,
                    'debit_amount' => 0,
                    'credit_amount' => $expense_amount,
                    'description' => 'Payment - ' . $expense_data['payment_mode']
                ];
            } else {
                // Credit purchase
                $payable_account = get_account_id_by_code('2000'); // Accounts Payable
                $journal_lines[] = [
                    'account_id' => $payable_account,
                    'debit_amount' => 0,
                    'credit_amount' => $expense_amount,
                    'description' => 'Expense on credit'
                ];
            }
            
            // Create the journal entry
            $journal_id = $CI->journal_model->createJournalEntry($journal_data, $journal_lines);
            
            if ($journal_id) {
                log_message('info', "Expense Integration: Successfully posted expense #{$expense_data['expense_id']} to GL as Journal #{$journal_id}");
                
                // Update expense record with journal reference
                update_expense_journal_reference($expense_data['expense_id'], $journal_id);
                
                return $journal_id;
            } else {
                log_message('error', "Expense Integration: Failed to create journal entry for expense #{$expense_data['expense_id']}");
                return false;
            }
            
        } catch (Exception $e) {
            log_message('error', "Expense Integration Error: " . $e->getMessage());
            return false;
        }
    }
}

// =============================================
// ACCOUNT MAPPING FUNCTIONS
// =============================================

if (!function_exists('get_billing_account_mapping')) {
    /**
     * Get account mapping for billing integration
     * 
     * @param string $module Module name
     * @param string $transaction_type Transaction type
     * @return array|false Account mapping or false if not found
     */
    function get_billing_account_mapping($module, $transaction_type) {
        $CI =& get_instance();
        
        $CI->db->select('debit_account_id, credit_account_id, tax_account_id');
        $CI->db->from('billing_account_mapping');
        $CI->db->where('module_name', $module);
        $CI->db->where('transaction_type', $transaction_type);
        $CI->db->where('is_active', 1);
        
        $query = $CI->db->get();
        
        if ($query->num_rows() > 0) {
            return $query->row_array();
        }
        
        // Return default mapping if not found
        return get_default_account_mapping($module, $transaction_type);
    }
}

if (!function_exists('get_default_account_mapping')) {
    /**
     * Get default account mapping for modules
     * 
     * @param string $module Module name
     * @param string $transaction_type Transaction type
     * @return array Default account mapping
     */
    function get_default_account_mapping($module, $transaction_type) {
        $default_mappings = [
            'opd' => [
                'debit_account_id' => get_account_id_by_code('1200'), // Accounts Receivable
                'credit_account_id' => get_account_id_by_code('4100'), // OPD Revenue
                'tax_account_id' => get_account_id_by_code('2100')     // Tax Payable
            ],
            'ipd' => [
                'debit_account_id' => get_account_id_by_code('1200'), // Accounts Receivable
                'credit_account_id' => get_account_id_by_code('4200'), // IPD Revenue
                'tax_account_id' => get_account_id_by_code('2100')     // Tax Payable
            ],
            'pathology' => [
                'debit_account_id' => get_account_id_by_code('1200'), // Accounts Receivable
                'credit_account_id' => get_account_id_by_code('4300'), // Pathology Revenue
                'tax_account_id' => get_account_id_by_code('2100')     // Tax Payable
            ],
            'radiology' => [
                'debit_account_id' => get_account_id_by_code('1200'), // Accounts Receivable
               'credit_account_id' => get_account_id_by_code('4400'), // Radiology Revenue
               'tax_account_id' => get_account_id_by_code('2100')     // Tax Payable
           ],
           'pharmacy' => [
               'debit_account_id' => get_account_id_by_code('1200'), // Accounts Receivable
               'credit_account_id' => get_account_id_by_code('4500'), // Pharmacy Revenue
               'tax_account_id' => get_account_id_by_code('2100')     // Tax Payable
           ],
           'blood_bank' => [
               'debit_account_id' => get_account_id_by_code('1200'), // Accounts Receivable
               'credit_account_id' => get_account_id_by_code('4600'), // Blood Bank Revenue
               'tax_account_id' => get_account_id_by_code('2100')     // Tax Payable
           ],
           'ambulance' => [
               'debit_account_id' => get_account_id_by_code('1200'), // Accounts Receivable
               'credit_account_id' => get_account_id_by_code('4700'), // Ambulance Revenue
               'tax_account_id' => get_account_id_by_code('2100')     // Tax Payable
           ]
       ];
       
       return $default_mappings[$module] ?? [
           'debit_account_id' => get_account_id_by_code('1200'), // Accounts Receivable
           'credit_account_id' => get_account_id_by_code('4000'), // General Revenue
           'tax_account_id' => get_account_id_by_code('2100')     // Tax Payable
       ];
   }
}

if (!function_exists('get_cash_account_by_payment_mode')) {
   /**
    * Get cash account ID based on payment mode
    * 
    * @param string $payment_mode Payment mode
    * @return int Account ID
    */
   function get_cash_account_by_payment_mode($payment_mode) {
       $CI =& get_instance();
       
       // Define payment mode to account mapping
       $payment_accounts = [
           'cash' => '1000',        // Cash
           'cheque' => '1010',      // Bank - Current Account
           'bank' => '1010',        // Bank - Current Account
           'card' => '1010',        // Bank - Current Account
           'credit_card' => '1010', // Bank - Current Account
           'debit_card' => '1010',  // Bank - Current Account
           'online' => '1010',      // Bank - Current Account
           'upi' => '1010',         // Bank - Current Account
           'net_banking' => '1010', // Bank - Current Account
           'wallet' => '1010',      // Bank - Current Account
           'paytm' => '1010',       // Bank - Current Account
           'gpay' => '1010',        // Bank - Current Account
           'phonepe' => '1010'      // Bank - Current Account
       ];
       
       $account_code = isset($payment_accounts[strtolower($payment_mode)]) 
                      ? $payment_accounts[strtolower($payment_mode)] 
                      : '1000'; // Default to cash
       
       return get_account_id_by_code($account_code);
   }
}

if (!function_exists('get_accounts_receivable_account')) {
   /**
    * Get Accounts Receivable account ID
    * 
    * @return int Account ID
    */
   function get_accounts_receivable_account() {
       return get_account_id_by_code('1200');
   }
}

if (!function_exists('get_discount_account')) {
   /**
    * Get discount account for a module
    * 
    * @param string $module Module name
    * @return int|false Account ID or false
    */
   function get_discount_account($module) {
       $discount_accounts = [
           'opd' => '6100',       // OPD Discount
           'ipd' => '6200',       // IPD Discount
           'pathology' => '6300', // Pathology Discount
           'radiology' => '6400', // Radiology Discount
           'pharmacy' => '6500',  // Pharmacy Discount
           'blood_bank' => '6600', // Blood Bank Discount
           'ambulance' => '6700'  // Ambulance Discount
       ];
       
       $account_code = isset($discount_accounts[$module]) 
                      ? $discount_accounts[$module] 
                      : '6800'; // General Discount
       
       return get_account_id_by_code($account_code);
   }
}

if (!function_exists('get_inventory_account_by_category')) {
   /**
    * Get inventory account by category
    * 
    * @param string $category Inventory category
    * @return int Account ID
    */
   function get_inventory_account_by_category($category) {
       $inventory_accounts = [
           'medical_supplies' => '1300', // Medical Supplies Inventory
           'medicines' => '1310',        // Medicines Inventory
           'equipment' => '1320',        // Equipment Inventory
           'consumables' => '1330',      // Consumables Inventory
           'lab_supplies' => '1340',     // Lab Supplies Inventory
           'office_supplies' => '1350',  // Office Supplies Inventory
           'general' => '1300'           // General Inventory
       ];
       
       $account_code = isset($inventory_accounts[strtolower($category)]) 
                      ? $inventory_accounts[strtolower($category)] 
                      : '1300'; // Default to Medical Supplies
       
       return get_account_id_by_code($account_code);
   }
}

if (!function_exists('get_expense_account_by_category')) {
   /**
    * Get expense account by category
    * 
    * @param string $category Expense category
    * @return int Account ID
    */
   function get_expense_account_by_category($category) {
       $expense_accounts = [
           'medical_supplies' => '5100',    // Medical Supplies Expense
           'salary' => '5010',              // Salary Expense
           'utilities' => '5020',           // Utilities Expense
           'rent' => '5030',                // Rent Expense
           'maintenance' => '5040',         // Maintenance Expense
           'marketing' => '5050',           // Marketing Expense
           'professional_fees' => '5060',  // Professional Fees
           'insurance' => '5070',           // Insurance Expense
           'depreciation' => '5080',        // Depreciation Expense
           'office_supplies' => '5090',     // Office Supplies Expense
           'travel' => '5110',              // Travel Expense
           'communication' => '5120',       // Communication Expense
           'training' => '5130',            // Training Expense
           'legal' => '5140',               // Legal Expense
           'audit' => '5150',               // Audit Expense
           'general' => '5000'              // General Expense
       ];
       
       $account_code = isset($expense_accounts[strtolower($category)]) 
                      ? $expense_accounts[strtolower($category)] 
                      : '5000'; // Default to General Expense
       
       return get_account_id_by_code($account_code);
   }
}

if (!function_exists('get_account_id_by_code')) {
   /**
    * Get account ID by account code
    * 
    * @param string $account_code Account code
    * @return int|false Account ID or false
    */
   function get_account_id_by_code($account_code) {
       $CI =& get_instance();
       
       // Check cache first
       static $account_cache = [];
       if (isset($account_cache[$account_code])) {
           return $account_cache[$account_code];
       }
       
       $CI->db->select('id');
       $CI->db->from('chart_of_accounts');
       $CI->db->where('account_code', $account_code);
       $CI->db->where('is_active', 1);
       
       $query = $CI->db->get();
       
       if ($query->num_rows() > 0) {
           $account_id = $query->row()->id;
           $account_cache[$account_code] = $account_id;
           return $account_id;
       }
       
       // Log missing account
       log_message('warning', "Account code '$account_code' not found in chart of accounts");
       return false;
   }
}

// =============================================
// REFERENCE AND DESCRIPTION GENERATORS
// =============================================

if (!function_exists('generate_billing_reference')) {
   /**
    * Generate billing reference number
    * 
    * @param string $module Module name
    * @param int $bill_id Bill ID
    * @return string Reference number
    */
   function generate_billing_reference($module, $bill_id) {
       $prefix_map = [
           'opd' => 'OPD',
           'ipd' => 'IPD',
           'pathology' => 'LAB',
           'radiology' => 'RAD',
           'pharmacy' => 'PHA',
           'blood_bank' => 'BLB',
           'ambulance' => 'AMB',
           'physiotherapy' => 'PHY',
           'dental' => 'DEN',
           'eye' => 'EYE'
       ];
       
       $prefix = isset($prefix_map[$module]) ? $prefix_map[$module] : strtoupper(substr($module, 0, 3));
       
       return $prefix . '-' . str_pad($bill_id, 8, '0', STR_PAD_LEFT);
   }
}

if (!function_exists('generate_billing_description')) {
   /**
    * Generate billing description
    * 
    * @param array $billing_data Billing data
    * @return string Description
    */
   function generate_billing_description($billing_data) {
       $module_names = [
           'opd' => 'OPD Consultation',
           'ipd' => 'IPD Services',
           'pathology' => 'Pathology Tests',
           'radiology' => 'Radiology Services',
           'pharmacy' => 'Pharmacy Sales',
           'blood_bank' => 'Blood Bank Services',
           'ambulance' => 'Ambulance Services',
           'physiotherapy' => 'Physiotherapy Services',
           'dental' => 'Dental Services',
           'eye' => 'Eye Care Services'
       ];
       
       $module_name = isset($module_names[$billing_data['module']]) 
                     ? $module_names[$billing_data['module']] 
                     : ucfirst($billing_data['module']) . ' Services';
       
       $description = $module_name;
       
       if (isset($billing_data['patient_name'])) {
           $description .= ' - ' . $billing_data['patient_name'];
       }
       
       if (isset($billing_data['bill_no'])) {
           $description .= ' (Bill: ' . $billing_data['bill_no'] . ')';
       }
       
       if (isset($billing_data['doctor_name'])) {
           $description .= ' [Dr. ' . $billing_data['doctor_name'] . ']';
       }
       
       return $description;
   }
}

// =============================================
// JOURNAL REFERENCE UPDATE FUNCTIONS
// =============================================

if (!function_exists('update_billing_journal_reference')) {
   /**
    * Update billing record with journal reference
    * 
    * @param string $module Module name
    * @param int $bill_id Bill ID
    * @param int $journal_id Journal ID
    * @return bool Success status
    */
   function update_billing_journal_reference($module, $bill_id, $journal_id) {
       $CI =& get_instance();
       
       // Define table mapping for different modules
       $table_map = [
           'opd' => 'opd_details',
           'ipd' => 'ipd_details',
           'pathology' => 'pathology_billing',
           'radiology' => 'radiology_billing',
           'pharmacy' => 'pharmacy_bill_basic',
           'blood_bank' => 'blood_issue',
           'ambulance' => 'ambulance_call',
           'physiotherapy' => 'physiotherapy_billing',
           'dental' => 'dental_billing',
           'eye' => 'eye_billing'
       ];
       
       if (!isset($table_map[$module])) {
           log_message('warning', "No table mapping found for module: $module");
           return false;
       }
       
       $table = $table_map[$module];
       
       // Check if table exists
       if (!$CI->db->table_exists($table)) {
           log_message('warning', "Table '$table' does not exist");
           return false;
       }
       
       // Update the billing record with journal reference
       $CI->db->where('id', $bill_id);
       $result = $CI->db->update($table, [
           'journal_id' => $journal_id,
           'gl_posted' => 1,
           'gl_posted_date' => date('Y-m-d H:i:s')
       ]);
       
       if ($result) {
           log_message('info', "Updated $module bill #$bill_id with journal reference #$journal_id");
       }
       
       return $result;
   }
}

if (!function_exists('update_payment_journal_reference')) {
   /**
    * Update payment record with journal reference
    * 
    * @param int $payment_id Payment ID
    * @param int $journal_id Journal ID
    * @return bool Success status
    */
   function update_payment_journal_reference($payment_id, $journal_id) {
       $CI =& get_instance();
       
       // Update the payment record
       $CI->db->where('id', $payment_id);
       $result = $CI->db->update('patient_charges', [
           'journal_id' => $journal_id,
           'gl_posted' => 1,
           'gl_posted_date' => date('Y-m-d H:i:s')
       ]);
       
       return $result;
   }
}

if (!function_exists('update_refund_journal_reference')) {
   /**
    * Update refund record with journal reference
    * 
    * @param int $refund_id Refund ID
    * @param int $journal_id Journal ID
    * @return bool Success status
    */
   function update_refund_journal_reference($refund_id, $journal_id) {
       $CI =& get_instance();
       
       // Update the refund record
       $CI->db->where('id', $refund_id);
       $result = $CI->db->update('patient_refunds', [
           'journal_id' => $journal_id,
           'gl_posted' => 1,
           'gl_posted_date' => date('Y-m-d H:i:s')
       ]);
       
       return $result;
   }
}

if (!function_exists('update_inventory_journal_reference')) {
   /**
    * Update inventory record with journal reference
    * 
    * @param int $transaction_id Transaction ID
    * @param int $journal_id Journal ID
    * @return bool Success status
    */
   function update_inventory_journal_reference($transaction_id, $journal_id) {
       $CI =& get_instance();
       
       // Update the inventory transaction record
       $CI->db->where('id', $transaction_id);
       $result = $CI->db->update('item_stock', [
           'journal_id' => $journal_id,
           'gl_posted' => 1,
           'gl_posted_date' => date('Y-m-d H:i:s')
       ]);
       
       return $result;
   }
}

if (!function_exists('update_expense_journal_reference')) {
   /**
    * Update expense record with journal reference
    * 
    * @param int $expense_id Expense ID
    * @param int $journal_id Journal ID
    * @return bool Success status
    */
   function update_expense_journal_reference($expense_id, $journal_id) {
       $CI =& get_instance();
       
       // Update the expense record
       $CI->db->where('id', $expense_id);
       $result = $CI->db->update('expenses', [
           'journal_id' => $journal_id,
           'gl_posted' => 1,
           'gl_posted_date' => date('Y-m-d H:i:s')
       ]);
       
       return $result;
   }
}

// =============================================
// VALIDATION AND UTILITY FUNCTIONS
// =============================================

if (!function_exists('is_financial_integration_enabled')) {
   /**
    * Check if financial integration is enabled
    * 
    * @return bool Integration status
    */
   function is_financial_integration_enabled() {
       $CI =& get_instance();
       
       // Check if financial tables exist
       if (!$CI->db->table_exists('chart_of_accounts') || !$CI->db->table_exists('journal_headers')) {
           return false;
       }
       
       // Check if integration is enabled in settings
       $CI->db->select('setting_value');
       $CI->db->from('app_settings');
       $CI->db->where('setting_key', 'financial_integration_enabled');
       
       $query = $CI->db->get();
       
       if ($query->num_rows() > 0) {
           return $query->row()->setting_value == '1';
       }
       
       // Default to enabled if setting doesn't exist
       return true;
   }
}

if (!function_exists('validate_billing_data')) {
   /**
    * Validate billing data before posting
    * 
    * @param array $billing_data Billing data
    * @return array Validation result
    */
   function validate_billing_data($billing_data) {
       $errors = [];
       
       // Check required fields
       $required_fields = ['module', 'bill_id', 'amount', 'patient_id', 'bill_date'];
       foreach ($required_fields as $field) {
           if (!isset($billing_data[$field]) || (is_string($billing_data[$field]) && trim($billing_data[$field]) === '')) {
               $errors[] = "Missing required field: $field";
           }
       }
       
       // Validate amount
       if (isset($billing_data['amount'])) {
           if (!is_numeric($billing_data['amount']) || floatval($billing_data['amount']) <= 0) {
               $errors[] = "Amount must be a positive number";
           }
       }
       
       // Validate tax amount
       if (isset($billing_data['tax_amount']) && !is_numeric($billing_data['tax_amount'])) {
           $errors[] = "Tax amount must be numeric";
       }
       
       // Validate discount amount
       if (isset($billing_data['discount_amount']) && !is_numeric($billing_data['discount_amount'])) {
           $errors[] = "Discount amount must be numeric";
       }
       
       // Validate date format
       if (isset($billing_data['bill_date'])) {
           $date = DateTime::createFromFormat('Y-m-d', $billing_data['bill_date']);
           if (!$date || $date->format('Y-m-d') !== $billing_data['bill_date']) {
               $errors[] = "Invalid date format. Use Y-m-d format";
           }
       }
       
       // Validate module
       $valid_modules = ['opd', 'ipd', 'pathology', 'radiology', 'pharmacy', 'blood_bank', 'ambulance', 'physiotherapy', 'dental', 'eye'];
       if (isset($billing_data['module']) && !in_array($billing_data['module'], $valid_modules)) {
           $errors[] = "Invalid module: " . $billing_data['module'];
       }
       
       // Validate patient ID
       if (isset($billing_data['patient_id']) && (!is_numeric($billing_data['patient_id']) || intval($billing_data['patient_id']) <= 0)) {
           $errors[] = "Invalid patient ID";
       }
       
       return [
           'valid' => empty($errors),
           'errors' => $errors
       ];
   }
}

if (!function_exists('get_financial_period_status')) {
   /**
    * Check if financial period is open for posting
    * 
    * @param string $date Date to check
    * @return bool Period status
    */
   function get_financial_period_status($date) {
       $CI =& get_instance();
       
       // Check if period is closed
       $CI->db->select('id');
       $CI->db->from('financial_period_closures');
       $CI->db->where('period_start_date <=', $date);
       $CI->db->where('period_end_date >=', $date);
       $CI->db->where('status', 'closed');
       
       $query = $CI->db->get();
       
       // If no closed periods found for this date, period is open
       return $query->num_rows() == 0;
   }
}

if (!function_exists('post_billing_analytics')) {
   /**
    * Post billing data to analytics
    * 
    * @param array $billing_data Billing data
    * @param int $journal_id Journal ID
    * @return bool Success status
    */
   function post_billing_analytics($billing_data, $journal_id) {
       $CI =& get_instance();
       
       try {
           // Check if analytics table exists
           if (!$CI->db->table_exists('billing_analytics')) {
               return false;
           }
           
           $analytics_data = [
               'module' => $billing_data['module'],
               'bill_id' => $billing_data['bill_id'],
               'journal_id' => $journal_id,
               'patient_id' => $billing_data['patient_id'],
               'bill_date' => $billing_data['bill_date'],
               'amount' => $billing_data['amount'],
               'tax_amount' => $billing_data['tax_amount'] ?? 0,
               'discount_amount' => $billing_data['discount_amount'] ?? 0,
               'payment_mode' => $billing_data['payment_mode'] ?? 'credit',
               'doctor_id' => $billing_data['doctor_id'] ?? null,
               'department_id' => $billing_data['department_id'] ?? null,
               'created_at' => date('Y-m-d H:i:s')
           ];
           
           $CI->db->insert('billing_analytics', $analytics_data);
           
           return $CI->db->affected_rows() > 0;
           
       } catch (Exception $e) {
           log_message('error', "Billing Analytics Error: " . $e->getMessage());
           return false;
       }
   }
}

if (!function_exists('reverse_billing_transaction')) {
   /**
    * Reverse a billing transaction
    * 
    * @param string $module Module name
    * @param int $bill_id Bill ID
    * @param string $reason Reason for reversal
    * @return bool|int Success status or journal ID
    */
   function reverse_billing_transaction($module, $bill_id, $reason = 'Transaction reversal') {
       $CI =& get_instance();
       
       // Load required models
       $CI->load->model('financial_model');
       $CI->load->model('journal_model');
       
       try {
           // Get original billing data
           $original_billing = get_original_billing_data($module, $bill_id);
           if (!$original_billing) {
               log_message('error', "Reversal Error: Original billing data not found for $module bill #$bill_id");
               return false;
           }
           
           // Create reversal journal entry
           $journal_data = [
               'reference_no' => 'REV-' . generate_billing_reference($module, $bill_id),
               'date' => date('Y-m-d'),
               'description' => 'Reversal: ' . $reason,
               'source_module' => $module . '_reversal',
               'source_id' => $bill_id,
               'patient_id' => $original_billing['patient_id'],
               'created_by' => $CI->session->userdata('staff_id') ?? 1
           ];
           
           // Get account mapping
           $account_mapping = get_billing_account_mapping($module, 'billing');
           if (!$account_mapping) {
               return false;
           }
           
           // Create reverse journal lines
           $journal_lines = [];
           $amount = floatval($original_billing['amount']);
           
           // Credit: Accounts Receivable (reverse debit)
           $journal_lines[] = [
               'account_id' => $account_mapping['debit_account_id'],
               'debit_amount' => 0,
               'credit_amount' => $amount,
               'description' => 'Reversal - Accounts Receivable',
               'patient_id' => $original_billing['patient_id']
           ];
           
           // Debit: Revenue Account (reverse credit)
           $journal_lines[] = [
               'account_id' => $account_mapping['credit_account_id'],
               'debit_amount' => $amount,
               'credit_amount' => 0,
               'description' => 'Reversal - ' . ucfirst($module) . ' Revenue',
               'patient_id' => $original_billing['patient_id']
           ];
           
           // Create the reversal journal entry
           $journal_id = $CI->journal_model->createJournalEntry($journal_data, $journal_lines);
           
           if ($journal_id) {
               log_message('info', "Successfully reversed $module billing #$bill_id as Journal #$journal_id");
               
               // Update original billing record
               update_billing_reversal_reference($module, $bill_id, $journal_id);
               
               return $journal_id;
           } else {
               log_message('error', "Failed to create reversal journal entry for $module billing #$bill_id");
               return false;
           }
           
       } catch (Exception $e) {
           log_message('error', "Billing Reversal Error: " . $e->getMessage());
           return false;
       }
   }
}

if (!function_exists('get_original_billing_data')) {
   /**
    * Get original billing data for reversal
    * 
    * @param string $module Module name
    * @param int $bill_id Bill ID
    * @return array|false Original billing data or false
    */
   function get_original_billing_data($module, $bill_id) {
       $CI =& get_instance();
       
       $table_map = [
           'opd' => 'opd_details',
           'ipd' => 'ipd_details',
           'pathology' => 'pathology_billing',
           'radiology' => 'radiology_billing',
           'pharmacy' => 'pharmacy_bill_basic',
           'blood_bank' => 'blood_issue',
           'ambulance' => 'ambulance_call'
       ];
       
       if (!isset($table_map[$module])) {
           return false;
       }
       
       $CI->db->select('*');
       $CI->db->from($table_map[$module]);
       $CI->db->where('id', $bill_id);
       
       $query = $CI->db->get();
       
       if ($query->num_rows() > 0) {
           return $query->row_array();
       }
       
       return false;
   }
}

if (!function_exists('update_billing_reversal_reference')) {
   /**
    * Update billing record with reversal reference
    * 
    * @param string $module Module name
    * @param int $bill_id Bill ID
    * @param int $reversal_journal_id Reversal Journal ID
    * @return bool Success status
    */
   function update_billing_reversal_reference($module, $bill_id, $reversal_journal_id) {
       $CI =& get_instance();
       
       $table_map = [
           'opd' => 'opd_details',
           'ipd' => 'ipd_details',
           'pathology' => 'pathology_billing',
           'radiology' => 'radiology_billing',
           'pharmacy' => 'pharmacy_bill_basic',
           'blood_bank' => 'blood_issue',
           'ambulance' => 'ambulance_call'
       ];
       
       if (!isset($table_map[$module])) {
           return false;
       }
       
       $CI->db->where('id', $bill_id);
       $result = $CI->db->update($table_map[$module], [
           'reversal_journal_id' => $reversal_journal_id,
           'is_reversed' => 1,
           'reversed_date' => date('Y-m-d H:i:s')
       ]);
       
       return $result;
   }
}

// =============================================
// BATCH PROCESSING FUNCTIONS
// =============================================

if (!function_exists('batch_post_billing_to_gl')) {
   /**
    * Batch post multiple billing transactions to GL
    * 
    * @param array $billing_transactions Array of billing data
    * @return array Results
    */
   function batch_post_billing_to_gl($billing_transactions) {
       $results = [
           'success_count' => 0,
           'error_count' => 0,
           'errors' => [],
           'success_ids' => []
       ];
       
       foreach ($billing_transactions as $billing_data) {
           $result = post_billing_to_gl($billing_data);
           
           if ($result) {
               $results['success_count']++;
               $results['success_ids'][] = $billing_data['bill_id'];
           } else {
               $results['error_count']++;
               $results['errors'][] = "Failed to post {$billing_data['module']} bill #{$billing_data['bill_id']}";
           }
       }
       
       log_message('info', "Batch billing post completed: {$results['success_count']} success, {$results['error_count']} errors");
       
       return $results;
   }
}

// =============================================
// USAGE EXAMPLES AND DOCUMENTATION
// =============================================

/*

// =============================================
// USAGE EXAMPLES
// =============================================

// Example 1: OPD billing after creating a bill
$billing_data = [
   'module' => 'opd',
   'bill_id' => $opd_bill_id,
   'bill_no' => $opd_bill_no,
   'amount' => $total_amount,
   'tax_amount' => $tax_amount,
   'discount_amount' => $discount_amount,
   'patient_id' => $patient_id,
   'patient_name' => $patient_name,
   'doctor_id' => $doctor_id,
   'doctor_name' => $doctor_name,
   'department_id' => $department_id,
   'bill_date' => $bill_date,
   'payment_mode' => $payment_mode, // 'credit' for billing, 'cash'/'card' for immediate payment
   'created_by' => $this->session->userdata('staff_id'),
   'currency' => 'INR',
   'exchange_rate' => 1.00
];

if (is_financial_integration_enabled()) {
   $journal_id = post_billing_to_gl($billing_data);
   if ($journal_id) {
       echo "Successfully posted to GL with Journal ID: $journal_id";
   }
}

// Example 2: Payment received
$payment_data = [
   'payment_id' => $payment_id,
   'amount' => $payment_amount,
   'payment_mode' => $payment_mode, // 'cash', 'card', 'cheque', 'online', etc.
   'patient_id' => $patient_id,
   'patient_name' => $patient_name,
   'payment_date' => $payment_date,
   'processing_fee' => $processing_fee, // Optional
   'created_by' => $this->session->userdata('staff_id')
];

if (is_financial_integration_enabled()) {
   $journal_id = post_payment_to_gl($payment_data);
   if ($journal_id) {
       echo "Payment posted to GL with Journal ID: $journal_id";
   }
}

// Example 3: Refund processing
$refund_data = [
   'refund_id' => $refund_id,
   'amount' => $refund_amount,
   'refund_mode' => $refund_mode, // 'cash', 'bank', 'cheque', etc.
   'patient_id' => $patient_id,
   'patient_name' => $patient_name,
   'refund_date' => $refund_date,
   'module' => $original_module, // Original billing module
   'reason' => $refund_reason,
   'created_by' => $this->session->userdata('staff_id')
];

if (is_financial_integration_enabled()) {
   $journal_id = post_refund_to_gl($refund_data);
   if ($journal_id) {
       echo "Refund posted to GL with Journal ID: $journal_id";
   }
}

// Example 4: Inventory purchase
$inventory_data = [
   'transaction_id' => $transaction_id,
   'transaction_type' => 'purchase', // 'purchase', 'usage', 'adjustment'
   'item_id' => $item_id,
   'item_name' => $item_name,
   'category' => 'medical_supplies', // 'medical_supplies', 'medicines', 'equipment', etc.
   'quantity' => $quantity,
   'unit_cost' => $unit_cost,
   'total_cost' => $total_cost,
   'supplier_id' => $supplier_id,
   'supplier_name' => $supplier_name,
   'date' => $transaction_date,
   'created_by' => $this->session->userdata('staff_id')
];

if (is_financial_integration_enabled()) {
   $journal_id = post_inventory_transaction_to_gl($inventory_data);
   if ($journal_id) {
       echo "Inventory transaction posted to GL with Journal ID: $journal_id";
   }
}

// Example 5: Expense recording
$expense_data = [
   'expense_id' => $expense_id,
   'amount' => $expense_amount,
   'expense_category' => 'utilities', // 'salary', 'utilities', 'rent', 'maintenance', etc.
   'description' => $expense_description,
   'expense_date' => $expense_date,
   'payment_mode' => $payment_mode, // 'cash', 'cheque', 'credit', etc.
   'vendor_id' => $vendor_id,
   'vendor_name' => $vendor_name,
   'created_by' => $this->session->userdata('staff_id')
];

if (is_financial_integration_enabled()) {
   $journal_id = post_expense_to_gl($expense_data);
   if ($journal_id) {
       echo "Expense posted to GL with Journal ID: $journal_id";
   }
}

// Example 6: Batch processing
$billing_transactions = [
   [
       'module' => 'opd',
       'bill_id' => 1001,
       'amount' => 500.00,
       'patient_id' => 123,
       'bill_date' => '2024-01-15',
       'payment_mode' => 'cash'
   ],
   [
       'module' => 'pathology',
       'bill_id' => 2001,
       'amount' => 750.00,
       'patient_id' => 124,
       'bill_date' => '2024-01-15',
       'payment_mode' => 'credit'
   ]
];

if (is_financial_integration_enabled()) {
   $results = batch_post_billing_to_gl($billing_transactions);
   echo "Batch processing completed: {$results['success_count']} success, {$results['error_count']} errors";
}

// Example 7: Transaction reversal
$reversal_result = reverse_billing_transaction('opd', 1001, 'Patient cancellation');
if ($reversal_result) {
   echo "Transaction reversed with Journal ID: $reversal_result";
}

// =============================================
// INTEGRATION SETUP CHECKLIST
// =============================================

1. Database Tables Required:
  - chart_of_accounts (Account structure)
  - journal_headers (Journal entry headers)
  - journal_lines (Journal entry details)
  - billing_account_mapping (Module to account mapping)
  - financial_period_closures (Period management)
  - billing_analytics (Optional - for analytics)

2. Account Setup:
  - Create chart of accounts with proper account codes
  - Set up account mappings for each module
  - Configure payment mode to account mappings

3. Settings Configuration:
  - Enable financial integration in app_settings
  - Set up financial periods
  - Configure account mappings

4. Module Integration:
  - Add journal_id field to billing tables
  - Add gl_posted flag to billing tables
  - Include helper in billing controllers
  - Call appropriate functions after transactions

5. Testing:
  - Test each module integration
  - Verify journal entries are created correctly
  - Check account balances
  - Test reversal functionality

// =============================================
// DATABASE SCHEMA ADDITIONS
// =============================================

-- Add to existing billing tables
ALTER TABLE opd_details ADD COLUMN journal_id INT NULL;
ALTER TABLE opd_details ADD COLUMN gl_posted TINYINT(1) DEFAULT 0;
ALTER TABLE opd_details ADD COLUMN gl_posted_date DATETIME NULL;
ALTER TABLE opd_details ADD COLUMN reversal_journal_id INT NULL;
ALTER TABLE opd_details ADD COLUMN is_reversed TINYINT(1) DEFAULT 0;
ALTER TABLE opd_details ADD COLUMN reversed_date DATETIME NULL;

-- Repeat for other billing tables (ipd_details, pathology_billing, etc.)

-- Create billing account mapping table
CREATE TABLE billing_account_mapping (
   id INT PRIMARY KEY AUTO_INCREMENT,
   module_name VARCHAR(50) NOT NULL,
   transaction_type VARCHAR(50) NOT NULL,
   debit_account_id INT NOT NULL,
   credit_account_id INT NOT NULL,
   tax_account_id INT NULL,
   is_active TINYINT(1) DEFAULT 1,
   created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
   updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
   UNIQUE KEY unique_mapping (module_name, transaction_type),
   FOREIGN KEY (debit_account_id) REFERENCES chart_of_accounts(id),
   FOREIGN KEY (credit_account_id) REFERENCES chart_of_accounts(id),
   FOREIGN KEY (tax_account_id) REFERENCES chart_of_accounts(id)
);

-- Insert default mappings
INSERT INTO billing_account_mapping (module_name, transaction_type, debit_account_id, credit_account_id, tax_account_id) VALUES
('opd', 'billing', 1200, 4100, 2100),
('ipd', 'billing', 1200, 4200, 2100),
('pathology', 'billing', 1200, 4300, 2100),
('radiology', 'billing', 1200, 4400, 2100),
('pharmacy', 'billing', 1200, 4500, 2100);

-- Create financial period closures table
CREATE TABLE financial_period_closures (
   id INT PRIMARY KEY AUTO_INCREMENT,
   period_name VARCHAR(100) NOT NULL,
   period_start_date DATE NOT NULL,
   period_end_date DATE NOT NULL,
   status ENUM('open', 'closed') DEFAULT 'open',
   closed_by INT NULL,
   closed_date DATETIME NULL,
   created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
   FOREIGN KEY (closed_by) REFERENCES staff(id)
);

-- Create billing analytics table (optional)
CREATE TABLE billing_analytics (
   id INT PRIMARY KEY AUTO_INCREMENT,
   module VARCHAR(50) NOT NULL,
   bill_id INT NOT NULL,
   journal_id INT NOT NULL,
   patient_id INT NOT NULL,
   bill_date DATE NOT NULL,
   amount DECIMAL(10,2) NOT NULL,
   tax_amount DECIMAL(10,2) DEFAULT 0,
   discount_amount DECIMAL(10,2) DEFAULT 0,
   payment_mode VARCHAR(50) NULL,
   doctor_id INT NULL,
   department_id INT NULL,
   created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
   INDEX idx_module_date (module, bill_date),
   INDEX idx_patient (patient_id),
   INDEX idx_doctor (doctor_id),
   FOREIGN KEY (journal_id) REFERENCES journal_headers(id),
   FOREIGN KEY (patient_id) REFERENCES patients(id)
);

*/

// =============================================
// ERROR HANDLING AND LOGGING
// =============================================

if (!function_exists('log_integration_error')) {
   /**
    * Log integration specific errors
    * 
    * @param string $module Module name
    * @param string $operation Operation type
    * @param string $error_message Error message
    * @param array $data Additional data
    */
   function log_integration_error($module, $operation, $error_message, $data = []) {
       $log_data = [
           'timestamp' => date('Y-m-d H:i:s'),
           'module' => $module,
           'operation' => $operation,
           'error' => $error_message,
           'data' => json_encode($data)
       ];
       
       log_message('error', "Financial Integration Error: " . json_encode($log_data));
       
       // Optionally store in database for better tracking
       $CI =& get_instance();
       if ($CI->db->table_exists('integration_error_log')) {
           $CI->db->insert('integration_error_log', $log_data);
       }
   }
}

if (!function_exists('get_integration_statistics')) {
   /**
    * Get integration statistics
    * 
    * @param string $date_from Start date
    * @param string $date_to End date
    * @return array Statistics
    */
   function get_integration_statistics($date_from = null, $date_to = null) {
       $CI =& get_instance();
       
       if (!$date_from) $date_from = date('Y-m-01');
       if (!$date_to) $date_to = date('Y-m-d');
       
       $stats = [
           'total_transactions' => 0,
           'total_amount' => 0,
           'by_module' => [],
           'by_payment_mode' => [],
           'errors' => 0
       ];
       
       // Get billing analytics if table exists
       if ($CI->db->table_exists('billing_analytics')) {
           $CI->db->select('module, COUNT(*) as count, SUM(amount) as total, payment_mode');
           $CI->db->from('billing_analytics');
           $CI->db->where('bill_date >=', $date_from);
           $CI->db->where('bill_date <=', $date_to);
           $CI->db->group_by('module, payment_mode');
           
           $query = $CI->db->get();
           
           foreach ($query->result_array() as $row) {
               $stats['total_transactions'] += $row['count'];
               $stats['total_amount'] += $row['total'];
               
               if (!isset($stats['by_module'][$row['module']])) {
                   $stats['by_module'][$row['module']] = ['count' => 0, 'amount' => 0];
               }
               $stats['by_module'][$row['module']]['count'] += $row['count'];
               $stats['by_module'][$row['module']]['amount'] += $row['total'];
               
               if (!isset($stats['by_payment_mode'][$row['payment_mode']])) {
                   $stats['by_payment_mode'][$row['payment_mode']] = ['count' => 0, 'amount' => 0];
               }
               $stats['by_payment_mode'][$row['payment_mode']]['count'] += $row['count'];
               $stats['by_payment_mode'][$row['payment_mode']]['amount'] += $row['total'];
           }
       }
       
       return $stats;
   }
}

if (!function_exists('validate_account_mappings')) {
   /**
    * Validate that all required account mappings exist
    * 
    * @return array Validation results
    */
   function validate_account_mappings() {
       $CI =& get_instance();
       
       $required_modules = ['opd', 'ipd', 'pathology', 'radiology', 'pharmacy', 'blood_bank', 'ambulance'];
       $missing_mappings = [];
       $invalid_accounts = [];
       
       foreach ($required_modules as $module) {
           $mapping = get_billing_account_mapping($module, 'billing');
           
           if (!$mapping) {
               $missing_mappings[] = $module;
           } else {
               // Validate that accounts exist
               foreach (['debit_account_id', 'credit_account_id'] as $account_field) {
                   if (isset($mapping[$account_field])) {
                       $CI->db->select('id');
                       $CI->db->from('chart_of_accounts');
                       $CI->db->where('id', $mapping[$account_field]);
                       $CI->db->where('is_active', 1);
                       
                       if ($CI->db->get()->num_rows() == 0) {
                           $invalid_accounts[] = "$module.$account_field ({$mapping[$account_field]})";
                       }
                   }
               }
           }
       }
       
       return [
           'valid' => empty($missing_mappings) && empty($invalid_accounts),
           'missing_mappings' => $missing_mappings,
           'invalid_accounts' => $invalid_accounts
       ];
   }
}

// =============================================
// HELPER FUNCTION FOR TESTING
// =============================================

if (!function_exists('test_financial_integration')) {
   /**
    * Test financial integration setup
    * 
    * @return array Test results
    */
   function test_financial_integration() {
       $tests = [
           'integration_enabled' => is_financial_integration_enabled(),
           'tables_exist' => [],
           'account_mappings' => validate_account_mappings(),
           'sample_accounts' => []
       ];
       
       $CI =& get_instance();
       
       // Test required tables
       $required_tables = ['chart_of_accounts', 'journal_headers', 'journal_lines'];
       foreach ($required_tables as $table) {
           $tests['tables_exist'][$table] = $CI->db->table_exists($table);
       }
       
       // Test sample accounts
       $sample_accounts = ['1000', '1200', '4100', '2100'];
       foreach ($sample_accounts as $code) {
           $account_id = get_account_id_by_code($code);
           $tests['sample_accounts'][$code] = $account_id !== false;
       }
       
       return $tests;
   }
}

// =============================================
// FINAL NOTES
// =============================================

/*
This billing integration helper provides comprehensive functionality for:

1. ✅ Automatic posting of billing transactions to General Ledger
2. ✅ Payment processing integration
3. ✅ Refund handling
4. ✅ Inventory transaction posting
5. ✅ Expense recording
6. ✅ Transaction reversal capability
7. ✅ Batch processing support
8. ✅ Comprehensive validation
9. ✅ Error handling and logging
10. ✅ Analytics and reporting
11. ✅ Account mapping management
12. ✅ Financial period controls
13. ✅ Multi-currency support (basic)
14. ✅ Testing and validation tools

To implement:
1. Include this helper in your billing modules
2. Set up the required database tables
3. Configure account mappings
4. Call the appropriate functions after transactions
5. Test thoroughly before going live

For support and customization, refer to the documentation and examples above.
*/
?>