<?php
defined('BASEPATH') OR exit('No direct script access allowed');

class Trial_balance extends Admin_Controller {

    public function __construct() {
        parent::__construct();
        $this->load->model('financial_model');
        $this->load->library('form_validation');
        $this->load->helper('form');
        
        // Check if user has permission to access this module
        if (!$this->rbac->hasPrivilege('trial_balance', 'can_view')) {
            access_denied();
        }
    }
    
    /**
     * Display the trial balance report
     */
    public function index() {
        $this->session->set_userdata('top_menu', 'Finance');
        $this->session->set_userdata('sub_menu', 'Finance/trial_balance');
        
        // Check if financial system is set up
        if (!$this->financial_model->isSystemSetup()) {
            // Try to create the tables
            if (!$this->financial_model->createFinancialTables()) {
                $this->session->set_flashdata('error', 'Financial system is not properly set up. Please contact your system administrator.');
                redirect('admin/dashboard');
            }
        }
        
        // Get the as_of_date from query string or use current date
        $as_of_date = $this->input->get('as_of_date') ?: date('Y-m-d');
        
        // Get trial balance data using the corrected method
        $data = $this->getTrialBalanceData($as_of_date);
        
        // Prepare view data
        $view_data = [
            'title' => 'Trial Balance',
            'as_of_date' => $as_of_date,
            'accounts' => $data['accounts'],
            'account_types' => $data['account_types'],
            'currency_symbol' => $this->customlib->getHospitalCurrencyFormat(),
            'is_balanced' => $data['is_balanced'],
            'total_debit' => $data['total_debit'],
            'total_credit' => $data['total_credit'],
            'difference' => $data['difference']
        ];
        
        // Load the view
        $this->load->view('layout/header', $view_data);
        $this->load->view('admin/financial/trialbalance', $view_data);
        $this->load->view('layout/footer', $view_data);
    }

    /**
     * Get trial balance data with corrected SQL syntax
     */
    private function getTrialBalanceData($as_of_date) {
        try {
            // Use the existing trial_balance_view but filter by date
            $sql = "
                SELECT 
                    coa.account_code,
                    coa.account_name,
                    coa.account_type,
                    CASE 
                        WHEN coa.account_type IN ('asset', 'expense') THEN 
                            GREATEST((coa.opening_balance + COALESCE(SUM(jl.debit_amount - jl.credit_amount), 0)), 0)
                        ELSE 0 
                    END as debit_balance,
                    CASE 
                        WHEN coa.account_type IN ('liability', 'equity', 'revenue') THEN 
                            GREATEST((coa.opening_balance + COALESCE(SUM(jl.credit_amount - jl.debit_amount), 0)), 0)
                        WHEN coa.account_type IN ('asset', 'expense') AND 
                             (coa.opening_balance + COALESCE(SUM(jl.debit_amount - jl.credit_amount), 0)) < 0 THEN 
                            ABS((coa.opening_balance + COALESCE(SUM(jl.debit_amount - jl.credit_amount), 0)))
                        ELSE 0 
                    END as credit_balance
                FROM chart_of_accounts coa
                LEFT JOIN journal_lines jl ON jl.account_id = coa.id
                LEFT JOIN journal_headers jh ON jh.id = jl.journal_id 
                    AND jh.status = 'posted' 
                    AND jh.date <= ?
                WHERE coa.is_active = 1
                GROUP BY coa.id, coa.account_code, coa.account_name, coa.account_type, coa.opening_balance
                HAVING debit_balance > 0 OR credit_balance > 0
                ORDER BY coa.account_code ASC
            ";

            $query = $this->db->query($sql, array($as_of_date));
            $accounts = $query->result_array();

            // Initialize account types
            $account_types = [
                'asset' => ['name' => 'Assets', 'accounts' => [], 'debit' => 0, 'credit' => 0],
                'liability' => ['name' => 'Liabilities', 'accounts' => [], 'debit' => 0, 'credit' => 0],
                'equity' => ['name' => 'Equity', 'accounts' => [], 'debit' => 0, 'credit' => 0],
                'revenue' => ['name' => 'Revenue', 'accounts' => [], 'debit' => 0, 'credit' => 0],
                'expense' => ['name' => 'Expenses', 'accounts' => [], 'debit' => 0, 'credit' => 0]
            ];

            // Group accounts by type and calculate totals
            $total_debit = 0;
            $total_credit = 0;

            foreach ($accounts as $account) {
                $type = $account['account_type'];
                if (isset($account_types[$type])) {
                    $account_types[$type]['accounts'][] = $account;
                    $account_types[$type]['debit'] += floatval($account['debit_balance']);
                    $account_types[$type]['credit'] += floatval($account['credit_balance']);
                }
                
                $total_debit += floatval($account['debit_balance']);
                $total_credit += floatval($account['credit_balance']);
            }

            // Check if balanced (allow for small rounding differences)
            $difference = abs($total_debit - $total_credit);
            $is_balanced = $difference < 0.01;

            return [
                'accounts' => $accounts,
                'account_types' => $account_types,
                'total_debit' => $total_debit,
                'total_credit' => $total_credit,
                'difference' => $difference,
                'is_balanced' => $is_balanced
            ];

        } catch (Exception $e) {
            log_message('error', 'Trial Balance Error: ' . $e->getMessage());
            
            // Return empty data structure on error
            return [
                'accounts' => [],
                'account_types' => [
                    'asset' => ['name' => 'Assets', 'accounts' => [], 'debit' => 0, 'credit' => 0],
                    'liability' => ['name' => 'Liabilities', 'accounts' => [], 'debit' => 0, 'credit' => 0],
                    'equity' => ['name' => 'Equity', 'accounts' => [], 'debit' => 0, 'credit' => 0],
                    'revenue' => ['name' => 'Revenue', 'accounts' => [], 'debit' => 0, 'credit' => 0],
                    'expense' => ['name' => 'Expenses', 'accounts' => [], 'debit' => 0, 'credit' => 0]
                ],
                'total_debit' => 0,
                'total_credit' => 0,
                'difference' => 0,
                'is_balanced' => true
            ];
        }
    }
    
    /**
     * Export trial balance to Excel
     */
    public function export() {
        // Check if user has permission to export
        if (!$this->rbac->hasPrivilege('trial_balance', 'can_export')) {
            access_denied();
        }
        
        // Get the as_of_date from query string or use current date
        $as_of_date = $this->input->get('as_of_date') ?: date('Y-m-d');
        
        // Get trial balance data
        $data = $this->getTrialBalanceData($as_of_date);
        
        // Load PHPExcel library
        $this->load->library('PHPExcel');
        
        // Create new PHPExcel object
        $objPHPExcel = new PHPExcel();
        
        // Set document properties
        $objPHPExcel->getProperties()
                   ->setCreator("Hospital Management System")
                   ->setLastModifiedBy("Hospital Management System")
                   ->setTitle("Trial Balance - " . date('Y-m-d'))
                   ->setSubject("Trial Balance");
        
        // Add a worksheet
        $objPHPExcel->setActiveSheetIndex(0);
        $sheet = $objPHPExcel->getActiveSheet();
        
        // Set title
        $sheet->setCellValue('A1', 'Trial Balance');
        $sheet->mergeCells('A1:D1');
        $sheet->getStyle('A1')->getFont()->setBold(true)->setSize(16);
        $sheet->getStyle('A1')->getAlignment()->setHorizontal(PHPExcel_Style_Alignment::HORIZONTAL_CENTER);
        
        // Set as of date
        $sheet->setCellValue('A2', 'As of ' . date('F j, Y', strtotime($as_of_date)));
        $sheet->mergeCells('A2:D2');
        $sheet->getStyle('A2')->getAlignment()->setHorizontal(PHPExcel_Style_Alignment::HORIZONTAL_CENTER);
        
        // Add headers
        $sheet->setCellValue('A4', 'Account Code')
              ->setCellValue('B4', 'Account Name')
              ->setCellValue('C4', 'Debit (' . $this->customlib->getHospitalCurrencyFormat() . ')')
              ->setCellValue('D4', 'Credit (' . $this->customlib->getHospitalCurrencyFormat() . ')');
        
        // Style headers
        $header_style = [
            'font' => ['bold' => true],
            'alignment' => ['horizontal' => PHPExcel_Style_Alignment::HORIZONTAL_CENTER],
            'borders' => ['allborders' => ['style' => PHPExcel_Style_Border::BORDER_THIN]],
            'fill' => ['type' => PHPExcel_Style_Fill::FILL_SOLID, 'color' => ['rgb' => 'EEEEEE']]
        ];
        $sheet->getStyle('A4:D4')->applyFromArray($header_style);
        
        // Add data
        $row = 5;
        $account_types = ['asset', 'liability', 'equity', 'revenue', 'expense'];
        
        foreach ($account_types as $type) {
            if (empty($data['account_types'][$type]['accounts'])) continue;
            
            $type_name = $data['account_types'][$type]['name'];
            $sheet->setCellValue("A$row", $type_name);
            $sheet->getStyle("A$row:D$row")->getFont()->setBold(true);
            $row++;
            
            // Accounts for this type
            foreach ($data['account_types'][$type]['accounts'] as $account) {
                $sheet->setCellValue("A$row", $account['account_code']);
                $sheet->setCellValue("B$row", $account['account_name']);
                $sheet->setCellValue("C$row", $account['debit_balance'] > 0 ? $account['debit_balance'] : '');
                $sheet->setCellValue("D$row", $account['credit_balance'] > 0 ? $account['credit_balance'] : '');
                $row++;
            }
            
            // Type totals
            $sheet->setCellValue("A$row", 'Total ' . $type_name);
            $sheet->setCellValue("C$row", $data['account_types'][$type]['debit']);
            $sheet->setCellValue("D$row", $data['account_types'][$type]['credit']);
            $sheet->getStyle("A$row:D$row")->getFont()->setBold(true);
            $sheet->getStyle("C$row:D$row")->getNumberFormat()->setFormatCode('#,##0.00');
            $row += 2;
        }
        
        // Grand totals
        $sheet->setCellValue("B$row", 'TOTAL');
        $sheet->setCellValue("C$row", $data['total_debit']);
        $sheet->setCellValue("D$row", $data['total_credit']);
        $sheet->getStyle("B$row:D$row")->getFont()->setBold(true);
        $sheet->getStyle("C$row:D$row")->getNumberFormat()->setFormatCode('#,##0.00');
        
        // Add a note if not balanced
        if (!$data['is_balanced']) {
            $row += 2;
            $sheet->setCellValue("A$row", 'WARNING: Trial Balance is not balanced!');
            $sheet->mergeCells("A$row:D$row");
            $sheet->getStyle("A$row")->getFont()->setBold(true)->setColor(new PHPExcel_Style_Color('FFFF0000'));
            $sheet->getStyle("A$row")->getAlignment()->setHorizontal(PHPExcel_Style_Alignment::HORIZONTAL_CENTER);
            
            $row++;
            $sheet->setCellValue("A$row", 'Difference: ' . $this->customlib->getHospitalCurrencyFormat() . ' ' . number_format($data['difference'], 2));
            $sheet->mergeCells("A$row:D$row");
            $sheet->getStyle("A$row")->getFont()->setBold(true);
            $sheet->getStyle("A$row")->getAlignment()->setHorizontal(PHPExcel_Style_Alignment::HORIZONTAL_CENTER);
        }
        
        // Auto-size columns
        foreach (range('A', 'D') as $col) {
            $sheet->getColumnDimension($col)->setAutoSize(true);
        }
        
        // Set number format for amounts
        $sheet->getStyle('C5:D' . ($row - 1))->getNumberFormat()->setFormatCode('#,##0.00');
        
        // Set borders
        $sheet->getStyle('A4:D' . ($row - 1))->getBorders()->getAllBorders()->setBorderStyle(PHPExcel_Style_Border::BORDER_THIN);
        
        // Set the file name and headers
        $filename = 'Trial_Balance_' . date('Y-m-d') . '.xls';
        
        // Redirect output to a client's web browser (Excel5)
        header('Content-Type: application/vnd.ms-excel');
        header('Content-Disposition: attachment;filename="' . $filename . '"');
        header('Cache-Control: max-age=0');
        
        $objWriter = PHPExcel_IOFactory::createWriter($objPHPExcel, 'Excel5');
        $objWriter->save('php://output');
        exit;
    }

    /**
     * Alternative method using your existing trial_balance_view
     */
    public function getTrialBalanceFromView($as_of_date) {
        try {
            // Since your view already handles the complex logic, we can use it
            // But we need to add date filtering manually
            $sql = "
                SELECT 
                    tbv.account_code,
                    tbv.account_name,
                    tbv.account_type,
                    tbv.debit_balance,
                    tbv.credit_balance
                FROM trial_balance_view tbv
                WHERE EXISTS (
                    SELECT 1 FROM chart_of_accounts coa 
                    WHERE coa.account_code = tbv.account_code 
                    AND coa.is_active = 1
                    AND NOT EXISTS (
                        SELECT 1 FROM journal_lines jl2
                        JOIN journal_headers jh2 ON jh2.id = jl2.journal_id
                        WHERE jl2.account_id = coa.id 
                        AND jh2.date > ? 
                        AND jh2.status = 'posted'
                    )
                )
                ORDER BY tbv.account_code ASC
            ";

            $query = $this->db->query($sql, array($as_of_date));
            return $query->result_array();

        } catch (Exception $e) {
            log_message('error', 'Trial Balance View Error: ' . $e->getMessage());
            return [];
        }
    }
}