<?php

if (!defined('BASEPATH'))
    exit('No direct script access allowed');

class Radiology_setup extends Admin_Controller {

    function __construct() {
        parent::__construct();
        
        // Load essential libraries and models
        try {
            $this->load->model('patient_model');
            $this->load->model('staff_model');
            $this->load->library('datatables');
            $this->load->library('customlib');
            $this->load->library('form_validation');
        } catch (Exception $e) {
            log_message('error', 'Radiology_setup controller error: ' . $e->getMessage());
            show_error('Failed to load required libraries: ' . $e->getMessage());
        }
    }

    /**
     * Check RBAC permissions with fallback
     */
    private function checkPermission($action) {
        try {
            if (method_exists($this->rbac, 'hasPrivilege')) {
                return $this->rbac->hasPrivilege('radiology_setup', $action);
            }
        } catch (Exception $e) {
            log_message('error', 'RBAC check failed: ' . $e->getMessage());
        }
        
        // Fallback: allow access if RBAC is not configured
        return true;
    }

    /**
     * Main index page for radiology test setup
     */
    public function index() {
        $this->session->set_userdata('top_menu', 'Setup');
        $this->session->set_userdata('sub_menu', 'radiology_setup');

        $data['title'] = 'Radiology Test Setup';
        
        try {
            if (!$this->checkPermission('can_view')) {
                show_error('Access denied. Please contact administrator to setup radiology_setup permissions.');
                return;
            }

            $data['categories'] = $this->getRadiologyCategories();
            
            $this->load->view('layout/header', $data);
            $this->load->view('admin/radiology/setup/index', $data);
            $this->load->view('layout/footer', $data);
            
        } catch (Exception $e) {
            log_message('error', 'Radiology setup view loading error: ' . $e->getMessage());
            show_error('Failed to load radiology setup view: ' . $e->getMessage());
        }
    }

    /**
     * Get radiology tests data for DataTables AJAX
     */
    public function get_radiology_tests_datatable() {
        header('Content-Type: application/json');
        
        if (!$this->checkPermission('can_view')) {
            echo json_encode(['error' => 'Access denied']);
            return;
        }

        try {
            $draw = intval($this->input->post('draw'));
            $start = intval($this->input->post('start'));
            $length = intval($this->input->post('length'));
            $search_value = $this->input->post('search')['value'];
            
            $this->db->select('id, test_name, test_code, category, estimated_duration, price, requires_contrast, requires_fasting, is_active, is_gcc');
            $this->db->from('radiology_tests');
            
            if (!empty($search_value)) {
                $this->db->group_start();
                $this->db->like('test_name', $search_value);
                $this->db->or_like('test_code', $search_value);
                $this->db->or_like('category', $search_value);
                $this->db->group_end();
            }
            
            $total_query = clone $this->db;
            $total_records = $total_query->count_all_results();
            
            $order_column = $this->input->post('order')[0]['column'];
            $order_dir = $this->input->post('order')[0]['dir'];
            $columns = ['test_name', 'test_code', 'category', 'estimated_duration', 'price', 'requires_contrast', 'requires_fasting', 'is_active', 'is_gcc'];
            
            if (isset($columns[$order_column])) {
                $this->db->order_by($columns[$order_column], $order_dir);
            } else {
                $this->db->order_by('test_name', 'asc');
            }
            
            $this->db->limit($length, $start);
            $query = $this->db->get();
            $tests = $query->result_array();
            
            $data = [];
            foreach ($tests as $test) {
                $price_formatted = 'Ksh. ' . number_format($test['price'], 2);
                
                $contrast_badge = $test['requires_contrast'] == 1 ? 
                    '<span class="label label-warning">Yes</span>' : 
                    '<span class="label label-default">No</span>';
                
                $fasting_badge = $test['requires_fasting'] == 1 ? 
                    '<span class="label label-info">Yes</span>' : 
                    '<span class="label label-default">No</span>';
                
                $gcc_badge = $test['is_gcc'] == 1 ? 
                    '<span class="label label-success"><i class="fa fa-star"></i> GCC</span>' : 
                    '<span class="label label-default">Regular</span>';
                
                $status_badge = $test['is_active'] == 1 ? 
                    '<span class="label label-success">Active</span>' : 
                    '<span class="label label-danger">Inactive</span>';
                
                $actions = '';
                if ($this->checkPermission('can_view')) {
                    $actions .= '<button class="btn btn-default btn-xs" onclick="viewRadiologyTest(' . $test['id'] . ')" title="View"><i class="fa fa-eye"></i></button> ';
                }
                if ($this->checkPermission('can_edit')) {
                    $actions .= '<button class="btn btn-primary btn-xs" onclick="editRadiologyTest(' . $test['id'] . ')" title="Edit"><i class="fa fa-edit"></i></button> ';
                    $toggle_class = $test['is_active'] == 1 ? 'warning' : 'success';
                    $toggle_icon = $test['is_active'] == 1 ? 'pause' : 'play';
                    $actions .= '<button class="btn btn-' . $toggle_class . ' btn-xs" onclick="toggleTestStatus(' . $test['id'] . ', ' . $test['is_active'] . ')" title="Toggle Status"><i class="fa fa-' . $toggle_icon . '"></i></button> ';
                    
                    $gcc_toggle_class = $test['is_gcc'] == 1 ? 'success' : 'default';
                    $gcc_toggle_icon = $test['is_gcc'] == 1 ? 'star' : 'star-o';
                    $actions .= '<button class="btn btn-' . $gcc_toggle_class . ' btn-xs" onclick="toggleGCCStatus(' . $test['id'] . ', ' . $test['is_gcc'] . ')" title="Toggle GCC"><i class="fa fa-' . $gcc_toggle_icon . '"></i></button> ';
                }
                if ($this->checkPermission('can_delete')) {
                    $actions .= '<button class="btn btn-danger btn-xs" onclick="deleteRadiologyTest(' . $test['id'] . ')" title="Delete"><i class="fa fa-trash"></i></button>';
                }
                
                $data[] = [
                    $test['test_name'],
                    $test['test_code'],
                    $test['category'],
                    $test['estimated_duration'],
                    $price_formatted,
                    $contrast_badge,
                    $fasting_badge,
                    $gcc_badge,
                    $status_badge,
                    $actions
                ];
            }
            
            echo json_encode([
                'draw' => $draw,
                'recordsTotal' => $total_records,
                'recordsFiltered' => $total_records,
                'data' => $data
            ]);
            
        } catch (Exception $e) {
            log_message('error', 'DataTables error: ' . $e->getMessage());
            echo json_encode([
                'draw' => 0,
                'recordsTotal' => 0,
                'recordsFiltered' => 0,
                'data' => [],
                'error' => 'Failed to load data: ' . $e->getMessage()
            ]);
        }
    }

    /**
     * Add new radiology test
     */
    public function add_radiology_test() {
        header('Content-Type: application/json');
        
        if (!$this->checkPermission('can_add')) {
            echo json_encode(['status' => 'error', 'message' => 'Access denied']);
            return;
        }

        try {
            $this->form_validation->set_rules('test_name', 'Test Name', 'required|trim|max_length[200]');
            $this->form_validation->set_rules('test_code', 'Test Code', 'required|trim|max_length[50]|is_unique[radiology_tests.test_code]');
            $this->form_validation->set_rules('category', 'Category', 'required|trim|max_length[100]');
            $this->form_validation->set_rules('price', 'Price', 'required|numeric|greater_than_equal_to[0]');
            $this->form_validation->set_rules('estimated_duration', 'Duration', 'integer|greater_than[0]|less_than_equal_to[180]');
            $this->form_validation->set_rules('requires_contrast', 'Contrast', 'integer|in_list[0,1]');
            $this->form_validation->set_rules('requires_fasting', 'Fasting', 'integer|in_list[0,1]');
            
            if ($this->form_validation->run() == FALSE) {
                $errors = array();
                $errors['test_name'] = form_error('test_name', '', '');
                $errors['test_code'] = form_error('test_code', '', '');
                $errors['category'] = form_error('category', '', '');
                $errors['price'] = form_error('price', '', '');
                $errors['estimated_duration'] = form_error('estimated_duration', '', '');
                
                echo json_encode([
                    'status' => 'error',
                    'message' => 'Validation failed',
                    'error' => array_filter($errors)
                ]);
                return;
            }

            $data = array(
                'test_name' => $this->input->post('test_name'),
                'test_code' => strtoupper($this->input->post('test_code')),
                'category' => $this->input->post('category'),
                'description' => $this->input->post('description'),
                'preparation_instructions' => $this->input->post('preparation_instructions'),
                'estimated_duration' => $this->input->post('estimated_duration') ?: 30,
                'price' => $this->input->post('price'),
                'is_active' => 1,
                'requires_contrast' => $this->input->post('requires_contrast') ?: 0,
                'requires_fasting' => $this->input->post('requires_fasting') ?: 0,
                'is_gcc' => $this->input->post('is_gcc') ?: 0,
                'created_at' => date('Y-m-d H:i:s'),
                'updated_at' => date('Y-m-d H:i:s')
            );

            $this->db->insert('radiology_tests', $data);
            $test_id = $this->db->insert_id();

            if ($test_id) {
                $this->logRadiologyActivity($test_id, 'test_created', 'Radiology test "' . $data['test_name'] . '" created');
                
                echo json_encode([
                    'status' => 'success',
                    'message' => 'Radiology test added successfully',
                    'test_id' => $test_id
                ]);
            } else {
                echo json_encode([
                    'status' => 'error',
                    'message' => 'Failed to add radiology test'
                ]);
            }

        } catch (Exception $e) {
            log_message('error', 'Add radiology test error: ' . $e->getMessage());
            echo json_encode([
                'status' => 'error',
                'message' => 'Database error: ' . $e->getMessage()
            ]);
        }
    }

    /**
     * Edit radiology test form
     */
    public function edit_radiology_test($id) {
        if (!$this->checkPermission('can_edit')) {
            show_error('Access denied');
        }

        try {
            $this->db->where('id', $id);
            $test = $this->db->get('radiology_tests')->row();

            if (!$test) {
                show_error('Radiology test not found');
            }

            $data['test'] = $test;
            $data['categories'] = $this->getRadiologyCategories();
            
            $this->load->view('admin/radiology/setup/edit_form', $data);

        } catch (Exception $e) {
            log_message('error', 'Edit radiology test error: ' . $e->getMessage());
            show_error('Failed to load edit form: ' . $e->getMessage());
        }
    }

    /**
     * Update radiology test
     */
    public function update_radiology_test() {
        header('Content-Type: application/json');
        
        if (!$this->checkPermission('can_edit')) {
            echo json_encode(['status' => 'error', 'message' => 'Access denied']);
            return;
        }

        try {
            $test_id = $this->input->post('test_id');
            
            log_message('debug', 'Update test POST data: ' . print_r($_POST, true));
            
            if (!$test_id) {
                echo json_encode(['status' => 'error', 'message' => 'Test ID is required']);
                return;
            }
            
            $this->db->where('id', $test_id);
            $existing_test = $this->db->get('radiology_tests')->row();
            
            if (!$existing_test) {
                echo json_encode(['status' => 'error', 'message' => 'Test not found']);
                return;
            }
            
            $this->form_validation->set_rules('test_name', 'Test Name', 'required|trim|max_length[200]');
            $this->form_validation->set_rules('test_code', 'Test Code', 'required|trim|max_length[50]|callback_check_test_code_unique[' . $test_id . ']');
            $this->form_validation->set_rules('category', 'Category', 'required|trim|max_length[100]');
            $this->form_validation->set_rules('price', 'Price', 'required|numeric|greater_than_equal_to[0]');
            $this->form_validation->set_rules('estimated_duration', 'Duration', 'integer|greater_than[0]|less_than_equal_to[180]');

            if ($this->form_validation->run() == FALSE) {
                $errors = array();
                $errors['test_name'] = form_error('test_name', '', '');
                $errors['test_code'] = form_error('test_code', '', '');
                $errors['category'] = form_error('category', '', '');
                $errors['price'] = form_error('price', '', '');
                $errors['estimated_duration'] = form_error('estimated_duration', '', '');
                
                log_message('error', 'Validation errors: ' . print_r($errors, true));
                
                echo json_encode([
                    'status' => 'error',
                    'message' => 'Validation failed',
                    'error' => array_filter($errors)
                ]);
                return;
            }

            $data = array(
                'test_name' => $this->input->post('test_name'),
                'test_code' => strtoupper($this->input->post('test_code')),
                'category' => $this->input->post('category'),
                'description' => $this->input->post('description'),
                'preparation_instructions' => $this->input->post('preparation_instructions'),
                'estimated_duration' => $this->input->post('estimated_duration') ?: 30,
                'price' => $this->input->post('price'),
                'requires_contrast' => $this->input->post('requires_contrast') ?: 0,
                'requires_fasting' => $this->input->post('requires_fasting') ?: 0,
                'is_gcc' => $this->input->post('is_gcc') ?: 0,
                'updated_at' => date('Y-m-d H:i:s')
            );
            
            log_message('debug', 'Update data: ' . print_r($data, true));

            $this->db->where('id', $test_id);
            $result = $this->db->update('radiology_tests', $data);
            
            $db_error = $this->db->error();
            if ($db_error['code'] !== 0) {
                log_message('error', 'Database error during update: ' . print_r($db_error, true));
                echo json_encode([
                    'status' => 'error',
                    'message' => 'Database error: ' . $db_error['message']
                ]);
                return;
            }

            if ($result || $this->db->affected_rows() >= 0) {
                $this->logRadiologyActivity($test_id, 'test_updated', 'Radiology test updated');
                
                echo json_encode([
                    'status' => 'success',
                    'message' => 'Radiology test updated successfully'
                ]);
            } else {
                echo json_encode([
                    'status' => 'error',
                    'message' => 'Failed to update radiology test - no changes made'
                ]);
            }

        } catch (Exception $e) {
            log_message('error', 'Update radiology test error: ' . $e->getMessage());
            echo json_encode([
                'status' => 'error',
                'message' => 'Database error: ' . $e->getMessage(),
                'debug' => [
                    'file' => $e->getFile(),
                    'line' => $e->getLine()
                ]
            ]);
        }
    }

    /**
     * View radiology test details
     */
    public function view_radiology_test($id) {
        if (!$this->checkPermission('can_view')) {
            show_error('Access denied');
        }

        try {
            $this->db->where('id', $id);
            $test = $this->db->get('radiology_tests')->row();

            if (!$test) {
                show_error('Radiology test not found');
            }

            $data['test'] = $test;
            
            $this->load->view('admin/radiology/setup/view_details', $data);

        } catch (Exception $e) {
            log_message('error', 'View radiology test error: ' . $e->getMessage());
            show_error('Failed to load test details: ' . $e->getMessage());
        }
    }

    /**
     * Delete radiology test
     */
    public function delete_radiology_test($id) {
        header('Content-Type: application/json');
        
        if (!$this->checkPermission('can_delete')) {
            echo json_encode(['status' => 'error', 'message' => 'Access denied']);
            return;
        }

        try {
            if ($this->db->table_exists('radiology_order_tests')) {
                $this->db->where('test_id', $id);
                $orders_count = $this->db->count_all_results('radiology_order_tests');

                if ($orders_count > 0) {
                    echo json_encode([
                        'status' => 'error',
                        'message' => 'Cannot delete test. It is being used in ' . $orders_count . ' order(s).'
                    ]);
                    return;
                }
            }

            $this->db->where('id', $id);
            $test = $this->db->get('radiology_tests')->row();

            if (!$test) {
                echo json_encode([
                    'status' => 'error',
                    'message' => 'Radiology test not found'
                ]);
                return;
            }

            $this->db->where('id', $id);
            $result = $this->db->delete('radiology_tests');

            if ($result) {
                $this->logRadiologyActivity($id, 'test_deleted', 'Radiology test "' . $test->test_name . '" deleted');
                
                echo json_encode([
                    'status' => 'success',
                    'message' => 'Radiology test deleted successfully'
                ]);
            } else {
                echo json_encode([
                    'status' => 'error',
                    'message' => 'Failed to delete radiology test'
                ]);
            }

        } catch (Exception $e) {
            log_message('error', 'Delete radiology test error: ' . $e->getMessage());
            echo json_encode([
                'status' => 'error',
                'message' => 'Database error: ' . $e->getMessage()
            ]);
        }
    }

    /**
     * Toggle test status (active/inactive)
     */
    public function toggle_test_status() {
        header('Content-Type: application/json');
        
        if (!$this->checkPermission('can_edit')) {
            echo json_encode(['status' => 'error', 'message' => 'Access denied']);
            return;
        }

        try {
            $test_id = $this->input->post('test_id');
            $status = $this->input->post('status');

            $data = array(
                'is_active' => $status,
                'updated_at' => date('Y-m-d H:i:s')
            );

            $this->db->where('id', $test_id);
            $result = $this->db->update('radiology_tests', $data);

            if ($result) {
                $status_text = $status == 1 ? 'activated' : 'deactivated';
                
                $this->logRadiologyActivity($test_id, 'test_status_changed', 'Test ' . $status_text);
                
                echo json_encode([
                    'status' => 'success',
                    'message' => 'Test ' . $status_text . ' successfully'
                ]);
            } else {
                echo json_encode([
                    'status' => 'error',
                    'message' => 'Failed to update test status'
                ]);
            }

        } catch (Exception $e) {
            log_message('error', 'Toggle test status error: ' . $e->getMessage());
            echo json_encode([
                'status' => 'error',
                'message' => 'Database error: ' . $e->getMessage()
            ]);
        }
    }

    /**
     * Toggle GCC status for a test
     */
    public function toggle_gcc_status() {
        header('Content-Type: application/json');
        
        if (!$this->checkPermission('can_edit')) {
            echo json_encode(['status' => 'error', 'message' => 'Access denied']);
            return;
        }

        try {
            $test_id = $this->input->post('test_id');
            $gcc_status = $this->input->post('gcc_status');

            $data = array(
                'is_gcc' => $gcc_status,
                'updated_at' => date('Y-m-d H:i:s')
            );

            $this->db->where('id', $test_id);
            $result = $this->db->update('radiology_tests', $data);

            if ($result) {
                $status_text = $gcc_status == 1 ? 'marked as GCC test' : 'unmarked as GCC test';
                
                $this->logRadiologyActivity($test_id, 'gcc_status_changed', 'Test ' . $status_text);
                
                echo json_encode([
                    'status' => 'success',
                    'message' => 'Test ' . $status_text . ' successfully'
                ]);
            } else {
                echo json_encode([
                    'status' => 'error',
                    'message' => 'Failed to update GCC status'
                ]);
            }

        } catch (Exception $e) {
            log_message('error', 'Toggle GCC status error: ' . $e->getMessage());
            echo json_encode([
                'status' => 'error',
                'message' => 'Database error: ' . $e->getMessage()
            ]);
        }
    }

    /**
     * Get GCC tests only
     */
    public function get_gcc_tests() {
        header('Content-Type: application/json');
        
        try {
            $this->db->select('id, test_name, test_code, category, estimated_duration, price, requires_contrast, requires_fasting, preparation_instructions');
            $this->db->from('radiology_tests');
            $this->db->where('is_active', 1);
            $this->db->where('is_gcc', 1);
            $this->db->order_by('category, test_name');
            
            $query = $this->db->get();
            $tests = $query->result_array();
            
            echo json_encode([
                'status' => 'success',
                'data' => $tests,
                'count' => count($tests)
            ]);
            
        } catch (Exception $e) {
            log_message('error', 'Get GCC tests error: ' . $e->getMessage());
            echo json_encode([
                'status' => 'error',
                'message' => 'Failed to load GCC tests: ' . $e->getMessage()
            ]);
        }
    }

    /**
     * Get GCC packages
     */
    public function get_gcc_packages() {
        header('Content-Type: application/json');
        
        try {
            if (!$this->db->table_exists('gcc_radiology_packages')) {
                echo json_encode([
                    'status' => 'error',
                    'message' => 'GCC packages table not found. Please run the GCC enhancement SQL script.'
                ]);
                return;
            }
            
            $this->db->select('p.*, GROUP_CONCAT(t.test_name ORDER BY pt.order_sequence SEPARATOR ", ") as included_tests');
            $this->db->from('gcc_radiology_packages p');
            $this->db->join('gcc_radiology_package_tests pt', 'p.id = pt.package_id', 'left');
            $this->db->join('radiology_tests t', 'pt.test_id = t.id', 'left');
            $this->db->where('p.is_active', 1);
            $this->db->group_by('p.id');
            $this->db->order_by('p.package_name');
            
            $query = $this->db->get();
            $packages = $query->result_array();
            
            echo json_encode([
                'status' => 'success',
                'data' => $packages,
                'count' => count($packages)
            ]);
            
        } catch (Exception $e) {
            log_message('error', 'Get GCC packages error: ' . $e->getMessage());
            echo json_encode([
                'status' => 'error',
                'message' => 'Failed to load GCC packages: ' . $e->getMessage()
            ]);
        }
    }

    /**
     * Export radiology tests to CSV
     */
    public function export_radiology_tests() {
        if (!$this->checkPermission('can_view')) {
            show_error('Access denied');
        }

        try {
            $this->load->helper('download');
            
            $this->db->select('test_name, test_code, category, description, estimated_duration, price, requires_contrast, requires_fasting, is_gcc, is_active, created_at');
            $this->db->order_by('test_name');
            $query = $this->db->get('radiology_tests');
            $tests = $query->result_array();

            $csv_data = "Test Name,Test Code,Category,Description,Duration (mins),Price (KES),Requires Contrast,Requires Fasting,GCC Test,Status,Created Date\n";
            
            foreach ($tests as $test) {
                $csv_data .= '"' . $test['test_name'] . '",';
                $csv_data .= '"' . $test['test_code'] . '",';
                $csv_data .= '"' . $test['category'] . '",';
                $csv_data .= '"' . str_replace('"', '""', $test['description']) . '",';
                $csv_data .= '"' . $test['estimated_duration'] . '",';
                $csv_data .= '"' . number_format($test['price'], 2) . '",';
                $csv_data .= '"' . ($test['requires_contrast'] ? 'Yes' : 'No') . '",';
                $csv_data .= '"' . ($test['requires_fasting'] ? 'Yes' : 'No') . '",';
                $csv_data .= '"' . ($test['is_gcc'] ? 'Yes' : 'No') . '",';
                $csv_data .= '"' . ($test['is_active'] ? 'Active' : 'Inactive') . '",';
                $csv_data .= '"' . date('Y-m-d', strtotime($test['created_at'])) . '"';
                $csv_data .= "\n";
            }

            $filename = 'radiology_tests_' . date('Y-m-d_H-i-s') . '.csv';
            force_download($filename, $csv_data);

        } catch (Exception $e) {
            log_message('error', 'Export radiology tests error: ' . $e->getMessage());
            show_error('Failed to export radiology tests: ' . $e->getMessage());
        }
    }

    /**
     * Friendly URL method aliases
     */
    public function get_tests_datatable() {
        return $this->get_radiology_tests_datatable();
    }

    public function add_test() {
        return $this->add_radiology_test();
    }

    public function edit_test($id) {
        return $this->edit_radiology_test($id);
    }

    public function update_test() {
        return $this->update_radiology_test();
    }

    public function view_test($id) {
        return $this->view_radiology_test($id);
    }

    public function delete_test($id) {
        return $this->delete_radiology_test($id);
    }

    public function toggle_status() {
        return $this->toggle_test_status();
    }

    public function export() {
        return $this->export_radiology_tests();
    }

    /**
     * Get radiology categories
     */
    private function getRadiologyCategories() {
        return array(
            'X-Ray' => 'X-Ray',
            'CT Scan' => 'CT Scan',
            'MRI' => 'MRI',
            'Ultrasound' => 'Ultrasound',
            'Mammography' => 'Mammography',
            'Fluoroscopy' => 'Fluoroscopy',
            'Nuclear Medicine' => 'Nuclear Medicine',
            'Interventional' => 'Interventional Radiology',
            'Special' => 'Special Procedures'
        );
    }

    /**
     * Custom validation callback for test code uniqueness during update
     */
    public function check_test_code_unique($test_code, $test_id) {
        $this->db->where('test_code', $test_code);
        $this->db->where('id !=', $test_id);
        $query = $this->db->get('radiology_tests');
        
        if ($query->num_rows() > 0) {
            $this->form_validation->set_message('check_test_code_unique', 'The Test Code is already in use.');
            return FALSE;
        }
        return TRUE;
    }

    /**
     * Log radiology activity
     */
    private function logRadiologyActivity($test_id, $action, $description) {
        try {
            $log_data = array(
                'test_id' => $test_id,
                'action' => $action,
                'description' => $description,
                'user_id' => method_exists($this->customlib, 'getStaffID') ? $this->customlib->getStaffID() : 1,
                'ip_address' => $this->input->ip_address(),
                'user_agent' => $this->input->user_agent(),
                'timestamp' => date('Y-m-d H:i:s')
            );
            
            if (!$this->db->table_exists('radiology_activity_logs')) {
                $this->createRadiologyActivityLogsTable();
            }
            
            $this->db->insert('radiology_activity_logs', $log_data);
        } catch (Exception $e) {
            log_message('error', 'Failed to log radiology activity: ' . $e->getMessage());
        }
    }

    /**
     * Create radiology activity logs table
     */
    private function createRadiologyActivityLogsTable() {
        $sql = "
            CREATE TABLE IF NOT EXISTS `radiology_activity_logs` (
                `id` int(11) NOT NULL AUTO_INCREMENT,
                `test_id` int(11) DEFAULT NULL,
                `patient_id` int(11) DEFAULT NULL,
                `order_id` int(11) DEFAULT NULL,
                `action` varchar(100) NOT NULL,
                `description` text NOT NULL,
                `user_id` int(11) DEFAULT NULL,
                `ip_address` varchar(45) DEFAULT NULL,
                `user_agent` text,
                `timestamp` datetime NOT NULL,
                PRIMARY KEY (`id`),
                KEY `test_id` (`test_id`),
                KEY `patient_id` (`patient_id`),
                KEY `order_id` (`order_id`),
                KEY `timestamp` (`timestamp`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8;
        ";
        
        $this->db->query($sql);
    }

    /**
     * Test update functionality
     */
    public function test_update() {
        header('Content-Type: application/json');
        
        try {
            $test_id = $this->input->post('test_id') ?: 1;
            
            echo json_encode([
                'status' => 'success',
                'message' => 'Update endpoint is working',
                'received_data' => $_POST,
                'test_id' => $test_id,
                'base_url' => base_url(),
                'method' => $_SERVER['REQUEST_METHOD'],
                'content_type' => isset($_SERVER['CONTENT_TYPE']) ? $_SERVER['CONTENT_TYPE'] : 'not set'
            ]);
            
        } catch (Exception $e) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Test failed: ' . $e->getMessage()
            ]);
        }
    }

    /**
     * Test method for debugging
     */
    public function test() {
        echo "<h1>Radiology Setup Controller Test</h1>";
        echo "<p>Controller is working!</p>";
        echo "<p>Base URL: " . base_url() . "</p>";
        echo "<p>Current time: " . date('Y-m-d H:i:s') . "</p>";
        
        try {
            if ($this->db->table_exists('radiology_tests')) {
                $this->db->select('COUNT(*) as test_count');
                $this->db->from('radiology_tests');
                $query = $this->db->get();
                $result = $query->row();
                echo "<p>Radiology tests in database: " . $result->test_count . "</p>";
                
                if ($result->test_count > 0) {
                    $this->db->select('*');
                    $this->db->from('radiology_tests');
                    $this->db->limit(3);
                    $query = $this->db->get();
                    $tests = $query->result_array();
                    
                    echo "<h3>Sample Tests:</h3>";
                    echo "<table border='1' style='border-collapse: collapse;'>";
                    echo "<tr><th>ID</th><th>Name</th><th>Code</th><th>Category</th><th>Price</th><th>GCC</th><th>Status</th></tr>";
                    foreach ($tests as $test) {
                        echo "<tr>";
                        echo "<td>" . $test['id'] . "</td>";
                        echo "<td>" . htmlspecialchars($test['test_name']) . "</td>";
                        echo "<td>" . htmlspecialchars($test['test_code']) . "</td>";
                        echo "<td>" . htmlspecialchars($test['category']) . "</td>";
                        echo "<td>Ksh. " . number_format($test['price'], 2) . "</td>";
                        echo "<td>" . (isset($test['is_gcc']) && $test['is_gcc'] ? 'GCC' : 'Regular') . "</td>";
                        echo "<td>" . ($test['is_active'] ? 'Active' : 'Inactive') . "</td>";
                        echo "</tr>";
                    }
                    echo "</table>";
                }
            } else {
                echo "<p style='color: red;'>Radiology tests table does not exist!</p>";
                echo "<h3>Required SQL:</h3>";
                echo "<pre>";
                echo "-- Add is_gcc column if missing:
ALTER TABLE radiology_tests ADD COLUMN is_gcc tinyint(1) NOT NULL DEFAULT '0' AFTER is_active;";
                echo "</pre>";
            }
            
            echo "<h3>RBAC Permissions:</h3>";
            echo "<ul>";
            try {
                echo "<li>Can View: " . ($this->checkPermission('can_view') ? 'Yes' : 'No') . "</li>";
                echo "<li>Can Add: " . ($this->checkPermission('can_add') ? 'Yes' : 'No') . "</li>";
                echo "<li>Can Edit: " . ($this->checkPermission('can_edit') ? 'Yes' : 'No') . "</li>";
                echo "<li>Can Delete: " . ($this->checkPermission('can_delete') ? 'Yes' : 'No') . "</li>";
            } catch (Exception $e) {
                echo "<li style='color: red;'>RBAC Error: " . $e->getMessage() . "</li>";
            }
            echo "</ul>";
            
            echo "<h3>Test AJAX Endpoint:</h3>";
            echo "<button onclick='testAjaxEndpoint()' style='padding: 10px; background: #007cba; color: white; border: none; border-radius: 3px;'>Test DataTables AJAX</button>";
            echo "<div id='ajax-result' style='margin-top: 10px; padding: 10px; background: #f9f9f9; border: 1px solid #ddd;'></div>";
            
            echo "<script>
function testAjaxEndpoint() {
    document.getElementById('ajax-result').innerHTML = 'Testing...';
    
    fetch('" . base_url() . "admin/radiology-setup/get-tests-datatable', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/x-www-form-urlencoded',
        },
        body: 'draw=1&start=0&length=5&search[value]='
    })
    .then(response => response.json())
    .then(data => {
        document.getElementById('ajax-result').innerHTML = 
            '<strong>Success!</strong><br><pre>' + JSON.stringify(data, null, 2) + '</pre>';
    })
    .catch(error => {
        document.getElementById('ajax-result').innerHTML = 
            '<strong style=\"color: red;\">Error:</strong> ' + error.message;
    });
}
</script>";
            
        } catch (Exception $e) {
            echo "<p style='color: red;'>Error: " . $e->getMessage() . "</p>";
        }
    }
}

?>