<?php
defined('BASEPATH') OR exit('No direct script access allowed');

/**
 * Enhanced Queue Management Controller
 * Handles all queue operations with real-time notifications for all departments
 */
class Queue extends Admin_Controller {

    public function __construct() {
        parent::__construct();
        $this->load->model('Queue_model');
        $this->load->library('session');
        $this->load->helper('url');
    }

    // =============================================
    // NOTIFICATION ENDPOINTS (ENHANCED)
    // =============================================

    /**
     * MAIN NOTIFICATION ENDPOINT - Enhanced for all departments
     * Returns notifications with patient details and sound triggers
     */
public function get_queue_notifications() {
        header('Content-Type: application/json');
        
        $department = $this->input->get('department', TRUE);
        $last_check = $this->input->get('last_check', TRUE) ?: (time() - 30);
        
        if (!$department) {
            echo json_encode(['status' => 'error', 'message' => 'Department required']);
            return;
        }
        
        try {
            // Get new patients since last check
            $notifications = $this->getNewPatientsForDepartment($department, $last_check);
            
            // Check for file-based notifications (urgent/manual notifications)
            $file_notifications = $this->getFileBasedNotifications($department);
            
            // Merge notifications
            $all_notifications = array_merge($notifications, $file_notifications);
            
            if (!empty($all_notifications)) {
                echo json_encode([
                    'status' => 'success',
                    'notifications' => $all_notifications,
                    'timestamp' => time(),
                    'count' => count($all_notifications)
                ]);
            } else {
                echo json_encode([
                    'status' => 'success',
                    'notifications' => [],
                    'timestamp' => time(),
                    'count' => 0
                ]);
            }
            
        } catch (Exception $e) {
            log_message('error', 'Queue notification error: ' . $e->getMessage());
            echo json_encode(['status' => 'error', 'message' => $e->getMessage()]);
        }
    }

    /**
     * Get new patients for specific department since last check
     */
private function getNewPatientsForDepartment($department, $last_check) {
        $notifications = [];
        
        try {
            // Query for new queue entries
            $this->db->select('pq.*, p.patient_name, p.mobileno, p.age, p.gender, p.patient_unique_id');
            $this->db->from('patient_queue pq');
            $this->db->join('patients p', 'p.id = pq.patient_id', 'left');
            $this->db->where('pq.department', $department);
            $this->db->where('pq.status', 'waiting');
            $this->db->where('UNIX_TIMESTAMP(pq.created_at) >', $last_check);
            $this->db->order_by('pq.created_at', 'DESC');
            $this->db->limit(10);
            
            $new_patients = $this->db->get()->result_array();
            
            foreach ($new_patients as $patient) {
                $notifications[] = [
                    'patient_id' => $patient['patient_id'],
                    'patient_name' => $patient['patient_name'] ?: 'Unknown Patient',
                    'queue_number' => $patient['queue_number'],
                    'department' => $patient['department'],
                    'priority' => $patient['priority'],
                    'phone' => $patient['mobileno'],
                    'age' => $patient['age'],
                    'gender' => $patient['gender'],
                    'patient_unique_id' => $patient['patient_unique_id'],
                    'workflow_type' => $patient['workflow_type'],
                    'time' => date('H:i:s', strtotime($patient['created_at'])),
                    'is_urgent' => in_array($patient['priority'], ['urgent', 'high']),
                    'notification_type' => 'new_patient'
                ];
            }
            
        } catch (Exception $e) {
            log_message('error', 'Error getting new patients: ' . $e->getMessage());
        }
        
        return $notifications;
    }

    /**
     * Get file-based notifications (urgent/manual)
     */
private function getFileBasedNotifications($department) {
        $notifications = [];
        
        try {
            $notification_file = APPPATH . 'cache/queue_notifications_' . $department . '.json';
            
            if (file_exists($notification_file)) {
                $file_content = file_get_contents($notification_file);
                $notification_data = json_decode($file_content, true);
                
                if ($notification_data) {
                    // Add to notifications array
                    $notifications[] = $notification_data;
                    
                    // Clear notification after reading
                    unlink($notification_file);
                }
            }
        } catch (Exception $e) {
            log_message('error', 'Error reading file notifications: ' . $e->getMessage());
        }
        
        return $notifications;
    }


    /**
     * Send notification to department (called by other controllers)
     */
public function send_notification_to_department($department, $notification_data) {
        try {
            // Ensure cache directory exists
            $cache_dir = APPPATH . 'cache';
            if (!is_dir($cache_dir)) {
                mkdir($cache_dir, 0755, true);
            }
            
            $notification = [
                'patient_id' => $notification_data['patient_id'] ?? null,
                'patient_name' => $notification_data['patient_name'] ?? 'Unknown Patient',
                'queue_number' => $notification_data['queue_number'] ?? 'N/A',
                'department' => $department,
                'priority' => $notification_data['priority'] ?? 'normal',
                'phone' => $notification_data['phone'] ?? '',
                'workflow_type' => $notification_data['workflow_type'] ?? 'general',
                'time' => date('H:i:s'),
                'is_urgent' => in_array($notification_data['priority'] ?? 'normal', ['urgent', 'high']),
                'notification_type' => $notification_data['notification_type'] ?? 'incoming_patient',
                'from_department' => $notification_data['from_department'] ?? 'system',
                'timestamp' => time(),
                'sound' => true
            ];
            
            // Store notification file for target department
            $notification_file = APPPATH . 'cache/queue_notifications_' . $department . '.json';
            file_put_contents($notification_file, json_encode($notification));
            
            log_message('info', "Notification sent to {$department}: {$notification['patient_name']}");
            
            return ['status' => 'success', 'message' => 'Notification sent'];
            
        } catch (Exception $e) {
            log_message('error', 'Failed to send notification: ' . $e->getMessage());
            return ['status' => 'error', 'message' => $e->getMessage()];
        }
    }
    
    /**
     * ENHANCED: Add patient to queue with notification
     */
    public function add_to_queue_with_notification() {
        $this->_ensure_ajax();
        
        $patient_id = $this->input->post('patient_id');
        $department = $this->input->post('department', TRUE) ?: 'registration';
        $priority = $this->input->post('priority', TRUE) ?: 'normal';
        $workflow_type = $this->input->post('workflow_type', TRUE) ?: 'general';
        
        if (!$patient_id) {
            $this->_json_response(['status' => 'error', 'message' => 'Patient ID required']);
            return;
        }

        try {
            // Add to queue
            $result = $this->Queue_model->addPatientToQueue($patient_id, $department, $priority, $workflow_type);
            
            if ($result['status'] === 'success') {
                // Get patient details for notification
                $patient = $this->getPatientDetails($patient_id);
                
                // Send notification to department
                $this->send_notification_to_department($department, [
                    'patient_id' => $patient_id,
                    'patient_name' => $patient['patient_name'] ?? 'Unknown',
                    'phone' => $patient['mobileno'] ?? '',
                    'queue_number' => $result['queue_number'],
                    'priority' => $priority,
                    'workflow_type' => $workflow_type,
                    'notification_type' => 'new_patient',
                    'from_department' => 'registration'
                ]);
                
                $result['notification_sent'] = true;
            }
            
            $this->_json_response($result);
            
        } catch (Exception $e) {
            $this->_json_response(['status' => 'error', 'message' => $e->getMessage()]);
        }
    }

/**
     * ENHANCED: Complete service with notification to next department
     */
    public function complete_service_with_notification() {
        $this->_ensure_ajax();
        
        $queue_id = $this->input->post('queue_id');
        $next_department = $this->input->post('next_department', TRUE);
        $notes = $this->input->post('notes', TRUE);
        
        if (!$queue_id) {
            $this->_json_response(['status' => 'error', 'message' => 'Queue ID required']);
            return;
        }

        try {
            $result = $this->Queue_model->completeService($queue_id, $next_department, $notes);
            
            // If routing to next department, send notification
            if ($result['status'] === 'success' && $next_department) {
                $queue_data = $this->getQueueById($queue_id);
                $patient = $this->getPatientDetails($queue_data['patient_id']);
                
                $this->send_notification_to_department($next_department, [
                    'patient_id' => $queue_data['patient_id'],
                    'patient_name' => $patient['patient_name'] ?? 'Unknown',
                    'phone' => $patient['mobileno'] ?? '',
                    'queue_number' => $result['new_queue_number'] ?? 'TRF-' . date('His'),
                    'priority' => $queue_data['priority'],
                    'workflow_type' => $queue_data['workflow_type'],
                    'notification_type' => 'transferred_patient',
                    'from_department' => $queue_data['department']
                ]);
                
                $result['notification_sent'] = true;
            }
            
            $this->_json_response($result);
            
        } catch (Exception $e) {
            $this->_json_response(['status' => 'error', 'message' => $e->getMessage()]);
        }
    }
    
    /**
     * Add test patient for debugging
     */
    public function add_test_patient_to_queue() {
        // Only allow in development or for admin users
        if (!$this->session->userdata('admin_id')) {
            show_404();
            return;
        }

        $department = $this->input->get('department', TRUE) ?: 'triage';
        
        try {
            // Create a test patient
            $test_patient_data = [
                'patient_name' => 'Test Patient ' . date('His'),
                'mobileno' => '0700' . rand(100000, 999999),
                'age' => rand(20, 80),
                'gender' => rand(0, 1) ? 'Male' : 'Female',
                'patient_unique_id' => 'TEST-' . date('His'),
                'is_active' => 'yes',
                'created_at' => date('Y-m-d H:i:s')
            ];
            
            $this->db->insert('patients', $test_patient_data);
            $patient_id = $this->db->insert_id();
            
            if ($patient_id) {
                // Add to queue
                $priority = ['normal', 'high', 'urgent'][rand(0, 2)];
                $result = $this->Queue_model->addPatientToQueue($patient_id, $department, $priority, 'test');
                
                if ($result['status'] === 'success') {
                    // Send notification
                    $this->send_notification_to_department($department, [
                        'patient_id' => $patient_id,
                        'patient_name' => $test_patient_data['patient_name'],
                        'phone' => $test_patient_data['mobileno'],
                        'queue_number' => $result['queue_number'],
                        'priority' => $priority,
                        'workflow_type' => 'test',
                        'notification_type' => 'test_patient',
                        'from_department' => 'debug'
                    ]);
                    
                    echo json_encode([
                        'status' => 'success',
                        'message' => "Test patient added to {$department}",
                        'patient_id' => $patient_id,
                        'queue_number' => $result['queue_number']
                    ]);
                } else {
                    echo json_encode(['status' => 'error', 'message' => 'Failed to add to queue']);
                }
            } else {
                echo json_encode(['status' => 'error', 'message' => 'Failed to create test patient']);
            }
            
        } catch (Exception $e) {
            echo json_encode(['status' => 'error', 'message' => $e->getMessage()]);
        }
    }
    
    /**
     * Get queue statistics for a department
     */
    private function getQueueStatisticsForDepartment($department) {
        try {
            // Get waiting count
            $this->db->where('department', $department);
            $this->db->where('status', 'waiting');
            $waiting_count = $this->db->count_all_results('patient_queue');
            
            // Get completed today count
            $this->db->where('department', $department);
            $this->db->where('status', 'completed');
            $this->db->where('DATE(completed_at)', date('Y-m-d'));
            $completed_today = $this->db->count_all_results('patient_queue');
            
            // Get average wait time
            $this->db->select('AVG(actual_wait_time) as avg_wait');
            $this->db->where('department', $department);
            $this->db->where('actual_wait_time IS NOT NULL');
            $this->db->where('DATE(completed_at)', date('Y-m-d'));
            $avg_result = $this->db->get('patient_queue')->row_array();
            $avg_wait_time = round($avg_result['avg_wait'] ?? 0, 1);
            
            return [
                'waiting_count' => $waiting_count,
                'completed_today' => $completed_today,
                'avg_wait_time' => $avg_wait_time
            ];
            
        } catch (Exception $e) {
            log_message('error', "Error getting stats for {$department}: " . $e->getMessage());
            return [
                'waiting_count' => 0,
                'completed_today' => 0,
                'avg_wait_time' => 0
            ];
        }
    }
    // =============================================
    // QUEUE MANAGEMENT METHODS
    // =============================================

    /**
     * Add patient to queue with automatic notification
     */
    public function add_to_queue() {
        $this->_ensure_ajax();
        
        $patient_id = $this->input->post('patient_id');
        $department = $this->input->post('department', TRUE) ?: 'registration';
        $priority = $this->input->post('priority', TRUE) ?: 'normal';
        $workflow_type = $this->input->post('workflow_type', TRUE) ?: 'general';
        
        if (!$patient_id) {
            $this->_json_response(['status' => 'error', 'message' => 'Patient ID required']);
            return;
        }

        try {
            // Add to queue
            $result = $this->Queue_model->addPatientToQueue($patient_id, $department, $priority, $workflow_type);
            
            if ($result['status'] === 'success') {
                // Get patient details for notification
                $patient = $this->getPatientDetails($patient_id);
                
                // Send notification to department
                $this->send_notification_to_department($department, [
                    'patient_id' => $patient_id,
                    'patient_name' => $patient['patient_name'] ?? 'Unknown',
                    'phone' => $patient['mobileno'] ?? '',
                    'queue_number' => $result['queue_number'],
                    'priority' => $priority,
                    'workflow_type' => $workflow_type,
                    'notification_type' => 'new_patient',
                    'from_department' => 'registration'
                ]);
                
                $result['notification_sent'] = true;
            }
            
            $this->_json_response($result);
            
        } catch (Exception $e) {
            $this->_json_response(['status' => 'error', 'message' => $e->getMessage()]);
        }
    }

    /**
     * Call next patient
     */
    public function call_next_patient() {
        $this->_ensure_ajax();
        
        $department = $this->input->post('department', TRUE);
        $staff_id = $this->input->post('staff_id') ?: $this->session->userdata('admin_id');
        
        if (!$department) {
            $this->_json_response(['status' => 'error', 'message' => 'Department required']);
            return;
        }

        try {
            $result = $this->Queue_model->callNextPatient($department, $staff_id);
            $this->_json_response($result);
        } catch (Exception $e) {
            $this->_json_response(['status' => 'error', 'message' => $e->getMessage()]);
        }
    }

    /**
     * Complete service and route to next department
     */
    public function complete_service() {
        $this->_ensure_ajax();
        
        $queue_id = $this->input->post('queue_id');
        $next_department = $this->input->post('next_department', TRUE);
        $notes = $this->input->post('notes', TRUE);
        
        if (!$queue_id) {
            $this->_json_response(['status' => 'error', 'message' => 'Queue ID required']);
            return;
        }

        try {
            $result = $this->Queue_model->completeService($queue_id, $next_department, $notes);
            
            // If routing to next department, send notification
            if ($result['status'] === 'success' && $next_department) {
                $queue_data = $this->Queue_model->getQueueById($queue_id);
                $patient = $this->getPatientDetails($queue_data['patient_id']);
                
                $this->send_notification_to_department($next_department, [
                    'patient_id' => $queue_data['patient_id'],
                    'patient_name' => $patient['patient_name'] ?? 'Unknown',
                    'phone' => $patient['mobileno'] ?? '',
                    'queue_number' => $result['new_queue_number'] ?? 'TRF-' . date('His'),
                    'priority' => $queue_data['priority'],
                    'workflow_type' => $queue_data['workflow_type'],
                    'notification_type' => 'transferred_patient',
                    'from_department' => $queue_data['department']
                ]);
            }
            
            $this->_json_response($result);
            
        } catch (Exception $e) {
            $this->_json_response(['status' => 'error', 'message' => $e->getMessage()]);
        }
    }
    
    /**
     * Get queue by ID
     */
    private function getQueueById($queue_id) {
        $this->db->where('id', $queue_id);
        return $this->db->get('patient_queue')->row_array();
    }

    /**
     * Skip patient in queue
     */
    public function skip_patient() {
        $this->_ensure_ajax();
        
        $queue_id = $this->input->post('queue_id');
        $reason = $this->input->post('reason', TRUE) ?: 'Patient skipped';
        
        if (!$queue_id) {
            $this->_json_response(['status' => 'error', 'message' => 'Queue ID required']);
            return;
        }

        try {
            $result = $this->Queue_model->skipPatient($queue_id, $reason);
            $this->_json_response($result);
        } catch (Exception $e) {
            $this->_json_response(['status' => 'error', 'message' => $e->getMessage()]);
        }
    }

    // =============================================
    // DEPARTMENT QUEUE VIEWS
    // =============================================

    /**
     * Main queue dashboard
     */
    public function index() {
        $data['title'] = 'Queue Management Dashboard';
        $data['queues_overview'] = $this->Queue_model->getAllQueuesOverview();
        
        $this->load->view('layout/header', $data);
        $this->load->view('admin/queue/dashboard', $data);
        $this->load->view('layout/footer');
    }

    /**
     * Department-specific queue management
     */
    public function department($department = null) {
        if (!$department) {
            redirect('admin/queue');
        }

        $data['title'] = ucfirst($department) . ' Queue Management';
        $data['department'] = $department;
        $data['queue_data'] = $this->Queue_model->getDepartmentQueue($department);
        $data['queue_config'] = $this->Queue_model->getQueueConfig($department);
        
        $this->load->view('layout/header', $data);
        $this->load->view('admin/queue/department', $data);
        $this->load->view('layout/footer');
    }

    // =============================================
    // DEBUG AND TESTING METHODS
    // =============================================


    /**
     * Debug queue status
     */
    public function debug_queue_status() {
        header('Content-Type: application/json');
        
        try {
            $stats = [];
            $departments = ['registration', 'triage', 'consultation', 'pharmacy', 'laboratory', 'radiology'];
            
            foreach ($departments as $dept) {
                $stats[$dept] = $this->Queue_model->getQueueStatistics($dept);
            }
            
            echo json_encode([
                'status' => 'success',
                'timestamp' => date('Y-m-d H:i:s'),
                'department_stats' => $stats,
                'total_waiting' => array_sum(array_column($stats, 'waiting_count'))
            ]);
            
        } catch (Exception $e) {
            echo json_encode(['status' => 'error', 'message' => $e->getMessage()]);
        }
    }

    // =============================================
    // UTILITY METHODS
    // =============================================

    /**
     * Get patient details
     */
    private function getPatientDetails($patient_id) {
        $this->db->where('id', $patient_id);
        return $this->db->get('patients')->row_array();
    }

    /**
     * Ensure request is AJAX
     */
    private function _ensure_ajax() {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }
    }

    /**
     * Send JSON response
     */
    private function _json_response($data) {
        header('Content-Type: application/json');
        echo json_encode($data);
        exit;
    }

    /**
     * Get available departments
     */
    public function get_departments() {
        $this->_ensure_ajax();
        
        $departments = [
            'registration' => 'Registration',
            'triage' => 'Triage',
            'consultation' => 'Consultation',
            'cardiology' => 'Cardiology',
            'neurology' => 'Neurology',
            'orthopedics' => 'Orthopedics',
            'dermatology' => 'Dermatology',
            'pediatrics' => 'Pediatrics',
            'gynecology' => 'Gynecology',
            'psychiatry' => 'Psychiatry',
            'psychology' => 'Psychology',
            'ent' => 'ENT',
            'optometrics' => 'Optometrics',
            'urology' => 'Urology',
            'physiotherapy' => 'Physiotherapy',
            'physician' => 'Physician',
            'dental' => 'Dental',
            'surgery' => 'Surgery',
            'nutrition' => 'Nutrition',
            'speech_therapy' => 'Speech Therapy',
            'laboratory' => 'Laboratory',
            'radiology' => 'Radiology',
            'pharmacy' => 'Pharmacy',
            'phlebotomy' => 'Phlebotomy'
        ];
        
        $this->_json_response(['status' => 'success', 'departments' => $departments]);
    }

    // =============================================
    // MISSING INTEGRATION ENDPOINTS FOR JAVASCRIPT
    // =============================================

    /**
     * Get department queue (for JavaScript integration)
     */
    public function getDepartmentQueue() {
        $this->_ensure_ajax();
        
        $department = $this->input->get('department', TRUE);
        if (!$department) {
            $this->_json_response(['status' => 'error', 'message' => 'Department required']);
            return;
        }

        try {
            $result = $this->Queue_model->getDepartmentQueue($department);
            $this->_json_response(['status' => 'success', 'data' => $result]);
        } catch (Exception $e) {
            $this->_json_response(['status' => 'error', 'message' => $e->getMessage()]);
        }
    }

    /**
     * Get triage queue (specific for triage integration)
     */
    public function getTriageQueue() {
        $this->_ensure_ajax();
        
        try {
            $this->db->select('pq.*, p.patient_name, p.mobileno, p.age, p.gender, p.patient_unique_id');
            $this->db->from('patient_queue pq');
            $this->db->join('patients p', 'p.id = pq.patient_id', 'left');
            $this->db->where('pq.department', 'triage');
            $this->db->where('pq.status', 'waiting');
            $this->db->order_by("FIELD(pq.priority, 'urgent', 'high', 'normal', 'low')", '', FALSE);
            $this->db->order_by('pq.created_at', 'ASC');
            $this->db->limit(20);
            
            $queue_patients = $this->db->get()->result_array();
            
            $this->_json_response([
                'status' => 'success', 
                'queue_patients' => $queue_patients,
                'count' => count($queue_patients)
            ]);
        } catch (Exception $e) {
            $this->_json_response(['status' => 'error', 'message' => $e->getMessage()]);
        }
    }

    /**
     * Update queue status (for JavaScript integration)
     */
    public function updateQueueStatus() {
        $this->_ensure_ajax();
        
        $queue_id = $this->input->post('queue_id');
        $status = $this->input->post('status', TRUE);
        
        if (!$queue_id || !$status) {
            $this->_json_response(['status' => 'error', 'message' => 'Queue ID and status required']);
            return;
        }

        try {
            $update_data = [
                'status' => $status,
                'updated_at' => date('Y-m-d H:i:s')
            ];
            
            if ($status === 'in_progress') {
                $update_data['called_at'] = date('Y-m-d H:i:s');
            }

            $this->db->where('id', $queue_id);
            $result = $this->db->update('patient_queue', $update_data);

            if ($result) {
                $this->_json_response(['status' => 'success', 'message' => 'Queue status updated']);
            } else {
                $this->_json_response(['status' => 'error', 'message' => 'Failed to update queue status']);
            }
        } catch (Exception $e) {
            $this->_json_response(['status' => 'error', 'message' => $e->getMessage()]);
        }
    }

    /**
     * Add patient to queue (alias for JavaScript integration)
     */
    public function addToQueue() {
        // This is an alias for the existing add_to_queue method
        $this->add_to_queue();
    }

    /**
     * Complete service (alias for JavaScript integration)  
     */
    public function completeService() {
        // This is an alias for the existing complete_service method
        $this->complete_service();
    }
}

/* End of file Queue.php */
/* Location: ./application/controllers/admin/Queue.php */