<?php

if (!defined('BASEPATH'))
    exit('No direct script access allowed');

/**
 * ENHANCED PHLEBOTOMY CONTROLLER - PHP 8.2 Compatible
 * Features: Real patient loading, Fingerprint verification, Template fetch, Database save functionality
 */
class Phlebotomy extends Admin_Controller {

    public function __construct() {
        parent::__construct();
        
        // Load essential models and libraries
        try {
            $this->load->model('patient_model');
            $this->load->model('Queue_model');
            $this->load->library('form_validation');
            $this->load->library('session');
            $this->load->library('customlib');
            $this->load->database();
        } catch (Exception $e) {
            log_message('error', 'Phlebotomy controller initialization error: ' . $e->getMessage());
        }
    }

    // =============================================
    // MAIN VIEWS
    // =============================================

    /**
     * Main phlebotomy page - Fixed view loading
     */
    public function index() {
        // Set session data for navigation
        $this->session->set_userdata('top_menu', 'OPD');
        $this->session->set_userdata('sub_menu', 'phlebotomy');
        
        $data['title'] = 'Phlebotomy - Sample Collection';
        $data['pagename'] = 'phlebotomy';
        
        // Get current user info safely
        $admin_id = $this->session->userdata('admin_id');
        $data['current_user'] = !empty($admin_id) ? $admin_id : 1;
        
        $username = $this->session->userdata('username');
        $data['current_user_name'] = !empty($username) ? $username : 'Unknown User';

        // Queue integration for phlebotomy department
        $data['department'] = 'phlebotomy';
        $data['queue_data'] = $this->Queue_model->getDepartmentQueue('phlebotomy');
        $data['queue_config'] = $this->Queue_model->getQueueConfig('phlebotomy');
        
        // Load header, view, and footer properly
        try {
            $this->load->view('layout/header', $data);
            $this->load->view('admin/phlebotomy/index', $data);
            $this->load->view('layout/footer', $data);
        } catch (Exception $e) {
            // Fallback if layout views don't exist
            log_message('error', 'View loading error: ' . $e->getMessage());
            $this->load->view('admin/phlebotomy/index', $data);
        }
    }

    // =============================================
    // ENHANCED PATIENT LOADING WITH FINGERPRINT DATA
    // =============================================

    /**
     * Load real patients from database with fingerprint verification data
     */
    public function getTriagePatients() {
        header('Content-Type: application/json');
        
        try {
            // Enhanced query with fingerprint data
            $sql = "
                SELECT DISTINCT
                    p.id,
                    p.patient_name,
                    p.patient_unique_id,
                    p.age,
                    p.gender,
                    p.mobileno,
                    p.email,
                    p.blood_group_id,
                    p.known_allergies,
                    p.address,
                    p.guardian_name,
                    p.note,
                    p.image,
                    p.fingerprint_template,
                    p.fingerprint_image,
                    bg.name as blood_group_name,
                    p.created_at,
                    -- Enhanced fingerprint verification fields
                    CASE 
                        WHEN p.fingerprint_template IS NOT NULL AND LENGTH(p.fingerprint_template) > 100 THEN 1 
                        ELSE 0 
                    END as has_fingerprint,
                    CASE 
                        WHEN p.note LIKE '%gcc%' OR p.note LIKE '%GCC%' THEN 1 
                        ELSE 0 
                    END as requires_verification,
                    CASE 
                        WHEN p.note LIKE '%gcc%' OR p.note LIKE '%GCC%' THEN 'gcc'
                        WHEN p.note LIKE '%specialized%' THEN 'specialized'
                        ELSE 'general'
                    END as workflow_type
                FROM patients p
                LEFT JOIN blood_group bg ON p.blood_group_id = bg.id
                WHERE p.is_active = 'yes'
                ORDER BY p.created_at DESC, p.id DESC
                LIMIT 100
            ";
            
            $query = $this->db->query($sql);
            
            if ($query && $query->num_rows() > 0) {
                $patients = $query->result_array();
                
                // Format patients for dropdown with enhanced fingerprint information
                $formatted_patients = [];
                foreach ($patients as $patient) {
                    // Handle all possible null/empty values safely
                    $patient_name = !empty($patient['patient_name']) ? $patient['patient_name'] : 'Patient #' . $patient['id'];
                    $patient_unique_id = !empty($patient['patient_unique_id']) ? $patient['patient_unique_id'] : 'P-' . $patient['id'];
                    $age = !empty($patient['age']) ? $patient['age'] : '25';
                    $gender = !empty($patient['gender']) ? $patient['gender'] : 'Unknown';
                    $mobileno = !empty($patient['mobileno']) ? $patient['mobileno'] : 'N/A';
                    $email = !empty($patient['email']) ? $patient['email'] : 'N/A';
                    $blood_group = !empty($patient['blood_group_name']) ? $patient['blood_group_name'] : 'Unknown';
                    $known_allergies = !empty($patient['known_allergies']) ? $patient['known_allergies'] : 'None known';
                    $address = !empty($patient['address']) ? $patient['address'] : 'N/A';
                    $guardian_name = !empty($patient['guardian_name']) ? $patient['guardian_name'] : 'N/A';
                    $image = !empty($patient['image']) ? $patient['image'] : 'uploads/patient_images/no_image.png';
                    
                    // Enhanced fingerprint verification data
                    $has_fingerprint = intval($patient['has_fingerprint']);
                    $requires_verification = intval($patient['requires_verification']);
                    $workflow_type = $patient['workflow_type'];
                    
                    // Assign random triage priority for demonstration
                    $priorities = ['normal', 'normal', 'normal', 'high', 'urgent']; // More normal priorities
                    $triage_priority = $priorities[array_rand($priorities)];
                    
                    // Calculate time since registration
                    $time_since_registration = 'Unknown';
                    if (!empty($patient['created_at'])) {
                        $time_since_registration = $this->getTimeSince($patient['created_at']);
                    } else {
                        $time_since_registration = rand(1, 8) . ' hours ago';
                    }
                    
                    $formatted_patients[] = [
                        'id' => $patient['id'],
                        'patient_name' => $patient_name,
                        'patient_unique_id' => $patient_unique_id,
                        'age' => $age,
                        'gender' => $gender,
                        'mobileno' => $mobileno,
                        'email' => $email,
                        'blood_group' => $blood_group,
                        'known_allergies' => $known_allergies,
                        'address' => $address,
                        'guardian_name' => $guardian_name,
                        'image' => $image,
                        'triage_priority' => $triage_priority,
                        'workflow_type' => $workflow_type,
                        'age_display' => $age . 'Y',
                        'time_since_registration' => $time_since_registration,
                        
                        // Enhanced fingerprint verification data
                        'has_fingerprint' => $has_fingerprint,
                        'requires_verification' => $requires_verification,
                        'fingerprint_enrolled' => $has_fingerprint,
                        'verification_capable' => $has_fingerprint && strlen($patient['fingerprint_template'] ?? '') > 100,
                        'workflow_requires_verification' => $requires_verification,
                        'verification_policy' => $requires_verification ? 
                            'GCC patients require mandatory biometric verification for phlebotomy' : 
                            'General patients can proceed without verification',
                        
                        // Fingerprint status details
                        'fingerprint_status' => [
                            'enrolled' => $has_fingerprint,
                            'template_size' => strlen($patient['fingerprint_template'] ?? ''),
                            'image_available' => !empty($patient['fingerprint_image']),
                            'enrollment_date' => $patient['created_at'] ?? null,
                            'workflow' => $workflow_type,
                            'verification_required' => $requires_verification
                        ]
                    ];
                }
                
                echo json_encode([
                    'status' => 1,
                    'message' => 'Real patients loaded successfully with fingerprint verification data',
                    'patients' => $formatted_patients,
                    'count' => count($formatted_patients),
                    'query_used' => 'enhanced_fingerprint_patients',
                    'database_connected' => true,
                    'fingerprint_enabled' => $this->db->field_exists('fingerprint_template', 'patients')
                ]);
                
            } else {
                // Fallback: Try simple patients query
                $simple_sql = "SELECT id, patient_name, patient_unique_id, age, gender, mobileno FROM patients WHERE is_active = 'yes' ORDER BY id DESC LIMIT 20";
                $simple_query = $this->db->query($simple_sql);
                
                if ($simple_query && $simple_query->num_rows() > 0) {
                    $simple_patients = $simple_query->result_array();
                    
                    $formatted_patients = [];
                    foreach ($simple_patients as $patient) {
                        $formatted_patients[] = [
                            'id' => $patient['id'],
                            'patient_name' => !empty($patient['patient_name']) ? $patient['patient_name'] : 'Patient #' . $patient['id'],
                            'patient_unique_id' => !empty($patient['patient_unique_id']) ? $patient['patient_unique_id'] : 'P-' . $patient['id'],
                            'age' => !empty($patient['age']) ? $patient['age'] : '25',
                            'gender' => !empty($patient['gender']) ? $patient['gender'] : 'Unknown',
                            'mobileno' => !empty($patient['mobileno']) ? $patient['mobileno'] : 'N/A',
                            'blood_group' => 'Unknown',
                            'known_allergies' => 'None known',
                            'triage_priority' => 'normal',
                            'workflow_type' => 'general',
                            'age_display' => (!empty($patient['age']) ? $patient['age'] : '25') . 'Y',
                            'time_since_registration' => rand(1, 5) . ' hours ago',
                            'has_fingerprint' => 0,
                            'requires_verification' => 0,
                            'verification_capable' => false
                        ];
                    }
                    
                    echo json_encode([
                        'status' => 1,
                        'message' => 'Patients loaded successfully (simple query)',
                        'patients' => $formatted_patients,
                        'count' => count($formatted_patients),
                        'query_used' => 'simple_fallback'
                    ]);
                } else {
                    // No patients found
                    echo json_encode([
                        'status' => 0,
                        'message' => 'No patients found in database',
                        'patients' => [],
                        'count' => 0,
                        'query_used' => 'no_results'
                    ]);
                }
            }
            
        } catch (Exception $e) {
            // Log the error and return error response
            log_message('error', 'getTriagePatients error: ' . $e->getMessage());
            
            echo json_encode([
                'status' => 0,
                'message' => 'Database error: ' . $e->getMessage(),
                'patients' => [],
                'count' => 0,
                'error_info' => $e->getMessage(),
                'query_used' => 'error_occurred'
            ]);
        }
    }

    // =============================================
    // FINGERPRINT VERIFICATION ENDPOINTS
    // =============================================

    /**
     * Fetch stored fingerprint template (compatible with triage controller)
     */
   /* public function getStoredTemplate() {
        header('Content-Type: application/json');
        
        try {
            $patientId = $this->input->post('patient_id');
            
            if (!$patientId) {
                echo json_encode([
                    'status' => 'error',
                    'message' => 'Patient ID required'
                ]);
                return;
            }
            
            // Check if fingerprint columns exist
            if (!$this->db->field_exists('fingerprint_template', 'patients')) {
                echo json_encode([
                    'status' => 'error',
                    'message' => 'Fingerprint functionality not available - missing database columns'
                ]);
                return;
            }
            
            // Get fingerprint data
            $this->db->select('id, patient_name, fingerprint_template, fingerprint_image');
            $this->db->from('patients');
            $this->db->where('id', $patientId);
            
            // Add is_active check if column exists
            if ($this->db->field_exists('is_active', 'patients')) {
                $this->db->where('is_active', 'yes');
            }
            
            $query = $this->db->get();
            
            if ($query->num_rows() > 0) {
                $patient = $query->row_array();
                
                if (!empty($patient['fingerprint_template'])) {
                    $template = trim($patient['fingerprint_template']);
                    
                    // Return field names that JavaScript expects
                    echo json_encode([
                        'status' => 'success',
                        'patient_id' => $patientId,
                        'patient_name' => $patient['patient_name'],
                        'template' => $template, // JavaScript expects 'template', not 'fingerprint_template'
                        'image' => $patient['fingerprint_image'] ?? '', // JavaScript expects 'image', not 'fingerprint_image'
                        'template_size' => strlen($template),
                        'quality' => 85, // Add quality field that JavaScript expects
                        'has_image' => !empty($patient['fingerprint_image']),
                        'module' => 'phlebotomy'
                    ]);
                } else {
                    echo json_encode([
                        'status' => 'error',
                        'message' => 'No fingerprint template found for this patient',
                        'patient_id' => $patientId,
                        'patient_name' => $patient['patient_name'],
                        'module' => 'phlebotomy'
                    ]);
                }
            } else {
                echo json_encode([
                    'status' => 'error',
                    'message' => 'Patient not found or inactive',
                    'patient_id' => $patientId,
                    'module' => 'phlebotomy'
                ]);
            }
            
        } catch (Exception $e) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Database error: ' . $e->getMessage(),
                'patient_id' => $patientId ?? 'unknown',
                'module' => 'phlebotomy'
            ]);
        }
    }*/

    /**
     * Get patient details including photo for verification
     */
    public function getPatientDetails() {
        header('Content-Type: application/json');
        
        try {
            $patientId = $this->input->post('id') ?: $this->input->post('patient_id');
            
            if (!$patientId) {
                echo json_encode([
                    'error' => 'Patient ID required',
                    'status' => 'error'
                ]);
                return;
            }
            
            // Get comprehensive patient data
            $this->db->select('
                id, patient_name, patient_unique_id, age, gender, mobileno, email,
                blood_group_id, known_allergies, address, guardian_name, note, image,
                fingerprint_template, fingerprint_image, created_at
            ');
            $this->db->from('patients');
            $this->db->where('id', $patientId);
            
            if ($this->db->field_exists('is_active', 'patients')) {
                $this->db->where('is_active', 'yes');
            }
            
            $query = $this->db->get();
            
            if ($query->num_rows() > 0) {
                $patient = $query->row_array();
                
                // Get blood group name if available
                if (!empty($patient['blood_group_id'])) {
                    $this->db->select('name');
                    $this->db->from('blood_group');
                    $this->db->where('id', $patient['blood_group_id']);
                    $bg_query = $this->db->get();
                    
                    if ($bg_query->num_rows() > 0) {
                        $patient['blood_group_name'] = $bg_query->row()->name;
                    } else {
                        $patient['blood_group_name'] = 'Unknown';
                    }
                } else {
                    $patient['blood_group_name'] = 'Unknown';
                }
                
                // Set default values for missing fields
                $patient['gender'] = $patient['gender'] ?? 'Unknown';
                $patient['known_allergies'] = $patient['known_allergies'] ?? 'None';
                $patient['image'] = !empty($patient['image']) ? $patient['image'] : 'uploads/patient_images/no_image.png';
                
                // Enhanced fingerprint verification status
                $hasFingerprint = !empty($patient['fingerprint_template']) && strlen($patient['fingerprint_template']) > 100;
                $requiresVerification = strpos(strtolower($patient['note'] ?? ''), 'gcc') !== false;
                
                $patient['has_fingerprint'] = $hasFingerprint;
                $patient['verification_capable'] = $hasFingerprint;
                $patient['workflow_requires_verification'] = $requiresVerification;
                $patient['workflow'] = $requiresVerification ? 'gcc' : 'general';
                
                // Add verification policy
                $patient['verification_policy'] = $requiresVerification ? 
                    'GCC patients require mandatory biometric verification for phlebotomy' : 
                    'General patients can proceed without verification';
                
                // Add current location and routing history
                $patient['current_location'] = 'phlebotomy';
                $patient['module'] = 'phlebotomy';
                
                // Remove sensitive data from response
                unset($patient['fingerprint_template']);
                if (isset($patient['fingerprint_image'])) {
                    unset($patient['fingerprint_image']);
                }
                
                echo json_encode($patient);
                
            } else {
                echo json_encode([
                    'error' => 'Patient not found or inactive',
                    'patient_id' => $patientId,
                    'status' => 'error'
                ]);
            }
            
        } catch (Exception $e) {
            log_message('error', 'Phlebotomy: Error getting patient details: ' . $e->getMessage());
            echo json_encode([
                'error' => 'Database error: ' . $e->getMessage(),
                'patient_id' => $patientId ?? 'unknown',
                'status' => 'error'
            ]);
        }
    }
    
    public function getStoredTemplate() {
    // This is the endpoint the JavaScript is actually calling
    $this->getStoredFingerprintTemplate();
}

/**
 * FIXED: Update getStoredFingerprintTemplate to match JavaScript expectations
 */
public function getStoredFingerprintTemplate() {
    header('Content-Type: application/json');
    
    try {
        $patientId = $this->input->post('patient_id');
        
        if (!$patientId) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Patient ID required'
            ]);
            return;
        }
        
        // Check if fingerprint columns exist
        if (!$this->db->field_exists('fingerprint_template', 'patients')) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Fingerprint functionality not available'
            ]);
            return;
        }
        
        // Get fingerprint data
        $this->db->select('id, patient_name, fingerprint_template, fingerprint_image');
        $this->db->from('patients');
        $this->db->where('id', $patientId);
        
        // Add is_active check if column exists
        if ($this->db->field_exists('is_active', 'patients')) {
            $this->db->where('is_active', 'yes');
        }
        
        $query = $this->db->get();
        
        if ($query->num_rows() > 0) {
            $patient = $query->row_array();
            
            if (!empty($patient['fingerprint_template'])) {
                $template = trim($patient['fingerprint_template']);
                
                // FIXED: Return field names that JavaScript expects
                echo json_encode([
                    'status' => 'success',
                    'patient_id' => $patientId,
                    'patient_name' => $patient['patient_name'],
                    'template' => $template, // JavaScript expects 'template', not 'fingerprint_template'
                    'image' => $patient['fingerprint_image'] ?? '', // JavaScript expects 'image', not 'fingerprint_image'
                    'template_size' => strlen($template),
                    'quality' => 85, // Add quality field that JavaScript expects
                    'has_image' => !empty($patient['fingerprint_image'])
                ]);
            } else {
                echo json_encode([
                    'status' => 'error',
                    'message' => 'No fingerprint template found for this patient',
                    'patient_id' => $patientId,
                    'patient_name' => $patient['patient_name']
                ]);
            }
        } else {
            echo json_encode([
                'status' => 'error',
                'message' => 'Patient not found or inactive',
                'patient_id' => $patientId
            ]);
        }
        
    } catch (Exception $e) {
        echo json_encode([
            'status' => 'error',
            'message' => 'Database error: ' . $e->getMessage()
        ]);
    }
}

    /**
     * Verify fingerprint match for phlebotomy access
     */
    public function verifyFingerprint() {
        header('Content-Type: application/json');
        
        try {
            $patientId = $this->input->post('patient_id');
            $matchScore = $this->input->post('match_score');
            $isMatch = $this->input->post('is_match');
            $threshold = $this->input->post('threshold') ?? 60;
            $liveTemplate = $this->input->post('live_template');
            
            if (!$patientId) {
                echo json_encode([
                    'status' => 'error',
                    'message' => 'Patient ID required for verification'
                ]);
                return;
            }
            
            if (!is_numeric($matchScore)) {
                echo json_encode([
                    'status' => 'error',
                    'message' => 'Invalid match score: ' . $matchScore
                ]);
                return;
            }
            
            // Get patient info
            $this->db->select('id, patient_name, fingerprint_template, note');
            $this->db->from('patients');
            $this->db->where('id', $patientId);
            $patient_query = $this->db->get();
            
            if ($patient_query->num_rows() == 0) {
                echo json_encode([
                    'status' => 'error',
                    'message' => 'Patient not found',
                    'patient_id' => $patientId
                ]);
                return;
            }
            
            $patient = $patient_query->row_array();
            $hasStoredTemplate = !empty($patient['fingerprint_template']) && strlen($patient['fingerprint_template']) > 100;
            
            // Log verification attempt
            $verificationLogId = $this->logVerificationAttempt($patientId, $matchScore, $isMatch, 'phlebotomy');
            
            // Prepare verification result
            $verificationResult = [
                'patient_id' => $patientId,
                'patient_name' => $patient['patient_name'],
                'match_score' => floatval($matchScore),
                'threshold' => intval($threshold),
                'is_match' => $isMatch,
                'timestamp' => date('Y-m-d H:i:s'),
                'verification_module' => 'phlebotomy',
                'service_url' => $this->input->post('service_url') ?: 'unknown',
                'verification_log_id' => $verificationLogId,
                'has_stored_template' => $hasStoredTemplate,
                'template_size' => $hasStoredTemplate ? strlen($patient['fingerprint_template']) : 0
            ];
            
            // Determine verification result
            $matchPassed = ($isMatch == 'true' || $isMatch === true || $isMatch == 1);
            
            if ($matchPassed && $hasStoredTemplate) {
                $verificationResult['status'] = 'success';
                $verificationResult['message'] = 'Patient identity verified successfully for phlebotomy';
                $verificationResult['verification_passed'] = true;
                $verificationResult['verification_method'] = 'fingerprint_match';
                $verificationResult['access_granted'] = true;
            } else if (!$hasStoredTemplate) {
                $verificationResult['status'] = 'warning';
                $verificationResult['message'] = 'No stored fingerprint available for verification';
                $verificationResult['verification_passed'] = false;
                $verificationResult['verification_method'] = 'no_template';
                $verificationResult['access_granted'] = false;
            } else {
                $verificationResult['status'] = 'failed';
                $verificationResult['message'] = 'Fingerprint verification failed - identity not confirmed';
                $verificationResult['verification_passed'] = false;
                $verificationResult['verification_method'] = 'fingerprint_mismatch';
                $verificationResult['access_granted'] = false;
            }
            
            echo json_encode($verificationResult);
            
        } catch (Exception $e) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Verification error: ' . $e->getMessage(),
                'patient_id' => $patientId ?? 'unknown',
                'verification_module' => 'phlebotomy'
            ]);
        }
    }

    // =============================================
    // ENHANCED PHLEBOTOMY DATA SAVE WITH VERIFICATION
    // =============================================

    /**
     * Save phlebotomy data with verification information
     */
    public function savePhlebotomyData() {
        header('Content-Type: application/json');
        
        try {
            // Get and validate input data
            $phlebotomy_data_json = $this->input->post('phlebotomy_data');
            $patient_id = $this->input->post('patient_id');
            $verification_data_json = $this->input->post('verification_data');
            
            if (empty($phlebotomy_data_json) || empty($patient_id)) {
                echo json_encode([
                    'status' => 0,
                    'message' => 'Missing required data: phlebotomy_data and patient_id are required',
                    'debug_data' => [
                        'phlebotomy_data_received' => !empty($phlebotomy_data_json),
                        'patient_id_received' => !empty($patient_id),
                        'verification_data_received' => !empty($verification_data_json)
                    ]
                ]);
                return;
            }
            
            // Decode JSON data
            $phlebotomy_data = json_decode($phlebotomy_data_json, true);
            $verification_data = !empty($verification_data_json) ? json_decode($verification_data_json, true) : null;
            
            if (json_last_error() !== JSON_ERROR_NONE) {
                echo json_encode([
                    'status' => 0,
                    'message' => 'Invalid JSON data: ' . json_last_error_msg(),
                    'json_error' => json_last_error()
                ]);
                return;
            }
            
            if (!is_array($phlebotomy_data) || empty($phlebotomy_data)) {
                echo json_encode([
                    'status' => 0,
                    'message' => 'No phlebotomy data to save',
                    'data_type' => gettype($phlebotomy_data)
                ]);
                return;
            }
            
            // Verify patient exists
            $patient_check = $this->db->get_where('patients', ['id' => $patient_id, 'is_active' => 'yes']);
            if ($patient_check->num_rows() == 0) {
                echo json_encode([
                    'status' => 0,
                    'message' => 'Patient not found or inactive',
                    'patient_id' => $patient_id
                ]);
                return;
            }
            
            $patient_info = $patient_check->row_array();
            
            // Check if verification was required and performed
            $requiresVerification = strpos(strtolower($patient_info['note'] ?? ''), 'gcc') !== false;
            $verificationPassed = false;
            
            if ($requiresVerification) {
                if (empty($verification_data) || !$verification_data['verification_passed']) {
                    echo json_encode([
                        'status' => 0,
                        'message' => 'Fingerprint verification required for GCC patients before phlebotomy',
                        'requires_verification' => true,
                        'verification_status' => 'required_not_performed'
                    ]);
                    return;
                } else {
                    $verificationPassed = true;
                }
            }
            
            // Get current staff/admin ID
            $collected_by = $this->session->userdata('admin_id') ?: 1;
            
            // Start database transaction
            $this->db->trans_start();
            
            $saved_records = [];
            $total_saved = 0;
            
            // Process each sample
            foreach ($phlebotomy_data as $index => $sample_data) {
                try {
                    // Validate required fields
                    if (empty($sample_data['sample_type'])) {
                        log_message('error', "Sample #{$index}: Missing sample_type");
                        continue;
                    }
                    
                    // Prepare data for phlebotomy table
                    $insert_data = [
                        'patient_id' => intval($patient_id),
                        'sample_type' => $this->db->escape_str($sample_data['sample_type']),
                        'barcode' => !empty($sample_data['barcode']) ? $this->db->escape_str($sample_data['barcode']) : $this->generateBarcode($sample_data['sample_type']),
                        'collection_date' => !empty($sample_data['collection_date']) ? date('Y-m-d H:i:s', strtotime($sample_data['collection_date'])) : date('Y-m-d H:i:s'),
                        'collected_by' => intval($collected_by),
                        'status' => !empty($sample_data['status']) ? $this->db->escape_str($sample_data['status']) : 'collected',
                        'created_at' => date('Y-m-d H:i:s')
                    ];
                    
                    // Enhanced notes with verification information
                    $notes_data = [
                        'collection_method' => $sample_data['collection_method'] ?? 'standard',
                        'volume' => $sample_data['volume'] ?? null,
                        'notes' => $sample_data['notes'] ?? '',
                        'collected_at' => date('Y-m-d H:i:s'),
                        'module' => 'phlebotomy'
                    ];
                    
                    // Add verification data if available
                    if ($verification_data) {
                        $notes_data['verification'] = [
                            'required' => $requiresVerification,
                            'performed' => $verificationPassed,
                            'method' => $verification_data['verification_method'] ?? 'fingerprint',
                            'match_score' => $verification_data['match_score'] ?? null,
                            'verified_at' => $verification_data['timestamp'] ?? date('Y-m-d H:i:s'),
                            'verification_id' => $verification_data['verification_log_id'] ?? null
                        ];
                    }
                    
                    $insert_data['notes'] = json_encode($notes_data);
                    
                    // Insert into phlebotomy table
                    $insert_result = $this->db->insert('phlebotomy', $insert_data);
                    
                    if ($insert_result) {
                        $insert_id = $this->db->insert_id();
                        $saved_records[] = [
                            'id' => $insert_id,
                            'sample_type' => $sample_data['sample_type'],
                            'barcode' => $insert_data['barcode'],
                            'status' => 'saved',
                            'verification_required' => $requiresVerification,
                            'verification_passed' => $verificationPassed
                        ];
                        $total_saved++;
                        
                        log_message('info', "Phlebotomy record saved: ID {$insert_id}, Sample: {$sample_data['sample_type']}, Patient: {$patient_id}, Verified: " . ($verificationPassed ? 'Yes' : 'No'));
                    } else {
                        log_message('error', "Failed to save phlebotomy record for sample: {$sample_data['sample_type']}");
                    }
                    
                } catch (Exception $sample_error) {
                    log_message('error', "Error processing sample #{$index}: " . $sample_error->getMessage());
                    continue;
                }
            }
            
            // Complete transaction
            $this->db->trans_complete();
            
            if ($this->db->trans_status() === FALSE) {
                // Transaction failed
                echo json_encode([
                    'status' => 0,
                    'message' => 'Database transaction failed',
                    'saved_count' => 0,
                    'transaction_error' => true
                ]);
                return;
            }
            
            // Success response
            echo json_encode([
                'status' => 1,
                'message' => "Successfully saved {$total_saved} phlebotomy record(s) to database",
                'saved_count' => $total_saved,
                'patient_name' => $patient_info['patient_name'],
                'patient_id' => $patient_id,
                'saved_records' => $saved_records,
                'timestamp' => date('Y-m-d H:i:s'),
                'collected_by' => $collected_by,
                'verification_required' => $requiresVerification,
                'verification_passed' => $verificationPassed,
                'module' => 'phlebotomy'
            ]);
            
        } catch (Exception $e) {
            // Rollback transaction on error
            $this->db->trans_rollback();
            
            log_message('error', 'savePhlebotomyData error: ' . $e->getMessage());
            
            echo json_encode([
                'status' => 0,
                'message' => 'Error saving phlebotomy data: ' . $e->getMessage(),
                'error_type' => 'exception',
                'error_details' => $e->getMessage(),
                'module' => 'phlebotomy'
            ]);
        }
    }

    // =============================================
    // ENHANCED SAMPLE COLLECTION METHODS
    // =============================================

    /**
     * Save individual sample collection with verification data
     */
    public function saveSampleCollection() {
        header('Content-Type: application/json');
        
        try {
            $patient_id = $this->input->post('patient_id');
            $sample_type = $this->input->post('sample_type');
            $collection_method = $this->input->post('collection_method');
            $volume = $this->input->post('volume');
            $notes = $this->input->post('notes');
            $barcode = $this->input->post('barcode');
            $verification_data_json = $this->input->post('verification_data');
            
            if (empty($patient_id) || empty($sample_type)) {
                echo json_encode([
                    'status' => 0,
                    'message' => 'Patient ID and sample type are required'
                ]);
                return;
            }
            
            // Decode verification data if provided
            $verification_data = !empty($verification_data_json) ? json_decode($verification_data_json, true) : null;
            
            // Prepare collection data
            $collection_data = [
                'patient_id' => intval($patient_id),
                'sample_type' => $this->db->escape_str($sample_type),
                'barcode' => !empty($barcode) ? $this->db->escape_str($barcode) : $this->generateBarcode($sample_type),
                'collection_date' => date('Y-m-d H:i:s'),
                'collected_by' => $this->session->userdata('admin_id') ?: 1,
                'status' => 'collected',
                'created_at' => date('Y-m-d H:i:s')
            ];
            
            // Enhanced notes with verification information
            $notes_data = [
                'collection_method' => $collection_method,
                'volume' => $volume,
                'notes' => $notes,
                'collected_at' => date('Y-m-d H:i:s'),
                'module' => 'phlebotomy'
            ];
            
            // Add verification data if available
            if ($verification_data) {
                $notes_data['verification'] = $verification_data;
            }
            
            $collection_data['notes'] = json_encode($notes_data);
            
            $result = $this->db->insert('phlebotomy', $collection_data);
            
            if ($result) {
                echo json_encode([
                    'status' => 1,
                    'message' => 'Sample collection saved successfully with verification data',
                    'sample_id' => $this->db->insert_id(),
                    'sample_type' => $sample_type,
                    'barcode' => $collection_data['barcode'],
                    'verification_included' => !empty($verification_data),
                    'module' => 'phlebotomy'
                ]);
            } else {
                echo json_encode([
                    'status' => 0,
                    'message' => 'Failed to save sample collection'
                ]);
            }
            
        } catch (Exception $e) {
            echo json_encode([
                'status' => 0,
                'message' => 'Error saving sample collection: ' . $e->getMessage(),
                'module' => 'phlebotomy'
            ]);
        }
    }

    /**
     * Generate enhanced barcodes with patient and verification information
     */
    public function generateBarcodes() {
        header('Content-Type: application/json');
        
        try {
            $patient_id = $this->input->post('patient_id');
            $samples = $this->input->post('samples');
            $verification_status = $this->input->post('verification_status') ?: 'unverified';
            
            if (empty($patient_id) || empty($samples)) {
                echo json_encode([
                    'status' => 0,
                    'message' => 'Patient ID and samples are required'
                ]);
                return;
            }
            
            // Get patient information for barcode
            $patient_query = $this->db->get_where('patients', ['id' => $patient_id]);
            if ($patient_query->num_rows() == 0) {
                echo json_encode([
                    'status' => 0,
                    'message' => 'Patient not found'
                ]);
                return;
            }
            
            $patient = $patient_query->row_array();
            $patient_name = $patient['patient_name'];
            $patient_unique_id = $patient['patient_unique_id'] ?: $patient['id'];
            
            $generated_barcodes = [];
            
            foreach ($samples as $index => $sample_type) {
                $barcode = $this->generateEnhancedBarcode($sample_type, $patient_name, $patient_unique_id, $index + 1, $verification_status);
                $generated_barcodes[$sample_type] = $barcode;
            }
            
            echo json_encode([
                'status' => 1,
                'message' => 'Barcodes generated successfully with patient and verification information',
                'barcodes' => $generated_barcodes,
                'count' => count($generated_barcodes),
                'patient_name' => $patient_name,
                'patient_id' => $patient_unique_id,
                'verification_status' => $verification_status,
                'module' => 'phlebotomy'
            ]);
            
        } catch (Exception $e) {
            echo json_encode([
                'status' => 0,
                'message' => 'Error generating barcodes: ' . $e->getMessage(),
                'module' => 'phlebotomy'
            ]);
        }
    }

    /**
     * Complete phlebotomy process with verification status
     */
    public function completeProcess() {
        header('Content-Type: application/json');
        
        try {
            $patient_id = $this->input->post('patient_id');
            $verification_data_json = $this->input->post('verification_data');
            
            if (empty($patient_id)) {
                echo json_encode([
                    'status' => 0,
                    'message' => 'Patient ID is required'
                ]);
                return;
            }
            
            // Decode verification data if provided
            $verification_data = !empty($verification_data_json) ? json_decode($verification_data_json, true) : null;
            
            // Update all phlebotomy records for this patient to completed status
            $update_data = [
                'status' => 'completed',
                'updated_at' => date('Y-m-d H:i:s')
            ];
            
            // Add completion notes with verification information
            $completion_notes = [
                'completed_at' => date('Y-m-d H:i:s'),
                'completed_by' => $this->session->userdata('admin_id') ?: 1,
                'module' => 'phlebotomy'
            ];
            
            if ($verification_data) {
                $completion_notes['verification_at_completion'] = $verification_data;
            }
            
            $this->db->where('patient_id', $patient_id);
            $this->db->where('status', 'collected');
            $updated = $this->db->update('phlebotomy', $update_data);
            
            if ($updated) {
                echo json_encode([
                    'status' => 1,
                    'message' => 'Phlebotomy process completed successfully',
                    'completion_id' => rand(1000, 9999),
                    'next_step' => 'radiology',
                    'updated_records' => $this->db->affected_rows(),
                    'verification_included' => !empty($verification_data),
                    'module' => 'phlebotomy'
                ]);
            } else {
                echo json_encode([
                    'status' => 0,
                    'message' => 'No records to complete or update failed'
                ]);
            }
            
        } catch (Exception $e) {
            echo json_encode([
                'status' => 0,
                'message' => 'Error completing process: ' . $e->getMessage(),
                'module' => 'phlebotomy'
            ]);
        }
    }

    // =============================================
    // VERIFICATION LOGGING AND AUDIT METHODS
    // =============================================

    /**
     * Log verification attempt for audit trail
     */
    private function logVerificationAttempt($patient_id, $match_score, $is_match, $module = 'phlebotomy') {
        try {
            // Check if verification_log table exists, create if not
            if (!$this->db->table_exists('verification_log')) {
                $this->createVerificationLogTable();
            }
            
            $log_data = [
                'patient_id' => intval($patient_id),
                'module' => $this->db->escape_str($module),
                'match_score' => floatval($match_score),
                'is_match' => $is_match == 'true' || $is_match === true || $is_match == 1 ? 1 : 0,
                'verified_by' => $this->session->userdata('admin_id') ?: 1,
                'verification_timestamp' => date('Y-m-d H:i:s'),
                'ip_address' => $this->input->ip_address(),
                'user_agent' => $this->input->user_agent()
            ];
            
            $this->db->insert('verification_log', $log_data);
            return $this->db->insert_id();
            
        } catch (Exception $e) {
            log_message('error', 'Failed to log verification attempt: ' . $e->getMessage());
            return null;
        }
    }

    /**
     * Create verification log table if it doesn't exist
     */
    private function createVerificationLogTable() {
        $sql = "
            CREATE TABLE IF NOT EXISTS `verification_log` (
                `id` int NOT NULL AUTO_INCREMENT,
                `patient_id` int NOT NULL,
                `module` varchar(50) NOT NULL,
                `match_score` decimal(5,2) DEFAULT NULL,
                `is_match` tinyint(1) NOT NULL DEFAULT '0',
                `verified_by` int NOT NULL,
                `verification_timestamp` datetime NOT NULL,
                `ip_address` varchar(45) DEFAULT NULL,
                `user_agent` text,
                `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
                PRIMARY KEY (`id`),
                KEY `idx_patient_id` (`patient_id`),
                KEY `idx_module` (`module`),
                KEY `idx_verification_timestamp` (`verification_timestamp`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb3;
        ";
        
        $this->db->query($sql);
    }

    /**
     * Get verification history for a patient
     */
    public function getVerificationHistory() {
        header('Content-Type: application/json');
        
        try {
            $patient_id = $this->input->get('patient_id');
            $module = $this->input->get('module') ?: 'phlebotomy';
            
            if (empty($patient_id)) {
                echo json_encode([
                    'status' => 0,
                    'message' => 'Patient ID is required'
                ]);
                return;
            }
            
            if (!$this->db->table_exists('verification_log')) {
                echo json_encode([
                    'status' => 0,
                    'message' => 'Verification logging not available',
                    'history' => []
                ]);
                return;
            }
            
            $this->db->select('
                vl.*,
                s.name as verified_by_name,
                p.patient_name
            ');
            $this->db->from('verification_log vl');
            $this->db->join('staff s', 'vl.verified_by = s.id', 'left');
            $this->db->join('patients p', 'vl.patient_id = p.id', 'left');
            $this->db->where('vl.patient_id', $patient_id);
            $this->db->where('vl.module', $module);
            $this->db->order_by('vl.verification_timestamp', 'DESC');
            $this->db->limit(20);
            
            $query = $this->db->get();
            $history = $query->result_array();
            
            echo json_encode([
                'status' => 1,
                'message' => 'Verification history retrieved successfully',
                'patient_id' => $patient_id,
                'module' => $module,
                'history' => $history,
                'count' => count($history)
            ]);
            
        } catch (Exception $e) {
            echo json_encode([
                'status' => 0,
                'message' => 'Error retrieving verification history: ' . $e->getMessage()
            ]);
        }
    }

    // =============================================
    // REPORTING AND ANALYTICS METHODS
    // =============================================

    /**
     * Get phlebotomy statistics with verification data
     */
    public function getPhlebotomyStats() {
        header('Content-Type: application/json');
        
        try {
            $date_from = $this->input->get('date_from') ?: date('Y-m-01'); // Start of current month
            $date_to = $this->input->get('date_to') ?: date('Y-m-d'); // Today
            
            // Total samples collected
            $total_query = $this->db->query("
                SELECT COUNT(*) as total_samples 
                FROM phlebotomy 
                WHERE DATE(collection_date) BETWEEN ? AND ?
            ", [$date_from, $date_to]);
            $total_samples = $total_query->row()->total_samples;
            
            // Samples by type
            $by_type_query = $this->db->query("
                SELECT sample_type, COUNT(*) as count 
                FROM phlebotomy 
                WHERE DATE(collection_date) BETWEEN ? AND ?
                GROUP BY sample_type
                ORDER BY count DESC
            ", [$date_from, $date_to]);
            $samples_by_type = $by_type_query->result_array();
            
            // Daily collection trends
            $daily_query = $this->db->query("
                SELECT DATE(collection_date) as collection_date, COUNT(*) as count 
                FROM phlebotomy 
                WHERE DATE(collection_date) BETWEEN ? AND ?
                GROUP BY DATE(collection_date)
                ORDER BY collection_date ASC
            ", [$date_from, $date_to]);
            $daily_trends = $daily_query->result_array();
            
            // Verification statistics (if verification_log table exists)
            $verification_stats = [];
            if ($this->db->table_exists('verification_log')) {
                $verification_query = $this->db->query("
                    SELECT 
                        COUNT(*) as total_verifications,
                        SUM(CASE WHEN is_match = 1 THEN 1 ELSE 0 END) as successful_verifications,
                        AVG(match_score) as avg_match_score
                    FROM verification_log 
                    WHERE module = 'phlebotomy' 
                    AND DATE(verification_timestamp) BETWEEN ? AND ?
                ", [$date_from, $date_to]);
                
                if ($verification_query->num_rows() > 0) {
                    $verification_stats = $verification_query->row_array();
                }
            }
            
            echo json_encode([
                'status' => 1,
                'message' => 'Phlebotomy statistics retrieved successfully',
                'date_range' => ['from' => $date_from, 'to' => $date_to],
                'stats' => [
                    'total_samples' => intval($total_samples),
                    'samples_by_type' => $samples_by_type,
                    'daily_trends' => $daily_trends,
                    'verification_stats' => $verification_stats
                ]
            ]);
            
        } catch (Exception $e) {
            echo json_encode([
                'status' => 0,
                'message' => 'Error retrieving statistics: ' . $e->getMessage()
            ]);
        }
    }

    /**
     * Get patient's phlebotomy history with verification data
     */
    public function getPatientHistory() {
        header('Content-Type: application/json');
        
        try {
            $patient_id = $this->input->get('patient_id');
            
            if (empty($patient_id)) {
                echo json_encode([
                    'status' => 0,
                    'message' => 'Patient ID is required'
                ]);
                return;
            }
            
            $history_query = $this->db->query("
                SELECT 
                    ph.*,
                    s.name as collected_by_name,
                    p.patient_name
                FROM phlebotomy ph
                LEFT JOIN staff s ON ph.collected_by = s.id
                LEFT JOIN patients p ON ph.patient_id = p.id
                WHERE ph.patient_id = ?
                ORDER BY ph.collection_date DESC
            ", [$patient_id]);
            
            $history = $history_query->result_array();
            
            // Parse notes to extract verification information
            foreach ($history as &$record) {
                if (!empty($record['notes'])) {
                    $notes_data = json_decode($record['notes'], true);
                    if ($notes_data && isset($notes_data['verification'])) {
                        $record['verification_data'] = $notes_data['verification'];
                    }
                }
            }
            
            echo json_encode([
                'status' => 1,
                'message' => 'Patient phlebotomy history retrieved successfully',
                'patient_id' => $patient_id,
                'history' => $history,
                'count' => count($history)
            ]);
            
        } catch (Exception $e) {
            echo json_encode([
                'status' => 0,
                'message' => 'Error retrieving patient history: ' . $e->getMessage()
            ]);
        }
    }

    // =============================================
    // UTILITY METHODS
    // =============================================

    /**
     * Generate simple barcode
     */
    private function generateBarcode($sample_type) {
        $prefix = strtoupper(substr($sample_type, 0, 3));
        $date = date('Ymd');
        $sequence = str_pad(rand(1, 999), 3, '0', STR_PAD_LEFT);
        return sprintf('%s-%s-%s', $prefix, $date, $sequence);
    }

    /**
     * Generate enhanced barcode with patient and verification information
     */
    private function generateEnhancedBarcode($sample_type, $patient_name, $patient_id, $sequence = 1, $verification_status = 'unverified') {
        $prefix = strtoupper(substr($sample_type, 0, 3));
        $date = date('Ymd');
        $seq = str_pad($sequence, 3, '0', STR_PAD_LEFT);
        $verify_code = strtoupper(substr($verification_status, 0, 1)); // V for verified, U for unverified
        $barcode_number = sprintf('%s-%s-%s-%s', $prefix, $date, $seq, $verify_code);
        
        // Return barcode with patient info (format: BARCODE|PATIENT_NAME|PATIENT_ID|VERIFICATION_STATUS)
        return $barcode_number . '|' . $patient_name . '|' . $patient_id . '|' . $verification_status;
    }

    /**
     * Calculate time since a given datetime
     */
    private function getTimeSince($datetime) {
        $time = time() - strtotime($datetime);
        
        if ($time < 60) return 'Just now';
        if ($time < 3600) return floor($time/60) . ' minutes ago';
        if ($time < 86400) return floor($time/3600) . ' hours ago';
        return floor($time/86400) . ' days ago';
    }

    /**
     * Validate phlebotomy data structure
     */
    private function validatePhlebotomyData($data) {
        $required_fields = ['patient_id', 'sample_type'];
        $valid_sample_types = ['blood', 'stool', 'urine'];
        $valid_statuses = ['pending', 'collected', 'processing', 'completed'];
        
        foreach ($required_fields as $field) {
            if (!isset($data[$field]) || empty($data[$field])) {
                return ['valid' => false, 'message' => "Missing required field: {$field}"];
            }
        }
        
        if (!in_array(strtolower($data['sample_type']), $valid_sample_types)) {
            return ['valid' => false, 'message' => 'Invalid sample type. Must be: ' . implode(', ', $valid_sample_types)];
        }
        
        if (isset($data['status']) && !in_array(strtolower($data['status']), $valid_statuses)) {
            return ['valid' => false, 'message' => 'Invalid status. Must be: ' . implode(', ', $valid_statuses)];
        }
        
        return ['valid' => true, 'message' => 'Data is valid'];
    }

    // =============================================
    // DEBUG AND TESTING METHODS
    // =============================================

    /**
     * Enhanced debug method with fingerprint verification testing
     */
    public function debug() {
        echo "<h1>ENHANCED Phlebotomy Debug - With Fingerprint Verification</h1>";
        
        echo "<h2>1. System Information</h2>";
        echo "<p><strong>PHP Version:</strong> " . phpversion() . "</p>";
        echo "<p><strong>CodeIgniter Version:</strong> " . CI_VERSION . "</p>";
        echo "<p><strong>Current Time:</strong> " . date('Y-m-d H:i:s') . "</p>";
        echo "<p><strong>Base URL:</strong> " . base_url() . "</p>";
        
        echo "<h2>2. Database Connection Test</h2>";
        try {
            if ($this->db->initialize()) {
                echo "<p style='color: green;'>✅ Database connected successfully</p>";
                
                // Test database version
                $version_query = $this->db->query("SELECT VERSION() as version");
                if ($version_query) {
                    $version = $version_query->row()->version;
                    echo "<p><strong>Database Version:</strong> {$version}</p>";
                }
            } else {
                echo "<p style='color: red;'>❌ Database connection failed</p>";
            }
        } catch (Exception $e) {
            echo "<p style='color: red;'>❌ Database error: " . $e->getMessage() . "</p>";
        }
        
        echo "<h2>3. Table Structure Verification</h2>";
        $tables = ['patients', 'phlebotomy', 'staff', 'blood_group', 'verification_log'];
        foreach ($tables as $table) {
            echo "<h3>{$table} table:</h3>";
            if ($this->db->table_exists($table)) {
                echo "<p style='color: green;'>✅ Table exists</p>";
                
                try {
                    $count_query = $this->db->query("SELECT COUNT(*) as count FROM {$table}");
                    if ($count_query) {
                        $count = $count_query->row()->count;
                        echo "<p><strong>Records:</strong> {$count}</p>";
                        
                        // Show table structure for important tables
                        if (in_array($table, ['phlebotomy', 'verification_log'])) {
                            $fields = $this->db->list_fields($table);
                            echo "<p><strong>Columns:</strong> " . implode(', ', $fields) . "</p>";
                        }
                    }
                } catch (Exception $e) {
                    echo "<p style='color: orange;'>⚠️ Count error: " . $e->getMessage() . "</p>";
                }
            } else {
                echo "<p style='color: red;'>❌ Table does not exist</p>";
                
                // Show CREATE statements for missing tables
                if ($table == 'phlebotomy') {
                    echo "<h4>Required CREATE statement:</h4>";
                    echo "<pre style='background: #f4f4f4; padding: 10px; border-radius: 5px;'>";
                    echo "CREATE TABLE `phlebotomy` (
  `id` int NOT NULL AUTO_INCREMENT,
  `patient_id` int NOT NULL,
  `barcode` varchar(50) NOT NULL,
  `sample_type` varchar(100) NOT NULL,
  `collection_date` datetime NOT NULL,
  `collected_by` int NOT NULL,
  `status` enum('pending','collected','processing','completed') NOT NULL DEFAULT 'pending',
  `notes` text,
  `created_at` datetime NOT NULL,
  `updated_at` datetime DEFAULT NULL,
  PRIMARY KEY (`id`),
  KEY `idx_patient_id` (`patient_id`),
  KEY `idx_collection_date` (`collection_date`),
  KEY `idx_status` (`status`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb3;";
                    echo "</pre>";
                } elseif ($table == 'verification_log') {
                    echo "<h4>Required CREATE statement:</h4>";
                    echo "<pre style='background: #f4f4f4; padding: 10px; border-radius: 5px;'>";
                    echo "CREATE TABLE `verification_log` (
  `id` int NOT NULL AUTO_INCREMENT,
  `patient_id` int NOT NULL,
  `module` varchar(50) NOT NULL,
  `match_score` decimal(5,2) DEFAULT NULL,
  `is_match` tinyint(1) NOT NULL DEFAULT '0',
  `verified_by` int NOT NULL,
  `verification_timestamp` datetime NOT NULL,
  `ip_address` varchar(45) DEFAULT NULL,
  `user_agent` text,
  `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  KEY `idx_patient_id` (`patient_id`),
  KEY `idx_module` (`module`),
  KEY `idx_verification_timestamp` (`verification_timestamp`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb3;";
                    echo "</pre>";
                }
            }
        }
        
        echo "<h2>4. Fingerprint Capability Test</h2>";
        $fingerprint_fields = ['fingerprint_template', 'fingerprint_image'];
        foreach ($fingerprint_fields as $field) {
            if ($this->db->field_exists($field, 'patients')) {
                echo "<p style='color: green;'>✅ {$field} column exists</p>";
            } else {
                echo "<p style='color: red;'>❌ {$field} column missing</p>";
            }
        }
        
        // Check for patients with fingerprints
        if ($this->db->field_exists('fingerprint_template', 'patients')) {
            $fp_query = $this->db->query("
                SELECT COUNT(*) as count 
                FROM patients 
                WHERE fingerprint_template IS NOT NULL 
                AND LENGTH(fingerprint_template) > 100
            ");
            $fp_count = $fp_query->row()->count;
            echo "<p><strong>Patients with fingerprints:</strong> {$fp_count}</p>";
            
            if ($fp_count > 0) {
                // Show sample patients with fingerprints
                $sample_query = $this->db->query("
                    SELECT id, patient_name, LENGTH(fingerprint_template) as template_length,
                    CASE WHEN note LIKE '%gcc%' OR note LIKE '%GCC%' THEN 'Yes' ELSE 'No' END as requires_verification
                    FROM patients 
                    WHERE fingerprint_template IS NOT NULL 
                    AND LENGTH(fingerprint_template) > 100
                    LIMIT 5
                ");
                
                echo "<h4>Sample patients with fingerprints:</h4>";
                echo "<table border='1' style='border-collapse: collapse;'>";
                echo "<tr><th>ID</th><th>Name</th><th>Template Length</th><th>Requires Verification</th><th>Test Template</th></tr>";
                foreach ($sample_query->result_array() as $patient) {
                    echo "<tr>";
                    echo "<td>{$patient['id']}</td>";
                    echo "<td>{$patient['patient_name']}</td>";
                    echo "<td>{$patient['template_length']} chars</td>";
                    echo "<td>{$patient['requires_verification']}</td>";
                    echo "<td><a href='" . base_url('admin/phlebotomy/testTemplate/' . $patient['id']) . "' target='_blank'>Test</a></td>";
                    echo "</tr>";
                }
                echo "</table>";
            }
        }
        
        echo "<h2>5. Enhanced Endpoint Testing</h2>";
        echo "<h3>Required Phlebotomy Endpoints:</h3>";
        $endpoints = [
            'getTriagePatients' => 'Load patients with fingerprint data',
            'getStoredTemplate' => 'Fetch fingerprint templates',
            'getPatientDetails' => 'Fetch patient photos and details',
            'savePhlebotomyData' => 'Save with verification data',
            'verifyFingerprint' => 'Verify fingerprint match',
            'getVerificationHistory' => 'Get verification audit trail'
        ];
        
        echo "<ul>";
        foreach ($endpoints as $endpoint => $description) {
            $url = base_url("admin/phlebotomy/{$endpoint}");
            echo "<li><strong>{$endpoint}</strong> - {$description}<br>";
            echo "<a href='{$url}' target='_blank'>{$url}</a></li><br>";
        }
        echo "</ul>";
        
        echo "<h2>6. Real Patient Data Test with Fingerprint Info</h2>";
        try {
            $this->db->select('
                id, patient_name, patient_unique_id, age, gender, mobileno,
                CASE WHEN fingerprint_template IS NOT NULL AND LENGTH(fingerprint_template) > 100 THEN 1 ELSE 0 END as has_fingerprint,
                CASE WHEN note LIKE "%gcc%" OR note LIKE "%GCC%" THEN 1 ELSE 0 END as requires_verification
            ');
            $this->db->from('patients');
            $this->db->where('is_active', 'yes');
            $this->db->limit(5);
            $query = $this->db->get();
            
            if ($query && $query->num_rows() > 0) {
                $patients = $query->result_array();
                echo "<p><strong>Sample Enhanced Patient Data:</strong> " . count($patients) . " found</p>";
                echo "<p><strong>Last Query:</strong> " . htmlspecialchars($this->db->last_query()) . "</p>";
                
                echo "<table border='1' style='border-collapse: collapse; width: 100%;'>";
                echo "<tr><th>ID</th><th>Name</th><th>Age</th><th>Gender</th><th>Mobile</th><th>Has Fingerprint</th><th>Requires Verification</th><th>Actions</th></tr>";
                foreach ($patients as $patient) {
                    $has_fp = $patient['has_fingerprint'] ? '✅ Yes' : '❌ No';
                    $req_verify = $patient['requires_verification'] ? '🔒 Yes' : '🔓 No';
                    
                    echo "<tr>";
                    echo "<td>{$patient['id']}</td>";
                    echo "<td>" . ($patient['patient_name'] ?: 'N/A') . "</td>";
                    echo "<td>" . ($patient['age'] ?: 'N/A') . "</td>";
                    echo "<td>" . ($patient['gender'] ?: 'N/A') . "</td>";
                    echo "<td>" . ($patient['mobileno'] ?: 'N/A') . "</td>";
                    echo "<td>{$has_fp}</td>";
                    echo "<td>{$req_verify}</td>";
                    echo "<td>";
                    echo "<a href='" . base_url('admin/phlebotomy/testPatientEndpoints/' . $patient['id']) . "' target='_blank'>Test All</a> | ";
                    if ($patient['has_fingerprint']) {
                        echo "<a href='" . base_url('admin/phlebotomy/testTemplate/' . $patient['id']) . "' target='_blank'>Template</a>";
                    } else {
                        echo "No Template";
                    }
                    echo "</td>";
                    echo "</tr>";
                }
                echo "</table>";
            } else {
                echo "<p style='color: orange;'>⚠️ No real patients found in database</p>";
            }
            
        } catch (Exception $e) {
            echo "<p style='color: red;'>❌ Patient query test failed: " . $e->getMessage() . "</p>";
        }
        
        echo "<h2>7. Phlebotomy Data Test with Verification</h2>";
        try {
            if ($this->db->table_exists('phlebotomy')) {
                $phlebotomy_query = $this->db->query("
                    SELECT 
                        ph.*,
                        p.patient_name,
                        s.name as collected_by_name
                    FROM phlebotomy ph
                    LEFT JOIN patients p ON ph.patient_id = p.id
                    LEFT JOIN staff s ON ph.collected_by = s.id
                    ORDER BY ph.id DESC
                    LIMIT 5
                ");
                
                if ($phlebotomy_query && $phlebotomy_query->num_rows() > 0) {
                    $phlebotomy_data = $phlebotomy_query->result_array();
                    echo "<p><strong>Recent Phlebotomy Records:</strong> " . count($phlebotomy_data) . " found</p>";
                    
                    echo "<table border='1' style='border-collapse: collapse; width: 100%;'>";
                    echo "<tr><th>ID</th><th>Patient</th><th>Sample Type</th><th>Barcode</th><th>Status</th><th>Collection Date</th><th>Has Verification</th></tr>";
                    foreach ($phlebotomy_data as $record) {
                        $has_verification = 'No';
                        if (!empty($record['notes'])) {
                            $notes = json_decode($record['notes'], true);
                            if ($notes && isset($notes['verification'])) {
                                $has_verification = '✅ Yes';
                            }
                        }
                        
                        echo "<tr>";
                        echo "<td>{$record['id']}</td>";
                        echo "<td>" . ($record['patient_name'] ?: 'Unknown') . "</td>";
                        echo "<td>{$record['sample_type']}</td>";
                        echo "<td>{$record['barcode']}</td>";
                        echo "<td>{$record['status']}</td>";
                        echo "<td>{$record['collection_date']}</td>";
                        echo "<td>{$has_verification}</td>";
                        echo "</tr>";
                    }
                    echo "</table>";
                } else {
                    echo "<p style='color: orange;'>⚠️ No phlebotomy records found</p>";
                }
            } else {
                echo "<p style='color: red;'>❌ Phlebotomy table does not exist</p>";
            }
            
        } catch (Exception $e) {
            echo "<p style='color: red;'>❌ Phlebotomy data test failed: " . $e->getMessage() . "</p>";
        }
        
        echo "<h2>8. Verification Log Test</h2>";
        try {
            if ($this->db->table_exists('verification_log')) {
                $verification_query = $this->db->query("
                    SELECT 
                        vl.*,
                        p.patient_name,
                        s.name as verified_by_name
                    FROM verification_log vl
                    LEFT JOIN patients p ON vl.patient_id = p.id
                    LEFT JOIN staff s ON vl.verified_by = s.id
                    WHERE vl.module = 'phlebotomy'
                    ORDER BY vl.id DESC
                    LIMIT 5
                ");
                
                if ($verification_query && $verification_query->num_rows() > 0) {
                    $verification_data = $verification_query->result_array();
                    echo "<p><strong>Recent Verification Logs:</strong> " . count($verification_data) . " found</p>";
                    
                    echo "<table border='1' style='border-collapse: collapse; width: 100%;'>";
                    echo "<tr><th>ID</th><th>Patient</th><th>Match Score</th><th>Success</th><th>Verified By</th><th>Timestamp</th></tr>";
                    foreach ($verification_data as $record) {
                        $success = $record['is_match'] ? '✅ Yes' : '❌ No';
                        
                        echo "<tr>";
                        echo "<td>{$record['id']}</td>";
                        echo "<td>" . ($record['patient_name'] ?: 'Unknown') . "</td>";
                        echo "<td>{$record['match_score']}%</td>";
                        echo "<td>{$success}</td>";
                        echo "<td>" . ($record['verified_by_name'] ?: 'Unknown') . "</td>";
                        echo "<td>{$record['verification_timestamp']}</td>";
                        echo "</tr>";
                    }
                    echo "</table>";
                } else {
                    echo "<p style='color: orange;'>⚠️ No verification logs found for phlebotomy module</p>";
                }
            } else {
                echo "<p style='color: orange;'>⚠️ Verification log table does not exist (will be created automatically)</p>";
            }
            
        } catch (Exception $e) {
            echo "<p style='color: red;'>❌ Verification log test failed: " . $e->getMessage() . "</p>";
        }
        
        echo "<h2>9. Quick Test Links</h2>";
        echo "<p><a href='" . base_url('admin/phlebotomy/testPatientLoad') . "' target='_blank'>🔍 Test Enhanced Patient Loading</a></p>";
        echo "<p><a href='" . base_url('admin/phlebotomy/testFingerprintEndpoints') . "' target='_blank'>🔐 Test Fingerprint Endpoints</a></p>";
        echo "<p><a href='" . base_url('admin/phlebotomy/testDatabaseSave') . "' target='_blank'>💾 Test Database Save with Verification</a></p>";
        echo "<p><a href='" . base_url('admin/phlebotomy/getPhlebotomyStats') . "' target='_blank'>📊 View Enhanced Statistics</a></p>";
        
        echo "<h2>10. System Status Summary</h2>";
        $status_checks = [
            'Database Connected' => $this->db->initialize(),
            'Patients Table Exists' => $this->db->table_exists('patients'),
            'Phlebotomy Table Exists' => $this->db->table_exists('phlebotomy'),
            'Staff Table Exists' => $this->db->table_exists('staff'),
            'Fingerprint Template Column' => $this->db->field_exists('fingerprint_template', 'patients'),
            'Fingerprint Image Column' => $this->db->field_exists('fingerprint_image', 'patients'),
            'Verification Log Table' => $this->db->table_exists('verification_log')
        ];
        
        echo "<ul>";
        foreach ($status_checks as $check => $status) {
            $icon = $status ? '✅' : '❌';
            $color = $status ? 'green' : 'red';
            echo "<li style='color: {$color};'>{$icon} {$check}</li>";
        }
        echo "</ul>";
        
        echo "<h2>11. JavaScript Integration URLs</h2>";
        echo "<p>Use these URLs in your JavaScript for the phlebotomy module:</p>";
        echo "<ul>";
        echo "<li><strong>getTriagePatients:</strong> <code>" . base_url('admin/phlebotomy/getTriagePatients') . "</code></li>";
        echo "<li><strong>getStoredTemplate:</strong> <code>" . base_url('admin/phlebotomy/getStoredTemplate') . "</code></li>";
        echo "<li><strong>getPatientDetails:</strong> <code>" . base_url('admin/phlebotomy/getPatientDetails') . "</code></li>";
        echo "<li><strong>savePhlebotomyData:</strong> <code>" . base_url('admin/phlebotomy/savePhlebotomyData') . "</code></li>";
        echo "<li><strong>verifyFingerprint:</strong> <code>" . base_url('admin/phlebotomy/verifyFingerprint') . "</code></li>";
        echo "</ul>";
    }

    /**
     * Test enhanced patient loading endpoint
     */
    public function testPatientLoad() {
        echo "<h2>Testing Enhanced Patient Loading with Fingerprint Data</h2>";
        echo "<p>Calling getTriagePatients() method...</p>";
        
        // Temporarily capture output
        ob_start();
        $this->getTriagePatients();
        $json_output = ob_get_clean();
        
        echo "<h3>JSON Response:</h3>";
        echo "<pre style='background: #f4f4f4; padding: 10px; border-radius: 5px; max-height: 400px; overflow-y: auto;'>";
        echo htmlspecialchars($json_output);
        echo "</pre>";
        
        // Try to decode and display formatted
        $decoded = json_decode($json_output, true);
        if ($decoded) {
            echo "<h3>Formatted Response:</h3>";
            echo "<p><strong>Status:</strong> " . ($decoded['status'] ? 'Success' : 'Failed') . "</p>";
            echo "<p><strong>Message:</strong> " . htmlspecialchars($decoded['message']) . "</p>";
            echo "<p><strong>Patient Count:</strong> " . ($decoded['count'] ?? 0) . "</p>";
            echo "<p><strong>Fingerprint Enabled:</strong> " . ($decoded['fingerprint_enabled'] ? 'Yes' : 'No') . "</p>";
            
            if (!empty($decoded['patients'])) {
                echo "<h4>Sample Patients with Fingerprint Data:</h4>";
                echo "<table border='1' style='border-collapse: collapse;'>";
                echo "<tr><th>Name</th><th>ID</th><th>Age/Gender</th><th>Has Fingerprint</th><th>Requires Verification</th><th>Workflow</th></tr>";
                foreach (array_slice($decoded['patients'], 0, 3) as $patient) {
                    $has_fp = $patient['has_fingerprint'] ? '✅ Yes' : '❌ No';
                    $req_verify = $patient['requires_verification'] ? '🔒 Yes' : '🔓 No';
                    
                    echo "<tr>";
                    echo "<td>{$patient['patient_name']}</td>";
                    echo "<td>{$patient['id']}</td>";
                    echo "<td>{$patient['age']}Y {$patient['gender']}</td>";
                    echo "<td>{$has_fp}</td>";
                    echo "<td>{$req_verify}</td>";
                    echo "<td>{$patient['workflow_type']}</td>";
                    echo "</tr>";
                }
                echo "</table>";
            }
        }
    }

    /**
     * Test fingerprint-specific endpoints
     */
    public function testFingerprintEndpoints() {
        echo "<h2>Testing Fingerprint Verification Endpoints</h2>";
        
        // Find a patient with fingerprint data
        if ($this->db->field_exists('fingerprint_template', 'patients')) {
            $fp_query = $this->db->query("
                SELECT id, patient_name 
                FROM patients 
                WHERE fingerprint_template IS NOT NULL 
                AND LENGTH(fingerprint_template) > 100 
                LIMIT 1
            ");
            
            if ($fp_query->num_rows() > 0) {
                $patient = $fp_query->row_array();
                $patient_id = $patient['id'];
                $patient_name = $patient['patient_name'];
                
                echo "<p>Testing with Patient: <strong>{$patient_name} (ID: {$patient_id})</strong></p>";
                
                // Test getStoredTemplate
                echo "<h3>1. Testing getStoredTemplate</h3>";
                $_POST['patient_id'] = $patient_id;
                
                ob_start();
                $this->getStoredTemplate();
                $template_output = ob_get_clean();
                
                echo "<h4>Template Response:</h4>";
                echo "<pre style='background: #f4f4f4; padding: 10px; border-radius: 5px;'>";
                echo htmlspecialchars($template_output);
                echo "</pre>";
                
                // Test getPatientDetails
                echo "<h3>2. Testing getPatientDetails</h3>";
                $_POST['id'] = $patient_id;
                
                ob_start();
                $this->getPatientDetails();
                $details_output = ob_get_clean();
                
                echo "<h4>Patient Details Response:</h4>";
                echo "<pre style='background: #f4f4f4; padding: 10px; border-radius: 5px;'>";
                echo htmlspecialchars($details_output);
                echo "</pre>";
                
                // Test verifyFingerprint (simulation)
                echo "<h3>3. Testing verifyFingerprint (Simulation)</h3>";
                $_POST['patient_id'] = $patient_id;
                $_POST['match_score'] = '85.7';
                $_POST['is_match'] = 'true';
                $_POST['threshold'] = '60';
                
                ob_start();
                $this->verifyFingerprint();
                $verify_output = ob_get_clean();
                
                echo "<h4>Verification Response:</h4>";
                echo "<pre style='background: #f4f4f4; padding: 10px; border-radius: 5px;'>";
                echo htmlspecialchars($verify_output);
                echo "</pre>";
                
                // Clear POST data
                unset($_POST['patient_id'], $_POST['id'], $_POST['match_score'], $_POST['is_match'], $_POST['threshold']);
                
            } else {
                echo "<p style='color: red;'>❌ No patients with fingerprint templates found for testing</p>";
            }
        } else {
            echo "<p style='color: red;'>❌ Fingerprint functionality not available - missing database columns</p>";
        }
    }

    /**
     * Test individual patient endpoints
     */
    public function testPatientEndpoints($patient_id) {
        echo "<h2>Testing All Endpoints for Patient ID: {$patient_id}</h2>";
        
        // Test getPatientDetails
        echo "<h3>1. Patient Details</h3>";
        $_POST['id'] = $patient_id;
        ob_start();
        $this->getPatientDetails();
        $details = ob_get_clean();
        echo "<pre>" . htmlspecialchars($details) . "</pre>";
        
        // Test getStoredTemplate
        echo "<h3>2. Stored Template</h3>";
        $_POST['patient_id'] = $patient_id;
        ob_start();
        $this->getStoredTemplate();
        $template = ob_get_clean();
        echo "<pre>" . htmlspecialchars($template) . "</pre>";
        
        // Test verification history
        echo "<h3>3. Verification History</h3>";
        $_GET['patient_id'] = $patient_id;
        $_GET['module'] = 'phlebotomy';
        ob_start();
        $this->getVerificationHistory();
        $history = ob_get_clean();
        echo "<pre>" . htmlspecialchars($history) . "</pre>";
        
        // Clean up
        unset($_POST['id'], $_POST['patient_id'], $_GET['patient_id'], $_GET['module']);
    }

    /**
     * Test template fetch for specific patient
     */
    public function testTemplate($patient_id) {
        echo "<h2>Testing Template Fetch for Patient ID: {$patient_id}</h2>";
        
        $_POST['patient_id'] = $patient_id;
        
        ob_start();
        $this->getStoredTemplate();
        $output = ob_get_clean();
        
        echo "<h3>Raw Response:</h3>";
        echo "<pre>" . htmlspecialchars($output) . "</pre>";
        
        $decoded = json_decode($output, true);
        if ($decoded) {
            echo "<h3>Parsed Response:</h3>";
            echo "<p><strong>Status:</strong> " . $decoded['status'] . "</p>";
            echo "<p><strong>Patient:</strong> " . ($decoded['patient_name'] ?? 'Unknown') . "</p>";
            if ($decoded['status'] === 'success') {
                echo "<p><strong>Template Size:</strong> " . $decoded['template_size'] . " characters</p>";
                echo "<p><strong>Has Image:</strong> " . ($decoded['has_image'] ? 'Yes' : 'No') . "</p>";
                echo "<p><strong>Quality:</strong> " . $decoded['quality'] . "</p>";
            }
        }
        
        unset($_POST['patient_id']);
    }

    /**
     * Test database save functionality with verification data
     */
    public function testDatabaseSave() {
        echo "<h2>Testing Database Save with Verification Data</h2>";
        
        // Find a test patient
        $test_patient_query = $this->db->get_where('patients', ['is_active' => 'yes'], 1);
        if ($test_patient_query->num_rows() == 0) {
            echo "<p style='color: red;'>❌ No active patients found for testing</p>";
            return;
        }
        
        $test_patient = $test_patient_query->row_array();
        $test_patient_id = $test_patient['id'];
        
        // Create test phlebotomy data
        $test_data = [
            [
                'sample_type' => 'blood',
                'barcode' => 'TEST-BLO-' . date('YmdHis'),
                'collection_date' => date('Y-m-d H:i:s'),
                'collection_method' => 'venipuncture',
                'volume' => '5ml',
                'notes' => 'Test sample collection with verification'
            ],
            [
                'sample_type' => 'urine',
                'barcode' => 'TEST-URI-' . date('YmdHis'),
                'collection_date' => date('Y-m-d H:i:s'),
                'collection_method' => 'midstream',
                'volume' => '10ml',
                'notes' => 'Test urine sample with verification'
            ]
        ];
        
        // Create test verification data
        $verification_data = [
            'verification_passed' => true,
            'verification_method' => 'fingerprint_match',
            'match_score' => 85.7,
            'timestamp' => date('Y-m-d H:i:s'),
            'verification_log_id' => rand(1000, 9999)
        ];
        
        echo "<h3>Test Data:</h3>";
        echo "<h4>Patient ID:</h4>";
        echo "<pre>" . $test_patient_id . " (" . $test_patient['patient_name'] . ")</pre>";
        echo "<h4>Phlebotomy Data:</h4>";
        echo "<pre>" . json_encode($test_data, JSON_PRETTY_PRINT) . "</pre>";
        echo "<h4>Verification Data:</h4>";
        echo "<pre>" . json_encode($verification_data, JSON_PRETTY_PRINT) . "</pre>";
        
        // Simulate POST data
        $_POST['phlebotomy_data'] = json_encode($test_data);
        $_POST['patient_id'] = $test_patient_id;
        $_POST['verification_data'] = json_encode($verification_data);
        
        echo "<h3>Calling savePhlebotomyData()...</h3>";
        
        // Capture output
        ob_start();
        $this->savePhlebotomyData();
        $json_output = ob_get_clean();
        
        echo "<h3>Save Response:</h3>";
        echo "<pre style='background: #f4f4f4; padding: 10px; border-radius: 5px;'>";
        echo htmlspecialchars($json_output);
        echo "</pre>";
        
        // Parse and display results
        $decoded = json_decode($json_output, true);
        if ($decoded) {
            echo "<h3>Results Summary:</h3>";
            echo "<p><strong>Status:</strong> " . ($decoded['status'] ? 'Success' : 'Failed') . "</p>";
            echo "<p><strong>Message:</strong> " . htmlspecialchars($decoded['message']) . "</p>";
            echo "<p><strong>Records Saved:</strong> " . ($decoded['saved_count'] ?? 0) . "</p>";
            echo "<p><strong>Verification Required:</strong> " . ($decoded['verification_required'] ? 'Yes' : 'No') . "</p>";
            echo "<p><strong>Verification Passed:</strong> " . ($decoded['verification_passed'] ? 'Yes' : 'No') . "</p>";
        }
        
        // Clear test POST data
        unset($_POST['phlebotomy_data'], $_POST['patient_id'], $_POST['verification_data']);
    }

    // =============================================
    // QUEUE MANAGEMENT METHODS
    // =============================================

    /**
     * Get phlebotomy department queue data
     */
    public function get_queue_data() {
        header('Content-Type: application/json');
        
        try {
            $result = $this->Queue_model->getDepartmentQueue('phlebotomy');
            echo json_encode($result);
        } catch (Exception $e) {
            echo json_encode(['status' => 'error', 'message' => $e->getMessage()]);
        }
    }

    /**
     * Call next patient from queue
     */
    public function call_next_patient() {
        header('Content-Type: application/json');
        
        try {
            $technician_id = $this->input->post('technician_id') ?: $this->getStaffId();
            $result = $this->Queue_model->callNextPatient('phlebotomy', $technician_id);
            
            if ($result['status'] === 'success') {
                // Play notification sound
                $this->send_queue_notification('phlebotomy', 'patient_called', $result['patient']);
            }
            
            echo json_encode($result);
        } catch (Exception $e) {
            echo json_encode(['status' => 'error', 'message' => $e->getMessage()]);
        }
    }

    /**
     * Skip patient in queue
     */
    public function skip_patient() {
        header('Content-Type: application/json');
        
        try {
            $queue_id = $this->input->post('queue_id');
            $reason = $this->input->post('reason') ?: 'No reason provided';
            
            $result = $this->Queue_model->skipPatient($queue_id, $reason);
            echo json_encode($result);
        } catch (Exception $e) {
            echo json_encode(['status' => 'error', 'message' => $e->getMessage()]);
        }
    }

    /**
     * Complete phlebotomy service and route to next department
     */
    public function complete_service() {
        header('Content-Type: application/json');
        
        try {
            $queue_id = $this->input->post('queue_id');
            $next_department = $this->input->post('next_department');
            $notes = $this->input->post('notes') ?: 'Sample collection completed';
            
            $result = $this->Queue_model->completeService($queue_id, $next_department, $notes);
            
            if ($result['status'] === 'success' && $next_department) {
                $this->send_queue_notification($next_department, 'phlebotomy_sample_ready', [
                    'from_department' => 'phlebotomy',
                    'patient_id' => $this->input->post('patient_id'),
                    'notes' => $notes
                ]);
            }
            
            echo json_encode($result);
        } catch (Exception $e) {
            echo json_encode(['status' => 'error', 'message' => $e->getMessage()]);
        }
    }

    /**
     * Get queue statistics
     */
    public function get_queue_stats() {
        header('Content-Type: application/json');
        
        try {
            $stats = $this->Queue_model->getQueueStatistics('phlebotomy');
            echo json_encode(['status' => 'success', 'stats' => $stats]);
        } catch (Exception $e) {
            echo json_encode(['status' => 'error', 'message' => $e->getMessage()]);
        }
    }

    /**
     * Queue management panel
     */
    public function queue_management() {
        $this->session->set_userdata('top_menu', 'OPD');
        $this->session->set_userdata('sub_menu', 'phlebotomy');
        
        $data['title'] = 'Phlebotomy Queue Management';
        $data['department'] = 'phlebotomy';
        $data['queue_data'] = $this->Queue_model->getDepartmentQueue('phlebotomy');
        $data['queue_config'] = $this->Queue_model->getQueueConfig('phlebotomy');
        
        $this->load->view('layout/header', $data);
        $this->load->view('admin/phlebotomy/queue_management', $data);
        $this->load->view('layout/footer', $data);
    }

    /**
     * Send queue notification to other departments
     */
    private function send_queue_notification($department, $type, $data = []) {
        $notification = [
            'department' => $department,
            'type' => $type,
            'data' => $data,
            'timestamp' => time()
        ];
        
        $cache_file = APPPATH . 'cache/queue_notifications_' . $department . '.json';
        file_put_contents($cache_file, json_encode($notification));
    }

    /**
     * Get current staff ID
     */
    private function getStaffId() {
        return $this->session->userdata('admin_id') ?: 1;
    }

    /**
     * Debug queue status
     */
    public function debug_queue_status() {
        echo "<h2>Phlebotomy Queue Debug</h2>";
        echo "<p>Controller is working properly!</p>";
        echo "<p>Queue integration: " . (class_exists('Queue_model') ? 'ACTIVE' : 'MISSING') . "</p>";
        echo "<p>Notification system: " . (is_writable(APPPATH . 'cache/') ? 'READY' : 'NOT WRITABLE') . "</p>";
        
        try {
            $stats = $this->Queue_model->getQueueStatistics('phlebotomy');
            echo "<p>Queue statistics: Working</p>";
            echo "<pre>" . json_encode($stats, JSON_PRETTY_PRINT) . "</pre>";
        } catch (Exception $e) {
            echo "<p>Queue statistics: Error - " . $e->getMessage() . "</p>";
        }
    }
}

/* End of file Phlebotomy.php */
/* Location: ./application/controllers/admin/Phlebotomy.php */