<?php

defined('BASEPATH') OR exit('No direct script access allowed');

class Nursing extends Admin_Controller {

    public function __construct() {
        parent::__construct();
        
        try {
            $this->load->model(['patient_model', 'staff_model', 'Queue_model']);
            $this->load->library(['datatables', 'form_validation', 'customlib']);
            $this->load->helper(['url', 'form', 'date']);
            $this->load->model('nursing_model');
        } catch (Exception $e) {
            log_message('error', 'Nursing controller initialization error: ' . $e->getMessage());
        }
        
       /* if (!$this->rbac->hasPrivilege('nursing', 'can_view')) {
            access_denied();
        }*/
    }

    public function index() {
        $this->session->set_userdata('top_menu', 'Nursing');
        $this->session->set_userdata('sub_menu', 'nursing/dashboard');
        
        $data['title'] = 'Nursing Dashboard';
        $data['page_title'] = 'Comprehensive Nursing Care Management';
        $data['nursing_staff'] = $this->getNursingStaffList();
        $data['stats'] = $this->getDashboardStats();

        // Queue integration for nursing department
        $data['department'] = 'nursing';
        $data['queue_data'] = $this->Queue_model->getDepartmentQueue('nursing');
        $data['queue_config'] = $this->Queue_model->getQueueConfig('nursing');
        
        $this->load->view('layout/header', $data);
        $this->load->view('admin/nursing/dashboard', $data);
        $this->load->view('layout/footer', $data);
    }

    private function getDashboardStats() {
        try {
            $stats = [
                'active_ipd_patients' => $this->getActiveIPDCount(),
                'pending_procedures' => $this->getPendingProceduresCount(),
                'todays_assessments' => $this->getTodayAssessmentsCount(),
                'critical_alerts' => $this->getCriticalAlertsCount(),
                'vitals_today' => $this->getVitalsToday(),
                'medications_today' => $this->getMedicationsToday()
            ];
            return $stats;
        } catch (Exception $e) {
            log_message('error', 'Dashboard stats error: ' . $e->getMessage());
            return [
                'active_ipd_patients' => 0,
                'pending_procedures' => 0,
                'todays_assessments' => 0,
                'critical_alerts' => 0,
                'vitals_today' => 0,
                'medications_today' => 0
            ];
        }
    }

    public function searchPatients() {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }

        try {
            $search_term = trim($this->input->post('search'));
            $search_type = $this->input->post('search_type', TRUE);
            
            if (empty($search_term) || strlen($search_term) < 2) {
                echo json_encode(array(
                    'status' => 'error',
                    'message' => 'Please enter at least 2 characters to search'
                ));
                return;
            }

            $patients = $this->searchPatientsInDatabase($search_term, $search_type);
            
            echo json_encode(array(
                'status' => 'success',
                'patients' => $patients,
                'count' => count($patients)
            ));

        } catch (Exception $e) {
            log_message('error', 'Patient search error: ' . $e->getMessage());
            echo json_encode(array(
                'status' => 'error',
                'message' => 'Search failed. Please try again.'
            ));
        }
    }
    
    private function searchPatientsInDatabase($search_term, $search_type = 'all') {
        try {
            if (method_exists($this->patient_model, 'searchPatientsForConsultation')) {
                $limit = 20;
                $patients = $this->patient_model->searchPatientsForConsultation($search_term, $limit);
                
                foreach ($patients as &$patient) {
                    $patient['is_ipd'] = $this->checkIPDStatus($patient['id']);
                    $patient['age_display'] = $this->formatPatientAge($patient);
                }
                
                return $patients;
            }
            
            return $this->directPatientSearch($search_term, $search_type);
            
        } catch (Exception $e) {
            log_message('error', 'Patient search database error: ' . $e->getMessage());
            return [];
        }
    }
    
    /**
 * Get pending procedures
 */
public function get_pending_procedures() {
    try {
        $this->db->select('
            np.id,
            np.procedure_name,
            np.procedure_type,
            np.status,
            np.created_at,
            np.scheduled_date,
            np.priority,
            p.patient_name,
            p.patient_unique_id,
            s.name as assigned_to_name,
            s.surname as assigned_to_surname
        ');
        $this->db->from('nursing_procedures np');
        $this->db->join('patients p', 'p.id = np.patient_id', 'left');
        $this->db->join('staff s', 's.id = np.assigned_to', 'left');
        $this->db->where_in('np.status', ['pending', 'scheduled']);
        $this->db->order_by('np.priority', 'DESC');
        $this->db->order_by('np.scheduled_date', 'ASC');
        $this->db->limit(50);
        
        $query = $this->db->get();
        
        if ($query === false) {
            throw new Exception('Database query failed: ' . $this->db->error()['message']);
        }
        
        echo json_encode([
            'status' => 'success',
            'procedures' => $query->result_array()
        ]);
        
    } catch (Exception $e) {
        echo json_encode([
            'status' => 'error',
            'message' => 'Failed to load pending procedures: ' . $e->getMessage()
        ]);
    }
}

/**
 * Get today's assessments
 */
public function get_todays_assessments() {
    try {
        $this->db->select('
            na.id,
            na.assessment_type,
            na.assessment_category,
            na.risk_level,
            na.assessment_date,
            na.created_at,
            p.patient_name,
            p.patient_unique_id,
            s.name as assessed_by_name,
            s.surname as assessed_by_surname
        ');
        $this->db->from('nursing_assessments na');
        $this->db->join('patients p', 'p.id = na.patient_id', 'left');
        $this->db->join('staff s', 's.id = na.assessed_by', 'left');
        $this->db->where('DATE(na.assessment_date)', date('Y-m-d'));
        $this->db->order_by('na.created_at', 'DESC');
        
        $query = $this->db->get();
        
        if ($query === false) {
            throw new Exception('Database query failed: ' . $this->db->error()['message']);
        }
        
        echo json_encode([
            'status' => 'success',
            'assessments' => $query->result_array()
        ]);
        
    } catch (Exception $e) {
        echo json_encode([
            'status' => 'error',
            'message' => 'Failed to load today\'s assessments: ' . $e->getMessage()
        ]);
    }
}

/**
 * Get critical alerts
 */
public function get_critical_alerts() {
    try {
        $alerts = [];
        
        // Critical vital signs (high/low values)
        if ($this->db->table_exists('nursing_vital_signs')) {
            $this->db->select('
                nvs.id,
                nvs.temperature,
                nvs.systolic_bp,
                nvs.diastolic_bp,
                nvs.pulse_rate,
                nvs.respiratory_rate,
                nvs.oxygen_saturation,
                nvs.recorded_at,
                p.patient_name,
                p.patient_unique_id,
                "vital_signs" as alert_type
            ');
            $this->db->from('nursing_vital_signs nvs');
            $this->db->join('patients p', 'p.id = nvs.patient_id', 'left');
            $this->db->where('DATE(nvs.recorded_at) >=', date('Y-m-d', strtotime('-1 day')));
            $this->db->where('(
                nvs.temperature > 39.0 OR nvs.temperature < 35.0 OR
                nvs.systolic_bp > 180 OR nvs.systolic_bp < 90 OR
                nvs.diastolic_bp > 110 OR nvs.diastolic_bp < 60 OR
                nvs.pulse_rate > 120 OR nvs.pulse_rate < 50 OR
                nvs.respiratory_rate > 30 OR nvs.respiratory_rate < 8 OR
                nvs.oxygen_saturation < 90
            )');
            $this->db->order_by('nvs.recorded_at', 'DESC');
            $this->db->limit(20);
            
            $vital_alerts = $this->db->get();
            if ($vital_alerts !== false) {
                foreach ($vital_alerts->result_array() as $vital) {
                    $alert_details = [];
                    if ($vital['temperature'] > 39.0) $alert_details[] = "High fever: {$vital['temperature']}°C";
                    if ($vital['temperature'] < 35.0) $alert_details[] = "Low temperature: {$vital['temperature']}°C";
                    if ($vital['systolic_bp'] > 180) $alert_details[] = "High systolic BP: {$vital['systolic_bp']}";
                    if ($vital['systolic_bp'] < 90) $alert_details[] = "Low systolic BP: {$vital['systolic_bp']}";
                    if ($vital['oxygen_saturation'] < 90) $alert_details[] = "Low O2 sat: {$vital['oxygen_saturation']}%";
                    
                    $alerts[] = [
                        'id' => $vital['id'],
                        'type' => 'critical_vitals',
                        'severity' => 'critical',
                        'patient_name' => $vital['patient_name'],
                        'patient_id' => $vital['patient_unique_id'],
                        'message' => implode(', ', $alert_details),
                        'timestamp' => $vital['recorded_at'],
                        'details' => $vital
                    ];
                }
            }
        }
        
        // High-risk assessments
        if ($this->db->table_exists('nursing_assessments')) {
            $this->db->select('
                na.id,
                na.assessment_type,
                na.risk_level,
                na.assessment_details,
                na.assessment_date,
                p.patient_name,
                p.patient_unique_id
            ');
            $this->db->from('nursing_assessments na');
            $this->db->join('patients p', 'p.id = na.patient_id', 'left');
            $this->db->where('na.risk_level', 'critical');
            $this->db->where('DATE(na.assessment_date) >=', date('Y-m-d', strtotime('-2 days')));
            $this->db->order_by('na.assessment_date', 'DESC');
            $this->db->limit(10);
            
            $assessment_alerts = $this->db->get();
            if ($assessment_alerts !== false) {
                foreach ($assessment_alerts->result_array() as $assessment) {
                    $alerts[] = [
                        'id' => $assessment['id'],
                        'type' => 'high_risk_assessment',
                        'severity' => 'critical',
                        'patient_name' => $assessment['patient_name'],
                        'patient_id' => $assessment['patient_unique_id'],
                        'message' => "Critical {$assessment['assessment_type']} assessment",
                        'timestamp' => $assessment['assessment_date'],
                        'details' => $assessment
                    ];
                }
            }
        }
        
        // Sort alerts by timestamp (newest first)
        usort($alerts, function($a, $b) {
            return strtotime($b['timestamp']) - strtotime($a['timestamp']);
        });
        
        echo json_encode([
            'status' => 'success',
            'alerts' => array_slice($alerts, 0, 30) // Limit to 30 most recent alerts
        ]);
        
    } catch (Exception $e) {
        echo json_encode([
            'status' => 'error',
            'message' => 'Failed to load critical alerts: ' . $e->getMessage()
        ]);
    }
}

/**
 * Get procedure templates
 */
public function get_procedure_templates() {
    try {
        if (!$this->db->table_exists('nursing_procedure_templates')) {
            // Return default templates if table doesn't exist
            $default_templates = [
                [
                    'id' => 1,
                    'template_name' => 'Wound Care - Standard',
                    'procedure_type' => 'wound_care',
                    'category' => 'Basic Care',
                    'duration_minutes' => 15,
                    'equipment_needed' => 'Sterile gloves, gauze, saline, antiseptic',
                    'steps' => "1. Wash hands and wear sterile gloves\n2. Remove old dressing\n3. Clean wound with saline\n4. Apply antiseptic\n5. Apply new sterile dressing\n6. Document wound condition",
                    'precautions' => 'Maintain sterile technique throughout',
                    'is_active' => true
                ],
                [
                    'id' => 2,
                    'template_name' => 'Vital Signs - Complete',
                    'procedure_type' => 'assessment',
                    'category' => 'Assessment',
                    'duration_minutes' => 10,
                    'equipment_needed' => 'Thermometer, BP cuff, stethoscope, pulse oximeter',
                    'steps' => "1. Explain procedure to patient\n2. Take temperature\n3. Measure blood pressure\n4. Count pulse rate\n5. Assess respiratory rate\n6. Measure oxygen saturation\n7. Document all findings",
                    'precautions' => 'Ensure patient is relaxed before BP measurement',
                    'is_active' => true
                ],
                [
                    'id' => 3,
                    'template_name' => 'Medication Administration - Oral',
                    'procedure_type' => 'medication',
                    'category' => 'Medication',
                    'duration_minutes' => 5,
                    'equipment_needed' => 'Medication, water, medication cup',
                    'steps' => "1. Verify 5 rights of medication\n2. Check patient identification\n3. Explain medication to patient\n4. Administer medication\n5. Ensure patient swallows\n6. Document administration\n7. Monitor for immediate reactions",
                    'precautions' => 'Always verify patient allergies before administration',
                    'is_active' => true
                ]
            ];
            
            echo json_encode([
                'status' => 'success',
                'templates' => $default_templates
            ]);
            return;
        }
        
        $this->db->select('*');
        $this->db->from('nursing_procedure_templates');
        $this->db->where('is_active', 1);
        $this->db->order_by('category', 'ASC');
        $this->db->order_by('template_name', 'ASC');
        
        $query = $this->db->get();
        
        if ($query === false) {
            throw new Exception('Database query failed: ' . $this->db->error()['message']);
        }
        
        echo json_encode([
            'status' => 'success',
            'templates' => $query->result_array()
        ]);
        
    } catch (Exception $e) {
        echo json_encode([
            'status' => 'error',
            'message' => 'Failed to load procedure templates: ' . $e->getMessage()
        ]);
    }
}

/**
 * Get staff workload analysis
 */
public function get_staff_workload() {
    try {
        $workload_data = [];
        $today = date('Y-m-d');
        
        // Get nursing staff
        $this->db->select('id, name, surname, employee_id');
        $this->db->from('staff');
        $this->db->where('is_active', 'yes');
        $this->db->where_in('role_id', [2, 9]); // Assuming 2 = nurse, 9 = head nurse
        $staff_query = $this->db->get();
        
        if ($staff_query === false) {
            throw new Exception('Failed to fetch staff data');
        }
        
        $nursing_staff = $staff_query->result_array();
        
        foreach ($nursing_staff as $staff) {
            $staff_id = $staff['id'];
            $workload = [
                'staff_id' => $staff_id,
                'staff_name' => $staff['name'] . ' ' . $staff['surname'],
                'employee_id' => $staff['employee_id'],
                'vital_signs_count' => 0,
                'nursing_notes_count' => 0,
                'procedures_count' => 0,
                'assessments_count' => 0,
                'total_activities' => 0,
                'patients_assigned' => 0,
                'workload_level' => 'low'
            ];
            
            // Count vital signs recorded today
            if ($this->db->table_exists('nursing_vital_signs')) {
                $this->db->where('recorded_by', $staff_id);
                $this->db->where('DATE(recorded_at)', $today);
                $vital_count = $this->db->count_all_results('nursing_vital_signs');
                $workload['vital_signs_count'] = $vital_count;
            }
            
            // Count nursing notes added today
            if ($this->db->table_exists('nurse_note')) {
                $this->db->where('staff_id', $staff_id);
                $this->db->where('DATE(date)', $today);
                $notes_count = $this->db->count_all_results('nurse_note');
                $workload['nursing_notes_count'] = $notes_count;
            }
            
            // Count procedures performed today
            if ($this->db->table_exists('nursing_procedures')) {
                $this->db->where('performed_by', $staff_id);
                $this->db->where('DATE(created_at)', $today);
                $procedures_count = $this->db->count_all_results('nursing_procedures');
                $workload['procedures_count'] = $procedures_count;
            }
            
            // Count assessments done today
            if ($this->db->table_exists('nursing_assessments')) {
                $this->db->where('assessed_by', $staff_id);
                $this->db->where('DATE(assessment_date)', $today);
                $assessments_count = $this->db->count_all_results('nursing_assessments');
                $workload['assessments_count'] = $assessments_count;
            }
            
            // Calculate total activities
            $total_activities = $workload['vital_signs_count'] + 
                              $workload['nursing_notes_count'] + 
                              $workload['procedures_count'] + 
                              $workload['assessments_count'];
            $workload['total_activities'] = $total_activities;
            
            // Determine workload level
            if ($total_activities > 20) {
                $workload['workload_level'] = 'high';
            } elseif ($total_activities > 10) {
                $workload['workload_level'] = 'medium';
            } else {
                $workload['workload_level'] = 'low';
            }
            
            $workload_data[] = $workload;
        }
        
        // Sort by total activities (highest first)
        usort($workload_data, function($a, $b) {
            return $b['total_activities'] - $a['total_activities'];
        });
        
        echo json_encode([
            'status' => 'success',
            'workload_data' => $workload_data,
            'summary' => [
                'total_staff' => count($nursing_staff),
                'high_workload' => count(array_filter($workload_data, function($w) { return $w['workload_level'] === 'high'; })),
                'medium_workload' => count(array_filter($workload_data, function($w) { return $w['workload_level'] === 'medium'; })),
                'low_workload' => count(array_filter($workload_data, function($w) { return $w['workload_level'] === 'low'; }))
            ]
        ]);
        
    } catch (Exception $e) {
        echo json_encode([
            'status' => 'error',
            'message' => 'Failed to load staff workload: ' . $e->getMessage()
        ]);
    }
}

/**
 * Create procedure from template
 */
public function create_procedure_from_template() {
    $template_id = $this->input->post('template_id');
    $patient_id = $this->input->post('patient_id');
    $scheduled_date = $this->input->post('scheduled_date');
    $notes = $this->input->post('notes');
    
    if (!$template_id || !$patient_id) {
        echo json_encode([
            'status' => 'error',
            'message' => 'Template ID and Patient ID are required'
        ]);
        return;
    }
    
    try {
        // Get template details
        $this->db->select('*');
        $this->db->from('nursing_procedure_templates');
        $this->db->where('id', $template_id);
        $template_query = $this->db->get();
        
        if ($template_query->num_rows() === 0) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Template not found'
            ]);
            return;
        }
        
        $template = $template_query->row_array();
        
        // Create procedure from template
        $procedure_data = [
            'patient_id' => $patient_id,
            'procedure_name' => $template['template_name'],
            'procedure_type' => $template['procedure_type'],
            'procedure_description' => $template['steps'],
            'equipment_needed' => $template['equipment_needed'],
            'precautions' => $template['precautions'],
            'estimated_duration' => $template['duration_minutes'],
            'status' => 'scheduled',
            'scheduled_date' => $scheduled_date ?: date('Y-m-d H:i:s'),
            'created_by' => $this->getStaffId(),
            'created_at' => date('Y-m-d H:i:s'),
            'procedure_notes' => $notes
        ];
        
        $this->db->insert('nursing_procedures', $procedure_data);
        $procedure_id = $this->db->insert_id();
        
        if ($procedure_id) {
            echo json_encode([
                'status' => 'success',
                'message' => 'Procedure scheduled successfully',
                'procedure_id' => $procedure_id
            ]);
        } else {
            echo json_encode([
                'status' => 'error',
                'message' => 'Failed to create procedure'
            ]);
        }
        
    } catch (Exception $e) {
        echo json_encode([
            'status' => 'error',
            'message' => 'Error creating procedure: ' . $e->getMessage()
        ]);
    }
}

/**
 * Update procedure status
 */
public function update_procedure_status() {
    $procedure_id = $this->input->post('procedure_id');
    $status = $this->input->post('status');
    $notes = $this->input->post('notes');
    
    if (!$procedure_id || !$status) {
        echo json_encode([
            'status' => 'error',
            'message' => 'Procedure ID and status are required'
        ]);
        return;
    }
    
    try {
        $update_data = [
            'status' => $status,
            'updated_at' => date('Y-m-d H:i:s')
        ];
        
        if ($status === 'completed') {
            $update_data['completed_at'] = date('Y-m-d H:i:s');
            $update_data['performed_by'] = $this->getStaffId();
        }
        
        if ($notes) {
            $update_data['completion_notes'] = $notes;
        }
        
        $this->db->where('id', $procedure_id);
        $result = $this->db->update('nursing_procedures', $update_data);
        
        if ($result) {
            echo json_encode([
                'status' => 'success',
                'message' => 'Procedure status updated successfully'
            ]);
        } else {
            echo json_encode([
                'status' => 'error',
                'message' => 'Failed to update procedure status'
            ]);
        }
        
    } catch (Exception $e) {
        echo json_encode([
            'status' => 'error',
            'message' => 'Error updating procedure: ' . $e->getMessage()
        ]);
    }
}

/**
 * Acknowledge critical alert
 */
public function acknowledge_alert() {
    $alert_id = $this->input->post('alert_id');
    $alert_type = $this->input->post('alert_type');
    $notes = $this->input->post('notes');
    
    if (!$alert_id || !$alert_type) {
        echo json_encode([
            'status' => 'error',
            'message' => 'Alert ID and type are required'
        ]);
        return;
    }
    
    try {
        // Create alert acknowledgment record
        $acknowledgment_data = [
            'alert_id' => $alert_id,
            'alert_type' => $alert_type,
            'acknowledged_by' => $this->getStaffId(),
            'acknowledged_at' => date('Y-m-d H:i:s'),
            'notes' => $notes
        ];
        
        // Check if alerts table exists, create if not
        if (!$this->db->table_exists('nursing_alert_acknowledgments')) {
            $this->create_alert_acknowledgments_table();
        }
        
        $this->db->insert('nursing_alert_acknowledgments', $acknowledgment_data);
        
        echo json_encode([
            'status' => 'success',
            'message' => 'Alert acknowledged successfully'
        ]);
        
    } catch (Exception $e) {
        echo json_encode([
            'status' => 'error',
            'message' => 'Error acknowledging alert: ' . $e->getMessage()
        ]);
    }
}

/**
 * Create alert acknowledgments table if it doesn't exist
 */
private function create_alert_acknowledgments_table() {
    $sql = "
        CREATE TABLE IF NOT EXISTS `nursing_alert_acknowledgments` (
            `id` int(11) NOT NULL AUTO_INCREMENT,
            `alert_id` int(11) NOT NULL,
            `alert_type` varchar(50) NOT NULL,
            `acknowledged_by` int(11) NOT NULL,
            `acknowledged_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
            `notes` text,
            PRIMARY KEY (`id`),
            KEY `idx_alert_id` (`alert_id`),
            KEY `idx_alert_type` (`alert_type`),
            KEY `idx_acknowledged_by` (`acknowledged_by`)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;
    ";
    
    $this->db->query($sql);
}

    private function formatPatientAge($patient) {
        try {
            if (method_exists($this->customlib, 'getPatientAge')) {
                $age = $patient['age'] ?? 0;
                $month = $patient['month'] ?? 0;
                $day = $patient['day'] ?? 0;
                
                return $this->customlib->getPatientAge($age, $month, $day);
            }
            
            if (!empty($patient['age'])) {
                return $patient['age'] . ' years';
            }
            
            return 'N/A';
            
        } catch (Exception $e) {
            log_message('error', 'Format patient age error: ' . $e->getMessage());
            return !empty($patient['age']) ? $patient['age'] . ' years' : 'N/A';
        }
    }

    private function checkIPDStatus($patient_id) {
        try {
            if ($this->db->table_exists('ipd_details')) {
                $this->db->select('id');
                $this->db->from('ipd_details');
                $this->db->where('patient_id', $patient_id);
                $this->db->where('discharged', 'no');
                $query = $this->db->get();
                return $query->num_rows() > 0;
            }
            return false;
        } catch (Exception $e) {
            return false;
        }
    }

    private function directPatientSearch($search_term, $search_type) {
        try {
            $this->db->select('p.id, p.patient_name, p.mobileno, p.id, p.gender, p.age, p.month, p.day, p.dob');
            $this->db->from('patients p');
            
            $this->db->group_start();
            if ($search_type == 'name') {
                $this->db->like('p.patient_name', $search_term);
            } elseif ($search_type == 'id') {
                $this->db->like('p.id', $search_term);
                $this->db->or_like('p.id', $search_term);
            } elseif ($search_type == 'phone') {
                $this->db->like('p.mobileno', $search_term);
            } else {
                $this->db->like('p.patient_name', $search_term);
                $this->db->or_like('p.id', $search_term);
                $this->db->or_like('p.mobileno', $search_term);
            }
            $this->db->group_end();
            
            $this->db->where('p.is_active', 'yes');
            $this->db->order_by('p.patient_name', 'ASC');
            $this->db->limit(20);
            
            $query = $this->db->get();
            $patients = $query->result_array();
            
            foreach ($patients as &$patient) {
                $patient['is_ipd'] = $this->checkIPDStatus($patient['id']);
                $patient['age_display'] = $this->formatPatientAge($patient);
            }
            
            return $patients;
            
        } catch (Exception $e) {
            log_message('error', 'Direct patient search error: ' . $e->getMessage());
            return [];
        }
    }

    public function getPatientDetails() {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }

        try {
            $patient_id = $this->input->post('patient_id');
            
            if (empty($patient_id)) {
                echo json_encode(array(
                    'status' => 'error',
                    'message' => 'Patient ID is required'
                ));
                return;
            }

            $patient = $this->getPatientById($patient_id);
            
            if (empty($patient)) {
                echo json_encode(array(
                    'status' => 'error',
                    'message' => 'Patient not found'
                ));
                return;
            }

            echo json_encode(array(
                'status' => 'success',
                'patient' => $patient
            ));

        } catch (Exception $e) {
            log_message('error', 'Get patient details error: ' . $e->getMessage());
            echo json_encode(array(
                'status' => 'error',
                'message' => 'Failed to load patient details'
            ));
        }
    }

    private function getPatientById($patient_id) {
        try {
            if (method_exists($this->patient_model, 'getpatientDetails')) {
                $patient = $this->patient_model->getpatientDetails($patient_id);
                if ($patient) {
                    $patient['is_ipd'] = $this->checkIPDStatus($patient_id);
                    $patient['age_display'] = $this->formatPatientAge($patient);
                    return $patient;
                }
            }
            
            return $this->directGetPatient($patient_id);
            
        } catch (Exception $e) {
            log_message('error', 'Get patient by ID error: ' . $e->getMessage());
            return null;
        }
    }

    private function directGetPatient($patient_id) {
        try {
            $this->db->select('*');
            $this->db->from('patients');
            $this->db->where('id', $patient_id);
            $query = $this->db->get();
            
            if ($query && $query->num_rows() > 0) {
                $patient = $query->row_array();
                $patient['is_ipd'] = $this->checkIPDStatus($patient_id);
                $patient['age_display'] = $this->formatPatientAge($patient);
                return $patient;
            }
            
            return null;
            
        } catch (Exception $e) {
            log_message('error', 'Direct get patient error: ' . $e->getMessage());
            return null;
        }
    }

    public function add_vital_signs() {
        header('Content-Type: application/json');

        $this->form_validation->set_rules('patient_id', 'Patient', 'required');
        $this->form_validation->set_rules('temperature', 'Temperature', 'required|numeric');
        $this->form_validation->set_rules('systolic_bp', 'Systolic BP', 'required|numeric');
        $this->form_validation->set_rules('diastolic_bp', 'Diastolic BP', 'required|numeric');
        $this->form_validation->set_rules('pulse_rate', 'Pulse Rate', 'required|numeric');
        $this->form_validation->set_rules('respiratory_rate', 'Respiratory Rate', 'required|numeric');

        if ($this->form_validation->run() == false) {
            $errors = validation_errors();
            echo json_encode([
                'status' => 'error', 
                'message' => 'Validation failed: ' . strip_tags($errors),
                'errors' => $errors,
                'form_data' => $this->input->post()
            ]);
            return;
        }

        try {
            if (!isset($this->nursing_model)) {
                $this->load->model('nursing_model');
            }

            $vital_data = [
                'patient_id' => $this->input->post('patient_id'),
                'ipd_id' => $this->input->post('ipd_id') ?: null,
                'opd_id' => $this->input->post('opd_id') ?: null,
                'temperature' => $this->input->post('temperature'),
                'temperature_unit' => $this->input->post('temperature_unit') ?: 'C',
                'systolic_bp' => $this->input->post('systolic_bp'),
                'diastolic_bp' => $this->input->post('diastolic_bp'),
                'pulse_rate' => $this->input->post('pulse_rate'),
                'respiratory_rate' => $this->input->post('respiratory_rate'),
                'oxygen_saturation' => $this->input->post('oxygen_saturation') ?: null,
                'pain_scale' => $this->input->post('pain_scale') ?: null,
                'weight' => $this->input->post('weight') ?: null,
                'height' => $this->input->post('height') ?: null,
                'notes' => $this->input->post('notes') ?: null,
                'recorded_by' => $this->getStaffId(),
                'recorded_at' => date('Y-m-d H:i:s')
            ];

            $vital_id = $this->nursing_model->addVitalSigns($vital_data);

            echo json_encode([
                'status' => 'success',
                'message' => 'Vital signs recorded successfully',
                'vital_id' => $vital_id
            ]);

        } catch (Exception $e) {
            log_message('error', 'Add vital signs error: ' . $e->getMessage());
            echo json_encode([
                'status' => 'error',
                'message' => 'Failed to record vital signs: ' . $e->getMessage(),
                'debug' => [
                    'error' => $e->getMessage(),
                    'file' => $e->getFile(),
                    'line' => $e->getLine()
                ]
            ]);
        }
    }

    public function add_nurse_note() {
        header('Content-Type: application/json');

        $this->form_validation->set_rules('patient_id', 'Patient', 'required');
        $this->form_validation->set_rules('note', 'Note', 'required');

        if ($this->form_validation->run() == false) {
            $errors = validation_errors();
            echo json_encode([
                'status' => 'error', 
                'message' => 'Validation failed: ' . strip_tags($errors),
                'errors' => $errors,
                'form_data' => $this->input->post()
            ]);
            return;
        }

        try {
            if (!isset($this->nursing_model)) {
                $this->load->model('nursing_model');
            }

            $patient_id = $this->input->post('patient_id');
            
            $ipd_id = $this->input->post('ipd_id');
            if (empty($ipd_id)) {
                $ipd_details = $this->nursing_model->getPatientIPDDetails($patient_id);
                if ($ipd_details) {
                    $ipd_id = $ipd_details['id'];
                }
            }

            if (empty($ipd_id)) {
                if ($this->db->table_exists('nurse_note')) {
                    $fields = $this->db->field_data('nurse_note');
                    $ipd_nullable = false;
                    foreach ($fields as $field) {
                        if ($field->name == 'ipd_id' && $field->null == 1) {
                            $ipd_nullable = true;
                            break;
                        }
                    }
                    
                    if (!$ipd_nullable) {
                        echo json_encode([
                            'status' => 'error',
                            'message' => 'This patient is not admitted (IPD). Nursing notes can only be added for IPD patients.',
                            'debug' => 'No IPD record found and ipd_id is required'
                        ]);
                        return;
                    }
                }
            }

            $note_data = [
                'ipd_id' => $ipd_id,
                'note' => $this->input->post('note'),
                'comment' => $this->input->post('comment') ?: '',
                'staff_id' => $this->getStaffId(),
                'date' => date('Y-m-d H:i:s')
            ];

            $note_id = $this->nursing_model->addNurseNote($note_data);

            echo json_encode([
                'status' => 'success',
                'message' => 'Nursing note added successfully',
                'note_id' => $note_id
            ]);

        } catch (Exception $e) {
            log_message('error', 'Add nurse note error: ' . $e->getMessage());
            echo json_encode([
                'status' => 'error',
                'message' => 'Failed to add nursing note: ' . $e->getMessage(),
                'debug' => [
                    'error' => $e->getMessage(),
                    'file' => $e->getFile(),
                    'line' => $e->getLine(),
                    'post_data' => $this->input->post()
                ]
            ]);
        }
    }

public function patient_profile($patient_id = null) {
    if (!$patient_id) {
        $this->session->set_flashdata('error', 'Patient ID required');
        redirect('admin/nursing');
    }

    $this->session->set_userdata('top_menu', 'Nursing');
    $this->session->set_userdata('sub_menu', 'nursing/patient_profile');
    
    $data['title'] = 'Patient Nursing Profile';
    
    $data['patient'] = $this->getPatientById($patient_id);
    
    if (!$data['patient']) {
        $this->session->set_flashdata('error', 'Patient not found');
        redirect('admin/nursing');
    }

    $data['ipd_details'] = $this->nursing_model->getPatientIPDDetails($patient_id);
    
    // Existing nursing data
    log_message('debug', 'Getting nursing notes for patient: ' . $patient_id);
    $data['nurse_notes'] = $this->nursing_model->getPatientNurseNotes($patient_id);
    log_message('debug', 'Notes found: ' . count($data['nurse_notes']));
    
    $data['vital_signs'] = $this->nursing_model->getPatientVitalSigns($patient_id);
    $data['medications'] = $this->nursing_model->getPatientMedications($patient_id);
    $data['procedures'] = $this->nursing_model->getPatientProcedures($patient_id);
    $data['care_plans'] = $this->nursing_model->getPatientCarePlans($patient_id);
    $data['nursing_assessments'] = $this->nursing_model->getPatientAssessments($patient_id);
    
    // NEW: Doctor and clinical notes with proper error handling
    $data['doctor_consultations'] = $this->getPatientConsultationsSafe($patient_id);
    $data['prescriptions'] = $this->getPatientPrescriptionsSafe($patient_id);
    $data['clinical_notes'] = $this->getPatientClinicalNotesSafe($patient_id);
    $data['opd_visits'] = $this->getPatientOPDVisitsSafe($patient_id);
    $data['ipd_visits'] = $this->getPatientIPDVisitsSafe($patient_id);
    
    // Supporting data
    $data['nursing_staff'] = $this->getNursingStaffList();
    $data['procedure_templates'] = $this->nursing_model->getProcedureTemplates();
    
    $this->load->view('layout/header', $data);
    $this->load->view('admin/nursing/patient_profile', $data);
    $this->load->view('layout/footer', $data);
}

/**
 * Get patient consultations with proper error handling
 */
private function getPatientConsultationsSafe($patient_id) {
    try {
        if (!$this->db->table_exists('consultations')) {
            log_message('debug', 'Consultations table does not exist');
            return [];
        }
        
        $this->db->select('
            c.id,
            c.date as consultation_date,
            c.symptoms,
            c.diagnosis,
            c.notes as consultation_notes,
            c.verification_status,
            c.verification_score,
            c.has_prescription,
            c.prescription_count,
            c.prescription_total_cost,
            c.general_prescription_notes,
            s.name as doctor_name,
            s.surname as doctor_surname,
            s.employee_id as doctor_employee_id
        ');
        $this->db->from('consultations c');
        $this->db->join('staff s', 's.id = c.doctor', 'left');
        $this->db->where('c.patient_id', $patient_id);
        $this->db->order_by('c.date', 'DESC');
        $this->db->limit(20);
        
        $query = $this->db->get();
        
        if ($query === false) {
            log_message('error', 'Query failed in getPatientConsultationsSafe: ' . $this->db->error()['message']);
            return [];
        }
        
        return $query->result_array();
        
    } catch (Exception $e) {
        log_message('error', 'Error getting patient consultations: ' . $e->getMessage());
        return [];
    }
}

/**
 * Get patient prescriptions with proper error handling
 */
private function getPatientPrescriptionsSafe($patient_id) {
    try {
        // Try modern prescriptions table first
        if ($this->db->table_exists('prescriptions')) {
            $this->db->select('
                p.id,
                p.prescription_date,
                p.prescription_time,
                p.status,
                p.total_cost,
                p.general_instructions,
                p.pharmacy_notes,
                p.dispensed_at,
                s1.name as doctor_name,
                s1.surname as doctor_surname,
                s2.name as dispensed_by_name,
                s2.surname as dispensed_by_surname
            ');
            $this->db->from('prescriptions p');
            $this->db->join('staff s1', 's1.id = p.doctor_id', 'left');
            $this->db->join('staff s2', 's2.id = p.dispensed_by', 'left');
            $this->db->where('p.patient_id', $patient_id);
            $this->db->order_by('p.prescription_date', 'DESC');
            $this->db->limit(15);
            
            $query = $this->db->get();
            
            if ($query === false) {
                log_message('error', 'Prescriptions query failed: ' . $this->db->error()['message']);
                return [];
            }
            
            $prescriptions = $query->result_array();
            
            // Get prescription items for each prescription
            foreach ($prescriptions as &$prescription) {
                $prescription['items'] = $this->getPrescriptionItemsSafe($prescription['id']);
            }
            
            return $prescriptions;
        }
        
        // Fallback to consultation_prescriptions if available
        if ($this->db->table_exists('consultation_prescriptions')) {
            $this->db->select('
                cp.id,
                cp.created_at as prescription_date,
                cp.status,
                cp.total_cost,
                cp.dosage_instructions,
                cp.duration,
                cp.special_instructions,
                cp.dispensed_at,
                m.medicine_name,
                m.medicine_category,
                s1.name as doctor_name,
                s1.surname as doctor_surname,
                c.date as consultation_date
            ');
            $this->db->from('consultation_prescriptions cp');
            $this->db->join('medicine m', 'm.id = cp.medicine_id', 'left');
            $this->db->join('consultations c', 'c.id = cp.consultation_id', 'left');
            $this->db->join('staff s1', 's1.id = c.doctor', 'left');
            $this->db->where('cp.patient_id', $patient_id);
            $this->db->order_by('cp.created_at', 'DESC');
            $this->db->limit(15);
            
            $query = $this->db->get();
            
            if ($query === false) {
                log_message('error', 'Consultation prescriptions query failed: ' . $this->db->error()['message']);
                return [];
            }
            
            return $query->result_array();
        }
        
        // Try to get from ipd_prescription_basic as fallback
        if ($this->db->table_exists('ipd_prescription_basic')) {
            $this->db->select('
                ipb.id,
                ipb.date as prescription_date,
                ipb.header_note as general_instructions,
                ipb.footer_note as pharmacy_notes,
                s.name as doctor_name,
                s.surname as doctor_surname
            ');
            $this->db->from('ipd_prescription_basic ipb');
            $this->db->join('ipd_details ipd', 'ipd.id = ipb.ipd_id', 'left');
            $this->db->join('staff s', 's.id = ipb.generated_by', 'left');
            $this->db->where('ipd.patient_id', $patient_id);
            $this->db->order_by('ipb.date', 'DESC');
            $this->db->limit(15);
            
            $query = $this->db->get();
            
            if ($query === false) {
                log_message('error', 'IPD prescriptions query failed: ' . $this->db->error()['message']);
                return [];
            }
            
            return $query->result_array();
        }
        
        return [];
        
    } catch (Exception $e) {
        log_message('error', 'Error getting patient prescriptions: ' . $e->getMessage());
        return [];
    }
}

/**
 * Get prescription items with error handling
 */
private function getPrescriptionItemsSafe($prescription_id) {
    try {
        if ($this->db->table_exists('prescription_items')) {
            $this->db->select('
                pi.id,
                pi.quantity,
                pi.dosage_instructions,
                pi.duration_days,
                pi.special_instructions,
                pi.unit_cost,
                pi.total_cost,
                m.medicine_name,
                m.medicine_category,
                m.medicine_type
            ');
            $this->db->from('prescription_items pi');
            $this->db->join('medicine m', 'm.id = pi.medicine_id', 'left');
            $this->db->where('pi.prescription_id', $prescription_id);
            $this->db->order_by('pi.id', 'ASC');
            
            $query = $this->db->get();
            
            if ($query === false) {
                log_message('error', 'Prescription items query failed: ' . $this->db->error()['message']);
                return [];
            }
            
            return $query->result_array();
        }
        
        return [];
        
    } catch (Exception $e) {
        log_message('error', 'Error getting prescription items: ' . $e->getMessage());
        return [];
    }
}

/**
 * Get patient clinical notes with proper table structure handling
 */
private function getPatientClinicalNotesSafe($patient_id) {
    try {
        $clinical_notes = [];
        
        // Get OPD visit notes - FIXED to use correct table structure
        if ($this->db->table_exists('opd_details') && $this->db->table_exists('visit_details')) {
            // First check what columns exist in opd_details
            $opd_fields = $this->db->list_fields('opd_details');
            $visit_fields = $this->db->list_fields('visit_details');
            
            // Build the select statement based on available columns
            $select_fields = [
                'od.id',
                'od.case_reference_id',
                'od.patient_id'
            ];
            
            // Add optional fields if they exist
            if (in_array('symptoms', $opd_fields)) {
                $select_fields[] = 'od.symptoms';
            } else {
                $select_fields[] = '"No symptoms recorded" as symptoms';
            }
            
            if (in_array('known_allergies', $opd_fields)) {
                $select_fields[] = 'od.known_allergies as any_known_allergies';
            } elseif (in_array('any_known_allergies', $opd_fields)) {
                $select_fields[] = 'od.any_known_allergies';
            } else {
                $select_fields[] = '"" as any_known_allergies';
            }
            
            if (in_array('note', $opd_fields)) {
                $select_fields[] = 'od.note';
            } else {
                $select_fields[] = '"" as note';
            }
            
            if (in_array('refference', $opd_fields)) {
                $select_fields[] = 'od.refference';
            } else {
                $select_fields[] = '"" as refference';
            }
            
            if (in_array('cons_doctor', $opd_fields)) {
                $select_fields[] = 'od.cons_doctor';
            } elseif (in_array('doctor', $opd_fields)) {
                $select_fields[] = 'od.doctor as cons_doctor';
            } else {
                $select_fields[] = 'NULL as cons_doctor';
            }
            
            // Add visit details fields
            if (in_array('appointment_date', $visit_fields)) {
                $select_fields[] = 'vd.appointment_date';
            } elseif (in_array('date', $visit_fields)) {
                $select_fields[] = 'vd.date as appointment_date';
            } else {
                $select_fields[] = 'NOW() as appointment_date';
            }
            
            // Add staff fields
            $select_fields[] = 's.name as doctor_name';
            $select_fields[] = 's.surname as doctor_surname';
            $select_fields[] = '"opd_visit" as note_type';
            
            $this->db->select(implode(', ', $select_fields));
            $this->db->from('opd_details od');
            $this->db->join('visit_details vd', 'vd.opd_details_id = od.id', 'left');
            $this->db->join('staff s', 's.id = od.cons_doctor', 'left');
            $this->db->where('od.patient_id', $patient_id);
            $this->db->order_by('vd.appointment_date', 'DESC');
            $this->db->limit(10);
            
            $query = $this->db->get();
            
            if ($query !== false) {
                $opd_notes = $query->result_array();
                
                foreach ($opd_notes as $note) {
                    $clinical_notes[] = [
                        'id' => $note['id'],
                        'date' => $note['appointment_date'],
                        'type' => 'OPD Visit',
                        'content' => $note['symptoms'] ?? 'No symptoms recorded',
                        'additional_info' => $note['note'] ?? '',
                        'doctor_name' => ($note['doctor_name'] ?? '') . ' ' . ($note['doctor_surname'] ?? ''),
                        'reference' => $note['case_reference_id'] ?? '',
                        'allergies' => $note['any_known_allergies'] ?? ''
                    ];
                }
            } else {
                log_message('error', 'Query error: ' . $this->db->error()['message'] . ' - Invalid query: ' . $this->db->last_query());
            }
        }
        
        // Get IPD consultant notes
        if ($this->db->table_exists('consultant_register') && $this->db->table_exists('ipd_details')) {
            $this->db->select('
                cr.id,
                cr.date,
                cr.instruction,
                s.name as doctor_name,
                s.surname as doctor_surname,
                ipd.case_reference_id,
                "consultant_note" as note_type
            ');
            $this->db->from('consultant_register cr');
            $this->db->join('ipd_details ipd', 'ipd.id = cr.ipd_id', 'left');
            $this->db->join('staff s', 's.id = cr.cons_doctor', 'left');
            $this->db->where('ipd.patient_id', $patient_id);
            $this->db->order_by('cr.date', 'DESC');
            $this->db->limit(10);
            
            $query = $this->db->get();
            
            if ($query !== false) {
                $consultant_notes = $query->result_array();
                
                foreach ($consultant_notes as $note) {
                    $clinical_notes[] = [
                        'id' => $note['id'],
                        'date' => $note['date'],
                        'type' => 'Consultant Note',
                        'content' => $note['instruction'],
                        'doctor_name' => ($note['doctor_name'] ?? '') . ' ' . ($note['doctor_surname'] ?? ''),
                        'reference' => $note['case_reference_id']
                    ];
                }
            } else {
                log_message('error', 'Consultant notes query failed: ' . $this->db->error()['message']);
            }
        }
        
        // Sort all notes by date descending
        usort($clinical_notes, function($a, $b) {
            return strtotime($b['date']) - strtotime($a['date']);
        });
        
        return $clinical_notes;
        
    } catch (Exception $e) {
        log_message('error', 'Error getting clinical notes: ' . $e->getMessage());
        return [];
    }
}

/**
 * Get patient OPD visits with proper error handling
 */
private function getPatientOPDVisitsSafe($patient_id) {
    try {
        if (!$this->db->table_exists('opd_details')) {
            return [];
        }
        
        // Check available fields
        $opd_fields = $this->db->list_fields('opd_details');
        $visit_fields = [];
        
        if ($this->db->table_exists('visit_details')) {
            $visit_fields = $this->db->list_fields('visit_details');
        }
        
        // Build select based on available fields
        $select_fields = [
            'od.id',
            'od.case_reference_id',
            'od.patient_id'
        ];
        
        // Add available fields
        foreach (['symptoms', 'note', 'height', 'weight', 'bp', 'pulse', 'temperature', 'respiration', 'cons_doctor'] as $field) {
            if (in_array($field, $opd_fields)) {
                $select_fields[] = 'od.' . $field;
            } else {
                $select_fields[] = 'NULL as ' . $field;
            }
        }
        
        // Handle allergies field variations
        if (in_array('any_known_allergies', $opd_fields)) {
            $select_fields[] = 'od.any_known_allergies';
        } elseif (in_array('known_allergies', $opd_fields)) {
            $select_fields[] = 'od.known_allergies as any_known_allergies';
        } else {
            $select_fields[] = 'NULL as any_known_allergies';
        }
        
        // Add visit details if table exists
        if (!empty($visit_fields)) {
            if (in_array('appointment_date', $visit_fields)) {
                $select_fields[] = 'vd.appointment_date';
            } elseif (in_array('date', $visit_fields)) {
                $select_fields[] = 'vd.date as appointment_date';
            }
            
            if (in_array('appointment_time', $visit_fields)) {
                $select_fields[] = 'vd.appointment_time';
            } elseif (in_array('time', $visit_fields)) {
                $select_fields[] = 'vd.time as appointment_time';
            }
        }
        
        // Add staff fields
        $select_fields[] = 's.name as doctor_name';
        $select_fields[] = 's.surname as doctor_surname';
        
        $this->db->select(implode(', ', $select_fields));
        $this->db->from('opd_details od');
        
        if (!empty($visit_fields)) {
            $this->db->join('visit_details vd', 'vd.opd_details_id = od.id', 'left');
        }
        
        $this->db->join('staff s', 's.id = od.cons_doctor', 'left');
        $this->db->where('od.patient_id', $patient_id);
        
        if (!empty($visit_fields) && in_array('appointment_date', $visit_fields)) {
            $this->db->order_by('vd.appointment_date', 'DESC');
        } elseif (!empty($visit_fields) && in_array('date', $visit_fields)) {
            $this->db->order_by('vd.date', 'DESC');
        } else {
            $this->db->order_by('od.id', 'DESC');
        }
        
        $this->db->limit(10);
        
        $query = $this->db->get();
        
        if ($query === false) {
            log_message('error', 'OPD visits query failed: ' . $this->db->error()['message']);
            return [];
        }
        
        return $query->result_array();
        
    } catch (Exception $e) {
        log_message('error', 'Error getting OPD visits: ' . $e->getMessage());
        return [];
    }
}

/**
 * Get patient IPD visits with proper error handling
 */
private function getPatientIPDVisitsSafe($patient_id) {
    try {
        if (!$this->db->table_exists('ipd_details')) {
            return [];
        }
        
        $this->db->select('
            ipd.id,
            ipd.case_reference_id,
            ipd.date as admission_date,
            ipd.discharged,
            ipd.discharge_date,
            ipd.symptoms,
            ipd.any_known_allergies,
            ipd.note,
            ipd.height,
            ipd.weight,
            ipd.bp,
            ipd.pulse,
            ipd.temperature,
            ipd.respiration,
            ipd.bed,
            ipd.bed_group_id,
            s.name as doctor_name,
            s.surname as doctor_surname
        ');
        $this->db->from('ipd_details ipd');
        $this->db->join('staff s', 's.id = ipd.cons_doctor', 'left');
        $this->db->where('ipd.patient_id', $patient_id);
        $this->db->order_by('ipd.date', 'DESC');
        $this->db->limit(10);
        
        $query = $this->db->get();
        
        if ($query === false) {
            log_message('error', 'IPD visits query failed: ' . $this->db->error()['message']);
            return [];
        }
        
        return $query->result_array();
        
    } catch (Exception $e) {
        log_message('error', 'Error getting IPD visits: ' . $e->getMessage());
        return [];
    }
}

/**
 * Get patient consultations with doctor notes
 */
private function getPatientConsultations($patient_id) {
    try {
        $this->db->select('
            c.id,
            c.date as consultation_date,
            c.symptoms,
            c.diagnosis,
            c.notes as consultation_notes,
            c.verification_status,
            c.verification_score,
            c.has_prescription,
            c.prescription_count,
            c.prescription_total_cost,
            c.general_prescription_notes,
            s.name as doctor_name,
            s.surname as doctor_surname,
            s.employee_id as doctor_employee_id
        ');
        $this->db->from('consultations c');
        $this->db->join('staff s', 's.id = c.doctor', 'left');
        $this->db->where('c.patient_id', $patient_id);
        $this->db->order_by('c.date', 'DESC');
        $this->db->limit(20); // Last 20 consultations
        
        $query = $this->db->get();
        return $query->result_array();
        
    } catch (Exception $e) {
        log_message('error', 'Error getting patient consultations: ' . $e->getMessage());
        return [];
    }
}

/**
 * Get patient prescriptions
 */
private function getPatientPrescriptions($patient_id) {
    try {
        // Try modern prescriptions table first
        if ($this->db->table_exists('prescriptions')) {
            $this->db->select('
                p.id,
                p.prescription_date,
                p.prescription_time,
                p.status,
                p.total_cost,
                p.general_instructions,
                p.pharmacy_notes,
                p.dispensed_at,
                s1.name as doctor_name,
                s1.surname as doctor_surname,
                s2.name as dispensed_by_name,
                s2.surname as dispensed_by_surname
            ');
            $this->db->from('prescriptions p');
            $this->db->join('staff s1', 's1.id = p.doctor_id', 'left');
            $this->db->join('staff s2', 's2.id = p.dispensed_by', 'left');
            $this->db->where('p.patient_id', $patient_id);
            $this->db->order_by('p.prescription_date', 'DESC');
            $this->db->limit(15);
            
            $query = $this->db->get();
            $prescriptions = $query->result_array();
            
            // Get prescription items for each prescription
            foreach ($prescriptions as &$prescription) {
                $prescription['items'] = $this->getPrescriptionItems($prescription['id']);
            }
            
            return $prescriptions;
        }
        
        // Fallback to consultation_prescriptions if available
        if ($this->db->table_exists('consultation_prescriptions')) {
            $this->db->select('
                cp.id,
                cp.created_at as prescription_date,
                cp.status,
                cp.total_cost,
                cp.dosage_instructions,
                cp.duration,
                cp.special_instructions,
                cp.dispensed_at,
                m.medicine_name,
                m.medicine_category,
                s1.name as doctor_name,
                s1.surname as doctor_surname,
                c.date as consultation_date
            ');
            $this->db->from('consultation_prescriptions cp');
            $this->db->join('medicine m', 'm.id = cp.medicine_id', 'left');
            $this->db->join('consultations c', 'c.id = cp.consultation_id', 'left');
            $this->db->join('staff s1', 's1.id = c.doctor', 'left');
            $this->db->where('cp.patient_id', $patient_id);
            $this->db->order_by('cp.created_at', 'DESC');
            $this->db->limit(15);
            
            $query = $this->db->get();
            return $query->result_array();
        }
        
        return [];
        
    } catch (Exception $e) {
        log_message('error', 'Error getting patient prescriptions: ' . $e->getMessage());
        return [];
    }
}

/**
 * Get prescription items for a prescription
 */
private function getPrescriptionItems($prescription_id) {
    try {
        if ($this->db->table_exists('prescription_items')) {
            $this->db->select('
                pi.id,
                pi.quantity,
                pi.dosage_instructions,
                pi.duration_days,
                pi.special_instructions,
                pi.unit_cost,
                pi.total_cost,
                m.medicine_name,
                m.medicine_category,
                m.medicine_type
            ');
            $this->db->from('prescription_items pi');
            $this->db->join('medicine m', 'm.id = pi.medicine_id', 'left');
            $this->db->where('pi.prescription_id', $prescription_id);
            $this->db->order_by('pi.id', 'ASC');
            
            $query = $this->db->get();
            return $query->result_array();
        }
        
        return [];
        
    } catch (Exception $e) {
        log_message('error', 'Error getting prescription items: ' . $e->getMessage());
        return [];
    }
}

/**
 * Get patient clinical notes from various sources
 */
private function getPatientClinicalNotes($patient_id) {
    try {
        $clinical_notes = [];
        
        // Get OPD visit notes
        if ($this->db->table_exists('opd_details')) {
            $this->db->select('
                od.id,
                od.opd_no,
                od.case_reference_id,
                od.appointment_date,
                od.symptoms,
                od.any_known_allergies,
                od.note,
                od.refference,
                od.cons_doctor,
                s.name as doctor_name,
                s.surname as doctor_surname,
                "opd_visit" as note_type
            ');
            $this->db->from('opd_details od');
            $this->db->join('staff s', 's.id = od.cons_doctor', 'left');
            $this->db->where('od.patient_id', $patient_id);
            $this->db->order_by('od.appointment_date', 'DESC');
            $this->db->limit(10);
            
            $query = $this->db->get();
            $opd_notes = $query->result_array();
            
            foreach ($opd_notes as $note) {
                $clinical_notes[] = [
                    'id' => $note['id'],
                    'date' => $note['appointment_date'],
                    'type' => 'OPD Visit',
                    'content' => $note['symptoms'],
                    'additional_info' => $note['note'],
                    'doctor_name' => ($note['doctor_name'] ?? '') . ' ' . ($note['doctor_surname'] ?? ''),
                    'reference' => $note['opd_no'],
                    'allergies' => $note['any_known_allergies']
                ];
            }
        }
        
        // Get IPD consultant notes
        if ($this->db->table_exists('consultant_register')) {
            $this->db->select('
                cr.id,
                cr.date,
                cr.instruction,
                s.name as doctor_name,
                s.surname as doctor_surname,
                ipd.case_reference_id,
                "consultant_note" as note_type
            ');
            $this->db->from('consultant_register cr');
            $this->db->join('ipd_details ipd', 'ipd.id = cr.ipd_id', 'left');
            $this->db->join('staff s', 's.id = cr.cons_doctor', 'left');
            $this->db->where('ipd.patient_id', $patient_id);
            $this->db->order_by('cr.date', 'DESC');
            $this->db->limit(10);
            
            $query = $this->db->get();
            $consultant_notes = $query->result_array();
            
            foreach ($consultant_notes as $note) {
                $clinical_notes[] = [
                    'id' => $note['id'],
                    'date' => $note['date'],
                    'type' => 'Consultant Note',
                    'content' => $note['instruction'],
                    'doctor_name' => ($note['doctor_name'] ?? '') . ' ' . ($note['doctor_surname'] ?? ''),
                    'reference' => $note['case_reference_id']
                ];
            }
        }
        
        // Sort all notes by date descending
        usort($clinical_notes, function($a, $b) {
            return strtotime($b['date']) - strtotime($a['date']);
        });
        
        return $clinical_notes;
        
    } catch (Exception $e) {
        log_message('error', 'Error getting clinical notes: ' . $e->getMessage());
        return [];
    }
}

/**
 * Get patient OPD visits
 */
private function getPatientOPDVisits($patient_id) {
    try {
        if (!$this->db->table_exists('opd_details')) {
            return [];
        }
        
        $this->db->select('
            od.id,
            od.opd_no,
            od.case_reference_id,
            od.appointment_date,
            od.appointment_time,
            od.symptoms,
            od.any_known_allergies,
            od.note,
            od.height,
            od.weight,
            od.bp,
            od.pulse,
            od.temperature,
            od.respiration,
            s.name as doctor_name,
            s.surname as doctor_surname
        ');
        $this->db->from('opd_details od');
        $this->db->join('staff s', 's.id = od.cons_doctor', 'left');
        $this->db->where('od.patient_id', $patient_id);
        $this->db->order_by('od.appointment_date', 'DESC');
        $this->db->limit(10);
        
        $query = $this->db->get();
        return $query->result_array();
        
    } catch (Exception $e) {
        log_message('error', 'Error getting OPD visits: ' . $e->getMessage());
        return [];
    }
}

/**
 * Get patient IPD visits
 */
private function getPatientIPDVisits($patient_id) {
    try {
        if (!$this->db->table_exists('ipd_details')) {
            return [];
        }
        
        $this->db->select('
            ipd.id,
            ipd.case_reference_id,
            ipd.date as admission_date,
            ipd.discharged,
            ipd.discharge_date,
            ipd.symptoms,
            ipd.any_known_allergies,
            ipd.note,
            ipd.height,
            ipd.weight,
            ipd.bp,
            ipd.pulse,
            ipd.temperature,
            ipd.respiration,
            ipd.bed,
            ipd.bed_group_id,
            s.name as doctor_name,
            s.surname as doctor_surname
        ');
        $this->db->from('ipd_details ipd');
        $this->db->join('staff s', 's.id = ipd.cons_doctor', 'left');
        $this->db->where('ipd.patient_id', $patient_id);
        $this->db->order_by('ipd.date', 'DESC');
        $this->db->limit(10);
        
        $query = $this->db->get();
        return $query->result_array();
        
    } catch (Exception $e) {
        log_message('error', 'Error getting IPD visits: ' . $e->getMessage());
        return [];
    }
}

/**
 * AJAX endpoint to get detailed consultation info with error handling
 */
public function get_consultation_details() {
    $consultation_id = $this->input->post('consultation_id');
    
    if (!$consultation_id) {
        echo json_encode(['status' => 'error', 'message' => 'Consultation ID required']);
        return;
    }
    
    try {
        if (!$this->db->table_exists('consultations')) {
            echo json_encode(['status' => 'error', 'message' => 'Consultations feature not available']);
            return;
        }
        
        // Get consultation details
        $this->db->select('
            c.*,
            s.name as doctor_name,
            s.surname as doctor_surname,
            s.employee_id as doctor_employee_id,
            p.patient_name,
            p.patient_unique_id
        ');
        $this->db->from('consultations c');
        $this->db->join('staff s', 's.id = c.doctor', 'left');
        $this->db->join('patients p', 'p.id = c.patient_id', 'left');
        $this->db->where('c.id', $consultation_id);
        
        $query = $this->db->get();
        
        if ($query === false) {
            echo json_encode(['status' => 'error', 'message' => 'Database error: ' . $this->db->error()['message']]);
            return;
        }
        
        if ($query->num_rows() == 0) {
            echo json_encode(['status' => 'error', 'message' => 'Consultation not found']);
            return;
        }
        
        $consultation = $query->row_array();
        
        // Get related prescriptions if any
        $consultation['prescriptions'] = $this->getPrescriptionsByConsultationSafe($consultation_id);
        
        echo json_encode([
            'status' => 'success',
            'data' => $consultation
        ]);
        
    } catch (Exception $e) {
        echo json_encode([
            'status' => 'error',
            'message' => 'Error loading consultation details: ' . $e->getMessage()
        ]);
    }
}

/**
 * Get prescriptions by consultation ID with error handling
 */
private function getPrescriptionsByConsultationSafe($consultation_id) {
    try {
        if ($this->db->table_exists('consultation_prescriptions')) {
            $this->db->select('
                cp.*,
                m.medicine_name,
                m.medicine_category
            ');
            $this->db->from('consultation_prescriptions cp');
            $this->db->join('medicine m', 'm.id = cp.medicine_id', 'left');
            $this->db->where('cp.consultation_id', $consultation_id);
            $this->db->order_by('cp.id', 'ASC');
            
            $query = $this->db->get();
            
            if ($query === false) {
                log_message('error', 'Consultation prescriptions query failed: ' . $this->db->error()['message']);
                return [];
            }
            
            return $query->result_array();
        }
        
        return [];
        
    } catch (Exception $e) {
        log_message('error', 'Error getting consultation prescriptions: ' . $e->getMessage());
        return [];
    }
}

/**
 * Get prescriptions by consultation ID
 */
private function getPrescriptionsByConsultation($consultation_id) {
    try {
        if ($this->db->table_exists('consultation_prescriptions')) {
            $this->db->select('
                cp.*,
                m.medicine_name,
                m.medicine_category
            ');
            $this->db->from('consultation_prescriptions cp');
            $this->db->join('medicine m', 'm.id = cp.medicine_id', 'left');
            $this->db->where('cp.consultation_id', $consultation_id);
            $this->db->order_by('cp.id', 'ASC');
            
            $query = $this->db->get();
            return $query->result_array();
        }
        
        return [];
        
    } catch (Exception $e) {
        log_message('error', 'Error getting consultation prescriptions: ' . $e->getMessage());
        return [];
    }
}

/**
 * AJAX endpoint to get detailed prescription info
 */
public function get_prescription_details() {
    $prescription_id = $this->input->post('prescription_id');
    
    if (!$prescription_id) {
        echo json_encode(['status' => 'error', 'message' => 'Prescription ID required']);
        return;
    }
    
    try {
        // Get prescription details
        $this->db->select('
            p.*,
            s1.name as doctor_name,
            s1.surname as doctor_surname,
            s2.name as dispensed_by_name,
            s2.surname as dispensed_by_surname,
            pt.patient_name
        ');
        $this->db->from('prescriptions p');
        $this->db->join('staff s1', 's1.id = p.doctor_id', 'left');
        $this->db->join('staff s2', 's2.id = p.dispensed_by', 'left');
        $this->db->join('patients pt', 'pt.id = p.patient_id', 'left');
        $this->db->where('p.id', $prescription_id);
        
        $query = $this->db->get();
        
        if ($query->num_rows() == 0) {
            echo json_encode(['status' => 'error', 'message' => 'Prescription not found']);
            return;
        }
        
        $prescription = $query->row_array();
        
        // Get prescription items
        $prescription['items'] = $this->getPrescriptionItems($prescription_id);
        
        echo json_encode([
            'status' => 'success',
            'data' => $prescription
        ]);
        
    } catch (Exception $e) {
        echo json_encode([
            'status' => 'error',
            'message' => 'Error loading prescription details: ' . $e->getMessage()
        ]);
    }
}

public function debug_nursing_notes($patient_id = null) {
    if (!$patient_id) {
        echo "<h1>Debug Nursing Notes</h1>";
        echo "<p>Please provide a patient ID: /admin/nursing/debug_nursing_notes/PATIENT_ID</p>";
        return;
    }
    
    echo "<h1>Debug Nursing Notes for Patient ID: {$patient_id}</h1>";
    
    try {
        // Check if nurse_note table exists
        if (!$this->db->table_exists('nurse_note')) {
            echo "<p>❌ nurse_note table does not exist</p>";
            return;
        }
        
        // Show table structure
        $fields = $this->db->list_fields('nurse_note');
        echo "<h2>nurse_note Table Structure:</h2>";
        echo "<p>Fields: " . implode(', ', $fields) . "</p>";
        
        // Show all notes in the table
        $this->db->select('*');
        $this->db->from('nurse_note');
        $this->db->order_by('date', 'DESC');
        $this->db->limit(10);
        $query = $this->db->get();
        $all_notes = $query->result_array();
        
        echo "<h2>Recent Notes in Database (Last 10):</h2>";
        if (empty($all_notes)) {
            echo "<p>❌ No notes found in database</p>";
        } else {
            echo "<table border='1' style='border-collapse: collapse; width: 100%;'>";
            echo "<tr><th>ID</th><th>Staff ID</th><th>IPD ID</th><th>Patient ID</th><th>Note</th><th>Date</th></tr>";
            foreach ($all_notes as $note) {
                echo "<tr>";
                echo "<td>" . ($note['id'] ?? 'N/A') . "</td>";
                echo "<td>" . ($note['staff_id'] ?? 'N/A') . "</td>";
                echo "<td>" . ($note['ipd_id'] ?? 'N/A') . "</td>";
                echo "<td>" . (isset($note['patient_id']) ? $note['patient_id'] : 'N/A') . "</td>";
                echo "<td>" . substr($note['note'] ?? 'N/A', 0, 50) . "...</td>";
                echo "<td>" . ($note['date'] ?? 'N/A') . "</td>";
                echo "</tr>";
            }
            echo "</table>";
        }
        
        // Try to get notes for specific patient
        echo "<h2>Notes for Patient {$patient_id}:</h2>";
        $patient_notes = $this->nursing_model->getPatientNurseNotes($patient_id);
        
        if (empty($patient_notes)) {
            echo "<p>❌ No notes found for this patient</p>";
            
            // Check if patient has IPD record
            $ipd_details = $this->nursing_model->getPatientIPDDetails($patient_id);
            if ($ipd_details) {
                echo "<p>✅ Patient has IPD record: " . json_encode($ipd_details) . "</p>";
                
                // Check if there are notes with this IPD ID
                if (in_array('ipd_id', $fields)) {
                    $this->db->select('*');
                    $this->db->from('nurse_note');
                    $this->db->where('ipd_id', $ipd_details['id']);
                    $query = $this->db->get();
                    $ipd_notes = $query->result_array();
                    
                    echo "<p>Notes with IPD ID {$ipd_details['id']}: " . count($ipd_notes) . "</p>";
                    if (!empty($ipd_notes)) {
                        foreach ($ipd_notes as $note) {
                            echo "<p>- " . $note['note'] . " (Date: " . $note['date'] . ")</p>";
                        }
                    }
                }
            } else {
                echo "<p>❌ Patient has no IPD record</p>";
            }
        } else {
            echo "<p>✅ Found " . count($patient_notes) . " notes for this patient</p>";
            foreach ($patient_notes as $note) {
                echo "<p>- " . $note['note'] . " (Date: " . $note['date'] . ")</p>";
            }
        }
        
    } catch (Exception $e) {
        echo "<p>❌ Error: " . $e->getMessage() . "</p>";
    }
}

/**
 * Fix nurse_note table structure and clean existing data
 * Add this method to your Nursing controller
 */
public function fix_nurse_note_table() {
    echo "<h1>Fixing Nurse Note Table Structure</h1>";
    
    try {
        // 1. Make ipd_id nullable since not all patients need to be IPD
        echo "<h2>1. Making ipd_id field nullable</h2>";
        $sql = "ALTER TABLE `nurse_note` MODIFY `ipd_id` int(11) DEFAULT NULL";
        $this->db->query($sql);
        echo "<p>✅ ipd_id field is now nullable</p>";
        
        // 2. Add patient_id field if it doesn't exist (it already exists based on your debug)
        echo "<h2>2. Ensuring patient_id field exists and is properly configured</h2>";
        $fields = $this->db->list_fields('nurse_note');
        if (in_array('patient_id', $fields)) {
            echo "<p>✅ patient_id field already exists</p>";
        } else {
            $sql = "ALTER TABLE `nurse_note` ADD `patient_id` int(11) DEFAULT NULL AFTER `ipd_id`";
            $this->db->query($sql);
            echo "<p>✅ Added patient_id field</p>";
        }
        
        // 3. Add index for better performance
        echo "<h2>3. Adding database indexes</h2>";
        try {
            $this->db->query("ALTER TABLE `nurse_note` ADD INDEX `idx_patient_id` (`patient_id`)");
            echo "<p>✅ Added index for patient_id</p>";
        } catch (Exception $e) {
            echo "<p>⚠️ Index might already exist: " . $e->getMessage() . "</p>";
        }
        
        try {
            $this->db->query("ALTER TABLE `nurse_note` ADD INDEX `idx_ipd_id` (`ipd_id`)");
            echo "<p>✅ Added index for ipd_id</p>";
        } catch (Exception $e) {
            echo "<p>⚠️ Index might already exist: " . $e->getMessage() . "</p>";
        }
        
        // 4. Clean up existing data - Fix NULL patient_id records
        echo "<h2>4. Cleaning existing data</h2>";
        
        // First, let's see what we can recover
        $this->db->select('nn.id, nn.ipd_id, nn.patient_id, nn.note, nn.date, ipd.patient_id as recovered_patient_id');
        $this->db->from('nurse_note nn');
        $this->db->join('ipd_details ipd', 'ipd.id = nn.ipd_id', 'left');
        $this->db->where('nn.patient_id IS NULL');
        $query = $this->db->get();
        $null_records = $query->result_array();
        
        echo "<p>Found " . count($null_records) . " records with NULL patient_id</p>";
        
        $recovered = 0;
        $deleted = 0;
        
        foreach ($null_records as $record) {
            if (!empty($record['recovered_patient_id'])) {
                // We can recover this record
                $this->db->where('id', $record['id']);
                $this->db->update('nurse_note', ['patient_id' => $record['recovered_patient_id']]);
                $recovered++;
                echo "<p>✅ Recovered patient_id for note ID {$record['id']}: set to {$record['recovered_patient_id']}</p>";
            } else {
                // Check if it's a test record we can safely delete
                if (strpos($record['note'], 'test') !== false || strpos($record['note'], 'Test') !== false) {
                    $this->db->where('id', $record['id']);
                    $this->db->delete('nurse_note');
                    $deleted++;
                    echo "<p>🗑️ Deleted test record ID {$record['id']}</p>";
                } else {
                    echo "<p>⚠️ Cannot recover note ID {$record['id']} - no IPD link and not a test record</p>";
                }
            }
        }
        
        echo "<p><strong>Summary:</strong> Recovered {$recovered} records, deleted {$deleted} test records</p>";
        
        // 5. Show updated table structure
        echo "<h2>5. Updated Table Structure</h2>";
        $field_data = $this->db->field_data('nurse_note');
        echo "<table border='1' style='border-collapse: collapse; width: 100%;'>";
        echo "<tr><th>Field</th><th>Type</th><th>Null Allowed</th><th>Default</th></tr>";
        foreach ($field_data as $field) {
            $null_status = $field->null ? '✅ YES' : '❌ NO';
            if ($field->name == 'ipd_id' && $field->null) {
                $null_status = '✅ YES (FIXED)';
            }
            echo "<tr>";
            echo "<td>" . $field->name . "</td>";
            echo "<td>" . $field->type . "</td>";
            echo "<td>" . $null_status . "</td>";
            echo "<td>" . ($field->default ?? 'NULL') . "</td>";
            echo "</tr>";
        }
        echo "</table>";
        
        // 6. Test with patient 56 (no IPD)
        echo "<h2>6. Testing with Patient 56 (No IPD)</h2>";
        
        $test_data = [
            'patient_id' => 56,
            'ipd_id' => null, // This should now work since ipd_id is nullable
            'note' => 'Test note for OPD patient (no IPD required)',
            'comment' => 'This should work now',
            'staff_id' => $this->getStaffId(),
            'date' => date('Y-m-d H:i:s')
        ];
        
        try {
            $result = $this->nursing_model->addNurseNote($test_data);
            echo "<p>✅ Successfully added note for non-IPD patient with ID: " . $result . "</p>";
            
            // Verify
            $this->db->select('*');
            $this->db->from('nurse_note');
            $this->db->where('id', $result);
            $query = $this->db->get();
            $inserted = $query->row_array();
            
            if ($inserted['patient_id'] == 56 && $inserted['ipd_id'] === null) {
                echo "<p>✅ Perfect! Non-IPD patient notes now work correctly</p>";
            }
            
        } catch (Exception $e) {
            echo "<p>❌ Test failed: " . $e->getMessage() . "</p>";
        }
        
        echo "<h2>7. Final Verification</h2>";
        
        // Count current records
        $this->db->select('
            COUNT(*) as total,
            SUM(CASE WHEN patient_id IS NOT NULL THEN 1 ELSE 0 END) as with_patient_id,
            SUM(CASE WHEN ipd_id IS NOT NULL THEN 1 ELSE 0 END) as with_ipd_id
        ');
        $this->db->from('nurse_note');
        $query = $this->db->get();
        $stats = $query->row_array();
        
        echo "<p><strong>Current Database Stats:</strong></p>";
        echo "<ul>";
        echo "<li>Total nursing notes: " . $stats['total'] . "</li>";
        echo "<li>Notes with patient_id: " . $stats['with_patient_id'] . "</li>";
        echo "<li>Notes with ipd_id: " . $stats['with_ipd_id'] . "</li>";
        echo "<li>Notes for non-IPD patients: " . ($stats['total'] - $stats['with_ipd_id']) . "</li>";
        echo "</ul>";
        
        echo "<h2>✅ Table Structure Fix Complete!</h2>";
        echo "<p>You can now:</p>";
        echo "<ul>";
        echo "<li>✅ Add nursing notes for IPD patients (with ipd_id)</li>";
        echo "<li>✅ Add nursing notes for OPD patients (without ipd_id)</li>";
        echo "<li>✅ Patient_id will always be saved correctly</li>";
        echo "<li>✅ Historical data has been cleaned up</li>";
        echo "</ul>";
        
        echo "<p><a href='" . base_url('admin/nursing/patient_profile/56') . "'>Test with Patient 56 Profile</a></p>";
        
    } catch (Exception $e) {
        echo "<p>❌ Error: " . $e->getMessage() . "</p>";
        echo "<p>Stack trace: " . $e->getTraceAsString() . "</p>";
    }
}

    public function vital_signs($patient_id = null) {
        $this->session->set_userdata('top_menu', 'Nursing');
        $this->session->set_userdata('sub_menu', 'nursing/vital_signs');
        
        $data['title'] = 'Nursing Vital Signs';
        
        if ($patient_id) {
            $data['patient'] = $this->getPatientById($patient_id);
            
            if (!$data['patient']) {
                $this->session->set_flashdata('error', 'Patient not found');
                redirect('admin/nursing');
            }
            
            $data['vital_signs'] = $this->nursing_model->getPatientVitalSigns($patient_id, 50);
        } else {
            $data['patient'] = null;
            $data['vital_signs'] = $this->nursing_model->getAllRecentVitalSigns(50);
        }
        
        $this->load->view('layout/header', $data);
        $this->load->view('admin/nursing/vital_signs', $data);
        $this->load->view('layout/footer', $data);
    }

public function get_vital_details() {
    header('Content-Type: application/json');
    
    if (!$this->input->is_ajax_request()) {
        echo json_encode(['status' => 'error', 'message' => 'Direct access not allowed']);
        return;
    }

    try {
        $vital_id = $this->input->post('vital_id');
        
        if (empty($vital_id)) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Vital signs ID is required'
            ]);
            return;
        }

        $vital = $this->nursing_model->getVitalSignsById($vital_id);
        
        if (!$vital) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Vital signs record not found'
            ]);
            return;
        }

        echo json_encode([
            'status' => 'success',
            'data' => $vital
        ]);

    } catch (Exception $e) {
        log_message('error', 'Get vital details error: ' . $e->getMessage());
        echo json_encode([
            'status' => 'error',
            'message' => 'Failed to load vital details: ' . $e->getMessage()
        ]);
    }
}

/**
 * Get procedure details for modal display
 */
public function get_procedure_details() {
    header('Content-Type: application/json');
    
    if (!$this->input->is_ajax_request()) {
        echo json_encode(['status' => 'error', 'message' => 'Direct access not allowed']);
        return;
    }

    try {
        $procedure_id = $this->input->post('procedure_id');
        
        if (empty($procedure_id)) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Procedure ID is required'
            ]);
            return;
        }

        $procedure = $this->nursing_model->getProcedureById($procedure_id);
        
        if (!$procedure) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Procedure record not found'
            ]);
            return;
        }

        echo json_encode([
            'status' => 'success',
            'data' => $procedure
        ]);

    } catch (Exception $e) {
        log_message('error', 'Get procedure details error: ' . $e->getMessage());
        echo json_encode([
            'status' => 'error',
            'message' => 'Failed to load procedure details: ' . $e->getMessage()
        ]);
    }
}

/**
 * Get assessment details for modal display
 */
public function get_assessment_details() {
    header('Content-Type: application/json');
    
    if (!$this->input->is_ajax_request()) {
        echo json_encode(['status' => 'error', 'message' => 'Direct access not allowed']);
        return;
    }

    try {
        $assessment_id = $this->input->post('assessment_id');
        
        if (empty($assessment_id)) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Assessment ID is required'
            ]);
            return;
        }

        $assessment = $this->nursing_model->getAssessmentById($assessment_id);
        
        if (!$assessment) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Assessment record not found'
            ]);
            return;
        }

        echo json_encode([
            'status' => 'success',
            'data' => $assessment
        ]);

    } catch (Exception $e) {
        log_message('error', 'Get assessment details error: ' . $e->getMessage());
        echo json_encode([
            'status' => 'error',
            'message' => 'Failed to load assessment details: ' . $e->getMessage()
        ]);
    }
}

/**
 * Get care plan details for modal display
 */
public function get_care_plan_details() {
    header('Content-Type: application/json');
    
    if (!$this->input->is_ajax_request()) {
        echo json_encode(['status' => 'error', 'message' => 'Direct access not allowed']);
        return;
    }

    try {
        $plan_id = $this->input->post('plan_id');
        
        if (empty($plan_id)) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Care plan ID is required'
            ]);
            return;
        }

        $care_plan = $this->nursing_model->getCarePlanById($plan_id);
        
        if (!$care_plan) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Care plan record not found'
            ]);
            return;
        }

        echo json_encode([
            'status' => 'success',
            'data' => $care_plan
        ]);

    } catch (Exception $e) {
        log_message('error', 'Get care plan details error: ' . $e->getMessage());
        echo json_encode([
            'status' => 'error',
            'message' => 'Failed to load care plan details: ' . $e->getMessage()
        ]);
    }
}

/**
 * Get medication details for modal display
 */
public function get_medication_details() {
    header('Content-Type: application/json');
    
    if (!$this->input->is_ajax_request()) {
        echo json_encode(['status' => 'error', 'message' => 'Direct access not allowed']);
        return;
    }

    try {
        $medication_id = $this->input->post('medication_id');
        
        if (empty($medication_id)) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Medication ID is required'
            ]);
            return;
        }

        $medication = $this->nursing_model->getMedicationById($medication_id);
        
        if (!$medication) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Medication record not found'
            ]);
            return;
        }

        echo json_encode([
            'status' => 'success',
            'data' => $medication
        ]);

    } catch (Exception $e) {
        log_message('error', 'Get medication details error: ' . $e->getMessage());
        echo json_encode([
            'status' => 'error',
            'message' => 'Failed to load medication details: ' . $e->getMessage()
        ]);
    }
}

public function test_nursing_note_route() {
    header('Content-Type: application/json');
    echo json_encode([
        'status' => 'success',
        'message' => 'Nursing note route is working',
        'timestamp' => date('Y-m-d H:i:s'),
        'method' => $_SERVER['REQUEST_METHOD'],
        'post_data' => $_POST
    ]);
}

    public function delete_vital_signs() {
        header('Content-Type: application/json');
        
        if (!$this->input->is_ajax_request()) {
            show_404();
        }

        try {
            $vital_id = $this->input->post('vital_id');
            
            if (empty($vital_id)) {
                echo json_encode(['status' => 'error', 'message' => 'Vital signs ID is required']);
                return;
            }

            $result = $this->nursing_model->deleteVitalSigns($vital_id);
            
            if ($result) {
                echo json_encode(['status' => 'success', 'message' => 'Vital signs deleted successfully']);
            } else {
                echo json_encode(['status' => 'error', 'message' => 'Failed to delete vital signs']);
            }

        } catch (Exception $e) {
            echo json_encode(['status' => 'error', 'message' => 'Error: ' . $e->getMessage()]);
        }
    }

    public function all_vital_signs() {
        $this->session->set_userdata('top_menu', 'Nursing');
        $this->session->set_userdata('sub_menu', 'nursing/vital_signs');
        
        $data['title'] = 'All Nursing Vital Signs';
        $data['patient'] = null;
        $data['vital_signs'] = $this->nursing_model->getAllRecentVitalSigns(100);
        
        $this->load->view('layout/header', $data);
        $this->load->view('admin/nursing/vital_signs', $data);
        $this->load->view('layout/footer', $data);
    }

    public function reports() {
        if (!$this->rbac->hasPrivilege('nursing', 'can_view')) {
            access_denied();
        }

        $this->session->set_userdata('top_menu', 'Nursing');
        $this->session->set_userdata('sub_menu', 'nursing/reports');
        
        $data['title'] = 'Nursing Reports';
        $data['daily_stats'] = $this->nursing_model->getDailyNursingStats();
        $data['procedure_stats'] = $this->nursing_model->getProcedureStats();
        $data['staff_workload'] = $this->nursing_model->getStaffWorkloadStats();
        
        $this->load->view('layout/header', $data);
        $this->load->view('admin/nursing/reports', $data);
        $this->load->view('layout/footer', $data);
    }

    private function getNursingStaffList() {
        try {
            if (method_exists($this->staff_model, 'get')) {
                $staff = $this->staff_model->get();
                return $staff;
            }
            
            $this->db->select('id, name, surname, employee_id');
            $this->db->from('staff');
            $this->db->where('is_active', 'yes');
            $this->db->order_by('name', 'ASC');
            
            $query = $this->db->get();
            return $query->result_array();
            
        } catch (Exception $e) {
            log_message('error', 'Error getting nursing staff: ' . $e->getMessage());
            return [];
        }
    }

    private function getStaffId() {
        try {
            if (method_exists($this->customlib, 'getStaffID')) {
                return $this->customlib->getStaffID();
            } elseif ($this->session->userdata('admin_id')) {
                return $this->session->userdata('admin_id');
            } else {
                return 1;
            }
        } catch (Exception $e) {
            log_message('error', 'Error getting staff ID: ' . $e->getMessage());
            return 1;
        }
    }

    private function getActiveIPDCount() {
        try {
            $this->db->select('COUNT(*) as count');
            $this->db->from('ipd_details');
            $this->db->where('discharged', 'no');
            $query = $this->db->get();
            $result = $query->row();
            return intval($result->count ?? 0);
        } catch (Exception $e) {
            return 0;
        }
    }

    private function getPendingProceduresCount() {
        try {
            if (!$this->db->table_exists('nursing_procedures')) {
                return 0;
            }
            $this->db->select('COUNT(*) as count');
            $this->db->from('nursing_procedures');
            $this->db->where_in('status', ['scheduled', 'in_progress']);
            $query = $this->db->get();
            $result = $query->row();
            return intval($result->count ?? 0);
        } catch (Exception $e) {
            return 0;
        }
    }

    private function getTodayAssessmentsCount() {
        try {
            if (!$this->db->table_exists('nursing_assessments')) {
                return 0;
            }
            $this->db->select('COUNT(*) as count');
            $this->db->from('nursing_assessments');
           $this->db->where('DATE(assessment_date)', date('Y-m-d'));
           $query = $this->db->get();
           $result = $query->row();
           return intval($result->count ?? 0);
       } catch (Exception $e) {
           return 0;
       }
   }

   private function getCriticalAlertsCount() {
       try {
           if (!$this->db->table_exists('nursing_assessments')) {
               return 0;
           }
           $this->db->select('COUNT(*) as count');
           $this->db->from('nursing_assessments');
           $this->db->where('risk_level', 'critical');
           $this->db->where('DATE(assessment_date) >=', date('Y-m-d', strtotime('-7 days')));
           $query = $this->db->get();
           $result = $query->row();
           return intval($result->count ?? 0);
       } catch (Exception $e) {
           return 0;
       }
   }

   private function getVitalsToday() {
       try {
           if (!$this->db->table_exists('nursing_vital_signs')) {
               return 0;
           }
           $this->db->select('COUNT(*) as count');
           $this->db->from('nursing_vital_signs');
           $this->db->where('DATE(recorded_at)', date('Y-m-d'));
           $query = $this->db->get();
           $result = $query->row();
           return intval($result->count ?? 0);
       } catch (Exception $e) {
           return 0;
       }
   }

   private function getMedicationsToday() {
       try {
           if (!$this->db->table_exists('medication_report')) {
               return 0;
           }
           $this->db->select('COUNT(*) as count');
           $this->db->from('medication_report');
           $this->db->where('DATE(date)', date('Y-m-d'));
           $query = $this->db->get();
           $result = $query->row();
           return intval($result->count ?? 0);
       } catch (Exception $e) {
           return 0;
       }
   }

   public function debug_forms() {
       echo "<h1>Nursing Forms Debug</h1>";
       
       echo "<h2>Vital Signs Table Structure</h2>";
       if ($this->db->table_exists('nursing_vital_signs')) {
           $fields = $this->db->list_fields('nursing_vital_signs');
           echo "<p>Table exists. Fields: " . implode(', ', $fields) . "</p>";
       } else {
           echo "<p>❌ nursing_vital_signs table does not exist</p>";
       }
       
       echo "<h2>Nurse Note Table Structure</h2>";
       if ($this->db->table_exists('nurse_note')) {
           $fields = $this->db->list_fields('nurse_note');
           echo "<p>Table exists. Fields: " . implode(', ', $fields) . "</p>";
           
           $field_data = $this->db->field_data('nurse_note');
           echo "<h3>Field Details:</h3>";
           echo "<table border='1' style='border-collapse: collapse;'>";
           echo "<tr><th>Field</th><th>Type</th><th>Max Length</th><th>Null</th><th>Default</th></tr>";
           foreach ($field_data as $field) {
               echo "<tr>";
               echo "<td>" . $field->name . "</td>";
               echo "<td>" . $field->type . "</td>";
               echo "<td>" . $field->max_length . "</td>";
               echo "<td>" . ($field->null ? 'YES' : 'NO') . "</td>";
               echo "<td>" . ($field->default ?? 'NULL') . "</td>";
               echo "</tr>";
           }
           echo "</table>";
       } else {
           echo "<p>❌ nurse_note table does not exist</p>";
       }
       
       echo "<h2>Staff ID Test</h2>";
       echo "<p>Current Staff ID: " . $this->getStaffId() . "</p>";
       
       echo "<h2>Patient Search Test</h2>";
       try {
           $patients = $this->searchPatientsInDatabase('test', 'all');
           echo "<p>✅ Patient search working - found " . count($patients) . " patients</p>";
       } catch (Exception $e) {
           echo "<p>❌ Patient search error: " . $e->getMessage() . "</p>";
       }
       
       echo "<h2>Form Validation Test</h2>";
       echo "<h3>POST Data:</h3>";
       echo "<pre>" . print_r($_POST, true) . "</pre>";
       
       echo "<h2>Database Connection Test</h2>";
       try {
           $this->db->select('COUNT(*) as count');
           $this->db->from('patients');
           $query = $this->db->get();
           $result = $query->row();
           echo "<p>✅ Database connected - " . $result->count . " patients found</p>";
       } catch (Exception $e) {
           echo "<p>❌ Database error: " . $e->getMessage() . "</p>";
       }
       
       echo "<h2>Permissions Test</h2>";
       try {
           if (method_exists($this->rbac, 'hasPrivilege')) {
               $nursing_view = $this->rbac->hasPrivilege('nursing', 'can_view');
               $nursing_add = $this->rbac->hasPrivilege('nursing', 'can_add');
               $note_add = $this->rbac->hasPrivilege('nurse_note', 'can_add');
               
               echo "<p>Nursing View: " . ($nursing_view ? '✅' : '❌') . "</p>";
               echo "<p>Nursing Add: " . ($nursing_add ? '✅' : '❌') . "</p>";
               echo "<p>Note Add: " . ($note_add ? '✅' : '❌') . "</p>";
           } else {
               echo "<p>❌ RBAC not available</p>";
           }
       } catch (Exception $e) {
           echo "<p>❌ Permission check error: " . $e->getMessage() . "</p>";
       }
       
       echo "<h2>Test Table Creation</h2>";
       try {
           $this->nursing_model->forceInitialization();
           echo "<p>✅ Table creation successful</p>";
       } catch (Exception $e) {
           echo "<p>❌ Table creation error: " . $e->getMessage() . "</p>";
       }
   }

   public function test_vital_signs() {
       echo "<h1>Test Vital Signs Insertion</h1>";
       
       try {
           $test_data = [
               'patient_id' => 1,
               'temperature' => 37.5,
               'systolic_bp' => 120,
               'diastolic_bp' => 80,
               'pulse_rate' => 72,
               'respiratory_rate' => 16,
               'recorded_by' => $this->getStaffId(),
               'recorded_at' => date('Y-m-d H:i:s')
           ];
           
           echo "<h2>Test Data:</h2>";
           echo "<pre>" . print_r($test_data, true) . "</pre>";
           
           $result = $this->nursing_model->addVitalSigns($test_data);
           echo "<p>✅ Vital signs inserted successfully with ID: " . $result . "</p>";
           
       } catch (Exception $e) {
           echo "<p>❌ Error: " . $e->getMessage() . "</p>";
       }
   }

   public function test_nursing_note() {
       echo "<h1>Test Nursing Note Insertion</h1>";
       
       try {
           $test_data = [
               'patient_id' => 1,
               'note' => 'Test nursing note',
               'comment' => 'Test comment',
               'staff_id' => $this->getStaffId(),
               'date' => date('Y-m-d H:i:s')
           ];
           
           echo "<h2>Test Data:</h2>";
           echo "<pre>" . print_r($test_data, true) . "</pre>";
           
           $result = $this->nursing_model->addNurseNote($test_data);
           echo "<p>✅ Nursing note inserted successfully with ID: " . $result . "</p>";
           
       } catch (Exception $e) {
           echo "<p>❌ Error: " . $e->getMessage() . "</p>";
       }
   }

   public function test() {
       echo "<h1>Nursing Module Test</h1>";
       echo "<p>Testing nursing module functionality with HMS integration...</p>";
       
       try {
           echo "<h2>Database Status</h2>";
           $tables = ['patients', 'nurse_note', 'staff', 'ipd_details'];
           
           foreach ($tables as $table) {
               if ($this->db->table_exists($table)) {
                   echo "<p>✓ {$table} table exists</p>";
               } else {
                   echo "<p>⚠ {$table} table missing</p>";
               }
           }

           echo "<h2>HMS Integration Status</h2>";
           if (method_exists($this->patient_model, 'getpatientDetails')) {
               echo "<p>✓ patient_model->getpatientDetails() available</p>";
           } else {
               echo "<p>⚠ patient_model->getpatientDetails() not available</p>";
           }
           
           if (method_exists($this->customlib, 'getStaffID')) {
               echo "<p>✓ customlib->getStaffID() available</p>";
           } else {
               echo "<p>⚠ customlib->getStaffID() not available</p>";
           }

           echo "<h2>Patient Search Test</h2>";
           $test_patients = $this->searchPatientsInDatabase('test', 'all');
           echo "<p>✓ Patient search working - found " . count($test_patients) . " patients</p>";

           echo "<h2>System Status</h2>";
           echo "<p>✓ Nursing Controller: Active</p>";
           echo "<p>✓ HMS Integration: Active</p>";
           echo "<p>✓ Patient Search: Ready</p>";
           echo "<p>✓ Vital Signs: Ready</p>";
           echo "<p>✓ Nursing Notes: Ready</p>";

           echo "<p>Current Staff ID: " . $this->getStaffId() . "</p>";
           echo "<p>Nursing Staff Count: " . count($this->getNursingStaffList()) . "</p>";
           
           echo "<h2>Debug Links</h2>";
           echo "<p><a href='" . base_url('admin/nursing/debug_forms') . "'>Debug Forms</a></p>";
           echo "<p><a href='" . base_url('admin/nursing/test_vital_signs') . "'>Test Vital Signs</a></p>";
           echo "<p><a href='" . base_url('admin/nursing/test_nursing_note') . "'>Test Nursing Note</a></p>";

       } catch (Exception $e) {
           echo "<p style='color: red;'>Error: " . $e->getMessage() . "</p>";
       }
   }

   public function setup() {
       echo "<h1>Nursing Module Setup</h1>";
       
       try {
           $this->nursing_model->forceInitialization();
           echo "<p>✓ Nursing tables created successfully</p>";
           echo "<p>✓ Default data inserted</p>";
           echo "<p>✓ HMS integration verified</p>";
           echo "<p><a href='" . base_url('admin/nursing') . "'>Go to Nursing Dashboard</a></p>";
           
       } catch (Exception $e) {
           echo "<p style='color: red;'>Setup failed: " . $e->getMessage() . "</p>";
       }
   }
   
    public function debugJsonOutput() {
        // Clear any previous output
        ob_clean();
        
        // Set proper headers
        header('Content-Type: application/json');
        header('Cache-Control: no-cache, must-revalidate');
        
        // Disable error display
        ini_set('display_errors', 0);
        error_reporting(0);
        
        try {
            echo json_encode([
                'status' => 'success',
                'message' => 'JSON output is working correctly',
                'timestamp' => date('Y-m-d H:i:s'),
                'test_data' => [
                    'php_version' => phpversion(),
                    'codeigniter_version' => CI_VERSION,
                    'json_last_error' => json_last_error_msg()
                ]
            ]);
        } catch (Exception $e) {
            echo json_encode([
                'status' => 'error',
                'message' => 'JSON test failed: ' . $e->getMessage()
            ]);
        }
        exit;
    }

    // CORRECTED getAllNursingNotes with proper JSON handling
    public function getAllNursingNotes() {
        // Clear any previous output and set headers
        ob_clean();
        header('Content-Type: application/json');
        header('Cache-Control: no-cache, must-revalidate');
        
        // Disable error display to prevent JSON corruption
        ini_set('display_errors', 0);
        error_reporting(0);

        if (!$this->input->is_ajax_request()) {
            echo json_encode(['status' => 'error', 'message' => 'Direct access not allowed']);
            exit;
        }

        try {
            $page = $this->input->post('page') ?: 1;
            $perPage = $this->input->post('per_page') ?: 25;
            $dateFilter = $this->input->post('date_filter') ?: 'week';
            $staffFilter = $this->input->post('staff_filter') ?: '';
            $patientSearch = $this->input->post('patient_search') ?: '';

            // Build date filter condition
            $dateCondition = '1=1'; // Default to show all
            switch($dateFilter) {
                case 'today':
                    $dateCondition = "DATE(nn.date) = '" . date('Y-m-d') . "'";
                    break;
                case 'yesterday':
                    $dateCondition = "DATE(nn.date) = '" . date('Y-m-d', strtotime('-1 day')) . "'";
                    break;
                case 'week':
                    $dateCondition = "DATE(nn.date) >= '" . date('Y-m-d', strtotime('-7 days')) . "'";
                    break;
                case 'month':
                    $dateCondition = "DATE(nn.date) >= '" . date('Y-m-d', strtotime('-30 days')) . "'";
                    break;
            }

            // Start with simple query to avoid JOIN issues
            $this->db->select('
                nn.id, 
                nn.date, 
                nn.note, 
                nn.comment, 
                nn.staff_id,
                nn.ipd_id,
                COALESCE(s.name, "Unknown") as nurse_name, 
                COALESCE(s.surname, "") as nurse_surname
            ');
            
            $this->db->from('nurse_note nn');
            $this->db->join('staff s', 's.id = nn.staff_id', 'left');
            
            // Apply filters
            if ($dateCondition !== '1=1') {
                $this->db->where($dateCondition);
            }
            
            if (!empty($staffFilter)) {
                $this->db->where('nn.staff_id', $staffFilter);
            }
            
            if (!empty($patientSearch)) {
                $this->db->like('nn.note', $patientSearch);
            }

            // Get total count
            $totalQuery = clone $this->db;
            $totalRecords = $totalQuery->count_all_results();

            // Apply pagination
            $offset = ($page - 1) * $perPage;
            $this->db->limit($perPage, $offset);
            $this->db->order_by('nn.date', 'DESC');

            $query = $this->db->get();
            $notes = $query->result_array();

            // Process each note to add patient information
            foreach ($notes as &$note) {
                $note['patient_name'] = 'General Note';
                $note['id'] = 'N/A';
               // $note['ipd_no'] = 'N/A';
                $note['patient_id'] = 0;
                $note['note_type'] = 'general';
                
                // Try to get patient info if ipd_id exists
                if (!empty($note['ipd_id'])) {
                    $patient_info = $this->getPatientFromIPD($note['ipd_id']);
                    if ($patient_info) {
                        $note['patient_name'] = $patient_info['patient_name'];
                        $note['id'] = $patient_info['id'];
                       // $note['ipd_no'] = $patient_info['ipd_no'];
                        $note['patient_id'] = $patient_info['patient_id'];
                    }
                }
            }

            $totalPages = ceil($totalRecords / $perPage);

            $response = [
                'status' => 'success',
                'notes' => $notes,
                'pagination' => [
                    'current_page' => (int)$page,
                    'total_pages' => $totalPages,
                    'total_records' => $totalRecords,
                    'per_page' => $perPage
                ]
            ];

            echo json_encode($response);

        } catch (Exception $e) {
            $response = [
                'status' => 'error',
                'message' => 'Failed to load nursing notes: ' . $e->getMessage(),
                'notes' => [],
                'pagination' => [
                    'current_page' => 1,
                    'total_pages' => 1,
                    'total_records' => 0,
                    'per_page' => 25
                ]
            ];
            
            echo json_encode($response);
        }
        
        exit; // Important: prevent any additional output
    }
    
        private function getPatientFromIPD($ipd_id) {
        if (empty($ipd_id)) {
            return null;
        }
        
        try {
            $this->db->select('p.patient_name, p.id, p.id as patient_id, ipd.id');
            $this->db->from('ipd_details ipd');
            $this->db->join('patients p', 'p.id = ipd.patient_id', 'left');
            $this->db->where('ipd.id', $ipd_id);
            $this->db->limit(1);
            
            $query = $this->db->get();
            
            if ($query && $query->num_rows() > 0) {
                return $query->row_array();
            }
        } catch (Exception $e) {
            // Ignore errors in helper method
        }
        
        return null;
    }

    // Simple version for testing
    public function getAllNursingNotesSimple() {
        ob_clean();
        header('Content-Type: application/json');
        ini_set('display_errors', 0);

        try {
            $this->db->select('nn.id, nn.date, nn.note, nn.comment, nn.staff_id, s.name as nurse_name, s.surname as nurse_surname');
            $this->db->from('nurse_note nn');
            $this->db->join('staff s', 's.id = nn.staff_id', 'left');
            $this->db->order_by('nn.date', 'DESC');
            $this->db->limit(25);

            $query = $this->db->get();
            $notes = $query->result_array();

            // Add required fields
            foreach ($notes as &$note) {
                $note['patient_name'] = 'General Note';
                $note['id'] = 'N/A';
              //  $note['ipd_no'] = 'N/A';
                $note['patient_id'] = 0;
                $note['note_type'] = 'general';
            }

            echo json_encode([
                'status' => 'success',
                'notes' => $notes,
                'pagination' => [
                    'current_page' => 1,
                    'total_pages' => 1,
                    'total_records' => count($notes),
                    'per_page' => 25
                ]
            ]);

        } catch (Exception $e) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Simple query failed: ' . $e->getMessage()
            ]);
        }
        
        exit;
    }

    // Raw test method to check what's being output
    public function testRawOutput() {
        echo "Raw PHP output test - if you see this, PHP is working";
        echo "\n";
        echo "Current time: " . date('Y-m-d H:i:s');
        exit;
    }

    public function getNursingStaff() {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }

        try {
            $staff = $this->getNursingStaffList();
            
            echo json_encode([
                'status' => 'success',
                'staff' => $staff
            ]);

        } catch (Exception $e) {
            log_message('error', 'Get nursing staff error: ' . $e->getMessage());
            echo json_encode([
                'status' => 'error',
                'message' => 'Failed to load staff list: ' . $e->getMessage()
            ]);
        }
    }

    public function getNoteDetails() {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }

        try {
            $noteId = $this->input->post('note_id');
            
            if (empty($noteId)) {
                echo json_encode([
                    'status' => 'error',
                    'message' => 'Note ID is required'
                ]);
                return;
            }

            // Check if table exists
            if (!$this->db->table_exists('nurse_note')) {
                echo json_encode([
                    'status' => 'error',
                    'message' => 'Nurse note table does not exist'
                ]);
                return;
            }

            $this->db->select('
                nn.id, 
                nn.date, 
                nn.note, 
                nn.comment, 
                nn.staff_id,
                COALESCE(s.name, "Unknown") as nurse_name, 
                COALESCE(s.surname, "") as nurse_surname,
                COALESCE(p.patient_name, "Unknown Patient") as patient_name, 
                COALESCE(p.id, 0) as patient_id,
                COALESCE(ipd.id, "N/A") as ipd_no,
                "general" as note_type
            ');
            $this->db->from('nurse_note nn');
            $this->db->join('staff s', 's.id = nn.staff_id', 'left');
            $this->db->join('ipd_details ipd', 'ipd.id = nn.ipd_id', 'left');
            $this->db->join('patients p', 'p.id = ipd.patient_id', 'left');
            $this->db->where('nn.id', $noteId);

            $query = $this->db->get();
            
            if (!$query) {
                $db_error = $this->db->error();
                log_message('error', 'Database error in getNoteDetails: ' . json_encode($db_error));
                
                echo json_encode([
                    'status' => 'error',
                    'message' => 'Database error occurred',
                    'debug' => $db_error
                ]);
                return;
            }

            $note = $query->row_array();

            if (!$note) {
                echo json_encode([
                    'status' => 'error',
                    'message' => 'Note not found'
                ]);
                return;
            }

            echo json_encode([
                'status' => 'success',
                'note' => $note
            ]);

        } catch (Exception $e) {
            log_message('error', 'Get note details error: ' . $e->getMessage());
            echo json_encode([
                'status' => 'error',
                'message' => 'Failed to load note details: ' . $e->getMessage()
            ]);
        }
    }
    
        public function debug_nursing_tables() {
        echo "<h1>Nursing Tables Debug</h1>";
        
        // Check nurse_note table
        echo "<h2>nurse_note table:</h2>";
        if ($this->db->table_exists('nurse_note')) {
            echo "<p>✅ Table exists</p>";
            
            // Get table structure
            $fields = $this->db->list_fields('nurse_note');
            echo "<p><strong>Fields:</strong> " . implode(', ', $fields) . "</p>";
            
            // Get sample data
            $this->db->limit(5);
            $query = $this->db->get('nurse_note');
            $sample_data = $query->result_array();
            
            echo "<p><strong>Sample records:</strong> " . count($sample_data) . "</p>";
            if (!empty($sample_data)) {
                echo "<pre>" . print_r($sample_data[0], true) . "</pre>";
            }
            
        } else {
            echo "<p>❌ nurse_note table does not exist</p>";
        }
        
        // Check related tables
        echo "<h2>Related tables:</h2>";
        $tables = ['staff', 'ipd_details', 'patients'];
        foreach ($tables as $table) {
            if ($this->db->table_exists($table)) {
                $this->db->select('COUNT(*) as count');
                $this->db->from($table);
                $query = $this->db->get();
                $count = $query->row()->count;
                echo "<p>✅ {$table}: {$count} records</p>";
            } else {
                echo "<p>❌ {$table} table missing</p>";
            }
        }
        
        // Test the query
        echo "<h2>Test Query:</h2>";
        try {
            $this->db->select('nn.id, nn.date, nn.note');
            $this->db->from('nurse_note nn');
            $this->db->limit(1);
            $query = $this->db->get();
            
            if ($query) {
                echo "<p>✅ Basic query works</p>";
                echo "<p>Last query: " . $this->db->last_query() . "</p>";
            } else {
                echo "<p>❌ Query failed</p>";
            }
        } catch (Exception $e) {
            echo "<p>❌ Query error: " . $e->getMessage() . "</p>";
        }
    }

    public function exportNursingNotes() {
        try {
            $dateFilter = $this->input->get('date_filter') ?: 'week';
            $staffFilter = $this->input->get('staff_filter') ?: '';
            $patientSearch = $this->input->get('patient_search') ?: '';

            // Build date filter condition
            $dateCondition = '';
            switch($dateFilter) {
                case 'today':
                    $dateCondition = "DATE(nn.date) = '" . date('Y-m-d') . "'";
                    break;
                case 'yesterday':
                    $dateCondition = "DATE(nn.date) = '" . date('Y-m-d', strtotime('-1 day')) . "'";
                    break;
                case 'week':
                    $dateCondition = "DATE(nn.date) >= '" . date('Y-m-d', strtotime('-7 days')) . "'";
                    break;
                case 'month':
                    $dateCondition = "DATE(nn.date) >= '" . date('Y-m-d', strtotime('-30 days')) . "'";
                    break;
                case 'all':
                default:
                    $dateCondition = '1=1';
                    break;
            }

            // Build the query
            $this->db->select('
                nn.date as "Date/Time",
                p.patient_name as "Patient Name",
                p.id as "Patient ID",
                ipd.id as "IPD No",
                CONCAT(s.name, " ", s.surname) as "Staff Name",
                nn.note as "Note",
                nn.comment as "Follow-up"
            ');
            $this->db->from('nurse_note nn');
            $this->db->join('staff s', 's.id = nn.staff_id', 'left');
            $this->db->join('ipd_details ipd', 'ipd.id = nn.ipd_id', 'left');
            $this->db->join('patients p', 'p.id = ipd.patient_id', 'left');
            
            // Apply filters
            if (!empty($dateCondition)) {
                $this->db->where($dateCondition);
            }
            
            if (!empty($staffFilter)) {
                $this->db->where('nn.staff_id', $staffFilter);
            }
            
            if (!empty($patientSearch)) {
                $this->db->group_start();
                $this->db->like('p.patient_name', $patientSearch);
                $this->db->or_like('p.id', $patientSearch);
                $this->db->or_like('ipd.id', $patientSearch);
                $this->db->group_end();
            }

            $this->db->order_by('nn.date', 'DESC');
            $query = $this->db->get();
            $data = $query->result_array();

            // Generate CSV
            $filename = 'nursing_notes_' . date('Y-m-d_H-i-s') . '.csv';
            header('Content-Type: text/csv');
            header('Content-Disposition: attachment; filename="' . $filename . '"');

            $output = fopen('php://output', 'w');
            
            // Write headers
            if (!empty($data)) {
                fputcsv($output, array_keys($data[0]));
                
                // Write data
                foreach ($data as $row) {
                    fputcsv($output, $row);
                }
            }
            
            fclose($output);

        } catch (Exception $e) {
            log_message('error', 'Export nursing notes error: ' . $e->getMessage());
            $this->session->set_flashdata('error', 'Failed to export nursing notes');
            redirect('admin/nursing/reports');
        }
    }

    public function exportVitalSigns() {
        try {
            $this->db->select('
                vs.recorded_at as "Date/Time",
                p.patient_name as "Patient Name",
                p.id as "Patient ID",
                vs.temperature as "Temperature (°C)",
                CONCAT(vs.systolic_bp, "/", vs.diastolic_bp) as "Blood Pressure",
                vs.pulse_rate as "Pulse Rate",
                vs.respiratory_rate as "Respiratory Rate",
                vs.oxygen_saturation as "O2 Saturation (%)",
                vs.weight as "Weight (kg)",
                vs.height as "Height (cm)",
                vs.bmi as "BMI",
                vs.pain_scale as "Pain Scale",
                CONCAT(s.name, " ", s.surname) as "Recorded By",
                vs.notes as "Notes"
            ');
            $this->db->from('nursing_vital_signs vs');
            $this->db->join('patients p', 'p.id = vs.patient_id', 'left');
            $this->db->join('staff s', 's.id = vs.recorded_by', 'left');
            $this->db->where('DATE(vs.recorded_at) >=', date('Y-m-d', strtotime('-30 days')));
            $this->db->order_by('vs.recorded_at', 'DESC');

            $query = $this->db->get();
            $data = $query->result_array();

            // Generate CSV
            $filename = 'vital_signs_' . date('Y-m-d_H-i-s') . '.csv';
            header('Content-Type: text/csv');
            header('Content-Disposition: attachment; filename="' . $filename . '"');

            $output = fopen('php://output', 'w');
            
            if (!empty($data)) {
                fputcsv($output, array_keys($data[0]));
                foreach ($data as $row) {
                    fputcsv($output, $row);
                }
            }
            
            fclose($output);

        } catch (Exception $e) {
            log_message('error', 'Export vital signs error: ' . $e->getMessage());
            $this->session->set_flashdata('error', 'Failed to export vital signs');
            redirect('admin/nursing/reports');
        }
    }

    public function exportProcedures() {
        try {
            if (!$this->db->table_exists('nursing_procedures')) {
                $this->session->set_flashdata('error', 'Procedures table not available');
                redirect('admin/nursing/reports');
                return;
            }

            $this->db->select('
                p.created_at as "Date Created",
                p.performed_at as "Date Performed",
                pat.patient_name as "Patient Name",
                pat.id as "Patient ID",
                p.procedure_name as "Procedure",
                p.procedure_type as "Type",
                p.status as "Status",
                p.outcome as "Outcome",
                p.duration_minutes as "Duration (min)",
                CONCAT(s.name, " ", s.surname) as "Performed By",
                p.description as "Description",
                p.complications as "Complications"
            ');
            $this->db->from('nursing_procedures p');
            $this->db->join('patients pat', 'pat.id = p.patient_id', 'left');
            $this->db->join('staff s', 's.id = p.performed_by', 'left');
            $this->db->where('DATE(p.created_at) >=', date('Y-m-d', strtotime('-30 days')));
            $this->db->order_by('p.created_at', 'DESC');

            $query = $this->db->get();
            $data = $query->result_array();

            // Generate CSV
            $filename = 'nursing_procedures_' . date('Y-m-d_H-i-s') . '.csv';
            header('Content-Type: text/csv');
            header('Content-Disposition: attachment; filename="' . $filename . '"');

            $output = fopen('php://output', 'w');
            
            if (!empty($data)) {
                fputcsv($output, array_keys($data[0]));
                foreach ($data as $row) {
                    fputcsv($output, $row);
                }
            }
            
            fclose($output);

        } catch (Exception $e) {
            log_message('error', 'Export procedures error: ' . $e->getMessage());
            $this->session->set_flashdata('error', 'Failed to export procedures');
            redirect('admin/nursing/reports');
        }
    }

    public function exportAssessments() {
        try {
            if (!$this->db->table_exists('nursing_assessments')) {
                $this->session->set_flashdata('error', 'Assessments table not available');
                redirect('admin/nursing/reports');
                return;
            }

            $this->db->select('
                a.assessment_date as "Date",
                p.patient_name as "Patient Name",
                p.id as "Patient ID",
                a.assessment_type as "Assessment Type",
                a.assessment_category as "Category",
                a.risk_level as "Risk Level",
                a.findings as "Findings",
                a.recommendations as "Recommendations",
                a.risk_factors as "Risk Factors",
                a.follow_up_required as "Follow-up Required",
                a.follow_up_date as "Follow-up Date",
                CONCAT(s.name, " ", s.surname) as "Assessed By"
            ');
            $this->db->from('nursing_assessments a');
            $this->db->join('patients p', 'p.id = a.patient_id', 'left');
            $this->db->join('staff s', 's.id = a.assessed_by', 'left');
            $this->db->where('DATE(a.assessment_date) >=', date('Y-m-d', strtotime('-30 days')));
            $this->db->order_by('a.assessment_date', 'DESC');

            $query = $this->db->get();
            $data = $query->result_array();

            // Generate CSV
            $filename = 'nursing_assessments_' . date('Y-m-d_H-i-s') . '.csv';
            header('Content-Type: text/csv');
            header('Content-Disposition: attachment; filename="' . $filename . '"');

            $output = fopen('php://output', 'w');
            
            if (!empty($data)) {
                fputcsv($output, array_keys($data[0]));
                foreach ($data as $row) {
                    fputcsv($output, $row);
                }
            }
            
            fclose($output);

        } catch (Exception $e) {
            log_message('error', 'Export assessments error: ' . $e->getMessage());
            $this->session->set_flashdata('error', 'Failed to export assessments');
            redirect('admin/nursing/reports');
        }
    }

    public function generateDailyReport() {
        try {
            $date = date('Y-m-d');
            $data = [
                'report_title' => 'Daily Nursing Report - ' . date('F j, Y'),
                'date' => $date,
                'vital_signs' => $this->getDailyVitalSigns($date),
                'nursing_notes' => $this->getDailyNursingNotes($date),
                'procedures' => $this->getDailyProcedures($date),
                'medications' => $this->getDailyMedications($date),
                'stats' => $this->nursing_model->getDailyNursingStats($date)
            ];

            // Load PDF library if available, otherwise show in browser
            $this->load->view('admin/nursing/daily_report', $data);

        } catch (Exception $e) {
            log_message('error', 'Generate daily report error: ' . $e->getMessage());
            show_error('Failed to generate daily report');
        }
    }

    private function getDailyVitalSigns($date) {
        try {
            if (!$this->db->table_exists('nursing_vital_signs')) {
                return [];
            }

            $this->db->select('
                vs.*, p.patient_name, p.id,
                CONCAT(s.name, " ", s.surname) as staff_name
            ');
            $this->db->from('nursing_vital_signs vs');
            $this->db->join('patients p', 'p.id = vs.patient_id', 'left');
            $this->db->join('staff s', 's.id = vs.recorded_by', 'left');
            $this->db->where('DATE(vs.recorded_at)', $date);
            $this->db->order_by('vs.recorded_at', 'DESC');

            $query = $this->db->get();
            return $query->result_array();

        } catch (Exception $e) {
            return [];
        }
    }

    private function getDailyNursingNotes($date) {
        try {
            $this->db->select('
                nn.*, p.patient_name, p.id,
                CONCAT(s.name, " ", s.surname) as staff_name
            ');
            $this->db->from('nurse_note nn');
            $this->db->join('ipd_details ipd', 'ipd.id = nn.ipd_id', 'left');
            $this->db->join('patients p', 'p.id = ipd.patient_id', 'left');
            $this->db->join('staff s', 's.id = nn.staff_id', 'left');
            $this->db->where('DATE(nn.date)', $date);
            $this->db->order_by('nn.date', 'DESC');

            $query = $this->db->get();
            return $query->result_array();

        } catch (Exception $e) {
            return [];
        }
    }

    private function getDailyProcedures($date) {
        try {
            if (!$this->db->table_exists('nursing_procedures')) {
                return [];
            }

            $this->db->select('
                p.*, pat.patient_name, pat.id,
                CONCAT(s.name, " ", s.surname) as staff_name
            ');
            $this->db->from('nursing_procedures p');
            $this->db->join('patients pat', 'pat.id = p.patient_id', 'left');
            $this->db->join('staff s', 's.id = p.performed_by', 'left');
            $this->db->where('DATE(p.performed_at)', $date);
            $this->db->order_by('p.performed_at', 'DESC');

            $query = $this->db->get();
            return $query->result_array();

        } catch (Exception $e) {
            return [];
        }
    }

    private function getDailyMedications($date) {
        try {
            if (!$this->db->table_exists('medication_report')) {
                return [];
            }

            $this->db->select('
                mr.*, p.patient_name, p.id,
                CONCAT(s.name, " ", s.surname) as staff_name
            ');
            $this->db->from('medication_report mr');
            $this->db->join('ipd_details ipd', 'ipd.id = mr.ipd_id', 'left');
            $this->db->join('patients p', 'p.id = ipd.patient_id', 'left');
            $this->db->join('staff s', 's.id = mr.given_by', 'left');
            $this->db->where('DATE(mr.date)', $date);
            $this->db->order_by('mr.date', 'DESC');

            $query = $this->db->get();
            return $query->result_array();

        } catch (Exception $e) {
            return [];
        }
    }
    
    /**
 * Save vital signs - matches the form action in HTML
 */
public function save_vital_signs() {
    header('Content-Type: application/json');
    
    // Clear any previous output
    ob_clean();

    if (!$this->input->is_ajax_request()) {
        echo json_encode(['status' => 'error', 'message' => 'Direct access not allowed']);
        return;
    }

    $this->form_validation->set_rules('patient_id', 'Patient', 'required');
    $this->form_validation->set_rules('temperature', 'Temperature', 'numeric');
    $this->form_validation->set_rules('systolic_bp', 'Systolic BP', 'numeric');
    $this->form_validation->set_rules('diastolic_bp', 'Diastolic BP', 'numeric');
    $this->form_validation->set_rules('pulse_rate', 'Pulse Rate', 'numeric');
    $this->form_validation->set_rules('respiratory_rate', 'Respiratory Rate', 'numeric');

    if ($this->form_validation->run() == false) {
        $errors = validation_errors();
        echo json_encode([
            'status' => 'error', 
            'message' => 'Validation failed: ' . strip_tags($errors)
        ]);
        return;
    }

    try {
        if (!isset($this->nursing_model)) {
            $this->load->model('nursing_model');
        }

        $vital_data = [
            'patient_id' => $this->input->post('patient_id'),
            'ipd_id' => $this->input->post('ipd_id') ?: null,
            'temperature' => $this->input->post('temperature') ?: null,
            'temperature_unit' => 'C',
            'systolic_bp' => $this->input->post('systolic_bp') ?: null,
            'diastolic_bp' => $this->input->post('diastolic_bp') ?: null,
            'pulse_rate' => $this->input->post('pulse_rate') ?: null,
            'respiratory_rate' => $this->input->post('respiratory_rate') ?: null,
            'oxygen_saturation' => $this->input->post('oxygen_saturation') ?: null,
            'notes' => $this->input->post('notes') ?: null,
            'recorded_by' => $this->getStaffId(),
            'recorded_at' => date('Y-m-d H:i:s')
        ];

        $vital_id = $this->nursing_model->addVitalSigns($vital_data);

        echo json_encode([
            'status' => 'success',
            'message' => 'Vital signs recorded successfully',
            'vital_id' => $vital_id
        ]);

    } catch (Exception $e) {
        log_message('error', 'Save vital signs error: ' . $e->getMessage());
        echo json_encode([
            'status' => 'error',
            'message' => 'Failed to record vital signs: ' . $e->getMessage()
        ]);
    }
}

/**
 * Save nursing note - FIXED VERSION with detailed debugging
 */
public function save_nursing_note() {
    header('Content-Type: application/json');
    
    // Clear any previous output and enable error reporting for debugging
    ob_clean();
    error_reporting(E_ALL);
    ini_set('display_errors', 0); // Don't display but log them

    if (!$this->input->is_ajax_request()) {
        echo json_encode(['status' => 'error', 'message' => 'Direct access not allowed']);
        return;
    }

    // Log all incoming POST data for debugging
    log_message('debug', '=== NURSING NOTE SAVE DEBUG START ===');
    log_message('debug', 'POST Data: ' . print_r($_POST, true));
    log_message('debug', 'Input post data: ' . print_r($this->input->post(), true));

    $this->form_validation->set_rules('patient_id', 'Patient', 'required|integer');
    $this->form_validation->set_rules('note', 'Note', 'required');

    if ($this->form_validation->run() == false) {
        $errors = validation_errors();
        log_message('error', 'Validation failed: ' . $errors);
        echo json_encode([
            'status' => 'error', 
            'message' => 'Validation failed: ' . strip_tags($errors),
            'validation_errors' => $errors,
            'post_data' => $this->input->post()
        ]);
        return;
    }

    try {
        if (!isset($this->nursing_model)) {
            $this->load->model('nursing_model');
        }

        $patient_id = intval($this->input->post('patient_id'));
        $ipd_id = $this->input->post('ipd_id');
        $note_content = trim($this->input->post('note'));
        $comment = trim($this->input->post('comment')) ?: '';
        $note_type = $this->input->post('note_type') ?: 'general';

        log_message('debug', "Processing note for patient_id: {$patient_id}");
        log_message('debug', "IPD ID from form: {$ipd_id}");
        log_message('debug', "Note content: {$note_content}");

        // Validate patient exists
        if (!$this->patientExists($patient_id)) {
            throw new Exception('Patient not found with ID: ' . $patient_id);
        }

        // Get or validate IPD details
        if (empty($ipd_id)) {
            log_message('debug', 'No IPD ID provided, checking for active IPD record...');
            $ipd_details = $this->nursing_model->getPatientIPDDetails($patient_id);
            if ($ipd_details) {
                $ipd_id = $ipd_details['id'];
                log_message('debug', "Found active IPD record: {$ipd_id}");
            } else {
                log_message('debug', 'No active IPD record found');
                // Check if nurse_note table allows NULL ipd_id
                $fields = $this->db->list_fields('nurse_note');
                $field_data = $this->db->field_data('nurse_note');
                $ipd_nullable = false;
                foreach ($field_data as $field) {
                    if ($field->name == 'ipd_id' && $field->null == 1) {
                        $ipd_nullable = true;
                        break;
                    }
                }
                
                if (!$ipd_nullable) {
                    throw new Exception('Patient is not currently admitted (no active IPD record). Nursing notes require an IPD admission.');
                }
            }
        } else {
            // Validate provided IPD ID belongs to this patient
            if (!$this->validateIPDPatientMatch($ipd_id, $patient_id)) {
                throw new Exception('IPD record does not belong to the specified patient');
            }
        }

        $staff_id = $this->getStaffId();
        log_message('debug', "Staff ID: {$staff_id}");

        // Prepare note data - CRITICAL: Make sure patient_id is included
        $note_data = [
            'patient_id' => $patient_id,  // EXPLICITLY SET THIS
            'ipd_id' => $ipd_id,
            'note' => $note_content,
            'comment' => $comment,
            'note_type' => $note_type,
            'staff_id' => $staff_id,
            'date' => date('Y-m-d H:i:s'),
            'created_at' => date('Y-m-d H:i:s'),
            'created_by' => $staff_id
        ];

        log_message('debug', 'Final note data: ' . json_encode($note_data));

        // Check table structure before inserting
        $table_fields = $this->db->list_fields('nurse_note');
        log_message('debug', 'Table fields: ' . json_encode($table_fields));

        // Filter data to only include existing fields
        $filtered_data = [];
        foreach ($note_data as $key => $value) {
            if (in_array($key, $table_fields)) {
                $filtered_data[$key] = $value;
            } else {
                log_message('debug', "Skipping field '{$key}' - not in table");
            }
        }

        log_message('debug', 'Filtered data: ' . json_encode($filtered_data));

        $note_id = $this->nursing_model->addNurseNote($filtered_data);

        // Verify the insertion by reading it back
        $inserted_note = $this->getInsertedNote($note_id);
        log_message('debug', 'Inserted note verification: ' . json_encode($inserted_note));

        echo json_encode([
            'status' => 'success',
            'message' => 'Nursing note added successfully',
            'note_id' => $note_id,
            'debug_info' => [
                'patient_id' => $patient_id,
                'ipd_id' => $ipd_id,
                'staff_id' => $staff_id,
                'inserted_note' => $inserted_note
            ]
        ]);

        log_message('debug', '=== NURSING NOTE SAVE DEBUG END ===');

    } catch (Exception $e) {
        log_message('error', 'Save nursing note error: ' . $e->getMessage());
        log_message('error', 'Stack trace: ' . $e->getTraceAsString());
        echo json_encode([
            'status' => 'error',
            'message' => 'Failed to add nursing note: ' . $e->getMessage(),
            'debug_info' => [
                'patient_id' => $patient_id ?? 'not_set',
                'ipd_id' => $ipd_id ?? 'not_set',
                'staff_id' => $this->getStaffId(),
                'post_data' => $this->input->post()
            ]
        ]);
    }
}

public function fix_orphaned_notes() {
    try {
        $fields = $this->db->list_fields('nurse_note');
        
        if (!in_array('patient_id', $fields)) {
            echo "<p>❌ patient_id field doesn't exist. Run fix_nurse_note_table first.</p>";
            return;
        }
        
        // For notes that have NULL ipd_id, we need to determine which patient they belong to
        // This is tricky because we need to know which patient was being worked on
        
        // Show current orphaned notes
        $this->db->select('*');
        $this->db->from('nurse_note');
        $this->db->where('patient_id IS NULL');
        $this->db->where('ipd_id IS NULL');
        $query = $this->db->get();
        $orphaned = $query->result_array();
        
        echo "<h2>Orphaned Notes (no patient_id or ipd_id):</h2>";
        if (empty($orphaned)) {
            echo "<p>✅ No orphaned notes found</p>";
        } else {
            echo "<p>Found " . count($orphaned) . " orphaned notes:</p>";
            echo "<table border='1' style='border-collapse: collapse;'>";
            echo "<tr><th>ID</th><th>Date</th><th>Note</th><th>Staff ID</th></tr>";
            foreach ($orphaned as $note) {
                echo "<tr>";
                echo "<td>" . $note['id'] . "</td>";
                echo "<td>" . $note['date'] . "</td>";
                echo "<td>" . substr($note['note'], 0, 50) . "...</td>";
                echo "<td>" . $note['staff_id'] . "</td>";
                echo "</tr>";
            }
            echo "</table>";
            
            // Manual fix - you'll need to determine which patient these belong to
            // For now, let's assume the most recent ones belong to patient 56
            echo "<h3>Fixing Recent Notes for Patient 56:</h3>";
            
            $this->db->where('patient_id IS NULL');
            $this->db->where('ipd_id IS NULL');
            $this->db->where('date >=', date('Y-m-d', strtotime('-1 day'))); // Notes from last 24 hours
            $this->db->update('nurse_note', ['patient_id' => 56]);
            
            $affected = $this->db->affected_rows();
            echo "<p>✅ Updated {$affected} recent notes to belong to patient 56</p>";
        }
        
    } catch (Exception $e) {
        echo "<p>❌ Error: " . $e->getMessage() . "</p>";
    }
}

/**
 * Save assessment - matches the form action in HTML
 */
public function save_assessment() {
    header('Content-Type: application/json');
    
    // Clear any previous output
    ob_clean();

    if (!$this->input->is_ajax_request()) {
        echo json_encode(['status' => 'error', 'message' => 'Direct access not allowed']);
        return;
    }

    $this->form_validation->set_rules('patient_id', 'Patient', 'required');
    $this->form_validation->set_rules('assessment_type', 'Assessment Type', 'required');
    $this->form_validation->set_rules('assessment_category', 'Assessment Category', 'required');
    $this->form_validation->set_rules('risk_level', 'Risk Level', 'required');
    $this->form_validation->set_rules('assessment_details', 'Assessment Details', 'required');

    if ($this->form_validation->run() == false) {
        $errors = validation_errors();
        echo json_encode([
            'status' => 'error', 
            'message' => 'Validation failed: ' . strip_tags($errors)
        ]);
        return;
    }

    try {
        if (!isset($this->nursing_model)) {
            $this->load->model('nursing_model');
        }

        $assessment_data = [
            'patient_id' => $this->input->post('patient_id'),
            'ipd_id' => $this->input->post('ipd_id') ?: null,
            'assessment_type' => $this->input->post('assessment_type'),
            'assessment_category' => $this->input->post('assessment_category'),
            'risk_level' => $this->input->post('risk_level'),
            'assessment_score' => $this->input->post('assessment_score') ?: null,
            'assessment_details' => $this->input->post('assessment_details'),
            'recommendations' => $this->input->post('recommendations') ?: null,
            'assessed_by' => $this->getStaffId(),
            'assessment_date' => date('Y-m-d H:i:s')
        ];

        $assessment_id = $this->nursing_model->addAssessment($assessment_data);

        echo json_encode([
            'status' => 'success',
            'message' => 'Assessment added successfully',
            'assessment_id' => $assessment_id
        ]);

    } catch (Exception $e) {
        log_message('error', 'Save assessment error: ' . $e->getMessage());
        echo json_encode([
            'status' => 'error',
            'message' => 'Failed to add assessment: ' . $e->getMessage()
        ]);
    }
}

/**
 * Save procedure - matches the form action in HTML
 */
public function save_procedure() {
    header('Content-Type: application/json');
    
    // Clear any previous output
    ob_clean();

    if (!$this->input->is_ajax_request()) {
        echo json_encode(['status' => 'error', 'message' => 'Direct access not allowed']);
        return;
    }

    $this->form_validation->set_rules('patient_id', 'Patient', 'required');
    $this->form_validation->set_rules('procedure_name', 'Procedure Name', 'required');
    $this->form_validation->set_rules('procedure_type', 'Procedure Type', 'required');
    $this->form_validation->set_rules('status', 'Status', 'required');

    if ($this->form_validation->run() == false) {
        $errors = validation_errors();
        echo json_encode([
            'status' => 'error', 
            'message' => 'Validation failed: ' . strip_tags($errors)
        ]);
        return;
    }

    try {
        if (!isset($this->nursing_model)) {
            $this->load->model('nursing_model');
        }

        $procedure_data = [
            'patient_id' => $this->input->post('patient_id'),
            'ipd_id' => $this->input->post('ipd_id') ?: null,
            'procedure_name' => $this->input->post('procedure_name'),
            'procedure_type' => $this->input->post('procedure_type'),
            'status' => $this->input->post('status'),
            'duration_minutes' => $this->input->post('duration_minutes') ?: null,
            'procedure_description' => $this->input->post('procedure_description') ?: null,
            'procedure_notes' => $this->input->post('procedure_notes') ?: null,
            'performed_by' => $this->getStaffId(),
            'created_at' => date('Y-m-d H:i:s'),
            'performed_at' => ($this->input->post('status') == 'completed') ? date('Y-m-d H:i:s') : null
        ];

        $procedure_id = $this->nursing_model->addProcedure($procedure_data);

        echo json_encode([
            'status' => 'success',
            'message' => 'Procedure added successfully',
            'procedure_id' => $procedure_id
        ]);

    } catch (Exception $e) {
        log_message('error', 'Save procedure error: ' . $e->getMessage());
        echo json_encode([
            'status' => 'error',
            'message' => 'Failed to add procedure: ' . $e->getMessage()
        ]);
    }
}

/**
 * Save care plan - matches the form action in HTML
 */
public function save_care_plan() {
    header('Content-Type: application/json');
    
    // Clear any previous output
    ob_clean();

    if (!$this->input->is_ajax_request()) {
        echo json_encode(['status' => 'error', 'message' => 'Direct access not allowed']);
        return;
    }

    $this->form_validation->set_rules('patient_id', 'Patient', 'required');
    $this->form_validation->set_rules('plan_name', 'Plan Name', 'required');
    $this->form_validation->set_rules('priority', 'Priority', 'required');
    $this->form_validation->set_rules('start_date', 'Start Date', 'required');
    $this->form_validation->set_rules('nursing_diagnosis', 'Nursing Diagnosis', 'required');
    $this->form_validation->set_rules('goals_outcomes', 'Goals/Outcomes', 'required');
    $this->form_validation->set_rules('interventions', 'Interventions', 'required');

    if ($this->form_validation->run() == false) {
        $errors = validation_errors();
        echo json_encode([
            'status' => 'error', 
            'message' => 'Validation failed: ' . strip_tags($errors)
        ]);
        return;
    }

    try {
        if (!isset($this->nursing_model)) {
            $this->load->model('nursing_model');
        }

        $care_plan_data = [
            'patient_id' => $this->input->post('patient_id'),
            'ipd_id' => $this->input->post('ipd_id') ?: null,
            'plan_name' => $this->input->post('plan_name'),
            'priority' => $this->input->post('priority'),
            'status' => 'active',
            'start_date' => $this->input->post('start_date'),
            'target_date' => $this->input->post('target_date') ?: null,
            'nursing_diagnosis' => $this->input->post('nursing_diagnosis'),
            'goals_outcomes' => $this->input->post('goals_outcomes'),
            'interventions' => $this->input->post('interventions'),
            'evaluation_criteria' => $this->input->post('evaluation_criteria') ?: null,
            'created_by' => $this->getStaffId(),
            'created_at' => date('Y-m-d H:i:s')
        ];

        $care_plan_id = $this->nursing_model->addCarePlan($care_plan_data);

        echo json_encode([
            'status' => 'success',
            'message' => 'Care plan created successfully',
            'care_plan_id' => $care_plan_id
        ]);

    } catch (Exception $e) {
        log_message('error', 'Save care plan error: ' . $e->getMessage());
        echo json_encode([
            'status' => 'error',
            'message' => 'Failed to create care plan: ' . $e->getMessage()
        ]);
    }
}
/**
 * Helper method to verify inserted note
 */
private function getInsertedNote($note_id) {
    try {
        $this->db->select('*');
        $this->db->from('nurse_note');
        $this->db->where('id', $note_id);
        $query = $this->db->get();
        
        if ($query->num_rows() > 0) {
            return $query->row_array();
        }
        return null;
    } catch (Exception $e) {
        log_message('error', 'Get inserted note error: ' . $e->getMessage());
        return null;
    }
}

/**
 * Helper method to check if patient exists
 */
private function patientExists($patient_id) {
    try {
        $this->db->select('id');
        $this->db->from('patients');
        $this->db->where('id', $patient_id);
        $query = $this->db->get();
        return $query->num_rows() > 0;
    } catch (Exception $e) {
        log_message('error', 'Patient exists check error: ' . $e->getMessage());
        return false;
    }
}

/**
 * Helper method to validate IPD and patient match
 */
private function validateIPDPatientMatch($ipd_id, $patient_id) {
    try {
        $this->db->select('patient_id');
        $this->db->from('ipd_details');
        $this->db->where('id', $ipd_id);
        $query = $this->db->get();
        
        if ($query->num_rows() > 0) {
            $row = $query->row();
            return $row->patient_id == $patient_id;
        }
        return false;
    } catch (Exception $e) {
        log_message('error', 'IPD patient match validation error: ' . $e->getMessage());
        return false;
    }
}

public function debug_nursing_note_issue($patient_id = 56) {
    echo "<h1>Nursing Note Debug Analysis</h1>";
    echo "<p><strong>Patient ID:</strong> {$patient_id}</p>";
    
    try {
        // 1. Check table structure
        echo "<h2>1. Table Structure Analysis</h2>";
        if (!$this->db->table_exists('nurse_note')) {
            echo "<p>❌ nurse_note table does not exist</p>";
            return;
        }
        
        $fields = $this->db->list_fields('nurse_note');
        $field_data = $this->db->field_data('nurse_note');
        
        echo "<h3>Table Fields:</h3>";
        echo "<table border='1' style='border-collapse: collapse; width: 100%;'>";
        echo "<tr><th>Field</th><th>Type</th><th>Null Allowed</th><th>Default</th></tr>";
        foreach ($field_data as $field) {
            echo "<tr>";
            echo "<td>" . $field->name . "</td>";
            echo "<td>" . $field->type . "</td>";
            echo "<td>" . ($field->null ? 'YES' : 'NO') . "</td>";
            echo "<td>" . ($field->default ?? 'NULL') . "</td>";
            echo "</tr>";
        }
        echo "</table>";
        
        // 2. Check recent insertions
        echo "<h2>2. Recent Nursing Notes Analysis</h2>";
        $this->db->select('*');
        $this->db->from('nurse_note');
        $this->db->order_by('id', 'DESC');
        $this->db->limit(5);
        $query = $this->db->get();
        $recent_notes = $query->result_array();
        
        echo "<h3>Last 5 Notes:</h3>";
        if (empty($recent_notes)) {
            echo "<p>❌ No notes found in database</p>";
        } else {
            echo "<table border='1' style='border-collapse: collapse; width: 100%;'>";
            echo "<tr><th>ID</th><th>Patient ID</th><th>IPD ID</th><th>Staff ID</th><th>Note</th><th>Date</th></tr>";
            foreach ($recent_notes as $note) {
                $patient_id_display = ($note['patient_id'] === null) ? '❌ NULL' : '✅ ' . $note['patient_id'];
                $ipd_id_display = ($note['ipd_id'] === null) ? '❌ NULL' : '✅ ' . $note['ipd_id'];
                
                echo "<tr>";
                echo "<td>" . $note['id'] . "</td>";
                echo "<td>" . $patient_id_display . "</td>";
                echo "<td>" . $ipd_id_display . "</td>";
                echo "<td>" . $note['staff_id'] . "</td>";
                echo "<td>" . substr($note['note'], 0, 30) . "...</td>";
                echo "<td>" . $note['date'] . "</td>";
                echo "</tr>";
            }
            echo "</table>";
        }
        
        // 3. Test patient existence
        echo "<h2>3. Patient Validation</h2>";
        $this->db->select('id, patient_name, mobileno');
        $this->db->from('patients');
        $this->db->where('id', $patient_id);
        $query = $this->db->get();
        
        if ($query->num_rows() > 0) {
            $patient = $query->row_array();
            echo "<p>✅ Patient exists: " . $patient['patient_name'] . " (ID: " . $patient['id'] . ")</p>";
        } else {
            echo "<p>❌ Patient {$patient_id} does not exist</p>";
            return;
        }
        
        // 4. Check IPD status
        echo "<h2>4. IPD Status Check</h2>";
        $this->db->select('*');
        $this->db->from('ipd_details');
        $this->db->where('patient_id', $patient_id);
        $this->db->order_by('date', 'DESC');
        $query = $this->db->get();
        
        if ($query->num_rows() > 0) {
            $ipd_records = $query->result_array();
            echo "<p>✅ Found " . count($ipd_records) . " IPD record(s)</p>";
            
            echo "<table border='1' style='border-collapse: collapse; width: 100%;'>";
            echo "<tr><th>IPD ID</th><th>Admission Date</th><th>Discharged</th><th>Status</th></tr>";
            
            $active_ipd = null;
            foreach ($ipd_records as $ipd) {
                $status = ($ipd['discharged'] == 'no') ? '✅ Active' : '❌ Discharged';
                if ($ipd['discharged'] == 'no') {
                    $active_ipd = $ipd;
                }
                
                echo "<tr>";
                echo "<td>" . $ipd['id'] . "</td>";
                echo "<td>" . $ipd['date'] . "</td>";
                echo "<td>" . $ipd['discharged'] . "</td>";
                echo "<td>" . $status . "</td>";
                echo "</tr>";
            }
            echo "</table>";
            
            if ($active_ipd) {
                echo "<p>✅ Active IPD found: ID " . $active_ipd['id'] . "</p>";
            } else {
                echo "<p>⚠️ No active IPD record (all discharged)</p>";
            }
        } else {
            echo "<p>❌ No IPD records found for patient {$patient_id}</p>";
        }
        
        // 5. Test note insertion
        echo "<h2>5. Test Note Insertion</h2>";
        try {
            $test_data = [
                'patient_id' => $patient_id,
                'ipd_id' => $active_ipd['id'] ?? null,
                'note' => 'DEBUG TEST NOTE - ' . date('Y-m-d H:i:s'),
                'comment' => 'This is a test insertion',
                'staff_id' => $this->getStaffId(),
                'date' => date('Y-m-d H:i:s')
            ];
            
            echo "<h3>Test Data:</h3>";
            echo "<pre>" . print_r($test_data, true) . "</pre>";
            
            $result = $this->nursing_model->addNurseNote($test_data);
            echo "<p>✅ Test insertion successful with ID: " . $result . "</p>";
            
            // Verify the test insertion
            $this->db->select('*');
            $this->db->from('nurse_note');
            $this->db->where('id', $result);
            $query = $this->db->get();
            
            if ($query->num_rows() > 0) {
                $inserted = $query->row_array();
                echo "<h3>Verification - Inserted Record:</h3>";
                echo "<table border='1' style='border-collapse: collapse;'>";
                foreach ($inserted as $field => $value) {
                    $display_value = ($value === null) ? '❌ NULL' : '✅ ' . $value;
                    echo "<tr><td><strong>{$field}</strong></td><td>{$display_value}</td></tr>";
                }
                echo "</table>";
                
                // Check if patient_id was saved correctly
                if ($inserted['patient_id'] == $patient_id) {
                    echo "<p>✅ SUCCESS: patient_id correctly saved</p>";
                } else {
                    echo "<p>❌ FAILURE: patient_id not saved correctly</p>";
                    echo "<p>Expected: {$patient_id}, Got: " . ($inserted['patient_id'] ?? 'NULL') . "</p>";
                }
            }
            
        } catch (Exception $e) {
            echo "<p>❌ Test insertion failed: " . $e->getMessage() . "</p>";
        }
        
        // 6. Check form handling
        echo "<h2>6. Form Data Simulation</h2>";
        $_POST = [
            'patient_id' => $patient_id,
            'ipd_id' => $active_ipd['id'] ?? '',
            'note' => 'Simulated form submission',
            'comment' => 'Test comment'
        ];
        
        echo "<h3>Simulated POST data:</h3>";
        echo "<pre>" . print_r($_POST, true) . "</pre>";
        
        echo "<h3>CodeIgniter Input Processing:</h3>";
        echo "<p>patient_id: " . $this->input->post('patient_id') . "</p>";
        echo "<p>ipd_id: " . $this->input->post('ipd_id') . "</p>";
        echo "<p>note: " . $this->input->post('note') . "</p>";
        echo "<p>comment: " . $this->input->post('comment') . "</p>";
        
        // 7. Summary and recommendations
        echo "<h2>7. Summary & Recommendations</h2>";
        
        $issues = [];
        if (empty($recent_notes)) {
            $issues[] = "No nursing notes in database";
        }
        
        $null_patient_count = 0;
        foreach ($recent_notes as $note) {
            if ($note['patient_id'] === null) {
                $null_patient_count++;
            }
        }
        
        if ($null_patient_count > 0) {
            $issues[] = "{$null_patient_count} notes have NULL patient_id";
        }
        
        if (!$active_ipd && in_array('ipd_id', $fields)) {
            $field_info = null;
            foreach ($field_data as $field) {
                if ($field->name == 'ipd_id') {
                    $field_info = $field;
                    break;
                }
            }
            if ($field_info && !$field_info->null) {
                $issues[] = "ipd_id field is required but patient has no active IPD";
            }
        }
        
        if (empty($issues)) {
            echo "<p>✅ No major issues detected</p>";
        } else {
            echo "<h3>Issues Found:</h3>";
            echo "<ul>";
            foreach ($issues as $issue) {
                echo "<li>❌ {$issue}</li>";
            }
            echo "</ul>";
        }
        
        echo "<h3>Recommendations:</h3>";
        echo "<ol>";
        echo "<li>Ensure the HTML form includes proper patient_id hidden field</li>";
        echo "<li>Verify JavaScript is passing correct data to AJAX request</li>";
        echo "<li>Check that patient_id field in nurse_note table allows the intended data type</li>";
        echo "<li>Ensure the nursing_model->addNurseNote method properly handles patient_id</li>";
        echo "<li>Test with browser developer tools to verify form submission data</li>";
        echo "</ol>";
        
    } catch (Exception $e) {
        echo "<p>❌ Debug failed: " . $e->getMessage() . "</p>";
    }
}

    // =============================================
    // QUEUE MANAGEMENT METHODS
    // =============================================

    /**
     * Get nursing department queue data
     */
    public function get_queue_data() {
        header('Content-Type: application/json');
        
        try {
            $result = $this->Queue_model->getDepartmentQueue('nursing');
            echo json_encode($result);
        } catch (Exception $e) {
            echo json_encode(['status' => 'error', 'message' => $e->getMessage()]);
        }
    }

    /**
     * Call next patient from queue
     */
    public function call_next_patient() {
        header('Content-Type: application/json');
        
        try {
            $nurse_id = $this->input->post('nurse_id') ?: $this->getStaffId();
            $result = $this->Queue_model->callNextPatient('nursing', $nurse_id);
            
            if ($result['status'] === 'success') {
                // Play notification sound
                $this->send_queue_notification('nursing', 'patient_called', $result['patient']);
            }
            
            echo json_encode($result);
        } catch (Exception $e) {
            echo json_encode(['status' => 'error', 'message' => $e->getMessage()]);
        }
    }

    /**
     * Skip patient in queue
     */
    public function skip_patient() {
        header('Content-Type: application/json');
        
        try {
            $queue_id = $this->input->post('queue_id');
            $reason = $this->input->post('reason') ?: 'No reason provided';
            
            $result = $this->Queue_model->skipPatient($queue_id, $reason);
            echo json_encode($result);
        } catch (Exception $e) {
            echo json_encode(['status' => 'error', 'message' => $e->getMessage()]);
        }
    }

    /**
     * Complete nursing service and route to next department
     */
    public function complete_service() {
        header('Content-Type: application/json');
        
        try {
            $queue_id = $this->input->post('queue_id');
            $next_department = $this->input->post('next_department');
            $notes = $this->input->post('notes') ?: 'Nursing care completed';
            
            $result = $this->Queue_model->completeService($queue_id, $next_department, $notes);
            
            if ($result['status'] === 'success' && $next_department) {
                $this->send_queue_notification($next_department, 'nursing_care_completed', [
                    'from_department' => 'nursing',
                    'patient_id' => $this->input->post('patient_id'),
                    'notes' => $notes
                ]);
            }
            
            echo json_encode($result);
        } catch (Exception $e) {
            echo json_encode(['status' => 'error', 'message' => $e->getMessage()]);
        }
    }

    /**
     * Get queue statistics
     */
    public function get_queue_stats() {
        header('Content-Type: application/json');
        
        try {
            $stats = $this->Queue_model->getQueueStatistics('nursing');
            echo json_encode(['status' => 'success', 'stats' => $stats]);
        } catch (Exception $e) {
            echo json_encode(['status' => 'error', 'message' => $e->getMessage()]);
        }
    }

    /**
     * Queue management panel
     */
    public function queue_management() {
        $this->session->set_userdata('top_menu', 'Nursing');
        $this->session->set_userdata('sub_menu', 'nursing/queue');
        
        $data['title'] = 'Nursing Queue Management';
        $data['department'] = 'nursing';
        $data['queue_data'] = $this->Queue_model->getDepartmentQueue('nursing');
        $data['queue_config'] = $this->Queue_model->getQueueConfig('nursing');
        
        $this->load->view('layout/header', $data);
        $this->load->view('admin/nursing/queue_management', $data);
        $this->load->view('layout/footer', $data);
    }

    /**
     * Send queue notification to other departments
     */
    private function send_queue_notification($department, $type, $data = []) {
        $notification = [
            'department' => $department,
            'type' => $type,
            'data' => $data,
            'timestamp' => time()
        ];
        
        $cache_file = APPPATH . 'cache/queue_notifications_' . $department . '.json';
        file_put_contents($cache_file, json_encode($notification));
    }

    /**
     * Get current staff ID
     */
    private function getStaffId() {
        return $this->session->userdata('admin_id') ?: 1;
    }

    /**
     * Debug queue status
     */
    public function debug_queue_status() {
        echo "<h2>Nursing Queue Debug</h2>";
        echo "<p>Controller is working properly!</p>";
        echo "<p>Queue integration: " . (class_exists('Queue_model') ? 'ACTIVE' : 'MISSING') . "</p>";
        echo "<p>Notification system: " . (is_writable(APPPATH . 'cache/') ? 'READY' : 'NOT WRITABLE') . "</p>";
        
        try {
            $stats = $this->Queue_model->getQueueStatistics('nursing');
            echo "<p>Queue statistics: Working</p>";
            echo "<pre>" . json_encode($stats, JSON_PRETTY_PRINT) . "</pre>";
        } catch (Exception $e) {
            echo "<p>Queue statistics: Error - " . $e->getMessage() . "</p>";
        }
    }

}
?>