<?php defined('BASEPATH') OR exit('No direct script access allowed');

class Mpesa extends CI_Controller {
    
    public function __construct() {
        parent::__construct();
        $this->load->library('mpesa_lib');
        $this->load->model('mpesa_model');
    }
    
    /**
     * Initiate STK Push
     */
    public function initiate_payment() {
        $phone_number = $this->input->post('phone_number');
        $amount = $this->input->post('amount');
        $patient_id = $this->input->post('patient_id');
        $workflow_type = $this->input->post('workflow_type');
        
        // Validate inputs
        if (empty($phone_number) || empty($amount)) {
            echo json_encode(array('status' => 'error', 'message' => 'Phone number and amount are required'));
            return;
        }
        
        // Create payment record
        $payment_data = array(
            'patient_id' => $patient_id,
            'phone_number' => $phone_number,
            'amount' => $amount,
            'workflow_type' => $workflow_type,
            'status' => 'pending',
            'created_at' => date('Y-m-d H:i:s')
        );
        
        $payment_id = $this->mpesa_model->create_payment($payment_data);
        
        // Set account reference with payment ID
        $account_reference = 'PATIENT_' . $patient_id . '_' . $payment_id;
        
        // Initiate STK Push
        $response = $this->mpesa_lib->stkPush($phone_number, $amount, $account_reference);
        
        if (isset($response['ResponseCode']) && $response['ResponseCode'] == '0') {
            // Update payment record with M-Pesa details
            $update_data = array(
                'merchant_request_id' => $response['MerchantRequestID'],
                'checkout_request_id' => $response['CheckoutRequestID'],
                'response_code' => $response['ResponseCode'],
                'response_description' => $response['ResponseDescription'],
                'customer_message' => $response['CustomerMessage']
            );
            
            $this->mpesa_model->update_payment($payment_id, $update_data);
            
            echo json_encode(array(
                'status' => 'success',
                'message' => $response['CustomerMessage'],
                'payment_id' => $payment_id,
                'checkout_request_id' => $response['CheckoutRequestID']
            ));
        } else {
            echo json_encode(array(
                'status' => 'error',
                'message' => 'STK Push failed',
                'error' => $response
            ));
        }
    }
    
    /**
     * Check Payment Status
     */
    public function check_payment_status() {
        $payment_id = $this->input->post('payment_id');
        $checkout_request_id = $this->input->post('checkout_request_id');
        
        if (empty($payment_id) || empty($checkout_request_id)) {
            echo json_encode(array('status' => 'error', 'message' => 'Payment ID and Checkout Request ID are required'));
            return;
        }
        
        // Query M-Pesa for payment status
        $response = $this->mpesa_lib->stkQuery($checkout_request_id);
        
        if (isset($response['ResponseCode']) && $response['ResponseCode'] == '0') {
            $status = 'failed'; // Default to failed
            
            if (isset($response['ResultCode'])) {
                if ($response['ResultCode'] == '0') {
                    $status = 'completed';
                } elseif ($response['ResultCode'] == '1032') {
                    $status = 'cancelled';
                } else {
                    $status = 'failed';
                }
            }
            
            // Update payment record
            $update_data = array(
                'result_code' => $response['ResultCode'] ?? null,
                'result_desc' => $response['ResultDesc'] ?? null,
                'status' => $status,
                'updated_at' => date('Y-m-d H:i:s')
            );
            
            $this->mpesa_model->update_payment($payment_id, $update_data);
            
            echo json_encode(array(
                'status' => 'success',
                'payment_status' => $status,
                'message' => $response['ResultDesc'] ?? 'Payment status retrieved'
            ));
        } else {
            echo json_encode(array(
                'status' => 'error',
                'message' => 'Failed to check payment status',
                'error' => $response
            ));
        }
    }
    
    /**
     * M-Pesa Callback
     */
    public function callback() {
        $response = json_decode(file_get_contents('php://input'), true);
        
        // Log the callback
        log_message('info', 'M-Pesa Callback: ' . json_encode($response));
        
        if (isset($response['Body']['stkCallback'])) {
            $callback_data = $response['Body']['stkCallback'];
            
            // Find payment record by CheckoutRequestID
            $payment = $this->mpesa_model->get_payment_by_checkout_request_id($callback_data['CheckoutRequestID']);
            
            if ($payment) {
                $status = 'failed'; // Default to failed
                $mpesa_receipt_number = null;
                $transaction_date = null;
                
                if ($callback_data['ResultCode'] == 0) {
                    $status = 'completed';
                    
                    // Extract callback metadata
                    if (isset($callback_data['CallbackMetadata']['Item'])) {
                        foreach ($callback_data['CallbackMetadata']['Item'] as $item) {
                            if ($item['Name'] == 'MpesaReceiptNumber') {
                                $mpesa_receipt_number = $item['Value'];
                            }
                            if ($item['Name'] == 'TransactionDate') {
                                $transaction_date = $item['Value'];
                            }
                        }
                    }
                }
                
                // Update payment record
                $update_data = array(
                    'result_code' => $callback_data['ResultCode'],
                    'result_desc' => $callback_data['ResultDesc'],
                    'mpesa_receipt_number' => $mpesa_receipt_number,
                    'transaction_date' => $transaction_date,
                    'status' => $status,
                    'callback_data' => json_encode($callback_data),
                    'updated_at' => date('Y-m-d H:i:s')
                );
                
                $this->mpesa_model->update_payment($payment->id, $update_data);
            }
        }
        
        // Return success response to M-Pesa
        echo json_encode(array('ResultCode' => 0, 'ResultDesc' => 'Success'));
    }
}