<?php

if (!defined('BASEPATH'))
    exit('No direct script access allowed');

class Lab_reports extends Admin_Controller {

    function __construct() {
        parent::__construct();
        
        // Load required models and libraries
        $this->load->model('lab_reports_model');
        $this->load->model('patient_model');
        $this->load->model('staff_model');
        $this->load->library('datatables');
        $this->load->library('customlib');
        $this->load->helper('file');
        $this->load->helper('download');
    }

    /**
     * Main lab reports page
     */
    public function index() {
        // Set session data for navigation
        $this->session->set_userdata('top_menu', 'Lab Reports');
        $this->session->set_userdata('sub_menu', 'lab_reports');

        // Prepare data for view
        $data['title'] = 'Lab Reports Management';
        
        // Get current user info
        $data['current_user'] = $this->getStaffInfo();
        
        // Check if models are loaded properly
        if (!$this->load->is_loaded('lab_reports_model')) {
            $this->load->model('lab_reports_model');
        }
        
        // Load view with error handling
        try {
            $this->load->view('layout/header', $data);
            $this->load->view('admin/lab_reports/index', $data);
            $this->load->view('layout/footer', $data);
        } catch (Exception $e) {
            log_message('error', 'Lab Reports view loading error: ' . $e->getMessage());
            show_error('Failed to load lab reports page: ' . $e->getMessage());
        }
    }

    /**
     * FIXED: Get all reports for DataTables - Simplified version
     */
    public function get_all_reports() {
        header('Content-Type: application/json');
        
        try {
            // Get POST data from DataTables
            $request = $this->input->post();
            
            // Simplified direct query approach
            $result = $this->getReportsDirectly($request);
            
            // Format data for DataTables
            $formatted_data = [];
            
            if (!empty($result['data'])) {
                foreach ($result['data'] as $row) {
                    $status_class = $this->getStatusClass($row['status']);
                    $critical_badge = ($row['is_critical'] == 1) ? ' <span class="label label-danger"><i class="fa fa-exclamation-triangle"></i> Critical</span>' : '';
                    
                    // Generate actions
                    $actions = '
                        <div class="btn-group">
                            <button class="btn btn-xs btn-primary" onclick="viewReport(' . $row['id'] . ')" title="View Report">
                                <i class="fa fa-eye"></i>
                            </button>
                            <button class="btn btn-xs btn-success" onclick="downloadReport(' . $row['id'] . ')" title="Download">
                                <i class="fa fa-download"></i>
                            </button>';
                    $actions .= '<button class="btn btn-info btn-xs" onclick="generateProfessionalReport(' . $row['id'] . ')" title="Professional Report">
                                <i class="fa fa-file-text"></i>
                            </button>';        
                    
                    if ($row['status'] !== 'verified') {
                        $actions .= '
                            <button class="btn btn-xs btn-warning" onclick="verifyReport(' . $row['id'] . ')" title="Verify">
                                <i class="fa fa-check"></i>
                            </button>';
                    }
                    
                    $actions .= '</div>';
                    
                    $formatted_data[] = [
                        'id' => $row['id'],
                        'patient_name' => $row['patient_name'] ?: 'Unknown Patient',
                        'report_title' => $row['report_title'] . $critical_badge,
                        'report_type' => ucfirst(str_replace('_', ' ', $row['report_type'])),
                        'department' => ucfirst($row['department']),
                        'report_date' => date('Y-m-d', strtotime($row['report_date'])),
                        'status' => '<span class="label ' . $status_class . '">' . ucfirst($row['status']) . '</span>',
                        'actions' => $actions
                    ];
                }
            }
            
            $response = [
                'draw' => intval($request['draw'] ?? 1),
                'recordsTotal' => $result['recordsTotal'] ?? 0,
                'recordsFiltered' => $result['recordsFiltered'] ?? 0,
                'data' => $formatted_data
            ];
            
            echo json_encode($response);
            
        } catch (Exception $e) {
            log_message('error', 'DataTables error: ' . $e->getMessage());
            echo json_encode([
                'draw' => 1,
                'recordsTotal' => 0,
                'recordsFiltered' => 0,
                'data' => [],
                'error' => $e->getMessage()
            ]);
        }
    }
    
/**
 * FIXED: Generate professional lab report method
 */
public function generateProfessionalReport($report_id = null) {
    try {
        $report_id = $report_id ?: $this->input->get('report_id');
        
        if (!$report_id) {
            show_error('Report ID is required');
            return;
        }
        
        // Get report details with patient and staff info
        $this->db->select('plr.*, 
                          p.patient_name, p.mobileno, p.age, p.gender, p.address, p.patient_unique_id,
                          CONCAT(COALESCE(s1.name, ""), " ", COALESCE(s1.surname, "")) as doctor_name,
                          CONCAT(COALESCE(s2.name, ""), " ", COALESCE(s2.surname, "")) as technician_name,
                          CONCAT(COALESCE(s3.name, ""), " ", COALESCE(s3.surname, "")) as verified_by_name');
        $this->db->from('patient_lab_reports plr');
        $this->db->join('patients p', 'plr.patient_id = p.id', 'left');
        $this->db->join('staff s1', 'plr.doctor_id = s1.id', 'left');
        $this->db->join('staff s2', 'plr.lab_technician = s2.id', 'left');
        $this->db->join('staff s3', 'plr.verified_by = s3.id', 'left');
        $this->db->where('plr.id', $report_id);
        
        $report_data = $this->db->get()->row();
        
        if (!$report_data) {
            show_error('Lab report not found');
            return;
        }
        
        // CRITICAL FIX: Properly prepare data for the view
        $data = [
            'report' => $report_data,
            'patient' => $report_data, // Patient data is included in the report object
            // FIXED: Pass the raw JSON strings - the view will handle parsing
            'test_results' => $report_data->test_results,
            'test_interpretations' => $report_data->test_interpretations,
            'reference_ranges' => $report_data->reference_ranges
        ];
        
        // Load the professional report view
        $this->load->view('admin/lab_reports/professional_report', $data);
        
    } catch (Exception $e) {
        log_message('error', 'Error generating professional report: ' . $e->getMessage());
        show_error('Error generating report: ' . $e->getMessage());
    }
}

/**
 * DEBUGGING: Add this method to check if test results are being saved correctly
 */
public function debug_report($report_id) {
    if (!$report_id) {
        show_error('Report ID required');
        return;
    }
    
    echo "<h1>Debug Report #" . $report_id . "</h1>";
    
    $this->db->select('*');
    $this->db->from('patient_lab_reports');
    $this->db->where('id', $report_id);
    $report = $this->db->get()->row();
    
    if (!$report) {
        echo "<p>Report not found!</p>";
        return;
    }
    
    echo "<h2>Raw Data from Database:</h2>";
    echo "<pre>";
    print_r($report);
    echo "</pre>";
    
    echo "<h2>Test Results Field:</h2>";
    echo "<pre>";
    echo "Raw: " . $report->test_results . "\n\n";
    if (!empty($report->test_results)) {
        $decoded = json_decode($report->test_results, true);
        echo "Decoded: ";
        print_r($decoded);
    } else {
        echo "Test results field is empty!";
    }
    echo "</pre>";
    
    echo "<h2>Test Interpretations Field:</h2>";
    echo "<pre>";
    echo "Raw: " . $report->test_interpretations . "\n\n";
    if (!empty($report->test_interpretations)) {
        $decoded = json_decode($report->test_interpretations, true);
        echo "Decoded: ";
        print_r($decoded);
    } else {
        echo "Test interpretations field is empty!";
    }
    echo "</pre>";
}

/**
 * VERIFY: Check if the database has the required fields
 */
public function check_database() {
    echo "<h1>Database Structure Check</h1>";
    
    // Check if table exists
    if (!$this->db->table_exists('patient_lab_reports')) {
        echo "<p style='color: red;'>❌ Table 'patient_lab_reports' does not exist!</p>";
        return;
    }
    
    echo "<p style='color: green;'>✅ Table 'patient_lab_reports' exists</p>";
    
    // Check fields
    $fields = $this->db->list_fields('patient_lab_reports');
    
    $required_fields = ['test_results', 'test_interpretations', 'reference_ranges'];
    
    echo "<h2>Required Fields Check:</h2>";
    foreach ($required_fields as $field) {
        if (in_array($field, $fields)) {
            echo "<p style='color: green;'>✅ Field '$field' exists</p>";
        } else {
            echo "<p style='color: red;'>❌ Field '$field' missing!</p>";
        }
    }
    
    echo "<h2>All Fields in Table:</h2>";
    echo "<ul>";
    foreach ($fields as $field) {
        echo "<li>$field</li>";
    }
    echo "</ul>";
    
    // Sample data check
    echo "<h2>Sample Report Data:</h2>";
    $this->db->select('id, report_title, test_results, test_interpretations');
    $this->db->from('patient_lab_reports');
    $this->db->limit(5);
    $reports = $this->db->get()->result();
    
    if (empty($reports)) {
        echo "<p>No reports found in database</p>";
    } else {
        foreach ($reports as $report) {
            echo "<div style='border: 1px solid #ccc; padding: 10px; margin: 10px 0;'>";
            echo "<strong>ID:</strong> " . $report->id . "<br>";
            echo "<strong>Title:</strong> " . $report->report_title . "<br>";
            echo "<strong>Has Test Results:</strong> " . (!empty($report->test_results) ? "Yes" : "No") . "<br>";
            echo "<strong>Has Interpretations:</strong> " . (!empty($report->test_interpretations) ? "Yes" : "No") . "<br>";
            echo "</div>";
        }
    }
}

/**
 * ADD MISSING DATABASE FIELDS (if needed)
 * Run this once if the fields are missing from your database
 */
public function add_missing_fields() {
    echo "<h1>Adding Missing Database Fields</h1>";
    
    try {
        // Add test_results field
        $this->db->query("ALTER TABLE patient_lab_reports ADD COLUMN test_results LONGTEXT COMMENT 'JSON field to store structured test results'");
        echo "<p style='color: green;'>✅ Added test_results field</p>";
    } catch (Exception $e) {
        echo "<p style='color: orange;'>⚠️ test_results field may already exist: " . $e->getMessage() . "</p>";
    }
    
    try {
        // Add reference_ranges field
        $this->db->query("ALTER TABLE patient_lab_reports ADD COLUMN reference_ranges LONGTEXT COMMENT 'JSON field to store reference ranges for tests'");
        echo "<p style='color: green;'>✅ Added reference_ranges field</p>";
    } catch (Exception $e) {
        echo "<p style='color: orange;'>⚠️ reference_ranges field may already exist: " . $e->getMessage() . "</p>";
    }
    
    try {
        // Add test_interpretations field
        $this->db->query("ALTER TABLE patient_lab_reports ADD COLUMN test_interpretations LONGTEXT COMMENT 'JSON field to store high/low/normal interpretations'");
        echo "<p style='color: green;'>✅ Added test_interpretations field</p>";
    } catch (Exception $e) {
        echo "<p style='color: orange;'>⚠️ test_interpretations field may already exist: " . $e->getMessage() . "</p>";
    }
    
    echo "<h2>Database Update Complete!</h2>";
    echo "<p><a href='" . base_url('admin/lab_reports/check_database') . "'>Check Database Structure</a></p>";
}

    /**
     * Simplified method to get reports directly
     */
    private function getReportsDirectly($request) {
        try {
            // Simple direct query
            $this->db->select('pr.id, pr.patient_id, pr.report_title, pr.report_type, 
                              pr.department, pr.report_date, pr.status, pr.is_critical,
                              pr.created_at, p.patient_name, p.mobileno');
            $this->db->from('patient_lab_reports pr');
            $this->db->join('patients p', 'pr.patient_id = p.id', 'left');
            
            // Apply filters if provided
            $status = $this->input->post('status');
            $department = $this->input->post('department');
            $date_from = $this->input->post('date_from');
            $date_to = $this->input->post('date_to');
            
            if (!empty($status)) {
                $this->db->where('pr.status', $status);
            }
            
            if (!empty($department)) {
                $this->db->where('pr.department', $department);
            }
            
            if (!empty($date_from)) {
                $this->db->where('pr.report_date >=', $date_from);
            }
            
            if (!empty($date_to)) {
                $this->db->where('pr.report_date <=', $date_to);
            }
            
            // Count total records before pagination
            $total_query = clone $this->db;
            $recordsTotal = $total_query->count_all_results();
            
            // Apply search if provided
            if (!empty($request['search']['value'])) {
                $search = $request['search']['value'];
                $this->db->group_start();
                $this->db->like('pr.report_title', $search);
                $this->db->or_like('p.patient_name', $search);
                $this->db->or_like('pr.report_type', $search);
                $this->db->group_end();
            }
            
            // Apply ordering
            $this->db->order_by('pr.id', 'DESC');
            
            // Apply pagination
            if (isset($request['length']) && $request['length'] != -1) {
                $this->db->limit($request['length'], $request['start'] ?? 0);
            } else {
                $this->db->limit(25, 0); // Default limit
            }
            
            $query = $this->db->get();
            $data = $query ? $query->result_array() : [];
            
            return [
                'draw' => intval($request['draw'] ?? 1),
                'recordsTotal' => $recordsTotal,
                'recordsFiltered' => $recordsTotal,
                'data' => $data
            ];
            
        } catch (Exception $e) {
            log_message('error', 'Direct query error: ' . $e->getMessage());
            return [
                'draw' => 1,
                'recordsTotal' => 0,
                'recordsFiltered' => 0,
                'data' => []
            ];
        }
    }

    /**
     * FIXED: Get dashboard statistics
     */
    public function get_dashboard_stats() {
        header('Content-Type: application/json');
        
        try {
            $stats = [];

            // Total reports
            $this->db->from('patient_lab_reports');
            $stats['total'] = $this->db->count_all_results();

            // Pending reports
            $this->db->from('patient_lab_reports');
            $this->db->where('status', 'pending');
            $stats['pending'] = $this->db->count_all_results();

            // Verified today
            $this->db->from('patient_lab_reports');
            $this->db->where('status', 'verified');
            $this->db->where('DATE(updated_at)', date('Y-m-d'));
            $stats['verified_today'] = $this->db->count_all_results();

            // Critical reports
            $this->db->from('patient_lab_reports');
            $this->db->where('is_critical', 1);
            $this->db->where('status !=', 'cancelled');
            $stats['critical'] = $this->db->count_all_results();

            echo json_encode(['success' => true, 'data' => $stats]);
            
        } catch (Exception $e) {
            log_message('error', 'Dashboard stats error: ' . $e->getMessage());
            echo json_encode([
                'success' => false, 
                'message' => 'Error fetching stats: ' . $e->getMessage(),
                'data' => [
                    'total' => 0,
                    'pending' => 0,
                    'verified_today' => 0,
                    'critical' => 0
                ]
            ]);
        }
    }

    /**
     * FIXED: Search patients for dropdown
     */
    public function search_patients() {
        header('Content-Type: application/json');
        
        $search_term = $this->input->get('q');
        $page = $this->input->get('page', 1);
        $per_page = 10;
        
        try {
            $this->db->select('id, patient_name, mobileno, age, gender');
            $this->db->from('patients');
            
            // Check if is_active column exists
            if ($this->db->field_exists('is_active', 'patients')) {
                $this->db->where('is_active', 'yes');
            }
            
            if (!empty($search_term)) {
                $this->db->group_start();
                $this->db->like('patient_name', $search_term);
                $this->db->or_like('mobileno', $search_term);
                $this->db->or_like('id', $search_term);
                $this->db->group_end();
            }
            
            $this->db->order_by('patient_name', 'ASC');
            $this->db->limit($per_page);
            
            $query = $this->db->get();
            $patients = $query ? $query->result_array() : [];
            
            $items = [];
            foreach ($patients as $patient) {
                $items[] = [
                    'id' => $patient['id'],
                    'text' => $patient['patient_name'] . ' (' . $patient['id'] . ') - ' . ($patient['mobileno'] ?: 'No phone'),
                    'patient_name' => $patient['patient_name'],
                    'mobileno' => $patient['mobileno'],
                    'age' => $patient['age'],
                    'gender' => $patient['gender']
                ];
            }
            
            echo json_encode([
                'items' => $items,
                'total_count' => count($items)
            ]);
            
        } catch (Exception $e) {
            log_message('error', 'Patient search error: ' . $e->getMessage());
            echo json_encode([
                'items' => [],
                'total_count' => 0,
                'error' => $e->getMessage()
            ]);
        }
    }

    /**
     * Upload report with enhanced error handling
     */
    public function upload_report() {
        header('Content-Type: application/json');
        
        try {
            // Get basic form data
            $patient_id = $this->input->post('patient_id');
            $report_type = $this->input->post('report_type');
            $report_title = $this->input->post('report_title');
            $report_description = $this->input->post('report_description');
            $test_names = $this->input->post('test_names');
            $report_date = $this->input->post('report_date');
            $department = $this->input->post('department');
            $is_critical = $this->input->post('is_critical') ? 1 : 0;
            $notes = $this->input->post('notes');

            // Get structured test results
            $test_results_json = $this->input->post('test_results');
            $test_interpretations_json = $this->input->post('test_interpretations');
            $reference_ranges_json = $this->input->post('reference_ranges');

            // Validate required fields
            if (empty($patient_id)) {
                echo json_encode(['success' => false, 'message' => 'Patient ID is required']);
                return;
            }
            
            if (empty($report_type)) {
                echo json_encode(['success' => false, 'message' => 'Report type is required']);
                return;
            }
            
            if (empty($report_title)) {
                echo json_encode(['success' => false, 'message' => 'Report title is required']);
                return;
            }
            
            if (empty($report_date)) {
                echo json_encode(['success' => false, 'message' => 'Report date is required']);
                return;
            }

            // Handle file upload
            $upload_result = $this->handleFileUpload();
            if (!$upload_result['success']) {
                echo json_encode($upload_result);
                return;
            }

            // Prepare data for insertion
            $report_data = [
                'patient_id' => $patient_id,
                'report_type' => $report_type,
                'report_title' => $report_title,
                'report_description' => $report_description,
                'file_name' => $upload_result['file_name'],
                'file_path' => $upload_result['file_path'],
                'file_size' => $upload_result['file_size'],
                'file_type' => $upload_result['file_type'],
                'original_filename' => $upload_result['original_filename'],
                'report_date' => $report_date,
                'test_names' => $test_names,
                'department' => $department ?: 'laboratory',
                'status' => 'pending',
                'is_critical' => $is_critical,
                'notes' => $notes,
                'uploaded_by' => $this->getStaffId(),
                'created_at' => date('Y-m-d H:i:s'),
                'updated_at' => date('Y-m-d H:i:s')
            ];

            // Add structured test results if present
            if (!empty($test_results_json)) {
                $report_data['test_results'] = $test_results_json;
            }
            
            if (!empty($test_interpretations_json)) {
                $report_data['test_interpretations'] = $test_interpretations_json;
            }
            
            if (!empty($reference_ranges_json)) {
                $report_data['reference_ranges'] = $reference_ranges_json;
            }

            // Insert report using direct query
            $this->db->insert('patient_lab_reports', $report_data);
            $report_id = $this->db->insert_id();

            if ($report_id) {
                echo json_encode([
                    'success' => true, 
                    'message' => 'Lab report uploaded successfully',
                    'report_id' => $report_id,
                    'has_structured_results' => !empty($test_results_json)
                ]);
            } else {
                // Clean up uploaded file if database insert failed
                if (file_exists($upload_result['file_path'])) {
                    unlink($upload_result['file_path']);
                }
                echo json_encode(['success' => false, 'message' => 'Failed to save report to database']);
            }

        } catch (Exception $e) {
            log_message('error', 'Lab report upload error: ' . $e->getMessage());
            echo json_encode(['success' => false, 'message' => 'Error: ' . $e->getMessage()]);
        }
    }

    /**
     * Handle file upload with comprehensive validation
     */
    private function handleFileUpload() {
        if (!isset($_FILES['lab_report_file'])) {
            return ['success' => false, 'message' => 'No file selected for upload'];
        }

        $file = $_FILES['lab_report_file'];

        if ($file['error'] != UPLOAD_ERR_OK) {
            return ['success' => false, 'message' => 'File upload error: ' . $file['error']];
        }

        $allowed_extensions = ['pdf', 'jpg', 'jpeg', 'png', 'doc', 'docx'];
        $max_size = 10 * 1024 * 1024; // 10MB

        if ($file['size'] > $max_size) {
            return ['success' => false, 'message' => 'File size exceeds 10MB limit'];
        }

        $file_ext = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
        if (!in_array($file_ext, $allowed_extensions)) {
            return ['success' => false, 'message' => 'Invalid file type. Allowed: PDF, JPG, PNG, DOC, DOCX'];
        }

        $upload_dir = './uploads/lab_reports/' . date('Y/m/');
        
        if (!is_dir($upload_dir)) {
            if (!mkdir($upload_dir, 0755, true)) {
                return ['success' => false, 'message' => 'Failed to create upload directory'];
            }
        }

        $file_name = 'lab_report_' . date('YmdHis') . '_' . uniqid() . '.' . $file_ext;
        $file_path = $upload_dir . $file_name;

        if (move_uploaded_file($file['tmp_name'], $file_path)) {
            chmod($file_path, 0644);
            
            return [
                'success' => true,
                'file_name' => $file_name,
                'file_path' => $file_path,
                'file_size' => $file['size'],
                'file_type' => $file['type'],
                'original_filename' => $file['name']
            ];
        } else {
            return ['success' => false, 'message' => 'Failed to move uploaded file'];
        }
    }

    /**
     * Get patient reports
     */
    public function get_patient_reports() {
        header('Content-Type: application/json');
        
        $patient_id = $this->input->get('patient_id');
        
        if (!$patient_id) {
            echo json_encode(['success' => false, 'message' => 'Patient ID required']);
            return;
        }

        try {
            $this->db->select('*');
            $this->db->from('patient_lab_reports');
            $this->db->where('patient_id', $patient_id);
            $this->db->order_by('report_date', 'DESC');
            $this->db->order_by('created_at', 'DESC');
            
            $query = $this->db->get();
            $reports = $query ? $query->result_array() : [];
            
            echo json_encode(['success' => true, 'data' => $reports]);
        } catch (Exception $e) {
            echo json_encode(['success' => false, 'message' => 'Error fetching reports: ' . $e->getMessage()]);
        }
    }

    /**
     * Get single report
     */
    public function get_report($report_id = null) {
        header('Content-Type: application/json');
        
        if (!$report_id) {
            $report_id = $this->input->get('report_id') ?: $this->input->post('report_id');
        }
        
        if (!$report_id) {
            echo json_encode(['success' => false, 'message' => 'Report ID required']);
            return;
        }

        try {
            $this->db->select('pr.*, p.patient_name, p.mobileno, p.age, p.gender');
            $this->db->from('patient_lab_reports pr');
            $this->db->join('patients p', 'pr.patient_id = p.id', 'left');
            $this->db->where('pr.id', $report_id);
            
            $query = $this->db->get();
            $report = $query ? $query->row_array() : null;
            
            if ($report) {
                // Process file paths
                if (!empty($report['file_path'])) {
                    $file_path = str_replace('./', '', $report['file_path']);
                    $report['file_url'] = base_url($file_path);
                    $report['file_exists'] = file_exists($report['file_path']);
                }
                
                echo json_encode(['success' => true, 'data' => $report]);
            } else {
                echo json_encode(['success' => false, 'message' => 'Report not found']);
            }
            
        } catch (Exception $e) {
            echo json_encode(['success' => false, 'message' => 'Error: ' . $e->getMessage()]);
        }
    }

    /**
     * Verify report
     */
    public function verify_report() {
        header('Content-Type: application/json');
        
        $report_id = $this->input->post('report_id');
        $notes = $this->input->post('notes');
        
        if (!$report_id) {
            echo json_encode(['success' => false, 'message' => 'Report ID required']);
            return;
        }

        try {
            $update_data = [
                'status' => 'verified',
                'verified_by' => $this->getStaffId(),
                'verified_at' => date('Y-m-d H:i:s'),
                'updated_at' => date('Y-m-d H:i:s')
            ];

            if ($notes) {
                $update_data['verification_notes'] = $notes;
            }

            $this->db->where('id', $report_id);
            $result = $this->db->update('patient_lab_reports', $update_data);
            
            if ($result) {
                echo json_encode(['success' => true, 'message' => 'Report verified successfully']);
            } else {
                echo json_encode(['success' => false, 'message' => 'Failed to verify report']);
            }
        } catch (Exception $e) {
            echo json_encode(['success' => false, 'message' => 'Error verifying report: ' . $e->getMessage()]);
        }
    }

    /**
     * Download report file
     */
    public function download_report($report_id) {
        if (!$report_id) {
            show_404();
            return;
        }

        try {
            $this->db->select('file_path, original_filename');
            $this->db->from('patient_lab_reports');
            $this->db->where('id', $report_id);
            $query = $this->db->get();
            $report = $query->row_array();
            
            if (!$report) {
                show_404();
                return;
            }

            $file_path = $report['file_path'];
            
            if (!file_exists($file_path)) {
                show_error('File not found on server', 404);
                return;
            }

            $this->load->helper('download');
            force_download($report['original_filename'], file_get_contents($file_path));
            
        } catch (Exception $e) {
            show_error('Download failed: ' . $e->getMessage());
        }
    }

    /**
     * Debug endpoint
     */
    public function test() {
        echo "<h1>Lab Reports Controller Test</h1>";
        echo "<p>Controller is working!</p>";
        echo "<p>Base URL: " . base_url() . "</p>";
        echo "<p>Current time: " . date('Y-m-d H:i:s') . "</p>";
        
        try {
            // Test database connection
            $this->db->select('COUNT(*) as patient_count');
            $this->db->from('patients');
            $query = $this->db->get();
            $result = $query->row();
            echo "<p>Patients in database: " . $result->patient_count . "</p>";
            
            // Test lab reports table
            if ($this->db->table_exists('patient_lab_reports')) {
                $this->db->select('COUNT(*) as report_count');
                $this->db->from('patient_lab_reports');
                $query = $this->db->get();
                $result = $query->row();
                echo "<p>Lab reports in database: " . $result->report_count . "</p>";
            } else {
                echo "<p>❌ patient_lab_reports table does not exist</p>";
            }
            
            // Test upload directory
            $upload_dir = './uploads/lab_reports/';
            if (is_dir($upload_dir)) {
                echo "<p>✅ Upload directory exists and is " . (is_writable($upload_dir) ? 'writable' : 'not writable') . "</p>";
            } else {
                echo "<p>❌ Upload directory does not exist</p>";
            }
            
        } catch (Exception $e) {
            echo "<p style='color: red;'>Error: " . $e->getMessage() . "</p>";
        }
    }

    // Helper methods
    
    private function getStaffId() {
        try {
            $staff_id = $this->session->userdata('admin_id');
            return $staff_id ?: 1;
        } catch (Exception $e) {
            return 1;
        }
    }

    private function getStaffInfo() {
        try {
            $staff_id = $this->getStaffId();
            $this->db->select('*');
            $this->db->from('staff');
            $this->db->where('id', $staff_id);
            $query = $this->db->get();
            return $query->row_array();
        } catch (Exception $e) {
            return null;
        }
    }

    private function getStatusClass($status) {
        $classes = [
            'pending' => 'label-warning',
            'completed' => 'label-primary',
            'verified' => 'label-success',
            'cancelled' => 'label-danger'
        ];
        return isset($classes[$status]) ? $classes[$status] : 'label-default';
    }

    /**
     * Export reports to CSV
     */
    public function export_reports() {
        try {
            $filters = [
                'status' => $this->input->get('status'),
                'department' => $this->input->get('department'),
                'date_from' => $this->input->get('date_from'),
                'date_to' => $this->input->get('date_to')
            ];
            
            $this->db->select('pr.*, p.patient_name');
            $this->db->from('patient_lab_reports pr');
            $this->db->join('patients p', 'pr.patient_id = p.id', 'left');
            
            // Apply filters
            if (!empty($filters['status'])) {
                $this->db->where('pr.status', $filters['status']);
            }
            if (!empty($filters['department'])) {
                $this->db->where('pr.department', $filters['department']);
            }
            if (!empty($filters['date_from'])) {
                $this->db->where('pr.report_date >=', $filters['date_from']);
            }
            if (!empty($filters['date_to'])) {
                $this->db->where('pr.report_date <=', $filters['date_to']);
            }
            
            $this->db->order_by('pr.report_date', 'DESC');
            $this->db->limit(1000); // Limit export to 1000 records
            
            $query = $this->db->get();
            $reports = $query->result_array();
            
            // Set headers for CSV download
            $filename = 'lab_reports_' . date('Y-m-d_H-i-s') . '.csv';
            header('Content-Type: text/csv');
            header('Content-Disposition: attachment; filename="' . $filename . '"');
            header('Pragma: no-cache');
            header('Expires: 0');
            
            $output = fopen('php://output', 'w');
            
            // CSV Headers
            fputcsv($output, [
                'ID', 'Patient Name', 'Report Title', 'Type', 'Department', 
                'Report Date', 'Status', 'Critical', 'Created At'
            ]);
            
            // CSV Data
            foreach ($reports as $report) {
                fputcsv($output, [
                    $report['id'],
                    $report['patient_name'],
                    $report['report_title'],
                    ucfirst(str_replace('_', ' ', $report['report_type'])),
                    ucfirst($report['department']),
                    $report['report_date'],
                    ucfirst($report['status']),
                    $report['is_critical'] ? 'Yes' : 'No',
                    $report['created_at']
                ]);
            }
            
            fclose($output);
            
        } catch (Exception $e) {
            show_error('Export failed: ' . $e->getMessage());
        }
    }

    /**
     * Get reference ranges for tests
     */
    public function get_reference_ranges() {
        header('Content-Type: application/json');
        
        try {
            // Return default ranges
            $ranges = [
                'hiv' => ['type' => 'select', 'normal' => 'negative'],
                'rbs' => ['type' => 'numeric', 'min' => 70, 'max' => 140, 'unit' => 'mg/dL'],
                'haemoglobin' => ['type' => 'numeric', 'min_male' => 13.5, 'max_male' => 17.5, 'min_female' => 12, 'max_female' => 15.5, 'unit' => 'g/dL'],
                'vdrl' => ['type' => 'select', 'normal' => 'negative'],
                'creatinine' => ['type' => 'numeric', 'min_male' => 0.7, 'max_male' => 1.3, 'min_female' => 0.6, 'max_female' => 1.1, 'unit' => 'mg/dL'],
                'hbsag' => ['type' => 'select', 'normal' => 'negative'],
                'hcv_antibody' => ['type' => 'select', 'normal' => 'negative'],
                'pregnancy_test' => ['type' => 'select', 'normal' => 'negative'],
                'urine_ph' => ['type' => 'numeric', 'min' => 4.5, 'max' => 8.0],
                'specific_gravity' => ['type' => 'numeric', 'min' => 1.003, 'max' => 1.030],
                'nitrite' => ['type' => 'select', 'normal' => 'negative'],
                'gpt_alt' => ['type' => 'numeric', 'min' => 7, 'max' => 35, 'unit' => 'U/L'],
                'got_ast' => ['type' => 'numeric', 'min' => 8, 'max' => 40, 'unit' => 'U/L'],
                'total_bilirubin' => ['type' => 'numeric', 'min' => 0.2, 'max' => 1.2, 'unit' => 'mg/dL']
            ];
            
            echo json_encode(['success' => true, 'data' => $ranges]);
            
        } catch (Exception $e) {
            echo json_encode(['success' => false, 'message' => 'Error: ' . $e->getMessage()]);
        }
    }
}