<?php

if (!defined('BASEPATH')) {
    exit('No direct script access allowed');
}

class Lab extends Admin_Controller {

    public function __construct() {
        parent::__construct();
        $this->load->model('lab_model');
        $this->load->model('patient_model');
        $this->load->model('staff_model');
        $this->load->library('form_validation');
        $this->load->library('datatables');
        $this->config->load('payroll');
        $this->config->load('mailsms');
        $this->load->helper('customfield_helper');
        $this->load->helper('custom');
        $this->patient_data = $this->session->userdata('patient');
    }

    public function index() {
        if (!$this->rbac->hasPrivilege('lab_tests', 'can_view')) {
            access_denied();
        }
        $this->session->set_userdata('top_menu', 'lab');
        $this->session->set_userdata('sub_menu', 'lab/index');
        $data['title'] = 'Lab Tests';
        $data['lab_tests'] = $this->lab_model->getLabTests();
        $data['categories'] = $this->lab_model->getLabTestCategories();
        $this->load->view('layout/header', $data);
        $this->load->view('admin/lab/index', $data);
        $this->load->view('layout/footer', $data);
    }

    public function add_test() {
        if (!$this->rbac->hasPrivilege('lab_tests', 'can_add')) {
            access_denied();
        }
        $this->form_validation->set_rules('name', 'Test Name', 'trim|required|xss_clean');
        $this->form_validation->set_rules('short_name', 'Short Name', 'trim|required|xss_clean');
        $this->form_validation->set_rules('category_id', 'Category', 'trim|required|xss_clean');
        $this->form_validation->set_rules('charge', 'Charge', 'trim|required|numeric|xss_clean');

        if ($this->form_validation->run() == FALSE) {
            $msg = array(
                'name' => form_error('name'),
                'short_name' => form_error('short_name'),
                'category_id' => form_error('category_id'),
                'charge' => form_error('charge'),
            );
            $array = array('status' => 'fail', 'error' => $msg, 'message' => '');
        } else {
            $data = array(
                'name' => $this->input->post('name'),
                'short_name' => $this->input->post('short_name'),
                'category_id' => $this->input->post('category_id'),
                'charge' => $this->input->post('charge'),
                'description' => $this->input->post('description'),
            );
            $this->lab_model->addLabTest($data);
            $array = array('status' => 'success', 'error' => '', 'message' => 'Lab Test added successfully.');
        }
        echo json_encode($array);
    }

    public function edit_test($id) {
        if (!$this->rbac->hasPrivilege('lab_tests', 'can_edit')) {
            access_denied();
        }
        $data['test'] = $this->lab_model->getLabTests($id)[0];
        $data['categories'] = $this->lab_model->getLabTestCategories();
        $this->load->view('admin/lab/edit_test_modal', $data); // Create this view for modal
    }

    public function update_test($id) {
        if (!$this->rbac->hasPrivilege('lab_tests', 'can_edit')) {
            access_denied();
        }
        $this->form_validation->set_rules('name', 'Test Name', 'trim|required|xss_clean');
        $this->form_validation->set_rules('short_name', 'Short Name', 'trim|required|xss_clean');
        $this->form_validation->set_rules('category_id', 'Category', 'trim|required|xss_clean');
        $this->form_validation->set_rules('charge', 'Charge', 'trim|required|numeric|xss_clean');

        if ($this->form_validation->run() == FALSE) {
            $msg = array(
                'name' => form_error('name'),
                'short_name' => form_error('short_name'),
                'category_id' => form_error('category_id'),
                'charge' => form_error('charge'),
            );
            $array = array('status' => 'fail', 'error' => $msg, 'message' => '');
        } else {
            $data = array(
                'name' => $this->input->post('name'),
                'short_name' => $this->input->post('short_name'),
                'category_id' => $this->input->post('category_id'),
                'charge' => $this->input->post('charge'),
                'description' => $this->input->post('description'),
            );
            $this->lab_model->updateLabTest($id, $data);
            $array = array('status' => 'success', 'error' => '', 'message' => 'Lab Test updated successfully.');
        }
        echo json_encode($array);
    }

    public function delete_test($id) {
        if (!$this->rbac->hasPrivilege('lab_tests', 'can_delete')) {
            access_denied();
        }
        $this->lab_model->deleteLabTest($id);
        echo json_encode(array('status' => 'success', 'message' => 'Lab Test deleted successfully.'));
    }

    public function results() {
        if (!$this->rbac->hasPrivilege('lab_results', 'can_view')) {
            access_denied();
        }
        $this->session->set_userdata('top_menu', 'lab');
        $this->session->set_userdata('sub_menu', 'lab/results');
        $data['title'] = 'Lab Results';
        $data['patients'] = $this->patient_model->getPatientList();
        $data['lab_tests'] = $this->lab_model->getLabTests();
        $data['technicians'] = $this->lab_model->getLabTechnicians(); 
        $this->load->view('layout/header', $data);
        $this->load->view('admin/lab/results', $data);
        $this->load->view('layout/footer', $data);
    }

    public function get_results_datatable(){
        $dt_response = $this->lab_model->getAllLabResults(); // Ensure Lab_model has getAllLabResults method for datatables
        $data = array();
        if (!empty($dt_response)) {
            foreach ($dt_response as $key => $value) {
                $row = array();
                $row[] = $value->id; // Assuming 'id' is the direct property
                $row[] = $value->patient_name; // Adjust if property names differ
                $row[] = $value->test_name;
                $row[] = $value->result_value;
                $row[] = $value->reference_range;
                $row[] = $value->technician_name;
                $row[] = date($this->customlib->getSchoolDateFormat(), strtotime($value->result_date));
                // Add actions button
                $action = '<a href="#" onclick="view_result('.$value->id.')" class="btn btn-default btn-xs" data-toggle="tooltip" title="View"><i class="fa fa-eye"></i></a>';
                $action .= ' <a href="#" onclick="edit_result('.$value->id.')" class="btn btn-default btn-xs" data-toggle="tooltip" title="Edit"><i class="fa fa-pencil"></i></a>';
                $action .= ' <a href="#" onclick="delete_result('.$value->id.')" class="btn btn-default btn-xs" data-toggle="tooltip" title="Delete"><i class="fa fa-trash"></i></a>';
                $row[] = $action;
                $data[] = $row;
            }
        }
        $output = array(
            "draw" => isset($_POST['draw']) ? intval($_POST['draw']) : 0,
            "recordsTotal" => $this->lab_model->countAllLabResults(), // Method to count total records
            "recordsFiltered" => $this->lab_model->countFilteredLabResults(), // Method to count filtered records for server-side processing
            "data" => $data,
        );
        echo json_encode($output);
    }

    public function add_result() {
        if (!$this->rbac->hasPrivilege('lab_results', 'can_add')) {
            access_denied();
        }
        $this->form_validation->set_rules('patient_id', 'Patient', 'trim|required|xss_clean');
        $this->form_validation->set_rules('test_id', 'Test', 'trim|required|xss_clean');
        $this->form_validation->set_rules('result_value', 'Result Value', 'trim|required|xss_clean');
        $this->form_validation->set_rules('technician_id', 'Technician', 'trim|required|xss_clean');
        $this->form_validation->set_rules('result_date', 'Result Date', 'trim|required|xss_clean');

        if ($this->form_validation->run() == FALSE) {
            $msg = array(
                'patient_id' => form_error('patient_id'),
                'test_id' => form_error('test_id'),
                'result_value' => form_error('result_value'),
                'technician_id' => form_error('technician_id'),
                'result_date' => form_error('result_date'),
            );
            $array = array('status' => 'fail', 'error' => $msg, 'message' => '');
        } else {
            $data = array(
                'patient_id' => $this->input->post('patient_id'),
                'test_id' => $this->input->post('test_id'),
                'result_value' => $this->input->post('result_value'),
                'reference_range' => $this->input->post('reference_range'),
                'technician_id' => $this->input->post('technician_id'),
                'result_date' => date('Y-m-d', strtotime($this->input->post('result_date'))),
                'comments' => $this->input->post('comments'),
            );
            $this->lab_model->addLabResult($data);
            $array = array('status' => 'success', 'error' => '', 'message' => 'Lab Result added successfully.');
        }
        echo json_encode($array);
    }

    public function get_lab_result($id){
        if (!$this->rbac->hasPrivilege('lab_results', 'can_view')) {
            access_denied();
        }
        $result = $this->lab_model->getLabResults($id); // Assuming getLabResults can fetch a single result by ID
        echo json_encode($result[0]); // Assuming it returns an array with one element
    }

    public function update_lab_result($id){
        if (!$this->rbac->hasPrivilege('lab_results', 'can_edit')) {
            access_denied();
        }
         $this->form_validation->set_rules('patient_id', 'Patient', 'trim|required|xss_clean');
        $this->form_validation->set_rules('test_id', 'Test', 'trim|required|xss_clean');
        $this->form_validation->set_rules('result_value', 'Result Value', 'trim|required|xss_clean');
        $this->form_validation->set_rules('technician_id', 'Technician', 'trim|required|xss_clean');
        $this->form_validation->set_rules('result_date', 'Result Date', 'trim|required|xss_clean');

        if ($this->form_validation->run() == FALSE) {
            $msg = array(
                'patient_id' => form_error('patient_id'),
                'test_id' => form_error('test_id'),
                'result_value' => form_error('result_value'),
                'technician_id' => form_error('technician_id'),
                'result_date' => form_error('result_date'),
            );
            $array = array('status' => 'fail', 'error' => $msg, 'message' => '');
        } else {
            $data = array(
                'patient_id' => $this->input->post('patient_id'),
                'test_id' => $this->input->post('test_id'),
                'result_value' => $this->input->post('result_value'),
                'reference_range' => $this->input->post('reference_range'),
                'technician_id' => $this->input->post('technician_id'),
                'result_date' => date('Y-m-d', strtotime($this->input->post('result_date'))),
                'comments' => $this->input->post('comments'),
            );
            $this->lab_model->updateLabResult($id, $data);
            $array = array('status' => 'success', 'error' => '', 'message' => 'Lab Result updated successfully.');
        }
        echo json_encode($array);
    }

    public function delete_lab_result($id){
        if (!$this->rbac->hasPrivilege('lab_results', 'can_delete')) {
            access_denied();
        }
        $this->lab_model->deleteLabResult($id);
        echo json_encode(array('status' => 'success', 'message' => 'Lab Result deleted successfully.'));
    }

}
