<?php
defined('BASEPATH') OR exit('No direct script access allowed');

/**
 * Import Controller
 * Handles bulk import of consumables, pharmacy stock, and lab supplies
 * 
 * @author Your Name
 * @version 1.0
 */
class Import extends Admin_Controller {

    protected $allowed_file_types = array('xls', 'xlsx', 'csv');
    protected $max_file_size = 5120; // 5MB in KB
    protected $upload_path = './uploads/imports/';

    public function __construct() {
        parent::__construct();
        
        // Load required libraries and models
        $this->load->library(['upload', 'form_validation']);
        $this->load->helper(['url', 'file', 'form']);
        $this->load->model('Import_model');
        
        // Load additional models if they exist
        if (file_exists(APPPATH . 'models/Consumables_model.php')) {
            $this->load->model('Consumables_model');
        }
        if (file_exists(APPPATH . 'models/Pharmacy_model.php')) {
            $this->load->model('Pharmacy_model');
        }
        if (file_exists(APPPATH . 'models/Lab_model.php')) {
            $this->load->model('Lab_model');
        }
        
        // Create upload directory if it doesn't exist
        if (!is_dir($this->upload_path)) {
            mkdir($this->upload_path, 0755, true);
        }
    }

    /**
     * Main import interface
     */
    public function index() {
        // Set menu navigation
        $this->session->set_userdata('top_menu', 'Import');
        $this->session->set_userdata('sub_menu', 'import/index');

        $data['title'] = 'Import Consumables & Stock';
        $data['import_modules'] = $this->get_import_modules();
        $data['recent_imports'] = $this->Import_model->get_recent_imports(10);
        
        // Load views
        $this->load->view('layout/header', $data);
        $this->load->view('admin/import/index', $data);
        $this->load->view('layout/footer', $data);
    }
    
    /**
     * Test method for debugging
     */
    public function test() {
        echo "Import controller loaded successfully!<br>";
        echo "Session staff_id: " . $this->session->userdata('staff_id') . "<br>";
        echo "Session user_id: " . $this->session->userdata('user_id') . "<br>";
        echo "Session admin: " . $this->session->userdata('admin') . "<br>";
        
        // Test if we can access the parent
        echo "Admin_Controller parent: " . (class_exists('Admin_Controller') ? 'EXISTS' : 'NOT FOUND');
    }

    /**
     * Handle file upload and preview
     */
public function upload_file() {
    header('Content-Type: application/json');
    
    try {
        // Configure upload settings
        $config = array(
            'upload_path' => $this->upload_path,
            'allowed_types' => implode('|', $this->allowed_file_types),
            'max_size' => $this->max_file_size,
            'encrypt_name' => true,
            'remove_spaces' => true
        );
        
        $this->upload->initialize($config);
        
        if (!$this->upload->do_upload('import_file')) {
            throw new Exception($this->upload->display_errors('', ''));
        }
        
        $upload_data = $this->upload->data();
        $file_path = $upload_data['full_path'];
        $import_type = $this->input->post('import_type');
        
        // Validate import type
        if (!in_array($import_type, array_keys($this->get_import_modules()))) {
            throw new Exception('Invalid import type selected');
        }
        
        // Parse file and get preview data
        $preview_data = $this->parse_file_preview($file_path, $import_type);
        
        // Get total record count from the file
        $total_records = $this->get_total_record_count($file_path);
        
        // Store import session data
        $import_session = array(
            'file_path' => $file_path,
            'original_filename' => $upload_data['orig_name'],
            'import_type' => $import_type,
            'preview_data' => $preview_data,
            'total_records' => $total_records,
            'upload_time' => time()
        );
        
        $this->session->set_userdata('import_session', $import_session);
        
        echo json_encode(array(
            'status' => 'success',
            'message' => 'File uploaded successfully',
            'preview_data' => $preview_data,
            'total_records' => $total_records, // Send actual total count
            'preview_count' => count($preview_data['data']), // Send preview count
            'columns' => $preview_data['columns']
        ));
        
    } catch (Exception $e) {
        // Clean up uploaded file if exists
        if (isset($upload_data['full_path']) && file_exists($upload_data['full_path'])) {
            unlink($upload_data['full_path']);
        }
        
        echo json_encode(array(
            'status' => 'error',
            'message' => $e->getMessage()
        ));
    }
}

/**
 * Get total record count from file (excluding headers)
 */
protected function get_total_record_count($file_path) {
    $file_extension = pathinfo($file_path, PATHINFO_EXTENSION);
    
    if (in_array($file_extension, ['xls', 'xlsx'])) {
        return $this->get_excel_total_count($file_path);
    } else {
        return $this->get_csv_total_count($file_path);
    }
}

/**
 * Get total count from CSV file
 */
protected function get_csv_total_count($file_path) {
    $count = 0;
    
    if (($handle = fopen($file_path, "r")) !== FALSE) {
        // Skip header row
        fgetcsv($handle, 1000, ",");
        
        // Count remaining rows
        while (($row = fgetcsv($handle, 1000, ",")) !== FALSE) {
            // Only count rows that have actual data
            if (array_filter($row, function($value) { return !empty(trim($value)); })) {
                $count++;
            }
        }
        fclose($handle);
    }
    
    return $count;
}

/**
 * Get total count from Excel file
 */
protected function get_excel_total_count($file_path) {
    // If you have PhpSpreadsheet available, use this version:
    if (file_exists(APPPATH . 'third_party/PhpSpreadsheet/vendor/autoload.php')) {
        require_once APPPATH . 'third_party/PhpSpreadsheet/vendor/autoload.php';
        
        try {
            $reader = \PhpOffice\PhpSpreadsheet\IOFactory::createReader('Xlsx');
            $reader->setReadDataOnly(true);
            $spreadsheet = $reader->load($file_path);
            $worksheet = $spreadsheet->getActiveSheet();
            
            // Find where data starts (skip company headers)
            $data_start_row = $this->find_data_start_row($worksheet);
            
            if ($data_start_row === false) {
                return 0;
            }
            
            // Count rows with actual data
            $count = 0;
            $highest_row = $worksheet->getHighestRow();
            
            for ($row = $data_start_row + 1; $row <= $highest_row; $row++) {
                $has_data = false;
                
                // Check first few columns to see if row has data
                for ($col = 1; $col <= 5; $col++) {
                    $cell_value = trim($worksheet->getCellByColumnAndRow($col, $row)->getCalculatedValue());
                    if (!empty($cell_value)) {
                        $has_data = true;
                        break;
                    }
                }
                
                if ($has_data) {
                    $count++;
                }
            }
            
            return $count;
            
        } catch (Exception $e) {
            // Fallback to line counting method
            return $this->get_excel_line_count($file_path);
        }
    } else {
        // Fallback method without PhpSpreadsheet
        return $this->get_excel_line_count($file_path);
    }
}

/**
 * Fallback method to count Excel rows (approximate)
 */
protected function get_excel_line_count($file_path) {
    // This is a rough estimation - you might want to implement a better method
    // or ensure PhpSpreadsheet is available
    
    // For now, return a reasonable estimate based on file size
    $file_size = filesize($file_path);
    
    // Very rough estimation: assume each row is about 200 bytes on average
    $estimated_rows = intval($file_size / 200);
    
    // Subtract header rows (estimated)
    return max(0, $estimated_rows - 5);
}

    /**
     * Process the actual import
     */
public function process_import() {
    header('Content-Type: application/json');
    
    try {
        $import_session = $this->session->userdata('import_session');
        
        if (!$import_session || !file_exists($import_session['file_path'])) {
            throw new Exception('Import session expired or file not found');
        }
        
        $import_options = $this->input->post('import_options', true);
        
        if (empty($import_options)) {
            throw new Exception('Import options are required');
        }
        
        // Start database transaction
        $this->db->trans_start();
        
        // Process import based on type
        $result = $this->process_import_by_type(
            $import_session['import_type'],
            $import_session['file_path'],
            $import_options
        );
        
        // Complete transaction
        $this->db->trans_complete();
        
        if ($this->db->trans_status() === FALSE) {
            throw new Exception('Database transaction failed');
        }
        
        // Log successful import
        $this->Import_model->log_import_activity([
            'import_type' => $import_session['import_type'],
            'filename' => $import_session['original_filename'],
            'records_processed' => $result['processed'],
            'records_successful' => $result['successful'],
            'records_failed' => $result['failed'],
            'status' => 'completed',
            'user_id' => $this->session->userdata('staff_id')
        ]);
        
        // Clean up
        $this->cleanup_import_session();
        
        echo json_encode(array(
            'status' => 'success',
            'message' => 'Import completed successfully',
            'summary' => $result
        ));
        
    } catch (Exception $e) {
        $this->db->trans_rollback();
        
        // Log failed import
        if (isset($import_session)) {
            $this->Import_model->log_import_activity([
                'import_type' => $import_session['import_type'] ?? 'unknown',
                'filename' => $import_session['original_filename'] ?? 'unknown',
                'status' => 'failed',
                'error_message' => $e->getMessage(),
                'user_id' => $this->session->userdata('staff_id') ?? null
            ]);
        }
        
        echo json_encode(array(
            'status' => 'error',
            'message' => $e->getMessage()
        ));
    }
}


    /**
     * Import history page
     */
    public function history() {
        // Set menu navigation
        $this->session->set_userdata('top_menu', 'Import');
        $this->session->set_userdata('sub_menu', 'import/history');

        $data['title'] = 'Import History';
        $data['imports'] = $this->Import_model->get_import_history();
        
        $this->load->view('layout/header', $data);
        $this->load->view('admin/import/history', $data);
        $this->load->view('layout/footer', $data);
    }

    /**
     * View import details
     */
    public function view_import($import_id) {
        $import = $this->Import_model->get_import_by_id($import_id);
        
        if (!$import) {
            show_404();
        }
        
        // Set menu navigation
        $this->session->set_userdata('top_menu', 'Import');
        $this->session->set_userdata('sub_menu', 'import/history');
        
        $data['title'] = 'Import Details - ' . $import->filename;
        $data['import'] = $import;
        
        $this->load->view('layout/header', $data);
        $this->load->view('admin/import/view', $data);
        $this->load->view('layout/footer', $data);
    }

    /**
     * Get import history via AJAX (for filtering and pagination)
     */
    public function get_history_ajax() {
        header('Content-Type: application/json');
        
        $filters = array(
            'type' => $this->input->get('type'),
            'status' => $this->input->get('status'),
            'date_from' => $this->input->get('date_from'),
            'date_to' => $this->input->get('date_to')
        );
        
        $page = $this->input->get('page') ?: 1;
        $limit = $this->input->get('limit') ?: 25;
        $offset = ($page - 1) * $limit;
        
        $imports = $this->Import_model->get_filtered_imports($filters, $limit, $offset);
        $total = $this->Import_model->get_filtered_imports_count($filters);
        $statistics = $this->Import_model->get_import_statistics();
        
        echo json_encode(array(
            'status' => 'success',
            'data' => $imports,
            'pagination' => array(
                'current_page' => $page,
                'per_page' => $limit,
                'total_items' => $total,
                'total_pages' => ceil($total / $limit)
            ),
            'statistics' => $statistics
        ));
    }

    /**
     * Get error details for a failed import
     */
    public function get_error_details($import_id) {
        header('Content-Type: application/json');
        
        $import = $this->Import_model->get_import_by_id($import_id);
        
        if ($import && $import->status == 'failed') {
            echo json_encode(array(
                'status' => 'success',
                'error_message' => $import->error_message
            ));
        } else {
            echo json_encode(array(
                'status' => 'error',
                'message' => 'Import not found or no error details available'
            ));
        }
    }

    /**
     * Get import statistics
     */
    public function get_statistics() {
        header('Content-Type: application/json');
        
        $statistics = $this->Import_model->get_import_statistics();
        
        echo json_encode(array(
            'status' => 'success',
            'statistics' => $statistics
        ));
    }

    /**
     * Cancel current import and clean up
     */
    public function cancel_import() {
        header('Content-Type: application/json');
        
        $this->cleanup_import_session();
        
        echo json_encode(array(
            'status' => 'success',
            'message' => 'Import cancelled successfully'
        ));
    }

    /**
     * Download import template
     */
    public function download_template($import_type) {
        $modules = $this->get_import_modules();
        
        if (!isset($modules[$import_type])) {
            show_404();
        }
        
        $module = $modules[$import_type];
        
        // Create CSV template
        $filename = strtolower(str_replace(' ', '_', $module['name'])) . '_template.csv';
        
        header('Content-Type: text/csv');
        header('Content-Disposition: attachment; filename="' . $filename . '"');
        
        $output = fopen('php://output', 'w');
        
        // Write headers
        $headers = array_merge($module['required_columns'], $module['optional_columns']);
        fputcsv($output, $headers);
        
        // Write sample data
        $sample_data = $this->get_sample_data($import_type);
        fputcsv($output, $sample_data);
        
        fclose($output);
    }

    // =====================================================
    // PRIVATE HELPER METHODS
    // =====================================================

    /**
     * Get available import modules
     */
protected function get_import_modules() {
    return array(
        'nursing_consumables' => array(
            'name' => 'Nursing Consumables',
            'description' => 'Import nursing consumables and medical supplies',
            'target_table' => 'pharmacy_stock_items',
            'group_name' => 'Nursing Consumables',
            'required_columns' => ['item_name', 'closing_stock'],
            'optional_columns' => ['opening_stock', 'added', 'transfer_in', 'transfer_out', 'sales', 'adjusted', 'value']
        ),
        'pharmacy_stock' => array(
            'name' => 'Pharmacy Stock',
            'description' => 'Import pharmacy medications and stock',
            'target_table' => 'pharmacy_stock_items',
            'group_name' => 'Pharmacy',
            'required_columns' => ['item_name', 'closing_stock'],
            'optional_columns' => ['opening_stock', 'added', 'transfer_in', 'transfer_out', 'assembly', 'sales', 'adjusted', 'value']
        ),
        'lab_supplies' => array(
            'name' => 'Laboratory Supplies',
            'description' => 'Import laboratory supplies and consumables',
            'target_table' => 'pharmacy_stock_items',
            'group_name' => 'Lab Supplies',
            'required_columns' => ['item_name', 'closing_stock'],
            'optional_columns' => ['opening_stock', 'added', 'transfer_in', 'transfer_out', 'sales', 'adjusted', 'value']
        )
    );
}

    /**
     * Check if pharmacy item exists - THIS WAS MISSING!
     */
    protected function check_pharmacy_item_exists($item_name, $group_name) {
        $this->db->where('item_name', $item_name);
        $this->db->where('group_name', $group_name);
        $query = $this->db->get('pharmacy_stock_items');
        
        if ($query === FALSE) {
            return false;
        }
        
        return $query->num_rows() > 0 ? $query->row() : false;
    }

    /**
     * Parse file and return preview data
     */
    private function parse_file_preview($file_path, $import_type) {
        $file_extension = pathinfo($file_path, PATHINFO_EXTENSION);
        
        if (in_array($file_extension, ['xls', 'xlsx'])) {
            return $this->parse_excel_preview($file_path, $import_type);
        } elseif ($file_extension == 'csv') {
            return $this->parse_csv_preview($file_path, $import_type);
        } else {
            throw new Exception('Unsupported file format');
        }
    }

    /**
     * Parse Excel file for preview
     */
    private function parse_excel_preview($file_path, $import_type) {
        require_once APPPATH . 'third_party/PhpSpreadsheet/vendor/autoload.php';
        
        try {
            $reader = \PhpOffice\PhpSpreadsheet\IOFactory::createReader('Xlsx');
            $reader->setReadDataOnly(true);
            $spreadsheet = $reader->load($file_path);
            $worksheet = $spreadsheet->getActiveSheet();
            
            // Find the data start row (skip headers like "CITY HEALTH LTD", etc.)
            $data_start_row = $this->find_data_start_row($worksheet);
            
            if ($data_start_row === false) {
                throw new Exception('Could not find data table in the Excel file');
            }
            
            // Get column headers
            $columns = array();
            $col_index = 1;
            while (true) {
                $cell_value = $worksheet->getCellByColumnAndRow($col_index, $data_start_row)->getCalculatedValue();
                if (empty($cell_value)) break;
                $columns[] = trim($cell_value);
                $col_index++;
            }
            
            // Get preview data (first 10 rows)
            $data = array();
            $max_preview_rows = 10;
            $current_preview = 0;
            
            for ($row = $data_start_row + 1; $row <= $worksheet->getHighestRow() && $current_preview < $max_preview_rows; $row++) {
                $row_data = array();
                $has_data = false;
                
                for ($col = 1; $col <= count($columns); $col++) {
                    $cell_value = $worksheet->getCellByColumnAndRow($col, $row)->getCalculatedValue();
                    $row_data[] = $cell_value;
                    if (!empty($cell_value)) $has_data = true;
                }
                
                if ($has_data) {
                    $data[] = $row_data;
                    $current_preview++;
                }
            }
            
            return array(
                'columns' => $columns,
                'data' => $data,
                'data_start_row' => $data_start_row,
                'total_data_rows' => $worksheet->getHighestRow() - $data_start_row
            );
            
        } catch (Exception $e) {
            throw new Exception('Error reading Excel file: ' . $e->getMessage());
        }
    }

    /**
     * Find the row where actual data starts in Excel file
     */
    private function find_data_start_row($worksheet) {
        for ($row = 1; $row <= 20; $row++) { // Check first 20 rows
            $cell_value = strtolower(trim($worksheet->getCell('A' . $row)->getCalculatedValue()));
            
            // Look for common header indicators
            if (strpos($cell_value, 'group') !== false || 
                strpos($cell_value, 'item') !== false ||
                strpos($cell_value, 'product') !== false ||
                strpos($cell_value, 'description') !== false) {
                return $row;
            }
            
            // Alternative check: look for "GROUP" in column B
            $cell_b_value = strtolower(trim($worksheet->getCell('B' . $row)->getCalculatedValue()));
            if ($cell_b_value === 'group') {
                return $row;
            }
        }
        
        return false;
    }

    /**
     * Parse CSV file for preview
     */
    private function parse_csv_preview($file_path, $import_type) {
        $data = array();
        $columns = array();
        
        if (($handle = fopen($file_path, "r")) !== FALSE) {
            // Get headers
            $columns = fgetcsv($handle, 1000, ",");
            
            // Get preview data
            $preview_count = 0;
            while (($row = fgetcsv($handle, 1000, ",")) !== FALSE && $preview_count < 10) {
                $data[] = $row;
                $preview_count++;
            }
            fclose($handle);
        }
        
        return array(
            'columns' => $columns,
            'data' => $data,
            'data_start_row' => 1
        );
    }

/**
 * Process import based on type
 */
protected function process_import_by_type($import_type, $file_path, $options) {
    switch ($import_type) {
        case 'nursing_consumables':
            return $this->process_nursing_consumables($file_path, $options);
        case 'pharmacy_stock':
            return $this->process_pharmacy_stock($file_path, $options);
        case 'lab_supplies':
            return $this->process_lab_supplies($file_path, $options);
        default:
            throw new Exception('Unknown import type: ' . $import_type);
    }
}

/**
 * Process lab supplies import
 */
protected function process_lab_supplies($file_path, $options) {
    $processed = 0;
    $successful = 0;
    $failed = 0;
    $errors = array();
    
    // Generate unique batch ID for this import
    $batch_id = 'BATCH_' . date('Ymd_His') . '_' . uniqid();
    
    // Get all data from file
    $all_data = $this->get_all_file_data($file_path);
    
    foreach ($all_data as $row_index => $row) {
        try {
            $processed++;
            
            // Skip empty rows
            if (empty(array_filter($row, function($value) { return !empty(trim($value)); }))) {
                continue;
            }
            
            // For lab supplies, we'll use the same table structure
            $item_data = array(
                'group_name' => 'Lab Supplies',
                'item_name' => trim($row[0] ?? ''),
                'opening_stock' => floatval($row[2] ?? 0),
                'added' => floatval($row[3] ?? 0),
                'transfer_in' => floatval($row[4] ?? 0),
                'transfer_out' => floatval($row[5] ?? 0),
                'assembly' => floatval($row[6] ?? 0), // Lab supplies might not use assembly, but keep for consistency
                'sales' => floatval($row[7] ?? 0),
                'adjusted' => floatval($row[8] ?? 0),
                'closing_stock' => floatval($row[1] ?? 0),
                'value' => floatval($row[9] ?? 0),
                'import_batch_id' => $batch_id,
                'created_at' => date('Y-m-d H:i:s'),
                'updated_at' => date('Y-m-d H:i:s')
            );
            
            // Validate required fields
            if (empty($item_data['item_name'])) {
                throw new Exception('Item name is required');
            }
            
            if ($item_data['closing_stock'] < 0) {
                throw new Exception('Closing stock cannot be negative');
            }
            
            // Check if item already exists
            $existing_item = $this->check_pharmacy_item_exists($item_data['item_name'], $item_data['group_name']);
            
            if ($existing_item) {
                // Handle duplicate based on options
                if ($options['duplicate_action'] === 'skip') {
                    continue;
                } elseif ($options['duplicate_action'] === 'error') {
                    throw new Exception('Item already exists: ' . $item_data['item_name']);
                } elseif ($options['duplicate_action'] === 'update') {
                    $this->db->where('id', $existing_item->id);
                    $this->db->update('pharmacy_stock_items', $item_data);
                }
            } else {
                // Insert new item
                $this->db->insert('pharmacy_stock_items', $item_data);
            }
            
            $successful++;
            
        } catch (Exception $e) {
            $failed++;
            $errors[] = "Row " . ($row_index + 1) . ": " . $e->getMessage();
        }
    }
    
    return array(
        'processed' => $processed,
        'successful' => $successful,
        'failed' => $failed,
        'errors' => $errors,
        'batch_id' => $batch_id
    );
}

/**
 * Check if item exists
 */
protected function check_item_exists($item_name, $category_id) {
    // Try different table names based on category
    $tables = array(
        1 => 'consumables',      // Nursing
        2 => 'pharmacy_stock_items', // Pharmacy  
        4 => 'consumables'       // Lab
    );
    
    $table = $tables[$category_id] ?? 'consumables';
    
    $this->db->where('item_name', $item_name);
    $this->db->where('category_id', $category_id);
    $query = $this->db->get($table);
    
    return $query->num_rows() > 0 ? $query->row() : false;
}

/**
 * Process nursing consumables import
 */
protected function process_nursing_consumables($file_path, $options) {
    $processed = 0;
    $successful = 0;
    $failed = 0;
    $errors = array();
    
    // Generate unique batch ID for this import
    $batch_id = 'BATCH_' . date('Ymd_His') . '_' . uniqid();
    
    // Get all data from file
    $all_data = $this->get_all_file_data($file_path);
    
    foreach ($all_data as $row_index => $row) {
        try {
            $processed++;
            
            // Skip empty rows
            if (empty(array_filter($row, function($value) { return !empty(trim($value)); }))) {
                continue;
            }
            
            // For nursing consumables, we'll use the same table structure as pharmacy
            // but with a different group_name
            $item_data = array(
                'group_name' => 'Nursing Consumables',
                'item_name' => trim($row[0] ?? ''),
                'opening_stock' => floatval($row[2] ?? 0),
                'added' => floatval($row[3] ?? 0),
                'transfer_in' => floatval($row[4] ?? 0),
                'transfer_out' => floatval($row[5] ?? 0),
                'assembly' => floatval($row[6] ?? 0),
                'sales' => floatval($row[7] ?? 0),
                'adjusted' => floatval($row[8] ?? 0),
                'closing_stock' => floatval($row[1] ?? 0),
                'value' => floatval($row[9] ?? 0),
                'import_batch_id' => $batch_id,
                'created_at' => date('Y-m-d H:i:s'),
                'updated_at' => date('Y-m-d H:i:s')
            );
            
            // Validate required fields
            if (empty($item_data['item_name'])) {
                throw new Exception('Item name is required');
            }
            
            if ($item_data['closing_stock'] < 0) {
                throw new Exception('Closing stock cannot be negative');
            }
            
            // Check if item already exists
            $existing_item = $this->check_pharmacy_item_exists($item_data['item_name'], $item_data['group_name']);
            
            if ($existing_item) {
                // Handle duplicate based on options
                if ($options['duplicate_action'] === 'skip') {
                    continue;
                } elseif ($options['duplicate_action'] === 'error') {
                    throw new Exception('Item already exists: ' . $item_data['item_name']);
                } elseif ($options['duplicate_action'] === 'update') {
                    $this->db->where('id', $existing_item->id);
                    $this->db->update('pharmacy_stock_items', $item_data);
                }
            } else {
                // Insert new item
                $this->db->insert('pharmacy_stock_items', $item_data);
            }
            
            $successful++;
            
        } catch (Exception $e) {
            $failed++;
            $errors[] = "Row " . ($row_index + 1) . ": " . $e->getMessage();
        }
    }
    
    return array(
        'processed' => $processed,
        'successful' => $successful,
        'failed' => $failed,
        'errors' => $errors,
        'batch_id' => $batch_id
    );
}

protected function get_all_file_data($file_path) {
    $file_extension = pathinfo($file_path, PATHINFO_EXTENSION);
    
    if (in_array($file_extension, ['xls', 'xlsx'])) {
        return $this->get_all_excel_data($file_path);
    } else {
        return $this->get_all_csv_data($file_path);
    }
}

    /**
     * Process lab consumables import
     */
    private function process_lab_consumables($file_path, $column_mapping, $options) {
        $modules = $this->get_import_modules();
        $module_config = $modules['lab_consumables'];
        
        return $this->process_consumables_import($file_path, $column_mapping, $options, $module_config);
    }

    /**
     * Generic consumables import processor
     */
    private function process_consumables_import($file_path, $column_mapping, $options, $module_config) {
        $file_extension = pathinfo($file_path, PATHINFO_EXTENSION);
        $processed = 0;
        $successful = 0;
        $failed = 0;
        $errors = array();
        
        // Get all data from file
        if (in_array($file_extension, ['xls', 'xlsx'])) {
            $all_data = $this->get_all_excel_data($file_path);
        } else {
            $all_data = $this->get_all_csv_data($file_path);
        }
        
        foreach ($all_data as $row_index => $row_data) {
            $processed++;
            
            try {
                // Map columns to database fields
                $mapped_data = array();
                foreach ($column_mapping as $file_column => $db_field) {
                    if ($db_field && isset($row_data[$file_column])) {
                        $mapped_data[$db_field] = $row_data[$file_column];
                    }
                }
                
                // Validate required fields
                if (empty($mapped_data['item_name'])) {
                    throw new Exception('Item name is required');
                }
                
                // Prepare consumable data
                $consumable_data = array(
                    'consumable_name' => trim($mapped_data['item_name']),
                    'category_id' => $module_config['category_id'],
                    'unit_of_measure' => 'pieces',
                    'unit_cost' => !empty($mapped_data['unit_cost']) ? floatval($mapped_data['unit_cost']) : 0.00,
                    'current_stock' => !empty($mapped_data['closing_stock']) ? intval($mapped_data['closing_stock']) : 0,
                    'minimum_stock' => !empty($mapped_data['minimum_stock']) ? intval($mapped_data['minimum_stock']) : 0,
                    'reorder_level' => !empty($mapped_data['reorder_level']) ? intval($mapped_data['reorder_level']) : 0,
                    'is_active' => 'yes',
                    'created_at' => date('Y-m-d H:i:s'),
                    'updated_at' => date('Y-m-d H:i:s')
                );
                
                // Generate consumable code if not provided
                if (empty($consumable_data['consumable_code'])) {
                    $consumable_data['consumable_code'] = $this->generate_consumable_code($module_config['category_id']);
                }
                
                // Check if item already exists
                $existing_item = $this->db->get_where('consumables', array(
                    'consumable_name' => $consumable_data['consumable_name'],
                    'category_id' => $consumable_data['category_id']
                ))->row();
                
                if ($existing_item && $options['update_existing'] != '1') {
                    if ($options['skip_existing'] == '1') {
                        continue; // Skip this item
                    } else {
                        throw new Exception('Item already exists: ' . $consumable_data['consumable_name']);
                    }
                } elseif ($existing_item && $options['update_existing'] == '1') {
                    // Update existing item
                    unset($consumable_data['created_at']);
                    $this->db->where('id', $existing_item->id);
                    $this->db->update('consumables', $consumable_data);
                } else {
                    // Insert new item
                    $this->db->insert('consumables', $consumable_data);
                }
                
                $successful++;
                
            } catch (Exception $e) {
                $failed++;
                $errors[] = "Row " . ($row_index + 1) . ": " . $e->getMessage();
            }
        }
        
        return array(
            'processed' => $processed,
            'successful' => $successful,
            'failed' => $failed,
            'errors' => $errors
        );
    }

/**
 * Process pharmacy stock import
 */
protected function process_pharmacy_stock($file_path, $options) {
    $processed = 0;
    $successful = 0;
    $failed = 0;
    $errors = array();
    
    // Generate unique batch ID for this import
    $batch_id = 'BATCH_' . date('Ymd_His') . '_' . uniqid();
    
    // Get all data from file
    $all_data = $this->get_all_file_data($file_path);
    
    foreach ($all_data as $row_index => $row) {
        try {
            $processed++;
            
            // Skip empty rows
            if (empty(array_filter($row, function($value) { return !empty(trim($value)); }))) {
                continue;
            }
            
            // Map the data for pharmacy stock based on your table structure
            $item_data = array(
                'group_name' => 'Pharmacy', // Default group name
                'item_name' => trim($row[0] ?? ''),
                'opening_stock' => floatval($row[2] ?? 0),
                'added' => floatval($row[3] ?? 0),
                'transfer_in' => floatval($row[4] ?? 0),
                'transfer_out' => floatval($row[5] ?? 0),
                'assembly' => floatval($row[6] ?? 0),
                'sales' => floatval($row[7] ?? 0),
                'adjusted' => floatval($row[8] ?? 0),
                'closing_stock' => floatval($row[1] ?? 0), // Closing stock is usually column 2
                'value' => floatval($row[9] ?? 0),
                'import_batch_id' => $batch_id,
                'created_at' => date('Y-m-d H:i:s'),
                'updated_at' => date('Y-m-d H:i:s')
            );
            
            // Validate required fields
            if (empty($item_data['item_name'])) {
                throw new Exception('Item name is required');
            }
            
            if ($item_data['closing_stock'] < 0) {
                throw new Exception('Closing stock cannot be negative');
            }
            
            // Check if item already exists (using group_name and item_name)
            $existing_item = $this->check_pharmacy_item_exists($item_data['item_name'], $item_data['group_name']);
            
            if ($existing_item) {
                // Handle duplicate based on options
                if ($options['duplicate_action'] === 'skip') {
                    continue; // Skip this item
                } elseif ($options['duplicate_action'] === 'error') {
                    throw new Exception('Item already exists: ' . $item_data['item_name']);
                } elseif ($options['duplicate_action'] === 'update') {
                    // Update existing item
                    $this->db->where('id', $existing_item->id);
                    $this->db->update('pharmacy_stock_items', $item_data);
                }
            } else {
                // Insert new item
                $this->db->insert('pharmacy_stock_items', $item_data);
            }
            
            $successful++;
            
        } catch (Exception $e) {
            $failed++;
            $errors[] = "Row " . ($row_index + 1) . ": " . $e->getMessage();
        }
    }
    
    return array(
        'processed' => $processed,
        'successful' => $successful,
        'failed' => $failed,
        'errors' => $errors,
        'batch_id' => $batch_id
    );
}
/**
 * Generate unique stock code for pharmacy
 */
protected function generate_stock_code($category_id) {
    $prefix = 'PHR';
    
    // Get next sequence number
    $this->db->select('MAX(CAST(SUBSTRING(stock_code, 4) AS UNSIGNED)) as max_num');
    $this->db->where('stock_code LIKE', $prefix . '%');
    $result = $this->db->get('pharmacy_stock_items')->row();
    
    $next_num = ($result && $result->max_num) ? $result->max_num + 1 : 1;
    
    return $prefix . str_pad($next_num, 4, '0', STR_PAD_LEFT);
}


    /**
     * Get all data from Excel file
     */
protected function get_all_excel_data($file_path) {
    // If PhpSpreadsheet is available
    if (file_exists(APPPATH . 'third_party/PhpSpreadsheet/vendor/autoload.php')) {
        require_once APPPATH . 'third_party/PhpSpreadsheet/vendor/autoload.php';
        
        $reader = \PhpOffice\PhpSpreadsheet\IOFactory::createReader('Xlsx');
        $reader->setReadDataOnly(true);
        $spreadsheet = $reader->load($file_path);
        $worksheet = $spreadsheet->getActiveSheet();
        
        $data_start_row = $this->find_data_start_row($worksheet);
        
        // Get column count
        $col_count = 0;
        $col_index = 1;
        while (true) {
            $cell_value = $worksheet->getCellByColumnAndRow($col_index, $data_start_row)->getCalculatedValue();
            if (empty($cell_value)) break;
            $col_count++;
            $col_index++;
        }
        
        $all_data = array();
        for ($row = $data_start_row + 1; $row <= $worksheet->getHighestRow(); $row++) {
            $row_data = array();
            $has_data = false;
            
            for ($col = 1; $col <= $col_count; $col++) {
                $cell_value = $worksheet->getCellByColumnAndRow($col, $row)->getCalculatedValue();
                $row_data[] = $cell_value;
                if (!empty($cell_value)) $has_data = true;
            }
            
            if ($has_data) {
                $all_data[] = $row_data;
            }
        }
        
        return $all_data;
    } else {
        // Fallback - treat as CSV
        return $this->get_all_csv_data($file_path);
    }
}

    /**
     * Get all data from CSV file
     */
protected function get_all_csv_data($file_path) {
    $all_data = array();
    
    if (($handle = fopen($file_path, "r")) !== FALSE) {
        // Skip header row
        fgetcsv($handle, 1000, ",");
        
        // Get all data
        while (($row = fgetcsv($handle, 1000, ",")) !== FALSE) {
            $all_data[] = $row;
        }
        fclose($handle);
    }
    
    return $all_data;
}

/**
 * Generate unique consumable code
 */
protected function generate_consumable_code($category_id) {
    $category_codes = array(
        1 => 'MED',  // Medical Supplies
        2 => 'PHR',  // Pharmaceuticals
        4 => 'LAB',  // Laboratory Supplies
    );
    
    $prefix = isset($category_codes[$category_id]) ? $category_codes[$category_id] : 'CON';
    
    // Get next sequence number
    $this->db->select('MAX(CAST(SUBSTRING(consumable_code, 4) AS UNSIGNED)) as max_num');
    $this->db->where('consumable_code LIKE', $prefix . '%');
    $this->db->where('category_id', $category_id);
    $result = $this->db->get('consumables')->row();
    
    $next_num = ($result && $result->max_num) ? $result->max_num + 1 : 1;
    
    return $prefix . str_pad($next_num, 4, '0', STR_PAD_LEFT);
}


    /**
     * Get sample data for templates
     */
    private function get_sample_data($import_type) {
        switch ($import_type) {
            case 'nursing_consumables':
                return array('5ML SYRINGE', '10', '0', '0', '0', '0', '0', '10', '150.00');
            case 'pharmacy_stock':
                return array('Paracetamol 500mg', '100', '50', '10', '5', '2', '20', '5', '138', '6900.00');
            case 'lab_consumables':
                return array('Blood Test Tubes', '50', '0', '0', '0', '5', '0', '45', '1575.00');
            default:
                return array();
        }
    }

    /**
     * Clean up import session and temporary files
     */
protected function cleanup_import_session() {
    $import_session = $this->session->userdata('import_session');
    
    if ($import_session && isset($import_session['file_path']) && file_exists($import_session['file_path'])) {
        unlink($import_session['file_path']);
    }
    
    $this->session->unset_userdata('import_session');
}
}