<?php
defined('BASEPATH') OR exit('No direct script access allowed');

/**
 * Front Office Controller
 * Handles front office operations including copay collection, cash management, and dashboard
 */
class Front_office extends Admin_Controller {

    public function __construct() {
        parent::__construct();
        
        // Check if user is logged in using your session structure
        $hospitaladmin = $this->session->userdata('hospitaladmin');
        if (!$hospitaladmin || !isset($hospitaladmin['id'])) {
            log_message('debug', 'Front Office: No hospitaladmin session found');
            redirect('site/userlogin');
        }
        
        $this->load->model('Front_office_model');
        $this->load->model('Patient_model');
        $this->load->model('Queue_model');
        $this->load->library('form_validation');
    }

    /**
     * Get current admin ID from session
     */
    private function getCurrentAdminId() {
        $hospitaladmin = $this->session->userdata('hospitaladmin');
        return $hospitaladmin['id'] ?? null;
    }

    /**
     * Get current admin username from session
     */
    private function getCurrentUsername() {
        $hospitaladmin = $this->session->userdata('hospitaladmin');
        return $hospitaladmin['username'] ?? 'Unknown User';
    }

    /**
     * Front Office Dashboard
     * Route: /admin/front_office/dashboard
     */
    public function dashboard() {
        // Temporarily comment out RBAC check for testing
        // if (!$this->rbac->hasPrivilege('front_office_dashboard', 'can_view')) {
        //     access_denied();
        // }

        $this->session->set_userdata('top_menu', 'Front Office');
        $this->session->set_userdata('sub_menu', 'front_office/dashboard');

        $data['title'] = 'Front Office Dashboard';
        $data['current_staff_id'] = $this->getCurrentAdminId();
        $data['current_staff_name'] = $this->getCurrentUsername();
        
        // Check if view file exists
        $view_path = APPPATH . 'views/admin/front_office/dashboard.php';
        if (!file_exists($view_path)) {
            // Create the directory if it doesn't exist
            $dir_path = APPPATH . 'views/admin/front_office/';
            if (!is_dir($dir_path)) {
                mkdir($dir_path, 0755, true);
            }
            
            // For now, show a simple message
            echo "<h1>Front Office Dashboard</h1>";
            echo "<p>Dashboard view file needs to be created at: {$view_path}</p>";
            echo "<p>Current Admin ID: " . $data['current_staff_id'] . "</p>";
            echo "<p>Current Username: " . $data['current_staff_name'] . "</p>";
            echo "<p>Session is working correctly!</p>";
            return;
        }
        
        $this->load->view('layout/header', $data);
        $this->load->view('admin/front_office/dashboard', $data);
        $this->load->view('layout/footer', $data);
    }

    /**
     * Index method as fallback
     */
    public function index() {
        $this->dashboard();
    }

    /**
     * Debug session method
     */
    public function debug_session() {
        echo "<h3>Session Debug Information</h3>";
        echo "<pre>";
        print_r($this->session->all_userdata());
        echo "</pre>";
        
        $hospitaladmin = $this->session->userdata('hospitaladmin');
        
        echo "<h3>Required Session Variables</h3>";
        echo "hospitaladmin array exists: " . (is_array($hospitaladmin) ? 'YES' : 'NO') . "<br>";
        echo "admin_id (hospitaladmin[id]): " . ($hospitaladmin['id'] ?? 'NOT SET') . "<br>";
        echo "username (hospitaladmin[username]): " . ($hospitaladmin['username'] ?? 'NOT SET') . "<br>";
        echo "email (hospitaladmin[email]): " . ($hospitaladmin['email'] ?? 'NOT SET') . "<br>";
        
        echo "<h3>Current URI</h3>";
        echo "URI: " . uri_string() . "<br>";
        echo "Base URL: " . base_url() . "<br>";
        
        echo "<h3>Methods Test</h3>";
        echo "getCurrentAdminId(): " . $this->getCurrentAdminId() . "<br>";
        echo "getCurrentUsername(): " . $this->getCurrentUsername() . "<br>";
    }

    /**
     * Get Dashboard Statistics (AJAX)
     * Route: /admin/front_office/get_dashboard_stats
     */
    public function get_dashboard_stats() {
        header('Content-Type: application/json');
        
        // Temporarily comment out RBAC check
        // if (!$this->rbac->hasPrivilege('front_office_dashboard', 'can_view')) {
        //     echo json_encode(['status' => 'error', 'message' => 'Access denied']);
        //     return;
        // }

        try {
            $staff_id = $this->getCurrentAdminId();
            $today = date('Y-m-d');
            
            if (!$staff_id) {
                echo json_encode(['status' => 'error', 'message' => 'No staff ID found in session']);
                return;
            }
            
            $stats = $this->Front_office_model->getDashboardStats($staff_id, $today);
            
            echo json_encode([
                'status' => 'success',
                'data' => $stats
            ]);
        } catch (Exception $e) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Failed to load dashboard stats: ' . $e->getMessage()
            ]);
        }
    }

    /**
     * Get Recent Copay Collections (AJAX)
     * Route: /admin/front_office/get_recent_copays
     */
    public function get_recent_copays() {
        header('Content-Type: application/json');
        
        try {
            $staff_id = $this->getCurrentAdminId();
            $today = date('Y-m-d');
            
            if (!$staff_id) {
                echo json_encode(['status' => 'error', 'message' => 'No staff ID found in session']);
                return;
            }
            
            $copays = $this->Front_office_model->getRecentCopays($staff_id, $today);
            
            echo json_encode([
                'status' => 'success',
                'data' => $copays
            ]);
        } catch (Exception $e) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Failed to load copay collections: ' . $e->getMessage()
            ]);
        }
    }

    /**
     * Get Activity Log (AJAX)
     * Route: /admin/front_office/get_activity_log
     */
    public function get_activity_log() {
        header('Content-Type: application/json');
        
        try {
            $staff_id = $this->getCurrentAdminId();
            $today = date('Y-m-d');
            
            if (!$staff_id) {
                echo json_encode(['status' => 'error', 'message' => 'No staff ID found in session']);
                return;
            }
            
            $activities = $this->Front_office_model->getActivityLog($staff_id, $today);
            
            echo json_encode([
                'status' => 'success',
                'data' => $activities
            ]);
        } catch (Exception $e) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Failed to load activity log: ' . $e->getMessage()
            ]);
        }
    }

    /**
     * Start Staff Shift (AJAX)
     * Route: /admin/front_office/start_shift
     */
    public function start_shift() {
        header('Content-Type: application/json');
        
        try {
            $staff_id = $this->getCurrentAdminId();
            $opening_cash = (float) $this->input->post('opening_cash');
            
            if (!$staff_id) {
                echo json_encode(['status' => 'error', 'message' => 'No staff ID found in session']);
                return;
            }
            
            if ($opening_cash <= 0) {
                echo json_encode(['status' => 'error', 'message' => 'Invalid opening cash amount']);
                return;
            }

            $result = $this->Front_office_model->startShift($staff_id, $opening_cash);
            
            if ($result) {
                echo json_encode([
                    'status' => 'success',
                    'message' => 'Shift started successfully',
                    'shift_start_time' => date('H:i:s')
                ]);
            } else {
                echo json_encode(['status' => 'error', 'message' => 'Failed to start shift']);
            }
        } catch (Exception $e) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Error starting shift: ' . $e->getMessage()
            ]);
        }
    }

    /**
     * End Staff Shift (AJAX)
     * Route: /admin/front_office/end_shift
     */
    public function end_shift() {
        header('Content-Type: application/json');
        
        try {
            $staff_id = $this->getCurrentAdminId();
            $closing_cash = (float) $this->input->post('closing_cash');
            
            if (!$staff_id) {
                echo json_encode(['status' => 'error', 'message' => 'No staff ID found in session']);
                return;
            }
            
            if ($closing_cash < 0) {
                echo json_encode(['status' => 'error', 'message' => 'Invalid closing cash amount']);
                return;
            }

            $result = $this->Front_office_model->endShift($staff_id, $closing_cash);
            
            if ($result) {
                echo json_encode([
                    'status' => 'success',
                    'message' => 'Shift ended successfully',
                    'reconciliation_report' => $result['report_url'] ?? null
                ]);
            } else {
                echo json_encode(['status' => 'error', 'message' => 'Failed to end shift']);
            }
        } catch (Exception $e) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Error ending shift: ' . $e->getMessage()
            ]);
        }
    }

    /**
     * Process Copay Payment (AJAX)
     * Route: /admin/front_office/process_copay
     */
    public function process_copay() {
        header('Content-Type: application/json');
        
        try {
            $this->form_validation->set_rules('patient_id', 'Patient ID', 'required|numeric');
            $this->form_validation->set_rules('payment_method', 'Payment Method', 'required');
            $this->form_validation->set_rules('amount_received', 'Amount Received', 'required|numeric|greater_than[0]');

            if ($this->form_validation->run() == FALSE) {
                echo json_encode([
                    'status' => 'error',
                    'message' => 'Validation failed',
                    'errors' => validation_errors()
                ]);
                return;
            }

            $staff_id = $this->getCurrentAdminId();
            if (!$staff_id) {
                echo json_encode(['status' => 'error', 'message' => 'No staff ID found in session']);
                return;
            }

            $patient_id = $this->input->post('patient_id');
            $payment_method = $this->input->post('payment_method');
            $amount_received = (float) $this->input->post('amount_received');
            $notes = $this->input->post('notes');

            // Get patient details
            $patient = $this->Patient_model->getpatientDetails($patient_id);
            if (!$patient) {
                echo json_encode(['status' => 'error', 'message' => 'Patient not found']);
                return;
            }

            // Process copay collection
            $result = $this->Front_office_model->processCopayCollection(
                $patient_id,
                $amount_received,
                $payment_method,
                $staff_id,
                $notes
            );

            if ($result['status'] === 'success') {
                echo json_encode([
                    'status' => 'success',
                    'message' => 'Copay payment processed successfully',
                    'receipt_number' => $result['receipt_number']
                ]);
            } else {
                echo json_encode([
                    'status' => 'error',
                    'message' => $result['message']
                ]);
            }
        } catch (Exception $e) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Error processing copay: ' . $e->getMessage()
            ]);
        }
    }

    /**
     * Search Patients for Copay Collection (AJAX)
     * Route: /admin/front_office/search_patients
     */
    public function search_patients() {
        header('Content-Type: application/json');
        
        try {
            $search_term = $this->input->get('search');
            
            if (strlen($search_term) < 3) {
                echo json_encode(['status' => 'error', 'message' => 'Search term must be at least 3 characters']);
                return;
            }

            $patients = $this->Front_office_model->searchPatientsForCopay($search_term);
            
            echo json_encode([
                'status' => 'success',
                'patients' => $patients
            ]);
        } catch (Exception $e) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Error searching patients: ' . $e->getMessage()
            ]);
        }
    }

    /**
     * Insurance Verification Page
     * Route: /admin/front_office/insurance_verification
     */
    public function insurance_verification() {
        $this->session->set_userdata('top_menu', 'Front Office');
        $this->session->set_userdata('sub_menu', 'front_office/insurance_verification');

        $data['title'] = 'Insurance Verification';
        
        echo "<h1>Insurance Verification</h1>";
        echo "<p>This feature is under development.</p>";
        echo "<p>Current User: " . $this->getCurrentUsername() . "</p>";
    }

    /**
     * Front Office Reports
     * Route: /admin/front_office/reports
     */
    public function reports() {
        $this->session->set_userdata('top_menu', 'Front Office');
        $this->session->set_userdata('sub_menu', 'front_office/reports');

        $data['title'] = 'Front Office Reports';
        $data['staff_list'] = $this->Front_office_model->getFrontOfficeStaff();
        
        echo "<h1>Front Office Reports</h1>";
        echo "<p>Reports feature is under development.</p>";
        echo "<p>Current User: " . $this->getCurrentUsername() . "</p>";
    }

    /**
     * General Registration Route
     * Route: /admin/front_office/general_registration
     */
    public function general_registration() {
        $this->session->set_userdata('top_menu', 'Front Office');
        $this->session->set_userdata('sub_menu', 'front_office/general_registration');

        $data['title'] = 'General Patient Registration';
        $data['organisations'] = $this->Front_office_model->getActiveInsuranceProviders();
        
        echo "<h1>Enhanced Patient Registration</h1>";
        echo "<p>Enhanced registration form is under development.</p>";
        echo "<p>Current User: " . $this->getCurrentUsername() . "</p>";
        echo "<p>Available Insurance Providers: " . count($data['organisations']) . "</p>";
    }

    /**
     * Cash Reconciliation Report
     * Route: /admin/front_office/cash_reconciliation
     */
    public function cash_reconciliation() {
        $this->session->set_userdata('top_menu', 'Front Office');
        $this->session->set_userdata('sub_menu', 'front_office/cash_reconciliation');

        $data['title'] = 'Cash Reconciliation';
        $data['staff_id'] = $this->getCurrentAdminId();
        $data['reconciliation_data'] = $this->Front_office_model->getCashReconciliationData();
        
        echo "<h1>Cash Reconciliation</h1>";
        echo "<p>Cash reconciliation feature is under development.</p>";
        echo "<p>Current Staff ID: " . $data['staff_id'] . "</p>";
        echo "<p>Reconciliation Records: " . count($data['reconciliation_data']) . "</p>";
    }

    /**
     * Database Status Check (for troubleshooting)
     * Route: /admin/front_office/db_status
     */
    public function db_status() {
        if (!$this->getCurrentAdminId()) {
            echo "Access denied - not logged in";
            return;
        }

        echo "<h3>Database Status Check</h3>";
        
        $status = $this->Front_office_model->getDatabaseStatus();
        
        echo "<h4>Table Status:</h4>";
        echo "<ul>";
        foreach ($status['tables'] as $table => $exists) {
            $status_text = $exists ? '✅ Exists' : '❌ Missing';
            echo "<li>{$table}: {$status_text}</li>";
        }
        echo "</ul>";
        
        if (isset($status['patient_columns'])) {
            echo "<h4>Patient Table Columns:</h4>";
            echo "<ul>";
            foreach ($status['patient_columns'] as $column => $exists) {
                $status_text = $exists ? '✅ Exists' : '❌ Missing';
                echo "<li>{$column}: {$status_text}</li>";
            }
            echo "</ul>";
        }
        
        echo "<h4>Core Tables:</h4>";
        echo "<ul>";
        echo "<li>Patients: " . ($status['patients_table'] ? '✅' : '❌') . "</li>";
        echo "<li>Staff: " . ($status['staff_table'] ? '✅' : '❌') . "</li>";
        echo "<li>Organisation: " . ($status['organisation_table'] ? '✅' : '❌') . "</li>";
        echo "</ul>";
        
        echo "<p><strong>Database Connection:</strong> " . ($status['database_connection'] ? '✅ Connected' : '❌ Failed') . "</p>";
    }
    
    /**
     * Verify Insurance Coverage (AJAX)
     * Route: /admin/front_office/verify_insurance
     */
    public function verify_insurance() {
        header('Content-Type: application/json');
        
        try {
            $this->form_validation->set_rules('patient_id', 'Patient ID', 'required|numeric');
            $this->form_validation->set_rules('organisation_id', 'Organisation ID', 'required|numeric');
            $this->form_validation->set_rules('insurance_id', 'Insurance ID', 'required');

            if ($this->form_validation->run() == FALSE) {
                echo json_encode([
                    'status' => 'error',
                    'message' => 'Validation failed',
                    'errors' => validation_errors()
                ]);
                return;
            }

            $patient_id = $this->input->post('patient_id');
            $organisation_id = $this->input->post('organisation_id');
            $insurance_id = $this->input->post('insurance_id');
            $staff_id = $this->getCurrentAdminId();

            if (!$staff_id) {
                echo json_encode(['status' => 'error', 'message' => 'No staff ID found in session']);
                return;
            }

            $result = $this->Front_office_model->verifyInsurance(
                $patient_id,
                $organisation_id,
                $insurance_id,
                $staff_id
            );

            echo json_encode($result);
        } catch (Exception $e) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Error verifying insurance: ' . $e->getMessage()
            ]);
        }
    }

    /**
     * Generate Daily Report
     * Route: /admin/front_office/generate_daily_report
     */
    public function generate_daily_report() {
        $date = $this->input->get('date') ?: date('Y-m-d');
        $staff_id = $this->input->get('staff_id');

        try {
            $report_data = $this->Front_office_model->generateDailyReport($date, $staff_id);
            
            $data['report_data'] = $report_data;
            $data['report_date'] = $date;
            $data['title'] = 'Daily Front Office Report - ' . date('d/m/Y', strtotime($date));
            $data['current_user'] = $this->getCurrentUsername();
            
            echo "<h1>Daily Front Office Report</h1>";
            echo "<p>Date: " . date('d/m/Y', strtotime($date)) . "</p>";
            echo "<p>Generated by: " . $data['current_user'] . "</p>";
            echo "<p>Cash Tracking Records: " . count($report_data['cash_tracking']) . "</p>";
            echo "<p>Registration Records: " . count($report_data['registrations']) . "</p>";
            
        } catch (Exception $e) {
            show_error('Error generating report: ' . $e->getMessage());
        }
    }

    /**
     * Update Cash Tracking (AJAX)
     * Route: /admin/front_office/update_cash_tracking
     */
    public function update_cash_tracking() {
        header('Content-Type: application/json');
        
        try {
            $staff_id = $this->getCurrentAdminId();
            $cash_dispensed = (float) $this->input->post('cash_dispensed');
            
            if (!$staff_id) {
                echo json_encode(['status' => 'error', 'message' => 'No staff ID found in session']);
                return;
            }
            
            $result = $this->Front_office_model->updateCashDispensed($staff_id, $cash_dispensed);
            
            if ($result) {
                echo json_encode(['status' => 'success', 'message' => 'Cash tracking updated']);
            } else {
                echo json_encode(['status' => 'error', 'message' => 'Failed to update cash tracking']);
            }
        } catch (Exception $e) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Error updating cash tracking: ' . $e->getMessage()
            ]);
        }
    }

    /**
     * Get Queue Widget Data for Dashboard
     * Route: /admin/front_office/queue_widget
     */
    public function queue_widget() {
        header('Content-Type: application/json');
        
        try {
            // Check if Queue_model exists
            if (!class_exists('Queue_model')) {
                echo json_encode([
                    'status' => 'success', 
                    'widget' => [
                        'department' => 'registration',
                        'display_name' => 'Registration Queue',
                        'waiting_count' => 0,
                        'completed_today' => 0,
                        'avg_wait_time' => 0
                    ]
                ]);
                return;
            }

            $stats = $this->Queue_model->getQueueStatistics('registration');
            
            $widget_data = [
                'department' => 'registration',
                'display_name' => 'Registration Queue',
                'waiting_count' => $stats['waiting_count'] ?? 0,
                'completed_today' => $stats['completed_today'] ?? 0,
                'avg_wait_time' => $stats['avg_wait_time'] ?? 0
            ];
            
            echo json_encode(['status' => 'success', 'widget' => $widget_data]);
        } catch (Exception $e) {
            echo json_encode(['status' => 'error', 'message' => $e->getMessage()]);
        }
    }

    /**
     * Initialize database tables (for setup)
     * Route: /admin/front_office/initialize_tables
     */
    public function initialize_tables() {
        // Only allow Super Admin
        $hospitaladmin = $this->session->userdata('hospitaladmin');
        if (!$hospitaladmin || $hospitaladmin['username'] !== 'Super Admin ') {
            echo "Access denied - Super Admin only";
            return;
        }

        echo "<h3>Initializing Front Office Database Tables</h3>";
        
        try {
            $results = $this->Front_office_model->initializeMissingTables();
            
            echo "<h4>Table Creation Results:</h4>";
            echo "<ul>";
            foreach ($results as $table => $result) {
                echo "<li>{$table}: {$result}</li>";
            }
            echo "</ul>";
            
            $column_results = $this->Front_office_model->addMissingColumns();
            
            echo "<h4>Column Addition Results:</h4>";
            echo "<ul>";
            foreach ($column_results as $column => $result) {
                echo "<li>{$column}: {$result}</li>";
            }
            echo "</ul>";
            
            echo "<p><strong>Initialization completed!</strong></p>";
            echo "<p><a href='" . base_url('admin/front_office/dashboard') . "'>Go to Front Office Dashboard</a></p>";
            
        } catch (Exception $e) {
            echo "<p style='color: red;'>Error: " . $e->getMessage() . "</p>";
        }
    }

    /**
     * Test method to verify everything is working
     * Route: /admin/front_office/test
     */
    public function test() {
        echo "<h1>Front Office Test Page</h1>";
        echo "<p>✅ Controller is working!</p>";
        echo "<p>✅ Session is working!</p>";
        echo "<p>Current User: " . $this->getCurrentUsername() . "</p>";
        echo "<p>Current Admin ID: " . $this->getCurrentAdminId() . "</p>";
        
        echo "<h3>Available Routes:</h3>";
        echo "<ul>";
        echo "<li><a href='" . base_url('admin/front_office/dashboard') . "'>Dashboard</a></li>";
        echo "<li><a href='" . base_url('admin/front_office/debug_session') . "'>Debug Session</a></li>";
        echo "<li><a href='" . base_url('admin/front_office/db_status') . "'>Database Status</a></li>";
        echo "<li><a href='" . base_url('admin/front_office/initialize_tables') . "'>Initialize Tables</a></li>";
        echo "</ul>";
        
        echo "<h3>AJAX Endpoints:</h3>";
        echo "<ul>";
        echo "<li>" . base_url('admin/front_office/get_dashboard_stats') . "</li>";
        echo "<li>" . base_url('admin/front_office/get_recent_copays') . "</li>";
        echo "<li>" . base_url('admin/front_office/get_activity_log') . "</li>";
        echo "</ul>";
    }
}
