<?php

if (!defined('BASEPATH'))
    exit('No direct script access allowed');

class Dispensary extends Admin_Controller {

    function __construct() {
        parent::__construct();
        
        // Load essential libraries and models
        try {
            $this->load->model('patient_model');
            $this->load->model('dispensary_model');
            $this->load->model('staff_model');
            $this->load->model('Financial_model'); // Load Financial model for accounting integration
            $this->load->library('datatables');
            $this->load->library('customlib');
            $this->load->model('Queue_model');
            $this->load->helper('form');
            $this->load->helper('url');
        } catch (Exception $e) {
            log_message('error', 'Dispensary controller error: ' . $e->getMessage());
            show_error('Failed to load required libraries: ' . $e->getMessage());
        }
    }

    /**
     * Dispensary Dashboard
     */
public function index() {
    $this->session->set_userdata('top_menu', 'Medical Dispensary');
    $this->session->set_userdata('sub_menu', 'dispensary/dashboard');

    $data['title'] = 'Medical Dispensary Dashboard';
    
    // Get queue data for pharmacy department
        $data['queue_data'] = $this->Queue_model->getDepartmentQueue('pharmacy');
        $data['queue_config'] = $this->Queue_model->getQueueConfig('pharmacy');
    
    try {
        // Get currency symbol
        $data['currency_symbol'] = $this->customlib->getHospitalCurrencyFormat();
        
        // Get total medicines count
        $data['total_medicines'] = $this->dispensary_model->countMedicines();
        
        // Get low stock medicines
        $data['low_stock_medicines'] = $this->dispensary_model->getLowStockMedicines();
        
        // Get expired medicines
        $data['expired_medicines'] = $this->dispensary_model->getExpiredMedicines();
        
        // Get today's sales - THIS IS THE KEY DATA YOUR DASHBOARD NEEDS
        $data['todays_sales'] = $this->getTodaysSalesData();
        
        // Get pending bills
        $data['pending_bills'] = $this->getPendingBillsData();
        
        // Get recent transactions (last 10)
        $data['recent_transactions'] = $this->getRecentTransactionsData();
        
        // Get low stock count
        $data['low_stock_count'] = count($data['low_stock_medicines']);
        
        // Get expired count  
        $data['expired_count'] = count($data['expired_medicines']);
        
        $this->load->view('layout/header', $data);
        $this->load->view('admin/dispensary/dashboard', $data);
        $this->load->view('layout/footer', $data);
        
    } catch (Exception $e) {
        log_message('error', 'Dispensary dashboard error: ' . $e->getMessage());
        show_error('Failed to load dispensary dashboard: ' . $e->getMessage());
    }
}

/**
 * Get today's sales data from pharmacy_bill_basic table
 */
private function getTodaysSalesData() {
    $today = date('Y-m-d');
    
    // Query pharmacy_bill_basic table for today's sales
    $this->db->select('COUNT(*) as total_bills, COALESCE(SUM(net_amount), 0) as total_sales');
    $this->db->from('pharmacy_bill_basic');
    $this->db->where('DATE(date)', $today);
    $this->db->where('payment_status !=', 'cancelled');
    
    $result = $this->db->get()->row();
    
    if (!$result) {
        return (object)array('total_bills' => 0, 'total_sales' => 0);
    }
    
    return $result;
}

/**
     * Queue management for pharmacy
     */
    public function queue() {
        $this->session->set_userdata('top_menu', 'Medical Dispensary');
        $this->session->set_userdata('sub_menu', 'dispensary/queue');
        
        $data['title'] = 'Pharmacy Queue Management';
        $data['department'] = 'pharmacy';
        $data['queue_data'] = $this->Queue_model->getDepartmentQueue('pharmacy');
        $data['queue_config'] = $this->Queue_model->getQueueConfig('pharmacy');
        
        $this->load->view('layout/header', $data);
        $this->load->view('admin/dispensary/queue_management', $data);
        $this->load->view('layout/footer');
    }

    // =============================================
    // QUEUE OPERATIONS (AJAX)
    // =============================================

    /**
     * AJAX: Call next patient for pharmacy
     */
    public function call_next_patient() {
        header('Content-Type: application/json');
        
        try {
            $pharmacist_id = $this->input->post('pharmacist_id') ?: $this->customlib->getStaffID();
            
            $result = $this->Queue_model->callNextPatient('pharmacy', $pharmacist_id);
            
            if ($result['status'] === 'success') {
                $this->send_queue_notification('pharmacy', 'patient_called', $result['patient']);
            }
            
            echo json_encode($result);
        } catch (Exception $e) {
            echo json_encode(['status' => 'error', 'message' => $e->getMessage()]);
        }
    }

/**
     * Send notification to departments
     */
    private function send_queue_notification($department, $action, $data = []) {
        try {
            $notification = [
                'department' => $department,
                'action' => $action,
                'data' => $data,
                'timestamp' => time(),
                'from_department' => 'pharmacy',
                'sound' => true
            ];
            
            $notification_file = APPPATH . 'cache/queue_notifications_' . $department . '.json';
            file_put_contents($notification_file, json_encode($notification));
            
            return true;
        } catch (Exception $e) {
            log_message('error', 'Pharmacy notification failed: ' . $e->getMessage());
            return false;
        }
    }


/**
 * Get pending bills data
 */
private function getPendingBillsData() {
    $this->db->select('COUNT(*) as count, COALESCE(SUM(net_amount), 0) as total_balance');
    $this->db->from('pharmacy_bill_basic');
    $this->db->where('payment_status', 'pending');
    
    $result = $this->db->get()->row();
    
    if (!$result) {
        return (object)array('count' => 0, 'total_balance' => 0);
    }
    
    return $result;
}

/**
 * Get recent transactions data
 */
private function getRecentTransactionsData() {
    $this->db->select('pbb.id, pbb.date as bill_date, pbb.net_amount, pbb.payment_status as status,
                      p.patient_name, p.patient_unique_id');
    $this->db->from('pharmacy_bill_basic pbb');
    $this->db->join('patients p', 'p.id = pbb.patient_id', 'left');
    $this->db->order_by('pbb.date', 'DESC');
    $this->db->limit(10);
    
    $result = $this->db->get()->result();
    
    // Add bill numbers to the results
    foreach ($result as $key => $transaction) {
        $result[$key]->bill_no = 'MDB' . str_pad($transaction->id, 6, '0', STR_PAD_LEFT);
    }
    
    return $result;
}

/**
 * Save New Medicine with Batches
 */
public function saveMedicine() {
    header('Content-Type: application/json');
    
    try {
        // Get and validate input data
        $medicine_name = trim($this->input->post('medicine_name'));
        $medicine_category_id = $this->input->post('medicine_category_id');
        $unit = trim($this->input->post('unit'));
        $batches = $this->input->post('batches');
        
        // Enhanced validation
        if (empty($medicine_name)) {
            echo json_encode(['status' => 0, 'message' => 'Medicine name is required']);
            return;
        }
        
        if (empty($medicine_category_id) || !is_numeric($medicine_category_id)) {
            echo json_encode(['status' => 0, 'message' => 'Valid medicine category is required']);
            return;
        }
        
        if (empty($unit)) {
            echo json_encode(['status' => 0, 'message' => 'Unit is required']);
            return;
        }
        
        if (empty($batches) || !is_array($batches)) {
            echo json_encode(['status' => 0, 'message' => 'At least one batch is required']);
            return;
        }
        
        // Check if medicine category exists
        $this->db->where('id', $medicine_category_id);
        $category_exists = $this->db->get('medicine_category')->row();
        
        if (!$category_exists) {
            echo json_encode(['status' => 0, 'message' => 'Selected medicine category does not exist']);
            return;
        }
        
        // Check if medicine already exists (case-insensitive)
        $this->db->where('LOWER(medicine_name)', strtolower($medicine_name));
        $this->db->where('medicine_category_id', $medicine_category_id);
        $existing = $this->db->get('medicines')->row();
        
        if ($existing) {
            echo json_encode(['status' => 0, 'message' => 'Medicine with this name already exists in the selected category']);
            return;
        }
        
        // Start database transaction
        $this->db->trans_start();
        
        // Prepare medicine data - MATCHES YOUR EXACT DATABASE STRUCTURE
        $medicine_data = [
            'medicine_name'         => $medicine_name,
            'medicine_category_id'  => (int)$medicine_category_id,
            'medicine_company'      => trim($this->input->post('medicine_company')) ?: null,
            'medicine_composition'  => trim($this->input->post('medicine_composition')) ?: null,
            'unit'                  => $unit,
            'is_active'             => 'yes',
            'created_at'            => date('Y-m-d H:i:s'),
            'updated_at'            => date('Y-m-d H:i:s')
        ];
        
        // Remove any NULL values that might cause issues
        $medicine_data = array_filter($medicine_data, function($value) {
            return $value !== null;
        });
        
        // Debug: Log what we're trying to insert
        log_message('info', 'Attempting to insert medicine: ' . json_encode($medicine_data));
        
        // Insert medicine record
        $insert_result = $this->db->insert('medicines', $medicine_data);
        $medicine_id = $this->db->insert_id();
        
        // Check if insert was successful
        if (!$insert_result || !$medicine_id) {
            $db_error = $this->db->error();
            log_message('error', 'Medicine insert failed: ' . $db_error['message']);
            throw new Exception('Failed to create medicine record: ' . $db_error['message']);
        }
        
        log_message('info', 'Medicine created successfully with ID: ' . $medicine_id);
        
        // Process and insert batches - USING YOUR EXACT TABLE STRUCTURE
        $batch_count = 0;
        $batch_errors = [];
        
        foreach ($batches as $index => $batch) {
            // Validate batch data
            if (empty($batch['batch_no']) || empty($batch['quantity']) || empty($batch['sale_rate'])) {
                $batch_errors[] = "Batch " . ($index + 1) . ": Missing required fields (batch_no, quantity, sale_rate)";
                continue;
            }
            
            if (!is_numeric($batch['quantity']) || $batch['quantity'] <= 0) {
                $batch_errors[] = "Batch " . ($index + 1) . ": Invalid quantity";
                continue;
            }
            
            if (!is_numeric($batch['sale_rate']) || $batch['sale_rate'] <= 0) {
                $batch_errors[] = "Batch " . ($index + 1) . ": Invalid sale rate";
                continue;
            }
            
            // Prepare batch data - MATCHES YOUR EXACT MEDICINE_BATCH_DETAILS TABLE STRUCTURE
            $purchase_rate = isset($batch['purchase_rate']) ? (float)$batch['purchase_rate'] : 0.00;
            $sale_rate = (float)$batch['sale_rate'];
            $quantity = (int)$batch['quantity'];
            $tax_rate = isset($batch['tax']) ? (float)$batch['tax'] : 0.00;
            
            $batch_data = [
                // Core identifiers
                'medicine_id'           => $medicine_id,
                'pharmacy_id'           => null, // Your table allows NULL for this
                'supplier_bill_basic_id' => null, // Your table allows NULL for this
                
                // Batch information
                'batch_no'              => trim($batch['batch_no']),
                'inward_date'           => date('Y-m-d H:i:s'), // DATETIME field
                'expiry'                => !empty($batch['expiry']) ? $batch['expiry'] : '2099-12-31', // DATE field
                
                // Quantities - your table has quantity as VARCHAR(200)
                'quantity'              => (string)$quantity,
                'available_quantity'    => (int)$quantity, // This is INT(11)
                
                // Packing info - your table has these as VARCHAR(100)
                'packing_qty'           => '1', // Default packing quantity
                'purchase_rate_packing' => (string)$purchase_rate, // Note: VARCHAR field, not decimal!
                
                // Financial amounts - your table has these as FLOAT(10,2)
                'mrp'                   => $sale_rate,
                'purchase_price'        => $purchase_rate * $quantity,
                'sale_rate'             => $sale_rate,
                'tax'                   => $tax_rate,
                'batch_amount'          => $sale_rate * $quantity,
                'amount'                => ($sale_rate * $quantity) + (($sale_rate * $quantity) * $tax_rate / 100),
                
                // Timestamp
                'created_at'            => date('Y-m-d H:i:s')
            ];
            
            // Add supplier if provided
            if (isset($batch['supplier']) && !empty($batch['supplier'])) {
                // Note: Your table doesn't have a supplier field, but let's log it
                log_message('info', 'Supplier info provided but table does not have supplier field: ' . $batch['supplier']);
            }
            
            // Debug log for batch data
            log_message('info', 'Attempting to insert batch: ' . json_encode($batch_data));
            
            // Insert batch
            $batch_insert = $this->db->insert('medicine_batch_details', $batch_data);
            
            if ($batch_insert) {
                $batch_count++;
                log_message('info', 'Batch inserted successfully: ' . $batch['batch_no']);
            } else {
                $db_error = $this->db->error();
                $batch_errors[] = "Batch " . ($index + 1) . " (" . $batch['batch_no'] . "): " . $db_error['message'];
                log_message('error', 'Batch insert failed: ' . $db_error['message']);
                log_message('error', 'Failed batch data: ' . json_encode($batch_data));
            }
        }
        
        // Check if at least one batch was inserted successfully
        if ($batch_count === 0) {
            $this->db->trans_rollback();
            $error_message = "No valid batches were created.";
            if (!empty($batch_errors)) {
                $error_message .= " Errors: " . implode(', ', $batch_errors);
            }
            echo json_encode(['status' => 0, 'message' => $error_message]);
            return;
        }
        
        // Commit transaction
        $this->db->trans_complete();
        
        // Check transaction status
        if ($this->db->trans_status() === FALSE) {
            $error_message = 'Transaction failed to complete';
            $db_error = $this->db->error();
            if (!empty($db_error['message'])) {
                $error_message .= ': ' . $db_error['message'];
            }
            echo json_encode(['status' => 0, 'message' => $error_message]);
            return;
        }
        
        // Success response
        $response = [
            'status' => 1,
            'message' => 'Medicine saved successfully',
            'medicine_id' => $medicine_id,
            'batches_added' => $batch_count
        ];
        
        // Add warnings if some batches failed
        if (!empty($batch_errors)) {
            $response['warnings'] = $batch_errors;
            $response['message'] .= ' (Some batches had errors)';
        }
        
        echo json_encode($response);
        
        // Optional: Trigger notification
        $this->triggerMedicineNotification('add_medicine', $medicine_data);
        
    } catch (Exception $e) {
        // Rollback transaction on error
        if ($this->db->trans_status() !== FALSE) {
            $this->db->trans_rollback();
        }
        
        $error_message = 'Error saving medicine: ' . $e->getMessage();
        log_message('error', 'Save medicine error: ' . $error_message);
        
        echo json_encode([
            'status' => 0, 
            'message' => $error_message,
            'debug_info' => [
                'line' => $e->getLine(),
                'file' => basename($e->getFile()),
                'db_error' => $this->db->error()
            ]
        ]);
    }
}

/**
 * Trigger Medicine Notification (Optional)
 */
private function triggerMedicineNotification($event, $medicine_data) {
    try {
        // Only proceed if notification system exists
        if (!$this->db->table_exists('notification_setting')) {
            return;
        }
        
        $this->db->where('event', $event);
        $this->db->where('is_active', 1);
        $notification_setting = $this->db->get('notification_setting')->row();
        
        if ($notification_setting) {
            // Get category name
            $this->db->select('medicine_category');
            $this->db->where('id', $medicine_data['medicine_category_id']);
            $category = $this->db->get('medicine_category')->row();
            
            // Prepare notification data
            $notification_data = [
                'medicine_name' => $medicine_data['medicine_name'],
                'medicine_category' => $category ? $category->medicine_category : '',
                'medicine_company' => $medicine_data['medicine_company'] ?: '',
                'medicine_composition' => $medicine_data['medicine_composition'] ?: '',
                'medicine_group' => $medicine_data['medicine_group'] ?: '',
                'unit' => $medicine_data['unit'],
                'unit_packing' => $medicine_data['unit_packing'] ?: ''
            ];
            
            // Send notification if system_notification exists
            if (class_exists('System_notification')) {
                $this->load->library('system_notification');
                $this->system_notification->send_system_notification($event, $notification_data);
            }
        }
    } catch (Exception $e) {
        log_message('error', 'Medicine notification error: ' . $e->getMessage());
        // Don't throw error, just log it
    }
}

/**
 * Get Medicine Categories for Dropdown
 */
public function getMedicineCategoriesForDropdown() {
    header('Content-Type: application/json');
    
    try {
        $this->db->select('id, medicine_category');
        $this->db->from('medicine_category');
        $this->db->order_by('medicine_category', 'ASC');
        $categories = $this->db->get()->result();
        
        echo json_encode(['status' => 1, 'categories' => $categories]);
        
    } catch (Exception $e) {
        log_message('error', 'Get categories error: ' . $e->getMessage());
        echo json_encode(['status' => 0, 'message' => 'Error loading categories']);
    }
}

/**
 * Update Medicine Information
 */
public function updateMedicine() {
    header('Content-Type: application/json');
    
    try {
        $medicine_id = $this->input->post('medicine_id');
        $medicine_name = $this->input->post('medicine_name');
        $medicine_category_id = $this->input->post('medicine_category_id');
        $unit = $this->input->post('unit');
        
        if (empty($medicine_id) || empty($medicine_name) || empty($medicine_category_id) || empty($unit)) {
            echo json_encode(['status' => 0, 'message' => 'Required fields are missing']);
            return;
        }
        
        // Check if medicine exists
        $this->db->where('id', $medicine_id);
        $existing_medicine = $this->db->get('medicines')->row();
        
        if (!$existing_medicine) {
            echo json_encode(['status' => 0, 'message' => 'Medicine not found']);
            return;
        }
        
        // Check for duplicate name in same category (excluding current medicine)
        $this->db->where('medicine_name', $medicine_name);
        $this->db->where('medicine_category_id', $medicine_category_id);
        $this->db->where('id !=', $medicine_id);
        $duplicate = $this->db->get('medicines')->row();
        
        if ($duplicate) {
            echo json_encode(['status' => 0, 'message' => 'Medicine with this name already exists in the selected category']);
            return;
        }
        
        $update_data = [
            'medicine_name' => $medicine_name,
            'medicine_category_id' => $medicine_category_id,
            'medicine_company' => $this->input->post('medicine_company'),
            'medicine_composition' => $this->input->post('medicine_composition'),
            'medicine_group' => $this->input->post('medicine_group'),
            'unit' => $unit,
            'unit_packing' => $this->input->post('unit_packing'),
            'min_level' => $this->input->post('min_level') ?: 10,
            'reorder_level' => $this->input->post('min_level') ?: 10,
            'updated_at' => date('Y-m-d H:i:s')
        ];
        
        $this->db->where('id', $medicine_id);
        $this->db->update('medicines', $update_data);
        
        if ($this->db->affected_rows() >= 0) {
            echo json_encode(['status' => 1, 'message' => 'Medicine updated successfully']);
        } else {
            echo json_encode(['status' => 0, 'message' => 'Failed to update medicine']);
        }
        
    } catch (Exception $e) {
        log_message('error', 'Update medicine error: ' . $e->getMessage());
        echo json_encode(['status' => 0, 'message' => 'Error updating medicine: ' . $e->getMessage()]);
    }
}

/**
 * Delete Medicine (with validation)
 */
public function deleteMedicine() {
    header('Content-Type: application/json');
    
    try {
        $medicine_id = $this->input->post('medicine_id');
        
        if (empty($medicine_id)) {
            echo json_encode(['status' => 0, 'message' => 'Medicine ID is required']);
            return;
        }
        
        // Check if medicine exists
        $this->db->where('id', $medicine_id);
        $medicine = $this->db->get('medicines')->row();
        
        if (!$medicine) {
            echo json_encode(['status' => 0, 'message' => 'Medicine not found']);
            return;
        }
        
        // Check if medicine has any stock
        $this->db->where('medicine_id', $medicine_id);
        $this->db->where('available_quantity >', 0);
        $stock_count = $this->db->count_all_results('medicine_batch_details');
        
        if ($stock_count > 0) {
            echo json_encode(['status' => 0, 'message' => 'Cannot delete medicine with available stock. Please clear all stock first.']);
            return;
        }
        
        // Check if medicine is used in any prescriptions
        if ($this->db->table_exists('opd_prescription_details')) {
            $this->db->where('pharmacy_id', $medicine_id);
            $prescription_count = $this->db->count_all_results('opd_prescription_details');
            
            if ($prescription_count > 0) {
                echo json_encode(['status' => 0, 'message' => 'Cannot delete medicine that has been prescribed to patients.']);
                return;
            }
        }
        
        $this->db->trans_start();
        
        // Delete related batch details first
        $this->db->where('medicine_id', $medicine_id);
        $this->db->delete('medicine_batch_details');
        
        // Delete stock movements if table exists
        if ($this->db->table_exists('stock_movements')) {
            $this->db->where('medicine_id', $medicine_id);
            $this->db->delete('stock_movements');
        }
        
        // Finally delete the medicine
        $this->db->where('id', $medicine_id);
        $this->db->delete('medicines');
        
        $this->db->trans_complete();
        
        if ($this->db->trans_status() === FALSE) {
            echo json_encode(['status' => 0, 'message' => 'Failed to delete medicine']);
            return;
        }
        
        echo json_encode(['status' => 1, 'message' => 'Medicine deleted successfully']);
        
    } catch (Exception $e) {
        $this->db->trans_rollback();
        log_message('error', 'Delete medicine error: ' . $e->getMessage());
        echo json_encode(['status' => 0, 'message' => 'Error deleting medicine: ' . $e->getMessage()]);
    }
}

/**
 * Get Medicine Details for Edit
 */
public function getMedicineForEdit() {
    header('Content-Type: application/json');
    
    try {
        $medicine_id = $this->input->post('medicine_id');
        
        if (empty($medicine_id)) {
            echo json_encode(['status' => 0, 'message' => 'Medicine ID is required']);
            return;
        }
        
        // Get medicine details
        $this->db->select('m.*, mc.medicine_category');
        $this->db->from('medicines m');
        $this->db->join('medicine_category mc', 'm.medicine_category_id = mc.id', 'left');
        $this->db->where('m.id', $medicine_id);
        $medicine = $this->db->get()->row();
        
        if (!$medicine) {
            echo json_encode(['status' => 0, 'message' => 'Medicine not found']);
            return;
        }
        
        // Get batch details
        $this->db->select('*');
        $this->db->from('medicine_batch_details');
        $this->db->where('medicine_id', $medicine_id);
        $this->db->order_by('expiry', 'ASC');
        $batches = $this->db->get()->result();
        
        $medicine->batches = $batches;
        
        echo json_encode(['status' => 1, 'medicine' => $medicine]);
        
    } catch (Exception $e) {
        log_message('error', 'Get medicine for edit error: ' . $e->getMessage());
        echo json_encode(['status' => 0, 'message' => 'Error loading medicine details']);
    }
}

/**
 * Bulk Import Medicines from CSV/Excel
 */
public function importMedicines() {
    header('Content-Type: application/json');
    
    try {
        if (!isset($_FILES['import_file']) || $_FILES['import_file']['error'] !== UPLOAD_ERR_OK) {
            echo json_encode(['status' => 0, 'message' => 'No file uploaded or upload error']);
            return;
        }
        
        $file = $_FILES['import_file'];
        $file_extension = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
        
        if (!in_array($file_extension, ['csv', 'xlsx', 'xls'])) {
            echo json_encode(['status' => 0, 'message' => 'Only CSV and Excel files are allowed']);
            return;
        }
        
        // Load PHPSpreadsheet or similar library for Excel files
        // For CSV, use built-in PHP functions
        $imported_count = 0;
        $error_count = 0;
        $errors = [];
        
        if ($file_extension === 'csv') {
            $imported_count = $this->importFromCSV($file['tmp_name'], $errors);
        } else {
            // Handle Excel files (requires PHPSpreadsheet library)
            echo json_encode(['status' => 0, 'message' => 'Excel import not yet implemented. Please use CSV format.']);
            return;
        }
        
        echo json_encode([
            'status' => 1, 
            'message' => "Import completed. $imported_count medicines imported successfully.",
            'imported_count' => $imported_count,
            'error_count' => count($errors),
            'errors' => $errors
        ]);
        
    } catch (Exception $e) {
        log_message('error', 'Import medicines error: ' . $e->getMessage());
        echo json_encode(['status' => 0, 'message' => 'Error importing medicines: ' . $e->getMessage()]);
    }
}

/**
 * Import medicines from CSV file
 */
private function importFromCSV($file_path, &$errors) {
    $imported_count = 0;
    
    if (($handle = fopen($file_path, "r")) !== FALSE) {
        $header = fgetcsv($handle, 1000, ","); // Read header row
        
        // Expected columns: medicine_name, category, company, composition, unit, batch_no, quantity, sale_rate, etc.
        $row_number = 1;
        
        while (($data = fgetcsv($handle, 1000, ",")) !== FALSE) {
            $row_number++;
            
            try {
                // Map CSV columns to database fields
                $medicine_data = [
                    'medicine_name' => isset($data[0]) ? trim($data[0]) : '',
                    'category_name' => isset($data[1]) ? trim($data[1]) : '',
                    'medicine_company' => isset($data[2]) ? trim($data[2]) : '',
                    'medicine_composition' => isset($data[3]) ? trim($data[3]) : '',
                    'unit' => isset($data[4]) ? trim($data[4]) : '',
                    'batch_no' => isset($data[5]) ? trim($data[5]) : '',
                    'quantity' => isset($data[6]) ? intval($data[6]) : 0,
                    'sale_rate' => isset($data[7]) ? floatval($data[7]) : 0
                ];
                
                // Validate required fields
                if (empty($medicine_data['medicine_name']) || empty($medicine_data['unit']) || 
                    empty($medicine_data['batch_no']) || $medicine_data['quantity'] <= 0 || 
                    $medicine_data['sale_rate'] <= 0) {
                    $errors[] = "Row $row_number: Missing required fields";
                    continue;
                }
                
                // Get or create category
                $category_id = $this->getOrCreateCategory($medicine_data['category_name']);
                
                // Check if medicine exists
                $this->db->where('medicine_name', $medicine_data['medicine_name']);
                $this->db->where('medicine_category_id', $category_id);
                $existing_medicine = $this->db->get('medicines')->row();
                
                if (!$existing_medicine) {
                    // Create new medicine
                    $new_medicine = [
                        'medicine_name' => $medicine_data['medicine_name'],
                        'medicine_category_id' => $category_id,
                        'medicine_company' => $medicine_data['medicine_company'],
                        'medicine_composition' => $medicine_data['medicine_composition'],
                        'unit' => $medicine_data['unit'],
                        'min_level' => 10,
                        'reorder_level' => 10,
                        'is_active' => 'yes',
                        'created_at' => date('Y-m-d H:i:s')
                    ];
                    
                    $this->db->insert('medicines', $new_medicine);
                    $medicine_id = $this->db->insert_id();
                } else {
                    $medicine_id = $existing_medicine->id;
                }
                
                // Add batch
                $batch_data = [
                    'medicine_id' => $medicine_id,
                    'batch_no' => $medicine_data['batch_no'],
                    'quantity' => $medicine_data['quantity'],
                    'available_quantity' => $medicine_data['quantity'],
                    'sale_rate' => $medicine_data['sale_rate'],
                    'mrp' => $medicine_data['sale_rate'],
                    'inward_date' => date('Y-m-d'),
                    'created_at' => date('Y-m-d H:i:s')
                ];
                
                $this->db->insert('medicine_batch_details', $batch_data);
                $imported_count++;
                
            } catch (Exception $e) {
                $errors[] = "Row $row_number: " . $e->getMessage();
            }
        }
        
        fclose($handle);
    }
    
    return $imported_count;
}

/**
 * Get or create medicine category
 */
private function getOrCreateCategory($category_name) {
    if (empty($category_name)) {
        $category_name = 'General';
    }
    
    // Check if category exists
    $this->db->where('medicine_category', $category_name);
    $category = $this->db->get('medicine_category')->row();
    
    if ($category) {
        return $category->id;
    }
    
    // Create new category
    $new_category = [
        'medicine_category' => $category_name,
        'description' => 'Auto-created during import',
        'created_at' => date('Y-m-d H:i:s')
    ];
    
    $this->db->insert('medicine_category', $new_category);
    return $this->db->insert_id();
}

/**
 * Get Stock Summary Report
 */
public function getStockSummaryReport() {
    header('Content-Type: application/json');
    
    try {
        $date_from = $this->input->get('date_from');
        $date_to = $this->input->get('date_to');
        
        // Get medicine stock summary
        $this->db->select('
            m.id,
            m.medicine_name,
            m.medicine_company,
            mc.medicine_category,
            m.unit,
            m.min_level,
            COALESCE(SUM(mbd.available_quantity), 0) as total_stock,
            COALESCE(SUM(mbd.quantity), 0) as total_purchased,
            MIN(mbd.expiry) as nearest_expiry,
            COUNT(mbd.id) as batch_count
        ');
        $this->db->from('medicines m');
        $this->db->join('medicine_category mc', 'm.medicine_category_id = mc.id', 'left');
        $this->db->join('medicine_batch_details mbd', 'm.id = mbd.medicine_id', 'left');
        
        if ($date_from && $date_to) {
            $this->db->where('mbd.inward_date >=', $date_from);
            $this->db->where('mbd.inward_date <=', $date_to);
        }
        
        $this->db->where('m.is_active', 'yes');
        $this->db->group_by('m.id');
        $this->db->order_by('m.medicine_name', 'ASC');
        
        $medicines = $this->db->get()->result();
        
        // Calculate summary statistics
        $summary = [
            'total_medicines' => count($medicines),
            'in_stock' => 0,
            'low_stock' => 0,
            'out_of_stock' => 0,
            'expired' => 0,
            'expiring_soon' => 0
        ];
        
        $today = new DateTime();
        
        foreach ($medicines as $medicine) {
            if ($medicine->total_stock <= 0) {
                $summary['out_of_stock']++;
            } elseif ($medicine->total_stock <= $medicine->min_level) {
                $summary['low_stock']++;
            } else {
                $summary['in_stock']++;
            }
            
            if ($medicine->nearest_expiry) {
                $expiry_date = new DateTime($medicine->nearest_expiry);
                $diff = $today->diff($expiry_date);
                
                if ($expiry_date < $today) {
                    $summary['expired']++;
                } elseif ($diff->days <= 30) {
                    $summary['expiring_soon']++;
                }
            }
        }
        
        echo json_encode([
            'status' => 1, 
            'medicines' => $medicines,
            'summary' => $summary
        ]);
        
    } catch (Exception $e) {
        log_message('error', 'Get stock summary report error: ' . $e->getMessage());
        echo json_encode(['status' => 0, 'message' => 'Error generating report']);
    }
}

/**
 * Generate Stock Alert Report
 */
public function getStockAlertReport() {
    header('Content-Type: application/json');
    
    try {
        $alert_type = $this->input->get('alert_type'); // low_stock, expired, expiring_soon
        
        $alerts = [];
        
        if ($alert_type === 'low_stock' || empty($alert_type)) {
            // Get low stock medicines
            $this->db->select('
                m.id,
                m.medicine_name,
                m.medicine_company,
                mc.medicine_category,
                m.unit,
                m.min_level,
                COALESCE(SUM(mbd.available_quantity), 0) as total_stock
            ');
            $this->db->from('medicines m');
            $this->db->join('medicine_category mc', 'm.medicine_category_id = mc.id', 'left');
            $this->db->join('medicine_batch_details mbd', 'm.id = mbd.medicine_id', 'left');
            $this->db->where('m.is_active', 'yes');
            $this->db->group_by('m.id');
            $this->db->having('total_stock <= m.min_level');
            $this->db->having('total_stock > 0');
            $this->db->order_by('total_stock', 'ASC');
            
            $low_stock = $this->db->get()->result();
            
            foreach ($low_stock as $item) {
                $alerts[] = [
                    'type' => 'Low Stock',
                    'priority' => 'medium',
                    'medicine_name' => $item->medicine_name,
                    'company' => $item->medicine_company,
                    'category' => $item->medicine_category,
                    'current_stock' => $item->total_stock,
                    'min_level' => $item->min_level,
                    'message' => "Stock is below minimum level ({$item->min_level})"
                ];
            }
        }
        
        if ($alert_type === 'expired' || empty($alert_type)) {
            // Get expired medicines
            $this->db->select('
                m.medicine_name,
                m.medicine_company,
                mc.medicine_category,
                mbd.batch_no,
                mbd.available_quantity,
                mbd.expiry
            ');
            $this->db->from('medicine_batch_details mbd');
            $this->db->join('medicines m', 'mbd.medicine_id = m.id');
            $this->db->join('medicine_category mc', 'm.medicine_category_id = mc.id', 'left');
            $this->db->where('mbd.expiry <', date('Y-m-d'));
            $this->db->where('mbd.available_quantity >', 0);
            $this->db->where('m.is_active', 'yes');
            $this->db->order_by('mbd.expiry', 'ASC');
            
            $expired = $this->db->get()->result();
            
            foreach ($expired as $item) {
                $alerts[] = [
                    'type' => 'Expired',
                    'priority' => 'high',
                    'medicine_name' => $item->medicine_name,
                    'company' => $item->medicine_company,
                    'category' => $item->medicine_category,
                    'batch_no' => $item->batch_no,
                    'quantity' => $item->available_quantity,
                    'expiry_date' => $item->expiry,
                    'message' => "Batch expired on " . date('d/m/Y', strtotime($item->expiry))
                ];
            }
        }
        
        if ($alert_type === 'expiring_soon' || empty($alert_type)) {
            // Get medicines expiring in next 30 days
            $this->db->select('
                m.medicine_name,
                m.medicine_company,
                mc.medicine_category,
                mbd.batch_no,
                mbd.available_quantity,
                mbd.expiry
            ');
            $this->db->from('medicine_batch_details mbd');
            $this->db->join('medicines m', 'mbd.medicine_id = m.id');
            $this->db->join('medicine_category mc', 'm.medicine_category_id = mc.id', 'left');
            $this->db->where('mbd.expiry >=', date('Y-m-d'));
            $this->db->where('mbd.expiry <=', date('Y-m-d', strtotime('+30 days')));
            $this->db->where('mbd.available_quantity >', 0);
            $this->db->where('m.is_active', 'yes');
            $this->db->order_by('mbd.expiry', 'ASC');
            
            $expiring_soon = $this->db->get()->result();
            
            foreach ($expiring_soon as $item) {
                $days_to_expiry = ceil((strtotime($item->expiry) - time()) / (60 * 60 * 24));
                $alerts[] = [
                    'type' => 'Expiring Soon',
                    'priority' => 'medium',
                    'medicine_name' => $item->medicine_name,
                    'company' => $item->medicine_company,
                    'category' => $item->medicine_category,
                    'batch_no' => $item->batch_no,
                    'quantity' => $item->available_quantity,
                    'expiry_date' => $item->expiry,
                    'days_to_expiry' => $days_to_expiry,
                    'message' => "Expires in $days_to_expiry days"
                ];
            }
        }
        
        // Sort alerts by priority and date
        usort($alerts, function($a, $b) {
            $priority_order = ['high' => 1, 'medium' => 2, 'low' => 3];
            return $priority_order[$a['priority']] - $priority_order[$b['priority']];
        });
        
        echo json_encode([
            'status' => 1, 
            'alerts' => $alerts,
            'total_alerts' => count($alerts)
        ]);
        
    } catch (Exception $e) {
        log_message('error', 'Get stock alert report error: ' . $e->getMessage());
        echo json_encode(['status' => 0, 'message' => 'Error generating alerts']);
    }
}
// =============================================
    // AJAX: Statistics and Status
    // =============================================

    /**
     * AJAX: Get pharmacy queue statistics
     */
    public function get_queue_stats() {
        header('Content-Type: application/json');
        
        try {
            $stats = $this->Queue_model->getQueueStatistics('pharmacy');
            echo json_encode(['status' => 'success', 'stats' => $stats]);
        } catch (Exception $e) {
            echo json_encode(['status' => 'error', 'message' => $e->getMessage()]);
        }
    }

    /**
     * AJAX: Get current pharmacy queue
     */
    public function get_current_queue() {
        header('Content-Type: application/json');
        
        try {
            $queue_data = $this->Queue_model->getDepartmentQueue('pharmacy');
            echo json_encode(['status' => 'success', 'queue' => $queue_data]);
        } catch (Exception $e) {
            echo json_encode(['status' => 'error', 'message' => $e->getMessage()]);
        }
    }

/*public function getPatientPrescriptions() {
    $patient_id = $this->input->post('patient_id');
    
    // Query to get prescriptions for this patient
    $query = "
        SELECT 
            pb.*,
            s.name as doctor_name,
            s.surname as doctor_surname,
            CONCAT(s.name, ' ', s.surname) as doctor_name,
            CASE 
                WHEN pb.ipd_id IS NOT NULL THEN 'IPD'
                ELSE 'OPD'
            END as visit_type
        FROM ipd_prescription_basic pb
        LEFT JOIN staff s ON pb.doctor = s.id
        LEFT JOIN visit_details vd ON pb.visit_details_id = vd.id
        LEFT JOIN opd_details od ON vd.opd_details_id = od.id
        WHERE od.patient_id = ? OR pb.patient_id = ?
        ORDER BY pb.date DESC
        LIMIT 10
    ";
    
    $prescriptions = $this->db->query($query, [$patient_id, $patient_id])->result();
    
    // Get medicines for each prescription
    foreach ($prescriptions as &$prescription) {
        $medicines_query = "
            SELECT 
                pd.*,
                p.medicine_name,
                mc.medicine_category
            FROM ipd_prescription_detail pd
            LEFT JOIN pharmacy p ON pd.medicine = p.id
            LEFT JOIN medicine_category mc ON p.medicine_category_id = mc.id
            WHERE pd.ipd_prescription_basic_id = ?
        ";
        
        $prescription->medicines = $this->db->query($medicines_query, [$prescription->id])->result();
    }
    
    echo json_encode([
        'status' => 'success',
        'prescriptions' => $prescriptions
    ]);
}*/

    /**
 * AJAX: Get queue notifications for department
 * This method checks for actual waiting patients and returns them as notifications
 */
public function get_queue_notifications() {
    header('Content-Type: application/json');
    
    try {
        $department = $this->input->get('department', TRUE);
        $last_check = $this->input->get('last_check', TRUE);
        
        // Default to checking all departments if none specified
        if (empty($department) || $department === 'all') {
            $department = null;
        }
        
        // If no last_check provided, use current time minus 30 seconds
        if (empty($last_check)) {
            $last_check = time() - 30;
        }
        
        // Convert timestamp to datetime for database query
        $last_check_datetime = date('Y-m-d H:i:s', $last_check);
        
        // Query for new patients added to queue since last check
        $this->db->select('pq.*, p.patient_name, p.mobileno, p.age, p.gender, p.patient_unique_id');
        $this->db->from('patient_queue pq');
        $this->db->join('patients p', 'p.id = pq.patient_id', 'left');
        $this->db->where('pq.status', 'waiting');
        $this->db->where('pq.created_at >', $last_check_datetime);
        
        // Filter by department if specified
        if ($department) {
            $this->db->where('pq.department', $department);
        }
        
        $this->db->order_by('pq.created_at', 'DESC');
        $this->db->limit(10);
        
        $query = $this->db->get();
        
        if (!$query) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Database query failed',
                'notifications' => [],
                'timestamp' => time()
            ]);
            return;
        }
        
        $new_patients = $query->result_array();
        $notifications = [];
        
        // Convert database results to notification format
        foreach ($new_patients as $patient) {
            $notifications[] = [
                'patient_id' => $patient['patient_id'],
                'patient_name' => $patient['patient_name'] ?: 'Unknown Patient',
                'queue_number' => $patient['queue_number'],
                'department' => $patient['department'],
                'priority' => $patient['priority'],
                'workflow_type' => $patient['queue_type'] ?: 'general',
                'time' => date('H:i:s', strtotime($patient['created_at'])),
                'created_at' => $patient['created_at'],
                'phone' => $patient['mobileno'] ?: '',
                'age' => $patient['age'] ?: '',
                'gender' => $patient['gender'] ?: '',
                'patient_unique_id' => $patient['patient_unique_id'] ?: ''
            ];
        }
        
        // Also check for priority patients (urgent/high priority)
        if (empty($new_patients)) {
            // Check for any urgent patients that might have been missed
            $this->db->select('pq.*, p.patient_name, p.mobileno, p.age, p.gender, p.patient_unique_id');
            $this->db->from('patient_queue pq');
            $this->db->join('patients p', 'p.id = pq.patient_id', 'left');
            $this->db->where('pq.status', 'waiting');
            $this->db->where_in('pq.priority', ['urgent', 'high']);
            
            if ($department) {
                $this->db->where('pq.department', $department);
            }
            
            $this->db->order_by('pq.created_at', 'DESC');
            $this->db->limit(5);
            
            $urgent_query = $this->db->get();
            $urgent_patients = $urgent_query->result_array();
            
            foreach ($urgent_patients as $patient) {
                $notifications[] = [
                    'patient_id' => $patient['patient_id'],
                    'patient_name' => $patient['patient_name'] ?: 'Unknown Patient',
                    'queue_number' => $patient['queue_number'],
                    'department' => $patient['department'],
                    'priority' => $patient['priority'],
                    'workflow_type' => $patient['queue_type'] ?: 'urgent',
                    'time' => date('H:i:s', strtotime($patient['created_at'])),
                    'created_at' => $patient['created_at'],
                    'phone' => $patient['mobileno'] ?: '',
                    'age' => $patient['age'] ?: '',
                    'gender' => $patient['gender'] ?: '',
                    'patient_unique_id' => $patient['patient_unique_id'] ?: '',
                    'is_urgent' => true
                ];
            }
        }
        
        echo json_encode([
            'status' => 'success',
            'notifications' => $notifications,
            'count' => count($notifications),
            'timestamp' => time(),
            'last_check' => $last_check,
            'department' => $department ?: 'all',
            'debug' => [
                'query_time' => $last_check_datetime,
                'current_time' => date('Y-m-d H:i:s'),
                'department_filter' => $department
            ]
        ]);
        
    } catch (Exception $e) {
        echo json_encode([
            'status' => 'error',
            'message' => 'Exception: ' . $e->getMessage(),
            'notifications' => [],
            'timestamp' => time()
        ]);
    }
}

public function debug_queue_status() {
    header('Content-Type: application/json');
    
    try {
        // Get all waiting patients
        $this->db->select('pq.*, p.patient_name, p.mobileno');
        $this->db->from('patient_queue pq');
        $this->db->join('patients p', 'p.id = pq.patient_id', 'left');
        $this->db->where('pq.status', 'waiting');
        $this->db->order_by('pq.created_at', 'DESC');
        $this->db->limit(20);
        
        $query = $this->db->get();
        $waiting_patients = $query->result_array();
        
        // Get queue counts by department
        $this->db->select('department, COUNT(*) as count');
        $this->db->from('patient_queue');
        $this->db->where('status', 'waiting');
        $this->db->group_by('department');
        $dept_counts = $this->db->get()->result_array();
        
        echo json_encode([
            'status' => 'success',
            'waiting_patients' => $waiting_patients,
            'department_counts' => $dept_counts,
            'total_waiting' => count($waiting_patients),
            'timestamp' => time(),
            'current_time' => date('Y-m-d H:i:s')
        ]);
        
    } catch (Exception $e) {
        echo json_encode([
            'status' => 'error',
            'message' => $e->getMessage()
        ]);
    }
}

public function searchMedicineByName() {
    header('Content-Type: application/json');
    
    try {
        $medicine_name = trim($this->input->post('medicine_name'));
        $search_term = trim($this->input->post('search_term'));
        
        // Use medicine_name if provided, otherwise use search_term
        $search = $medicine_name ?: $search_term;
        
        if (empty($search)) {
            echo json_encode([
                'status' => 0,
                'message' => 'Medicine name is required',
                'debug' => 'No search term provided'
            ]);
            return;
        }
        
        log_message('info', 'Searching for medicine: ' . $search);
        
        // Check what medicine tables are available and find the medicine
        $medicine_found = $this->findMedicineInAvailableTables($search);
        
        if ($medicine_found) {
            echo json_encode([
                'status' => 1,
                'medicine' => $medicine_found,
                'message' => 'Medicine found',
                'debug' => 'Found in table: ' . ($medicine_found['table_source'] ?? 'unknown')
            ]);
        } else {
            // Provide detailed debug info about what tables were checked
            $debug_info = $this->debugMedicineSearch($search);
            
            echo json_encode([
                'status' => 0,
                'message' => 'Medicine not found in stock',
                'debug' => $debug_info,
                'search_term' => $search
            ]);
        }
        
    } catch (Exception $e) {
        log_message('error', 'Search medicine by name error: ' . $e->getMessage());
        echo json_encode([
            'status' => 0,
            'message' => 'Error searching medicine: ' . $e->getMessage(),
            'debug' => 'Exception occurred'
        ]);
    }
}

/**
 * Find medicine in available tables with comprehensive search
 */
private function findMedicineInAvailableTables($search) {
    // Define possible table combinations
    $table_combinations = [
        ['medicine' => 'medicines', 'batch' => 'medicine_batch_details'],
        ['medicine' => 'pharmacy', 'batch' => 'medicine_batch_details'],
        ['medicine' => 'medicines', 'batch' => 'pharmacy_stock'],
        ['medicine' => 'pharmacy', 'batch' => 'pharmacy_stock'],
        ['medicine' => 'medicines', 'batch' => null], // No batch table
        ['medicine' => 'pharmacy', 'batch' => null]   // No batch table
    ];
    
    foreach ($table_combinations as $combo) {
        $medicine_table = $combo['medicine'];
        $batch_table = $combo['batch'];
        
        // Check if tables exist
        if (!$this->db->table_exists($medicine_table)) {
            continue;
        }
        
        if ($batch_table && !$this->db->table_exists($batch_table)) {
            continue;
        }
        
        // Try to find medicine in this combination
        $medicine = $this->searchInTableCombination($search, $medicine_table, $batch_table);
        
        if ($medicine) {
            $medicine['table_source'] = $medicine_table . ($batch_table ? " + {$batch_table}" : '');
            return $medicine;
        }
    }
    
    return null;
}

/**
 * Search in specific table combination
 */
private function searchInTableCombination($search, $medicine_table, $batch_table) {
    try {
        // Build base query
        $this->db->select("
            m.id,
            m.medicine_name,
            m.medicine_company,
            m.unit
        ");
        
        // Add category if available
        if ($this->db->field_exists('medicine_category_id', $medicine_table)) {
            $this->db->select('mc.medicine_category');
            $this->db->join('medicine_category mc', 'm.medicine_category_id = mc.id', 'left');
        }
        
        // Add stock information if batch table available
        if ($batch_table) {
            $this->db->select('COALESCE(SUM(b.available_quantity), 0) as total_stock');
            $this->db->join("{$batch_table} b", 'm.id = b.medicine_id', 'left');
        } else {
            $this->db->select('1 as total_stock'); // Default stock
        }
        
        $this->db->from("{$medicine_table} m");
        
        // Search conditions - try exact match first
        $this->db->where('m.medicine_name', $search);
        
        // Add stock conditions if batch table exists
        if ($batch_table) {
            $this->db->where('b.available_quantity >', 0);
            $this->db->where('b.expiry >', date('Y-m-d'));
            $this->db->group_by('m.id');
        }
        
        // Add active condition if field exists
        if ($this->db->field_exists('is_active', $medicine_table)) {
            $this->db->where('m.is_active', 'yes');
        }
        
        $this->db->limit(1);
        
        $medicine = $this->db->get()->row_array();
        
        if ($medicine) {
            log_message('info', 'Found exact match in ' . $medicine_table . ': ' . json_encode($medicine));
            return $medicine;
        }
        
        // If no exact match, try partial match
        return $this->searchPartialInTableCombination($search, $medicine_table, $batch_table);
        
    } catch (Exception $e) {
        log_message('error', 'Error searching in ' . $medicine_table . ': ' . $e->getMessage());
        return null;
    }
}

/**
 * Search with partial matching
 */
private function searchPartialInTableCombination($search, $medicine_table, $batch_table) {
    try {
        // Build base query for partial search
        $this->db->select("
            m.id,
            m.medicine_name,
            m.medicine_company,
            m.unit
        ");
        
        // Add category if available
        if ($this->db->field_exists('medicine_category_id', $medicine_table)) {
            $this->db->select('mc.medicine_category');
            $this->db->join('medicine_category mc', 'm.medicine_category_id = mc.id', 'left');
        }
        
        // Add stock information if batch table available
        if ($batch_table) {
            $this->db->select('COALESCE(SUM(b.available_quantity), 0) as total_stock');
            $this->db->join("{$batch_table} b", 'm.id = b.medicine_id', 'left');
        } else {
            $this->db->select('1 as total_stock'); // Default stock
        }
        
        $this->db->from("{$medicine_table} m");
        
        // Partial search conditions
        $this->db->group_start();
        $this->db->like('m.medicine_name', $search);
        $this->db->or_like('m.medicine_company', $search);
        
        // Try searching by composition if field exists
        if ($this->db->field_exists('medicine_composition', $medicine_table)) {
            $this->db->or_like('m.medicine_composition', $search);
        }
        
        $this->db->group_end();
        
        // Add stock conditions if batch table exists
        if ($batch_table) {
            $this->db->where('b.available_quantity >', 0);
            $this->db->where('b.expiry >', date('Y-m-d'));
            $this->db->group_by('m.id');
        }
        
        // Add active condition if field exists
        if ($this->db->field_exists('is_active', $medicine_table)) {
            $this->db->where('m.is_active', 'yes');
        }
        
        $this->db->order_by('m.medicine_name', 'ASC');
        $this->db->limit(5);
        
        $medicines = $this->db->get()->result_array();
        
        if (!empty($medicines)) {
            log_message('info', 'Found partial matches in ' . $medicine_table . ': ' . count($medicines));
            return $medicines[0]; // Return first match
        }
        
        return null;
        
    } catch (Exception $e) {
        log_message('error', 'Error in partial search for ' . $medicine_table . ': ' . $e->getMessage());
        return null;
    }
}

/**
 * Debug medicine search to help troubleshoot
 */
private function debugMedicineSearch($search) {
    $debug_info = [
        'search_term' => $search,
        'tables_checked' => [],
        'total_medicines' => 0,
        'sample_medicines' => []
    ];
    
    // Check medicine tables
    $medicine_tables = ['medicines', 'pharmacy'];
    $batch_tables = ['medicine_batch_details', 'pharmacy_stock'];
    
    foreach ($medicine_tables as $table) {
        if ($this->db->table_exists($table)) {
            $debug_info['tables_checked'][$table] = 'exists';
            
            // Get total count
            $count = $this->db->count_all($table);
            $debug_info['total_medicines'] += $count;
            
            // Get sample medicines that might match
            $this->db->select('medicine_name');
            $this->db->from($table);
            $this->db->like('medicine_name', $search);
            $this->db->limit(5);
            $sample = $this->db->get()->result_array();
            
            if (!empty($sample)) {
                $debug_info['sample_medicines'][$table] = $sample;
            }
        } else {
            $debug_info['tables_checked'][$table] = 'missing';
        }
    }
    
    foreach ($batch_tables as $table) {
        if ($this->db->table_exists($table)) {
            $debug_info['tables_checked'][$table] = 'exists';
        } else {
            $debug_info['tables_checked'][$table] = 'missing';
        }
    }
    
    return $debug_info;
}

/**
 * Mark prescription medicine as dispensed
 */
public function markMedicineDispensed() {
    header('Content-Type: application/json');
    
    try {
        $prescription_id = $this->input->post('prescription_id');
        $medicine_id = $this->input->post('medicine_id');
        $dispensed_quantity = $this->input->post('dispensed_quantity');
        $bill_id = $this->input->post('bill_id'); // From pharmacy billing
        
        if (!$prescription_id || !$medicine_id || !$dispensed_quantity) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Missing required fields'
            ]);
            return;
        }
        
        $this->db->trans_start();
        
        // Get current medicine details
        $this->db->select('*');
        $this->db->from('opd_prescription_details');
        $this->db->where('id', $medicine_id);
        $this->db->where('prescription_id', $prescription_id);
        $medicine = $this->db->get()->row_array();
        
        if (!$medicine) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Medicine not found in prescription'
            ]);
            return;
        }
        
        // Calculate new dispensed quantity
        $new_dispensed = $medicine['quantity_dispensed'] + $dispensed_quantity;
        $is_fully_dispensed = $new_dispensed >= $medicine['quantity_prescribed'];
        
        // Update medicine dispensing record
        $update_data = [
            'quantity_dispensed' => $new_dispensed,
            'dispensed_by' => $this->getStaffId(),
            'dispensed_at' => date('Y-m-d H:i:s'),
            'is_dispensed' => $is_fully_dispensed ? 'yes' : 'partial'
        ];
        
        if ($bill_id) {
            $update_data['bill_reference'] = $bill_id;
        }
        
        $this->db->where('id', $medicine_id);
        $this->db->update('opd_prescription_details', $update_data);
        
        // Update prescription overall status
        $this->updatePrescriptionStatus($prescription_id);
        
        $this->db->trans_complete();
        
        if ($this->db->trans_status() === FALSE) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Failed to update dispensing record'
            ]);
        } else {
            echo json_encode([
                'status' => 'success',
                'message' => 'Medicine dispensing recorded successfully',
                'is_fully_dispensed' => $is_fully_dispensed,
                'remaining_quantity' => $medicine['quantity_prescribed'] - $new_dispensed
            ]);
        }
        
    } catch (Exception $e) {
        $this->db->trans_rollback();
        echo json_encode([
            'status' => 'error',
            'message' => 'Error recording dispensing: ' . $e->getMessage()
        ]);
    }
}

/**
 * Update prescription overall dispensing status
 */
private function updatePrescriptionStatus($prescription_id) {
    try {
        // Check dispensing status of all medicines
        $this->db->select('COUNT(*) as total, SUM(CASE WHEN is_dispensed = "yes" THEN 1 ELSE 0 END) as fully_dispensed,
                          SUM(CASE WHEN is_dispensed = "partial" THEN 1 ELSE 0 END) as partially_dispensed');
        $this->db->from('opd_prescription_details');
        $this->db->where('prescription_id', $prescription_id);
        $result = $this->db->get()->row();
        
        $status = 'fresh';
        if ($result->fully_dispensed == $result->total) {
            $status = 'dispensed'; // All medicines fully dispensed
        } elseif ($result->fully_dispensed > 0 || $result->partially_dispensed > 0) {
            $status = 'partial'; // Some medicines dispensed
        }
        
        // Update prescription status
        $this->db->where('id', $prescription_id);
        $this->db->update('opd_prescription_basic', [
            'dispensed_status' => $status
        ]);
        
        log_message('info', "Prescription {$prescription_id} status updated to: {$status}");
        
    } catch (Exception $e) {
        log_message('error', 'Error updating prescription status: ' . $e->getMessage());
    }
}

/**
 * Process bill with prescription tracking
 */
public function processBillWithPrescription() {
    header('Content-Type: application/json');
    
    try {
        // Get form data
        $patient_id = $this->input->post('patient_id');
        $bill_items = json_decode($this->input->post('bill_items'), true);
        $from_prescription = $this->input->post('from_prescription') == '1';
        
        if (!$patient_id || empty($bill_items)) {
            echo json_encode([
                'status' => 0,
                'message' => 'Patient ID and bill items required'
            ]);
            return;
        }
        
        $this->db->trans_start();
        
        // Process regular pharmacy bill
        $bill_data = [
            'date' => date('Y-m-d H:i:s'),
            'patient_id' => $patient_id,
            'customer_type' => 'patient',
            'total' => floatval($this->input->post('subtotal')),
            'discount_percentage' => floatval($this->input->post('discount_percentage')),
            'discount' => floatval($this->input->post('discount_amount')),
            'tax_percentage' => floatval($this->input->post('tax_percentage')),
            'tax' => floatval($this->input->post('tax_amount')),
            'net_amount' => floatval($this->input->post('net_amount')),
            'note' => $this->input->post('notes') ?: '',
            'payment_status' => 'paid',
            'generated_by' => $this->getStaffId(),
            'created_at' => date('Y-m-d H:i:s')
        ];
        
        $this->db->insert('pharmacy_bill_basic', $bill_data);
        $bill_id = $this->db->insert_id();
        
        if (!$bill_id) {
            throw new Exception('Failed to create pharmacy bill');
        }
        
        // Save bill items and track prescription dispensing
        foreach ($bill_items as $item) {
            // Save bill detail
            $bill_detail = [
                'pharmacy_bill_basic_id' => $bill_id,
                'medicine_batch_detail_id' => $item['medicine_batch_detail_id'],
                'quantity' => $item['quantity'],
                'sale_price' => $item['sale_price'],
                'amount' => $item['amount'],
                'created_at' => date('Y-m-d H:i:s')
            ];
            
            $this->db->insert('pharmacy_bill_detail', $bill_detail);
            
            // If this item came from a prescription, mark it as dispensed
            if ($from_prescription && isset($item['prescription_id']) && isset($item['prescription_medicine_id'])) {
                $this->markMedicineDispensedFromBill(
                    $item['prescription_id'],
                    $item['prescription_medicine_id'],
                    $item['quantity'],
                    $bill_id
                );
            }
        }
        
        $this->db->trans_complete();
        
        if ($this->db->trans_status() === FALSE) {
            echo json_encode([
                'status' => 0,
                'message' => 'Failed to process bill'
            ]);
        } else {
            echo json_encode([
                'status' => 1,
                'message' => 'Bill processed successfully',
                'bill_id' => $bill_id
            ]);
        }
        
    } catch (Exception $e) {
        $this->db->trans_rollback();
        echo json_encode([
            'status' => 0,
            'message' => 'Error processing bill: ' . $e->getMessage()
        ]);
    }
}

/**
 * Mark medicine as dispensed from bill processing
 */
private function markMedicineDispensedFromBill($prescription_id, $medicine_id, $quantity, $bill_id) {
    try {
        // Get current medicine details
        $this->db->select('*');
        $this->db->from('opd_prescription_details');
        $this->db->where('id', $medicine_id);
        $this->db->where('prescription_id', $prescription_id);
        $medicine = $this->db->get()->row_array();
        
        if ($medicine) {
            $new_dispensed = $medicine['quantity_dispensed'] + $quantity;
            $is_fully_dispensed = $new_dispensed >= $medicine['quantity_prescribed'];
            
            $this->db->where('id', $medicine_id);
            $this->db->update('opd_prescription_details', [
                'quantity_dispensed' => $new_dispensed,
                'dispensed_by' => $this->getStaffId(),
                'dispensed_at' => date('Y-m-d H:i:s'),
                'is_dispensed' => $is_fully_dispensed ? 'yes' : 'partial',
                'bill_reference' => $bill_id
            ]);
            
            // Update prescription status
            $this->updatePrescriptionStatus($prescription_id);
        }
        
    } catch (Exception $e) {
        log_message('error', 'Error marking medicine as dispensed: ' . $e->getMessage());
    }
}

/**
 * Get prescription dispensing history
 */
public function getPrescriptionDispensingHistory() {
    header('Content-Type: application/json');
    
    try {
        $patient_id = $this->input->post('patient_id');
        $limit = $this->input->post('limit') ?: 10;
        
        if (!$patient_id) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Patient ID required'
            ]);
            return;
        }
        
        // Get dispensing history
        $this->db->select('pb.*, s.name as doctor_name, s.surname as doctor_surname,
                          COUNT(pd.id) as total_medicines,
                          SUM(CASE WHEN pd.is_dispensed = "yes" THEN 1 ELSE 0 END) as dispensed_count,
                          SUM(pd.total_amount) as total_cost');
        $this->db->from('opd_prescription_basic pb');
        $this->db->join('staff s', 'pb.doctor_id = s.id', 'left');
        $this->db->join('opd_prescription_details pd', 'pb.id = pd.prescription_id', 'left');
        $this->db->where('pb.patient_id', $patient_id);
        $this->db->group_by('pb.id');
        $this->db->order_by('pb.created_at', 'DESC');
        $this->db->limit($limit);
        
        $query = $this->db->get();
        $history = $query->result_array();
        
        // Format the data
        foreach ($history as &$record) {
            $record['doctor_name'] = trim(($record['doctor_name'] ?: '') . ' ' . ($record['doctor_surname'] ?: ''));
            $record['date_formatted'] = date('M j, Y', strtotime($record['date']));
            $record['status_label'] = ucfirst($record['dispensed_status']);
            $record['dispensing_progress'] = $record['total_medicines'] > 0 ? 
                round(($record['dispensed_count'] / $record['total_medicines']) * 100, 1) : 0;
        }
        
        echo json_encode([
            'status' => 'success',
            'history' => $history,
            'total_found' => count($history)
        ]);
        
    } catch (Exception $e) {
        echo json_encode([
            'status' => 'error',
            'message' => 'Error loading dispensing history: ' . $e->getMessage()
        ]);
    }
}

/**
 * Debug prescription system
 */
public function debugPrescriptionSystem() {
    header('Content-Type: application/json');
    
    try {
        $results = [
            'timestamp' => date('Y-m-d H:i:s'),
            'table_checks' => [],
            'sample_data' => [],
            'endpoints' => []
        ];
        
        // Check table existence
        $results['table_checks']['opd_prescription_basic'] = $this->db->table_exists('opd_prescription_basic');
        $results['table_checks']['opd_prescription_details'] = $this->db->table_exists('opd_prescription_details');
        $results['table_checks']['pharmacy_bill_basic'] = $this->db->table_exists('pharmacy_bill_basic');
        $results['table_checks']['consultations'] = $this->db->table_exists('consultations');
        
        // Get sample data if tables exist
        if ($results['table_checks']['opd_prescription_basic']) {
            $this->db->from('opd_prescription_basic');
            $results['sample_data']['total_prescriptions'] = $this->db->count_all_results();
            
            $this->db->select('dispensed_status, COUNT(*) as count');
            $this->db->from('opd_prescription_basic');
            $this->db->group_by('dispensed_status');
            $query = $this->db->get();
            $results['sample_data']['status_breakdown'] = $query->result_array();
        }
        
        // List available endpoints
        $results['endpoints'] = [
            'getPatientPrescriptions' => base_url() . 'admin/dispensary/getPatientPrescriptions',
            'markMedicineDispensed' => base_url() . 'admin/dispensary/markMedicineDispensed',
            'processBillWithPrescription' => base_url() . 'admin/dispensary/processBillWithPrescription',
            'getPrescriptionDispensingHistory' => base_url() . 'admin/dispensary/getPrescriptionDispensingHistory'
        ];
        
        echo json_encode([
            'status' => 'success',
            'message' => 'Prescription system debug completed',
            'results' => $results
        ]);
        
    } catch (Exception $e) {
        echo json_encode([
            'status' => 'error',
            'message' => 'Debug failed: ' . $e->getMessage()
        ]);
    }
}
/**
 * Reports page
 */
public function reports() {
    $this->session->set_userdata('top_menu', 'Medical Dispensary');
    $this->session->set_userdata('sub_menu', 'dispensary/reports');

    $data['title'] = 'Dispensary Reports & Analytics';
    
    try {
        $data['medicine_categories'] = $this->dispensary_model->getMedicineCategories();
        
        $this->load->view('layout/header', $data);
        $this->load->view('admin/dispensary/reports', $data);
        $this->load->view('layout/footer', $data);
        
    } catch (Exception $e) {
        log_message('error', 'Dispensary reports error: ' . $e->getMessage());
        show_error('Failed to load reports: ' . $e->getMessage());
    }
}

    /**
     * Medicine Stock Management
     */
    public function medicineStock() {
        $this->session->set_userdata('top_menu', 'Medical Dispensary');
        $this->session->set_userdata('sub_menu', 'dispensary/stock');

        $data['title'] = 'Medicine Stock Management';
        
        try {
            $data['medicine_categories'] = $this->dispensary_model->getMedicineCategories();
            $data['medicines'] = $this->dispensary_model->getAllMedicinesWithStock();
            $data['low_stock_count'] = $this->dispensary_model->getLowStockCount();
            $data['expired_count'] = $this->dispensary_model->getExpiredCount();
            
            $this->load->view('layout/header', $data);
            $this->load->view('admin/dispensary/medicine_stock', $data);
            $this->load->view('layout/footer', $data);
            
        } catch (Exception $e) {
            log_message('error', 'Medicine stock error: ' . $e->getMessage());
            show_error('Failed to load medicine stock: ' . $e->getMessage());
        }
    }

    /**
     * Billing/Dispensing Interface
     */
    public function dispensing() {
        $this->session->set_userdata('top_menu', 'Medical Dispensary');
        $this->session->set_userdata('sub_menu', 'dispensary/billing');

        $data['title'] = 'Medicine Dispensing & Billing';
        
        try {
            $data['medicine_categories'] = $this->dispensary_model->getMedicineCategories();
            $data['available_medicines'] = $this->dispensary_model->getAvailableMedicines();
            $data['currency_symbol'] = $this->customlib->getHospitalCurrencyFormat();
            
            // Get the next bill number
            $data['next_bill_no'] = $this->dispensary_model->generateBillNumber();
            
            $this->load->view('layout/header', $data);
            $this->load->view('admin/dispensary/dispensing', $data);
            $this->load->view('layout/footer', $data);
            
        } catch (Exception $e) {
            log_message('error', 'Dispensing interface error: ' . $e->getMessage());
            show_error('Failed to load dispensing interface: ' . $e->getMessage());
        }
    }
    
    public function test() {
        echo "<h1>Dispensary (Pharmacy) Controller Test</h1>";
        echo "<p>Controller is working properly!</p>";
        echo "<p>Queue integration: " . (class_exists('Queue_model') ? 'ACTIVE' : 'MISSING') . "</p>";
        echo "<p>Notification system: " . (is_writable(APPPATH . 'cache/') ? 'READY' : 'NOT WRITABLE') . "</p>";
        
        // Test database tables
        $tables_to_check = ['medication_dispensing', 'dispensing_completions', 'medications'];
        foreach ($tables_to_check as $table) {
            echo "<p>Table $table: " . ($this->db->table_exists($table) ? 'EXISTS' : 'MISSING') . "</p>";
        }
        
        try {
            $stats = $this->Queue_model->getQueueStatistics('pharmacy');
            echo "<p>Queue statistics: Working</p>";
        } catch (Exception $e) {
            echo "<p>Queue statistics: Error - " . $e->getMessage() . "</p>";
        }
    }
    
    /**
     * AJAX: Remove patient from pharmacy queue
     */
    public function remove_from_queue() {
        header('Content-Type: application/json');
        
        try {
            $queue_id = $this->input->post('queue_id');
            $reason = $this->input->post('reason', TRUE) ?: 'Medication dispensing completed';
            
            $result = $this->Queue_model->skipPatient($queue_id, $reason);
            echo json_encode($result);
        } catch (Exception $e) {
            echo json_encode(['status' => 'error', 'message' => $e->getMessage()]);
        }
    }
    
    /**
 * Check for incoming patients from triage/consultation
 */
public function check_incoming_notifications() {
    header('Content-Type: application/json');
    
    try {
        $notification_file = APPPATH . 'cache/queue_notifications_pharmacy.json';
        
        if (file_exists($notification_file)) {
            $notification = json_decode(file_get_contents($notification_file), true);
            unlink($notification_file);
            
            echo json_encode([
                'status' => 'success',
                'notification' => $notification,
                'has_notification' => true,
                'sound' => true
            ]);
        } else {
            echo json_encode([
                'status' => 'success',
                'has_notification' => false
            ]);
        }
    } catch (Exception $e) {
        echo json_encode(['status' => 'error', 'message' => $e->getMessage()]);
    }
}

    /**
     * Search Patient for Dispensing
     */
public function searchPatient() {
    $search_term = $this->input->post('search_term');
    
    if (empty($search_term)) {
        echo json_encode(array('status' => 0, 'message' => 'Search term is required'));
        return;
    }
    
    try {
        // Direct query instead of using model method for testing
        $this->db->select('id, patient_unique_id, patient_name, guardian_name, mobileno, age, gender');
        $this->db->from('patients');
        $this->db->group_start();
            $this->db->like('patient_name', $search_term);
            $this->db->or_like('patient_unique_id', $search_term);
            $this->db->or_like('mobileno', $search_term);
        $this->db->group_end();
        $this->db->where('is_active', 'yes');
        $this->db->order_by('patient_name', 'ASC');
        $this->db->limit(20);
        
        $query = $this->db->get();
        $patients = $query->result();
        
        echo json_encode(array(
            'status' => 1, 
            'patients' => $patients,
            'count' => count($patients),
            'search_term' => $search_term,
            'query' => $this->db->last_query() // For debugging
        ));
        
    } catch (Exception $e) {
        log_message('error', 'Patient search error: ' . $e->getMessage());
        echo json_encode(array(
            'status' => 0, 
            'message' => 'Search failed: ' . $e->getMessage()
        ));
    }
}
    
public function testPatientData() {
    // Test if we have any patients at all
    $this->db->select('COUNT(*) as total');
    $total_patients = $this->db->get('patients')->row()->total;
    
    // Get a few sample patients
    $this->db->select('id, patient_name, patient_unique_id, mobileno, is_active');
    $this->db->limit(5);
    $sample_patients = $this->db->get('patients')->result();
    
    echo json_encode(array(
        'total_patients' => $total_patients,
        'sample_patients' => $sample_patients,
        'table_exists' => $this->db->table_exists('patients')
    ));
}

    /**
     * Get Patient Details
     */
    public function getPatientDetails() {
        $patient_id = $this->input->post('patient_id');
        
        if (empty($patient_id)) {
            echo json_encode(array('status' => 0, 'message' => 'Patient ID is required'));
            return;
        }
        
        try {
            $patient = $this->dispensary_model->getPatientById($patient_id);
            if ($patient) {
                echo json_encode(array('status' => 1, 'patient' => $patient));
            } else {
                echo json_encode(array('status' => 0, 'message' => 'Patient not found'));
            }
        } catch (Exception $e) {
            log_message('error', 'Get patient details error: ' . $e->getMessage());
            echo json_encode(array('status' => 0, 'message' => 'Failed to get patient details'));
        }
    }
    
    /**
 * Get stored prescriptions for a patient - FIXED VERSION
 */
public function getPatientPrescriptions() {
    header('Content-Type: application/json');
    error_reporting(0);
    ini_set('display_errors', 0);
    
    try {
        $patient_id = $this->input->post('patient_id');
        
        if (!$patient_id) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Patient ID required',
                'prescriptions' => []
            ]);
            return;
        }
        
        // Check if prescription tables exist
        if (!$this->db->table_exists('opd_prescription_basic')) {
            echo json_encode([
                'status' => 'success',
                'prescriptions' => [],
                'message' => 'No prescription system available - please check with consultation module'
            ]);
            return;
        }
        
        // Get fresh and partial prescriptions for this patient
        $this->db->select('pb.*, s.name as doctor_name, s.surname as doctor_surname, 
                          p.patient_name, c.symptoms, c.diagnosis');
        $this->db->from('opd_prescription_basic pb');
        $this->db->join('staff s', 'pb.doctor_id = s.id', 'left');
        $this->db->join('patients p', 'pb.patient_id = p.id', 'left');
        $this->db->join('consultations c', 'pb.consultation_id = c.id', 'left');
        $this->db->where('pb.patient_id', $patient_id);
        $this->db->where_in('pb.dispensed_status', ['fresh', 'partial']); // Only non-dispensed prescriptions
        $this->db->order_by('pb.created_at', 'DESC');
        $this->db->limit(10);
        
        $query = $this->db->get();
        
        if (!$query) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Database query failed',
                'prescriptions' => []
            ]);
            return;
        }
        
        $prescriptions = $query->result_array();
        
        // Get medicines for each prescription
        foreach ($prescriptions as &$prescription) {
            // Format doctor name
            $prescription['doctor_name'] = trim(($prescription['doctor_name'] ?: '') . ' ' . ($prescription['doctor_surname'] ?: ''));
            if (empty(trim($prescription['doctor_name']))) {
                $prescription['doctor_name'] = 'Unknown Doctor';
            }
            
            // Get non-dispensed medicines for this prescription
            $this->db->select('*');
            $this->db->from('opd_prescription_details');
            $this->db->where('prescription_id', $prescription['id']);
            $this->db->where('is_dispensed !=', 'yes'); // Only medicines not fully dispensed
            $this->db->order_by('id', 'ASC');
            
            $medicines_query = $this->db->get();
            $prescription['medicines'] = $medicines_query ? $medicines_query->result_array() : [];
            
            // Calculate remaining quantities
            foreach ($prescription['medicines'] as &$medicine) {
                $medicine['remaining_quantity'] = $medicine['quantity_prescribed'] - $medicine['quantity_dispensed'];
                $medicine['medicine_name'] = $medicine['medicine_name']; // For compatibility
                $medicine['dosage'] = $medicine['dosage'] ?: 'As directed';
                $medicine['instruction'] = $medicine['instructions'] ?: 'Take as prescribed';
            }
            
            // Format dates
            $prescription['prescription_date'] = $prescription['date'];
            $prescription['prescription_date_formatted'] = date('M j, Y', strtotime($prescription['date']));
            
            // Set defaults
            $prescription['visit_type'] = $prescription['visit_type'] ?: 'OPD';
            $prescription['finding_description'] = $prescription['finding_description'] ?: ($prescription['diagnosis'] ?: 'General consultation');
        }
        
        echo json_encode([
            'status' => 'success',
            'prescriptions' => $prescriptions,
            'total_found' => count($prescriptions),
            'message' => count($prescriptions) > 0 ? 'Found ' . count($prescriptions) . ' prescription(s) for dispensing' : 'No prescriptions available for dispensing'
        ]);
        
    } catch (Exception $e) {
        echo json_encode([
            'status' => 'error',
            'message' => 'Error loading prescriptions: ' . $e->getMessage(),
            'prescriptions' => []
        ]);
    }
}

/**
 * Try to get OPD prescriptions
 */
private function getOPDPrescriptions($patient_id) {
    try {
        // Check if opd_prescription_basic table exists
        if (!$this->db->table_exists('opd_prescription_basic')) {
            log_message('info', 'opd_prescription_basic table does not exist');
            return [];
        }
        
        // Simple query to get OPD prescriptions
        $this->db->select('opb.*, s.name as doctor_name, s.surname as doctor_surname');
        $this->db->from('opd_prescription_basic opb');
        $this->db->join('staff s', 's.id = opb.generated_by', 'left');
        $this->db->join('opd_details od', 'od.id = opb.opd_details_id', 'left');
        $this->db->where('od.patient_id', $patient_id);
        $this->db->order_by('opb.date', 'DESC');
        $this->db->limit(5);
        
        $prescriptions = $this->db->get()->result_array();
        
        // Get medicines for each prescription
        foreach ($prescriptions as &$prescription) {
            $prescription['medicines'] = $this->getOPDPrescriptionMedicines($prescription['id']);
            $prescription['doctor_name'] = trim($prescription['doctor_name'] . ' ' . $prescription['doctor_surname']);
            $prescription['prescription_date'] = $prescription['date'];
            $prescription['visit_type'] = 'OPD';
        }
        
        return $prescriptions;
        
    } catch (Exception $e) {
        log_message('error', 'OPD prescriptions error: ' . $e->getMessage());
        return [];
    }
}

/**
 * Try to get IPD prescriptions
 */
private function getIPDPrescriptions($patient_id) {
    try {
        // Check if ipd_prescription_basic table exists
        if (!$this->db->table_exists('ipd_prescription_basic')) {
            log_message('info', 'ipd_prescription_basic table does not exist');
            return [];
        }
        
        // Simple query to get IPD prescriptions
        $this->db->select('ipb.*, s.name as doctor_name, s.surname as doctor_surname');
        $this->db->from('ipd_prescription_basic ipb');
        $this->db->join('staff s', 's.id = ipb.generated_by', 'left');
        $this->db->join('ipd_details id', 'id.id = ipb.ipd_id', 'left');
        $this->db->where('id.patient_id', $patient_id);
        $this->db->order_by('ipb.date', 'DESC');
        $this->db->limit(5);
        
        $prescriptions = $this->db->get()->result_array();
        
        // Get medicines for each prescription
        foreach ($prescriptions as &$prescription) {
            $prescription['medicines'] = $this->getIPDPrescriptionMedicines($prescription['id']);
            $prescription['doctor_name'] = trim($prescription['doctor_name'] . ' ' . $prescription['doctor_surname']);
            $prescription['prescription_date'] = $prescription['date'];
            $prescription['visit_type'] = 'IPD';
        }
        
        return $prescriptions;
        
    } catch (Exception $e) {
        log_message('error', 'IPD prescriptions error: ' . $e->getMessage());
        return [];
    }
}

/**
 * Get medicines for OPD prescription
 */
private function getOPDPrescriptionMedicines($prescription_id) {
    try {
        if (!$this->db->table_exists('opd_prescription_details')) {
            return [];
        }
        
        $this->db->select('opd.*, m.medicine_name, mc.medicine_category');
        $this->db->from('opd_prescription_details opd');
        $this->db->join('medicines m', 'm.id = opd.medicine_id', 'left');
        $this->db->join('medicine_category mc', 'mc.id = m.medicine_category_id', 'left');
        $this->db->where('opd.opd_prescription_basic_id', $prescription_id);
        
        return $this->db->get()->result_array();
        
    } catch (Exception $e) {
        log_message('error', 'OPD prescription medicines error: ' . $e->getMessage());
        return [];
    }
}

/**
 * Get medicines for IPD prescription
 */
private function getIPDPrescriptionMedicines($prescription_id) {
    try {
        if (!$this->db->table_exists('ipd_prescription_details')) {
            return [];
        }
        
        $this->db->select('ipd.*, m.medicine_name, mc.medicine_category');
        $this->db->from('ipd_prescription_details ipd');
        $this->db->join('medicines m', 'm.id = ipd.medicine_id', 'left');
        $this->db->join('medicine_category mc', 'mc.id = m.medicine_category_id', 'left');
        $this->db->where('ipd.ipd_prescription_basic_id', $prescription_id);
        
        return $this->db->get()->result_array();
        
    } catch (Exception $e) {
        log_message('error', 'IPD prescription medicines error: ' . $e->getMessage());
        return [];
    }
}

private function createMockPrescription($patient_id) {
    // Create a sample prescription for testing
    return [
        [
            'id' => 'mock_1',
            'prescription_no' => 'RX' . date('Ymd') . '001',
            'doctor_name' => 'Dr. Sample Doctor',
            'prescription_date' => date('Y-m-d'),
            'visit_type' => 'OPD',
            'finding_description' => 'Sample prescription for testing dispensing system',
            'dispensed_status' => 'fresh',
            'medicines' => [
                [
                    'medicine_name' => 'Paracetamol 500mg',
                    'dosage' => '1 tablet twice daily',
                    'duration' => '5 days',
                    'instruction' => 'Take after meals',
                    'medicine_category' => 'Analgesics'
                ],
                [
                    'medicine_name' => 'Amoxicillin 250mg',
                    'dosage' => '1 capsule three times daily',
                    'duration' => '7 days',
                    'instruction' => 'Complete the course',
                    'medicine_category' => 'Antibiotics'
                ]
            ]
        ]
    ];
}



    /**
     * Search Medicine by Name/Category
     */
    public function searchMedicine() {
    $search_term = $this->input->post('search_term');
    $category_id = $this->input->post('category_id');
    
    // Debug logging
    log_message('info', 'Search medicine called with term: ' . $search_term . ', category: ' . $category_id);
    
    try {
        // Build the query step by step for debugging
        $this->db->select('m.id, m.medicine_name, m.medicine_company, m.unit,
                          mc.medicine_category,
                          COALESCE(SUM(mbd.available_quantity), 0) as available_stock,
                          MIN(mbd.sale_rate) as min_price,
                          MAX(mbd.sale_rate) as max_price');
        $this->db->from('medicines m');
        $this->db->join('medicine_category mc', 'm.medicine_category_id = mc.id', 'left');
        $this->db->join('medicine_batch_details mbd', 'm.id = mbd.medicine_id', 'left');
        
        // Add search conditions
        if (!empty($search_term)) {
            $this->db->group_start();
                $this->db->like('m.medicine_name', $search_term);
                $this->db->or_like('m.medicine_company', $search_term);
                if (isset($m->medicine_composition)) {
                    $this->db->or_like('m.medicine_composition', $search_term);
                }
            $this->db->group_end();
        }
        
        if (!empty($category_id)) {
            $this->db->where('m.medicine_category_id', $category_id);
        }
        
        $this->db->where('m.is_active', 'yes');
        // Remove the stock condition for now to see all medicines
        // $this->db->where('mbd.available_quantity >', 0);
        // $this->db->where('mbd.expiry >', date('Y-m-d'));
        
        $this->db->group_by('m.id');
        $this->db->order_by('m.medicine_name', 'ASC');
        $this->db->limit(50);
        
        // Log the query for debugging
        $query_string = $this->db->get_compiled_select();
        log_message('info', 'Medicine search query: ' . $query_string);
        
        // Reset and execute the query
        $this->db->select('m.id, m.medicine_name, m.medicine_company, m.unit,
                          mc.medicine_category,
                          COALESCE(SUM(mbd.available_quantity), 0) as available_stock,
                          MIN(mbd.sale_rate) as min_price,
                          MAX(mbd.sale_rate) as max_price');
        $this->db->from('medicines m');
        $this->db->join('medicine_category mc', 'm.medicine_category_id = mc.id', 'left');
        $this->db->join('medicine_batch_details mbd', 'm.id = mbd.medicine_id', 'left');
        
        if (!empty($search_term)) {
            $this->db->group_start();
                $this->db->like('m.medicine_name', $search_term);
                $this->db->or_like('m.medicine_company', $search_term);
            $this->db->group_end();
        }
        
        if (!empty($category_id)) {
            $this->db->where('m.medicine_category_id', $category_id);
        }
        
        $this->db->where('m.is_active', 'yes');
        $this->db->group_by('m.id');
        $this->db->order_by('m.medicine_name', 'ASC');
        $this->db->limit(50);
        
        $medicines = $this->db->get()->result();
        
        log_message('info', 'Medicine search found: ' . count($medicines) . ' results');
        
        echo json_encode(array('status' => 1, 'medicines' => $medicines));
        
    } catch (Exception $e) {
        log_message('error', 'Medicine search error: ' . $e->getMessage());
        echo json_encode(array('status' => 0, 'message' => 'Search failed: ' . $e->getMessage()));
    }
}

    /**
     * Get Medicine Details with Stock
     */
    public function getMedicineDetails() {
        $medicine_id = $this->input->post('medicine_id');
        
        if (empty($medicine_id)) {
            echo json_encode(array('status' => 0, 'message' => 'Medicine ID is required'));
            return;
        }
        
        try {
            $medicine = $this->dispensary_model->getMedicineWithStock($medicine_id);
            if ($medicine) {
                echo json_encode(array('status' => 1, 'medicine' => $medicine));
            } else {
                echo json_encode(array('status' => 0, 'message' => 'Medicine not found or out of stock'));
            }
        } catch (Exception $e) {
            log_message('error', 'Get medicine details error: ' . $e->getMessage());
            echo json_encode(array('status' => 0, 'message' => 'Failed to get medicine details'));
        }
    }

/**
 * Process Dispensing Bill - FIXED VERSION
 */
public function processBill() {
    // Log the incoming request for debugging
    log_message('info', 'Process bill request received');
    log_message('info', 'POST data: ' . json_encode($this->input->post()));
    
    // Get customer information
    $customer_id = $this->input->post('customer_id'); // This could be patient_id or walkin identifier
    $customer_type = $this->input->post('customer_type'); // 'registered' or 'walkin'
    
    // Walk-in customer details (only if customer_type is 'walkin')
    $walkin_name = $this->input->post('walkin_name');
    $walkin_phone = $this->input->post('walkin_phone');
    $walkin_age = $this->input->post('walkin_age');
    $walkin_gender = $this->input->post('walkin_gender');
    
    $bill_data = array(
        'date' => date('Y-m-d H:i:s'),
        'total' => $this->input->post('subtotal'),
        'discount_percentage' => $this->input->post('discount_percentage') ?: 0,
        'discount' => $this->input->post('discount_amount') ?: 0,
        'tax_percentage' => $this->input->post('tax_percentage') ?: 0,
        'tax' => $this->input->post('tax_amount') ?: 0,
        'net_amount' => $this->input->post('net_amount'),
        'note' => $this->input->post('notes'),
        'generated_by' => $this->session->userdata('admin')['id'] ?? 1,
        'payment_status' => 'paid',
        'payment_method' => $this->input->post('payment_method') ?: 'cash',
        'amount_paid' => $this->input->post('paid_amount'),
        'balance_amount' => $this->input->post('balance_amount')
    );
    
    // Handle customer data based on type
    if ($customer_type === 'walkin') {
        // For walk-in customers, store their details in the bill
        $bill_data['customer_type'] = 'walkin';
        $bill_data['customer_name'] = $walkin_name ?: 'Walk-in Customer';
        $bill_data['customer_phone'] = $walkin_phone;
        $bill_data['customer_age'] = $walkin_age;
        $bill_data['customer_gender'] = $walkin_gender;
        $bill_data['patient_id'] = null; // No patient ID for walk-ins
    } else {
        // For registered patients
        $bill_data['customer_type'] = 'registered';
        $bill_data['patient_id'] = $customer_id;
        $bill_data['customer_name'] = null; // Will be fetched from patient table
        $bill_data['customer_phone'] = null;
        $bill_data['customer_age'] = null;
        $bill_data['customer_gender'] = null;
    }
    
    $bill_items_json = $this->input->post('bill_items');
    $bill_items = json_decode($bill_items_json, true);
    
    log_message('info', 'Bill data: ' . json_encode($bill_data));
    log_message('info', 'Bill items: ' . $bill_items_json);
    
    // Validate required data
    if (empty($bill_items)) {
        echo json_encode(array('status' => 0, 'message' => 'Medicine items are required'));
        return;
    }
    
    if ($customer_type === 'registered' && empty($customer_id)) {
        echo json_encode(array('status' => 0, 'message' => 'Patient ID is required for registered patients'));
        return;
    }
    
    // Validate bill items
    if (!is_array($bill_items) || count($bill_items) == 0) {
        echo json_encode(array('status' => 0, 'message' => 'Invalid bill items data'));
        return;
    }
    
    // Start transaction
    $this->db->trans_start();
    
    try {
        // Generate bill number
        $bill_data['bill_no'] = $this->generateBillNumber();
        
        // Insert bill into pharmacy_bill_basic table
        $this->db->insert('pharmacy_bill_basic', $bill_data);
        $bill_id = $this->db->insert_id();
        
        log_message('info', 'Bill inserted with ID: ' . $bill_id);
        
        if ($bill_id) {
            // Insert bill items and update stock
            $item_success = true;
            foreach ($bill_items as $item) {
                // Validate item data
                if (empty($item['medicine_id']) || empty($item['medicine_batch_detail_id']) || empty($item['quantity'])) {
                    log_message('error', 'Invalid item data: ' . json_encode($item));
                    $item_success = false;
                    break;
                }
                
                // Insert bill item
                $bill_item_data = array(
                    'pharmacy_bill_basic_id' => $bill_id,
                    'medicine_batch_detail_id' => $item['medicine_batch_detail_id'],
                    'quantity' => $item['quantity'],
                    'sale_price' => $item['sale_price'] ?? $item['unit_price'],
                    'amount' => $item['amount'] ?? $item['total_price']
                );
                
                $this->db->insert('pharmacy_bill_detail', $bill_item_data);
                
                // Update medicine stock
                $this->updateMedicineStock($item['medicine_batch_detail_id'], $item['quantity']);
            }
            
            if ($item_success) {
                $this->db->trans_complete();
                
                if ($this->db->trans_status() === FALSE) {
                    echo json_encode(array('status' => 0, 'message' => 'Transaction failed'));
                } else {
                    // Success response
                    echo json_encode(array(
                        'status' => 1, 
                        'message' => 'Bill processed successfully',
                        'bill_id' => $bill_id,
                        'bill_no' => $bill_data['bill_no'],
                        'customer_type' => $customer_type
                    ));
                    
                    // Log successful transaction
                    log_message('info', 'Bill processed successfully - ID: ' . $bill_id . ', Type: ' . $customer_type);
                }
            } else {
                $this->db->trans_rollback();
                echo json_encode(array('status' => 0, 'message' => 'Failed to process bill items'));
            }
        } else {
            $this->db->trans_rollback();
            echo json_encode(array('status' => 0, 'message' => 'Failed to create bill'));
        }
        
    } catch (Exception $e) {
        $this->db->trans_rollback();
        log_message('error', 'Process bill error: ' . $e->getMessage());
        echo json_encode(array('status' => 0, 'message' => 'Error processing bill: ' . $e->getMessage()));
    }
}

/**
 * Get customer history - works for both registered patients and walk-in customers
 */
public function getCustomerHistory() {
    header('Content-Type: application/json');
    
    try {
        $customer_id = $this->input->post('customer_id');
        $customer_type = $this->input->post('customer_type');
        $limit = $this->input->post('limit') ?: 20;
        
        if (empty($customer_id) || empty($customer_type)) {
            echo json_encode([
                'status' => 0,
                'message' => 'Customer ID and type are required'
            ]);
            return;
        }
        
        log_message('info', "Loading history for customer ID: {$customer_id}, Type: {$customer_type}");
        
        // Get bills based on customer type
        if ($customer_type === 'registered') {
            $bills = $this->getPatientBillsFromDatabase($customer_id, $limit);
        } else {
            // For walk-in customers, we can't really track history since they don't have persistent IDs
            $bills = [];
        }
        
        if (!empty($bills)) {
            echo json_encode([
                'status' => 1,
                'bills' => $bills,
                'total_found' => count($bills),
                'message' => 'History loaded successfully'
            ]);
        } else {
            $message = $customer_type === 'walkin' ? 
                'History not available for walk-in customers' : 
                'No billing history found for this customer';
                
            echo json_encode([
                'status' => 0,
                'bills' => [],
                'message' => $message
            ]);
        }
        
    } catch (Exception $e) {
        log_message('error', 'Get customer history error: ' . $e->getMessage());
        echo json_encode([
            'status' => 0,
            'message' => 'Error loading history: ' . $e->getMessage()
        ]);
    }
}
    
    /**
     * Create financial entries for pharmacy bill
     * This integrates with your existing financial system
     */
    private function createPharmacyFinancialEntries($bill_id, $bill_data) {
        try {
            log_message('info', "Creating financial entries for pharmacy bill ID: {$bill_id}");
            
            // Ensure we have the Financial_model loaded
            if (!isset($this->Financial_model)) {
                $this->load->model('Financial_model');
            }
            
            // Call the Financial_model method to create pharmacy entries
            $result = $this->Financial_model->create_pharmacy_financial_entries($bill_id, $bill_data);
            
            if ($result) {
                log_message('info', "Financial entries created successfully for pharmacy bill: {$bill_id}");
                return true;
            } else {
                log_message('error', "Failed to create financial entries for pharmacy bill: {$bill_id}");
                return false;
            }
            
        } catch (Exception $e) {
            log_message('error', "Exception creating financial entries for pharmacy bill {$bill_id}: " . $e->getMessage());
            return false;
        }
    }

    /**
     * Get required pharmacy accounts for financial integration
     */
    private function ensurePharmacyAccounts() {
        try {
            // Define required accounts for pharmacy billing
            $required_accounts = [
                [
                    'account_code' => '1001',
                    'account_name' => 'Cash',
                    'account_type' => 'asset',
                    'description' => 'Cash received from pharmacy sales'
                ],
                [
                    'account_code' => '4005',
                    'account_name' => 'Pharmacy Revenue',
                    'account_type' => 'revenue',
                    'description' => 'Revenue from pharmacy medicine sales'
                ],
                [
                    'account_code' => '2001',
                    'account_name' => 'Accounts Receivable',
                    'account_type' => 'asset',
                    'description' => 'Outstanding payments from patients'
                ],
                [
                    'account_code' => '5001',
                    'account_name' => 'Cost of Goods Sold - Pharmacy',
                    'account_type' => 'expense',
                    'description' => 'Cost of medicines sold'
                ]
            ];
            
            // Check and create accounts if they don't exist
            foreach ($required_accounts as $account) {
                $this->db->where('account_code', $account['account_code']);
                $existing = $this->db->get('chart_of_accounts')->row();
                
                if (!$existing) {
                    $account['is_active'] = 1;
                    $account['opening_balance'] = 0.00;
                    $account['created_at'] = date('Y-m-d H:i:s');
                    
                    $this->db->insert('chart_of_accounts', $account);
                    log_message('info', "Created pharmacy account: {$account['account_code']} - {$account['account_name']}");
                }
            }
            
            return true;
            
        } catch (Exception $e) {
            log_message('error', 'Error ensuring pharmacy accounts: ' . $e->getMessage());
            return false;
        }
    }

/**
 * Print Bill Receipt - FIXED VERSION
 */
public function printBill($bill_id) {
    try {
        // Get bill details
        $this->db->select('pbb.*, p.patient_name, p.patient_unique_id, p.mobileno as patient_phone');
        $this->db->from('pharmacy_bill_basic pbb');
        $this->db->join('patients p', 'pbb.patient_id = p.id', 'left');
        $this->db->where('pbb.id', $bill_id);
        $bill_query = $this->db->get();
        
        if ($bill_query->num_rows() == 0) {
            show_error('Bill not found');
            return;
        }
        
        $bill = $bill_query->row();
        
        // Get bill items
        $this->db->select('pbd.*, m.medicine_name, mb.batch_no, mb.expiry');
        $this->db->from('pharmacy_bill_detail pbd');
        $this->db->join('medicine_batch_details mb', 'pbd.medicine_batch_detail_id = mb.id');
        $this->db->join('medicines m', 'mb.medicine_id = m.id');
        $this->db->where('pbd.pharmacy_bill_basic_id', $bill_id);
        $items_query = $this->db->get();
        $bill_items = $items_query->result();
        
        // Determine customer information based on bill type
        if ($bill->customer_type === 'walkin') {
            $customer_name = $bill->customer_name ?: 'Walk-in Customer';
            $customer_id = 'WALK-IN';
            $customer_phone = $bill->customer_phone ?: 'N/A';
        } else {
            $customer_name = $bill->patient_name ?: 'Unknown Patient';
            $customer_id = $bill->patient_unique_id ?: 'N/A';
            $customer_phone = $bill->patient_phone ?: 'N/A';
        }
        
        // Prepare data for receipt view
        $data = [
            'bill' => $bill,
            'bill_items' => $bill_items,
            'customer_name' => $customer_name,
            'customer_id' => $customer_id,
            'customer_phone' => $customer_phone,
            'customer_type' => $bill->customer_type,
            'currency_symbol' => $this->customlib->getHospitalCurrencyFormat(),
            'hospital_name' => $this->customlib->getHospitalName(),
            'hospital_address' => $this->customlib->getHospitalAddress()
        ];
        
        // Load receipt view
        $this->load->view('admin/dispensary/bill_receipt', $data);
        
    } catch (Exception $e) {
        log_message('error', 'Print bill error: ' . $e->getMessage());
        show_error('Error loading bill receipt: ' . $e->getMessage());
    }
}

/**
 * Get daily sales report with walk-in vs registered patient breakdown
 */
public function getDailySalesReport($date = null) {
    $date = $date ?: date('Y-m-d');
    
    try {
        // Get total sales by customer type
        $this->db->select('
            customer_type,
            COUNT(*) as total_bills,
            SUM(net_amount) as total_amount,
            AVG(net_amount) as average_bill
        ');
        $this->db->from('pharmacy_bill_basic');
        $this->db->where('DATE(date)', $date);
        $this->db->where('payment_status', 'paid');
        $this->db->group_by('customer_type');
        $query = $this->db->get();
        
        $sales_by_type = [];
        foreach ($query->result() as $row) {
            $sales_by_type[$row->customer_type] = [
                'total_bills' => $row->total_bills,
                'total_amount' => $row->total_amount,
                'average_bill' => $row->average_bill
            ];
        }
        
        return $sales_by_type;
        
    } catch (Exception $e) {
        log_message('error', 'Daily sales report error: ' . $e->getMessage());
        return [];
    }
}
/**
 * Database table creation helper for walk-in support
 * Add this method and run it once to update your database schema
 */
public function updateDatabaseForWalkIn() {
    // Only run this if you're an admin and need to update the database
    if (!$this->session->userdata('admin')) {
        show_error('Access denied');
        return;
    }
    
    try {
        // Add new columns to pharmacy_bill_basic table
        $sql = "ALTER TABLE pharmacy_bill_basic 
                ADD COLUMN customer_type ENUM('registered', 'walkin') DEFAULT 'registered' AFTER patient_id,
                ADD COLUMN customer_name VARCHAR(255) NULL AFTER customer_type,
                ADD COLUMN customer_phone VARCHAR(20) NULL AFTER customer_name,
                ADD COLUMN customer_age INT NULL AFTER customer_phone,
                ADD COLUMN customer_gender ENUM('Male', 'Female', 'Other') NULL AFTER customer_age,
                ADD COLUMN payment_method VARCHAR(50) DEFAULT 'cash' AFTER payment_status,
                ADD COLUMN amount_paid DECIMAL(10,2) NULL AFTER payment_method,
                ADD COLUMN balance_amount DECIMAL(10,2) NULL AFTER amount_paid,
                ADD COLUMN bill_no VARCHAR(50) UNIQUE NULL AFTER id";
        
        $this->db->query($sql);
        
        echo json_encode([
            'status' => 1,
            'message' => 'Database updated successfully for walk-in support'
        ]);
        
    } catch (Exception $e) {
        echo json_encode([
            'status' => 0, 
            'message' => 'Database update failed: ' . $e->getMessage()
        ]);
    }
}

/**
 * Get walk-in customers summary for dashboard
 */
public function getWalkInSummary($date_from = null, $date_to = null) {
    $date_from = $date_from ?: date('Y-m-d');
    $date_to = $date_to ?: date('Y-m-d');
    
    try {
        $this->db->select('
            COUNT(*) as total_walkin_bills,
            SUM(net_amount) as total_walkin_amount,
            AVG(net_amount) as average_walkin_bill
        ');
        $this->db->from('pharmacy_bill_basic');
        $this->db->where('customer_type', 'walkin');
        $this->db->where('DATE(date) >=', $date_from);
        $this->db->where('DATE(date) <=', $date_to);
        $this->db->where('payment_status', 'paid');
        
        $query = $this->db->get();
        
        if ($query->num_rows() > 0) {
            return $query->row();
        }
        
        return (object)[
            'total_walkin_bills' => 0,
            'total_walkin_amount' => 0,
            'average_walkin_bill' => 0
        ];
        
    } catch (Exception $e) {
        log_message('error', 'Walk-in summary error: ' . $e->getMessage());
        return (object)[
            'total_walkin_bills' => 0,
            'total_walkin_amount' => 0,
            'average_walkin_bill' => 0
        ];
    }
}
/**
 * Export sales report with customer type breakdown
 */
public function exportSalesReport() {
    $date_from = $this->input->get('date_from') ?: date('Y-m-d', strtotime('-30 days'));
    $date_to = $this->input->get('date_to') ?: date('Y-m-d');
    
    try {
        // Get detailed sales data
        $this->db->select('
            pbb.bill_no,
            pbb.date,
            pbb.customer_type,
            CASE 
                WHEN pbb.customer_type = "registered" THEN p.patient_name
                ELSE pbb.customer_name
            END as customer_name,
            CASE 
                WHEN pbb.customer_type = "registered" THEN p.patient_unique_id
                ELSE "WALK-IN"
            END as customer_id,
            pbb.total,
            pbb.discount,
            pbb.tax,
            pbb.net_amount,
            pbb.payment_method
        ');
        $this->db->from('pharmacy_bill_basic pbb');
        $this->db->join('patients p', 'pbb.patient_id = p.id', 'left');
        $this->db->where('DATE(pbb.date) >=', $date_from);
        $this->db->where('DATE(pbb.date) <=', $date_to);
        $this->db->where('pbb.payment_status', 'paid');
        $this->db->order_by('pbb.date', 'DESC');
        
        $query = $this->db->get();
        $sales_data = $query->result();
        
        // Set headers for CSV download
        header('Content-Type: text/csv');
        header('Content-Disposition: attachment; filename="pharmacy_sales_report_' . $date_from . '_to_' . $date_to . '.csv"');
        
        $output = fopen('php://output', 'w');
        
        // CSV headers
        fputcsv($output, [
            'Bill No',
            'Date',
            'Customer Type',
            'Customer Name',
            'Customer ID',
            'Subtotal',
            'Discount',
            'Tax',
            'Net Amount',
            'Payment Method'
        ]);
        
        // CSV data
        foreach ($sales_data as $row) {
            fputcsv($output, [
                $row->bill_no,
                $row->date,
                ucfirst($row->customer_type),
                $row->customer_name ?: 'N/A',
                $row->customer_id,
                number_format($row->total, 2),
                number_format($row->discount, 2),
                number_format($row->tax, 2),
                number_format($row->net_amount, 2),
                ucfirst($row->payment_method)
            ]);
        }
        
        fclose($output);
        
    } catch (Exception $e) {
        log_message('error', 'Export sales report error: ' . $e->getMessage());
        show_error('Error exporting sales report: ' . $e->getMessage());
    }
}
/**
 * Get Medicine Full Details for Modal
 */
public function getMedicineFullDetails() {
    $medicine_id = $this->input->post('medicine_id');
    
    if (empty($medicine_id)) {
        echo json_encode(array('status' => 0, 'message' => 'Medicine ID is required'));
        return;
    }
    
    try {
        $medicine = $this->dispensary_model->getMedicineFullDetails($medicine_id);
        
        if ($medicine) {
            // Generate HTML content for modal
            $content = $this->generateMedicineDetailsHTML($medicine);
            echo json_encode(array('status' => 1, 'content' => $content));
        } else {
            echo json_encode(array('status' => 0, 'message' => 'Medicine not found'));
        }
    } catch (Exception $e) {
        log_message('error', 'Get medicine full details error: ' . $e->getMessage());
        echo json_encode(array('status' => 0, 'message' => 'Error loading medicine details'));
    }
}

/**
 * Get Stock Adjustment Form
 */
public function getStockAdjustmentForm() {
    $medicine_id = $this->input->post('medicine_id');
    
    if (empty($medicine_id)) {
        echo json_encode(array('status' => 0, 'message' => 'Medicine ID is required'));
        return;
    }
    
    try {
        $data = $this->dispensary_model->getStockAdjustmentData($medicine_id);
        
        if ($data) {
            $content = $this->generateStockAdjustmentHTML($data);
            echo json_encode(array('status' => 1, 'content' => $content));
        } else {
            echo json_encode(array('status' => 0, 'message' => 'Medicine not found or no batches available'));
        }
    } catch (Exception $e) {
        log_message('error', 'Get stock adjustment form error: ' . $e->getMessage());
        echo json_encode(array('status' => 0, 'message' => 'Error loading stock adjustment form'));
    }
}

/**
 * Process Stock Adjustment
 */
public function processStockAdjustment() {
    $adjustment_data = array(
        'batch_id' => $this->input->post('batch_id'),
        'adjustment_type' => $this->input->post('adjustment_type'),
        'quantity' => $this->input->post('quantity'),
        'reason' => $this->input->post('reason'),
        'staff_id' => $this->session->userdata('admin')['id'] ?? 1
    );
    
    // Validate input
    if (empty($adjustment_data['batch_id']) || empty($adjustment_data['adjustment_type']) || empty($adjustment_data['quantity'])) {
        echo json_encode(array('status' => 0, 'message' => 'All fields are required'));
        return;
    }
    
    if (!is_numeric($adjustment_data['quantity']) || $adjustment_data['quantity'] <= 0) {
        echo json_encode(array('status' => 0, 'message' => 'Invalid quantity'));
        return;
    }
    
    try {
        $result = $this->dispensary_model->processStockAdjustment($adjustment_data);
        
        if ($result['status']) {
            echo json_encode(array(
                'status' => 1, 
                'message' => $result['message'],
                'old_quantity' => $result['old_quantity'],
                'new_quantity' => $result['new_quantity']
            ));
        } else {
            echo json_encode(array('status' => 0, 'message' => $result['message']));
        }
    } catch (Exception $e) {
        log_message('error', 'Process stock adjustment error: ' . $e->getMessage());
        echo json_encode(array('status' => 0, 'message' => 'Error processing stock adjustment'));
    }
}

/**
 * Mark Batch as Expired
 */
public function markBatchExpired() {
    $batch_id = $this->input->post('batch_id');
    
    if (empty($batch_id)) {
        echo json_encode(array('status' => 0, 'message' => 'Batch ID is required'));
        return;
    }
    
    try {
        $result = $this->dispensary_model->markBatchExpired($batch_id);
        
        if ($result) {
            echo json_encode(array('status' => 1, 'message' => 'Batch marked as expired successfully'));
        } else {
            echo json_encode(array('status' => 0, 'message' => 'Failed to mark batch as expired'));
        }
    } catch (Exception $e) {
        log_message('error', 'Mark batch expired error: ' . $e->getMessage());
        echo json_encode(array('status' => 0, 'message' => 'Error marking batch as expired'));
    }
}

/**
 * Export Stock Data
 */
public function exportStock() {
    $format = $this->input->get('format') ?: 'excel';
    
    try {
        $this->dispensary_model->exportStock($format);
    } catch (Exception $e) {
        log_message('error', 'Export stock error: ' . $e->getMessage());
        show_error('Error exporting stock data: ' . $e->getMessage());
    }
}

/**
 * Get Stock Movement History
 */
public function getStockMovementHistory() {
    $medicine_id = $this->input->post('medicine_id');
    $limit = $this->input->post('limit') ?: 50;
    
    try {
        $movements = $this->dispensary_model->getStockMovementHistory($medicine_id, $limit);
        echo json_encode(array('status' => 1, 'movements' => $movements));
    } catch (Exception $e) {
        log_message('error', 'Get stock movement history error: ' . $e->getMessage());
        echo json_encode(array('status' => 0, 'message' => 'Error loading stock movement history'));
    }
}

/**
 * Get Dashboard Analytics
 */
public function getDashboardAnalytics() {
    try {
        $analytics = $this->dispensary_model->getDashboardAnalytics();
        echo json_encode(array('status' => 1, 'analytics' => $analytics));
    } catch (Exception $e) {
        log_message('error', 'Get dashboard analytics error: ' . $e->getMessage());
        echo json_encode(array('status' => 0, 'message' => 'Error loading dashboard analytics'));
    }
}

/**
 * Generate Medicine Details HTML for Modal
 */
private function generateMedicineDetailsHTML($medicine) {
    $html = '<div class="medicine-details">';
    $html .= '<div class="row">';
    $html .= '<div class="col-md-6">';
    $html .= '<h5><i class="fa fa-medkit"></i> ' . $medicine->medicine_name . '</h5>';
    $html .= '<table class="table table-bordered">';
    $html .= '<tr><th width="40%">Company:</th><td>' . ($medicine->medicine_company ?: 'N/A') . '</td></tr>';
    $html .= '<tr><th>Category:</th><td>' . ($medicine->medicine_category ?: 'N/A') . '</td></tr>';
    $html .= '<tr><th>Unit:</th><td>' . ($medicine->unit ?: 'N/A') . '</td></tr>';
    $html .= '<tr><th>Total Available:</th><td><span class="badge bg-blue">' . ($medicine->total_available ?: 0) . '</span></td></tr>';
    $html .= '</table>';
    $html .= '</div>';
    $html .= '<div class="col-md-6">';
    
    if (!empty($medicine->batches)) {
        $html .= '<h6>Available Batches:</h6>';
        $html .= '<div class="table-responsive" style="max-height: 300px; overflow-y: auto;">';
        $html .= '<table class="table table-condensed table-striped">';
        $html .= '<thead><tr><th>Batch</th><th>Expiry</th><th>Stock</th><th>Price</th></tr></thead>';
        $html .= '<tbody>';
        
        foreach ($medicine->batches as $batch) {
            $expiry_class = '';
            if ($batch->expiry) {
                $expiry_date = new DateTime($batch->expiry);
                $today = new DateTime();
                $diff = $today->diff($expiry_date);
                
                if ($expiry_date < $today) {
                    $expiry_class = 'text-danger';
                } elseif ($diff->days <= 30) {
                    $expiry_class = 'text-warning';
                }
            }
            
            $html .= '<tr>';
            $html .= '<td>' . $batch->batch_no . '</td>';
            $html .= '<td class="' . $expiry_class . '">' . ($batch->expiry ? date('d/m/Y', strtotime($batch->expiry)) : 'N/A') . '</td>';
            $html .= '<td>' . $batch->available_quantity . '</td>';
            $html .= '<td>' . ($this->customlib->getHospitalCurrencyFormat() . number_format($batch->sale_rate, 2)) . '</td>';
            $html .= '</tr>';
        }
        
        $html .= '</tbody></table>';
        $html .= '</div>';
    } else {
        $html .= '<div class="alert alert-warning"><i class="fa fa-warning"></i> No batches available</div>';
    }
    
    $html .= '</div></div></div>';
    
    return $html;
}

/**
 * Generate Stock Adjustment HTML for Modal
 */
private function generateStockAdjustmentHTML($data) {
    $medicine = $data['medicine'];
    $adjustment_types = $data['adjustment_types'];
    
    $html = '<form id="stockAdjustmentForm">';
    $html .= '<div class="row">';
    $html .= '<div class="col-md-12">';
    $html .= '<h5><i class="fa fa-medkit"></i> ' . $medicine->medicine_name . '</h5>';
    $html .= '<p class="text-muted">' . ($medicine->medicine_company ?: 'N/A') . '</p>';
    $html .= '</div>';
    $html .= '</div>';
    
    $html .= '<div class="row">';
    $html .= '<div class="col-md-6">';
    $html .= '<div class="form-group">';
    $html .= '<label for="adjustmentBatch">Select Batch <span class="text-red">*</span></label>';
    $html .= '<select class="form-control" id="adjustmentBatch" name="batch_id" required onchange="updateCurrentStock()">';
    $html .= '<option value="">-- Select Batch --</option>';
    
    foreach ($medicine->batches as $batch) {
        $html .= '<option value="' . $batch->batch_id . '" data-stock="' . $batch->available_quantity . '">';
        $html .= 'Batch: ' . $batch->batch_no . ' | Stock: ' . $batch->available_quantity . ' | Expiry: ' . date('m/Y', strtotime($batch->expiry));
        $html .= '</option>';
    }
    
    $html .= '</select>';
    $html .= '</div>';
    $html .= '</div>';
    
    $html .= '<div class="col-md-6">';
    $html .= '<div class="form-group">';
    $html .= '<label for="adjustmentType">Adjustment Type <span class="text-red">*</span></label>';
    $html .= '<select class="form-control" id="adjustmentType" name="adjustment_type" required>';
    
    foreach ($adjustment_types as $key => $label) {
        $html .= '<option value="' . $key . '">' . $label . '</option>';
    }
    
    $html .= '</select>';
    $html .= '</div>';
    $html .= '</div>';
    $html .= '</div>';
    
    $html .= '<div class="row">';
    $html .= '<div class="col-md-6">';
    $html .= '<div class="form-group">';
    $html .= '<label>Current Stock</label>';
    $html .= '<input type="text" class="form-control" id="currentStock" readonly>';
    $html .= '</div>';
    $html .= '</div>';
    
    $html .= '<div class="col-md-6">';
    $html .= '<div class="form-group">';
    $html .= '<label for="adjustmentQuantity">Quantity <span class="text-red">*</span></label>';
    $html .= '<input type="number" class="form-control" id="adjustmentQuantity" name="quantity" min="1" required>';
    $html .= '</div>';
    $html .= '</div>';
    $html .= '</div>';
    
    $html .= '<div class="row">';
    $html .= '<div class="col-md-12">';
    $html .= '<div class="form-group">';
    $html .= '<label for="adjustmentReason">Reason</label>';
    $html .= '<textarea class="form-control" id="adjustmentReason" name="reason" rows="3" placeholder="Enter reason for stock adjustment..."></textarea>';
    $html .= '</div>';
    $html .= '</div>';
    $html .= '</div>';
    
    $html .= '<div class="row">';
    $html .= '<div class="col-md-6">';
    $html .= '<button type="button" class="btn btn-default btn-block" data-dismiss="modal">Cancel</button>';
    $html .= '</div>';
    $html .= '<div class="col-md-6">';
    $html .= '<button type="submit" class="btn btn-success btn-block">Adjust Stock</button>';
    $html .= '</div>';
    $html .= '</div>';
    
    $html .= '</form>';
    
    $html .= '<script>';
    $html .= 'function updateCurrentStock() {';
    $html .= '    var selectedOption = $("#adjustmentBatch option:selected");';
    $html .= '    var stock = selectedOption.data("stock");';
    $html .= '    $("#currentStock").val(stock || 0);';
    $html .= '}';
    
    $html .= '$("#stockAdjustmentForm").off("submit").on("submit", function(e) {';
    $html .= '    e.preventDefault();';
    $html .= '    var formData = $(this).serialize();';
    $html .= '    $.ajax({';
    $html .= '        url: "' . base_url('admin/dispensary/processStockAdjustment') . '",';
    $html .= '        type: "POST",';
    $html .= '        data: formData,';
    $html .= '        dataType: "json",';
    $html .= '        success: function(response) {';
    $html .= '            if (response.status === 1) {';
    $html .= '                showAlert("Stock adjusted successfully", "success");';
    $html .= '                $("#stockAdjustmentModal").modal("hide");';
    $html .= '                setTimeout(function() { location.reload(); }, 1500);';
    $html .= '            } else {';
    $html .= '                showAlert("Error: " + response.message, "danger");';
    $html .= '            }';
    $html .= '        },';
    $html .= '        error: function() {';
    $html .= '            showAlert("Error processing stock adjustment", "danger");';
    $html .= '        }';
    $html .= '    });';
    $html .= '});';
    $html .= '</script>';
    
    return $html;
}


    /**
     * Dispensary Settings
     */
    public function settings() {
        $this->session->set_userdata('top_menu', 'Medical Dispensary');
        $this->session->set_userdata('sub_menu', 'dispensary/settings');

        $data['title'] = 'Dispensary Settings';
        
        try {
            $data['medicine_categories'] = $this->dispensary_model->getMedicineCategories();
            $data['pharmacy_units'] = $this->dispensary_model->getPharmacyUnits();
            
            $this->load->view('layout/header', $data);
            $this->load->view('admin/dispensary/settings', $data);
            $this->load->view('layout/footer', $data);
            
        } catch (Exception $e) {
            log_message('error', 'Dispensary settings error: ' . $e->getMessage());
            show_error('Failed to load settings: ' . $e->getMessage());
        }
    }

    /**
     * Get Bill History for Patient
     */
public function getPatientBillHistory() {
    header('Content-Type: application/json');
    
    try {
        $patient_id = $this->input->post('patient_id');
        $limit = $this->input->post('limit') ?: 20;
        
        if (empty($patient_id)) {
            echo json_encode([
                'status' => 0,
                'message' => 'Patient ID is required'
            ]);
            return;
        }
        
        log_message('info', 'Loading bill history for patient ID: ' . $patient_id);
        
        // Get patient bills from your pharmacy_bill_basic table
        $bills = $this->getPatientBillsFromDatabase($patient_id, $limit);
        
        if (!empty($bills)) {
            echo json_encode([
                'status' => 1,
                'bills' => $bills,
                'total_found' => count($bills),
                'message' => 'Bill history loaded successfully'
            ]);
        } else {
            echo json_encode([
                'status' => 0,
                'bills' => [],
                'message' => 'No billing history found for this patient'
            ]);
        }
        
    } catch (Exception $e) {
        log_message('error', 'Get patient bill history error: ' . $e->getMessage());
        echo json_encode([
            'status' => 0,
            'message' => 'Error loading bill history: ' . $e->getMessage()
        ]);
    }
}

/**
 * Get patient bills from database
 */
private function getPatientBillsFromDatabase($patient_id, $limit) {
    try {
        // Check what bill table exists in your system
        if ($this->db->table_exists('pharmacy_bill_basic')) {
            return $this->getBillsFromPharmacyBillBasic($patient_id, $limit);
        } elseif ($this->db->table_exists('pharmacy_bills')) {
            return $this->getBillsFromPharmacyBills($patient_id, $limit);
        } else {
            log_message('info', 'No pharmacy bill tables found');
            return [];
        }
    } catch (Exception $e) {
        log_message('error', 'Error getting patient bills: ' . $e->getMessage());
        return [];
    }
}

/**
 * Get bills from pharmacy_bill_basic table
 */
private function getBillsFromPharmacyBillBasic($patient_id, $limit) {
    try {
        $this->db->select('
            pbb.id,
            pbb.date as bill_date,
            pbb.total as subtotal,
            pbb.discount,
            pbb.tax,
            pbb.net_amount,
            pbb.payment_status as status,
            pbb.note,
            pbb.created_at,
            p.patient_name,
            p.patient_unique_id,
            s.name as generated_by_name
        ');
        
        $this->db->from('pharmacy_bill_basic pbb');
        $this->db->join('patients p', 'p.id = pbb.patient_id', 'left');
        $this->db->join('staff s', 's.id = pbb.generated_by', 'left');
        $this->db->where('pbb.patient_id', $patient_id);
        $this->db->order_by('pbb.date', 'DESC');
        $this->db->limit($limit);
        
        $bills = $this->db->get()->result_array();
        
        // Format bills and add bill numbers
        foreach ($bills as &$bill) {
            $bill['bill_no'] = 'MDB' . str_pad($bill['id'], 6, '0', STR_PAD_LEFT);
            $bill['bill_date_formatted'] = date('M j, Y', strtotime($bill['bill_date']));
            $bill['status_label'] = ucfirst($bill['status']);
            
            // Get bill items for this bill
            $bill['items'] = $this->getBillItems($bill['id'], 'pharmacy_bill_basic');
            $bill['total_items'] = count($bill['items']);
        }
        
        return $bills;
        
    } catch (Exception $e) {
        log_message('error', 'Error getting bills from pharmacy_bill_basic: ' . $e->getMessage());
        return [];
    }
}

/**
 * Get bills from pharmacy_bills table (alternative structure)
 */
private function getBillsFromPharmacyBills($patient_id, $limit) {
    try {
        $this->db->select('
            pb.id,
            pb.bill_date,
            pb.subtotal,
            pb.discount_amount,
            pb.tax_amount,
            pb.net_amount,
            pb.status,
            pb.notes,
            pb.created_at,
            p.patient_name,
            p.patient_unique_id
        ');
        
        $this->db->from('pharmacy_bills pb');
        $this->db->join('patients p', 'p.id = pb.patient_id', 'left');
        $this->db->where('pb.patient_id', $patient_id);
        $this->db->order_by('pb.bill_date', 'DESC');
        $this->db->limit($limit);
        
        $bills = $this->db->get()->result_array();
        
        // Format bills
        foreach ($bills as &$bill) {
            $bill['bill_no'] = 'PB' . str_pad($bill['id'], 6, '0', STR_PAD_LEFT);
            $bill['bill_date_formatted'] = date('M j, Y', strtotime($bill['bill_date']));
            $bill['status_label'] = ucfirst($bill['status']);
            
            // Get bill items
            $bill['items'] = $this->getBillItems($bill['id'], 'pharmacy_bills');
            $bill['total_items'] = count($bill['items']);
        }
        
        return $bills;
        
    } catch (Exception $e) {
        log_message('error', 'Error getting bills from pharmacy_bills: ' . $e->getMessage());
        return [];
    }
}

/**
 * Get bill items for a specific bill
 */
private function getBillItems($bill_id, $bill_table) {
    try {
        $items = [];
        
        if ($bill_table === 'pharmacy_bill_basic') {
            // Get items from pharmacy_bill_detail
            if ($this->db->table_exists('pharmacy_bill_detail')) {
                $this->db->select('
                    pbd.quantity,
                    pbd.sale_price,
                    pbd.amount,
                    m.medicine_name,
                    mbd.batch_no
                ');
                
                $this->db->from('pharmacy_bill_detail pbd');
                
                // Join with medicine tables
                if ($this->db->table_exists('medicine_batch_details')) {
                    $this->db->join('medicine_batch_details mbd', 'mbd.id = pbd.medicine_batch_detail_id', 'left');
                    
                    if ($this->db->table_exists('medicines')) {
                        $this->db->join('medicines m', 'm.id = mbd.medicine_id', 'left');
                    } elseif ($this->db->table_exists('pharmacy')) {
                        $this->db->join('pharmacy m', 'm.id = mbd.medicine_id', 'left');
                    }
                }
                
                $this->db->where('pbd.pharmacy_bill_basic_id', $bill_id);
                $items = $this->db->get()->result_array();
            }
        } elseif ($bill_table === 'pharmacy_bills') {
            // Get items from pharmacy_bill_items
            if ($this->db->table_exists('pharmacy_bill_items')) {
                $this->db->select('
                    pbi.quantity,
                    pbi.unit_price,
                    pbi.total_price,
                    pm.medicine_name,
                    ps.batch_no
                ');
                
                $this->db->from('pharmacy_bill_items pbi');
                $this->db->join('pharmacy_medicines pm', 'pm.id = pbi.medicine_id', 'left');
                $this->db->join('pharmacy_stock ps', 'ps.id = pbi.batch_id', 'left');
                $this->db->where('pbi.bill_id', $bill_id);
                
                $items = $this->db->get()->result_array();
            }
        }
        
        return $items;
        
    } catch (Exception $e) {
        log_message('error', 'Error getting bill items: ' . $e->getMessage());
        return [];
    }
}

/**
 * Get bill details for modal display
 */
public function getBillDetails() {
    header('Content-Type: application/json');
    
    try {
        $bill_id = $this->input->post('bill_id');
        
        if (empty($bill_id)) {
            echo json_encode([
                'status' => 0,
                'message' => 'Bill ID is required'
            ]);
            return;
        }
        
        // Get bill details
        $bill_details = $this->getBillDetailsFromDatabase($bill_id);
        
        if ($bill_details) {
            echo json_encode([
                'status' => 1,
                'bill' => $bill_details,
                'message' => 'Bill details loaded successfully'
            ]);
        } else {
            echo json_encode([
                'status' => 0,
                'message' => 'Bill not found'
            ]);
        }
        
    } catch (Exception $e) {
        log_message('error', 'Get bill details error: ' . $e->getMessage());
        echo json_encode([
            'status' => 0,
            'message' => 'Error loading bill details: ' . $e->getMessage()
        ]);
    }
}

/**
 * Get bill details from database
 */
private function getBillDetailsFromDatabase($bill_id) {
    try {
        $bill = null;
        
        // Try pharmacy_bill_basic first
        if ($this->db->table_exists('pharmacy_bill_basic')) {
            $this->db->select('
                pbb.*,
                p.patient_name,
                p.patient_unique_id,
                p.age,
                p.gender,
                s.name as generated_by_name
            ');
            $this->db->from('pharmacy_bill_basic pbb');
            $this->db->join('patients p', 'p.id = pbb.patient_id', 'left');
            $this->db->join('staff s', 's.id = pbb.generated_by', 'left');
            $this->db->where('pbb.id', $bill_id);
            
            $bill = $this->db->get()->row_array();
        }
        
        if ($bill) {
            // Format bill
            $bill['bill_no'] = 'MDB' . str_pad($bill['id'], 6, '0', STR_PAD_LEFT);
            $bill['bill_date_formatted'] = date('M j, Y g:i A', strtotime($bill['date']));
            $bill['status_label'] = ucfirst($bill['payment_status']);
            
            // Get detailed items
            $bill['items'] = $this->getBillItems($bill_id, 'pharmacy_bill_basic');
            
            return $bill;
        }
        
        return null;
        
    } catch (Exception $e) {
        log_message('error', 'Error getting bill details: ' . $e->getMessage());
        return null;
    }
}

    /**
     * Check medicine availability and expiry
     */
    public function checkMedicineAvailability() {
        $medicine_id = $this->input->post('medicine_id');
        $quantity = $this->input->post('quantity');
        
        if (empty($medicine_id) || empty($quantity)) {
            echo json_encode(array('status' => 0, 'message' => 'Medicine ID and quantity are required'));
            return;
        }
        
        try {
            $availability = $this->dispensary_model->checkMedicineAvailability($medicine_id, $quantity);
            echo json_encode(array('status' => 1, 'availability' => $availability));
        } catch (Exception $e) {
            log_message('error', 'Check medicine availability error: ' . $e->getMessage());
            echo json_encode(array('status' => 0, 'message' => 'Availability check failed'));
        }
    }
    
/**
 * Get Sales Report Data for Cards
 */
public function getSalesReport() {
    $date_from = $this->input->post('date_from') ?: date('Y-m-01');
    $date_to = $this->input->post('date_to') ?: date('Y-m-d');
    $medicine_category = $this->input->post('medicine_category');
    
    try {
        // Get sales summary
        $this->db->select('
            COUNT(*) as total_bills,
            COALESCE(SUM(net_amount), 0) as total_revenue,
            COUNT(DISTINCT patient_id) as unique_patients
        ');
        $this->db->from('pharmacy_bill_basic');
        $this->db->where('DATE(date) >=', $date_from);
        $this->db->where('DATE(date) <=', $date_to);
        $this->db->where('payment_status !=', 'cancelled');
        
        $sales_summary = $this->db->get()->row();
        
        // Get medicines sold count
        $this->db->select('COALESCE(SUM(pbd.quantity), 0) as medicines_sold, COALESCE(SUM(pbd.quantity), 0) as total_units');
        $this->db->from('pharmacy_bill_detail pbd');
        $this->db->join('pharmacy_bill_basic pbb', 'pbb.id = pbd.pharmacy_bill_basic_id');
        $this->db->where('DATE(pbb.date) >=', $date_from);
        $this->db->where('DATE(pbb.date) <=', $date_to);
        $this->db->where('pbb.payment_status !=', 'cancelled');
        
        $medicines_data = $this->db->get()->row();
        
        // Calculate growth percentage (compare with previous period)
        $period_days = (strtotime($date_to) - strtotime($date_from)) / (60 * 60 * 24);
        $prev_date_from = date('Y-m-d', strtotime($date_from . " -{$period_days} days"));
        $prev_date_to = date('Y-m-d', strtotime($date_to . " -{$period_days} days"));
        
        $this->db->select('COALESCE(SUM(net_amount), 0) as prev_revenue');
        $this->db->from('pharmacy_bill_basic');
        $this->db->where('DATE(date) >=', $prev_date_from);
        $this->db->where('DATE(date) <=', $prev_date_to);
        $this->db->where('payment_status !=', 'cancelled');
        
        $prev_data = $this->db->get()->row();
        
        $growth_percentage = 0;
        if ($prev_data && $prev_data->prev_revenue > 0) {
            $growth_percentage = (($sales_summary->total_revenue - $prev_data->prev_revenue) / $prev_data->prev_revenue) * 100;
        }
        
        $data = array(
            'total_bills' => $sales_summary->total_bills,
            'total_revenue' => $sales_summary->total_revenue,
            'unique_patients' => $sales_summary->unique_patients,
            'medicines_sold' => $medicines_data->medicines_sold,
            'total_units' => $medicines_data->total_units,
            'growth_percentage' => round($growth_percentage, 1)
        );
        
        echo json_encode(array('status' => 1, 'data' => $data));
        
    } catch (Exception $e) {
        log_message('error', 'Sales report error: ' . $e->getMessage());
        echo json_encode(array('status' => 0, 'message' => 'Error generating sales report'));
    }
}

/**
 * Get Detailed Report Data
 */
public function getDetailedReport() {
    $date_from = $this->input->post('date_from') ?: date('Y-m-01');
    $date_to = $this->input->post('date_to') ?: date('Y-m-d');
    $medicine_category = $this->input->post('medicine_category');
    $report_type = $this->input->post('report_type') ?: 'summary';
    
    try {
        $this->db->select('
            pbb.id,
            pbb.date as bill_date,
            pbb.total as subtotal,
            pbb.discount as discount_amount,
            pbb.tax as tax_amount,
            pbb.net_amount,
            pbb.payment_status as status,
            p.patient_name,
            p.patient_unique_id,
            COUNT(pbd.id) as item_count
        ');
        $this->db->from('pharmacy_bill_basic pbb');
        $this->db->join('patients p', 'p.id = pbb.patient_id', 'left');
        $this->db->join('pharmacy_bill_detail pbd', 'pbd.pharmacy_bill_basic_id = pbb.id', 'left');
        $this->db->where('DATE(pbb.date) >=', $date_from);
        $this->db->where('DATE(pbb.date) <=', $date_to);
        $this->db->where('pbb.payment_status !=', 'cancelled');
        
        if ($medicine_category) {
            $this->db->join('medicine_batch_details mbd', 'mbd.id = pbd.medicine_batch_detail_id');
            $this->db->join('medicines m', 'm.id = mbd.medicine_id');
            $this->db->where('m.medicine_category_id', $medicine_category);
        }
        
        $this->db->group_by('pbb.id');
        $this->db->order_by('pbb.date', 'DESC');
        $this->db->limit(100); // Limit for performance
        
        $bills = $this->db->get()->result();
        
        // Add bill numbers and payment method
        foreach ($bills as $key => $bill) {
            $bills[$key]->bill_no = 'MDB' . str_pad($bill->id, 6, '0', STR_PAD_LEFT);
            $bills[$key]->payment_method = 'Cash'; // Default since we don't have payment method in bill_basic
        }
        
        echo json_encode(array('status' => 1, 'bills' => $bills));
        
    } catch (Exception $e) {
        log_message('error', 'Detailed report error: ' . $e->getMessage());
        echo json_encode(array('status' => 0, 'message' => 'Error generating detailed report'));
    }
}

/**
 * Get Top Medicines
 */
public function getTopMedicines() {
    $date_from = $this->input->post('date_from') ?: date('Y-m-01');
    $date_to = $this->input->post('date_to') ?: date('Y-m-d');
    $limit = $this->input->post('limit') ?: 10;
    
    try {
        $this->db->select('
            m.medicine_name,
            m.medicine_company,
            SUM(pbd.quantity) as total_sold,
            SUM(pbd.amount) as total_revenue,
            COUNT(DISTINCT pbb.id) as bill_count
        ');
        $this->db->from('pharmacy_bill_detail pbd');
        $this->db->join('pharmacy_bill_basic pbb', 'pbb.id = pbd.pharmacy_bill_basic_id');
        $this->db->join('medicine_batch_details mbd', 'mbd.id = pbd.medicine_batch_detail_id');
        $this->db->join('medicines m', 'm.id = mbd.medicine_id');
        $this->db->where('DATE(pbb.date) >=', $date_from);
        $this->db->where('DATE(pbb.date) <=', $date_to);
        $this->db->where('pbb.payment_status !=', 'cancelled');
        $this->db->group_by('m.id');
        $this->db->order_by('total_sold', 'DESC');
        $this->db->limit($limit);
        
        $medicines = $this->db->get()->result();
        
        echo json_encode(array('status' => 1, 'medicines' => $medicines));
        
    } catch (Exception $e) {
        log_message('error', 'Top medicines error: ' . $e->getMessage());
        echo json_encode(array('status' => 0, 'message' => 'Error loading top medicines'));
    }
}

/**
 * Get Critical Stock Alerts
 */
public function getCriticalStock() {
    try {
        $this->db->select('
            m.medicine_name,
            m.medicine_company,
            SUM(mbd.available_quantity) as stock
        ');
        $this->db->from('medicines m');
        $this->db->join('medicine_batch_details mbd', 'm.id = mbd.medicine_id', 'left');
        $this->db->where('m.is_active', 'yes');
        $this->db->group_by('m.id');
        $this->db->having('stock < 50 OR stock IS NULL');
        $this->db->order_by('stock', 'ASC');
        $this->db->limit(20);
        
        $items = $this->db->get()->result();
        
        echo json_encode(array('status' => 1, 'items' => $items));
        
    } catch (Exception $e) {
        log_message('error', 'Critical stock error: ' . $e->getMessage());
        echo json_encode(array('status' => 0, 'message' => 'Error loading critical stock'));
    }
}

/**
 * Get Expiry Alerts
 */
public function getExpiryAlerts() {
    $days = $this->input->post('days') ?: 90;
    
    try {
        $expiry_date = date('Y-m-d', strtotime("+{$days} days"));
        
        $this->db->select('
            m.medicine_name,
            m.medicine_company,
            mbd.batch_no,
            mbd.expiry,
            mbd.available_quantity
        ');
        $this->db->from('medicine_batch_details mbd');
        $this->db->join('medicines m', 'm.id = mbd.medicine_id');
        $this->db->where('mbd.expiry <=', $expiry_date);
        $this->db->where('mbd.expiry >', date('Y-m-d'));
        $this->db->where('mbd.available_quantity >', 0);
        $this->db->where('m.is_active', 'yes');
        $this->db->order_by('mbd.expiry', 'ASC');
        $this->db->limit(20);
        
        $items = $this->db->get()->result();
        
        echo json_encode(array('status' => 1, 'items' => $items));
        
    } catch (Exception $e) {
        log_message('error', 'Expiry alerts error: ' . $e->getMessage());
        echo json_encode(array('status' => 0, 'message' => 'Error loading expiry alerts'));
    }
}

/**
 * Get Financial Summary
 */
public function getFinancialSummary() {
    $date_from = $this->input->post('date_from') ?: date('Y-m-01');
    $date_to = $this->input->post('date_to') ?: date('Y-m-d');
    
    try {
        $this->db->select('
            COALESCE(SUM(total), 0) as gross_revenue,
            COALESCE(SUM(discount), 0) as total_discounts,
            COALESCE(SUM(tax), 0) as total_taxes,
            COALESCE(SUM(net_amount), 0) as net_revenue,
            COALESCE(AVG(net_amount), 0) as avg_bill_value,
            COUNT(*) as total_bills
        ');
        $this->db->from('pharmacy_bill_basic');
        $this->db->where('DATE(date) >=', $date_from);
        $this->db->where('DATE(date) <=', $date_to);
        $this->db->where('payment_status !=', 'cancelled');
        
        $data = $this->db->get()->row();
        
        // Add payment breakdown (simplified since we don't have payment method data)
        $data->payment_breakdown = 'Cash: 100%';
        
        echo json_encode(array('status' => 1, 'data' => $data));
        
    } catch (Exception $e) {
        log_message('error', 'Financial summary error: ' . $e->getMessage());
        echo json_encode(array('status' => 0, 'message' => 'Error loading financial summary'));
    }
}

/**
 * Get Sales Trend Data
 */
public function getSalesTrend() {
    $date_from = $this->input->post('date_from') ?: date('Y-m-01');
    $date_to = $this->input->post('date_to') ?: date('Y-m-d');
    
    try {
        $this->db->select('
            DATE(date) as sale_date,
            COALESCE(SUM(net_amount), 0) as daily_sales
        ');
        $this->db->from('pharmacy_bill_basic');
        $this->db->where('DATE(date) >=', $date_from);
        $this->db->where('DATE(date) <=', $date_to);
        $this->db->where('payment_status !=', 'cancelled');
        $this->db->group_by('DATE(date)');
        $this->db->order_by('DATE(date)', 'ASC');
        
        $results = $this->db->get()->result();
        
        $labels = array();
        $data = array();
        
        foreach ($results as $result) {
            $labels[] = date('M d', strtotime($result->sale_date));
            $data[] = floatval($result->daily_sales);
        }
        
        echo json_encode(array(
            'status' => 1, 
            'labels' => $labels, 
            'data' => $data
        ));
        
    } catch (Exception $e) {
        log_message('error', 'Sales trend error: ' . $e->getMessage());
        echo json_encode(array('status' => 0, 'message' => 'Error loading sales trend'));
    }
}

/**
 * Get Chart Data
 */
public function getChartData() {
    $date_from = $this->input->post('date_from') ?: date('Y-m-01');
    $date_to = $this->input->post('date_to') ?: date('Y-m-d');
    $chart_type = $this->input->post('chart_type') ?: 'categories';
    
    try {
        $labels = array();
        $data = array();
        
        switch ($chart_type) {
            case 'categories':
                $this->db->select('
                    mc.medicine_category as category_name,
                    COALESCE(SUM(pbd.amount), 0) as total_sales
                ');
                $this->db->from('pharmacy_bill_detail pbd');
                $this->db->join('pharmacy_bill_basic pbb', 'pbb.id = pbd.pharmacy_bill_basic_id');
                $this->db->join('medicine_batch_details mbd', 'mbd.id = pbd.medicine_batch_detail_id');
                $this->db->join('medicines m', 'm.id = mbd.medicine_id');
                $this->db->join('medicine_category mc', 'mc.id = m.medicine_category_id', 'left');
                $this->db->where('DATE(pbb.date) >=', $date_from);
                $this->db->where('DATE(pbb.date) <=', $date_to);
                $this->db->where('pbb.payment_status !=', 'cancelled');
                $this->db->group_by('mc.id');
                $this->db->order_by('total_sales', 'DESC');
                
                $results = $this->db->get()->result();
                
                foreach ($results as $result) {
                    $labels[] = $result->category_name ?: 'Uncategorized';
                    $data[] = floatval($result->total_sales);
                }
                break;
                
            case 'revenue':
                $this->db->select('
                    DATE(date) as sale_date,
                    COALESCE(SUM(net_amount), 0) as revenue
                ');
                $this->db->from('pharmacy_bill_basic');
                $this->db->where('DATE(date) >=', $date_from);
                $this->db->where('DATE(date) <=', $date_to);
                $this->db->where('payment_status !=', 'cancelled');
                $this->db->group_by('DATE(date)');
                $this->db->order_by('DATE(date)', 'ASC');
                
                $results = $this->db->get()->result();
                
                foreach ($results as $result) {
                    $labels[] = date('M d', strtotime($result->sale_date));
                    $data[] = floatval($result->revenue);
                }
                break;
                
            case 'profit':
                // Simplified profit calculation (revenue - discounts)
                $this->db->select('
                    DATE(date) as sale_date,
                    COALESCE(SUM(net_amount - discount), 0) as profit
                ');
                $this->db->from('pharmacy_bill_basic');
                $this->db->where('DATE(date) >=', $date_from);
                $this->db->where('DATE(date) <=', $date_to);
                $this->db->where('payment_status !=', 'cancelled');
                $this->db->group_by('DATE(date)');
                $this->db->order_by('DATE(date)', 'ASC');
                
                $results = $this->db->get()->result();
                
                foreach ($results as $result) {
                    $labels[] = date('M d', strtotime($result->sale_date));
                    $data[] = floatval($result->profit);
                }
                break;
        }
        
        echo json_encode(array(
            'status' => 1, 
            'labels' => $labels, 
            'data' => $data
        ));
        
    } catch (Exception $e) {
        log_message('error', 'Chart data error: ' . $e->getMessage());
        echo json_encode(array('status' => 0, 'message' => 'Error loading chart data'));
    }
}

/**
 * Export Report
 */
public function exportReport() {
    $date_from = $this->input->get('date_from') ?: date('Y-m-01');
    $date_to = $this->input->get('date_to') ?: date('Y-m-d');
    $format = $this->input->get('export_format') ?: 'pdf';
    $medicine_category = $this->input->get('medicine_category');
    
    try {
        $data = $this->dispensary_model->getDetailedReportData($date_from, $date_to, $medicine_category);
        
        if ($format === 'csv') {
            $this->exportCSV($data, $date_from, $date_to);
        } elseif ($format === 'excel') {
            $this->exportExcel($data, $date_from, $date_to);
        } else {
            $this->exportPDF($data, $date_from, $date_to);
        }
    } catch (Exception $e) {
        log_message('error', 'Export error: ' . $e->getMessage());
        show_error('Error exporting report: ' . $e->getMessage());
    }
}

/**
 * Get System Information
 */
public function getSystemInfo() {
    try {
        $info = array(
            'total_medicines' => $this->dispensary_model->countMedicines(),
            'active_categories' => $this->dispensary_model->countActiveCategories(),
            'todays_bills' => $this->dispensary_model->getTodaysBillsCount(),
            'db_size' => $this->dispensary_model->getDatabaseSize(),
            'cache_size' => $this->dispensary_model->getCacheSize(),
            'last_backup' => $this->dispensary_model->getLastBackupDate()
        );
        
        echo json_encode(array('status' => 1, 'data' => $info));
    } catch (Exception $e) {
        log_message('error', 'System info error: ' . $e->getMessage());
        echo json_encode(array('status' => 0, 'message' => 'Error loading system information'));
    }
}

/**
 * Get Settings
 */
public function getSettings() {
    try {
        $settings = $this->dispensary_model->getDispensarySettings();
        echo json_encode(array('status' => 1, 'settings' => $settings));
    } catch (Exception $e) {
        log_message('error', 'Get settings error: ' . $e->getMessage());
        echo json_encode(array('status' => 0, 'message' => 'Error loading settings'));
    }
}

/**
 * Save Settings
 */
public function saveSettings() {
    $settings = $this->input->post('settings');
    $type = $this->input->post('type');
    
    if (empty($settings) || empty($type)) {
        echo json_encode(array('status' => 0, 'message' => 'Settings data and type are required'));
        return;
    }
    
    try {
        $result = $this->dispensary_model->saveDispensarySettings($settings, $type);
        if ($result) {
            echo json_encode(array('status' => 1, 'message' => 'Settings saved successfully'));
        } else {
            echo json_encode(array('status' => 0, 'message' => 'Failed to save settings'));
        }
    } catch (Exception $e) {
        log_message('error', 'Save settings error: ' . $e->getMessage());
        echo json_encode(array('status' => 0, 'message' => 'Error saving settings'));
    }
}

/**
 * Save Category
 */
public function saveCategory() {
    $id = $this->input->post('id');
    $name = $this->input->post('name');
    $description = $this->input->post('description');
    
    if (empty($name)) {
        echo json_encode(array('status' => 0, 'message' => 'Category name is required'));
        return;
    }
    
    try {
        $data = array(
            'medicine_category' => $name,
            'description' => $description
        );
        
        if (empty($id)) {
            // Insert new category
            $result = $this->db->insert('medicine_category', $data);
        } else {
            // Update existing category
            $this->db->where('id', $id);
            $result = $this->db->update('medicine_category', $data);
        }
        
        if ($result) {
            echo json_encode(array('status' => 1, 'message' => 'Category saved successfully'));
        } else {
            echo json_encode(array('status' => 0, 'message' => 'Failed to save category'));
        }
    } catch (Exception $e) {
        log_message('error', 'Save category error: ' . $e->getMessage());
        echo json_encode(array('status' => 0, 'message' => 'Error saving category'));
    }
}

/**
 * Delete Category
 */
public function deleteCategory() {
    $id = $this->input->post('id');
    
    if (empty($id)) {
        echo json_encode(array('status' => 0, 'message' => 'Category ID is required'));
        return;
    }
    
    try {
        // Check if category is being used
        $this->db->where('medicine_category_id', $id);
        $medicines_count = $this->db->count_all_results('medicines');
        
        if ($medicines_count > 0) {
            echo json_encode(array('status' => 0, 'message' => 'Cannot delete category. It is being used by ' . $medicines_count . ' medicines.'));
            return;
        }
        
        $this->db->where('id', $id);
        $result = $this->db->delete('medicine_category');
        
        if ($result) {
            echo json_encode(array('status' => 1, 'message' => 'Category deleted successfully'));
        } else {
            echo json_encode(array('status' => 0, 'message' => 'Failed to delete category'));
        }
    } catch (Exception $e) {
        log_message('error', 'Delete category error: ' . $e->getMessage());
        echo json_encode(array('status' => 0, 'message' => 'Error deleting category'));
    }
}

/**
 * Save Unit
 */
public function saveUnit() {
    $id = $this->input->post('id');
    $unit_name = $this->input->post('unit_name');
    $abbreviation = $this->input->post('abbreviation');
    $unit_type = $this->input->post('unit_type');
    $description = $this->input->post('description');
    $is_active = $this->input->post('is_active');
    
    if (empty($unit_name)) {
        echo json_encode(array('status' => 0, 'message' => 'Unit name is required'));
        return;
    }
    
    try {
        $data = array(
            'unit_name' => $unit_name,
            'abbreviation' => $abbreviation,
            'unit_type' => $unit_type,
            'description' => $description,
            'is_active' => $is_active
        );
        
        if (empty($id)) {
            // Insert new unit
            $result = $this->db->insert('pharmacy_unit', $data);
        } else {
            // Update existing unit
            $data['updated_at'] = date('Y-m-d H:i:s');
            $this->db->where('id', $id);
            $result = $this->db->update('pharmacy_unit', $data);
        }
        
        if ($result) {
            echo json_encode(array('status' => 1, 'message' => 'Unit saved successfully'));
        } else {
            echo json_encode(array('status' => 0, 'message' => 'Failed to save unit'));
        }
    } catch (Exception $e) {
        log_message('error', 'Save unit error: ' . $e->getMessage());
        echo json_encode(array('status' => 0, 'message' => 'Error saving unit'));
    }
}

/**
 * Get Unit
 */
public function getUnit() {
    $id = $this->input->post('id');
    
    if (empty($id)) {
        echo json_encode(array('status' => 0, 'message' => 'Unit ID is required'));
        return;
    }
    
    try {
        $this->db->where('id', $id);
        $unit = $this->db->get('pharmacy_unit')->row();
        
        if ($unit) {
            echo json_encode(array('status' => 1, 'unit' => $unit));
        } else {
            echo json_encode(array('status' => 0, 'message' => 'Unit not found'));
        }
    } catch (Exception $e) {
        log_message('error', 'Get unit error: ' . $e->getMessage());
        echo json_encode(array('status' => 0, 'message' => 'Error loading unit'));
    }
}

/**
 * Delete Unit
 */
public function deleteUnit() {
    $id = $this->input->post('id');
    
    if (empty($id)) {
        echo json_encode(array('status' => 0, 'message' => 'Unit ID is required'));
        return;
    }
    
    try {
        // Check if unit is being used
        $this->db->where('unit', $id);
        $medicines_count = $this->db->count_all_results('medicines');
        
        if ($medicines_count > 0) {
            echo json_encode(array('status' => 0, 'message' => 'Cannot delete unit. It is being used by ' . $medicines_count . ' medicines.'));
            return;
        }
        
        $this->db->where('id', $id);
        $result = $this->db->delete('pharmacy_unit');
        
        if ($result) {
            echo json_encode(array('status' => 1, 'message' => 'Unit deleted successfully'));
        } else {
            echo json_encode(array('status' => 0, 'message' => 'Failed to delete unit'));
        }
    } catch (Exception $e) {
        log_message('error', 'Delete unit error: ' . $e->getMessage());
        echo json_encode(array('status' => 0, 'message' => 'Error deleting unit'));
    }
}

/**
 * Clear Cache
 */
public function clearCache() {
    try {
        // Clear CodeIgniter cache
        $this->load->driver('cache');
        $this->cache->clean();
        
        // Clear file cache if exists
        $cache_path = APPPATH . 'cache/';
        if (is_dir($cache_path)) {
            $files = glob($cache_path . '*');
            foreach ($files as $file) {
                if (is_file($file)) {
                    unlink($file);
                }
            }
        }
        
        echo json_encode(array('status' => 1, 'message' => 'Cache cleared successfully'));
    } catch (Exception $e) {
        log_message('error', 'Clear cache error: ' . $e->getMessage());
        echo json_encode(array('status' => 0, 'message' => 'Error clearing cache'));
    }
}

/**
 * Optimize Database
 */
public function optimizeDatabase() {
    try {
        $tables = array(
            'pharmacy_bills', 'pharmacy_bill_items', 'medicines', 
            'medicine_batch_details', 'medicine_category', 'pharmacy_unit'
        );
        
        foreach ($tables as $table) {
            $this->db->query("OPTIMIZE TABLE `{$table}`");
        }
        
        echo json_encode(array('status' => 1, 'message' => 'Database optimized successfully'));
    } catch (Exception $e) {
        log_message('error', 'Optimize database error: ' . $e->getMessage());
        echo json_encode(array('status' => 0, 'message' => 'Error optimizing database'));
    }
}

/**
 * Clean Temporary Files
 */
public function cleanTempFiles() {
    try {
        $temp_paths = array(
            APPPATH . 'logs/',
            APPPATH . 'cache/',
            FCPATH . 'uploads/temp/'
        );
        
        $deleted_count = 0;
        foreach ($temp_paths as $path) {
            if (is_dir($path)) {
                $files = glob($path . '*');
                foreach ($files as $file) {
                    if (is_file($file) && (time() - filemtime($file)) > (24 * 3600)) { // Older than 24 hours
                        if (unlink($file)) {
                            $deleted_count++;
                        }
                    }
                }
            }
        }
        
        echo json_encode(array('status' => 1, 'message' => "Cleaned {$deleted_count} temporary files"));
    } catch (Exception $e) {
        log_message('error', 'Clean temp files error: ' . $e->getMessage());
        echo json_encode(array('status' => 0, 'message' => 'Error cleaning temporary files'));
    }
}

/**
 * Rebuild Indexes
 */
public function rebuildIndexes() {
    try {
        $tables = array('pharmacy_bills', 'pharmacy_bill_items', 'medicines', 'medicine_batch_details');
        
        foreach ($tables as $table) {
            $this->db->query("ALTER TABLE `{$table}` DISABLE KEYS");
            $this->db->query("ALTER TABLE `{$table}` ENABLE KEYS");
        }
        
        echo json_encode(array('status' => 1, 'message' => 'Database indexes rebuilt successfully'));
    } catch (Exception $e) {
        log_message('error', 'Rebuild indexes error: ' . $e->getMessage());
        echo json_encode(array('status' => 0, 'message' => 'Error rebuilding indexes'));
    }
}

/**
 * Create Backup
 */
public function createBackup() {
    try {
        $this->load->dbutil();
        
        $tables = array(
            'pharmacy_bills', 'pharmacy_bill_items', 'medicines', 
            'medicine_batch_details', 'medicine_category', 'pharmacy_unit'
        );
        
        $backup = $this->dbutil->backup(array(
            'tables' => $tables,
            'format' => 'sql',
            'add_drop' => TRUE,
            'add_insert' => TRUE
        ));
        
        $filename = 'dispensary_backup_' . date('Y_m_d_H_i_s') . '.sql';
        $backup_path = FCPATH . 'uploads/backups/';
        
        if (!is_dir($backup_path)) {
            mkdir($backup_path, 0755, true);
        }
        
        $filepath = $backup_path . $filename;
        
        if (write_file($filepath, $backup)) {
            echo json_encode(array(
                'status' => 1, 
                'message' => 'Backup created successfully',
                'download_url' => base_url('uploads/backups/' . $filename)
            ));
        } else {
            echo json_encode(array('status' => 0, 'message' => 'Failed to create backup file'));
        }
    } catch (Exception $e) {
        log_message('error', 'Create backup error: ' . $e->getMessage());
        echo json_encode(array('status' => 0, 'message' => 'Error creating backup'));
    }
}

/**
 * Test Notifications
 */
public function testNotifications() {
    try {
        // Send test email if configured
        $settings = $this->dispensary_model->getDispensarySettings();
        
        if (!empty($settings['alert_email'])) {
            $this->load->library('email');
            
            $this->email->from('system@hospital.com', 'Medical Dispensary System');
            $this->email->to($settings['alert_email']);
            $this->email->subject('Test Notification - Medical Dispensary');
            $this->email->message('This is a test notification from your Medical Dispensary system. If you received this, notifications are working correctly.');
            
            if ($this->email->send()) {
                echo json_encode(array('status' => 1, 'message' => 'Test notification sent successfully'));
            } else {
                echo json_encode(array('status' => 0, 'message' => 'Failed to send test notification'));
            }
        } else {
            echo json_encode(array('status' => 0, 'message' => 'No email address configured for notifications'));
        }
    } catch (Exception $e) {
        log_message('error', 'Test notifications error: ' . $e->getMessage());
        echo json_encode(array('status' => 0, 'message' => 'Error sending test notification'));
    }
}

/**
 * Export Settings
 */
public function exportSettings() {
    try {
        $settings = $this->dispensary_model->getDispensarySettings();
        
        $filename = 'dispensary_settings_' . date('Y_m_d') . '.json';
        
        header('Content-Type: application/json');
        header('Content-Disposition: attachment; filename="' . $filename . '"');
        header('Content-Length: ' . strlen(json_encode($settings)));
        
        echo json_encode($settings, JSON_PRETTY_PRINT);
    } catch (Exception $e) {
        log_message('error', 'Export settings error: ' . $e->getMessage());
        show_error('Error exporting settings: ' . $e->getMessage());
    }
}

/**
 * Export to CSV
 */
private function exportCSV($data, $date_from, $date_to) {
    $filename = 'dispensary_report_' . $date_from . '_to_' . $date_to . '.csv';
    
    header('Content-Type: text/csv');
    header('Content-Disposition: attachment; filename="' . $filename . '"');
    
    $output = fopen('php://output', 'w');
    
    // CSV headers
    fputcsv($output, array(
        'Date', 'Bill No', 'Patient Name', 'Patient ID', 'Items Count', 
        'Subtotal', 'Discount', 'Tax', 'Net Amount', 'Payment Method', 'Status'
    ));
    
    // CSV data
    foreach ($data as $row) {
        fputcsv($output, array(
            date('d/m/Y', strtotime($row->bill_date)),
            $row->bill_no,
            $row->patient_name ?: 'Walk-in',
            $row->patient_unique_id ?: 'N/A',
            $row->item_count,
            $row->subtotal,
            $row->discount_amount,
            $row->tax_amount,
            $row->net_amount,
            $row->payment_method,
            $row->status
        ));
    }
    
    fclose($output);
}

/**
 * Export to Excel (simplified - you may want to use PHPSpreadsheet library)
 */
private function exportExcel($data, $date_from, $date_to) {
    // For a simple Excel export, we'll use CSV with .xls extension
    // For full Excel functionality, implement PHPSpreadsheet library
    
    $filename = 'dispensary_report_' . $date_from . '_to_' . $date_to . '.xls';
    
    header('Content-Type: application/vnd.ms-excel');
    header('Content-Disposition: attachment; filename="' . $filename . '"');
    
    echo "<table border='1'>";
    echo "<tr><th>Date</th><th>Bill No</th><th>Patient Name</th><th>Patient ID</th><th>Items Count</th><th>Subtotal</th><th>Discount</th><th>Tax</th><th>Net Amount</th><th>Payment Method</th><th>Status</th></tr>";
    
    foreach ($data as $row) {
        echo "<tr>";
        echo "<td>" . date('d/m/Y', strtotime($row->bill_date)) . "</td>";
        echo "<td>" . $row->bill_no . "</td>";
        echo "<td>" . ($row->patient_name ?: 'Walk-in') . "</td>";
        echo "<td>" . ($row->patient_unique_id ?: 'N/A') . "</td>";
        echo "<td>" . $row->item_count . "</td>";
        echo "<td>" . $row->subtotal . "</td>";
        echo "<td>" . $row->discount_amount . "</td>";
        echo "<td>" . $row->tax_amount . "</td>";
        echo "<td>" . $row->net_amount . "</td>";
        echo "<td>" . $row->payment_method . "</td>";
        echo "<td>" . $row->status . "</td>";
        echo "</tr>";
    }
    
    echo "</table>";
}

/**
 * Export to PDF (simplified - you may want to use TCPDF or similar library)
 */
private function exportPDF($data, $date_from, $date_to) {
    // For a simple PDF export, we'll generate HTML and use browser print
    // For full PDF functionality, implement TCPDF or similar library
    
    $this->load->library('customlib');
    
    $html = '<!DOCTYPE html>
    <html>
    <head>
        <title>Dispensary Report</title>
        <style>
            body { font-family: Arial, sans-serif; font-size: 12px; }
            table { width: 100%; border-collapse: collapse; margin-top: 20px; }
            th, td { border: 1px solid #ddd; padding: 8px; text-align: left; }
            th { background-color: #f2f2f2; }
            .header { text-align: center; margin-bottom: 30px; }
        </style>
    </head>
    <body>
        <div class="header">
            <h2>' . $this->customlib->getSchoolName() . '</h2>
            <h3>Medical Dispensary Report</h3>
            <p>Period: ' . date('d/m/Y', strtotime($date_from)) . ' to ' . date('d/m/Y', strtotime($date_to)) . '</p>
        </div>
        
        <table>
            <thead>
                <tr>
                    <th>Date</th>
                    <th>Bill No</th>
                    <th>Patient</th>
                    <th>Items</th>
                    <th>Net Amount</th>
                    <th>Status</th>
                </tr>
            </thead>
            <tbody>';
    
    foreach ($data as $row) {
        $html .= '<tr>
            <td>' . date('d/m/Y', strtotime($row->bill_date)) . '</td>
            <td>' . $row->bill_no . '</td>
            <td>' . ($row->patient_name ?: 'Walk-in') . '</td>
            <td>' . $row->item_count . '</td>
            <td>' . $this->customlib->getSchoolCurrencyFormat() . number_format($row->net_amount, 2) . '</td>
            <td>' . ucfirst($row->status) . '</td>
        </tr>';
    }
    
    $html .= '</tbody>
        </table>
        
        <script>
            window.onload = function() { window.print(); }
        </script>
    </body>
    </html>';
    
    echo $html;
}

/**
 * Test pharmacy financial integration
 * Usage: /admin/dispensary/test_pharmacy_financials/[bill_id]
 */
public function test_pharmacy_financials($bill_id = null) {
    if (!$bill_id) {
        echo "<h3>Test Pharmacy Financial Integration</h3>";
        echo "Usage: /admin/dispensary/test_pharmacy_financials/[bill_id]<br>";
        echo "Example: /admin/dispensary/test_pharmacy_financials/123<br>";
        return;
    }
    
    // Enable error reporting for debugging
    error_reporting(E_ALL);
    ini_set('display_errors', 1);
    
    echo "<h3>Testing Pharmacy Financial Integration for Bill ID: {$bill_id}</h3>";
    echo "<hr>";
    
    try {
        // Get bill details
        $this->db->where('id', $bill_id);
        $bill = $this->db->get('pharmacy_bill_basic')->row_array();
        
        if (!$bill) {
            echo "<div style='color: red;'>❌ Bill not found with ID: {$bill_id}</div>";
            return;
        }
        
        echo "<h4>Bill Details:</h4>";
        echo "<ul>";
        echo "<li><strong>Bill ID:</strong> {$bill['id']}</li>";
        echo "<li><strong>Patient ID:</strong> {$bill['patient_id']}</li>";
        echo "<li><strong>Date:</strong> {$bill['date']}</li>";
        echo "<li><strong>Net Amount:</strong> KES " . number_format($bill['net_amount'], 2) . "</li>";
        echo "<li><strong>Payment Status:</strong> {$bill['payment_status']}</li>";
        echo "</ul>";
        
        // Check if pharmacy accounts exist - using correct account codes
        echo "<h4>Checking Required Accounts:</h4>";
        $account_codes = ['1001', '4005', '1200', '1201'];
        $accounts_exist = true;
        
        echo "<table border='1' cellpadding='5' cellspacing='0'>";
        echo "<tr><th>Account Code</th><th>Account Name</th><th>Type</th><th>Status</th><th>ID</th></tr>";
        
        $found_accounts = [];
        foreach ($account_codes as $code) {
            $this->db->select('id, account_code, account_name, account_type, is_active');
            $account = $this->db->get_where('chart_of_accounts', ['account_code' => $code])->row_array();
            
            if ($account) {
                $found_accounts[$code] = $account;
                $status = $account['is_active'] ? "<span style='color: green;'>✅ Active</span>" : "<span style='color: orange;'>⚠️ Inactive</span>";
                echo "<tr>";
                echo "<td>{$account['account_code']}</td>";
                echo "<td>{$account['account_name']}</td>";
                echo "<td>{$account['account_type']}</td>";
                echo "<td>{$status}</td>";
                echo "<td>{$account['id']}</td>";
                echo "</tr>";
            } else {
                $accounts_exist = false;
                echo "<tr>";
                echo "<td>{$code}</td>";
                echo "<td colspan='3'><span style='color: red;'>❌ NOT FOUND</span></td>";
                echo "<td>-</td>";
                echo "</tr>";
            }
        }
        echo "</table>";
        
        // Show what accounts we actually found
        echo "<h4>Account Analysis:</h4>";
        echo "<ul>";
        foreach ($found_accounts as $code => $account) {
            $expected_type = '';
            switch($code) {
                case '1001': $expected_type = 'asset (Cash)'; break;
                case '4005': $expected_type = 'revenue (Pharmacy Revenue)'; break;
                case '1200': $expected_type = 'asset (Accounts Receivable) OR liability (Accounts Payable)'; break;
                case '1201': $expected_type = 'expense (Cost of Goods Sold)'; break;
            }
            
            $type_ok = true;
            if ($code == '1001' && $account['account_type'] != 'asset') $type_ok = false;
            if ($code == '4005' && $account['account_type'] != 'revenue') $type_ok = false;
            if ($code == '5001' && $account['account_type'] != 'expense') $type_ok = false;
            
            $type_status = $type_ok ? '✅' : '⚠️';
            echo "<li>{$type_status} <strong>{$code}:</strong> Found as '{$account['account_type']}' (Expected: {$expected_type})</li>";
        }
        echo "</ul>";
        
        // Note about account 2001
        if (isset($found_accounts['2001'])) {
            if ($found_accounts['2001']['account_type'] == 'liability') {
                echo "<div style='background: #fff3cd; padding: 10px; border: 1px solid #ffeeba; margin: 10px 0;'>";
                echo "⚠️ <strong>Note:</strong> Account 2001 is set as 'liability' (Accounts Payable) but for pharmacy sales we need 'asset' (Accounts Receivable). ";
                echo "This might cause issues. Consider creating a separate account code for Accounts Receivable.";
                echo "</div>";
            }
        }
        
        if (!$accounts_exist) {
            echo "<div style='color: red; margin: 10px 0;'>⚠️ Some required accounts are missing. Please run the pharmacy accounts setup SQL first.</div>";
            return;
        }
        
        // Test creating financial entries with enhanced debugging
        echo "<h4>Testing Financial Entry Creation:</h4>";
        echo "<div style='border: 1px solid #ccc; padding: 10px; margin: 10px 0;'>";
        
        try {
            // Load Financial model with error checking
            echo "<li>Loading Financial_model...</li>";
            if (!isset($this->Financial_model)) {
                $this->load->model('Financial_model');
                echo "<li>✅ Financial_model loaded successfully</li>";
            } else {
                echo "<li>✅ Financial_model already loaded</li>";
            }
            
            // Check if the method exists
            echo "<li>Checking if create_pharmacy_financial_entries method exists...</li>";
            if (method_exists($this->Financial_model, 'create_pharmacy_financial_entries')) {
                echo "<li>✅ Method create_pharmacy_financial_entries found</li>";
            } else {
                echo "<li>❌ Method create_pharmacy_financial_entries NOT found in Financial_model</li>";
                echo "<li>Available methods in Financial_model:</li>";
                echo "<ul>";
                $methods = get_class_methods($this->Financial_model);
                foreach ($methods as $method) {
                    if (!in_array($method, ['__construct', 'get_instance'])) {
                        echo "<li>{$method}</li>";
                    }
                }
                echo "</ul>";
                return;
            }
            
            // Try calling the method
            echo "<li>Calling create_pharmacy_financial_entries...</li>";
            
            // Add more detailed logging
            log_message('debug', "About to call create_pharmacy_financial_entries for bill {$bill_id}");
            
            $result = $this->Financial_model->create_pharmacy_financial_entries($bill_id, $bill);
            
            echo "<li>Method call completed. Result: " . ($result ? 'TRUE' : 'FALSE') . "</li>";
            
        } catch (Exception $e) {
            echo "<li>❌ Exception occurred: " . $e->getMessage() . "</li>";
            echo "<li>Stack trace: " . $e->getTraceAsString() . "</li>";
            return;
        }
        
        echo "</div>";
        
        if ($result) {
            echo "<div style='color: green;'>✅ Financial entries created successfully!</div>";
            
            // Check what was created
            echo "<h4>Checking Created Records:</h4>";
            
            // Journal headers
            $this->db->where('source_module', 'pharmacy');
            $this->db->where('source_id', $bill_id);
            $journals = $this->db->get('journal_headers')->result_array();
            
            echo "<ul>";
            echo "<li><strong>Journal Headers:</strong> " . count($journals) . " records</li>";
            
            if (!empty($journals)) {
                foreach ($journals as $journal) {
                    echo "<ul><li>ID: {$journal['id']}, Reference: {$journal['reference_no']}, Amount: " . number_format($journal['total_amount'], 2) . "</li></ul>";
                }
                
                // Journal lines
                $journal_ids = array_column($journals, 'id');
                $this->db->where_in('journal_id', $journal_ids);
                $this->db->join('chart_of_accounts', 'chart_of_accounts.id = journal_lines.account_id');
                $this->db->select('journal_lines.*, chart_of_accounts.account_code, chart_of_accounts.account_name');
                $lines = $this->db->get('journal_lines')->result_array();
                
                echo "<li><strong>Journal Lines:</strong> " . count($lines) . " records</li>";
                
                if (!empty($lines)) {
                    echo "<h5>Detailed Journal Entries:</h5>";
                    echo "<table border='1' cellpadding='5' cellspacing='0'>";
                    echo "<tr><th>Account</th><th>Description</th><th>Debit</th><th>Credit</th></tr>";
                    
                    $total_debits = 0;
                    $total_credits = 0;
                    
                    foreach ($lines as $line) {
                        echo "<tr>";
                        echo "<td>{$line['account_code']} - {$line['account_name']}</td>";
                        echo "<td>{$line['description']}</td>";
                        echo "<td>" . ($line['debit_amount'] > 0 ? number_format($line['debit_amount'], 2) : '-') . "</td>";
                        echo "<td>" . ($line['credit_amount'] > 0 ? number_format($line['credit_amount'], 2) : '-') . "</td>";
                        echo "</tr>";
                        
                        $total_debits += $line['debit_amount'];
                        $total_credits += $line['credit_amount'];
                    }
                    
                    echo "<tr style='font-weight: bold; background: #f0f0f0;'>";
                    echo "<td colspan='2'>TOTALS</td>";
                    echo "<td>" . number_format($total_debits, 2) . "</td>";
                    echo "<td>" . number_format($total_credits, 2) . "</td>";
                    echo "</tr>";
                    echo "</table>";
                    
                    // Verify balance
                    if (abs($total_debits - $total_credits) < 0.01) {
                        echo "<div style='color: green; margin: 10px 0;'>✅ <strong>Balanced:</strong> Debits equal Credits</div>";
                    } else {
                        echo "<div style='color: red; margin: 10px 0;'>❌ <strong>Not Balanced:</strong> Difference of " . number_format(abs($total_debits - $total_credits), 2) . "</div>";
                    }
                }
            }
            
            // General ledger
            $this->db->where('reference_type', 'pharmacy_bill');
            $this->db->where('reference_id', $bill_id);
            $gl_entries = $this->db->get('general_ledger')->result_array();
            echo "<li><strong>General Ledger:</strong> " . count($gl_entries) . " records</li>";
            
            // Financial transactions
            $this->db->where('reference_type', 'pharmacy_bill');
            $this->db->where('reference_id', $bill_id);
            $ft_entries = $this->db->get('financial_transactions')->result_array();
            echo "<li><strong>Financial Transactions:</strong> " . count($ft_entries) . " records</li>";
            
            echo "</ul>";
            
        } else {
            echo "<div style='color: red;'>❌ Failed to create financial entries!</div>";
            
            // Check database logs
            echo "<h4>Checking Database Logs:</h4>";
            echo "<p>Last database error: " . $this->db->error()['message'] . "</p>";
            
            // Check application logs
            echo "<h4>Recent Application Logs:</h4>";
            $log_file = APPPATH . 'logs/log-' . date('Y-m-d') . '.php';
            if (file_exists($log_file)) {
                $logs = file_get_contents($log_file);
                $recent_logs = array_slice(explode("\n", $logs), -20);
                echo "<pre style='background: #f8f8f8; padding: 10px; font-size: 12px;'>";
                echo implode("\n", $recent_logs);
                echo "</pre>";
            } else {
                echo "<p>No log file found for today</p>";
            }
        }
        
    } catch (Exception $e) {
        echo "<div style='color: red;'>❌ Error during testing: " . $e->getMessage() . "</div>";
        echo "<div>Stack trace:</div>";
        echo "<pre style='background: #f8f8f8; padding: 10px; font-size: 12px;'>";
        echo $e->getTraceAsString();
        echo "</pre>";
    }
    
    echo "<hr>";
    echo "<h4>Debug Information:</h4>";
    echo "<ul>";
    echo "<li><strong>PHP Version:</strong> " . phpversion() . "</li>";
    echo "<li><strong>CodeIgniter Version:</strong> " . CI_VERSION . "</li>";
    echo "<li><strong>Database:</strong> " . $this->db->database . "</li>";
    echo "<li><strong>Current Time:</strong> " . date('Y-m-d H:i:s') . "</li>";
    echo "</ul>";
}

/**
 * Setup pharmacy accounts automatically
 */
public function setup_pharmacy_accounts() {
    echo "<h3>Setting Up Pharmacy Accounts</h3>";
    echo "<hr>";
    
    try {
        $this->load->model('Financial_model');
        $result = $this->Financial_model->ensure_pharmacy_accounts();
        
        if ($result) {
            echo "<div style='color: green;'>✅ Pharmacy accounts setup completed successfully!</div>";
            echo "<br><a href='" . base_url() . "admin/dispensary/test_pharmacy_financials/1'>Test Financial Integration</a>";
        } else {
            echo "<div style='color: red;'>❌ Failed to setup pharmacy accounts</div>";
        }
        
    } catch (Exception $e) {
        echo "<div style='color: red;'>❌ Error setting up accounts: " . $e->getMessage() . "</div>";
    }
}

/**
 * View financial records for a specific pharmacy bill
 */
public function view_pharmacy_financials($bill_id = null) {
    if (!$bill_id) {
        echo "Usage: /admin/dispensary/view_pharmacy_financials/[bill_id]";
        return;
    }
    
    echo "<h3>Financial Records for Pharmacy Bill ID: {$bill_id}</h3>";
    echo "<hr>";
    
    // Journal Headers
    echo "<h4>Journal Headers:</h4>";
    $journals = $this->db->get_where('journal_headers', [
        'source_module' => 'pharmacy', 
        'source_id' => $bill_id
    ])->result_array();
    
    if (!empty($journals)) {
        echo "<table border='1' cellpadding='5' cellspacing='0'>";
        echo "<tr><th>ID</th><th>Reference No</th><th>Date</th><th>Description</th><th>Amount</th><th>Status</th></tr>";
        foreach ($journals as $journal) {
            echo "<tr>";
            echo "<td>{$journal['id']}</td>";
            echo "<td>{$journal['reference_no']}</td>";
            echo "<td>{$journal['date']}</td>";
            echo "<td>{$journal['description']}</td>";
            echo "<td>" . number_format($journal['total_amount'], 2) . "</td>";
            echo "<td>{$journal['status']}</td>";
            echo "</tr>";
        }
        echo "</table>";
    } else {
        echo "<div>No journal headers found</div>";
    }
    
    // Journal Lines
    echo "<br><h4>Journal Lines:</h4>";
    if (!empty($journals)) {
        $journal_ids = array_column($journals, 'id');
        $this->db->where_in('journal_id', $journal_ids);
        $this->db->join('chart_of_accounts', 'chart_of_accounts.id = journal_lines.account_id');
        $this->db->select('journal_lines.*, chart_of_accounts.account_code, chart_of_accounts.account_name');
        $lines = $this->db->get('journal_lines')->result_array();
        
        if (!empty($lines)) {
            echo "<table border='1' cellpadding='5' cellspacing='0'>";
            echo "<tr><th>Journal ID</th><th>Account</th><th>Debit</th><th>Credit</th><th>Description</th></tr>";
            foreach ($lines as $line) {
                echo "<tr>";
                echo "<td>{$line['journal_id']}</td>";
                echo "<td>{$line['account_code']} - {$line['account_name']}</td>";
                echo "<td>" . number_format($line['debit_amount'], 2) . "</td>";
                echo "<td>" . number_format($line['credit_amount'], 2) . "</td>";
                echo "<td>{$line['description']}</td>";
                echo "</tr>";
            }
            echo "</table>";
        } else {
            echo "<div>No journal lines found</div>";
        }
    }
    
    // General Ledger
    echo "<br><h4>General Ledger Entries:</h4>";
    $this->db->where('reference_type', 'pharmacy_bill');
    $this->db->where('reference_id', $bill_id);
    $this->db->join('chart_of_accounts', 'chart_of_accounts.id = general_ledger.account_id');
    $this->db->select('general_ledger.*, chart_of_accounts.account_code, chart_of_accounts.account_name');
    $gl_entries = $this->db->get('general_ledger')->result_array();
    
    if (!empty($gl_entries)) {
        echo "<table border='1' cellpadding='5' cellspacing='0'>";
        echo "<tr><th>Date</th><th>Account</th><th>Debit</th><th>Credit</th><th>Description</th></tr>";
        foreach ($gl_entries as $entry) {
            echo "<tr>";
            echo "<td>{$entry['transaction_date']}</td>";
            echo "<td>{$entry['account_code']} - {$entry['account_name']}</td>";
            echo "<td>" . number_format($entry['debit_amount'], 2) . "</td>";
            echo "<td>" . number_format($entry['credit_amount'], 2) . "</td>";
            echo "<td>{$entry['description']}</td>";
            echo "</tr>";
        }
        echo "</table>";
    } else {
        echo "<div>No general ledger entries found</div>";
    }
    
    // Financial Transactions
    echo "<br><h4>Financial Transactions:</h4>";
    $ft_entries = $this->db->get_where('financial_transactions', [
        'reference_type' => 'pharmacy_bill',
        'reference_id' => $bill_id
    ])->result_array();
    
    if (!empty($ft_entries)) {
        echo "<table border='1' cellpadding='5' cellspacing='0'>";
        echo "<tr><th>Date</th><th>Type</th><th>Category</th><th>Amount</th><th>Description</th><th>Receipt</th></tr>";
        foreach ($ft_entries as $entry) {
            echo "<tr>";
            echo "<td>{$entry['transaction_date']}</td>";
            echo "<td>{$entry['transaction_type']}</td>";
            echo "<td>{$entry['category']}/{$entry['subcategory']}</td>";
            echo "<td>" . number_format($entry['amount'], 2) . "</td>";
            echo "<td>{$entry['description']}</td>";
            echo "<td>{$entry['receipt_number']}</td>";
            echo "</tr>";
        }
        echo "</table>";
    } else {
        echo "<div>No financial transactions found</div>";
    }
}

/**
 * Clear financial data for a pharmacy bill (USE WITH CAUTION)
 */
public function clear_pharmacy_financials($bill_id = null, $confirm = null) {
    if (!$bill_id || $confirm !== 'YES') {
        echo "<h3>Clear Financial Data for Pharmacy Bill</h3>";
        echo "<div style='color: red;'><strong>WARNING: This will delete all financial records for the bill!</strong></div>";
        echo "<br>Usage: /admin/dispensary/clear_pharmacy_financials/[bill_id]/YES";
        echo "<br>Example: /admin/dispensary/clear_pharmacy_financials/123/YES";
        return;
    }

    echo "<h3>Clearing Financial Data for Pharmacy Bill ID: {$bill_id}</h3>";
    echo "<hr>";
    
    // Delete in reverse order to respect foreign keys
    $tables = [
        'financial_transactions' => ['reference_type' => 'pharmacy_bill', 'reference_id' => $bill_id],
        'general_ledger' => ['reference_type' => 'pharmacy_bill', 'reference_id' => $bill_id]
    ];
    
    // Get journal IDs first
    $journal_ids = [];
    $this->db->select('id');
    $this->db->where('source_module', 'pharmacy');
    $this->db->where('source_id', $bill_id);
    $journals = $this->db->get('journal_headers')->result_array();
    
    if (!empty($journals)) {
        $journal_ids = array_column($journals, 'id');
    }
    
    foreach ($tables as $table => $conditions) {
        foreach ($conditions as $field => $value) {
            $this->db->where($field, $value);
        }
        $this->db->delete($table);
        $affected = $this->db->affected_rows();
        echo "<li>Deleted {$affected} records from {$table}</li>";
    }
    
    // Delete journal lines
    if (!empty($journal_ids)) {
        $this->db->where_in('journal_id', $journal_ids);
        $this->db->delete('journal_lines');
        $affected = $this->db->affected_rows();
        echo "<li>Deleted {$affected} records from journal_lines</li>";
    }
    
    // Delete journal headers
    if (!empty($journal_ids)) {
        $this->db->where_in('id', $journal_ids);
        $this->db->delete('journal_headers');
        $affected = $this->db->affected_rows();
        echo "<li>Deleted {$affected} records from journal_headers</li>";
    }
    
    echo "<br><div style='color: green;'>✅ Financial data cleared for pharmacy bill {$bill_id}</div>";
}
}