<?php

defined('BASEPATH') OR exit('No direct script access allowed');

/**
 * Consumable Store Controller
 * 
 * Handles all operations for the consumable store module
 * 
 * @author Hospital Management System
 * @version 1.0
 */
class Consumable extends Admin_Controller
{
    
    public function __construct()
    {
        parent::__construct();
        $this->load->model('consumable_model');
        $this->load->library(['form_validation', 'upload']);
        $this->load->helper(['url', 'form']);
        
        // Set session menu
        $this->session->set_userdata('top_menu', 'inventory');
        $this->session->set_userdata('sub_menu', 'consumable');
    }

    // ========================================================
    // MAIN DASHBOARD
    // ========================================================

    /**
     * Consumable Store Dashboard
     */
    public function index()
    {
        if (!$this->rbac->hasPrivilege('consumable_store', 'can_view')) {
            access_denied();
        }

        $data['title'] = 'Consumable Store Dashboard';
        
        // Get dashboard statistics
        $data['total_consumables'] = $this->consumable_model->getConsumables(['active_only' => true]);
        $data['low_stock_items'] = $this->consumable_model->getConsumables(['low_stock' => true, 'active_only' => true]);
        $data['recent_transactions'] = $this->consumable_model->getTransactions(['limit' => 10]);
        $data['active_alerts'] = $this->consumable_model->getActiveAlerts();
        $data['stock_summary'] = $this->consumable_model->getStockSummaryReport();
        $data['expiring_items'] = $this->consumable_model->getExpiringItemsReport(30);
        
        $this->load->view('layout/header', $data);
        $this->load->view('admin/consumable/dashboard', $data);
        $this->load->view('layout/footer');
    }

    // ========================================================
    // CONSUMABLES MANAGEMENT
    // ========================================================

    /**
     * List all consumables
     */
    public function consumables()
    {
        if (!$this->rbac->hasPrivilege('consumable_store', 'can_view')) {
            access_denied();
        }

        $data['title'] = 'Manage Consumables';
        
        // Get filters from request
        $filters = [];
        if ($this->input->get('category_id')) {
            $filters['category_id'] = $this->input->get('category_id');
        }
        if ($this->input->get('search')) {
            $filters['search'] = $this->input->get('search');
        }
        if ($this->input->get('low_stock')) {
            $filters['low_stock'] = true;
        }
        $filters['active_only'] = true;
        
        $data['consumables'] = $this->consumable_model->getConsumables($filters);
        $data['categories'] = $this->consumable_model->getConsumableCategories();
        $data['current_filters'] = $filters;
        
        $this->load->view('layout/header', $data);
        $this->load->view('admin/consumable/consumables', $data);
        $this->load->view('layout/footer');
    }

    /**
     * Add new consumable
     */
    public function add_consumable()
    {
        if (!$this->rbac->hasPrivilege('consumable_store', 'can_add')) {
            access_denied();
        }

        $data['title'] = 'Add New Consumable';
        $data['categories'] = $this->consumable_model->getConsumableCategories();
        $data['suppliers'] = $this->consumable_model->getSuppliers();

        if ($this->input->post()) {
            $this->form_validation->set_rules('consumable_name', 'Consumable Name', 'required|min_length[3]');
            $this->form_validation->set_rules('category_id', 'Category', 'required|numeric');
            $this->form_validation->set_rules('unit_of_measure', 'Unit of Measure', 'required');
            $this->form_validation->set_rules('unit_cost', 'Unit Cost', 'required|numeric|greater_than[0]');
            $this->form_validation->set_rules('minimum_stock', 'Minimum Stock', 'numeric');
            $this->form_validation->set_rules('maximum_stock', 'Maximum Stock', 'numeric');
            $this->form_validation->set_rules('reorder_level', 'Reorder Level', 'numeric');

            if ($this->form_validation->run() == TRUE) {
                $consumable_data = [
                    'consumable_name' => $this->input->post('consumable_name'),
                    'generic_name' => $this->input->post('generic_name'),
                    'category_id' => $this->input->post('category_id'),
                    'description' => $this->input->post('description'),
                    'unit_of_measure' => $this->input->post('unit_of_measure'),
                    'unit_cost' => $this->input->post('unit_cost'),
                    'markup_percentage' => $this->input->post('markup_percentage') ?: 0,
                    'minimum_stock' => $this->input->post('minimum_stock') ?: 0,
                    'maximum_stock' => $this->input->post('maximum_stock') ?: 0,
                    'reorder_level' => $this->input->post('reorder_level') ?: 0,
                    'current_stock' => $this->input->post('current_stock') ?: 0,
                    'manufacturer' => $this->input->post('manufacturer'),
                    'brand' => $this->input->post('brand'),
                    'strength' => $this->input->post('strength'),
                    'formulation' => $this->input->post('formulation'),
                    'batch_tracking' => $this->input->post('batch_tracking') ?: 'no',
                    'expiry_tracking' => $this->input->post('expiry_tracking') ?: 'no',
                    'storage_temperature' => $this->input->post('storage_temperature'),
                    'storage_conditions' => $this->input->post('storage_conditions'),
                    'barcode' => $this->input->post('barcode'),
                    'therapeutic_class' => $this->input->post('therapeutic_class'),
                    'contraindications' => $this->input->post('contraindications'),
                    'side_effects' => $this->input->post('side_effects'),
                    'dosage_instructions' => $this->input->post('dosage_instructions'),
                    'requires_prescription' => $this->input->post('requires_prescription') ?: 'no',
                    'is_controlled_substance' => $this->input->post('is_controlled_substance') ?: 'no',
                    'controlled_schedule' => $this->input->post('controlled_schedule'),
                    'is_active' => 'yes'
                ];

                if ($this->consumable_model->addConsumable($consumable_data)) {
                    $this->session->set_flashdata('msg', '<div class="alert alert-success">Consumable added successfully!</div>');
                    redirect('admin/consumable/consumables');
                } else {
                    $this->session->set_flashdata('msg', '<div class="alert alert-danger">Failed to add consumable. Please try again.</div>');
                }
            }
        }

        $this->load->view('layout/header', $data);
        $this->load->view('admin/consumable/add_consumable', $data);
        $this->load->view('layout/footer');
    }

    /**
     * Edit consumable
     */
    public function edit_consumable($id)
    {
        if (!$this->rbac->hasPrivilege('consumable_store', 'can_edit')) {
            access_denied();
        }

        $data['title'] = 'Edit Consumable';
        $data['consumable'] = $this->consumable_model->getConsumableById($id);
        
        if (!$data['consumable']) {
            show_404();
        }

        $data['categories'] = $this->consumable_model->getConsumableCategories();
        $data['suppliers'] = $this->consumable_model->getSuppliers();

        if ($this->input->post()) {
            $this->form_validation->set_rules('consumable_name', 'Consumable Name', 'required|min_length[3]');
            $this->form_validation->set_rules('category_id', 'Category', 'required|numeric');
            $this->form_validation->set_rules('unit_of_measure', 'Unit of Measure', 'required');
            $this->form_validation->set_rules('unit_cost', 'Unit Cost', 'required|numeric|greater_than[0]');

            if ($this->form_validation->run() == TRUE) {
                $consumable_data = [
                    'consumable_name' => $this->input->post('consumable_name'),
                    'generic_name' => $this->input->post('generic_name'),
                    'category_id' => $this->input->post('category_id'),
                    'description' => $this->input->post('description'),
                    'unit_of_measure' => $this->input->post('unit_of_measure'),
                    'unit_cost' => $this->input->post('unit_cost'),
                    'markup_percentage' => $this->input->post('markup_percentage') ?: 0,
                    'minimum_stock' => $this->input->post('minimum_stock') ?: 0,
                    'maximum_stock' => $this->input->post('maximum_stock') ?: 0,
                    'reorder_level' => $this->input->post('reorder_level') ?: 0,
                    'manufacturer' => $this->input->post('manufacturer'),
                    'brand' => $this->input->post('brand'),
                    'strength' => $this->input->post('strength'),
                    'formulation' => $this->input->post('formulation'),
                    'batch_tracking' => $this->input->post('batch_tracking') ?: 'no',
                    'expiry_tracking' => $this->input->post('expiry_tracking') ?: 'no',
                    'storage_temperature' => $this->input->post('storage_temperature'),
                    'storage_conditions' => $this->input->post('storage_conditions'),
                    'barcode' => $this->input->post('barcode'),
                    'therapeutic_class' => $this->input->post('therapeutic_class'),
                    'contraindications' => $this->input->post('contraindications'),
                    'side_effects' => $this->input->post('side_effects'),
                    'dosage_instructions' => $this->input->post('dosage_instructions'),
                    'requires_prescription' => $this->input->post('requires_prescription') ?: 'no',
                    'is_controlled_substance' => $this->input->post('is_controlled_substance') ?: 'no',
                    'controlled_schedule' => $this->input->post('controlled_schedule'),
                    'is_active' => $this->input->post('is_active') ?: 'yes'
                ];

                if ($this->consumable_model->updateConsumable($id, $consumable_data)) {
                    $this->session->set_flashdata('msg', '<div class="alert alert-success">Consumable updated successfully!</div>');
                    redirect('admin/consumable/consumables');
                } else {
                    $this->session->set_flashdata('msg', '<div class="alert alert-danger">Failed to update consumable. Please try again.</div>');
                }
            }
        }

        $this->load->view('layout/header', $data);
        $this->load->view('admin/consumable/edit_consumable', $data);
        $this->load->view('layout/footer');
    }

    // ========================================================
    // DEPARTMENT STOCK MANAGEMENT
    // ========================================================

    /**
     * Department stock overview  
     */
    public function department_stock($department_id = null)
    {
        if (!$this->rbac->hasPrivilege('consumable_store', 'can_view')) {
            access_denied();
        }

        $data['title'] = 'Department Stock Management';
        $data['departments'] = $this->consumable_model->getDepartments();
        
        if ($department_id) {
            $data['selected_department'] = $department_id;
            $data['department_stock'] = $this->consumable_model->getDepartmentStock($department_id);
            
            // Get department info
            $dept_info = array_filter($data['departments'], function($dept) use ($department_id) {
                return $dept->id == $department_id;
            });
            $data['department_info'] = reset($dept_info);
        }

        $this->load->view('layout/header', $data);
        $this->load->view('admin/consumable/department_stock', $data);
        $this->load->view('layout/footer');
    }

    /**
     * Release consumables to department
     */
    public function release_to_department()
    {
        if (!$this->rbac->hasPrivilege('consumable_store', 'can_edit')) {
            access_denied();
        }

        $data['title'] = 'Release Consumables to Department';
        $data['departments'] = $this->consumable_model->getDepartments();
        $data['consumables'] = $this->consumable_model->getConsumables(['active_only' => true]);

        if ($this->input->post()) {
            $this->form_validation->set_rules('department_id', 'Department', 'required|numeric');
            $this->form_validation->set_rules('consumable_id', 'Consumable', 'required|numeric');
            $this->form_validation->set_rules('quantity', 'Quantity', 'required|numeric|greater_than[0]');

            if ($this->form_validation->run() == TRUE) {
                $department_id = $this->input->post('department_id');
                $consumable_id = $this->input->post('consumable_id');
                $quantity = $this->input->post('quantity');
                $staff_id = $this->session->userdata('admin')['id'];
                $notes = $this->input->post('notes');

                $result = $this->consumable_model->releaseToDepar tment($department_id, $consumable_id, $quantity, $staff_id, $notes);

                if ($result['success']) {
                    $this->session->set_flashdata('msg', '<div class="alert alert-success">Consumables released successfully! Transaction #: ' . $result['transaction_number'] . '</div>');
                    redirect('admin/consumable/department_stock/' . $department_id);
                } else {
                    $this->session->set_flashdata('msg', '<div class="alert alert-danger">Error: ' . $result['error'] . '</div>');
                }
            }
        }

        $this->load->view('layout/header', $data);
        $this->load->view('admin/consumable/release_to_department', $data);
        $this->load->view('layout/footer');
    }

    // ========================================================
    // REQUISITIONS MANAGEMENT
    // ========================================================

    /**
     * List requisitions
     */
    public function requisitions()
    {
        if (!$this->rbac->hasPrivilege('consumable_store', 'can_view')) {
            access_denied();
        }

        $data['title'] = 'Consumable Requisitions';
        
        $filters = [];
        if ($this->input->get('department_id')) {
            $filters['department_id'] = $this->input->get('department_id');
        }
        if ($this->input->get('status')) {
            $filters['status'] = $this->input->get('status');
        }
        
        $data['requisitions'] = $this->consumable_model->getRequisitions($filters);
        $data['departments'] = $this->consumable_model->getDepartments();
        $data['current_filters'] = $filters;

        $this->load->view('layout/header', $data);
        $this->load->view('admin/consumable/requisitions', $data);
        $this->load->view('layout/footer');
    }

    /**
     * Create new requisition
     */
    public function create_requisition()
    {
        if (!$this->rbac->hasPrivilege('consumable_store', 'can_add')) {
            access_denied();
        }

        $data['title'] = 'Create Consumable Requisition';
        $data['departments'] = $this->consumable_model->getDepartments();
        $data['consumables'] = $this->consumable_model->getConsumables(['active_only' => true]);

        if ($this->input->post()) {
            $this->form_validation->set_rules('requesting_department_id', 'Department', 'required|numeric');
            $this->form_validation->set_rules('request_date', 'Request Date', 'required');
            $this->form_validation->set_rules('priority', 'Priority', 'required');
            $this->form_validation->set_rules('purpose', 'Purpose', 'required');

            if ($this->form_validation->run() == TRUE) {
                $requisition_data = [
                    'requesting_department_id' => $this->input->post('requesting_department_id'),
                    'requested_by' => $this->session->userdata('admin')['id'],
                    'request_date' => $this->input->post('request_date'),
                    'required_date' => $this->input->post('required_date'),
                    'priority' => $this->input->post('priority'),
                    'purpose' => $this->input->post('purpose'),
                    'justification' => $this->input->post('justification'),
                    'status' => 'draft'
                ];

                // Process requisition items
                $items_data = [];
                $consumable_ids = $this->input->post('consumable_id');
                $quantities = $this->input->post('quantity_requested');
                $justifications = $this->input->post('item_justification');

                if ($consumable_ids && $quantities) {
                    $total_estimated_cost = 0;
                    
                    for ($i = 0; $i < count($consumable_ids); $i++) {
                        if ($consumable_ids[$i] && $quantities[$i] > 0) {
                            $consumable = $this->consumable_model->getConsumableById($consumable_ids[$i]);
                            $item_cost = $consumable->unit_cost * $quantities[$i];
                            
                            $items_data[] = [
                                'consumable_id' => $consumable_ids[$i],
                                'quantity_requested' => $quantities[$i],
                                'unit_cost_estimated' => $consumable->unit_cost,
                                'total_cost_estimated' => $item_cost,
                                'justification' => $justifications[$i] ?? ''
                            ];
                            
                            $total_estimated_cost += $item_cost;
                        }
                    }
                    
                    $requisition_data['total_estimated_cost'] = $total_estimated_cost;
                }

                if (!empty($items_data)) {
                    $result = $this->consumable_model->addRequisition($requisition_data, $items_data);
                    
                    if ($result['success']) {
                        $this->session->set_flashdata('msg', '<div class="alert alert-success">Requisition created successfully!</div>');
                        redirect('admin/consumable/requisitions');
                    } else {
                        $this->session->set_flashdata('msg', '<div class="alert alert-danger">Error: ' . $result['error'] . '</div>');
                    }
                } else {
                    $this->session->set_flashdata('msg', '<div class="alert alert-warning">Please add at least one item to the requisition.</div>');
                }
            }
        }

        $this->load->view('layout/header', $data);
        $this->load->view('admin/consumable/create_requisition', $data);
        $this->load->view('layout/footer');
    }

    /**
     * View requisition details
     */
    public function view_requisition($id)
    {
        if (!$this->rbac->hasPrivilege('consumable_store', 'can_view')) {
            access_denied();
        }

        $data['title'] = 'Requisition Details';
        $data['requisition'] = $this->consumable_model->getRequisitionById($id);
        
        if (!$data['requisition']) {
            show_404();
        }

        $this->load->view('layout/header', $data);
        $this->load->view('admin/consumable/view_requisition', $data);
        $this->load->view('layout/footer');
    }

    /**
     * Approve requisition
     */
    public function approve_requisition($id)
    {
        if (!$this->rbac->hasPrivilege('consumable_store', 'can_approve')) {
            access_denied();
        }

        if ($this->input->post()) {
            $approved_by = $this->session->userdata('admin')['id'];
            $comments = $this->input->post('approval_comments');

            if ($this->consumable_model->approveRequisition($id, $approved_by, $comments)) {
                $this->session->set_flashdata('msg', '<div class="alert alert-success">Requisition approved successfully!</div>');
            } else {
                $this->session->set_flashdata('msg', '<div class="alert alert-danger">Failed to approve requisition.</div>');
            }
        }

        redirect('admin/consumable/view_requisition/' . $id);
    }

    // ========================================================
    // TRANSACTIONS AND REPORTS
    // ========================================================

    /**
     * View transactions
     */
    public function transactions()
    {
        if (!$this->rbac->hasPrivilege('consumable_store', 'can_view')) {
            access_denied();
        }

        $data['title'] = 'Consumable Transactions';
        
        $filters = [];
        if ($this->input->get('transaction_type')) {
            $filters['transaction_type'] = $this->input->get('transaction_type');
        }
        if ($this->input->get('department_id')) {
            $filters['department_id'] = $this->input->get('department_id');
        }
        if ($this->input->get('date_from')) {
            $filters['date_from'] = $this->input->get('date_from');
        }
        if ($this->input->get('date_to')) {
            $filters['date_to'] = $this->input->get('date_to');
        }
        
        $data['transactions'] = $this->consumable_model->getTransactions($filters);
        $data['departments'] = $this->consumable_model->getDepartments();
        $data['current_filters'] = $filters;

        $this->load->view('layout/header', $data);
        $this->load->view('admin/consumable/transactions', $data);
        $this->load->view('layout/footer');
    }

    /**
     * Reports dashboard
     */
    public function reports()
    {
        if (!$this->rbac->hasPrivilege('consumable_store', 'can_view')) {
            access_denied();
        }

        $data['title'] = 'Consumable Reports';
        $data['stock_summary'] = $this->consumable_model->getStockSummaryReport();
        $data['expiring_items'] = $this->consumable_model->getExpiringItemsReport(90);
        $data['low_stock_items'] = $this->consumable_model->getConsumables(['low_stock' => true, 'active_only' => true]);

        $this->load->view('layout/header', $data);
        $this->load->view('admin/consumable/reports', $data);
        $this->load->view('layout/footer');
    }

    // ========================================================
    // AJAX METHODS
    // ========================================================

    /**
     * Get consumable details (AJAX)
     */
    public function get_consumable_details()
    {
        if (!$this->rbac->hasPrivilege('consumable_store', 'can_view')) {
            echo json_encode(['error' => 'Access denied']);
            return;
        }

        $id = $this->input->post('consumable_id');
        $consumable = $this->consumable_model->getConsumableById($id);
        
        if ($consumable) {
            echo json_encode([
                'success' => true,
                'data' => $consumable
            ]);
        } else {
            echo json_encode([
                'success' => false,
                'message' => 'Consumable not found'
            ]);
        }
    }

    /**
     * Search consumables (AJAX)
     */
    public function search_consumables()
    {
        if (!$this->rbac->hasPrivilege('consumable_store', 'can_view')) {
            echo json_encode(['error' => 'Access denied']);
            return;
        }

        $search = $this->input->get('q');
        $category_id = $this->input->get('category_id');
        
        $filters = [
            'active_only' => true,
            'search' => $search
        ];
        
        if ($category_id) {
            $filters['category_id'] = $category_id;
        }
        
        $consumables = $this->consumable_model->getConsumables($filters);
        
        $results = [];
        foreach ($consumables as $consumable) {
            $results[] = [
                'id' => $consumable->id,
                'text' => $consumable->consumable_name . ' (' . $consumable->consumable_code . ')',
                'unit_cost' => $consumable->unit_cost,
                'current_stock' => $consumable->current_stock,
                'unit_of_measure' => $consumable->unit_of_measure
            ];
        }
        
        echo json_encode(['results' => $results]);
    }

    /**
     * Get department consumables (AJAX)
     */
    public function get_department_consumables()
    {
        if (!$this->rbac->hasPrivilege('consumable_store', 'can_view')) {
            echo json_encode(['error' => 'Access denied']);
            return;
        }

        $department_id = $this->input->post('department_id');
        $consumables = $this->consumable_model->getDepartmentStock($department_id);
        
        echo json_encode([
            'success' => true,
            'data' => $consumables
        ]);
    }
}

?>