<?php

if (!defined('BASEPATH'))
    exit('No direct script access allowed');

class Consultation extends Admin_Controller {

    function __construct() {
        parent::__construct();
        
        // Load essential libraries
        try {
            $this->load->model('patient_model');
            $this->load->model('staff_model');
            $this->load->library('datatables');
            $this->load->library('customlib');
            // Add queue model
            $this->load->model('Queue_model');
    
        } catch (Exception $e) {
            log_message('error', 'Consultation controller error: ' . $e->getMessage());
            show_error('Failed to load required libraries: ' . $e->getMessage());
        }
    }

    public function index() {
        // Set session data
        $this->session->set_userdata('top_menu', 'OPD');
        $this->session->set_userdata('sub_menu', 'consultation');

        // Prepare data for view
        $data['title'] = 'Patient Consultation';
        
        // Get doctors
        try {
            if (method_exists($this->staff_model, 'getStaffbyrole')) {
                $data['doctors'] = $this->staff_model->getStaffbyrole(3); // Role 3 = doctors
            } else {
                // Alternative query to get doctors
                $this->db->select('s.*, r.name as role_name');
                $this->db->from('staff s');
                $this->db->join('staff_roles sr', 'sr.staff_id = s.id', 'left');
                $this->db->join('roles r', 'r.id = sr.role_id', 'left');
                $this->db->where('r.name', 'Doctor');
                $this->db->or_where('r.id', 3);
                $this->db->where('s.is_active', 'yes');
                $query = $this->db->get();
                $data['doctors'] = $query->result_array();
            }
        } catch (Exception $e) {
            // Fallback - get all active staff
            $this->db->select('id, name, surname, employee_id');
            $this->db->from('staff');
            $this->db->where('is_active', 'yes');
            $query = $this->db->get();
            $data['doctors'] = $query->result_array();
        }
        
        // Get gender list
        try {
            if (method_exists($this->customlib, 'getGender_Patient')) {
                $data['genderList'] = $this->customlib->getGender_Patient();
            } else {
                $data['genderList'] = array(
                    'male' => 'Male',
                    'female' => 'Female',
                    'other' => 'Other'
                );
            }
        } catch (Exception $e) {
            $data['genderList'] = array('male' => 'Male', 'female' => 'Female');
        }
        
        // Load views
        try {
            $this->load->view('layout/header', $data);
            $this->load->view('admin/consultation/index', $data);
            $this->load->view('layout/footer', $data);
        } catch (Exception $e) {
            log_message('error', 'View loading error: ' . $e->getMessage());
            show_error('Failed to load consultation view: ' . $e->getMessage());
        }
    }
    
    /**
 * Consultation Queue Management View
 * URL: /admin/consultation/queue
 */
public function queue() {
    $this->session->set_userdata('top_menu', 'OPD');
    $this->session->set_userdata('sub_menu', 'consultation/queue');
    
    $data['title'] = 'Consultation Queue Management';
    $data['department'] = 'consultation';
    $data['queue_data'] = $this->Queue_model->getDepartmentQueue('consultation');
    $data['queue_config'] = $this->Queue_model->getQueueConfig('consultation');
    
    // Get doctors for assignment
    $data['doctors'] = $this->staff_model->getStaffbyrole(3); // Role 3 = Doctor
    
    $this->load->view('layout/header', $data);
    $this->load->view('admin/consultation/queue_management', $data);
    $this->load->view('layout/footer');
}

    // =============================================
    // PATIENT MANAGEMENT METHODS
    // =============================================

    public function getAllPatients() {
    // CRITICAL: Clean output buffer and set headers
    ob_clean();
    header('Content-Type: application/json');
    header('Cache-Control: no-cache, must-revalidate');
    
    // Suppress any error output that could corrupt JSON
    error_reporting(0);
    ini_set('display_errors', 0);
 
    try {
        // Get all active patients with fingerprint info
        $this->db->select('id, patient_name, mobileno, patient_workflow, gender, age, image, patient_unique_id, patient_type, identification_number');
        $this->db->from('patients');
        $this->db->where('is_active', 'yes');
        $this->db->order_by('patient_name', 'ASC');
        $this->db->limit(800);
        
        $query = $this->db->get();
        
        if (!$query) {
            ob_clean();
            echo json_encode([
                'patients' => [],
                'count' => 0,
                'error' => 'Database query failed: ' . $this->db->error()['message']
            ]);
            exit;
        }
        
        $patients = $query->result_array();
        
        // Add fingerprint status to each patient if fingerprint columns exist
        foreach ($patients as &$patient) {
            if ($this->db->field_exists('fingerprint_template', 'patients')) {
                // Get fingerprint status separately to avoid large data transfer
                $this->db->select('fingerprint_template');
                $this->db->from('patients');
                $this->db->where('id', $patient['id']);
                $fp_query = $this->db->get();
                
                if ($fp_query && $fp_query->num_rows() > 0) {
                    $fp_data = $fp_query->row();
                    $patient['has_fingerprint'] = !empty($fp_data->fingerprint_template);
                } else {
                    $patient['has_fingerprint'] = false;
                }
            } else {
                $patient['has_fingerprint'] = false;
            }
            
            if (!empty($patient['image'])) {
                // Log the original image value for debugging
                error_log("Original image value: " . $patient['image']);
                
                // Handle different image path scenarios
                if (str_starts_with($patient['image'], 'http')) {
                    // Already a full URL, keep as is
                    $patient['image_url'] = $patient['image'];
                } else if (str_starts_with($patient['image'], '../uploads/')) {
                    // Already has relative path, keep as is
                    $patient['image_url'] = $patient['image'];
                } else if (str_starts_with($patient['image'], 'uploads/')) {
                    // Missing ./ prefix
                    $patient['image_url'] = './' . $patient['image'];
                } else {
                    // Just filename, add full path
                    $patient['image_url'] = '../uploads/patient_images/' . $patient['image'];
                }
                
                // Check if file actually exists (optional)
                $file_path = $_SERVER['DOCUMENT_ROOT'] . '/' . ltrim($patient['image_url'], './');
                if (!file_exists($file_path)) {
                    error_log("Image file not found: " . $file_path);
                    $patient['image_url'] = '../uploads/patient_images/no_image.png';
                }
            } else {
                $patient['image_url'] = '../uploads/patient_images/no_image.png';
            }
        }
        
        // Clean output buffer before sending JSON
        ob_clean();
        echo json_encode([
            'patients' => $patients,
            'count' => count($patients),
            'message' => 'Found ' . count($patients) . ' patients'
        ]);
        exit;
        
    } catch (Exception $e) {
        ob_clean();
        echo json_encode([
            'patients' => [],
            'count' => 0,
            'error' => 'Exception: ' . $e->getMessage()
        ]);
        exit;
    }
}

/**
 * Get patient verification history
 */
public function getPatientVerifications() {
    header('Content-Type: application/json');
    
    try {
        $patientId = $this->input->post('patient_id');
        $limit = $this->input->post('limit') ?? 10;
        
        if (!$patientId) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Patient ID required'
            ]);
            return;
        }
        
        // Check if verification log table exists
        if (!$this->db->table_exists('fingerprint_verification_log')) {
            echo json_encode([
                'status' => 'success',
                'verifications' => [],
                'message' => 'No verification history available - table not created yet'
            ]);
            return;
        }
        
        // Get verification history
        $this->db->select('v.*, CONCAT(s.name, " ", s.surname) as staff_name_full');
        $this->db->from('fingerprint_verification_log v');
        $this->db->join('staff s', 's.id = v.staff_id', 'left');
        $this->db->where('v.patient_id', $patientId);
        $this->db->order_by('v.verification_date', 'DESC');
        $this->db->limit($limit);
        
        $query = $this->db->get();
        
        if ($query) {
            $verifications = $query->result_array();
            
            // Format the data
            foreach ($verifications as &$verification) {
                $verification['match_percentage'] = round(($verification['match_score'] / 199) * 100, 1);
                $verification['verification_date_formatted'] = date('M j, Y g:i A', strtotime($verification['verification_date']));
            }
            
            echo json_encode([
                'status' => 'success',
                'verifications' => $verifications,
                'total_found' => count($verifications)
            ]);
        } else {
            echo json_encode([
                'status' => 'error',
                'message' => 'Database query failed',
                'verifications' => []
            ]);
        }
        
    } catch (Exception $e) {
        log_message('error', 'Error getting patient verifications: ' . $e->getMessage());
        echo json_encode([
            'status' => 'error',
            'message' => 'Database error: ' . $e->getMessage(),
            'verifications' => []
        ]);
    }
}

public function getAllDoctors() {
    // Clean output buffer and set headers
    ob_clean();
    header('Content-Type: application/json');
    header('Cache-Control: no-cache, must-revalidate');
    
    // Suppress any error output that could corrupt JSON
    error_reporting(0);
    ini_set('display_errors', 0);
    
    try {
        $doctors = [];
        
        // Try to use the staff model method first
        if (method_exists($this->staff_model, 'getStaffbyrole')) {
            // Get doctors using the staff model - role 3 = doctors
            $doctors = $this->staff_model->getStaffbyrole(3); // Role 3 = Doctor
            
            if (!empty($doctors)) {
                // Format the response from staff model
                $formatted_doctors = [];
                foreach ($doctors as $doctor) {
                    $full_name = trim(($doctor['name'] ?? 'Unknown') . ' ' . ($doctor['surname'] ?? ''));
                    $display_name = 'Dr. ' . $full_name;
                    
                    // Add specialization info if available
                    if (!empty($doctor['specialist'])) {
                        $display_name .= ' (' . $doctor['specialist'] . ')';
                    } elseif (!empty($doctor['specialization'])) {
                        $display_name .= ' (' . $doctor['specialization'] . ')';
                    }
                    
                    $formatted_doctors[] = [
                        'id' => $doctor['id'],
                        'name' => $doctor['name'] ?? 'Unknown',
                        'surname' => $doctor['surname'] ?? '',
                        'employee_id' => $doctor['employee_id'] ?? ('EMP-' . $doctor['id']),
                        'full_name' => $full_name,
                        'display_name' => $display_name,
                        'specialist' => $doctor['specialist'] ?? '',
                        'specialization' => $doctor['specialization'] ?? '',
                        'designation' => 'Doctor'
                    ];
                }
                $doctors = $formatted_doctors;
            }
        }
        
        // If staff model didn't work or returned empty, use direct database query
        if (empty($doctors)) {
            // Direct database query with proper joins to staff_roles and roles tables
            $this->db->select('s.id, s.name, s.surname, s.employee_id, s.specialist, s.specialization, r.name as role_name');
            $this->db->from('staff s');
            $this->db->join('staff_roles sr', 's.id = sr.staff_id', 'inner');
            $this->db->join('roles r', 'sr.role_id = r.id', 'inner');
            $this->db->where('s.is_active', 'yes');
            $this->db->where('r.id', 3); // Role ID 3 = Doctor
            $this->db->order_by('s.name, s.surname', 'ASC');
            
            $query = $this->db->get();
            
            if ($query && $query->num_rows() > 0) {
                $staff_doctors = $query->result_array();
                
                // Format doctors for dropdown
                $formatted_doctors = [];
                foreach ($staff_doctors as $doctor) {
                    $full_name = trim(($doctor['name'] ?? 'Unknown') . ' ' . ($doctor['surname'] ?? ''));
                    $display_name = 'Dr. ' . $full_name;
                    
                    // Add specialization info if available
                    if (!empty($doctor['specialist'])) {
                        $display_name .= ' (' . $doctor['specialist'] . ')';
                    } elseif (!empty($doctor['specialization'])) {
                        $display_name .= ' (' . $doctor['specialization'] . ')';
                    }
                    
                    // Add employee ID if available
                    if (!empty($doctor['employee_id'])) {
                        $display_name .= ' - ID: ' . $doctor['employee_id'];
                    }
                    
                    $formatted_doctors[] = [
                        'id' => $doctor['id'],
                        'name' => $doctor['name'] ?? 'Unknown',
                        'surname' => $doctor['surname'] ?? '',
                        'employee_id' => $doctor['employee_id'] ?? ('EMP-' . $doctor['id']),
                        'full_name' => $full_name,
                        'display_name' => $display_name,
                        'specialist' => $doctor['specialist'] ?? '',
                        'specialization' => $doctor['specialization'] ?? '',
                        'role_name' => $doctor['role_name'] ?? 'Doctor',
                        'designation' => 'Doctor'
                    ];
                }
                $doctors = $formatted_doctors;
            }
        }
        
        // Final fallback - get all active staff if still no doctors found
        if (empty($doctors)) {
            $this->db->select('id, name, surname, employee_id, specialist, specialization');
            $this->db->where('is_active', 'yes');
            $this->db->order_by('name, surname', 'ASC');
            $this->db->limit(20); // Limit fallback results
            $query = $this->db->get('staff');
            
            if ($query && $query->num_rows() > 0) {
                $all_staff = $query->result_array();
                
                $formatted_doctors = [];
                foreach ($all_staff as $staff) {
                    $full_name = trim(($staff['name'] ?? 'Unknown') . ' ' . ($staff['surname'] ?? ''));
                    $display_name = $full_name; // Don't add "Dr." for fallback staff
                    
                    if (!empty($staff['specialist'])) {
                        $display_name .= ' (' . $staff['specialist'] . ')';
                    }
                    
                    $formatted_doctors[] = [
                        'id' => $staff['id'],
                        'name' => $staff['name'] ?? 'Unknown',
                        'surname' => $staff['surname'] ?? '',
                        'employee_id' => $staff['employee_id'] ?? ('EMP-' . $staff['id']),
                        'full_name' => $full_name,
                        'display_name' => $display_name,
                        'specialist' => $staff['specialist'] ?? '',
                        'specialization' => $staff['specialization'] ?? '',
                        'designation' => 'Staff (Fallback)'
                    ];
                }
                $doctors = $formatted_doctors;
            }
        }
        
        // Clean output buffer before sending JSON
        ob_clean();
        echo json_encode([
            'status' => 'success',
            'doctors' => $doctors,
            'count' => count($doctors),
            'message' => 'Found ' . count($doctors) . ' doctors',
            'debug' => [
                'method_used' => !empty($doctors) ? 'success' : 'fallback',
                'role_id_searched' => 3
            ]
        ]);
        exit;
        
    } catch (Exception $e) {
        ob_clean();
        echo json_encode([
            'status' => 'error',
            'doctors' => [],
            'count' => 0,
            'error' => 'Exception: ' . $e->getMessage(),
            'debug' => [
                'line' => $e->getLine(),
                'file' => basename($e->getFile()),
                'role_id_searched' => 3
            ]
        ]);
        exit;
    }
}

/**
 * Save consultation with integrated prescription data
 */
public function saveConsultationWithPrescription()
{
    if (!$this->rbac->hasPrivilege('consultant_register', 'can_add')) {
        access_denied();
    }

    $this->form_validation->set_rules('patient_id', 'Patient', 'required|integer');
    $this->form_validation->set_rules('consultation_date', 'Consultation Date', 'required');
    $this->form_validation->set_rules('doctor_id', 'Doctor', 'required|integer');
    $this->form_validation->set_rules('symptoms', 'Symptoms', 'required');

    if ($this->form_validation->run() == false) {
        $response = array(
            'status' => 'error',
            'message' => validation_errors(),
            'errors' => $this->form_validation->error_array()
        );
        echo json_encode($response);
        return;
    }

    try {
        $this->db->trans_start();

        // Prepare consultation data
        $consultation_data = array(
            'patient_id' => $this->input->post('patient_id'),
            'consultation_date' => $this->input->post('consultation_date'),
            'doctor_id' => $this->input->post('doctor_id'),
            'symptoms' => $this->input->post('symptoms'),
            'diagnosis' => $this->input->post('diagnosis'),
            'notes' => $this->input->post('notes'),
            'created_at' => date('Y-m-d H:i:s'),
            'updated_at' => date('Y-m-d H:i:s')
        );

        // Insert consultation
        $this->db->insert('consultations', $consultation_data);
        $consultation_id = $this->db->insert_id();

        // Handle prescriptions if provided
        $prescriptions_json = $this->input->post('prescriptions');
        $prescription_ids = array();
        
        if (!empty($prescriptions_json)) {
            $prescriptions = json_decode($prescriptions_json, true);
            
            if (is_array($prescriptions) && count($prescriptions) > 0) {
                foreach ($prescriptions as $prescription) {
                    $prescription_data = array(
                        'consultation_id' => $consultation_id,
                        'patient_id' => $this->input->post('patient_id'),
                        'medicine_id' => $prescription['drugId'],
                        'quantity' => $prescription['quantity'],
                        'unit_price' => $prescription['unitPrice'],
                        'total_cost' => $prescription['totalCost'],
                        'dosage_instructions' => $prescription['dosageInstructions'],
                        'duration' => $prescription['duration'],
                        'duration_unit' => $prescription['durationUnit'],
                        'administration_time' => $prescription['administrationTime'] ?? null,
                        'special_instructions' => $prescription['specialInstructions'] ?? null,
                        'status' => 'pending',
                        'created_at' => date('Y-m-d H:i:s'),
                        'updated_at' => date('Y-m-d H:i:s')
                    );

                    $this->db->insert('consultation_prescriptions', $prescription_data);
                    $prescription_ids[] = $this->db->insert_id();
                }
            }
        }

        $this->db->trans_complete();

        if ($this->db->trans_status() === false) {
            throw new Exception('Database transaction failed');
        }

        $response = array(
            'status' => 'success',
            'message' => 'Consultation and prescription saved successfully',
            'consultation_id' => $consultation_id,
            'prescription_ids' => $prescription_ids,
            'total_prescriptions' => count($prescription_ids)
        );

    } catch (Exception $e) {
        $this->db->trans_rollback();
        $response = array(
            'status' => 'error',
            'message' => 'Failed to save consultation: ' . $e->getMessage()
        );
    }

    echo json_encode($response);
}
    
    public function getIcdDiagnoses() {
        header('Content-Type: application/json');
        
        try {
            // Load the consultation model if not already loaded
            if (!isset($this->consultation_model)) {
                $this->load->model('consultation_model');
            }
            
            $icdVersion = $this->input->get('version') ?: 'ICD10';
            $limit = $this->input->get('limit') ?: 1000;
            
            $diagnoses = $this->consultation_model->getIcdDiagnosisCodes($icdVersion, $limit);
            
            echo json_encode([
                'status' => 'success',
                'message' => 'ICD diagnoses loaded successfully',
                'diagnoses' => $diagnoses,
                'count' => count($diagnoses)
            ]);
            
        } catch (Exception $e) {
            log_message('error', 'Error loading ICD diagnoses: ' . $e->getMessage());
            echo json_encode([
                'status' => 'error',
                'message' => 'Failed to load ICD diagnoses: ' . $e->getMessage(),
                'diagnoses' => []
            ]);
        }
    }
    
    /**
 * Create OPD prescription table if it doesn't exist
 */
private function createOpdPrescriptionTables() {
    try {
        // Create OPD prescription basic table
        if (!$this->db->table_exists('opd_prescription_basic')) {
            $sql = "CREATE TABLE `opd_prescription_basic` (
                `id` int(11) NOT NULL AUTO_INCREMENT,
                `patient_id` varchar(50) NOT NULL,
                `consultation_id` int(11) DEFAULT NULL,
                `prescription_no` varchar(100) DEFAULT NULL,
                `doctor_id` int(11) NOT NULL,
                `date` date NOT NULL,
                `finding_description` text,
                `header_note` text,
                `footer_note` text,
                `total_medicines` int(11) DEFAULT 0,
                `dispensed_status` enum('fresh','partial','dispensed') DEFAULT 'fresh',
                `visit_type` varchar(50) DEFAULT 'OPD',
                `created_by` int(11) DEFAULT 1,
                `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
                `updated_at` timestamp NULL DEFAULT NULL ON UPDATE CURRENT_TIMESTAMP,
                PRIMARY KEY (`id`),
                KEY `patient_id` (`patient_id`),
                KEY `consultation_id` (`consultation_id`),
                KEY `doctor_id` (`doctor_id`),
                KEY `date` (`date`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8";
            
            $this->db->query($sql);
            log_message('info', 'Created opd_prescription_basic table');
        }
        
        // Create OPD prescription details table
        if (!$this->db->table_exists('opd_prescription_details')) {
            $sql = "CREATE TABLE `opd_prescription_details` (
                `id` int(11) NOT NULL AUTO_INCREMENT,
                `prescription_id` int(11) NOT NULL,
                `medicine_name` varchar(255) NOT NULL,
                `medicine_id` int(11) DEFAULT NULL,
                `dosage` varchar(100) DEFAULT NULL,
                `frequency` varchar(100) DEFAULT NULL,
                `duration` varchar(100) DEFAULT NULL,
                `quantity_prescribed` int(11) DEFAULT NULL,
                `quantity_dispensed` int(11) DEFAULT 0,
                `instructions` text,
                `unit_price` decimal(10,2) DEFAULT 0.00,
                `total_amount` decimal(10,2) DEFAULT 0.00,
                `is_dispensed` enum('no','partial','yes') DEFAULT 'no',
                `dispensed_by` int(11) DEFAULT NULL,
                `dispensed_at` timestamp NULL DEFAULT NULL,
                `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
                PRIMARY KEY (`id`),
                KEY `prescription_id` (`prescription_id`),
                KEY `medicine_id` (`medicine_id`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8";
            
            $this->db->query($sql);
            log_message('info', 'Created opd_prescription_details table');
        }
        
        // Add prescription columns to consultations table if they don't exist
        if (!$this->db->field_exists('has_prescription', 'consultations')) {
            $sql = "ALTER TABLE `consultations` 
                    ADD COLUMN `has_prescription` tinyint(1) DEFAULT 0 AFTER `verification_status`";
            $this->db->query($sql);
        }
        
        if (!$this->db->field_exists('prescription_id', 'consultations')) {
            $sql = "ALTER TABLE `consultations` 
                    ADD COLUMN `prescription_id` int(11) DEFAULT NULL AFTER `has_prescription`";
            $this->db->query($sql);
        }
        
        log_message('info', 'Prescription tables created successfully');
        
    } catch (Exception $e) {
        log_message('error', 'Error creating prescription tables: ' . $e->getMessage());
    }
}

/**
 * Complete saveWithPrescription method - Replace your existing one with this
 */
public function saveWithPrescription() {
    // Clean output buffer and set headers FIRST
    ob_clean();
    header('Content-Type: application/json');
    header('Cache-Control: no-cache, must-revalidate');
    error_reporting(0);
    ini_set('display_errors', 0);

    try {
        log_message('debug', 'saveWithPrescription called with data: ' . json_encode($_POST));
        
        // Create prescription tables if they don't exist
        $this->createOpdPrescriptionTables();
        
        // Start transaction
        $this->db->trans_start();

        // Get and validate form data
        $patient_id = $this->input->post('patient_id');
        $symptoms = $this->input->post('symptoms');
        $doctor_id = $this->input->post('doctor');
        $consultation_date = $this->input->post('consultation_date') ?: date('Y-m-d');
        $notes = $this->input->post('notes') ?: '';
        
        // Get prescription data - check multiple possible field names
        $selected_prescriptions = $this->input->post('selected_prescriptions');
        if (empty($selected_prescriptions) || $selected_prescriptions === '[]') {
            $selected_prescriptions = $this->input->post('selectedPrescriptionsInput') ?: '[]';
        }
        
        log_message('debug', 'Raw prescription data: ' . $selected_prescriptions);
        
        // Validation
        if (!$patient_id || !$symptoms || !$doctor_id) {
            throw new Exception('Missing required fields: patient_id, symptoms, or doctor');
        }

        // Prepare consultation data
        $consultation_data = [
            'patient_id' => $patient_id,
            'doctor' => (int)$doctor_id,
            'date' => $consultation_date,
            'symptoms' => $symptoms,
            'notes' => $notes,
            'verification_status' => $this->input->post('verification_status') == '1' ? 'verified' : 'unverified',
            'verification_score' => $this->input->post('verification_score') ?: 0,
            'verification_quality' => $this->input->post('verification_quality') ?: '',
            'created_by' => $this->getStaffId(),
            'created_at' => date('Y-m-d H:i:s')
        ];

        // Handle diagnosis
        $diagnosis_type = $this->input->post('diagnosis_type');
        if ($diagnosis_type === 'custom' && $this->input->post('diagnosis_custom')) {
            $consultation_data['diagnosis'] = $this->input->post('diagnosis_custom');
        } elseif ($diagnosis_type === 'icd10') {
            $diagnosis_code = $this->input->post('diagnosis_code');
            $diagnosis_title = $this->input->post('diagnosis_title');
            if ($diagnosis_code && $diagnosis_title) {
                $consultation_data['diagnosis'] = $diagnosis_code . ' - ' . $diagnosis_title;
            }
        } elseif ($diagnosis_type === 'gcc_workflow') {
            $consultation_data['diagnosis'] = 'GCC Patient - No ICD diagnosis required';
        }

        // Insert consultation
        $this->db->insert('consultations', $consultation_data);
        $consultation_id = $this->db->insert_id();

        if (!$consultation_id) {
            throw new Exception('Failed to save consultation');
        }

        log_message('debug', 'Consultation saved with ID: ' . $consultation_id);

        // Process prescription data
        $prescription_id = null;
        $prescription_medicines = json_decode($selected_prescriptions, true);
        $prescription_count = 0;
        $prescription_total_cost = 0.00;
        
        log_message('debug', 'Decoded prescriptions: ' . json_encode($prescription_medicines));
        
        if (!empty($prescription_medicines) && is_array($prescription_medicines)) {
            try {
                // Validate prescription data structure
                foreach ($prescription_medicines as $medicine) {
                    if (!isset($medicine['drug_name']) || !isset($medicine['quantity'])) {
                        log_message('error', 'Invalid prescription data structure: ' . json_encode($medicine));
                        continue;
                    }
                }
                
                $prescription_id = $this->savePrescriptionData($consultation_id, $patient_id, $doctor_id, $consultation_date, $prescription_medicines);
                
                // Calculate prescription summary
                $prescription_count = count($prescription_medicines);
                foreach ($prescription_medicines as $medicine) {
                    $prescription_total_cost += floatval($medicine['total_cost'] ?? 0);
                }
                
                log_message('debug', 'Prescription saved with ID: ' . $prescription_id . ', Count: ' . $prescription_count);
                
            } catch (Exception $e) {
                log_message('error', 'Error saving prescription: ' . $e->getMessage());
                // Continue without prescription if it fails
                $prescription_id = null;
            }
        }
        
        // Update consultation with prescription info
        $prescription_update = [
            'has_prescription' => ($prescription_count > 0) ? 1 : 0,
            'prescription_id' => $prescription_id,
            'prescription_count' => $prescription_count,
            'prescription_total_cost' => $prescription_total_cost,
            'prescription_status' => ($prescription_count > 0) ? 'pending' : 'none',
            'general_prescription_notes' => $this->input->post('prescription_notes') ?: ($prescription_count > 0 ? 'Take medicines as prescribed by the doctor' : ''),
            'pharmacy_notes' => $this->input->post('pharmacy_notes') ?: ''
        ];
        
        $this->db->where('id', $consultation_id);
        $this->db->update('consultations', $prescription_update);
        
        log_message('debug', 'Consultation updated with prescription info');

        // Complete transaction
        $this->db->trans_complete();

        if ($this->db->trans_status() === FALSE) {
            throw new Exception('Transaction failed');
        }

        // Prepare success response
        $response = [
            'status' => 'success',
            'message' => 'Consultation saved successfully',
            'consultation_id' => $consultation_id,
            'prescription_saved' => ($prescription_id ? true : false),
            'prescription_id' => $prescription_id,
            'total_medicines' => $prescription_count,
            'prescription_total_cost' => $prescription_total_cost
        ];

        if ($prescription_count > 0) {
            $response['message'] .= ' with ' . $prescription_count . ' prescribed medicines';
        }

        log_message('debug', 'Success response: ' . json_encode($response));

        ob_clean();
        echo json_encode($response);
        exit;

    } catch (Exception $e) {
        $this->db->trans_rollback();
        
        log_message('error', 'Error saving consultation with prescription: ' . $e->getMessage());
        
        ob_clean();
        echo json_encode([
            'status' => 'error',
            'message' => 'Error saving consultation: ' . $e->getMessage(),
            'debug' => [
                'line' => $e->getLine(),
                'file' => basename($e->getFile()),
                'post_data' => $_POST
            ]
        ]);
        exit;
    }
}

/**
 * Debug method to see what prescription data is being received
 * Add this to your Consultation controller temporarily
 */
public function debugPrescriptionSubmission() {
    ob_clean();
    header('Content-Type: application/json');
    header('Cache-Control: no-cache, must-revalidate');
    error_reporting(E_ALL);
    ini_set('display_errors', 0);

    try {
        $debug_info = [
            'timestamp' => date('Y-m-d H:i:s'),
            'request_method' => $_SERVER['REQUEST_METHOD'],
            'content_type' => $_SERVER['CONTENT_TYPE'] ?? 'not_set',
            'all_post_data' => $_POST,
            'all_get_data' => $_GET,
            'raw_input' => file_get_contents('php://input'),
            'prescription_analysis' => []
        ];

        // Check specific prescription fields
        $prescription_fields = [
            'selected_prescriptions',
            'total_prescriptions', 
            'prescription_notes',
            'pharmacy_notes'
        ];

        foreach ($prescription_fields as $field) {
            $value = $this->input->post($field);
            $debug_info['prescription_analysis'][$field] = [
                'exists' => isset($_POST[$field]),
                'value' => $value,
                'length' => strlen($value ?? ''),
                'type' => gettype($value)
            ];
        }

        // Try to decode prescription JSON
        $selected_prescriptions = $this->input->post('selected_prescriptions');
        if ($selected_prescriptions) {
            $decoded = json_decode($selected_prescriptions, true);
            $debug_info['prescription_analysis']['json_decode'] = [
                'success' => (json_last_error() === JSON_ERROR_NONE),
                'error' => json_last_error_msg(),
                'decoded_data' => $decoded,
                'is_array' => is_array($decoded),
                'count' => is_array($decoded) ? count($decoded) : 0
            ];

            if (is_array($decoded) && !empty($decoded)) {
                $debug_info['prescription_analysis']['first_prescription'] = $decoded[0];
                $debug_info['prescription_analysis']['required_fields_check'] = [];
                
                $required_fields = ['drug_id', 'drug_name', 'quantity', 'unit_price', 'total_cost'];
                foreach ($required_fields as $req_field) {
                    $debug_info['prescription_analysis']['required_fields_check'][$req_field] = 
                        isset($decoded[0][$req_field]) ? 'present' : 'missing';
                }
            }
        }

        // Check form validation
        $debug_info['form_validation'] = [
            'patient_id' => $this->input->post('patient_id') ? 'present' : 'missing',
            'symptoms' => $this->input->post('symptoms') ? 'present' : 'missing', 
            'doctor' => $this->input->post('doctor') ? 'present' : 'missing',
            'consultation_date' => $this->input->post('consultation_date') ? 'present' : 'missing'
        ];

        // Test if the data would pass validation
        $patient_id = $this->input->post('patient_id');
        $symptoms = $this->input->post('symptoms');
        $doctor_id = $this->input->post('doctor');
        
        $debug_info['would_pass_validation'] = (!empty($patient_id) && !empty($symptoms) && !empty($doctor_id));

        ob_clean();
        echo json_encode([
            'status' => 'success',
            'message' => 'Prescription submission debug completed',
            'debug_info' => $debug_info
        ], JSON_PRETTY_PRINT);
        exit;

    } catch (Exception $e) {
        ob_clean();
        echo json_encode([
            'status' => 'error',
            'message' => 'Debug failed: ' . $e->getMessage(),
            'debug_info' => $debug_info ?? []
        ]);
        exit;
    }
}

/**
 * Simple test to simulate prescription save
 */
public function testPrescriptionSave() {
    ob_clean();
    header('Content-Type: application/json');
    header('Cache-Control: no-cache, must-revalidate');
    error_reporting(0);
    ini_set('display_errors', 0);

    try {
        // Create test prescription data
        $test_prescription = [
            [
                'drug_id' => 1,
                'drug_name' => 'Test Medicine',
                'quantity' => 10,
                'unit_price' => 5.50,
                'total_cost' => 55.00,
                'dosage_instructions' => 'Take once daily after meals',
                'duration' => '7',
                'duration_unit' => 'days',
                'special_instructions' => 'Take with water'
            ]
        ];

        // Simulate the prescription save process
        $this->createOpdPrescriptionTables();
        
        $consultation_id = 999; // Test consultation ID
        $patient_id = 134;
        $doctor_id = 3;
        $date = date('Y-m-d');
        
        $prescription_id = $this->savePrescriptionData($consultation_id, $patient_id, $doctor_id, $date, $test_prescription);
        
        ob_clean();
        echo json_encode([
            'status' => 'success',
            'message' => 'Test prescription save completed',
            'test_data' => $test_prescription,
            'prescription_id' => $prescription_id,
            'tables_exist' => [
                'opd_prescription_basic' => $this->db->table_exists('opd_prescription_basic'),
                'opd_prescription_details' => $this->db->table_exists('opd_prescription_details')
            ]
        ]);
        exit;

    } catch (Exception $e) {
        ob_clean();
        echo json_encode([
            'status' => 'error',
            'message' => 'Test failed: ' . $e->getMessage(),
            'line' => $e->getLine(),
            'file' => basename($e->getFile())
        ]);
        exit;
    }
}

/**
 * Get medicine details for prescription modal
 */
public function getMedicineDetails() {
    header('Content-Type: application/json');
    
    try {
        $medicine_id = $this->input->post('medicine_id') ?: $this->input->get('medicine_id');
        
        if (!$medicine_id) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Medicine ID required'
            ]);
            return;
        }
        
        // Check if medicine tables exist
        $medicine_table = 'medicine';
        if (!$this->db->table_exists('medicine')) {
            if ($this->db->table_exists('medicines')) {
                $medicine_table = 'medicines';
            } else {
                // Return sample data
                $sample_medicines = $this->getSampleMedicines();
                $medicine = null;
                foreach ($sample_medicines as $med) {
                    if ($med['id'] == $medicine_id) {
                        $medicine = $med;
                        break;
                    }
                }
                
                if ($medicine) {
                    echo json_encode([
                        'status' => 'success',
                        'medicine' => $medicine
                    ]);
                } else {
                    echo json_encode([
                        'status' => 'error',
                        'message' => 'Medicine not found'
                    ]);
                }
                return;
            }
        }
        
        // Get medicine details
        $this->db->select('m.*, mc.medicine_category');
        $this->db->from($medicine_table . ' m');
        $this->db->join('medicine_category mc', 'm.medicine_category_id = mc.id', 'left');
        $this->db->where('m.id', $medicine_id);
        
        $query = $this->db->get();
        
        if ($query->num_rows() > 0) {
            $medicine = $query->row_array();
            
            // Format medicine data
            $medicine['name'] = $medicine['medicine_name'];
            $medicine['category'] = $medicine['medicine_category'] ?? 'Unknown';
            $medicine['unit'] = $medicine['medicine_unit'] ?? 'Unit';
            $medicine['cost'] = floatval($medicine['sell_price'] ?? $medicine['price'] ?? 0);
            $medicine['stock_quantity'] = intval($medicine['available_quantity'] ?? 100);
            $medicine['composition'] = $medicine['composition'] ?? $medicine['generic_name'] ?? '';
            
            echo json_encode([
                'status' => 'success',
                'medicine' => $medicine
            ]);
        } else {
            echo json_encode([
                'status' => 'error',
                'message' => 'Medicine not found'
            ]);
        }
        
    } catch (Exception $e) {
        echo json_encode([
            'status' => 'error',
            'message' => 'Error loading medicine details: ' . $e->getMessage()
        ]);
    }
}

public function getMedicinesForPrescription() {
    ob_clean();
    header('Content-Type: application/json');
    header('Cache-Control: no-cache, must-revalidate');
    error_reporting(0);
    ini_set('display_errors', 0);
    
    try {
        $search = $this->input->get('search') ?: $this->input->post('search');
        $category_id = $this->input->get('category_id') ?: $this->input->post('category_id');
        $page = intval($this->input->get('page') ?: $this->input->post('page') ?: 1);
        $limit = 20;
        $offset = ($page - 1) * $limit;
        
        // Check if medicine tables exist
        $medicine_table = 'medicine';
        if (!$this->db->table_exists('medicine')) {
            if ($this->db->table_exists('medicines')) {
                $medicine_table = 'medicines';
            } else {
                // Create sample medicines data
                ob_clean();
                echo json_encode([
                    'status' => 'success',
                    'medicines' => $this->getSampleMedicines($search),
                    'pagination' => [
                        'current_page' => 1,
                        'total_pages' => 1,
                        'total_records' => count($this->getSampleMedicines($search))
                    ],
                    'categories' => $this->getSampleCategories(),
                    'message' => 'Using sample medicine data - no medicine table found'
                ]);
                exit;
            }
        }
        
        // Build query
        $this->db->select('m.*, mc.medicine_category');
        $this->db->from($medicine_table . ' m');
        $this->db->join('medicine_category mc', 'm.medicine_category_id = mc.id', 'left');
        $this->db->where('m.is_active', 'yes');
        
        // Add search conditions
        if (!empty($search)) {
            $this->db->group_start();
            $this->db->like('m.medicine_name', $search);
            $this->db->or_like('m.generic_name', $search);
            $this->db->or_like('m.medicine_type', $search);
            $this->db->or_like('mc.medicine_category', $search);
            $this->db->group_end();
        }
        
        if (!empty($category_id)) {
            $this->db->where('m.medicine_category_id', $category_id);
        }
        
        // Get total count for pagination
        $total_query = clone $this->db;
        $total_records = $total_query->count_all_results();
        $total_pages = ceil($total_records / $limit);
        
        // Apply pagination
        $this->db->limit($limit, $offset);
        $this->db->order_by('m.medicine_name', 'ASC');
        
        $query = $this->db->get();
        
        if (!$query) {
            throw new Exception('Database query failed: ' . $this->db->error()['message']);
        }
        
        $medicines = $query->result_array();
        
        // Format medicines data
        foreach ($medicines as &$medicine) {
            $medicine['id'] = $medicine['id'];
            $medicine['name'] = $medicine['medicine_name'];
            $medicine['generic_name'] = $medicine['generic_name'] ?? '';
            $medicine['category'] = $medicine['medicine_category'] ?? 'Unknown';
            $medicine['unit'] = $medicine['medicine_unit'] ?? 'Unit';
            $medicine['cost'] = floatval($medicine['sell_price'] ?? $medicine['price'] ?? 0);
            $medicine['stock_quantity'] = intval($medicine['available_quantity'] ?? 100);
            $medicine['composition'] = $medicine['composition'] ?? $medicine['generic_name'] ?? '';
            $medicine['dosage_form'] = $medicine['dosage_form'] ?? $medicine['medicine_type'] ?? 'Tablet';
            
            // Stock status
            if ($medicine['stock_quantity'] <= 0) {
                $medicine['stock_status'] = 'out_of_stock';
            } elseif ($medicine['stock_quantity'] <= 10) {
                $medicine['stock_status'] = 'low_stock';
            } else {
                $medicine['stock_status'] = 'in_stock';
            }
        }
        
        // Get categories for filter
        $categories = $this->getMedicineCategories();
        
        ob_clean();
        echo json_encode([
            'status' => 'success',
            'medicines' => $medicines,
            'pagination' => [
                'current_page' => $page,
                'total_pages' => $total_pages,
                'total_records' => $total_records,
                'per_page' => $limit
            ],
            'categories' => $categories,
            'search_term' => $search,
            'message' => 'Medicines loaded successfully'
        ]);
        exit;
        
    } catch (Exception $e) {
        ob_clean();
        echo json_encode([
            'status' => 'error',
            'message' => 'Error loading medicines: ' . $e->getMessage(),
            'medicines' => [],
            'categories' => []
        ]);
        exit;
    }
}

public function checkConsultationPrescriptions() {
    ob_clean();
    header('Content-Type: application/json');
    header('Cache-Control: no-cache, must-revalidate');
    error_reporting(0);
    ini_set('display_errors', 0);

    try {
        $consultation_id = $this->input->post('consultation_id') ?: $this->input->get('consultation_id');
        
        if ($consultation_id) {
            // Get specific consultation
            $this->db->select('*');
            $this->db->from('consultations');
            $this->db->where('id', $consultation_id);
            $consultation = $this->db->get()->row_array();
            
            $result = [
                'consultation_id' => $consultation_id,
                'consultation_found' => !empty($consultation),
                'consultation_data' => $consultation
            ];
            
            if ($consultation) {
                // Check prescription fields specifically
                $result['prescription_analysis'] = [
                    'has_prescription' => $consultation['has_prescription'] ?? 'field_missing',
                    'prescription_id' => $consultation['prescription_id'] ?? 'field_missing',
                    'prescription_count' => $consultation['prescription_count'] ?? 'field_missing',
                    'prescription_total_cost' => $consultation['prescription_total_cost'] ?? 'field_missing',
                    'prescription_status' => $consultation['prescription_status'] ?? 'field_missing'
                ];
                
                // If prescription_id exists, try to get prescription details
                if (!empty($consultation['prescription_id'])) {
                    if ($this->db->table_exists('opd_prescription_basic')) {
                        $this->db->select('*');
                        $this->db->from('opd_prescription_basic');
                        $this->db->where('id', $consultation['prescription_id']);
                        $prescription_basic = $this->db->get()->row_array();
                        
                        $result['prescription_details'] = [
                            'prescription_found' => !empty($prescription_basic),
                            'prescription_data' => $prescription_basic
                        ];
                        
                        if ($prescription_basic) {
                            // Get prescription medicines
                            $this->db->select('*');
                            $this->db->from('opd_prescription_details');
                            $this->db->where('prescription_id', $consultation['prescription_id']);
                            $medicines = $this->db->get()->result_array();
                            
                            $result['prescription_details']['medicines'] = $medicines;
                            $result['prescription_details']['medicine_count'] = count($medicines);
                        }
                    } else {
                        $result['prescription_details'] = ['error' => 'opd_prescription_basic table does not exist'];
                    }
                }
            }
        } else {
            // Get recent consultations with prescription data
            $this->db->select('id, patient_id, date, has_prescription, prescription_id, prescription_count, prescription_total_cost, prescription_status');
            $this->db->from('consultations');
            $this->db->order_by('created_at', 'DESC');
            $this->db->limit(10);
            $recent_consultations = $this->db->get()->result_array();
            
            $result = [
                'recent_consultations' => $recent_consultations,
                'consultations_with_prescriptions' => []
            ];
            
            foreach ($recent_consultations as $consultation) {
                if (!empty($consultation['has_prescription']) && $consultation['has_prescription'] == 1) {
                    $result['consultations_with_prescriptions'][] = $consultation;
                }
            }
            
            $result['summary'] = [
                'total_recent_consultations' => count($recent_consultations),
                'consultations_with_prescriptions_count' => count($result['consultations_with_prescriptions'])
            ];
        }

        ob_clean();
        echo json_encode([
            'status' => 'success',
            'message' => 'Consultation prescription check completed',
            'data' => $result
        ], JSON_PRETTY_PRINT);
        exit;

    } catch (Exception $e) {
        ob_clean();
        echo json_encode([
            'status' => 'error',
            'message' => 'Check failed: ' . $e->getMessage()
        ]);
        exit;
    }
}

public function debugPrescriptionData() {
    ob_clean();
    header('Content-Type: application/json');
    header('Cache-Control: no-cache, must-revalidate');
    error_reporting(E_ALL);
    ini_set('display_errors', 0);

    try {
        $debug_info = [
            'timestamp' => date('Y-m-d H:i:s'),
            'request_method' => $_SERVER['REQUEST_METHOD'],
            'post_data' => $_POST,
            'prescription_data_analysis' => []
        ];

        // Check if prescription data is being sent
        $selected_prescriptions = $this->input->post('selected_prescriptions');
        
        if ($selected_prescriptions) {
            $debug_info['prescription_data_analysis']['raw_prescription_data'] = $selected_prescriptions;
            
            // Try to decode JSON
            $decoded_prescriptions = json_decode($selected_prescriptions, true);
            $debug_info['prescription_data_analysis']['json_decode_success'] = (json_last_error() === JSON_ERROR_NONE);
            $debug_info['prescription_data_analysis']['json_error'] = json_last_error_msg();
            $debug_info['prescription_data_analysis']['decoded_data'] = $decoded_prescriptions;
            $debug_info['prescription_data_analysis']['prescription_count'] = is_array($decoded_prescriptions) ? count($decoded_prescriptions) : 0;
            
            if (is_array($decoded_prescriptions) && !empty($decoded_prescriptions)) {
                $debug_info['prescription_data_analysis']['first_prescription'] = $decoded_prescriptions[0];
                $debug_info['prescription_data_analysis']['total_cost'] = array_sum(array_column($decoded_prescriptions, 'total_cost'));
            }
        } else {
            $debug_info['prescription_data_analysis']['error'] = 'No prescription data received';
        }

        // Check consultation table structure
        if ($this->db->table_exists('consultations')) {
            $debug_info['table_structure'] = [
                'consultations_exists' => true,
                'prescription_fields' => []
            ];
            
            $prescription_fields = [
                'has_prescription',
                'prescription_id', 
                'prescription_count',
                'prescription_total_cost',
                'prescription_status',
                'general_prescription_notes',
                'pharmacy_notes'
            ];
            
            foreach ($prescription_fields as $field) {
                $debug_info['table_structure']['prescription_fields'][$field] = $this->db->field_exists($field, 'consultations');
            }
        } else {
            $debug_info['table_structure'] = ['consultations_exists' => false];
        }

        // Check if opd_prescription tables exist
        $debug_info['prescription_tables'] = [
            'opd_prescription_basic' => $this->db->table_exists('opd_prescription_basic'),
            'opd_prescription_details' => $this->db->table_exists('opd_prescription_details')
        ];

        ob_clean();
        echo json_encode([
            'status' => 'success',
            'message' => 'Prescription debug completed',
            'debug_info' => $debug_info
        ], JSON_PRETTY_PRINT);
        exit;

    } catch (Exception $e) {
        ob_clean();
        echo json_encode([
            'status' => 'error',
            'message' => 'Debug failed: ' . $e->getMessage(),
            'debug_info' => $debug_info ?? []
        ]);
        exit;
    }
}

/**
 * Save prescription data to database
 */
private function savePrescriptionData($consultation_id, $patient_id, $doctor_id, $date, $medicines) {
    try {
        // Generate prescription number
        $prescription_no = $this->generatePrescriptionNumber();
        
        // Create prescription basic record
        $prescription_basic = [
            'patient_id' => $patient_id,
            'consultation_id' => $consultation_id,
            'prescription_no' => $prescription_no,
            'doctor_id' => $doctor_id,
            'date' => $date,
            'finding_description' => $this->input->post('diagnosis') ?: $this->input->post('diagnosis_custom'),
            'header_note' => $this->input->post('prescription_header') ?: '',
            'footer_note' => $this->input->post('prescription_footer') ?: 'Take medicines as directed by the doctor.',
            'total_medicines' => count($medicines),
            'dispensed_status' => 'fresh',
            'visit_type' => 'OPD',
            'created_by' => $this->getStaffId(),
            'created_at' => date('Y-m-d H:i:s')
        ];
        
        $this->db->insert('opd_prescription_basic', $prescription_basic);
        $prescription_id = $this->db->insert_id();
        
        if (!$prescription_id) {
            throw new Exception('Failed to create prescription record');
        }
        
        // Save prescription medicines
        foreach ($medicines as $medicine) {
            $medicine_detail = [
                'prescription_id' => $prescription_id,
                'medicine_name' => $medicine['drug_name'],
                'medicine_id' => isset($medicine['drug_id']) ? $medicine['drug_id'] : null,
                'dosage' => $medicine['dosage_instructions'] ?: 'As directed',
                'frequency' => $this->extractFrequency($medicine['dosage_instructions']),
                'duration' => ($medicine['duration'] ?? '') . ' ' . ($medicine['duration_unit'] ?? ''),
                'quantity_prescribed' => $medicine['quantity'],
                'instructions' => $medicine['special_instructions'] ?: 'Take as prescribed',
                'unit_price' => floatval($medicine['unit_price'] ?? 0),
                'total_amount' => floatval($medicine['total_cost'] ?? 0),
                'is_dispensed' => 'no',
                'created_at' => date('Y-m-d H:i:s')
            ];
            
            $this->db->insert('opd_prescription_details', $medicine_detail);
        }
        
        log_message('info', 'Prescription saved with ID: ' . $prescription_id);
        return $prescription_id;
        
    } catch (Exception $e) {
        log_message('error', 'Error saving prescription: ' . $e->getMessage());
        throw new Exception('Failed to save prescription: ' . $e->getMessage());
    }
}

/**
 * Generate unique prescription number
 */
private function generatePrescriptionNumber() {
    $prefix = 'OPDP'; // OPD Prescription prefix
    $year = date('Y');
    $month = date('m');
    
    // Get the last prescription number for current month
    $this->db->select('prescription_no');
    $this->db->from('opd_prescription_basic');
    $this->db->like('prescription_no', $prefix . $year . $month, 'after');
    $this->db->order_by('id', 'DESC');
    $this->db->limit(1);
    
    $query = $this->db->get();
    
    if ($query->num_rows() > 0) {
        $last_no = $query->row()->prescription_no;
        $last_sequence = (int)substr($last_no, -4);
        $new_sequence = $last_sequence + 1;
    } else {
        $new_sequence = 1;
    }
    
    return $prefix . $year . $month . str_pad($new_sequence, 4, '0', STR_PAD_LEFT);
}

/**
 * Extract frequency from dosage instructions
 */
private function extractFrequency($dosage) {
    if (empty($dosage)) return 'As directed';
    
    $dosage_lower = strtolower($dosage);
    
    if (strpos($dosage_lower, 'once daily') !== false || strpos($dosage_lower, '1 time daily') !== false) {
        return 'Once daily';
    } elseif (strpos($dosage_lower, 'twice daily') !== false || strpos($dosage_lower, '2 times daily') !== false) {
        return 'Twice daily';
    } elseif (strpos($dosage_lower, 'three times daily') !== false || strpos($dosage_lower, '3 times daily') !== false) {
        return 'Three times daily';
    } elseif (strpos($dosage_lower, 'four times daily') !== false || strpos($dosage_lower, '4 times daily') !== false) {
        return 'Four times daily';
    } elseif (strpos($dosage_lower, 'every 4 hours') !== false) {
        return 'Every 4 hours';
    } elseif (strpos($dosage_lower, 'every 6 hours') !== false) {
        return 'Every 6 hours';
    } elseif (strpos($dosage_lower, 'every 8 hours') !== false) {
        return 'Every 8 hours';
    } elseif (strpos($dosage_lower, 'as needed') !== false || strpos($dosage_lower, 'prn') !== false) {
        return 'As needed';
    } else {
        return 'As directed';
    }
}
/**
 * Get patient prescriptions for dispensing
 */
public function getPatientPrescriptions() {
    ob_clean();
    header('Content-Type: application/json');
    header('Cache-Control: no-cache, must-revalidate');
    error_reporting(0);
    ini_set('display_errors', 0);

    try {
        $patient_id = $this->input->post('patient_id');
        
        if (!$patient_id) {
            ob_clean();
            echo json_encode([
                'status' => 'error',
                'message' => 'Patient ID required',
                'prescriptions' => []
            ]);
            exit;
        }
        
        // Check if prescription tables exist
        if (!$this->db->table_exists('opd_prescription_basic')) {
            // Create tables if they don't exist
            $this->createOpdPrescriptionTables();
            
            ob_clean();
            echo json_encode([
                'status' => 'success',
                'prescriptions' => [],
                'message' => 'No prescriptions found - tables created'
            ]);
            exit;
        }
        
        // Get prescriptions for this patient
        $this->db->select('pb.*, s.name as doctor_name, s.surname as doctor_surname, c.symptoms, c.diagnosis');
        $this->db->from('opd_prescription_basic pb');
        $this->db->join('staff s', 'pb.doctor_id = s.id', 'left');
        $this->db->join('consultations c', 'pb.consultation_id = c.id', 'left');
        $this->db->where('pb.patient_id', $patient_id);
        $this->db->order_by('pb.created_at', 'DESC');
        $this->db->limit(10); // Last 10 prescriptions
        
        $query = $this->db->get();
        
        if (!$query) {
            ob_clean();
            echo json_encode([
                'status' => 'error',
                'message' => 'Database query failed',
                'prescriptions' => []
            ]);
            exit;
        }
        
        $prescriptions = $query->result_array();
        
        // Get medicines for each prescription
        foreach ($prescriptions as &$prescription) {
            // Format doctor name
            $prescription['doctor_name'] = trim(($prescription['doctor_name'] ?: '') . ' ' . ($prescription['doctor_surname'] ?: ''));
            if (empty(trim($prescription['doctor_name']))) {
                $prescription['doctor_name'] = 'Unknown Doctor';
            }
            
            // Get medicines for this prescription
            $this->db->select('*');
            $this->db->from('opd_prescription_details');
            $this->db->where('prescription_id', $prescription['id']);
            $this->db->order_by('id', 'ASC');
            
            $medicines_query = $this->db->get();
            $prescription['medicines'] = $medicines_query ? $medicines_query->result_array() : [];
            
            // Format prescription date
            $prescription['prescription_date'] = $prescription['date'];
            $prescription['prescription_date_formatted'] = date('M j, Y', strtotime($prescription['date']));
            
            // Set defaults
            $prescription['visit_type'] = $prescription['visit_type'] ?: 'OPD';
            $prescription['finding_description'] = $prescription['finding_description'] ?: $prescription['diagnosis'];
        }
        
        ob_clean();
        echo json_encode([
            'status' => 'success',
            'prescriptions' => $prescriptions,
            'total_found' => count($prescriptions),
            'message' => 'Prescriptions loaded successfully'
        ]);
        exit;
        
    } catch (Exception $e) {
        ob_clean();
        echo json_encode([
            'status' => 'error',
            'message' => 'Error loading prescriptions: ' . $e->getMessage(),
            'prescriptions' => []
        ]);
        exit;
    }
}

/**
 * Get prescription details by ID
 */
public function getPrescriptionDetails() {
    header('Content-Type: application/json');
    
    try {
        $prescription_id = $this->input->post('prescription_id');
        
        if (!$prescription_id) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Prescription ID required'
            ]);
            return;
        }
        
        // Get prescription basic info
        $this->db->select('pb.*, s.name as doctor_name, s.surname as doctor_surname, p.patient_name');
        $this->db->from('opd_prescription_basic pb');
        $this->db->join('staff s', 'pb.doctor_id = s.id', 'left');
        $this->db->join('patients p', 'pb.patient_id = p.id', 'left');
        $this->db->where('pb.id', $prescription_id);
        
        $query = $this->db->get();
        
        if ($query->num_rows() == 0) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Prescription not found'
            ]);
            return;
        }
        
        $prescription = $query->row_array();
        
        // Format doctor name
        $prescription['doctor_name'] = trim(($prescription['doctor_name'] ?: '') . ' ' . ($prescription['doctor_surname'] ?: ''));
        
        // Get prescription medicines
        $this->db->select('*');
        $this->db->from('opd_prescription_details');
        $this->db->where('prescription_id', $prescription_id);
        $this->db->order_by('id', 'ASC');
        
        $medicines_query = $this->db->get();
        $prescription['medicines'] = $medicines_query ? $medicines_query->result_array() : [];
        
        echo json_encode([
            'status' => 'success',
            'prescription' => $prescription
        ]);
        
    } catch (Exception $e) {
        echo json_encode([
            'status' => 'error',
            'message' => 'Error loading prescription: ' . $e->getMessage()
        ]);
    }
}

/**
 * Update prescription dispensing status
 */
public function updatePrescriptionDispensing() {
    header('Content-Type: application/json');
    
    try {
        $prescription_id = $this->input->post('prescription_id');
        $medicine_id = $this->input->post('medicine_id');
        $dispensed_quantity = $this->input->post('dispensed_quantity');
        $dispensed_by = $this->getStaffId();
        
        if (!$prescription_id || !$medicine_id || !$dispensed_quantity) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Missing required fields'
            ]);
            return;
        }
        
        $this->db->trans_start();
        
        // Update medicine dispensing record
        $this->db->where('id', $medicine_id);
        $this->db->where('prescription_id', $prescription_id);
        $this->db->update('opd_prescription_details', [
            'quantity_dispensed' => $dispensed_quantity,
            'dispensed_by' => $dispensed_by,
            'dispensed_at' => date('Y-m-d H:i:s'),
            'is_dispensed' => 'yes'
        ]);
        
        // Check if all medicines in prescription are dispensed
        $this->db->select('COUNT(*) as total, SUM(CASE WHEN is_dispensed = "yes" THEN 1 ELSE 0 END) as dispensed');
        $this->db->from('opd_prescription_details');
        $this->db->where('prescription_id', $prescription_id);
        $result = $this->db->get()->row();
        
        $dispensed_status = 'partial';
        if ($result->dispensed == $result->total) {
            $dispensed_status = 'dispensed';
        } elseif ($result->dispensed == 0) {
            $dispensed_status = 'fresh';
        }
        
        // Update prescription status
        $this->db->where('id', $prescription_id);
        $this->db->update('opd_prescription_basic', [
            'dispensed_status' => $dispensed_status
        ]);
        
        $this->db->trans_complete();
        
        if ($this->db->trans_status() === FALSE) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Failed to update dispensing status'
            ]);
        } else {
            echo json_encode([
                'status' => 'success',
                'message' => 'Dispensing status updated successfully',
                'prescription_status' => $dispensed_status
            ]);
        }
        
    } catch (Exception $e) {
        $this->db->trans_rollback();
        echo json_encode([
            'status' => 'error',
            'message' => 'Error updating dispensing: ' . $e->getMessage()
        ]);
    }
}

/**
 * Get prescription history for a patient
 */
public function getPrescriptionHistory() {
    header('Content-Type: application/json');
    
    try {
        $patient_id = $this->input->post('patient_id');
        $limit = $this->input->post('limit') ?: 20;
        
        if (!$patient_id) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Patient ID required'
            ]);
            return;
        }
        
        // Get prescription history
        $this->db->select('pb.*, s.name as doctor_name, s.surname as doctor_surname, 
                          COUNT(pd.id) as total_medicines,
                          SUM(CASE WHEN pd.is_dispensed = "yes" THEN 1 ELSE 0 END) as dispensed_medicines');
        $this->db->from('opd_prescription_basic pb');
        $this->db->join('staff s', 'pb.doctor_id = s.id', 'left');
        $this->db->join('opd_prescription_details pd', 'pb.id = pd.prescription_id', 'left');
        $this->db->where('pb.patient_id', $patient_id);
        $this->db->group_by('pb.id');
        $this->db->order_by('pb.created_at', 'DESC');
        $this->db->limit($limit);
        
        $query = $this->db->get();
        $prescriptions = $query->result_array();
        
        // Format the data
        foreach ($prescriptions as &$prescription) {
            $prescription['doctor_name'] = trim(($prescription['doctor_name'] ?: '') . ' ' . ($prescription['doctor_surname'] ?: ''));
            $prescription['date_formatted'] = date('M j, Y', strtotime($prescription['date']));
        }
        
        echo json_encode([
            'status' => 'success',
            'prescriptions' => $prescriptions,
            'total_found' => count($prescriptions)
        ]);
        
    } catch (Exception $e) {
        echo json_encode([
            'status' => 'error',
            'message' => 'Error loading prescription history: ' . $e->getMessage()
        ]);
    }
}

/**
 * Test prescription functionality
 */
public function testPrescriptionSystem() {
    header('Content-Type: application/json');
    
    try {
        $results = [
            'timestamp' => date('Y-m-d H:i:s'),
            'table_checks' => [],
            'sample_data' => []
        ];
        
        // Check if tables exist
        $results['table_checks']['opd_prescription_basic'] = $this->db->table_exists('opd_prescription_basic');
        $results['table_checks']['opd_prescription_details'] = $this->db->table_exists('opd_prescription_details');
        $results['table_checks']['consultations'] = $this->db->table_exists('consultations');
        
        // Create tables if they don't exist
        if (!$results['table_checks']['opd_prescription_basic'] || !$results['table_checks']['opd_prescription_details']) {
            $this->createOpdPrescriptionTables();
            $results['action_taken'] = 'Created missing prescription tables';
        }
        
        // Get sample data
        if ($this->db->table_exists('opd_prescription_basic')) {
            $this->db->from('opd_prescription_basic');
            $results['sample_data']['total_prescriptions'] = $this->db->count_all_results();
            
            $this->db->select('*');
            $this->db->from('opd_prescription_basic');
            $this->db->limit(5);
            $query = $this->db->get();
            $results['sample_data']['recent_prescriptions'] = $query->result_array();
        }
        
        echo json_encode([
            'status' => 'success',
            'message' => 'Prescription system test completed',
            'results' => $results
        ]);
        
    } catch (Exception $e) {
        echo json_encode([
            'status' => 'error',
            'message' => 'Test failed: ' . $e->getMessage()
        ]);
    }
}
    
/**
 * ENHANCED VERSION with error handling - Save consultation with financials
 */
public function saveWithFinancials() {
    // Set proper headers first
    header('Content-Type: application/json');
    header('Cache-Control: no-cache, must-revalidate');
    
    // Handle errors properly
    error_reporting(E_ALL);
    ini_set('display_errors', 0);
    
    // Start output buffering to catch any unwanted output
    ob_start();
    
    try {
        // Clear any previous output
        ob_clean();
        
        log_message('debug', 'saveWithFinancials started');
        log_message('debug', 'POST data: ' . json_encode($_POST));
        
        // Ensure tables exist
        $this->ensureTablesExist();
        
        // Validate required fields
        $patient_id = $this->input->post('patient_id');
        $symptoms = $this->input->post('symptoms');
        $doctor = $this->input->post('doctor');
        $consultation_date = $this->input->post('consultation_date'); // This comes from form
        $consultation_charge = $this->input->post('consultation_charge');
        
        if (!$patient_id || !$symptoms || !$doctor) { // Remove consultation_date requirement if optional
            ob_clean();
            echo json_encode(array(
                'status' => 'error',
                'message' => 'Missing required fields: patient_id, symptoms, or doctor'
            ));
            ob_end_flush();
            return;
        }
        
        // Start transaction
        $this->db->trans_start();
        
        // Prepare consultation data - FIXED: Use correct database column names
        $consultationData = array(
            'patient_id' => (int)$patient_id,
            'doctor' => (int)$doctor,
            'date' => $consultation_date ?: date('Y-m-d'), // Use 'date' not 'consultation_date'
            'symptoms' => $symptoms,
            'notes' => $this->input->post('notes') ?: '',
            'verification_status' => $this->input->post('verification_status') == '1' ? 'verified' : 'unverified',
            'created_by' => $this->session->userdata('admin_id') ?: 1,
            'created_at' => date('Y-m-d H:i:s')
        );
        
        // Handle diagnosis
        $diagnosisType = $this->input->post('diagnosis_type');
        if ($diagnosisType === 'custom' && $this->input->post('diagnosis_custom')) {
            $consultationData['diagnosis'] = $this->input->post('diagnosis_custom');
        } elseif ($diagnosisType === 'icd10' && $this->input->post('diagnosis_title')) {
            $diagnosisCode = $this->input->post('diagnosis_code');
            $diagnosisTitle = $this->input->post('diagnosis_title');
            $consultationData['diagnosis'] = $diagnosisCode . ' - ' . $diagnosisTitle;
        } else {
            $consultationData['diagnosis'] = '';
        }
        
        // Insert consultation
        $this->db->insert('consultations', $consultationData);
        $consultationId = $this->db->insert_id();
        
        if (!$consultationId) {
            throw new Exception('Failed to insert consultation');
        }
        
        // Handle financial data if provided
        if ($consultation_charge && floatval($consultation_charge) > 0) {
            $this->saveFinancialData($consultationId, $consultation_charge);
        }
        
        // Complete transaction
        $this->db->trans_complete();
        
        if ($this->db->trans_status() === FALSE) {
            throw new Exception('Transaction failed');
        }
        
        // Clear buffer and send success response
        ob_clean();
        echo json_encode(array(
            'status' => 'success',
            'message' => 'Consultation saved successfully',
            'consultation_id' => $consultationId
        ));
        ob_end_flush();
        
    } catch (Exception $e) {
        // Rollback transaction
        $this->db->trans_rollback();
        
        log_message('error', 'saveWithFinancials error: ' . $e->getMessage());
        
        // Clear buffer and send error response
        ob_clean();
        echo json_encode(array(
            'status' => 'error',
            'message' => 'Error saving consultation: ' . $e->getMessage(),
            'debug' => array(
                'file' => $e->getFile(),
                'line' => $e->getLine()
            )
        ));
        ob_end_flush();
    }
}

/**
 * Save financial data separately
 */
private function saveFinancialData($consultationId, $consultationCharge) {
    $baseAmount = floatval($consultationCharge);
    $discountPercent = floatval($this->input->post('discount_percentage')) ?: 0;
    $taxPercent = floatval($this->input->post('tax_percentage')) ?: 0;
    
    // Calculate amounts
    $discountAmount = ($baseAmount * $discountPercent) / 100;
    $amountAfterDiscount = $baseAmount - $discountAmount;
    $taxAmount = ($amountAfterDiscount * $taxPercent) / 100;
    $totalAmount = $amountAfterDiscount + $taxAmount;
    
    $financialData = array(
        'consultation_id' => $consultationId,
        'base_amount' => $baseAmount,
        'discount_percentage' => $discountPercent,
        'discount_amount' => $discountAmount,
        'tax_percentage' => $taxPercent,
        'tax_amount' => $taxAmount,
        'total_amount' => $totalAmount,
        'payment_mode' => $this->input->post('payment_mode') ?: '',
        'payment_status' => $this->input->post('payment_status') ?: 'pending',
        'payment_notes' => $this->input->post('payment_notes') ?: '',
        'created_at' => date('Y-m-d H:i:s')
    );
    
    $this->db->insert('consultation_financials', $financialData);
    
    if ($this->db->error()['code'] != 0) {
        throw new Exception('Failed to insert financial data: ' . $this->db->error()['message']);
    }
}

public function saveWithLabTests() {
    // CRITICAL: Clean output buffer and set headers FIRST
    ob_clean();
    header('Content-Type: application/json');
    header('Cache-Control: no-cache, must-revalidate');
    error_reporting(0);
    ini_set('display_errors', 0);

    try {
        // Log input data for debugging
        log_message('debug', 'Input data: ' . print_r($_POST, true));

        // Start database transaction
        $this->db->trans_start();

        // Save consultation first
        $consultation_data = [
            'patient_id' => $this->input->post('patient_id'),
            'symptoms' => $this->input->post('symptoms'),
            'doctor' => $this->input->post('doctor'),
            'date' => $this->input->post('consultation_date') ?: date('Y-m-d'),
            'notes' => $this->input->post('notes') ?: '',
            'verification_status' => $this->input->post('verification_status') ?: 'unverified',
            'verification_score' => $this->input->post('verification_score') ?: 0,
            'verification_quality' => $this->input->post('verification_quality') ?: '',
            'created_by' => $this->getStaffId(),
            'created_at' => date('Y-m-d H:i:s')
        ];

        // Insert consultation
        $this->db->insert('consultations', $consultation_data);

        // Check for database errors
        if ($this->db->error()['code'] !== 0) {
            throw new Exception('Database error: ' . $this->db->error()['message']);
        }

        $consultation_id = $this->db->insert_id();

        if ($consultation_id === 0) {
            throw new Exception('Failed to insert consultation data.');
        }

        // Save selected lab tests
        $selected_tests = json_decode($this->input->post('selected_tests'), true);

        if (json_last_error() !== JSON_ERROR_NONE) {
            throw new Exception('Invalid JSON for selected tests: ' . json_last_error_msg());
        }

        $lab_test_ids = [];

        if (!empty($selected_tests)) {
            foreach ($selected_tests as $test) {
                $lab_data = [
                    'patient_id' => $consultation_data['patient_id'],
                    'test_id' => isset($test['id']) ? $test['id'] : null,
                    'test_name' => $test['test_name'],
                    'short_name' => isset($test['short_name']) ? $test['short_name'] : '',
                    'test_result' => '',
                    'notes' => $this->input->post('test_instructions') ?: '',
                    'test_date' => date('Y-m-d'),
                    'priority' => $this->input->post('test_priority') ?: 'normal',
                    'base_cost' => floatval($test['cost']),
                    'total_cost' => floatval($test['cost']),
                    'final_amount' => floatval($test['cost']),
                    'result_status' => 'pending',
                    'doctor_id' => $consultation_data['doctor'],
                    'verification_status' => 'unverified',
                    'tested_by' => $this->getStaffId(),
                    'created_at' => date('Y-m-d H:i:s')
                ];

                $this->db->insert('laboratory', $lab_data);

                if ($this->db->error()['code'] !== 0) {
                    throw new Exception('Database error: ' . $this->db->error()['message']);
                }

                $lab_test_ids[] = $this->db->insert_id();
            }
        }

        // Complete transaction
        $this->db->trans_complete();

        if ($this->db->trans_status() === FALSE) {
            throw new Exception('Transaction failed');
        }

        // Clean output buffer before sending response
        ob_clean();
        echo json_encode([
            'status' => 'success',
            'message' => 'Consultation and lab tests saved successfully',
            'consultation_id' => $consultation_id,
            'lab_test_ids' => $lab_test_ids,
            'total_tests_saved' => count($lab_test_ids)
        ]);
        exit;

    } catch (Exception $e) {
        // Rollback transaction on error
        $this->db->trans_rollback();
        
        log_message('error', 'Error saving consultation: ' . $e->getMessage());
        
        ob_clean();
        echo json_encode([
            'status' => 'error',
            'message' => 'Error saving consultation: ' . $e->getMessage()
        ]);
        exit;
    }
}

/**
 * ALTERNATIVE: Create a link between consultations and lab tests using a separate table
 * This method creates a consultation_lab_tests table to link consultations with lab tests
 */
public function saveWithLabTestsLinked() {
    header('Content-Type: application/json');

    try {
        // Start database transaction
        $this->db->trans_start();

        // Create consultation_lab_tests table if it doesn't exist
        $this->createConsultationLabTestsTable();

        // Save consultation first
        $consultation_data = [
            'patient_id' => $this->input->post('patient_id'),
            'symptoms' => $this->input->post('symptoms'),
            'doctor' => $this->input->post('doctor'),
            'date' => $this->input->post('consultation_date') ?: date('Y-m-d'),
            'notes' => $this->input->post('notes') ?: '',
            'verification_status' => $this->input->post('verification_status') ?: 'unverified',
            'created_by' => $this->getStaffId(),
            'created_at' => date('Y-m-d H:i:s')
        ];

        $this->db->insert('consultations', $consultation_data);
        $consultation_id = $this->db->insert_id();

        if (!$consultation_id) {
            throw new Exception('Failed to insert consultation data.');
        }

        // Save lab tests
        $selected_tests = json_decode($this->input->post('selected_tests'), true);
        $lab_test_ids = [];

        if (!empty($selected_tests)) {
            foreach ($selected_tests as $test) {
                // Insert into laboratory table
                $lab_data = [
                    'patient_id' => $consultation_data['patient_id'],
                    'test_id' => isset($test['id']) ? $test['id'] : null,
                    'test_name' => $test['test_name'],
                    'short_name' => isset($test['short_name']) ? $test['short_name'] : '',
                    'test_result' => '',
                    'notes' => $this->input->post('test_instructions') ?: '',
                    'test_date' => date('Y-m-d'),
                    'priority' => $this->input->post('test_priority') ?: 'normal',
                    'base_cost' => floatval($test['cost']),
                    'total_cost' => floatval($test['cost']),
                    'final_amount' => floatval($test['cost']),
                    'result_status' => 'pending',
                    'doctor_id' => $consultation_data['doctor'],
                    'verification_status' => 'unverified',
                    'tested_by' => $this->getStaffId(),
                    'created_at' => date('Y-m-d H:i:s')
                ];

                $this->db->insert('laboratory', $lab_data);
                $lab_id = $this->db->insert_id();
                $lab_test_ids[] = $lab_id;

                // Create link between consultation and lab test
                $link_data = [
                    'consultation_id' => $consultation_id,
                    'lab_test_id' => $lab_id,
                    'created_at' => date('Y-m-d H:i:s')
                ];

                $this->db->insert('consultation_lab_tests', $link_data);
            }
        }

        // Complete transaction
        $this->db->trans_complete();

        if ($this->db->trans_status() === FALSE) {
            throw new Exception('Transaction failed');
        }

        echo json_encode([
            'status' => 'success',
            'message' => 'Consultation and lab tests saved successfully with links',
            'consultation_id' => $consultation_id,
            'lab_test_ids' => $lab_test_ids,
            'total_tests_saved' => count($lab_test_ids)
        ]);

    } catch (Exception $e) {
        $this->db->trans_rollback();
        
        log_message('error', 'Error saving consultation with lab tests: ' . $e->getMessage());
        echo json_encode([
            'status' => 'error',
            'message' => 'Error saving consultation: ' . $e->getMessage()
        ]);
    }
}

/**
 * Create consultation_lab_tests linking table
 */
private function createConsultationLabTestsTable() {
    if (!$this->db->table_exists('consultation_lab_tests')) {
        $sql = "CREATE TABLE `consultation_lab_tests` (
            `id` int(11) NOT NULL AUTO_INCREMENT,
            `consultation_id` int(11) NOT NULL,
            `lab_test_id` int(11) NOT NULL,
            `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (`id`),
            KEY `consultation_id` (`consultation_id`),
            KEY `lab_test_id` (`lab_test_id`),
            CONSTRAINT `fk_consultation_lab_consultation` FOREIGN KEY (`consultation_id`) REFERENCES `consultations` (`id`) ON DELETE CASCADE,
            CONSTRAINT `fk_consultation_lab_laboratory` FOREIGN KEY (`lab_test_id`) REFERENCES `laboratory` (`id`) ON DELETE CASCADE
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8";
        
        $this->db->query($sql);
        
        if ($this->db->error()['code'] != 0) {
            log_message('error', 'Failed to create consultation_lab_tests table: ' . $this->db->error()['message']);
        } else {
            log_message('info', 'Created consultation_lab_tests table successfully');
        }
    }
}


/**
 * Save diagnosis relationship if consultation_diagnosis table exists
 */
private function saveDiagnosisRelation($consultationId, $diagnosisCodeId) {
    try {
        if ($this->db->table_exists('consultation_diagnosis')) {
            $diagnosisRelation = [
                'consultation_id' => $consultationId,
                'diagnosis_code_id' => (int)$diagnosisCodeId,
                'is_primary' => 1,
                'created_at' => date('Y-m-d H:i:s')
            ];
            
            $this->db->insert('consultation_diagnosis', $diagnosisRelation);
            log_message('debug', 'Diagnosis relationship saved for consultation: ' . $consultationId);
        }
    } catch (Exception $e) {
        // Don't fail the main transaction for diagnosis relationship issues
        log_message('warning', 'Could not save diagnosis relationship: ' . $e->getMessage());
    }
}

private function createFinancialTable() {
    if (!$this->db->table_exists('consultation_financials')) {
        $sql = "CREATE TABLE IF NOT EXISTS `consultation_financials` (
            `id` int(11) NOT NULL AUTO_INCREMENT,
            `consultation_id` int(11) NOT NULL,
            `base_amount` decimal(10,2) NOT NULL DEFAULT 0.00,
            `discount_percentage` decimal(5,2) NOT NULL DEFAULT 0.00,
            `discount_amount` decimal(10,2) NOT NULL DEFAULT 0.00,
            `tax_percentage` decimal(5,2) NOT NULL DEFAULT 0.00,
            `tax_amount` decimal(10,2) NOT NULL DEFAULT 0.00,
            `total_amount` decimal(10,2) NOT NULL DEFAULT 0.00,
            `payment_mode` varchar(50) DEFAULT NULL,
            `payment_status` enum('pending','paid','partial') DEFAULT 'pending',
            `payment_notes` text DEFAULT NULL,
            `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
            `updated_at` timestamp NULL DEFAULT NULL ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (`id`),
            KEY `consultation_id` (`consultation_id`),
            KEY `payment_status` (`payment_status`)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
        
        $this->db->query($sql);
        
        if ($this->db->error()['code'] != 0) {
            log_message('error', 'Failed to create consultation_financials table: ' . $this->db->error()['message']);
        } else {
            log_message('info', 'Created consultation_financials table successfully');
        }
    }
}

private function ensureDiagnosisCodesTables() {
    try {
        // Create diagnosis_codes table if it doesn't exist
        if (!$this->db->table_exists('diagnosis_codes')) {
            $sql = "CREATE TABLE IF NOT EXISTS `diagnosis_codes` (
                `id` int(11) NOT NULL AUTO_INCREMENT,
                `icd_version` enum('ICD10','ICD11') NOT NULL DEFAULT 'ICD10',
                `code` varchar(20) NOT NULL,
                `title` varchar(500) NOT NULL,
                `description` text,
                `category` varchar(100) DEFAULT NULL,
                `subcategory` varchar(100) DEFAULT NULL,
                `is_active` tinyint(1) NOT NULL DEFAULT '1',
                `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
                `updated_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                PRIMARY KEY (`id`),
                UNIQUE KEY `unique_code_version` (`code`, `icd_version`),
                KEY `idx_icd_version` (`icd_version`),
                KEY `idx_category` (`category`),
                KEY `idx_active` (`is_active`),
                KEY `idx_code` (`code`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
            
            $this->db->query($sql);
            
            if ($this->db->error()['code'] == 0) {
                log_message('info', 'Created diagnosis_codes table successfully');
                $this->insertSampleDiagnosisCodes();
            }
        }
        
        // Create consultation_diagnosis table if it doesn't exist
        if (!$this->db->table_exists('consultation_diagnosis')) {
            $sql = "CREATE TABLE IF NOT EXISTS `consultation_diagnosis` (
                `id` int(11) NOT NULL AUTO_INCREMENT,
                `consultation_id` int(11) NOT NULL,
                `diagnosis_code_id` int(11) NOT NULL,
                `is_primary` tinyint(1) NOT NULL DEFAULT '0',
                `notes` text,
                `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
                PRIMARY KEY (`id`),
                KEY `fk_consultation_diagnosis_consultation` (`consultation_id`),
                KEY `fk_consultation_diagnosis_code` (`diagnosis_code_id`),
                KEY `idx_consultation_primary` (`consultation_id`, `is_primary`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
            
            $this->db->query($sql);
            
            if ($this->db->error()['code'] == 0) {
                log_message('info', 'Created consultation_diagnosis table successfully');
            }
        }
        
    } catch (Exception $e) {
        log_message('warning', 'Could not create diagnosis tables: ' . $e->getMessage());
    }
}

/**
 * Search ICD-10 diagnosis codes
 */
public function searchIcdDiagnoses() {
    header('Content-Type: application/json');
    
    try {
        if (!isset($this->consultation_model)) {
            $this->load->model('consultation_model');
        }
        
        $searchTerm = $this->input->get('term') ?: $this->input->post('term');
        $category = $this->input->get('category');
        $icdVersion = $this->input->get('version') ?: 'ICD10';
        $limit = $this->input->get('limit') ?: 50;
        
        if (empty($searchTerm)) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Search term required',
                'diagnoses' => []
            ]);
            return;
        }
        
        $diagnoses = $this->consultation_model->searchIcdDiagnosisCodes($searchTerm, $icdVersion, $category, $limit);
        
        echo json_encode([
            'status' => 'success',
            'message' => 'Search completed',
            'diagnoses' => $diagnoses,
            'count' => count($diagnoses),
            'search_term' => $searchTerm
        ]);
        
    } catch (Exception $e) {
        log_message('error', 'Error searching ICD diagnoses: ' . $e->getMessage());
        echo json_encode([
            'status' => 'error',
            'message' => 'Search failed: ' . $e->getMessage(),
            'diagnoses' => []
        ]);
    }
}

    /**
     * FIXED: Get patient details with proper error handling
     */
public function getPatientDetails() {
    header('Content-Type: application/json');
    
    try {
        $patientId = $this->input->post('id');
        
        if (!$patientId) {
            echo json_encode([
                'error' => 'Patient ID required',
                'debug' => [
                    'post_data' => $_POST,
                    'get_data' => $_GET
                ]
            ]);
            return;
        }
        
        // Check if patients table exists
        if (!$this->db->table_exists('patients')) {
            echo json_encode([
                'error' => 'Patients table does not exist',
                'patient_id' => $patientId
            ]);
            return;
        }
        
        // Enhanced query with workflow information
        $this->db->select('p.*');
        
        // Try to join with patient_types table if it exists
        if ($this->db->table_exists('patient_types')) {
            $this->db->select('pt.type_name, pt.type_code, pt.requires_fingerprint, pt.requires_photo, pt.workflow_steps, pt.description');
            $this->db->from('patients p');
            $this->db->join('patient_types pt', 'p.patient_type_id = pt.id', 'left');
        } else {
            $this->db->from('patients p');
        }
        
        $this->db->where('p.id', $patientId);
        
        $query = $this->db->get();
        
        if (!$query) {
            echo json_encode([
                'error' => 'Query failed: ' . $this->db->error()['message'],
                'patient_id' => $patientId,
                'last_query' => $this->db->last_query()
            ]);
            return;
        }
        
        if ($query->num_rows() == 0) {
            echo json_encode([
                'error' => 'Patient not found with ID: ' . $patientId,
                'patient_id' => $patientId,
                'last_query' => $this->db->last_query()
            ]);
            return;
        }
        
        $patient = $query->row_array();
        
        // =============================================
        // WORKFLOW DETECTION AND ENFORCEMENT
        // =============================================
        
        // Determine if this is a GCC patient through multiple methods
        $isGCCPatient = $this->determineGCCWorkflow($patient);
        
        // Set workflow information
        $patient['is_gcc_patient'] = $isGCCPatient;
        $patient['workflow_type'] = $this->getPatientWorkflowType($patient);
        $patient['verification_required'] = $isGCCPatient;
        $patient['allows_admin_override'] = !$isGCCPatient;
        
        // Add workflow requirements
        $patient['workflow_requirements'] = [
            'requires_fingerprint' => $isGCCPatient || ($patient['requires_fingerprint'] ?? 0),
            'requires_photo' => $isGCCPatient || ($patient['requires_photo'] ?? 0),
            'allows_admin_override' => !$isGCCPatient,
            'mandatory_verification' => $isGCCPatient
        ];
        
        // =============================================
        // EXISTING PATIENT DATA PROCESSING
        // =============================================
        
        // Get blood group name if blood_group table exists
        if ($this->db->table_exists('blood_group') && !empty($patient['blood_group'])) {
            try {
                $this->db->select('name');
                $this->db->from('blood_group');
                $this->db->where('id', $patient['blood_group']);
                $bg_query = $this->db->get();
                
                if ($bg_query && $bg_query->num_rows() > 0) {
                    $patient['blood_group_name'] = $bg_query->row()->name;
                } else {
                    $patient['blood_group_name'] = $patient['blood_group'] ?? 'Unknown';
                }
            } catch (Exception $e) {
                $patient['blood_group_name'] = $patient['blood_group'] ?? 'Unknown';
            }
        } else {
            $patient['blood_group_name'] = $patient['blood_group'] ?? 'Unknown';
        }
        
        // Process patient data
        $patient['patient_age'] = !empty($patient['age']) ? $patient['age'] . ' years' : 'Unknown';
        $patient['image'] = !empty($patient['image']) ? $patient['image'] : 'uploads/patient_images/no_image.png';
        
        // Set defaults for missing fields
        $patient['patient_name'] = $patient['patient_name'] ?? 'Unknown';
        $patient['gender'] = $patient['gender'] ?? 'Unknown';
        $patient['mobileno'] = $patient['mobileno'] ?? '';
        $patient['known_allergies'] = $patient['known_allergies'] ?? 'None';
        
        // =============================================
        // ENHANCED FINGERPRINT STATUS CHECKING
        // =============================================
        
        $hasFingerprint = false;
        $fingerprintTemplateSize = 0;
        
        if ($this->db->field_exists('fingerprint_template', 'patients')) {
            $hasFingerprint = !empty($patient['fingerprint_template']);
            $fingerprintTemplateSize = strlen($patient['fingerprint_template'] ?? '');
        }
        
        $patient['has_fingerprint'] = $hasFingerprint;
        $patient['verification_capable'] = $hasFingerprint && $fingerprintTemplateSize > 100;
        
        // Enhanced fingerprint status for workflow enforcement
        $patient['fingerprint_status'] = [
            'enrolled' => $hasFingerprint,
            'template_size' => $fingerprintTemplateSize,
            'image_available' => !empty($patient['fingerprint_image']),
            'enrollment_date' => $patient['created_at'] ?? null,
            'verification_ready' => $hasFingerprint && $fingerprintTemplateSize > 100,
            'gcc_compliant' => $isGCCPatient ? ($hasFingerprint && $fingerprintTemplateSize > 100) : true
        ];
        
        // =============================================
        // WORKFLOW COMPLIANCE CHECKING
        // =============================================
        
        if ($isGCCPatient) {
            // For GCC patients, check compliance
            $patient['workflow_compliance'] = [
                'fingerprint_enrolled' => $hasFingerprint,
                'template_valid' => $fingerprintTemplateSize > 100,
                'photo_available' => !empty($patient['image']) && $patient['image'] !== 'uploads/patient_images/no_image.png',
                'ready_for_verification' => $hasFingerprint && $fingerprintTemplateSize > 100,
                'compliance_status' => ($hasFingerprint && $fingerprintTemplateSize > 100) ? 'compliant' : 'non_compliant'
            ];
        } else {
            // For general patients, all optional
            $patient['workflow_compliance'] = [
                'fingerprint_enrolled' => $hasFingerprint,
                'template_valid' => $fingerprintTemplateSize > 100,
                'photo_available' => !empty($patient['image']),
                'ready_for_verification' => true, // Always ready for general patients
                'compliance_status' => 'compliant' // General patients are always compliant
            ];
        }
        
        // Security: Remove sensitive data from response - template will be fetched separately when needed
        unset($patient['fingerprint_template']);
        if (isset($patient['fingerprint_image'])) {
            unset($patient['fingerprint_image']);
        }
        
        // Add debug information for workflow detection
        $patient['workflow_debug'] = [
            'detection_method' => $this->getWorkflowDetectionMethod($patient),
            'patient_type_field' => $patient['patient_type'] ?? 'not_set',
            'type_name' => $patient['type_name'] ?? 'not_set',
            'requires_fingerprint_flag' => $patient['requires_fingerprint'] ?? 'not_set',
            'note_contains_gcc' => !empty($patient['note']) && stripos($patient['note'], 'gcc') !== false,
            'workflow_conclusion' => $isGCCPatient ? 'GCC_PATIENT' : 'GENERAL_PATIENT'
        ];
        
        echo json_encode($patient);
        
    } catch (Exception $e) {
        echo json_encode([
            'error' => 'Database error: ' . $e->getMessage(),
            'patient_id' => $patientId ?? 'undefined',
            'line' => $e->getLine(),
            'file' => $e->getFile(),
            'trace' => $e->getTraceAsString()
        ]);
    }
}

/**
 * Determine if patient follows GCC workflow requiring fingerprint verification
 */
private function determineGCCWorkflow($patient) {
    // Method 1: Check patient_type field directly
    if (isset($patient['patient_type']) && 
        (strtoupper($patient['patient_type']) === 'GCC' || strtolower($patient['patient_type']) === 'gcc')) {
        return true;
    }
    
    // Method 2: Check type_name from patient_types join
    if (isset($patient['type_name']) && 
        (stripos($patient['type_name'], 'GCC') !== false || stripos($patient['type_name'], 'gulf') !== false)) {
        return true;
    }
    
    // Method 3: Check type_code field
    if (isset($patient['type_code']) && strtoupper($patient['type_code']) === 'GCC') {
        return true;
    }
    
    // Method 4: Check requires_fingerprint flag from patient_types
    if (isset($patient['requires_fingerprint']) && $patient['requires_fingerprint'] == 1) {
        return true;
    }
    
    // Method 5: Check patient_workflow column if it exists
    if (isset($patient['patient_workflow']) && strtolower($patient['patient_workflow']) === 'gcc') {
        return true;
    }
    
    // Method 6: Check note field for GCC workflow data
    if (!empty($patient['note'])) {
        $note_lower = strtolower($patient['note']);
        if (stripos($note_lower, 'gcc') !== false || 
            stripos($note_lower, 'gulf cooperation') !== false ||
            stripos($note_lower, 'workflow') !== false && stripos($note_lower, 'gcc') !== false) {
            return true;
        }
    }
    
    // Method 7: Check custom_fields if it exists and is valid JSON
    if (!empty($patient['custom_fields'])) {
        try {
            $custom_fields = json_decode($patient['custom_fields'], true);
            if (is_array($custom_fields)) {
                // Check nested patient workflow
                if (isset($custom_fields['patient']['gcc_workflow']) && 
                    $custom_fields['patient']['gcc_workflow'] === 'gcc') {
                    return true;
                }
                
                // Check direct gcc_workflow field
                if (isset($custom_fields['gcc_workflow']) && 
                    $custom_fields['gcc_workflow'] === 'gcc') {
                    return true;
                }
            }
        } catch (Exception $e) {
            // JSON parsing failed, continue with other methods
        }
    }
    
    return false; // Default to general workflow
}

/**
 * Get the patient's workflow type as a readable string
 */
private function getPatientWorkflowType($patient) {
    if ($this->determineGCCWorkflow($patient)) {
        return 'GCC Process';
    }
    
    // Check if type_name is available from patient_types join
    if (!empty($patient['type_name'])) {
        return $patient['type_name'];
    }
    
    // Check direct patient_type field
    if (!empty($patient['patient_workflow'])) {
        switch (strtoupper($patient['patient_workflow'])) {
            case 'SPECIALIZED':
                return 'Specialized Medical Process';
            case 'GENERAL':
                return 'General Process';
            default:
                return ucfirst($patient['patient_workflow']) . ' Process';
        }
    }
    
    return 'General Process'; // Default workflow type
}

/**
 * Get the method used to detect the workflow (for debugging)
 */
private function getWorkflowDetectionMethod($patient) {
    if (isset($patient['patient_workflow']) && strtoupper($patient['patient_workflow']) === 'gcc') {
        return 'patient_type_field';
    }
    
    if (isset($patient['type_name']) && stripos($patient['type_name'], 'GCC') !== false) {
        return 'type_name_join';
    }
    
    if (isset($patient['requires_fingerprint']) && $patient['requires_fingerprint'] == 1) {
        return 'requires_fingerprint_flag';
    }
    
    if (!empty($patient['note']) && stripos($patient['note'], 'gcc') !== false) {
        return 'note_field_content';
    }
    
    if (!empty($patient['custom_fields'])) {
        try {
            $custom_fields = json_decode($patient['custom_fields'], true);
            if (is_array($custom_fields) && 
                (isset($custom_fields['patient']['gcc_workflow']) || isset($custom_fields['gcc_workflow']))) {
                return 'custom_fields_json';
            }
        } catch (Exception $e) {
            // Continue
        }
    }
    
    return 'default_general_workflow';
}

/**
 * ENHANCED: Get stored fingerprint template with better error handling
 */
public function getStoredFingerprintTemplate() {
    header('Content-Type: application/json');
    
    try {
        $patientId = $this->input->post('patient_id');
        
        if (!$patientId) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Patient ID required'
            ]);
            return;
        }
        
        // Check if fingerprint columns exist
        if (!$this->db->field_exists('fingerprint_template', 'patients')) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Fingerprint functionality not available - fingerprint_template column missing',
                'debug' => [
                    'table_exists' => $this->db->table_exists('patients'),
                    'available_fields' => $this->db->list_fields('patients')
                ]
            ]);
            return;
        }
        
        // Get only fingerprint data with better query
        $this->db->select('id, patient_name, fingerprint_template, fingerprint_image');
        $this->db->from('patients');
        $this->db->where('id', $patientId);
        $this->db->where('is_active', 'yes');
        
        $query = $this->db->get();
        
        if ($query->num_rows() > 0) {
            $patient = $query->row_array();
            
            if (!empty($patient['fingerprint_template'])) {
                // Clean the template data
                $template = trim($patient['fingerprint_template']);
                
                echo json_encode([
                    'status' => 'success',
                    'patient_id' => $patientId,
                    'patient_name' => $patient['patient_name'],
                    'fingerprint_template' => $template,
                    'fingerprint_image' => $patient['fingerprint_image'] ?? '',
                    'template_size' => strlen($template),
                    'has_image' => !empty($patient['fingerprint_image']),
                    'template_type' => $this->detectTemplateType($template),
                    'debug' => [
                        'template_preview' => substr($template, 0, 50) . '...',
                        'encoding_detected' => $this->detectEncoding($template)
                    ]
                ]);
            } else {
                echo json_encode([
                    'status' => 'error',
                    'message' => 'No fingerprint template found for this patient',
                    'patient_id' => $patientId,
                    'patient_name' => $patient['patient_name'],
                    'debug' => [
                        'template_exists' => isset($patient['fingerprint_template']),
                        'template_empty' => empty($patient['fingerprint_template'])
                    ]
                ]);
            }
        } else {
            echo json_encode([
                'status' => 'error',
                'message' => 'Patient not found or inactive',
                'patient_id' => $patientId,
                'debug' => [
                    'last_query' => $this->db->last_query()
                ]
            ]);
        }
        
    } catch (Exception $e) {
        echo json_encode([
            'status' => 'error',
            'message' => 'Database error: ' . $e->getMessage(),
            'debug' => [
                'exception' => $e->getMessage(),
                'line' => $e->getLine(),
                'file' => $e->getFile()
            ]
        ]);
    }
}

/**
 * ENHANCED: Verify fingerprint with comprehensive logging
 */
public function verifyFingerprint() {
    header('Content-Type: application/json');
    
    try {
        $patientId = $this->input->post('patient_id');
        $matchScore = $this->input->post('match_score');
        $isMatch = $this->input->post('is_match');
        $threshold = $this->input->post('threshold') ?? 60;
        $serviceUrl = $this->input->post('service_url');
        
        // Enhanced validation
        if (!$patientId) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Patient ID required',
                'received_data' => $_POST
            ]);
            return;
        }
        
        if (!is_numeric($matchScore)) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Invalid match score: ' . $matchScore,
                'received_data' => $_POST
            ]);
            return;
        }
        
        // Validate patient exists and get details
        $this->db->select('id, patient_name, fingerprint_template');
        $this->db->from('patients');
        $this->db->where('id', $patientId);
        $this->db->where('is_active', 'yes');
        $patient_query = $this->db->get();
        
        if ($patient_query->num_rows() == 0) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Patient not found or inactive',
                'patient_id' => $patientId
            ]);
            return;
        }
        
        $patient = $patient_query->row_array();
        
        // Check if patient has fingerprint template
        $hasStoredTemplate = !empty($patient['fingerprint_template']);
        
        // Log verification attempt with detailed information
        $verificationLogId = $this->logFingerprintVerification($patientId, $matchScore, $isMatch, $threshold, $serviceUrl, $hasStoredTemplate);
        
        // Prepare detailed response
        $verificationResult = [
            'patient_id' => $patientId,
            'patient_name' => $patient['patient_name'],
            'verification_type' => '1:1',
            'timestamp' => date('Y-m-d H:i:s'),
            'staff_id' => $this->getStaffId(),
            'match_score' => floatval($matchScore),
            'threshold_used' => intval($threshold),
            'max_possible_score' => 199,
            'service_url' => $serviceUrl ?? 'unknown',
            'verification_log_id' => $verificationLogId,
            'has_stored_template' => $hasStoredTemplate,
            'template_size' => $hasStoredTemplate ? strlen($patient['fingerprint_template']) : 0
        ];
        
        // Determine verification result
        $matchPassed = ($isMatch == 'true' || $isMatch === true || $isMatch == 1);
        
        if ($matchPassed && $hasStoredTemplate) {
            $verificationResult['status'] = 'success';
            $verificationResult['message'] = 'Patient identity verified successfully';
            $verificationResult['verification_passed'] = true;
            $verificationResult['verification_method'] = 'fingerprint_match';
        } else if (!$hasStoredTemplate) {
            $verificationResult['status'] = 'warning';
            $verificationResult['message'] = 'No stored fingerprint available for verification';
            $verificationResult['verification_passed'] = false;
            $verificationResult['verification_method'] = 'no_template';
        } else {
            $verificationResult['status'] = 'failed';
            $verificationResult['message'] = 'Fingerprint verification failed - identity not confirmed';
            $verificationResult['verification_passed'] = false;
            $verificationResult['verification_method'] = 'fingerprint_mismatch';
        }
        
        // Add quality assessment
        $verificationResult['quality_assessment'] = $this->assessVerificationQuality($matchScore, $threshold, $hasStoredTemplate);
        
        echo json_encode($verificationResult);
        
    } catch (Exception $e) {
        echo json_encode([
            'status' => 'error',
            'message' => 'Verification error: ' . $e->getMessage(),
            'patient_id' => $patientId ?? 'unknown',
            'debug' => [
                'exception' => $e->getMessage(),
                'line' => $e->getLine(),
                'file' => $e->getFile()
            ]
        ]);
    }
}

/**
 * ENHANCED: Log fingerprint verification with detailed tracking
 */
private function logFingerprintVerification($patientId, $matchScore, $isMatch, $threshold = 60, $serviceUrl = null, $hasStoredTemplate = true) {
    try {
        // Create verification log table if it doesn't exist
        $this->createVerificationLogTable();
        
        $logData = [
            'patient_id' => $patientId,
            'verification_type' => '1:1',
            'match_score' => floatval($matchScore),
            'is_match' => ($isMatch == 'true' || $isMatch === true || $isMatch == 1) ? 1 : 0,
            'threshold_used' => intval($threshold),
            'service_url' => $serviceUrl,
            'verification_date' => date('Y-m-d H:i:s'),
            'staff_id' => $this->getStaffId(),
            'ip_address' => $this->input->ip_address(),
            'user_agent' => $this->input->user_agent(),
            'has_stored_template' => $hasStoredTemplate ? 1 : 0,
            'verification_status' => ($isMatch == 'true' || $isMatch === true || $isMatch == 1) ? 'success' : 'failed'
        ];
        
        $this->db->insert('fingerprint_verification_log', $logData);
        return $this->db->insert_id();
        
    } catch (Exception $e) {
        log_message('error', 'Failed to log verification: ' . $e->getMessage());
        return null;
    }
}

/**
 * Create enhanced verification log table
 */
private function createVerificationLogTable() {
    if (!$this->db->table_exists('fingerprint_verification_log')) {
        $sql = "CREATE TABLE `fingerprint_verification_log` (
            `id` int(11) NOT NULL AUTO_INCREMENT,
            `patient_id` varchar(50) NOT NULL,
            `verification_type` enum('1:1','1:N') DEFAULT '1:1',
            `match_score` decimal(5,2) DEFAULT NULL,
            `is_match` tinyint(1) DEFAULT 0,
            `threshold_used` int(3) DEFAULT 60,
            `service_url` varchar(255) DEFAULT NULL,
            `verification_date` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
            `staff_id` int(11) DEFAULT NULL,
            `ip_address` varchar(45) DEFAULT NULL,
            `user_agent` text DEFAULT NULL,
            `has_stored_template` tinyint(1) DEFAULT 1,
            `verification_status` enum('success','failed','error') DEFAULT 'failed',
            `notes` text DEFAULT NULL,
            PRIMARY KEY (`id`),
            KEY `patient_id` (`patient_id`),
            KEY `verification_date` (`verification_date`),
            KEY `staff_id` (`staff_id`),
            KEY `verification_status` (`verification_status`)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8";
        
        $this->db->query($sql);
    } else {
        // Add missing columns to existing table
        $this->addVerificationLogColumns();
    }
}

/**
 * Add missing columns to verification log table
 */
private function addVerificationLogColumns() {
    try {
        if (!$this->db->field_exists('has_stored_template', 'fingerprint_verification_log')) {
            $sql = "ALTER TABLE `fingerprint_verification_log` 
                    ADD COLUMN `has_stored_template` TINYINT(1) DEFAULT 1 AFTER `user_agent`";
            $this->db->query($sql);
        }
        
        if (!$this->db->field_exists('verification_status', 'fingerprint_verification_log')) {
            $sql = "ALTER TABLE `fingerprint_verification_log` 
                    ADD COLUMN `verification_status` ENUM('success','failed','error') DEFAULT 'failed' AFTER `has_stored_template`";
            $this->db->query($sql);
        }
    } catch (Exception $e) {
        log_message('error', 'Error adding verification log columns: ' . $e->getMessage());
    }
}

/**
 * Assess verification quality
 */
private function assessVerificationQuality($matchScore, $threshold, $hasStoredTemplate) {
    if (!$hasStoredTemplate) {
        return [
            'quality' => 'no_template',
            'description' => 'No stored fingerprint template available',
            'recommendation' => 'Enroll patient fingerprint for future verifications'
        ];
    }
    
    $score = floatval($matchScore);
    
    if ($score >= ($threshold + 30)) {
        return [
            'quality' => 'excellent',
            'description' => 'Very high confidence match',
            'recommendation' => 'Proceed with confidence'
        ];
    } else if ($score >= ($threshold + 10)) {
        return [
            'quality' => 'good',
            'description' => 'High confidence match',
            'recommendation' => 'Proceed normally'
        ];
    } else if ($score >= $threshold) {
        return [
            'quality' => 'acceptable',
            'description' => 'Acceptable match above threshold',
            'recommendation' => 'Proceed with normal caution'
        ];
    } else if ($score >= ($threshold - 10)) {
        return [
            'quality' => 'low',
            'description' => 'Below threshold but close',
            'recommendation' => 'Consider retry or manual verification'
        ];
    } else {
        return [
            'quality' => 'poor',
            'description' => 'Well below threshold',
            'recommendation' => 'Retry verification or use alternative method'
        ];
    }
}

/**
 * Detect template type (ISO, ANSI, etc.)
 */
private function detectTemplateType($template) {
    if (empty($template)) {
        return 'none';
    }
    
    // Basic detection based on common patterns
    if (strpos($template, 'FMR') === 0) {
        return 'ISO_19794-2';
    } else if (strpos($template, 'ANSI') === 0) {
        return 'ANSI_378';
    } else if (strlen($template) > 100 && ctype_alnum(str_replace(['+', '/', '='], '', $template))) {
        return 'base64_encoded';
    } else {
        return 'unknown';
    }
}

/**
 * Detect encoding type
 */
private function detectEncoding($data) {
    if (empty($data)) {
        return 'none';
    }
    
    if (base64_encode(base64_decode($data, true)) === $data) {
        return 'base64';
    } else if (ctype_xdigit($data)) {
        return 'hexadecimal';
    } else {
        return 'binary_or_text';
    }
}

/**
 * Get verification history for a patient
 */
public function getVerificationHistory() {
    header('Content-Type: application/json');
    
    try {
        $patientId = $this->input->post('patient_id');
        $limit = $this->input->post('limit') ?? 10;
        
        if (!$patientId) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Patient ID required'
            ]);
            return;
        }
        
        if (!$this->db->table_exists('fingerprint_verification_log')) {
            echo json_encode([
                'status' => 'success',
                'verifications' => [],
                'message' => 'No verification history available'
            ]);
            return;
        }
        
        $this->db->select('v.*, s.name as staff_name, s.surname as staff_surname');
        $this->db->from('fingerprint_verification_log v');
        $this->db->join('staff s', 's.id = v.staff_id', 'left');
        $this->db->where('v.patient_id', $patientId);
        $this->db->order_by('v.verification_date', 'DESC');
        $this->db->limit($limit);
        
        $query = $this->db->get();
        $verifications = $query->result_array();
        
        // Format the data
        foreach ($verifications as &$verification) {
            $verification['staff_name_full'] = trim($verification['staff_name'] . ' ' . $verification['staff_surname']);
            $verification['verification_date_formatted'] = date('M j, Y g:i A', strtotime($verification['verification_date']));
            $verification['match_percentage'] = round(($verification['match_score'] / 199) * 100, 1);
        }
        
        echo json_encode([
            'status' => 'success',
            'verifications' => $verifications,
            'total_found' => count($verifications)
        ]);
        
    } catch (Exception $e) {
        echo json_encode([
            'status' => 'error',
            'message' => 'Error retrieving verification history: ' . $e->getMessage()
        ]);
    }
}

/**
 * Test fingerprint system connectivity
 */
public function testFingerprintSystem() {
    header('Content-Type: application/json');
    
    try {
        $results = [
            'timestamp' => date('Y-m-d H:i:s'),
            'database_checks' => [],
            'patient_fingerprint_stats' => [],
            'system_info' => []
        ];
        
        // Check database structure
        $results['database_checks']['patients_table_exists'] = $this->db->table_exists('patients');
        $results['database_checks']['fingerprint_template_column'] = $this->db->field_exists('fingerprint_template', 'patients');
        $results['database_checks']['fingerprint_image_column'] = $this->db->field_exists('fingerprint_image', 'patients');
        $results['database_checks']['verification_log_table'] = $this->db->table_exists('fingerprint_verification_log');
        
        // Get fingerprint statistics
        if ($results['database_checks']['patients_table_exists']) {
            $this->db->from('patients');
            $this->db->where('is_active', 'yes');
            $total_patients = $this->db->count_all_results();
            
            if ($results['database_checks']['fingerprint_template_column']) {
                $this->db->from('patients');
                $this->db->where('is_active', 'yes');
                $this->db->where('fingerprint_template IS NOT NULL');
                $this->db->where('fingerprint_template !=', '');
                $patients_with_fingerprints = $this->db->count_all_results();
            } else {
                $patients_with_fingerprints = 0;
            }
            
            $results['patient_fingerprint_stats'] = [
                'total_active_patients' => $total_patients,
                'patients_with_fingerprints' => $patients_with_fingerprints,
                'fingerprint_enrollment_rate' => $total_patients > 0 ? round(($patients_with_fingerprints / $total_patients) * 100, 1) : 0
            ];
        }
        
        // System information
        $results['system_info'] = [
            'php_version' => PHP_VERSION,
            'codeigniter_version' => CI_VERSION ?? 'Unknown',
            'server_time' => date('Y-m-d H:i:s'),
            'user_agent' => $this->input->user_agent(),
            'ip_address' => $this->input->ip_address()
        ];
        
        echo json_encode([
            'status' => 'success',
            'message' => 'System test completed',
            'results' => $results
        ]);
        
    } catch (Exception $e) {
        echo json_encode([
            'status' => 'error',
            'message' => 'System test failed: ' . $e->getMessage(),
            'debug' => [
                'exception' => $e->getMessage(),
                'line' => $e->getLine(),
                'file' => $e->getFile()
            ]
        ]);
    }
}

    // =============================================
    // CONSULTATION MANAGEMENT METHODS  
    // =============================================

   /**
     * FIXED: Save consultation with proper field mapping
     */
public function save() {
    header('Content-Type: application/json');
    
    try {
        // Create consultations table if it doesn't exist
        $this->createConsultationTable();
        
        // Enhanced validation
        $patient_id = $this->input->post('patient_id');
        $symptoms = $this->input->post('symptoms');
        $doctor_id = $this->input->post('doctor'); // This maps to doctor field in database
        $consultation_date = $this->input->post('consultation_date');
        $consultation_id = $this->input->post('id');
        $verification_status = $this->input->post('verification_status');
        
        if (!$patient_id || !$symptoms || !$doctor_id) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Please fill all required fields',
                'required_fields' => [
                    'patient_id' => $patient_id ? 'present' : 'missing',
                    'symptoms' => $symptoms ? 'present' : 'missing',
                    'doctor' => $doctor_id ? 'present' : 'missing'
                ]
            ]);
            return;
        }
        
        // Handle diagnosis - ICD-10 or custom
        $diagnosisCodeId = $this->input->post('diagnosis_code_id');
        $customDiagnosis = $this->input->post('diagnosis_custom');
        $diagnosis = '';
        
        if ($diagnosisCodeId) {
            // ICD-10 diagnosis selected
            $diagnosisDetails = $this->getDiagnosisById($diagnosisCodeId);
            if ($diagnosisDetails) {
                $diagnosis = $diagnosisDetails['code'] . ' - ' . $diagnosisDetails['title'];
            }
        } elseif ($customDiagnosis) {
            // Custom diagnosis entered
            $diagnosis = $customDiagnosis;
        }
        
        // Prepare consultation data with CORRECT field names for your table structure
        $data = [
            'patient_id' => $patient_id,
            'doctor' => (int)$doctor_id,  // Using 'doctor' field name as per your table
            'date' => $consultation_date ?: date('Y-m-d'),  // Using 'date' field name as per your table
            'symptoms' => $symptoms,
            'diagnosis' => $diagnosis,
            'notes' => $this->input->post('notes') ?: '',
            'verification_status' => $verification_status == '1' ? 'verified' : 'unverified',
            'created_by' => $this->getStaffId()
        ];
        
        if ($consultation_id) {
            // Complete consultation queue
            if ($queue_id) {
                $result = $this->Queue_model->completeService($queue_id, $next_department, $consultation_notes);
                
                if ($next_department) {
                    // Send notification to next department
                    $this->send_queue_notification($next_department, 'new_patient_from_consultation', [
                        'patient_id' => $patient_id,
                        'consultation_id' => $consultation_id,
                        'doctor_name' => $this->get_doctor_name($doctor_id),
                        'diagnosis' => $diagnosis,
                        'priority' => $this->determine_next_priority($diagnosis, $next_department)
                    ]);
                }
            }
            
            // Update existing consultation
            $data['updated_at'] = date('Y-m-d H:i:s');
            
            $this->db->where('id', $consultation_id);
            $result = $this->db->update('consultations', $data);
            
            if ($result) {
                echo json_encode([
                    'status' => 'success',
                    'message' => 'Consultation updated successfully',
                    'id' => $consultation_id
                ]);
            } else {
                echo json_encode([
                    'status' => 'error',
                    'message' => 'Failed to update consultation',
                    'db_error' => $this->db->error()
                ]);
            }
        } else {
            // Insert new consultation
            $data['created_at'] = date('Y-m-d H:i:s');
            
            $insertResult = $this->db->insert('consultations', $data);
            $consultationId = $this->db->insert_id();
            
            if ($insertResult && $consultationId) {
                echo json_encode([
                    'status' => 'success',
                    'message' => 'Consultation saved successfully',
                    'id' => $consultationId
                ]);
            } else {
                echo json_encode([
                    'status' => 'error',
                    'message' => 'Failed to save consultation',
                    'debug_info' => [
                        'insert_result' => $insertResult,
                        'consultation_id' => $consultationId,
                        'last_query' => $this->db->last_query(),
                        'db_error' => $this->db->error()
                    ]
                ]);
            }
        }
        
    } catch (Exception $e) {
        log_message('error', 'Consultation save error: ' . $e->getMessage());
        echo json_encode([
            'status' => 'error',
            'message' => 'Error: ' . $e->getMessage(),
            'debug_info' => [
                'exception' => $e->getMessage(),
                'line' => $e->getLine(),
                'file' => $e->getFile()
            ]
        ]);
    }
}

/**
 * Get doctor name for notifications
 */
private function get_doctor_name($doctor_id) {
    $this->db->select('name, surname');
    $this->db->where('id', $doctor_id);
    $doctor = $this->db->get('staff')->row();
    
    return $doctor ? $doctor->name . ' ' . $doctor->surname : 'Doctor';
}

/**
 * AJAX: Get consultation queue status
 */
public function get_queue_status() {
    header('Content-Type: application/json');
    
    try {
        $result = $this->Queue_model->getDepartmentQueue('consultation');
        
        // Add doctor assignment info
        if ($result['status'] === 'success' && isset($result['queue_list'])) {
            foreach ($result['queue_list'] as &$patient) {
                $patient['assigned_doctor'] = $this->get_assigned_doctor($patient['id']);
            }
        }
        
        echo json_encode($result);
    } catch (Exception $e) {
        echo json_encode(['status' => 'error', 'message' => $e->getMessage()]);
    }
}

/**
 * Get assigned doctor for queue item
 */
private function get_assigned_doctor($queue_id) {
    // Check if there's a doctor assignment in queue
    $this->db->select('served_by');
    $this->db->where('id', $queue_id);
    $queue = $this->db->get('patient_queue')->row();
    
    if ($queue && $queue->served_by) {
        return $this->get_doctor_name($queue->served_by);
    }
    
    return null;
}

/**
 * AJAX: Call next patient for consultation
 */
public function call_next_patient() {
    header('Content-Type: application/json');
    
    try {
        $doctor_id = $this->input->post('doctor_id') ?: $this->getStaffId();
        $result = $this->Queue_model->callNextPatient('consultation', $doctor_id);
        
        if ($result['status'] === 'success') {
            // Play notification sound
            $this->send_queue_notification('consultation', 'patient_called_for_consultation', array_merge($result['patient'], [
                'doctor_name' => $this->get_doctor_name($doctor_id)
            ]));
        }
        
        echo json_encode($result);
    } catch (Exception $e) {
        echo json_encode(['status' => 'error', 'message' => $e->getMessage()]);
    }
}

/**
 * AJAX: Assign doctor to patient in queue
 */
public function assign_doctor() {
    header('Content-Type: application/json');
    
    try {
        $queue_id = $this->input->post('queue_id');
        $doctor_id = $this->input->post('doctor_id');
        
        $this->db->where('id', $queue_id);
        $this->db->update('patient_queue', [
            'served_by' => $doctor_id,
            'status' => 'in_progress',
            'called_at' => date('Y-m-d H:i:s'),
            'updated_at' => date('Y-m-d H:i:s')
        ]);
        
        if ($this->db->affected_rows() > 0) {
            echo json_encode([
                'status' => 'success',
                'message' => 'Doctor assigned successfully',
                'doctor_name' => $this->get_doctor_name($doctor_id)
            ]);
        } else {
            echo json_encode(['status' => 'error', 'message' => 'Failed to assign doctor']);
        }
        
    } catch (Exception $e) {
        echo json_encode(['status' => 'error', 'message' => $e->getMessage()]);
    }
}

/**
 * AJAX: Remove patient from consultation queue
 */
public function remove_from_queue() {
    header('Content-Type: application/json');
    
    try {
        $queue_id = $this->input->post('queue_id');
        $reason = $this->input->post('reason', TRUE) ?: 'Consultation completed';
        
        $result = $this->Queue_model->skipPatient($queue_id, $reason);
        echo json_encode($result);
    } catch (Exception $e) {
        echo json_encode(['status' => 'error', 'message' => $e->getMessage()]);
    }
}

/**
 * AJAX: Quick consultation completion
 */
public function quick_complete() {
    header('Content-Type: application/json');
    
    try {
        $queue_id = $this->input->post('queue_id');
        $next_department = $this->input->post('next_department', TRUE);
        $notes = $this->input->post('notes', TRUE);
        
        // Get queue details
        $this->db->where('id', $queue_id);
        $queue = $this->db->get('patient_queue')->row_array();
        
        if (!$queue) {
            echo json_encode(['status' => 'error', 'message' => 'Queue record not found']);
            return;
        }
        
        // Create basic consultation record
        $consultation_data = [
            'patient_id' => $queue['patient_id'],
            'doctor_id' => $queue['served_by'] ?: $this->getStaffId(),
            'consultation_notes' => $notes ?: 'Quick consultation completed',
            'status' => 'completed',
            'consultation_date' => date('Y-m-d'),
            'consultation_time' => date('H:i:s'),
            'created_at' => date('Y-m-d H:i:s')
        ];
        
        $this->db->insert('consultations', $consultation_data);
        
        // Complete queue
        $result = $this->Queue_model->completeService($queue_id, $next_department, $notes);
        
        if ($result['status'] === 'success' && $next_department) {
            $this->send_queue_notification($next_department, 'quick_consultation_patient_incoming', [
                'patient_id' => $queue['patient_id'],
                'from_quick_consultation' => true
            ]);
        }
        
        echo json_encode($result);
    } catch (Exception $e) {
        echo json_encode(['status' => 'error', 'message' => $e->getMessage()]);
    }
}

/**
 * Check for incoming patients from triage/other departments
 */
public function check_incoming_notifications() {
    header('Content-Type: application/json');
    
    try {
        $notification_file = APPPATH . 'cache/queue_notifications_consultation.json';
        
        if (file_exists($notification_file)) {
            $notification = json_decode(file_get_contents($notification_file), true);
            unlink($notification_file);
            
            echo json_encode([
                'status' => 'success',
                'notification' => $notification,
                'has_notification' => true,
                'sound' => true
            ]);
        } else {
            echo json_encode([
                'status' => 'success',
                'has_notification' => false
            ]);
        }
    } catch (Exception $e) {
        echo json_encode(['status' => 'error', 'message' => $e->getMessage()]);
    }
}

/**
 * Get doctor availability and workload
 */
public function get_doctor_workload() {
    header('Content-Type: application/json');
    
    try {
        $today = date('Y-m-d');
        $doctors = $this->staff_model->getStaffbyrole(3); // Get all doctors
        
        $doctor_stats = [];
        
        foreach ($doctors as $doctor) {
            // Get current patients assigned to doctor
            $this->db->where('served_by', $doctor['id']);
            $this->db->where('department', 'consultation');
            $this->db->where_in('status', ['waiting', 'in_progress']);
            $current_load = $this->db->count_all_results('patient_queue');
            
            // Get consultations completed today
            $this->db->where('doctor_id', $doctor['id']);
            $this->db->where('DATE(consultation_date)', $today);
            $completed_today = $this->db->count_all_results('consultations');
            
            $doctor_stats[] = [
                'doctor_id' => $doctor['id'],
                'doctor_name' => $doctor['name'] . ' ' . $doctor['surname'],
                'current_load' => $current_load,
                'completed_today' => $completed_today,
                'availability' => $current_load < 5 ? 'available' : 'busy'
            ];
        }
        
        echo json_encode(['status' => 'success', 'doctors' => $doctor_stats]);
    } catch (Exception $e) {
        echo json_encode(['status' => 'error', 'message' => $e->getMessage()]);
    }
}

/**
 * Send notification to departments
 */
private function send_queue_notification($department, $action, $data = []) {
    try {
        $notification = [
            'department' => $department,
            'action' => $action,
            'data' => $data,
            'timestamp' => time(),
            'from_department' => 'consultation',
            'sound' => true
        ];
        
        $notification_file = APPPATH . 'cache/queue_notifications_' . $department . '.json';
        file_put_contents($notification_file, json_encode($notification));
        
        return true;
    } catch (Exception $e) {
        log_message('error', 'Consultation notification failed: ' . $e->getMessage());
        return false;
    }
}

/**
 * Get consultation statistics
 */
public function get_consultation_stats() {
    header('Content-Type: application/json');
    
    try {
        $stats = $this->Queue_model->getQueueStatistics('consultation');
        
        // Add consultation-specific stats
        $today = date('Y-m-d');
        
        // Consultations by doctor today
        $this->db->select('s.name, s.surname, COUNT(*) as consultation_count');
        $this->db->from('consultations c');
        $this->db->join('staff s', 's.id = c.doctor_id', 'left');
        $this->db->where('DATE(c.consultation_date)', $today);
        $this->db->group_by('c.doctor_id');
        $this->db->order_by('consultation_count', 'DESC');
        $doctor_stats = $this->db->get()->result_array();
        
        $stats['doctor_consultations'] = $doctor_stats;
        
        echo json_encode(['status' => 'success', 'stats' => $stats]);
    } catch (Exception $e) {
        echo json_encode(['status' => 'error', 'message' => $e->getMessage()]);
    }
}

 // =============================================
    // AJAX: Get queue statistics
    // =============================================

    /**
     * AJAX: Get consultation queue statistics
     */
    public function get_queue_stats() {
        header('Content-Type: application/json');
        
        try {
            $stats = $this->Queue_model->getQueueStatistics('consultation');
            echo json_encode(['status' => 'success', 'stats' => $stats]);
        } catch (Exception $e) {
            echo json_encode(['status' => 'error', 'message' => $e->getMessage()]);
        }
    }
    
    /**
     * AJAX: Get current consultation queue
     */
    public function get_current_queue() {
        header('Content-Type: application/json');
        
        try {
            $queue_data = $this->Queue_model->getDepartmentQueue('consultation');
            echo json_encode(['status' => 'success', 'queue' => $queue_data]);
        } catch (Exception $e) {
            echo json_encode(['status' => 'error', 'message' => $e->getMessage()]);
        }
    }

    /**
     * FIXED: Get patient consultations
     */
public function getPatientConsultations() {
    ob_clean();
    header('Content-Type: application/json');
    header('Cache-Control: no-cache, must-revalidate');
    error_reporting(0);
    ini_set('display_errors', 0);
    
    try {
        $patient_id = $this->input->post('patient_id');
        
        if (!$patient_id) {
            ob_clean();
            echo json_encode([
                'status' => 'error',
                'message' => 'Patient ID required'
            ]);
            exit;
        }
        
        // Check if consultations table exists
        if (!$this->db->table_exists('consultations')) {
            ob_clean();
            echo json_encode([
                'status' => 'success',
                'consultations' => [],
                'message' => 'No consultations table found'
            ]);
            exit;
        }
        
        // Query consultations
        $this->db->select('c.*, c.date as consultation_date');
        $this->db->from('consultations c');
        $this->db->where('c.patient_id', $patient_id);
        $this->db->order_by('c.date', 'DESC');
        $this->db->limit(5);
        
        $query = $this->db->get();
        
        // Debug: Log the generated SQL query
        error_log('SQL Query: ' . $this->db->last_query());
        
        if (!$query) {
            ob_clean();
            echo json_encode([
                'status' => 'error',
                'message' => 'Database query failed'
            ]);
            exit;
        }
        
        $consultations = $query->result_array();
        
        // Debug: Log the number of consultations found
        error_log('Consultations found: ' . count($consultations));
        
        // Get doctor names separately to avoid join issues
        foreach ($consultations as &$consultation) {
            if (!empty($consultation['doctor'])) {
                $this->db->select('name, surname');
                $this->db->from('staff');
                $this->db->where('id', $consultation['doctor']);
                $doctor_query = $this->db->get();
                
                if ($doctor_query && $doctor_query->num_rows() > 0) {
                    $doctor = $doctor_query->row();
                    $consultation['doctor_name'] = $doctor->name;
                    if (!empty($doctor->surname)) {
                        $consultation['doctor_name'] .= ' ' . $doctor->surname;
                    }
                } else {
                    $consultation['doctor_name'] = 'Unknown Doctor';
                }
            } else {
                $consultation['doctor_name'] = 'Unknown Doctor';
            }
        }
        
        ob_clean();
        echo json_encode([
            'status' => 'success',
            'consultations' => $consultations,
            'debug' => [
                'patient_id' => $patient_id,
                'count' => count($consultations)
            ]
        ]);
        exit;
        
    } catch (Exception $e) {
        ob_clean();
        echo json_encode([
            'status' => 'error',
            'message' => $e->getMessage()
        ]);
        exit;
    }
}

    /**
     * FIXED: Get consultation list for DataTable
     */
    public function getConsultationList() {
        header('Content-Type: application/json');
        
        try {
            // Check if consultations table exists
            if (!$this->db->table_exists('consultations')) {
                echo json_encode([
                    'draw' => 1,
                    'recordsTotal' => 0,
                    'recordsFiltered' => 0,
                    'data' => [],
                    'message' => 'Consultations table does not exist'
                ]);
                return;
            }
            
            // Query consultations with verification status if available
            if ($this->db->field_exists('verification_status', 'consultations')) {
                $this->db->select('c.id, c.patient_id, c.doctor, c.date, c.symptoms, c.diagnosis, c.notes, c.verification_status, c.created_at');
            } else {
                $this->db->select('c.id, c.patient_id, c.doctor, c.date, c.symptoms, c.diagnosis, c.notes, c.created_at');
            }
            
            $this->db->from('consultations c');
            $this->db->order_by('c.created_at', 'DESC');
            $this->db->limit(50);
            
            $query = $this->db->get();
            
            if (!$query) {
                echo json_encode([
                    'draw' => 1,
                    'recordsTotal' => 0,
                   'recordsFiltered' => 0,
                   'data' => [],
                   'error' => 'Query failed: ' . $this->db->error()['message']
               ]);
               return;
           }
           
           $consultations = $query->result_array();
           
           // Build data array for DataTable
           $data = [];
           foreach ($consultations as $consultation) {
               // Get patient name
               $patientName = 'Unknown Patient';
               if ($consultation['patient_id']) {
                   $this->db->select('patient_name');
                   $this->db->from('patients');
                   $this->db->where('id', $consultation['patient_id']);
                   $patientQuery = $this->db->get();
                   if ($patientQuery && $patientQuery->num_rows() > 0) {
                       $patient = $patientQuery->row();
                       $patientName = $patient->patient_name;
                   }
               }
               
               // Get doctor name
               $doctorName = 'Unknown Doctor';
               if ($consultation['doctor']) {
                   $this->db->select('name, surname');
                   $this->db->from('staff');
                   $this->db->where('id', $consultation['doctor']);
                   $doctorQuery = $this->db->get();
                   if ($doctorQuery && $doctorQuery->num_rows() > 0) {
                       $doctor = $doctorQuery->row();
                       $doctorName = 'Dr. ' . $doctor->name;
                       if (!empty($doctor->surname)) {
                           $doctorName .= ' ' . $doctor->surname;
                       }
                   }
               }
               
               // Format verification badge
               $verificationBadge = '';
               if (isset($consultation['verification_status'])) {
                   switch ($consultation['verification_status']) {
                       case 'verified':
                           $verificationBadge = '<br><span class="label label-success">Verified</span>';
                           break;
                       case 'skipped':
                           $verificationBadge = '<br><span class="label label-warning">Skipped</span>';
                           break;
                       default:
                           $verificationBadge = '<br><span class="label label-danger">Unverified</span>';
                   }
               }
               
               $data[] = [
                   $consultation['id'],
                   date('d M Y', strtotime($consultation['date'])),
                   $patientName . ' (' . $consultation['patient_id'] . ')' . $verificationBadge,
                   $doctorName,
                   substr($consultation['symptoms'] ?? 'No symptoms', 0, 50) . 
                   (strlen($consultation['symptoms'] ?? '') > 50 ? '...' : ''),
                   substr($consultation['diagnosis'] ?? 'No diagnosis', 0, 50),
                   '<button class="btn btn-xs btn-info" onclick="viewConsultation(' . $consultation['id'] . ')">
                       <i class="fa fa-eye"></i> View
                   </button>
                   <button class="btn btn-xs btn-primary" onclick="editConsultation(' . $consultation['id'] . ')">
                       <i class="fa fa-edit"></i> Edit
                   </button>'
               ];
           }
           
           echo json_encode([
               'draw' => 1,
               'recordsTotal' => count($data),
               'recordsFiltered' => count($data),
               'data' => $data,
               'success' => true
           ]);
           
       } catch (Exception $e) {
           echo json_encode([
               'draw' => 1,
               'recordsTotal' => 0,
               'recordsFiltered' => 0,
               'data' => [],
               'error' => 'Exception: ' . $e->getMessage() . ' on line ' . $e->getLine()
           ]);
       }
   }

   /**
    * FIXED: Get consultation details for viewing/editing
    */
public function getConsultationDetails() {
    header('Content-Type: application/json');
    
    try {
        $consultationId = $this->input->post('consultation_id');
        
        if (!$consultationId) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Consultation ID required'
            ]);
            return;
        }
        
        // Get consultation details - FIXED: Use correct column name 'date'
        if ($this->db->field_exists('verification_status', 'consultations')) {
            $this->db->select('c.id, c.patient_id, c.doctor, c.date, c.symptoms, c.diagnosis, c.notes, c.verification_status, c.created_at, c.updated_at');
        } else {
            $this->db->select('c.id, c.patient_id, c.doctor, c.date, c.symptoms, c.diagnosis, c.notes, c.created_at');
        }
        
        $this->db->from('consultations c');
        $this->db->where('c.id', $consultationId);
        
        $query = $this->db->get();
        
        if ($query->num_rows() > 0) {
            $consultation = $query->row_array();
            
            // Get patient name separately
            $this->db->select('patient_name');
            $this->db->from('patients');
            $this->db->where('id', $consultation['patient_id']);
            $patientQuery = $this->db->get();
            
            if ($patientQuery && $patientQuery->num_rows() > 0) {
                $patient = $patientQuery->row();
                $consultation['patient_name'] = $patient->patient_name;
            } else {
                $consultation['patient_name'] = 'Unknown Patient';
            }
            
            // Get doctor name separately
            $this->db->select('name, surname');
            $this->db->from('staff');
            $this->db->where('id', $consultation['doctor']);
            $doctorQuery = $this->db->get();
            
            if ($doctorQuery && $doctorQuery->num_rows() > 0) {
                $doctor = $doctorQuery->row();
                $consultation['doctor_name'] = $doctor->name;
                if (!empty($doctor->surname)) {
                    $consultation['doctor_name'] .= ' ' . $doctor->surname;
                }
            } else {
                $consultation['doctor_name'] = 'Unknown Doctor';
            }
            
            // Format date for display - FIXED: Use correct column name 'date'
            $consultation['consultation_date'] = $consultation['date'];
            
            // Set default values for optional fields
            if (!isset($consultation['verification_status'])) {
                $consultation['verification_status'] = 'unverified';
            }
            
            if (!isset($consultation['updated_at'])) {
                $consultation['updated_at'] = null;
            }
            
            echo json_encode([
                'status' => 'success',
                'consultation' => $consultation
            ]);
            
        } else {
            echo json_encode([
                'status' => 'error',
                'message' => 'Consultation not found with ID: ' . $consultationId
            ]);
        }
        
    } catch (Exception $e) {
        echo json_encode([
            'status' => 'error',
            'message' => 'Database error: ' . $e->getMessage(),
            'line' => $e->getLine(),
            'consultation_id' => $consultationId ?? 'undefined'
        ]);
    }
}

   /**
    * FIXED: Add new patient
    */
   public function addPatient() {
       header('Content-Type: application/json');
       
       try {
           $patientName = $this->input->post('patientName');
           $patientAge = $this->input->post('patientAge');
           $patientGender = $this->input->post('patientGender');
           $patientPhone = $this->input->post('patientPhone');
           
           if (!$patientName || !$patientAge || !$patientGender || !$patientPhone) {
               echo json_encode([
                   'status' => 'error',
                   'message' => 'Please fill all required fields'
               ]);
               return;
           }
           
           // Generate patient ID
           $patientId = $this->generatePatientId();
           
           $patientData = [
               'id' => $patientId,
               'patient_name' => $patientName,
               'age' => $patientAge,
               'gender' => strtolower($patientGender),
               'mobileno' => $patientPhone,
               'is_active' => 'yes',
               'created_at' => date('Y-m-d H:i:s')
           ];
           
           $this->db->insert('patients', $patientData);
           
           if ($this->db->affected_rows() > 0) {
               echo json_encode([
                   'status' => 'success',
                   'message' => 'Patient added successfully',
                   'patient_id' => $patientId
               ]);
           } else {
               echo json_encode([
                   'status' => 'error',
                   'message' => 'Failed to add patient'
               ]);
           }
           
       } catch (Exception $e) {
           echo json_encode([
               'status' => 'error',
               'message' => 'Error: ' . $e->getMessage()
           ]);
       }
   }

   /**
    * Generate unique patient ID
    */
   private function generatePatientId() {
       $currentYear = date('Y');
       
       // Get the last patient ID for current year
       $this->db->select('id');
       $this->db->from('patients');
       $this->db->like('id', $currentYear, 'after');
       $this->db->order_by('id', 'DESC');
       $this->db->limit(1);
       
       $query = $this->db->get();
       
       if ($query->num_rows() > 0) {
           $lastId = $query->row()->id;
           $lastNumber = (int)substr($lastId, 4);
           $newNumber = $lastNumber + 1;
       } else {
           $newNumber = 1;
       }
       
       return $currentYear . str_pad($newNumber, 4, '0', STR_PAD_LEFT);
   }

   /**
    * Get staff ID safely
    */
   private function getStaffId() {
       try {
           if (method_exists($this->customlib, 'getStaffID')) {
               return $this->customlib->getStaffID();
           } else {
               // Fallback - try to get from session
               $staffId = $this->session->userdata('admin_id');
               return $staffId ?: 1; // Default to 1 if no staff ID found
           }
       } catch (Exception $e) {
           log_message('error', 'Error getting staff ID: ' . $e->getMessage());
           return 1; // Default staff ID
       }
   }

   // =============================================
   // DEBUGGING AND TESTING METHODS
   // =============================================

   /**
    * Test controller functionality
    */
public function test() {
        echo "<h1>Consultation Controller Test</h1>";
        echo "<p>Controller is working properly!</p>";
        echo "<p>Queue integration: " . (class_exists('Queue_model') ? 'ACTIVE' : 'MISSING') . "</p>";
        echo "<p>Notification system: " . (is_writable(APPPATH . 'cache/') ? 'READY' : 'NOT WRITABLE') . "</p>";
        
        try {
            $stats = $this->Queue_model->getQueueStatistics('consultation');
            echo "<p>Queue statistics: Working</p>";
        } catch (Exception $e) {
            echo "<p>Queue statistics: Error - " . $e->getMessage() . "</p>";
        }
    }
    
    /**
 * AJAX: Get queue notifications for department
 * This method checks for actual waiting patients and returns them as notifications
 */
public function get_queue_notifications() {
    header('Content-Type: application/json');
    
    try {
        $department = $this->input->get('department', TRUE);
        $last_check = $this->input->get('last_check', TRUE);
        
        // Default to checking all departments if none specified
        if (empty($department) || $department === 'all') {
            $department = null;
        }
        
        // If no last_check provided, use current time minus 30 seconds
        if (empty($last_check)) {
            $last_check = time() - 30;
        }
        
        // Convert timestamp to datetime for database query
        $last_check_datetime = date('Y-m-d H:i:s', $last_check);
        
        // Query for new patients added to queue since last check
        $this->db->select('pq.*, p.patient_name, p.mobileno, p.age, p.gender, p.patient_unique_id');
        $this->db->from('patient_queue pq');
        $this->db->join('patients p', 'p.id = pq.patient_id', 'left');
        $this->db->where('pq.status', 'waiting');
        $this->db->where('pq.created_at >', $last_check_datetime);
        
        // Filter by department if specified
        if ($department) {
            $this->db->where('pq.department', $department);
        }
        
        $this->db->order_by('pq.created_at', 'DESC');
        $this->db->limit(10);
        
        $query = $this->db->get();
        
        if (!$query) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Database query failed',
                'notifications' => [],
                'timestamp' => time()
            ]);
            return;
        }
        
        $new_patients = $query->result_array();
        $notifications = [];
        
        // Convert database results to notification format
        foreach ($new_patients as $patient) {
            $notifications[] = [
                'patient_id' => $patient['patient_id'],
                'patient_name' => $patient['patient_name'] ?: 'Unknown Patient',
                'queue_number' => $patient['queue_number'],
                'department' => $patient['department'],
                'priority' => $patient['priority'],
                'workflow_type' => $patient['queue_type'] ?: 'general',
                'time' => date('H:i:s', strtotime($patient['created_at'])),
                'created_at' => $patient['created_at'],
                'phone' => $patient['mobileno'] ?: '',
                'age' => $patient['age'] ?: '',
                'gender' => $patient['gender'] ?: '',
                'patient_unique_id' => $patient['patient_unique_id'] ?: ''
            ];
        }
        
        // Also check for priority patients (urgent/high priority)
        if (empty($new_patients)) {
            // Check for any urgent patients that might have been missed
            $this->db->select('pq.*, p.patient_name, p.mobileno, p.age, p.gender, p.patient_unique_id');
            $this->db->from('patient_queue pq');
            $this->db->join('patients p', 'p.id = pq.patient_id', 'left');
            $this->db->where('pq.status', 'waiting');
            $this->db->where_in('pq.priority', ['urgent', 'high']);
            
            if ($department) {
                $this->db->where('pq.department', $department);
            }
            
            $this->db->order_by('pq.created_at', 'DESC');
            $this->db->limit(5);
            
            $urgent_query = $this->db->get();
            $urgent_patients = $urgent_query->result_array();
            
            foreach ($urgent_patients as $patient) {
                $notifications[] = [
                    'patient_id' => $patient['patient_id'],
                    'patient_name' => $patient['patient_name'] ?: 'Unknown Patient',
                    'queue_number' => $patient['queue_number'],
                    'department' => $patient['department'],
                    'priority' => $patient['priority'],
                    'workflow_type' => $patient['queue_type'] ?: 'urgent',
                    'time' => date('H:i:s', strtotime($patient['created_at'])),
                    'created_at' => $patient['created_at'],
                    'phone' => $patient['mobileno'] ?: '',
                    'age' => $patient['age'] ?: '',
                    'gender' => $patient['gender'] ?: '',
                    'patient_unique_id' => $patient['patient_unique_id'] ?: '',
                    'is_urgent' => true
                ];
            }
        }
        
        echo json_encode([
            'status' => 'success',
            'notifications' => $notifications,
            'count' => count($notifications),
            'timestamp' => time(),
            'last_check' => $last_check,
            'department' => $department ?: 'all',
            'debug' => [
                'query_time' => $last_check_datetime,
                'current_time' => date('Y-m-d H:i:s'),
                'department_filter' => $department
            ]
        ]);
        
    } catch (Exception $e) {
        echo json_encode([
            'status' => 'error',
            'message' => 'Exception: ' . $e->getMessage(),
            'notifications' => [],
            'timestamp' => time()
        ]);
    }
}

public function debug_queue_status() {
    header('Content-Type: application/json');
    
    try {
        // Get all waiting patients
        $this->db->select('pq.*, p.patient_name, p.mobileno');
        $this->db->from('patient_queue pq');
        $this->db->join('patients p', 'p.id = pq.patient_id', 'left');
        $this->db->where('pq.status', 'waiting');
        $this->db->order_by('pq.created_at', 'DESC');
        $this->db->limit(20);
        
        $query = $this->db->get();
        $waiting_patients = $query->result_array();
        
        // Get queue counts by department
        $this->db->select('department, COUNT(*) as count');
        $this->db->from('patient_queue');
        $this->db->where('status', 'waiting');
        $this->db->group_by('department');
        $dept_counts = $this->db->get()->result_array();
        
        echo json_encode([
            'status' => 'success',
            'waiting_patients' => $waiting_patients,
            'department_counts' => $dept_counts,
            'total_waiting' => count($waiting_patients),
            'timestamp' => time(),
            'current_time' => date('Y-m-d H:i:s')
        ]);
        
    } catch (Exception $e) {
        echo json_encode([
            'status' => 'error',
            'message' => $e->getMessage()
        ]);
    }
}


   /**
    * Debug patient details
    */
   public function debug_patient($patientId = null) {
       if (!$patientId) {
           $patientId = $this->input->get('id') ?: $this->input->post('id');
       }
       
       echo "<h2>Patient Details Debug</h2>";
       
       if (!$patientId) {
           echo "<p>No patient ID provided. Add ?id=PATIENT_ID to the URL</p>";
           return;
       }
       
       echo "<p><strong>Debugging Patient ID:</strong> " . $patientId . "</p>";
       
       try {
           // Check if patient exists
           $this->db->select('*');
           $this->db->from('patients');
           $this->db->where('id', $patientId);
           $query = $this->db->get();
           
           echo "<p><strong>SQL Query:</strong> " . $this->db->last_query() . "</p>";
           echo "<p><strong>Rows found:</strong> " . $query->num_rows() . "</p>";
           
           if ($query->num_rows() > 0) {
               $patient = $query->row_array();
               
               echo "<h3>Patient Data:</h3>";
               echo "<table border='1' style='border-collapse: collapse; width: 100%;'>";
               foreach ($patient as $key => $value) {
                   echo "<tr>";
                   echo "<td><strong>" . htmlspecialchars($key) . "</strong></td>";
                   echo "<td>" . htmlspecialchars($value ?? 'NULL') . "</td>";
                   echo "</tr>";
               }
               echo "</table>";
               
               // Test the actual API call
               echo "<h3>API Response Test:</h3>";
               echo "<p><strong>Testing getPatientDetails API...</strong></p>";
               
               // Simulate the API call
               $_POST['id'] = $patientId;
               ob_start();
               $this->getPatientDetails();
               $apiResponse = ob_get_clean();
               
               echo "<p><strong>API Response:</strong></p>";
               echo "<pre>" . htmlspecialchars($apiResponse) . "</pre>";
               
           } else {
               echo "<p style='color: red;'>Patient not found!</p>";
               
               // Show some sample patients
               echo "<h3>Sample Patients:</h3>";
               $this->db->select('id, patient_name, mobileno');
               $this->db->from('patients');
               $this->db->limit(10);
               $query = $this->db->get();
               
               if ($query->num_rows() > 0) {
                  echo "<table border='1' style='border-collapse: collapse;'>";
                  echo "<tr><th>ID</th><th>Name</th><th>Phone</th></tr>";
                  foreach ($query->result_array() as $sample) {
                      echo "<tr>";
                      echo "<td><a href='?id=" . $sample['id'] . "'>" . $sample['id'] . "</a></td>";
                      echo "<td>" . $sample['patient_name'] . "</td>";
                      echo "<td>" . $sample['mobileno'] . "</td>";
                      echo "</tr>";
                  }
                  echo "</table>";
              } else {
                  echo "<p>No patients found in database!</p>";
              }
          }
          
      } catch (Exception $e) {
          echo "<p style='color: red;'>Error: " . $e->getMessage() . "</p>";
          echo "<p><strong>Last Query:</strong> " . $this->db->last_query() . "</p>";
      }
   }

   /**
    * Check current table structure
    */
   public function check_table_structure() {
       try {
           echo "<h2>Current Table Structure</h2>";
           
           // Check patients table
           if ($this->db->table_exists('patients')) {
               echo "<h3>Patients Table:</h3>";
               $fields = $this->db->field_data('patients');
               
               echo "<table border='1' style='border-collapse: collapse; width: 100%;'>";
               echo "<tr><th>Field</th><th>Type</th><th>Null</th><th>Key</th><th>Default</th></tr>";
               
               foreach ($fields as $field) {
                   echo "<tr>";
                   echo "<td>" . $field->name . "</td>";
                   echo "<td>" . $field->type . "</td>";
                   echo "<td>" . ($field->null ? 'YES' : 'NO') . "</td>";
                   echo "<td>" . ($field->primary_key ? 'PRI' : '') . "</td>";
                   echo "<td>" . ($field->default ?? 'NULL') . "</td>";
                   echo "</tr>";
               }
               echo "</table>";
           } else {
               echo "<p style='color: red;'>Patients table does not exist!</p>";
           }
           
           // Check consultations table
           if ($this->db->table_exists('consultations')) {
               echo "<h3>Consultations Table:</h3>";
               $fields = $this->db->field_data('consultations');
               
               echo "<table border='1' style='border-collapse: collapse; width: 100%;'>";
               echo "<tr><th>Field</th><th>Type</th><th>Null</th><th>Key</th><th>Default</th></tr>";
               
               foreach ($fields as $field) {
                   echo "<tr>";
                   echo "<td>" . $field->name . "</td>";
                   echo "<td>" . $field->type . "</td>";
                   echo "<td>" . ($field->null ? 'YES' : 'NO') . "</td>";
                   echo "<td>" . ($field->primary_key ? 'PRI' : '') . "</td>";
                   echo "<td>" . ($field->default ?? 'NULL') . "</td>";
                   echo "</tr>";
               }
               echo "</table>";
           } else {
               echo "<p style='color: red;'>Consultations table does not exist!</p>";
           }
           
       } catch (Exception $e) {
           echo "<p style='color: red;'>Error: " . $e->getMessage() . "</p>";
       }
   }

   /**
    * Create test consultation for debugging
    */
   public function create_test_consultation() {
       header('Content-Type: application/json');
       
       try {
           // Create table first
           $this->createConsultationTable();
           
           // Get a sample patient
           $this->db->select('id');
           $this->db->from('patients');
           $this->db->limit(1);
           $patient = $this->db->get()->row();
           
           if (!$patient) {
               echo json_encode(['error' => 'No patients found in database']);
               return;
           }
           
           // Get a sample doctor
           $this->db->select('id');
           $this->db->from('staff');
           $this->db->limit(1);
           $doctor = $this->db->get()->row();
           
           if (!$doctor) {
               echo json_encode(['error' => 'No staff found in database']);
               return;
           }
           
           // Create test consultation
           $testData = [
               'patient_id' => $patient->id,
               'doctor' => $doctor->id,
               'date' => date('Y-m-d'),
               'symptoms' => 'Test symptoms for debugging',
               'diagnosis' => 'Test diagnosis',
               'notes' => 'Test consultation created for debugging',
               'verification_status' => 'unverified',
               'created_by' => 1,
               'created_at' => date('Y-m-d H:i:s')
           ];
           
           $result = $this->db->insert('consultations', $testData);
           $insertId = $this->db->insert_id();
           
           echo json_encode([
               'sample_patient_id' => $patient->id,
               'sample_doctor_id' => $doctor->id,
               'insert_result' => $result,
               'insert_id' => $insertId,
               'affected_rows' => $this->db->affected_rows(),
               'last_query' => $this->db->last_query(),
               'test_data' => $testData,
               'table_fields' => $this->db->list_fields('consultations')
           ], JSON_PRETTY_PRINT);
           
       } catch (Exception $e) {
           echo json_encode([
               'error' => $e->getMessage(),
               'line' => $e->getLine(),
               'file' => $e->getFile(),
               'last_query' => $this->db->last_query()
           ]);
       }
   }
   
       /**
     * Create consultations table if it doesn't exist
     */
    private function createConsultationTable() {
        if (!$this->db->table_exists('consultations')) {
            $sql = "CREATE TABLE IF NOT EXISTS `consultations` (
                `id` int(11) NOT NULL AUTO_INCREMENT,
                `patient_id` varchar(50) NOT NULL,
                `doctor` int(11) NOT NULL,
                `date` date NOT NULL,
                `symptoms` text NOT NULL,
                `diagnosis` text DEFAULT NULL,
                `notes` text DEFAULT NULL,
                `verification_status` enum('verified','unverified','skipped') DEFAULT 'unverified',
                `created_by` int(11) DEFAULT 1,
                `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
                `updated_at` timestamp NULL DEFAULT NULL ON UPDATE CURRENT_TIMESTAMP,
                PRIMARY KEY (`id`),
                KEY `patient_id` (`patient_id`),
                KEY `doctor` (`doctor`),
                KEY `date` (`date`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8";
            
            $this->db->query($sql);
        } else {
            // Add missing columns to existing table
            $this->addConsultationColumns();
        }
    }
   
   private function addConsultationColumns() {
        try {
            if (!$this->db->field_exists('verification_status', 'consultations')) {
                $sql = "ALTER TABLE `consultations` 
                        ADD COLUMN `verification_status` ENUM('verified','unverified','skipped') DEFAULT 'unverified' AFTER `notes`";
                $this->db->query($sql);
            }
            
            if (!$this->db->field_exists('created_by', 'consultations')) {
                $sql = "ALTER TABLE `consultations` 
                        ADD COLUMN `created_by` INT(11) DEFAULT 1 AFTER `verification_status`";
                $this->db->query($sql);
            }
            
            if (!$this->db->field_exists('created_at', 'consultations')) {
                $sql = "ALTER TABLE `consultations` 
                        ADD COLUMN `created_at` TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP AFTER `created_by`";
                $this->db->query($sql);
            }
            
            if (!$this->db->field_exists('updated_at', 'consultations')) {
                $sql = "ALTER TABLE `consultations` 
                        ADD COLUMN `updated_at` TIMESTAMP NULL DEFAULT NULL ON UPDATE CURRENT_TIMESTAMP AFTER `created_at`";
                $this->db->query($sql);
            }
            
        } catch (Exception $e) {
            log_message('error', 'Error adding consultation columns: ' . $e->getMessage());
        }
    }
    
    /**
 * Debug method to check what's wrong
 */
public function debugSystem() {
    // Force JSON output and error reporting
    header('Content-Type: application/json');
    error_reporting(E_ALL);
    ini_set('display_errors', 0); // Don't display errors in HTML
    
    $debug_info = array();
    
    try {
        // Check if tables exist
        $debug_info['tables'] = array(
            'consultations' => $this->db->table_exists('consultations'),
            'consultation_financials' => $this->db->table_exists('consultation_financials'),
            'patients' => $this->db->table_exists('patients'),
            'staff' => $this->db->table_exists('staff'),
            'diagnosis_codes' => $this->db->table_exists('diagnosis_codes')
        );
        
        // Check database connection
        $debug_info['database'] = array(
            'connected' => $this->db->initialize(),
            'database' => $this->db->database,
            'hostname' => $this->db->hostname
        );
        
        // Check if methods exist
        $debug_info['methods'] = array(
            'saveWithFinancials' => method_exists($this, 'saveWithFinancials'),
            'getFinancialSummary' => method_exists($this, 'getFinancialSummary'),
            'generateFinancialReport' => method_exists($this, 'generateFinancialReport')
        );
        
        // Test simple query
        $debug_info['simple_query'] = $this->db->query("SELECT 1 as test")->row();
        
        echo json_encode(array(
            'status' => 'success',
            'message' => 'Debug completed successfully',
            'debug_info' => $debug_info,
            'timestamp' => date('Y-m-d H:i:s')
        ));
        
    } catch (Exception $e) {
        echo json_encode(array(
            'status' => 'error',
            'message' => 'Debug failed: ' . $e->getMessage(),
            'debug_info' => $debug_info,
            'error_details' => array(
                'file' => $e->getFile(),
                'line' => $e->getLine(),
                'trace' => $e->getTraceAsString()
            )
        ));
    }
}

/**
 * Simple financial summary
 */
public function getFinancialSummary() {
    header('Content-Type: application/json');
    ob_start();
    
    try {
        ob_clean();
        
        // Check if financial table exists
        if (!$this->db->table_exists('consultation_financials')) {
            echo json_encode(array(
                'status' => 'success',
                'summary' => array(
                    'total_revenue' => '0.00',
                    'paid_amount' => '0.00',
                    'pending_amount' => '0.00',
                    'outstanding_amount' => '0.00'
                ),
                'charts' => array(
                    'revenue_labels' => array(),
                    'revenue_data' => array(),
                    'payment_status_data' => array(0, 0, 0)
                ),
                'message' => 'Financial table not found'
            ));
            ob_end_flush();
            return;
        }
        
        // Get basic financial summary
        $this->db->select('
            COALESCE(SUM(total_amount), 0) as total_revenue,
            COALESCE(SUM(CASE WHEN payment_status = "paid" THEN total_amount ELSE 0 END), 0) as paid_amount,
            COALESCE(SUM(CASE WHEN payment_status = "pending" THEN total_amount ELSE 0 END), 0) as pending_amount,
            COALESCE(SUM(CASE WHEN payment_status = "partial" THEN total_amount ELSE 0 END), 0) as partial_amount
        ');
        $this->db->from('consultation_financials');
        $summary = $this->db->get()->row_array();
        
        $outstanding = floatval($summary['pending_amount']) + floatval($summary['partial_amount']);
        
        ob_clean();
        echo json_encode(array(
            'status' => 'success',
            'summary' => array(
                'total_revenue' => number_format($summary['total_revenue'], 2),
                'paid_amount' => number_format($summary['paid_amount'], 2),
                'pending_amount' => number_format($summary['pending_amount'], 2),
                'outstanding_amount' => number_format($outstanding, 2)
            ),
            'charts' => array(
                'revenue_labels' => array(),
                'revenue_data' => array(),
                'payment_status_data' => array(
                    floatval($summary['paid_amount']),
                    floatval($summary['pending_amount']),
                    floatval($summary['partial_amount'])
                )
            )
        ));
        ob_end_flush();
        
    } catch (Exception $e) {
        ob_clean();
        echo json_encode(array(
            'status' => 'error',
            'message' => 'Error loading financial summary: ' . $e->getMessage()
        ));
        ob_end_flush();
    }
}

/**
 * Ensure all required tables exist - ADD THIS METHOD TO YOUR CONTROLLER
 */
private function ensureTablesExist() {
    try {
        // Create consultation table
        $this->createConsultationTable();
        
        // Create financial table
        $this->createFinancialTable();
        
        // Create diagnosis tables
        $this->ensureDiagnosisCodesTables();
        
        log_message('debug', 'All required tables ensured');
        
    } catch (Exception $e) {
        log_message('error', 'Error ensuring tables exist: ' . $e->getMessage());
        throw new Exception('Failed to ensure required tables exist: ' . $e->getMessage());
    }
}
/**
 * Simple financial report
 */
public function generateFinancialReport() {
    header('Content-Type: application/json');
    ob_start();
    
    try {
        ob_clean();
        
        $dateFrom = $this->input->post('date_from');
        $dateTo = $this->input->post('date_to');
        
        if (!$dateFrom || !$dateTo) {
            echo json_encode(array(
                'status' => 'error',
                'message' => 'Date range is required'
            ));
            ob_end_flush();
            return;
        }
        
        // Simple report with basic data
        $report = array(
            'total_consultations' => 0,
            'total_revenue' => '0.00',
            'paid_amount' => '0.00',
            'outstanding_amount' => '0.00'
        );
        
        // Get consultations count
        $this->db->where('date >=', $dateFrom);
        $this->db->where('date <=', $dateTo);
        $this->db->from('consultations');
        $report['total_consultations'] = $this->db->count_all_results();
        
        // Get financial data if table exists
        if ($this->db->table_exists('consultation_financials')) {
            $this->db->select('
                COALESCE(SUM(total_amount), 0) as total_revenue,
                COALESCE(SUM(CASE WHEN payment_status = "paid" THEN total_amount ELSE 0 END), 0) as paid_amount
            ');
            $this->db->from('consultation_financials cf');
            $this->db->join('consultations c', 'c.id = cf.consultation_id');
            $this->db->where('c.date >=', $dateFrom);
            $this->db->where('c.date <=', $dateTo);
            $financial = $this->db->get()->row_array();
            
            if ($financial) {
                $report['total_revenue'] = number_format($financial['total_revenue'], 2);
                $report['paid_amount'] = number_format($financial['paid_amount'], 2);
                $outstanding = floatval($financial['total_revenue']) - floatval($financial['paid_amount']);
                $report['outstanding_amount'] = number_format($outstanding, 2);
            }
        }
        
        ob_clean();
        echo json_encode(array(
            'status' => 'success',
            'message' => 'Report generated successfully',
            'report' => $report
        ));
        ob_end_flush();
        
    } catch (Exception $e) {
        ob_clean();
        echo json_encode(array(
            'status' => 'error',
            'message' => 'Error generating report: ' . $e->getMessage()
        ));
        ob_end_flush();
    }
}

/*public function getLabTestsForConsultation() {
    header('Content-Type: application/json');
    
    try {
        // Check if lab_tests table exists, fallback to pathology
        if ($this->db->table_exists('lab_tests')) {
            $this->loadLabTestsFromLabTestsTable();
        } else {
            $this->loadLabTestsFromPathologyTable();
        }
    } catch (Exception $e) {
        echo json_encode([
            'status' => 'error',
            'message' => 'Error loading lab tests: ' . $e->getMessage()
        ]);
    }
}*/

public function getLabList() {
    ob_clean();
    header('Content-Type: application/json');
    header('Cache-Control: no-cache, must-revalidate');
    error_reporting(0);
    ini_set('display_errors', 0);
    
    try {
        // Check if laboratory table exists
        if (!$this->db->table_exists('laboratory')) {
            ob_clean();
            echo json_encode([
                'draw' => 1,
                'recordsTotal' => 0,
                'recordsFiltered' => 0,
                'data' => [],
                'message' => 'Laboratory table does not exist'
            ]);
            exit;
        }
        
        // Query laboratory records
        $this->db->select('l.*, p.patient_name');
        $this->db->from('laboratory l');
        $this->db->join('patients p', 'p.id = l.patient_id', 'left');
        $this->db->order_by('l.created_at', 'DESC');
        $this->db->limit(50);
        
        $query = $this->db->get();
        
        if (!$query) {
            ob_clean();
            echo json_encode([
                'draw' => 1,
                'recordsTotal' => 0,
                'recordsFiltered' => 0,
                'data' => [],
                'error' => 'Query failed: ' . $this->db->error()['message']
            ]);
            exit;
        }
        
        $labs = $query->result_array();
        
        // Build data array for DataTable
        $data = [];
        foreach ($labs as $lab) {
            // Format verification badge
            $verificationBadge = $this->getVerificationBadge($lab['verification_status'] ?? 'unverified');
            
            // Get patient name
            $patientName = $lab['patient_name'] ?: 'Unknown Patient';
            
            // Get staff name if tested_by exists
            $staffName = 'Unknown Staff';
            if (!empty($lab['tested_by'])) {
                $this->db->select('name, surname');
                $this->db->from('staff');
                $this->db->where('id', $lab['tested_by']);
                $staffQuery = $this->db->get();
                if ($staffQuery && $staffQuery->num_rows() > 0) {
                    $staff = $staffQuery->row();
                    $staffName = $staff->name;
                    if (!empty($staff->surname)) {
                        $staffName .= ' ' . $staff->surname;
                    }
                }
            }
            
            $testDate = !empty($lab['test_date']) ? 
                       date('d M Y', strtotime($lab['test_date'])) : 
                       date('d M Y H:i', strtotime($lab['created_at']));
            
            $data[] = [
                $lab['id'],
                $testDate,
                $patientName . ' (' . $lab['patient_id'] . ')' . $verificationBadge,
                $lab['test_name'],
                substr($lab['test_result'] ?? 'Pending', 0, 50) . 
                (strlen($lab['test_result'] ?? '') > 50 ? '...' : ''),
                $staffName,
                '<button class="btn btn-xs btn-info" onclick="viewLab(' . $lab['id'] . ')">
                    <i class="fa fa-eye"></i> View
                </button>
                <button class="btn btn-xs btn-primary" onclick="editLab(' . $lab['id'] . ')">
                    <i class="fa fa-edit"></i> Edit
                </button>'
            ];
        }
        
        ob_clean();
        echo json_encode([
            'draw' => 1,
            'recordsTotal' => count($data),
            'recordsFiltered' => count($data),
            'data' => $data,
            'success' => true
        ]);
        exit;
        
    } catch (Exception $e) {
        ob_clean();
        echo json_encode([
            'draw' => 1,
            'recordsTotal' => 0,
            'recordsFiltered' => 0,
            'data' => [],
            'error' => 'Exception: ' . $e->getMessage() . ' on line ' . $e->getLine()
        ]);
        exit;
    }
}

public function getPatientLabTests() {
    ob_clean();
    header('Content-Type: application/json');
    header('Cache-Control: no-cache, must-revalidate');
    error_reporting(0);
    ini_set('display_errors', 0);
    
    try {
        $patient_id = $this->input->post('patient_id') ?: $this->input->get('patient_id');
        
        if (!$patient_id) {
            ob_clean();
            echo json_encode([
                'status' => 'error',
                'message' => 'Patient ID required',
                'lab_tests' => []
            ]);
            exit;
        }
        
        // Check if laboratory table exists
        if (!$this->db->table_exists('laboratory')) {
            ob_clean();
            echo json_encode([
                'status' => 'success',
                'lab_tests' => [],
                'message' => 'Laboratory table not found'
            ]);
            exit;
        }
        
        // Query lab tests for this patient
        $this->db->select('*');
        $this->db->from('laboratory');
        $this->db->where('patient_id', $patient_id);
        $this->db->order_by('created_at', 'DESC');
        $this->db->limit(10);
        
        $query = $this->db->get();
        
        if (!$query) {
            ob_clean();
            echo json_encode([
                'status' => 'error',
                'message' => 'Database query failed',
                'lab_tests' => []
            ]);
            exit;
        }
        
        $lab_tests = $query->result_array();
        
        // Format the lab tests data
        foreach ($lab_tests as &$test) {
            $test['test_date_formatted'] = !empty($test['test_date']) ? 
                date('M j, Y', strtotime($test['test_date'])) : 
                date('M j, Y', strtotime($test['created_at']));
                
            $test['status_label'] = ucfirst($test['result_status'] ?? 'pending');
            $test['verification_badge'] = $this->getVerificationBadge($test['verification_status'] ?? 'unverified');
        }
        
        ob_clean();
        echo json_encode([
            'status' => 'success',
            'lab_tests' => $lab_tests,
            'total_found' => count($lab_tests)
        ]);
        exit;
        
    } catch (Exception $e) {
        ob_clean();
        echo json_encode([
            'status' => 'error',
            'message' => 'Error loading lab tests: ' . $e->getMessage(),
            'lab_tests' => []
        ]);
        exit;
    }
}

private function loadLabTestsFromLabTestsTable() {
    // Get tests grouped by category
    $this->db->select('lt.*, ltc.category_name');
    $this->db->from('lab_tests lt');
    $this->db->join('lab_test_categories ltc', 'ltc.id = lt.category_id', 'left');
    $this->db->where('lt.is_active', 1);
    $this->db->order_by('ltc.category_name, lt.test_name');
    
    $query = $this->db->get();
    $all_tests = $query->result_array();
    
    // Group by category
    $tests_by_category = [];
    $categories = [];
    
    foreach ($all_tests as $test) {
        $category_id = $test['category_id'] ?: 'other';
        $category_name = $test['category_name'] ?: 'Other Tests';
        
        if (!isset($tests_by_category[$category_id])) {
            $tests_by_category[$category_id] = [];
            $categories[$category_id] = $category_name;
        }
        
        $tests_by_category[$category_id][] = $test;
    }
    
    echo json_encode([
        'status' => 'success',
        'tests_by_category' => $tests_by_category,
        'categories' => $categories
    ]);
}

private function loadLabTestsFromPathologyTable() {
    // Fallback to pathology table
    $this->db->select('p.*, pc.category_name');
    $this->db->from('pathology p');
    $this->db->join('pathology_category pc', 'pc.id = p.pathology_category_id', 'left');
    $this->db->where('p.is_active', 1);
    $this->db->order_by('pc.category_name, p.test_name');
    
    $query = $this->db->get();
    $all_tests = $query->result_array();
    
    // Group by category and map to lab_tests format
    $tests_by_category = [];
    $categories = [];
    
    foreach ($all_tests as $test) {
        $category_id = $test['pathology_category_id'] ?: 'other';
        $category_name = $test['category_name'] ?: 'Other Tests';
        
        if (!isset($tests_by_category[$category_id])) {
            $tests_by_category[$category_id] = [];
            $categories[$category_id] = $category_name;
        }
        
        // Map pathology fields to lab_tests format
        $mapped_test = [
            'id' => $test['id'],
            'test_name' => $test['test_name'],
            'short_name' => $test['short_name'],
            'cost' => $test['cost'],
            'sample_type' => $test['sample_type'],
            'method' => $test['method'],
            'category_id' => $category_id
        ];
        
        $tests_by_category[$category_id][] = $mapped_test;
    }
    
    echo json_encode([
        'status' => 'success',
        'tests_by_category' => $tests_by_category,
        'categories' => $categories
    ]);
}

public function testJsonResponse() {
    ob_clean();
    header('Content-Type: application/json');
    header('Cache-Control: no-cache, must-revalidate');
    error_reporting(0);
    ini_set('display_errors', 0);
    
    try {
        ob_clean();
        echo json_encode([
            'status' => 'success',
            'message' => 'JSON response test successful',
            'timestamp' => date('Y-m-d H:i:s'),
            'method' => $_SERVER['REQUEST_METHOD'],
            'data' => [
                'test' => true,
                'number' => 123,
                'array' => [1, 2, 3]
            ]
        ]);
        exit;
        
    } catch (Exception $e) {
        ob_clean();
        echo json_encode([
            'status' => 'error',
            'message' => $e->getMessage()
        ]);
        exit;
    }
}

public function search() {
    try {
        $search = $this->input->get('search');
        $limit = $this->input->get('limit') ?: 20;
        
        if (strlen($search) < 2) {
            echo json_encode([]);
            return;
        }
        
        $patients = $this->db->select('id, patient_name as text, patient_name as name, mobileno as phone, age, gender')
                            ->like('patient_name', $search)
                            ->or_like('mobileno', $search)
                            ->or_like('id', $search)
                            ->order_by('patient_name')
                            ->limit($limit)
                            ->get('patients')
                            ->result_array();
        
        // Format for select2
        foreach ($patients as &$patient) {
            $patient['text'] = $patient['name'] . ' (ID: ' . $patient['id'] . ')';
        }
        
        echo json_encode($patients);
        
    } catch (Exception $e) {
        echo json_encode([]);
    }
}

/**
 * Save patient notes
 */
public function saveNotes() {
    try {
        $patient_id = $this->input->post('patient_id');
        $notes = $this->input->post('notes');
        
        if (!$patient_id) {
            echo json_encode([
                'status' => false,
                'message' => 'Patient ID is required'
            ]);
            return;
        }
        
        // Check if notes exist
        $existing = $this->db->where('patient_id', $patient_id)->get('patient_notes')->row();
        
        if ($existing) {
            // Update existing notes
            $this->db->where('patient_id', $patient_id)
                    ->update('patient_notes', [
                        'notes' => $notes,
                        'updated_at' => date('Y-m-d H:i:s'),
                        'updated_by' => $this->session->userdata('user_id')
                    ]);
        } else {
            // Insert new notes
            $this->db->insert('patient_notes', [
                'patient_id' => $patient_id,
                'notes' => $notes,
                'created_at' => date('Y-m-d H:i:s'),
                'created_by' => $this->session->userdata('user_id')
            ]);
        }
        
        if ($this->db->affected_rows() > 0) {
            echo json_encode([
                'status' => true,
                'message' => 'Notes saved successfully'
            ]);
        } else {
            echo json_encode([
                'status' => false,
                'message' => 'No changes made to notes'
            ]);
        }
        
    } catch (Exception $e) {
        echo json_encode([
            'status' => false,
            'message' => 'Error saving notes: ' . $e->getMessage()
        ]);
    }
}

/**
 * Get consultations with lab tests
 */
public function getConsultationsWithLabTests() {
    try {
        $consultations = $this->db->select('c.*, p.patient_name, s.name as doctor_name, s.surname as doctor_surname,
                                           COUNT(ptr.id) as tests_count')
                                 ->from('consultations c')
                                 ->join('patients p', 'c.patient_id = p.id', 'left')
                                 ->join('staff s', 'c.doctor = s.id', 'left')
                                 ->join('pathology_test_reports ptr', 'c.id = ptr.consultation_id', 'left')
                                 ->group_by('c.id')
                                 ->order_by('c.consultation_date', 'DESC')
                                 ->limit(100)
                                 ->get()
                                 ->result_array();
        
        // Format data
        foreach ($consultations as &$consultation) {
            $consultation['doctor_name'] = trim(($consultation['doctor_name'] ?? '') . ' ' . ($consultation['doctor_surname'] ?? ''));
            $consultation['verification_status'] = $consultation['verification_status'] ? 'verified' : 'unverified';
        }
        
        echo json_encode([
            'status' => 'success',
            'consultations' => $consultations
        ]);
        
    } catch (Exception $e) {
        echo json_encode([
            'status' => 'error',
            'message' => 'Error loading consultations: ' . $e->getMessage()
        ]);
    }
}

/**
 * Save consultation with lab tests
 */
/*public function saveWithLabTests() {
    try {
        $this->db->trans_start();
        
        // Get form data
        $patient_id = $this->input->post('patient_id');
        $consultation_date = $this->input->post('consultation_date');
        $doctor = $this->input->post('doctor');
        $symptoms = $this->input->post('symptoms');
        $notes = $this->input->post('notes');
        $selected_tests = $this->input->post('selected_tests');
        
        if (!$patient_id || !$consultation_date || !$doctor || !$symptoms) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Required fields are missing'
            ]);
            return;
        }
        
        // Save consultation
        $consultation_data = [
            'patient_id' => $patient_id,
            'consultation_date' => $consultation_date,
            'doctor' => $doctor,
            'symptoms' => $symptoms,
            'notes' => $notes,
            'verification_status' => $this->input->post('verification_status') ?: 0,
            'verification_score' => $this->input->post('verification_score') ?: 0,
            'created_at' => date('Y-m-d H:i:s')
        ];
        
        // Add diagnosis data if available
        if ($this->input->post('diagnosis_code')) {
            $consultation_data['diagnosis_code'] = $this->input->post('diagnosis_code');
            $consultation_data['diagnosis_title'] = $this->input->post('diagnosis_title');
        } elseif ($this->input->post('diagnosis_custom')) {
            $consultation_data['diagnosis_custom'] = $this->input->post('diagnosis_custom');
        }
        
        $this->db->insert('consultations', $consultation_data);
        $consultation_id = $this->db->insert_id();
        
        // Save lab tests if selected
        if ($selected_tests) {
            $tests = json_decode($selected_tests, true);
            if (is_array($tests) && !empty($tests)) {
                foreach ($tests as $test) {
                    $test_data = [
                        'patient_id' => $patient_id,
                        'consultation_id' => $consultation_id,
                        'test_id' => $test['id'],
                        'test_name' => $test['test_name'],
                        'test_category_id' => $test['category'],
                        'test_date' => $consultation_date,
                        'cost' => $test['cost'] ?: 0,
                        'priority' => $this->input->post('test_priority') ?: 'normal',
                        'instructions' => $this->input->post('test_instructions'),
                        'result_status' => 'pending',
                        'ordered_by' => $doctor,
                        'created_at' => date('Y-m-d H:i:s')
                    ];
                    
                    $this->db->insert('pathology_test_reports', $test_data);
                }
            }
        }
        
        $this->db->trans_complete();
        
        if ($this->db->trans_status() === FALSE) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Failed to save consultation and tests'
            ]);
        } else {
            $test_count = is_array(json_decode($selected_tests, true)) ? count(json_decode($selected_tests, true)) : 0;
            echo json_encode([
                'status' => 'success',
                'message' => "Consultation saved successfully with {$test_count} lab tests ordered",
                'consultation_id' => $consultation_id
            ]);
        }
        
    } catch (Exception $e) {
        $this->db->trans_rollback();
        echo json_encode([
            'status' => 'error',
            'message' => 'Error saving consultation: ' . $e->getMessage()
        ]);
    }
}*/

/**
 * Get patient review data for the review tab
 */
public function getPatientReview() {
    try {
        $patient_id = $this->input->post('patient_id');
        $time_period = $this->input->post('time_period', true) ?: 3; // Default 3 months
        
        if (!$patient_id) {
            echo json_encode([
                'status' => false,
                'message' => 'Patient ID is required'
            ]);
            return;
        }
        
        // Calculate date range
        $end_date = date('Y-m-d');
        if ($time_period == 0) {
            $start_date = '1900-01-01'; // All time
        } else {
            $start_date = date('Y-m-d', strtotime("-{$time_period} months"));
        }
        
        // Get patient basic information
        $patient = $this->db->select('id, patient_name as name, age, gender, mobileno as phone, blood_group_name, known_allergies')
                           ->where('id', $patient_id)
                           ->get('patients')
                           ->row_array();
        
        if (!$patient) {
            echo json_encode([
                'status' => false,
                'message' => 'Patient not found'
            ]);
            return;
        }
        
        // Get consultations count and data
        $consultations = $this->db->select('c.*, s.name as doctor_name, s.surname as doctor_surname')
                                 ->from('consultations c')
                                 ->join('staff s', 'c.doctor = s.id', 'left')
                                 ->where('c.patient_id', $patient_id)
                                 ->where('c.consultation_date >=', $start_date)
                                 ->where('c.consultation_date <=', $end_date)
                                 ->order_by('c.consultation_date', 'DESC')
                                 ->limit(10)
                                 ->get()
                                 ->result_array();
        
        // Format consultations
        foreach ($consultations as &$consultation) {
            $consultation['doctor'] = trim(($consultation['doctor_name'] ?? '') . ' ' . ($consultation['doctor_surname'] ?? ''));
            $consultation['date'] = date('M d, Y', strtotime($consultation['consultation_date']));
        }
        
        // Get triage records
        $triage_records = $this->db->select('tr.*, s.name as nurse_name, s.surname as nurse_surname')
                                  ->from('triage_records tr')
                                  ->join('staff s', 'tr.staff_id = s.id', 'left')
                                  ->where('tr.patient_id', $patient_id)
                                  ->where('tr.date >=', $start_date)
                                  ->where('tr.date <=', $end_date)
                                  ->order_by('tr.date', 'DESC')
                                  ->limit(10)
                                  ->get()
                                  ->result_array();
        
        // Format triage records
        foreach ($triage_records as &$record) {
            $record['nurse_name'] = trim(($record['nurse_name'] ?? '') . ' ' . ($record['nurse_surname'] ?? ''));
            $record['date'] = date('M d, Y', strtotime($record['date']));
            
            // Calculate BMI if weight and height available
            if ($record['weight'] && $record['height']) {
                $height_m = $record['height'] / 100;
                $record['bmi'] = round($record['weight'] / ($height_m * $height_m), 1);
            }
        }
        
        // Get medical history
        $medical_history = [];
        
        // Chronic conditions
        $medical_history['chronic_conditions'] = $this->db->select('*')
                                                         ->where('patient_id', $patient_id)
                                                         ->get('patient_chronic_conditions')
                                                         ->result_array();
        
        // Current medications
        $medical_history['medications'] = $this->db->select('pm.*, m.medicine_name as name')
                                                  ->from('patient_medications pm')
                                                  ->join('medicine_list m', 'pm.medicine_id = m.id', 'left')
                                                  ->where('pm.patient_id', $patient_id)
                                                  ->where('pm.status', 'active')
                                                  ->get()
                                                  ->result_array();
        
        // Allergies
        $medical_history['allergies'] = $this->db->select('*')
                                                ->where('patient_id', $patient_id)
                                                ->get('patient_allergies')
                                                ->result_array();
        
        // Past procedures
        $medical_history['procedures'] = $this->db->select('pp.*, s.name as doctor_name, s.surname as doctor_surname')
                                                 ->from('patient_procedures pp')
                                                 ->join('staff s', 'pp.doctor_id = s.id', 'left')
                                                 ->where('pp.patient_id', $patient_id)
                                                 ->order_by('pp.procedure_date', 'DESC')
                                                 ->get()
                                                 ->result_array();
        
        // Format procedures
        foreach ($medical_history['procedures'] as &$procedure) {
            $procedure['doctor'] = trim(($procedure['doctor_name'] ?? '') . ' ' . ($procedure['doctor_surname'] ?? ''));
            $procedure['date'] = date('M d, Y', strtotime($procedure['procedure_date']));
        }
        
        // Get patient notes
        $notes_result = $this->db->select('notes')
                               ->where('patient_id', $patient_id)
                               ->get('patient_notes')
                               ->row();
        $notes = $notes_result ? $notes_result->notes : '';
        
        // Calculate summary
        $summary = [
            'consultations' => count($consultations),
            'prescriptions' => $this->db->where('patient_id', $patient_id)
                                       ->where('date >=', $start_date)
                                       ->where('date <=', $end_date)
                                       ->count_all_results('prescriptions'),
            'lab_tests' => $this->db->where('patient_id', $patient_id)
                                   ->where('test_date >=', $start_date)
                                   ->where('test_date <=', $end_date)
                                   ->count_all_results('pathology_test_reports')
        ];
        
        echo json_encode([
            'status' => 'success',
            'data' => [
                'patient' => $patient,
                'summary' => $summary,
                'consultations' => $consultations,
                'triage_records' => $triage_records,
                'medical_history' => $medical_history,
                'notes' => $notes
            ]
        ]);
        
    } catch (Exception $e) {
        echo json_encode([
            'status' => false,
            'message' => 'Error loading patient review: ' . $e->getMessage()
        ]);
    }
}

/**
 * Get lab summary for dashboard
 */
public function getLabSummary() {
    try {
        $today = date('Y-m-d');
        $this_month_start = date('Y-m-01');
        
        // Get summary counts
        $summary = [
            'total_tests' => $this->db->count_all_results('pathology_test_reports'),
            'pending_tests' => $this->db->where('result_status', 'pending')
                                       ->count_all_results('pathology_test_reports'),
            'completed_tests' => $this->db->where('result_status', 'completed')
                                         ->count_all_results('pathology_test_reports'),
            'critical_results' => $this->db->where('result_status', 'critical')
                                          ->count_all_results('pathology_test_reports')
        ];
        
        // Get chart data - tests by category
        $category_data = $this->db->select('pc.category_name, COUNT(ptr.id) as test_count')
                                 ->from('pathology_test_reports ptr')
                                 ->join('pathology_categories pc', 'ptr.test_category_id = pc.id', 'left')
                                 ->group_by('ptr.test_category_id')
                                 ->get()
                                 ->result_array();
        
        $category_labels = [];
        $category_counts = [];
        foreach ($category_data as $data) {
            $category_labels[] = $data['category_name'] ?: 'Other';
            $category_counts[] = (int)$data['test_count'];
        }
        
        // Get status data
        $status_data = [
            $summary['pending_tests'],
            $summary['completed_tests'], 
            $summary['critical_results']
        ];
        
        $charts = [
            'category_labels' => $category_labels,
            'category_data' => $category_counts,
            'status_data' => $status_data
        ];
        
        echo json_encode([
            'status' => 'success',
            'summary' => $summary,
            'charts' => $charts
        ]);
        
    } catch (Exception $e) {
        echo json_encode([
            'status' => 'error',
            'message' => 'Error loading lab summary: ' . $e->getMessage()
        ]);
    }
}

/**
 * Get recent lab tests for table
 */
public function getRecentLabTests() {
    try {
        $tests = $this->db->select('ptr.*, p.patient_name, pc.category_name, lt.test_name')
                         ->from('pathology_test_reports ptr')
                         ->join('patients p', 'ptr.patient_id = p.id', 'left')
                         ->join('pathology_categories pc', 'ptr.test_category_id = pc.id', 'left')
                         ->join('lab_tests lt', 'ptr.test_id = lt.id', 'left')
                         ->order_by('ptr.test_date', 'DESC')
                         ->limit(50)
                         ->get()
                         ->result_array();
        
        // Format test dates
        foreach ($tests as &$test) {
            $test['test_date'] = date('M d, Y', strtotime($test['test_date']));
        }
        
        echo json_encode([
            'status' => 'success',
            'tests' => $tests
        ]);
        
    } catch (Exception $e) {
        echo json_encode([
            'status' => 'error',
            'message' => 'Error loading lab tests: ' . $e->getMessage()
        ]);
    }
}

/**
 * Get test details for results entry
 */
public function getTestDetails() {
    try {
        $test_id = $this->input->post('test_id');
        
        if (!$test_id) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Test ID is required'
            ]);
            return;
        }
        
        $test = $this->db->select('ptr.*, p.patient_name, lt.test_name, pc.category_name')
                        ->from('pathology_test_reports ptr')
                        ->join('patients p', 'ptr.patient_id = p.id', 'left')
                        ->join('lab_tests lt', 'ptr.test_id = lt.id', 'left')
                        ->join('pathology_categories pc', 'ptr.test_category_id = pc.id', 'left')
                        ->where('ptr.id', $test_id)
                        ->get()
                        ->row_array();
        
        if (!$test) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Test not found'
            ]);
            return;
        }
        
        echo json_encode([
            'status' => 'success',
            'test' => $test
        ]);
        
    } catch (Exception $e) {
        echo json_encode([
            'status' => 'error',
            'message' => 'Error loading test details: ' . $e->getMessage()
        ]);
    }
}

/**
 * Save test results
 */
public function saveTestResults() {
    try {
        $test_id = $this->input->post('test_id');
        $test_result = $this->input->post('test_result');
        $result_status = $this->input->post('result_status');
        $reference_range = $this->input->post('reference_range');
        $lab_notes = $this->input->post('lab_notes');
        
        if (!$test_id || !$test_result) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Test ID and result are required'
            ]);
            return;
        }
        
        $update_data = [
            'test_result' => $test_result,
            'result_status' => $result_status ?: 'completed',
            'reference_range' => $reference_range,
            'notes' => $lab_notes,
            'result_date' => date('Y-m-d H:i:s'),
            'technician_id' => $this->session->userdata('user_id'),
            'updated_at' => date('Y-m-d H:i:s')
        ];
        
        $this->db->where('id', $test_id)->update('pathology_test_reports', $update_data);
        
        if ($this->db->affected_rows() > 0) {
            echo json_encode([
                'status' => 'success',
                'message' => 'Test results saved successfully'
            ]);
        } else {
            echo json_encode([
                'status' => 'error',
                'message' => 'Failed to save test results'
            ]);
        }
        
    } catch (Exception $e) {
        echo json_encode([
            'status' => 'error',
            'message' => 'Error saving test results: ' . $e->getMessage()
        ]);
    }
}

/**
 * Generate lab report
 */
public function generateLabReport() {
    try {
        $date_from = $this->input->post('date_from');
        $date_to = $this->input->post('date_to');
        $category_id = $this->input->post('category_id');
        $test_status = $this->input->post('test_status');
        
        if (!$date_from || !$date_to) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Date range is required'
            ]);
            return;
        }
        
        // Build query
        $this->db->select('ptr.*, p.patient_name, pc.category_name, lt.test_name')
                ->from('pathology_test_reports ptr')
                ->join('patients p', 'ptr.patient_id = p.id', 'left')
                ->join('pathology_categories pc', 'ptr.test_category_id = pc.id', 'left')
                ->join('lab_tests lt', 'ptr.test_id = lt.id', 'left')
                ->where('ptr.test_date >=', $date_from)
                ->where('ptr.test_date <=', $date_to);
        
        if ($category_id) {
            $this->db->where('ptr.test_category_id', $category_id);
        }
        
        if ($test_status) {
            $this->db->where('ptr.result_status', $test_status);
        }
        
        $tests = $this->db->order_by('ptr.test_date', 'DESC')->get()->result_array();
        
        // Calculate report summary
        $total_tests = count($tests);
        $completed_tests = count(array_filter($tests, function($test) {
            return $test['result_status'] === 'completed';
        }));
        $pending_tests = count(array_filter($tests, function($test) {
            return $test['result_status'] === 'pending';
        }));
        $critical_tests = count(array_filter($tests, function($test) {
            return $test['result_status'] === 'critical';
        }));
        
        $report = [
            'total_tests' => $total_tests,
            'completed_tests' => $completed_tests,
            'pending_tests' => $pending_tests,
            'critical_tests' => $critical_tests,
            'tests' => $tests,
            'date_range' => [
                'from' => $date_from,
                'to' => $date_to
            ]
        ];
        
        echo json_encode([
            'status' => 'success',
            'report' => $report
        ]);
        
    } catch (Exception $e) {
        echo json_encode([
            'status' => 'error',
            'message' => 'Error generating report: ' . $e->getMessage()
        ]);
    }
}

/**
 * Get lab tests for consultation (for the modal)
 */
public function getLabTestsForConsultation() {
    try {
        // Get test categories
        $categories = $this->db->select('id, category_name')
                              ->where('is_active', 1)
                              ->get('pathology_categories')
                              ->result_array();
        
        $categories_list = [];
        foreach ($categories as $category) {
            $categories_list[$category['id']] = $category['category_name'];
        }
        
        // Get tests by category
        $tests_by_category = [];
        foreach ($categories as $category) {
            $tests = $this->db->select('id, test_name, short_name, cost, sample_type, method')
                             ->where('category_id', $category['id'])
                             ->where('is_active', 1)
                             ->order_by('test_name')
                             ->get('lab_tests')
                             ->result_array();
            
            if (!empty($tests)) {
                $tests_by_category[$category['id']] = $tests;
            }
        }
        
        echo json_encode([
            'status' => 'success',
            'categories' => $categories_list,
            'tests_by_category' => $tests_by_category
        ]);
        
    } catch (Exception $e) {
        echo json_encode([
            'status' => 'error',
            'message' => 'Error loading lab tests: ' . $e->getMessage()
        ]);
    }
}

}