<?php

if (!defined('BASEPATH')) {
    exit('No direct script access allowed');
}

class Agent extends Admin_Controller
{
    public function __construct()
    {
        parent::__construct();
        $this->load->model('Agent_model');
        $this->load->model('Patient_model');
    }

    /**
     * Agent management dashboard
     */
    public function index()
    {
        if (!$this->rbac->hasPrivilege('agent_management', 'can_view')) {
            access_denied();
        }

        $data['title'] = 'Agent Management';
        $data['agents'] = $this->Agent_model->get_active_agents();
        
        $this->load->view('layout/header', $data);
        $this->load->view('admin/agent/index', $data);
        $this->load->view('layout/footer', $data);
    }

    /**
     * AJAX: Get active agents for registration form
     */
    public function get_active_agents()
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }

        try {
            $agents = $this->Agent_model->get_active_agents();
            
            if ($agents) {
                echo json_encode(['status' => 'success', 'agents' => $agents]);
            } else {
                echo json_encode(['status' => 'error', 'message' => 'No active agents found']);
            }
        } catch (Exception $e) {
            echo json_encode(['status' => 'error', 'message' => $e->getMessage()]);
        }
    }

    /**
     * AJAX: Get countries available for a specific agent
     */
    public function get_agent_countries()
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }

        $agent_id = $this->input->post('agent_id');

        if (!$agent_id) {
            echo json_encode(['status' => 'error', 'message' => 'Agent ID is required']);
            return;
        }

        try {
            $countries = $this->Agent_model->get_agent_countries($agent_id);
            
            if ($countries) {
                echo json_encode(['status' => 'success', 'countries' => $countries]);
            } else {
                echo json_encode(['status' => 'error', 'message' => 'No available destinations for this agent']);
            }
        } catch (Exception $e) {
            echo json_encode(['status' => 'error', 'message' => $e->getMessage()]);
        }
    }

    /**
     * AJAX: Get contract details
     */
    public function get_contract_details()
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }

        $contract_id = $this->input->post('contract_id');

        if (!$contract_id) {
            echo json_encode(['status' => 'error', 'message' => 'Contract ID is required']);
            return;
        }

        try {
            $contract = $this->Agent_model->get_contract_details($contract_id);
            
            if ($contract) {
                echo json_encode(['status' => 'success', 'contract' => $contract]);
            } else {
                echo json_encode(['status' => 'error', 'message' => 'Contract not found']);
            }
        } catch (Exception $e) {
            echo json_encode(['status' => 'error', 'message' => $e->getMessage()]);
        }
    }

    /**
     * Agent configuration page
     */
    public function configure()
    {
        if (!$this->rbac->hasPrivilege('agent_management', 'can_edit')) {
            access_denied();
        }

        $data['title'] = 'Agent Configuration';
        $data['agents'] = $this->Agent_model->get_active_agents();
        $data['countries'] = $this->Agent_model->get_all_countries();
        
        $this->load->view('layout/header', $data);
        $this->load->view('admin/agent/configure', $data);
        $this->load->view('layout/footer', $data);
    }

    /**
     * Create new agent
     */
    public function create_agent()
    {
        if (!$this->rbac->hasPrivilege('agent_management', 'can_add')) {
            access_denied();
        }

        $this->form_validation->set_rules('agent_name', 'Agent Name', 'trim|required|xss_clean');
        $this->form_validation->set_rules('contact_person', 'Contact Person', 'trim|required|xss_clean');
        $this->form_validation->set_rules('phone', 'Phone', 'trim|required|xss_clean');
        $this->form_validation->set_rules('email', 'Email', 'trim|required|valid_email|xss_clean');

        if ($this->form_validation->run() === FALSE) {
            $error_array = array();
            $error_array['agent_name'] = form_error('agent_name');
            $error_array['contact_person'] = form_error('contact_person');
            $error_array['phone'] = form_error('phone');
            $error_array['email'] = form_error('email');

            echo json_encode(array('status' => 'fail', 'error' => $error_array));
        } else {
            $agent_data = array(
                'agent_name' => $this->input->post('agent_name'),
                'company_name' => $this->input->post('company_name'),
                'contact_person' => $this->input->post('contact_person'),
                'phone' => $this->input->post('phone'),
                'email' => $this->input->post('email'),
                'address' => $this->input->post('address'),
                'license_number' => $this->input->post('license_number'),
                'license_expiry' => $this->input->post('license_expiry'),
                'commission_type' => $this->input->post('commission_type') ?: 'per_patient',
                'notes' => $this->input->post('notes'),
                'status' => 'active',
                'created_by' => $this->session->userdata('admin_id')
            );

            $agent_id = $this->Agent_model->create_agent($agent_data);

            if ($agent_id) {
                echo json_encode(array('status' => 'success', 'message' => 'Agent created successfully', 'agent_id' => $agent_id));
            } else {
                echo json_encode(array('status' => 'fail', 'message' => 'Failed to create agent'));
            }
        }
    }

    /**
     * Create contract for agent-country combination
     */
    public function create_contract()
    {
        if (!$this->rbac->hasPrivilege('agent_management', 'can_add')) {
            access_denied();
        }

        $this->form_validation->set_rules('agent_id', 'Agent', 'trim|required|xss_clean|numeric');
        $this->form_validation->set_rules('country_id', 'Country', 'trim|required|xss_clean|numeric');
        $this->form_validation->set_rules('contract_name', 'Contract Name', 'trim|required|xss_clean');
        $this->form_validation->set_rules('total_cost_per_patient', 'Total Cost', 'trim|required|xss_clean|numeric');
        $this->form_validation->set_rules('agent_commission_per_patient', 'Agent Commission', 'trim|required|xss_clean|numeric');
        $this->form_validation->set_rules('patient_limit', 'Patient Limit', 'trim|required|xss_clean|numeric');
        $this->form_validation->set_rules('contract_start_date', 'Start Date', 'trim|required|xss_clean');
        $this->form_validation->set_rules('contract_end_date', 'End Date', 'trim|required|xss_clean');

        if ($this->form_validation->run() === FALSE) {
            $error_array = array();
            $error_array['agent_id'] = form_error('agent_id');
            $error_array['country_id'] = form_error('country_id');
            $error_array['contract_name'] = form_error('contract_name');
            $error_array['total_cost_per_patient'] = form_error('total_cost_per_patient');
            $error_array['agent_commission_per_patient'] = form_error('agent_commission_per_patient');
            $error_array['patient_limit'] = form_error('patient_limit');
            $error_array['contract_start_date'] = form_error('contract_start_date');
            $error_array['contract_end_date'] = form_error('contract_end_date');

            echo json_encode(array('status' => 'fail', 'error' => $error_array));
        } else {
            $total_cost = $this->input->post('total_cost_per_patient');
            $commission = $this->input->post('agent_commission_per_patient');
            
            $contract_data = array(
                'agent_id' => $this->input->post('agent_id'),
                'country_id' => $this->input->post('country_id'),
                'contract_name' => $this->input->post('contract_name'),
                'total_cost_per_patient' => $total_cost,
                'agent_commission_per_patient' => $commission,
                'hospital_amount_per_patient' => $total_cost - $commission,
                'patient_limit' => $this->input->post('patient_limit'),
                'contract_start_date' => $this->input->post('contract_start_date'),
                'contract_end_date' => $this->input->post('contract_end_date'),
                'special_conditions' => $this->input->post('special_conditions'),
                'auto_renew' => $this->input->post('auto_renew') ?: 'no',
                'renewal_notice_days' => $this->input->post('renewal_notice_days') ?: 30,
                'status' => 'active',
                'created_by' => $this->session->userdata('admin_id'),
                'approved_by' => $this->session->userdata('admin_id'),
                'approved_at' => date('Y-m-d H:i:s')
            );

            $contract_id = $this->Agent_model->create_contract($contract_data);

            if ($contract_id) {
                echo json_encode(array('status' => 'success', 'message' => 'Contract created successfully', 'contract_id' => $contract_id));
            } else {
                echo json_encode(array('status' => 'fail', 'message' => 'Failed to create contract'));
            }
        }
    }

    /**
     * Agent dashboard - statistics and overview
     */
    public function dashboard($agent_id = null)
    {
        if (!$this->rbac->hasPrivilege('agent_management', 'can_view')) {
            access_denied();
        }

        if (!$agent_id) {
            show_404();
        }

        $data['title'] = 'Agent Dashboard';
        $data['agent'] = $this->Agent_model->get_agent_with_contracts($agent_id);
        
        if (!$data['agent']) {
            show_404();
        }

        $data['stats'] = $this->Agent_model->get_agent_stats($agent_id);
        $data['recent_patients'] = $this->Agent_model->get_patient_history($agent_id, 10);
        
        $this->load->view('layout/header', $data);
        $this->load->view('admin/agent/dashboard', $data);
        $this->load->view('layout/footer', $data);
    }

    /**
     * Update contract status
     */
    public function update_contract_status()
    {
        if (!$this->rbac->hasPrivilege('agent_management', 'can_edit')) {
            access_denied();
        }

        $contract_id = $this->input->post('contract_id');
        $status = $this->input->post('status');

        if (!$contract_id || !$status) {
            echo json_encode(['status' => 'error', 'message' => 'Missing required parameters']);
            return;
        }

        $valid_statuses = ['draft', 'active', 'expired', 'terminated', 'full'];
        if (!in_array($status, $valid_statuses)) {
            echo json_encode(['status' => 'error', 'message' => 'Invalid status']);
            return;
        }

        try {
            $result = $this->Agent_model->update_contract_status($contract_id, $status);
            
            if ($result) {
                echo json_encode(['status' => 'success', 'message' => 'Contract status updated successfully']);
            } else {
                echo json_encode(['status' => 'error', 'message' => 'Failed to update contract status']);
            }
        } catch (Exception $e) {
            echo json_encode(['status' => 'error', 'message' => $e->getMessage()]);
        }
    }

    /**
     * Get expiring contracts for notifications
     */
    public function get_expiring_contracts()
    {
        if (!$this->rbac->hasPrivilege('agent_management', 'can_view')) {
            access_denied();
        }

        $days_ahead = $this->input->get('days') ?: 30;
        $contracts = $this->Agent_model->get_expiring_contracts($days_ahead);
        
        echo json_encode(['status' => 'success', 'contracts' => $contracts]);
    }

    /**
     * Export agent performance report
     */
    public function export_performance_report($agent_id = null)
    {
        if (!$this->rbac->hasPrivilege('agent_management', 'can_view')) {
            access_denied();
        }

        // Implementation for exporting performance reports
        // This would generate CSV/PDF reports for agent performance
        
        echo json_encode(['status' => 'success', 'message' => 'Report generation started']);
    }

    /**
     * AJAX: Get agents with their contracts for admin view
     */
    public function get_agents_with_contracts()
    {
        if (!$this->rbac->hasPrivilege('agent_management', 'can_view')) {
            access_denied();
        }

        try {
            $agents = $this->Agent_model->get_active_agents();
            
            // Get contracts for each agent
            foreach ($agents as &$agent) {
                $agent['contracts'] = $this->Agent_model->get_agent_countries($agent['id']);
                
                // Add days to expiry for each contract
                foreach ($agent['contracts'] as &$contract) {
                    $days_to_expiry = (strtotime($contract['contract_end_date']) - time()) / (60 * 60 * 24);
                    $contract['days_to_expiry'] = floor($days_to_expiry);
                }
            }
            
            echo json_encode(['status' => 'success', 'agents' => $agents]);
        } catch (Exception $e) {
            echo json_encode(['status' => 'error', 'message' => $e->getMessage()]);
        }
    }

    /**
     * AJAX: Get quick statistics for dashboard
     */
    public function get_quick_stats()
    {
        if (!$this->rbac->hasPrivilege('agent_management', 'can_view')) {
            access_denied();
        }

        try {
            // Total agents
            $this->db->where('status', 'active');
            $total_agents = $this->db->count_all_results('agents');
            
            // Active contracts
            $this->db->where('status', 'active');
            $this->db->where('contract_end_date >', date('Y-m-d'));
            $active_contracts = $this->db->count_all_results('agent_country_contracts');
            
            // Total patients registered this year
            $this->db->select_sum('patients_registered');
            $this->db->where('status', 'active');
            $result = $this->db->get('agent_country_contracts')->row_array();
            $total_patients = $result['patients_registered'] ?: 0;
            
            // Expiring contracts (within 30 days)
            $expiring_contracts = count($this->Agent_model->get_expiring_contracts(30));
            
            $stats = [
                'total_agents' => $total_agents,
                'active_contracts' => $active_contracts,
                'total_patients' => $total_patients,
                'expiring_contracts' => $expiring_contracts
            ];
            
            echo json_encode(['status' => 'success', 'stats' => $stats]);
        } catch (Exception $e) {
            echo json_encode(['status' => 'error', 'message' => $e->getMessage()]);
        }
    }
}