// SecuGen WebAPI JavaScript Integration

// Global variables
let secugenConfig = {
    baseUrl: 'https://localhost:8443', // fallback default
    quality: 50,
    timeout: 10000,
    templateFormat: 'ISO19794'
};

// Helper to get the current endpoint
function getSecuGenBaseUrl() {
    if (window.secuGenEndpoint && typeof window.secuGenEndpoint === 'string') {
        return window.secuGenEndpoint;
    }
    return secugenConfig.baseUrl;
}

// Initialize SecuGen WebAPI
function initSecuGenAPI() {
    // Create necessary HTML elements if not present
    if (!document.getElementById('fingerprintImage')) {
        console.error('Required HTML elements not found');
        return false;
    }
    return true;
}

// Call SGIFPGetData to capture fingerprint
async function CallSGIFPGetData() {
    try {
        // Update status
        document.getElementById('fingerprintStatus').innerHTML = 'Scanning fingerprint...';
        
        // Get dynamic endpoint
        const baseUrl = getSecuGenBaseUrl();
        if (!baseUrl) {
            throw new Error('No SecuGen WebAPI endpoint found.');
        }
        // Call capture endpoint
        const response = await fetch(`${baseUrl}/SGIFPCapture`, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify({
                quality: secugenConfig.quality,
                timeout: secugenConfig.timeout,
                templateFormat: secugenConfig.templateFormat
            })
        });

        if (!response.ok) {
            throw new Error('Network response was not ok');
        }

        const data = await response.json();
        
        // Handle successful capture
        SuccessFunc(data);
    } catch (error) {
        // Handle error
        ErrorFunc(error);
    }
}

// Success handler
function SuccessFunc(result) {
    if (result && result.imageData) {
        // Display fingerprint image
        const fingerprintImage = document.getElementById('fingerprintImage');
        fingerprintImage.src = 'data:image/png;base64,' + result.imageData;
        fingerprintImage.style.display = 'block';

        // Store template
        document.getElementById('fingerprint_template').value = result.templateData;

        // Update status
        document.getElementById('fingerprintStatus').innerHTML = 'Fingerprint captured successfully!';
        document.getElementById('fingerprintStatus').style.color = 'green';
    } else {
        ErrorFunc(new Error('Invalid response data'));
    }
}

// Error handler
function ErrorFunc(error) {
    console.error('Fingerprint capture error:', error);
    document.getElementById('fingerprintStatus').innerHTML = 'Error capturing fingerprint. Please try again.';
    document.getElementById('fingerprintStatus').style.color = 'red';
}

// Initialize when document is ready
document.addEventListener('DOMContentLoaded', function() {
    // Initialize SecuGen API
    if (initSecuGenAPI()) {
        // Add click handler for scan button
        document.getElementById('scanFingerprint').addEventListener('click', function() {
            CallSGIFPGetData();
        });
    }
});