# 🏥 Hospital Queue Management System Implementation

## Overview
This document provides complete setup and usage instructions for the integrated Queue Management System in your City Health Hospital Management System.

## ✅ What's Been Implemented

### 1. Core Queue System
- **Multi-department queue management** for all hospital departments
- **Real-time notifications** with sound alerts
- **Priority-based queue ordering** (urgent, high, normal, low)
- **Automatic patient routing** between departments
- **Queue status tracking** and analytics

### 2. Integrated Departments
- ✅ **Registration** - Auto-adds patients to triage queue
- ✅ **Triage** - Queue-aware patient selection and routing
- ✅ **Consultation** - Displays waiting patients from triage
- ✅ **Pharmacy** - Shows prescription patients
- ✅ **Laboratory** - Lab test queue management
- ✅ **Radiology** - Imaging appointment queue
- ✅ **Phlebotomy** - Blood draw queue
- ✅ **Emergency** - Urgent patient handling

### 3. JavaScript Integration
- **Automatic department detection** based on current page
- **Queue panels** appear on relevant department pages
- **Sound notifications** for new patients
- **Real-time queue updates** every 30 seconds
- **Patient selection integration** with existing forms

## 🚀 Installation & Setup

### Step 1: Database Setup
1. Run the setup script:
   ```bash
   # Via browser
   http://yourdomain.com/setup_queue_system.php
   
   # Or via CLI
   php setup_queue_system.php
   ```

2. Verify tables created:
   - `patient_queue` - Main queue records
   - `queue_config` - Department configurations
   - `queue_display_settings` - UI display settings
   - `queue_flow_rules` - Workflow routing rules
   - `queue_notifications` - Real-time notifications

### Step 2: File Verification
Ensure these files exist and are in place:
- ✅ `assets/js/multi_department_queue_integration.js`
- ✅ `application/controllers/admin/Queue.php`
- ✅ `application/models/Queue_model.php`
- ✅ Footer integration added to `layout/footer.php`

### Step 3: Test the System
1. **Register a new patient** - Should auto-add to triage queue
2. **Visit triage page** - Should show queue panel with patients
3. **Select patient from queue** - Should mark as in-progress
4. **Complete triage** - Should route to next department

## 📋 How It Works

### Patient Registration Flow
```
1. Patient registered via admin/patient/create
2. System determines initial department (usually triage)
3. Patient added to queue with appropriate priority
4. Notification sent to department staff
5. Queue panel updates in real-time
```

### Department Integration Flow
```
1. Staff visits department page (e.g., /admin/triage)
2. JavaScript detects current department
3. Queue panel loads showing waiting patients
4. Staff selects patient from queue
5. Patient marked as in-progress
6. Upon completion, routed to next department
```

### Priority System
- **Urgent** - Emergency cases, immediate attention
- **High** - Elderly patients (65+), infants (≤2), specialized cases
- **Normal** - Standard workflow patients
- **Low** - Non-urgent cases

## 🎛️ Configuration Options

### Queue Configuration (`queue_config` table)
- `queue_prefix` - Department queue number prefix (e.g., TRI001)
- `average_service_time` - Expected service duration in minutes
- `max_queue_size` - Maximum patients in queue
- `sound_enabled` - Enable/disable notification sounds
- `counter_reset` - Daily/weekly/monthly queue number reset

### Display Settings (`queue_display_settings` table)
- `display_name` - Department name shown in queue
- `theme_color` - Queue panel color scheme
- `icon_class` - FontAwesome icon for department
- `display_order` - Order in queue dashboard

### Flow Rules (`queue_flow_rules` table)
- **Automatic** - Patients auto-route after service completion
- **Manual** - Staff manually routes patients
- **Conditional** - Route based on specific conditions

## 🔧 Customization Guide

### Adding New Departments
1. Add to queue config:
   ```sql
   INSERT INTO queue_config (department, queue_prefix, average_service_time) 
   VALUES ('cardiology', 'CAR', 30);
   ```

2. Add display settings:
   ```sql
   INSERT INTO queue_display_settings (department, display_name, theme_color, icon_class) 
   VALUES ('cardiology', 'Cardiology', '#ff6b6b', 'fa-heart');
   ```

3. Update JavaScript integration in your controller:
   ```php
   $this->load->model('Queue_model');
   $data['queue_data'] = $this->Queue_model->getDepartmentQueue('cardiology');
   ```

### Custom Priority Logic
Modify the `getPatientPriority()` method in `Patient.php`:
```php
private function getPatientPriority($patient_id, $workflow_type) {
    // Add your custom priority logic here
    // e.g., check for VIP status, insurance type, etc.
}
```

### Custom Workflow Routing
Add new flow rules:
```sql
INSERT INTO queue_flow_rules 
(rule_name, from_department, to_department, condition_type, workflow_type)
VALUES 
('Cardiology to ICU', 'cardiology', 'icu', 'conditional', 'critical_care');
```

## 📊 Usage Instructions

### For Registration Staff
1. Register patients normally
2. System automatically adds to triage queue
3. Priority assigned based on age and workflow type
4. Queue number displayed in success message

### For Department Staff
1. **Queue Panel** appears on department pages
2. **Patient List** shows waiting patients in priority order
3. **Click/Double-click** patient to load their details
4. **Complete Service** to route to next department
5. **Sound Toggle** available in top navigation

### For Administrators
1. **Queue Dashboard** - `admin/queue` for overview
2. **Department Queues** - `admin/queue/department/triage`
3. **Configuration** - Modify settings in database
4. **Reports** - Queue statistics and performance metrics

## 🔊 Sound Notifications

### How It Works
- New patients trigger sound alerts
- Different sounds for different priority levels
- Sound toggle available in navbar
- Stored preference in localStorage

### Sound Types
- **New Patient** - Single beep (600Hz)
- **Patient Called** - Double beep (1000Hz)
- **Urgent Alert** - Rapid beeps (1200Hz)
- **Service Completed** - Success tone (800Hz)

## 📱 Mobile & Responsive
- Queue panels adapt to screen size
- Touch-friendly patient selection
- Mobile notification sounds work
- Responsive queue dashboard

## 🐛 Troubleshooting

### Common Issues

#### Queue Not Showing
1. Check JavaScript console for errors
2. Verify database tables exist
3. Ensure `Queue_model` is loaded in controller
4. Check file permissions on assets/js/

#### Notifications Not Working
1. Verify cache directory exists (`application/cache/`)
2. Check file write permissions
3. Ensure notification endpoints are accessible
4. Browser may block programmatic sounds - user interaction required

#### Patients Not Auto-Added to Queue
1. Check `addPatientToQueue()` method in Patient controller
2. Verify queue model methods work
3. Check database foreign key constraints
4. Review error logs for exceptions

### Debug Tools
- **Queue Status Debug** - `admin/queue/debug_queue_status`
- **Test Patient Creation** - `admin/queue/add_test_patient_to_queue`
- **JavaScript Debug** - `window.debugQueue()` in browser console

## 🔐 Security Considerations

### Access Control
- Queue endpoints require authentication
- Department access controlled by user roles
- Sensitive patient data protected

### Data Protection
- Queue records linked to patient data
- Automatic cleanup of old queue records
- No sensitive data in JavaScript variables

## 📈 Performance Optimization

### Database Optimization
- Indexed frequently queried columns
- Automated cleanup of completed queues
- Efficient joins for patient details

### JavaScript Performance
- 30-second polling intervals
- Debounced API calls
- Minimal DOM manipulation

### Caching Strategy
- File-based notification caching
- Queue statistics caching
- Local storage for user preferences

## 🎯 Best Practices

### For Development
1. Always test queue integration after code changes
2. Monitor error logs for queue-related issues
3. Keep queue JavaScript updated across modules
4. Use consistent naming for queue-related functions

### For Staff Training
1. Show staff how to use queue panels
2. Explain priority system importance
3. Train on patient routing workflows
4. Demonstrate sound notification controls

### For Maintenance
1. Regular cleanup of old queue records
2. Monitor queue performance metrics
3. Update department configurations as needed
4. Backup queue configuration data

## 📞 Support & Maintenance

### Regular Tasks
- **Daily**: Monitor queue performance
- **Weekly**: Clean old completed queues
- **Monthly**: Review queue configuration
- **Quarterly**: Update department workflows

### Monitoring
- Queue length by department
- Average wait times
- Patient flow through departments
- Staff utilization metrics

---

## 🎉 Success! Your Queue System is Ready

The queue management system is now fully integrated into your hospital management system. Staff will see queue panels on their department pages, patients will be automatically routed through the hospital workflow, and you'll have real-time visibility into patient flow.

**Need Help?** 
- Check the troubleshooting section above
- Review error logs in `application/logs/`
- Test with the debug endpoints provided

**Want to Customize?**
- Modify department configurations in the database
- Update JavaScript integration for new workflows  
- Add custom priority rules as needed

Your hospital now has a modern, efficient queue management system! 🏥✨